/*
 * Flash partitions described by the OF (or flattened) device tree
 *
 * Copyright © 2006 MontaVista Software Inc.
 * Author: Vitaly Wool <vwool@ru.mvista.com>
 *
 * Revised to handle newer style flash binding by:
 *   Copyright © 2007 David Gibson, IBM Corporation.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/of.h>
#include <linux/mtd/mtd.h>
#include <linux/slab.h>
#include <linux/mtd/partitions.h>

extern char *flash_type_name;

static bool node_has_compatible(struct device_node *pp)
{
	return of_get_property(pp, "compatible", NULL);
}

#define DNI_NAND_PARTITION
#ifdef DNI_NAND_PARTITION

#define DNI_BOARD_DATA1_PARTITION_OFFSET 0x00d00000
#define DNI_BOARD_DATA1_PARTITION_LENGTH 0x00080000
#define DNI_MODEL_ID_OFFSET 85
#define DNI_MODEL_ID_LENGTH 16
struct dni_nor_nand_partition_entry {
	u32	offset;	/* bytes */
	u32	size;	/* bytes */
	char	*name;
};

struct dni_nor_nand_partition_entry dni_rbs10_nor_partition[] = {
	{ 0x00180000, 0x00040000, "config" },
	{ 0x001c0000, 0x00010000, "pot" },
	{ 0x001d0000, 0x00010000, "boarddata1" },
	{ 0x001e0000, 0x001f0000, "language" },
	{ 0x003d0000, 0x00010000, "https_cert" },
	{ 0x003e0000, 0x00010000, "oopsdump" },
	{ 0x003f0000, 0x01c10000, "firmware" },
	{ 0x003f0000, 0x00320000, "kernel" },
	{ 0x00710000, 0x018f0000, "rootfs" }
};

struct dni_nor_nand_partition_entry dni_rbr10_nand_partition[] = {
	{ 0x00b80000, 0x00080000, "0:ART.bak" },
	{ 0x00c00000, 0x00100000, "config" },
	{ 0x00d00000, 0x00080000, "boarddata1" },
	{ 0x00d80000, 0x00080000, "pot" },
	{ 0x00e00000, 0x00080000, "boarddata1.bak" },
	{ 0x00e80000, 0x00300000, "language" },
	{ 0x01180000, 0x00080000, "cert" },
	{ 0x01200000, 0x02800000, "firmware" },
	{ 0x01200000, 0x00320000, "kernel" },
	{ 0x01520000, 0x024e0000, "rootfs" },
	{ 0x03a00000, 0x00a00000, "secdata" },
	{ 0x04400000, 0x03c00000, "ntgrdata" }
};
#endif

static int parse_ofpart_partitions(struct mtd_info *master,
				   struct mtd_partition **pparts,
				   struct mtd_part_parser_data *data)
{
	struct device_node *node;
	const char *partname;
	char dni_model_id[DNI_MODEL_ID_LENGTH + 1] = {0};
	struct device_node *pp;
	int nr_parts, i, blocks, ret_len, count;
	struct dni_nor_nand_partition_entry *dni_nor_nand_partition; 


	if (!data)
		return 0;

	node = data->of_node;
	if (!node)
		return 0;

	/* First count the subnodes */
	nr_parts = 0;
	for_each_child_of_node(node,  pp) {
		if (node_has_compatible(pp))
			continue;

		nr_parts++;
	}

	if (nr_parts == 0)
		return 0;

	*pparts = kzalloc(32 * sizeof(**pparts), GFP_KERNEL);
	if (!*pparts)
		return -ENOMEM;

	i = 0;
	for_each_child_of_node(node,  pp) {
		const __be32 *reg;
		int len;
		int a_cells, s_cells;

		if (node_has_compatible(pp))
			continue;

		reg = of_get_property(pp, "reg", &len);
		if (!reg) {
			nr_parts--;
			continue;
		}

		a_cells = of_n_addr_cells(pp);
		s_cells = of_n_size_cells(pp);
		(*pparts)[i].offset = of_read_number(reg, a_cells);
		(*pparts)[i].size = of_read_number(reg + a_cells, s_cells);

		partname = of_get_property(pp, "label", &len);
		if (!partname)
			partname = of_get_property(pp, "name", &len);
		(*pparts)[i].name = partname;

		if (of_get_property(pp, "read-only", &len))
			(*pparts)[i].mask_flags |= MTD_WRITEABLE;

		if (of_get_property(pp, "lock", &len))
			(*pparts)[i].mask_flags |= MTD_POWERUP_LOCK;

		i++;
	}

	if (!i) {
		of_node_put(pp);
		pr_err("No valid partition found on %s\n", node->full_name);
		kfree(*pparts);
		*pparts = NULL;
		return -EINVAL;
	}

#ifdef DNI_NAND_PARTITION
	/* default get 14 mtd partitions.
	 * 0	0:SBL1
	 * 1	0:MIBIB
	 * 2	0:BOOTCONFIG
	 * 3	0:QSEE
	 * 4	0:QSEE_1
	 * 5	0:CDT
	 * 6	0:CDT_1
	 * 7	0:BOOTCONFIG1
	 * 8	0:APPSBLENV
	 * 9	0:APPSBL
	 * 10	0:APPSBL_1
	 * 11	0:ART
	 * 12	rootfs
	 * 13	rootfs_1
	 */

	blocks = (DNI_BOARD_DATA1_PARTITION_LENGTH / master->erasesize); 
	for(i = 0; i < blocks; i++)
		/* To Do: check the backup partition of boarddata1 if all all blocks are bad block? */
		if(!mtd_block_isbad(master,DNI_BOARD_DATA1_PARTITION_OFFSET + i * master->erasesize))
				break;
	mtd_read(master, DNI_BOARD_DATA1_PARTITION_OFFSET + i * master->erasesize + DNI_MODEL_ID_OFFSET, sizeof(dni_model_id) - 1,
			&ret_len, (void *)dni_model_id);

    printk("[DNI MTD PARTITION] model id:|%s|  flash_type:|%s| dni_model_id[0]=%x dni_model_id[1]=%x dni_model_id[2]=%x\n", 
            dni_model_id, flash_type_name, dni_model_id[0], dni_model_id[1], dni_model_id[2]);
	if ((strcmp(dni_model_id, "RBR10") == 0) || 
            (dni_model_id[0] == 0xff && dni_model_id[1] == 0xff && dni_model_id[2] == 0xff && strcmp(flash_type_name,"NAND_FLASH") == 0)){
		strcpy(dni_model_id, "RBR10");
		nr_parts = 12;
		dni_nor_nand_partition = dni_rbr10_nand_partition;
		count = sizeof(dni_rbr10_nand_partition)/sizeof(struct dni_nor_nand_partition_entry);
	}
	else {
		strcpy(dni_model_id, "RBS10");
		nr_parts = 8;
		dni_nor_nand_partition = dni_rbs10_nor_partition;
		count = sizeof(dni_rbs10_nor_partition)/sizeof(struct dni_nor_nand_partition_entry);
	}

	printk(KERN_INFO "use %s partition table\n", dni_model_id);

	for (i = 0; i < count; i++, nr_parts++) {
		(*pparts)[nr_parts].offset = dni_nor_nand_partition[i].offset;
		(*pparts)[nr_parts].size = dni_nor_nand_partition[i].size;
		(*pparts)[nr_parts].name = dni_nor_nand_partition[i].name;
	}

#endif
	
	return nr_parts;
}

static struct mtd_part_parser ofpart_parser = {
	.owner = THIS_MODULE,
	.parse_fn = parse_ofpart_partitions,
	.name = "ofpart",
};

static int parse_ofoldpart_partitions(struct mtd_info *master,
				      struct mtd_partition **pparts,
				      struct mtd_part_parser_data *data)
{
	struct device_node *dp;
	int i, plen, nr_parts;
	const struct {
		__be32 offset, len;
	} *part;
	const char *names;

	if (!data)
		return 0;

	dp = data->of_node;
	if (!dp)
		return 0;

	part = of_get_property(dp, "partitions", &plen);
	if (!part)
		return 0; /* No partitions found */

	pr_warning("Device tree uses obsolete partition map binding: %s\n",
			dp->full_name);

	nr_parts = plen / sizeof(part[0]);

	*pparts = kzalloc(nr_parts * sizeof(*(*pparts)), GFP_KERNEL);
	if (!*pparts)
		return -ENOMEM;

	names = of_get_property(dp, "partition-names", &plen);

	for (i = 0; i < nr_parts; i++) {
		(*pparts)[i].offset = be32_to_cpu(part->offset);
		(*pparts)[i].size   = be32_to_cpu(part->len) & ~1;
		/* bit 0 set signifies read only partition */
		if (be32_to_cpu(part->len) & 1)
			(*pparts)[i].mask_flags = MTD_WRITEABLE;

		if (names && (plen > 0)) {
			int len = strlen(names) + 1;

			(*pparts)[i].name = names;
			plen -= len;
			names += len;
		} else {
			(*pparts)[i].name = "unnamed";
		}

		part++;
	}

	return nr_parts;
}

static struct mtd_part_parser ofoldpart_parser = {
	.owner = THIS_MODULE,
	.parse_fn = parse_ofoldpart_partitions,
	.name = "ofoldpart",
};

static int __init ofpart_parser_init(void)
{
	register_mtd_parser(&ofpart_parser);
	register_mtd_parser(&ofoldpart_parser);
	return 0;
}

static void __exit ofpart_parser_exit(void)
{
	deregister_mtd_parser(&ofpart_parser);
	deregister_mtd_parser(&ofoldpart_parser);
}

module_init(ofpart_parser_init);
module_exit(ofpart_parser_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Parser for MTD partitioning information in device tree");
MODULE_AUTHOR("Vitaly Wool, David Gibson");
/*
 * When MTD core cannot find the requested parser, it tries to load the module
 * with the same name. Since we provide the ofoldpart parser, we should have
 * the corresponding alias.
 */
MODULE_ALIAS("ofoldpart");
