/*
 * ntpclient.c - NTP client
 *
 * Copyright (C) 1997, 1999, 2000, 2003, 2006, 2007, 2010, 2015  Larry Doolittle  <larry@doolittle.boa.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License (Version 2,
 *  June 1991) as published by the Free Software Foundation.  At the
 *  time of writing, that license was published by the FSF with the URL
 *  http://www.gnu.org/copyleft/gpl.html, and is incorporated herein by
 *  reference.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  Possible future improvements:
 *      - Write more documentation  :-(
 *      - Support leap second processing
 *      - Support IPv6
 *      - Support multiple (interleaved) servers
 *
 *  Compile with -DPRECISION_SIOCGSTAMP if your machine really has it.
 *  Older kernels (before the tickless era, pre 3.0?) only give an answer
 *  to the nearest jiffy (1/100 second), not so interesting for us.
 *
 *  If the compile gives you any flak, check below in the section
 *  labelled "XXX fixme - non-automatic build configuration".
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>     /* gethostbyname */
#include <arpa/inet.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#include <fcntl.h>
#include <net/if.h>

#include "product_config.h"

#if PRECISION_SIOCGSTAMP
#include <sys/ioctl.h>
#endif
#include <sys/time.h>

#include "ntpclient.h"
/*
 * dni_system
 * output: the file that the output will be redirected to, or fill with NULL if you don't want to redirect the output
 * mode: the mode that you want to redirect. R_NORMAL = 'cmd > output', R_STDERR = 'cmd > output 2>&1', R_APPEND = 'cmd >> output', R_OUTPUT = 'cmd 1> output 2>err'
 * cmd: the absolute path of the command (executable binary or script)
 * ...: the argument strings passed to the command, must end with NULL
 * example 1: dni_safe_system(NULL, NULL, R_NORMAL, "/bin/ls", NULL);
 * example 2: dni_safe_system("/tmp/result", NULL, R_NORMAL, "/bin/ls", "-l", NULL);
 */
typedef void (*sighandler_t)(int);
#include <stdarg.h>
#include <assert.h>
#include <stddef.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/stat.h>
extern char **__environ;
#define MAX_SYSTEM_ARG 100
#ifndef R_NORMAL
#define R_NORMAL 0
#define R_STDERR 0x01
#define R_APPEND 0x02
#define R_OUTPUT 0x03
#endif

int dni_safe_system(const char *output, const char *output2, unsigned char mode, const char *cmd,
                    ...)
{
    int wait_val, pid;
    sighandler_t save_quit, save_int, save_chld;
    save_quit = signal(SIGQUIT, SIG_IGN);
    save_int = signal(SIGINT, SIG_IGN);
    save_chld = signal(SIGCHLD, SIG_DFL);

    if ((pid = vfork()) < 0) {
        signal(SIGQUIT, save_quit);
        signal(SIGINT, save_int);
        signal(SIGCHLD, save_chld);
        return -1;
    }
    if (pid == 0) {
        int fd, i = 0;
        char *cmdpath;
        va_list args;
        char *argv[MAX_SYSTEM_ARG] = {NULL};
        char *p;

        signal(SIGQUIT, SIG_DFL);
        signal(SIGINT, SIG_DFL);
        signal(SIGCHLD, SIG_DFL);

        argv[0] = cmdpath = (char *)cmd;

        va_start(args, cmd);
        for (i = 1;;) {
            p = va_arg(args, char *);
            if (p == NULL) {
                break;
            }
            if (i < MAX_SYSTEM_ARG) {
                argv[i++] = p;
            } else {
                printf("warning: drop some argv!\n");
                break;
            }
        }
        va_end(args);

        if (output) {
            if (!(mode & R_APPEND)) {
                unlink(output);
            }

            if ((fd = open(output, O_WRONLY | O_CREAT | ((mode & R_APPEND) ? O_APPEND : 0), 0666)) < 0) {
                printf("can not open %s %s\n", output, strerror(errno));
                _exit(127);
            }

            dup2(fd, STDOUT_FILENO);
            if (mode & R_STDERR) {
                dup2(fd, STDERR_FILENO);
            }
            close(fd);
        }
        if (output2) {
            if (!(mode & R_APPEND)) {
                unlink(output2);
            }
            if ((fd = open(output2, O_WRONLY | O_CREAT | ((mode & R_APPEND) ? O_APPEND : 0), 0666)) < 0) {
                printf("can not open %s %s\n", output, strerror(errno));
                _exit(127);
            }
            if (mode & R_OUTPUT) {
                dup2(fd, STDERR_FILENO);
            }
            close(fd);
        }

        execve(cmdpath, (char *const *)argv, __environ);
        _exit(127);
    }
    /* Signals are not absolutly guarenteed with vfork */
    signal(SIGQUIT, SIG_IGN);
    signal(SIGINT, SIG_IGN);

    if (wait4(pid, &wait_val, 0, 0) == -1) {
        wait_val = -1;
    }

    signal(SIGQUIT, save_quit);
    signal(SIGINT, save_int);
    signal(SIGCHLD, save_chld);
    return wait_val;
}

#if ENABLE_NETGEAR_XCLOUD_TIME
#define DEV_URL "devicelocation.dev.ngxcld.com"
#define QA_URL "devicelocation.qa.ngxcld.com"
#define PROD_URL "devicelocation.ngxcld.com"
/* the content of file is stored in static array */
static char *cat_file(char *name)
{
    int i;
    FILE *fp;
    static char buf[512];

    buf[0] = '\0';

    fp = fopen(name, "r");
    if (fp == NULL) {
        return buf;
    }
    if(fgets(buf, sizeof(buf), fp)) {};
    fclose(fp);

    i = 0;
    while (buf[i] != '\0' && buf[i] != '\r' && buf[i] != '\n') {
        i++;
    }
    buf[i] = '\0';

    return buf;
}


static char *get_value_from_dto(char *string)
{
    char cmd[128] = {0};

    snprintf(cmd, sizeof(cmd) - 1, "d2 -s %s > /tmp/ra_d2_info", string);
    if(system(cmd)) {};

    return cat_file("/tmp/ra_d2_info");
}

#if OPENWRT_UCI
static char *get_value_from_config(char *string)
{
    dni_safe_system("/tmp/cache/ntpclient/config_value", NULL, R_NORMAL, "/bin/config", "get", string,
                    NULL);

    return cat_file("/tmp/cache/ntpclient/config_value");
}
#endif

static void getRaCloudTime()
{
#define CURRENT_TIME_PATH "device-location/syncTime"
    char user[64] = {0}, pw[64] = {0}, sn[32] = {0}, model[8] = {0}, *p = NULL;
    int env = 0;
    p = get_value_from_dto("xagentcfg[0].x_agent_id");

    if(p != NULL) {
        strncpy(user, p, sizeof(user) - 1);
    } else {
        user[0] = '0';
    }

    if(strncmp(user, "DEV-", 4) == 0) {
        env = 1;
    } else if(strncmp(user, "QA-", 3) == 0) {
        env = 2;
    } else {
        env = 0;
    }

    p = get_value_from_dto("xagentcfg[0].x_agent_claim_code");
    if(p != NULL) {
        strncpy(pw, p, sizeof(pw) - 1);
    } else {
        pw[0] = '0';
    }

#if OPENWRT_UCI
    p = get_value_from_config("dgc.project.board_data.sn");
    if(p != NULL) {
        strlcpy(sn, p, sizeof(sn) - 1);
    } else {
        sn[0] = '\0';
    }

    p = get_value_from_config("dgc.project.board_data.module_name");
    if(p != NULL) {
        strlcpy(model, p, sizeof(model) - 1);
    } else {
        model[0] = '\0';
    }
#else
    strncpy(sn, cat_file("/tmp/Seria_Number"), sizeof(sn) - 1);
    sn[sizeof(sn) - 1] = '\0';
    strncpy(model, cat_file("/module_name"), sizeof(model) - 1);
    model[sizeof(model) - 1] = '\0';
#endif

    if(access("/tmp/xCloud_time", F_OK) == 0) {
        remove("/tmp/xCloud_time");
    }

    /* sizeof(xlcoud_link) = sizeof("format string")+sizeof(DEV_URL) + sizeof(CURRENT_TIME_PATH)
     * format string include "{haedwareId: model: }", It have 19 bytes,
     * max len of DEV_URL(devicelocation.dev.ngxcld.com) = 29
     * max len of CURRENT_TIME_PATH = 24
     * they maybe change by ntgr, should notice!
     * */
    char xcloud_link[82] = {0};
    /* sizeof(xlcoud_data) = sizeof("format string")+sizeof(sn) + sizeof(model)
     * format string include "https:///", it have 8 bytes
     * max len of sn = 32
     * max len of model = 8
     * */
    char xcloud_data[64] = {0};
    snprintf(xcloud_data, sizeof(xcloud_data) - 1, "{\"hardwareId\":\"%s\",\"model\":\"%s\"}", sn,
             model);
    if(env == 1) {
        snprintf(xcloud_link, sizeof(xcloud_link) - 1, "https://%s/%s", DEV_URL, CURRENT_TIME_PATH);
    } else if (env == 2) {
        snprintf(xcloud_link, sizeof(xcloud_link) - 1, "https://%s/%s", QA_URL, CURRENT_TIME_PATH);
    } else {
        snprintf(xcloud_link, sizeof(xcloud_link) - 1, "https://%s/%s", PROD_URL, CURRENT_TIME_PATH);
    }

    /*"curl
     * --capath /etc/ssl/certs
     * -X POST https://%s/%s
     * -H \"content-type: application/json\"
     * -d \'{\"hardwareId\":\"%32s\",\"model\":\"%8s\"}\'
     * -o /tmp/xCloud_time
     * --connect-timeout 3 */
    /* Through system function's block mechaism to wait server reponse*/
    if(system("echo '[ntpclient] getRaCloudTime, Query...' > /dev/console")) {};
    dni_safe_system("/tmp/xCloud_time", NULL, R_NORMAL,
                    "/bin/curl", "-k",
                    "-X", "POST", xcloud_link,
                    "-H", "content-type: application/json",
                    "-d",  xcloud_data,
                    "--connect-timeout", "5",
                    NULL);
}

static void setRaCloudTime()
{
    FILE *ct_fp;
    char time_buf[256] = {0};
    char *data = NULL;
    struct timeval tm_ra = {0};
    int data_flag = 0;

    ct_fp = fopen("/tmp/xCloud_time", "r");
    if(ct_fp) {
        while(fgets(time_buf, sizeof(time_buf) - 1, ct_fp)) {
            if(strstr(time_buf, "timestamp")) {
                strtok(time_buf, ": ");
                data = strtok(NULL, ",");
                if(data != NULL) {
                    tm_ra.tv_sec = atol(data);
                    data_flag = 1;
                }
            }
        }
        fclose(ct_fp);
    }
    if(data_flag == 1) {
        if(settimeofday(&tm_ra, NULL) < 0) {
            perror("[ntp] set ra time error");
        }
        /* do corresponding actions in ntp_updated once updating the time by xCloud */
        if(system("ntp_updated xcloud")) {};
    }
}

static inline void updateRaCloudTime()
{
    if(access("/tmp/system_time_updated", F_OK) != 0) {
        getRaCloudTime();
        setRaCloudTime();
    }
}
#endif

#if OPENWRT_UCI
void ntp_syn_status(unsigned short fail_type)
{
    char buf[72] = {0};
    snprintf(buf, sizeof(buf) - 1, "{ \"FailReason\": \"%u\" }", fail_type);
    dni_safe_system("/tmp/cache/ntpclient/status", NULL, R_NORMAL, "/bin/echo", buf, NULL);
}

void update_ntp_port_status(unsigned short port)
{
    char buf[72] = {0};
    snprintf(buf, sizeof(buf) - 1, "{ \"PortNumber\": \"%u\" }", port);
    dni_safe_system("/tmp/cache/ntpclient/port", NULL, R_NORMAL, "/bin/echo", buf, NULL);
}
#endif

/* Default to the RFC-4330 specified value */
#ifndef MIN_INTERVAL
#define MIN_INTERVAL 15
#endif

#if ENABLE_DEBUG
#define DEBUG_OPTION "d"
int debug = 0;
#else
#define DEBUG_OPTION
#endif

#if ENABLE_REPLAY
#define  REPLAY_OPTION   "r"
#else
#define  REPLAY_OPTION
#endif

extern char *optarg;  /* according to man 2 getopt */

#include <stdint.h>
typedef uint32_t u32;  /* universal for C99 */
/* typedef u_int32_t u32;   older Linux installs? */

/* XXX fixme - non-automatic build configuration */
#ifdef __linux__
#include <sys/utsname.h>
#include <sys/time.h>
#include <sys/timex.h>
#include <netdb.h>
#else
extern int getaddrinfo(const char *hostname, const char *service, const struct addrinfo *hints,
                       struct addrinfo **result);
extern int h_errno;
#define herror(hostname) \
	fprintf(stderr,"Error %d looking up hostname %s\n", h_errno,hostname)
#endif
/* end configuration for host systems */

#define JAN_1970        0x83aa7e80      /* 2208988800 1970 - 1900 in seconds */
#define NTP_PORT (123)
#define DAY_TIME 86400
// According to NTGR demand, reduce period to 1s.
#define NETGEAR_PERIOD 1

/* How to multiply by 4294.967296 quickly (and not quite exactly)
 * without using floating point or greater than 32-bit integers.
 * If you want to fix the last 12 microseconds of error, add in
 * (2911*(x))>>28)
 */
#define NTPFRAC(x) ( 4294*(x) + ( (1981*(x))>>11 ) )

/* The reverse of the above, needed if we want to set our microsecond
 * clock (via clock_settime) based on the incoming time in NTP format.
 * Basically exact.
 */
#define USEC(x) ( ( (x) >> 12 ) - 759 * ( ( ( (x) >> 10 ) + 32768 ) >> 16 ) )

/* Converts NTP delay and dispersion, apparently in seconds scaled
 * by 65536, to microseconds.  RFC-1305 states this time is in seconds,
 * doesn't mention the scaling.
 * Should somehow be the same as 1000000 * x / 65536
 */
#define sec2u(x) ( (x) * 15.2587890625 )

struct ntptime {
    unsigned int coarse;
    unsigned int fine;
};

struct ntp_control {
    u32 time_of_send[2];
    int live;
    int set_clock;   /* non-zero presumably needs root privs */
    int probe_count;
    int cycle_time;
    int goodness;
    int cross_check;
    char serv_addr[4];
};

/* prototypes for some local routines */
static void send_packet(int usd, u32 time_sent[2]);
static int rfc1305print(u32 *data, struct ntptime *arrival, struct ntp_control *ntpc, int *error);
/* static void udp_handle(int usd, char *data, int data_len, struct sockaddr *sa_source, int sa_len); */

#if ENABLE_BOOT_RELAY
#define BOOT_RELAY_OPTION "y"                  // add an option to force to do ntp boot  relay 
int boot_relay = 0;
#else
#define BOOT_RELAY_OPTION
#endif

int ap_mode = 0;
char *wan_proto = "dhcp";
char *wan_ifname = "brwan";
char *lan_ifname = "br0";

char *time_zone = "GMT+8";
#if NETGEAR_DAYLIGHT_SAVING_TIME
#define DST_OPTION "e:"
int dst_flag = 0;
#else
#define DST_OPTION
#endif

enum NTP_FailReason {
    FAIL_none = 0,
    FAIL_wan_conn_no_up,		/*The WAN connection is NOT up!*/
    FAIL_no_server_response		/*Do not receive response from NTP server*/
};

#define PPP_STATUS	"/etc/ppp/ppp0-status"
#define PPP1_STATUS      "/etc/ppp/pppoe1-status"
#define BPA_STATUS	"/tmp/bpa_info"
#define CABLE_FILE	"/tmp/port_status"
#define DSL_CABLE_FILE  "/tmp/dsl_port_status"
#define RA_PORTNUM  "/tmp/ntp/ra_portnum"
#define RA_FAILREASON  "/tmp/ntp/ra_failreason"

#if OPENWRT_UCI==0
static struct in_addr get_ipaddr(char *ifname)
{
    int fd;
    struct ifreq ifr;
    struct in_addr pa;

    pa.s_addr = 0;
    fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0) {
        return pa;
    }

    memset(&ifr, 0, sizeof(ifr));
    ifr.ifr_addr.sa_family = AF_INET;
    strcpy(ifr.ifr_name, ifname);
    if (ioctl(fd, SIOCGIFADDR, &ifr) == 0) {
        pa = ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr;
    }
    close(fd);

    return pa;
}

int get_ipv6addr(char *ifname)
{
    int result = 0;
    FILE *fp;
    char buf[256] = {0};

    fp = fopen("/proc/net/if_inet6", "r");
    if(fp == NULL) {
        perror("read if_inet6_addr");
        return 0;
    }
    while(fgets(buf, sizeof(buf), fp)) {
        if(strstr(buf, ifname)) {
            if(strncmp(buf, "fe80", 4) != 0) {
                result = 1;
                break;
            }
        }
    }
    fclose(fp);
    return result;
}
#endif

/* '\0' means read failed */
static char readc(char *file)
{
    int fd;
    char value;

    fd = open(file, O_RDONLY, 0666);
    if (fd < 0) {
        return 0;
    }
    if (read(fd, &value, 1) != 1) {
        value = 0;
    }
    close(fd);

    return value;
}

/* DHCP / StaticIP ... */
static inline int eth_up(void)
{
    return readc(CABLE_FILE) == '1';
}

/* DSL-DHCP / DSL-StaticIP ... */
static inline int dsl_up(void)
{
    return readc(DSL_CABLE_FILE) == '1';
}

/* It is ONLY used for PPPoE/PPTP mode. */
static inline int ppp_up(void)
{
    return readc(PPP_STATUS) == '1';
}

/* It is ONLY used for mulpppoe mode. */
static inline int ppp_up_mul(void)
{
    return readc(PPP1_STATUS) == '1';
}

/* 1). `up time: %lu`; 2). `down time: %lu` */
static inline int bpa_up(void)
{
    return readc(BPA_STATUS) == 'u';
}

/*under ap mode, wan_ifname is br0, wan_proto is dhcp*/
static inline int apmode_up(void)
{
    return ap_mode == 1;
}

#if OPENWRT_UCI==0
static int net_verified(void)
{
    int alive;

    if (!strcmp(wan_proto, "pppoe") || !strcmp(wan_proto, "pptp") || !strcmp(wan_proto, "pppoa")
        || !strcmp(wan_proto, "ipoa")) {
        alive = ppp_up();
    } else if (!strcmp(wan_proto, "bigpond")) {
        alive = bpa_up();
    } else if (!strcmp(wan_proto, "mulpppoe1")) {
        alive = ppp_up_mul();
    } else {
        alive = eth_up() || dsl_up() || apmode_up();
    }

    return alive;
}
#endif

int wan_conn_up(void)
{
#if OPENWRT_UCI
    FILE *fp = NULL;
    char buf[128] = {0};
    int ret_val = 0;
    fp = fopen("/tmp/cache/sysmon_server/internet.status", "r");
    if(fp == NULL) {
        if(debug) {
            printf("[ntpclient] open internet.status failed. \n");
        }
        return ret_val;
    }
    while(fgets(buf, sizeof(buf) - 1, fp)) {
        if(strstr(buf, "gw_status")) {
            if(sscanf(buf, "%*[^:]:%d", &ret_val) != 1) {
                ret_val = 0;
                if(debug) {
                    printf("[ntpclient] get gw_status fail, buf:%s\n", buf);
                }
            }
            break;
        }
    }
    fclose(fp);
    return ret_val;
#else
    int alive;
    struct in_addr ip;
    int result;

    alive = net_verified();
    if (alive == 0) {
        ip.s_addr = 0;
    } else {
        ip = get_ipaddr(wan_ifname);
    }

    /*check ipv6 address for some ipv6 type without ipv4 address, such as v6plus & Dual-Stack Lite mode */
    if(ip.s_addr == 0 && get_ipv6addr(wan_ifname) == 0) {
        result = 0;
    } else {
        result = 1;
    }
    return result;
#endif
}

#if NETGEAR_DAYLIGHT_SAVING_TIME
void daylight_saving_setting(void)
{
    time_t now;
    struct tm *tm;
    char date_buf[128];
    time(&now);
    /*the config time_zone includes the daylight saving time offset, so DUT will get the correct locale time by TZ*/
    setenv("TZ", time_zone, 1);
    tm = localtime(&now);
    sprintf(date_buf, "%.2d%.2d%.2d%.2d%d", tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min,
            tm->tm_year + 1900);
    dni_safe_system(NULL, NULL, R_NORMAL, "/bin/date", "-s", date_buf, NULL);
}
#endif

/*
  * [NETGEAR SPEC V1.6] 8.6 NTP:
  *
  * NETGEAR NTP Servers
  *	time-a.netgear.com
  *	time-b.netgear.com
  *	time-c.netgear.com
  *	time-d.netgear.com
  *	time-e.netgear.com
  * 	time-f.netgear.com
  *	time-g.netgear.com
  *	time-h.netgear.com
  *
  * NETGEAR NTP Server Assignments
  * The primary and secondary Netgear NTP servers will be selected based upon
  * the user-selected time zone.
  */
struct server_struct {
    char *primary;
    char *secondary;
#if ENABLE_THIRD_NTP_SERVER==1
    char *thirdary;
#endif
};

#if ENABLE_THIRD_NTP_SERVER==1
void select_ntp_servers(char **primary, char **secondary, char **thirdary)
#else
void select_ntp_servers(char **primary, char **secondary)
#endif
{
    char *p;
    int tmzone = 0;

#if ENABLE_THIRD_NTP_SERVER==1
    static struct server_struct ntpsvrs[] = {
        /*00 ~ 02: GMT+0,1,2	Greenwich, Amsterdam, Athens */
        { "time-g.netgear.com", "time-h.netgear.com", "0.0.0.0" },
        { "time-g.netgear.com", "time-h.netgear.com", "0.0.0.0" },
        { "time-g.netgear.com", "time-h.netgear.com", "0.0.0.0" },

        /* 03 ~ 05: GMT+3,4,5	Baghdad, Abu Dhabi, Ekaterinaburg */
        { "time-f.netgear.com", "time-g.netgear.com", "0.0.0.0" },
        { "time-f.netgear.com", "time-g.netgear.com", "0.0.0.0" },
        { "time-f.netgear.com", "time-g.netgear.com", "0.0.0.0" },

        /* 06 ~ 08: GMT+6,7,8	Almaty, Bangkok, Beijing */
        { "time-e.netgear.com", "time-f.netgear.com", "0.0.0.0" },
        { "time-e.netgear.com", "time-f.netgear.com", "0.0.0.0" },
        { "time-e.netgear.com", "time-f.netgear.com", "0.0.0.0" },

        /* 09 ~ 13: GMT+9,10,11,12,13	Tokyo, Brisbane, Solomon Islands */
        { "time-d.netgear.com", "time-e.netgear.com", "0.0.0.0" },
        { "time-d.netgear.com", "time-e.netgear.com", "0.0.0.0" },
        { "time-d.netgear.com", "time-e.netgear.com", "0.0.0.0" },
        { "time-d.netgear.com", "time-e.netgear.com", "0.0.0.0" },
        { "time-d.netgear.com", "time-e.netgear.com", "0.0.0.0" },

        /* 14 ~ 16: GMT-1,2,3	Azores, Mid-Atlantic, Brazil */
        { "time-h.netgear.com", "time-a.netgear.com", "0.0.0.0" },
        { "time-h.netgear.com", "time-a.netgear.com", "0.0.0.0" },
        { "time-h.netgear.com", "time-a.netgear.com", "0.0.0.0" },

        /* 17 ~ 19: GMT-4,5,6	Canada, USA/Eastern, USA/Central */
        { "time-a.netgear.com", "time-b.netgear.com", "time-e.netgear.com" },
        { "time-a.netgear.com", "time-b.netgear.com", "time-e.netgear.com" },
        { "time-a.netgear.com", "time-b.netgear.com", "time-e.netgear.com" },

        /* 20 ~ 22: GMT-7,8,9	USA/Mountain, USA/Pacific, Alaska */
        { "time-b.netgear.com", "time-c.netgear.com", "time-a.netgear.com" },
        { "time-b.netgear.com", "time-c.netgear.com", "time-a.netgear.com" },
        { "time-b.netgear.com", "time-c.netgear.com", "time-a.netgear.com" },

        /* 23 ~ 25: GMT-10,11,12	Hawaii, Samoa, Eniwetok */
        { "time-c.netgear.com", "time-d.netgear.com", "0.0.0.0" },
        { "time-c.netgear.com", "time-d.netgear.com", "0.0.0.0" },
        { "time-c.netgear.com", "time-d.netgear.com", "0.0.0.0" },
    };
#else
    static struct server_struct ntpsvrs[] = {
        /*00 ~ 02: GMT+0,1,2	Greenwich, Amsterdam, Athens */
        { "time-g.netgear.com", "time-h.netgear.com" },
        { "time-g.netgear.com", "time-h.netgear.com" },
        { "time-g.netgear.com", "time-h.netgear.com" },

        /* 03 ~ 05: GMT+3,4,5	Baghdad, Abu Dhabi, Ekaterinaburg */
        { "time-f.netgear.com", "time-g.netgear.com" },
        { "time-f.netgear.com", "time-g.netgear.com" },
        { "time-f.netgear.com", "time-g.netgear.com" },

        /* 06 ~ 08: GMT+6,7,8	Almaty, Bangkok, Beijing */
        { "time-e.netgear.com", "time-f.netgear.com" },
        { "time-e.netgear.com", "time-f.netgear.com" },
        { "time-e.netgear.com", "time-f.netgear.com" },

        /* 09 ~ 13: GMT+9,10,11,12,13	Tokyo, Brisbane, Solomon Islands */
        { "time-d.netgear.com", "time-e.netgear.com" },
        { "time-d.netgear.com", "time-e.netgear.com" },
        { "time-d.netgear.com", "time-e.netgear.com" },
        { "time-d.netgear.com", "time-e.netgear.com" },
        { "time-d.netgear.com", "time-e.netgear.com" },

        /* 14 ~ 16: GMT-1,2,3	Azores, Mid-Atlantic, Brazil */
        { "time-h.netgear.com", "time-a.netgear.com" },
        { "time-h.netgear.com", "time-a.netgear.com" },
        { "time-h.netgear.com", "time-a.netgear.com" },

        /* 17 ~ 19: GMT-4,5,6	Canada, USA/Eastern, USA/Central */
        { "time-a.netgear.com", "time-b.netgear.com" },
        { "time-a.netgear.com", "time-b.netgear.com" },
        { "time-a.netgear.com", "time-b.netgear.com" },

        /* 20 ~ 22: GMT-7,8,9	USA/Mountain, USA/Pacific, Alaska */
        { "time-b.netgear.com", "time-c.netgear.com" },
        { "time-b.netgear.com", "time-c.netgear.com" },
        { "time-b.netgear.com", "time-c.netgear.com" },

        /* 23 ~ 25: GMT-10,11,12	Hawaii, Samoa, Eniwetok */
        { "time-c.netgear.com", "time-d.netgear.com" },
        { "time-c.netgear.com", "time-d.netgear.com" },
        { "time-c.netgear.com", "time-d.netgear.com" },
    };
#endif

    /*
      *  The config data is opposite with the real time zone value, so ...
      * [GMT-0 --> 00]
      * [GMT+1 ~ +12 --> 14 ~ 25]
      * [GMT-1 ~ -13 --> 1 ~ 13 ]
      */
    p = time_zone;
    if (strncmp(p, "GMT", 3) == 0) {
        p += 3;
        if (strcmp(p, "-0") == 0) {
            tmzone = 0;
        } else if (*p == '-') {
            tmzone = atoi(++p);
        } else {
            tmzone = 13 + atoi(++p);
        }
    } else {
        p = 0;
    }
    //(GMT-03:30) Newfoundland
    //Primary: time-a.netgear.com
    //Secondary: time-b.netgear.com
    if(strstr(time_zone, "GMT+3:30") || strstr(time_zone, "GMT+03:30")) {
        tmzone = 17;
    }

    printf("time zone index is : %d\n", tmzone);
    if (tmzone < 0 || tmzone > 25) {
        tmzone = 0;
    }

    *primary = ntpsvrs[tmzone].primary;
    *secondary = ntpsvrs[tmzone].secondary;
#if ENABLE_THIRD_NTP_SERVER==1
    *thirdary = ntpsvrs[tmzone].thirdary;
#endif
}

void apply_settings(void)
{
    char *pid_file = "/tmp/run/syslogd.pid";
    if (access(pid_file, F_OK) == 0) {
        if(system("kill -USR1 $(cat /tmp/run/syslogd.pid)")) {};
        sleep(1);
    }
    syslog(LOG_WARNING, "[Time synchronized with NTP server]");

#if OPENWRT_UCI
    if(system("[ -f /tmp/cache/ntpclient/ntp_updated ] || { "
              "touch /tmp/cache/ntpclient/ntp_updated; "
              "}")) {};
#else
    /* log the first entry */
    if(system("[ -f /tmp/ntp_updated ] || { "
              "touch /tmp/ntp_updated;"
              "/usr/sbin/ntpst set;"
              "/etc/init.d/cron restart;"
              "}")) {};
#endif

#if NETGEAR_DAYLIGHT_SAVING_TIME
    /* check the daylight saving time*/
    if (dst_flag == 1) {
        daylight_saving_setting();

        /*just for traffic meter when time change*/
        if(system("ntp_updated daylight")) {};
    }
#endif

    /* do corresponding actions in ntp_updated once updating the time */
    if(system("ntp_updated common")) {};
}

static int get_current_freq(void)
{
    /* OS dependent routine to get the current value of clock frequency.
     */
#ifdef __linux__
    struct timex txc;
    txc.modes = 0;
    if (adjtimex(&txc) < 0) {
        perror("adjtimex");
        exit(1);
    }
    return txc.freq;
#else
    return 0;
#endif
}

static int set_freq(int new_freq)
{
    /* OS dependent routine to set a new value of clock frequency.
     */
#ifdef __linux__
    struct timex txc;
    txc.modes = ADJ_FREQUENCY;
    txc.freq = new_freq;
    if (adjtimex(&txc) < 0) {
        perror("adjtimex");
        exit(1);
    }
    return txc.freq;
#else
    return 0;
#endif
}

static void set_time(struct ntptime *new)
{
#if USE_OBSOLETE_GETTIMEOFDAY==0
    /* POSIX 1003.1-2001 way to set the system clock
     */
    struct timespec tv_set;
    /* it would be even better to subtract half the slop */
    tv_set.tv_sec  = new->coarse - JAN_1970;
    /* divide xmttime.fine by 4294.967296 */
    tv_set.tv_nsec = USEC(new->fine) * 1000;
    if (clock_settime(CLOCK_REALTIME, &tv_set) < 0) {
        perror("clock_settime");
        exit(1);
    }
    if (debug) {
        printf("set time to %lu.%.9lu\n", tv_set.tv_sec, tv_set.tv_nsec);
    }
#else
    /* Traditional Linux way to set the system clock
     */
    struct timeval tv_set;
    /* it would be even better to subtract half the slop */
    tv_set.tv_sec  = new->coarse - JAN_1970;
    /* divide xmttime.fine by 4294.967296 */
    tv_set.tv_usec = USEC(new->fine);
    if (settimeofday(&tv_set, NULL) < 0) {
        perror("settimeofday");
        exit(1);
    }
    if (debug) {
        printf("set time to %lu.%.6lu\n", tv_set.tv_sec, tv_set.tv_usec);
    }
#endif
}

static void ntpc_gettime(u32 *time_coarse, u32 *time_fine)
{
#if USE_OBSOLETE_GETTIMEOFDAY==0
    /* POSIX 1003.1-2001 way to get the system time
     */
    struct timespec now;
    clock_gettime(CLOCK_REALTIME, &now);
    *time_coarse = now.tv_sec + JAN_1970;
    *time_fine   = NTPFRAC(now.tv_nsec / 1000);
#else
    /* Traditional Linux way to get the system time
     */
    struct timeval now;
    gettimeofday(&now, NULL);
    *time_coarse = now.tv_sec + JAN_1970;
    *time_fine   = NTPFRAC(now.tv_usec);
#endif
}

static void send_packet(int usd, u32 time_sent[2])
{
    u32 data[12];
#define LI 0
#define VN 4
#define MODE 3
#define STRATUM 0
#define POLL 4
#define PREC -6

    if (debug) {
        fprintf(stderr, "Sending ...\n");
    }
    if (sizeof data != 48) {
        fprintf(stderr, "size error\n");
        return;
    }
    memset(data, 0, sizeof data);
    data[0] = htonl (
                  ( LI << 30 ) | ( VN << 27 ) | ( MODE << 24 ) |
                  ( STRATUM << 16) | ( POLL << 8 ) | ( PREC & 0xff ) );
    data[1] = htonl(1 << 16); /* Root Delay (seconds) */
    data[2] = htonl(1 << 16); /* Root Dispersion (seconds) */
    ntpc_gettime(time_sent, time_sent + 1);
    data[10] = htonl(time_sent[0]); /* Transmit Timestamp coarse */
    data[11] = htonl(time_sent[1]); /* Transmit Timestamp fine   */
    send(usd, data, 48, 0);
}

static void get_packet_timestamp(int usd, struct ntptime *udp_arrival_ntp)
{
#if PRECISION_SIOCGSTAMP
    struct timeval udp_arrival;
    if ( ioctl(usd, SIOCGSTAMP, &udp_arrival) < 0 ) {
        perror("ioctl-SIOCGSTAMP");
        ntpc_gettime(&udp_arrival_ntp->coarse, &udp_arrival_ntp->fine);
    } else {
        udp_arrival_ntp->coarse = udp_arrival.tv_sec + JAN_1970;
        udp_arrival_ntp->fine   = NTPFRAC(udp_arrival.tv_usec);
    }
#else
    (void) usd;  /* not used */
    ntpc_gettime(&udp_arrival_ntp->coarse, &udp_arrival_ntp->fine);
#endif
}

static int check_source(int data_len, struct sockaddr_in *sa_in, unsigned int sa_len,
                        struct ntp_control *ntpc)
{
    struct sockaddr *sa_source = (struct sockaddr *) sa_in;
    (void) sa_len;  /* not used */
    if (debug) {
        printf("packet of length %d received\n", data_len);
        if (sa_source->sa_family == AF_INET) {
            printf("Source: INET Port %d host %s\n",
                   ntohs(sa_in->sin_port), inet_ntoa(sa_in->sin_addr));
        } else {
            printf("Source: Address family %d\n", sa_source->sa_family);
        }
    }
    /* we could check that the source is the server we expect, but
     * Denys Vlasenko recommends against it: multihomed hosts get it
     * wrong too often. */
#if 0
    if (memcmp(ntpc->serv_addr, &(sa_in->sin_addr), 4) != 0) {
        return 1;  /* fault */
    }
#else
    (void) ntpc; /* not used */
#endif
    if (NTP_PORT != ntohs(sa_in->sin_port)) {
        return 1;  /* fault */
    }
    return 0;
}

static double ntpdiff( struct ntptime *start, struct ntptime *stop)
{
    int a;
    unsigned int b;
    a = stop->coarse - start->coarse;
    if (stop->fine >= start->fine) {
        b = stop->fine - start->fine;
    } else {
        b = start->fine - stop->fine;
        b = ~b;
        a -= 1;
    }

    return a * 1.e6 + b * (1.e6 / 4294967296.0);
}

/* Does more than print, so this name is bogus.
 * It also makes time adjustments, both sudden (-s)
 * and phase-locking (-l).
 * sets *error to the number of microseconds uncertainty in answer
 * returns 0 normally, 1 if the message fails sanity checks
 */
static int rfc1305print(u32 *data, struct ntptime *arrival, struct ntp_control *ntpc, int *error)
{
    /* straight out of RFC-1305 Appendix A */
    int li, vn, mode, stratum, poll, prec;
    int delay, disp, refid;
    struct ntptime reftime, orgtime, rectime, xmttime;
    double el_time, st_time, skew1, skew2;
    int freq;
#if ENABLE_DEBUG
    const char *drop_reason = NULL;
#endif

#define Data(i) ntohl(((u32 *)data)[i])
    li      = Data(0) >> 30 & 0x03;
    vn      = Data(0) >> 27 & 0x07;
    mode    = Data(0) >> 24 & 0x07;
    stratum = Data(0) >> 16 & 0xff;
    poll    = Data(0) >>  8 & 0xff;
    prec    = Data(0)       & 0xff;
    if (prec & 0x80) {
        prec |= 0xffffff00;
    }
    delay   = Data(1);
    disp    = Data(2);
    refid   = Data(3);
    reftime.coarse = Data(4);
    reftime.fine   = Data(5);
    orgtime.coarse = Data(6);
    orgtime.fine   = Data(7);
    rectime.coarse = Data(8);
    rectime.fine   = Data(9);
    xmttime.coarse = Data(10);
    xmttime.fine   = Data(11);
#undef Data

    if (debug) {
        printf("LI=%d  VN=%d  Mode=%d  Stratum=%d  Poll=%d  Precision=%d\n",
               li, vn, mode, stratum, poll, prec);
        printf("Delay=%.1f  Dispersion=%.1f  Refid=%u.%u.%u.%u\n",
               sec2u(delay), sec2u(disp),
               refid >> 24 & 0xff, refid >> 16 & 0xff, refid >> 8 & 0xff, refid & 0xff);
        printf("Reference %u.%.6u\n", reftime.coarse, USEC(reftime.fine));
        printf("(sent)    %u.%.6u\n", ntpc->time_of_send[0], USEC(ntpc->time_of_send[1]));
        printf("Originate %u.%.6u\n", orgtime.coarse, USEC(orgtime.fine));
        printf("Receive   %u.%.6u\n", rectime.coarse, USEC(rectime.fine));
        printf("Transmit  %u.%.6u\n", xmttime.coarse, USEC(xmttime.fine));
        printf("Our recv  %u.%.6u\n", arrival->coarse, USEC(arrival->fine));
    }
    el_time = ntpdiff(&orgtime, arrival); /* elapsed */
    st_time = ntpdiff(&rectime, &xmttime); /* stall */
    skew1 = ntpdiff(&orgtime, &rectime);
    skew2 = ntpdiff(&xmttime, arrival);
    freq = get_current_freq();
    if (debug) {
        printf("Total elapsed: %9.2f\n"
               "Server stall:  %9.2f\n"
               "Slop:          %9.2f\n",
               el_time, st_time, el_time - st_time);
        printf("Skew:          %9.2f\n"
               "Frequency:     %9d\n"
               " day   second     elapsed    stall     skew  dispersion  freq\n",
               (skew1 - skew2) / 2, freq);
    }

    /* error checking, see RFC-4330 section 5 */
#if ENABLE_DEBUG
#define FAIL(x) do { drop_reason=(x); goto fail;} while (0)
#else
#define FAIL(x) goto fail;
#endif
    if (ntpc->cross_check) {
        if (li == 3) {
            FAIL("LI==3");    /* unsynchronized */
        }
        if (vn < 3) {
            FAIL("VN<3");    /* RFC-4330 documents SNTP v4, but we interoperate with NTP v3 */
        }
        if (mode != 4) {
            FAIL("MODE!=3");
        }
        if (orgtime.coarse != ntpc->time_of_send[0] ||
            orgtime.fine   != ntpc->time_of_send[1] ) {
            FAIL("ORG!=sent");
        }
        if (xmttime.coarse == 0 && xmttime.fine == 0) {
            FAIL("XMT==0");
        }
        if (delay > 65536 || delay < -65536) {
            FAIL("abs(DELAY)>65536");
        }
        if (disp  > 65536 || disp  < -65536) {
            FAIL("abs(DISP)>65536");
        }
        if (stratum == 0) {
            FAIL("STRATUM==0");    /* kiss o' death */
        }
#undef FAIL
    }

    /* XXX should I do this if debug flag is set? */
    //if (ntpc->set_clock) { /* you'd better be root, or ntpclient will exit here! */
    set_time(&xmttime);
    //}

    /* Not the ideal order for printing, but we want to be sure
     * to do all the time-sensitive thinking (and time setting)
     * before we start the output, especially fflush() (which
     * could be slow).  Of course, if debug is turned on, speed
     * has gone down the drain anyway. */
    if (ntpc->live) {
        int new_freq;
        new_freq = contemplate_data(arrival->coarse, (skew1 - skew2) / 2,
                                    el_time + sec2u(disp), freq);
        if (!debug && new_freq != freq) {
            set_freq(new_freq);
        }
    }

    if (debug)
        printf("%d %.5d.%.3d  %8.1f %8.1f  %8.1f %8.1f %9d\n",
               arrival->coarse / 86400, arrival->coarse % 86400,
               arrival->fine / 4294967, el_time, st_time,
               (skew1 - skew2) / 2, sec2u(disp), freq);

    fflush(stdout);
    *error = el_time - st_time;

    return 0;
fail:
#if ENABLE_DEBUG
    printf("%d %.5d.%.3d  rejected packet: %s\n",
           arrival->coarse / 86400, arrival->coarse % 86400,
           arrival->fine / 4294967, drop_reason);
#else
    printf("%d %.5d.%.3d  rejected packet\n",
           arrival->coarse / 86400, arrival->coarse % 86400,
           arrival->fine / 4294967);
#endif
    return 1;
}

int check_valid_ipaddr(char *ip, unsigned int *ip_addr)
{
    *ip_addr = inet_addr(ip);

    if (INADDR_NONE == * ip_addr || * ip_addr == 0) {
        return 0;
    }

    return 1;
}

static int stuff_net_addr(struct in_addr *p, char *hostname)
{
    struct addrinfo *answer, hint;
    bzero(&hint, sizeof(hint));
    hint.ai_family = AF_INET;
    hint.ai_socktype = SOCK_STREAM;
    int ret = getaddrinfo(hostname, NULL, &hint, &answer);
    struct sockaddr_in *sockaddr_ipv4;
    if (ret != 0) {
        /* avoid printing: "time-h.netgear.com: Unknown host" */
        /* herror(hostname); */
        return 0;
    }
    if (answer->ai_family != AF_INET) {
        /* IPv4 only, until I get a chance to test IPv6 */
        fprintf(stderr, "oops %d\n", answer->ai_family);
        return 0;
    }
    sockaddr_ipv4 = (struct sockaddr_in *)answer->ai_addr;
    memcpy(p, &(sockaddr_ipv4->sin_addr), 4);
    return 1;
}

static int setup_receive(int usd, unsigned int interface, unsigned short port)
{
    struct sockaddr_in sa_rcvr;
    memset(&sa_rcvr, 0, sizeof sa_rcvr);
    sa_rcvr.sin_family = AF_INET;
    sa_rcvr.sin_addr.s_addr = htonl(interface);
    sa_rcvr.sin_port = htons(port);
    if(bind(usd, (struct sockaddr *) &sa_rcvr, sizeof sa_rcvr) == -1) {
        perror("bind");
        fprintf(stderr, "could not bind to udp port %d\n", port);
        return 0;
    }
    /* listen(usd,3); this isn't TCP; thanks Alexander! */

    /* Make "usd" close on child process when call system(),
     * so that the child process will not inherit the parent resource */
    fcntl(usd, F_SETFD, FD_CLOEXEC);

    return 1;
}

static int setup_transmit(int usd, char *host, unsigned short port, struct ntp_control *ntpc)
{
    struct sockaddr_in sa_dest;
    unsigned int ip_addr = 0;

    memset(&sa_dest, 0, sizeof sa_dest);
    sa_dest.sin_family = AF_INET;

    /*Spec NTP V2 the input field of the "Set your preferred NTP server" option shall support both entering*/
    /*an IP address or DNS address*/
    if(check_valid_ipaddr(host, &ip_addr)) {
        sa_dest.sin_addr.s_addr = inet_addr(host);
    } else {
        if (!stuff_net_addr(&(sa_dest.sin_addr), host)) {
            return 0;
        }
    }
    memcpy(ntpc->serv_addr, &(sa_dest.sin_addr), 4); /* XXX asumes IPv4 */
    sa_dest.sin_port = htons(port);
    if (connect(usd, (struct sockaddr *)&sa_dest, sizeof sa_dest) == -1) {
        perror("connect");
        return 0;
    }

    return 1;
}

void primary_loop(int usd, struct ntp_control *ntpc)
{
    fd_set fds;
    struct sockaddr_in sa_xmit_in;
    int i, pack_len, probes_sent, error;
    socklen_t sa_xmit_len;
    struct timeval to;
    struct ntptime udp_arrival_ntp;
    static u32 incoming_word[325];
    int steady_state = 0;
#define incoming ((char *) incoming_word)
#define sizeof_incoming (sizeof incoming_word)

    if (debug) {
        printf("Listening...\n");
    }

    probes_sent = 0;
    sa_xmit_len = sizeof sa_xmit_in;
    to.tv_sec = 0;
    to.tv_usec = 0;
    for (;;) {
        FD_ZERO(&fds);
        FD_SET(usd, &fds);
        i = select(usd + 1, &fds, NULL, NULL, &to); /* Wait on read or error */
        if ((i != 1) || (!FD_ISSET(usd, &fds))) {
            if (i < 0) {
                if (errno != EINTR) {
                    perror("select");
                }
                continue;
            }
            if ((to.tv_sec == 0) || (to.tv_sec == 5) || (to.tv_sec == DAY_TIME)) {
                if (steady_state != 1 && probes_sent >= ntpc->probe_count &&
                    ntpc->probe_count != 0) {
                    break;
                }

                steady_state = 0;
                send_packet(usd, ntpc->time_of_send);
                ++probes_sent;
                to.tv_sec = 5;
                to.tv_usec = 0;
            }
            continue;
        }
        pack_len = recvfrom(usd, incoming, sizeof_incoming, 0,
                            (struct sockaddr *) &sa_xmit_in, &sa_xmit_len);
        error = ntpc->goodness;
        if (pack_len < 0) {
            perror("recvfrom");
            /* A query receives no successful response, the retry algorithm must
            *  wait that random delay period before initiating the first retry query.
            */
            select(1, NULL, NULL, NULL, &to);
        } else if (pack_len > 0 && (unsigned)pack_len < sizeof_incoming) {
            get_packet_timestamp(usd, &udp_arrival_ntp);
            if (check_source(pack_len, &sa_xmit_in, sa_xmit_len, ntpc) != 0) {
                continue;
            }
            if (rfc1305print(incoming_word, &udp_arrival_ntp, ntpc, &error) != 0) {
                continue;
            }
            steady_state = 1;
            apply_settings();
            /* udp_handle(usd,incoming,pack_len,&sa_xmit,sa_xmit_len); */
        } else {
            printf("Ooops.  pack_len=%d\n", pack_len);
            fflush(stdout);
        }

        if (steady_state == 1) {
            to.tv_sec = DAY_TIME;
            to.tv_usec = 0;
        } else if ((error < ntpc->goodness && ntpc->goodness != 0) ||
                   (probes_sent >= ntpc->probe_count && ntpc->probe_count != 0)) {
            /* best rollover option: specify -g, -s, and -l.
             * simpler rollover option: specify -s and -l, which
             * triggers a magic -c 1 */
            ntpc->set_clock = 0;
            if (!ntpc->live) {
                break;
            }
        }
    }
#undef incoming
#undef sizeof_incoming
}

#if ENABLE_REPLAY
static void do_replay(void)
{
    char line[100];
    int n, day, freq, absolute;
    float sec, el_time, st_time, disp;
    double skew, errorbar;
    int simulated_freq = 0;
    unsigned int last_fake_time = 0;
    double fake_delta_time = 0.0;

    while (fgets(line, sizeof line, stdin)) {
        n = sscanf(line, "%d %f %f %f %lf %f %d",
                   &day, &sec, &el_time, &st_time, &skew, &disp, &freq);
        if (n == 7) {
            fputs(line, stdout);
            absolute = day * 86400 + (int)sec;
            errorbar = el_time + disp;
            if (debug) printf("contemplate %u %.1f %.1f %d\n",
                                  absolute, skew, errorbar, freq);
            if (last_fake_time == 0) {
                simulated_freq = freq;
            }
            fake_delta_time += (absolute - last_fake_time) * ((double)(freq - simulated_freq)) / 65536;
            if (debug) {
                printf("fake %f %d \n", fake_delta_time, simulated_freq);
            }
            skew += fake_delta_time;
            freq = simulated_freq;
            last_fake_time = absolute;
            simulated_freq = contemplate_data(absolute, skew, errorbar, freq);
        } else {
            fprintf(stderr, "Replay input error\n");
            exit(2);
        }
    }
}
#endif

static const char *ntp_pid_file = "/tmp/run/ntpclient.pid";
static void save_ntp_pid(void)
{
    int pid_fd;
    FILE *pid = NULL;

    if (ntp_pid_file != NULL) {
        unlink(ntp_pid_file);
        pid_fd = open(ntp_pid_file, O_CREAT | O_EXCL | O_WRONLY, 0600);
        if (pid_fd == -1) {
            perror("open ntp_pid_file");
            exit(1);
        } else {
            if ((pid = fdopen(pid_fd, "w")) == NULL) {
                perror("open ntp_pid");
                exit(1);
            } else {
                fprintf(pid, "%d\n", (int) getpid());
                fclose(pid);
            }
            close(pid_fd);
        }
    }
}

unsigned int dni_random(void)
{
    unsigned int randnum = 0;
    int fd_rand = open("/dev/urandom", O_RDONLY);

    if (fd_rand < 0) {
        return 0;
    }
    if (read(fd_rand, &randnum, sizeof(randnum)) < 0) {
        close(fd_rand);
        return 0;
    }
    close(fd_rand);
    return randnum;
}

static void usage(char *argv0)
{
    fprintf(stderr,
            "Usage: %s [-c count]"
#if ENABLE_DEBUG
            " [-d]"
#endif
            " [-f frequency] [-g goodness] [-h hostname] [-b sec_host]\n"
            "\t[-m max_interval] [-i min_interval] [-l] [-p port] [-q min_delay]"
#if ENABLE_REPLAY
            " [-r]"
#endif
            " [ -w wan_proto] [ -n interface]\n"
            "\t[-j prefer_server_type] [-k prefer_server_name]"
#if NETGEAR_DAYLIGHT_SAVING_TIME
            " [-e dst_set]"
#endif
#if ENABLE_BOOT_RELAY
            " [-y]"
#endif
            " [-s] [-t] [-z time_zone]\n",
            argv0);
}
/*
 *	@brief  get a random and availdable port
 *		it's for switching port fot NTPv4,because some ISP will
 *		block NTP's src port packet.
 *	@return if success, return a port value,
 *		if failed ,while() in the funciton.
 *	@implementation
 *		1.Create socket. ->
 *		2.Create sockaddr & set addr.sin_port=htons(0). ->
 *		3.Bind. ->
 *			(When addr.sin_port=htons(0), kernel will randomly assign an available port)
 *		4.Getsockname() to get local port number assigned by kernel. ->
 *		5.If port < 1024, go back to step1.
 *		  If port > 1024, return port.
 * */
unsigned short get_random_port(void)
{
    int sockfd = 0;
    struct sockaddr_in addr;
    unsigned short port = 0;
    int addr_len = sizeof(struct sockaddr_in);
    do {
        while((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
            perror("socket for switching port");
            sleep(1);
        }
        addr.sin_family = AF_INET;
        addr.sin_port = htons(port);
        addr.sin_addr.s_addr = htonl(INADDR_ANY);
        while(bind(sockfd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
            perror("bind for switching port");
            sleep(1);
        }
        while(getsockname(sockfd, (struct sockaddr *)&addr, (socklen_t *)&addr_len) != 0) {
            perror("ntp get sock name error");
            sleep(1);
        }
        port = ntohs(addr.sin_port);
        close(sockfd);
    } while(port < 1024);
    return(port);
}

/* write a word/integer value to `file'. */
void writew(char *file, unsigned short value)
{
    FILE *fp;

    fp = fopen(file, "w+");
    if (fp == NULL) {
        return;
    }

    fprintf(fp, "%d", value);
    fclose(fp);
}

int main(int argc, char *argv[])
{
    int usd;  /* socket */
    int c;
    /* These parameters are settable from the command line
       the initializations here provide default behavior */
    unsigned short udp_local_port = 0; /* default of 0 means kernel chooses */
    int initial_freq = 0;           /* initial freq value to use */
    struct ntp_control ntpc;
    int min_interval = 0;
    int max_interval = 0;
    char *hostname = NULL;        /* must be set */
    char *sec_host = "0.0.0.0";
#if ENABLE_THIRD_NTP_SERVER==1
    char *third_host = "0.0.0.0";
#endif
    char *ntps = "0.0.0.0";
    char *prefer_server_type =
        NULL;	/*0 means use default NETGEAR NTP server, 1 means use user's preferred NTP server*/
    char *prefer_server_name = NULL;	/*The name of user's preferred NTP server*/
    char ntp_server[128] = {}; //The length of perferred server have checked with GUI, it is less than 128 bytes.
    unsigned int retry_count = 0;
    unsigned short use_default_server = 1;

    struct timeval to;

    /* ntpclient -h "time-g.netgear.com" -b "time-h.netgear.com" -i 15 -m 60 -p 123 -s */
    // use the default values to work correctly
    ntpc.live = 0;
    ntpc.set_clock = 1;
    ntpc.probe_count = 1;         /* default of 0 means loop forever */
    ntpc.cycle_time = MIN_INTERVAL;        /* seconds */
    ntpc.goodness = 0;
    ntpc.cross_check = 1;

    min_interval = 1;
    max_interval = 60;
    udp_local_port = NTP_PORT;
#if ENABLE_BOOT_RELAY
    FILE *fp = NULL;
    boot_relay = 0;
#endif

    for (;;) {
        c = getopt( argc, argv, "c:" DEBUG_OPTION "j:k:" "b:m:i:w:n:a:z:" BOOT_RELAY_OPTION DST_OPTION
                    "f:g:h:lp:q:" REPLAY_OPTION "st");
        if (c == EOF) {
            break;
        }
        switch (c) {
        case 'c':
            ntpc.probe_count = atoi(optarg);
            break;
#if ENABLE_DEBUG
        case 'd':
            ++debug;
            break;
#endif
        case 'f':
            initial_freq = atoi(optarg);
            if (debug) printf("initial frequency %d\n",
                                  initial_freq);
            set_freq(initial_freq);
            break;
        case 'g':
            ntpc.goodness = atoi(optarg);
            break;
        case 'h':
            hostname = optarg;
            break;
        case 'j':
            prefer_server_type = optarg;
            break;
        case 'k':
            prefer_server_name = optarg;
            break;
        case 'l':
            (ntpc.live)++;
            break;
        case 'p':
            udp_local_port = atoi(optarg);
            break;
        case 'q':
            min_delay = atof(optarg);
            break;
#if ENABLE_REPLAY
        case 'r':
            do_replay();
            exit(0);
            break;
#endif
        case 's':
            ntpc.set_clock++;
            break;

        case 't':
            ntpc.cross_check = 0;
            break;

#if ENABLE_BOOT_RELAY
        case 'y':
            boot_relay = 1;
            break;
#endif

        case 'b':
            sec_host = optarg;
            break;
        case 'm':
            max_interval = atoi(optarg);
            break;
        case 'i':
            min_interval = atoi(optarg);
            break;

        case 'w':
            wan_proto = strdup(optarg);
            break;

        case 'n':
            wan_ifname = strdup(optarg);
            break;

        case 'a':
            lan_ifname = strdup(optarg);
            break;

        case 'z':
            time_zone = strdup(optarg);
            break;

#if NETGEAR_DAYLIGHT_SAVING_TIME
        case 'e':
            dst_flag = atoi(optarg);
            break;
#endif
        default:
            usage(argv[0]);
            exit(1);
        }
    }

#if OPENWRT_UCI
    if(system("[ -d /tmp/cache/ntpclient ] || mkdir -p /tmp/cache/ntpclient")) {};
#else
    if(system("mkdir /tmp/ntp")) {};
#endif

    /* When prefer_server_type is NULL, use default NETGEAR NTP server*/
    if (prefer_server_type == NULL) {
        prefer_server_type = "0";
    }

    /* select ntpserver by default or by manual through NET-CGI */
    if (strcmp(prefer_server_type, "1") != 0) {
#if ENABLE_THIRD_NTP_SERVER==1
        select_ntp_servers(&hostname, &sec_host, &third_host);
#else
        select_ntp_servers(&hostname, &sec_host);
#endif
        use_default_server = 1;
    } else if (prefer_server_name != NULL) {
        hostname = prefer_server_name;
        strcpy(ntp_server, hostname);
        hostname = ntp_server;
        sec_host = hostname;
#if ENABLE_THIRD_NTP_SERVER==1
        third_host = hostname;
#endif
        use_default_server = 0;
    } else {
        printf("Please set your preferred NTP server name behind -k!\n");
        usage(argv[0]);
        exit(1);
    }

    if (hostname == NULL) {
        usage(argv[0]);
        exit(1);
    }
    //printf("Run NTP Client with setting: pri:%s sec:%s\n", hostname ? : "", sec_host ? : "");

    if (strcmp(sec_host, "0.0.0.0") == 0) {
        sec_host = hostname;
    }
#if ENABLE_THIRD_NTP_SERVER==1
    if (strncmp(third_host, "0.0.0.0", 32) == 0) {
        third_host = hostname;
    }
#endif

    if (min_interval > max_interval || min_interval < 0 || max_interval < 0) {
        usage(argv[0]);
        exit(1);
    } else if (max_interval == 0) {
        max_interval = MIN_INTERVAL;
        min_interval = MIN_INTERVAL;
    } else {
        ntpc.cycle_time = min_interval + dni_random() % (max_interval - min_interval + 1);
    }

    if (!strcmp(wan_ifname, lan_ifname)) {
        ap_mode = 1;
    }

    if (debug) {
        printf("Configuration:\n"
               "  -c probe_count       %d\n"
               "  -d (debug)           %d\n"
               "  -f frequency         %d\n"
               "  -g goodness          %d\n"
               "  -h hostname          %s\n"
               "  -b second hostname   %s\n"
#if ENABLE_THIRD_NTP_SERVER==1
               "  -r third hostname    %s\n"
#endif
               "  -i interval(min)     %d\n"
               "  -m interval(max)     %d\n"
               "  -l live				%d\n"
               "  -e dst enable		%d\n"
               "  -z time_zone			%s\n"
               "  -w wan_proto			%s\n"
               "  -n wan interface		%s\n"
               "  -a lan interface		%s\n"
               "  -j prefer_server_type	%s\n"
               "  -k prefer_server_name	%s\n"
               "  -p local_port        %d\n"
               "  -q min_delay        	%.21f\n"
               "  -y boot_relay        %d\n"
               "  -s set_clock         %d\n"
               "  -t cross_check		%d\n",
               ntpc.probe_count, debug, initial_freq, ntpc.goodness, hostname, sec_host,
#if ENABLE_THIRD_NTP_SERVER==1
               third_host,
#endif
               min_interval, max_interval, ntpc.live, dst_flag, time_zone, wan_proto, wan_ifname, lan_ifname,
               prefer_server_type,
               prefer_server_name, udp_local_port, min_delay, boot_relay, ntpc.set_clock, ntpc.cross_check);
    } else {
        if(daemon(1, 1)) {};
    }

#if OPENWRT_UCI
    if(access("/tmp/cache/ntpclient/ntp_updated", F_OK) == 0) {
        unlink("/tmp/cache/ntpclient/ntp_updated");
    }
#else
    if(access("/tmp/ntp_updated", F_OK) == 0) {
        unlink("/tmp/ntp_updated");
    }
#endif

#if ENABLE_BOOT_RELAY
    if (boot_relay == 0) {
        //detect whether the router is in the booting process
        fp = fopen("/tmp/boot_status", "r");
        if (fp != NULL) {
            boot_relay = 1;
            fclose(fp);
            fp = NULL;
        }
    }
#endif

    if (ntpc.set_clock && !ntpc.live && !ntpc.goodness && !ntpc.probe_count) {
        ntpc.probe_count = 1;
    }

    /* respect only applicable MUST of RFC-4330 */
    if (ntpc.probe_count != 1 && ntpc.cycle_time < MIN_INTERVAL) {
        ntpc.cycle_time = MIN_INTERVAL;
    }

    save_ntp_pid();

    while(1) {
        /* if use default server, client's requests should like following
         *  |  NTP Server   time-e.netgear.com  |
         *  |  local port    123                |
         *  |  NTP Server   time-f.netgear.com  |
         *  |  local port    123                |
         *  |  NTP Server   time-e.netgear.com  |
         *  |  local port    random-port        |
         *  |  NTP Server   time-f.netgear.com  |
         *  |  local port    random-port        |
         * * *
         * */
        if(use_default_server) {
#if ENABLE_THIRD_NTP_SERVER==1
            if((retry_count / 3) % 2) {
                while((retry_count % 6) == 3) {
                    udp_local_port = get_random_port();
                    break;
                }
                ntps = (strncmp(ntps, hostname, 32) == 0) ? sec_host : hostname;
                if((retry_count % 6) == 5) {
                    ntps = third_host;
                }
            } else {
                udp_local_port = NTP_PORT;
                ntps = (strncmp(ntps, hostname, 32) == 0) ? sec_host : hostname;
                if((retry_count % 6) == 2) {
                    ntps = third_host;
                }
            }
#else
            if((retry_count / 2) % 2) {
                while((retry_count % 4) == 2) {
                    udp_local_port = get_random_port();
                    break;
                }
                ntps = (strcmp(ntps, hostname) == 0) ? sec_host : hostname;
            } else {
                udp_local_port = NTP_PORT;
                ntps = (strcmp(ntps, hostname) == 0) ? sec_host : hostname;
            }
#endif
            retry_count++;
        }
        /* if use manual server, client's requests should like following
         *  |  NTP Server   manual-server       |
         *  |  local port    123                |
         *  |  NTP Server   manual-server       |
         *  |  local port    random-port        |
         * * *
         * */
        else {
            if(retry_count % 2) {
                udp_local_port = get_random_port();
            } else {
                udp_local_port = NTP_PORT;
            }
            ntps = hostname;
            retry_count++;
        }

        if (debug) {
            printf("Configuration:\n"
                   "  -c probe_count       %d\n"
                   "  -d (debug)           %d\n"
                   "  -f frequency         %d\n"
                   "  -g goodness          %d\n"
                   "  -h hostname          %s\n"
                   "  -b second hostname   %s\n"
                   "  -i interval(min)     %d\n"
                   "  -m interval(max)     %d\n"
                   "  -l live				%d\n"
                   "  -e dst enable		%d\n"
                   "  -z time_zone			%s\n"
                   "  -w wan_proto			%s\n"
                   "  -n wan interface		%s\n"
                   "  -a lan interface		%s\n"
                   "  -j prefer_server_type	%s\n"
                   "  -k prefer_server_name	%s\n"
                   "  -p local_port        %d\n"
                   "  -q min_delay        	%.21f\n"
                   "  -y boot_relay        %d\n"
                   "  -s set_clock         %d\n"
                   "  -t cross_check		%d\n",
                   ntpc.probe_count, debug, initial_freq, ntpc.goodness, hostname, sec_host, min_interval,
                   max_interval, ntpc.live, dst_flag, time_zone, wan_proto, wan_ifname, lan_ifname, prefer_server_type,
                   prefer_server_name, udp_local_port, min_delay, boot_relay, ntpc.set_clock, ntpc.cross_check);
        }

//	printf("Configuration: NTP server : %s Interval : %d Local port :%d\n", ntps, ntpc.cycle_time, udp_local_port);
#if OPENWRT_UCI
        update_ntp_port_status(udp_local_port);
#else
        writew(RA_PORTNUM, udp_local_port);
#endif
        /* Startup sequence */
        if ((usd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
            perror ("socket");
            goto cont;
        }

        if (!wan_conn_up()) {
            //		printf("The WAN connection is NOT up!\n");
#if OPENWRT_UCI
            ntp_syn_status(FAIL_wan_conn_no_up);
#else
            writew(RA_FAILREASON, FAIL_wan_conn_no_up);
#endif
            close(usd);
            goto cont;
        }
#if ENABLE_BOOT_RELAY
        if (boot_relay == 1) {
            ntps = "0.0.0.0";
            close(usd);
            goto cont;
        }
#endif

        if (!setup_receive(usd, INADDR_ANY, udp_local_port)
            || !setup_transmit(usd, ntps, NTP_PORT, &ntpc)) {
            close(usd);
#if ENABLE_NETGEAR_XCLOUD_TIME
            /* Get Cloud time from xagent.
             * if ntp fail ,use ra method to update coarsely.
             * it will just updated once.
             * */
            updateRaCloudTime();
#endif
            to.tv_sec = ntpc.cycle_time;
            to.tv_usec = 0;
            select(1, NULL, NULL, NULL, &to);
            goto loop;
        }

        primary_loop(usd, &ntpc);

#if OPENWRT_UCI
        ntp_syn_status(FAIL_no_server_response);
#else
        writew(RA_FAILREASON, FAIL_no_server_response);
#endif
        /*when program is out of primary loop,the NTP server is fail,so delete the file.*/
#if OPENWRT_UCI
        if(system("rm -f /tmp/cache/ntpclient/ntp_updated")) {};
#else
        if(system("rm -f /tmp/ntp_updated")) {};
#endif

        close(usd);

loop:
#if ENABLE_NETGEAR_XCLOUD_TIME
        /* Get Cloud time from xagent.
         * if ntp fail ,use ra method to update coarsely.
         * it will just updated once.
         * */
        updateRaCloudTime();
#endif
        /* [NETGEAR Spec 12]:Subsequent queries will double the preceding query interval
         * until the interval has exceeded the steady state query interval, at which point
         * and new random interval between 0.00 and 60.00 seconds is selected and the
         * process repeats.
         */
        to.tv_sec = ntpc.cycle_time;
        to.tv_usec = 0;
        select(1, NULL, NULL, NULL, &to);


        if ((ntpc.cycle_time * 2) > DAY_TIME) {
            ntpc.cycle_time = min_interval + dni_random() % (max_interval - min_interval + 1);
        } else {
            ntpc.cycle_time = ntpc.cycle_time * 2;
        }
        continue;

cont:
        /* [NETGEAR Spec 12]: we will wait randomly calculated period of 0 to 20 seconds
         * before issuing the first NTP query upon subsequent power-ons or resets.
         */
#if ENABLE_BOOT_RELAY
        if (boot_relay == 1) {
            boot_relay = 0;
        }
#endif
        to.tv_sec = dni_random() % (NETGEAR_PERIOD + 1);
        to.tv_usec = 0;
        select(1, NULL, NULL, NULL, &to);

    }

    unlink(ntp_pid_file); /*remove the pid file*/
    return 0;
}
