/*
 *  linux/kernel/panic.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

/*
 * This function is used through-out the kernel (including mm and fs)
 * to indicate a major problem.
 */
#include <linux/debug_locks.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/mtd/mtd.h>
#include <linux/version.h>
#include <linux/sched/debug.h>
#include <linux/interrupt.h>
#include <linux/kmsg_dump.h>
#include <linux/kallsyms.h>
#include <linux/notifier.h>
#include <linux/vt_kern.h>
#include <linux/module.h>
#include <linux/random.h>
#include <linux/ftrace.h>
#include <linux/reboot.h>
#include <linux/delay.h>
#include <linux/kexec.h>
#include <linux/sched.h>
#include <linux/sysrq.h>
#include <linux/init.h>
#include <linux/nmi.h>
#include <linux/console.h>
#include <linux/bug.h>
#include <linux/ratelimit.h>
#include <linux/debugfs.h>
#include <asm/sections.h>

#define PANIC_TIMER_STEP 100
#define PANIC_BLINK_SPD 18

int panic_on_oops = CONFIG_PANIC_ON_OOPS_VALUE;
static unsigned long tainted_mask =
	IS_ENABLED(CONFIG_GCC_PLUGIN_RANDSTRUCT) ? (1 << TAINT_RANDSTRUCT) : 0;
static int pause_on_oops;
static int pause_on_oops_flag;
static DEFINE_SPINLOCK(pause_on_oops_lock);
bool crash_kexec_post_notifiers;
int panic_on_warn __read_mostly;

int panic_timeout = CONFIG_PANIC_TIMEOUT;
EXPORT_SYMBOL_GPL(panic_timeout);

ATOMIC_NOTIFIER_HEAD(panic_notifier_list);

EXPORT_SYMBOL(panic_notifier_list);

static long no_blink(int state)
{
	return 0;
}

/* Returns how long it waited in ms */
long (*panic_blink)(int state);
EXPORT_SYMBOL(panic_blink);

/*
 * Stop ourself in panic -- architecture code may override this
 */
void __weak panic_smp_self_stop(void)
{
	while (1)
		cpu_relax();
}

/*
 * Stop ourselves in NMI context if another CPU has already panicked. Arch code
 * may override this to prepare for crash dumping, e.g. save regs info.
 */
void __weak nmi_panic_self_stop(struct pt_regs *regs)
{
	panic_smp_self_stop();
}

/*
 * Stop other CPUs in panic.  Architecture dependent code may override this
 * with more suitable version.  For example, if the architecture supports
 * crash dump, it should save registers of each stopped CPU and disable
 * per-CPU features such as virtualization extensions.
 */
void __weak crash_smp_send_stop(void)
{
	static int cpus_stopped;

	/*
	 * This function can be called twice in panic path, but obviously
	 * we execute this only once.
	 */
	if (cpus_stopped)
		return;

	/*
	 * Note smp_send_stop is the usual smp shutdown function, which
	 * unfortunately means it may not be hardened to work in a panic
	 * situation.
	 */
	smp_send_stop();
	cpus_stopped = 1;
}

atomic_t panic_cpu = ATOMIC_INIT(PANIC_CPU_INVALID);

/*
 * A variant of panic() called from NMI context. We return if we've already
 * panicked on this CPU. If another CPU already panicked, loop in
 * nmi_panic_self_stop() which can provide architecture dependent code such
 * as saving register state for crash dump.
 */
void nmi_panic(struct pt_regs *regs, const char *msg)
{
	int old_cpu, cpu;

	cpu = raw_smp_processor_id();
	old_cpu = atomic_cmpxchg(&panic_cpu, PANIC_CPU_INVALID, cpu);

	if (old_cpu == PANIC_CPU_INVALID)
		panic("%s", msg);
	else if (old_cpu != cpu)
		nmi_panic_self_stop(regs);
}
EXPORT_SYMBOL(nmi_panic);

/*
 * @mtd: MTD device structure
 * @offset: erase start place
*/
int mtd_erase_block(struct mtd_info *mtd,int offset)
{
    struct erase_info erase;
    int sts;

    /* Erase the flash block. */
    memset(&erase, 0x00, sizeof(erase));
    erase.addr = offset;
    erase.len = mtd->erasesize;
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 19, 0)
    erase.mtd = mtd;
#endif

    if( (sts = mtd_block_isbad(mtd, offset)) == 0 )
    {
        sts = mtd_erase(mtd, &erase);

        /* Function local_bh_disable has been called and this
         * is the only operation that should be occurring.
         * Therefore, spin waiting for erase to complete.
         */
        if( 0 == sts )
        {
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 19, 0)
            int i;

            for(i = 0; i < 10000 && erase.state != MTD_ERASE_DONE &&
                erase.state != MTD_ERASE_FAILED; i++ )
            {
                udelay(100);
            }

            if( erase.state != MTD_ERASE_DONE )
            {
                sts = -1;
                printk("nandEraseBlk - Block 0x%8.8x. Error erase block timeout.\n", offset);
            }
#endif
        }
        else
            printk("nandEraseBlk - Block 0x%8.8x. Error erasing block.\n", offset);
    }
    else
        printk("nandEraseBlk - Block 0x%8.8x. Bad block.\n", offset);

    return (sts);
}

EXPORT_SYMBOL(mtd_erase_block);

#define MAGIC_NUM 120
int find_position(struct mtd_info  *mtd,long unsigned int offset,int len)
{
	uint8_t buf[1];
	size_t retlen;
	int ret;
	int count = 0;
	ret = mtd_read(mtd, offset, len, &retlen, buf);
	if((retlen != 1) || ((ret < 0) && (ret != -EUCLEAN))){
		printk("read failure at %lx\n",offset);
		return count;
	}
	count = *buf;
	return count;
}

int log_buf_copy(char *dest, int idx, int size);
#define PANIC_MSG_LEN 12288
#define SKIP_LENGTH 4096
#define START_WRITE 0    
#define BLOCK 131072
#define MTD_DUMPPART  "oopsdump"

void skip_log_level(char msg[PANIC_MSG_LEN])
{
	int i;
	/* skip the log level */
	for(i=0; i<PANIC_MSG_LEN ; ){
		if (msg[i + 0] == '<' && msg[i + 1] >= '0' &&
				msg[i + 1] <= '7' && msg[i + 2] == '>') {
			msg[i] = msg[i+1] = msg[i+2] = ' ';
			i +=3;
			continue;
		}
		else
			i++;
	}

}
/*----------------echo panic to /proc/reboot_reason------------------------------------------*/
int echo_panic_to_reboot_reason(void)
{
	struct mtd_info *mtd = NULL;
	int len = 0;
	int count, retbad;
	long unsigned int offset;
	offset = START_WRITE ;
	char reboot_msg[128];

	mtd = get_mtd_device_nm(MTD_DUMPPART);

    if (IS_ERR(mtd))
		printk("can not get %s\n", MTD_DUMPPART);
	else
	{
		printk("the name of mtd is %s\n", MTD_DUMPPART);
		count = START_WRITE / BLOCK + 1;

		while(1){
			//flag = find_position(mtd,offset,1);
			if(count < 4){
				/*block check : check to avoid to write log into a bad block
				 *if the block's  first two bytes is not 0xff,we regard it as
				 * a bad block
				 */
				if(mtd_block_isbad(mtd, count*BLOCK)){
					printk(KERN_WARNING "mtdoops: bad block at %08lx\n",offset);
					offset += BLOCK;
					count++;
				}
				else
					break;

			} else
				return -1;
		}
		retbad = mtd_erase_block(mtd, offset / BLOCK * BLOCK);
		if( retbad < 0 ){
			printk( KERN_WARNING "erase failed ,please try agin.\n");
			retbad  = mtd_erase_block(mtd, offset / BLOCK * BLOCK);
		}
		else
			printk(KERN_WARNING "have erased successfully11111111.\n");

		strncpy(reboot_msg, "0x2", 4);
		len = 0;

		//write a block full
		if(offset >= ((count + 1)*128*1024)){
			offset = count*128*1024;
			retbad = mtd_erase_block(mtd, offset);
			if( retbad < 0 ){
				printk( KERN_WARNING "erase failed ,please try agin.\n");
				retbad  = mtd_erase_block(mtd, offset);
			}
			else
				printk(KERN_WARNING "have erased successfully2222222.\n");
		}

		mtd_write(mtd, offset, PANIC_MSG_LEN, &len, reboot_msg);
		printk(KERN_WARNING "mtd write successfully!!!!!.\n");
	}
	return 0;
}

static char dni_msg[PANIC_MSG_LEN + 1];
int save_kernel_msg_to_flash(void)
{
	struct mtd_info *mtd = NULL;
	int ret, len = 0, idx = 0;
	int count, retbad;
	long unsigned int offset;
	offset = START_WRITE + BLOCK;

	mtd = get_mtd_device_nm(MTD_DUMPPART);
	echo_panic_to_reboot_reason();

    if (IS_ERR(mtd))
		printk("can not get %s\n", MTD_DUMPPART);
	else
	{
		printk("the name of mtd is %s\n", MTD_DUMPPART);
		count = START_WRITE / BLOCK + 1;

		while(1){
			//flag = find_position(mtd,offset,1);
			if(count < 4){
				/*block check : check to avoid to write log into a bad block
				 *if the block's  first two bytes is not 0xff,we regard it as
				 * a bad block
				 */
				if(mtd_block_isbad(mtd, count*BLOCK)){
					printk(KERN_WARNING "mtdoops: bad block at %08lx\n",offset);
					offset += BLOCK;
					count++;
				}
				else
					break;

			} else 
				return -1;
		}
		while(1)
		{
			if( find_position(mtd,offset,1) == MAGIC_NUM ){
				offset += SKIP_LENGTH;
				printk(KERN_WARNING "this page have been written,please check another.\n ");
			}
			else
				break;
			
		}
			retbad = mtd_erase_block(mtd, offset / BLOCK * BLOCK);
			if( retbad < 0 ){
				printk( KERN_WARNING "erase failed ,please try agin.\n");
				retbad  = mtd_erase_block(mtd, offset / BLOCK * BLOCK);
			}
			else
				printk(KERN_WARNING "have erased successfully11111111.\n");

//		while (1) {
			ret = log_buf_copy(dni_msg, idx, PANIC_MSG_LEN); 
//			printk("log_buf_copy ret %lu.\n" , offset);
//			if (ret <= 0 || offset >= 256*1024*1024)
//				break;
			skip_log_level(dni_msg);
			//personally set a magic number to know if this page has been written
			dni_msg[0] = MAGIC_NUM;
			len = 0;

			//write a block full
			if(offset >= ((count + 1)*128*1024)){
				offset = count*128*1024;
				retbad = mtd_erase_block(mtd, offset);
				if( retbad < 0 ){
					printk( KERN_WARNING "erase failed ,please try agin.\n");
					retbad  = mtd_erase_block(mtd, offset);
				}
				else
					printk(KERN_WARNING "have erased successfully2222222.\n");

			}	

			mtd_write(mtd, offset, PANIC_MSG_LEN, &len, dni_msg);
			offset += PANIC_MSG_LEN;
			idx += ret;
			printk(KERN_WARNING "mtd write successfully!!!!!.\n");
//		}
	}
	return 0;

}
/**
 *	panic - halt the system
 *	@fmt: The text string to print
 *
 *	Display a message, then perform cleanups.
 *
 *	This function never returns.
 */
void panic(const char *fmt, ...)
{
	static char buf[1024];
	va_list args;
	long i, i_next = 0;
	int state = 0;
	int old_cpu, this_cpu;
	bool _crash_kexec_post_notifiers = crash_kexec_post_notifiers;

	/*
	 * Disable local interrupts. This will prevent panic_smp_self_stop
	 * from deadlocking the first cpu that invokes the panic, since
	 * there is nothing to prevent an interrupt handler (that runs
	 * after setting panic_cpu) from invoking panic() again.
	 */
	local_irq_disable();
	preempt_disable_notrace();

	/*
	 * It's possible to come here directly from a panic-assertion and
	 * not have preempt disabled. Some functions called from here want
	 * preempt to be disabled. No point enabling it later though...
	 *
	 * Only one CPU is allowed to execute the panic code from here. For
	 * multiple parallel invocations of panic, all other CPUs either
	 * stop themself or will wait until they are stopped by the 1st CPU
	 * with smp_send_stop().
	 *
	 * `old_cpu == PANIC_CPU_INVALID' means this is the 1st CPU which
	 * comes here, so go ahead.
	 * `old_cpu == this_cpu' means we came from nmi_panic() which sets
	 * panic_cpu to this CPU.  In this case, this is also the 1st CPU.
	 */
	this_cpu = raw_smp_processor_id();
	old_cpu  = atomic_cmpxchg(&panic_cpu, PANIC_CPU_INVALID, this_cpu);

	if (old_cpu != PANIC_CPU_INVALID && old_cpu != this_cpu)
		panic_smp_self_stop();

	console_verbose();
	bust_spinlocks(1);
	va_start(args, fmt);
	vsnprintf(buf, sizeof(buf), fmt, args);
	va_end(args);
	pr_emerg("Kernel panic - not syncing: %s\n", buf);
#ifdef CONFIG_DEBUG_BUGVERBOSE
	/*
	 * Avoid nested stack-dumping if a panic occurs during oops processing
	 */
	if (!test_taint(TAINT_DIE) && oops_in_progress <= 1)
		dump_stack();
#endif

	/*
	 * If we have crashed and we have a crash kernel loaded let it handle
	 * everything else.
	 * If we want to run this after calling panic_notifiers, pass
	 * the "crash_kexec_post_notifiers" option to the kernel.
	 *
	 * Bypass the panic_cpu check and call __crash_kexec directly.
	 */
	if (!_crash_kexec_post_notifiers) {
		printk_safe_flush_on_panic();
		__crash_kexec(NULL);

		/*
		 * Note smp_send_stop is the usual smp shutdown function, which
		 * unfortunately means it may not be hardened to work in a
		 * panic situation.
		 */
		smp_send_stop();
	} else {
		/*
		 * If we want to do crash dump after notifier calls and
		 * kmsg_dump, we will need architecture dependent extra
		 * works in addition to stopping other CPUs.
		 */
		crash_smp_send_stop();
	}

	/*
	 * Run any panic handlers, including those that might need to
	 * add information to the kmsg dump output.
	 */
	atomic_notifier_call_chain(&panic_notifier_list, 0, buf);

	/* Call flush even twice. It tries harder with a single online CPU */
	printk_safe_flush_on_panic();
	kmsg_dump(KMSG_DUMP_PANIC);

	/*
	 * If you doubt kdump always works fine in any situation,
	 * "crash_kexec_post_notifiers" offers you a chance to run
	 * panic_notifiers and dumping kmsg before kdump.
	 * Note: since some panic_notifiers can make crashed kernel
	 * more unstable, it can increase risks of the kdump failure too.
	 *
	 * Bypass the panic_cpu check and call __crash_kexec directly.
	 */
	if (_crash_kexec_post_notifiers)
		__crash_kexec(NULL);

	save_kernel_msg_to_flash();
#ifdef CONFIG_VT
	unblank_screen();
#endif
	console_unblank();

	/*
	 * We may have ended up stopping the CPU holding the lock (in
	 * smp_send_stop()) while still having some valuable data in the console
	 * buffer.  Try to acquire the lock then release it regardless of the
	 * result.  The release will also print the buffers out.  Locks debug
	 * should be disabled to avoid reporting bad unlock balance when
	 * panic() is not being callled from OOPS.
	 */
	debug_locks_off();
	console_flush_on_panic();

	if (!panic_blink)
		panic_blink = no_blink;

	if (panic_timeout > 0) {
		/*
		 * Delay timeout seconds before rebooting the machine.
		 * We can't use the "normal" timers since we just panicked.
		 */
		pr_emerg("Rebooting in %d seconds..\n", panic_timeout);

		for (i = 0; i < panic_timeout * 1000; i += PANIC_TIMER_STEP) {
			touch_nmi_watchdog();
			if (i >= i_next) {
				i += panic_blink(state ^= 1);
				i_next = i + 3600 / PANIC_BLINK_SPD;
			}
			mdelay(PANIC_TIMER_STEP);
		}
	}
	if (panic_timeout != 0) {
		/*
		 * This will not be a clean reboot, with everything
		 * shutting down.  But if there is a chance of
		 * rebooting the system it will be rebooted.
		 */
		emergency_restart();
	}
#ifdef __sparc__
	{
		extern int stop_a_enabled;
		/* Make sure the user can actually press Stop-A (L1-A) */
		stop_a_enabled = 1;
		pr_emerg("Press Stop-A (L1-A) from sun keyboard or send break\n"
			 "twice on console to return to the boot prom\n");
	}
#endif
#if defined(CONFIG_S390)
	{
		unsigned long caller;

		caller = (unsigned long)__builtin_return_address(0);
		disabled_wait(caller);
	}
#endif
	pr_emerg("---[ end Kernel panic - not syncing: %s ]---\n", buf);
	local_irq_enable();
	for (i = 0; ; i += PANIC_TIMER_STEP) {
		touch_softlockup_watchdog();
		if (i >= i_next) {
			i += panic_blink(state ^= 1);
			i_next = i + 3600 / PANIC_BLINK_SPD;
		}
		mdelay(PANIC_TIMER_STEP);
	}
}

EXPORT_SYMBOL(panic);

/*
 * TAINT_FORCED_RMMOD could be a per-module flag but the module
 * is being removed anyway.
 */
const struct taint_flag taint_flags[TAINT_FLAGS_COUNT] = {
	[ TAINT_PROPRIETARY_MODULE ]	= { 'P', 'G', true },
	[ TAINT_FORCED_MODULE ]		= { 'F', ' ', true },
	[ TAINT_CPU_OUT_OF_SPEC ]	= { 'S', ' ', false },
	[ TAINT_FORCED_RMMOD ]		= { 'R', ' ', false },
	[ TAINT_MACHINE_CHECK ]		= { 'M', ' ', false },
	[ TAINT_BAD_PAGE ]		= { 'B', ' ', false },
	[ TAINT_USER ]			= { 'U', ' ', false },
	[ TAINT_DIE ]			= { 'D', ' ', false },
	[ TAINT_OVERRIDDEN_ACPI_TABLE ]	= { 'A', ' ', false },
	[ TAINT_WARN ]			= { 'W', ' ', false },
	[ TAINT_CRAP ]			= { 'C', ' ', true },
	[ TAINT_FIRMWARE_WORKAROUND ]	= { 'I', ' ', false },
	[ TAINT_OOT_MODULE ]		= { 'O', ' ', true },
	[ TAINT_UNSIGNED_MODULE ]	= { 'E', ' ', true },
	[ TAINT_SOFTLOCKUP ]		= { 'L', ' ', false },
	[ TAINT_LIVEPATCH ]		= { 'K', ' ', true },
	[ TAINT_AUX ]			= { 'X', ' ', true },
	[ TAINT_RANDSTRUCT ]		= { 'T', ' ', true },
};

/**
 * print_tainted - return a string to represent the kernel taint state.
 *
 * For individual taint flag meanings, see Documentation/sysctl/kernel.txt
 *
 * The string is overwritten by the next call to print_tainted(),
 * but is always NULL terminated.
 */
const char *print_tainted(void)
{
	static char buf[TAINT_FLAGS_COUNT + sizeof("Tainted: ")];

	BUILD_BUG_ON(ARRAY_SIZE(taint_flags) != TAINT_FLAGS_COUNT);

	if (tainted_mask) {
		char *s;
		int i;

		s = buf + sprintf(buf, "Tainted: ");
		for (i = 0; i < TAINT_FLAGS_COUNT; i++) {
			const struct taint_flag *t = &taint_flags[i];
			*s++ = test_bit(i, &tainted_mask) ?
					t->c_true : t->c_false;
		}
		*s = 0;
	} else
		snprintf(buf, sizeof(buf), "Not tainted");

	return buf;
}

int test_taint(unsigned flag)
{
	return test_bit(flag, &tainted_mask);
}
EXPORT_SYMBOL(test_taint);

unsigned long get_taint(void)
{
	return tainted_mask;
}

/**
 * add_taint: add a taint flag if not already set.
 * @flag: one of the TAINT_* constants.
 * @lockdep_ok: whether lock debugging is still OK.
 *
 * If something bad has gone wrong, you'll want @lockdebug_ok = false, but for
 * some notewortht-but-not-corrupting cases, it can be set to true.
 */
void add_taint(unsigned flag, enum lockdep_ok lockdep_ok)
{
	if (lockdep_ok == LOCKDEP_NOW_UNRELIABLE && __debug_locks_off())
		pr_warn("Disabling lock debugging due to kernel taint\n");

	set_bit(flag, &tainted_mask);
}
EXPORT_SYMBOL(add_taint);

static void spin_msec(int msecs)
{
	int i;

	for (i = 0; i < msecs; i++) {
		touch_nmi_watchdog();
		mdelay(1);
	}
}

/*
 * It just happens that oops_enter() and oops_exit() are identically
 * implemented...
 */
static void do_oops_enter_exit(void)
{
	unsigned long flags;
	static int spin_counter;

	if (!pause_on_oops)
		return;

	spin_lock_irqsave(&pause_on_oops_lock, flags);
	if (pause_on_oops_flag == 0) {
		/* This CPU may now print the oops message */
		pause_on_oops_flag = 1;
	} else {
		/* We need to stall this CPU */
		if (!spin_counter) {
			/* This CPU gets to do the counting */
			spin_counter = pause_on_oops;
			do {
				spin_unlock(&pause_on_oops_lock);
				spin_msec(MSEC_PER_SEC);
				spin_lock(&pause_on_oops_lock);
			} while (--spin_counter);
			pause_on_oops_flag = 0;
		} else {
			/* This CPU waits for a different one */
			while (spin_counter) {
				spin_unlock(&pause_on_oops_lock);
				spin_msec(1);
				spin_lock(&pause_on_oops_lock);
			}
		}
	}
	spin_unlock_irqrestore(&pause_on_oops_lock, flags);
}

/*
 * Return true if the calling CPU is allowed to print oops-related info.
 * This is a bit racy..
 */
int oops_may_print(void)
{
	return pause_on_oops_flag == 0;
}

/*
 * Called when the architecture enters its oops handler, before it prints
 * anything.  If this is the first CPU to oops, and it's oopsing the first
 * time then let it proceed.
 *
 * This is all enabled by the pause_on_oops kernel boot option.  We do all
 * this to ensure that oopses don't scroll off the screen.  It has the
 * side-effect of preventing later-oopsing CPUs from mucking up the display,
 * too.
 *
 * It turns out that the CPU which is allowed to print ends up pausing for
 * the right duration, whereas all the other CPUs pause for twice as long:
 * once in oops_enter(), once in oops_exit().
 */
void oops_enter(void)
{
	tracing_off();
	/* can't trust the integrity of the kernel anymore: */
	debug_locks_off();
	do_oops_enter_exit();
}

/*
 * 64-bit random ID for oopses:
 */
static u64 oops_id;

static int init_oops_id(void)
{
	if (!oops_id)
		get_random_bytes(&oops_id, sizeof(oops_id));
	else
		oops_id++;

	return 0;
}
late_initcall(init_oops_id);

void print_oops_end_marker(void)
{
	init_oops_id();
	pr_warn("---[ end trace %016llx ]---\n", (unsigned long long)oops_id);
}

/*
 * Called when the architecture exits its oops handler, after printing
 * everything.
 */
void oops_exit(void)
{
	do_oops_enter_exit();
	print_oops_end_marker();
	kmsg_dump(KMSG_DUMP_OOPS);
}

struct warn_args {
	const char *fmt;
	va_list args;
};

void __warn(const char *file, int line, void *caller, unsigned taint,
	    struct pt_regs *regs, struct warn_args *args)
{
	disable_trace_on_warning();

	if (args)
		pr_warn(CUT_HERE);

	if (file)
		pr_warn("WARNING: CPU: %d PID: %d at %s:%d %pS\n",
			raw_smp_processor_id(), current->pid, file, line,
			caller);
	else
		pr_warn("WARNING: CPU: %d PID: %d at %pS\n",
			raw_smp_processor_id(), current->pid, caller);

	if (args)
		vprintk(args->fmt, args->args);

	if (panic_on_warn) {
		/*
		 * This thread may hit another WARN() in the panic path.
		 * Resetting this prevents additional WARN() from panicking the
		 * system on this thread.  Other threads are blocked by the
		 * panic_mutex in panic().
		 */
		panic_on_warn = 0;
		panic("panic_on_warn set ...\n");
	}

	print_modules();

	if (regs)
		show_regs(regs);
	else
		dump_stack();

	print_irqtrace_events(current);

	print_oops_end_marker();

	/* Just a warning, don't kill lockdep. */
	add_taint(taint, LOCKDEP_STILL_OK);
}

#ifdef WANT_WARN_ON_SLOWPATH
void warn_slowpath_fmt(const char *file, int line, const char *fmt, ...)
{
	struct warn_args args;

	args.fmt = fmt;
	va_start(args.args, fmt);
	__warn(file, line, __builtin_return_address(0), TAINT_WARN, NULL,
	       &args);
	va_end(args.args);
}
EXPORT_SYMBOL(warn_slowpath_fmt);

void warn_slowpath_fmt_taint(const char *file, int line,
			     unsigned taint, const char *fmt, ...)
{
	struct warn_args args;

	args.fmt = fmt;
	va_start(args.args, fmt);
	__warn(file, line, __builtin_return_address(0), taint, NULL, &args);
	va_end(args.args);
}
EXPORT_SYMBOL(warn_slowpath_fmt_taint);

void warn_slowpath_null(const char *file, int line)
{
	pr_warn(CUT_HERE);
	__warn(file, line, __builtin_return_address(0), TAINT_WARN, NULL, NULL);
}
EXPORT_SYMBOL(warn_slowpath_null);
#else
void __warn_printk(const char *fmt, ...)
{
	va_list args;

	pr_warn(CUT_HERE);

	va_start(args, fmt);
	vprintk(fmt, args);
	va_end(args);
}
EXPORT_SYMBOL(__warn_printk);
#endif

#ifdef CONFIG_BUG

/* Support resetting WARN*_ONCE state */

static int clear_warn_once_set(void *data, u64 val)
{
	generic_bug_clear_once();
	memset(__start_once, 0, __end_once - __start_once);
	return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(clear_warn_once_fops,
			NULL,
			clear_warn_once_set,
			"%lld\n");

static __init int register_warn_debugfs(void)
{
	/* Don't care about failure */
	debugfs_create_file("clear_warn_once", 0200, NULL,
			    NULL, &clear_warn_once_fops);
	return 0;
}

device_initcall(register_warn_debugfs);
#endif

#ifdef CONFIG_STACKPROTECTOR

/*
 * Called when gcc's -fstack-protector feature is used, and
 * gcc detects corruption of the on-stack canary value
 */
__visible void __stack_chk_fail(void)
{
	panic("stack-protector: Kernel stack is corrupted in: %pB",
		__builtin_return_address(0));
}
EXPORT_SYMBOL(__stack_chk_fail);

#endif

#ifdef CONFIG_ARCH_HAS_REFCOUNT
void refcount_error_report(struct pt_regs *regs, const char *err)
{
	WARN_RATELIMIT(1, "refcount_t %s at %pB in %s[%d], uid/euid: %u/%u\n",
		err, (void *)instruction_pointer(regs),
		current->comm, task_pid_nr(current),
		from_kuid_munged(&init_user_ns, current_uid()),
		from_kuid_munged(&init_user_ns, current_euid()));
}
#endif

core_param(panic, panic_timeout, int, 0644);
core_param(pause_on_oops, pause_on_oops, int, 0644);
core_param(panic_on_warn, panic_on_warn, int, 0644);
core_param(crash_kexec_post_notifiers, crash_kexec_post_notifiers, bool, 0644);

static int __init oops_setup(char *s)
{
	if (!s)
		return -EINVAL;
	if (!strcmp(s, "panic"))
		panic_on_oops = 1;
	return 0;
}
early_param("oops", oops_setup);

/*----------------creat /proc/reboot_reason, save reboot reason------------------------------------------*/
static unsigned long reboot_reason_flags = 0;
static char dni_panic_flag[128];
static char dni_panic_buf[128];

int dni_write_for_proc(void)
{
	struct mtd_info *mtd = NULL;
	int len = 0;
	int count, retbad;
	long unsigned int offset;
	offset = START_WRITE;

	mtd = get_mtd_device_nm(MTD_DUMPPART);

    if (IS_ERR(mtd))
		printk("can not get %s\n", MTD_DUMPPART);
	else
	{
		printk("the name of mtd is %s\n", MTD_DUMPPART);
		count = START_WRITE / BLOCK + 1;

		while(1){
			if(count < 4){
				/*block check : check to avoid to write log into a bad block
				 *if the block's  first two bytes is not 0xff,we regard it as
				 * a bad block
				 */
				if(mtd_block_isbad(mtd, count*BLOCK)){
					printk(KERN_WARNING "mtdoops: bad block at %08lx\n",offset);
					offset += BLOCK;
					count++;
				}
				else
					break;
			} else 
				return -1;
		}

		retbad = mtd_erase_block(mtd, offset / BLOCK * BLOCK);
		len = 0;

		if( retbad < 0 ){
			printk( KERN_WARNING "erase failed ,please try agin.\n");
			retbad  = mtd_erase_block(mtd, offset / BLOCK * BLOCK);
		}

		if(offset >= ((count + 1)*128*1024)){
			offset = count*128*1024;
			retbad = mtd_erase_block(mtd, offset);
			if( retbad < 0 ){
				printk( KERN_WARNING "erase failed ,please try agin.\n");
				retbad  = mtd_erase_block(mtd, offset);
			}
		}

		mtd_write(mtd, offset, PANIC_MSG_LEN, &len, dni_panic_buf);
		printk(KERN_WARNING "reboot_reason:mtd write successfully!!!!!.\n");
	}
	return 0;
}

int dni_read_for_proc(void)
{
	struct mtd_info *mtd = NULL;
	long unsigned int offset;
	int ret;
	offset = START_WRITE;
	char scratch[128];
	size_t sz;

	mtd = get_mtd_device_nm(MTD_DUMPPART);

     if (IS_ERR(mtd)){
		printk("can not get %s\n", MTD_DUMPPART);
		return -1;
	}

	ret = mtd_read(mtd, offset, sizeof(scratch), &sz, scratch);

	if (ret >= 0 && sz != sizeof(scratch)){
		printk(KERN_ERR "reboot_reason: mtd read valid argument at 0x%lx: %s\n",
			offset, scratch);
		return -EINVAL;
	}

	if (ret < 0) {
		printk(KERN_ERR "reboot_reason: mtd read failed at 0x%lx: %d\n",
			offset, ret);
		return ret;
	}

	if(strlen(scratch)>2)
	{
		strncpy(dni_panic_flag,scratch,4);
	}
	else
	{
		strncpy(dni_panic_flag,"0x00",4);
	}
	return 0;
}

static int reboot_reason_read(struct file *file, char __user *buf, size_t count, loff_t *ppos)
{
	char buffer[8];
	int len = 4;

	dni_read_for_proc();
	reboot_reason_flags = simple_strtoul(dni_panic_flag, NULL, 16);
	len = snprintf(buffer, sizeof(buffer), "0x%lx", reboot_reason_flags);

	reboot_reason_flags = 0x00;
	sprintf(dni_panic_buf, "0x%lx", reboot_reason_flags);
	dni_write_for_proc();

	return simple_read_from_buffer(buf, count, ppos, buffer, len);
}

static int reboot_reason_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	unsigned long tmp_bits = 0;
	char c[5]={0};

	if (copy_from_user(c, buf, sizeof(c)))
		return -EFAULT;
	tmp_bits = simple_strtoul(c, NULL, 16);
	reboot_reason_flags = reboot_reason_flags | (0x1 << tmp_bits);
	sprintf(dni_panic_buf, "0x%lx\n", reboot_reason_flags);
	dni_write_for_proc();

	return count;
}

static const struct file_operations proc_reboot_reason_operations = {
	.read       = reboot_reason_read,
	.write      = reboot_reason_write,
};

static int __init reboot_reason_proc_init(void)
{
	struct proc_dir_entry *reboot_reason;

	reboot_reason = proc_create("reboot_reason", 0666, NULL, &proc_reboot_reason_operations);

	if (!reboot_reason)
		return -ENOMEM;

//	proc_set_size(reboot_reason, reboot_reason_proc_data_size);

	return 0;
}

static void __exit reboot_reason_proc_cleanup(void)
{
	remove_proc_entry("config.gz", NULL);
}

module_init(reboot_reason_proc_init);
module_exit(reboot_reason_proc_cleanup);


MODULE_LICENSE("DNI");
MODULE_AUTHOR("xinping wan");
MODULE_DESCRIPTION("Save reboot reason for aws-iot");
