/*
 *   Copyright 2005 The Apache Software Foundation
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 */
package org.apache.felix.tool.mangen;

import java.io.InputStream;
import java.io.IOException;

/**
 * Wrapper interface to class bytecode scanning tools. This interface exposes
 * just the methods required by mangen for class scanning. Specific implementations
 * should be created for different scanners e.g. ASM, BCEL etc.
 * <p>
 * At present the interface has been deliberately kept small and simple. It would be
 * possible to use a {@link java.util.Collection#toArray()} or {@link java.util.Iterator} style models
 * for retrieving class information but in both cases these would be likely to require
 * additional object creation with possibly detrimental performance impacts.
 * <p>
 * The ordering of return values for each of the methods taking an <code>index</code> is not
 * important. All that matters is that the method can be called with an index of <code>0</code> 
 * up to <code>getXXCount()</code> times to retrieve the complete set of values.
 * 
 * @version $Revision: 14 $
 * @author <A HREF="mailto:robw@ascert.com">Rob Walker</A> 
 * @see <a href="http://java.sun.com/docs/books/vmspec/2nd-edition/html/VMSpecTOC.doc.html">The JavaTM Virtual Machine Specification, Second Edition</a> 
 */
public interface ClassScanner
{
    //////////////////////////////////////////////////
    // STATIC VARIABLES
    //////////////////////////////////////////////////

    //////////////////////////////////////////////////
    // SIGNATURES
    //////////////////////////////////////////////////

    /**
    * Scan the class bytes by reading the supplied {@link InputStream} The name field
     * specifies the name of the class to be scanned, and may not be needed 
     * by all scanners.
     */
    public void scan(InputStream is, String name)
            throws IOException;

    /**
     * Return the number of methods for the scanned class
     */
    public int getMethodCount();
    
    /**
     * Return the name of the specific method index.
     */
    public String getMethodName(int index);

    /**
     * Return the signature of the specific method index. 
     * <p>
     * Method signatures should be returned in the form as specified in 
     * <a href="http://java.sun.com/docs/books/vmspec/2nd-edition/html/ClassFile.doc.html#7035">section 4.3.3</a>
     * of the VM spec.
     */
    public String getMethodSignature(int index);

    /**
     * Determines whether the specified method has a Synthetic attribute present.
     * <p>
     * Section <a href="http://java.sun.com/docs/books/vmspec/2nd-edition/html/ClassFile.doc.html#80128">section 4.7.6</a>
     * describes the <code>Synthetic</code> attribute. A typical use of this is to mark
     * methods and fields generated by the Java compiler which weren't actually present in
     * the source code. There are a few cases where <code>mangen</code> uses these to
     * look for <i>generated name</i> patterns to detect class references that would
     * otherwise be missed.
     */
    public boolean isSyntheticMethod(int index);
    
    /**
     * Return the number of fields for the scanned class
     */
    public int getFieldCount();

    /**
     * Return the name of the specific field index.
     */
    public String getFieldName(int index);

    /**
     * Return the signature of the specific method index.
     * <p>
     * Field signatures should be returned in the form as specified in 
     * <a href="http://java.sun.com/docs/books/vmspec/2nd-edition/html/ClassFile.doc.html#14152">section 4.3.2</a>
     * of the VM spec.
     */
    public String getFieldSignature(int index);

    /**
     * Determines whether the specified field has a Synthetic attribute present.
     * {@see #getMethodSignature(int)}
     */
    public boolean isSyntheticField(int index);

    /**
     * Return the number of constant classes for the scanned class. Depending on the
     * approach taking by specific scanner implementations it may be possible to detect all
     * required class references without accessing the constant classes. If so, 
     * the implementation of this method should return 0.
     */
    public int getConstantClassCount();
    
    /**
     * Return the signature of the specific constant class.
     * <p>
     * ConstantClass signatures should be returned in the form as specified in 
     * <a href="http://java.sun.com/docs/books/vmspec/2nd-edition/html/ClassFile.doc.html#1221">section 4.4.1</a>
     * of the VM spec.
     * <p>
     * <i>Note:</i> in general, a simple class name should be returned as a name string rather
     * than an <code>L&lt;classname&gt;;</code> signature string. However, if an implementation
     * has a signature format and wishes to avoid parsing it then simply pre-pending a <code>[</code> will
     * force the {@link BundleJar} calling method to treat it as a signature and perform the
     * parsing.
     */
    public String getConstantClassSignature(int index);
    
}
