#!/bin/sh

set -x

CRASH_DIR="$(ls -l /opt/bitdefender/var/crash | awk '{print $NF}')"
ARKS_DIR="$(ls -l /opt/bitdefender/var/arks | awk '{print $NF}')"

CHARON_BIN="/opt/bitdefender/bin/charon"
KARMA_BIN="/opt/bitdefender/bin/karma"
STACK_HASH_BIN="/opt/bitdefender/bin/bdstack_hash"
DUMP_CONFIG_BIN="/opt/bitdefender/bin/dump_config"
KATASTIF_SERVICE="${1}"
[ -z "${KATASTIF_SERVICE}" ] && KATASTIF_SERVICE="chs-minidump"
IS_KARMA_DASHBOARD="${2}"
[ -z "${IS_KARMA_DASHBOARD}" ] && IS_KARMA_DASHBOARD="no"

run() {
    if [ -x "$1" ]; then
        LD_LIBRARY_PATH="/opt/bitdefender/lib:${LD_LIBRARY_PATH}" "$@"
    fi
}

get_module_version() {
    local version_file="/opt/bitdefender/bitdefender-release"
    if [ -f "/etc/bitdefender-release" ]; then
        version_file="/etc/bitdefender-release"
    fi
    grep -E '^VERSION=' "${version_file}" 2>/dev/null | awk -F'[=~'] '{print $2}'
}


try_upload(){
    local dump="${1}"
    local app_id="$2"
    local hash="$(run ${STACK_HASH_BIN} "${CRASH_DIR}/${dump}")"

    # If not gzip'ed, gzip it (breakpad dumps).
    # Unfortunately, 'gzip -t' is not available on all routers.
    echo "${CRASH_DIR}/${dump}" | grep -E "\.gz$" || {
        gzip -c -f "${CRASH_DIR}/${dump}" > "${CRASH_DIR}/${dump}.gz"
        rm -f "${CRASH_DIR}/${dump}"
        dump="${dump}.gz"
    }

    # Create the config.cfg file
    run "${DUMP_CONFIG_BIN}" -target_file="${CRASH_DIR}/config.cfg"

    # Core dumps format: $EXE.$PID.$TIMESTAMP.core.gz (BOXv2)
    # Breakpad dumps format: $EXE.$INDEX.$TIMESTAMP.dmp.gz (Netgear & PoCs)
    local exe_name="$(basename ${dump} | cut -d '.' -f 1)"
    local patch_ver="$(basename ${dump} | cut -d '.' -f 1-3).patch.ver"
    local timestamp="$(basename ${dump} | cut -d '.' -f 3)"

    local archivename="${hash}"
    [ -z "${archivename}" ] && {
        # Available from kernel 2.3.16.
        local random="$(cat /proc/sys/kernel/random/uuid | cut -c1-6)"
        archivename="d_${random}_${timestamp}"
    }

    tar -cf "${ARKS_DIR}/${archivename}" -C "${CRASH_DIR}" "${dump}" "${patch_ver}" "config.cfg" 2> /dev/null
    rm -f "${CRASH_DIR}/${dump}"
    rm -f "${CRASH_DIR}/${patch_ver}"

    local uuid="$(run ${CHARON_BIN} -submit -s=${KATASTIF_SERVICE} -i=${ARKS_DIR}/${archivename} -sig=${archivename})"

    local module_version="$(get_module_version)"
    local extra_json=""
    if [ $IS_KARMA_DASHBOARD = "no" ]; then
        extra_json="{ \"crash_info\" :                            \
                      {                                           \
                        \"module_name\":\"${exe_name}\",          \
                        \"module_version\":\"${module_version}\", \
                        \"katastif\":\"${uuid}\",                 \
                        \"stack_hash\":\"${hash}\"                \
                      }                                           \
                    }"
    else
        extra_json="{ \"crash_info\" :                            \
                      {                                           \
                        \"module_name\":\"${exe_name}\",          \
                        \"module_version\":\"${module_version}\"  \
                      },                                          \
                      \"event_version\": 1                        \
                    }"
    fi

    run "${KARMA_BIN}" -submit -event_type=bd_module_crashed -extra_json="${extra_json}"
}

check_dumps() {
    local files="$(ls ${CRASH_DIR} | grep -vE '.cfg|.patch.ver')"

    for i in $files; do
        try_upload "${i}"
    done
}

check_pending_dumps() {
    local files="$(ls ${ARKS_DIR} 2> /dev/null)"

    for i in ${files}; do
        run "${CHARON_BIN}" -submit -s="${KATASTIF_SERVICE}" -i="${ARKS_DIR}/${i}" -sig="${i}"
    done
}

check_dumps

check_pending_dumps
