#!/bin/sh

exclude_credentials() {
    local FILE="$1"
    local PATTERN="$2"
    local DELIM="$3"

    local TMP=$(cat /dev/urandom | tr -cd 'a-f0-9' | head -c 12)
    grep -i "${PATTERN}" "${FILE}" | while read -r LINE; do
        local TOKEN=$(echo "${LINE}" | cut -d "${DELIM}" -f 2)
        grep -i -v "${TOKEN}" "${FILE}" > "/tmp/${TMP}"
        mv "/tmp/${TMP}" "${FILE}"
    done
}

LOGS_TYPE="BDAGENT"
LOGS_ENCRYPT="true"

# BD Agent is installed two directories below this script's location
THIS_SCRIPT_DIR="$(cd "$(dirname "$0")"; pwd -P)"
BDROOT="$(dirname "$(dirname "$THIS_SCRIPT_DIR")")"

# On BOXv2, ${BDROOT}/share is a symlink to ${BDROOT}/patches/base/share/
# Two directories below script location would be ${BDROOT}/patches/base
if [ "${LOGS_TYPE}" = "BOXV2" ]; then
    BDROOT="/opt/bitdefender"
fi

# BD_LOG_ARCHIVE_INTERNAL_PATH shall be set via Docker's env.list file.
ARCHIVE_DIR="/tmp"
ARCHIVE_NAME="bitdefender_logs.tar.gz"
[ -w "${BD_LOG_ARCHIVE_INTERNAL_PATH}" ] && ARCHIVE_DIR="${BD_LOG_ARCHIVE_INTERNAL_PATH}"

ROOTNAME="bitdefender_logs"
ROOTDIR="/tmp/${ROOTNAME}"
mkdir -p "${ROOTDIR}"

STORAGEDATA="${BDROOT}/etc/storage.data"
[ -e "${BDROOT}/etc/updater.conf" ] && UPDATERCONF="${BDROOT}/etc/updater.conf"
ETC_LOGGING="${BDROOT}/etc/logging"
[ -e "${BDROOT}/etc/jsonrules" ] && ETC_JSONRULES="${BDROOT}/etc/jsonrules"
[ -e "${BDROOT}/var/log" ] && DAEMONSLOGS_PERSISTENT="${BDROOT}/var/log"
BDTMP="${BDROOT}/var/tmp"
# 08.07.2021 - dneatu: Please keep both temporary log locations
# (e.g. backward compatibility with old bootstrap images)
[ -e "/tmp/opt/bitdefender/var/log" ] && DAEMONSLOGS_TMP_OLD="/tmp/opt/bitdefender/var/log"

# Remove old logs archive if exists
if [ ! -L "${ARCHIVE_DIR:?}/${ARCHIVE_NAME}" ]; then
    rm -f "${ARCHIVE_DIR:?}/${ARCHIVE_NAME}"
fi

if [ "${LOGS_TYPE}" = "BOXV2" ]; then
    PATCH_INFO="${ROOTDIR}/PATCH_INFO"
    mkdir -p "${PATCH_INFO}"

    WSDIR="${ROOTDIR}/ws" && mkdir -p "${WSDIR}"
    cp /tmp/*.json  "${WSDIR}"
    cp /tmp/*.out   "${WSDIR}"
    cp /tmp/*.err   "${WSDIR}"

    # Daemons
    cp "${DAEMONSLOGS_PERSISTENT}/init_config/03-user-network-properties.log" "${BDTMP}"
    exclude_credentials "${BDTMP}/03-user-network-properties.log" "pass=" "="
    exclude_credentials "${BDTMP}/03-user-network-properties.log" "user=" "="

    # Config
    CONFIGDIR="${ROOTDIR}/config" && mkdir -p "${CONFIGDIR}"
    cp /etc/config/dhcp                                                 "${CONFIGDIR}"
    grep -v -i -E "option (user|pass)" /etc/config/network            > "${CONFIGDIR}/network"
    cp /etc/config/firewall                                             "${CONFIGDIR}"
    cp /etc/config/wireless                                             "${CONFIGDIR}"
    cp /etc/config/upnpd                                                "${CONFIGDIR}"
    grep -v -e "WPAPSK[0-9][0-9]*=" /etc/Wireless/iNIC/iNIC_ap.dat    > "${CONFIGDIR}/iNIC_ap.dat"
    grep -v -e "WPAPSK[0-9][0-9]*=" /etc/Wireless/RT2860AP/RT2860.dat > "${CONFIGDIR}/RT2860.dat"

    MISCDIR="${ROOTDIR}/misc" && mkdir -p "${MISCDIR}"
    uci export | grep -v -i -E "option (user|pass)" > "${MISCDIR}/uci"
    logread                                         > "${MISCDIR}/logread"
    "${BDROOT}/bin/iptables"  -nvL -t nat           > "${MISCDIR}/iptables-nat"
    "${BDROOT}/bin/ip6tables"  -nvL -t nat          > "${MISCDIR}/ip6tables-nat"
    ethtool eth0                                    > "${MISCDIR}/ethtool-eth0"
    ethtool eth2                                    > "${MISCDIR}/ethtool-eth2"
    conntrack -L                                    > "${MISCDIR}/conntrack"
    ip neigh show                                   > "${MISCDIR}/neigh"
    cp /etc/crontabs/root                             "${MISCDIR}/crontabs"
    cp /tmp/dhcp.leases                               "${MISCDIR}"
    cp /tmp/resolv.conf                               "${MISCDIR}"
    cp /tmp/resolv.conf.auto                          "${MISCDIR}"
    cp /proc/net/arp                                  "${MISCDIR}/arp"
    cp /var/log/messages                              "${MISCDIR}"

    # Patches info
    cp "${BDROOT}/patches/latest/patch.info" "${PATCH_INFO}"
    cp "${BDROOT}/etc/patch.server" "${PATCH_INFO}"
    cp "${BDROOT}/var/update/patch.csum" "${PATCH_INFO}"

    cp /etc/bitdefender-release                                         "${ROOTDIR}"
    cp /etc/bitdefender-release-bootstrap                               "${ROOTDIR}"
    grep -v -E "/network/.*/(user|pass)" "${BDROOT}/etc/storage.data" > "${ROOTDIR}/storage.data"
    find "${ROOTDIR}" -name 'install_full_ws.*' -exec rm {} \;
fi

# Other miscellaneous info.
[ -e "${BDROOT}/bitdefender-release" ] && cp "${BDROOT}/bitdefender-release" "${ROOTDIR}"
ps                                                    > "${ROOTDIR}/ps"
dmesg                                                 > "${ROOTDIR}/dmesg"
top -b -n 1                                           > "${ROOTDIR}/top"
uptime                                                > "${ROOTDIR}/uptime"
date -R                                               > "${ROOTDIR}/date"
df -h                                                 > "${ROOTDIR}/df"

"${BDROOT}/bin/iptables" -nvL -t filter                                             > "${ROOTDIR}/iptables-filter"
"${BDROOT}/bin/ip6tables" -nvL -t filter                                            > "${ROOTDIR}/ip6tables-filter"
[ -e "${BDROOT}/bin/ebtables" ] && "${BDROOT}/bin/ebtables" -L                      > "${ROOTDIR}/ebtables"

LD_LIBRARY_PATH="/opt/bitdefender/lib" "${BDROOT}/bin/gtables" -l -t detection           > "${ROOTDIR}/gtables.detection"
LD_LIBRARY_PATH="/opt/bitdefender/lib" "${BDROOT}/bin/gtables" -l -t traffic             > "${ROOTDIR}/gtables.traffic"
cp "${STORAGEDATA}" "${ROOTDIR}/storage.data"
LD_LIBRARY_PATH="/opt/bitdefender/lib" "${BDROOT}/bin/bdsett" -get-key -recursive=1 /tmp >> "${ROOTDIR}/storage.data"

if [ -e "${BDROOT}/etc/bdupd.server" ]; then
    BDUPD_INFO="${ROOTDIR}/BDUPD_INFO"
    mkdir -p "${BDUPD_INFO}"
    cp "${BDROOT}/etc/bdupd.server" "${BDUPD_INFO}"
    cp "${BDROOT}"/var/upd/* "${BDUPD_INFO}"
fi

TAR_FLAGS="-czhf"

ARCHIVE_COMPONENTS="${ETC_LOGGING} ${ETC_JSONRULES} ${UPDATERCONF} ${DAEMONSLOGS_PERSISTENT} ${BDTMP} ${DAEMONSLOGS_TMP_OLD}"
if [ "${BDAGENT_DOCKER}" = "true" ]; then
    ARCHIVE_COMPONENTS="/persistent-data/bd_init_sync.sh.log.* ${ARCHIVE_COMPONENTS}"
fi

# Create the archive.
tar ${TAR_FLAGS} "${ARCHIVE_DIR}/${ARCHIVE_NAME}" -C "${ROOTDIR}" . ${ARCHIVE_COMPONENTS}

rm -rf "${ROOTDIR}"

if [ "${LOGS_ENCRYPT}" = "true" ]; then
    # Encrypt the archive.
    if [ "${LOGS_TYPE}" = "BOXV2" ]; then
        source "${BDROOT}/share/scripts/lib/key.sh" || exit 1
        get_platform_key || exit 1
        PUBKEY="${PLATFORM_KEY}"
    else
        PUBKEY="${BDROOT}/etc/logs-pub.pem"
    fi
    ENCSYMKEY=bitdefender_key.enc
    TEMPKEY=/tmp/bitdefender-sym-key-temp

    LD_LIBRARY_PATH="${BDROOT}/lib" "${BDROOT}/bin/bdcrypto" gen-sym-key -k "${TEMPKEY}"
    LD_LIBRARY_PATH="${BDROOT}/lib" "${BDROOT}/bin/bdcrypto" aes-encrypt -k "${TEMPKEY}" -i "/tmp/${ARCHIVE_NAME}" -o "/tmp/${ARCHIVE_NAME}.enc"
    LD_LIBRARY_PATH="${BDROOT}/lib" "${BDROOT}/bin/bdcrypto" rsa-encrypt -k "${PUBKEY}" -i "${TEMPKEY}" -o "/tmp/${ENCSYMKEY}"

    tar -czf "/${ARCHIVE_DIR}/${ARCHIVE_NAME}" -C /tmp "${ENCSYMKEY}" "${ARCHIVE_NAME}.enc"
    rm -f "/tmp/${ENCSYMKEY}" "/${ARCHIVE_DIR}/${ARCHIVE_NAME}.enc" "${TEMPKEY}" 
fi
