/*
   Copyright (c) 2014 Broadcom
   All Rights Reserved

    <:label-BRCM:2014:DUAL/GPL:standard
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2, as published by
    the Free Software Foundation (the "GPL").
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    
    A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
    writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
    
:>
*/

/******************************************************************************/
/*                                                                            */
/* File Description:                                                          */
/*                                                                            */
/* This file contains the implementation of misc shell commands for Lilac     */
/*                                                                            */
/******************************************************************************/

/*****************************************************************************/
/*                                                                           */
/* Include files                                                             */
/*                                                                           */
/*****************************************************************************/

#define BDMF_SESSION_DISABLE_FORMAT_CHECK

#include "bdmf_shell.h"
#include "rdpa_port_int.h"
#include "access_macros.h"
#if (defined(BCM_DSL_XRDP) || defined(CONFIG_BCM96858) || \
    defined(CONFIG_BCM96856) || defined(CONFIG_BCM96846) || defined(CONFIG_BCM96878)) && !defined(BDMF_SYSTEM_SIM)
#include "bcm_OS_Deps.h" /* We need the following 3 #includes for PSP_BUFLEN_16 / RDPA_WAN_TYPE_PSP_KEY definitions */
#include "board.h"
#include "wan_drv.h"
#endif
#if defined(XRDP) && !defined(BDMF_SYSTEM_SIM)
#include "rdp_drv_rnr.h"
#endif
#include <rdpa_api.h>
#include "bdmf_dev.h"
#if defined(XRDP) && defined(BCM_PON_XRDP)
#include "rdpa_vlan_ex.h"
#include "rdd_ag_ds_tm.h"
#include "rdd_ag_us_tm.h"
#endif

#ifdef USE_BDMF_SHELL

#if defined(BDMF_SYSTEM_SIM)
#ifndef XRDP
#include "rdd.h"
#endif
#ifndef LEGACY_RDP
#ifdef XRDP
#include "XRDP_AG.h"
uint8_t *soc_base_address;
#include "rdp_drv_tcam.h"
#ifdef RDP_SIM
#include "rdp_cpu_sim.h"
#endif
#endif
#else
#include "rdd_common.h"
#include "rdd_legacy_conv.h"
#endif
#else
#ifdef XRDP
#include "rdd_runner_proj_defs.h"
#endif
#endif

typedef union
{
    uint32_t word;
    uint8_t word_byte[4];
    uint16_t word_half[2];
} word_t;

#ifdef XRDP
#define TEXT_OFFSET 48
#define LINE_LENGTH 66
#else
#define TEXT_OFFSET 32
#define LINE_LENGTH 50
#endif
#define TEXT_END 17

enum
{
    SRC_DFP,
    SRC_SHELL
};

static void p_chr(char *line, uint32_t index, char cc)
{
    if ((isgraph((int)cc) != 0) || (cc == ' '))
        line[index] = cc;
    else
        line[index] = '.';
}

static void prn(uint32_t source, bdmf_session_handle session, char *line)
{
#if defined(DFP)
    switch (source)
    {
    case SRC_DFP:
        /*pi_dfp_shell_printf(line);*/
        break;
    case SRC_SHELL:
        bdmf_session_print(session, "%s", line);
        break;
    }
#else
    bdmf_session_print(session, "%s", line);
#endif
}

static void read_mem(bdmf_session_handle session, void *adrs, uint32_t nwords, uint32_t width, uint32_t source)
{
    uint32_t value = 0;
    word_t pvalue[4];
    void *read_address = adrs;
    int32_t length = nwords;
    uint32_t linecounter = 0;
    uint32_t l = 0;
    uint16_t k16;
    uint32_t k32;
    char *line;
    char tline[10];

    line = bdmf_alloc(LINE_LENGTH);
    if (!line)
        return;
    memset((void *)line, ' ', LINE_LENGTH);
    line[LINE_LENGTH - 1] = '\0';
    for (; length > 0; read_address += width, length--)
    {
        switch (width)
        {
        case 1:
#ifdef RDP_SIM
            READ_8((uint8_t *)read_address, value);
#else
            value = *(uint8_t *)read_address;
#endif
            if (linecounter % 8 == 0)
            {
                if (linecounter != 0)
                {
                    line[strlen(line)] = ' ';
                    line[TEXT_OFFSET] = '*';
                    memset(&line[TEXT_OFFSET + 1], '.', TEXT_END);
                    for (l = 0; l < linecounter; l++)
                        p_chr(line, TEXT_OFFSET + 1 + l, (char)pvalue[l / 4].word_byte[l % 4]);
                    line[TEXT_OFFSET + TEXT_END] = '*';
                    line[TEXT_OFFSET + TEXT_END + 1] = '\0';
                    prn(source, session, line);
                }
                memset((void *)line, ' ', LINE_LENGTH);
                switch (source)
                {
                case SRC_DFP:
                    sprintf(line, "%px: %02x", read_address, (unsigned int)value);
                    break;
                case SRC_SHELL:
                    sprintf(line, "\n\r%px: %02x", read_address, (unsigned int)value);
                    break;
                }
                linecounter = 1;
            }
            else
            {
                sprintf(tline, " %02x", (unsigned int)value);
                strcat(line, tline);
                linecounter++;
            }
            pvalue[(linecounter - 1) / 4].word_byte[(linecounter - 1) % 4] = (uint8_t)value;
            break;

        case 2:
#ifdef RDP_SIM
            READ_16((uint16_t *)read_address, value);
#else
            value = swap2bytes(*(uint16_t *)read_address);
#endif
            if (linecounter % 4 == 0)
            {
                if (linecounter != 0)
                {
                    line[strlen(line)] = ' ';
                    line[TEXT_OFFSET] = '*';
                    memset(&line[TEXT_OFFSET + 1], '.', TEXT_END);
                    for (l = 0; l < linecounter; l++)
                    {
                        k16 = pvalue[l / 2].word_half[l % 2];
                        p_chr(line, TEXT_OFFSET + 1 + l*2, (char)((k16 & 0xFF00) >> 8));
                        p_chr(line, TEXT_OFFSET + 2 + l*2, (char)(k16 & 0xFF));
                    }
                    line[TEXT_OFFSET + TEXT_END] = '*';
                    line[TEXT_OFFSET + TEXT_END + 1] = '\0';
                    prn(source, session, line);
                }
                memset((void *)line, ' ', LINE_LENGTH);
                switch (source)
                {
                case SRC_DFP:
                    sprintf(line, "%px: %04x", read_address, (unsigned int)value);
                    break;
                case SRC_SHELL:
                    sprintf(line, "\n\r%px: %04x", read_address, (unsigned int)value);
                    break;
                }
                linecounter = 1;
            }
            else
            {
                sprintf(tline, " %04x", (unsigned int)value);
                strcat(line, tline);
                linecounter++;
            }
            pvalue[(linecounter - 1) / 2].word_half[(linecounter - 1) % 2] = (uint16_t)value;
            break;

        case 4:
#ifdef RDP_SIM
            READ_32((uint32_t *)read_address, value);
#else
            value = swap4bytes(*(uint32_t *)read_address);
#endif
            if (linecounter % 2 == 0)
            {
                if (linecounter != 0)
                {
                    line[strlen(line)] = ' ';
                    line[TEXT_OFFSET] = '*';
                    memset(&line[TEXT_OFFSET + 1], '.', TEXT_END);
                    for (l = 0; l < linecounter; l++)
                    {
                        k32 = pvalue[l].word;
                        p_chr(line, TEXT_OFFSET + 1 + l*4, (char)((k32 & 0xFF000000) >> 24));
                        p_chr(line, TEXT_OFFSET + 2 + l*4, (char)((k32 & 0xFF0000) >> 16));
                        p_chr(line, TEXT_OFFSET + 3 + l*4, (char)((k32 & 0xFF00) >> 8));
                        p_chr(line, TEXT_OFFSET + 4 + l*4, (char)(k32 & 0xFF));
                    }
                    line[TEXT_OFFSET + TEXT_END] = '*';
                    line[TEXT_OFFSET + TEXT_END + 1] = '\0';
                    prn(source, session, line);
                }
                memset((void *)line, ' ', LINE_LENGTH);
                switch (source)
                {
                case SRC_DFP:
                    sprintf(line, "%px: %08x", read_address, (unsigned int)value);
                    break;
                case SRC_SHELL:
                    sprintf(line, "\n\r%px: %08x", read_address, (unsigned int)value);
                    break;
                }
                linecounter = 1;
            }
            else
            {
                sprintf(tline, " %08x", (unsigned int)value);
                strcat(line, tline);
                linecounter++;
            }
            pvalue[linecounter - 1].word = (uint32_t)value;
            break;

        default:
            break;
        }
    }
    switch (width)
    {
    case 1:
        if (linecounter > 0)
        {
            line[strlen(line)] = ' ';
            line[TEXT_OFFSET] = '*';
            memset(&line[TEXT_OFFSET + 1], '.', TEXT_END);
            for (l = 0; l < linecounter; l++)
            {
                p_chr(line, TEXT_OFFSET + 1 + l, (char) pvalue[l / 4].word_byte[l % 4]);
            }
            line[TEXT_OFFSET + TEXT_END] = '*';
            line[TEXT_OFFSET + TEXT_END + 1] = '\0';
            prn(source, session, line);
        }
        break;

    case 2:
        if (linecounter > 0)
        {
            line[strlen(line)] = ' ';
            line[TEXT_OFFSET] = '*';
            memset(&line[TEXT_OFFSET + 1], '.', TEXT_END);
            for (l = 0; l < linecounter; l++)
            {
                k16 = pvalue[l / 2].word_half[l % 2];
                p_chr(line, TEXT_OFFSET + 1 + l*2, (char)((k16 & 0xFF00) >> 8));
                p_chr(line, TEXT_OFFSET + 2 + l*2, (char)(k16 & 0xFF));
            }
            line[TEXT_OFFSET + TEXT_END] = '*';
            line[TEXT_OFFSET + TEXT_END + 1] = '\0';
            prn(source, session, line);
        }
        break;

    case 4:
        if (linecounter > 0)
        {
            line[strlen(line)] = ' ';
            line[TEXT_OFFSET] = '*';
            memset(&line[TEXT_OFFSET + 1], '.', TEXT_END);
            for (l = 0; l < linecounter; l++)
            {
                k32 = pvalue[l].word;
                p_chr(line, TEXT_OFFSET + 1 + l*4, (char)((k32 & 0xFF000000) >> 24));
                p_chr(line, TEXT_OFFSET + 2 + l*4, (char)((k32 & 0xFF0000) >> 16));
                p_chr(line, TEXT_OFFSET + 3 + l*4, (char)((k32 & 0xFF00) >> 8));
                p_chr(line, TEXT_OFFSET + 4 + l*4, (char)(k32 & 0xFF));
            }
            line[TEXT_OFFSET + TEXT_END] = '*';
            line[TEXT_OFFSET + TEXT_END + 1] = '\0';
            prn(source, session, line);
        }
        break;

    default:
        break;
    }
    bdmf_free(line);
}

static void write_mem(void *write_address, uint32_t value, uint32_t width)
{
   switch (width)
   {
#ifdef RDP_SIM
   case 0:
   case 1:
       WRITE_8((uint8_t *)write_address, value);
       break;
   default:
   case 4:
   case 5:
       WRITE_32((uint16_t *)write_address, value);
       break;
   case 2:
   case 3:
       WRITE_16((uint32_t *)write_address, value);
       break;
#else
   case 0:
   case 1:
       *(uint8_t *)write_address = (uint8_t)value;
       break;
   default:
   case 4:
   case 5:
       *(uint32_t *)write_address = value;
       break;
   case 2:
   case 3:
       *(uint16_t *)write_address = (uint16_t)value;
       break;
#endif
   }
}

typedef enum {
    MISC_MEMORY_BYTE = 0,
    MISC_MEMORY_WORD,
    MISC_MEMORY_HEX,
} misc_mem_width_t;

#define WIDTH_FROM_ENUM(x) (x == MISC_MEMORY_BYTE ? 1 : (x == MISC_MEMORY_WORD ? 4 : (x == MISC_MEMORY_HEX ? 2 : 0)))

static int misc_read_memory_command(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    uint8_t parameter_index = 0;
#if defined(__LP64__) || defined(_LP64)
    uint8_t *address = (uint8_t *)(uintptr_t)parm[parameter_index++].value.unumber64; /* input */
#else
    uint8_t *address = (uint8_t *)parm[parameter_index++].value.unumber; /* input */
#endif
    misc_mem_width_t width = parm[parameter_index++].value.unumber; /* input */
    uint32_t number_of_words = parm[parameter_index++].value.unumber; /* input */

    read_mem(session, address, number_of_words, WIDTH_FROM_ENUM(width), SRC_SHELL);
    bdmf_session_print(session, "\n\r");
    return 0;
}

static int misc_write_memory_command(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    uint8_t parameter_index = 0;
#if defined(__LP64__) || defined(_LP64)
    uint8_t *address = (uint8_t *)(uintptr_t)parm[parameter_index++].value.unumber64; /* input */
#else
    uint8_t *address = (uint8_t *)parm[parameter_index++].value.unumber; /* input */
#endif
    misc_mem_width_t width = parm[parameter_index++].value.unumber; /* input */
    uint32_t value = parm[parameter_index++].value.unumber; /* input */

    write_mem(address, value, WIDTH_FROM_ENUM(width));
    bdmf_session_print(session, "Writing to memory done\n\r");
    return 0;
}

void misc_mr_cmd(void *address, int num_of_words)
{
    read_mem(NULL, address, num_of_words, WIDTH_FROM_ENUM(MISC_MEMORY_WORD), SRC_SHELL);
}

void misc_mw_cmd(void *address, uint32_t value)
{
    write_mem(address, value, WIDTH_FROM_ENUM(MISC_MEMORY_WORD));
}

#if defined(BDMF_SYSTEM_SIM)
static int misc_runner_memory_dump_command(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[],
    uint16_t n_parms)
{
    if (_segment_file_init("sim_mem", "w+b", (uint8_t *)DEVICE_ADDRESS(0), SIM_MEM_SIZE))
        return -1;

    bdmf_session_print(session, "Runner mem dump done\n\r");
    return 0;
}

#ifdef CONFIG_DHD_RUNNER
void rdd_sim_dhd_tx_flow_ring_mgmt_init(void);
#endif

#define MAX_NUM_OF_WAN_CHANNELS 40 /* Temporary, to cover all platforms */
static int misc_runner_segs_dump_command(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int rc = -1;
#ifndef XRDP
    uint16_t *tx_pointers_tbl;
    int tx_pointers_tbl_size;
#else
    int debug = (int)parm[0].value.number;
    char file_name[100] =  {};
    uint32_t rnr_idx, mem_high_byte_num, mem_cntxt_byte_num;
#endif
     
#ifndef XRDP
    tx_pointers_tbl_size = sizeof(uint16_t) * MAX_NUM_OF_WAN_CHANNELS *
        RDD_WAN_CHANNEL_0_7_DESCRIPTOR_RATE_CONTROLLER_ADDR_NUMBER *
        RDD_US_RATE_CONTROLLER_DESCRIPTOR_TX_QUEUE_ADDR_NUMBER;
    tx_pointers_tbl = (uint16_t *)bdmf_alloc(tx_pointers_tbl_size);
    if (!tx_pointers_tbl)
        goto exit;

#ifndef LEGACY_RDP
#ifdef CONFIG_DHD_RUNNER
    rdd_sim_dhd_tx_flow_ring_mgmt_init();
    rdd_save_wifi_dongle_config();
#endif
#endif

    if (rdd_sim_save_ddr_tables())
        goto exit;
    if (rdd_sim_save_hw_cfg())
        goto exit;
    rdd_sim_save_tx_pointers(tx_pointers_tbl);

    if (_segment_file_init("tx_pointers_table", "w+b", (uint8_t *)tx_pointers_tbl, tx_pointers_tbl_size))
        goto exit;
    if (segment_file_init("data_common", "w+b", RUNNER_COMMON_0_OFFSET, sizeof(RUNNER_COMMON)))
        goto exit;
    if (segment_file_init("data_private_A", "w+b", RUNNER_PRIVATE_0_OFFSET, sizeof(RUNNER_PRIVATE)))
        goto exit;
    if (segment_file_init("context_segment_A", "w+b", RUNNER_CNTXT_MAIN_0_OFFSET, sizeof(RUNNER_CNTXT_MAIN)))
        goto exit;
    if (segment_file_init("context_segment_C", "w+b", RUNNER_CNTXT_PICO_0_OFFSET, sizeof(RUNNER_CNTXT_PICO)))
        goto exit;
    if (segment_file_init("data_common", "a+b", RUNNER_COMMON_1_OFFSET, sizeof(RUNNER_COMMON)))
        goto exit;
    if (segment_file_init("data_private_B", "w+b", RUNNER_PRIVATE_1_OFFSET, sizeof(RUNNER_PRIVATE)))
        goto exit;
    if (segment_file_init("context_segment_B", "w+b", RUNNER_CNTXT_MAIN_1_OFFSET, sizeof(RUNNER_CNTXT_MAIN)))
        goto exit;
    if (segment_file_init("context_segment_D", "w+b", RUNNER_CNTXT_PICO_1_OFFSET, sizeof(RUNNER_CNTXT_PICO)))
        goto exit;
    if (segment_file_init("sim_mem", "w+b", 0, SIM_MEM_SIZE))
        goto exit;
#else
    mem_high_byte_num = (RU_REG_RAM_CNT(RNR_MEM, HIGH) + 1) * sizeof(uint32_t);
    mem_cntxt_byte_num = (RU_REG_RAM_CNT(RNR_CNTXT, MEM_ENTRY) + 1) * sizeof(uint32_t);

    for (rnr_idx = 0; rnr_idx < RNR_MEM_BLOCK.addr_count; rnr_idx++)
    {
        sprintf(file_name, "data_mem_%d", rnr_idx);
        if (segment_file_init(file_name, "w+b", (RU_BLK(RNR_MEM).addr[rnr_idx] + RU_REG_OFFSET(RNR_MEM, HIGH)),
            mem_high_byte_num * 2))
            goto exit;
        sprintf(file_name, "context_segment_%d", rnr_idx);
        if (segment_file_init(file_name, "w+b", (RU_BLK(RNR_CNTXT).addr[rnr_idx] + RU_REG_OFFSET(RNR_CNTXT, MEM_ENTRY)),
            mem_cntxt_byte_num))
            goto exit;
    }
    if (debug)
    {
        if (rdd_sim_save_ddr_tables())
            goto exit;
    }
    if (_segment_file_init("sim_mem", "w+b", (uint8_t *)soc_base_address, SIM_MEM_SIZE))
        goto exit;
    if (drv_tcam_mem_dump(session, "tcam"))
        goto exit;
#endif
    rc = 0;

exit:
#ifndef XRDP
    if (tx_pointers_tbl)
        bdmf_free(tx_pointers_tbl);
#endif
    bdmf_session_print(session, "Dump segments %s\n\r", rc ? "failed" : "done");
    return rc;
}
#endif
static bdmfmon_handle_t misc_dir;


#if defined(XRDP)
/* write SRAM
 *   BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, 15),
 *   BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
 *   BDMFMON_MAKE_PARM("data", "Data to write", BDMFMON_PARM_STRING, 0));
 */
static int misc_sram_write_handler(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int core = (int)parm[0].value.number;
    uint32_t addr = (uint32_t)parm[1].value.number;
    const char *data_str = (const char *)parm[2].value.string;

    uint8_t *start_addr = (uint8_t *)(DEVICE_ADDRESS(rdp_runner_core_addr[core]) + addr);
    uint32_t len = strlen(data_str);
    int n;

    if ((len % 2) == 1)
    {
        bdmf_session_print(session, "Hex string must have even number of bytes. Got %u\n", len);
        return BDMF_ERR_PARM;
    }
    n =  bdmf_strhex(data_str, start_addr, len/2);
    if (n != len/2)
    {
        bdmf_session_print(session, "Hex string must have even number of bytes. Got %u\n", len);
        return BDMF_ERR_PARM;
    }

    return BDMF_ERR_OK;
}

/* write SRAM
 *   BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, 15),
 *   BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
 *   BDMFMON_MAKE_PARM("data", "Data to write", BDMFMON_PARM_STRING, 0));
 */
static int misc_image_sram_write_handler(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int core = (int)parm[0].value.number;
    uint32_t addr = (uint32_t)parm[1].value.number;
    const char *data_str = (const char *)parm[2].value.string;
    rdp_runner_image_e image_id = rdp_core_to_image_map[core];
    uint8_t *start_addr;
    uint32_t len = strlen(data_str);
    int i, n;

    if ((len % 2) == 1)
    {
        bdmf_session_print(session, "Hex string must have even number of bytes. Got %u\n", len);
        return BDMF_ERR_PARM;
    }
    bdmf_session_print(session, "image_id = %d cores: ", image_id);
    for (i = 0; i < RNR_LAST; i++)
    {
        if (image_id != rdp_core_to_image_map[i])
            continue;
        start_addr = (uint8_t *)(DEVICE_ADDRESS(rdp_runner_core_addr[i]) + addr);
        bdmf_session_print(session, "%d, ", i);
        n =  bdmf_strhex(data_str, start_addr, len/2);
        if (n != len/2)
        {
            bdmf_session_print(session, "Hex string must have even number of bytes. Got %u\n", len);
            return BDMF_ERR_PARM;
        }
    }
    bdmf_session_print(session, "\n");
    return BDMF_ERR_OK;
}

/*   SRAM MEMSET
 *   BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, 15),
 *   BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
 *   BDMFMON_MAKE_PARM("data", "Data to write", BDMFMON_PARM_STRING, 0));
 *   BDMFMON_MAKE_PARM("size", "size of data to fill"),
 */
static int misc_sram_memset_write_handler(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int core = (int)parm[0].value.number;
    uint32_t addr = (uint32_t)parm[1].value.number;
    uint8_t data = (uint8_t)parm[2].value.number;
    uint16_t size = (uint32_t)parm[3].value.number;
    uint8_t *w_addr = (uint8_t *)(DEVICE_ADDRESS(rdp_runner_core_addr[core]) + addr);
    int i;

    if ((addr + size) >= 0x4000)    /* SRAM size is 0x4000 */
    {
        bdmf_session_print(session, "SRAM memory was exceeded. Got to address 0x%x\n", addr + size);
        return BDMF_ERR_PARM;
    }

    for (i = 0; i < size; i++)
        *w_addr++ = data;

    return BDMF_ERR_OK;
}

/* Read SRAM
 *   BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, 15),
 *   BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
 *   BDMFMON_MAKE_PARM_RANGE("size", "Area size", BDMFMON_PARM_NUMBER, 0, 1, 512));
 */
static int misc_sram_read_handler(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int core = (int)parm[0].value.number;
    uint32_t addr = (uint32_t)parm[1].value.number;
    uint32_t size = (uint32_t)parm[2].value.number;

    uint8_t *start_addr = (uint8_t *)(DEVICE_ADDRESS(rdp_runner_core_addr[core]) + addr);

    bdmf_session_hexdump(session, start_addr, 0, size);

    return BDMF_ERR_OK;
}

#if !defined(BDMF_SYSTEM_SIM)
/* WAkeup runner task
 *   BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, 15),
 *   BDMFMON_MAKE_PARM_RANGE("task", "Task index", BDMFMON_PARM_NUMBER, 0, 0, 31));
 */
static int misc_task_wakeup_handler(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int core = (int)parm[0].value.number;
    int task = (int)parm[1].value.number;
    ag_drv_rnr_regs_cfg_cpu_wakeup_set(core, task);
    return BDMF_ERR_OK;
}
#endif

#endif

/* Setting system global EPON Mode from shell commands*/
#if defined(BDMF_SYSTEM_SIM) && defined(EPON)
extern rdpa_epon_mode g_epon_mode;
static struct bdmfmon_enum_val epon_mode_enum_table[] = {
    {"none", rdpa_epon_none},
    {"ctc", rdpa_epon_ctc},
    {"cuc", rdpa_epon_cuc},
    {"dpoe", rdpa_epon_dpoe},
    {"bcm", rdpa_epon_bcm},
    {"ctc_dyn", rdpa_epon_ctc_dyn},
    {"cuc_dyn", rdpa_epon_cuc_dyn},
    {NULL, 0},
};
static int misc_epon_mode_set(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    g_epon_mode = parm[0].value.unumber;
    return BDMF_ERR_OK;
}
#endif

#if defined(XRDP) && defined(BCM_PON_XRDP) && !defined(RDP_UFC)
/* List of actions after invoke of rdpa_forward_action2rdd_action */
static struct bdmfmon_enum_val __rdd_action_enum_table[] =
{
    {"forward", ACTION_FORWARD},
    {"host", ACTION_TRAP},
    {"drop", ACTION_DROP},
    {"flood", ACTION_MULTICAST}, /* use only for bridge DAL */
    {NULL, 0}
};

/* Read VLAN Hash entry by port and VID */
static int misc_vlan_hash_entry_handler_read(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[],
    uint16_t n_parms)
{
    BRIDGE_AND_VLAN_LKP_RESULT_STRUCT bridge_and_vlan_lkp_result;
    const char *port_name = (const char *)parm[0].value.string;
    int16_t vid = (int16_t)parm[1].value.number;
    bdmf_object_handle port_obj;
    port_drv_priv_t *port;
    int rc;

    rdpa_port_get(port_name, &port_obj);
    if (port_obj == NULL)
    {
        bdmf_session_print(session, "Port object not found\n");
        return BDMF_ERR_OK;
    }
    port = (port_drv_priv_t *)bdmf_obj_data(port_obj);

    bdmf_put(port_obj);

    rc = rdpa_vlan_hash_entry_read(PORT_OBJECT_VPORT_GET(port), vid, &bridge_and_vlan_lkp_result);
    if (rc)
    {
        if (rc == BDMF_ERR_NOENT)
            bdmf_session_print(session, "Entry not found\n");
        else
            bdmf_session_print(session, "Entry lookup in hash failed, rc = %d\n", rc);
        return BDMF_ERR_OK;
    }

    bdmf_session_print(session, "== Result:\n\t"
        "bridge_id = %d, port_isolation_map = 0x%x,\n\t"
        "ingress_filter_profile = 0x%x, protocol_filters_dis = 0x%x,\n\t"
        "sa_lookup_en = %d, sa_lookup_miss_action = %d,\n\t"
        "da_lookup_en = %d, da_lookup_miss_action = %d,\n\t"
        "aggregation_en = %d, counter_id = %d, counter_id_valid = %d\n",
        bridge_and_vlan_lkp_result.bridge_id, bridge_and_vlan_lkp_result.port_isolation_map,
        bridge_and_vlan_lkp_result.ingress_filter_profile, bridge_and_vlan_lkp_result.protocol_filters_dis,
        bridge_and_vlan_lkp_result.sa_lookup_en, bridge_and_vlan_lkp_result.sa_lookup_miss_action,
        bridge_and_vlan_lkp_result.da_lookup_en, bridge_and_vlan_lkp_result.da_lookup_miss_action,
        bridge_and_vlan_lkp_result.aggregation_en, 
        bridge_and_vlan_lkp_result.counter_id, bridge_and_vlan_lkp_result.counter_id_valid);

    return BDMF_ERR_OK;
}

static int misc_vlan_hash_entry_handler_write(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[],
    uint16_t n_parms)
{
    BRIDGE_AND_VLAN_LKP_RESULT_STRUCT bridge_and_vlan_lkp_result;
    int parm_idx = 0;
    const char *port_name = (const char *)parm[parm_idx++].value.string;
    int16_t vid = (int16_t)parm[parm_idx++].value.number;
    uint32_t vport_mask = (uint32_t)parm[parm_idx++].value.number;
    bdmf_object_handle port_obj;
    port_drv_priv_t *port;
    int rc;

    rdpa_port_get(port_name, &port_obj);
    if (port_obj == NULL)
    {
        bdmf_session_print(session, "Port object not found\n");
        return BDMF_ERR_OK;
    }
    port = (port_drv_priv_t *)bdmf_obj_data(port_obj);

    bdmf_put(port_obj);

    rdpa_vlan_hash_entry_isolation_vector_set(vport_mask, &bridge_and_vlan_lkp_result);
    bridge_and_vlan_lkp_result.bridge_id = (int16_t)parm[parm_idx++].value.number;
    bridge_and_vlan_lkp_result.ingress_filter_profile = (uint8_t)parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.protocol_filters_dis = (uint8_t)parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.sa_lookup_en = (uint8_t)parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.da_lookup_en = (uint8_t)parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.sa_lookup_miss_action = parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.da_lookup_miss_action = parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.aggregation_en = (uint8_t)parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.counter_id = (uint8_t)parm[parm_idx++].value.unumber;
    bridge_and_vlan_lkp_result.counter_id_valid = (uint8_t)parm[parm_idx++].value.unumber;


    bdmf_session_print(session, "== Writing new entry, Key (port = %s, vid = %d), Result:\n\t"
        "bridge_id = %d, port_isolation_map = 0x%x,\n\t"
        "ingress_filter_profile = 0x%x, protocol_filters_dis = 0x%x,\n\t"
        "sa_lookup_en = %d, sa_lookup_miss_action = %d,\n\t"
        "da_lookup_en = %d, da_lookup_miss_action = %d,\n\t"
        "aggregation_en = %d, counter_id = %d, counter_id_valid = %d\n",
        port_name, vid,
        bridge_and_vlan_lkp_result.bridge_id, bridge_and_vlan_lkp_result.port_isolation_map,
        bridge_and_vlan_lkp_result.ingress_filter_profile, bridge_and_vlan_lkp_result.protocol_filters_dis,
        bridge_and_vlan_lkp_result.sa_lookup_en, bridge_and_vlan_lkp_result.sa_lookup_miss_action,
        bridge_and_vlan_lkp_result.da_lookup_en, bridge_and_vlan_lkp_result.da_lookup_miss_action,
        bridge_and_vlan_lkp_result.aggregation_en, 
        bridge_and_vlan_lkp_result.counter_id, bridge_and_vlan_lkp_result.counter_id_valid);

    rc = rdpa_vlan_hash_entry_write(PORT_OBJECT_VPORT_GET(port), vid, &bridge_and_vlan_lkp_result);
    if (rc)
    {
        bdmf_session_print(session, "Failed to write new entry to hash, rc = %d\n", rc);
        return BDMF_ERR_OK;
    }

    bdmf_session_print(session, "Done\n");
    return BDMF_ERR_OK;
}

/* Delete VLAN Hash entry by port and VID */
static int misc_vlan_hash_entry_handler_del(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[],
    uint16_t n_parms)
{
    const char *port_name = (const char *)parm[0].value.string;
    int16_t vid = (int16_t)parm[1].value.number;
    bdmf_object_handle port_obj;
    port_drv_priv_t *port;
    int rc;

    rdpa_port_get(port_name, &port_obj);
    if (port_obj == NULL)
    {
        bdmf_session_print(session, "Port object not found\n");
        return BDMF_ERR_OK;
    }
    port = (port_drv_priv_t *)bdmf_obj_data(port_obj);
    bdmf_put(port_obj);

    rc = rdpa_vlan_hash_entry_delete(port->index, vid);
    if (rc)
    {
        if (rc == BDMF_ERR_NOENT)
            bdmf_session_print(session, "Entry not found\n");
        else
            bdmf_session_print(session, "Entry delete in hash failed, rc = %d\n", rc);
    }
    else
        bdmf_session_print(session, "Done\n");

    return BDMF_ERR_OK;
}

#if !defined(HW_AGGREGATION_FLUSH)
/* disable flush task */
static int disable_flush_task_write(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[],
    uint16_t n_parms)
{
    int rc;
    uint16_t disable = parm[0].value.unumber;

    rc = rdd_ag_us_tm_flush_aggregation_task_disable_set(disable);
    if (rc)
        return rc;

    rc = rdd_ag_ds_tm_flush_aggregation_task_disable_set(disable);

    return BDMF_ERR_OK;
}
#endif /*!defined(HW_AGGREGATION_FLUSH)*/

#endif /* defined(XRDP) && defined(BCM_PON_XRDP) */


void misc_shell_init(void)
{
    static struct bdmfmon_enum_val size_enum_table[] = {
        {"w",   MISC_MEMORY_WORD},
        {"h",   MISC_MEMORY_HEX},
        {"b",   MISC_MEMORY_BYTE},
        {NULL, 0},
    };

    misc_dir = NULL;

    misc_dir = bdmfmon_dir_find(NULL, "misc");
    if (!misc_dir)
    {
        misc_dir = bdmfmon_dir_add(NULL, "misc", "miscellaneous", BDMF_ACCESS_ADMIN, NULL);
        if (!misc_dir)
        {
            bdmf_session_print(NULL, "Can't create %s directory\n", "misc");
            return;
        }
    }

    BDMFMON_MAKE_CMD(misc_dir, "mr", "read memory", misc_read_memory_command,
#if defined(__LP64__) || defined(_LP64)
        BDMFMON_MAKE_PARM("address", "address: Hex64", BDMFMON_PARM_HEX64, 0),
#else
        BDMFMON_MAKE_PARM("address", "address: Hex", BDMFMON_PARM_HEX, 0),
#endif
        BDMFMON_MAKE_PARM_ENUM("width", "width", size_enum_table, 0),
        BDMFMON_MAKE_PARM("length", "number of words", BDMFMON_PARM_NUMBER, 0));

    BDMFMON_MAKE_CMD(misc_dir, "mw", "write memory", misc_write_memory_command,
#if defined(__LP64__) || defined(_LP64)
        BDMFMON_MAKE_PARM("address", "address: Hex64", BDMFMON_PARM_HEX64, 0),
#else
        BDMFMON_MAKE_PARM("address", "address: Hex", BDMFMON_PARM_HEX, 0),
#endif
        BDMFMON_MAKE_PARM_ENUM("width", "width", size_enum_table, 0),
        BDMFMON_MAKE_PARM("value", "value: Hex", BDMFMON_PARM_HEX, 0));
#if defined(BDMF_SYSTEM_SIM)
    BDMFMON_MAKE_CMD_NOPARM(misc_dir, "rmd", "runner memory dump", misc_runner_memory_dump_command);
    BDMFMON_MAKE_CMD(misc_dir, "rsmd", "runner segments memory dump", misc_runner_segs_dump_command,
        BDMFMON_MAKE_PARM_DEFVAL("debug", "Debug level", BDMFMON_PARM_NUMBER, 0, 0));
#ifdef XRDP
    BDMFMON_MAKE_CMD(misc_dir, "connect", "set connect mode", cpu_runner_sim_connect,
        BDMFMON_MAKE_PARM("port", "session port", BDMFMON_PARM_NUMBER, BDMFMON_PARM_FLAG_OPTIONAL));
#endif
#if defined(EPON)
    BDMFMON_MAKE_CMD(misc_dir, "sem", "set epon mode", misc_epon_mode_set,
        BDMFMON_MAKE_PARM_ENUM("mode", "epon mode", epon_mode_enum_table, 0));
#endif
#endif

#if defined(XRDP)
    BDMFMON_MAKE_CMD(misc_dir, "sramw", "Write SRAM", misc_sram_write_handler,
        BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, (NUM_OF_RUNNER_CORES-1)),
        BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
        BDMFMON_MAKE_PARM("data", "Data to write", BDMFMON_PARM_STRING, 0));

    BDMFMON_MAKE_CMD(misc_dir, "sramr", "Read SRAM", misc_sram_read_handler,
        BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, (NUM_OF_RUNNER_CORES-1)),
        BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
        BDMFMON_MAKE_PARM_RANGE("size", "Area size", BDMFMON_PARM_NUMBER, 0, 1, 512));

    BDMFMON_MAKE_CMD(misc_dir, "isramw", "Image write SRAM", misc_image_sram_write_handler,
        BDMFMON_MAKE_PARM_RANGE("core", "Core index (image inferred from core)", BDMFMON_PARM_NUMBER, 0, 0,
        (NUM_OF_RUNNER_CORES-1)),
        BDMFMON_MAKE_PARM("address", "SRAM address", BDMFMON_PARM_HEX, 0),
        BDMFMON_MAKE_PARM("data", "Data to write", BDMFMON_PARM_STRING, 0));

    BDMFMON_MAKE_CMD(misc_dir, "sramws", "Write SRAM memset", misc_sram_memset_write_handler,
            BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, (NUM_OF_RUNNER_CORES-1)),
            BDMFMON_MAKE_PARM("address", "SRAM address (Hex)", BDMFMON_PARM_HEX, 0),
            BDMFMON_MAKE_PARM("data", "Data to write (Hex)", BDMFMON_PARM_HEX, 0),
            BDMFMON_MAKE_PARM("size", "Number of bytes to be filled (Dec)", BDMFMON_PARM_NUMBER, 0));

#if !defined(BDMF_SYSTEM_SIM)
    BDMFMON_MAKE_CMD(misc_dir, "wakeup", "Wakeup task", misc_task_wakeup_handler,
        BDMFMON_MAKE_PARM_RANGE("core", "Core index", BDMFMON_PARM_NUMBER, 0, 0, (NUM_OF_RUNNER_CORES-1)),
        BDMFMON_MAKE_PARM_RANGE("task", "Task index", BDMFMON_PARM_NUMBER, 0, 0, 31));
#endif
#endif
#if defined(XRDP) && defined(BCM_PON_XRDP) && !defined(RDP_UFC)
    BDMFMON_MAKE_CMD(misc_dir, "rvh", "Read VLAN Hash entry", misc_vlan_hash_entry_handler_read,
        BDMFMON_MAKE_PARM("port_name", "port name", BDMFMON_PARM_STRING, 0),
        BDMFMON_MAKE_PARM_RANGE("vid", "VID", BDMFMON_PARM_NUMBER, 0, 0, RDPA_MAX_VID));

    BDMFMON_MAKE_CMD(misc_dir, "wvh", "Write VLAN Hash entry", misc_vlan_hash_entry_handler_write,
        BDMFMON_MAKE_PARM("port_mame", "port name", BDMFMON_PARM_STRING, 0),
        BDMFMON_MAKE_PARM_RANGE("vid", "VID", BDMFMON_PARM_NUMBER, 0, 0, RDPA_MAX_VID),
        BDMFMON_MAKE_PARM("vports_vector", "Vport vector", BDMFMON_PARM_HEX, 0), /* TODO: Add enum mask */
        BDMFMON_MAKE_PARM_RANGE("bridge_id", "BRIDGE ID", BDMFMON_PARM_NUMBER, 0, 0, RDPA_BRIDGE_MAX_BRIDGES),
        BDMFMON_MAKE_PARM("filter_profile", "Ingress filter profile (0-f, 0x3f for invalid)", BDMFMON_PARM_HEX, 0),
        BDMFMON_MAKE_PARM("dis_proto", "Disabled protocols mask (bits: 0-ipv4, 1-ipv6, 2-pppoe, 3-non-ip)",
            BDMFMON_PARM_HEX, 0), /* TODO: Add enum mask */
        BDMFMON_MAKE_PARM_RANGE("sal", "SA Lookup (0-disabled, 1-enabled)", BDMFMON_PARM_NUMBER, 0, 0, 1),
        BDMFMON_MAKE_PARM_RANGE("dal", "DA Lookup (0-disabled, 1-enabled)", BDMFMON_PARM_NUMBER, 0, 0, 1),
        BDMFMON_MAKE_PARM_ENUM("sal_action", "SA Lookup action", __rdd_action_enum_table, 0),
        BDMFMON_MAKE_PARM_ENUM("dal_action", "DA Lookup action", __rdd_action_enum_table, 0),
        BDMFMON_MAKE_PARM_RANGE("aggr", "Aggregation (0-disabled, 1-enabled)", BDMFMON_PARM_NUMBER, 0, 0, 1),
        BDMFMON_MAKE_PARM_RANGE("vlan_counter_index", "Vlan counter Index 0-127", BDMFMON_PARM_NUMBER, 0, 0, 127),
        BDMFMON_MAKE_PARM_RANGE("counter_id_valid", "counter id valid (0-no, 1-yes)", BDMFMON_PARM_NUMBER, 0, 0, 1));

    BDMFMON_MAKE_CMD(misc_dir, "dvh", "Delete VLAN Hash entry", misc_vlan_hash_entry_handler_del,
        BDMFMON_MAKE_PARM("port_mame", "port name", BDMFMON_PARM_STRING, 0),
        BDMFMON_MAKE_PARM_RANGE("vid", "VID", BDMFMON_PARM_NUMBER, 0, 0, RDPA_MAX_VID));
#if !defined(HW_AGGREGATION_FLUSH)
    BDMFMON_MAKE_CMD(misc_dir, "disable_flush_task", "disable flush task", disable_flush_task_write,
        BDMFMON_MAKE_PARM_RANGE("disable", "disable flush task (0-no, 1-yes)", BDMFMON_PARM_NUMBER, 0, 0, 1));
#endif
#endif
}

void misc_shell_uninit(void)
{
    if (misc_dir)
    {
        bdmfmon_token_destroy(misc_dir);
        misc_dir = NULL;
    }
}

#endif
