/*
    <:copyright-BRCM:2015:DUAL/GPL:standard
    
       Copyright (c) 2015 Broadcom 
       All Rights Reserved
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2, as published by
    the Free Software Foundation (the "GPL").
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    
    A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
    writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
    
:>
*/

#ifndef _RDP_DRV_TCAM_H
#define _RDP_DRV_TCAM_H

#include "access_macros.h"
#include "bdmf_interface.h"
#include "rdp_common.h"
#include "xrdp_drv_drivers_common_ag.h"
#include "xrdp_drv_tcam_ag.h"
#include "rdp_platform.h"

#if defined(BCM63146) || defined(BCM4912)
#define TCAM_KEY_SHADOW_HW_WAR
#endif

/* Defines for compatibility between different versions of the HAL generator. */
#ifdef TCAM_CONTEXT_RAM_CONTEXT_REG_RAM_CNT
#define TCAM_CONTEXT_RAM_TCAM_TCAM_CONTEXT_RAM_CONTEXT_REG_RAM_CNT TCAM_CONTEXT_RAM_CONTEXT_REG_RAM_CNT
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_OP_DONE_REG               TCAM_INDIRECT_OP_DONE_REG
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_OP_DONE_REG_OFFSET        TCAM_INDIRECT_OP_DONE_REG_OFFSET
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_OP_DONE_DONE_FIELD        TCAM_INDIRECT_OP_DONE_DONE_FIELD
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_OP_DONE_DONE_FIELD_MASK   TCAM_INDIRECT_OP_DONE_DONE_FIELD_MASK
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_OP_DONE_DONE_FIELD_SHIFT  TCAM_INDIRECT_OP_DONE_DONE_FIELD_SHIFT
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_KEY_OUT_REG               TCAM_INDIRECT_KEY_OUT_REG
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_KEY_OUT_REG_OFFSET        TCAM_INDIRECT_KEY_OUT_REG_OFFSET
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_MATCH_FIELD          TCAM_INDIRECT_RSLT_MATCH_FIELD
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_MATCH_FIELD_MASK     TCAM_INDIRECT_RSLT_MATCH_FIELD_MASK
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_MATCH_FIELD_SHIFT    TCAM_INDIRECT_RSLT_MATCH_FIELD_SHIFT
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_INDEX_FIELD          TCAM_INDIRECT_RSLT_INDEX_FIELD
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_INDEX_FIELD_MASK     TCAM_INDIRECT_RSLT_INDEX_FIELD_MASK
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_INDEX_FIELD_SHIFT    TCAM_INDIRECT_RSLT_INDEX_FIELD_SHIFT
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_REG                  TCAM_INDIRECT_RSLT_REG
#define TCAM_INDIRECT_TCAM_TCAM_INDIRECT_RSLT_REG_OFFSET           TCAM_INDIRECT_RSLT_REG_OFFSET
#define ag_drv_tcam_debug_bus_tcam_debug_bus_select_set            ag_drv_tcam_tcam_debug_bus_select_set
#define ag_drv_tcam_debug_bus_tcam_debug_bus_select_get            ag_drv_tcam_tcam_debug_bus_select_get
#endif

#define RDP_TCAM_CONTEXT_SIZE           4

/* Max TCAM key size in bytes */
#define RDP_TCAM_MAX_KEY_SIZE           64

/* Max number of TCAM priorities.
 * TCAM rules are sorted in order of priorities.
 * Lesser number means higher priority
 */
#define RDP_TCAM_MAX_PRIORITIES         128

/* TCAM key size */
typedef enum
{
#if CHIP_VER >= RDP_GEN_62
    RDP_TCAM_KEY_128,   /**< 128 bit key */
#endif

    RDP_TCAM_KEY_256,   /**< 256 bit key */

#if CHIP_VER < RDP_GEN_62
    RDP_TCAM_KEY_512    /**< 512 bit key */
#endif
} rdp_tcam_key_type;

/* TCAM rule priority. The value range is 0..RDP_TCAM_MAX_PRIORITIES-1 */
typedef uint8_t rdp_tcam_priority;

/* TCAM key / mask area */
typedef union rdp_tcam_key_area
{
    uint8_t b[RDP_TCAM_MAX_KEY_SIZE];
    uint32_t w[RDP_TCAM_MAX_KEY_SIZE / 4];
} rdp_tcam_key_area_t;

#if defined(TCAM_KEY_SHADOW_HW_WAR)
/* TCAM key / mask shadow area */
typedef struct rdp_tcam_key_shadow_area
{
    rdp_tcam_key_area_t tcam_key_mem;    
    uint32_t valid;
} rdp_tcam_key_shadow_area_t;
#endif

/* TCAM context */
typedef struct rdp_tcam_context
{
    uint32_t word[RDP_TCAM_CONTEXT_SIZE];   /**< Context words in CPU byte order */
} rdp_tcam_context_t;

/* TCAM commands */
typedef enum
{
    TCAM_CMD_READ       = 0,
    TCAM_CMD_WRITE      = 1,
    TCAM_CMD_COMPARE    = 2,
    TCAM_CMD_INVALIDATE = 3,
} tcam_cmd;

/* Set tcam mode
 * \param[in]   key_type             Key size
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_mode_set(rdp_tcam_key_type key_type);

/* Get tcam mode
 * \param[out]  key_type             Key size
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_mode_get(rdp_tcam_key_type *key_type);

/* Get TCAM key size
 * \param[out]  key_size             Key size (bytes)
 * \returns: 0=0 or error < 0
 */
static inline bdmf_error_t drv_tcam_keysize_get(uint32_t *key_size)
{
    rdp_tcam_key_type key_type;
    bdmf_error_t err = drv_tcam_mode_get(&key_type);
    if (err)
        return err;
    *key_size = (key_type == RDP_TCAM_KEY_256) ? 256 / 8 : 512 / 8;
    return BDMF_ERR_OK;
}

/* Add classification rule
 *
 * The function inserts new entry into the correct place in TCAM table(s)
 * based on the entry's priority.
 * The tables are updated safely. Firmware can perform searches while
 * the tables are being updated.
 *
 * The function must be called under lock!
 *
 * \param[in]   priority             Rule priority
 * \param[in]   key                  Key area
 * \param[in]   mask                 Mask area
 * \param[in]   context              Rule context
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_rule_add(
    rdp_tcam_priority                       priority,
    const rdp_tcam_key_area_t              *key,
    const rdp_tcam_key_area_t              *mask,
    const rdp_tcam_context_t               *context);

/* Modify classification context
 *
 * The function inserts new context into the correct place in TCAM table(s)
 * based on the entry's index.
 *
 * The function must be called under lock!
 *
 * \param[in]   rule_index           Index in TCAM table
 * \param[in]   context              Rule context
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_rule_modify(
    uint16_t                               rule_index,
    rdp_tcam_context_t                     *context);

/* Delete classification rule
 *
 * The function deletes existing entry from TCAM table(s).
 * The tables are updated safely. Firmware can perform searches while
 * the tables are being updated.
 *
 * The function must be called under lock!
 *
 * \param[in]   key                  Key area
 * \param[in]   mask                 Mask area
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_rule_delete(
    const rdp_tcam_key_area_t              *key,
    const rdp_tcam_key_area_t              *mask);

/* Find classification rule
 *
 * The function must be called under lock!
 *
 * \param[in]   key                  Key area
 * \param[in]   rule_index           Index in TCAM table
 * \param[in]   context              Rule context
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_rule_find(
    const rdp_tcam_key_area_t              *key,
    uint16_t                               *rule_index,
    rdp_tcam_context_t                     *context);

/* Find TCAM rule and return index
 *
 * \param[in]   key                  Key area
 * \param[in]   mask                 Mask area
 * \param[in]   rule_index           Index in TCAM table
 * \returns: 0=0 or error < 0
 */
bdmf_error_t drv_tcam_rule_lkup(
    const rdp_tcam_key_area_t              *key,
    const rdp_tcam_key_area_t              *mask,
    uint16_t  *rule_index);

/*
 * CLI support
 */
#ifdef USE_BDMF_SHELL
void drv_tcam_cli_init(bdmfmon_handle_t driver_dir);
void drv_tcam_cli_exit(bdmfmon_handle_t driver_dir);
bdmf_error_t drv_tcam_mem_dump(bdmf_session_handle session, const char *filename);
#endif

#endif /* _RDP_DRV_TCAM_H */

