/*
    <:copyright-BRCM:2015:DUAL/GPL:standard
    
       Copyright (c) 2015 Broadcom 
       All Rights Reserved
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2, as published by
    the Free Software Foundation (the "GPL").
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    
    A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
    writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
    
    :>
*/

#include "rdp_subsystem_common.h"
#include "rdp_drv_cnpl.h"
#include "rdp_common.h"
#include "rdd_defs.h"
#include "XRDP_AG.h"
#include "rdp_drv_cntr.h"
#include "xrdp_drv_cnpl_ag.h"
#include "bdmf_dev.h"

DEFINE_BDMF_FASTLOCK(counter_read_lock);

/*
    counter configuration example:
        call the function: ag_drv_cnpl_counter_cfg_set to set up a new counter group
            cnt_loc_profile: group number (0-15)
            cn_double: set each counter to be single or double (boolean)
            cn0_byts: number of bytes for each counter (1, 2, 4)
            ba: base address for the counter group (10bit in 8B resolution)
            wrap: freeze or wrap around (boolean)
            clr: clear on read (boolean)

    counter read example:
        1. send read command by calling the function: drv_cnpl_counter_read_command_set
            group: choose the counter group (4bit)
            cntr: choose start counter (14bit)
            size: number of counters to read (8bit)
        2. check status register: ag_drv_cnpl_sw_stat_get
            cn_rd_st: counter read status (0, 1, 2)
        3. read the result from: ag_drv_cnpl_block_sw_if_sw_cnt_rd_get
            rd_idx: read bytes rd_idx*4--rd_idx*4+3 (0-7)
*/

/*
    policer configuration example:
        1. memory reset
        2. call the function: ag_drv_cnpl_policer_cfg_set to set up a new policer
            conf_idx: profile number (0, 1)
            bk_ba: bucket base address (10bit in 8B resolution)
            pa_ba: parameters base address (10bit)
            pl_double: set each policer to be single or double (boolean)
            pl_st: first policer in the group (8bit)
            pl_end: last policer in the group (8bit)
        3. call the function: ag_drv_cnpl_block_policers_configurations_per_up_set to start periodic update
            en: periodic update enable (boolean)
            N: period in 8k cycles quanta (8bit)

    policer read example:
        1. send read command by calling the function: drv_cnpl_policer_read_command_set
            group: choose the policer group (2bit)
            policer_num: choose start policer (8bit)
            reset_after_read: set reset (boolean)
        2. check status register: ag_drv_cnpl_sw_stat_get
            pl_rd_st: policer read status (0, 1, 2)
        3. read the result from: ag_drv_cnpl_block_sw_if_sw_pl_rd_get
            bucket: read bytes bucket*4--bucket*4+3 (0-7)

    policer police example:
        1. send police command by calling the function: drv_cnpl_policer_police_command_set
            group: choose the policer group (2bit)
            policer_num: choose start policer (8bit)
            packet_len: packet length (16bit)
        2. check status register: ag_drv_cnpl_sw_stat_get
            pl_plc_st: policer police status (0, 1, 2)
        3. read the color from: ag_drv_cnpl_block_sw_if_sw_pl_rslt_get
*/

#if CHIP_VER >= RDP_GEN_62
/* cpu_port enum values */
bdmf_attr_enum_table_t bufmng_enum_table = {
    .values = {
        {"CPU RX     ", BUF_MNG_CPU_RX_CNTR},
        {"CPU RX HIGH", BUF_MNG_CPU_RX_HIGH_CNTR},
        {"WLAN_0     ", BUF_MNG_WLAN_0_CNTR},
        {"WLAN_0 HIGH", BUF_MNG_WLAN_HIGH_0_CNTR},
        {"WLAN_0 RX  ", BUF_MNG_WLAN_RX_0_CNTR},
        {"WLAN_1     ", BUF_MNG_WLAN_1_CNTR},
        {"WLAN_1 HIGH", BUF_MNG_WLAN_HIGH_1_CNTR},
        {"WLAN_1 RX  ", BUF_MNG_WLAN_RX_1_CNTR},
        {"WLAN_2     ", BUF_MNG_WLAN_2_CNTR},
        {"WLAN_2 HIGH", BUF_MNG_WLAN_HIGH_2_CNTR},
        {"WLAN_2 RX  ", BUF_MNG_WLAN_RX_2_CNTR},
        {"WLAN_3     ", BUF_MNG_WLAN_3_CNTR},
        {"WLAN_3 HIGH", BUF_MNG_WLAN_HIGH_3_CNTR},
        {"WLAN_3 RX  ", BUF_MNG_WLAN_RX_3_CNTR},
        {"WLAN GROUP ", BUF_MNG_WLAN_TOTAL_CNTR},
        {"BE1 GROUP  ", BUF_MNG_BE1_GROUP_CNTR},
        {"BE2 GROUP  ", BUF_MNG_BE2_GROUP_CNTR},
        {"WIRED GROUP", BUF_MNG_WIRED_GROUP_CNTR},
        {"ALL PASS   ", BUF_MNG_ALL_PASS_CNTR},
        {"TOTAL      ", BUF_MNG_TOTAL_BUDGET_CNTR},
        {NULL,      0}
    }
};
#elif CHIP_VER >= RDP_GEN_60
/* cpu_port enum values */
bdmf_attr_enum_table_t bufmng_enum_table = {
    .values = {
        {"CPU RX     ", BUF_MNG_CPU_RX_CNTR},
        {"CPU RX HIGH", BUF_MNG_CPU_RX_HIGH_CNTR},
        {"WLAN_0     ", BUF_MNG_WLAN_0_CNTR},
        {"WLAN_0 HIGH", BUF_MNG_WLAN_HIGH_0_CNTR},
        {"WLAN_0 RX  ", BUF_MNG_WLAN_RX_0_CNTR},
        {"WLAN_1     ", BUF_MNG_WLAN_1_CNTR},
        {"WLAN_1 HIGH", BUF_MNG_WLAN_HIGH_1_CNTR},
        {"WLAN_1 RX  ", BUF_MNG_WLAN_RX_1_CNTR},
        {"WLAN_2     ", BUF_MNG_WLAN_2_CNTR},
        {"WLAN_2 HIGH", BUF_MNG_WLAN_HIGH_2_CNTR},
        {"WLAN_2 RX  ", BUF_MNG_WLAN_RX_2_CNTR},
        {"WLAN_3     ", BUF_MNG_WLAN_3_CNTR},
        {"WLAN_3 HIGH", BUF_MNG_WLAN_HIGH_3_CNTR},
        {"WLAN_3 RX  ", BUF_MNG_WLAN_RX_3_CNTR},
        {"WLAN TOTAL ", BUF_MNG_WLAN_TOTAL_CNTR},
        {"ALL PASS   ", BUF_MNG_ALL_PASS_CNTR},
        {"TOTAL      ", BUF_MNG_TOTAL_BUDGET_CNTR},
        {NULL,      0}
    }
};
#endif

bdmf_error_t drv_cnpl_policer_police(uint8_t *result, uint8_t group, uint8_t policer_num, uint16_t packet_len)
{
    uint16_t time_out = 0;
    bdmf_error_t rc;
    cnpl_sw_stat read_status;

    /* set read request to the policer */
    rc = drv_cnpl_policer_police_command_set(group, policer_num, packet_len);

    /* wait for read done */
    time_out = 0;
    while (!rc && time_out < CNPL_READ_TIMEOUT)
    {
        rc = ag_drv_cnpl_sw_stat_get(&read_status);
        if (rc || read_status.pl_plc_st == CNPL_READ_DONE)
            break;

        time_out++;
    }
    if (time_out == CNPL_READ_TIMEOUT)
        rc = BDMF_ERR_PARM;

    /* read the policer data */
    rc = rc ? rc : ag_drv_cnpl_sw_if_sw_pl_rslt_get(result);

    return rc;
}

bdmf_error_t drv_cnpl_policer_read(void *policers, uint8_t group, uint8_t num_of_policers, bdmf_boolean reset)
{
    uint16_t time_out = 0;
    bdmf_error_t rc;
    cnpl_sw_stat read_status;
    cnpl_policer_cfg policer_cfg;

    rc = ag_drv_cnpl_policer_cfg_get(group, &policer_cfg);

    /* check burst size */
    if (num_of_policers * (policer_cfg.pl_double + 1) > CNPL_READ_POLICER_BUFFER)
        rc = BDMF_ERR_PARM;

    /* set read request to the policer */
    rc = rc ? rc : drv_cnpl_policer_read_command_set(group, num_of_policers, reset);

    /* wait for read done */
    time_out = 0;
    while (!rc && time_out < CNPL_READ_TIMEOUT)
    {
        rc = ag_drv_cnpl_sw_stat_get(&read_status);
        if (rc || read_status.pl_rd_st == CNPL_READ_DONE)
            break;

        time_out++;
    }
    if (time_out == CNPL_READ_TIMEOUT)
        rc = BDMF_ERR_PARM;

    /* read the policers data */
    rc = rc ? rc : drv_cnpl_policer_read_command_get(policers, num_of_policers, !policer_cfg.pl_double);

    return rc;
}

bdmf_error_t drv_cnpl_policer_read_command_get(void *policers, uint8_t num_of_policers, bdmf_boolean single)
{
    bdmf_error_t rc = BDMF_ERR_OK;
    uint8_t size, i;
    uint32_t *policer_ptr = (uint32_t *)policers;

    /* bucket size is 4B or 8B */
    if (single)
        size = num_of_policers;
    else
        size = num_of_policers * 2;

    for (i = 0; !rc && i < size && i < CNPL_READ_POLICER_BUFFER; i++)
        rc = ag_drv_cnpl_sw_if_sw_cnt_rd_get(i, &policer_ptr[i]);

    return rc;
}

bdmf_error_t drv_cnpl_counter_clr(uint8_t group, uint32_t cntr_id)
{
    /* This function clears CNPL counter by reading it.
       Do not write to CNPL memory from Host. This does not work on 6878, 63146, and 4912
    */

    bdmf_error_t rc=0;
#ifndef RDP_SIM
    cnpl_counter_cfg counter_cfg = {};
    uint32_t temp_cntr_arr[MAX_NUM_OF_COUNTERS_PER_READ];
    int b_clear_on_read_counter;

    rc = ag_drv_cnpl_counter_cfg_get(group, &counter_cfg);
    if (rc)
    {
        return rc;
    }

    b_clear_on_read_counter = counter_cfg.clr;

    if (!b_clear_on_read_counter)
    {
        /* Temporary configure the counter as clear_on_read */
        counter_cfg.clr = 1;
        rc = ag_drv_cnpl_counter_cfg_set(group, &counter_cfg);
        if (rc)
        {
            return rc;
        }
    }

    rc = drv_cnpl_counter_read(temp_cntr_arr, group, cntr_id, 1);

    if (rc)
    {
        return rc;
    }

    if (!b_clear_on_read_counter)
    {
        /* Restore counter configuration */
        counter_cfg.clr = 0;
        rc = ag_drv_cnpl_counter_cfg_set(group, &counter_cfg);
    }
#else
    rdp_cpu_counter_clear(group, cntr_id);
#endif
    return rc;
}

bdmf_error_t drv_cnpl_counter_set(uint8_t group, uint32_t cntr_id, uint32_t value)
{
    uint8_t cntr_bytes_num;
    uint32_t cntr_mem_offset;
    bdmf_error_t rc;
    cnpl_counter_cfg counter_cfg = {};

    bdmf_fastlock_lock(&counter_read_lock);

    rc = ag_drv_cnpl_counter_cfg_get(group, &counter_cfg);
    if (rc)
    {
        bdmf_fastlock_unlock(&counter_read_lock);
        return rc;
    }

    /* check burst size */
    cntr_bytes_num = (counter_cfg.cn_double + 1) << counter_cfg.cn0_byts;

    cntr_mem_offset = (counter_cfg.ba << 3) + (cntr_id * cntr_bytes_num);

    if (counter_cfg.cn0_byts == 1)
    {
        ag_drv_cnpl_memory_data_set((cntr_mem_offset / sizeof(uint32_t)), (uint32_t)(value << 16));
    }
    else
    {
        ag_drv_cnpl_memory_data_set((cntr_mem_offset / sizeof(uint32_t)), (uint32_t)value);
    }

    bdmf_fastlock_unlock(&counter_read_lock);

    return BDMF_ERR_OK;
}

bdmf_error_t drv_cnpl_counter_read(void *counters, uint8_t group, uint16_t start_counter, uint8_t num_of_counters)
{
    bdmf_error_t rc = BDMF_ERR_OK;
    cnpl_counter_cfg counter_cfg = {};

#if !defined(RDP_SIM)
    cnpl_sw_stat read_status;
    uint16_t time_out = 0, burst;
#endif

    bdmf_fastlock_lock(&counter_read_lock);

#if !defined(RDP_SIM)
    rc = ag_drv_cnpl_counter_cfg_get(group, &counter_cfg);
    if (rc)
    {
        bdmf_fastlock_unlock(&counter_read_lock);
        return rc;
    }

    /* check burst size */
    burst = num_of_counters << (counter_cfg.cn0_byts + counter_cfg.cn_double);

    if (burst > CNPL_READ_COUNTER_BUFFER)
        rc = BDMF_ERR_PARM;

    /* set read request */
    rc = rc ? rc : drv_cnpl_counter_read_command_set(group, start_counter, num_of_counters);

    /* wait for read done */
    time_out = 0;
    while (!rc && time_out < CNPL_READ_TIMEOUT)
    {
        read_status.cn_rd_st = CNPL_READ_PROCESS;
        rc = ag_drv_cnpl_sw_stat_get(&read_status);
        if (rc || read_status.cn_rd_st == CNPL_READ_DONE)
            break;

        time_out++;
    }
    if (time_out >= CNPL_READ_TIMEOUT)
        rc = BDMF_ERR_INTERNAL;
#else
    rc = rdp_cpu_counter_read(group, start_counter, counters, num_of_counters, &counter_cfg.cn_double, &counter_cfg.cn0_byts);
#endif

    /* read the HW registers data */
    rc = rc ? rc : drv_cnpl_counter_read_command_get(counters, num_of_counters, counter_cfg.cn_double, counter_cfg.cn0_byts, (start_counter % 2));

    bdmf_fastlock_unlock(&counter_read_lock);
    return rc;
}

bdmf_error_t drv_cnpl_counter_read_command_set(uint8_t group, uint16_t start_counter, uint8_t num_of_counters)
{
    uint32_t cmd;

    if (group >= CNPL_MAX_COUNTER_GROUPS ||
        start_counter >= CNPL_MAX_COUNTER_INDEX)
    {
        return BDMF_ERR_PARM;
    }

    cmd = (CNPL_COUNTER_READ_COMMAND << CNPL_COMMAND_OFFSET) |
          (group << CNPL_COUNTER_READ_COMMAND_GROUP_OFFSET) |
          (start_counter << CNPL_COUNTER_READ_COMMAND_START_OFFSET) |
          (num_of_counters << CNPL_COUNTER_READ_COMMAND_SIZE_OFFSET);

    return ag_drv_cnpl_sw_if_sw_cmd_set(cmd);
}

bdmf_error_t drv_cnpl_counter_read_command_get(uint32_t *counters, uint8_t num_of_counters, bdmf_boolean cn_double, uint8_t cn0_byts, bdmf_boolean is_start_counter_odd)
{
    bdmf_error_t rc = BDMF_ERR_OK;
    uint8_t i, j = 0;
    uint32_t counter_buff[CNPL_READ_COUNTER_BUFFER] = {};
    uint32_t buff_size_word = 0, buff_size_bytes;

    num_of_counters *= (cn_double + 1);
    buff_size_bytes = num_of_counters << cn0_byts;

    buff_size_word = (buff_size_bytes + 3/*roundup*/) / 4; 

#if !defined(RDP_SIM)
    for (i = 0; !rc && i < buff_size_word && i < CNPL_READ_COUNTER_BUFFER; i++)
        rc = ag_drv_cnpl_sw_if_sw_cnt_rd_get(i, &counter_buff[i]);

    if (rc)
        return rc;
#else
    for (i = 0; i < buff_size_word && i < CNPL_READ_COUNTER_BUFFER; i++)
    {
        counter_buff[i] = __swap4bytes(counters[i]);
    }
#endif

    switch (cn0_byts)
    {
    case 0:
        for (i=0; i < buff_size_word; i++)
        {
            counters[j++] = (counter_buff[i] >> 24) & 0xFF;
            counters[j++] = (counter_buff[i] >> 16) & 0xFF;
            counters[j++] = (counter_buff[i] >> 8) & 0xFF;
            counters[j++] = counter_buff[i] & 0xFF;
        }
        break;
    case 1:
        for (i=0; i < buff_size_word; i++)
        {
            if (is_start_counter_odd && num_of_counters == 1)/* adjust for odd counter due to memory alignment */
            {                                                /* no support for burst mode (num_of_counters > 1)*/
                 counters[j++] = counter_buff[i] & 0xFFFF;
                 counters[j++] = 0;
            }
            else
            {
                counters[j++] = counter_buff[i] >> 16;
                counters[j++] = counter_buff[i] & 0xFFFF;
            }
        }
        break;
    case 2:
    case 3:
        for (i=0; i < buff_size_word; i++)
            counters[i] = counter_buff[i];
        break;

    default:
        return BDMF_ERR_INTERNAL;
    }
    return rc;
}

bdmf_error_t drv_cnpl_policer_police_command_set(uint8_t group, uint8_t policer_num, uint16_t packet_len)
{
    uint32_t cmd;

    if (group >= CNPL_MAX_POLICER_GROUPS)
    {
        return BDMF_ERR_PARM;
    }

    cmd = (CNPL_POLICER_POLICE_COMMAND << CNPL_COMMAND_OFFSET) |
          (group << CNPL_POLICER_COMMAND_GROUP_OFFSET) |
          (policer_num << CNPL_POLICER_COMMAND_START_OFFSET) |
          (packet_len << CNPL_POLICER_COMMAND_PACKET_LEN_OFFSET);

    return ag_drv_cnpl_sw_if_sw_cmd_set(cmd);
}

bdmf_error_t drv_cnpl_policer_read_command_set(uint8_t group, uint8_t policer_num, bdmf_boolean reset_after_read)
{
    uint32_t cmd;

    if (group >= CNPL_MAX_POLICER_GROUPS)
    {
        return BDMF_ERR_PARM;
    }

    cmd = (CNPL_POLICER_READ_COMMAND << CNPL_COMMAND_OFFSET) |
          (group << CNPL_POLICER_COMMAND_GROUP_OFFSET) |
          (policer_num << CNPL_POLICER_COMMAND_START_OFFSET) |
          (reset_after_read << CNPL_POLICER_COMMAND_CLEAR_OFFSET);

    return ag_drv_cnpl_sw_if_sw_cmd_set(cmd);
}

uint8_t drv_cnpl_periodic_update_us_to_n_get(uint32_t microseconds)
{
#if defined (POLICER_SHIFT_N_WA)
    int _ns = microseconds;
#else
    int _ns = microseconds * 1000;
#endif
#if CHIP_VER < RDP_GEN_62
    if (_ns > CNPL_PERIODIC_UPDATE_QUANTA_NS)
        return (_ns + CNPL_PERIODIC_UPDATE_HALF_QUANTA_NS) / CNPL_PERIODIC_UPDATE_QUANTA_NS;
    return CNPL_PERIODIC_UPDATE_MINIMUM;
#else
    int n;
    for (n = 0; n <= BITS_4_VAL; n++)
    {
        if ((CNPL_PERIODIC_UPDATE_QUANTA_NS << n) >= _ns)
            break;
    }
#if defined (POLICER_SHIFT_N_WA)
    n = 0;
#endif
    return n;
#endif
}

bdmf_error_t drv_cnpl_memory_data_init()
{
#if ((CHIP_VER >= RDP_GEN_62) && (!defined(RDP_SIM)))
    bdmf_boolean init0 = 1, init1 = 1;     
    ag_drv_cnpl_misc_mem_init_set(init0, init1);
    do
    {
        ag_drv_cnpl_misc_mem_init_get(&init0, &init1);
    } while ((init0) == 1 || (init1 == 1));
#else
    MEMSET((uint32_t *)DEVICE_ADDRESS(RU_BLK(CNPL).addr[0] + RU_REG_OFFSET(CNPL, MEMORY_DATA)), 0, (RU_REG_RAM_CNT(CNPL, MEMORY_DATA) + 1) * sizeof(uint32_t));
#endif
    return BDMF_ERR_OK;
}

#if CHIP_VER >= RDP_GEN_60
int drv_cnpl_cli_bufmng_sanity_get(bdmf_session_handle session, bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int rc = BDMF_ERR_OK;
    uint8_t idx1, idx2, idx3;
    uint32_t val1, val2, val3;

#if CHIP_VER >= RDP_GEN_62
    uint32_t val = 0;
    rc = ag_drv_cnpl_buf_mng_counters_cfg_stat_cntr_neg_st_get(&val);
#else
    uint16_t val = 0;
    rc = ag_drv_cnpl_buf_mng_counters_cfg_stat_cntr_neg_st_get(&val);
#endif

    if (val && !rc)
    {
        rc = ag_drv_cnpl_cnpl_stat_neg_cap_cnt_get(0, &idx1, &val1);
        rc = rc ? rc : ag_drv_cnpl_cnpl_stat_neg_cap_cnt_get(1, &idx2, &val2);
        rc = rc ? rc : ag_drv_cnpl_cnpl_stat_neg_cap_cnt_get(2, &idx3, &val3);

        bdmf_session_print(session, "\nBufmng negative error: %d (cnt_%d: %d, cnt_%d :%d, cnt_%d :%d)\n",
                        val, idx1, val1, idx2, val2, idx3, val3);
    }

    return rc;
}
#endif

#ifdef USE_BDMF_SHELL

int drv_cnpl_cli_config_get(bdmf_session_handle session, bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    static uint32_t cntr_cfg[] = {cli_cnpl_counter_cfg};
    static uint32_t policer_cfg[] = {cli_cnpl_policer_cfg};
    static uint32_t policer_calc_type[] = {cli_cnpl_policers_configurations_pl_calc_type};
    static uint32_t misc_cfg[] = {cli_cnpl_policers_configurations_per_up, cli_cnpl_misc_arb_prm};

    /* get counters configurations */
    bdmf_session_print(session, "\nCounters configurations:\n");
    HAL_CLI_PRINT_NUM_OF_LIST(session, cnpl, cntr_cfg, CNPL_MAX_COUNTER_GROUPS);

    /* get policer configurations */
    bdmf_session_print(session, "\nPolicers configurations:\n");
    HAL_CLI_PRINT_NUM_OF_LIST(session, cnpl, policer_cfg, CNPL_MAX_POLICER_GROUPS);
    HAL_CLI_PRINT_NUM_OF_LIST(session, cnpl, policer_calc_type, (CNPL_MAX_POLICER_INDEX / 32));

    /* get misc condigurations */
    bdmf_session_print(session, "\nMisc configurations:\n");
    HAL_CLI_PRINT_LIST(session, cnpl, misc_cfg);

    return 0;
}

#if CHIP_VER >= RDP_GEN_60

int drv_cnpl_cli_bufmng_config_get(bdmf_session_handle session, bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    uint8_t lvl, nxtlvl, ctr, ctr_idx;
    uint32_t max_thr, rsrv_thr, hipri_thr, ctr_val;

    /* print order configurations */
    bdmf_session_print(session, "\nBufmng order:\n");
    for (ctr_idx = 0; ctr_idx < CNPL_MAX_BUFMNG_INDEX; ctr_idx++)
    {
        bdmf_session_print(session, "[%2d] %-11s->", ctr_idx, bdmf_attr_get_enum_text_hlp(&bufmng_enum_table, ctr_idx));
        ctr = ctr_idx;

        for (lvl = 0; lvl < 3; lvl++)
        {
            ag_drv_cnpl_cnpl_stat_order_get(ctr, &nxtlvl);
            if (nxtlvl == BUFMNG_NULL_BUFF)
            {
                bdmf_session_print(session, " NULL\n");
                break;
            }
            else
            {
                bdmf_session_print(session, " (%2d) %-11s ->", nxtlvl, bdmf_attr_get_enum_text_hlp(&bufmng_enum_table, nxtlvl));
                ctr = nxtlvl;
            }
        }

        if (nxtlvl != BUFMNG_NULL_BUFF)
        {
            bdmf_session_print(session, "Wrong configuration - up to 3 levels available\n");
        }
    }

    /* print order configurations */
    bdmf_session_print(session, "\nCounter  Name         Max_thr   Reserved_thr  High_prio_thr  Value\n");
    for (ctr_idx = 0; ctr_idx < CNPL_MAX_BUFMNG_INDEX; ctr_idx++)
    {
        ag_drv_cnpl_cnpl_stat_max_thr_get(ctr_idx, &max_thr);
        ag_drv_cnpl_cnpl_stat_rsrv_thr_get(ctr_idx, &rsrv_thr);
        ag_drv_cnpl_cnpl_stat_hipri_thr_get(ctr_idx, &hipri_thr);
        ag_drv_cnpl_cnpl_stat_ctrs_val_get(ctr_idx, &ctr_val);

        bdmf_session_print(session, "[%2d]     %-11s  %6d    %6d        %6d       %6d\n", ctr_idx, bdmf_attr_get_enum_text_hlp(&bufmng_enum_table, ctr_idx), max_thr, rsrv_thr, hipri_thr, ctr_val);
    }
    return 0;
}
#endif

static int drv_cnpl_cli_group_counter_get(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    uint8_t parameter_index = 0;
    uint32_t cntr_arr[4];
    bdmf_error_t rc;
    cnpl_counter_cfg counter_cfg = {};

    uint8_t cnpl_group_number = parm[parameter_index++].value.unumber; /* input */
    uint8_t cnpl_counter_number = parm[parameter_index++].value.unumber; /* input */

    rc = ag_drv_cnpl_counter_cfg_get(cnpl_group_number, &counter_cfg);
    if (rc)
        return rc;

    rc = drv_cnpl_counter_read(cntr_arr, cnpl_group_number, cnpl_counter_number, 1);
    if (!rc)
    {
        if (counter_cfg.cn0_byts == 1)
            bdmf_session_print(session, "counter 1: %d\n\r", cntr_arr[cnpl_counter_number%2]);
        else
            bdmf_session_print(session, "counter 1: %d\n\r", cntr_arr[0]);
        if (counter_cfg.cn_double)
            bdmf_session_print(session, "counter 2: %d\n\r", cntr_arr[1]);
    }
    else
        bdmf_session_print(session, "Can't read counter. err : %d\n\r", rc);

    return 0;
}

static bdmfmon_handle_t cnpl_dir;

#if defined(CONFIG_CPU_RX_FROM_XPM)
extern void drv_fpm_print_non_empty_token_info(bdmf_session_handle session);
extern void drv_fpm_print_packet_from_token_idx(bdmf_session_handle session, uint32_t token_idx, uint32_t offset);
static int rdp_drv_shell_print_xpm_db(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    int rc = BDMF_ERR_OK;

    drv_fpm_print_non_empty_token_info(session);

    return rc;
}

static int rdp_drv_shell_print_xpm_data(bdmf_session_handle session, const bdmfmon_cmd_parm_t parm[], uint16_t n_parms)
{
    drv_fpm_print_packet_from_token_idx(session, parm[0].value.unumber, parm[1].value.unumber);
    return 0;
}
#endif

void drv_cnpl_cli_init(bdmfmon_handle_t driver_dir)
{
    cnpl_dir = ag_drv_cnpl_cli_init(driver_dir);

    BDMFMON_MAKE_CMD_NOPARM(cnpl_dir, "cfg_get", "cnpl configuration", (bdmfmon_cmd_cb_t)drv_cnpl_cli_config_get);
#if CHIP_VER >= RDP_GEN_60
    BDMFMON_MAKE_CMD_NOPARM(cnpl_dir, "bufmng_cfg_get", "BUFMNG configuration", (bdmfmon_cmd_cb_t)drv_cnpl_cli_bufmng_config_get);
#endif
    BDMFMON_MAKE_CMD(cnpl_dir, "rgc", "read group counter", drv_cnpl_cli_group_counter_get,
        BDMFMON_MAKE_PARM("group", "group number", BDMFMON_PARM_NUMBER, 0),
        BDMFMON_MAKE_PARM("counter", "counter offset", BDMFMON_PARM_NUMBER, 0));
#if defined(CONFIG_CPU_RX_FROM_XPM)
    BDMFMON_MAKE_CMD_NOPARM(cnpl_dir, "xpm_db",   "print non empty xpms from database", rdp_drv_shell_print_xpm_db);
    BDMFMON_MAKE_CMD(cnpl_dir, "xpm_print", "print packet data from XPM", rdp_drv_shell_print_xpm_data,
    BDMFMON_MAKE_PARM_RANGE("token_idx" ,"xpm token idx", BDMFMON_PARM_UNUMBER, 0, 0, 500000),
    BDMFMON_MAKE_PARM_RANGE("data_offset" ,"data offset in the XPM (320 recommended)", BDMFMON_PARM_UNUMBER, 0, 0, 1000));
#endif
}

void drv_cnpl_cli_exit(bdmfmon_handle_t driver_dir)
{
    if (cnpl_dir)
    {
        bdmfmon_token_destroy(cnpl_dir);
        cnpl_dir = NULL;
    }
}

#endif /* USE_BDMF_SHELL */

/* Data path init helpers */
int drv_cnpl_data_path_init(bdmf_boolean is_gateway, bdmf_boolean vlan_stats_enable, bdmf_boolean en_color_aware, 
    bdmf_boolean reset_arb_prm)
{
    int rc;

    /* reset counters-policers memory */
    rc = drv_cnpl_memory_data_init();

#if CHIP_VER >= RDP_GEN_62
    if (reset_arb_prm)
    {
        uint8_t sw_prio, mem_addr_bit_sel;

        rc = rc ? rc : ag_drv_cnpl_misc_arb_prm_get(&sw_prio, &mem_addr_bit_sel);
        rc = rc ? rc : ag_drv_cnpl_misc_arb_prm_set(sw_prio, 0);
    }
#endif

#if CHIP_VER >= RDP_GEN_43
    rc = rc ? rc : ag_drv_cnpl_misc_col_awr_en_set(en_color_aware);
#endif

    /* init counter groups according to project */
    rc = rc ? rc : drv_cntr_group_init(is_gateway, vlan_stats_enable);
    return rc;
}


