/*
 * <:copyright-BRCM:2019:DUAL/GPL:standard
 * 
 *    Copyright (c) 2019 Broadcom 
 *    All Rights Reserved
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation (the "GPL").
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * 
 * A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
 * writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * :>
 */

/*******************************************************************
 * bdmf_buf.h
 *
 * BDMF - user-space sk_buff emulation
 *
 * This file is derived from linux skbuff.h
 *
 * This file is free software: you can redistribute and/or modify it
 * under the terms of the GNU Public License, Version 2, as published
 * by the Free Software Foundation, unless a different license
 * applies as provided above.
 *
 * This program is distributed in the hope that it will be useful,
 * but AS-IS and WITHOUT ANY WARRANTY; without even the implied
 * warranties of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * TITLE or NONINFRINGEMENT. Redistribution, except as permitted by
 * the GNU Public License.
 *
 * You should have received a copy of the GNU Public License,
 * Version 2 along with this file; if not, see
 * <http://www.gnu.org/licenses>.
 *******************************************************************/

#ifndef _BL_BUF_H_
#define _BL_BUF_H_

#define SKB_ALLOC_LEN    2048
#define SKB_ALLOC_ALIGN  256
#define SKB_RESERVE      64
#define SKB_MAGIC        ((' '<<24) | ('s'<<16) | ('k'<<8) | ('b'))


#define BCM_SKB_TAILROOM (32)
#define BCM_SKB_SHAREDINFO  256
#define PFKBUFF_PHEAD_OFFSET 64
#define BCM_PKT_HEADROOM 256
#define BCM_PKT_HEADROOM_RSV 320
#define PFKBUFF_TO_PDATA(pFkb,headroom)     \
            (uint8_t *)((uint8_t*)(pFkb) + PFKBUFF_PHEAD_OFFSET + (headroom))
#define PDATA_TO_PFKBUFF(pData,headroom)    \
            (uint8_t *)((uint8_t*)(pData)-(headroom)-PFKBUFF_PHEAD_OFFSET)
#define BCM_PKT_TAILROOM	(BCM_SKB_TAILROOM + BCM_SKB_SHAREDINFO)
#define BCM_DCACHE_ALIGN(len)  ((len + 31) & ~31)

typedef union wlFlowInf {
	uint32_t u32;
	union {
		union {
			struct {
				/* Start - Shared fields between ucast and mcast */
				uint32_t is_ucast:1;
				/* wl_prio is 4 bits for nic and 3 bits for dhd. Plan is
				 * to make NIC as 3 bits after more analysis */
				uint32_t wl_prio:4;
				/* End - Shared fields between ucast and mcast */
				uint32_t nic_reserved1:11;
				uint32_t wl_chainidx:16;
			};
			struct {
				uint32_t overlayed_field:16;
				uint32_t ssid_dst:16; /* For bridged traffic we don't have chainidx (0xFE) */
			};
		} nic;

		struct {
			/* Start - Shared fields between ucast and mcast */
			uint32_t is_ucast:1;
			uint32_t wl_prio:4;
			/* End - Shared fields between ucast and mcast */
			/* Start - Shared fields between dhd ucast and dhd mcast */
			uint32_t flowring_idx:10;
			/* End - Shared fields between dhd ucast and dhd mcast */
			uint32_t dhd_reserved:13;
			uint32_t ssid:4;
		} dhd;
	} ucast;
	struct {
		/* Start - Shared fields between ucast and mcast */
		/* for multicast, WFD does not need to populate this flowring_idx, it is used internally by dhd driver */
		uint32_t is_ucast:1;
		uint32_t wl_prio:4;
		/* End - Shared fields between ucast and mcast */
		/* Start - Shared fields between dhd ucast and dhd mcast */
		uint32_t flowring_idx:10;
		/* End - Shared fields between dhd ucast and dhd mcast */
		uint32_t mcast_reserved:1;
		uint32_t ssid_vector:16;
	} mcast;

	struct {
		/* Start - Shared fields b/w ucast, mcast & pktfwd */
		uint32_t is_ucast           : 1;    /* Start - Shared fields b/w ucast, mcast */
		uint32_t wl_prio            : 4;    /* packet priority */
		/* End - Shared fields between ucast, mcast & pktfwd */
		uint32_t pktfwd_reserved    : 7;
		uint32_t ssid               : 4;
		uint32_t pktfwd_key         : 16;   /* pktfwd_key_t : 2b domain, 2b incarn, 12b index */
	} pktfwd;

	struct {
		/* Start - Shared fields b/w ucast, mcast, pktfwd & awl */
		uint32_t is_ucast           : 1;    /* is unicast packet */
		uint32_t wl_prio            : 4;    /* packet priority */
		/* End - Shared fields between ucast, mcast, pktfwd & awl */
		uint32_t awl_reserved       : 21;   /* Archer WLAN Reserved */
		uint32_t radio              : 2;    /* Radio Index */
		uint32_t ifidx              : 4;    /* Interface Index */
	} awl;
} wlFlowInf_t;

struct sk_buff {
        struct sk_buff *next;
        uint32_t magic;
        uint32_t len;
        uint8_t *data;
        uint8_t *tail;
        uint8_t *end;
        uint8_t cb[16];  /* control buffer - for driver use */
        int priority;
        wlFlowInf_t	wl;
};

struct sk_buff_head {
        struct sk_buff *head;
        struct sk_buff *tail;
};

void dev_kfree_skb(struct sk_buff *skb);

#define dev_kfree_skb_irq dev_kfree_skb

/* returns skb pointer data belongs to or NULL if data doesn't belong to skb */
struct sk_buff *data_to_skb(void *data);

static inline void skb_queue_head_init(struct sk_buff_head *q)
{
        q->head = q->tail = NULL;
}

static inline int skb_queue_empty(const struct sk_buff_head *q)
{
    return (q->head == NULL);
}

static inline void __skb_queue_tail(struct sk_buff_head *q, struct sk_buff *skb)
{
        skb->next = NULL;
        if (q->head)
                q->tail->next = skb;
        else
                q->head = skb;
        q->tail = skb;
}

static inline struct sk_buff *__skb_dequeue(struct sk_buff_head *q)
{
        struct sk_buff *skb = q->head;
        if (!skb)
                return NULL;
        q->head = skb->next;
        return skb;
}


struct sk_buff *dev_alloc_skb(uint32_t length);

static inline void skb_set_tail_pointer(struct sk_buff *skb, const int offset)
{
       skb->tail = skb->data + offset;
}

static inline uint8_t *skb_put(struct sk_buff *skb, uint32_t size)
{
        uint8_t *data=skb->tail;
        assert(data+size <= skb->end);
        skb->tail += size;
        skb->len += size;
        return data;
}

/*
 * Buffer data pointer accessor
 */
static inline  uint8_t *skb_data(struct sk_buff *skb)
{
        return skb->data;
}


/*
 * Buffer length accessor
 */
static inline  uint32_t skb_length(struct sk_buff *skb)
{
        return skb->len;
}


static inline void skb_trim(struct sk_buff *skb, uint32_t size)
{
        assert(skb->len >= size);
        skb->len = size;
}


/**
 *  skb_reserve - adjust headroom
 *  @skb: buffer to alter
 *  @len: bytes to move
 *
 *  Increase the headroom of an empty &sk_buff by reducing the tail
 *  room. This is only allowed for an empty buffer.
 */
static inline void skb_reserve(struct sk_buff *skb, unsigned int len)
{
    skb->data += len;
    skb->tail += len;
}

/**
*    skb_push - add data to the start of a buffer
*    @skb: buffer to use
*    @len: amount of data to add
*
*    This function extends the used data area of the buffer at the buffer
*    start. If this would exceed the total buffer headroom the kernel will
*    panic. A pointer to the first byte of the extra data is returned.
*/
static inline unsigned char *skb_push(struct sk_buff *skb, unsigned int len)
{
   skb->data -= len;
   skb->len  += len;
   assert(skb->data>=(uint8_t *)skb + sizeof(struct sk_buff));
   return skb->data;
}


/**
*    dev_kfree_skb_data - release skb given the data pointer
*    the data pointer should not be farer than 256 bytes from
*    the beginning of skb buffer it is part of
*    @data: buffer pointer. originally skb_data(skb)
*           where skb is allocated by dev_alloc_skb
*           possibly modified later, but not more that 64 bytes in either direction
*/
void dev_kfree_skb_data(uint8_t *data);



/**
*    skb_make - fill in skb header given data pointer and length
*    the data pointer should not be farer than 256 bytes from
*    the beginning of skb buffer it is part of
*    @data: buffer pointer. originally skb_data(skb)
*           where skb is allocated by dev_alloc_skb
*           possibly modified later, but not more that 64 bytes in either direction
*/
struct sk_buff *skb_make(uint8_t *data, uint32_t len);



/**
*    skb_stat - get skb pool statistics
*    @alloc_count: number of outstanding skb allocations
*    @free_count: number of buffers on skb free list
*/
void skb_stat(uint32_t *alloc_count, uint32_t *free_count);

/* returns the length of the ethernet + ipv4 + transport headers */
int skb_transport_offset(const struct sk_buff *skb);

#endif /* _BL_BUF_H_ */
