/*
 * <:copyright-BRCM:2019:DUAL/GPL:standard
 * 
 *    Copyright (c) 2019 Broadcom 
 *    All Rights Reserved
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation (the "GPL").
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * 
 * A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
 * writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * :>
 */


#include <linux/version.h>
#include "bdmf_system.h"
#include "bdmf_system_common.h"
#include "bcm_intr.h"
#if defined(CONFIG_BCM963138) || defined(CONFIG_BCM963148)
#include "board.h"
#include "bcm_mm.h"
#endif
#if defined(CONFIG_BCM_PKTRUNNER_GSO) || defined(CONFIG_RUNNER_CPU_TX_FRAG_GATHER)
#include "access_macros.h"
#include "linux/highmem.h"
#include "bcm_mm.h"
#include <rdpa_cpu_basic.h>
#endif
#include "bpm.h"

#ifdef XRDP
void rdp_recycle_buf_dummy(void *pFkb);

void (*sysb_recycle_function_cb)(void *pFkb) = rdp_recycle_buf_dummy;
EXPORT_SYMBOL(sysb_recycle_function_cb);
#endif

/* sanity compilation flags checks */
#if defined(XRDP)
#if defined(CONFIG_CPU_RX_FROM_XPM) && !defined(CC_NBUFF_FLUSH_OPTIMIZATION)
#error "CC_NBUFF_FLUSH_OPTIMIZATION shouldbe be defined too with CONFIG_CPU_RX_FROM_XPM"
#endif
#if defined(CONFIG_CPU_RX_FROM_XPM) && !defined(CONFIG_BCM_BPM_SW_BACKUP) && !defined(CONFIG_BRCM_QEMU)
#error "CONFIG_CPU_RX_FROM_XPM must be defined together with CONFIG_BCM_BPM_SW_BACKUP"
#endif
#endif
void rdp_recycle_buf_dummy(void *pFkb)
{
    return;
}

#if defined(CONFIG_RUNNER_CPU_TX_FRAG_GATHER)
extern int bdmf_sg_csum(struct sk_buff *skb, runner_sg_desc_t *sg_desc_p);
extern int bdmf_sg_gso(struct sk_buff *skb, rnr_cpu_tx_func xmit_fn, pbuf_t *pbuf, const rdpa_cpu_tx_info_t *info);
#endif

/*
 * System buffer abstraction
 */

/** Recycle the data buffer
 * \param[in]   data   Data buffer
 * \param[in]   context -unused,for future use
 */
void bdmf_sysb_databuf_recycle(void *datap)
{
    __bdmf_sysb_databuf_recycle(datap);
}

static inline void *data_endp_calc(struct sk_buff *skb, void *data_startp)
{
    void *data_endp;
#if defined(CC_NBUFF_FLUSH_OPTIMIZATION)
    void *dirty_p = skb_shinfo(skb)->dirty_p;
    void *shinfoBegin = (void *)skb_shinfo(skb);

    /* BPM skbs benefit from below. Linux skbs continue to relocate
     * via skb_headerinit(). Caller would need to inform skb_headerinit
     * of what is the true end of the data buffer. skb_headerinit
     * may not assume bcm_pkt_lengths.h definitions.
     */
    if ((uintptr_t)shinfoBegin < ((uintptr_t)data_startp +
                (BPM_BUF_TO_END_OFFSET)))
    {
        void *shinfoEnd;

        /* skb_shared_info is located in a DMA-able region.
         * nr_frags could be ++/-- during sk_buff life.
         */
        shinfoEnd = shinfoBegin + sizeof(struct skb_shared_info);
        cache_invalidate_region(shinfoBegin, shinfoEnd);
    }

    if (dirty_p) 
    {
        if ((dirty_p < (void *)skb->head) || (dirty_p > shinfoBegin)) 
        {
            printk("invalid dirty_p detected: %px valid=[%px %px]\n",
                    dirty_p, skb->head, shinfoBegin);
            data_endp = shinfoBegin;
        } 
        else 
            data_endp = (dirty_p < data_startp) ? data_startp : dirty_p;
    } 
    else 
        data_endp = shinfoBegin;
#else
     data_endp = (void *)(skb_shinfo(skb)) + sizeof(struct skb_shared_info);
#endif
     return data_endp;
}

/** Recycle the system buffer and associated data buffer
 * \param[in]   data   Data buffer
 * \param[in]   context - unused,for future use
 * \param[in]   flags   - indicates what to recyle
 */
void bdmf_sysb_recycle(bdmf_sysb sysb, unsigned long context, uint32_t flags)
{
    struct sk_buff *skb;
    void *data_startp; 

    if ( IS_FKBUFF_PTR(sysb) )
    {
        /* Transmit driver is expected to perform cache invalidations */

        __bdmf_sysb_databuf_recycle(PNBUFF_2_FKBUFF(sysb));
        return;
    }

    /* skb may be a BPM sk_buff, kernel sk_buff or skb located inside of the data buffer (SKB_INPLACE) */
    skb = PNBUFF_2_SKBUFF(sysb);

     if (!(flags & SKB_DATA_RECYCLE))
     {
        /* Warn because only DATA recycle is supported if we are here */
        WARN_ON(1);
        return;
     }

     /* Transmit driver is not required to perform cache invalidations. */

     /* DMA-able start region need not be cacheline aligned.
      * A data buffer may only be recycled by the last referring sk_buff.
      */
     data_startp = (void *)((uint8_t *)skb->head + BCM_PKT_HEADROOM);
     cache_invalidate_region(data_startp, data_endp_calc(skb, data_startp));

     __bdmf_sysb_databuf_recycle(PDATA_TO_PFKBUFF(skb->head,0));
}

void bdmf_sysb_recycle_skb(bdmf_sysb sysb, unsigned long context, uint32_t flags)
{
    if (flags & SKB_DATA_RECYCLE) {
        bdmf_sysb_recycle(sysb, context, flags); /* BPM | FEED | kmem */
        return;
    }
#if (defined(CONFIG_BCM_BPM) || defined(CONFIG_BCM_BPM_MODULE))
    if (flags & SKB_RECYCLE) {
        gbpm_recycle_skb(sysb, context, flags); /* BPM */
    }
#endif
}

#if defined(CONFIG_BCM_PKTRUNNER_GSO) || defined(CONFIG_RUNNER_CPU_TX_FRAG_GATHER)

/* bdmf_kmap_skb_frag bdmf_kunmap_skb_frag  functions are copied
 * from linux. Here we are not using local_bh_disable/local_bh_enable
 * when HIGHMEM is enabled. This is because currently these functions
 * are called with irq's disabled (f_rdd_lock_irq), kernel will generate
 * a warning if local_bh_enable is called when irq's are disabled
 */

/*TODO: optimize bdmf_kmap_skb_frag bdmf_kunmap_skb_frag */

/** map a page to virtual address
 * \param[in]   skb_frag_t
 * return : virtual address
 */
void *bdmf_kmap_skb_frag(const skb_frag_t *frag)
{
#ifdef CONFIG_HIGHMEM
    BUG_ON(in_irq());

    //local_bh_disable();
#endif
    return kmap_atomic(skb_frag_page(frag));
}

/** Free virtual address mapping
 * \param[in]   vaddr virtual address
 */
void bdmf_kunmap_skb_frag(void *vaddr)
{
    kunmap_atomic(vaddr);
#ifdef CONFIG_HIGHMEM
    //local_bh_enable();
#endif
}
#endif /* BCM_PKTRUNNER_GSO || RUNNER_FRAG_GATHER */


/*
 * CONFIG_BCM_PKTRUNNER_GSO enables Runner Gen3 gso feature,
 * and supports 63138/148, 4908. 
 */
#if defined(CONFIG_BCM_PKTRUNNER_GSO)

static DEFINE_SPINLOCK(bdmf_runner_gso_desc_pool_lock);

#define BDMF_RUNNER_GSO_DESC_POOL_LOCK(flags)    spin_lock_irqsave(&bdmf_runner_gso_desc_pool_lock, flags)
#define BDMF_RUNNER_GSO_DESC_POOL_UNLOCK(flags)  spin_unlock_irqrestore(&bdmf_runner_gso_desc_pool_lock, flags)

runner_gso_desc_t *runner_gso_desc_pool=NULL;

//#define CONFIG_BCM_PKTRUNNER_GSO_DEBUG 1

/** Recycle a GSO Descriptor
 * \param[in]   gso_desc_p   pointer to GSO Descriptor
 */
void bdmf_runner_gso_desc_free(runner_gso_desc_t *gso_desc_p)
{
    if(gso_desc_p)
    {
        unsigned long flags;

        BDMF_RUNNER_GSO_DESC_POOL_LOCK(flags);
        gso_desc_p->isAllocated = 0;
        cache_flush_len(gso_desc_p, sizeof(runner_gso_desc_t));
        BDMF_RUNNER_GSO_DESC_POOL_UNLOCK(flags);
    }
}

/** Allocate a GSO Descriptor
 * \returns pointer to allocated GSO Descriptor
 */
static inline runner_gso_desc_t *bdmf_runner_gso_desc_alloc(void)
{
    static int alloc_index = 0;
    int i=0;
    unsigned long flags;
    runner_gso_desc_t *gso_desc_p;

    BDMF_RUNNER_GSO_DESC_POOL_LOCK(flags);

    while(i < RUNNER_MAX_GSO_DESC)
    {
        if(alloc_index == RUNNER_MAX_GSO_DESC)
            alloc_index = 0;

        gso_desc_p = &runner_gso_desc_pool[alloc_index];
        cache_invalidate_len(gso_desc_p, sizeof(runner_gso_desc_t));
        if(!gso_desc_p->isAllocated)
        {
    #if 1
            memset(gso_desc_p, 0, sizeof(runner_gso_desc_t));
    #else
            gso_desc_p->nr_frags = 0;
            gso_desc_p->flags = 0;
            gso_desc_p->mss = 0;
    #endif

            gso_desc_p->isAllocated = 1;
            alloc_index++;
            BDMF_RUNNER_GSO_DESC_POOL_UNLOCK(flags);
            return gso_desc_p;
        }
        alloc_index++;
        i++;
    }
    BDMF_RUNNER_GSO_DESC_POOL_UNLOCK(flags);
    return NULL;
}

#if defined(CONFIG_BCM_PKTRUNNER_GSO_DEBUG)
static int bdmf_gso_desc_dump_enabled=0;

/**  Print information in GSO Descriptor
 * \param[in]   gso_desc_p   pointer to GSO Descriptor
 */
void bdmf_gso_desc_dump(runner_gso_desc_t *gso_desc_p)
{
    int i;

    if (bdmf_gso_desc_dump_enabled)
    {
        printk("******* Dumping gso_desc_p =%px  runner phys-addr =%x*********\n", gso_desc_p,
            swap4bytes(VIRT_TO_PHYS(gso_desc_p)));

        printk("gso_desc_p ->len:        host(%d ) runner(%d) \n", swap2bytes(gso_desc_p->len), gso_desc_p->len);
        printk("gso_desc_p ->linear_len: host(%d ) runner(%d) \n", swap2bytes(gso_desc_p->linear_len),
            gso_desc_p->linear_len);
        printk("gso_desc_p ->mss:        host(%d ) runner(%d)\n", swap2bytes(gso_desc_p->mss), gso_desc_p->mss);
        printk("gso_desc_p ->nr_frags:   %d \n", gso_desc_p->nr_frags);

        printk("gso_desc_p ->data:       host(%x ) runner(%x)\n", swap4bytes(gso_desc_p->data), gso_desc_p->data);
        for (i=0; i < gso_desc_p ->nr_frags; i++)
        {
            printk("Hostfrag_data[%d]=%x, frag_len=%d \n", i, swap4bytes(gso_desc_p->frag_data[i]),
                swap2bytes(gso_desc_p->frag_len[i]));
            printk("frag_data[%d]=%x, frag_len=%d \n", i, gso_desc_p->frag_data[i], gso_desc_p->frag_len[i]);
        }
        cache_invalidate_len(gso_desc_p, sizeof(runner_gso_desc_t));
    }
}
#endif

/** Create a pool of GSO Descriptors
 * \param[in]   num_desc  number of descriptors to be created
 * \returns 0=Success -1=failure
 */
int bdmf_gso_desc_pool_create( uint32_t num_desc)
{
    uint32_t mem_size = num_desc * sizeof(runner_gso_desc_t);

    runner_gso_desc_pool = CACHED_MALLOC_ATOMIC(mem_size);
    if(!runner_gso_desc_pool)
    {
        printk(KERN_ERR "###### ERROR:Failed to allocate runner_gso_desc_pool\n");
        return -1;
    }
    memset(runner_gso_desc_pool, 0, mem_size);
    cache_flush_len(runner_gso_desc_pool,  mem_size);

    printk("++++Runner gso_desc_pool created successfully\n");
#if defined(CONFIG_BCM_PKTRUNNER_GSO_DEBUG)
    printk("&bdmf_gso_desc_dump_enabled=%px bdmf_gso_desc_dump_enabled=%d\n",
            &bdmf_gso_desc_dump_enabled, bdmf_gso_desc_dump_enabled);
#endif

    return 0;
}
EXPORT_SYMBOL(bdmf_gso_desc_pool_create);

/** Free a pool of GSO Descriptors
 */
void bdmf_gso_desc_pool_destroy(void)
{
    CACHED_FREE(runner_gso_desc_pool);
}
EXPORT_SYMBOL(bdmf_gso_desc_pool_destroy);


/** Checks if a packet needs GSO processing and convert skb to GSO Descriptor
 * \param[in]   sysb  system buffer
 * \param[out]  is_gso_pkt_p indicates to caller if sysb is a GSO packet
 * \returns for Non-GSO: sysb->data GSO: GSO Desciptor or NULL
 */
void *bdmf_sysb_data_to_gsodesc(const bdmf_sysb sysb, uint32_t *is_gso_pkt_p)
{
#if 0
    if(IS_FKBUFF_PTR(sysb))
    {
        printk("GSO not supported with fkbs's \n");
        return NULL;
    }
    else
#endif
    {
        struct sk_buff *skb = PNBUFF_2_SKBUFF(sysb);
        uint32_t linear_len, total_frag_len;
        uint16_t nr_frags, i;
        runner_gso_desc_t *gso_desc_p;
        skb_frag_t *frag;
        uint8_t *vaddr;
	unsigned int bv_len=0, bv_offset=0;

        if(unlikely(skb_is_nonlinear(skb) && (skb->ip_summed != CHECKSUM_PARTIAL)))
        {
            /* forwarded pkts which are received as frags need to be linearized,
            * just check for checksum not needed and frags
            *
            * Note:we don't expect a packet with GSO skb without
            * CHECKSUM_PARTIAL even for UDP pkts
            */
            if(__skb_linearize(skb)){
                printk(KERN_ERR "%s: skb_linearize failed \n",__func__);
                return NULL;
            }
        }        
        
        if((skb->ip_summed == CHECKSUM_PARTIAL) || skb_is_gso(skb) || skb_shinfo(skb)->nr_frags)
        {

#if 0
            /*TODO remove this check after verfiying with 8K page sizes */
            if( skb->len > 64*1024)
            {
                printk("%s: ERROR: skb->len %d  >64K\n",__FUNCTION__, skb->len);
            }
#endif

#if defined(CONFIG_BCM_PKTRUNNER_GSO_DEBUG)
            if (bdmf_gso_desc_dump_enabled)
            {
                printk("skb->ip_summed == CHECKSUM_PARTIAL = %d\n", skb->ip_summed == CHECKSUM_PARTIAL ? 1 : 0);
                printk("skb->ip_summed = %d\n", skb->ip_summed);
                printk("skb_is_gso(skb) = %d\n", skb_is_gso(skb) ? 1 : 0);
                printk("skb_shinfo(skb)->gso_size = %d\n", skb_shinfo(skb)->gso_size);
                printk("skb_shinfo(skb)->nr_frags = %d\n", skb_shinfo(skb)->nr_frags);
            }
#endif

            /* TODO when FRAGLIST is supported add nr_frags of all skb's in fraglist */
            nr_frags = skb_shinfo(skb)->nr_frags;

            if(nr_frags >  RUNNER_MAX_GSO_FRAGS)
            {
                printk("%s: ERROR: nr_frags %d exceed max\n",__FUNCTION__, nr_frags);
                return NULL;
            }

            gso_desc_p = bdmf_runner_gso_desc_alloc();
            if(!gso_desc_p)
            {
                printk("%s: ERROR: failed to allocate gso_desc_p\n",__FUNCTION__);
                return NULL;
            }

            if(nr_frags)
            {
                total_frag_len = 0;

                for(i=0; i < nr_frags; i++ )
                {
                    frag = &skb_shinfo(skb)->frags[i];
                    vaddr = bdmf_kmap_skb_frag(frag);
#if defined(__KERNEL__) && (LINUX_VERSION_CODE >= KERNEL_VERSION(5,10,0))
                    bv_len = frag->bv_len;
                    bv_offset = frag->bv_offset;
#else
                    bv_len = frag->page_offset;
                    bv_offset = frag->size;
#endif
                    cache_flush_len((vaddr + bv_offset), bv_len);
                    gso_desc_p->frag_data[i]= swap4bytes(VIRT_TO_PHYS(vaddr + bv_offset));
                    gso_desc_p->frag_len[i] = swap2bytes((uint16_t)(bv_len));
                    total_frag_len += bv_len;
                    bdmf_kunmap_skb_frag(vaddr);
                }

                linear_len = skb->len - total_frag_len;
            }
            else
            {
                linear_len = skb->len;
            }
#if defined(CONFIG_BCM_PKTRUNNER_GSO_DEBUG)
            if (bdmf_gso_desc_dump_enabled)
                printk("skb_shinfo(skb)->nr_frags = %d\n", skb_shinfo(skb)->nr_frags);
#endif
            /*We expect skb->data to be flushed already in RDPA */
            //cache_flush_len(skb->data, linear_len);


            gso_desc_p->data = swap4bytes(VIRT_TO_PHYS(skb->data));
            gso_desc_p->linear_len = swap2bytes(linear_len);
            gso_desc_p->len = swap2bytes(skb->len);
            gso_desc_p->nr_frags = nr_frags;
            gso_desc_p->mss = swap2bytes(skb_shinfo(skb)->gso_size);
            *is_gso_pkt_p = 1;
            cache_flush_len(gso_desc_p, sizeof(runner_gso_desc_t));

#if defined(CONFIG_BCM_PKTRUNNER_GSO_DEBUG)
            bdmf_gso_desc_dump(gso_desc_p);
#endif

            return gso_desc_p;
        }
        else
        {
            *is_gso_pkt_p = 0;
            return skb->data;
        }
    }
}
EXPORT_SYMBOL(bdmf_sysb_data_to_gsodesc);
#endif


/*
 * CONFIG_RUNNER_CPU_TX_FRAG_GATHER enables Runner support for SKB frag data
 * Enabled for Runner Gen6: 63146, 4912, 6813, etc 
 */
#if defined(CONFIG_RUNNER_CPU_TX_FRAG_GATHER)

#if defined(CC_CONFIG_BCM_SG_FRAG_GATHER_DEBUG)
static int bdmf_sg_desc_dump_enabled = 0;

#define __debug(fmt, arg...)                    \
        if(bdmf_sg_desc_dump_enabled)           \
            printk(fmt, ##arg)
#else
#define __debug(fmt, arg...)
#endif

static DEFINE_SPINLOCK(bdmf_runner_sg_desc_pool_lock);

#define BDMF_RUNNER_SG_DESC_POOL_LOCK(flags)    spin_lock_irqsave(&bdmf_runner_sg_desc_pool_lock, flags)
#define BDMF_RUNNER_SG_DESC_POOL_UNLOCK(flags)  spin_unlock_irqrestore(&bdmf_runner_sg_desc_pool_lock, flags)

runner_sg_desc_t *runner_sg_desc_pool=NULL;
uint8_t *runner_sg_hdr_pool=NULL;

/** Recycle a SG Descriptor
 * \param[in]   sg_desc_p   pointer to SG Descriptor
 */
void bdmf_runner_sg_desc_free(runner_sg_desc_t *sg_desc_p)
{
    if(sg_desc_p)
    {
        unsigned long flags;

        BDMF_RUNNER_SG_DESC_POOL_LOCK(flags);
        sg_desc_p->is_allocated = 0;
        cache_flush_len(sg_desc_p, sizeof(runner_sg_desc_t));
        BDMF_RUNNER_SG_DESC_POOL_UNLOCK(flags);
    }
}

/** Allocate a SG Descriptor
 * \returns pointer to allocated SG Descriptor
 */
runner_sg_desc_t *bdmf_runner_sg_desc_alloc(void)
{
    static int alloc_index = 0;
    int i=0;
    unsigned long flags;
    runner_sg_desc_t *sg_desc_p;

    BDMF_RUNNER_SG_DESC_POOL_LOCK(flags);

    while(i < RUNNER_MAX_SG_DESC)
    {
        if(alloc_index == RUNNER_MAX_SG_DESC)
            alloc_index = 0;

        sg_desc_p = &runner_sg_desc_pool[alloc_index];
        cache_invalidate_len(sg_desc_p, sizeof(runner_sg_desc_t));
        
        if(!sg_desc_p->is_allocated)
        {
            memset(sg_desc_p, 0, sizeof(runner_sg_desc_t));
            /* link sg_desc with header buffer */
            /* header buffer grows upwards with packet L4/L3/L2 hdr push operation */
            sg_desc_p->gso_template_hdr = (uint8_t *)(runner_sg_hdr_pool + (alloc_index + 1) * RUNNER_SG_HDR_SIZE);

            sg_desc_p->is_allocated = 1;
            alloc_index++;
            BDMF_RUNNER_SG_DESC_POOL_UNLOCK(flags);
            return sg_desc_p;
        }
        alloc_index++;
        i++;
    }
    BDMF_RUNNER_SG_DESC_POOL_UNLOCK(flags);
    return NULL;
}


/** Check SG Descriptor pool
 * \returns Num_of_free SG descriptors
 */
int bdmf_runner_check_sg_desc_pool(void)
{
    int cnt=0, i;
    runner_sg_desc_t *sg_desc_p;

    for(i = 0; i < RUNNER_MAX_SG_DESC; i++)
    {
        sg_desc_p = &runner_sg_desc_pool[i];
        cache_invalidate_len(sg_desc_p, 32);
        
        if(!sg_desc_p->is_allocated)
            cnt += 1;
    }

    return cnt;
}
EXPORT_SYMBOL(bdmf_runner_check_sg_desc_pool);

/** Create a pool of SG Descriptors
 * \param[in]   num_desc  number of descriptors to be created
 * \returns 0=Success -1=failure
 */
int bdmf_sg_desc_pool_create( uint32_t num_desc)
{
    uint32_t mem_size = num_desc * sizeof(runner_sg_desc_t);
    uint32_t hdr_mem_size = num_desc * RUNNER_SG_HDR_SIZE;

    runner_sg_desc_pool = CACHED_MALLOC(mem_size);
    if(!runner_sg_desc_pool)
    {
        printk(KERN_ERR "###### ERROR:Failed to allocate runner_sg_desc_pool\n");
        return -1;
    }
    memset(runner_sg_desc_pool, 0, mem_size);
    cache_flush_len(runner_sg_desc_pool,  mem_size);

    runner_sg_hdr_pool = CACHED_MALLOC(hdr_mem_size);
    if(!runner_sg_hdr_pool)
    {
        printk(KERN_ERR "###### ERROR:Failed to allocate runner_sg_header_pool\n");
        return -1;
    }
    memset(runner_sg_hdr_pool, 0, hdr_mem_size);
    cache_flush_len(runner_sg_hdr_pool,  hdr_mem_size);

    printk("\n\n\n\n++++Runner sg_desc_pool and sg_header_pool created successfully !!!!!!!!!!!!!!!!\n");
    printk("SG descriptor Num_of_entries %u, sz_of_entry %u, mem_size %u, addr 0x%px\n", 
            num_desc, (uint32_t)sizeof(runner_sg_desc_t), mem_size, runner_sg_desc_pool);
    printk("SG header pool Num_of_entries %u, sz_of_entry %u, mem_size %u, addr 0x%px\n", 
            num_desc, RUNNER_SG_HDR_SIZE, hdr_mem_size, runner_sg_hdr_pool);
#if defined(CC_CONFIG_BCM_SG_FRAG_GATHER_DEBUG)
    printk("&bdmf_sg_desc_dump_enabled=%px bdmf_sg_desc_dump_enabled=%d\n\n\n\n\n",
            &bdmf_sg_desc_dump_enabled, bdmf_sg_desc_dump_enabled);
#endif

    return 0;
}
EXPORT_SYMBOL(bdmf_sg_desc_pool_create);

/** Free a pool of SG Descriptors
 */
void bdmf_sg_desc_pool_destroy(void)
{
    CACHED_FREE(runner_sg_desc_pool);
}
EXPORT_SYMBOL(bdmf_sg_desc_pool_destroy);

/** Checks if a packet needs SG processing and convert skb to SG Descriptor
 * \param[in]   sg_descriptor pointer
 * \param[in]   xmit_fn, pbuf, cpu_tx_info, passed from rdpa_cpu driver
 * \returns rc
 */
int inline bdmf_sg_send_desc(runner_sg_desc_t *sg_desc_p, rnr_cpu_tx_func xmit_fn, pbuf_t *pbuf, const void *info)
{
    int rc;

    pbuf->data = sg_desc_p;
    /* use special/impossible pkt_length as SG_DESC marker */
    pbuf->length = CPU_TX_SG_PACKET_TYPE_LEN;

    rc = xmit_fn(pbuf, (const rdpa_cpu_tx_info_t *)info, bdmf_sysb_mark(pbuf->sysb));
    if (rc)
    {
        __debug("%s: xmit_fn rc %d\n", __func__, rc);
        /* free SG_DESC as xmit fails */
        bdmf_runner_sg_desc_free(sg_desc_p);
    }
    return rc;
}

/** Checks if a packet needs SG processing and convert skb to SG Descriptor
 * \param[in]   sysb  system buffer
 * \param[out]  is_sg_pkt_p indicates to caller if sysb is a SG packet
 * \returns for Non-SG: sysb->data SG: SG Desciptor or NULL
 */
void *bdmf_sysb_data_to_sgdesc(const bdmf_sysb sysb, uint32_t *is_sg_pkt_p, rnr_cpu_tx_func xmit_fn, pbuf_t *pbuf, const void *info, uint8_t recycle_bit)
{
    struct sk_buff *skb = PNBUFF_2_SKBUFF(sysb);
    uint32_t linear_len, total_frag_len;
    uint16_t nr_frags, i, frag_idx;
    runner_sg_desc_t *sg_desc_p;
    skb_frag_t *frag;
    uint8_t *vaddr;
    static uint32_t cnt_fail_alloc_sg = 0;
    int rc;
    unsigned int bv_len=0, bv_offset=0;
    
    if (skb_is_gso(skb) ||
        ( ((skb->ip_summed == CHECKSUM_PARTIAL) ||
           (skb->ip_summed == CHECKSUM_NONE)) && skb_is_nonlinear(skb) ))
    {
        /* GSO */
        rc = bdmf_sg_gso(skb, xmit_fn, pbuf, info);
        if (!rc)
            *is_sg_pkt_p = 1;

        /* GSO segment completed in function call and no more process needed */
        return NULL;
    }

    /* single packet csum offload, csum_partial, including tcp and udp, has no frag data */
    else if ((skb->ip_summed == CHECKSUM_PARTIAL) && (!skb_is_nonlinear(skb)))
    {
        sg_desc_p = bdmf_runner_sg_desc_alloc();
        if (!sg_desc_p)
        {
            printk("%s: ERROR: failed to allocate sg_desc_p cnt %d\n",__FUNCTION__, cnt_fail_alloc_sg++);
            return NULL;
        }

        if (bdmf_sg_csum(skb, sg_desc_p) != 0)
        {
            goto done_exit;
        }
        cache_flush_len(sg_desc_p, sizeof(runner_sg_desc_t));
        
        /* xmit packet */
        pbuf->is_sg_desc = 1;
        pbuf->do_not_recycle = recycle_bit;
        rc = bdmf_sg_send_desc(sg_desc_p, xmit_fn, pbuf, info);
        if (!rc)
            *is_sg_pkt_p = 1;

        return NULL;
    }

    /* Forwarding packet, Scatter-Gather frag data only, checksum is completed */
    else if ((skb->ip_summed != CHECKSUM_PARTIAL) && (skb_is_nonlinear(skb)))
    {
        __debug("%s: SG pkt, skb->nr_frag %d\n", __FUNCTION__, skb_shinfo(skb)->nr_frags);
        __debug("skb->ip_summed = %d\n", skb->ip_summed);

        nr_frags = skb_shinfo(skb)->nr_frags;

        if (nr_frags > RUNNER_MAX_SG_FRAGS)
        {
            printk("%s: ERROR: skb->nr_frags %d exceed max_sg_array %d\n",__FUNCTION__, nr_frags, RUNNER_MAX_SG_FRAGS);
            return NULL;
        }

        sg_desc_p = bdmf_runner_sg_desc_alloc();
        if (!sg_desc_p)
        {
            printk("%s: ERROR: failed to allocate sg_desc_p cnt %d\n",__FUNCTION__, cnt_fail_alloc_sg++);
            return NULL;
        }

        /* linear data first */
        /* Note: 5G/LTE doesn't have linear data */
        frag_idx = 0;
        total_frag_len = 0;

        linear_len = skb->len - skb->data_len;
        if (linear_len)
        {
            sg_desc_p->frag_data[0] = __swap4bytes64(VIRT_TO_PHYS(skb->data));
            sg_desc_p->frag_len[0]  = swap2bytes(linear_len);
            frag_idx = 1;
            nr_frags += 1;  /* count skb linear data */
        }
        
        /* frag data next */
        for (i = 0; i < skb_shinfo(skb)->nr_frags; i++ )
        {
            frag = &skb_shinfo(skb)->frags[i];
            vaddr = bdmf_kmap_skb_frag(frag);
#if defined(__KERNEL__) && (LINUX_VERSION_CODE >= KERNEL_VERSION(5,10,0))
                    bv_len = frag->bv_len;
                    bv_offset = frag->bv_offset;
#else
                    bv_len = frag->page_offset;
                    bv_offset = frag->size;
#endif
            cache_flush_len((vaddr + bv_offset), bv_len);
            sg_desc_p->frag_data[i + frag_idx]= __swap4bytes64(VIRT_TO_PHYS(vaddr + bv_offset));
            sg_desc_p->frag_len[i + frag_idx] = swap2bytes((uint16_t)(bv_len));
            total_frag_len += bv_len;
            bdmf_kunmap_skb_frag(vaddr);
        }
                    
        sg_desc_p->total_len = swap2bytes(skb->len);
        sg_desc_p->nr_frags = nr_frags;
        cache_flush_len(sg_desc_p, sizeof(runner_sg_desc_t));

        __debug("sg_desc: total_len %d, sg_frag %d, is_allocated %d\n", 
                swap2bytes(sg_desc_p->total_len), sg_desc_p->nr_frags, sg_desc_p->is_allocated);
        
        /* xmit packet */
        pbuf->is_sg_desc = 1;
        pbuf->do_not_recycle = recycle_bit;
        rc = bdmf_sg_send_desc(sg_desc_p, xmit_fn, pbuf, info);
        if (!rc)
            *is_sg_pkt_p = 1;

        return NULL;
    }

done_exit:

    /* rest of SKB type, or non-TCP/UDP packet type */
    *is_sg_pkt_p = 0;
    return skb->data;
}
EXPORT_SYMBOL(bdmf_sysb_data_to_sgdesc);
#endif

/** Add data to sysb
 *
 * The function will is similar to skb_put()
 *
 * \param[in]   sysb        System buffer
 * \param[in]   bytes       Bytes to add
 * \returns added block pointer
 */
static inline void *bdmf_sysb_put(const bdmf_sysb sysb, uint32_t bytes)
{
    return nbuff_put((struct sk_buff *)sysb, bytes);
}

int bdmf_int_connect(int irq, int cpu, int flags,
    int (*isr)(int irq, void *priv), const char *name, void *priv)
{
    int rc;
    /* Supposingly should work for all BCM platforms.
     * If it is not the case - mode ifdefs can be added later.
     * We might also switch to
     * BcmHalMapInterruptEx in order to handle affinity.
     */
    rc = BcmHalMapInterrupt((FN_HANDLER)isr, (void *)priv, irq);
    return rc ? BDMF_ERR_INTERNAL : 0;
}

/** Unmask IRQ
 * \param[in]   irq IRQ
 */
void bdmf_int_enable(int irq)
{
    /*
     * in case of Oren the BcmHalMapInterrupt already enables the interrupt.
     */
    /* Supposingly should work for all BCM platforms.
     * If it is not the case - mode ifdefs can be added later.
     */
}

/** Mask IRQ
 * \param[in]   irq IRQ
 */
void bdmf_int_disable(int irq)
{
    /* Supposingly should work for all BCM platforms.
     * If it is not the case - mode ifdefs can be added later.
     */
    BcmHalInterruptDisable(irq);
}
EXPORT_SYMBOL(bdmf_sysb_headroom_size_set);
EXPORT_SYMBOL(bdmf_sysb_databuf_recycle);
EXPORT_SYMBOL(bdmf_sysb_recycle);
EXPORT_SYMBOL(bdmf_sysb_recycle_skb);
EXPORT_SYMBOL(bdmf_int_connect);
EXPORT_SYMBOL(bdmf_int_enable);
EXPORT_SYMBOL(bdmf_int_disable);
