/*
 * ntpclient.c - NTP client
 *
 * Copyright (C) 1997, 1999, 2000, 2003, 2006, 2007, 2010, 2015  Larry Doolittle  <larry@doolittle.boa.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License (Version 2,
 *  June 1991) as published by the Free Software Foundation.  At the
 *  time of writing, that license was published by the FSF with the URL
 *  http://www.gnu.org/copyleft/gpl.html, and is incorporated herein by
 *  reference.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  Possible future improvements:
 *      - Write more documentation  :-(
 *      - Support leap second processing
 *      - Support IPv6
 *      - Support multiple (interleaved) servers
 *
 *  Compile with -DPRECISION_SIOCGSTAMP if your machine really has it.
 *  Older kernels (before the tickless era, pre 3.0?) only give an answer
 *  to the nearest jiffy (1/100 second), not so interesting for us.
 *
 *  If the compile gives you any flak, check below in the section
 *  labelled "XXX fixme - non-automatic build configuration".
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include "dniconfig.h"
#include <json-c/json.h>

#ifdef linux
#include <sys/utsname.h>
#include <sys/time.h>
typedef u_int32_t __u32;
#include <sys/timex.h>
#else
#define main ntpclient
extern struct hostent *gethostbyname(const char *name);
extern int h_errno;
#define herror(hostname) \
	fprintf(stderr,"Error %d looking up hostname %s\n", h_errno,hostname)
typedef uint32_t __u32;
#endif

#define ENABLE_DEBUG

extern char *optarg;

char *wan_proto = NULL;
 
#define DEV_URL "devicelocation.dev.ngxcld.com"
#define QA_URL "devicelocation.qa.ngxcld.com"
#define PROD_URL "devicelocation.ngxcld.com"
/* the content of file is stored in static array */
static char *cat_file(char *name)
{
        int i;
        FILE *fp;
        static char buf[512];

        buf[0] = '\0';

        fp = fopen(name, "r");
        if (fp == NULL)
                return buf;
        fgets(buf, sizeof(buf), fp);
        fclose(fp);

        i = 0;
        while (buf[i] != '\0' && buf[i] != '\r' && buf[i] != '\n')
                i++;
        buf[i] = '\0';

        return buf;
}

char* get_sn(char *buf, char buf_size)
{
	FILE *fp = NULL;
	fp = popen("artmtd -r sn", "r");
	if(fp != NULL)
		fgets(buf, buf_size, fp);
	else
		return NULL;
	char *p_sn = strstr(buf, "sn:");
	if(p_sn != NULL)
		p_sn += 3;
	else
		return NULL;
	
	int i = 0;
	char *p_line = p_sn;
	for(i =0; i < buf; i++){
		if(*p_line == '\n' || *p_line == '\r'){
			*p_line = 0;
			break;
		}
		p_line++;
	}

	pclose(fp);
	return p_sn;
}

static char *get_value_from_dto(char *string)
{
	FILE *fp;
	char cmd[128] = {0};

	snprintf(cmd, sizeof(cmd)-1, "d2 -s %s > /tmp/cache/ntpclient/ra_d2_info", string);
	system(cmd);

	return cat_file("/tmp/cache/ntpclient/ra_d2_info");
}

static void getRaCloudTime()
{
#define CURRENT_TIME_PATH "device-location/syncTime"
	char user[64] = {0}, pw[64] = {0}, cmd[512] = {0}, sn[32] = {0}, model[8] = {0}, *p = NULL;
	int env = 0;
	p = get_value_from_dto("xagentcfg[0].x_agent_id");

	if(p != NULL)
		strncpy(user, p, sizeof(user)-1);
	else
		user[0] = '0';

	if(strncmp(user, "DEV-", 4) == 0)
		env = 1;
	else if(strncmp(user, "QA-", 3) == 0)
		env = 2;
	else
		env = 0;

	p = get_value_from_dto("xagentcfg[0].x_agent_claim_code");
	if(p != NULL)
		strncpy(pw, p, sizeof(pw)-1);
	else
		pw[0] = '0';

	char sn_data[20] = {0};
	strncpy(sn, get_sn(sn_data, sizeof(sn_data)), sizeof(sn)-1);
	strncpy(model, cat_file("/module_name"), sizeof(model)-1);

	if(access("/tmp/cache/ntpclient/xCloud_time", F_OK) == 0)
		remove("/tmp/cache/ntpclient/xCloud_time");

/*	if(env == 1)
		snprintf(cmd, sizeof(cmd)-1, "curl -k -X POST https://%s/%s -H \"content-type: application/json\" -H \"Authorization:%s:%s\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/xCloud_time --connect-timeout 3 &", DEV_URL, CURRENT_TIME_PATH, user, pw, sn, model);
	else if (env == 2)
		snprintf(cmd, sizeof(cmd)-1, "curl -k -X POST https://%s/%s -H \"content-type: application/json\" -H \"Authorization:%s:%s\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/xCloud_time --connect-timeout 3 &",  QA_URL, CURRENT_TIME_PATH, user, pw, sn, model);
	else
		snprintf(cmd, sizeof(cmd)-1, "curl -k -X POST https://%s/%s -H \"content-type: application/json\" -H \"Authorization:%s:%s\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/xCloud_time --connect-timeout 3 &",PROD_URL, CURRENT_TIME_PATH, user, pw, sn, model);
*/
/*	if(env == 1)
		snprintf(cmd, sizeof(cmd)-1, "curl --capath /etc/ssl/certs -X POST https://%s/%s -H \"content-type: application/json\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/xCloud_time --connect-timeout 3 &", DEV_URL, CURRENT_TIME_PATH, sn, model);
	else if (env == 2)
		snprintf(cmd, sizeof(cmd)-1, "curl --capath /etc/ssl/certs -X POST https://%s/%s -H \"content-type: application/json\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/xCloud_time --connect-timeout 3 &",  QA_URL, CURRENT_TIME_PATH, sn, model);
	else
		snprintf(cmd, sizeof(cmd)-1, "curl --capath /etc/ssl/certs -X POST https://%s/%s -H \"content-type: application/json\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/xCloud_time --connect-timeout 3 &",PROD_URL, CURRENT_TIME_PATH, sn, model);
*/
	if(env == 1)
		snprintf(cmd, sizeof(cmd)-1, "curl -k -X POST https://%s/%s -H \"content-type: application/json\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/cache/ntpclient/xCloud_time --connect-timeout 3", DEV_URL, CURRENT_TIME_PATH, sn, model);
	else if (env == 2)
		snprintf(cmd, sizeof(cmd)-1, "curl -k -X POST https://%s/%s -H \"content-type: application/json\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/cache/ntpclient/xCloud_time --connect-timeout 3",  QA_URL, CURRENT_TIME_PATH, sn, model);
	else
		snprintf(cmd, sizeof(cmd)-1, "curl -k -X POST https://%s/%s -H \"content-type: application/json\" -d \'{\"hardwareId\":\"%s\",\"model\":\"%s\"}\' -o /tmp/cache/ntpclient/xCloud_time --connect-timeout 3",PROD_URL, CURRENT_TIME_PATH, sn, model);
	system(cmd);
	system("echo '[ntpclient] getRaCloudTime, Query...' > /dev/console");
}

static void setRaCloudTime()
{
	FILE *ct_fp;
	char time_buf[256] = {0}, sztime[24] = {0};
	char *data = NULL, *zoneoffset = NULL;
	struct timeval tm_ra = {0};
	int data_flag = 0, zone_flag = 0;

	ct_fp = fopen("/tmp/cache/ntpclient/xCloud_time", "r");
	if(ct_fp) {
		while(fgets(time_buf, sizeof(time_buf)-1, ct_fp)) {
			if(strstr(time_buf, "timestamp")) {
				strtok(time_buf, ": ");
				data = strtok(NULL, ",");
				if(data != NULL) {
					tm_ra.tv_sec = atol(data);
					data_flag = 1;
				}
			}
	/* set 0 time-zone's time ,comment following*/
	//		else if(strstr(time_buf, "zoneOffset")) {
	//			strtok(time_buf, ": ");
	//			zoneoffset = strtok(NULL, "\n");
	//			if(zoneoffset != NULL) {
	//				time = time + atol(zoneoffset);
	//				zone_flag = 1;
	//			}
	//		}
		}
		fclose(ct_fp);
	}
	if(data_flag == 1) {
		if(settimeofday(&tm_ra, NULL) < 0)
			perror("[ntp] set ra time error");
		system("touch /tmp/cache/ntpclient/ntp_RaUpdated");
	}
}

static inline void updateRaCloudTime()
{
	if(access("/tmp/cache/ntpclient/ntp_RaUpdated", F_OK) != 0)
	{
		getRaCloudTime();
		setRaCloudTime();
	}
}
 
/**
 * @brief dni function to get random num.
 *
 * @return random num
 */
unsigned int DNI_random()
{
	unsigned int randnum = 0;
	int fd_rand = open("/dev/urandom", O_RDONLY);

	if (fd_rand < 0) return 0;
	if (read(fd_rand, &randnum, sizeof(randnum)) < 0) {
		close(fd_rand);
		return 0;
	}
	close(fd_rand);
	return randnum;
}


void ntp_syn_status(char *fail_type){
	char buf[72] = {0};
	snprintf(buf, sizeof(buf)-1, "echo '{ \"FailReason\": \"%s\" }' > /tmp/cache/ntpclient/status", fail_type);
	system(buf);
}

void update_ntp_port_status(unsigned short port)
{
	char buf[72] = {0};

	snprintf(buf, sizeof(buf)-1, "echo '{ \"PortNumber\": \"%u\" }' > /tmp/cache/ntpclient/port", port);
	system(buf);
}

#define JAN_1970        0x83aa7e80      /* 2208988800 1970 - 1900 in seconds */
#define NTP_PORT (123)
#define DAY_TIME 86400
// According to NTGR demand, reduce period to 1s.
#define NETGEAR_PERIOD 1

/* How to multiply by 4294.967296 quickly (and not quite exactly)
 * without using floating point or greater than 32-bit integers.
 * If you want to fix the last 12 microseconds of error, add in
 * (2911*(x))>>28)
 */
#define NTPFRAC(x) ( 4294*(x) + ( (1981*(x))>>11 ) )

/* The reverse of the above, needed if we want to set our microsecond
 * clock (via settimeofday) based on the incoming time in NTP format.
 * Basically exact.
 */
#define USEC(x) ( ( (x) >> 12 ) - 759 * ( ( ( (x) >> 10 ) + 32768 ) >> 16 ) )

/* Converts NTP delay and dispersion, apparently in seconds scaled
 * by 65536, to microseconds.  RFC1305 states this time is in seconds,
 * doesn't mention the scaling.
 * Should somehow be the same as 1000000 * x / 65536
 */
#define sec2u(x) ( (x) * 15.2587890625 )

struct ntptime {
	unsigned int coarse;
	unsigned int fine;
};

void send_packet(int usd, struct ntptime *udp_send_ntp);
int rfc1305print(char *data, struct ntptime *arrival, struct ntptime *udp_send_ntp);
int udp_handle(int usd, char *data, int data_len, struct sockaddr *sa_source, int sa_len, struct ntptime *udp_send_ntp);

/* global variables (I know, bad form, but this is a short program) */
char incoming[1500];
struct timeval time_of_send;
int set_clock=0;   /* non-zero presumably needs root privs */

#ifdef ENABLE_BOOT_RELAY
#define BOOT_RELAY_OPTION "f"                  // add an option to force to do ntp boot  relay 
int boot_relay=0;
#else
#define BOOT_RELAY_OPTION
#endif

#ifdef ENABLE_DEBUG
int debug=0;
#define DEBUG_OPTION "d"
#else
#define debug 0
#define DEBUG_OPTION
#endif

int get_current_freq()
{
	/* OS dependent routine to get the current value of clock frequency.
	 */
#ifdef linux
	struct timex txc;
	txc.modes = 0;
	if (adjtimex(&txc) < 0) {
		perror("adjtimex"); exit(1);
	}
	return txc.freq;
#else
	return 0;
#endif
}

void send_packet(int usd, struct ntptime *udp_send_ntp)
{
	__u32 data[12];
	struct timeval now;
#define LI 0
#define VN 4
#define MODE 3
#define STRATUM 0
#define POLL 4 
#define PREC -6

	if (debug) fprintf(stderr,"Sending ...\n");
	if (sizeof(data) != 48) {
		fprintf(stderr,"size error\n");
		return;
	}
	bzero((char*)data,sizeof(data));
	data[0] = htonl (
		( LI << 30 ) | ( VN << 27 ) | ( MODE << 24 ) |
		( STRATUM << 16) | ( POLL << 8 ) | ( PREC & 0xff ) );
	data[1] = htonl(1<<16);  /* Root Delay (seconds) */
	data[2] = htonl(1<<16);  /* Root Dispersion (seconds) */
	gettimeofday(&now,NULL);
	data[10] = htonl(now.tv_sec + JAN_1970); /* Transmit Timestamp coarse */
	data[11] = htonl(NTPFRAC(now.tv_usec));  /* Transmit Timestamp fine   */
	
	udp_send_ntp->coarse = now.tv_sec + JAN_1970;
	udp_send_ntp->fine   = NTPFRAC(now.tv_usec);
	
	send(usd, data, 48, 0);
	time_of_send = now;
}


struct sockaddr *syslog_sa;
int udp_handle(int usd, char *data, int data_len, 
				  struct sockaddr *sa_source, int sa_len, struct ntptime *udp_send_ntp)
{
	struct timeval udp_arrival;
	struct ntptime udp_arrival_ntp;

	syslog_sa = sa_source;
#ifdef _PRECISION_SIOCGSTAMP
	if ( ioctl(usd, SIOCGSTAMP, &udp_arrival) < 0 ) {
		perror("ioctl-SIOCGSTAMP");
		gettimeofday(&udp_arrival, NULL);
	}
#else
	gettimeofday(&udp_arrival, NULL);
#endif
	udp_arrival_ntp.coarse = udp_arrival.tv_sec + JAN_1970;
	udp_arrival_ntp.fine = NTPFRAC(udp_arrival.tv_usec);
	
	if ((udp_arrival_ntp.coarse > (udp_send_ntp->coarse + 5)) 
		||((udp_arrival_ntp.coarse == (udp_send_ntp->coarse + 5))&&(udp_arrival_ntp.fine > udp_send_ntp->fine)))
		return -1;

	if (debug) {
		struct sockaddr_in *sa_in = (struct sockaddr_in *)sa_source;

		printf("packet of length %d received\n", data_len);
		if (sa_source->sa_family == AF_INET) {
			printf("Source: INET Port %d host %s\n",
			ntohs(sa_in->sin_port), inet_ntoa(sa_in->sin_addr));
		} else {
			printf("Source: Address family %d\n", sa_source->sa_family);
		}
	}
	
	return(rfc1305print(data, &udp_arrival_ntp, udp_send_ntp));
}

double ntpdiff( struct ntptime *start, struct ntptime *stop)
{
	int a;
	unsigned int b;
	a = stop->coarse - start->coarse;
	if (stop->fine >= start->fine) {
		b = stop->fine - start->fine;
	} else {
		b = start->fine - stop->fine;
		b = ~b;
		a -= 1;
	}
	
	return a*1.e6 + b * (1.e6/4294967296.0);
}

int rfc1305print(char *data, struct ntptime *arrival, struct ntptime *udp_send_ntp)
{
/* straight out of RFC-1305 Appendix A */
	int li, vn, mode, stratum, poll, prec;
	int delay, disp, refid;
	struct ntptime reftime, orgtime, rectime, xmttime;
	double etime,stime,skew1,skew2;
	int freq;
	FILE *fp;
	char cmd[128];
	char *pid_file = "/tmp/run/syslogd.pid";

#define Data(i) ntohl(((unsigned int *)data)[i])
	li      = Data(0) >> 30 & 0x03;
	vn      = Data(0) >> 27 & 0x07;
	mode    = Data(0) >> 24 & 0x07;
	stratum = Data(0) >> 16 & 0xff;
	poll    = Data(0) >>  8 & 0xff;
	prec    = Data(0)       & 0xff;
	if (prec & 0x80) prec|=0xffffff00;
	delay   = Data(1);
	disp    = Data(2);
	refid   = Data(3);
	reftime.coarse = Data(4);
	reftime.fine   = Data(5);
	orgtime.coarse = Data(6);
	orgtime.fine   = Data(7);
	rectime.coarse = Data(8);
	rectime.fine   = Data(9);
	xmttime.coarse = Data(10);
	xmttime.fine   = Data(11);
#undef Data

	if ((orgtime.coarse != udp_send_ntp->coarse)
		|| ((orgtime.coarse == udp_send_ntp->coarse) && (orgtime.fine != udp_send_ntp->fine))) {
		if (debug) {
			printf("The Originate Timestamp in the server reply");
			printf("doesn't match the Transmit Timestamp used in the request\n");
		}
		return -1;
	}
	if ((xmttime.coarse == 0) && (xmttime.fine == 0)) {
		if (debug) {
			printf("The Transmit Timestamp in the server reply is zero\n");
		}
		return -1;
	}
	if (mode != 4) {
		if (debug) {
			printf("The Mode in the server reply is not 4 (unicast)\n");
		}
		return -1;
	}

	/* Error Checking, see RFC-4330 section 5, KOD code check  */
	if (stratum == 0) {  //kiss of death check mechanism.
		struct sockaddr_in *syslog_sa_in = (struct sockaddr_in *)syslog_sa;
		if (debug) {
			printf("The Stratum is error for KOD check mechanism\n");
		}
		if (syslog_sa->sa_family == AF_INET){
		    syslog(LOG_WARNING, "[DoS Attack: Kiss of Death] from source: %s, port: %d", inet_ntoa(syslog_sa_in->sin_addr),ntohs(syslog_sa_in->sin_port));
		}
		else{
		    syslog(LOG_WARNING, "[DoS Attack: Kiss of Death] from source family: %d",syslog_sa->sa_family);
		}
		return -1;
	}

	if (set_clock) {   /* you'd better be root, or ntpclient will crash! */
		struct timeval tv_set;
		/* it would be even better to subtract half the slop */
		tv_set.tv_sec  = xmttime.coarse - JAN_1970;
		/* divide xmttime.fine by 4294.967296 */
		tv_set.tv_usec = USEC(xmttime.fine);
		
		if (settimeofday(&tv_set,NULL) < 0) {
			perror("settimeofday");
			exit(1);
		}
		if (debug) {
			printf("set time to %lu.%.6lu\n", tv_set.tv_sec, tv_set.tv_usec);
		}
	}

	if (debug) {
		printf("LI=%d  VN=%d  Mode=%d  Stratum=%d  Poll=%d  Precision=%d\n",
			li, vn, mode, stratum, poll, prec);
		printf("Delay=%.1f  Dispersion=%.1f  Refid=%u.%u.%u.%u\n",
			sec2u(delay),sec2u(disp),
			refid>>24&0xff, refid>>16&0xff, refid>>8&0xff, refid&0xff);
		printf("Reference %u.%.10u\n", reftime.coarse, reftime.fine);
		printf("Originate %u.%.10u\n", orgtime.coarse, orgtime.fine);
		printf("Receive   %u.%.10u\n", rectime.coarse, rectime.fine);
		printf("Transmit  %u.%.10u\n", xmttime.coarse, xmttime.fine);
		printf("Our recv  %u.%.10u\n", arrival->coarse, arrival->fine);
	}
	
	etime = ntpdiff(&orgtime, arrival);
	stime = ntpdiff(&rectime, &xmttime);
	skew1 = ntpdiff(&orgtime, &rectime);
	skew2 = ntpdiff(&xmttime, arrival);
	freq = get_current_freq();
	
	if (debug) {
		printf("Total elapsed: %9.2f\n"
	       		"Server stall:  %9.2f\n"
	      		"Slop:          %9.2f\n",
			etime, stime, etime-stime);
		printf("Skew:          %9.2f\n"
	       		"Frequency:     %9d\n"
	       		" day   second     elapsed    stall     skew  dispersion  freq\n",
			(skew1-skew2)/2, freq);
		printf("%d %5d.%.3d  %8.1f %8.1f  %8.1f %8.1f %9d\n",
			arrival->coarse/86400+15020, arrival->coarse%86400,
			arrival->fine/4294967, etime, stime,
			(skew1-skew2)/2, sec2u(disp), freq);
	}
	
	/* log the first entry */
	system("[ -f /tmp/cache/ntpclient/ntp_updated ] || { "
			"touch /tmp/cache/ntpclient/ntp_updated; "
			"/usr/sbin/ntpst set; "
			"}");

	if (access(pid_file,F_OK) == 0){
		sprintf(cmd,"kill -USR1 $(cat %s)",pid_file);
		system(cmd);
		sleep(1);
	}
	syslog(LOG_WARNING, "[Time synchronized with NTP server]");
	fflush(stdout);

	/* use hot plug mechaism to boot ntpclient_update*/		
	char *script = "/usr/sbin/ntpclient-hotplug-call";
	char *action = "offset"; //default action so far, may add more.
	char call_cmd[50] = {0};
	snprintf(call_cmd, sizeof(call_cmd), "%s %s", script, action);
	system(call_cmd);

	return 1;
}


int stuff_net_addr(struct in_addr *p, char *hostname)
{
	struct hostent *ntpserver;
	
	ntpserver = gethostbyname(hostname);
	if (ntpserver == NULL) {
		/* avoid printing: "time-h.netgear.com: Unknown host" */
		/* herror(hostname); */
		return 0;
	}
	if (ntpserver->h_length != 4) {
		fprintf(stderr, "oops %d\n", ntpserver->h_length);
		return 0;
	}
	
	memcpy(&(p->s_addr), ntpserver->h_addr_list[0], 4);
	return 1;
}

int setup_receive(int usd, unsigned int interface, unsigned short port)
{
	struct sockaddr_in sa_rcvr;

	bzero((char *) &sa_rcvr, sizeof(sa_rcvr));
	sa_rcvr.sin_family = AF_INET;
	sa_rcvr.sin_addr.s_addr = htonl(interface);
	sa_rcvr.sin_port = htons(port);

	if (bind(usd, (struct sockaddr *) &sa_rcvr, sizeof(sa_rcvr)) == -1) {
		fprintf(stderr, "Could not bind to udp port %d\n", port);
		perror("bind");
		return 0;
	}

	listen(usd, 3);

	/* Make "usd" close on child process when call system(),
	 * so that the child process will not inherit the parent resource */
	fcntl(usd, F_SETFD, FD_CLOEXEC);

	return 1;
}

int setup_transmit(int usd, char *host, unsigned short port)
{
	struct sockaddr_in sa_dest;
	
	bzero((char *) &sa_dest, sizeof(sa_dest));
	sa_dest.sin_family = AF_INET;
	
	if (!stuff_net_addr(&(sa_dest.sin_addr), host))
		return 0;
	
	sa_dest.sin_port = htons(port);
	
	if (connect(usd, (struct sockaddr *)&sa_dest, sizeof(sa_dest)) == -1) {
		perror("connect");
		return 0;
	}

	return 1;
}

void primary_loop(int usd, int num_probes, int cycle_time)
{
	fd_set fds;
	struct sockaddr sa_xmit;
	int i, pack_len, sa_xmit_len, probes_sent;
	struct timeval to;
	struct ntptime udp_send_ntp;
	int steady_state = 0;

	if (debug) printf("Listening...\n");

	probes_sent = 0;
	sa_xmit_len = sizeof(sa_xmit);
	to.tv_sec = 0;
	to.tv_usec = 0;
	
	for (;;) {
		FD_ZERO(&fds);
		FD_SET(usd, &fds);
		i = select(usd+1, &fds, NULL, NULL, &to);  /* Wait on read or error */
		if ((i != 1) || (!FD_ISSET(usd,&fds))) {
			if (i == EINTR) continue;
			if (i < 0) perror("select");
			
			if ((to.tv_sec == 0) || (to.tv_sec == cycle_time) 
					|| (to.tv_sec == DAY_TIME)) {
				if (steady_state != 1 
					&& probes_sent >= num_probes && num_probes != 0) 
					break;
				
				steady_state = 0;
				send_packet(usd, &udp_send_ntp);
				++probes_sent;
				to.tv_sec = cycle_time;
				to.tv_usec = 0;
			}
			continue;
		}
		
		pack_len = recvfrom(usd, incoming, sizeof(incoming), 0, &sa_xmit, &sa_xmit_len);
		
		if (pack_len < 0) {
			perror("recvfrom");
			updateRaCloudTime();
		    /* A query receives no successful response, the retry algorithm must 
			*  wait that random delay period before initiating the first retry query.
			*/
			select(1, NULL, NULL, NULL, &to);
		} else if (pack_len > 0 && pack_len < sizeof(incoming)) {
			steady_state = udp_handle(usd, incoming, pack_len, 
						&sa_xmit, sa_xmit_len, &udp_send_ntp);
		} else {
			printf("Ooops.  pack_len=%d\n", pack_len);
			fflush(stdout);
		}

		/* error check for rfc4330*/
		if (steady_state == -1){
			to.tv_sec = cycle_time;
			to.tv_usec = 0;
			select(1, NULL, NULL, NULL, &to);
		}

		if (steady_state == 1) {
			to.tv_sec = DAY_TIME;
			to.tv_usec = 0;
			ntp_syn_status("0"); // ntp syn sucessfully for gui warning.
		} else if (probes_sent >= num_probes && num_probes != 0) 
			break;
	}
	/*when program is out of primary loop,the NTP server is fail,so delete the file.*/
	system("rm -f /tmp/cache/ntpclient/ntp_updated");
	ntp_syn_status("2");
}

static int wan_conn_up(void)
{
	struct json_object *objRoot;
	objRoot = json_object_from_file("/tmp/cache/network/network.status");
	if(objRoot == NULL){
		printf("[ntpclient] open netwan status failed. \r\n");
	}
	struct json_object *objWanstatus;
	json_object_object_get_ex(objRoot, "wan_updown", &objWanstatus);
	if(objWanstatus == NULL){
		printf("[ntpclient] wan status isn't set.");
	}
	int ret_val = strstr(json_object_to_json_string(objWanstatus),"up") == NULL?0:1;
	json_object_put(objRoot);
	return ret_val;
}

static int is_normalmode()
{
	struct json_object *objRoot;
	objRoot = json_object_from_file("/tmp/cache/network/network.conf");
	if(objRoot == NULL){
		printf("[ntpclient] open netwan conf failed. \r\n");
	}
	struct json_object *objOpmode;
	json_object_object_get_ex(objRoot, "opmode", &objOpmode);
	if(objOpmode == NULL){
		printf("[ntpclient] wan conf isn't set.");
	}
	int ret_val = strstr(json_object_to_json_string(objOpmode),"normal") == NULL?0:1;
	json_object_put(objRoot);
	return ret_val;
}

void usage(char *argv0)
{
	fprintf(stderr,
				"\t\t -t time_zone -o opmode\n"
				"\t\t -h hostname [-b second_hostname]\n"
                "\t\t   [-i min_interval -m max_interval (0<=min_interval<=max_interval)]\n"
                "\t\t   [-c count] [-p local_port] [-d] [-s]\n",
		argv0);
}

/*	
 *	@brief  get a random and availdable port
 *		it's for switching port fot NTPv4,because some ISP will 
 *		block NTP's src port packet. 
 *	@return if success, return a port value,
 *		if failed ,while() in the funciton.
 * */
unsigned short get_random_port(void)
{
	int sockfd = 0;
	struct sockaddr_in addr;
	unsigned short port = 0;
	int addr_len = sizeof(struct sockaddr_in);
	do{
	    while((sockfd = socket(AF_INET,SOCK_DGRAM,0)) < 0)
	    {
		    perror("socket for switching port");
		    sleep(1);
	    }
	    addr.sin_family = AF_INET;
	    addr.sin_port = htons(port);
	    addr.sin_addr.s_addr = htonl(INADDR_ANY);
	    while(bind(sockfd,(struct sockaddr*)&addr,sizeof(addr))< 0)
	    {
		    perror("bind for switching port");
		    sleep(1);
	    }
	    while(getsockname(sockfd,(struct sockaddr*)&addr,&addr_len)!= 0)
	    {
		    perror("ntp get sock name error");
		    sleep(1);
	    }
	    port = ntohs(addr.sin_port);
	    close(sockfd);
	}while(port<1024);
	return(port);
}

inline void SetSystemTzDst(const char *tz_p){
	FILE *fp = fopen("/tmp/TZ", "w+");
	if(fp){
		fprintf(fp, tz_p);
		fclose(fp);
	}
	else{
		if(debug)
		   printf("set system tz&dst error\r\n");
	}
}

int main(int argc, char *argv[]) {
	int usd;  /* socket */
	int c;
	/* Tnese parameters are settable from the command line
	   the initializations here provide default behavior */
	unsigned short udp_local_port = NTP_PORT; 
	int probe_count = 1;            /* default of 0 means loop forever */
	int cycle_time = 15;          /* request timeout in seconds */
	int min_interval = 15;
	int max_interval = 60;
	/* int debug=0; is a global above */
	char *hostname = NULL;          /* must be set */
	char *sec_host = "0.0.0.0";
	char *ntps = "0.0.0.0";
	const char *time_zone = NULL;
	char *op_mode = NULL;
	struct timeval to;
	FILE *fp = NULL, *fd = NULL;
	char *manual_ntp = NULL;
	char manual_ntp_server_tmp[128]={0};  // The length of manual server have checked with GUI,it's less than 128 bytes.
	char buff[32] = {0};
	int is_base = 0;

	unsigned short use_default_server = 1;

	unsigned long seed;

	/* ntpclient -o opmode -w dhcp -h "time-g.netgear.com" -b "time-h.netgear.com" -i 15 -m 60 -p 123 -s */
	for (;;) {
		c = getopt(argc, argv, "t:w:o:c:"DEBUG_OPTION BOOT_RELAY_OPTION "h:i:p:b:m:s");
		if (c == EOF) break;
		switch (c) {
			case 't':
				time_zone = optarg;
				break;
			case 'w':
				wan_proto = optarg;
				break;
			case 'o':
 				op_mode = optarg;
				break;
			case 'c':
 				probe_count = atoi(optarg);
				break;
#ifdef ENABLE_DEBUG
			case 'd':
				++debug;
				break;
#endif
			case 'h':
				hostname = optarg;
				break;
#ifdef ENABLE_BOOT_RELAY
			case 'f':
				boot_relay = 1;
				break;
#endif
			case 'i':
				min_interval = atoi(optarg);
				break;
			case 'p':
				udp_local_port = atoi(optarg);
				break;
			case 'b':
				sec_host = optarg;
				break;
			case 'm':
				max_interval = atoi(optarg);
				break;
			case 's':
				set_clock = 1;
				probe_count = 1;
				break;
			default:
				usage(argv[0]);
				exit(1);
		}
	}

	if (hostname == NULL || time_zone == NULL || op_mode == NULL) {
		usage(argv[0]);
		exit(1);
	}
	SetSystemTzDst(time_zone);
	ntps = hostname;
	if (min_interval > max_interval || min_interval < 0 || max_interval < 0) {
		usage(argv[0]);
		exit(1);
	} else if (max_interval == 0) {
		max_interval = cycle_time;
		min_interval = cycle_time;
	} else
		cycle_time = min_interval + DNI_random()%(max_interval-min_interval+1);
	
	if (debug) {
		printf("Configuration:\n"
			"  -t time_zone         %s\n"
			"  -o op_mode           %s\n"
			"  -c probe_count       %d\n"
			"  -d (debug)           %d\n"
			"  -h hostname          %s\n"
			"  -b second hostname   %s\n"
			"  -i interval(min)     %d\n"
			"  -m interval(max)     %d\n"
			"  -p local_port        %d\n"
			"  -s set_clock         %d\n",
			time_zone,op_mode, probe_count, debug, hostname, sec_host, 
			min_interval, max_interval, udp_local_port, set_clock);
	} else {
		daemon(1, 1);
	}

	if((fd = fopen("/tmp/orbi_type", "r")) != NULL)
	{
		fgets(buff, sizeof(buff)-1, fd);
		if(strstr(buff, "Base"))
			is_base = 1;
		fclose(fd);
	}

	system("[ -d /tmp/cache/ntpclient ] || mkdir -p /tmp/cache/ntpclient");
	system("[ -f /tmp/cache/ntpclient/ntp_updated ] && rm -f /tmp/cache/ntpclient/ntp_updated");
#ifdef ENABLE_BOOT_RELAY
	if (boot_relay == 0){
		//detect whether the router is in the booting process
		fp=fopen("/tmp/boot_status","r");
		if (fp != NULL){
			boot_relay = 1;
			fclose(fp);
			fp = NULL;
		}
	}
#endif

	while(1) {
		/* if use default server, client's requests should like following
		 *  |  NTP Server   time-e.netgear.com  |
		 *  |  local port    123                |
		 *  |  NTP Server   time-f.netgear.com  |
		 *  |  local port    123                |
		 *  |  NTP Server   time-e.netgear.com  |
		 *  |  local port    random-port        |
		 *  |  NTP Server   time-f.netgear.com  |
		 *  |  local port    random-port        |
		 * * * 
		 * */
		static char poll_count = 0;
		if(poll_count > 0)
			ntps = (strcmp(ntps, hostname) == 0) ? sec_host : hostname;
		poll_count++;
		if(poll_count > 2){
			udp_local_port = ((udp_local_port == NTP_PORT)?get_random_port() : NTP_PORT);
			poll_count = 1;
		}
		update_ntp_port_status(udp_local_port);
		if (strcmp(ntps, "0.0.0.0") == 0)
			continue;

		if (debug) {
			printf("Configuration:\n"
				"  Probe count          %d\n"
				"  Debug                %d\n"
				"  NTP server           %s\n"
				"  Interval             %d\n"
				"  Local port           %d\n"
				"  Set clock            %d\n",
				probe_count, debug, ntps, cycle_time, 
				udp_local_port, set_clock);
		}

		/* Startup sequence */
		if ((usd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
			perror ("socket");
			goto cont;
		}

		if ((!wan_conn_up() && is_base) && strcmp("ap_mode", op_mode) != 0 && strcmp("bridge_mode", op_mode) != 0) {
			/* printf("The WAN connection is NOT up!\n"); */
			ntp_syn_status("1");
			close(usd);
			goto cont;
		}

#ifdef ENABLE_BOOT_RELAY
		if (boot_relay == 1){
			ntps = "0.0.0.0";
			close(usd);
			goto cont;
		}
#endif

		if (!setup_receive(usd, INADDR_ANY, udp_local_port)
				|| !setup_transmit(usd, ntps, NTP_PORT)) {
			close(usd);
			ntp_syn_status("1");
			/* Get Cloud time from xagent.
			 * if ntp fail ,use ra method to update coarsely.
			 * it will just updated once.
			 * */
			updateRaCloudTime();
			to.tv_sec = cycle_time;
			to.tv_usec = 0;
			select(1, NULL, NULL, NULL, &to);
			goto loop;
		}

		primary_loop(usd, probe_count, cycle_time);
		close(usd);
	loop:
		/* Get Cloud time from xagent.
		 * if ntp fail ,use ra method to update coarsely.
		 * it will just updated once.
		 * */
		updateRaCloudTime();
		/* [NETGEAR Spec 8.6]:Subsequent queries will double the preceding query interval 
		 * until the interval has exceeded the steady state query interval, at which point 
		 * and new random interval between 15.00 and 60.00 seconds is selected and the 
		 * process repeats.
		 */
		if ((cycle_time * 2) > DAY_TIME)
			cycle_time = min_interval + DNI_random()%(max_interval-min_interval+1);
		else
			cycle_time = cycle_time * 2;
		continue;

	cont:	
		/* [NETGEAR Spec 8.6]: we will wait randomly calculated period of 0 to 240 seconds 
		 * before issuing the first NTP query upon subsequent power-ons or resets. 
		 */
#ifdef ENABLE_BOOT_RELAY
		if (boot_relay == 1){
			boot_relay = 0;
		}
#endif
		to.tv_sec = DNI_random() % (NETGEAR_PERIOD + 1);
		to.tv_usec = 0;
		select(1, NULL, NULL, NULL, &to);

	}
	
	return 0;
}


