/* clientpacket.c
 *
 * Packet generation and dispatching functions for the DHCP client.
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#define _GNU_SOURCE
#include <string.h>
#include <sys/socket.h>
#include <features.h>
#if __GLIBC__ >=2 && __GLIBC_MINOR >= 1
#include <netpacket/packet.h>
#include <net/ethernet.h>
#else
#include <asm/types.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#endif
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "dhcpd.h"
#include "packet.h"
#include "options.h"
#include "dhcpc.h"
#include "debug.h" 


/* Create a random xid */
unsigned long random_xid(void)
{
	static int initialized;
	if (!initialized) {
		int fd;
		unsigned long seed;

		fd = open("/dev/urandom", 0);
		if (fd < 0 || read(fd, &seed, sizeof(seed)) < 0) {
			LOG(LOG_WARNING, "Could not load seed from /dev/urandom: %s",
				strerror(errno));
			seed = time(0);
		}
		if (fd >= 0) close(fd);
		srand(seed);
		initialized++;
	}
	return rand();
}

#if defined(CONFIG_ORANGE_ISP)

struct dchp_option_90_auth_id
{
    char code;
    char length;
    char protocol;
    char algorithm;
    char RDM;
    char replay_detection[8];
    char authentication_info[51]; /* assume the struct size is 64 byte */
};

static void gen_option90_string(char* option_str)
{
    extern char service_type[32];
    extern char option90_autheticationId[32];
    char option90_auth_info[64]="";
    char option90_auth_id[40]="";
    int len;
    char prefix_orange_dhcp[]={0x1a,0x09,0x00,0x00,0x05,0x58,0x01,0x03,0x41,0x01,0x0d};
    struct dchp_option_90_auth_id dhcp90_content;
    time_t uTime;
    int i = 0;

    if(strlen(service_type)==0)
        return;

    if(strlen(option90_autheticationId)!=0)
    {
        memset(&dhcp90_content, 0x00, sizeof(struct dchp_option_90_auth_id));
    
        dhcp90_content.code = DHCP_AUTH_ID;

        memset(&dhcp90_content.protocol, 0, sizeof(dhcp90_content.protocol));
        memset(&dhcp90_content.algorithm, 0, sizeof(dhcp90_content.algorithm));
        memset(&dhcp90_content.RDM, 0, sizeof(dhcp90_content.RDM));
        memset(&dhcp90_content.replay_detection, 0, sizeof(dhcp90_content.replay_detection));
        //sprintf(option90_auth_info,"%sfti/%s",prefix_orange_dhcp,option90_autheticationId);
	    memset(option90_auth_info,0x1a,1);
	    memset(option90_auth_info+1,0x09,1);
	    memset(option90_auth_info+2,0x00,1);
	    memset(option90_auth_info+3,0x00,1);			
	    memset(option90_auth_info+4,0x05,1);
	    memset(option90_auth_info+5,0x58,1);	 
	    memset(option90_auth_info+6,0x01,1);	 
	    memset(option90_auth_info+7,0x03,1);	
	    memset(option90_auth_info+8,0x41,1);	 
	    memset(option90_auth_info+9,0x01,1);
		memset(option90_auth_info+10,0x0d,1);	
		sprintf(option90_auth_id,"%s",option90_autheticationId);
		memcpy(option90_auth_info+11,option90_autheticationId,strlen(option90_auth_id));		       
        
		memcpy(dhcp90_content.authentication_info, option90_auth_info, strlen(option90_auth_id)+11);

        dhcp90_content.length = sizeof(dhcp90_content.protocol) + sizeof(dhcp90_content.algorithm) \
            + sizeof(dhcp90_content.RDM) + sizeof(dhcp90_content.replay_detection) + 11 + strlen(option90_auth_id);
        memcpy(option_str, &dhcp90_content, dhcp90_content.length + 2);
    }

    return;
}

static void gen_option77_string(char* option_str)
{
    extern char service_type[32];
    int len;

    if(strstr(service_type, "Internet"))
        len = strlen(ORANGE_INTERNET_USER_CLASS);
    else if(strstr(service_type, "IPTV"))
        len = strlen(ORANGE_IPTV_USER_CLASS);
    else
        return;
        
    if( len!=0 )
    {
        option_str[0] = DHCP_USER_CLASS;
        option_str[1] = len+1;

        /* user class data */
        option_str[2] = len;

        
        if(strstr(service_type, "Internet"))
            strncpy(option_str + 3, ORANGE_INTERNET_USER_CLASS, len);
        else if(strstr(service_type, "IPTV"))
            strncpy(option_str + 3, ORANGE_IPTV_USER_CLASS, len);
    }

    return;
}



struct dhcp_option_125_vendor_id
{
    char code;
    char opt_len;
    char enterprise_N[4];
    char data_len;
	char data[57];	 
};



static void gen_option125_string(char* option_str)
{
    
    extern char service_type[32];
    extern char serial_number[14];
    char option125_sub_opt[64]="";
    char value[4]={0x00,0x00,0x00,0x00};
      
    struct dhcp_option_125_vendor_id dhcp125_content;
    int opt_len = 0;
    int data_len = 0;
	int enterprise_N = NTGR_ENTERPRISE_NUM;
	    
    char sn[20]="";
    if(strlen(service_type)==0)
        return;
    if(strlen(serial_number)==0)
        return;
    memset(&dhcp125_content, 0x00, sizeof(struct dhcp_option_125_vendor_id));
    memset(sn, 0x00, sizeof(sn));
    memset(option125_sub_opt,0x00,sizeof(option125_sub_opt));
    dhcp125_content.code = DHCP_VENDOR_ID;
    value[0] =  (enterprise_N >> 24) & 0xFF;
    value[1] =  (enterprise_N >> 16) & 0xFF;
    value[2] =  (enterprise_N >> 8) & 0xFF;
    value[3] =  enterprise_N  & 0xFF;
    memcpy(dhcp125_content.enterprise_N,value,4);
    /*sub_option 4*/
    option125_sub_opt[0] = 0x04;
    option125_sub_opt[1] = strlen(ORANGE_OUI);
    memcpy(option125_sub_opt+2,ORANGE_OUI,strlen(ORANGE_OUI));
    data_len+=strlen(ORANGE_OUI);
    data_len+=2;
    /*sub_option 5*/
    option125_sub_opt[data_len] = 0x05;
    data_len+=1;
    sprintf(sn,"NQ%s",serial_number);
    option125_sub_opt[data_len] = strlen(sn);
    data_len+=1;
    memcpy(option125_sub_opt+data_len,sn,strlen(sn));
    data_len+=strlen(sn);
     /*sub_option 6*/
    option125_sub_opt[data_len] = 0x06; 
    data_len+=1;
    option125_sub_opt[data_len] = strlen(PRODUCT_CLASS);
    data_len+=1;
    memcpy(option125_sub_opt+data_len,PRODUCT_CLASS,strlen(PRODUCT_CLASS));
    data_len+=strlen(PRODUCT_CLASS);
    memcpy(dhcp125_content.data, option125_sub_opt, data_len);
    dhcp125_content.data_len = data_len;
    opt_len = data_len + 5;
	dhcp125_content.opt_len = opt_len;
	memcpy(option_str, &dhcp125_content, dhcp125_content.opt_len + 2);	 
    return;
}
#endif /* CONFIG_ORANGE_ISP */


/* initialize a packet with the proper defaults */
static void init_packet(struct dhcpMessage *packet, char type)
{
	struct vendor  {
		char vendor, length;
		char str[sizeof("udhcp "VERSION)];
	} vendor_id = { DHCP_VENDOR,  sizeof("udhcp "VERSION) - 1, "udhcp "VERSION};

    /* @option60 */
#if defined(_XDSL_PRODUCT)
    extern char option60_venderId[64];
	unsigned char vendor_id1[66];
	int len;
	len=strlen(option60_venderId);
	if( len!=0 )
    {
			vendor_id1[0] = DHCP_VENDOR;
			vendor_id1[1] = len;
			strncpy(vendor_id1 + 2, option60_venderId, len);
	}
#endif	    
#if defined(CONFIG_ORANGE_ISP)
    extern char service_type[32];
	unsigned char vendor_id1[66];
	int len;
    if(strstr(service_type, "Internet"))
        len = strlen(ORANGE_INTERNET_VCI);
    else if(strstr(service_type, "IPTV"))
        len = strlen(ORANGE_IPTV_VCI);
    else
        len = strlen(service_type);
	
	if(strstr(service_type, "PPPoE"))
        len = 0;
	
	if( len!=0 )
    {
		vendor_id1[0] = DHCP_VENDOR;
		vendor_id1[1] = len;
        if(strstr(service_type, "Internet"))
		    strncpy(vendor_id1 + 2, ORANGE_INTERNET_VCI, len);
        else if(strstr(service_type, "IPTV"))
		    strncpy(vendor_id1 + 2, ORANGE_IPTV_VCI, len);
        else
            strncpy(vendor_id1 + 2, service_type, len);
	}
#endif
#if defined(CONFIG_ORANGE_ISP)
    /* @option77 user class */
    unsigned char userClass1[66];
    memset(userClass1, 0, sizeof(userClass1));
    
    if(strstr(service_type, "Internet") || strstr(service_type, "IPTV"))
        gen_option77_string(userClass1);
    /* @option125 user class */
    unsigned char VendorID1[66];        
    if(strstr(service_type, "Internet") || strstr(service_type, "IPTV"))
        gen_option125_string(VendorID1);        
    
    /* @option90 authentication id */
    unsigned char autheticationId1[66];
    memset(autheticationId1, 0x00, sizeof(autheticationId1));
    if(strstr(service_type, "Internet") || strstr(service_type, "IPTV"))
        gen_option90_string(autheticationId1);
#endif
	init_header(packet, type);
	memcpy(packet->chaddr, client_config.arp, 6);
	add_option_string(packet->options, client_config.clientid);
    /* foxconn wklin modified start, 08/10/2007 */
    if (type != DHCPDECLINE) {
	    if (client_config.hostname) add_option_string(packet->options, client_config.hostname);
#if defined(_XDSL_PRODUCT) || defined(CONFIG_ORANGE_ISP)     /* @option60 */
	    if( len!=0 )
	    	add_option_string(packet->options, vendor_id1);
	    else
#endif
            add_option_string(packet->options, (unsigned char *) &vendor_id);
    
#if defined(CONFIG_ORANGE_ISP)
        /* @option77 user class */
        if(strstr(service_type, "Internet") || strstr(service_type, "IPTV"))
        {
  		//	printf("%s:%d service_type(%s)\n",__func__,__LINE__,service_type);
            add_option_string(packet->options, userClass1);
		}  
        
        /* @option90 authentication id */
        if(strstr(service_type, "Internet") || strstr(service_type, "IPTV"))
        {
          //  printf("%s:%d service_type(%s)\n",__func__,__LINE__,service_type);
            add_option_string(packet->options, VendorID1);
            add_option_string(packet->options, autheticationId1);
		}  
#endif          
    }
    /* foxconn wklin modified end, 08/10/2007 */

}


/* Add a paramater request list for stubborn DHCP servers. Pull the data
 * from the struct in options.c. Don't do bounds checking here because it
 * goes towards the head of the packet. */
static void add_requests(struct dhcpMessage *packet)
{
	int end = end_option(packet->options);
	int i, len = 0;

	packet->options[end + OPT_CODE] = DHCP_PARAM_REQ;
	for (i = 0; options[i].code; i++)
		if (options[i].flags & OPTION_REQ)
			packet->options[end + OPT_DATA + len++] = options[i].code;
	packet->options[end + OPT_LEN] = len;
	packet->options[end + OPT_DATA + len] = DHCP_END;

}


/* Broadcast a DHCP discover packet to the network, with an optionally requested IP */
int send_discover(unsigned long xid, unsigned long requested)
{
	struct dhcpMessage packet;

	init_packet(&packet, DHCPDISCOVER);
	packet.xid = xid;
	if (requested)
		add_simple_option(packet.options, DHCP_REQUESTED_IP, requested);

	add_requests(&packet);
	return raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST, 
				SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
}


/* Broadcasts a DHCP request message */
int send_selecting(unsigned long xid, unsigned long server, unsigned long requested)
{
	struct dhcpMessage packet;
	struct in_addr addr;

	init_packet(&packet, DHCPREQUEST);
	packet.xid = xid;

	add_simple_option(packet.options, DHCP_REQUESTED_IP, requested);
	add_simple_option(packet.options, DHCP_SERVER_ID, server);
	
	add_requests(&packet);
	addr.s_addr = requested;
	LOG(LOG_DEBUG, "Sending select for %s...", inet_ntoa(addr));
	return raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST, 
				SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
}


/* Unicasts or broadcasts a DHCP renew message */
int send_renew(unsigned long xid, unsigned long server, unsigned long ciaddr)
{
	struct dhcpMessage packet;
	int ret = 0;

	init_packet(&packet, DHCPREQUEST);
	packet.xid = xid;
	packet.ciaddr = ciaddr;

	add_requests(&packet);
	LOG(LOG_DEBUG, "Sending renew...");
	if (server) 
		ret = kernel_packet(&packet, ciaddr, CLIENT_PORT, server, SERVER_PORT);
	else ret = raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST,
				SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
	return ret;
}	


/* Unicasts a DHCP release message */
int send_release(unsigned long server, unsigned long ciaddr)
{
	struct dhcpMessage packet;

	init_packet(&packet, DHCPRELEASE);
	packet.xid = random_xid();
	packet.ciaddr = ciaddr;
	
	add_simple_option(packet.options, DHCP_REQUESTED_IP, ciaddr);
	add_simple_option(packet.options, DHCP_SERVER_ID, server);

	LOG(LOG_DEBUG, "Sending release...");
	return kernel_packet(&packet, ciaddr, CLIENT_PORT, server, SERVER_PORT);
}

/* foxconn wklin added start, 08/07/2007 */
/* Unicasts a DHCP decline message */
int send_decline(unsigned long xid, unsigned long server, unsigned long ciaddr)
{
	struct dhcpMessage packet;
    int ret = 0;

	init_packet(&packet, DHCPDECLINE);
	packet.xid = xid;
	packet.ciaddr = 0; /* 0 per RFC2131 p.37 */
	
	add_simple_option(packet.options, DHCP_REQUESTED_IP, ciaddr);
	add_simple_option(packet.options, DHCP_SERVER_ID, server);

	LOG(LOG_DEBUG, "Sending decline...");
	ret = raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST,
	        SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
    return ret;
}
/* foxconn wklin added end, 08/07/2007 */

/* return -1 on errors that are fatal for the socket, -2 for those that aren't */
int get_raw_packet(struct dhcpMessage *payload, int fd)
{
	int bytes;
	struct udp_dhcp_packet_rcv packet;/*foxconn wklin modified, 10/03/2007*/
	u_int32_t source, dest;
	u_int16_t check;
	
	/* foxconn wklin modified start, 10/03/2007 */
	memset(&packet, 0, sizeof(struct udp_dhcp_packet_rcv));
	bytes = read(fd, &packet, sizeof(struct udp_dhcp_packet_rcv));
	/* foxconn wklin modified end, 10/03/2007 */
	if (bytes < 0) {
		DEBUG(LOG_INFO, "couldn't read on raw listening socket -- ignoring");
		usleep(500000); /* possible down interface, looping condition */
		return -1;
	}
	
	if (bytes < (int) (sizeof(struct iphdr) + sizeof(struct udphdr))) {
		DEBUG(LOG_INFO, "message too short, ignoring");
		return -2;
	}
	
	if (bytes < ntohs(packet.ip.tot_len)) {
		DEBUG(LOG_INFO, "Truncated packet");
		return -2;
	}
	
	/* ignore any extra garbage bytes */
	bytes = ntohs(packet.ip.tot_len);
	
	/* foxconn wklin modified start, 10/03/2007 */
	/* use udp_dhcp_packet_rcv data structure */
	/* Make sure its the right packet for us, and that it passes sanity checks */
	if (packet.ip.protocol != IPPROTO_UDP || packet.ip.version != IPVERSION ||
	    packet.ip.ihl != sizeof(packet.ip) >> 2 || packet.udp.dest != htons(CLIENT_PORT) ||
	    bytes > (int) sizeof(struct udp_dhcp_packet_rcv) ||
	    ntohs(packet.udp.len) != (short) (bytes - sizeof(packet.ip))) {
	    	DEBUG(LOG_INFO, "unrelated/bogus packet");
	    	return -2;
	}
	/* foxconn wklin modified end, 10/03/2007 */

	/* check IP checksum */
	check = packet.ip.check;
	packet.ip.check = 0;
	if (check != checksum(&(packet.ip), sizeof(packet.ip))) {
		DEBUG(LOG_INFO, "bad IP header checksum, ignoring");
		return -1;
	}
	
	/* verify the UDP checksum by replacing the header with a psuedo header */
	source = packet.ip.saddr;
	dest = packet.ip.daddr;
	check = packet.udp.check;
	packet.udp.check = 0;
	memset(&packet.ip, 0, sizeof(packet.ip));

	packet.ip.protocol = IPPROTO_UDP;
	packet.ip.saddr = source;
	packet.ip.daddr = dest;
	packet.ip.tot_len = packet.udp.len; /* cheat on the psuedo-header */
	if (check && check != checksum(&packet, bytes)) {
		DEBUG(LOG_ERR, "packet with bad UDP checksum received, ignoring");
		return -2;
	}
	
	memcpy(payload, &(packet.data), bytes - (sizeof(packet.ip) + sizeof(packet.udp)));
	
	if (ntohl(payload->cookie) != DHCP_MAGIC) {
		LOG(LOG_ERR, "received bogus message (bad magic) -- ignoring");
		return -2;
	}
	DEBUG(LOG_INFO, "oooooh!!! got some!");
	return bytes - (sizeof(packet.ip) + sizeof(packet.udp));
	
}

