/*
 * RTL9311B USB PHY Device Driver for Realtek RTL9311B SoC
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/spinlock.h>
#include <linux/types.h>
#include <linux/uaccess.h>
#include <linux/usb/phy.h>
#include "../../soc/realtek/chip_probe.h"
#include "../../soc/realtek/error.h"

#define RTL9311B_USB_PHY_NAME "rtl9311b_usb_phy"
#define RTK_USB_PHY_VER_UNKNOWN "unknown"

#define SYS_IPECR        0x01000600
#define EN_USBHOST       BIT(26)
#define EN_USBPHY        BIT(1)

#define USBEXT_CFG1           0x0
#define MS_ENDIAN_HOST        BIT(17)
#define SS_ENDIAN_HOST        BIT(16)
#define USBEXT_PHY_FORCE_MODE 0x10
#define PORT0_F_TERM_SEL      BIT(9)
#define PORT0_F_SUSPEND       BIT(7)
#define PORT0_F_DISCON        BIT(5)

#define INSNREG05        0x14

struct phy_data {
	int size;
	u8 *addr;
	u8 *val;
};

struct rtk_usb_phy {
	struct usb_phy	phy;
	struct device	*dev;
	struct phy_data *phy_cali;
	struct phy_data *phy_param;
	void __iomem	*u2_ext;
	void __iomem	*phy_ctrl;
	void __iomem	*insnreg;
	const char      *ver;
	const char      *name;
	u8 port;
	u8 vport;
};

bool rtl9311b_usb_phy_big_endian_desc = false;

#define LOOP_LIMIT 100000
#define UTMI_NOP   (1<<12)
#define UTMI_LOAD  (0<<12)
#define BUILD_CMD(vp,vctrl) (((vp)&0xf)<<13)|(((vctrl)&0xf)<<8)
#define UTMI_BUSY(x) (x&(1<<17))

void __iomem *utmi_ctrl = 0;

static u32 utmi_wait(void)
{
	u32 __reg, __c = 0;

	void __iomem *addr = utmi_ctrl;
	do {
		__c++;
		__reg = readl(addr);
		if (unlikely(__c>LOOP_LIMIT)) {
			printk("utmi_wait timeout\n");
			return 0;
		}
	} while (UTMI_BUSY(__reg));

	return __reg;
}

static void utmi_set(u32 v)
{
	void __iomem *addr;

	addr = utmi_ctrl;

	utmi_wait();

	writel(v, addr);
}

static u32 utmi_get(void)
{
	u32 reg;
	reg = utmi_wait();
	return reg;
}

static int ehci_phy_get(struct rtk_usb_phy *phy, u32 port, u8 reg, u8 *data)
{

	// send [3:0]
	utmi_set(BUILD_CMD(phy->vport,reg&0xf)|UTMI_NOP);
	utmi_set(BUILD_CMD(phy->vport,reg&0xf)|UTMI_LOAD);
	utmi_set(BUILD_CMD(phy->vport,reg&0xf)|UTMI_NOP);

	// send [7:4]
	utmi_set(BUILD_CMD(phy->vport,reg>>4)|UTMI_NOP);
	utmi_set(BUILD_CMD(phy->vport,reg>>4)|UTMI_LOAD);
	utmi_set(BUILD_CMD(phy->vport,reg>>4)|UTMI_NOP);

	*data = utmi_get() & 0xff;

	return 0;
}

static int ehci_is_valid_param(u8 port, u8 reg)
{

	if (port > 1)
		return 0;

	if (((reg >= 0xe0) && (reg <= 0xe7)) ||
	        ((reg >= 0xf0) && (reg <= 0xf7)) ||
	        ((reg >= 0xb0) && (reg <= 0xb7)))
		return 1;

	return 0;
}

static int usb_phy_reg_read(struct rtk_usb_phy *phy, u8 port, u8 reg, u8 *val)
{
	int ret = 0;
	u8 data;

	if (!ehci_is_valid_param(port, reg))
		return -EINVAL;

	if (!((reg >= 0xb0) && (reg <= 0xb7)))
		reg -= 0x20;

	ehci_phy_get(phy, port, reg, &data);

	*val = data;

	return ret;
}

static int usb_phy_reg_write(struct rtk_usb_phy *phy, u8 port, u8 reg, u8 val)
{
	void __iomem *addr = phy->phy_ctrl;
	u32 v;

	if (!ehci_is_valid_param(port, reg)) {
		dev_err(phy->dev, "echi phy param error\n");
		return -EINVAL;
	}

	if (port) {
		v = (readl(addr) & ~0xff0000) | ((val & 0xff) << 16);
		writel(v, addr);
	} else {
		v = (readl(addr) & ~0xff) | (val & 0xff);
		writel(v, addr);
	}

	// send [3:0]
	utmi_set(BUILD_CMD(phy->vport,reg&0xf)|UTMI_NOP);
	utmi_set(BUILD_CMD(phy->vport,reg&0xf)|UTMI_LOAD);
	utmi_set(BUILD_CMD(phy->vport,reg&0xf)|UTMI_NOP);

	// send [7:4]
	utmi_set(BUILD_CMD(phy->vport,reg>>4)|UTMI_NOP);
	utmi_set(BUILD_CMD(phy->vport,reg>>4)|UTMI_LOAD);
	utmi_set(BUILD_CMD(phy->vport,reg>>4)|UTMI_NOP);

	return 0;

}

static void rtk_sysreg_writel(u32 paddr, u32 val)
{
	void __iomem *addr;
	addr = ioremap(paddr, 4);
	if (addr) {
		writel(val, addr);
		iounmap(addr);
	} else {
		printk("ioremap(0x%08x) fail!\n", paddr);
	}
}

static int rtk_sysreg_readl(u32 paddr, u32 *val)
{
	void __iomem *addr;

	addr = ioremap(paddr, 4);
	if (addr) {
		*val = readl(addr);
		iounmap(addr);
		return 0;
	} else {
		return -ENOMEM;
	}
}

bool rtk_usb_byte_order_quirk(void) {
	return rtl9311b_usb_phy_big_endian_desc;
}
EXPORT_SYMBOL(rtk_usb_byte_order_quirk);

static void usb_host_setup(struct rtk_usb_phy *phy)
{
	void __iomem *reg;
	u32 val;
	int ret;

	/* Enable host and PHY */
	ret = rtk_sysreg_readl(SYS_IPECR, &val);
	if (ret)
		dev_err(phy->dev, "Unable to get IP_ENABLE_REG val\n");

	val |= EN_USBHOST | EN_USBPHY;
	rtk_sysreg_writel(SYS_IPECR, val);

	/* Set endian */
	reg = phy->u2_ext + USBEXT_CFG1;
	val = readl(reg);
	if (rtk_usb_byte_order_quirk()) {
		val |= (SS_ENDIAN_HOST);
		val &= ~MS_ENDIAN_HOST;
	} else {
	val |= (MS_ENDIAN_HOST | SS_ENDIAN_HOST);
	}
	writel(val, reg);

	return;
}

static void usb_phy_force_mode_setup(struct rtk_usb_phy *phy)
{
	void __iomem *reg;
	u32 val = 0;

	reg = phy->u2_ext + USBEXT_PHY_FORCE_MODE;
	val = readl(reg);
	val &= ~(PORT0_F_TERM_SEL | PORT0_F_DISCON);
	val |= PORT0_F_SUSPEND;
	writel(val, reg);
}

static void usb_host_led_config(void)
{
	//leave this implementation to SDK team
}

static int usb_phy_patch(struct rtk_usb_phy *phy)
{
	int i;

	dev_info(phy->dev, "USB PHY patch\n");

	/* PHY calibariton */
	for (i = 0; i < phy->phy_cali->size; i++)
		usb_phy_reg_write(phy, phy->port, phy->phy_cali->addr[i], phy->phy_cali->val[i]);

	/* PHY parameter setup*/
	for (i = 0; i < phy->phy_param->size; i++)
		usb_phy_reg_write(phy, phy->port, phy->phy_param->addr[i], phy->phy_param->val[i]);

	return 0;
}

__weak struct proc_dir_entry *realtek_proc;

static void show_usage(void)
{

	printk("	w [addr] [data]: write [value] to [reg] (data should be 8 bits, addr should be in [E0-E7], [F0-F6])\n");
}

static ssize_t rtk_usb_phy_proc_write(struct file * file, const char __user * userbuf, size_t count, loff_t * off)
{
	char buf[32];
	int len, ret;
	u8 addr, data;
	struct rtk_usb_phy *phy = PDE_DATA(file_inode(file));

	len = min(sizeof(buf), count);
	if (copy_from_user(buf, userbuf, len))
		return -E2BIG;

	if (strncmp(buf, "help", 4) == 0) {
		show_usage();
	} else if (strncmp(buf, "w ", 2) == 0) {
		if( 2 == sscanf(buf, "w %hhx %hhx", &addr, &data)) {
			ret = usb_phy_reg_write(phy, phy->port, addr, data);
			if (ret)
				return ret;
		} else {
			goto ERROR_PARA;
		}
	} else if (strncmp(buf, "r ", 2) == 0) {
		if(1 == sscanf(buf, "r %hhx", &addr)) {
			ret = usb_phy_reg_read(phy, phy->port, addr, &data);
			if (ret)
				return ret;
			printk("%02x = %02x\n", addr, data);
		}
	} else {
		goto ERROR_PARA;
	}
	return count;

ERROR_PARA:
	printk("error parameter...\n");
	show_usage();
	return -EPERM;
}


#define ITEM_PER_LINE 4
static int rtk_usb_phy_show(struct seq_file *s, void *v)
{
	struct rtk_usb_phy *phy = s->private;
	int addr, n = 0;
	u8 data;

	seq_printf(s, "USB PHY version: %s\n\n", phy->ver);

	seq_printf(s, "Port[%d]:\n", phy->port);

	/*swtich to page 0*/
	usb_phy_reg_write(phy, phy->port, 0xF4, 0x9b);
	seq_printf(s, "Page 0\n");
	n = 0;
	for (addr = 0xE0; addr <= 0xE7; addr++) {
		n++;
		usb_phy_reg_read(phy, phy->port, addr, &data);
		seq_printf(s, "%02X: %4x   ", addr, data);
		if ((n % ITEM_PER_LINE) == 0)
			seq_printf(s, "\n");
	}
	for (addr = 0xF0; addr <= 0xF7; addr++) {
		n++;
		usb_phy_reg_read(phy, phy->port, addr, &data);
		seq_printf(s, "%02X: %4x   ", addr, data);
		if ((n % ITEM_PER_LINE) == 0)
			seq_printf(s, "\n");
	}

	/*swtich to page 1*/
	usb_phy_reg_write(phy, phy->port, 0xF4, 0xbb);
	seq_printf(s, "\nPage 1\n");
	for (addr = 0xE0; addr <= 0xE7; addr++) {
		n++;
		usb_phy_reg_read(phy, phy->port, addr, &data);
		seq_printf(s, "%02X: %4x   ", addr, data);
		if ((n % ITEM_PER_LINE) == 0)
			seq_printf(s, "\n");
	}
	seq_printf(s, "\n");

	/*swtich to page 2*/
	usb_phy_reg_write(phy, phy->port, 0xF4, 0xdb);
	seq_printf(s, "\nPage 2\n");
	for (addr = 0xE0; addr <= 0xE7; addr++) {
		n++;
		usb_phy_reg_read(phy, phy->port, addr, &data);
		seq_printf(s, "%02X: %4x   ", addr, data);
		if ((n % ITEM_PER_LINE) == 0)
			seq_printf(s, "\n");
	}
	seq_printf(s, "\n");

	/*swtich back to page 0*/
	usb_phy_reg_write(phy, phy->port, 0xF4, 0x9b);
	return 0;
}

static int rtk_usb_phy_open(struct inode *inode, struct file *file)
{
	return(single_open(file, rtk_usb_phy_show, PDE_DATA(inode)));
}


static const struct proc_ops fops_rtk_usb_phy = {
	.proc_open  		= rtk_usb_phy_open,
	.proc_write 		= rtk_usb_phy_proc_write,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= single_release,
};


static int rtk_usb_host_init(struct rtk_usb_phy *uphy)
{
	int ret;
	struct proc_dir_entry *e;

	usb_host_setup(uphy);

	usb_host_led_config();

	ret = usb_phy_patch(uphy);
	if (ret) {
		dev_err(uphy->dev, "PHY configure failed\n");
		return ret;
	}

	usb_phy_force_mode_setup(uphy);

	e = proc_create_data("usb_phy", S_IRUGO | S_IWUSR, realtek_proc, &fops_rtk_usb_phy, uphy);

	return 0;
}

static int rtk_usb_phy_init(struct usb_phy *phy)
{
	/* Todo */
	dev_info(phy->dev, "%s()\n", __func__);
	return 0;
}

static void rtk_usb_phy_shutdown(struct usb_phy *phy)
{
	/* Todo */
	dev_info(phy->dev, "%s()\n", __func__);
}

static int rtk_usb_phy_set_suspend(struct usb_phy *phy, int suspend)
{
	/* Todo */
	dev_info(phy->dev, "%s()\n", __func__);

	return 0;
}

static int rtk_usb_phy_set_wakeup(struct usb_phy *phy, bool enabled)
{
	/* Todo */
	dev_info(phy->dev, "%s()\n", __func__);

	return 0;
}

static int rtk_usb_phy_set_vbus(struct usb_phy *phy, int on)
{
	/* Todo */
	return 0;
}

static int rtk_usb_phy_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *node = dev->of_node;
	struct rtk_usb_phy *uphy;
	struct phy_data *param_data, *cali_data;
	struct resource *res;
	int ret = 0;
    uint32 chip_id, chip_rev_id;

	dev_info(dev, "RTL9311B USB PHY Probe...\n");

    if (drv_swcore_cid_get((uint32)0, (uint32 *)&chip_id, (uint32 *)&chip_rev_id) != RT_ERR_OK)
    {
        dev_err(dev, "9311B USB get chip id/rev fail ret=%x\n", uphy->vport, ret);
        return -EPERM;
    }

    if ((drv_swcore_chipFamilyId_get(chip_id) == RTL9311B_FAMILY_ID) && (chip_rev_id == 0)) 
    {
        rtl9311b_usb_phy_big_endian_desc = true;
        dev_info(dev, "Use BIG ENDIAN DESC\n");
    }

	uphy = devm_kzalloc(dev, sizeof(*uphy), GFP_KERNEL);
	if (IS_ERR(uphy))
		return -ENOMEM;

	/* USB EXT (AUX reg) */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM,
	                                   "usb2_ext");
	uphy->u2_ext = devm_ioremap_resource(dev, res);

	if (IS_ERR(uphy->u2_ext))
		return PTR_ERR(uphy->u2_ext);
	dev_info(dev, "usb2 ext resource - %pr mapped at 0x%pK\n", res,
	         uphy->u2_ext);

	/* PHY CTRL (system reg) */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM,
	                                   "phy_ctrl");
	uphy->phy_ctrl = devm_ioremap_resource(dev, res);

	if (IS_ERR(uphy->phy_ctrl))
		return PTR_ERR(uphy->phy_ctrl);
	dev_info(dev, "phy ctrl resource - %pr mapped at 0x%pK\n", res,
	         uphy->phy_ctrl);

	/* INSNREG (Synopsys-Specific reg) */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM,
	                                   "insnreg");
	uphy->insnreg = devm_ioremap_resource(dev, res);

	if (IS_ERR(uphy->insnreg))
		return PTR_ERR(uphy->insnreg);
	dev_info(dev, "insnreg resource - %pr mapped at 0x%pK\n", res,
	         uphy->insnreg);
	utmi_ctrl = uphy->insnreg + INSNREG05;

	/* Port */
	ret = of_property_read_u8(node, "port", &uphy->port);
	if (ret)
		goto err;
	dev_info(dev, "port: %d\n", uphy->port);

	/* vport for UTMI CTRL */
	ret = of_property_read_u8(node, "vport", &uphy->vport);
	if (ret)
		goto err;
	dev_info(dev, "vport: %d\n", uphy->vport);

	/* PHY parameters ver */
	ret = of_property_read_string(node, "ver", &uphy->ver);
	if (ret) {
		uphy->ver = RTK_USB_PHY_VER_UNKNOWN;
	}

	uphy->dev				= dev;
	uphy->phy.dev			= uphy->dev;
	uphy->phy.label		= RTL9311B_USB_PHY_NAME;
	uphy->phy.init		= rtk_usb_phy_init;
	uphy->phy.shutdown	= rtk_usb_phy_shutdown;
	uphy->phy.set_suspend	= rtk_usb_phy_set_suspend;
	uphy->phy.set_wakeup	= rtk_usb_phy_set_wakeup;
	uphy->phy.set_vbus	= rtk_usb_phy_set_vbus;
	uphy->phy.type        = USB_PHY_TYPE_USB2;

	cali_data = devm_kzalloc(dev, sizeof(*cali_data), GFP_KERNEL);
	if (IS_ERR(cali_data))
		return -ENOMEM;

	if (node) {
		ret = of_property_read_u32_index(node, "cali_size", 0, &cali_data->size);
		if (ret)
			goto err;

		cali_data->addr = devm_kzalloc(dev, sizeof(u8)*cali_data->size, GFP_KERNEL);
		if (!cali_data->addr)
			return -ENOMEM;

		cali_data->val = devm_kzalloc(dev, sizeof(u8)*cali_data->size, GFP_KERNEL);
		if (!cali_data->val)
			return -ENOMEM;

		ret = of_property_read_u8_array(node, "cali_addr",
		                                cali_data->addr, cali_data->size);
		if (ret)
			goto err;

		ret = of_property_read_u8_array(node, "cali_val",
		                                cali_data->val, cali_data->size);
		if (ret)
			goto err;

		uphy->phy_cali = cali_data;
	}

	param_data = devm_kzalloc(dev, sizeof(*param_data), GFP_KERNEL);
	if (IS_ERR(param_data))
		return -ENOMEM;

	if (node) {
		if (chip_rev_id == 0) {
			ret = of_property_read_u32_index(node, "param_a_size", 0, &param_data->size);
			if (ret)
				goto err;

			param_data->addr = devm_kzalloc(dev, sizeof(u8)*param_data->size, GFP_KERNEL);
			if (!param_data->addr)
				return -ENOMEM;

			param_data->val = devm_kzalloc(dev, sizeof(u8)*param_data->size, GFP_KERNEL);
			if (!param_data->val)
				return -ENOMEM;

			ret = of_property_read_u8_array(node, "param_a_addr",
											param_data->addr, param_data->size);
			if (ret)
				goto err;

			ret = of_property_read_u8_array(node, "param_a_val",
											param_data->val, param_data->size);
			if (ret)
				goto err;
		}
		else {
		ret = of_property_read_u32_index(node, "param_size", 0, &param_data->size);
		if (ret)
			goto err;

		param_data->addr = devm_kzalloc(dev, sizeof(u8)*param_data->size, GFP_KERNEL);
		if (!param_data->addr)
			return -ENOMEM;

		param_data->val = devm_kzalloc(dev, sizeof(u8)*param_data->size, GFP_KERNEL);
		if (!param_data->val)
			return -ENOMEM;

		ret = of_property_read_u8_array(node, "param_addr",
		                                param_data->addr, param_data->size);
		if (ret)
			goto err;

		ret = of_property_read_u8_array(node, "param_val",
		                                param_data->val, param_data->size);
		if (ret)
			goto err;
		}

		uphy->phy_param = param_data;
	}

	platform_set_drvdata(pdev, uphy);

	ret = usb_add_phy_dev(&uphy->phy);
	if (ret)
		goto err;

	ret = rtk_usb_host_init(uphy);
	if (ret) {
		usb_remove_phy(&uphy->phy);
		goto err;
	}

	dev_info(dev, "USB PHY version: %s\n", uphy->ver);
	dev_info(dev, "Finished rtl9311b USB PHY Probe\n");

	return 0;

err:
	dev_info(dev, "Error in rtl9311b USB PHY Probe\n");
	return ret;
}

static int rtk_usb_phy_remove(struct platform_device *pdev)
{
	struct rtk_usb_phy *uphy = platform_get_drvdata(pdev);

	usb_remove_phy(&uphy->phy);
	return 0;
}

static const struct of_device_id rtl9311b_usb_phy_dt_match[] = {
	{ .compatible = "rtk,rtl9311b-usb-phy", },
	{},
};
MODULE_DEVICE_TABLE(of, rtl9311b_usb_phy_dt_match);

static struct platform_driver rtl9311b_usb_phy_driver = {
	.probe		= rtk_usb_phy_probe,
	.remove		= rtk_usb_phy_remove,
	.driver		= {
		.name	= RTL9311B_USB_PHY_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(rtl9311b_usb_phy_dt_match),
	},
};
module_platform_driver(rtl9311b_usb_phy_driver);

MODULE_DESCRIPTION("RTL9311B USB PHY Device Driver");
MODULE_ALIAS("platform:" RTL9311B_USB_PHY_NAME);
