/*
 * Copyright (C) 2021 Realtek Semiconductor Corp.
 * All Rights Reserved.
 *
 * This program is the proprietary software of Realtek Semiconductor
 * Corporation and/or its licensors, and only be used, duplicated,
 * modified or distributed under the authorized license from Realtek.
 *
 * ANY USE OF THE SOFTWARE OTHER THAN AS AUTHORIZED UNDER
 * THIS LICENSE OR COPYRIGHT LAW IS PROHIBITED.
 *
 * $Revision$
 * $Date$
 *
 * Purpose : RTK switch soc driver
 *
 * Feature : RTK switch soc driver
 *
 */

/*
 * Include Files
 */
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/of_address.h>
#include <asm/io.h>

#include <soc/realtek/rtk_sw.h>


/*
 * Symbol Definition
 */
/* rtk switch soc data */
typedef struct rtk_sw_data_s {
    struct platform_device *pdev;
    void __iomem *soc_base_addr;
    void __iomem *swcore_base_addr;

    /* system restart notifier */
    struct notifier_block restart_nb;
} rtk_sw_data_t;

/*
 * Data Declaration
 */
rtk_sw_data_t *rtk_swsoc = NULL;
static const struct of_device_id rtk_sw_dt_match[];


/*
 * Macro Definition
 */


/*
 * Function Declaration
 */


/* Function Name:
 *      bsp_mem32_read
 * Description:
 *      Get the value from register.
 * Input:
 *      unit - unit id
 *      addr - register address
 * Output:
 *      pVal - pointer buffer of the register value
 * Return:
 *      RT_ERR_OK
 *      RT_ERR_FAILED
 * Note:
 *      1. Support single unit right now and ignore unit
 *      2. When we support the multiple chip in future, we will check the input unit
 */
int32_t
bsp_mem32_read(uint32_t unit, uint32_t addr, uint32_t *pVal)
{
    int32_t ret = RT_ERR_OK;

    *pVal = readl(rtk_swsoc->swcore_base_addr + (uintptr_t)addr);

    return ret;
} /* end of bsp_mem32_read */


/* Function Name:
 *      bsp_mem32_write
 * Description:
 *      Set the value to register.
 * Input:
 *      unit - unit id
 *      addr - register address
 *      val  - the value to write register
 * Output:
 *      None
 * Return:
 *      RT_ERR_OK
 *      RT_ERR_FAILED
 * Note:
 *      1. Support single unit right now and ignore unit
 *      2. When we support the multiple chip in future, we will check the input unit
 */
int32_t
bsp_mem32_write(uint32_t unit, uint32_t addr, uint32_t val)
{
    int32_t ret = 0;

    writel(val, (rtk_swsoc->swcore_base_addr + (uintptr_t)addr));

    return ret;
} /* end of bsp_mem32_write */


static int
rtk_sw_restart_handler(struct notifier_block *this,
                unsigned long mode, void *cmd)
{
    uint32_t  data;
    uint32_t reg, val;
    int32_t ret;

    if ((rtk_swsoc == NULL) || (rtk_swsoc->pdev == NULL))
    {
        printk("%s:%d: rtk_swsoc not init\n", __FUNCTION__, __LINE__);
        return NOTIFY_BAD;
    }

    ret = of_property_read_u32_array(rtk_swsoc->pdev->dev.of_node, "rst_glb_ctrl_reg", &reg, 1);
    if (ret)
    {
        dev_info(&rtk_swsoc->pdev->dev, "rtk_swsoc of get rst_glb_ctrl_reg of_property fail %d\n", ret);
        return NOTIFY_BAD;
    }

    ret = of_property_read_u32_array(rtk_swsoc->pdev->dev.of_node, "rst_glb_ctrl_val", &val, 1);
    if (ret)
    {
        dev_info(&rtk_swsoc->pdev->dev, "rtk_swsoc get rst_glb_ctrl_val of_property fail %d\n", ret);
        return NOTIFY_BAD;
    }

    bsp_mem32_read(RTK_BSP_DEFAULT_UNIT, reg, &data);
    data = data | val;
    bsp_mem32_write(RTK_BSP_DEFAULT_UNIT, reg, data);
  
    return NOTIFY_OK;
}

static int
rtk_sw_probe(struct platform_device *pdev)
{
    struct device *dev = &pdev->dev;
    struct device_node *dn;
    int32_t ret;

	rtk_swsoc = devm_kzalloc(dev, sizeof(rtk_sw_data_t), GFP_KERNEL);
	if (!rtk_swsoc)
		return -ENOMEM;

	rtk_swsoc->pdev = pdev;
    dn =  pdev->dev.of_node;
    if (!dn)
    {
        dev_err(dev, "socreg of_find_node_by_name BASE failed\n");
        return -ENODEV;
    }
    rtk_swsoc->soc_base_addr = of_iomap(dn, 0);
    if (!rtk_swsoc->soc_base_addr)
    {
        dev_err(dev, "socreg of_iomap failed\n");
        return -ENOMEM;
    }

    rtk_swsoc->swcore_base_addr = of_iomap(dn, 1);
    if (!rtk_swsoc->swcore_base_addr)
    {
        dev_err(dev, "switchreg of_iomap failed\n");
        return -ENOMEM;
    }


    /* register restart handler */
    rtk_swsoc->restart_nb.notifier_call = rtk_sw_restart_handler;
    rtk_swsoc->restart_nb.priority = 131;
    ret = register_restart_handler(&rtk_swsoc->restart_nb);
    if (ret)
    {
        dev_err(dev, "cannot register restart handler (err=%d)\n", ret);
    }

    platform_set_drvdata(pdev, rtk_swsoc);

	return 0;
}


static int
rtk_sw_remove(struct platform_device *pdev)
{
    /* unregister restart handler */
    unregister_restart_handler(&rtk_swsoc->restart_nb);
	return 0;
}

int32_t
rtk_sw_socBaseAddr_get(void __iomem **addr)
{
    *addr = rtk_swsoc->soc_base_addr;
    return 0;
}

int32_t
rtk_sw_swcoreBaseAddr_get(void __iomem **addr)
{
    *addr = rtk_swsoc->swcore_base_addr;
    return 0;
}

static const struct of_device_id rtk_sw_dt_match[] = {
	{ .compatible = "rtk,rtl9311b"},
	{ .compatible = "rtk,rtl9330"},
	{ },
};

MODULE_DEVICE_TABLE(of, rtk_sw_dt_match);

static struct platform_driver rtk_swsoc_driver = {
	.driver = {
		.name		= "rtk_sw",
		.owner		= THIS_MODULE,
		.of_match_table	= rtk_sw_dt_match,
	},
	.probe = rtk_sw_probe,
	.remove	= rtk_sw_remove,
};

module_platform_driver(rtk_swsoc_driver);

MODULE_DESCRIPTION("Realtek Switch SoC driver");
MODULE_LICENSE("GPL v2");

EXPORT_SYMBOL(bsp_mem32_read);
EXPORT_SYMBOL(bsp_mem32_write);
EXPORT_SYMBOL(rtk_sw_socBaseAddr_get);
EXPORT_SYMBOL(rtk_sw_swcoreBaseAddr_get);

