/*
 * PCIe host controller driver for Realtek rtl93xx SoCs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>
#include <linux/resource.h>
#include <linux/seq_file.h>
#include <linux/signal.h>
#include <linux/types.h>
#include <linux/reset.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/phy/phy.h>
#include <linux/uaccess.h>

//for avoid rt_type.h redefine linux/phy/phy.h
#define __COMMON_RT_TYPE_H__

#include "../../soc/realtek/chip_probe.h"
#include "../../soc/realtek/error.h"

#include "pcie-rtl93xx-dw.h"

//SYS regs
#define SYS_IPECR 0x1000600
#define EN_PCIE_RC          BIT(29)
#define EN_EPHY             BIT(0)
#define SYS_PPCR  0x1000508
#define EPHY_LANE1_EN       BIT(31)
#define EPHY_RX50_LINK      BIT(30)
#define EPHY_CMU_EN_REF_PLL BIT(27)
#define EPHY_CKRDY_LCCMU    BIT(23)
#define EPHY_PCLK_EN        BIT(22)
#define SYS_SSR1  0x1000040
#define PCIE_PHY_SOC_RST_N  BIT(1)
#define PCIE_MAC_SOC_RST_N  BIT(0)
#define SYS_PRCR  0x1000104
#define PCIE_SW_RST_N       0x1

//PCIe MAC regs
#define MAC_BASE_ADDR 0x01b20000
#define MAC_MDIO_C    0x0
#define MDIO_SRST     BIT(1)
#define MAC_INT       0x4
#define MAC_DWC_INFO  0x8
#define MAC_DWC_CFG   0xc
#define MAC_BDF       0x10
#define MAC_LTSSM_R   0x34

//MDIO cmd
#define MDIO_DATA_MASK 0xffff
#define MDIO_DATA      16
#define MDIO_ADDR_MASK 0x7f //bit[6] gen 1/2, bit[5] page, bit[4:0] reg
#define MDIO_REG_ADDR  6
#define MDIO_READY     BIT(2)
#define MDIO_WRITE     BIT(0)

#define SERDES_PHY_INIT		0
#define SERDES_PHY_AUTO_CAL	1

#define TOTAL_INTR_CNT      6
#define MAX_INTX_HOST_IRQS	4
#define INTX_START          2
#define INTX_END            (INTX_START + MAX_INTX_HOST_IRQS)

#define MAX_BER_POLL_COUNT	50
#define MAX_LINKUP_POLL_COUNT	500

#define MAX_LANE_NUM		2
#define MAX_NAME_LEN		16

#define to_rtl93xx_pcie(x)	container_of(x, struct rtl93xx_pcie, pp)

struct serdes_cfg {
	u32 addr;
	u32 val;
};

struct phy_data {
	int size;
	u8 *addr;
	u16 *val;
};

struct rtl93xx_pcie {
	void __iomem	*reg_base; /* ext reg */
	void __iomem	*serdes_base; /* serdes phy */
	phys_addr_t		dbi_start;
	phys_addr_t		dbi_end;

#ifdef DW_PCIE_V480
	phys_addr_t		iatu_unroll_start;
#endif

	struct clk		*bus_clk;
	struct pcie_port	pp;

	struct irq_domain	*intx_irq_domain;
	struct device		*dev;

	u32		idx;
	u8		lanes;
	bool	init;
	bool	do_link_rst;

	struct phy_data *phy_param;

	/* Device-Specific */
	/* linking up ready/stable time */
	u16		device_ready_time;
};

#ifdef DW_PCIE_V480
#define PCIE_GLBL_PCIE_CONTR_IATU_BASE_ADDR			0x48
#define RC_IATU_BASE_ADDR_MASK				  0xFFF80000
#endif

/* ltssm definition */
#define LTSSM_DETECT_QUIET	0x00
#define LTSSM_DETECT_ACT	0x01
#define LTSSM_POLL_ACTIVE	0x02
#define LTSSM_POLL_COMPLIANCE	0x03
#define LTSSM_POLL_CONFIG	0x04
#define LTSSM_PRE_DETECT_QUIET	0x05
#define LTSSM_DETECT_WAIT	0x06
#define LTSSM_CFG_LINKWD_START	0x07
#define LTSSM_CFG_LINKWD_ACEPT	0x08
#define LTSSM_CFG_LANENUM_WAI	0x09
#define LTSSM_CFG_LANENUM_ACEPT	0x0A
#define LTSSM_CFG_COMPLETE	0x0B
#define LTSSM_CFG_IDLE		0x0C
#define LTSSM_RCVRY_LOCK	0x0D
#define LTSSM_RCVRY_SPEED	0x0E
#define LTSSM_RCVRY_RCVRCFG	0x0F
#define LTSSM_RCVRY_IDLE	0x10
#define LTSSM_L0		0x11
#define LTSSM_L0S		0x12
#define LTSSM_L123_SEND_EIDLE	0x13
#define LTSSM_L1_IDLE		0x14
#define LTSSM_L2_IDLE		0x15
#define LTSSM_L2_WAKE		0x16
#define LTSSM_DISABLED_ENTRY	0x17
#define LTSSM_DISABLED_IDLE	0x18
#define LTSSM_DISABLED		0x19
#define LTSSM_ENTRY		0x1A
#define LTSSM_LPBK_ACTIVE	0x1B
#define LTSSM_LPBK_EXIT		0x1C
#define LTSSM_LPBK_EXIT_TIMEOUT	0x1D
#define LTSSM_HOT_RESET_ENTRY	0X1E
#define LTSSM_HOT_RESET		0x1F
#define LTSSM_RCVRY_EQ0		0x20
#define LTSSM_RCVRY_EQ1		0x21
#define LTSSM_RCVRY_EQ2		0x22
#define LTSSM_RCVRY_EQ3		0x23

#define LTSSM_R_DETECT_QUIET ((1 << LTSSM_DETECT_QUIET) | (1 << LTSSM_DETECT_ACT) | (1 << LTSSM_PRE_DETECT_QUIET))

const char *ltssm_str[] = {
	"DETECT_QUIET",
	"DETECT_ACT",
	"POLL_ACTIVE",
	"POLL_COMPLIANCE",
	"POLL_CONFIG",
	"PRE_DETECT_QUIET",
	"DETECT_WAIT",
	"CFG_LINKWD_START",
	"CFG_LINKWD_ACEPT",
	"CFG_LANENUM_WAI",
	"CFG_LANENUM_ACEPT",
	"CFG_COMPLETE",
	"CFG_IDLE",
	"RCVRY_LOCK",
	"RCVRY_SPEED",
	"RCVRY_RCVRCFG",
	"RCVRY_IDLE",
	"L0",
	"L0S",
	"L123_SEND_EIDLE",
	"L1_IDLE",
	"L2_IDLE",
	"L2_WAKE",
	"DISABLED_ENTRY",
	"DISABLED_IDLE",
	"DISABLED",
	"LPBK_ENTRY",
	"LPBK_ACTIVE",
	"LPBK_EXIT",
	"LPBK_EXIT_TIMEOUT",
	"HOT_RESET_ENTRY",
	"HOT_RESET",
	"RCVRY_EQ0",
	"RCVRY_EQ1",
	"RCVRY_EQ2",
	"RCVRY_EQ3"
};

static inline u32 rtl93xx_pcie_readl(struct rtl93xx_pcie *pcie, u32 reg);
static inline void rtl93xx_pcie_writel(struct rtl93xx_pcie *pcie, u32 val, u32 reg);

static int rtl93xx_pcie_device_reset(struct rtl93xx_pcie *rtl93xx_pcie);
static int rtl93xx_pcie_ltssm(struct pcie_port *pp);

static ssize_t rtl93xx_pcie_device_reset_store(struct device *dev,
        struct device_attribute *attr,
        const char *buf, size_t count)
{
	struct rtl93xx_pcie *rtl93xx_pcie = dev_get_drvdata(dev);

	rtl93xx_pcie_device_reset(rtl93xx_pcie);

	return count;
}
static DEVICE_ATTR(device_reset, S_IWUSR, NULL, rtl93xx_pcie_device_reset_store);

static struct attribute *rtl93xx_pcie_attributes[] = {
	&dev_attr_device_reset.attr,
	NULL
};

static const struct attribute_group rtl93xx_pcie_attr_group = {
	.attrs = rtl93xx_pcie_attributes,
};

static inline u32 rtl93xx_pcie_readl(struct rtl93xx_pcie *pcie, u32 reg)
{
	return readl(pcie->reg_base + reg);
}

static inline void rtl93xx_pcie_writel(struct rtl93xx_pcie *pcie, u32 val,
                                       u32 reg)
{
	writel(val, pcie->reg_base + reg);
}

static void rtk_sysreg_writel(u32 paddr, u32 val)
{
	void __iomem *addr;
	addr = ioremap(paddr, 4);
	if (addr) {
		writel(val, addr);
		iounmap(addr);
	} else {
		printk("ioremap(0x%08x) fail!\n", paddr);
	}
}

static int rtk_sysreg_readl(u32 paddr, u32 *val)
{
	void __iomem *addr;

	addr = ioremap(paddr, 4);
	if (addr) {
		*val = readl(addr);
		iounmap(addr);
		return 0;
	} else {
		return -ENOMEM;
	}
}

static u16 rtl93xx_pcie_phy_reg_read(struct rtl93xx_pcie *p, u8 addr)
{
	u32 val;
	u32 cnt;

	val = 0;
	val |= (addr & MDIO_ADDR_MASK) << MDIO_REG_ADDR;
	rtl93xx_pcie_writel(p, val, MAC_MDIO_C);

	usleep_range(1000, 2000);

	do {
		val = rtl93xx_pcie_readl(p, MAC_MDIO_C);
		cnt++;
	} while (!(val & MDIO_READY) && cnt < 1000);

	if (cnt >= 1000)
		printk("PHY read timeout\n");

	return (u16)((val >> MDIO_DATA) & MDIO_DATA_MASK);
}

static void rtl93xx_pcie_phy_reg_write(struct rtl93xx_pcie *p, u8 addr, u16 data)
{
	u32 val;

	val = 0;
	//data
	val |= (data & MDIO_DATA_MASK) << MDIO_DATA;
	val |= (addr & MDIO_ADDR_MASK) << MDIO_REG_ADDR;
	//write
	val |= MDIO_WRITE;

	rtl93xx_pcie_writel(p, val, MAC_MDIO_C);
	usleep_range(1000, 2000);
}

static int rtl93xx_pcie_device_reset(struct rtl93xx_pcie *rtl93xx_pcie)
{
	dev_info(rtl93xx_pcie->dev, "Assert PCIE Device Reset\n");
	rtk_sysreg_writel(SYS_PRCR, ~PCIE_SW_RST_N);
	usleep_range(1000, 2000);
	dev_info(rtl93xx_pcie->dev, "Deassert PCIe Device Reset\n");
	rtk_sysreg_writel(SYS_PRCR, PCIE_SW_RST_N);

	return 0;
}


static int rtl93xx_pcie_phy_patch(struct rtl93xx_pcie *p)
{
	int i;

	//do nothing if there is no patch to be done
	if (!p->phy_param)
		return 0;

	for (i = 0; i < p->phy_param->size; i++) {
		rtl93xx_pcie_phy_reg_write(p, p->phy_param->addr[i], p->phy_param->val[i]);
	}

	return 0;
}

static int rtl93xx_pcie_host_reset(struct rtl93xx_pcie *rtl93xx_pcie,
                                   int serdes_phase)
{
	u32 val;
	int ret, try;

	// 0. Assert PCIE Device Reset
	dev_info(rtl93xx_pcie->dev, "Assert PCIE Device Reset\n");
	rtk_sysreg_writel(SYS_PRCR, ~PCIE_SW_RST_N);

	// 1. PCIE MAC reset

	// disable then enable IP
	dev_info(rtl93xx_pcie->dev, "PCIE IP Reset\n");
	ret = rtk_sysreg_readl(SYS_IPECR, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get IPECR val\n");
		return ret;
	}
	val &= ~EN_PCIE_RC;
	rtk_sysreg_writel(SYS_IPECR, val);

	mb();

	ret = rtk_sysreg_readl(SYS_IPECR, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get IPECR val\n");
		return ret;
	}
	val |= EN_PCIE_RC;
	rtk_sysreg_writel(SYS_IPECR, val);
	usleep_range(1000, 2000);

	//trigger SOC MAC reset
	dev_info(rtl93xx_pcie->dev, "PCIe SoC MAC Reset\n");
	ret = rtk_sysreg_readl(SYS_SSR1, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get SSR1 val\n");
		return ret;
	}
	val &= ~PCIE_MAC_SOC_RST_N;
	rtk_sysreg_writel(SYS_SSR1, val);

	mb();

	ret = rtk_sysreg_readl(SYS_SSR1, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get SSR1 val\n");
		return ret;
	}
	val |= PCIE_MAC_SOC_RST_N;
	rtk_sysreg_writel(SYS_SSR1, val);
    usleep_range(1000, 2000);

	// 2. PCIE PHY control setup
	dev_info(rtl93xx_pcie->dev, "PHY control Reset\n");
	ret = rtk_sysreg_readl(SYS_PPCR, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get PPCR val\n");
		return ret;
	}
	val |= (EPHY_LANE1_EN | EPHY_RX50_LINK | EPHY_CMU_EN_REF_PLL);
	rtk_sysreg_writel(SYS_PPCR, val);

	// 3. PCIE phy mdio reset
	dev_info(rtl93xx_pcie->dev, "MDIO control Reset\n");
	val = rtl93xx_pcie_readl(rtl93xx_pcie, MAC_MDIO_C);
	val |= MDIO_SRST;
	rtl93xx_pcie_writel(rtl93xx_pcie, val, MAC_MDIO_C);
	usleep_range(10000, 20000);
	val &= ~(MDIO_SRST);
	rtl93xx_pcie_writel(rtl93xx_pcie, val, MAC_MDIO_C);
	usleep_range(10000, 20000);

	//4. write PHY parameters and reset PHY
	dev_info(rtl93xx_pcie->dev, "Update PHY parameters\n");
	rtl93xx_pcie_phy_patch(rtl93xx_pcie);

	try = 0;
	do {
		usleep_range(10000, 20000);
		ret = rtk_sysreg_readl(SYS_PPCR, &val);
		if (ret) {
			dev_err(rtl93xx_pcie->dev, "Unable to get PPCR val\n");
			return ret;
		}
		try++;
	} while (!(val & EPHY_CKRDY_LCCMU) && try < 3);

	if (try == 3) {
			dev_err(rtl93xx_pcie->dev, "EPHY clk not ready (PPCR: %0x)\n", val);
			return -ETIMEDOUT;
		}

	//trigger SOC PHY reset
	dev_info(rtl93xx_pcie->dev, "PCIe SoC PHY Reset\n");
	ret = rtk_sysreg_readl(SYS_SSR1, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get SSR1 val\n");
		return ret;
	}
	val &= ~PCIE_PHY_SOC_RST_N;
	rtk_sysreg_writel(SYS_SSR1, val);

	mb();

	ret = rtk_sysreg_readl(SYS_SSR1, &val);
	if (ret) {
		dev_err(rtl93xx_pcie->dev, "Unable to get SSR1 val\n");
		return ret;
	}
	val |= PCIE_PHY_SOC_RST_N;
	rtk_sysreg_writel(SYS_SSR1, val);
	usleep_range(1000, 2000);


	// 5. Deassert PCIE Device Reset
	dev_info(rtl93xx_pcie->dev, "Deassert PCIe Device Reset\n");
	rtk_sysreg_writel(SYS_PRCR, PCIE_SW_RST_N);

	return 0;
}

static int rtl93xx_pcie_host_setup(struct rtl93xx_pcie *rtl93xx_pcie)
{
	struct pcie_port *pp = &rtl93xx_pcie->pp;

	// Config DWC cfg
	dev_info(rtl93xx_pcie->dev, "Config DWC CFG\n");
	rtl93xx_pcie_writel(rtl93xx_pcie, 0x0, MAC_DWC_CFG);
	// Config PCIe tlp type
	// mw 0x01B20010 0x00007d10 // config pcie tlp type
	// Bus Number: 125
	// Device Number: 2
	// Function Number: 0
	dev_info(rtl93xx_pcie->dev, "Config MAC BDF\n");
	rtl93xx_pcie_writel(rtl93xx_pcie, 0x7d10, MAC_BDF);

#ifdef DW_PCIE_V480
	if (rtl93xx_pcie->iatu_unroll_start) {
		u32 base_addr = rtl93xx_pcie->iatu_unroll_start &
		                RC_IATU_BASE_ADDR_MASK;
	}
#endif

	dw_pcie_setup_rc(pp);

	return 0;
}

static int rtl93xx_pcie_establish_link(struct rtl93xx_pcie *rtl93xx_pcie)
{
	struct pcie_port *pp = &rtl93xx_pcie->pp;
	int count = 0;
	u32 ltssm;
	int rc;

	if (dw_pcie_link_up(pp)) {
		dev_err(pp->dev, "Link already up\n");
		return 0;
	}

	/* assert LTSSM enable */
	rtl93xx_pcie_writel(rtl93xx_pcie, 0x80, MAC_DWC_CFG);

	/* check if the link is up or not */
	while (!dw_pcie_link_up(pp)) {
		if (count == 0)
			dev_info(rtl93xx_pcie->dev, "Linking ...\n");

		usleep_range(1000, 2000);
		count++;

		if (count > MAX_LINKUP_POLL_COUNT) {
			ltssm = rtl93xx_pcie_ltssm(pp);
			dev_err(rtl93xx_pcie->dev,
			        "Link Fail(ltssm record = 0x%08x)\n",
			        ltssm);
			return (ltssm == LTSSM_R_DETECT_QUIET ?
			        -ENODEV : -EIO);
		}
	}

	if (rtl93xx_pcie->device_ready_time)
		msleep(rtl93xx_pcie->device_ready_time);

	/* Check link again for make sure the link is stable during init*/
	if (dw_pcie_link_up(pp)) {
		u8 speed, lanes;

		rc = dw_pcie_link_check(pp, &speed, &lanes);

		dev_info(rtl93xx_pcie->dev, "Link Up Gen%d x%d(%d)\n", speed,
		         lanes, count);
	} else {
		ltssm = rtl93xx_pcie_ltssm(pp);
		dev_info(rtl93xx_pcie->dev,
		         "Link Not Stable...Down(ltssm record = 0x%08x)\n",
		         ltssm);
		rc = -EAGAIN;
	}

	return rc;
}

static void	rtl93xx_pcie_msi_enable(struct rtl93xx_pcie *rtl93xx_pcie)
{
	return;
}

static void rtl93xx_pcie_intx_enable(struct rtl93xx_pcie *rtl93xx_pcie)
{
	struct pcie_port *pp = &rtl93xx_pcie->pp;
	int i;

#if 1
	return;
#else
	for (i = INTX_START; i < INTX_END; i++) {
		dev_info(rtl93xx_pcie->dev, "Enable IRQ(%d)\n", pp->irq[i]);
		enable_irq(pp->irq[i]);
	}

	return;
#endif
}

static void rtl93xx_pcie_intx_disable(struct rtl93xx_pcie *rtl93xx_pcie)
{
	struct pcie_port *pp = &rtl93xx_pcie->pp;
	int i;

#if 1
	return;
#else
	for (i = INTX_START; i < INTX_END; i++) {
		dev_info(rtl93xx_pcie->dev, "disable IRQ(%d)\n", pp->irq[i]);
		disable_irq_nosync(pp->irq[i]);
	}

	return;
#endif
}

static void rtl93xx_pcie_intx_nop(struct irq_data *irqd)
{
	return;
}

static void rtl93xx_pcie_mask_intx_irq(struct irq_data *irqd)
{
	struct rtl93xx_pcie *rtl93xx_pcie = irq_data_get_irq_chip_data(irqd);

	rtl93xx_pcie_intx_disable(rtl93xx_pcie);
}

static void rtl93xx_pcie_unmask_intx_irq(struct irq_data *irqd)
{
	struct rtl93xx_pcie *rtl93xx_pcie = irq_data_get_irq_chip_data(irqd);

	rtl93xx_pcie_intx_enable(rtl93xx_pcie);
}

static struct irq_chip rtl93xx_pcie_intx_irq_chip = {
	.name = "PCI-INTx",
	.irq_ack = rtl93xx_pcie_intx_nop,
	.irq_enable = rtl93xx_pcie_unmask_intx_irq,
	.irq_disable = rtl93xx_pcie_mask_intx_irq,
	.irq_mask = rtl93xx_pcie_mask_intx_irq,
	.irq_unmask = rtl93xx_pcie_unmask_intx_irq,
};

static int rtl93xx_pcie_intx_map(struct irq_domain *domain, unsigned int irq,
                                 irq_hw_number_t hwirq)
{
	irq_set_chip_and_handler(irq, &rtl93xx_pcie_intx_irq_chip,
	                         handle_simple_irq);
	irq_set_chip_data(irq, domain->host_data);

	return 0;
}

static const struct irq_domain_ops intx_domain_ops = {
	.map = rtl93xx_pcie_intx_map,
};

static int rtl93xx_pcie_intx_host_init(struct rtl93xx_pcie *rtl93xx_pcie,
                                       struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *node = dev->of_node;
	struct device_node *pcie_intc_node =  of_get_next_child(node, NULL);
	int ret;

	if (IS_ERR(pcie_intc_node)) {
		ret = PTR_ERR(pcie_intc_node);
		dev_err(dev, "No PCIe Intc node found(%d)\n", ret);
		return ret;
	}

	/*
	 * Add irq domain for INTx, which is used for PCI device IRQ parsing
	 * from device tree.
	 *
	 * PCI device gets its IRQ via pcibios_add_device() in arch/arm64/kernel/pci.c.
	 * This function assigns the irq returns by of_irq_parse_and_map_pci(), which
	 * decodes a PCI irq from the device tree and map to a virq.
	 */
	rtl93xx_pcie->intx_irq_domain =
	    irq_domain_add_linear(pcie_intc_node, MAX_INTX_HOST_IRQS,
	                          &intx_domain_ops, rtl93xx_pcie);
	if (IS_ERR(rtl93xx_pcie->intx_irq_domain)) {
		ret = PTR_ERR(rtl93xx_pcie->intx_irq_domain);
		dev_err(dev, "Failed to get a INTx IRQ domain(%d)\n", ret);
		return ret;
	}

	return 0;
}

static void rtl93xx_pcie_intx_host_exit(struct rtl93xx_pcie *rtl93xx_pcie)
{
	int i, irq;

	if (rtl93xx_pcie->intx_irq_domain) {
		/* added in pcibios_add_device() in arch/arm64/kernel/pci.c */
		for (i = 0; i < MAX_INTX_HOST_IRQS; i++) {
			irq = irq_find_mapping(rtl93xx_pcie->intx_irq_domain, i);
			if (irq > 0)
				irq_dispose_mapping(irq);
		}

		irq_domain_remove(rtl93xx_pcie->intx_irq_domain);

		rtl93xx_pcie->intx_irq_domain = NULL;
	}
}

static void rtl93xx_pcie_enable_interrupts(struct rtl93xx_pcie *rtl93xx_pcie)
{
	if (IS_ENABLED(CONFIG_PCI_MSI))
		rtl93xx_pcie_msi_enable(rtl93xx_pcie);
	else
		rtl93xx_pcie_intx_enable(rtl93xx_pcie);
}

const char *irq_str[] = {
	"pcie-msi",
	"pcie-link-rst",
	"pcie-int-a",
	"pcie-int-b",
	"pcie-int-c",
	"pcie-int-d",
};

static irqreturn_t rtl9311b_pcie_msi_isr(int irq, void *arg)
{
	struct rtl93xx_pcie *rtl93xx_pcie = arg;

	dw_handle_msi_irq(&rtl93xx_pcie->pp);

	return IRQ_HANDLED;
}

static irqreturn_t rtl9311b_pcie_link_rst_isr(int irq, void *arg)
{
	struct rtl93xx_pcie *rtl93xx_pcie = arg;
	int ltssm = rtl93xx_pcie_ltssm(&rtl93xx_pcie->pp);

	rtl93xx_pcie->pp.link_down = true;

	dev_info(rtl93xx_pcie->dev,
	         "Link Rst!!!(ltssm record = 0x%08x)\n",
	         ltssm);

	/* cleaer app_ltssm_enable */
	rtl93xx_pcie_writel(rtl93xx_pcie, 0x0, MAC_DWC_CFG);

	if (rtl93xx_pcie->do_link_rst)
		dw_pcie_host_link_notify(&rtl93xx_pcie->pp);

	return IRQ_HANDLED;
}

static irqreturn_t rtl9311b_pcie_inta_isr(int irq, void *arg)
{
	struct rtl93xx_pcie *rtl93xx_pcie = arg;
	struct irq_domain *intx_irq_domain = rtl93xx_pcie->intx_irq_domain;

#if 0
	printk("\n[%s] irq = %d, mapped irq = %d\n", __func__,
	       irq, irq_find_mapping(intx_irq_domain, 0));
	printk("    INT: 0x%08x\n", rtl93xx_pcie_readl(rtl93xx_pcie, MAC_INT));
#endif
	generic_handle_irq(irq_find_mapping(intx_irq_domain, 0));

	return IRQ_HANDLED;
}

static irqreturn_t rtl9311b_pcie_intb_isr(int irq, void *arg)
{
	struct rtl93xx_pcie *rtl93xx_pcie = arg;
	struct irq_domain *intx_irq_domain = rtl93xx_pcie->intx_irq_domain;

	generic_handle_irq(irq_find_mapping(intx_irq_domain, 1));

	return IRQ_HANDLED;
}

static irqreturn_t rtl9311b_pcie_intc_isr(int irq, void *arg)
{
	struct rtl93xx_pcie *rtl93xx_pcie = arg;
	struct irq_domain *intx_irq_domain = rtl93xx_pcie->intx_irq_domain;

	generic_handle_irq(irq_find_mapping(intx_irq_domain, 2));

	return IRQ_HANDLED;
}

static irqreturn_t rtl9311b_pcie_intd_isr(int irq, void *arg)
{
	struct rtl93xx_pcie *rtl93xx_pcie = arg;
	struct irq_domain *intx_irq_domain = rtl93xx_pcie->intx_irq_domain;

	generic_handle_irq(irq_find_mapping(intx_irq_domain, 3));

	return IRQ_HANDLED;
}

static irqreturn_t (*irq_handler[])(int, void *) = {
	rtl9311b_pcie_msi_isr,
	rtl9311b_pcie_link_rst_isr,
	rtl9311b_pcie_inta_isr,
	rtl9311b_pcie_intb_isr,
	rtl9311b_pcie_intc_isr,
	rtl9311b_pcie_intd_isr
};

static int rtl93xx_pcie_link_up(struct pcie_port *pp)
{
	struct rtl93xx_pcie *rtl93xx_pcie = to_rtl93xx_pcie(pp);
	u32 val;

	val = rtl93xx_pcie_readl(rtl93xx_pcie, MAC_DWC_INFO);
	val &= (1 << 5);
	rtl93xx_pcie->pp.link_down = val ? false : true;

	return val ? 1 :  0;
}

static int rtl93xx_pcie_ltssm(struct pcie_port *pp)
{
	struct rtl93xx_pcie *rtl93xx_pcie = to_rtl93xx_pcie(pp);
	u32 val;

	val = rtl93xx_pcie_readl(rtl93xx_pcie, MAC_LTSSM_R);

	return val;
}

__weak struct proc_dir_entry *realtek_proc;

static void show_usage(void)
{

	printk("	w [addr] [data]: write [value] to [reg] (data should be 16 bits, addr should be in [0x00-0x34])\n");
}

static ssize_t rtl93xx_pcie_phy_proc_write(struct file * file, const char __user * userbuf, size_t count, loff_t * off)
{
	char buf[32];
	int len;
	u8 addr;
	u16 data;
	struct rtl93xx_pcie *p = PDE_DATA(file_inode(file));

	len = min(sizeof(buf), count);
	if (copy_from_user(buf, userbuf, len))
		return -E2BIG;

	if (strncmp(buf, "help", 4) == 0) {
		show_usage();
	} else if (strncmp(buf, "w ", 2) == 0) {
		if( 2 == sscanf(buf, "w %hhx %hx", &addr, &data)) {
			rtl93xx_pcie_phy_reg_write(p, addr, data);
		} else {
			goto ERROR_PARA;
		}
	} else if (strncmp(buf, "r ", 2) == 0) {
		if(1 == sscanf(buf, "r %hhx", &addr)) {
			data = rtl93xx_pcie_phy_reg_read(p, addr);
			printk("%02x = %04x\n", addr, data);
		}
	} else {
		goto ERROR_PARA;
	}
	return count;

ERROR_PARA:
	printk("error parameter...\n");
	show_usage();
	return -EPERM;
}

static int rtl93xx_pcie_phy_show(struct seq_file *s, void *v)
{
	struct rtl93xx_pcie *p = s->private;
	u8 addr;
	u16 data;

	//seq_printf(s, "PCIe PHY version: %s\n\n", phy->ver);

	seq_printf(s, "Gen 1\n");
	seq_printf(s, "Page 0  \tPage 1\n");
	for (addr = 0x0; addr <= 0x14; addr++) {
		data = rtl93xx_pcie_phy_reg_read(p, addr);
		seq_printf(s, "%02x: %04x\t", addr, data);
		data = rtl93xx_pcie_phy_reg_read(p, addr+0x20);
		seq_printf(s, "%02x: %04x\n", addr, data);
	}

	for (addr = 0x15; addr <= 0x1f; addr++) {
		data = rtl93xx_pcie_phy_reg_read(p, addr);
		seq_printf(s, "%02x: %04x\n", addr, data);
	}

	seq_printf(s, "Gen 2\n");
	seq_printf(s, "Page 0  \tPage 1\n");
	for (addr = 0x0; addr <= 0x14; addr++) {
		data = rtl93xx_pcie_phy_reg_read(p, addr+0x40);
		seq_printf(s, "%02x: %04x\t", addr, data);
		data = rtl93xx_pcie_phy_reg_read(p, addr+0x60);
		seq_printf(s, "%02x: %04x\n", addr, data);
	}

	for (addr = 0x15; addr <= 0x1f; addr++) {
		data = rtl93xx_pcie_phy_reg_read(p, addr+0x40);
		seq_printf(s, "%02x: %04x\n", addr, data);
	}
	return 0;
}

static int rtl93xx_pcie_phy_open(struct inode *inode, struct file *file)
{
	return(single_open(file, rtl93xx_pcie_phy_show, PDE_DATA(inode)));
}


static const struct proc_ops fops_rtl93xx_pcie_phy = {
	.proc_open  		= rtl93xx_pcie_phy_open,
	.proc_write 		= rtl93xx_pcie_phy_proc_write,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= single_release,
};

static int rtl93xx_pcie_host_init(struct pcie_port *pp)
{
	struct rtl93xx_pcie *rtl93xx_pcie = to_rtl93xx_pcie(pp);
	int try_link = 0;
	int ret;

TRY_LINK:
	ret = rtl93xx_pcie_host_reset(rtl93xx_pcie, SERDES_PHY_INIT);
	if (ret)
		return ret;

	rtl93xx_pcie_host_setup(rtl93xx_pcie);
	ret = rtl93xx_pcie_establish_link(rtl93xx_pcie);
	if (((ret == -EIO) || (ret == -EAGAIN) || (ret == -EPERM)) &&
	        (try_link < 3)) {
		dev_info(rtl93xx_pcie->dev, "%s! retry(%d) ...\n",
		         ret == -EPERM ? "target miss" : "link fail",
		         try_link + 1);
		try_link++;
		goto TRY_LINK;
	}
	if (ret == -ENODEV || ret == -EIO)
		return ret;
	if (ret == -EPERM) /* not target link */
		ret = 0;

	/* A dummy function for showing the common driver framework */
	rtl93xx_pcie_enable_interrupts(rtl93xx_pcie);

	rtl93xx_pcie->init = true;
	rtl93xx_pcie->do_link_rst = true;

	return 0;
}

static struct pcie_host_ops rtl93xx_pcie_host_ops = {
	.link_up = rtl93xx_pcie_link_up,
	.host_init = rtl93xx_pcie_host_init,
};

static int __init rtl93xx_add_pcie_port(struct rtl93xx_pcie *rtl93xx_pcie,
                                        struct platform_device *pdev)
{
	struct pcie_port *pp = &rtl93xx_pcie->pp;
	struct device *dev = &pdev->dev;
	struct resource *res;
	int ret;
	int i;

	pp->dev = &pdev->dev;

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "rc_dbi");
	pp->dbi_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(pp->dbi_base)) {
		ret = PTR_ERR(pp->dbi_base);
		return ret;
	}
	dev_info(dev, "resource - %pr mapped at 0x%pK\n", res,
	         pp->dbi_base);
	rtl93xx_pcie->dbi_start = res->start;
	rtl93xx_pcie->dbi_end = res->end;

#ifdef DW_PCIE_V480
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "iatu_unroll");
	pp->iatu_unroll_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(pp->iatu_unroll_base)) {
		rtl93xx_pcie->iatu_unroll_start = 0;
	} else {
		dev_info(dev, "resource - %pr mapped at 0x%pK\n", res,
		         pp->iatu_unroll_base);
		rtl93xx_pcie->iatu_unroll_start = res->start;
	}
#endif

#if 1
	for (i = 0; i < TOTAL_INTR_CNT; i++) {
		pp->irq[i] = platform_get_irq(pdev, i);
		if (pp->irq[i] < 0)
			return pp->irq[i];

		dev_info(dev, "pcie irq[%d] is %d\n", i, pp->irq[i]);
		ret = devm_request_irq(dev, pp->irq[i], irq_handler[i],
		                       IRQF_NO_THREAD, irq_str[i], rtl93xx_pcie);

		if (ret) {
			dev_err(dev, "Failed to request irq(%d)\n", ret);
			return ret;
		}

	}
#endif

	ret = rtl93xx_pcie_intx_host_init(rtl93xx_pcie, pdev);

	if (ret < 0)
		return ret;

	pp->root_bus_nr = -1;
	pp->ops = &rtl93xx_pcie_host_ops;
	pp->lanes = rtl93xx_pcie->lanes;

#ifdef DW_PCIE_V480
	pp->num_viewport = 2;
#endif
	ret = dw_pcie_host_init(pp);
	if (ret) {
		dev_err(dev, "Failed to initialize host(%d)\n", ret);
		rtl93xx_pcie_intx_host_exit(rtl93xx_pcie);
		return ret;
	}
	return 0;
}

static int rtl93xx_phy_data_probe(struct device *dev, struct device_node *np, struct rtl93xx_pcie *p)
{
	struct phy_data *pd;
	int ret;
    uint32 chip_id, chip_rev_id;
    bool rtl9311b_a_cut_param = false;

    if (drv_swcore_cid_get((uint32)0, (uint32 *)&chip_id, (uint32 *)&chip_rev_id) != RT_ERR_OK)
    {
        dev_err(dev, "get chip id/rev fail ret=%x\n", ret);
        return -EPERM;
    }

    if ((drv_swcore_chipFamilyId_get(chip_id) == RTL9311B_FAMILY_ID) && (chip_rev_id == 0))
    {
        rtl9311b_a_cut_param = true;
        dev_info(dev, "Use A cut parameter\n");
    }

	pd = devm_kzalloc(dev, sizeof(*pd), GFP_KERNEL);
	if (IS_ERR(pd))
		return -ENOMEM;

    if (rtl9311b_a_cut_param)
    {
        ret = of_property_read_u32_index(np, "param_a_size", 0, &pd->size);
        if (ret)
            goto err;
    }
    else
    {
        ret = of_property_read_u32_index(np, "param_size", 0, &pd->size);
        if (ret)
            goto err;
    }

	pd->addr = devm_kzalloc(dev, sizeof(u8)*pd->size, GFP_KERNEL);
	if (!pd->addr)
		return -ENOMEM;

	pd->val = devm_kzalloc(dev, sizeof(u16)*pd->size, GFP_KERNEL);
	if (!pd->val)
		return -ENOMEM;

    if (rtl9311b_a_cut_param)
    {
        ret = of_property_read_u8_array(np, "param_a_addr",
                                        pd->addr, pd->size);
        if (ret)
            goto err;

        ret = of_property_read_u16_array(np, "param_a_val",
                                         pd->val, pd->size);
        if (ret)
            goto err;
    }
    else
    {
        ret = of_property_read_u8_array(np, "param_addr",
                                        pd->addr, pd->size);
        if (ret)
            goto err;

        ret = of_property_read_u16_array(np, "param_val",
                                         pd->val, pd->size);
        if (ret)
            goto err;
    }

	goto done;

err:
	dev_err(dev, "ERROR in PHY parameters probe (%d)\n", ret);
	pd->size = 0;

done:
	p->phy_param = pd;

	return ret;
}

static int __init rtl93xx_pcie_probe(struct platform_device *pdev)
{
	struct rtl93xx_pcie *rtl93xx_pcie;
	struct rtl93xx_pcie *dbg_pcie;
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct resource *reg_res;
	struct proc_dir_entry *e;
	int lanes, ret;

	rtl93xx_pcie = devm_kzalloc(dev, sizeof(*rtl93xx_pcie), GFP_KERNEL);
	if (!rtl93xx_pcie)
		return -ENOMEM;
	rtl93xx_pcie->dev = dev;

	reg_res = platform_get_resource_byname(pdev, IORESOURCE_MEM,
	                                       "ext_regs");

	rtl93xx_pcie->reg_base = devm_ioremap_resource(dev, reg_res);
	if (IS_ERR(rtl93xx_pcie->reg_base)) {
		ret = PTR_ERR(rtl93xx_pcie->reg_base);
		goto err_out;
	}
	dev_info(dev, "resource - %pr mapped at 0x%pK\n", reg_res,
	         rtl93xx_pcie->reg_base);

	if (of_property_read_u32(np, "id", &rtl93xx_pcie->idx)) {
		dev_err(dev, "missing id property\n");
		goto err_out;
	}
	dev_info(dev, "id %d\n", rtl93xx_pcie->idx);

	ret = of_property_read_u32(np, "num-lanes", &lanes);
	if (ret || (lanes < 1) || (lanes > MAX_LANE_NUM))
		rtl93xx_pcie->lanes = 1;
	else
		rtl93xx_pcie->lanes = lanes;
	dev_info(dev, "num-lanes %d\n", rtl93xx_pcie->lanes);

	ret = rtl93xx_phy_data_probe(dev, np, rtl93xx_pcie);
	if (ret)
		goto err_out;

	ret = of_property_read_u16(np, "ready-time",
	                           &rtl93xx_pcie->device_ready_time);
	if (ret)
		rtl93xx_pcie->device_ready_time = 0;

	platform_set_drvdata(pdev, rtl93xx_pcie);

	ret = rtl93xx_add_pcie_port(rtl93xx_pcie, pdev);
	if (ret < 0)
		goto err_out;

	ret = sysfs_create_group(&rtl93xx_pcie->dev->kobj,
	                         &rtl93xx_pcie_attr_group);

	if (ret) {
		dev_err(dev, "failed to register sysfs\n");
		return ret;
	}

	e = proc_create_data("pcie_phy", S_IRUGO | S_IWUSR, realtek_proc, &fops_rtl93xx_pcie_phy, rtl93xx_pcie);
	if (!e) {
		dev_err(dev, "failed to create procfs entry for PHY\n");
		return -EINVAL;
	}

	return 0;

err_out:
	/* create procfs for debug */
	dbg_pcie = kmalloc(sizeof(struct rtl93xx_pcie), GFP_KERNEL);
	dbg_pcie->reg_base = ioremap(MAC_BASE_ADDR, 4);
	if (!dbg_pcie->reg_base) {
		dev_err(dev, "failed to remap MAC base addr\n");
		return -ENOMEM;
	}

	e = proc_create_data("pcie_phy", S_IRUGO | S_IWUSR, realtek_proc, &fops_rtl93xx_pcie_phy, dbg_pcie);
	if (!e) {
		dev_err(dev, "failed to create procfs entry for PHY\n");
		return -EINVAL;
	}

	return ret;
}

static int __exit rtl93xx_pcie_remove(struct platform_device *pdev)
{
	struct rtl93xx_pcie *rtl93xx_pcie = platform_get_drvdata(pdev);
	struct pcie_port *pp = &rtl93xx_pcie->pp;

	sysfs_remove_group(&pdev->dev.kobj, &rtl93xx_pcie_attr_group);

	dw_pcie_host_exit(pp);

	rtl93xx_pcie_intx_host_exit(rtl93xx_pcie);

	rtl93xx_pcie->do_link_rst = false;
	rtk_sysreg_writel(SYS_PRCR, ~PCIE_SW_RST_N);

	return 0;
}

static void rtl93xx_pcie_shutdown(struct platform_device *pdev)
{
	struct rtl93xx_pcie *rtl93xx_pcie = platform_get_drvdata(pdev);

	rtl93xx_pcie->do_link_rst = false;
	rtk_sysreg_writel(SYS_PRCR, ~PCIE_SW_RST_N);
}

static const struct of_device_id rtl93xx_pcie_of_match[] = {
	{ .compatible = "realtek,9311b-pcie"},
	{},
};
MODULE_DEVICE_TABLE(of, rtl93xx_pcie_of_match);

static struct platform_driver rtl93xx_pcie_driver = {
	.probe = rtl93xx_pcie_probe,
	.remove		= __exit_p(rtl93xx_pcie_remove),
	.shutdown = rtl93xx_pcie_shutdown,
	.driver = {
		.name	= "9311b-pcie",
		.of_match_table = rtl93xx_pcie_of_match,
	},
};

static int __init rtl93xx_pcie_init(void)
{
	return platform_driver_register(&rtl93xx_pcie_driver);
}
late_initcall(rtl93xx_pcie_init);

static void __exit rtl93xx_pcie_exit(void)
{
	platform_driver_unregister(&rtl93xx_pcie_driver);
}
module_exit(rtl93xx_pcie_exit);

MODULE_DESCRIPTION("Realtek 9311B PCIe host controller driver");
MODULE_LICENSE("GPL v2");
