/*
 * Synopsys Designware PCIe host controller driver
 *
 * Copyright (C) 2013 Samsung Electronics Co., Ltd.
 *		http://www.samsung.com
 *
 * Author: Jingoo Han <jg1.han@samsung.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#ifndef _PCIE_DESIGNWARE_H
#define _PCIE_DESIGNWARE_H

/* Synopsis PCIe 4.80 support refer to kernel 4.11 drivers/pci/dwc */
//#define DW_PCIE_V480

/*
 * Maximum number of MSI IRQs can be 256 per controller. But keep
 * it 32 as of now. Probably we will never need more than 32. If needed,
 * then increment it in multiple of 32.
 */
#define MAX_MSI_IRQS			32
#define MAX_MSI_CTRLS			(MAX_MSI_IRQS / 32)

enum irq_type {
	IRQ_MSI = 0,
	IRQ_LINK_RST,
	IRQ_INTA,
	IRQ_INTB,
	IRQ_INTC,
	IRQ_INTD,
	IRQ_TYPE_CNT,
};

struct pcie_port {
	struct device		*dev;
	u8			root_bus_nr;
	void __iomem		*dbi_base;	/* rc */
	u64			cfg0_base;
	void __iomem		*va_cfg0_base;	/* device */
	u32			cfg0_size;
	u64			cfg1_base;
	void __iomem		*va_cfg1_base;	/* bridge */
	u32			cfg1_size;
	resource_size_t		io_base;
	phys_addr_t		io_bus_addr;
	u32			io_size;
	u64			mem_base;
	phys_addr_t		mem_bus_addr;
	u32			mem_size;
	struct resource		*cfg;
	struct resource		*io;
	struct resource		*mem;
	struct resource		*busn;
	int			irq[IRQ_TYPE_CNT];
	u32			lanes;
	struct pcie_host_ops	*ops;
	int			msi_irq;
	struct irq_domain	*irq_domain;
	unsigned long		msi_data;
	DECLARE_BITMAP(msi_irq_in_use, MAX_MSI_IRQS);

	bool			link_down;
	bool			forced_gen1;

	u8			target_gen;
	u8			target_lanes;

	struct pci_bus		*root_bus;
	struct pci_dev		*rc_pdev;
	struct work_struct	work;
	struct pci_host_bridge  *bridge;

	bool			recovery;

#ifdef DW_PCIE_V480
	void __iomem		*iatu_unroll_base;

	u32			num_viewport;
	u8			iatu_unroll_enabled;
#endif
};

struct pcie_host_ops {
	void (*readl_rc)(struct pcie_port *pp,
			 void __iomem *dbi_base, u32 *val);
	void (*writel_rc)(struct pcie_port *pp,
			  u32 val, void __iomem *dbi_base);
	int (*rd_own_conf)(struct pcie_port *pp, int where, int size, u32 *val);
	int (*wr_own_conf)(struct pcie_port *pp, int where, int size, u32 val);
	int (*rd_other_conf)(struct pcie_port *pp, struct pci_bus *bus,
			     unsigned int devfn, int where, int size, u32 *val);
	int (*wr_other_conf)(struct pcie_port *pp, struct pci_bus *bus,
			     unsigned int devfn, int where, int size, u32 val);
	int (*link_up)(struct pcie_port *pp);
	int (*host_init)(struct pcie_port *pp);
	void (*msi_set_irq)(struct pcie_port *pp, int irq);
	void (*msi_clear_irq)(struct pcie_port *pp, int irq);
	phys_addr_t (*get_msi_addr)(struct pcie_port *pp);
	u32 (*get_msi_data)(struct pcie_port *pp, int pos);
	void (*scan_bus)(struct pcie_port *pp);
	int (*msi_host_init)(struct pcie_port *pp, struct msi_controller *chip);
};

irqreturn_t dw_handle_msi_irq(struct pcie_port *pp);
void dw_pcie_msi_init(struct pcie_port *pp);
int dw_pcie_link_up(struct pcie_port *pp);
int dw_pcie_link_check(struct pcie_port *pp, u8 *speed, u8 *lanes);
int dw_pcie_host_link_notify(struct pcie_port *pp);
void dw_pcie_setup_rc(struct pcie_port *pp);
int dw_pcie_host_init(struct pcie_port *pp);
int dw_pcie_host_exit(struct pcie_port *pp);

#define PCI_DEVICE_ID_SYNOPSYS_PCIE_RC     0xabcd

void dw_dbg_set(int mode);

#endif /* _PCIE_DESIGNWARE_H */
