#include <linux/kernel.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/version.h>

#include <nor_rxi310_v1/nor_spif_core.h>
#if 1
#include <linux/module.h>
#include <linux/semaphore.h>
#include <linux/slab.h>
	#define ERASE_P _erase
	#define READ_P _read
	#define WRITE_P _write
	#define UNLOCK_P _unlock
	#define LOCK_P _lock
	#define SUSPEND_P _suspend
	#define RESUME_P _resume
	#define SYNC_P _sync
	#define POINT_P _point
	#define UNPOINT_P _unpoint
#endif

void __iomem *rxi310_flash_ctl_base;


#define MTD_DUMMY_FUNC(func_name) \
	static int luna_snof_mtd_##func_name(void) { \
		printk(KERN_WARNING "WW: %s: Invoked\n", \
					 __FUNCTION__); \
		/* dump_stack(); */ \
		return 0; \
	}
MTD_DUMMY_FUNC(lock);
MTD_DUMMY_FUNC(suspend);
MTD_DUMMY_FUNC(resume);
MTD_DUMMY_FUNC(destory);

#define LUNA_SNOF_G3_NULL_FUNC NULL


DEFINE_SEMAPHORE(snof_mutex);
#define SNOF_CNTLR_LOCK_INIT() do { sema_init(&snof_mutex, 1); } while (0)

#define SNOF_CNTLR_LOCK() do {									\
		down(&snof_mutex); \
	} while (0)
#define SNOF_CNTLR_UNLOCK() do { \
		up(&snof_mutex); \
	} while (0)


static int
sw_snof_mtd_erase(struct mtd_info *mtd, struct erase_info *instr) {
	unsigned long ofs, len;
	int ret;

	ofs = instr->addr;
	len = instr->len;

	SNOF_CNTLR_LOCK();
	ret = NORSF_ERASE(ofs, len, 0, 0);
	SNOF_CNTLR_UNLOCK();

	if (ret)
		return -EIO;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(4,17,1))
	instr->state = MTD_ERASE_DONE;

	instr->state = MTD_ERASE_DONE;
	mtd_erase_callback(instr);
#endif
	return 0;
}

#if 0
#include <linux/ctype.h>
void
print_data(size_t retlen, char *buf)
{
	int i;
	printk("\n--------(%d)\n", retlen);
	for (i = 0; i < retlen; i += 16) {
		   printk("%04X: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X | %c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c\n",
       i,
       buf[i+0],
       buf[i+1],
       buf[i+2],
       buf[i+3],
       buf[i+4],
       buf[i+5],
       buf[i+6],
       buf[i+7],
       buf[i+8],
       buf[i+9],
       buf[i+10],
       buf[i+11],
       buf[i+12],
       buf[i+13],
       buf[i+14],
       buf[i+15],
       isprint(buf[i+0] )?buf[i+0]  : ' ',
       isprint(buf[i+1] )?buf[i+1]  : ' ',
       isprint(buf[i+2] )?buf[i+2]  : ' ',
       isprint(buf[i+3] )?buf[i+3]  : ' ',
       isprint(buf[i+4] )?buf[i+4]  : ' ',
       isprint(buf[i+5] )?buf[i+5]  : ' ',
       isprint(buf[i+6] )?buf[i+6]  : ' ',
       isprint(buf[i+7] )?buf[i+7]  : ' ',
       isprint(buf[i+8] )?buf[i+8]  : ' ',
       isprint(buf[i+9] )?buf[i+9]  : ' ',
       isprint(buf[i+10])?buf[i+10] : ' ',
       isprint(buf[i+11])?buf[i+11] : ' ',
       isprint(buf[i+12])?buf[i+12] : ' ',
       isprint(buf[i+13])?buf[i+13] : ' ',
       isprint(buf[i+14])?buf[i+14] : ' ',
       isprint(buf[i+15])?buf[i+15] : ' '
       );
	}
	printk("\n--------\n");

}
#endif

static int
sw_snof_mtd_read(struct mtd_info *mtd, loff_t from, size_t len,
                              size_t *retlen, u_char *buf) {
	unsigned long ofs;
	int ret;

	ofs = from;
	if (retlen)
		*retlen = 0;

	SNOF_CNTLR_LOCK();
	ret = NORSF_READ(ofs, len, buf, 0);
	SNOF_CNTLR_UNLOCK();

	if (ret)
		return -EIO;

	if (retlen)
		*retlen = len;

	return ret;
}

static int
sw_snof_mtd_write(struct mtd_info *mtd, loff_t to, size_t len,
                               size_t *retlen, const u_char *buf) {
	unsigned long ofs;
	char rebuf[len];
	int ret;

	ofs = to;
	*retlen = 0;

	SNOF_CNTLR_LOCK();
	ret = NORSF_PROG(ofs, len, buf, 0);
	SNOF_CNTLR_UNLOCK();

	if (ret)
		return -EIO;

	if (retlen)
		*retlen = len;

#if 0 // check data correct
	sw_snof_mtd_read(mtd, to, len, NULL, rebuf);
	if(0 != memcmp(buf, rebuf, len)) {
		printk("[NOR] Write data error!\n")
		printk("[NOR] offset is %x, len is %x\n", to, len );
		printk("\nwrite data ----------------------\n",len, buf);
		print_data(len, buf);
		printk("\nread data ----------------------\n",len, buf);
		print_data(len, rebuf);
	}
#endif

	return ret;
}

static int
sw_snof_mtd_unlock(struct mtd_info *mtd, loff_t ofs, uint64_t len) {
	return 0;
}
static int luna_snof_erasesize(const norsf_info_t  *ni) {
	const norsf_erase_cmd_t *cmds = ni->cmd_info->cerase;
	int i = 0;
	int erasesize = 4096;

	for (i=0; i<ni->cmd_info->cerase_cmd_num; i++) {
		if (cmds[i].offset_lmt == 0) {
			erasesize = cmds[i].sz_b;
		}
	}
	return erasesize;
}



struct mtd_info *
snof_probe(struct map_info *map)
{
	struct mtd_info *mtd = NULL;
	rxi310_flash_ctl_base = map->virt;
	SNOF_CNTLR_LOCK_INIT();

	printk("SPI NOR FLASH driver\n");
	norsf_detect();

	mtd = kzalloc(sizeof(*mtd), GFP_KERNEL);
	if (!mtd) {
		printk(KERN_ERR "Failed to allocate memory for MTD device\n");
		return NULL;
	}

	mtd->priv = map;
	mtd->type = MTD_NORFLASH;
	mtd->flags = MTD_CAP_NORFLASH;
	mtd->name = map->name;
	mtd->writesize = 1;
	mtd->ERASE_P = sw_snof_mtd_erase;
	mtd->READ_P = sw_snof_mtd_read;
	mtd->WRITE_P = sw_snof_mtd_write;
	mtd->UNLOCK_P = sw_snof_mtd_unlock;
	mtd->LOCK_P = (typeof(mtd->LOCK_P))luna_snof_mtd_lock;
	mtd->SUSPEND_P = (typeof(mtd->SUSPEND_P))luna_snof_mtd_suspend;
	mtd->RESUME_P = (typeof(mtd->RESUME_P))luna_snof_mtd_resume;
	mtd->writebufsize = 256;

	mtd->SYNC_P = NULL;
	mtd->POINT_P = NULL;
	mtd->UNPOINT_P = NULL;

	mtd->erasesize = luna_snof_erasesize(&norsf_info);
	//printk("[NOR] erasesize is %d(%x)\n", mtd->erasesize, mtd->erasesize);
	if (mtd->erasesize != 4096) {
		printk(KERN_WARNING "\nWW: %s: Erase size is %d KB, make sure flash layout matched!\n",
		       __FUNCTION__,
		       mtd->erasesize / 1024);
	}
	mtd->numeraseregions = 0;


	mtd->size = (norsf_info.size_per_chip_b * norsf_info.num_chips);

	return mtd;
}


struct mtd_chip_driver sw_snof_g3_chipdrv = {
	.probe = snof_probe,
	.destroy = (typeof(sw_snof_g3_chipdrv.destroy))luna_snof_mtd_destory,
	.name = "map_rom",
	.module = THIS_MODULE
};


int __init sw_snof_init(void) {
	register_mtd_chip_driver(&sw_snof_g3_chipdrv);
	return 0;
}

void __exit sw_snof_exit(void) {
	unregister_mtd_chip_driver(&sw_snof_g3_chipdrv);
	return;
}

module_init(sw_snof_init);
module_exit(sw_snof_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Realtek Semiconductor Corp.");
MODULE_DESCRIPTION("NOR SPI-F Cntlr. Driver");
