#include <nor_spif_core.h>
#include <string.h>
#include <cross_env.h>

#if defined(__LUNA_KERNEL__) || defined(CONFIG_UNDER_UBOOT)
    #define NORSFG2_PROLOGUE_HINT()
    #define NORSFG2_EPILOGUE_HINT()
    #define NORSFG2_ACTION_PREFIX
    #define NORSFG2_PP_VERIFY_EN (0)
    #if defined(CONFIG_UNDER_UBOOT)
	#include <time.h>
	unsigned long timer_read_counter(void);
        #define GET_CPU_CYCLE() timer_read_counter()
    #else
        #define GET_CPU_CYCLE() jiffies
    #endif
#else
    #define NORSFG2_PROLOGUE_HINT() printf("NOTICE:  NOR SPI-F... ")
    #define NORSFG2_EPILOGUE_HINT() printf("done\n");
    #define NORSFG2_ACTION_PREFIX   "NOTICE:  "
    #define NORSFG2_PP_VERIFY_EN (0)
    static uint32_t get_timer_value(void) { return (uint32_t)(~read_cntpct_el0()); }
    #define GET_CPU_CYCLE() get_timer_value()
#endif

#if defined(__LUNA_KERNEL__)
    void schedule(void);
    #if(defined(CONFIG_DEFAULTS_KERNEL_3_18) && (CONFIG_DEFAULTS_KERNEL_3_18 == 1)||defined(CONFIG_DEFAULTS_KERNEL_4_4))
        DEFINE_SPINLOCK(spi_lock);
    #elif(defined(CONFIG_KERNEL_2_6_30) && (CONFIG_KERNEL_2_6_30 == 1))
        spinlock_t spi_lock = SPIN_LOCK_UNLOCKED;
    #else
        #error EE: UNKNOWN KERNEL version
    #endif
    unsigned long spi_lock_flags;
    #define INTERRUPT_DISABLE() do { \
            spin_lock_irqsave(&spi_lock, spi_lock_flags); \
        } while(0)
    #define INTERRUPT_ENABLE() do { \
            spin_unlock_irqrestore(&spi_lock, spi_lock_flags); \
        } while(0)
    #define OS_SCHEDULE_OUT() schedule()
    #undef RETURN_STACK_SAVE
    #undef RETURN_STACK_RESTORE
    #define RETURN_STACK_SAVE(_bpctl_backup)
    #define RETURN_STACK_RESTORE(_bpctl_backup)

static uint32_t us_to_c(uint32_t us) {
    const uint32_t us_per_tick = (1000 * 1000 / HZ);
    uint32_t tick = us / us_per_tick;
    return tick > 0? tick: 1;
}
#elif defined(CONFIG_UNDER_UBOOT)
    #define INTERRUPT_ENABLE()
    #define INTERRUPT_DISABLE()
    #define OS_SCHEDULE_OUT()
    unsigned long get_tbclk(void);
    #define GET_CPU_MHZ (get_tbclk()/1000000)
    static uint32_t us_to_c(uint32_t us) {
        return (us * GET_CPU_MHZ);
    }
#else
    #define INTERRUPT_ENABLE()
    #define INTERRUPT_DISABLE()
    #define OS_SCHEDULE_OUT()
    unsigned int plat_get_syscnt_freq2(void);
    #define GET_CPU_MHZ plat_get_syscnt_freq2
    static uint32_t us_to_c(uint32_t us) {
        return (us * GET_CPU_MHZ());
    }
#endif

#ifndef GET_CPU_CYCLE
    #error EE: Missing CPU cycle count retriever.
#endif

#ifndef NORSF_WBUF_LIM_B
    #define NORSF_WBUF_LIM_B 256
#endif

#if(NORSFG2_PP_VERIFY_EN == 1)
    #define NORSFG2_PP_VERIFY(src, dst, len) do { \
        norsf_int_memcmp(src, dst, len); \
    } while(0)
#else
    #define NORSFG2_PP_VERIFY(src, dst, len)
#endif

#define alloca(sz) __builtin_alloca(sz)
#define MEGABYTE(n) (n * 1024 * 1024)

#ifndef NOTICE
#define NOTICE printf
#endif

#define WAIT_FOR_CS() do { \
        SFCSR_T sfcsr_cs_mask = { .f.spi_csb0_sts = 1, \
                                  .f.spi_csb1_sts = 1 }; \
        SFCSR_T sfcsr_cur; \
        reg_soc_read(NOR_G2_SFCSRr, &sfcsr_cur.v); \
        while((sfcsr_cur.v & sfcsr_cs_mask.v) == sfcsr_cs_mask.v) { \
            reg_soc_read(NOR_G2_SFCSRr, &sfcsr_cur.v); \
        } \
    } while(0)

#define POS_ADVANCE(cs, offset, flash_sz_b, accum_b, res) do { \
        if(res < 0) goto t4_exception; \
        offset  += res; \
        accum_b += res; \
        if(offset >= flash_sz_b) { \
            cs++; \
            offset = 0; \
        } \
    } while(0)

#define NOP_DELAY(num) do { \
        uint32_t _num = (num); \
        while(_num--) { \
            asm volatile ("nop;\n"); \
        } \
    } while(0)


norsf_erase_cmd_t cmn_erase_cmds[] SECTION_NOR_SPIF_PARAM =
    {{.a.cmd       = 0xd8,
      .a.write_en  = 1,
      .a.pdir      = norsf_ptx,
      .a.cmode     = norsf_sio,
      .a.amode     = norsf_sio,
      .a.pmode     = norsf_sio,
      .sz_b        = 64*1024,
      .offset_lmt  = 0,
      .to_us       = -1},
     {.a.cmd       = 0x52,
      .a.write_en  = 1,
      .a.pdir      = norsf_ptx,
      .a.cmode     = norsf_sio,
      .a.amode     = norsf_sio,
      .a.pmode     = norsf_sio,
      .sz_b        = 32*1024,
      .offset_lmt  = 0,
      .to_us       = -1},
     {.a.cmd       = 0x20,
      .a.write_en  = 1,
      .a.pdir      = norsf_ptx,
      .a.cmode     = norsf_sio,
      .a.amode     = norsf_sio,
      .a.pmode     = norsf_sio,
      .sz_b        = 4*1024,
      .offset_lmt  = 0,
      .to_us       = -1}};

norsf_read_cmd_t cmn_read_cmd SECTION_NOR_SPIF_PARAM = {
    .a.cmd      = 0x03,
    .a.write_en = 0,
    .a.dummy_ck = 0,
    .a.pdir     = norsf_prx,
    .a.cmode    = norsf_sio,
    .a.amode    = norsf_sio,
    .a.pmode    = norsf_sio,
    .xread_en   = VZERO,
    .xread_ex   = VZERO,
};

norsf_cmd_info_t cmn_cmd_info SECTION_NOR_SPIF_PARAM = {
    .cerase = cmn_erase_cmds,

    .cerase_cmd_num = sizeof(cmn_erase_cmds)/sizeof(norsf_erase_cmd_t),

    .cprog_attr.cmd      = 0x02,
    .cprog_attr.pdir     = norsf_ptx,
    .cprog_attr.cmode    = norsf_sio,
    .cprog_attr.amode    = norsf_sio,
    .cprog_attr.pmode    = norsf_sio,
    .cprog_attr.write_en = 1,
    .cprog_attr.dummy_ck = 0,
    .cprog_to_us         = -1,
    .cprog_lim_b         = 256,

    .cread = &cmn_read_cmd,
};

norsf_info_t norsf_info SECTION_NOR_SPIF_COREDATA = {
    .num_chips      = 1,
    .addr_mode      = norsf_3b_addr,
    .sec_sz_b       = 4*1024, /* Set to the sector size for UBoot or Linux */
    .size_per_chip_b= 0,      /* Set to 0 for auto-size detection. */

    .arch_erase     = norsf_cmn_arch_erase,
    .arch_prog      = norsf_cmn_arch_prog,
    .arch_read      = norsf_cmn_arch_read,
    .arch_wip       = norsf_cmn_arch_wip,
    .pio_addr_len   = 2,
    .cmd_info       = &cmn_cmd_info,
};

#if(NORSFG2_PP_VERIFY_EN == 1)
SECTION_NOR_SPIF_MISC
static void norsf_int_memcmp(uint8_t *debug_src, uint8_t *debug_dst, uint32_t debug_len) {
    while(debug_len--) {
        if(*debug_src != *debug_dst) {
            printf("EE: debug_src: %p: %02x %02x %02x %02x *%02x* %02x %02x %02x %02x\n",
                   debug_src,
                   *(uint8_t *)(((uint32_t)debug_src - 4)),
                   *(uint8_t *)(((uint32_t)debug_src - 3)),
                   *(uint8_t *)(((uint32_t)debug_src - 2)),
                   *(uint8_t *)(((uint32_t)debug_src - 1)),
                   *(uint8_t *)(((uint32_t)debug_src)),
                   *(uint8_t *)(((uint32_t)debug_src + 1)),
                   *(uint8_t *)(((uint32_t)debug_src + 2)),
                   *(uint8_t *)(((uint32_t)debug_src + 3)),
                   *(uint8_t *)(((uint32_t)debug_src + 4)));
            printf("EE: debug_dst: %p: %02x %02x %02x %02x *%02x* %02x %02x %02x %02x\n",
                   debug_dst,
                   *(uint8_t *)(((uint32_t)debug_dst - 4)),
                   *(uint8_t *)(((uint32_t)debug_dst - 3)),
                   *(uint8_t *)(((uint32_t)debug_dst - 2)),
                   *(uint8_t *)(((uint32_t)debug_dst - 1)),
                   *(uint8_t *)(((uint32_t)debug_dst)),
                   *(uint8_t *)(((uint32_t)debug_dst + 1)),
                   *(uint8_t *)(((uint32_t)debug_dst + 2)),
                   *(uint8_t *)(((uint32_t)debug_dst + 3)),
                   *(uint8_t *)(((uint32_t)debug_dst + 4)));
            break;
        }
        debug_src++;
        debug_dst++;
    }
    return;
}
#endif

#define TX_MODE 0
#define RX_MODE 3
#define DISABLE_NOR_SPIC()  RMOD(SSIEN, spic_en, 0)
#define ENABLE_NOR_SPIC()   RMOD(SSIEN, spic_en, 1)
#define FIFO_EXTEND        (8) //RLXVM should be 0
#define WSIZE    (4)
#define HSIZE    (2)
#define BSIZE    (1)

int32_t norsf_cmn_arch_status(const uint32_t cs);
SECTION_NOR_SPIF_CORE
void norsf_wait_busy(const uint32_t cs) {
    while(1) {
        int32_t st = norsf_cmn_arch_status(cs);
        if(!(st&0x1)) break;
    }
}

SECTION_NOR_SPIF_CORE
static void norsf_check_ctrl_ready(void) {
    SR_T sr;

    sr.v = readl(RRVAL(SR));
    while(1) {
        sr.v = readl(RRVAL(SR));
        if(1==sr.f.txe) {
            //printf("EE: Transfer Error.");
            break;
        } else if(0==sr.f.busy) {
            break;
        }
    }
    return;
}

#ifndef AUTO_LENGTHdv
#define AUTO_LENGTHdv (0x28030000)
#endif

/* Tier-I function */
SECTION_NOR_SPIF_CORE
static int32_t norsf_single_cmd(uint32_t addr, uint8_t *payload, nsf_trx_attr_t attr) {
    volatile uint8_t *DR = (volatile uint8_t *)DR0ar;
    uint32_t addr_len_r, at_len_r;
    volatile uint32_t w_tx_fifo;
    const uint32_t fifo_len = (1<<RFLD(CTRLR2,fifo_entry_w_tx_fifo_entry))+FIFO_EXTEND;
    int res=0;
    uint32_t bpctl_backup __attribute__((unused));
    volatile uint8_t pf_dummy __attribute__((unused));

    INTERRUPT_DISABLE();
    RETURN_STACK_SAVE(bpctl_backup);

    /* Reset phase */
    addr_len_r = readl(RRVAL(ADDR_LENGTH));
    at_len_r = readl(RRVAL(AUTO_LENGTH));
    DISABLE_NOR_SPIC();

    writel(AUTO_LENGTHdv, RRVAL(AUTO_LENGTH));
    RMOD(CTRLR0, addr_ch, attr.amode,
                data_ch, attr.pmode,
                tmod, (attr.pdir == norsf_prx)?RX_MODE:TX_MODE);

    /* CS */
    writel(1 << attr.cs, RRVAL(SER));

    /* Write phase: cmd & addr */
    if(addr != -1) {
        uint32_t addr_len = norsf_info.pio_addr_len+1;
        RMOD(ADDR_LENGTH, addr_phase_length, addr_len&0x3);
        if(3==addr_len) {
            writel(__CPU_TO_BE32(attr.cmd << 24 | addr), RRVAL(DR0));
        } else if(4==addr_len) {
            writeb(attr.cmd, RRVAL(DR0));
            norsf_check_ctrl_ready();
            writel(__CPU_TO_BE32(addr), RRVAL(DR0));
        }
    } else { // cmd only
				writeb(attr.cmd, RRVAL(DR0));
    }

    /* Payload phase */
    if(payload != VZERO) {
        /* dummy cycle phase, if any. */
        if(attr.dummy_ck) {
            u32 baudr_v = readl(RRVAL(BAUDR));
            uint32_t dummy = attr.dummy_ck * baudr_v * 2;
            RMOD(AUTO_LENGTH, rd_dummy_length, dummy);
        } else {
            RMOD(AUTO_LENGTH, rd_dummy_length, 0);
        }

        uint32_t _len;
        /* continue demanded access. */
        if(attr.pdir == norsf_prx ) {
            if(attr.plen_b > fifo_len) {
                /* Enable data split */
                RMOD(CTRLR2, seq_en, 1);
            }

            writel(attr.plen_b, RRVAL(CTRLR1));
            ENABLE_NOR_SPIC();
            while(RFLD(SR, rfne)==0);	// polling RX FIFO util it is not empty

            while(attr.plen_b) {
                if(attr.plen_b >= WSIZE && 0==((uint64_t)payload&0x3)) {
										volatile uint32_t val_dr = readl(RRVAL(DR0));
										*(uint32_t *)(payload) =  __LE32_TO_CPU(val_dr);
                    _len = WSIZE;
                } else if(attr.plen_b >= HSIZE && 0==((uint64_t)payload&0x1)) {
										volatile uint16_t val_dr = readw(RRVAL(DR0));
                    *(uint16_t *)(payload) = __LE16_TO_CPU(val_dr);
                    _len = HSIZE;
                } else {
										volatile uint8_t val_dr = readb(RRVAL(DR0));
                    *payload = val_dr;
                    _len = BSIZE;
                }
                payload += _len;
                attr.plen_b -= _len;
            }
        } else {
            uint32_t fifo_cnt = fifo_len - (norsf_info.pio_addr_len+2);    // fifo count = total fifo len - (CMD + ADDR occupied)
            if(addr == -1) {    // for status register
                RMOD(ADDR_LENGTH, addr_phase_length, attr.plen_b);
            }

            uint32_t ds_en = 0;
            while(attr.plen_b) {
                uint32_t _plen;
                _plen = NORSF_MIN(attr.plen_b, fifo_cnt);
                if(_plen >= WSIZE && 0==((uint64_t)payload&0x3)) {
										writel(__CPU_TO_LE32(*(uint32_t *)(payload)), RRVAL(DR0));
                    _len = WSIZE;
                } else if(_plen >= HSIZE && 0==((uint64_t)payload&0x1)) {
										writew(__CPU_TO_LE16(*(uint16_t *)(payload)), RRVAL(DR0));
                    _len = HSIZE;
                } else {
										writeb(*(payload), RRVAL(DR0));
                    _len = BSIZE;
                }
                payload += _len;
                attr.plen_b -= _len;
                fifo_cnt -= _len;

                if(0==fifo_cnt){
                    if(ds_en==0) {  // program data split
                        RMOD(CTRLR2, seq_en, 1);
                        ENABLE_NOR_SPIC();
                        ds_en=1;
                    }
                    while(RFLD(SR, tfe)==0) { // wait fifo empty
                        norsf_check_ctrl_ready();
                    }
                    fifo_cnt = fifo_len;
                }
            }
						if (fifo_cnt != fifo_len) {
							ENABLE_NOR_SPIC();
							while(RFLD(SR, tfe)==0) { // wait fifo empty
								norsf_check_ctrl_ready();
							}
						}
        }
    } else {
        ENABLE_NOR_SPIC();
    }

    norsf_check_ctrl_ready();
    DISABLE_NOR_SPIC();
    RMOD(CTRLR2, seq_en, 0);
    writel(1, RRVAL(FLUSH_FIFO));

    /* Reset */
    writel( addr_len_r, RRVAL(ADDR_LENGTH));
    writel( at_len_r, RRVAL(AUTO_LENGTH));
    RETURN_STACK_RESTORE(bpctl_backup);
    INTERRUPT_ENABLE();
    return res;
}

#if !defined(__LUNA_KERNEL__) && !defined(CONFIG_UNDER_UBOOT)
SECTION_NOR_SPIF_CORE
static void norsf_int_configure_mmio_read(norsf_info_t *ni,
                                          const nsf_trx_attr_t cread) {
    uint32_t i;

    norsf_check_ctrl_ready();

    if(ni->addr_mode & norsf_4b_mode) {
        for(i=0; i<ni->num_chips; i++) {
            if(ni->is_mr4a) {
                ni->am.mode.arch_en4b(ni, i);
            } else {
                ni->am.mode.arch_ex4b(ni, i);
            }
        }
    }

#if(NORSF_QSPI_EN == 1)
    if(ni->cmd_info->cread->xread_en) {
        for(i=0; i<ni->num_chips; i++) {
            ni->cmd_info->cread->xread_en(ni, i);
        }
    }
#endif

    norsf_check_ctrl_ready();

    return;
}
#endif

#define ABS_DEV(a, b)   (((a)>(b))?((a)-(b)):((b)-(a)))
/* Tier-II function */
SECTION_NOR_SPIF_CORE
int32_t norsf_compound_cmd(const uint32_t addr,
                         uint8_t *payload,
                         const nsf_trx_attr_t attr,
                         const norsf_wip_info_t *wi) {
    uint32_t time_out, now_c;
    uint32_t res = 0;

    /* `write en' flow here assumes that all NOR SPI-F comes with the same style. */
    if(attr.write_en == 1) {
        nsf_trx_attr_t attr_wren = {
            .cs = attr.cs,
            .cmd  = 0x06,
            .plen_b = 0,
            .write_en = 0,
            .dummy_ck = 0,
            .pdir  = norsf_ptx,
            .cmode = norsf_sio,
            .amode = norsf_sio,
            .pmode = norsf_sio,
        };

        norsf_single_cmd(-1, VZERO, attr_wren);
    }

    res = norsf_single_cmd(addr, payload, attr);

    if(wi) {
        OS_SCHEDULE_OUT();
        now_c = time_out = GET_CPU_CYCLE();
        time_out += wi->to_c;
        while(wi->wip(attr.cs)) {
            if(time_after(now_c, time_out)) {
                return NORSFG2_T2_WIP_TIMEOUT;
            }
            OS_SCHEDULE_OUT();
            now_c = GET_CPU_CYCLE();
        }
    }

    return res;
}

#if defined(__LUNA_KERNEL__) || defined(CONFIG_UNDER_UBOOT)
extern norsf_probe_t *LS_start_of_norsf_pfunc[];
extern norsf_probe_t *LS_end_of_norsf_pfunc;
#endif

SECTION_NOR_SPIF_MISC
static uint32_t norsf_log2(uint32_t v) {
    uint32_t _v, f;

    if(v == 0) return 0;

    _v = v;
    f = 0;

    while(_v) {
        f++;
        _v = _v >> 1;
    }

    return f - 1;
}

SECTION_NOR_SPIF_MISC
static uint32_t norsf_addr_on_flash(const void *addr) {
    uint64_t _addr = (uint64_t)addr;
    return (_addr>=ORIN_FLASH_BASE) && (_addr<(ORIN_FLASH_BASE+ORIN_FLASH_SIZE));
}

#define VALID_RD_SINGLE     (1<<0)    // 1-1-1
#define VALID_RD_DUAL_O     (1<<1)    // 1-1-2
#define VALID_RD_DUAL_IO    (1<<2)    // 1-2-2
#define VALID_RD_QUAD_O     (1<<3)    // 1-1-4
#define VALID_RD_QUAD_IO    (1<<4)    // 1-4-4

SECTION_NOR_SPIF_MISC
int32_t norsf_configure_mmio_read(norsf_info_t *ni, uint32_t mmio32_en) {
    uint32_t id, sfsize_field;
    uint32_t valid;
    id = ni->id;

    //printf("%06x/", id);
    //printk("id is %06x/", id);
    if(ni->size_per_chip_b == 0) {
        ni->size_per_chip_b = 1 << (id & 0xff);
    }

    sfsize_field = norsf_log2(ni->size_per_chip_b) - 12;
    RMOD(FLASH_SIZE, flash_size, sfsize_field);

    if((mmio32_en && (ni->addr_mode & norsf_4b_mode)) ||
        (ni->addr_mode & norsf_4b_cmd)) {
        ni->is_mr4a = 1;
    } else {
        ni->is_mr4a = 0;
    }

    // configure Auto Mode (MMIO)
    RMOD(AUTO_LENGTH, auto_addr_length, (ni->is_mr4a)?0:3);
    uint32_t amode = ni->cmd_info->cread->a.amode;
    uint32_t pmode = ni->cmd_info->cread->a.pmode;
    uint32_t cmd = ni->cmd_info->cread->a.cmd;
    uint32_t dummy = ni->cmd_info->cread->a.dummy_ck;
    if(norsf_qio==pmode) {
        if(norsf_qio==amode) {    // 1-4-4
            valid = VALID_RD_QUAD_IO;
            RMOD(READ_QUAD_ADDR_DATA, read_quad_io_cmd, cmd);
        } else {    // 1-1-4
            valid = VALID_RD_QUAD_O;
            RMOD(READ_QUAD_DATA, read_quad_o_cmd, cmd);
        }
    } else if(norsf_dio==pmode) {
        if(norsf_dio==amode) {    // 1-2-2
            valid = VALID_RD_DUAL_IO;
            RMOD(READ_DUAL_ADDR_DATA, read_dual_io_cmd, cmd);
        } else {    // 1-1-2
            valid = VALID_RD_DUAL_O;
            RMOD(READ_DUAL_DATA, read_dual_o_cmd, cmd);
        }
    } else {
        valid = VALID_RD_SINGLE;
        RMOD(READ_FAST_SINGLE, frd_cmd, cmd);
    }
    u32 valid_cmd_v = readl(RRVAL(VALID_CMD));
    //valid = (RVAL(VALID_CMD) & ~(0x1F)) | valid;
    valid = (valid_cmd_v  & ~(0x1F)) | valid;
    //RVAL(VALID_CMD) = valid;
    writel( valid, RRVAL(VALID_CMD));
    if (norsf_sio == pmode){
        u32 fbaudr_v = readl(RRVAL(FBAUDR));
        //dummy = RVAL(FBAUDR)*2*dummy;
        dummy = fbaudr_v*2*dummy;
    }else{
        u32 baudr_v = readl(RRVAL(BAUDR));
        dummy = baudr_v*2*dummy;
    }
    RMOD(AUTO_LENGTH, rd_dummy_length, dummy);

    // Disable Write Auto Mode
    RMOD(WRITE_ENABLE, wr_en_cmd, 0);

    // Enable Software Reset
    PGM_RST_FIFO_1_T rst1 = { .f = {.cnt = 4, .sta = 1, .ch  = 0, .cmd = 0x66 }};
    PGM_RST_FIFO_2_T rst2 = { .f = {.cnt = 8, .sta = 1, .ch  = 0, .cmd = 0x99 }};
    writel( rst1.v, RRVAL(PGM_RST_FIFO_1));
    writel( rst2.v, RRVAL(PGM_RST_FIFO_2));

#if !defined(__LUNA_KERNEL__) && !defined(CONFIG_UNDER_UBOOT)
    printf("%dB-M%x%s ", (3+ni->is_mr4a), ni->addr_mode, NORSF_QSPI_EN?"Q":"D");

    norsf_int_configure_mmio_read(ni, ni->cmd_info->cread->a);
#endif
    IDCACHE_FLUSH();

    return 0;
}

SECTION_NOR_SPIF_MISC
int norsf_chip_num_det(uint32_t id) {
#if NORSF_CHIP_NUM > 1
    if(id == norsf_rdid(1)) {
        return 2;
    }
#endif
    return 1;
}

SECTION_NOR_SPIF_MISC
void norsf_detect(void) {
    writel(1, RRVAL(SER));

    NORSFG2_PROLOGUE_HINT();
#if defined(__LUNA_KERNEL__) || defined(CONFIG_UNDER_UBOOT)
    norsf_probe_t **pf = LS_start_of_norsf_pfunc;
    void *flash_info = VZERO;

    while(pf != &LS_end_of_norsf_pfunc) {
        flash_info = (void *)(*pf)();
        if(flash_info != VZERO) {
            memcpy(&norsf_info, flash_info, sizeof(norsf_info_t));
						printk("[NOR] flash chip found; id is %x\n", norsf_rdid(0));
            break;
        }
        pf++;
    }
#endif

    norsf_info.id = norsf_rdid(0);
    norsf_info.num_chips = norsf_chip_num_det(norsf_info.id);

    if(norsf_addr_on_flash(norsf_info.arch_wip) ||
        ((norsf_info.addr_mode & norsf_4b_mode) &&
         (norsf_addr_on_flash(norsf_info.am.mode.arch_en4b) ||
          norsf_addr_on_flash(norsf_info.am.mode.arch_ex4b))) ||
        (norsf_addr_on_flash(norsf_info.cmd_info->cread->xread_en) ||
         norsf_addr_on_flash(norsf_info.cmd_info->cread->xread_ex))) {
        puts("WW: arch_wip(), arch_en4b(), arch_ex4b(), xread_en(),"
             "and/or xread_ex() is on flash!\n");
    }

    if(((norsf_info.addr_mode & norsf_4b_mode) &&
         ((VZERO == norsf_info.am.mode.arch_en4b) ||
          (VZERO == norsf_info.am.mode.arch_ex4b))) ||
        ((norsf_info.addr_mode & norsf_4b_ear) &&
         (VZERO == norsf_info.am.ear.arch_wrear))) {
        puts("WW: 4B mode/ear selected but no arch_en/ex4b()/arch_wrear() functions defined!\n");
    }

    if(norsf_info.addr_mode & (norsf_4b_mode | norsf_4b_cmd)) {
			printk("[NOR] 4 bytes address\n");
        norsf_info.pio_addr_len = 3;
    } else {
			printk("[NOR] 3 bytes address\n");
        norsf_info.pio_addr_len = 2;
    }

    if((norsf_info.addr_mode & norsf_4b_ear) ||
        ((NORSF_MMIO_4B_EN == 0) && (norsf_info.addr_mode & norsf_4b_mode))) {
        norsf_info.dyn_4b_addr_switch = 1;
    } else {
        norsf_info.dyn_4b_addr_switch = 0;
    }

    norsf_configure_mmio_read(&norsf_info, NORSF_MMIO_4B_EN);

    NORSFG2_EPILOGUE_HINT();

    return;
}

SECTION_NOR_SPIF_MISC
static void show_progress(uint32_t done, uint32_t total, uint32_t verbose) {
    if(verbose) {
        printf("%3d%%\b\b\b\b", done*100/total);
    }
    return;
}

/* Common Tier-III functions. */
/* !!!! norsf_cmn_arch_wip() MUST NOT run from flash!!!! */
SECTION_NOR_SPIF_CORE
int32_t norsf_cmn_arch_wip(const uint32_t cs) {
    const nsf_trx_attr_t attr = {
        .cs       = cs,
        .cmd      = 0x05,
        .plen_b   = 1,
        .write_en = 0,
        .dummy_ck = 0,
        .pdir     = norsf_prx,
        .cmode    = norsf_sio,
        .amode    = norsf_sio,
        .pmode    = norsf_sio,
    };
    const uint8_t mask = 0x1;
    uint8_t payload;

    norsf_compound_cmd(-1, &payload, attr, 0);

    return ((payload & mask) == mask);
}

SECTION_NOR_SPIF_CORE
void norsf_cmn_arch_wren(const uint32_t cs) {
    const nsf_trx_attr_t attr = {
        .cs       = cs,
        .cmd      = 0x06,
        .plen_b   = 0,
        .write_en = 0,
        .dummy_ck = 0,
        .pdir     = norsf_ptx,
        .cmode    = norsf_sio,
        .amode    = norsf_sio,
        .pmode    = norsf_sio,
    };

    norsf_single_cmd(-1, VZERO, attr);

    return;
}

SECTION_NOR_SPIF_CORE
int32_t norsf_cmn_arch_status(const uint32_t cs) {
    const nsf_trx_attr_t attr = {
        .cs       = cs,
        .cmd      = 0x05,
        .plen_b   = 1,
        .write_en = 0,
        .dummy_ck = 0,
        .pdir     = norsf_prx,
        .cmode    = norsf_sio,
        .amode    = norsf_sio,
        .pmode    = norsf_sio,
    };
    uint8_t payload;

    norsf_compound_cmd(-1, &payload, attr, 0);

    return payload;
}

SECTION_NOR_SPIF_MISC
int32_t norsf_cmn_arch_erase(const norsf_erase_cmd_t *cmd,
                           const uint32_t cs,
                           const uint32_t offset,
                           const norsf_wip_info_t *wi) {
    int32_t res;
    nsf_trx_attr_t attr = cmd->a;
    attr.cs = cs;

    res = norsf_compound_cmd(offset, VZERO, attr, wi);

    res = (res < 0)? res: cmd->sz_b;

    return res;
}

SECTION_NOR_SPIF_MISC
int32_t norsf_cmn_arch_prog(const norsf_info_t *ni,
                          const uint32_t cs,
                          const uint32_t offset,
                          const uint32_t len,
                          const void *buf,
                          const norsf_wip_info_t *wi) {
    int32_t res;
    nsf_trx_attr_t attr = ni->cmd_info->cprog_attr;
    attr.cs     = cs;
    attr.plen_b = len;

    res = norsf_compound_cmd(offset, (void *)buf, attr, wi);

    res = (res < 0)? res: len;

    return res;
}

SECTION_NOR_SPIF_MISC
int32_t norsf_cmn_arch_read(const norsf_info_t *ni,
                          const uint32_t cs,
                          const uint32_t offset,
                          const uint32_t len,
                          void *buf) {
    uint32_t res=0;
    nsf_trx_attr_t attr = ni->cmd_info->cread->a;
    attr.cs = cs;
    attr.plen_b = len;

    do {    // split read
        attr.plen_b -= res;
        res = norsf_compound_cmd(offset+res, buf, attr, VZERO);
    } while(res);

    return len;
}

SECTION_NOR_SPIF_MISC
int32_t norsf_cmn_4b_mode_en(const norsf_info_t *ni, const uint32_t cs) {
    uint32_t res;
    nsf_trx_attr_t attr = {
        .cs       = cs,
        .cmd      = 0xb7,
        .write_en = 0,
        .dummy_ck = 0,
        .pdir     = norsf_ptx,
        .cmode    = norsf_sio,
        .amode    = norsf_sio,
        .pmode    = norsf_sio,
    };

    res = norsf_compound_cmd(-1, VZERO, attr, VZERO);

    return res;
}

SECTION_NOR_SPIF_CORE
int32_t norsf_cmn_4b_mode_ex(const norsf_info_t *ni, const uint32_t cs) {
    uint32_t res;
    nsf_trx_attr_t attr = {
        .cs       = cs,
        .cmd      = 0xe9,
        .write_en = 0,
        .dummy_ck = 0,
        .pdir     = norsf_ptx,
        .cmode    = norsf_sio,
        .amode    = norsf_sio,
        .pmode    = norsf_sio,
    };

    res = norsf_compound_cmd(-1, VZERO, attr, VZERO);

    return res;
}

SECTION_NOR_SPIF_MISC
int32_t norsf_rdid(const uint32_t cs) {

    uint32_t id;
    nsf_trx_attr_t attr = {
        .cs       = cs,
        .cmd      = 0x9f,
        .plen_b   = 3,
        .write_en = 0,
        .dummy_ck = 0,
        .pdir     = norsf_prx,
        .cmode    = norsf_sio,
        .amode    = norsf_sio,
        .pmode    = norsf_sio,
    };

    norsf_compound_cmd(-1, (uint8_t *)&id, attr, VZERO);
    return __BE32_TO_CPU(id) >> ((4 - attr.plen_b) * 8);
}

/* Tier-IV functions. */
SECTION_NOR_SPIF_MISC
static int32_t norsf_t4_epilogue(int32_t res,
                               const uint32_t cur_offset,
                               const uint32_t left_len,
                               const uint32_t cmd_offset,
                               const uint32_t accum_len,
                               const uint32_t verbose) {
    if(res > 0) {
        if(verbose) {
            printf("100%% ~ %08x/%d B\n", accum_len + cmd_offset - 1, accum_len);
        }
    } else {
        puts("EE: ");
        switch (res) {
        case NORSFG2_T2_WIP_TIMEOUT:
            printf("WIP() timeout");
            break;
        case NORSFG2_E4_OFFSET_NALIGN:
            printf("offset %08x is not aligned", cur_offset);
            break;
        case NORSFG2_E4_LEN_NALIGN:
            printf("erase size: %d B is not aligned", left_len);
            break;
        case NORSFG2_E4_BAD_CMD_AND_OFFSET:
            printf("can't erase %d B due to non-uniform sector", left_len);
            break;
        case NORSFG2_T4_LEN_OVERFLOW:
            printf("destination %08x overflows", left_len + cmd_offset);
            break;
        default:
            printf("unknown error: %d", res);
            break;
        }
        puts("\n");
    }
    return (res < 0);
}

SECTION_NOR_SPIF_MISC
static void norsf_addr_range_enable(const norsf_info_t *ni,
                                    const uint32_t cs,
                                    const uint32_t next_offset,
                                    norsf_usable_addr_t *ua_range) {
    uint32_t global_offset, bank_id;

    if(ni->dyn_4b_addr_switch) {
        global_offset = cs * ni->size_per_chip_b + next_offset;
        if((global_offset >= ua_range->start) &&
            (global_offset < ua_range->end)) {
            return;
        }

        ua_range->start = cs * ni->size_per_chip_b;

        if(ni->addr_mode == norsf_4b_mode) {
            ni->am.mode.arch_en4b(ni, cs);

            ua_range->end   = ua_range->start + ni->size_per_chip_b;
        } else {
            /* norsf_4b_ear */
            bank_id = next_offset/MEGABYTE(16);

            ni->am.ear.arch_wrear(ni, cs, bank_id);

            ua_range->start += bank_id * MEGABYTE(16);
            ua_range->end   = ua_range->start + MEGABYTE(16);
        }
    }

    return;
}

SECTION_NOR_SPIF_MISC
static void norsf_addr_range_restore(const norsf_info_t *ni) {
    uint32_t i;
    int32_t dyn_addr_mode = -1;

    if(ni->dyn_4b_addr_switch) {
        dyn_addr_mode = ni->addr_mode;
    }

    for(i=0; i<ni->num_chips; i++) {
        if(dyn_addr_mode == norsf_4b_mode) {
            ni->am.mode.arch_ex4b(ni, i);
        } else if(dyn_addr_mode == norsf_4b_ear) {
            ni->am.ear.arch_wrear(ni, i, 0);
        }
    }

    return;
}

SECTION_NOR_SPIF_MISC
static int32_t norsf_erase_cmd_sel(const norsf_info_t *ni,
                                 const uint32_t offset,
                                 const uint32_t len) {
    uint32_t i = 0;
    int res = NORSFG2_E4_UNKNOWN, offset_lmt;
    const norsf_erase_cmd_t *cmds = ni->cmd_info->cerase;

    for(i=0; i<ni->cmd_info->cerase_cmd_num; i++) {
        offset_lmt = (cmds[i].offset_lmt == 0 ? ni->size_per_chip_b : cmds[i].offset_lmt);
        if((((int32_t)offset) - offset_lmt) & ni->size_per_chip_b) {
            if((offset % cmds[i].sz_b) == 0) {
                res = i;
                if(len >= cmds[i].sz_b) {
                    break;
                }
            } else {
                res = NORSFG2_E4_OFFSET_NALIGN;
            }
        } else {
            /* res = NORSFG2_E4_BAD_CMD_AND_OFFSET; */
        }
    }

    return res;
}

SECTION_NOR_SPIF_MISC
int norsf_erase(const norsf_info_t *ni,
                const uint32_t offset,
                const uint32_t len,
                const uint32_t is_strict_len,
                const uint32_t verbose)
{
    const uint32_t nsf_sz_b = ni->size_per_chip_b;
    const norsf_erase_cmd_t *cmds = ni->cmd_info->cerase;
    norsf_wip_info_t wi;
    int32_t done_b = 0, res = 0;
    uint32_t next_offset = offset % nsf_sz_b;
    uint32_t cs = offset / nsf_sz_b;
    norsf_usable_addr_t ua_handler = {0};

    if((offset + len) > (nsf_sz_b * ni->num_chips)) {
        res = NORSFG2_T4_LEN_OVERFLOW;
        goto e4_end;
    }

    if(verbose) {
        printf(NORSFG2_ACTION_PREFIX "Erasing %d B from %08x... ", len, offset);
    }

    wi.wip = ni->arch_wip;

    while(done_b < len) {
        show_progress(done_b, len, verbose);
        res = norsf_erase_cmd_sel(ni, next_offset, len - done_b);
        if(res < 0) {
            break;
        } else {
            if((cmds[res].sz_b > len - done_b) && (is_strict_len)) {
                res = NORSFG2_E4_LEN_NALIGN;
                break;
            } else {
                /* 0 makes timeout become -1, i.e., 0xffff_ffff. */
                wi.to_c = us_to_c(cmds[res].to_us);

                norsf_addr_range_enable(ni, cs, next_offset, &ua_handler);

                res = ni->arch_erase(&cmds[res],
                                     cs,
                                     next_offset,
                                     &wi);
            }
        }

        POS_ADVANCE(cs, next_offset, nsf_sz_b, done_b, res);
    }

 t4_exception:
    norsf_addr_range_restore(ni);

    IDCACHE_FLUSH();
 e4_end:
    return norsf_t4_epilogue(res, next_offset, len - done_b, offset, done_b, verbose);
}

SECTION_NOR_SPIF_MISC
int norsf_prog(const norsf_info_t *ni,
               const uint32_t offset,
               const uint32_t len,
               const void *buf,
               const uint32_t verbose)
{
    const uint32_t nsf_sz_b = ni->size_per_chip_b;
    const norsf_cmd_info_t *cmd = ni->cmd_info;
    norsf_wip_info_t wi;
    uint16_t op_unit = ni->cmd_info->cprog_lim_b;
    int32_t done_b = 0, res = 0;
    uint32_t next_offset = offset % nsf_sz_b;
    uint32_t cs = offset / nsf_sz_b;
    uint32_t op_len;
    void *temp_buf = VZERO;
    void **buf_ptr;
    norsf_usable_addr_t ua_handler = {0};

#if(NORSFG2_PP_VERIFY_EN == 1)
    uint32_t debug_len = len;
    uint8_t *debug_src = (uint8_t *)buf;
    uint8_t *debug_dst = (uint8_t *)(NORSF_CFLASH_BASE + offset);
#endif

    if((offset + len) > (nsf_sz_b * ni->num_chips)) {
        res = NORSFG2_T4_LEN_OVERFLOW;
        goto p4_end;
    }

    buf_ptr = (void **)&buf;

    /* 0 makes timeout become -1, i.e., 0xffff_ffff. */
    wi.to_c = us_to_c(cmd->cprog_to_us);
    wi.wip = ni->arch_wip;

    op_unit = NORSF_MIN(op_unit, NORSF_WBUF_LIM_B);

    if(norsf_addr_on_flash(buf)) {
        temp_buf = alloca(op_unit);
        buf_ptr = &temp_buf;
    }

    if(verbose) {
        printf(NORSFG2_ACTION_PREFIX "Writing %d B from %p to %08x... ", len, buf, offset);
    }

    while(done_b < len) {
        op_len = NORSF_MIN((len - done_b), (op_unit - (next_offset % op_unit)));
        if(temp_buf) {
            memcpy(temp_buf, buf, op_len);
        }
        show_progress(done_b, len, verbose);

        norsf_addr_range_enable(ni, cs, next_offset, &ua_handler);

        res = ni->arch_prog(ni,
                            cs,
                            next_offset,
                            op_len,
                            *buf_ptr,
                            &wi);

        POS_ADVANCE(cs, next_offset, nsf_sz_b, done_b, res);
        buf += res;
    }

 t4_exception:
    norsf_addr_range_restore(ni);

    //IDCACHE_FLUSH();

#if(NORSFG2_PP_VERIFY_EN == 1)
    norsf_int_memcmp(debug_src, debug_dst, debug_len);
#endif

 p4_end:
    return norsf_t4_epilogue(res, next_offset, len - done_b, offset, done_b, verbose);
}

SECTION_NOR_SPIF_MISC
int norsf_read(const norsf_info_t *ni,
               const uint32_t offset,
               const uint32_t len,
               void *buf,
               const uint32_t verbose) {
    const nsf_trx_attr_t dummy_attr = { .plen_b = 1024 };
    const uint32_t op_unit = dummy_attr.plen_b;
    const uint32_t nsf_sz_b = ni->size_per_chip_b;
    uint32_t cs = offset / nsf_sz_b;
    uint32_t next_offset = offset % nsf_sz_b;
    uint32_t op_len;
    int32_t done_b = 0, res = 0;
    norsf_usable_addr_t ua_handler = {0};

    if((offset + len) > (nsf_sz_b * ni->num_chips)) {
        res = NORSFG2_T4_LEN_OVERFLOW;
        goto r4_end;
    }

    if(verbose) {
        printf(NORSFG2_ACTION_PREFIX "Reading %d B to %p from %08x... ", len, buf, offset);
    }

#ifdef MMIO_SPEEDUP
    if((offset + len) <= MEGABYTE(16)) {
        memcpy(buf, (const void *)((uint64_t)(ORIN_FLASH_BASE + offset)), len);
        res = len;
        done_b = len;
        goto r4_end;
    }
#endif

    while(done_b < len) {
        op_len = NORSF_MIN(NORSF_MIN((len - done_b), op_unit), (nsf_sz_b - next_offset));
        show_progress(done_b, len, verbose);

        norsf_addr_range_enable(ni, cs, next_offset, &ua_handler);
        res = ni->arch_read(ni, cs, next_offset, op_len, buf);
        POS_ADVANCE(cs, next_offset, nsf_sz_b, done_b, res);
        buf += res;
    }

t4_exception:
    norsf_addr_range_restore(ni);

 r4_end:
    return norsf_t4_epilogue(res, next_offset, len - done_b, offset, done_b, verbose);
}

int norsf_flashsize_get() {
    uint32_t id;
    rxi310_flash_ctl_base = ioremap(0x01020000, 4);
    writel(1, RRVAL(SER));
    #if defined(__LUNA_KERNEL__) || defined(CONFIG_UNDER_UBOOT)
        norsf_probe_t **pf = LS_start_of_norsf_pfunc;
        void *flash_info = VZERO;

        while(pf != &LS_end_of_norsf_pfunc) {
            flash_info = (void *)(*pf)();
            if(flash_info != VZERO) {
                memcpy(&norsf_info, flash_info, sizeof(norsf_info_t));
                break;
            }
            pf++;
        }
    #endif
    id = norsf_info.id;
    if(norsf_info.size_per_chip_b == 0) {
        norsf_info.size_per_chip_b = 1 << (id & 0xff);
    }
    return (norsf_info.size_per_chip_b * norsf_info.num_chips);
}
EXPORT_SYMBOL(norsf_flashsize_get);
