

#include <linux/mtd/spi-nand.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/nand_bbt.h>
#include <linux/slab.h>
#include "snand_dawu_v1/spi_nand_core.h"
#include "snand_dawu_v1/spi_nand_common.h"
#include "ecc_dawu_v1/ecc.h"
#include <flash/flash_partition.h>
#include <uapi/mtd/mtd-abi.h>


#define size64(a) ((((a)+255)>>8)<<8)

struct sw_spi_nand_chip{
	struct spi_nand_chip chip;
	uint64_t gap;
	snaf_info_t *info;
};

void __iomem *flash_ctl_base;
void static __iomem *flash_mem_base;
extern int add_mtd_partitions(struct mtd_info *, const struct mtd_partition *, int);

/**
*  Default OOB area specification layout
*/
static struct nand_ecclayout_user micron_ecc_layout_64 = {
	.eccbytes = 32,
	.eccpos = {
		   8, 9, 10, 11, 12, 13, 14, 15,
		   24, 25, 26, 27, 28, 29, 30, 21,
		   40, 41, 42, 43, 44, 45, 46, 47,
		   56, 57, 58, 59, 60, 61, 62, 63},
	.oobavail = 30,
	.oobfree = {
		{.offset = 2,
		 .length = 6},
		{.offset = 16,
		 .length = 8},
		{.offset = 32,
		 .length = 8},
		{.offset = 48,
		 .length = 8}, }
};

static struct nand_ecclayout_user mxic_ecc_layout_64 = {
	.eccbytes = 14,
	.eccpos = { 2, 3, 16, 17, 18, 19, 32, 33, 34, 35, 48, 49, 50, 51},
	.oobavail = 48,
	.oobfree = {
		{.offset = 4,
		 .length = 12},
		{.offset = 20,
		 .length = 12},
		{.offset = 36,
		 .length = 12},
		{.offset = 52,
		 .length = 12}, }
};

static struct nand_ecclayout_user micron_ecc_layout_128 = {
	.eccbytes = 64,
	.eccpos = {
		64, 65, 66, 67, 68, 69, 70, 71,
		72, 73, 74, 75, 76, 77, 78, 79,
		80, 81, 82, 83, 84, 85, 86, 87,
		88, 89, 90, 91, 92, 93, 94, 95,
		96, 97, 98, 99, 100, 101, 102, 103,
		104, 105, 106, 107, 108, 109, 110, 111,
		112, 113, 114, 115, 116, 117, 118, 119,
		120, 121, 122, 123, 124, 125, 126, 127},
	.oobavail = 62,
	.oobfree = {
		{.offset = 2,
		 .length = 62}, }
};

static struct nand_ecclayout_user gd5f_ecc_layout_64 = {
	.eccbytes = 64,
	.eccpos = {
		64, 65, 66, 67, 68, 69, 70, 71,
		72, 73, 74, 75, 76, 77, 78, 79,
		80, 81, 82, 83, 84, 85, 86, 87,
		88, 89, 90, 91, 92, 93, 94, 95,
		96, 97, 98, 99, 100, 101, 102, 103,
		104, 105, 106, 107, 108, 109, 110, 111,
		112, 113, 114, 115, 116, 117, 118, 119,
		120, 121, 122, 123, 124, 125, 126, 127},
	.oobavail = 63,
	.oobfree = { {1, 63} }
};

#ifndef CONFIG_NAND_HIDE_BAD
#define CONFIG_NAND_HIDE_BAD 1
#endif

#ifdef CONFIG_NAND_HIDE_BAD
#define BAD_BLOCK_LIST_ELEMENT	128
#define BBL0_NAME 0x42626c30 //'Bbl0'
#define BBL1_NAME 0x316C6242 //'1lbB
enum {
	FACTORY_1ST_SCAN = 1,
	FACTORY_FROM_BBT0= 2,
	FACTORY_FROM_BBT1= 3,
};
#endif

#ifdef CONFIG_NAND_HIDE_BAD
#define BBL_INDICATORS          4
#define VENUS_CONFIG_BBL_OFFSET         (0x180000)  /* 1.5MB */
#define VENUS_CONFIG_BBL_OFFSET_REDUND  (0x1C0000)
static u8 bbl_source=0;
static u8 acc_phy_offset=0;
static u8 bad_block_cnt=0;
static uint32_t bad_block_list[BAD_BLOCK_LIST_ELEMENT];

static int spi_nand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip);

static int xlat_offset_to_skip_factory_bad(struct spi_nand_chip *chip, uint32_t *page_addr)
{
	uint32_t i;
	uint32_t offset;
	/* Just return original offset:
		 1. if no bad block found
		 2. Physical flash offset access is assigned */
	if((bad_block_cnt==0) || (acc_phy_offset==1)){
		return 0;
	}

	offset = *page_addr<<chip->page_shift;
	for(i=0; i<bad_block_cnt; i++){
		if(bad_block_list[i] <= offset){
			offset += chip->block_size;
		}else{
		}
	}
	if(offset >= chip->size){
		return -EINVAL;
	}
	*page_addr = offset>>chip->page_shift;
	return 0;
}

#endif
/**
 * spi_nand_get_device - [GENERIC] Get chip for selected access
 * @mtd: MTD device structure
 * @new_state: the state which is requested
 *
 * Get the device and lock it for exclusive access
 */
static int spi_nand_get_device(struct mtd_info *mtd, int new_state)
{
	struct spi_nand_chip *this = mtd->priv;
	DECLARE_WAITQUEUE(wait, current);

	/*
	 * Grab the lock and see if the device is available
	 */
	while (1) {
		spin_lock(&this->chip_lock);
		if (this->state == FL_READY) {
			this->state = new_state;
			spin_unlock(&this->chip_lock);
			break;
		}
		if (new_state == FL_PM_SUSPENDED) {
			spin_unlock(&this->chip_lock);
			return (this->state == FL_PM_SUSPENDED) ? 0 : -EAGAIN;
		}
		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&this->wq, &wait);
		spin_unlock(&this->chip_lock);
		schedule();
		remove_wait_queue(&this->wq, &wait);
	}
	return 0;
}

static inline int spi_nand_issue_cmd(struct spi_nand_chip *chip,
				struct spi_nand_cmd *cmd)
{
	return chip->command_fn(chip, cmd);
}

/* "crc16_tab & cyg_crc16" are Inherited from u-boot-2015.04 */
/* Table of CRC constants - implements x^16+x^12+x^5+1 */
static const uint16_t crc16_tab[] = {
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7,
    0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,
    0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6,
    0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de,
    0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485,
    0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,
    0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4,
    0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc,
    0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823,
    0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b,
    0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12,
    0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a,
    0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41,
    0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49,
    0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70,
    0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78,
    0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f,
    0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067,
    0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e,
    0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256,
    0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d,
    0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
    0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c,
    0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634,
    0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,
    0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3,
    0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,
    0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92,
    0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9,
    0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,
    0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,
    0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0,
};
static uint16_t cyg_crc16(unsigned char *buf, int len)
{
    int i;
    uint16_t cksum;

    cksum = 0;
    for (i = 0;  i < len;  i++) {
		cksum = crc16_tab[((cksum>>8) ^ *buf++) & 0xFF] ^ (cksum << 8);
    }
    return cksum;
}



static void get_spi_nand_factory_bad_blocks(struct mtd_info *mtd, uint32_t start, uint32_t size)
{
    struct spi_nand_chip *chip = mtd->priv;

    /* Check factory bad stored in flash chip:
         * 1) Read Bbl0 if the factory bad had already been scanned
         * 2) If Bbl0 is bad. Read Bbl1 for the factory bad
         */
    loff_t offset;
    loff_t bbl_off;
    uint32_t ret=0;
    size_t retlen;
    uint32_t indicator[BBL_INDICATORS];
    uint32_t tempread[BAD_BLOCK_LIST_ELEMENT];
    memset(indicator, 0xF, sizeof(indicator));
    memset(tempread, 0xF, sizeof(tempread));

#if 0
    if (IS_TAURUS) {
        bbl_off = TAURUS_CONFIG_BBL_OFFSET;
    } else {
        bbl_off = VENUS_CONFIG_BBL_OFFSET;
    }
#endif
		bbl_off = VENUS_CONFIG_BBL_OFFSET;

    acc_phy_offset = 1;
    for(offset=bbl_off ; offset < chip->size ; offset += chip->block_size){
        ret = spi_nand_block_bad(mtd, offset, 1);
        if(ret == 1) continue;
        mtd->_read(mtd, offset, sizeof(indicator), &retlen, (u8 *)indicator);

        if(((indicator[0] == BBL0_NAME) || (indicator[0] == BBL1_NAME)) && (indicator[2] == indicator[3])){
            ret = mtd->_read(mtd, offset+chip->page_size, sizeof(tempread), &retlen, (u8 *)tempread);
            if(indicator[2] == cyg_crc16((u8 *)tempread, sizeof(tempread))){
                bad_block_cnt = indicator[1];
                bbl_source = (indicator[0]==BBL0_NAME)?FACTORY_FROM_BBT0:FACTORY_FROM_BBT1;
                memcpy(bad_block_list, tempread, sizeof(bad_block_list));
                acc_phy_offset = 0;
                return;
            }
        }
    }

    /* If the BBLn is not read from flash, scan the BBI in OOB */
    offset = start;
    bbl_source = FACTORY_1ST_SCAN;
    bad_block_cnt = 0;
    memset(bad_block_list, 0xF, sizeof(bad_block_list));
    while(((offset - start) < size) && (bad_block_cnt<BAD_BLOCK_LIST_ELEMENT)) {
         ret = spi_nand_block_bad(mtd, offset, 1);
         if(ret == 1){
            bad_block_list[bad_block_cnt++] = offset;
        }
        offset += chip->block_size;
    }
    mtd->size -= bad_block_cnt*chip->block_size;
    acc_phy_offset = 0;
    return;
}

/**
 * spi_nand_release_device - [GENERIC] release chip
 * @mtd: MTD device structure
 *
 * Deselect, release chip lock and wake up anyone waiting on the device
 */
static void spi_nand_release_device(struct mtd_info *mtd)
{
	struct spi_nand_chip *this = mtd->priv;

	/* Release the chip */
	spin_lock(&this->chip_lock);
	this->state = FL_READY;
	wake_up(&this->wq);
	spin_unlock(&this->chip_lock);
}

static int spi_nand_manufacture_init(struct spi_nand_chip *chip)
{

	switch (chip->mfr_id) {
	case SPI_NAND_MFR_TOSHIBA:
	case SPI_NAND_MFR_MICRON:
	case SPI_NAND_MFR_ETRON:
	case SPI_NAND_MFR_WINBOND:
		//chip->get_ecc_status = spi_nand_mt29f_ecc_status;

		if (chip->oob_size == 64)
			chip->ecclayout = &micron_ecc_layout_64;
		else if (chip->oob_size == 128)
			chip->ecclayout = &micron_ecc_layout_128;

		break;
	case SPI_NAND_MFR_MACRONIX:
		//chip->get_ecc_status = spi_nand_mt29f_ecc_status;
		if (chip->oob_size == 64)
			chip->ecclayout = &mxic_ecc_layout_64;
		break;
	case SPI_NAND_MFR_ESMT_GD_ZENTEL:
		//chip->get_ecc_status = spi_nand_gd5f_ecc_status;
		chip->ecc_strength = 8;
		if (chip->oob_size == 64)
			chip->ecclayout = &gd5f_ecc_layout_64;

		break;
	default:
		//chip->get_ecc_status = spi_nand_mt29f_ecc_status;
		chip->ecclayout = &micron_ecc_layout_64;
		break;
	}
	return 0;
}



/**
 * spi_nand_read_reg - send command 0Fh to read register
 * @chip: SPI-NAND device structure
 * @reg; register to read
 * @buf: buffer to store value
 */
static int spi_nand_read_reg(struct spi_nand_chip *chip,
			u8 reg, u8 *buf)
{
	struct spi_nand_cmd cmd;
	int ret;

	memset(&cmd, 0, sizeof(struct spi_nand_cmd));
	cmd.cmd = SPI_NAND_CMD_GET_FEATURE;
	cmd.n_addr = 1;
	cmd.addr[0] = reg;
	cmd.n_rx = 1;
	cmd.rx_buf = buf;

	ret = spi_nand_issue_cmd(chip, &cmd);
	if (ret < 0)
		pr_err("err: %d read register %d\n", ret, reg);

	return ret;
}

/**
 * spi_nand_write_reg - send command 1Fh to write register
 * @chip: SPI-NAND device structure
 * @reg; register to write
 * @buf: buffer stored value
 */
static int spi_nand_write_reg(struct spi_nand_chip *chip,
			u8 reg, u8 *buf)
{
	struct spi_nand_cmd cmd;
	int ret;

	memset(&cmd, 0, sizeof(struct spi_nand_cmd));
	cmd.cmd = SPI_NAND_CMD_SET_FEATURE;
	cmd.n_addr = 1;
	cmd.addr[0] = reg;
	cmd.n_tx = 1,
	cmd.tx_buf = buf,

	ret = spi_nand_issue_cmd(chip, &cmd);
	if (ret < 0)
		pr_err("err: %d write register %d\n", ret, reg);

	return ret;
}

/**
 * spi_nand_read_status - get status register value
 * @chip: SPI-NAND device structure
 * @status: buffer to store value
 * Description:
 *   After read, write, or erase, the Nand device is expected to set the
 *   busy status.
 *   This function is to allow reading the status of the command: read,
 *   write, and erase.
 *   Once the status turns to be ready, the other status bits also are
 *   valid status bits.
 */
static int spi_nand_read_status(struct spi_nand_chip *chip, u8 *status)
{
	return spi_nand_read_reg(chip, REG_STATUS, status);
}



/**
 * spi_nand_transfer_oob - transfer oob to client buffer
 * @chip: SPI-NAND device structure
 * @oob: oob destination address
 * @ops: oob ops structure
 * @len: size of oob to transfer
 */
static void spi_nand_transfer_oob(struct spi_nand_chip *chip, u8 *oob,
				  struct mtd_oob_ops *ops, size_t len)
{
	switch (ops->mode) {

	case MTD_OPS_PLACE_OOB:
	case MTD_OPS_RAW:
		memcpy(oob, chip->oobbuf + ops->ooboffs, len);
		return;

	case MTD_OPS_AUTO_OOB: {
		struct nand_oobfree *free = chip->ecclayout->oobfree;
		uint32_t boffs = 0, roffs = ops->ooboffs;
		size_t bytes = 0;

		for (; free->length && len; free++, len -= bytes) {
			/* Read request not from offset 0? */
			if (unlikely(roffs)) {
				if (roffs >= free->length) {
					roffs -= free->length;
					continue;
				}
				boffs = free->offset + roffs;
				bytes = min_t(size_t, len,
					      (free->length - roffs));
				roffs = 0;
			} else {
				bytes = min_t(size_t, len, free->length);
				boffs = free->offset;
			}
			memcpy(oob, chip->oobbuf + boffs, bytes);
			oob += bytes;
		}
		return;
	}
	default:
		BUG();
	}
}

/**
 * spi_nand_set_ds - set value to die select register
 * @chip: SPI-NAND device structure
 * @cfg: buffer stored value
 * Description:
 *   Configuration register includes OTP config, Lock Tight enable/disable
 *   and Internal ECC enable/disable.
 */
static int spi_nand_set_ds(struct spi_nand_chip *chip, u8 *ds)
{
	struct spi_nand_cmd cmd;

	if ((chip->mfr_id == SPI_NAND_MFR_MICRON)) {
		//return 0;
		return spi_nand_write_reg(chip, REG_DIE_SELECT, ds);
	} else {
		memset(&cmd, 0, sizeof(struct spi_nand_cmd));
		cmd.cmd = SPI_NAND_CMD_DIE_SELECT;
		cmd.n_addr = 0;
		cmd.addr[0] = *ds;
		cmd.n_tx = 0;
		//return 0;
		return spi_nand_issue_cmd(chip, &cmd);
	}
}

/**
 * spi_nand_lun_select - send die select command if needed
 * @chip: SPI-NAND device structure
 * @lun: lun need to access
 */
static int spi_nand_lun_select(struct spi_nand_chip *chip, u8 lun)
{
	u8 ds = 0;
	int ret = 0;

	if (chip->lun != lun) {
		if (chip->mfr_id == SPI_NAND_MFR_MICRON)
			ds = (lun == 1) ? DIE_SELECT_DS1 : DIE_SELECT_DS0;
		else
			ds = lun;
		ret = spi_nand_set_ds(chip, &ds);
		chip->lun = lun;
	}

	return ret;
}




/**
 * spi_nand_erase_block_erase - send command D8h to erase a block
 * @chip: SPI-NAND device structure
 * @page_addr: the page to erase.
 * Description:
 *   Need to wait for tERS.
 */
static int spi_nand_erase_block(struct spi_nand_chip *chip,
					uint32_t page_addr)
{
	struct spi_nand_cmd cmd;

	memset(&cmd, 0, sizeof(struct spi_nand_cmd));
	cmd.cmd = SPI_NAND_CMD_BLK_ERASE;
	cmd.n_addr = 3;
	cmd.addr[0] = (u8)(page_addr >> 16);
	cmd.addr[1] = (u8)(page_addr >> 8);
	cmd.addr[2] = (u8)page_addr;

	return spi_nand_issue_cmd(chip, &cmd);
}





/**
 * spi_nand_read_page_to_cache - send command 13h to read data from Nand
 * to cache
 * @chip: SPI-NAND device structure
 * @page_addr: page to read
 */
static int spi_nand_read_page_to_cache(struct spi_nand_chip *chip,
					uint32_t page_addr)
{
	struct spi_nand_cmd cmd;

	memset(&cmd, 0, sizeof(struct spi_nand_cmd));
	cmd.cmd = SPI_NAND_CMD_PAGE_READ;
	cmd.n_addr = 3;
	cmd.addr[0] = (u8)(page_addr >> 16);
	cmd.addr[1] = (u8)(page_addr >> 8);
	cmd.addr[2] = (u8)page_addr;

	return spi_nand_issue_cmd(chip, &cmd);
}

/**
 * spi_nand_read_from_cache - read data out from cache register
 * @chip: SPI-NAND device structure
 * @page_addr: page to read
 * @column: the location to read from the cache
 * @len: number of bytes to read
 * @rbuf: buffer held @len bytes
 * Description:
 *   Command can be 03h, 0Bh, 3Bh, 6Bh, BBh, EBh
 *   The read can specify 1 to (page size + spare size) bytes of data read at
 *   the corresponding locations.
 *   No tRd delay.
 */
static int spi_nand_read_from_cache(struct spi_nand_chip *chip,
		uint32_t page_addr, uint32_t column, size_t len, u8 *rbuf)
{
	struct spi_nand_cmd cmd;

	memset(&cmd, 0, sizeof(struct spi_nand_cmd));
	cmd.cmd = chip->read_cache_op;
	cmd.n_dummy_cycle = chip->dummy_cycle;
	cmd.n_addr = 2;
	cmd.addr[0] = (u8)(column >> 8);
	if (chip->options & SPI_NAND_NEED_PLANE_SELECT)
		cmd.addr[0] |= (u8)(((page_addr >>
			(chip->block_shift - chip->page_shift)) & 0x1) << 4);
	cmd.addr[1] = (u8)column;
	cmd.n_rx = len;
	cmd.rx_buf = rbuf;

	return spi_nand_issue_cmd(chip, &cmd);
}


static int spi_nand_wait(struct spi_nand_chip *chip, u8 *s);
/**
 * spi_nand_do_read_page - read page from flash to buffer
 * @mtd: MTD device structure
 * @page_addr: page address/raw address
 * @column: column address
 * @ecc_off: without ecc or not
 * @corrected: how many bit error corrected
 * @buf: data buffer
 * @len: data length to read
 */
static int spi_nand_do_read_page(struct mtd_info *mtd, uint32_t page_addr,
			bool ecc_off, int *corrected, bool oob_only)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = mtd->priv;
	int ret;
	u8 status;

	ret = spi_nand_read_page_to_cache(chip, page_addr);
    if (ret < 0) {
		pr_err("error %d read page 0x%x to cache\n",
			ret, page_addr);
        return ret;
    }
    ret = spi_nand_wait(chip, &status);
    if (ret < 0) {
        pr_err("error %d waiting page 0x%x to cache\n", ret, page_addr);
        return ret;
    }
	if (!ecc_off) {
	  uint8_t cs = 0, conf;
	  if (_chip->info->pm.ecc_bits>6) conf=ECC_BCH12;
	  else conf = ECC_BCH6;
	  ret = _chip->info->ecc_decode(_chip->info, cs, conf, chip->buf, chip->oobbuf);
	  if (ret < 0) {
		pr_err("internal ECC error reading page 0x%x\n",
			page_addr);
		mtd->ecc_stats.failed++;
	  } else{
			mtd->ecc_stats.corrected += ret;
	  }

	}

	if (!oob_only){
		spi_nand_read_from_cache(chip, page_addr, 0,
				chip->page_size + chip->oob_size, chip->buf);
	}else{
		spi_nand_read_from_cache(chip, page_addr, chip->page_size,
				chip->oob_size, chip->oobbuf);
	}

	return 0;
}


/**
 * spi_nand_do_read_oob - read out-of-band
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operations description structure
 * Description:
 *   Disable internal ECC before reading when MTD_OPS_RAW set.
 */
static int spi_nand_do_read_oob(struct mtd_info *mtd, loff_t from,
			  struct mtd_oob_ops *ops)
{
	struct spi_nand_chip *chip = mtd->priv;
	int page_addr;
	int corrected = 0;
	struct mtd_ecc_stats stats;
	int readlen = ops->ooblen;
	int len;
	int ret = 0;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int lun_num=0, lun_mask;
    uint32_t pa_be4xlat __attribute((unused));
    uint32_t pa_in_out __attribute((unused));

	pr_debug("%s: from = 0x%08Lx, len = %i\n",
			__func__, (unsigned long long)from, readlen);

	stats = mtd->ecc_stats;

	len = ops->mode == MTD_OPS_AUTO_OOB ? mtd->oobavail : mtd->oobsize;

	if (unlikely(ops->ooboffs >= len)) {
		pr_err("%s: attempt to start read outside oob\n",
				__func__);
		return -EINVAL;
	}

	/* Do not allow reads past end of device */
	if (unlikely(from >= mtd->size ||
		     ops->ooboffs + readlen > ((mtd->size >> chip->page_shift) -
					(from >> chip->page_shift)) * len)) {
		pr_err("%s: attempt to read beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	/* Shift to get page */
	page_addr = (from >> chip->page_shift);
    pa_be4xlat = page_addr;
    lun_mask = (1<<chip->lun_shift)-1;

	len -= ops->ooboffs;
	ops->oobretlen = 0;

	while (1) {
#ifdef CONFIG_NAND_HIDE_BAD
        pa_in_out = pa_be4xlat;
        ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
        if(ret == -EINVAL){
            pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
            return ret;
        }
        if(pa_be4xlat != pa_in_out){
            pr_debug("Read xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
        }
        page_addr = pa_in_out;
        pa_be4xlat++;
#endif

        if (chip->options & SPI_NAND_NEED_DIE_SELECT){
            lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
            spi_nand_lun_select(chip, lun_num);
        }
        if (ecc_off)
            chip->disable_ecc(chip);

		/*read data from chip*/
		ret = spi_nand_do_read_page(mtd, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr), ecc_off,
					&corrected, true);
		if (ret)
			goto out;
		len = min(len, readlen);
		spi_nand_transfer_oob(chip, ops->oobbuf + ops->oobretlen,
					ops, len);
		readlen -= len;
		ops->oobretlen += len;
		if (!readlen)
			break;
		page_addr++;

        if (ecc_off)
            chip->enable_ecc(chip);
	}
out:
	if (ret)
		return ret;

	if (mtd->ecc_stats.failed - stats.failed)
		return -EBADMSG;

	return  mtd->ecc_stats.corrected - stats.corrected ? -EUCLEAN : 0;
}


/**
 * spi_nand_block_bad - Check if block at offset is bad
 * @mtd: MTD device structure
 * @offs: offset relative to mtd start
 * @getchip: 0, if the chip is already selected
 */
static int spi_nand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct mtd_oob_ops ops = {0};
	uint32_t block_addr;
	u8 bad[2] = {0, 0};
	u8 ret = 0;

	block_addr = ofs >> chip->block_shift;
	ops.mode = MTD_OPS_PLACE_OOB;
	ops.ooblen = 2;
	ops.oobbuf = bad;

	if (getchip)
		spi_nand_get_device(mtd, FL_READING);
	spi_nand_do_read_oob(mtd, block_addr << chip->block_shift, &ops);
	if (getchip)
		spi_nand_release_device(mtd);
	if (bad[0] != 0xFF || bad[1] != 0xFF)
		ret =  1;

	return ret;
}


/**
 * spi_nand_is_bad_bbm - [BBT Interface] Check if block at offset is factory bad
 * @mtd: MTD device structure
 * @offs: offset relative to mtd start
 */
static int spi_nand_is_bad_bbm(struct mtd_info *mtd, loff_t ofs)
{
	return spi_nand_block_bad(mtd, ofs, 1);
}


/**
 * spi_nand_block_checkbad - Check if a block is marked bad
 * @mtd: MTD device structure
 * @ofs: offset from device start
 * @getchip: 0, if the chip is already selected
 * @allowbbt: 1, if its allowed to access the bbt area
 *
 * Check, if the block is bad. Either by reading the bad block table or
 * calling of the scan function.
 */
static int spi_nand_block_checkbad(struct mtd_info *mtd, loff_t ofs,
			int getchip, int allowbbt)
{
	struct spi_nand_chip *chip = mtd->priv;

	if (!chip->bbt)
		return spi_nand_block_bad(mtd, ofs, getchip);

	/* Return info from the table */
	if (nand_bbt_isbad(chip->bbt, ofs))
		return 1;
	else if (allowbbt)
		return 0;
	else
		return nand_bbt_isreserved(chip->bbt, ofs);
}
#define RTK_ERASE_FAILED    -1
#define RTK_ERASE_DONE       0
/**
 * __spi_nand_erase - erase block(s)
 * @mtd: MTD device structure
 * @einfo: erase instruction
 * @allowbbt: allow to access bbt
 *
 * Erase one or more blocks
 */
static int __spi_nand_erase(struct mtd_info *mtd, struct erase_info *einfo,
			int allowbbt)
{
	struct spi_nand_chip *chip = mtd->priv;
	int page_addr, pages_per_block;
	loff_t len;
	int ret = 0;
	int check_result = 0;
	int lun_num=0, lun_mask;
    uint32_t pa_be4xlat __attribute((unused));
    uint32_t pa_in_out __attribute((unused));

	/* check address align on block boundary */
	if (einfo->addr & (chip->block_size - 1)) {
		pr_err("%s: Unaligned address\n", __func__);
		return -EINVAL;
	}

	if (einfo->len & (chip->block_size - 1)) {
		pr_err("%s: Length not block aligned\n", __func__);
		return -EINVAL;
	}

	/* Do not allow erase past end of device */
	if ((einfo->len + einfo->addr) > chip->size) {
		pr_err("%s: Erase past end of device\n", __func__);
		return -EINVAL;
	}

	einfo->fail_addr = MTD_FAIL_ADDR_UNKNOWN;

	/* Grab the lock and see if the device is available */
	spi_nand_get_device(mtd, FL_ERASING);

	pages_per_block = 1 << (chip->block_shift - chip->page_shift);
	page_addr = einfo->addr >> chip->page_shift;
    pa_be4xlat = page_addr;
	len = einfo->len;
    lun_mask = (1<<chip->lun_shift)-1;
	chip->cached_page = -1;

	//einfo->state = MTD_ERASING;


	while (len) {
		/* Check if we have a bad block, we do not erase bad blocks! */
		if (spi_nand_block_checkbad(mtd, ((loff_t) page_addr) <<
					chip->page_shift, 0, allowbbt)) {
			pr_warn("%s: attempt to erase a bad block at 0x%012llx\n",
			__func__, ((loff_t) page_addr) << chip->page_shift);
			//einfo->state = MTD_ERASE_FAILED;
			check_result = RTK_ERASE_FAILED;
			goto erase_exit;
		}
        #ifdef CONFIG_NAND_HIDE_BAD
            pa_in_out = pa_be4xlat;
            ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
            if(ret == -EINVAL){
                pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
                return ret;
            }
            if(pa_be4xlat != pa_in_out){
                pr_debug("erase xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
            }
            page_addr = pa_in_out;
            pa_be4xlat += 64;
        #endif

        if (chip->options & SPI_NAND_NEED_DIE_SELECT){
            lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
            spi_nand_lun_select(chip, lun_num);
        }

		//spi_nand_write_enable(chip);
		ret = spi_nand_erase_block(chip, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr));
		if (ret < 0) {
			pr_err("erase block 0x%012llx failed\n",
				((loff_t) page_addr) << chip->page_shift);
			//einfo->state = MTD_ERASE_FAILED;
			check_result = RTK_ERASE_FAILED;
			einfo->fail_addr = (loff_t)page_addr
						<< chip->page_shift;
			goto erase_exit;
		}

		/* Increment page address and decrement length */
		len -= (1ULL << chip->block_shift);
		page_addr += pages_per_block;
        #ifdef CONFIG_NAND_HIDE_BAD
        page_addr = pa_be4xlat;
        #endif
    }

	//einfo->state = MTD_ERASE_DONE;
    check_result = RTK_ERASE_DONE;

erase_exit:

	ret = check_result == RTK_ERASE_DONE ? 0 : -EIO;

	spi_nand_release_device(mtd);

	/* Do call back function */
//	if (!ret)
//		mtd_erase_callback(einfo);

	/* Return more or less happy */
	return ret;
}


/**
 * spi_nand_erase - [MTD Interface] erase block(s)
 * @mtd: MTD device structure
 * @einfo: erase instruction
 *
 * Erase one ore more blocks
 */
static int spi_nand_erase(struct mtd_info *mtd, struct erase_info *einfo)
{
	return __spi_nand_erase(mtd, einfo, 0);
}

static int spi_nand_erase_bbt(struct mtd_info *mtd, loff_t ofs)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct erase_info einfo = {0};

//	einfo.mtd = mtd;
	einfo.addr = ofs;
	einfo.len = chip->block_size;

	return __spi_nand_erase(mtd, &einfo, 1);
}


static int spi_nand_default_bbt(struct mtd_info *mtd)
{
	struct nand_bbt *bbt = kzalloc(size64(sizeof(struct nand_bbt)), GFP_KERNEL);
	struct spi_nand_chip *chip = mtd->priv;

	if (!bbt)
		return -ENOMEM;

	bbt->bbt_options |= NAND_BBT_USE_FLASH | NAND_BBT_NO_OOB;
	bbt->mtd = mtd;
	bbt->numchips = 1;
	bbt->chipsize = mtd->size;
	bbt->chip_shift = ilog2(chip->size);
	bbt->bbt_erase_shift = chip->block_shift;
	bbt->page_shift = chip->page_shift;
	bbt->is_bad_bbm = spi_nand_is_bad_bbm;
	bbt->erase = spi_nand_erase_bbt;
	chip->bbt = bbt;


	return nand_bbt_init(chip->bbt);
}

/**
 * spi_nand_read_page_cache_random - send command 30h for data read
 * @chip: SPI-NAND device structure
 * @page_addr: the page to read to data register.
 * Description:
 *   Transfer data from data register to cache register and kick off the other
 *   page data transferring from array to data register.
 */
static int spi_nand_read_page_cache_random(struct spi_nand_chip *chip,
					uint32_t page_addr)
{
	struct spi_nand_cmd cmd;

	memset(&cmd, 0, sizeof(struct spi_nand_cmd));
	cmd.cmd = SPI_NAND_CMD_READ_PAGE_CACHE_RDM;
	cmd.addr[0] = (u8)(page_addr >> 16);
	cmd.addr[1] = (u8)(page_addr >> 8);
	cmd.addr[2] = (u8)page_addr;
	cmd.n_addr = 3;

	return spi_nand_issue_cmd(chip, &cmd);
}

/**
 * spi_nand_read_page_cache_last - send command 3Fh to end
 * READ PAGE CACHE RANDOM(30h) sequence
 * @chip: SPI-NAND device structure
 * Description:
 *   End the READ PAGE CACHE RANDOM sequence and copies a page from
 *   the data register to the cache register.
 */
static int spi_nand_read_page_cache_last(struct spi_nand_chip *chip)
{
  struct spi_nand_cmd cmd;

  memset(&cmd, 0, sizeof(struct spi_nand_cmd));
  cmd.cmd = SPI_NAND_CMD_READ_PAGE_CACHE_LAST;

  return spi_nand_issue_cmd(chip, &cmd);
}


/**
 * spi_nand_wait - wait until the command is done
 * @chip: SPI-NAND device structure
 * @s: buffer to store status register(can be NULL)
 */
static int spi_nand_wait(struct spi_nand_chip *chip, u8 *s)
{
  unsigned long timeo = jiffies;
  u8 status, state = chip->state;
  int ret = -ETIMEDOUT;
  int count = 0;

  if (state == FL_ERASING || state == FL_WRITING)
    timeo += msecs_to_jiffies(400);
  else
    timeo += msecs_to_jiffies(20);

  while (time_before(jiffies, timeo) || count < MIN_TRY_COUNT) {
    spi_nand_read_status(chip, &status);
    if ((status & STATUS_OIP_MASK) == STATUS_READY) {
      ret = 0;
      goto out;
    }
    count++;
  }
out:
  if (s)
    *s = status;

  return ret;
}

#if 0
/**
 * spi_nand_wait_crbusy - wait until CRBSY is clear
 * @chip: SPI-NAND device structure
 * Description:
 *   Used in READ PAGE CACHE RANDOM(30h) sequence, CRBSY bit clear
 *   means data is transferred from data register to cache register.
 */
static int spi_nand_wait_crbusy(struct spi_nand_chip *chip)
{
  unsigned long timeo = jiffies;
  u8 status;
  int ret = -ETIMEDOUT;
  int count = 0;

  timeo += msecs_to_jiffies(20);

  while (time_before(jiffies, timeo) || count < MIN_TRY_COUNT) {
    spi_nand_read_status(chip, &status);
    if ((status & STATUS_CRBSY_MASK) == STATUS_READY) {
      ret = 0;
      goto out;
    }
    count++;
  }
out:
  return ret;
}
#endif

/**
 * spi_nand_read_id - send 9Fh command to get ID
 * @chip: SPI-NAND device structure
 * @buf: buffer to store id
 */
static int spi_nand_read_id(struct spi_nand_chip *chip, u8 *buf)
{
  struct spi_nand_cmd cmd;

  memset(&cmd, 0, sizeof(struct spi_nand_cmd));
  cmd.cmd = SPI_NAND_CMD_READ_ID;
  cmd.n_addr = 1;
  cmd.addr[0]= 0;
  cmd.n_rx = 3;
  cmd.rx_buf = buf;

  return spi_nand_issue_cmd(chip, &cmd);
}

#if 0
/**
 * spi_nand_reset - send command FFh to reset chip.
 * @chip: SPI-NAND device structure
 */
static int spi_nand_reset(struct spi_nand_chip *chip)
{
  struct spi_nand_cmd cmd;

  memset(&cmd, 0, sizeof(struct spi_nand_cmd));
  cmd.cmd = SPI_NAND_CMD_RESET;

  if (spi_nand_issue_cmd(chip, &cmd) < 0)
    pr_err("spi_nand reset failed!\n");

  return 0;
}
#endif

/**
 * spi_nand_read_pages_fast - read data from flash to buffer
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operations description structure
 * @max_bitflips: maximum bitflip count
 * Description:
 *   Advanced read function, use READ PAGE CACHE RANDOM to
 *   speed up read.
 */
static int spi_nand_read_pages_fast(struct mtd_info *mtd, loff_t from,
			  struct mtd_oob_ops *ops, unsigned int *max_bitflips)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = mtd->priv;
	int page_addr, page_offset, size;
	int ret;
	unsigned int corrected = 0;
	int readlen = ops->len;
	int oobreadlen = ops->ooblen;
	bool ecc_off = ops->mode == MTD_OPS_RAW, cross_lun = false;
	bool read_ramdon_issued = false;
	int ooblen = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;
	u8 status;
	u8 *buf;
	/*use internal buffer when buffer from upper isn't phy continuous*/
	int use_in_buf = !virt_addr_valid(ops->datbuf);
	int lun_num;

	page_addr = from >> chip->page_shift;
	page_offset = from & chip->page_mask;
	ops->retlen = 0;
	lun_num = from >> chip->lun_shift;
again:
	if (chip->options & SPI_NAND_NEED_DIE_SELECT)
		spi_nand_lun_select(chip, lun_num);

	spi_nand_read_page_to_cache(chip, page_addr);
	ret = spi_nand_wait(chip, &status);
	if (ret < 0) {
		pr_err("error %d waiting page 0x%x to cache\n",
			ret, page_addr);
		goto out;
	}

	if (!ecc_off) {
	  uint8_t cs = 0, conf;
	  if (_chip->info->pm.ecc_bits>6) conf=ECC_BCH12;
	  else conf = ECC_BCH6;
	  ret = _chip->info->ecc_decode(_chip->info, cs, conf, chip->buf, chip->oobbuf);
	  if (ret < 0) {
		pr_err("internal ECC error reading page 0x%x\n",
			page_addr);
		mtd->ecc_stats.failed++;
	  } else{
			mtd->ecc_stats.corrected += ret;
	  }

	}

	while ((page_offset + readlen > chip->page_size) && !cross_lun) {
		if (!(chip->options & SPI_NAND_NEED_DIE_SELECT) ||
			(page_addr + 1) &
			((1 << (chip->lun_shift - chip->page_shift)) - 1)) {
			read_ramdon_issued = true;
			spi_nand_read_page_cache_random(chip, page_addr + 1);
			ret = spi_nand_wait(chip, &status);
			if (ret < 0) {
				pr_err("error %d waiting page 0x%x to data resigter\n",
					ret, page_addr + 1);
				goto out;
			}
		} else {
			cross_lun = true;
			break;
		}
		*max_bitflips = max(*max_bitflips, corrected);
		size = min(readlen, chip->page_size - page_offset);
		buf = use_in_buf ? chip->buf : ops->datbuf + ops->retlen;
		spi_nand_read_from_cache(chip,
				page_addr, page_offset, size, buf);
		if (use_in_buf)
			memcpy(ops->datbuf + ops->retlen, chip->buf, size);
		page_offset = 0;
		ops->retlen += size;
		readlen -= size;
		if (unlikely(ops->oobbuf)) {
			size = min(oobreadlen, ooblen);
			spi_nand_read_from_cache(chip, page_addr,
				chip->page_size, chip->oob_size, chip->oobbuf);
			spi_nand_transfer_oob(chip,
				ops->oobbuf + ops->oobretlen, ops, size);
			ops->oobretlen += size;
			oobreadlen -= size;
		}

		if (!cross_lun) {
			//ret = spi_nand_wait_crbusy(chip);
			if (ret < 0) {
				pr_err("error %d waiting page 0x%x to cache\n",
					ret, page_addr + 1);
				goto out;
			}
		}
		page_addr++;
	}
	if (read_ramdon_issued) {
		spi_nand_read_page_cache_last(chip);

		/*
		* Already check ecc status in loop, no need to check again
		*/
		//ret = spi_nand_wait(chip, &status);
		if (ret < 0) {
			pr_err("error %d waiting page 0x%x to cache\n",
				ret, page_addr);
			goto out;
		}
	}
	*max_bitflips = max(*max_bitflips, corrected);
	size = min(readlen, chip->page_size - page_offset);
	buf = use_in_buf ? chip->buf : ops->datbuf + ops->retlen;
	spi_nand_read_from_cache(chip, page_addr, page_offset, size, buf);
	if (use_in_buf)
		memcpy(ops->datbuf + ops->retlen, chip->buf, size);
	ops->retlen += size;
	readlen -= size;
	if (unlikely(ops->oobbuf)) {
		size = min(oobreadlen, ooblen);
		spi_nand_read_from_cache(chip, page_addr,
			chip->page_size, chip->oob_size, chip->oobbuf);
		spi_nand_transfer_oob(chip,
			ops->oobbuf + ops->oobretlen, ops, size);
		ops->oobretlen += size;
		oobreadlen -= size;
	}

	if (cross_lun) {
		cross_lun = false;
		page_addr++;
		page_offset = 0;
		lun_num++;
		goto again;
	}
out:
	chip->cached_page = -1;
	return ret;
}


static inline bool is_read_page_fast_benefit(struct spi_nand_chip *chip,
			loff_t from, size_t len)
{
	/* Disable random cache read which is incompatible with other Flash vendor. */
	return false;

	if (len < chip->page_size << 2)
		return false;
	if (from >> chip->lun_shift == (from + len) >> chip->lun_shift)
		return true;
	if (((1 << chip->lun_shift) - from) >= (chip->page_size << 2) ||
		(from + len - (1 << chip->lun_shift)) >= (chip->page_size << 2))
		return true;
	return false;
}


/**
 * spi_nand_read_pages - read data from flash to buffer
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operations description structure
 * @max_bitflips: maximum bitflip count
 * Description:
 *   Normal read function, read one page to buffer before issue
 *   another.
 */
static int spi_nand_read_pages(struct mtd_info *mtd, loff_t from,
			  struct mtd_oob_ops *ops, unsigned int *max_bitflips)
{
	struct spi_nand_chip *chip = mtd->priv;
	int page_addr, page_offset, size, ret=0;
	unsigned int corrected = 0;
	int readlen = ops->len;
	int oobreadlen = ops->ooblen;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int ooblen = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;
	int lun_num=0, lun_mask;
    uint32_t pa_be4xlat __attribute((unused));
    uint32_t pa_in_out __attribute((unused));

	page_addr = from >> chip->page_shift;
	page_offset = from & chip->page_mask;
    pa_be4xlat = page_addr;
    lun_mask = (1<<chip->lun_shift)-1;

	ops->retlen = 0;
	*max_bitflips = 0;


	while (1) {
        #ifdef CONFIG_NAND_HIDE_BAD
            pa_in_out = pa_be4xlat;
            ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
            if(ret == -EINVAL){
                pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
                return ret;
            }
            if(pa_be4xlat != pa_in_out){
                pr_debug("Read xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
            }
            page_addr = pa_in_out;
            pa_be4xlat++;
        #endif

        if (chip->options & SPI_NAND_NEED_DIE_SELECT){
            lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
            spi_nand_lun_select(chip, lun_num);
        }

    	if (ecc_off)
	    	chip->disable_ecc(chip);

		size = min(readlen, chip->page_size - page_offset);
		if (page_addr != chip->cached_page
			|| ecc_off != chip->cached_page_ecc_off) {
			ret = spi_nand_do_read_page(mtd, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr),
                ecc_off, &corrected, false);
			if (ret)
				break;
			chip->cached_page_bitflips = corrected;
            chip->cached_page = page_addr;
			chip->cached_page_ecc_off = ecc_off;
		}

		memcpy(ops->datbuf + ops->retlen,
			chip->buf + page_offset, size);
		*max_bitflips = max(*max_bitflips, chip->cached_page_bitflips);

		ops->retlen += size;
		readlen -= size;
		page_offset = 0;

		if (unlikely(ops->oobbuf)) {
			size = min(oobreadlen, ooblen);
			spi_nand_transfer_oob(chip,
				ops->oobbuf + ops->oobretlen, ops, size);
			ops->oobretlen += size;
			oobreadlen -= size;
		}
		if (!readlen)
			break;

		page_addr++;

    	if (ecc_off)
	    	chip->enable_ecc(chip);
	}

	return ret;
}


/**
 * spi_nand_do_read_ops - read data from flash to buffer
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob ops structure
 * Description:
 *   Disable internal ECC before reading when MTD_OPS_RAW set.
 */
static int spi_nand_do_read_ops(struct mtd_info *mtd, loff_t from,
			  struct mtd_oob_ops *ops)
{
	struct spi_nand_chip *chip = mtd->priv;
	int ret;
	struct mtd_ecc_stats stats;
	unsigned int max_bitflips = 0;
	int oobreadlen = ops->ooblen;
	int ooblen = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;

	/* Do not allow reads past end of device */
	if (unlikely(from >= mtd->size)) {
		pr_err("%s: attempt to read beyond end of device\n",
				__func__);
		return -EINVAL;
	}
	stats = mtd->ecc_stats;

	/* for oob */
	if (oobreadlen > 0) {
		if (unlikely(ops->ooboffs >= ooblen)) {
			pr_err("%s: attempt to start read outside oob\n",
					__func__);
			return -EINVAL;
		}

		if (unlikely(ops->ooboffs + oobreadlen >
		((mtd->size >> chip->page_shift) - (from >> chip->page_shift))
		* ooblen)) {
			pr_err("%s: attempt to read beyond end of device\n",
					__func__);
			return -EINVAL;
		}
		ooblen -= ops->ooboffs;
		ops->oobretlen = 0;
	}
#if 0 //20200304: Confifure ECC bit in spi_nand_read_pages
	if (ecc_off)
		chip->disable_ecc(chip);
#endif

	if (is_read_page_fast_benefit(chip, from, ops->len))
		ret = spi_nand_read_pages_fast(mtd, from, ops, &max_bitflips);
	else
		ret = spi_nand_read_pages(mtd, from, ops, &max_bitflips);

#if 0 //20200304: Confifure ECC bit in spi_nand_read_pages
	if (ecc_off)
		chip->enable_ecc(chip);
#endif
	if (ret)
		return ret;

	if (mtd->ecc_stats.failed - stats.failed)
		return -EBADMSG;

	return max_bitflips;
}


/**
 * spi_nand_read - [MTD Interface] SPI-NAND read
 * @mtd: MTD device structure
 * @from: offset to read from
 * @len: number of bytes to read
 * @retlen: pointer to variable to store the number of read bytes
 * @buf: the databuffer to put data
 */
static int spi_nand_read(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u8 *buf)
{
	struct mtd_oob_ops ops;
	int ret;
	// printk("%s:%d ==================\n", __FUNCTION__, __LINE__);
	spi_nand_get_device(mtd, FL_READING);

	memset(&ops, 0, sizeof(ops));
	ops.len = len;
	ops.datbuf = buf;
	ops.mode = MTD_OPS_PLACE_OOB;
 	ret = spi_nand_do_read_ops(mtd, from, &ops);

	*retlen = ops.retlen;

	spi_nand_release_device(mtd);
	return ret;
}



#if 1
/**
 * spi_nand_get_cfg - get configuration register value
 * @chip: SPI-NAND device structure
 * @cfg: buffer to store value
 * Description:
 *   Configuration register includes OTP config, Lock Tight enable/disable
 *   and Internal ECC enable/disable.
 */
static int spi_nand_get_cfg(struct spi_nand_chip *chip, u8 *cfg)
{
  return spi_nand_read_reg(chip, REG_CFG, cfg);
}

/**
 * spi_nand_set_cfg - set value to configuration register
 * @chip: SPI-NAND device structure
 * @cfg: buffer stored value
 * Description:
 *   Configuration register includes OTP config, Lock Tight enable/disable
 *   and Internal ECC enable/disable.
 */
static int spi_nand_set_cfg(struct spi_nand_chip *chip, u8 *cfg)
{
	return spi_nand_write_reg(chip, REG_CFG, cfg);
}
#endif



#if 1
/**
 * spi_nand_enable_ecc - enable internal ECC
 * @chip: SPI-NAND device structure
 * Description:
 *   There is one bit( bit 0x10 ) to set or to clear the internal ECC.
 *   Enable chip internal ECC, set the bit to 1
 *   Disable chip internal ECC, clear the bit to 0
 */
static void spi_nand_enable_ecc(struct spi_nand_chip *chip)
{
  u8 cfg = 0;

  spi_nand_get_cfg(chip, &cfg);
  if ((cfg & CFG_ECC_MASK) == CFG_ECC_ENABLE)
    return;
  cfg |= CFG_ECC_ENABLE;
  spi_nand_set_cfg(chip, &cfg);
}

/**
 * spi_nand_disable_ecc - disable internal ECC
 * @chip: SPI-NAND device structure
 * Description:
 *   There is one bit( bit 0x10 ) to set or to clear the internal ECC.
 *   Enable chip internal ECC, set the bit to 1
 *   Disable chip internal ECC, clear the bit to 0
 */
static void spi_nand_disable_ecc(struct spi_nand_chip *chip)
{
	u8 cfg = 0;

	spi_nand_get_cfg(chip, &cfg);
	if ((cfg & CFG_ECC_MASK) == CFG_ECC_ENABLE) {
		cfg &= ~CFG_ECC_ENABLE;
		spi_nand_set_cfg(chip, &cfg);
	}
}

#endif

/**
 * spi_nand_lock_block - write block lock register to
 * lock/unlock device
 * @spi: spi device structure
 * @lock: value to set to block lock register
 * Description:
 *   After power up, all the Nand blocks are locked.  This function allows
 *   one to unlock the blocks, and so it can be written or erased.
 */
static int spi_nand_lock_block(struct spi_nand_chip *chip, u8 lock)
{
	//return 0;
	return spi_nand_write_reg(chip, REG_BLOCK_LOCK, &lock);
}


/**
 * spi_nand_init_flash_chip - [SPI-NAND Interface] Init flash chip
 * @mtd: MTD device structure
 * Description:
 *   This is used to initialize the flash chip status, like protection/ecc/QE.
 */
int spi_nand_init_flash_chip(struct mtd_info *mtd)
{
    struct spi_nand_chip *chip = mtd->priv;

	spi_nand_lock_block(chip, BL_ALL_UNLOCKED);
    if(chip->options & SPI_NAND_NEED_QE_ENABLE){
        //spi_nand_quad_enable(chip);
    }

	if ((chip->options & SPI_NAND_NEED_DIE_SELECT)
		/*&& (chip->mfr_id != SPI_NAND_MFR_MICRON)*/ ){
		spi_nand_lun_select(chip, 1);
		spi_nand_lock_block(chip, BL_ALL_UNLOCKED);
        if(chip->options & SPI_NAND_NEED_QE_ENABLE){
            //spi_nand_quad_enable(chip);
        }
		spi_nand_lun_select(chip, 0);
	}
    return 0;
}

static void
snand_info_to_chip(struct mtd_info *mtd)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = (struct sw_spi_nand_chip*)mtd->priv;

	chip->block_size = SNAF_BLOCK_SIZE(_chip->info);
	chip->page_size = SNAF_PAGE_SIZE(_chip->info);
	chip->oob_size = SNAF_SPARE_SIZE(_chip->info);
	chip->size = SNAF_CHIP_SIZE(_chip->info);

}

/**
 * spi_nand_suspend - [MTD Interface] Suspend the SPI-NAND flash
 * @mtd: MTD device structure
 */
static int spi_nand_suspend(struct mtd_info *mtd)
{
	return spi_nand_get_device(mtd, FL_PM_SUSPENDED);
}

/**
 * spi_nand_resume - [MTD Interface] Resume the SPI-NAND flash
 * @mtd: MTD device structure
 */
static void spi_nand_resume(struct mtd_info *mtd)
{
	struct spi_nand_chip *this = mtd->priv;

	if (this->state == FL_PM_SUSPENDED)
		spi_nand_release_device(mtd);
	else
		pr_err("%s is not called in suspended state\n:", __func__);
}


//static int sw_spi_nand_program_execute(struct mtd_info *mtd, struct spi_nand_cmd *cmd);
/**
 * spi_nand_do_write_page - write data from buffer to flash
 * @mtd: MTD device structure
 * @page_addr: page address/raw address
 * @column: column address
 * @buf: data buffer
 * @len: data length to write
 * @clr_cache: clear cache register with 0xFF or not
 */
static int spi_nand_do_write_page(struct mtd_info *mtd, uint32_t page_addr,
						bool oob_only)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = mtd->priv;
	int ret = 0;

	if (!oob_only){
		//do nothing
	} else {
		//clean page buffer
		memset(chip->buf, 0xFF ,SNAF_PAGE_SIZE(_chip->info));
	}

	snand_page_prog(_chip->info, 0, page_addr<<SNAF_CA_BIT(_chip->info), chip->buf, chip->oobbuf);

	return ret;
}


/**
 * spi_nand_fill_oob - transfer client buffer to oob
 * @chip: SPI-NAND device structure
 * @oob: oob data buffer
 * @len: oob data write length
 * @ops: oob ops structure
 */
static void spi_nand_fill_oob(struct spi_nand_chip *chip, u8 *oob,
				size_t len, struct mtd_oob_ops *ops)
{
	memset(chip->oobbuf, 0xff, chip->oob_size);
	switch (ops->mode) {

	case MTD_OPS_PLACE_OOB:
	case MTD_OPS_RAW:
		memcpy(chip->oobbuf + ops->ooboffs, oob, len);
		return;

	case MTD_OPS_AUTO_OOB: {
		struct nand_oobfree *free = chip->ecclayout->oobfree;
		uint32_t boffs = 0, woffs = ops->ooboffs;
		size_t bytes = 0;

		for (; free->length && len; free++, len -= bytes) {
			/* Write request not from offset 0? */
			if (unlikely(woffs)) {
				if (woffs >= free->length) {
					woffs -= free->length;
					continue;
				}
				boffs = free->offset + woffs;
				bytes = min_t(size_t, len,
					      (free->length - woffs));
				woffs = 0;
			} else {
				bytes = min_t(size_t, len, free->length);
				boffs = free->offset;
			}
			memcpy(chip->oobbuf + boffs, oob, bytes);
			oob += bytes;
		}
		return;
	}
	default:
		BUG();
	}
}

/**
 * spi_nand_do_write_ops - write data from buffer to flash
 * @mtd: MTD device structure
 * @to: offset to write to
 * @ops: oob operations description structure
 * Description:
 *   Disable internal ECC before writing when MTD_OPS_RAW set.
 */
static int spi_nand_do_write_ops(struct mtd_info *mtd, loff_t to,
			 struct mtd_oob_ops *ops)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = mtd->priv;
	int page_addr, page_offset, size;
	int writelen = ops->len;
	int oobwritelen = ops->ooblen;
	int ret = 0;
	int ooblen = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int lun_num=0, lun_mask;
    uint32_t pa_be4xlat __attribute((unused));
    uint32_t pa_in_out __attribute((unused));

	/* Do not allow reads past end of device */
	if (unlikely(to >= mtd->size)) {
		pr_err("%s: attempt to write beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	page_addr = to >> chip->page_shift;
	page_offset = to & chip->page_mask;
    pa_be4xlat = page_addr;
    lun_mask = (1<<chip->lun_shift)-1;
	ops->retlen = 0;

	/* for oob */
	if (oobwritelen > 0) {
		/* Do not allow write past end of page */
		if ((ops->ooboffs + oobwritelen) > ooblen) {
			pr_err("%s: attempt to write past end of page\n",
					__func__);
			return -EINVAL;
		}

		if (unlikely(ops->ooboffs >= ooblen)) {
			pr_err("%s: attempt to start write outside oob\n",
					__func__);
			return -EINVAL;
		}
		if (unlikely(ops->ooboffs + oobwritelen >
		((mtd->size >> chip->page_shift) - (to >> chip->page_shift))
			* ooblen)) {
			pr_err("%s: attempt to write beyond end of device\n",
					__func__);
			return -EINVAL;
		}
		ooblen -= ops->ooboffs;
		ops->oobretlen = 0;
	}

	chip->cached_page = -1;
	while (1) {
#ifdef CONFIG_NAND_HIDE_BAD
        pa_in_out = pa_be4xlat;
        ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
        if(ret == -EINVAL){
            pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
            return ret;
        }
        if(pa_be4xlat != pa_in_out){
            pr_debug("Write xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
        }
        page_addr = pa_in_out;
        pa_be4xlat++;
#endif

        if (chip->options & SPI_NAND_NEED_DIE_SELECT){
            lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
            spi_nand_lun_select(chip, lun_num);
        }

	if (ecc_off)
		chip->disable_ecc(chip);

		if (unlikely(ops->oobbuf)) {
			size = min(oobwritelen, ooblen);
			spi_nand_fill_oob(chip, ops->oobbuf + ops->oobretlen,
					size, ops);
			ops->oobretlen += size;
			oobwritelen -= size;
		} else {
			memset(chip->oobbuf, 0xff, chip->oob_size);
		}
		size = min(writelen, chip->page_size - page_offset);
		memcpy(chip->buf + page_offset,
			ops->datbuf + ops->retlen, size);
		if (page_offset)
			memset(chip->buf, 0xff, page_offset);
		if (size < chip->page_size - page_offset)
			memset(chip->buf + page_offset + size, 0xff,
				chip->page_size - page_offset - size);
		if(!ecc_off){
			uint8_t cs = 0, conf;
			if (_chip->info->pm.ecc_bits>6) conf=ECC_BCH12;
			else conf = ECC_BCH6;
			_chip->info->ecc_encode(_chip->info, cs, conf, chip->buf, chip->oobbuf);
		}
		ret = spi_nand_do_write_page(mtd, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr), false);
		if (ret) {
			pr_err("error %d writing page 0x%x\n",
				ret, page_addr);
			goto out;
		}
		ops->retlen += size;
		writelen -= size;
		page_offset = 0;
		if (!writelen)
			break;
		page_addr++;

    	if (ecc_off)
    		chip->enable_ecc(chip);
	}
out:
	return ret;
}





/**
 * spi_nand_write - [MTD Interface] SPI-NAND write
 * @mtd: MTD device structure
 * @to: offset to write to
 * @len: number of bytes to write
 * @retlen: pointer to variable to store the number of written bytes
 * @buf: the data to write
 */
static int spi_nand_write(struct mtd_info *mtd, loff_t to, size_t len,
	size_t *retlen, const u8 *buf)
{
	struct mtd_oob_ops ops;
	int ret;

	spi_nand_get_device(mtd, FL_WRITING);

	memset(&ops, 0, sizeof(ops));
	ops.len = len;
	ops.datbuf = (u8 *)buf;
	ops.mode = MTD_OPS_PLACE_OOB;
	ret =  spi_nand_do_write_ops(mtd, to, &ops);

	*retlen = ops.retlen;

	spi_nand_release_device(mtd);

	return ret;
}


/**
 * spi_nand_read_oob - [MTD Interface] read data and/or out-of-band
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operation description structure
 */
static int spi_nand_read_oob(struct mtd_info *mtd, loff_t from,
			struct mtd_oob_ops *ops)
{
	int ret = -ENOTSUPP;

	ops->retlen = 0;

	/* Do not allow reads past end of device */
	if (ops->datbuf && (from + ops->len) > mtd->size) {
		pr_err("%s: attempt to read beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	spi_nand_get_device(mtd, FL_READING);

	switch (ops->mode) {
	case MTD_OPS_PLACE_OOB:
	case MTD_OPS_AUTO_OOB:
	case MTD_OPS_RAW:
		break;

	default:
		goto out;
	}

	if (!ops->datbuf)
		ret = spi_nand_do_read_oob(mtd, from, ops);
	else
		ret = spi_nand_do_read_ops(mtd, from, ops);

out:
	spi_nand_release_device(mtd);

	return ret;
}

/**
 * spi_nand_do_write_oob - write out-of-band
 * @mtd: MTD device structure
 * @to: offset to write to
 * @ops: oob operation description structure
 * Description:
 *   Disable internal ECC before writing when MTD_OPS_RAW set.
 */
static int spi_nand_do_write_oob(struct mtd_info *mtd, loff_t to,
			     struct mtd_oob_ops *ops)
{
	int page_addr, len, ret;
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = mtd->priv;
	int writelen = ops->ooblen;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int lun_num, lun_mask;
    uint32_t pa_in_out __attribute((unused));


	pr_debug("%s: to = 0x%08x, len = %i\n",
			 __func__, (unsigned int)to, (int)writelen);

	len = ops->mode == MTD_OPS_AUTO_OOB ? mtd->oobavail : mtd->oobsize;

	/* Do not allow write past end of page */
	if ((ops->ooboffs + writelen) > len) {
		pr_err("%s: attempt to write past end of page\n",
				__func__);
		return -EINVAL;
	}

	if (unlikely(ops->ooboffs >= len)) {
		pr_err("%s: attempt to start write outside oob\n",
				__func__);
		return -EINVAL;
	}

	/* Do not allow write past end of device */
	if (unlikely(to >= mtd->size ||
		     ops->ooboffs + writelen >
			((mtd->size >> chip->page_shift) -
			 (to >> chip->page_shift)) * len)) {
		pr_err("%s: attempt to write beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	/* Shift to get page */
	page_addr = to >> chip->page_shift;
#ifdef CONFIG_NAND_HIDE_BAD
    pa_in_out = page_addr;
    ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
    if(ret == -EINVAL){
        pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
        return ret;
    }
    if(page_addr != pa_in_out){
        pr_debug("Read xlat: 0x%x => 0x%x\n",page_addr,pa_in_out);
    }
    page_addr = pa_in_out;
#endif
    lun_mask = (1<<chip->lun_shift)-1;

	lun_num = to >> chip->lun_shift;
	spi_nand_fill_oob(chip, ops->oobbuf, writelen, ops);
    if (chip->options & SPI_NAND_NEED_DIE_SELECT){
        lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
        spi_nand_lun_select(chip, lun_num);
        if(lun_num){
            page_addr = ((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift;
        }
    }

	if (ecc_off)
		chip->disable_ecc(chip);

	if(!ecc_off){
		uint8_t cs = 0, conf;
		if (_chip->info->pm.ecc_bits>6) conf=ECC_BCH12;
		else conf = ECC_BCH6;
		_chip->info->ecc_encode(_chip->info, cs, conf, chip->buf, chip->oobbuf);
	}
	ret = spi_nand_do_write_page(mtd, page_addr, true);
	if (ret) {
		pr_err("error %d writing page 0x%x\n",
			ret, page_addr);
		goto out;
	}
	ops->oobretlen = writelen;

out:
	if (ecc_off)
		chip->enable_ecc(chip);

	return ret;
}


/**
 * spi_nand_write_oob - [MTD Interface] write data and/or out-of-band
 * @mtd: MTD device structure
 * @to: offset to write to
 * @ops: oob operation description structure
 */
static int spi_nand_write_oob(struct mtd_info *mtd, loff_t to,
			  struct mtd_oob_ops *ops)
{
	int ret = -ENOTSUPP;

	ops->retlen = 0;

	/* Do not allow writes past end of device */
	if (ops->datbuf && (to + ops->len) > mtd->size) {
		pr_err("%s: attempt to write beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	spi_nand_get_device(mtd, FL_WRITING);

	switch (ops->mode) {
	case MTD_OPS_PLACE_OOB:
	case MTD_OPS_AUTO_OOB:
	case MTD_OPS_RAW:
		break;

	default:
		goto out;
	}

	if (!ops->datbuf)
		ret = spi_nand_do_write_oob(mtd, to, ops);
	else
		ret = spi_nand_do_write_ops(mtd, to, ops);

out:
	spi_nand_release_device(mtd);

	return ret;
}

/**
 * spi_nand_sync - [MTD Interface] sync
 * @mtd: MTD device structure
 *
 * Sync is actually a wait for chip ready function
 */
static void spi_nand_sync(struct mtd_info *mtd)
{
	pr_debug("spi_nand_sync: called\n");

	/* Grab the lock and see if the device is available */
	spi_nand_get_device(mtd, FL_SYNCING);

	/* Release it and go back */
	spi_nand_release_device(mtd);
}

/**
 * spi_nand_block_isbad - [MTD Interface] Check if block at offset is bad
 * @mtd: MTD device structure
 * @offs: offset relative to mtd start
 */
static int spi_nand_block_isbad(struct mtd_info *mtd, loff_t offs)
{
	return spi_nand_block_checkbad(mtd, offs, 1, 0);
}

/**
 * spi_nand_block_markbad_lowlevel - mark a block bad
 * @mtd: MTD device structure
 * @ofs: offset from device start
 *
 * This function performs the generic bad block marking steps (i.e., bad
 * block table(s) and/or marker(s)). We only allow the hardware driver to
 * specify how to write bad block markers to OOB (chip->block_markbad).
 *
 * We try operations in the following order:
 *  (1) erase the affected block, to allow OOB marker to be written cleanly
 *  (2) write bad block marker to OOB area of affected block (unless flag
 *      NAND_BBT_NO_OOB_BBM is present)
 *  (3) update the BBT
 * Note that we retain the first error encountered in (2) or (3), finish the
 * procedures, and dump the error in the end.
*/
static int spi_nand_block_markbad_lowlevel(struct mtd_info *mtd, loff_t ofs)
{
	struct spi_nand_chip *chip = mtd->priv;
	struct nand_bbt *bbt = chip->bbt;
	struct mtd_oob_ops ops = {0};
	struct erase_info einfo = {0};
	uint32_t block_addr;
	u8 buf[2] = {0, 0};
	int res, ret = 0;

	if (!bbt || !(bbt->bbt_options & NAND_BBT_NO_OOB_BBM)) {
		/*erase bad block before mark bad block*/
//		einfo.mtd = mtd;
		einfo.addr = ofs;
		einfo.len = 1UL << chip->block_shift;
		spi_nand_erase(mtd, &einfo);

		block_addr = ofs >> chip->block_shift;
		ops.mode = MTD_OPS_PLACE_OOB;
		ops.ooblen = 2;
		ops.oobbuf = buf;
		spi_nand_get_device(mtd, FL_WRITING);
		ret = spi_nand_do_write_oob(mtd,
				block_addr << chip->block_shift, &ops);
		spi_nand_release_device(mtd);
	}

	/* Mark block bad in BBT */
	if (chip->bbt) {
		res = nand_bbt_markbad(chip->bbt, ofs);
		if (!ret)
			ret = res;
	}

	if (!ret)
		mtd->ecc_stats.badblocks++;

	return ret;
}


/**
 * spi_nand_block_markbad - [MTD Interface] Mark block at the given offset
 * as bad
 * @mtd: MTD device structure
 * @ofs: offset relative to mtd start
 */
static int spi_nand_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	int ret;

	ret = spi_nand_block_isbad(mtd, ofs);
	if (ret) {
		/* If it was bad already, return success and do nothing */
		if (ret > 0)
			return 0;
		return ret;
	}

	return spi_nand_block_markbad_lowlevel(mtd, ofs);
}

/**
 * spi_nand_block_isreserved - [MTD Interface] Check if a block is
 * marked reserved.
 * @mtd: MTD device structure
 * @ofs: offset from device start
 */
static int spi_nand_block_isreserved(struct mtd_info *mtd, loff_t ofs)
{
	struct spi_nand_chip *chip = mtd->priv;

	if (!chip->bbt)
		return 0;
	/* Return info from the table */
	return nand_bbt_isreserved(chip->bbt, ofs);
}


/**
 * spi_nand_scan_ident - [SPI-NAND Interface] Scan for the SPI-NAND device
 * @mtd: MTD device structure
 * Description:
 *   This is the first phase of the initiazation. It reads the flash ID and
 *   sets up spi_nand_chip fields accordingly.
 */
int spi_nand_scan_ident(struct mtd_info *mtd)
{
//	u8 id[SPI_NAND_MAX_ID_LEN] = {0};
	u8 id[SNAF_MAX_ID_LEN];
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = (struct sw_spi_nand_chip*)mtd->priv;

	if(spi_nand_probe(0)){
		return -ENODEV;
	}

	_chip->info = &snaf_info;

	snand_info_to_chip(mtd);

	spi_nand_read_id(chip, id);

	pr_info("SPI-NAND: %02X%02X is found.\n", id[0], id[1]);

	chip->mfr_id = id[0];
	chip->dev_id = id[1];
	chip->block_shift = ilog2(chip->block_size);
	chip->page_shift = ilog2(chip->page_size);
	chip->page_mask = chip->page_size - 1;
	chip->lun = 0;

#if 1

	if (!chip->enable_ecc){
		chip->enable_ecc = spi_nand_enable_ecc;
	}
	if (!chip->disable_ecc){
		chip->disable_ecc = spi_nand_disable_ecc;
	}
#endif
	chip->buf = kzalloc(size64(chip->page_size + chip->oob_size), GFP_KERNEL);
	if (!chip->buf)
		return -ENOMEM;

	chip->oobbuf = chip->buf + chip->page_size;

	return 0;
}
EXPORT_SYMBOL_GPL(spi_nand_scan_ident);




/**
 * spi_nand_scan_tail - [SPI-NAND Interface] Scan for the SPI-NAND device
 * @mtd: MTD device structure
 * Description:
 *   This is the second phase of the initiazation. It fills out all the
 *   uninitialized fields of spi_nand_chip and mtd fields.
 */
int spi_nand_scan_tail(struct mtd_info *mtd)
{
	struct spi_nand_chip *chip = mtd->priv;
//	struct sw_spi_nand_chip *_chip =  mtd->priv;
    uint32_t i __attribute((unused));

	/* Initialize state */
	chip->state = FL_READY;
	/* Invalidate the pagebuffer reference */
	chip->cached_page = -1;

	init_waitqueue_head(&(chip->wq));
	spin_lock_init(&(chip->chip_lock));

#if 1
	if (chip->options & SPI_NAND_ECC_TYPE_HRADWARE)
		spi_nand_disable_ecc(chip);
#endif

	mtd->name = "sw_spinand_flash";
	mtd->size = chip->size;
	mtd->erasesize = chip->block_size;
	mtd->writesize = chip->page_size;
	mtd->writebufsize = mtd->writesize;
	mtd->owner = THIS_MODULE;
	mtd->type = MTD_NANDFLASH;
	mtd->flags = MTD_CAP_NANDFLASH;
	if (!mtd->ecc_strength)
		mtd->ecc_strength = chip->ecc_strength ?
					chip->ecc_strength : 1;

//	mtd->ecclayout = chip->ecclayout;
	mtd->oobsize = chip->oob_size;
	mtd->oobavail = chip->ecclayout->oobavail;
	mtd->_erase = spi_nand_erase;
	mtd->_point = NULL;
	mtd->_unpoint = NULL;
	mtd->_read = spi_nand_read;
	mtd->_write = spi_nand_write;
	mtd->_read_oob = spi_nand_read_oob;
	mtd->_write_oob = spi_nand_write_oob;
	mtd->_sync = spi_nand_sync;
	mtd->_lock = NULL;
	mtd->_unlock = NULL;
	mtd->_suspend = spi_nand_suspend;
	mtd->_resume = spi_nand_resume;
	mtd->_block_isbad = spi_nand_block_isbad;
	mtd->_block_markbad = spi_nand_block_markbad;
	mtd->_block_isreserved = spi_nand_block_isreserved;

	if (!mtd->bitflip_threshold)
		mtd->bitflip_threshold = DIV_ROUND_UP(mtd->ecc_strength * 3, 4);

#ifdef CONFIG_NAND_HIDE_BAD
     get_spi_nand_factory_bad_blocks(mtd, 0, mtd->size);

    /* Update the the size information to the availabe capacity */
    mtd->size -= bad_block_cnt*mtd->erasesize;
    printk("spi nand total capacity is %dMB(0x%x)\n",(uint32_t)(chip->size/0x100000),(uint32_t)chip->size);
    printk("spi nand available size is %dMB(0x%x)\n",(uint32_t)(mtd->size/0x100000),(uint32_t)mtd->size);

    if(bbl_source == FACTORY_1ST_SCAN)
        printk("spi nand 1st scan factory bad to memory\n");
    else
        printk("spi nand %s from flash\n",(bbl_source==FACTORY_FROM_BBT0)?"Bbl0":"Bbl1");

    printk("spi nand hidden: ");
    for(i=0 ; i<bad_block_cnt ; i++){
        printk("0x%x ", bad_block_list[i]);
    }
    printk("\n");

    printk("spi nand Mode: ");
    switch(chip->read_cache_op){
        case SPI_NAND_CMD_READ_FROM_CACHE_QUAD_IO:
            printk("Q/");
            break;
        case SPI_NAND_CMD_READ_FROM_CACHE_X4:
            printk("4/");
            break;
        case SPI_NAND_CMD_READ_FROM_CACHE_DUAL_IO:
            printk("D/");
            break;
        case SPI_NAND_CMD_READ_FROM_CACHE_X2:
            printk("2/");
            break;
        case SPI_NAND_CMD_READ_FROM_CACHE:
        case SPI_NAND_CMD_READ_FROM_CACHE_FAST:
            printk("S/");
            break;
    }
    switch(chip->write_cache_op){
        case SPI_NAND_CMD_PROG_LOAD_X4:
            printk("4\n");
            break;
        case SPI_NAND_CMD_PROG_LOAD:
            printk("S\n");
            break;
    }
#endif

    /* Check, if we should skip the bad block table scan */
    if (chip->options & SPI_NAND_SKIP_BBTSCAN)
        return 0;
    /* Build bad block table */
    return spi_nand_default_bbt(mtd);
	return 0;
}
EXPORT_SYMBOL_GPL(spi_nand_scan_tail);


/**
 * spi_nand_scan_ident_release - [SPI-NAND Interface] Free resources
 * applied by spi_nand_scan_ident
 * @mtd: MTD device structure
 */
int spi_nand_scan_ident_release(struct mtd_info *mtd)
{
	struct spi_nand_chip *chip = mtd->priv;

	kfree(chip->oobbuf);
	kfree(chip->bbt);

	return 0;
}
EXPORT_SYMBOL_GPL(spi_nand_scan_ident_release);



/**
 * spi_nand_scan_tail_release - [SPI-NAND Interface] Free resources
 * applied by spi_nand_scan_tail
 * @mtd: MTD device structure
 */
int spi_nand_scan_tail_release(struct mtd_info *mtd)
{
	return 0;
}
EXPORT_SYMBOL_GPL(spi_nand_scan_tail_release);


/**
 * spi_nand_release - [SPI-NAND Interface] Free resources held by the SPI-NAND
 * device
 * @mtd: MTD device structure
 */
int spi_nand_release(struct mtd_info *mtd)
{
	struct spi_nand_chip *chip = mtd->priv;

	mtd_device_unregister(mtd);
	kfree(chip->oobbuf);
	kfree(chip->bbt);

	return 0;
}
EXPORT_SYMBOL_GPL(spi_nand_release);

/*
 * sw_spinand_reset - Reset SPI Nand
 * Description:
 *     Reset: reset SPI Nand device
 */
static int
sw_spinand_reset(void/* struct spi_nand_cmd *cmd */)
{
	snand_reset(NULL, 0);
	return 0;
}

/*
 * sw_spinand_get_feature_reg - send command 0xf with with address <addr> to get register value
 * Description:
 */
static int
sw_spinand_get_feature_reg(struct mtd_info *mtd, struct spi_nand_cmd *cmd)
{
	struct sw_spi_nand_chip *_chip = mtd->priv;
	uint8_t addr = cmd->addr[0];
	u8 *value = cmd->rx_buf;
	*value = snand_get_feature(_chip->info, 0, addr);
	return 0;
}

/*
 * sw_spinand_set_feature_reg - send command 0x1f with address <addr> to set register value
 * Description:
 */
static int
sw_spinand_set_feature_reg(struct mtd_info *mtd, struct spi_nand_cmd *cmd)
{
	struct sw_spi_nand_chip *_chip = mtd->priv;
	uint8_t addr = cmd->addr[0];
	const u8 *value = cmd->tx_buf;
	snand_set_feature(_chip->info, 0, addr, *value);
	return 0;
}

static int
sw_spinand_read_page_to_cache(struct mtd_info *mtd, struct spi_nand_cmd *cmd)
{
	u32 page_id = (u32)(cmd->addr[0] << 16) | (u32)(cmd->addr[1] << 8) | (u32)(cmd->addr[2]);
	struct spi_nand_chip *chip = mtd->priv;
	struct sw_spi_nand_chip *_chip = mtd->priv;
	snand_page_read(_chip->info, 0, page_id<<SNAF_CA_BIT(_chip->info), chip->buf, chip->oobbuf);
	return 0;

}

static int sw_spinand_read_cache(struct mtd_info *mtd, struct spi_nand_cmd *cmd)
{
	struct spi_nand_chip *chip = mtd->priv;
	//u32 offset = (u32)(cmd->addr[0] << 8) | (u32)(cmd->addr[1]);
	u8 *buf = cmd->rx_buf;
	u32 len = cmd->n_rx;

	memcpy(buf, chip->buf, len);

	return 0;
}

static int sw_spinand_erase_block(struct mtd_info *mtd, struct spi_nand_cmd *cmd)
{
	struct sw_spi_nand_chip *_chip = mtd->priv;
//	int ret = 0;
//	u8 status;
	u32 block_id = (u32)(cmd->addr[0] << 16) | (u32)(cmd->addr[1] << 8) | (u32)(cmd->addr[2]);
	return snand_block_erase(_chip->info, 0, block_id<<SNAF_CA_BIT(_chip->info));
}


static int sw_spinand_write_cache_x1(struct mtd_info *mtd, struct spi_nand_cmd *cmd)
{
	struct spi_nand_chip *chip = mtd->priv;
//	struct sw_spi_nand_chip *_chip = mtd->priv;
//	u32 offset = (u32)(cmd->addr[0] << 8) | (u32)(cmd->addr[1]);
	u8 *buf = (u8 *)cmd->tx_buf;
	u32 len = cmd->n_tx;

	memcpy(chip->buf, buf, len);

	return 0;
}

/*
 * sw_spinand_read_id- Read SPI Nand ID
 * Description:
 *    Read ID: read two ID bytes from the SPI Nand device
 */
static int
sw_spinand_read_id(struct mtd_info *mtd, u8 *id)
{
	struct sw_spi_nand_chip *_chip = mtd->priv;
	snand_rdid(_chip->info, 0, id);
	return 0;
}


/*
 * sw_spinand_cmd_fn - to process a command to send to the SPI-NAND
 * by cortina SPI NAND controller
 * @chip: SPI-NAND device structure
 * @cmd: command structure
 * Description:
 *   Set up the command buffer to send to the SPI controller.
 *   The command buffer has to initialized to 0.
 */
static int sw_spinand_cmd_fn(struct spi_nand_chip *chip,
				struct spi_nand_cmd *cmd)
{
	struct mtd_info *mtd = chip->mtd;
	int ret = 0;

	switch (cmd->cmd) {
	case SPI_NAND_CMD_RESET:
		ret = sw_spinand_reset(/* cmd */);
		break;
	case SPI_NAND_CMD_GET_FEATURE:
		ret = sw_spinand_get_feature_reg(mtd, cmd);
		break;
	case SPI_NAND_CMD_SET_FEATURE:
		ret = sw_spinand_set_feature_reg(mtd, cmd);
		break;
	case SPI_NAND_CMD_DIE_SELECT:
		//ret = ca_spinand_die_select(cmd);
		break;
	case SPI_NAND_CMD_PAGE_READ:
		ret = sw_spinand_read_page_to_cache(mtd, cmd);
		break;
	case SPI_NAND_CMD_READ_PAGE_CACHE_RDM:
		/* Todo */
		break;
	case SPI_NAND_CMD_READ_PAGE_CACHE_LAST:
		/* Todo */
		break;
	case SPI_NAND_CMD_READ_FROM_CACHE:
	case SPI_NAND_CMD_READ_FROM_CACHE_FAST:
	case SPI_NAND_CMD_READ_FROM_CACHE_X2:
	case SPI_NAND_CMD_READ_FROM_CACHE_DUAL_IO:
	case SPI_NAND_CMD_READ_FROM_CACHE_X4:
	case SPI_NAND_CMD_READ_FROM_CACHE_QUAD_IO:
		ret = sw_spinand_read_cache(mtd, cmd);
		break;
	case SPI_NAND_CMD_BLK_ERASE:
		ret = sw_spinand_erase_block(mtd, cmd);
		break;
	case SPI_NAND_CMD_PROG_EXC:
		break;
	case SPI_NAND_CMD_PROG_LOAD:
		ret = sw_spinand_write_cache_x1(mtd, cmd);
		break;
	case SPI_NAND_CMD_PROG_LOAD_RDM_DATA:
		/* Todo */
		break;
	case SPI_NAND_CMD_PROG_LOAD_X4:
		ret = sw_spinand_write_cache_x1(mtd, cmd);
		//ret = ca_spinand_write_cache_x4(cmd);
		break;
	case SPI_NAND_CMD_PROG_LOAD_RDM_DATA_X4:
		/* Todo */
		break;
	case SPI_NAND_CMD_READ_ID:
		ret = sw_spinand_read_id(mtd, cmd->rx_buf);
		break;
	case SPI_NAND_CMD_WR_DISABLE:
		/* Todo */
		break;
	case SPI_NAND_CMD_WR_ENABLE:
		//ret = ca_spinand_write_enable(cmd);
		break;
	case SPI_NAND_CMD_END:
	default:
		break;
	}

    return ret;
}


#ifdef CONFIG_OF
static struct of_device_id sw_spinand_dt_ids[] = {
    {.compatible = "rtk,sw_spi_nand",},
    {},
};

MODULE_DEVICE_TABLE(of, sw_spinand_dt_ids);
#endif


#define SPI_NANDINFO_PROC
#ifdef SPI_NANDINFO_PROC
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
static struct spi_nand_chip *private_chip = NULL;
static int rtk_read_proc_nandinfo(struct seq_file *m, void *v)
{
		if(private_chip){
			seq_printf(m, "%-15s: %llu\n", "nand_size"   , private_chip->size);
			seq_printf(m, "%-15s: %llu\n", "chip_size"   , private_chip->size);
			seq_printf(m, "%-15s: %u\n" , "block_size"  , private_chip->block_size);
			seq_printf(m, "%-15s: %u\n" , "chunk_size"  , private_chip->page_size);
			seq_printf(m, "%-15s: %u\n" , "oob_size"    , private_chip->oob_size);
			seq_printf(m, "%-15s: %u\n" , "oob_avail"   , private_chip->ecclayout->oobavail);
			seq_printf(m, "%-15s: %u\n" , "PagePerBlock", private_chip->block_size/private_chip->page_size);
		}
        return 0;
}

void rtk_mtd_set_dev_defaults(struct mtd_info *mtd)
{
    if (mtd->dev.parent)
    {
        if (!mtd->owner && mtd->dev.parent->driver)
            mtd->owner = mtd->dev.parent->driver->owner;
        if (!mtd->name)
            mtd->name = dev_name(mtd->dev.parent);
    }
    else
    {
        pr_debug("mtd device won't show a device symlink in sysfs");
    }

    INIT_LIST_HEAD(&mtd->partitions);
    mutex_init(&mtd->master.partitions_lock);
}

static int rtk_open_proc_nandinfo(struct inode *inode, struct file *file)
{
        return single_open(file, rtk_read_proc_nandinfo, NULL);
}

static const struct proc_ops rtk_nand_proc_ops = {
        .proc_open           = rtk_open_proc_nandinfo,
        .proc_read           = seq_read,
        .proc_lseek          = seq_lseek,
        .proc_release        = single_release,
};
#endif

static const char * const part_probes[] = {
  "ofpart", "cmdlinepart", "RedBoot", NULL };
static int sw_spinand_probe(struct platform_device *pdev)
{
	struct resource mem_resource;
	const struct of_device_id *match;
	struct device_node *np = pdev->dev.of_node;
	struct spi_nand_chip *chip;
	struct mtd_info *mtd;
	rtk_partition_entry_t partition_table[RTK_MTD_PARTITION_NUM];
	struct mtd_partition rtk_sdk_parts[RTK_MTD_PARTITION_NUM];
	int i ,ret_int = 0;
	u8 ret;

	match = of_match_device(sw_spinand_dt_ids, &pdev->dev);
	if (!match)
		return -EINVAL;

	dev_notice(&pdev->dev, "SW SPI-NAND Flash driver\n");

	/* SPI Nand Flash Controller addr base */
	ret = of_address_to_resource(np, 0, &mem_resource);

	if (ret) {
		dev_warn(&pdev->dev, "invalid address %d\n", ret);
		return ret;
	}

	flash_ctl_base = devm_ioremap(&pdev->dev, mem_resource.start,
	resource_size(&mem_resource));

	chip = kzalloc(size64(sizeof(struct sw_spi_nand_chip)), GFP_KERNEL);
	if (!chip) {
		ret = -ENOMEM;
		goto err1;
	}

	mtd = kzalloc(size64(sizeof(struct mtd_info)), GFP_KERNEL);
	if (!mtd) {
		ret = -ENOMEM;
		goto err2;
	}

	mtd->priv = chip;
	chip->mtd = mtd;
	chip->command_fn = sw_spinand_cmd_fn;

//	struct sw_spi_nand_chip *_chip = mtd->priv;
#if 0 // check io mode
	if (!of_property_read_u32(np, "io-mode", &spi_mode))
		printk("SPI-NAND io-mode:%x\n", spi_mode);
	else
		spi_mode = 0;

	if (spi_mode & SPI_RX_QUAD)
		chip->controller_caps |= SPI_NAND_RX_QUAD;
	if (spi_mode & SPI_RX_DUAL)
		chip->controller_caps |= SPI_NAND_RX_DUAL;
	if (spi_mode & SPI_TX_QUAD)
		chip->controller_caps |= SPI_NAND_TX_QUAD;
	if (spi_mode & SPI_TX_DUAL)
		chip->controller_caps |= SPI_NAND_TX_DUAL;
#endif
	chip->priv = pdev;
	mtd_set_of_node(mtd, pdev->dev.of_node);
	rtk_mtd_set_dev_defaults(mtd);
	platform_set_drvdata(pdev, chip);

	ret_int = spi_nand_scan_ident(mtd);
	if (ret_int)
		goto err3;

	ret_int = spi_nand_manufacture_init(chip);
	if (ret_int)
		goto err3;

    ret_int = spi_nand_init_flash_chip(mtd);
	if (ret_int)
		goto err3;

	ret_int = spi_nand_scan_tail(mtd);
	if (ret_int)
		goto err4;

#ifdef SPI_NANDINFO_PROC
	private_chip = chip;
	proc_create("nandinfo", 0, NULL, &rtk_nand_proc_ops);
#endif

	rtk_flash_partition_table_get(chip->size, partition_table);

	for (i = 0; i < RTK_MTD_PARTITION_NUM; i++) {
		rtk_sdk_parts[i].name = partition_table[i].name;
		rtk_sdk_parts[i].size = partition_table[i].size;
		rtk_sdk_parts[i].offset = partition_table[i].offset;
		rtk_sdk_parts[i].mask_flags = 0;
		rtk_sdk_parts[i].add_flags = MTD_WRITEABLE;
		rtk_sdk_parts[i].of_node = np;
		rtk_sdk_parts[i].types = (const char *const *)NULL;
	}

	if ( add_mtd_partitions(mtd, rtk_sdk_parts, ARRAY_SIZE(rtk_sdk_parts)) ) {
		printk("%s:%d Add partition Fail\n", __FUNCTION__, __LINE__);
	}
	else {
		printk("%s:%d Add partition Success\n", __FUNCTION__, __LINE__);
		return 0;
	}

	spi_nand_scan_tail_release(mtd);
err4:
	spi_nand_scan_ident_release(mtd);
err3:
	platform_set_drvdata(pdev, NULL);
	kfree(mtd);
err2:
	kfree((struct sw_spi_nand_chip*)chip);
err1:
	return ret;
}


static int
sw_spinand_remove(struct platform_device *pdev)
{
	struct spi_nand_chip *chip = platform_get_drvdata(pdev);
	struct mtd_info *mtd = chip->mtd;

	kfree(mtd);
	kfree((struct sw_spi_nand_chip*)chip);

	return 0;
}



static struct platform_driver sw_spinand_driver = {
	.probe	= sw_spinand_probe,
	.remove	= sw_spinand_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = "sw_spi_nand",
		.of_match_table = of_match_ptr(sw_spinand_dt_ids),
	},
};

static int __init sw_spinand_init(void)
{
	return platform_driver_register(&sw_spinand_driver);
}

static void __exit sw_spinand_exit(void)
{
	platform_driver_unregister(&sw_spinand_driver);
}

module_init(sw_spinand_init);
module_exit(sw_spinand_exit);



MODULE_DESCRIPTION("SPI NAND framework");
MODULE_LICENSE("GPL v2");
