#include <spi_nand/spi_nand_core.h>
#include <spi_nand/spi_nand_common.h>

/************************************************
  *  MICRON, Manufacture ID: 0x2C
  ***********************************************/

#define MICRON_ECC_1_3_BITFLIP  (1)
#define MICRON_ECC_4_6_BITFLIP  (3)
#define MICRON_ECC_7_8_BITFLIP  (5)

static void micron_ecc_encode(struct snaf_info_s *info, uint8_t cs, uint8_t cap, uint8_t *payload, uint8_t *spare){ return; }

static int micron_ecc_decode(struct snaf_info_s *info, uint8_t cs, uint8_t cap, uint8_t *payload, uint8_t *spare)
{
    uint8_t ecst = snand_get_ecfail2(info, cs);
    VERBOSE("%s ecfail %x\n", __FUNCTION__, ecst);
    switch(ecst)
    {
        case SNAF_ECC_NO_BITFLIPS:
            return 0;
        case SNAF_ECC_UNCOR_ERROR:
            ERROR("ECC Uncorrected Error Detected\n");
            return -2;
        case MICRON_ECC_1_3_BITFLIP:
            return 3;   // 1~3 bitflips, return max.
        case MICRON_ECC_4_6_BITFLIP:
            return 6;   // 4~6 bitflips, return max.
        case MICRON_ECC_7_8_BITFLIP:
            return 8;   // 7~8 bitflips, return max.
        default:
            break;
    }
    return -1;
}

static void micron_init(snaf_info_t *info)
{
    uint32_t cs;
    uint32_t ondie_en = CHECK_ARCH(info->pm.mode, SNAF_ON_DIE_ECC);
    uint32_t extra = CHECK_ARCH(info->pm.mode, SNAF_EXTRA_ECSP);
    for (cs=0; cs<info->num_lun; cs++) {
        snand_reset(info, cs);
        snand_wait_oip(info, cs, 635, __FUNCTION__);
        snand_unlock(info, cs);
        snand_quad_enable2(info, cs, CHECK_ARCH(info->pm.mode, SNAF_QUAD_IO));
        snand_set_ecc_en(info, cs, ondie_en);
    }
    if (ondie_en) {
        info->ecc_decode = micron_ecc_decode;
        info->ecc_encode = micron_ecc_encode;
        if(extra) info->pm.spare_size -= 1;    // hide internal ecc space
    }
    info->tprog = 600;
    info->tread = 170;
    info->tberase = 10000;
}

#define MICRON_OPS   (SNAF_ON_DIE_ECC|SNAF_EXTRA_ECSP|SNAF_SERIAL_IO)

const snaf_pm_t micron_chips_info[] = {
    /*********** 3.3 V ***********/
    SNDB(MID_MICRON, "MT29F1G01ABAFD", 0x14, 1024, 64, 2048, 128, 8, MICRON_OPS), // 1Gb
    SNDB(MID_MICRON, "MT29F2G01ABAGD", 0x24, 2048, 64, 2048, 128, 8, MICRON_OPS|SNAF_PLANE_ARCH), // 2Gb
    SNDB(MID_MICRON, "MT29F4G01ABAFD", 0x34, 2048, 64, 4096, 256, 8, MICRON_OPS), // 4Gb

    /*********** 1.8 V ***********/
    SNDB(MID_MICRON, "MT29F1G01ABBFD", 0x15, 1024, 64, 2048, 128, 8, MICRON_OPS), // 1Gb
    SNDB(MID_MICRON, "MT29F2G01ABBGD", 0x25, 2048, 64, 2048, 128, 8, MICRON_OPS|SNAF_PLANE_ARCH), // 2Gb
    SNDB(MID_MICRON, "MT29F4G01ABBFD", 0x35, 2048, 64, 4096, 256, 8, MICRON_OPS), // 4Gb

};

const snaf_manu_t micron_manu_info = {
    .mid = MID_MICRON,
    .mname = "MICRON",     // follow parameter page namespace
    .nchips = ARRAY_SIZE(micron_chips_info),
    .chips = micron_chips_info,
    .init = micron_init
};
