#include <spi_nand/spi_nand_core.h>
#include <spi_nand/spi_nand_common.h>

/************************************************
  *  KIOXIA(TOSHIBA), Manufacture ID: 0x98
  ***********************************************/
static void kioxia_ecc_encode(struct snaf_info_s *info, uint8_t cs, uint8_t cap, uint8_t *payload, uint8_t *spare){ return; }

static int kioxia_ecc_decode(struct snaf_info_s *info, uint8_t cs, uint8_t cap, uint8_t *payload, uint8_t *spare)
{
    uint8_t ecst = snand_get_ecfail(info, cs);
    uint32_t ret;
    VERBOSE("%s ecfail %x\n", __FUNCTION__, ecst);

    switch(ecst)
    {
        case SNAF_ECC_NO_BITFLIPS:
            return 0;
        case SNAF_ECC_UNCOR_ERROR:
            ERROR("ECC Uncorrected Error Detected\n");
            return -2;
        case SNAF_ECC_HAS_LT_BITFLIPS:
        case SNAF_ECC_HAS_ELT_BITFLIPS:
            ret = snand_get_feature(info, cs, 0x30);
            NOTICE("Feature 0x30: %x, MAX Bit-Flips %d bits\n", ret, ret>>4);
            return ret >> 4;    // max bit-flips
        default:
            break;
    }
    return -1;
}

static void kioxia_init(snaf_info_t *info)
{
    uint32_t cs;
    uint32_t ondie_en = CHECK_ARCH(info->pm.mode, SNAF_ON_DIE_ECC);
    uint32_t extra = CHECK_ARCH(info->pm.mode, SNAF_EXTRA_ECSP);
    for (cs=0; cs<info->num_lun; cs++) {
        snand_reset(info, cs);
        snand_wait_oip(info, cs, 550, __FUNCTION__);
        snand_unlock(info, cs);
        snand_quad_enable(info, cs, CHECK_ARCH(info->pm.mode, SNAF_QUAD_IO));
        snand_set_ecc_en(info, cs, CHECK_ARCH(info->pm.mode, SNAF_ON_DIE_ECC));
    }
    if (ondie_en) {
        info->ecc_decode = kioxia_ecc_decode;
        info->ecc_encode = kioxia_ecc_encode;
        if(extra) info->pm.spare_size -= 1;    // hide internal ecc space
    }
    info->tprog = 600;
    info->tread = 300;
    info->tberase = 10000;
}

#define KIOXIA_OPS   (SNAF_ON_DIE_ECC|SNAF_EXTRA_ECSP|SNAF_SERIAL_IO)

/*    Manu ID, Device ID, num of block, init func, options */
const snaf_pm_t kioxia_chips_info[] = {
    /*********** 3.3 V ***********/
    // 1st generation (G)
    SNDB(MID_KIOXIA, "TC58CVG0S3HRAIG", 0xC2, 1024, 64, 2048, 128, 8, KIOXIA_OPS), // 1Gb
    SNDB(MID_KIOXIA, "TC58CVG1S3HRAIG", 0xCB, 2048, 64, 2048, 128, 8, KIOXIA_OPS), // 2Gb
    SNDB(MID_KIOXIA, "TC58CVG2S0HRAIG", 0xCD, 2048, 64, 4096, 256, 8, KIOXIA_OPS), // 4Gb
    // 2nd generation (J)
    SNDB(MID_KIOXIA, "TC58CVG0S3HRAIJ", 0xE2, 1024, 64, 2048, 128, 8, KIOXIA_OPS), // 1Gb
    SNDB(MID_KIOXIA, "TC58CVG1S3HRAIJ", 0xEB, 2048, 64, 2048, 128, 8, KIOXIA_OPS), // 2Gb
    SNDB(MID_KIOXIA, "TC58CVG2S0HRAIJ", 0xED, 2048, 64, 4096, 256, 8, KIOXIA_OPS), // 4Gb
    SNDB(MID_KIOXIA, "TH58CVG3S0HRAIJ", 0xE4, 4096, 64, 4096, 256, 8, KIOXIA_OPS), // 8Gb

    /*********** 1.8 V ***********/
    // 1st generation (G)
    SNDB(MID_KIOXIA, "TC58CYG0S3HRAIG", 0xB2, 1024, 64, 2048, 128, 8, KIOXIA_OPS), // 1Gb
    SNDB(MID_KIOXIA, "TC58CYG1S3HRAIG", 0xBB, 2048, 64, 2048, 128, 8, KIOXIA_OPS), // 2Gb
    SNDB(MID_KIOXIA, "TC58CYG2S0HRAIG", 0xBD, 2048, 64, 4096, 256, 8, KIOXIA_OPS), // 4Gb
    // 2nd generation (J)
    SNDB(MID_KIOXIA, "TC58CYG0S3HRAIJ", 0xD2, 1024, 64, 2048, 128, 8, KIOXIA_OPS), // 1Gb
    SNDB(MID_KIOXIA, "TC58CYG1S3HRAIJ", 0xDB, 2048, 64, 2048, 128, 8, KIOXIA_OPS), // 2Gb
    SNDB(MID_KIOXIA, "TC58CYG2S0HRAIJ", 0xDD, 2048, 64, 4096, 256, 8, KIOXIA_OPS), // 4Gb
    SNDB(MID_KIOXIA, "TH58CYG3S0HRAIJ", 0xD4, 4096, 64, 4096, 256, 8, KIOXIA_OPS), // 8Gb
};

const snaf_manu_t kioxia_manu_info = {
    .mid = MID_KIOXIA,
    .mname = "TOSHIBA",     // follow parameter page namespace
    .nchips = ARRAY_SIZE(kioxia_chips_info),
    .chips = kioxia_chips_info,
    .init = kioxia_init,
};
