#ifndef ___SPI_NAND_CORE_H__
#define ___SPI_NAND_CORE_H__ 1

#define DCACHE_FLUSH(start, size) sihnah_flush_dcache_range(start, size)
#define MEMCPY sihnah_memcpy

#include <sihnah.h>
#include <spi_nand_register.h>

/***********************************************
  *  SPI NAND Common OP Code
  ***********************************************/
#define OP_READ             (0x03)
#define OP_FAST_READ_SIO    (0x0B)
#define OP_FAST_READ_DIO    (0xBB)
#define OP_FAST_READ_QIO    (0xEB)
#define OP_READ_X2          (0x3B)
#define OP_READ_X4          (0x6B)
#define OP_PROG_LOAD        (0x02)
#define OP_PROG_LOAD_X4     (0x32)
#define OP_PROG_LOADR       (0x84)
#define OP_PROG_LOADR_X4    (0x34)
#define OP_RESET            (0xFF)
#define OP_RDID             (0x9F)
#define OP_WRITE_ENABLE     (0x06)
#define OP_WRITE_DISABLE    (0x04)
#define OP_BLOCK_ERASE      (0xD8)
#define OP_PROG_EXEC        (0x10)
#define OP_PAGE_DATA_READ   (0x13)
#define OP_SET_FEATURE      (0x1F)
#define OP_GET_FEATURE      (0x0F)
#define OP_DIE_SELECT       (0xC2)

#define SNAF_MAX_ID_LEN     (4)
#define SNAF_MAX_NUM_LUN    (2)

/***********************************************
  *  SPI NAND Structure Delcaire
  ***********************************************/
enum {
    snaf_xfer_rx,       //0
    snaf_xfer_tx,       //1
    snaf_xfer_rx_dma,   //2
    snaf_xfer_tx_dma,   //3
};

enum {
    snaf_sio,       //0
    snaf_dio,       //1
    snaf_qio,       //2
};

typedef struct {
    uint8_t         op;
    uint32_t        addr;
    uint8_t         amode;
    uint8_t         dmode;
    uint8_t         dummy_bytes;
    uint8_t         *data;      // data address
    uint32_t        dsize;     // data size
    uint8_t         *spare;     // spare address
    uint32_t        ssize;     // spare size
    uint8_t         dma;        // dma enable
} snaf_payload_t;

typedef struct {
    uint8_t         rd_op;
    uint8_t         rd_amode;
    uint8_t         rd_dmode;
    uint8_t         rd_xfer_mode;
    uint8_t         dummy_bytes;
    uint8_t         wr_op;
    uint8_t         wr_amode;
    uint8_t         wr_dmode;
    uint8_t         wr_xfer_mode;
} snaf_dev_t;

typedef struct {
    //char             *mname;            // model name
    uint8_t         mid;               // Manufacture id
    uint8_t         did_len;
    uint16_t        did;               // Device id
    uint32_t        num_blk:3;         // num_block = 512*(1<<num_blk); (512~8192)
    uint32_t        num_page:3;        // num_page_per_block = 32*(1<<num_page+1); (32~1024)
    uint32_t        page_size:3;       // page_size = 1024*(1<<page_size); (1024~8192)
    uint32_t        spare_size:3;      // spare_size = 32*(1<<spare_size); (32~1024)
    uint32_t        ecc_bits:4;        // ecc requirement
    uint16_t        mode;              // support mode
} snaf_pm_t;

typedef struct snaf_info_s {
    snaf_pm_t        pm;
    uint8_t          num_lun;           // number of logical unit
    uint16_t         tprog;             // max page program time
    uint16_t         tberase;           // max block erase time
    uint16_t         tread;             // max page read time
    snaf_dev_t       *dev;
    void            (*ecc_encode)(struct snaf_info_s *info, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare);
    int             (*ecc_decode)(struct snaf_info_s *info, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare);
} snaf_info_t;

#define SNAF_MODEL_UNIT_NUM_BLK         (512)
#define SNAF_MODEL_NUM_BLK_512          (0) //512*(1<<0)
#define SNAF_MODEL_NUM_BLK_1024         (1) //512*(1<<1)
#define SNAF_MODEL_NUM_BLK_2048         (2) //512*(1<<2)
#define SNAF_MODEL_NUM_BLK_4096         (3) //512*(1<<3)
#define SNAF_MODEL_NUM_BLK_8192         (4) //512*(1<<3)

#define SNAF_MODEL_UNIT_NUM_PAGE        (32)
#define SNAF_MODEL_NUM_PAGE_32          (0) // 32*(1<<0)
#define SNAF_MODEL_NUM_PAGE_64          (1) // 32*(1<<1)
#define SNAF_MODEL_NUM_PAGE_128         (2) // 32*(1<<2)
#define SNAF_MODEL_NUM_PAGE_256         (3) // 32*(1<<3)
#define SNAF_MODEL_NUM_PAGE_512         (4) // 32*(1<<4)
#define SNAF_MODEL_NUM_PAGE_1024        (5) // 32*(1<<5)

#define SNAF_MODEL_UNIT_PAGE_SIZE       (1024)
#define SNAF_MODEL_PAGE_SIZE_1024B      (0) //1024*(1<<0)
#define SNAF_MODEL_PAGE_SIZE_2048B      (1) //1024*(1<<1)
#define SNAF_MODEL_PAGE_SIZE_4096B      (2) //1024*(1<<2)
#define SNAF_MODEL_PAGE_SIZE_8192B      (3) //1024*(1<<2)

#define SNAF_MODEL_UNIT_SPARE_SIZE      (32)
#define SNAF_MODEL_SPARE_SIZE_32B       (0) //  32*(1<<0)
#define SNAF_MODEL_SPARE_SIZE_64B       (1) //  32*(1<<1)
#define SNAF_MODEL_SPARE_SIZE_128B      (2) //  32*(1<<2)
#define SNAF_MODEL_SPARE_SIZE_256B      (3) //  32*(1<<3)
#define SNAF_MODEL_SPARE_SIZE_512B      (4) //  32*(1<<4)
#define SNAF_MODEL_SPARE_SIZE_1024B     (5) //  32*(1<<5)

#define SNAF_MODEL_UNIT_OOB_SIZE        (4)
#define SNAF_MODEL_OOB_6T               (5)   (((oob_byte)/SNAF_MODEL_UNIT_OOB_SIZE)-1)

#define SNAF_NUM_OF_PAGE_PER_BLK(__info) ((1<<(__info)->pm.num_page) * SNAF_MODEL_UNIT_NUM_PAGE)
#define SNAF_NUM_OF_BLOCK(__info)        ((1<<(__info)->pm.num_blk)* SNAF_MODEL_UNIT_NUM_BLK)
#define SNAF_PAGE_SIZE(__info)           ((1<<(__info)->pm.page_size) * SNAF_MODEL_UNIT_PAGE_SIZE)
#define SNAF_SPARE_SIZE(__info)          ((1<<(__info)->pm.spare_size) * SNAF_MODEL_UNIT_SPARE_SIZE)
#define SNAF_BLOCK_SIZE(__info)          (SNAF_NUM_OF_PAGE_PER_BLK(__info) * SNAF_PAGE_SIZE(__info))
#define SNAF_FULL_PAGE_SIZE(__info)      (SNAF_PAGE_SIZE(__info)+SNAF_SPARE_SIZE(__info))
#define SNAF_CHIP_SIZE(__info)           (SNAF_NUM_OF_BLOCK(__info)*SNAF_BLOCK_SIZE(__info))

#define SNAF_BA_BIT(__info)              ((__info)->pm.num_blk+9)
#define SNAF_PA_BIT(__info)              ((__info)->pm.num_page+5)
#define SNAF_CA_BIT(__info)              ((__info)->pm.page_size+10)
#define SNAF_BADDR(__info, __off)        ((__off)>>SNAF_CA_BIT(__info))
#define SNAF_CADDR(__info, __off)        ((__off)&((1<<SNAF_CA_BIT(__info))-1))

// PLANE
#define SNAF_PLANE_BIT(__info, _baddr)    ((_baddr) >> SNAF_PA_BIT(__info) & 0x1);
#define SNAF_PLANE_BIT_OFF(__info)        (SNAF_CA_BIT(__info)+1)

// ARCH
#define CHECK_ARCH(__m, __t)    ((__t)==((__m)&(__t)))
#define CHECK_DID(_p, _id)     ((_p)->did == ((_id&0xFFFFFF)>>((3-((_p)->did_len))*8)))

// ecc
#define SNAF_ECC_NO_BITFLIPS      (0)
#define SNAF_ECC_HAS_LT_BITFLIPS  (1)   // less than max bit-flips
#define SNAF_ECC_UNCOR_ERROR      (2)
#define SNAF_ECC_HAS_ELT_BITFLIPS (3)   // equal or less than max bit-flips

#define SNAF_SERIAL_IO            (1 << 0)
#define SNAF_DUAL_IO              (1 << 1)
#define SNAF_QUAD_IO              (1 << 2)
#define SNAF_FORCE_PIO            (1 << 4)
#define SNAF_EXTRA_ECSP           (1 << 5)  // extra ecc space
#define SNAF_ON_DIE_ECC           (1 << 6)  // LUT support on-die ecc
#define SNAF_PLANE_ARCH           (1 << 7)
#define SNAF_STACK_DIE            (1 << 8)

// default timing and parameters
#define SNAF_DEF_TPROG      (600)
#define SNAF_DEF_TBERASE    (10000)
#define SNAF_DEF_TREAD      (120)
#define SNAF_DEF_TRST       (600)
#define SNAF_DEF_NUM_LUN    (1)
#define SNAF_DEF_IO_MODE    (SNAF_SERIAL_IO)

#define MID_DEFAULT    (0x23)
#define MID_KIOXIA     (0x98)
#define MID_WINBOND    (0xEF)
#define MID_MXIC       (0xC2)
#define MID_ETRON      (0xD5)
#define MID_MICRON     (0x2C)
#define MID_DOSILICON  (0xE5)
#define MID_FORESEE    (0xCD)
#define MID_XTX        (0x0B)
#define MID_GD         (0xC8)
#define MID_ESMT       (0xC8)   //? from GD and Micron
#define MID_HEYANGTEK  (0xC9)
#define MID_END        (0)
#define DID_DEFAULT    (0x79)
#define DID_RPP        (0x9a2a)

#define ABS_DEV(a, b)   (((a)>(b))?((a)-(b)):((b)-(a)))

extern snaf_dev_t snaf_dev;
extern snaf_dev_t snaf_dev_serial;
extern snaf_dev_t snaf_dev_dual;
extern snaf_dev_t snaf_dev_quad;
extern snaf_info_t snaf_info;

// Tier-III (include several Commands)
void snand_get_version(void);

void snand_set_feature_bits(snaf_info_t *info, uint8_t cs, uint16_t f_addr, uint8_t bit_offset, uint8_t bit_len, uint8_t value);
void snand_set_feature(snaf_info_t *info, uint8_t cs, uint16_t f_addr, uint8_t v);
void snand_set_blk_lock(snaf_info_t *info, uint8_t cs, uint8_t v);
void snand_set_ecc_en(snaf_info_t *info, uint8_t cs, uint8_t v);
void snand_set_wel(snaf_info_t *info, uint8_t cs, uint8_t v);
void snand_quad_enable(snaf_info_t *info, uint8_t cs, uint8_t v);
void snand_quad_enable2(snaf_info_t *info, uint8_t cs, uint8_t v);

uint8_t snand_get_feature_bits(snaf_info_t *info, uint8_t cs, uint16_t f_addr, uint8_t bit_offset, uint8_t bit_len);
uint8_t snand_get_feature(snaf_info_t *info, uint8_t cs, uint16_t f_addr);
uint8_t snand_get_blk_lock(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_ecc_en(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_oip(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_wel(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_efail(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_pfail(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_ecfail(snaf_info_t *info, uint8_t cs);
uint8_t snand_get_ecfail2(snaf_info_t *info, uint8_t cs);
int snand_wait_oip(snaf_info_t *info, uint8_t cs, uint32_t to_us, const char *s);

void snand_page_read(snaf_info_t *info, uint8_t cs, uint32_t off, uint8_t *payload, uint8_t *spare);
int snand_page_prog(snaf_info_t *info, uint8_t cs, uint32_t off, uint8_t *payload, uint8_t *spare);
int snand_block_erase(snaf_info_t *info, uint8_t cs, uint32_t off);
void snand_lock(snaf_info_t *info, uint8_t cs, uint8_t lock);
void snand_unlock(snaf_info_t *info, uint8_t cs);
void snand_rdid(snaf_info_t *info, uint8_t cs, uint8_t *id);
void snand_reset(snaf_info_t *info, uint8_t cs);
void snand_read_para_page(snaf_info_t *info, uint32_t para_addr, uint8_t *payload);

#endif //___SPI_NAND_CORE_H__
