#include <spi_nand_core.h>
#include <spi_nand_xfer_cmd.h>
#include <sihnah.h>
#include <asm/io.h>
#include <asm/cacheflush.h>

/*************************************************************************
  *  Definitions of IO_WIDTH / Length / Status / PIO_Trigger / DMA_Trigger / CS_Select
  *************************************************************************/

#define LEN2REG(len) ((len)-1)
#define IO_WIDTH_LEN(io_width, len) (uint32_t)(((io_width)<<28)|(len))


#define WAIT_NAND_CTRLR_RDY() while(RFLD(SNAFSR, nfcos))
#define WAIT_DMA_CTRLR_RDY() while((readl(RRVAL(SNAFSR))&0x06)!=0)

#if 1
#define XFER_PIO_WR(unit) ({    \
    WAIT_NAND_CTRLR_RDY();\
    writel((volatile uint32_t)(((unit).mode<<28)|LEN2REG((unit).nbytes)), RRVAL(SNAFWCMR));\
    writel((unit).v<<(8*(4-(unit).nbytes)), RRVAL(SNAFWR));\
})
#else
    WAIT_NAND_CTRLR_RDY();\
    RVAL(SNAFWCMR) = (uint32_t)(((unit).mode<<28)|LEN2REG((unit).nbytes));\
    RVAL(SNAFWR) = (unit).v<<(8*(4-(unit).nbytes));\
})
#endif


#define SPI_NAND_CS(_cs, act_deact)({\
    if(_cs) {\
        RMOD(SNAFCCR, cecs1, act_deact);\
        while(RFLD(SNAFSR, cs1)!=act_deact);\
    } else {\
        RMOD(SNAFCCR, cecs0, act_deact);\
        while(RFLD(SNAFSR, cs0)!=act_deact);}\
})

#define ACT_CS(_cs) ({\
    WAIT_NAND_CTRLR_RDY();\
    SPI_NAND_CS(_cs, 1);\
    WAIT_NAND_CTRLR_RDY();\
    SPI_NAND_CS(_cs, 0);\
})

#define DEACT_CS(_cs) ({\
    WAIT_NAND_CTRLR_RDY();\
    SPI_NAND_CS(_cs, 1);\
    WAIT_NAND_CTRLR_RDY();\
})

/* Utilities */
#define UNIT_BLK_SHF     (9) // 512
#define UNIT_PGE_BLK_SHF (PAGE_SHF+UNIT_BLK_SHF)
#define PGE_BLK_SHF(b)   (b+UNIT_PGE_BLK_SHF)
#define WSIZE    (4)
#define HSIZE    (2)
#define BSIZE    (1)

snaf_dev_t
snaf_dev_serial = { .rd_op = OP_READ,
                    .rd_amode = snaf_sio,
                    .rd_dmode = snaf_sio,
                    .rd_xfer_mode = 0,
                    .dummy_bytes = 1,
                    .wr_op = OP_PROG_LOAD,
                    .wr_amode = snaf_sio,
                    .wr_dmode = snaf_sio,
                    .wr_xfer_mode = 0};

snaf_dev_t
snaf_dev_dual  = { .rd_op = OP_READ_X2,
                   .rd_amode = snaf_sio,
                   .rd_dmode = snaf_dio,
                   .rd_xfer_mode = 0,
                   .dummy_bytes = 1,
                   .wr_op = OP_PROG_LOAD,
                   .wr_amode = snaf_sio,
                   .wr_dmode = snaf_sio,
                   .wr_xfer_mode = 0};

snaf_dev_t
snaf_dev_quad = { .rd_op = OP_READ_X4,
                  .rd_amode = snaf_sio,
                  .rd_dmode = snaf_qio,
                  .rd_xfer_mode = 0,
                  .dummy_bytes = 1,
                  .wr_op = OP_PROG_LOAD_X4,
                  .wr_amode = snaf_sio,
                  .wr_dmode = snaf_qio,
                  .wr_xfer_mode = 0};


// Tier-I (SPI-NAND Single Transaction, CMD+ADDR+DUMMY+DATA)
#define WORD_SIZE_CHECK(remaining, payload)  ((remaining)>=WSIZE && 0==((uint64_t)(payload)&0x3))
#define HWORD_SIZE_CHECK(remaining, payload) ((remaining)>=HSIZE && 0==((uint64_t)(payload)&0x1))

#define WDR     *(volatile uint32_t *)DR
#define HDR     *((volatile uint16_t *)DR+1)   // offset for controller behavior
#define BDR     *(DR+3)

static void snc_pio_read_write(snaf_xfer_attr_t *xfer)
{
    uint32_t bytes, _len, i;
    uint8_t *payload;
    volatile uint8_t *DR;
    uint32_t rtmp_32;
    uint16_t rtmp16;
    uint8_t  rtmp8;

    for (i=0; i<2; i++) {
        bytes = xfer->data[i].nbytes;
        if (bytes==0) continue;
        payload = xfer->data[i].v;
        VERBOSE("%s, payload %llx, Data (%d) %d bytes, mode %d\n", __FUNCTION__, (uint64_t)payload, xfer->pdir, xfer->data[i].nbytes, xfer->data[i].mode);

        if (xfer->pdir == snaf_xfer_rx ) {
            //DR = (volatile uint8_t *)SNAFRDRar;
            while(bytes) {
                WAIT_NAND_CTRLR_RDY();
                if(WORD_SIZE_CHECK(bytes, payload)) {
                    writel(IO_WIDTH_LEN(xfer->data[i].mode, LEN2REG(WSIZE)), RRVAL(SNAFRCMR));
                    WAIT_NAND_CTRLR_RDY();
                    rtmp_32 = readl(RRVAL(SNAFRDR));
                    *(uint32_t *)(payload) = __BE32_TO_CPU(rtmp_32);
                    _len = WSIZE;
                } else if(HWORD_SIZE_CHECK(bytes, payload)) {
                    writew(IO_WIDTH_LEN(xfer->data[i].mode, LEN2REG(HSIZE)), RRVAL(SNAFRCMR));
                    WAIT_NAND_CTRLR_RDY();
                    rtmp16 = readw(RRVAL(SNAFRDR)+2);
                    *(uint16_t *)(payload) = __BE16_TO_CPU(rtmp16);
                    _len = HSIZE;
                } else {
                    writeb(IO_WIDTH_LEN(xfer->data[i].mode, LEN2REG(BSIZE)), RRVAL(SNAFRCMR));
                    WAIT_NAND_CTRLR_RDY();
                    rtmp8 = readb(RRVAL(SNAFRDR)+3);
                    *payload = rtmp8;
                    _len = BSIZE;
                }
                payload += _len;
                bytes -= _len;
            }
        } else if (xfer->pdir == snaf_xfer_tx ) {    // TX
            DR = (volatile uint8_t *)SNAFWRar;
            while(bytes) {
                WAIT_NAND_CTRLR_RDY();
                if(WORD_SIZE_CHECK(bytes, payload)) {
                    writel(IO_WIDTH_LEN(xfer->data[i].mode, LEN2REG(WSIZE)), RRVAL(SNAFWCMR));
                    WAIT_NAND_CTRLR_RDY();
                    writel(__CPU_TO_BE32(*(uint32_t *)(payload)), RRVAL(SNAFRDR));
                    _len = WSIZE;
                } else if(HWORD_SIZE_CHECK(bytes, payload)) {
                    writel(IO_WIDTH_LEN(xfer->data[i].mode, LEN2REG(HSIZE)), RRVAL(SNAFWCMR));
                    WAIT_NAND_CTRLR_RDY();
                    writew(__CPU_TO_BE16(*(uint16_t *)(payload)), RRVAL(SNAFRDR)+2);
                    _len = HSIZE;
                } else {
                    writel(IO_WIDTH_LEN(xfer->data[i].mode, LEN2REG(BSIZE)), RRVAL(SNAFWCMR));
                    WAIT_NAND_CTRLR_RDY();
                    writeb(*(payload), RRVAL(SNAFRDR)+3);
                    _len = BSIZE;
                }
                payload += _len;
                bytes -= _len;
            }
        } else {
            VERBOSE("WW: unknown !\n");
        }
    }
}

static void snc_dma_read_write(snaf_xfer_attr_t *xfer)
{
    uint32_t bytes, i;
    uint8_t *payload;

    for (i=0; i<2; i++) {
        bytes = xfer->data[i].nbytes;
        payload = xfer->data[i].v;
        if (bytes) {
            VERBOSE("%s, payload %llx ,Data (pdir %d) %d bytes, mode %d\n", __FUNCTION__, (uint64_t)payload, xfer->pdir, xfer->data[i].nbytes, xfer->data[i].mode);

            //DCACHE_FLUSH((uint64_t)payload, bytes);  // start, size
            __flush_dcache_area((uint64_t)payload, bytes);  // start, size
            WAIT_NAND_CTRLR_RDY();
            writel((uint64_t)payload & ((uint32_t)-1), RRVAL(SNAFDRSAR));
            writel(IO_WIDTH_LEN(xfer->data[i].mode, bytes), RRVAL(SNAFDLR));
            WAIT_NAND_CTRLR_RDY();
            writel(xfer->pdir & 0x1, RRVAL(SNAFDTR));
            WAIT_DMA_CTRLR_RDY();
        }
    }
}

uint32_t snc_single_cmd(snaf_xfer_attr_t *xfer)
{
    /* cs */
    ACT_CS(xfer->cs);

    /* cmd */
    VERBOSE("%s, CMD(%02x)\n", __FUNCTION__, xfer->cmd.v);
    XFER_PIO_WR(xfer->cmd);

    /* addr */
    if (xfer->addr.nbytes) {
        VERBOSE("%s, ADDR(%03x) %d bytes, mode %d\n", __FUNCTION__, xfer->addr.v, xfer->addr.nbytes, xfer->addr.mode);
        XFER_PIO_WR(xfer->addr);
    }

    /* dummy before data */
    if (xfer->dummy.nbytes) {
        VERBOSE("%s, Dummy %d bytes, mode %d\n", __FUNCTION__, xfer->dummy.nbytes, xfer->dummy.mode);
        XFER_PIO_WR(xfer->dummy);
    }

    /* data */
    uint32_t rd_val = 0;
    if (xfer->data[0].nbytes || xfer->data[1].nbytes) {
        VERBOSE("%s, DATA pdir = %d\n", __FUNCTION__, xfer->pdir);
        if (xfer->pdir>>1) {
            snc_dma_read_write(xfer);
        } else {
            snc_pio_read_write(xfer);
        }
    }

    DEACT_CS(xfer->cs);
    return rd_val;
}

// Tier-II (SPI-NAND Single Command)
static void sncmd_read_id(uint8_t cs, uint8_t *v, uint32_t len)
{
    snaf_xfer_attr_t xfer = XFER_CMD_RDID(cs, OP_RDID, 0, v, len);
    snc_single_cmd(&xfer);
}

static void sncmd_write_enable(uint8_t cs)
{
    snaf_xfer_attr_t xfer = XFER_CMD_ONLY(cs, OP_WRITE_ENABLE);
    snc_single_cmd(&xfer);
}

static void sncmd_write_disable(uint8_t cs)
{
    snaf_xfer_attr_t xfer = XFER_CMD_ONLY(cs, OP_WRITE_DISABLE);
    snc_single_cmd(&xfer);
}

static void sncmd_set_feature(uint8_t cs, uint32_t f_addr, uint32_t conf)
{
    uint8_t v = conf;
    snaf_xfer_attr_t xfer = XFER_CMD_SET(cs, OP_SET_FEATURE, f_addr, &v);
    snc_single_cmd(&xfer);
}

static uint8_t sncmd_get_feature(uint8_t cs, uint32_t f_addr)
{
    uint8_t v=0;
    snaf_xfer_attr_t xfer = XFER_CMD_GET(cs, OP_GET_FEATURE, f_addr, &v);
    snc_single_cmd(&xfer);
    return v;
}

static void sncmd_read_buffer(uint8_t cs, snaf_payload_t *p)
{
    uint8_t dir = snaf_xfer_rx;;
    if(p->dma) { dir = snaf_xfer_rx_dma; }
    snaf_xfer_attr_t xfer = _XFER_CMD_READ(cs, dir, p->op,
                                           p->addr, p->amode, \
                                           p->data, p->dsize, \
                                           p->spare, p->ssize, \
                                           p->dmode, p->dummy_bytes);
    snc_single_cmd(&xfer);
}

static void sncmd_program_load(uint8_t cs, snaf_payload_t *p)
{
    uint8_t dir = snaf_xfer_tx;;
    if(p->dma) { dir = snaf_xfer_tx_dma; }
    snaf_xfer_attr_t xfer = _XFER_CMD_PROG(cs, dir, p->op,
                                           p->addr, p->amode, \
                                           p->data, p->dsize, \
                                           p->spare, p->ssize, \
                                           p->dmode);
    snc_single_cmd(&xfer);
}

static void sncmd_page_data_read(uint8_t cs, uint32_t b_addr)
{
    snaf_xfer_attr_t xfer = XFER_CMD_EXEC(cs, OP_PAGE_DATA_READ, b_addr, 3);
    snc_single_cmd(&xfer);
}

static void sncmd_program_execute(uint8_t cs, uint32_t b_addr)
{
    snaf_xfer_attr_t xfer = XFER_CMD_EXEC(cs, OP_PROG_EXEC,b_addr, 3);
    snc_single_cmd(&xfer);
}

static void sncmd_block_erase(uint8_t cs, uint32_t b_addr)
{
    snaf_xfer_attr_t xfer = XFER_CMD_EXEC(cs, OP_BLOCK_ERASE,b_addr, 3);
    snc_single_cmd(&xfer);
}

static void sncmd_reset(uint8_t cs)
{
    snaf_xfer_attr_t xfer = XFER_CMD_ONLY(cs, OP_RESET);
    snc_single_cmd(&xfer);
}

static void sncmd_die_select(uint8_t cs, uint32_t die_id)
{
    snaf_xfer_attr_t xfer = XFER_OP(cs, snaf_xfer_tx,
                                    XFER_CMD(OP_DIE_SELECT),
                                    XFER_ADDR(die_id, 1, snaf_sio),
                                    XFER_NONE, XFER_NONE, XFER_NONE);
    snc_single_cmd(&xfer);
}

// Tier-III (include several Commands)
uint8_t snand_get_feature_bits(snaf_info_t *info, uint8_t cs, uint16_t f_addr, uint8_t bit_offset, uint8_t bit_len)
{
    uint8_t s = sncmd_get_feature(cs, f_addr);
    VERBOSE("%s %x = %x\n", __FUNCTION__, f_addr, s);
    return (s>>bit_offset)&((1<<bit_len)-1);
}

uint8_t snand_get_feature(snaf_info_t *info, uint8_t cs, uint16_t f_addr)
{
    return snand_get_feature_bits(info, cs, f_addr, 0, 8);
}

uint8_t snand_get_blk_lock(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xA0, 3, 3);
}

uint8_t snand_get_blk_protect_en(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xB0, 2, 1);
}

uint8_t snand_get_ecc_en(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xB0, 4, 1);
}

uint8_t snand_get_oip(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xC0, 0, 1);
}

uint8_t snand_get_wel(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xC0, 1, 1);
}

uint8_t snand_get_efail(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xC0, 2, 1);
}

uint8_t snand_get_pfail(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xC0, 3, 1);
}

uint8_t snand_get_ecfail(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xC0, 4, 2);
}

uint8_t snand_get_ecfail2(snaf_info_t *info, uint8_t cs)
{
    return snand_get_feature_bits(info, cs, 0xC0, 4, 3);
}

#if defined(CONFIG_UNDER_UBOOT)
    unsigned long get_tbclk(void);
    #define GET_CPU_MHZ() (get_tbclk()/1000000)
    unsigned long timer_read_counter(void);
    #define GET_CPU_CYCLE() timer_read_counter()
#else
    unsigned int plat_get_syscnt_freq2(void);
    #define GET_CPU_MHZ plat_get_syscnt_freq2
    //static uint32_t get_timer_value(void) { return (uint32_t)(~read_cntpct_el0()); }
    //#define GET_CPU_CYCLE() get_timer_value()
    #define GET_CPU_CYCLE() 1
#endif

//static uint32_t us_to_c(uint32_t us) { return (us * GET_CPU_MHZ() ); }
static uint32_t us_to_c(uint32_t us) { return 10; }

int snand_wait_oip(snaf_info_t *info, uint8_t cs, uint32_t to_us, const char *s)
{
    uint32_t start, now;
    uint32_t to = us_to_c(to_us) * 5;   // timeout * 5
    now = start = GET_CPU_CYCLE();
    while (1==snand_get_oip(info, cs))
    {
        if(ABS_DEV(start, now) > to) { // DOWNCOUNT!!
            ERROR("Wait oip Timeout! (%d us, %s)\n", to_us, s);
            return -1;  // TIMEOUT
        }
        now = GET_CPU_CYCLE();
    }
    return 0;
}

void snand_set_feature_bits(snaf_info_t *info, uint8_t cs, uint16_t f_addr, uint8_t bit_offset, uint8_t bit_len, uint8_t value)
{
    uint8_t s = sncmd_get_feature(cs, f_addr);
    VERBOSE("%s get_feature %x\n", __FUNCTION__, s);

    sncmd_write_enable(cs);
    s &= ~(((1<<bit_len)-1) << bit_offset);
    s |= (value&((1<<bit_len)-1))<<bit_offset;
    sncmd_set_feature(cs, f_addr, s);
    VERBOSE("%s %x = %x\n", __FUNCTION__, f_addr, s);

    //The users must use the Write Enable (06h) or the Write Disable (04h) command to switch the WEL bit since Set Feature command cannot change it.
    sncmd_write_disable(cs);
}

void snand_set_feature(snaf_info_t *info, uint8_t cs, uint16_t f_addr, uint8_t v)
{
    return snand_set_feature_bits(info, cs, f_addr, 0, 8, v);
}

void snand_set_blk_lock(snaf_info_t *info, uint8_t cs, uint8_t v)
{
    // winbond bit[7:3]
    // other bit[6:3]
    return snand_set_feature_bits(info, cs, 0xA0, 3, 4, v);
}

void snand_set_otp_en(snaf_info_t *info, uint8_t cs, uint8_t v)
{
    return snand_set_feature_bits(info, cs, 0xB0, 6, 1, v);
}

void snand_set_ecc_en(snaf_info_t *info, uint8_t cs, uint8_t v)
{
    return snand_set_feature_bits(info, cs, 0xB0, 4, 1, v);
}

void snand_quad_enable(snaf_info_t *info, uint8_t cs, uint8_t v)
{
    /* Type 1: B0 bit0 */
    return snand_set_feature_bits(info, cs, 0xB0, 0, 1, v);
}

void snand_quad_enable2(snaf_info_t *info, uint8_t cs, uint8_t v)
{
    /* Type 2: A0 bit1 */
    return snand_set_feature_bits(info, cs, 0xA0, 1, 1, v);
}

void snand_set_wel(snaf_info_t *info, uint8_t cs, uint8_t v)
{
    return snand_set_feature_bits(info, cs, 0xC0, 1, 1, v);
}

// support up to two dies
static void snand_die_xfer(snaf_info_t *info, uint32_t *off, uint8_t *die)
{
    uint32_t unit_die_size = SNAF_CHIP_SIZE(info)/2;
    if (*off > unit_die_size) {
        *die = 1;
    } else {
        *die = 0;
    }
    *off -= unit_die_size * *die;
}

void snand_page_read(snaf_info_t *info, uint8_t cs, uint32_t off, uint8_t *payload, uint8_t *spare)
{
    uint8_t die = CHECK_ARCH(info->pm.mode, SNAF_STACK_DIE);
    uint8_t dma = CHECK_ARCH(info->pm.mode, SNAF_FORCE_PIO)?0:1;
    uint8_t die_id = 0;
    uint8_t plane_bit;
    uint32_t b_addr, c_addr;
    uint32_t _off = off;

    if (die) {
        snand_die_xfer(info, &_off, &die_id);
        sncmd_die_select(cs, die_id);
    }

    b_addr = SNAF_BADDR(info, _off);
    c_addr = SNAF_CADDR(info, _off);
    /* data read from flash to flash cache */
    sncmd_page_data_read(cs, b_addr);

    snand_wait_oip(info, cs, info->tread, __FUNCTION__);

    /* process plane bit */
    plane_bit = SNAF_PLANE_BIT(info, b_addr);
    c_addr |=  plane_bit << SNAF_PLANE_BIT_OFF(info);

    VERBOSE("%s, off %x, die%d, ba %x, ca %x, p %x  with %s\n",
              __FUNCTION__, _off, die_id, b_addr, c_addr,
              plane_bit, dma?"dma":"pio");

    snaf_dev_t *d = info->dev;
    snaf_payload_t _p = { .op = d->rd_op, .dummy_bytes = d->dummy_bytes,
                          .addr = c_addr, .amode = d->rd_amode,
                          .data = payload, .dsize = SNAF_PAGE_SIZE(info),
                          .spare = spare,  .ssize = SNAF_SPARE_SIZE(info),
                          .dmode = d->rd_dmode, .dma = dma};

    if (payload==NULL) {
        _p.addr += SNAF_PAGE_SIZE(info); // read spare only
        _p.dsize = 0;
    }
    if (spare==NULL) { _p.ssize = 0; }

    /* do data read from cache to dest. */
    sncmd_read_buffer(cs, &_p);

    // select back to die0
    if (die) { sncmd_die_select(cs, 0); }
}

int snand_page_prog(snaf_info_t *info, uint8_t cs, uint32_t off, uint8_t *payload, uint8_t *spare)
{
    uint8_t die = CHECK_ARCH(info->pm.mode, SNAF_STACK_DIE);
    uint8_t dma = CHECK_ARCH(info->pm.mode, SNAF_FORCE_PIO)?0:1;
    uint8_t die_id = 0;
    uint8_t plane_bit;
    uint32_t b_addr, c_addr;
    uint32_t _off = off;
    int ret;

    if (die) {
        snand_die_xfer(info, &_off, &die_id);
        sncmd_die_select(cs, die_id);
    }

    b_addr = SNAF_BADDR(info, _off);
    c_addr = SNAF_CADDR(info, _off);

    /* write enable */
    sncmd_write_enable(cs);

    /* process plane bit */
    plane_bit = SNAF_PLANE_BIT(info, b_addr);
    c_addr |=  plane_bit << SNAF_PLANE_BIT_OFF(info);

    VERBOSE("%s, off %x, die%d, ba %x, ca %x, p %x  with %s\n",
              __FUNCTION__, _off, die_id, b_addr, c_addr,
              plane_bit, dma?"dma":"pio");

    snaf_dev_t *d = info->dev;
    snaf_payload_t _p = { .op = d->wr_op, .dummy_bytes = 0,
                          .addr = c_addr, .amode = d->wr_amode,
                          .data = payload, .dsize = SNAF_PAGE_SIZE(info),
                          .spare = spare,  .ssize = SNAF_SPARE_SIZE(info),
                          .dmode = d->wr_dmode, .dma = dma };
    /* data load from src. to flash cache */
    sncmd_program_load(cs, &_p);

    /* data program from flash cache to flash */
    sncmd_program_execute(cs, b_addr);
    if (snand_wait_oip(info, cs, info->tprog, __FUNCTION__)) {
        ret = -2; goto done;
    }
    ret = snand_get_pfail(info, cs);
done:
    // select back to die0
    if (die) { sncmd_die_select(cs, 0); }

    return ret;
}

int snand_block_erase(snaf_info_t *info, uint8_t cs, uint32_t off)
{
    uint8_t die = CHECK_ARCH(info->pm.mode, SNAF_STACK_DIE);
    uint8_t die_id = 0;
    uint32_t b_addr;
    uint32_t _off = off;
    int ret;
    if (die) {
        snand_die_xfer(info, &_off, &die_id);
        sncmd_die_select(cs, die_id);
    }

    b_addr = SNAF_BADDR(info, off);
    VERBOSE("%s off %x die%d, b_addr %x\n",
             __FUNCTION__, off, die_id, b_addr);

    sncmd_write_enable(cs);
    sncmd_block_erase(cs, b_addr);
    if (snand_wait_oip(info, cs, info->tberase, __FUNCTION__)) {
        ret = -2; goto done;
    }

    ret = snand_get_efail(info, cs);
done:
    // select back to die0
    if (die) { sncmd_die_select(cs, 0); }
    return ret;
}

void snand_lock(snaf_info_t *info, uint8_t cs, uint8_t lock)
{
    // winbond bit[7:3]
    // other bit[6:3]
    snand_set_blk_lock(info, cs, 0xf);
}

void snand_unlock(snaf_info_t *info, uint8_t cs)
{
    snand_set_blk_lock(info, cs, 0);
}

void snand_rdid(snaf_info_t *info, uint8_t cs, uint8_t *id)
{
    sncmd_read_id(cs, id, SNAF_MAX_ID_LEN);
}

void snand_reset(snaf_info_t *info, uint8_t cs)
{
    /* Data corruption may happen if there is an on-going internal Erase or Program operation when Reset command sequence is accepted by the device. It is recommended to check the BUSY bit in Status Register before issuing the Reset command. */

    snand_wait_oip(info, cs, 1000, __FUNCTION__);
    return sncmd_reset(cs);
}

void snand_read_para_page(snaf_info_t *info, uint32_t para_addr, uint8_t *payload)
{
    snand_set_otp_en(info, 0, 1);
    sncmd_page_data_read(0, para_addr);     // some used 0x0, and some used 0x01
    snand_wait_oip(info, 0, info->tread, __FUNCTION__);
    snaf_payload_t _p = { .op = OP_READ, .dummy_bytes = 1,
                          .addr = 0, .amode = snaf_sio,
                          .data = payload, .dsize = 256,
                          .spare = NULL,  .ssize = 0,
                          .dmode = snaf_sio, .dma = 1 };
    sncmd_read_buffer(0, &_p);
    snand_set_otp_en(info, 0, 0);
}

