#include <spi_nand_core.h>
#include <spi_nand_common.h>

//#include <ecc_core.h>
#include <ecc.h>

static int spi_nand_read_param_page(snaf_info_t *info, uint32_t mid);

snaf_dev_t snaf_dev = { .rd_op = OP_READ,
                        .rd_amode = snaf_sio,
                        .rd_dmode = snaf_sio,
                        .rd_xfer_mode = 0,
                        .dummy_bytes = 1,
                        .wr_op = OP_PROG_LOAD,
                        .wr_amode = snaf_sio,
                        .wr_dmode = snaf_sio,
                        .wr_xfer_mode = 0};

snaf_info_t snaf_info = {
    .pm = SNDB(MID_DEFAULT, "PARA-PAGE", DID_DEFAULT, 1024, 64, 2048,  64, 0, SNAF_DEF_IO_MODE|SNAF_ON_DIE_ECC),
    .num_lun = SNAF_DEF_NUM_LUN,
    .tprog = SNAF_DEF_TPROG,
    .tread = SNAF_DEF_TREAD,
    .tberase = SNAF_DEF_TBERASE,
    .ecc_decode = spi_nand_ecc_decode,
    .ecc_encode = spi_nand_ecc_encode,
    .dev = &snaf_dev
};

// Tier-IV (Function include Operations and ECC check)
void spi_nand_init(snaf_info_t *info)
{
    uint8_t cs;
    uint32_t id;
    snaf_dev_t *tmp_dev;

    // check cs count
    for (cs=1; cs<SNAF_MAX_NUM_LUN; cs++) {
        snand_rdid(&snaf_info, cs, (uint8_t*)&id);
        id = __CPU_TO_BE32(id);
        if(1==CHECK_DID(&(info->pm), id)) {
           info->num_lun += 1;
        }
    }

    if (0!=CHECK_ARCH(info->pm.mode, SNAF_QUAD_IO)) {
        tmp_dev = &snaf_dev_quad;
    } else if (0!=CHECK_ARCH(info->pm.mode, SNAF_DUAL_IO)) {
        tmp_dev = &snaf_dev_dual;
    } else {
        tmp_dev = &snaf_dev_serial;
    }
    MEMCPY(info->dev, tmp_dev, sizeof(snaf_dev_t));
}

#define XFER_SNAF_SETTING(__base, __v) \
    ({  uint32_t __i;        \
        for (__i=0; __i<5; __i++) {\
            if (__v == __base * (1<<__i)) break;\
        }\
        if (__i==5) __i=0;\
        __i;\
    })

static const snaf_manu_t *snaf_manu[] = {
    &dosilicon_manu_info,
    &esmt_manu_info,
    &etron_manu_info,
    &foresee_manu_info,
    &gd_manu_info,
    &hyt_manu_info,
    &kioxia_manu_info,
    &micron_manu_info,
    &mxic_manu_info,
    &winbond_manu_info,
    &xtx_manu_info,
};

__attribute__ ((weak))
int spi_nand_conjecture(snaf_info_t *info, uint32_t mode)
{
    // seeking the 1st valid page
    return 0;
}

int spi_nand_probe(uint32_t mode)
{
    snaf_info_t *info;
    uint32_t id, mid;
    const snaf_manu_t *manu = NULL;
    char *name;
    uint32_t i, j;
    RMOD(SNAFCFR, dma_endian, 1); // DMA to Little

    snand_rdid(&snaf_info, 0, (uint8_t*)&id);
    id = __CPU_TO_BE32(id);
    mid = id >> 24;

    // traverse ROM
    for (i=0; i<ARRAY_SIZE(snaf_manu); i++) {
        manu = snaf_manu[i];
        if (manu->mid != mid) continue;
        for(j=0; j<manu->nchips; j++) {
            const snaf_pm_t *chip = &(manu->chips[j]);
            if(1==CHECK_DID(chip, id)) {
                name = manu->mname;
                memcpy(&(snaf_info.pm), chip, sizeof(snaf_pm_t));
                goto done;
            }
        }
    }

    // no database found, try read param page
    if (0==spi_nand_read_param_page(&snaf_info, mid)) {
        goto manu_init;
    } else {
        ERROR("No Parameter Page Avialable (Unknown ID: %x)\n", id);
    }

    // get page size
    if (0==spi_nand_conjecture(&snaf_info, mode)) {
        (&snaf_info)->pm.mid = mid;
        (&snaf_info)->pm.did = (id>>16)&0xFF;
        goto manu_init;
    } else {
        ERROR("Failed!\n");
        return -1;
    }
manu_init:
    for (i=0; i<ARRAY_SIZE(snaf_manu); i++) {
        manu = snaf_manu[i];
        // try to do same manufacturer init here
        if (manu->mid == mid) break;
    }
done:
    info = &snaf_info;

    if (mode!=0) info->pm.mode = mode&0xFFFF;
    // common init flow
    spi_nand_init(info);
    // do manu specific init flow
    if (NULL!=manu && NULL!=manu->init) manu->init(&snaf_info);

    //NOTICE("SPI-NAND \"%s\"\n", info->mname);
    NOTICE("SPI-NAND \"%s\"\n", name);
    NOTICE("ID: %x%x, Size: %d MBx%d, ECC: %d bits, Mode: %x\n", info->pm.mid, info->pm.did, SNAF_CHIP_SIZE(info)>>20, info->num_lun, info->pm.ecc_bits, info->pm.mode);


    return 0;
}

void spi_nand_ecc_encode(struct snaf_info_s *info, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare)
{
    sihnah_ecc_t op = { .algo = conf, .act = ECC_ENCODE };
    uint32_t psize = SNAF_PAGE_SIZE(info);
    uint32_t u_tssize;
    uint32_t i, cnt = psize/ECC_DATA_SZ_B;

    if (conf == ECC_BCH12) {
        u_tssize = ECC_BCH12_SYNDROME_SZ_B + ECC_TAG_SZ_B;
    } else {
        u_tssize = ECC_BCH6_SYNDROME_SZ_B + ECC_TAG_SZ_B;
    }

    for(i=0; i<cnt; i++) {
        ecc_action(&op, payload, spare, spare+6);
        payload += ECC_DATA_SZ_B;
        spare += u_tssize;
    }
}

int spi_nand_ecc_decode(struct snaf_info_s *info, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare)
{
    uint32_t max=0;
    uint32_t sec=0;
    int ret;
    sihnah_ecc_t op = { .algo = conf, .act = ECC_DECODE };
    uint32_t psize = SNAF_PAGE_SIZE(info);
    uint32_t u_tssize;
    uint32_t i, cnt = psize/ECC_DATA_SZ_B;

    if (conf == ECC_BCH12) {
        u_tssize = ECC_BCH12_SYNDROME_SZ_B + ECC_TAG_SZ_B;
    } else {
        u_tssize = ECC_BCH6_SYNDROME_SZ_B + ECC_TAG_SZ_B;
    }

    for(i=0; i<cnt; i++)
    {
        ret = ecc_action(&op, payload, spare, spare+6);
        if (ret==-1) return ret;
        if (ret>max) {
            max = ret;
            sec = i;
        }
        payload += ECC_DATA_SZ_B;
        spare += u_tssize;
    }
    if (max != 0) return (sec<<16 | max);
    return 0;
}

int spi_nand_block_is_bad(struct snaf_info_s *info, uint8_t cs, uint32_t off)
{
    uint8_t buf[256]  __attribute__ ((aligned(4)));
    snand_page_read(info, cs, off, NULL, buf);
    if (*(buf) != 0xff)  {
        return 1;
    }
    return 0;
}

#define CRC_POLYNOM         (0x8005U)
#define CRC_INIT_VALUE      (0x4F4EU)

static uint16_t _check_crc(uint16_t crc, uint8_t *data_in,
                           uint32_t data_len)
{
    uint32_t i;
    uint32_t j;
    uint32_t bit;

    for (i = 0U; i < data_len; i++) {
        uint8_t cur_param = *data_in++;

        for (j = BIT(7); j != 0U; j >>= 1) {
            bit = crc & BIT(15);
            crc <<= 1;

            if ((cur_param & j) != 0U) {
                bit ^= BIT(15);
        }

        if (bit != 0U) {
            crc ^= CRC_POLYNOM;
        }
    }

    crc &= GENMASK(15, 0);
}

    return crc;
}

int spi_nand_read_param_page(snaf_info_t *info, uint32_t mid)
{
    struct nand_param_page para __attribute__ ((aligned(4)));
    uint32_t para_addr;
    uint8_t id[SNAF_MAX_ID_LEN];

    switch(mid)
    {
        case MID_ETRON:
            para_addr = 0;
            break;
        case MID_GD:
            para_addr = 4;
            break;
        default:
            para_addr = 1;
            break;
    }

    snand_read_para_page(info, para_addr, (uint8_t *)&para);
    VERBOSE("sigature          :%s\n", (char *)&para.page_sig);
    VERBOSE("manufacturer      :%s\n", (char *)para.manufacturer);
    VERBOSE("model             :%s\n", (char *)para.model);
    VERBOSE("manufacturer_id   :%x\n", para.manufacturer_id);
    VERBOSE("bytes_per_page    :%d\n", para.bytes_per_page);
    VERBOSE("spare_per_page    :%d\n", para.spare_per_page);
    VERBOSE("num_pages_per_blk :%d\n", para.num_pages_per_blk);
    VERBOSE("num_blk_in_lun    :%d\n", para.num_blk_in_lun);
    VERBOSE("num_lun           :%d\n", para.num_lun);
    VERBOSE("nb_ecc_bits       :%d\n", para.nb_ecc_bits);
    VERBOSE("io_pin_cap_max    :%d\n", para.io_pin_cap_max);
    VERBOSE("tprog             :%d\n", para.tprog);
    VERBOSE("tbers             :%d\n", para.tbers);
    VERBOSE("tr                :%d\n", para.tr);
    VERBOSE("crc16             :%x\n", para.crc16);

    // check signature, "ONFI" or "NAND"
    if (strncmp((char *)&para.page_sig, "NAND", 4) != 0 &&
        strncmp((char *)&para.page_sig, "ONFI", 4) != 0)
    {
        ERROR("Signature Check Failed\n");
        return -1;
    }
    if (_check_crc(CRC_INIT_VALUE, (uint8_t *)&para, 254U) != para.crc16)
    {
        ERROR("CRC Check Failed\n");
        return -2;
    }

    snand_rdid(&snaf_info, 0, id);
    if (id[0] != para.manufacturer_id)
    {
        NOTICE("Manu-ID mismatch! RDID(%x)!=PARA(%x)", id[0], para.manufacturer_id);
        id[0] = para.manufacturer_id;
        id[1] = 0xDD;
    }

    info->pm.mid = id[0];
    info->pm.did = id[1];
    info->pm.did_len = 1;

    // fill parameters
    info->pm.num_blk = XFER_SNAF_SETTING(SNAF_MODEL_UNIT_NUM_BLK, para.num_blk_in_lun);
    info->pm.num_page = XFER_SNAF_SETTING(SNAF_MODEL_UNIT_NUM_PAGE, para.num_pages_per_blk);
    info->pm.page_size = XFER_SNAF_SETTING(SNAF_MODEL_UNIT_PAGE_SIZE, para.bytes_per_page);
    info->pm.spare_size = XFER_SNAF_SETTING(SNAF_MODEL_UNIT_SPARE_SIZE, para.spare_per_page);
    info->pm.ecc_bits = para.nb_ecc_bits;

    if (0==info->pm.ecc_bits) { info->pm.mode |= SNAF_ON_DIE_ECC; info->pm.spare_size+=1; }

    info->tprog = para.tprog;
    info->tberase = para.tbers;
    info->tread = para.tr;
    return 0;
}
