#include <sihnah.h>
#include <ecc.h>
#include <asm/io.h>
#include <asm/cacheflush.h>

#define TRUNC_64_TO_32(_val) ((uint32_t)(((long)(_val)) & 0xffffffff))

/* return:
	 1...24: ecc decode with n correction
	 0: ecc decode without correction, or encode done
	 -1: FUBAR
 */
int ecc_action(const sihnah_ecc_t *op, uint8_t *data, uint8_t *tag, uint8_t *syndrome) {
	/* simply set to the max. tag + syndrome of this cntlr. */
	SIHNAH_DMA_ALLOC(tns, (ECC_TAG_SZ_B + ECC_BCH12_SYNDROME_SZ_B), SIHNAH_CACHE_LINE_SIZE);
	const ECCFR_T eccfr = {
		.f.rbo = 0,
		.f.wbo = 0,
		.f.slv_endian = 0,
		.f.dma_endian = 1,
		.f.ecc_cfg = op->algo,
		.f.ie = 0,
		.f.precise = 1,
		.f.lbc_bsz = 3,
	};
	ECSR_T ecsr;
	int syndrome_sz_b;

	if (((long)data) & (SIHNAH_CACHE_LINE_SIZE - 1)) {
		ERROR("data ptr. is not cache-line aligned.\n");
		return -2;
	}

	if (op->algo == ECC_BCH12) {
		syndrome_sz_b = ECC_BCH12_SYNDROME_SZ_B;
	} else {
		syndrome_sz_b = ECC_BCH6_SYNDROME_SZ_B;
	}

	sihnah_memcpy(tns, tag, ECC_TAG_SZ_B);
	if (op->act == ECC_DECODE) {
		sihnah_memcpy(tns + ECC_TAG_SZ_B, syndrome, syndrome_sz_b);
	}
	/* append 0xff for all one check */
	sihnah_memset(
		tns + ECC_TAG_SZ_B + syndrome_sz_b,
		0xff,
		sizeof(tns) - ECC_TAG_SZ_B - syndrome_sz_b);

	__flush_dcache_area((long)data, ECC_DATA_SZ_B);
	__flush_dcache_area((long)tns, sizeof(tns));

	writel(eccfr.v, RRVAL(ECCFR));
	RMOD(ECDSAR, addr, TRUNC_64_TO_32(data)); //data start addr.
	RMOD(ECDTAR, addr, TRUNC_64_TO_32(tns)); //tag + syndrome start addr.
	RMOD(ECDTR, dmare, op->act); //1 for encode

	do {
		ecsr.v = readl(RRVAL(ECSR));
	} while (ecsr.v & 0x3);

	if (op->act == ECC_ENCODE) {
		/* return syndrome for encode */;
		sihnah_memcpy(syndrome, tns+ECC_TAG_SZ_B, syndrome_sz_b);
		return 0;
	}

	sihnah_memcpy(tag, tns, ECC_TAG_SZ_B);

	/* return correction result for decode */
	if (ecsr.f.all_one) {
		return 0;
	} else if (ecsr.f.eccer) {
		return -1;
	} else {
		return ecsr.f.eccn;
	}
}
