#include <linux/mtd/nand.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/bbm.h>
#include <linux/mtd/partitions.h>
#include <linux/slab.h>
#include <flash/flash_partition.h>
#include <uapi/mtd/mtd-abi.h>
#include <internals.h>
#include <ecc.h>
#include <pnand.h>
#include <linux/dma-mapping.h>

static const struct nand_manufacturer_desc nand_manufacturer_descs[] = {
	{NAND_MFR_AMD, "AMD/Spansion", &amd_nand_manuf_ops},
	{NAND_MFR_ATO, "ATO"},
	{NAND_MFR_EON, "Eon"},
	{NAND_MFR_ESMT, "ESMT", &esmt_nand_manuf_ops},
	{NAND_MFR_FUJITSU, "Fujitsu"},
	{NAND_MFR_HYNIX, "Hynix", &hynix_nand_manuf_ops},
	{NAND_MFR_INTEL, "Intel"},
	{NAND_MFR_MACRONIX, "Macronix", &macronix_nand_manuf_ops},
	{NAND_MFR_MICRON, "Micron", &micron_nand_manuf_ops},
	{NAND_MFR_NATIONAL, "National"},
	{NAND_MFR_RENESAS, "Renesas"},
	{NAND_MFR_SAMSUNG, "Samsung", &samsung_nand_manuf_ops},
	{NAND_MFR_SANDISK, "SanDisk"},
	{NAND_MFR_STMICRO, "ST Micro"},
	{NAND_MFR_TOSHIBA, "Toshiba", &toshiba_nand_manuf_ops},
	{NAND_MFR_WINBOND, "Winbond"},
};

/* a shorthand to generate struct nand_ecc_caps with only one ECC stepsize */
#define NAND_ECC_CAPS_SINGLE(__name, __calc, __step, ...)	\
static const int __name##_strengths[] = { __VA_ARGS__ };	\
static const struct nand_ecc_step_info __name##_stepinfo = {	\
	.stepsize = __step,					\
	.strengths = __name##_strengths,			\
	.nstrengths = ARRAY_SIZE(__name##_strengths),		\
};								\
static const struct nand_ecc_caps __name = {			\
	.stepinfos = &__name##_stepinfo,			\
	.nstepinfos = 1,					\
	.calc_ecc_bytes = __calc,				\
}

/* status */
#define STATUS_OIP_MASK		0x01
#define STATUS_CRBSY_MASK	0x80
#define STATUS_READY		(0 << 0)
#define STATUS_BUSY		(1 << 0)
#define PNAND_OPTION_ONDIE_ECC (1<<0)
extern int add_mtd_partitions(struct mtd_info *, const struct mtd_partition *, int);

/* Define default oob placement schemes for large and small page devices */
static struct nand_ecclayout_user nand_oob_8 = {
	.eccbytes = 3,
	.eccpos = {0, 1, 2},
	.oobfree = {
		{.offset = 3,
		 .length = 2},
		{.offset = 6,
		 .length = 2},
		{.offset = 8,
		 .length = 0}
	}
};

static struct nand_ecclayout_user nand_oob_16 = {
	.eccbytes = 6,
	.eccpos = {0, 1, 2, 3, 6, 7},
	.oobfree = {
		{.offset = 8,
		 . length = 8},
		{.offset = 16,
		 . length = 0}
	}
};

static struct nand_ecclayout_user nand_oob_64 = {
	.eccbytes = 40,
	.eccpos = {
			24, 25, 26, 27, 28, 29, 30, 31,
			32, 33, 34, 35, 36, 37, 38, 39,
			40, 41, 42, 43, 44, 45, 46, 47,
			48, 49, 50, 51, 52, 53, 54, 55,
			56, 57, 58, 59, 60, 61, 62, 63 },
	.oobfree = {
		{.offset = 2,
		 .length = 14},
		{.offset = 64,
		 . length = 0}
	}
};

static struct nand_ecclayout_user nand_oob_128 = {
	.eccbytes = 80,
	.eccpos = {
		24, 25, 26, 27, 28 , 29 , 30 , 31   ,
		32, 33, 34, 35, 36 , 37 , 38 , 39   ,
		40, 41, 42, 43, 44 , 45 , 46 , 47   ,
		48, 49, 50, 51, 52 , 53 , 54 , 55   ,
		56, 57, 58, 59, 60 , 61 , 62 , 63   ,
		64, 65, 66, 67, 68 , 69 , 70 , 71   ,
		72, 73, 74, 75, 76 , 77 , 78 , 79   ,
		80, 81, 82, 83, 84 , 85 , 86 , 87   ,
		88, 89, 90, 91, 92 , 93 , 94 , 95   ,
		96, 97, 98, 99, 100, 101, 102, 103 },
	.oobfree = {
		{.offset = 2,
		 .length = 14},
        {.offset = 104,
		 .length = 24},
		{.offset = 128,
		 . length = 0}
	}
};

static struct nand_ecclayout_user nand_oob_224 = {
	.eccbytes = 160,
	.eccpos = {
		48 , 49 , 50 , 51 , 52 , 53 , 54 , 55  ,
		56 , 57 , 58 , 59 , 60 , 61 , 62 , 63  ,
		64 , 65 , 66 , 67 , 68 , 69 , 70 , 71  ,
		72 , 73 , 74 , 75 , 76 , 77 , 78 , 79  ,
		80 , 81 , 82 , 83 , 84 , 85 , 86 , 87  ,
		88 , 89 , 90 , 91 , 92 , 93 , 94 , 95  ,
		96 , 97 , 98 , 99 , 100, 101, 102, 103 ,
		104, 105, 106, 107, 108, 109, 110, 111 ,
		112, 113, 114, 115, 116, 117, 118, 119 ,
		120, 121, 122, 123, 124, 125, 126, 127 ,
		128, 129, 130, 131, 132, 133, 134, 135 ,
		136, 137, 138, 139, 140, 141, 142, 143 ,
		144, 145, 146, 147, 148, 149, 150, 151 ,
		152, 153, 154, 155, 156, 157, 158, 159 ,
		160, 161, 162, 163, 164, 165, 166, 167 ,
		168, 169, 170, 171, 172, 173, 174, 175 ,
		176, 177, 178, 179, 180, 181, 182, 183 ,
		184, 185, 186, 187, 188, 189, 190, 191 ,
		192, 193, 194, 195, 196, 197, 198, 199 ,
		200, 201, 202, 203, 204, 205, 206, 207},
	.oobfree = {
		{.offset = 2,
		 .length = 46},
		{.offset = 0xd0,
		 .length = 16},
		{.offset = 224,
		 .length = 0}
	}
};

static struct nand_ecclayout_user nand_oob_256 = {
	.eccbytes = 160,
	.eccpos = {
		48 , 49 , 50 , 51 , 52 , 53 , 54 , 55  ,
		56 , 57 , 58 , 59 , 60 , 61 , 62 , 63  ,
		64 , 65 , 66 , 67 , 68 , 69 , 70 , 71  ,
		72 , 73 , 74 , 75 , 76 , 77 , 78 , 79  ,
		80 , 81 , 82 , 83 , 84 , 85 , 86 , 87  ,
		88 , 89 , 90 , 91 , 92 , 93 , 94 , 95  ,
		96 , 97 , 98 , 99 , 100, 101, 102, 103 ,
		104, 105, 106, 107, 108, 109, 110, 111 ,
		112, 113, 114, 115, 116, 117, 118, 119 ,
		120, 121, 122, 123, 124, 125, 126, 127 ,
		128, 129, 130, 131, 132, 133, 134, 135 ,
		136, 137, 138, 139, 140, 141, 142, 143 ,
		144, 145, 146, 147, 148, 149, 150, 151 ,
		152, 153, 154, 155, 156, 157, 158, 159 ,
		160, 161, 162, 163, 164, 165, 166, 167 ,
		168, 169, 170, 171, 172, 173, 174, 175 ,
		176, 177, 178, 179, 180, 181, 182, 183 ,
		184, 185, 186, 187, 188, 189, 190, 191 ,
		192, 193, 194, 195, 196, 197, 198, 199 ,
		200, 201, 202, 203, 204, 205, 206, 207},
	.oobfree = {
		{.offset = 2,
		 .length = 46},
		{.offset = 0xd0,
		 .length = 48},
		{.offset = 256,
		 .length = 0}
	}
};

struct nand_flash_dev sw_nand_flash_ids[] = {
	[0] = { "Dosilicon FMND2G08S3D-IA 2Gb 1.8V",
		{.id = {0xf8, 0xaa, 0x90, 0x15, 0x46 }}, SZ_2K, SZ_256, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[1] = { "Dosilicon FMND4G08S3B-ID 4Gb 1.8V",
		{.id = {0xf8, 0xac, 0x90, 0x15, 0x46 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[2] = { "Dosilicon FMND2G08U3D-IA 2Gb 3.3V",
		{.id = {0xf8, 0xda, 0x90, 0x95, 0x46 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[3] = { "Dosilicon FMND4G08U3B-IA 4Gb 3.3V",
		{.id = {0xf8, 0xdc, 0x90, 0x95, 0x46 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[4] = { "ESMT F59L2G81KA 2Gb 3.3V",
		{.id = {0xc8, 0x6a, 0x90, 0x04, 0x34 }}, SZ_2K, SZ_256, SZ_128K, 0, 5, 128, NAND_ECC_INFO(8, SZ_512) },
	[5] = { "ESMT F59L4G81KA 4Gb 3.3V",
		{.id = {0xc8, 0xdc, 0x80, 0x19, 0x30 }}, SZ_4K, SZ_512, SZ_256K, 0, 5, 256, NAND_ECC_INFO(8, SZ_512) },
	[6] = { "ESMT F59L8G81XA 8Gb 3.3V",
		{.id = {0x2c, 0xd3, 0x90, 0xa6, 0x64 }}, SZ_4K, SZ_1K, SZ_256K, 0, 5, 224, NAND_ECC_INFO(8, SZ_512) }, //540
	[7] = { "JSC JS27HP2G08SDDA-45 2Gb 1.8V",
		{.id = {0xad, 0xda, 0x90, 0x15, 0x46 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[8] = { "JSC JS27HU2G08SCN-25 2Gb 3.3V",
		{.id = {0xad, 0xda, 0x90, 0x95, 0x46 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[9] = { "Kioxia TH58NYG2S3HBAI4 4Gb 1.8V",
		{.id = {0x98, 0xac, 0x91, 0x15, 0x76 }}, SZ_4K, SZ_512, SZ_256K, 0, 5, 128, NAND_ECC_INFO(8, SZ_512) }, //528
	[10] = { "Kioxia TC58BVG2S0HTA00 4Gb 3.3V",
		{.id = {0x98, 0xdc, 0x90, 0x26, 0xF6 }}, SZ_4K, SZ_512, SZ_256K, 0, 5, 128, NAND_ECC_INFO(8, SZ_512) }, //528
	[11] = { "Kioxia TH58NVG2S3HBAI4  4Gb 3.3V",
		{.id = {0x98, 0xdc, 0x91, 0x15, 0x76 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 128, NAND_ECC_INFO(8, SZ_512) }, //528
	[12] = { "Kioxia TH58BVG3S0HTA00 8Gb 3.3V",
		{.id = {0x98, 0xd3, 0x91, 0x26, 0xF4 }}, SZ_4K, SZ_1K, SZ_256K, 0, 5, 128, NAND_ECC_INFO(8, SZ_512) }, //528
	[13] = { "Micron MT29F8G08ABACAWP 8Gb 3.3V",
		{.id = {0x2c, 0xd3, 0x90, 0xa6, 0x64 }}, SZ_4K, SZ_1K, SZ_256K, 0, 5, 224, NAND_ECC_INFO(8, SZ_512) }, //540
	[14] = { "MXIC MX30LF1G18AC 1Gb 3.3V",
		{.id = {0xc2, 0xf1, 0x80, 0x95, 0x02 }}, SZ_2K, SZ_128, SZ_128K, 0, 5, 128, NAND_ECC_INFO(4, SZ_512) },
	[15] = { "MXIC MX30LF1G28AD 1Gb 3.3V",
		{.id = {0xc2, 0xf1, 0x80, 0x91, 0x03, 0x3 }}, SZ_2K, SZ_128, SZ_128K, 0, 6, 128, NAND_ECC_INFO(8, SZ_512) }, //512+32
	[16] = { "MXIC MX30LF2G28AD 2Gb 3.3V",
		{.id = {0xc2, 0xda, 0x90, 0x91, 0x07, 0x3 }}, SZ_2K, SZ_256, SZ_128K, 0, 6, 128, NAND_ECC_INFO(8, SZ_512) }, //512+32
	[17] = { "MXIC MX60LF8G28AD 8Gb 3.3V",
		{.id = {0xc2, 0xd3, 0xd1, 0xa2, 0x5b, 0x3 }}, SZ_4K, SZ_1K, SZ_256K, 0, 6, 128, NAND_ECC_INFO(8, SZ_512) }, //512+32
	[18] = { "Winbond W29N02GVSIAA 2Gb 3.3V",
		{.id = {0xef, 0xda, 0x90, 0x95, 0x04 }}, SZ_2K, SZ_256, SZ_128K, 0, 5, 64, NAND_ECC_INFO(1, SZ_512) },
	[19] = { "Winbond W29N04GVSIAF 4Gb 3.3V",
		{.id = {0xef, 0xdc, 0x90, 0x95, 0x54 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[20] = { "Winbond W29N08GVSIAA 8Gb 3.3V",
		{.id = {0xef, 0xd3, 0x91, 0x95, 0x58 }}, SZ_2K, SZ_1K, SZ_128K, 0, 5, 64, NAND_ECC_INFO(4, SZ_512) },
	[21] = { "XTX PN27Q02BBGITG 2Gb 1.8V",
		{.id = {0x98, 0xaa, 0x90, 0x15, 0xf6 }}, SZ_2K, SZ_256, SZ_128K, 0, 5, 64, NAND_ECC_INFO(8, SZ_512) }, //528
	[22] = { "XTX PN27G02BBGITG 2Gb 3.3V",
		{.id = {0x98, 0xda, 0x90, 0x15, 0xf6 }}, SZ_2K, SZ_256, SZ_128K, 0, 5, 64, NAND_ECC_INFO(8, SZ_512) }, //528
	[23] = { "XTX PN27G04ABGITG 4Gb 3.3V",
		{.id = {0x98, 0xda, 0x90, 0x15, 0xf6 }}, SZ_4K, SZ_512, SZ_256K, 0, 5, 256, NAND_ECC_INFO(8, SZ_512) }, //544
	[24] = { "XTX XT27G02ETSIGA  2Gb 3.3V",
		{.id = {0x2c, 0xda, 0x90, 0x95, 0x06 }}, SZ_2K, SZ_256, SZ_128K, 0, 5, 64, NAND_ECC_INFO(8, SZ_512) }, //528
	[25] = { "XTX XT27G02ETSIGA  4Gb 3.3V",
		{.id = {0xec, 0xdc, 0x10, 0x95, 0x56 }}, SZ_2K, SZ_512, SZ_128K, 0, 5, 64, NAND_ECC_INFO(8, SZ_512) }, //528
	{NULL}
};

uint32_t idx_option_flag[] = {
	[ 0] = 0,
	[ 1] = 0,
	[ 2] = 0,
	[ 3] = 0,
	[ 4] = 0,
	[ 5] = 0,
	[ 6] = 0,
	[ 7] = 0,
	[ 8] = 0,
	[ 9] = PNAND_OPTION_ONDIE_ECC,
	[10] = PNAND_OPTION_ONDIE_ECC,
	[11] = 0,
	[12] = PNAND_OPTION_ONDIE_ECC,
	[13] = 0,
	[14] = 0,
	[15] = 0,
	[16] = 0,
	[17] = 0,
	[18] = 0,
	[19] = 0,
	[20] = 0,
	[21] = 0,
	[22] = 0,
	[23] = 0,
	[24] = 0,
	[25] = 0,
	[26] = 0
};

#define MIN_TRY_COUNT		3
#define NAND_MAX_ADDR_LEN 4
struct pnand_cmd{
	uint8_t		cmd;
	uint8_t		n_addr;		/* Number of address */
	union {
		uint8_t addr[NAND_MAX_ADDR_LEN];	/* Reg Offset */
		uint32_t addrs;
	};
	//uint8_t		n_dummy_cycle;		/* Number of dummy cycles */
	uint32_t		n_tx;		/* Number of tx bytes */
	const uint8_t	*tx_buf;	/* Tx buf */
	uint32_t		n_rx;		/* Number of rx bytes */
	uint8_t		*rx_buf;	/* Rx buf */
};

struct sw_pnand_chip{
	struct nand_chip chip;
	uint64_t gap;
	pnand_info_t *info;
	struct mtd_info *mtd;
	int cached_page;
	uint8_t		*buf;
	uint8_t		*oobbuf;
	int (*command_fn)(struct sw_pnand_chip *this,
	struct pnand_cmd *cmd);
	uint32_t	cached_page_bitflips;
	uint8_t		cached_page_ecc_off;
	//uint8_t lun;
	uint8_t lun_shift;
	uint8_t read_cache_op;
	uint32_t block_size;
	spinlock_t *_lock;
	wait_queue_head_t *_wq;
	flstate_t _state;
	struct nand_ecclayout_user *_ecc_layout;
	int _numchips;
	uint8_t ce;
	uint32_t read_tmp;
	uint8_t read_left;
	uint8_t addr[6];
	uint8_t addr_byte_cnt;
	sihnah_ecc_t ecc_op;
	void (*ecc_encode)(struct mtd_info *mtd, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare);
	int (*ecc_decode)(struct mtd_info *mtd, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare);
};

typedef struct {
	uint32_t total_chunks;
	uint32_t page_idx_start;
} bfn_info_t;


#undef NAND_MAX_ADDR_LEN
void __iomem *flash_ctl_base;
void static __iomem *flash_mem_base;
struct platform_device *gpdev;
dma_addr_t tns_phy_base;
dma_addr_t data_phy_base;
uint8_t *tns_vir_base;
uint8_t *data_vir_base;

int dma_settings(struct device *dev) {
	int ret;
	dev->dma_mask = kmalloc(sizeof(u64), GFP_ATOMIC);
	if ((ret = dma_set_coherent_mask(dev, DMA_BIT_MASK(32))) != 0){
			goto out;
	}
	if ((ret = dma_set_mask(dev, DMA_BIT_MASK(32))) != 0){
		goto out;
	}
	tns_vir_base = DMA_ALLOC(dev, (ECC_TAG_SZ_B + ECC_BCH12_SYNDROME_SZ_B), &tns_phy_base);
	data_vir_base = DMA_ALLOC(dev, ECC_DATA_SZ_B, &data_phy_base);	
out:
	return -1;	
}

void rtk_mtd_set_dev_defaults(struct mtd_info *mtd) {
	if (mtd->dev.parent) 
	{
		if (!mtd->owner && mtd->dev.parent->driver)
			mtd->owner = mtd->dev.parent->driver->owner;
		if (!mtd->name)
			mtd->name = dev_name(mtd->dev.parent);
	}
	else
	{
		pr_debug("mtd device won't show a device symlink in sysfs");
	}

	INIT_LIST_HEAD(&mtd->partitions);
	mutex_init(&mtd->master.partitions_lock);
}

static void
pnand_ondie_ecc_encode(struct mtd_info *mtd, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare)
{
	return;
}

static int
pnand_ondie_ecc_decode(struct mtd_info *mtd, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare)
{
	struct nand_chip *chip = mtd->priv;
	uint32_t trunk_num = chip->ecc.steps;
	uint8_t ecc_info[trunk_num];
	bool ecc_fail = false;
	int max_bitflips = 0;
	// -- >> read ecc status
	pnand_write_cmd(cs, 0x7A);
	while (!pnand_ready()) {
		;
	}

	pnand_read_buf(ecc_info, trunk_num);
	// -- <<
	int i;
	int stat;
	for ( i = 0; i < trunk_num; i++){
		stat = ecc_info[i] & 0xF;
		if (stat == 0xF) {
			mtd->ecc_stats.failed++;
			ecc_fail = true;
		} else {
			mtd->ecc_stats.corrected += stat;
			max_bitflips = max_t(unsigned int, max_bitflips, stat);
		}
	}
	if (ecc_fail) {
		return -EBADMSG;
	}

	return max_bitflips;
}

static void
pnand_ecc_encode(struct mtd_info *mtd, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int i, eccsize = chip->ecc.size;
	int eccbytes = chip->ecc.bytes;
	int eccsteps = chip->ecc.steps;
	uint8_t *ecc_calc = chip->ecc.calc_buf;
	const uint8_t *p = payload;
	uint32_t *eccpos = _chip->_ecc_layout->eccpos;
	char *tag = chip->oob_poi;
	int ret;
	memset(ecc_calc, 0xAA, 4*eccbytes);
	SIHNAH_DMA_ALLOC(dat_aligned, ECC_DATA_SZ_B, SIHNAH_CACHE_LINE_SIZE);

	for (i = 0; eccsteps; eccsteps--, i += eccbytes, p += eccsize, tag += ECC_TAG_SZ_B) {
		chip->ecc.hwctl(chip, NAND_ECC_WRITE);
		memcpy(dat_aligned, p, ECC_DATA_SZ_B);
		ecc_action((const sihnah_ecc_t *)&_chip->ecc_op, dat_aligned, tag, &ecc_calc[i]);
	}

	for (i = 0; i < chip->ecc.total; i++){
		chip->oob_poi[eccpos[i]] = ecc_calc[i];
	}

	return ;

}

static int
pnand_ecc_decode(struct mtd_info *mtd, uint8_t cs, uint8_t conf, uint8_t *payload, uint8_t *spare)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int i, eccsize = chip->ecc.size;
	int eccbytes = chip->ecc.bytes;
	int eccsteps = chip->ecc.steps;
	uint8_t *p = payload;
	uint8_t *ecc_calc = chip->ecc.calc_buf;
	uint8_t *ecc_code = chip->ecc.code_buf;
	uint32_t *eccpos = _chip->_ecc_layout->eccpos;
	unsigned int max_bitflips = 0;
	int ret;
	bool ecc_fail = false;

	for (i = 0; eccsteps; eccsteps--, i += eccbytes, p += eccsize) {
		chip->ecc.hwctl(chip, NAND_ECC_READ);

		chip->ecc.calculate(chip, p, &ecc_calc[i]);
	}


	for (i = 0; i < chip->ecc.total; i++){
		ecc_code[i] = chip->oob_poi[eccpos[i]];
	}

	eccsteps = chip->ecc.steps;
	p = payload;

	for (i = 0 ; eccsteps; eccsteps--, i += eccbytes, p += eccsize) {
		int stat;

		stat = chip->ecc.correct(chip, p, &ecc_code[i], &ecc_calc[i]);
		if (stat == -EBADMSG /* &&
		    (chip->ecc.options & NAND_ECC_GENERIC_ERASED_CHECK) */) {
			/* check for empty pages with bitflips */
			stat = nand_check_erased_ecc_chunk(p, eccsize,
						&ecc_code[i], eccbytes,
						NULL, 0,
						chip->ecc.strength);
		}

		if (stat < 0) {
			mtd->ecc_stats.failed++;
			ecc_fail = true;
		} else {
			if (stat != 0) {
				printk("ecc error correct (%d)\n", stat );
			}
			mtd->ecc_stats.corrected += stat;
			max_bitflips = max_t(unsigned int, max_bitflips, stat);
		}
	}
	if (ecc_fail) {
		return -EBADMSG;
	}
	return max_bitflips;

}


/**
 * nand_check_ecc_caps - check the sanity of preset ECC settings
 * @chip: nand chip info structure
 * @caps: ECC caps info structure
 * @oobavail: OOB size that the ECC engine can use
 *
 * When ECC step size and strength are already set, check if they are supported
 * by the controller and the calculated ECC bytes fit within the chip's OOB.
 * On success, the calculated ECC bytes is set.
 */
int nand_check_ecc_caps(struct nand_chip *chip,
			const struct nand_ecc_caps *caps, int oobavail)
{
	//struct mtd_info *mtd = nand_to_mtd(chip);
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;
	struct mtd_info *mtd = _chip->mtd;
	const struct nand_ecc_step_info *stepinfo;
	int preset_step = chip->ecc.size;
	int preset_strength = chip->ecc.strength;
	int nsteps, ecc_bytes;
	int i, j;

	if (WARN_ON(oobavail < 0))
		return -EINVAL;

	if (!preset_step || !preset_strength)
		return -ENODATA;

	nsteps = mtd->writesize / preset_step;

	for (i = 0; i < caps->nstepinfos; i++) {
		stepinfo = &caps->stepinfos[i];

		if (stepinfo->stepsize != preset_step)
			continue;

		for (j = 0; j < stepinfo->nstrengths; j++) {
			if (stepinfo->strengths[j] != preset_strength)
				continue;

			ecc_bytes = caps->calc_ecc_bytes(preset_step,
							 preset_strength);
			if (WARN_ON_ONCE(ecc_bytes < 0))
				return ecc_bytes;

			if (ecc_bytes * nsteps > oobavail) {
				pr_err("ECC (step, strength) = (%d, %d) does not fit in OOB",
				       preset_step, preset_strength);
				return -ENOSPC;
			}

			chip->ecc.bytes = ecc_bytes;

			return 0;
		}
	}

	pr_err("ECC (step, strength) = (%d, %d) not supported on this controller",
	       preset_step, preset_strength);

	return -ENOTSUPP;
}

/**
 * nand_match_ecc_req - meet the chip's requirement with least ECC bytes
 * @chip: nand chip info structure
 * @caps: ECC engine caps info structure
 * @oobavail: OOB size that the ECC engine can use
 *
 * If a chip's ECC requirement is provided, try to meet it with the least
 * number of ECC bytes (i.e. with the largest number of OOB-free bytes).
 * On success, the chosen ECC settings are set.
 */
int nand_match_ecc_req(struct nand_chip *chip,
		       const struct nand_ecc_caps *caps, int oobavail)
{
	//struct mtd_info *mtd = nand_to_mtd(chip);
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;
	struct mtd_info *mtd = _chip->mtd;
	const struct nand_ecc_step_info *stepinfo;
	int req_step = chip->base.ecc.requirements.step_size;
	int req_strength = chip->base.ecc.requirements.strength;
	int req_corr, step_size, strength, nsteps, ecc_bytes, ecc_bytes_total;
	int best_step, best_strength, best_ecc_bytes;
	int best_ecc_bytes_total = INT_MAX;
	int i, j;

	if (WARN_ON(oobavail < 0))
		return -EINVAL;

	/* No information provided by the NAND chip */
	if (!req_step || !req_strength)
		return -ENOTSUPP;

	/* number of correctable bits the chip requires in a page */
	req_corr = mtd->writesize / req_step * req_strength;

	for (i = 0; i < caps->nstepinfos; i++) {
		stepinfo = &caps->stepinfos[i];
		step_size = stepinfo->stepsize;

		for (j = 0; j < stepinfo->nstrengths; j++) {
			strength = stepinfo->strengths[j];

			/*
			 * If both step size and strength are smaller than the
			 * chip's requirement, it is not easy to compare the
			 * resulted reliability.
			 */
			if (step_size < req_step && strength < req_strength)
				continue;

			if (mtd->writesize % step_size)
				continue;

			nsteps = mtd->writesize / step_size;

			ecc_bytes = caps->calc_ecc_bytes(step_size, strength);
			if (WARN_ON_ONCE(ecc_bytes < 0))
				continue;
			ecc_bytes_total = ecc_bytes * nsteps;

			if (ecc_bytes_total > oobavail ||
			    strength * nsteps < req_corr)
				continue;

			/*
			 * We assume the best is to meet the chip's requrement
			 * with the least number of ECC bytes.
			 */
			if (ecc_bytes_total < best_ecc_bytes_total) {
				best_ecc_bytes_total = ecc_bytes_total;
				best_step = step_size;
				best_strength = strength;
				best_ecc_bytes = ecc_bytes;
			}
		}
	}

	if (best_ecc_bytes_total == INT_MAX)
		return -ENOTSUPP;

	chip->ecc.size = best_step;
	chip->ecc.strength = best_strength;
	chip->ecc.bytes = best_ecc_bytes;

	return 0;
}
static int
sihnah_ecc_calc_ecc_bytes(int step_size, int strength)
{
	if (strength) {
		if (strength < 7) {
			return 10;
		} else if (strength < 13) {
			return 20;
		}
	}

	printk("EE: unsupported ECC strength: %d\n", strength);
	return -1;
}

NAND_ECC_CAPS_SINGLE(sihnah_ecc_caps, sihnah_ecc_calc_ecc_bytes, ECC_DATA_SZ_B, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12);

static inline struct sw_pnand_chip *to_sihnah_pnand(struct nand_chip *chip) {
	return container_of(chip, struct sw_pnand_chip, chip);
}


static void
sihnah_pnand_rwbuf_flush(struct sw_pnand_chip *sihnah)
{
	sihnah->read_tmp = 0;
	sihnah->read_left = 0;

	return;
}

static void
sihnah_pnand_addr_flush(struct sw_pnand_chip *sihnah)
{
	sihnah->addr_byte_cnt = pnand_write_addr(sihnah->addr, sihnah->addr_byte_cnt);
	return;
}


static void
sihnah_pnand_cmd_ctrl(struct mtd_info *mtd, int dat, unsigned int ctrl)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *sihnah = to_sihnah_pnand(chip);

	sihnah_pnand_rwbuf_flush(sihnah);

	if ((ctrl & NAND_CTRL_ALE) == NAND_CTRL_ALE) {
		/* addr. is inputted byte-by-byte */
		sihnah->addr[sihnah->addr_byte_cnt++] = (dat & 0xff);
	} else {
		sihnah_pnand_addr_flush(sihnah);

		if ((ctrl & NAND_CTRL_CLE) == NAND_CTRL_CLE) {
			pnand_write_cmd(sihnah->ce, dat & 0xff);
		} else if (dat == NAND_CMD_NONE) {
			/* printf("DD: %s:%d: ignore NAND_CMD_NONE\n", __FUNCTION__, __LINE__); */
		} else {
			printk("EE: %s:%d: unknown ctrl: %08x\n", __FUNCTION__, __LINE__, ctrl);
		}
	}

	return;
}


static uint8_t sihnah_pnand_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *sihnah = to_sihnah_pnand(chip);

	return pnand_read_byte(&sihnah->read_tmp, &sihnah->read_left);
}



static void
sihnah_pnand_write_buf(struct mtd_info *mtd, const uint8_t *buf, int len)
{
	pnand_write_buf(buf, len);

	return;
}

static int
sihnah_pnand_dev_ready(struct mtd_info *mtd)
{
	return pnand_ready();
}

static void
sihnah_ecc_hwctl(struct nand_chip *chip, int mode)
{
  struct sw_pnand_chip *sihnah = to_sihnah_pnand(chip);

  if (mode == NAND_ECC_WRITE) {
    sihnah->ecc_op.act = ECC_ENCODE;
  } else {
    sihnah->ecc_op.act = ECC_DECODE;
  }

  return;
}

static int
sihnah_ecc_calculate(struct nand_chip *chip, const uint8_t *dat, uint8_t *synd)
{
	struct sw_pnand_chip *sihnah = to_sihnah_pnand(chip);
	uint8_t tag[ECC_TAG_SZ_B];

	SIHNAH_DMA_ALLOC(dat_aligned, ECC_DATA_SZ_B, SIHNAH_CACHE_LINE_SIZE);

	if (sihnah->ecc_op.act != ECC_ENCODE) {
		return 0;
	}

	memset(tag, 0xff, sizeof(tag));
	memcpy(dat_aligned, dat, ECC_DATA_SZ_B);

	ecc_action((const sihnah_ecc_t *)&sihnah->ecc_op, dat_aligned, tag, synd);

	return 0;
}

static int sihnah_ecc_correct(
	struct nand_chip *chip, uint8_t *dat, uint8_t *read_ecc, uint8_t *calc_ecc)
{
	struct sw_pnand_chip *sihnah = to_sihnah_pnand(chip);
	int tag_idx;
	int res;
	int i = 0;

	SIHNAH_DMA_ALLOC(dat_aligned, ECC_DATA_SZ_B, SIHNAH_CACHE_LINE_SIZE);
	memcpy(dat_aligned, dat, ECC_DATA_SZ_B);

	tag_idx = ((read_ecc - chip->ecc.code_buf) / chip->ecc.bytes) * ECC_TAG_SZ_B;

	res = ecc_action((const sihnah_ecc_t *)&sihnah->ecc_op, dat_aligned, &chip->oob_poi[tag_idx], read_ecc);


	if (res == -1) {
		res = -EBADMSG;
	}
	memcpy(dat, dat_aligned, ECC_DATA_SZ_B);

	return res;
}

static pnand_info_t pnand_info = {0};
static bfn_info_t bfn_info = {0};

static int is_pnand_info_uninit(pnand_info_t *info) {
	if ((info->row_addr_c < 2) || (info->row_addr_c > 4)){
		printk("WW: run `nand atpset ...' first\n");
		return 1;
	} else {
		return 0;
	}
}



#ifndef CONFIG_NAND_HIDE_BAD
#define CONFIG_NAND_HIDE_BAD 1
#endif
//#undef CONFIG_NAND_HIDE_BAD

#ifdef CONFIG_NAND_HIDE_BAD
#define BAD_BLOCK_LIST_ELEMENT	128
#define BBL0_NAME 0x42626c30 //'Bbl0'
#define BBL1_NAME 0x316C6242 //'1lbB
enum {
	FACTORY_1ST_SCAN = 1,
	FACTORY_FROM_BBT0= 2,
	FACTORY_FROM_BBT1= 3,
};
#endif

static int pnand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip);

#ifdef CONFIG_NAND_HIDE_BAD
#define BBL_INDICATORS          4
#define VENUS_CONFIG_BBL_OFFSET         (0x180000)  /* 1.5MB */
#define VENUS_CONFIG_BBL_OFFSET_REDUND  (0x1C0000)
static uint8_t bbl_source=0;
static uint8_t acc_phy_offset=0;
static uint8_t bad_block_cnt=0;
static uint32_t bad_block_list[BAD_BLOCK_LIST_ELEMENT];
uint64_t nand_chipsize=0;

static int pnand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip);

static int xlat_offset_to_skip_factory_bad(struct nand_chip *chip, uint32_t *page_addr)
{
	uint32_t i;
	uint32_t offset;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;
	/* Just return original offset:
	   1. if no bad block found
	   2. Physical flash offset access is assigned */
	if((bad_block_cnt==0) || (acc_phy_offset==1)){
		return 0;
	}

	offset = *page_addr<<chip->page_shift;
	for(i=0; i<bad_block_cnt; i++){
		if(bad_block_list[i] <= offset){
			offset += _chip->block_size;
		}else{
		}
	}
	// if(offset >= chip->chipsize){
	// 	return -EINVAL;
	// }
	*page_addr = offset>>chip->page_shift;
	return 0;
}

#endif

/**
 * pnand_get_device - [GENERIC] Get chip for selected access
 * @mtd: MTD device structure
 * @new_state: the state which is requested
 *
 * Get the device and lock it for exclusive access
 */
static int pnand_get_device(struct mtd_info *mtd, int new_state)
{
	struct sw_pnand_chip *chip = mtd->priv;
	spinlock_t *lock = chip->_lock;
	wait_queue_head_t *wq = chip->_wq;
	DECLARE_WAITQUEUE(wait, current);
retry:
	spin_lock(lock);

	/* Hardware controller shared among independent devices */
	// if (!chip->controller->active){
	// 	chip->controller->active = chip;
	// }

	// if (chip->controller->active == chip && chip->_state == FL_READY) {
	if (chip->_state == FL_READY) {
		chip->_state = new_state;
		spin_unlock(lock);
		return 0;
	}
	if (new_state == FL_PM_SUSPENDED) {
		//if (chip->controller->active->state == FL_PM_SUSPENDED) {
		if (chip->chip.suspended) {
			chip->_state = FL_PM_SUSPENDED;
			spin_unlock(lock);
			return 0;
		}
	}
	set_current_state(TASK_UNINTERRUPTIBLE);
	add_wait_queue(wq, &wait);
	spin_unlock(lock);
	schedule();
	remove_wait_queue(wq, &wait);
	goto retry;
}

/**
 * nand_release_device - [GENERIC] release chip
 * @chip: NAND chip object
 *
 * Release chip lock and wake up anyone waiting on the device.
 */
static void pnand_release_device(struct mtd_info *mtd)
{
	struct sw_pnand_chip *chip = mtd->priv;

	/* Release the controller and the chip */
	spin_lock(chip->_lock);
	chip->_state = FL_READY;
	wake_up(chip->_wq);
	spin_unlock(chip->_lock);
}

static inline int pnand_issue_cmd(struct sw_pnand_chip *chip,
		struct pnand_cmd *cmd)
{
	struct sw_pnand_chip * _chip = chip;
	return _chip->command_fn(chip, cmd);
}

static void
set_ce_of_page_addr(struct mtd_info *mtd, int *pageAddr)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int chipSizePageAddr = 1<<(chip->chip_shift-chip->page_shift);
	static int testflag = 1000;
	if (!(--testflag)) {
		//printk("CY_Log: [%s:%d] pageAddr is %d(%xh) real %d(%xh), chip->chip_shift is %d, chip->page_shift is %d, 1<<(chip->chip_shift-chip->page_shift) is %d(%x)\n", __FUNCTION__, __LINE__, *pageAddr, *pageAddr, *pageAddr<<chip->page_shift, *pageAddr<<chip->page_shift, chip->chip_shift, chip->page_shift, chipSizePageAddr, chipSizePageAddr );
		// dump_stack();
	}
	if (testflag <= 0) {
		testflag = 1000;
	}
	if (_chip->_numchips == 1) {
		_chip->ce = 0;
		return;
	}
	if (*pageAddr >= chipSizePageAddr) {
		*pageAddr -= chipSizePageAddr;
		_chip->ce = 1;
	} else{
		_chip->ce = 0;
	}
}

/* "crc16_tab & cyg_crc16" are Inherited from u-boot-2015.04 */
/* Table of CRC constants - implements x^16+x^12+x^5+1 */
static const uint16_t crc16_tab[] = {
	0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7,
	0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,
	0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6,
	0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de,
	0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485,
	0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,
	0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4,
	0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc,
	0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823,
	0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b,
	0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12,
	0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a,
	0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41,
	0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49,
	0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70,
	0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78,
	0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f,
	0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067,
	0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e,
	0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256,
	0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d,
	0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
	0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c,
	0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634,
	0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,
	0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3,
	0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,
	0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92,
	0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9,
	0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,
	0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,
	0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0,
};
static uint16_t cyg_crc16(unsigned char *buf, int len)
{
	int i;
	uint16_t cksum;

	cksum = 0;
	for (i = 0;  i < len;  i++) {
		cksum = crc16_tab[((cksum>>8) ^ *buf++) & 0xFF] ^ (cksum << 8);
	}
	return cksum;
}


#ifdef  CONFIG_NAND_HIDE_BAD
static void get_pnand_factory_bad_blocks(struct mtd_info *mtd, uint32_t start, uint32_t size)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;

	/* Check factory bad stored in flash chip:
	 * 1) Read Bbl0 if the factory bad had already been scanned
	 * 2) If Bbl0 is bad. Read Bbl1 for the factory bad
	 */
	loff_t offset;
	loff_t bbl_off;
	uint32_t ret=0;
	size_t retlen;
	uint32_t indicator[BBL_INDICATORS];
	uint32_t tempread[BAD_BLOCK_LIST_ELEMENT];
	memset(indicator, 0xF, sizeof(indicator));
	memset(tempread, 0xF, sizeof(tempread));

#if 0
	if (IS_TAURUS) {
		bbl_off = TAURUS_CONFIG_BBL_OFFSET;
	} else {
		bbl_off = VENUS_CONFIG_BBL_OFFSET;
	}
#endif
	bbl_off = VENUS_CONFIG_BBL_OFFSET;

	acc_phy_offset = 1;
	for(offset=bbl_off ; offset < nand_chipsize ; offset += (mtd->erasesize)){\
		ret = pnand_block_bad(mtd, offset, 1);
		if(ret == 1) continue;
		mtd->_read(mtd, offset, sizeof(indicator), &retlen, (uint8_t *)indicator);
		if(((indicator[0] == BBL0_NAME) || (indicator[0] == BBL1_NAME)) && (indicator[2] == indicator[3])){
			ret = mtd->_read(mtd, offset+(mtd->writesize), sizeof(tempread), &retlen, (uint8_t *)tempread);
			if(indicator[2] == cyg_crc16((uint8_t *)tempread, sizeof(tempread))){
				bad_block_cnt = indicator[1];
				bbl_source = (indicator[0]==BBL0_NAME)?FACTORY_FROM_BBT0:FACTORY_FROM_BBT1;
				memcpy(bad_block_list, tempread, sizeof(bad_block_list));
				acc_phy_offset = 0;
				return;
			}
		}
	}

	/* If the BBLn is not read from flash, scan the BBI in OOB */
	offset = start;
	bbl_source = FACTORY_1ST_SCAN;
	bad_block_cnt = 0;
	memset(bad_block_list, 0xF, sizeof(bad_block_list));
	while(((offset - start) < size) && (bad_block_cnt<BAD_BLOCK_LIST_ELEMENT)) {
		ret = pnand_block_bad(mtd, offset, 1);
		if(ret == 1){
			bad_block_list[bad_block_cnt++] = offset;
		}
		offset += _chip->block_size;
	}
	mtd->size -= bad_block_cnt*_chip->block_size;
	acc_phy_offset = 0;
	return;
}
#endif


#if 0
/**
 * pnand_read_reg - send command 0Fh to read register
 * @chip: NAND device structure
 * @reg; register to read
 * @buf: buffer to store value
 */
static int pnand_read_reg(struct nand_chip *chip,
		uint8_t reg, uint8_t *buf)
{
	struct pnand_cmd cmd;
	int ret;

	memset(&cmd, 0, sizeof(struct pnand_cmd));
	cmd.cmd = NAND_CMD_GET_FEATURES;
	cmd.n_addr = 1;
	cmd.addr[0] = reg;
	cmd.n_rx = 1;
	cmd.rx_buf = buf;

	ret = pnand_issue_cmd(chip, &cmd);
	if (ret < 0)
		pr_err("err: %d read register %d\n", ret, reg);

	return ret;
}
#endif

#if 0
/**
 * pnand_write_reg - send command 1Fh to write register
 * @chip: NAND device structure
 * @reg; register to write
 * @buf: buffer stored value
 */
static int pnand_write_reg(struct nand_chip *chip,
		uint8_t reg, uint8_t *buf)
{
	struct pnand_cmd cmd;
	int ret;

	memset(&cmd, 0, sizeof(struct pnand_cmd));
	cmd.cmd = NAND_CMD_SET_FEATURES;
	cmd.n_addr = 1;
	cmd.addr[0] = reg;
	cmd.n_tx = 1,
		cmd.tx_buf = buf,

		ret = pnand_issue_cmd(chip, &cmd);
	if (ret < 0)
		pr_err("err: %d write register %d\n", ret, reg);

	return ret;
}
#endif

/**
 * pnand_read_status - get status register value
 * @chip: NAND device structure
 * @status: buffer to store value
 * Description:
 *   After read, write, or erase, the Nand device is expected to set the
 *   busy status.
 *   This function is to allow reading the status of the command: read,
 *   write, and erase.
 *   Once the status turns to be ready, the other status bits also are
 *   valid status bits.
 */
static int pnand_read_status(struct sw_pnand_chip *chip, uint8_t *status)
{
	//TODO : status register
#if 0
	return pnand_read_reg(chip, REG_STATUS, status);
#endif
	return 0;
}



/**
 * pnand_transfer_oob - transfer oob to client buffer
 * @chip: NAND device structure
 * @oob: oob destination address
 * @ops: oob ops structure
 * @len: size of oob to transfer
 */
static void pnand_transfer_oob(struct sw_pnand_chip *chip, uint8_t *oob,
		struct mtd_oob_ops *ops, size_t len)
{
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;
	switch (ops->mode) {
		case MTD_OPS_PLACE_OOB:
		case MTD_OPS_RAW:
			memcpy(oob, _chip->oobbuf + ops->ooboffs, len);
			return;
		case MTD_OPS_AUTO_OOB: {
			// TODO ecclayout free
#if 1
			struct nand_oobfree *free = (struct nand_oobfree *)_chip->_ecc_layout->oobfree;
#endif
			uint32_t boffs = 0, roffs = ops->ooboffs;
			size_t bytes = 0;

			// TODO ecclayout free
#if 1
			for (; free->length && len; free++, len -= bytes) {
				/* Read request not from offset 0? */
				if (unlikely(roffs)) {
					if (roffs >= free->length) {
						roffs -= free->length;
						continue;
					}
					boffs = free->offset + roffs;
					bytes = min_t(size_t, len,
							(free->length - roffs));
					roffs = 0;
				} else {
					bytes = min_t(size_t, len, free->length);
					boffs = free->offset;
				}
				memcpy(oob, _chip->oobbuf + boffs, bytes);
				oob += bytes;
			}
#endif
			return;
		}
		default:
			BUG();
	}
}

#if 0
/**
 * pnand_set_ds - set value to die select register
 * @chip: NAND device structure
 * @cfg: buffer stored value
 * Description:
 *   Configuration register includes OTP config, Lock Tight enable/disable
 *   and Internal ECC enable/disable.
 */
static int pnand_set_ds(struct nand_chip *chip, uint8_t *ds)
{
	struct pnand_cmd cmd;

	if ((chip->mfr_id == NAND_MFR_MICRON)) {
		return 0;
		//return pnand_write_reg(chip, REG_DIE_SELECT, ds);
	} else {
		memset(&cmd, 0, sizeof(struct pnand_cmd));
		cmd.cmd = NAND_CMD_DIE_SELECT;
		cmd.n_addr = 0;
		cmd.addr[0] = *ds;
		cmd.n_tx = 0;
		return 0;
		//return pnand_issue_cmd(chip, &cmd);
	}
}
#endif

#if 0
/**
 * pnand_lun_select - send die select command if needed
 * @chip: NAND device structure
 * @lun: lun need to access
 */
static int pnand_lun_select(struct nand_chip *chip, uint8_t lun)
{
	uint8_t ds = 0;
	int ret = 0;

#if 0
	if (chip->lun != lun) {
		if (chip->mfr_id == NAND_MFR_MICRON)
			ds = (lun == 1) ? DIE_SELECT_DS1 : DIE_SELECT_DS0;
		else
			ds = lun;
		ret = pnand_set_ds(chip, &ds);
		chip->lun = lun;
	}
#endif

	return ret;
}
#endif


/**
 * pnand_wait - wait until the command is done
 * @chip: NAND device structure
 * @s: buffer to store status register(can be NULL)
 */
static int pnand_wait(struct sw_pnand_chip *chip, uint8_t *s)
{
	unsigned long timeo = jiffies;
	uint8_t status, state = chip->_state;
	int ret = -ETIMEDOUT;
	int count = 0;

	if (state == FL_ERASING || state == FL_WRITING)
		timeo += msecs_to_jiffies(400);
	else
		timeo += msecs_to_jiffies(20);

	while (time_before(jiffies, timeo) || count < MIN_TRY_COUNT) {
		pnand_read_status(chip, &status);
		if ((status & STATUS_OIP_MASK) == STATUS_READY) {
			ret = 0;
			goto out;
		}
		count++;
	}
out:
	if (s)
		*s = status;

	return ret;
}



/**
 * pnand_erase_block - send command D8h to erase a block
 * @chip: NAND device structure
 * @page_addr: the page to erase.
 * Description:
 *   Need to wait for tERS.
 */
static int pnand_erase_block(struct sw_pnand_chip *chip,
		uint32_t page_addr)
{
	struct pnand_cmd cmd;

	memset(&cmd, 0, sizeof(struct pnand_cmd));
	cmd.cmd = NAND_CMD_ERASE1;
	cmd.addrs = page_addr;

	return pnand_issue_cmd(chip, &cmd);
}





/**
 * pnand_read_page_to_cache - send command 13h to read data from Nand
 * to cache
 * @chip: NAND device structure
 * @page_addr: page to read
 */
static int
pnand_read_page_to_cache(struct sw_pnand_chip *chip,
		uint32_t page_addr)
{
	struct pnand_cmd cmd;

	memset(&cmd, 0, sizeof(struct pnand_cmd));
	cmd.cmd = NAND_CMD_READ0;
	cmd.addrs = page_addr;

	return pnand_issue_cmd(chip, &cmd);
}

/**
 * pnand_read_from_cache - read data out from cache register
 * @chip: NAND device structure
 * @page_addr: page to read
 * @column: the location to read from the cache
 * @len: number of bytes to read
 * @rbuf: buffer held @len bytes
 * Description:
 *   Command can be 03h, 0Bh, 3Bh, 6Bh, BBh, EBh
 *   The read can specify 1 to (page size + spare size) bytes of data read at
 *   the corresponding locations.
 *   No tRd delay.
 */
static int
pnand_read_from_cache(struct sw_pnand_chip *chip,
		uint32_t page_addr, uint32_t column, size_t len, uint8_t *rbuf)
{
    struct sw_pnand_chip *_chip = chip;
    struct mtd_info *mtd = _chip->mtd;
	uint8_t addr[6] = {0};
	uint32_t pageAddr = page_addr;

	//uint32_t page_idx = cmd->addrs;
    set_ce_of_page_addr(mtd, &pageAddr);
    pnand_write_cmd(_chip->ce, 0);
    pnand_addr_to_array(&addr[2], pageAddr);
    pnand_write_addr(addr, _chip->info->row_addr_c + 2);

	pnand_write_cmd(_chip->ce, 0x30);

	pnand_wait_ready();

	pnand_read_buf(_chip->buf, mtd->writesize+mtd->oobsize);


	if (_chip->buf+column != rbuf) {
		memcpy(rbuf, _chip->buf+column, len);
	}
	return 0;
}


/**
 * pnand_do_read_page - read page from flash to buffer
 * @mtd: MTD device structure
 * @page_addr: page address/raw address
 * @column: column address
 * @ecc_off: without ecc or not
 * @corrected: how many bit error corrected
 * @buf: data buffer
 * @len: data length to read
 */
static int pnand_do_read_page(struct mtd_info *mtd, uint32_t page_addr,
		bool ecc_off, int *corrected, bool oob_only)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int ret, ecc_error = 0;
	uint8_t status;
	ret = pnand_read_page_to_cache(_chip, page_addr);
	if (ret < 0) {
		pr_err("error %d read page 0x%x to cache\n",
				ret, page_addr);
		return ret;
	}
	ret = pnand_wait(_chip, &status);
	if (ret < 0) {
		pr_err("error %d waiting page 0x%x to cache\n", ret, page_addr);
		return ret;
	}

	if (!ecc_off) {
		uint32_t conf;
		if (chip->base.ecc.requirements.strength > 6) {
			conf = ECC_BCH12;
		} else {
			conf = ECC_BCH6;
		}
		ret = _chip->ecc_decode(mtd, _chip->ce, conf, _chip->buf, _chip->oobbuf);
	  if (ret < 0) {
			pr_err("%s:%d internal ECC error reading page 0x%x\n", __FUNCTION__, __LINE__, page_addr);
	  } else{
			mtd->ecc_stats.corrected += ret;
	  }
	}	

	if (!oob_only) {
		pnand_read_from_cache(_chip, page_addr, 0,
				mtd->writesize + mtd->oobsize, _chip->buf);		
	}
	else {
		pnand_read_from_cache(_chip, page_addr, mtd->writesize,
				mtd->oobsize, _chip->oobbuf);		
	}

	if (!ecc_off) {
		//TODO ecc status
#if 0
		chip->get_ecc_status(chip, status, corrected, &ecc_error);
#endif
		/*
		 * If there's an ECC error, print a message and notify MTD
		 * about it. Then complete the read, to load actual data on
		 * the buffer (instead of the status result).
		 */
		if (ecc_error) {
			pr_err("%s:%d internal ECC error reading page 0x%x\n", __FUNCTION__, __LINE__, page_addr);
			mtd->ecc_stats.failed++;
		} else if (*corrected) {
			mtd->ecc_stats.corrected += *corrected;
		}
	}
	return 0;
}


/**
 * pnand_do_read_oob - read out-of-band
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operations description structure
 * Description:
 *   Disable internal ECC before reading when MTD_OPS_RAW set.
 */
static int pnand_do_read_oob(struct mtd_info *mtd, loff_t from,
		struct mtd_oob_ops *ops)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int page_addr;
	int corrected = 0;
	struct mtd_ecc_stats stats;
	int readlen = ops->ooblen;
	int len;
	int ret = 0;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int lun_num=0, lun_mask;
	uint32_t pa_be4xlat __attribute((unused));
	uint32_t pa_in_out __attribute((unused));

	pr_debug("%s: from = 0x%08Lx, len = %i\n",
			__func__, (unsigned long long)from, readlen);
	stats = mtd->ecc_stats;
	len = ops->mode == MTD_OPS_AUTO_OOB ? mtd->oobavail : mtd->oobsize;
	if (unlikely(ops->ooboffs >= len)) {
		pr_err("%s: attempt to start read outside oob\n",
				__func__);
		return -EINVAL;
	}

	/* Do not allow reads past end of device */
	if (unlikely(from >= mtd->size ||
				ops->ooboffs + readlen > ((mtd->size >> chip->page_shift) -
					(from >> chip->page_shift)) * len)) {
		pr_err("%s: attempt to read beyond end of device\n",
				__func__);
		return -EINVAL;
	}
	/* Shift to get page */
	page_addr = (from >> chip->page_shift);
	pa_be4xlat = page_addr;
	//lun_mask = (1<<chip->lun_shift)-1;

	len -= ops->ooboffs;
	ops->oobretlen = 0;
	while (1) {
#ifdef CONFIG_NAND_HIDE_BAD
		pa_in_out = pa_be4xlat;
		ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
		if(ret == -EINVAL){
			pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
			return ret;
		}
		if(pa_be4xlat != pa_in_out){
			pr_debug("Read xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
		}
		page_addr = pa_in_out;
		pa_be4xlat++;
#endif

#if 0
		if (chip->options & NAND_NEED_DIE_SELECT){
			lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
			pnand_lun_select(chip, lun_num);
		}
#endif
		//TODO : disable ECC
#if 0
		if (ecc_off)
			chip->disable_ecc(chip);
#endif
		/*read data from chip*/
		ret = pnand_do_read_page(mtd, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr), ecc_off,
				&corrected, true);
		if (ret)
			goto out;
		len = min(len, readlen);
		pnand_transfer_oob(_chip, ops->oobbuf + ops->oobretlen,
				ops, len);
		readlen -= len;
		ops->oobretlen += len;
		if (!readlen)
			break;
		page_addr++;

		//TODO: enable ECC
#if 0
		if (ecc_off)
			chip->enable_ecc(chip);
#endif
	}
out:
	if (ret)
		return ret;

	if (mtd->ecc_stats.failed - stats.failed)
		return -EBADMSG;

	return  mtd->ecc_stats.corrected - stats.corrected ? -EUCLEAN : 0;
}


/**
 * pnand_block_bad - Check if block at offset is bad
 * @mtd: MTD device structure
 * @offs: offset relative to mtd start
 * @getchip: 0, if the chip is already selected
 */
static int pnand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip)
{
	struct nand_chip *chip = mtd->priv;
	struct mtd_oob_ops ops = {0};
	uint32_t block_addr;
	uint8_t bad[2] = {0, 0};
	uint8_t ret = 0;

	block_addr = ofs >> chip->phys_erase_shift;
	ops.mode = MTD_OPS_PLACE_OOB;
	ops.ooblen = 2;
	ops.oobbuf = bad;

	if (getchip) {
		pnand_get_device(mtd, FL_READING);
		pnand_do_read_oob(mtd, block_addr << chip->phys_erase_shift, &ops);
	}

	if (getchip) {
		pnand_release_device(mtd);
	}

	if (bad[0] != 0xFF || bad[1] != 0xFF)
		ret =  1;

	return ret;
}


/**
 * pnand_is_bad_bbm - [BBT Interface] Check if block at offset is factory bad
 * @mtd: MTD device structure
 * @offs: offset relative to mtd start
 */
static int pnand_is_bad_bbm(struct mtd_info *mtd, loff_t ofs)
{
	return pnand_block_bad(mtd, ofs, 1);
}


/**
 * pnand_block_checkbad - Check if a block is marked bad
 * @mtd: MTD device structure
 * @ofs: offset from device start
 * @getchip: 0, if the chip is already selected
 * @allowbbt: 1, if its allowed to access the bbt area
 *
 * Check, if the block is bad. Either by reading the bad block table or
 * calling of the scan function.
 */
static int pnand_block_checkbad(struct mtd_info *mtd, loff_t ofs,
		int getchip, int allowbbt)
{
	struct nand_chip *chip = mtd->priv;

	if (!chip->bbt)
		return pnand_block_bad(mtd, ofs, getchip);

	/* Return info from the table */
	if (nand_isbad_bbt(chip, ofs, allowbbt))
		return 1;
	else if (allowbbt)
		return 0;
	else{
    return nand_isreserved_bbt(chip, ofs);
	}


}

static void
pnand_write_enable(struct nand_chip *chip)
{
  writel(0x00800012, RRVAL(NACFR));
}


/**
 * __pnand_erase - erase block(s)
 * @mtd: MTD device structure
 * @einfo: erase instruction
 * @allowbbt: allow to access bbt
 *
 * Erase one or more blocks
 */
static int __pnand_erase(struct mtd_info *mtd, struct erase_info *einfo,
		int allowbbt)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int page_addr, pages_per_block;
	loff_t len;
	int ret = 0;
	int lun_num=0, lun_mask;
	uint32_t pa_be4xlat __attribute((unused));
	uint32_t pa_in_out __attribute((unused));

	/* check address align on block boundary */
	if (einfo->addr & ((_chip->block_size) - 1)) {
		pr_err("%s: Unaligned address\n", __func__);
		return -EINVAL;
	}

	if (einfo->len & ((_chip->block_size) - 1)) {
		pr_err("%s: Length not block aligned\n", __func__);
		return -EINVAL;
	}

	/* Do not allow erase past end of device */ 
	if ((einfo->len + einfo->addr) > nand_chipsize) {
		pr_err("%s: Erase past end of device\n", __func__); return -EINVAL;
	}

	einfo->fail_addr = MTD_FAIL_ADDR_UNKNOWN;

	/* Grab the lock and see if the device is available */
	pnand_get_device(mtd, FL_ERASING);

	pages_per_block = 1 << (chip->phys_erase_shift- chip->page_shift);
	page_addr = einfo->addr >> chip->page_shift;
	pa_be4xlat = page_addr;
	len = einfo->len;
	//  lun_mask = (1<<chip->lun_shift)-1;
	_chip->cached_page = -1;

	// einfo->state = MTD_ERASING;


	while (len) {
		/* Check if we have a bad block, we do not erase bad blocks! */
		if (pnand_block_checkbad(mtd, ((loff_t) page_addr) <<
					chip->page_shift, 0, allowbbt)) {
			pr_warn("%s: attempt to erase a bad block at 0x%012llx\n",
					__func__, ((loff_t) page_addr) << chip->page_shift);
			// einfo->state = MTD_ERASE_FAILED;
			goto erase_exit;
		}
#ifdef CONFIG_NAND_HIDE_BAD
		pa_in_out = pa_be4xlat;
		ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
		if(ret == -EINVAL){
			pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
			return ret;
		}
		if(pa_be4xlat != pa_in_out){
			pr_debug("erase xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
			// printk("erase xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
		}
		page_addr = pa_in_out;
		pa_be4xlat += pages_per_block;
#endif

#if 0
		if (chip->options & NAND_NEED_DIE_SELECT){
			lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
			pnand_lun_select(chip, lun_num);
		}
#endif

		pnand_write_enable(chip);
		ret = pnand_erase_block(_chip, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr));
		if (ret < 0) {
			pr_err("erase block 0x%012llx failed\n",
					((loff_t) page_addr) << chip->page_shift);
			// einfo->state = MTD_ERASE_FAILED;
			einfo->fail_addr = (loff_t)page_addr
				<< chip->page_shift;
			goto erase_exit;
		}

		/* Increment page address and decrement length */
		len -= (1ULL << chip->phys_erase_shift);
		page_addr += pages_per_block;
#ifdef CONFIG_NAND_HIDE_BAD
		page_addr = pa_be4xlat;
#endif
	}

	// einfo->state = MTD_ERASE_DONE;

erase_exit:

	// ret = einfo->state == MTD_ERASE_DONE ? 0 : -EIO;
	ret = 0;
	pnand_release_device(mtd);

	/* Do call back function */
	// if (!ret)
	// 	mtd_erase_callback(einfo);

	/* Return more or less happy */
	return ret;
}


/**
 * pnand_erase - [MTD Interface] erase block(s)
 * @mtd: MTD device structure
 * @einfo: erase instruction
 *
 * Erase one ore more blocks
 */
static int pnand_erase(struct mtd_info *mtd, struct erase_info *einfo)
{
	return __pnand_erase(mtd, einfo, 0);
}

static int pnand_erase_bbt(struct mtd_info *mtd, loff_t ofs)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	struct erase_info einfo = {0};

	// einfo.mtd = mtd;
	einfo.addr = ofs;
	einfo.len = _chip->block_size;

	return __pnand_erase(mtd, &einfo, 1);
}




/**
 * pnand_read_pages_fast - read data from flash to buffer
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operations description structure
 * @max_bitflips: maximum bitflip count
 * Description:
 *   Advanced read function, use READ PAGE CACHE RANDOM to
 *   speed up read.
 */
static int pnand_read_pages_fast(struct mtd_info *mtd, loff_t from,
		struct mtd_oob_ops *ops, unsigned int *max_bitflips)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	int page_addr, page_offset, size;
	int ret;
	unsigned int corrected  = 0, ecc_error = 0;
	int readlen    = ops->len;
	int oobreadlen = ops->ooblen;
	bool ecc_off   = ops->mode == MTD_OPS_RAW, cross_lun = false;
	bool read_ramdon_issued = false;
	int ooblen              = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;
	uint8_t status;
	uint8_t *buf;
	/*use internal buffer when buffer from upper isn't phy continuous*/
	int use_in_buf = !virt_addr_valid(ops->datbuf);
	int lun_num;

	page_addr   = from >> chip->page_shift;
	page_offset = from & chip->pagemask;
	ops->retlen = 0;
	//lun_num = from >> chip->lun_shift;
again:
#if 0
	if (chip->options & NAND_NEED_DIE_SELECT)
		pnand_lun_select(chip, lun_num);
#endif

	pnand_read_page_to_cache(_chip, page_addr);
	ret = pnand_wait(_chip, &status);
	if (ret < 0) {
		pr_err("error %d waiting page 0x%x to cache\n",
				ret, page_addr);
		goto out;
	}
	while ((page_offset + readlen > mtd->writesize) && !cross_lun) {
		if (/*!(chip->options & NAND_NEED_DIE_SELECT) ||*/
				(page_addr + 1) &
				((1 << (_chip->lun_shift - chip->page_shift)) - 1)) {
			read_ramdon_issued = true;
			pnand_read_page_to_cache(_chip, page_addr + 1);
			ret = pnand_wait(_chip, &status);
			if (ret < 0) {
				pr_err("error %d waiting page 0x%x to data resigter\n",
						ret, page_addr + 1);
				goto out;
			}
		} else {
			cross_lun = true;
			break;
		}
		*max_bitflips = max(*max_bitflips, corrected);
		size = min(readlen, mtd->writesize - page_offset);
		buf = use_in_buf ? _chip->buf : ops->datbuf + ops->retlen;
		pnand_read_from_cache(_chip,
				page_addr, page_offset, size, buf);
		if (use_in_buf)
			memcpy(ops->datbuf + ops->retlen, _chip->buf, size);
		page_offset = 0;
		ops->retlen += size;
		readlen -= size;
		if (unlikely(ops->oobbuf)) {
			size = min(oobreadlen, ooblen);
			pnand_read_from_cache(_chip, page_addr,
				mtd->writesize, mtd->oobsize, _chip->oobbuf);
			pnand_transfer_oob(_chip,
				ops->oobbuf + ops->oobretlen, ops, size);
			ops->oobretlen += size;
			oobreadlen -= size;
		}
		if (!ecc_off) {
			//TODO: ecc status
#if 0
			chip->get_ecc_status(chip,
				status, &corrected, &ecc_error);
#endif
			if (ecc_error) {
				pr_err("%s:%d internal ECC error reading page 0x%x\n", __FUNCTION__, __LINE__, page_addr);
				mtd->ecc_stats.failed++;
			} else if (corrected) {
				mtd->ecc_stats.corrected += corrected;
			}
		}
		if (!cross_lun) {
			//ret = pnand_wait_crbusy(chip);
			if (ret < 0) {
				pr_err("error %d waiting page 0x%x to cache\n",
					ret, page_addr + 1);
				goto out;
			}
		}
		page_addr++;
	}
#if 0
	if (read_ramdon_issued) {
		pnand_read_page_cache_last(chip);
		/*
		 * Already check ecc status in loop, no need to check again
		 */
		ret = pnand_wait(chip, &status);
		if (ret < 0) {
			pr_err("error %d waiting page 0x%x to cache\n",
				ret, page_addr);
			goto out;
		}
	}
#endif
	*max_bitflips = max(*max_bitflips, corrected);
	size = min(readlen, mtd->writesize - page_offset);
	buf = use_in_buf ? _chip->buf : ops->datbuf + ops->retlen;
	pnand_read_from_cache(_chip, page_addr, page_offset, size, buf);
	if (use_in_buf)
		memcpy(ops->datbuf + ops->retlen, _chip->buf, size);
	ops->retlen += size;
	readlen -= size;
	if (unlikely(ops->oobbuf)) {
		size = min(oobreadlen, ooblen);
		pnand_read_from_cache(_chip, page_addr,
			mtd->writesize, mtd->oobsize, _chip->oobbuf);
		pnand_transfer_oob(_chip,
			ops->oobbuf + ops->oobretlen, ops, size);
		ops->oobretlen += size;
		oobreadlen -= size;
	}
	if (!ecc_off) {
		//TODO: ecc status
#if 0
		chip->get_ecc_status(chip, status, &corrected, &ecc_error);
#endif
		if (ecc_error) {
			pr_err("%s:%d internal ECC error reading page 0x%x\n", __FUNCTION__, __LINE__, page_addr);
			mtd->ecc_stats.failed++;
		} else if (corrected) {
			mtd->ecc_stats.corrected += corrected;
		}
	}
	if (cross_lun) {
		cross_lun = false;
		page_addr++;
		page_offset = 0;
		lun_num++;
		goto again;
	}
out:
	_chip->cached_page = -1;
	return ret;
}


static inline bool is_read_page_fast_benefit(struct nand_chip *chip,
		loff_t from, size_t len)
{
	/* Disable random cache read which is incompatible with other Flash vendor. */
	return false;

	if (len < (1<<(chip->page_shift)))
		return false;
#if 0 // no dia select
	if (from >> chip->lun_shift == (from + len) >> chip->lun_shift)
		return true;
	if (((1 << chip->lun_shift) - from) >= (1<<(chip->page_shift)) ||
			(from + len - (1 << chip->lun_shift)) >= (1<<(chip->page_shift)))
		return true;
#endif
	return false;
}


/**
 * pnand_read_pages - read data from flash to buffer
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operations description structure
 * @max_bitflips: maximum bitflip count
 * Description:
 *   Normal read function, read one page to buffer before issue
 *   another.
 */
static int pnand_read_pages(struct mtd_info *mtd, loff_t from,
		struct mtd_oob_ops *ops, unsigned int *max_bitflips)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;
	int page_addr, page_offset, size, ret=0;
	unsigned int corrected = 0;
	int readlen = ops->len;
	int oobreadlen = ops->ooblen;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int ooblen = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;
	int lun_num=0, lun_mask;
	uint32_t pa_be4xlat __attribute((unused));
	uint32_t pa_in_out __attribute((unused));

	page_addr = from >> chip->page_shift;
	page_offset = from & chip->pagemask;
	pa_be4xlat = page_addr;
	//lun_mask = (1<<chip->lun_shift)-1;

	ops->retlen = 0;
	*max_bitflips = 0;


	while (1) {
#ifdef CONFIG_NAND_HIDE_BAD
		pa_in_out = pa_be4xlat;
		ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
		if(ret == -EINVAL){
			pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
			return ret;
		}
		if(pa_be4xlat != pa_in_out){
			pr_debug("Read xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
		}
		page_addr = pa_in_out;
		pa_be4xlat++;
#endif

#if 0 // not dia select
		if (chip->options & NAND_NEED_DIE_SELECT){
			lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
			pnand_lun_select(chip, lun_num);
		}
#endif

		// TODO call ecc off
#if 0
		if (ecc_off)
			chip->disable_ecc(chip);
#endif

		size = min(readlen, mtd->writesize - page_offset);
		if (page_addr != _chip->cached_page
				|| ecc_off != _chip->cached_page_ecc_off) {
			ret = pnand_do_read_page(mtd, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr),
					ecc_off, &corrected, false);
			if (ret)
				break;

			if (!ecc_off) {
				uint32_t conf;
				if (chip->base.ecc.requirements.strength > 6) {
					conf = ECC_BCH12;
				} else {
					conf = ECC_BCH6;
				}
				ret = _chip->ecc_decode(mtd, _chip->ce, conf, _chip->buf, _chip->oobbuf);
				if (ret < 0) {
					pr_err("%s:%d internal ECC error reading page 0x%x\n", __FUNCTION__, __LINE__, page_addr);
					mtd->ecc_stats.failed++;
				} else{
					mtd->ecc_stats.corrected += ret;
				}
			}
			//TODO cached_page_bitflips
#if 0
			chip->cached_page_bitflips = corrected;
			chip->cached_page = page_addr;
#endif
			_chip->cached_page_ecc_off = ecc_off;
		}

		memcpy(ops->datbuf + ops->retlen,
				_chip->buf + page_offset, size);
		//TODO cached_page_bitflips
#if 0
		*max_bitflips = max(*max_bitflips, chip->cached_page_bitflips);
#endif

		ops->retlen += size;
		readlen -= size;
		page_offset = 0;

		if (unlikely(ops->oobbuf)) {
			size = min(oobreadlen, ooblen);
			pnand_transfer_oob(_chip,
					ops->oobbuf + ops->oobretlen, ops, size);
			ops->oobretlen += size;
			oobreadlen -= size;
		}
		if (!readlen)
			break;

		page_addr++;

		//TODO enable ecc
#if 0
		if (ecc_off)
			chip->enable_ecc(chip);
#endif
	}

	return ret;
}


/**
 * pnand_do_read_ops - read data from flash to buffer
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob ops structure
 * Description:
 *   Disable internal ECC before reading when MTD_OPS_RAW set.
 */
static int
pnand_do_read_ops(struct mtd_info *mtd, loff_t from,
		struct mtd_oob_ops *ops)
{
	struct nand_chip *chip = mtd->priv;
	int ret;
	struct mtd_ecc_stats stats;
	unsigned int max_bitflips = 0;

	int oobreadlen = ops->ooblen;
	bool ecc_off   = ops->mode == MTD_OPS_RAW;
	int ooblen     = ops->mode == MTD_OPS_AUTO_OOB ?  mtd->oobavail : mtd->oobsize;

	/* Do not allow reads past end of device */
	if (unlikely(from >= mtd->size)) {
		pr_err("%s: attempt to read beyond end of device\n",
				__func__);
		return -EINVAL;
	}
	stats = mtd->ecc_stats;

	/* for oob */
	if (oobreadlen > 0) {
		if (unlikely(ops->ooboffs >= ooblen)) {
			pr_err("%s: attempt to start read outside oob\n",
					__func__);
			return -EINVAL;
		}

		if (unlikely(ops->ooboffs + oobreadlen >
					((mtd->size >> chip->page_shift) - (from >> chip->page_shift))
					* ooblen)) {
			pr_err("%s: attempt to read beyond end of device\n",
					__func__);
			return -EINVAL;
		}
		ooblen -= ops->ooboffs;
		ops->oobretlen = 0;
	}
#if 0 //20200304: Confifure ECC bit in pnand_read_pages
	if (ecc_off)
		chip->disable_ecc(chip);
#endif

	if (is_read_page_fast_benefit(chip, from, ops->len))
		ret = pnand_read_pages_fast(mtd, from, ops, &max_bitflips);
	else
		ret = pnand_read_pages(mtd, from, ops, &max_bitflips);

#if 0 //20200304: Confifure ECC bit in pnand_read_pages
	if (ecc_off)
		chip->enable_ecc(chip);
#endif
	if (ret)
		return ret;

	if (mtd->ecc_stats.failed - stats.failed)
		return -EBADMSG;

	return max_bitflips;
}


/**
 * pnand_read - [MTD Interface] NAND read
 * @mtd: MTD device structure
 * @from: offset to read from
 * @len: number of bytes to read
 * @retlen: pointer to variable to store the number of read bytes
 * @buf: the databuffer to put data
 */
static int
pnand_read(struct mtd_info *mtd, loff_t from, size_t len,
		size_t *retlen, uint8_t *buf)
{
	struct mtd_oob_ops ops;
	int ret;

	pnand_get_device(mtd, FL_READING);

	memset(&ops, 0, sizeof(ops));
	ops.len = len;
	ops.datbuf = buf;
	ops.mode = MTD_OPS_PLACE_OOB;
	ret = pnand_do_read_ops(mtd, from, &ops);

	*retlen = ops.retlen;

	pnand_release_device(mtd);
	return ret;
}



#if 0
/**
 * pnand_set_cfg - set value to configuration register
 * @chip: NAND device structure
 * @cfg: buffer stored value
 * Description:
 *   Configuration register includes OTP config, Lock Tight enable/disable
 *   and Internal ECC enable/disable.
 */
static int pnand_set_cfg(struct pnand_chip *chip, uint8_t *cfg)
{
	return pnand_write_reg(chip, REG_CFG, cfg);
}
#endif



#if 0
/**
 * pnand_disable_ecc - disable internal ECC
 * @chip: NAND device structure
 * Description:
 *   There is one bit( bit 0x10 ) to set or to clear the internal ECC.
 *   Enable chip internal ECC, set the bit to 1
 *   Disable chip internal ECC, clear the bit to 0
 */
static void pnand_disable_ecc(struct nand_chip *chip)
{
	uint8_t cfg = 0;

	pnand_get_cfg(chip, &cfg);
	if ((cfg & CFG_ECC_MASK) == CFG_ECC_ENABLE) {
		cfg &= ~CFG_ECC_ENABLE;
		pnand_set_cfg(chip, &cfg);
	}
}

#endif

#if 0
/**
 * pnand_lock_block - write block lock register to
 * lock/unlock device
 * @chip: nand device structure
 * @lock: value to set to block lock register
 * Description:
 *   After power up, all the Nand blocks are locked.  This function allows
 *   one to unlock the blocks, and so it can be written or erased.
 */
static int pnand_lock_block(struct nand_chip *chip, uint8_t lock)
{
	return 0;
	//return pnand_write_reg(chip, REG_BLOCK_LOCK, &lock);
}
#endif


/**
 * pnand_init_flash_chip - [NAND Interface] Init flash chip
 * @mtd: MTD device structure
 * Description:
 *   This is used to initialize the flash chip status, like protection/ecc/QE.
 */
int pnand_init_flash_chip(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;

	//nand lock


#if 0
	if ((chip->options & NAND_NEED_DIE_SELECT)
			/*&& (chip->mfr_id != NAND_MFR_MICRON)*/ ){
		pnand_lun_select(chip, 1);
		pnand_lock_block(chip, BL_ALL_UNLOCKED);
		pnand_lun_select(chip, 0);
	}
#endif
	return 0;
}



/**
 * pnand_suspend - [MTD Interface] Suspend the NAND flash
 * @mtd: MTD device structure
 */
static int pnand_suspend(struct mtd_info *mtd)
{
	return pnand_get_device(mtd, FL_PM_SUSPENDED);
}

/**
 * pnand_resume - [MTD Interface] Resume the NAND flash
 * @mtd: MTD device structure
 */
static void pnand_resume(struct mtd_info *mtd)
{
	struct sw_pnand_chip *this = mtd->priv;

	if (this->_state == FL_PM_SUSPENDED)
		pnand_release_device(mtd);
	else
		pr_err("%s is not called in suspended state\n:", __func__);
}

char
read_status_common(struct mtd_info *mtd, pnand_info_t *info);

int
pnand_page_prog(struct mtd_info *mtd, uint32_t  page_addr, uint8_t *buf, uint8_t *oobbuf)
{
  uint8_t status;
  uint8_t addr[6] = {0};
  struct sw_pnand_chip *_chip = mtd->priv;
  pnand_info_t *info = _chip->info;
  uint32_t pageAddr = page_addr;

  set_ce_of_page_addr(mtd, &pageAddr);
  pnand_write_cmd(_chip->ce, 0x80);
  pnand_addr_to_array(&addr[2], pageAddr);
  pnand_write_addr(addr, info->row_addr_c + 2);

  pnand_write_buf((uint8_t*)_chip->buf, mtd->writesize+mtd->oobsize);

  pnand_write_cmd(_chip->ce, 0x10);
  pnand_wait_ready();

  status = read_status_common(mtd, info);
  if (status & 0x01) {
    printk("EE: page_program_common() blk_page_idx: %d, fail: %02x\n", page_addr, status);
    return status;
  }
  return 0;
}



/**
 * pnand_do_write_page - write data from buffer to flash
 * @mtd: MTD device structure
 * @page_addr: page address/raw address
 * @column: column address
 * @buf: data buffer
 * @len: data length to write
 * @clr_cache: clear cache register with 0xFF or not
 */
static int pnand_do_write_page(struct mtd_info *mtd, uint32_t page_addr,
		bool oob_only)
{
	struct nand_chip *chip = mtd->priv;
  	struct sw_pnand_chip *_chip = mtd->priv;
	int ret = 0;

	pnand_write_enable(chip);
	if (!oob_only){
    //do nothing
  } else {
    memset(_chip->buf, 0xFF, mtd->writesize);
  }
  ret = pnand_page_prog(mtd, page_addr, _chip->buf, _chip->oobbuf );

	if (ret < 0) {
		pr_err("error %d program page 0x%x to cache\n",
				ret, page_addr);
		return ret;
	}
	return 0;
}


/**
 * pnand_fill_oob - transfer client buffer to oob
 * @chip: NAND device structure
 * @oob: oob data buffer
 * @len: oob data write length
 * @ops: oob ops structure
 */
static void pnand_fill_oob(struct sw_pnand_chip *chip, uint8_t *oob,
		size_t len, struct mtd_oob_ops *ops)
{
	struct sw_pnand_chip *_chip = chip;
	memset(_chip->oobbuf, 0xff, _chip->info->spare_size_b);
	switch (ops->mode) {
		case MTD_OPS_PLACE_OOB:
		case MTD_OPS_RAW:
			memcpy(_chip->oobbuf + ops->ooboffs, oob, len);
			return;

		case MTD_OPS_AUTO_OOB: {
			struct nand_oobfree *free = (struct nand_oobfree *)_chip->_ecc_layout->oobfree;
			uint32_t boffs = 0, woffs = ops->ooboffs;
			size_t bytes = 0;

			for (; free->length && len; free++, len -= bytes) {
				/* Write request not from offset 0? */
				if (unlikely(woffs)) {
					if (woffs >= free->length) {
						woffs -= free->length;
						continue;
					}
					boffs = free->offset + woffs;
					bytes = min_t(size_t, len,
							(free->length - woffs));
					woffs = 0;
				} else {
					bytes = min_t(size_t, len, free->length);
					boffs = free->offset;
				}
				memcpy(_chip->oobbuf + boffs, oob, bytes);
				oob += bytes;
			}
			return;
		}
		default:
			BUG();
	}
}

/**
 * pnand_do_write_ops - write data from buffer to flash
 * @mtd: MTD device structure
 * @to: offset to write to
 * @ops: oob operations description structure
 * Description:
 *   Disable internal ECC before writing when MTD_OPS_RAW set.
 */
static int pnand_do_write_ops(struct mtd_info *mtd, loff_t to,
		struct mtd_oob_ops *ops)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)chip;
	int page_addr, page_offset, size;
	int writelen = ops->len;
	int oobwritelen = ops->ooblen;
	int ret = 0;
	int ooblen = ops->mode == MTD_OPS_AUTO_OOB ?
		mtd->oobavail : mtd->oobsize;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int lun_num=0, lun_mask;
	uint32_t pa_be4xlat __attribute((unused));
	uint32_t pa_in_out __attribute((unused));

	/* Do not allow reads past end of device */
	if (unlikely(to >= mtd->size)) {
		pr_err("%s: attempt to write beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	page_addr = to >> chip->page_shift;
	page_offset = to & chip->pagemask;
	pa_be4xlat = page_addr;
	//lun_mask = (1<<chip->lun_shift)-1;
	ops->retlen = 0;

	/* for oob */
	if (oobwritelen > 0) {
		/* Do not allow write past end of page */
		if ((ops->ooboffs + oobwritelen) > ooblen) {
			pr_err("%s: attempt to write past end of page\n",
					__func__);
			return -EINVAL;
		}

		if (unlikely(ops->ooboffs >= ooblen)) {
			pr_err("%s: attempt to start write outside oob\n",
					__func__);
			return -EINVAL;
		}
		if (unlikely(ops->ooboffs + oobwritelen >
					((mtd->size >> chip->page_shift) - (to >> chip->page_shift))
					* ooblen)) {
			pr_err("%s: attempt to write beyond end of device\n",
					__func__);
			return -EINVAL;
		}
		ooblen -= ops->ooboffs;
		ops->oobretlen = 0;
	}

	_chip->cached_page = -1;
	while (1) {
#ifdef CONFIG_NAND_HIDE_BAD
		pa_in_out = pa_be4xlat;
		ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
		if(ret == -EINVAL){
			pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
			return ret;
		}
		if(pa_be4xlat != pa_in_out){
			pr_debug("Write xlat: 0x%x => 0x%x\n",pa_be4xlat,pa_in_out);
		}
		page_addr = pa_in_out;
		pa_be4xlat++;
#endif

#if 0
		if (chip->options & NAND_NEED_DIE_SELECT){
			lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
			pnand_lun_select(chip, lun_num);
		}
#endif

		//TODO : disable ECC
#if 0
		if (ecc_off)
			chip->disable_ecc(chip);
#endif

		if (unlikely(ops->oobbuf)) {
			size = min(oobwritelen, ooblen);
			pnand_fill_oob(_chip, ops->oobbuf + ops->oobretlen,
					size, ops);
			ops->oobretlen += size;
			oobwritelen -= size;
		} else {
			memset(_chip->oobbuf, 0xff, mtd->oobsize);
		}
		size = min(writelen, mtd->writesize - page_offset);
		memcpy(_chip->buf + page_offset,
				ops->datbuf + ops->retlen, size);
		if (page_offset)
			memset(_chip->buf, 0xff, page_offset);
		if (size < mtd->writesize - page_offset)
			memset(_chip->buf + page_offset + size, 0xff,
					mtd->writesize - page_offset - size);

		if (!ecc_off) {
			uint32_t conf;
			if (chip->base.ecc.requirements.strength > 6) {
				conf = ECC_BCH12;
			} else {
				conf = ECC_BCH6;
			}
			_chip->ecc_encode(mtd, _chip->ce, conf, _chip->buf, _chip->oobbuf);
		}

		ret = pnand_do_write_page(mtd, (lun_num?(((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift):page_addr), false);
		if (ret) {
			pr_err("error %d writing page 0x%x\n",
					ret, page_addr);
			goto out;
		}
		ops->retlen += size;
		writelen -= size;
		page_offset = 0;
		if (!writelen)
			break;
		page_addr++;

		//TODO: enable ecc
#if 0
		if (ecc_off)
			chip->enable_ecc(chip);
#endif
	}
out:
	return ret;
}





/**
 * pnand_write - [MTD Interface] NAND write
 * @mtd: MTD device structure
 * @to: offset to write to
 * @len: number of bytes to write
 * @retlen: pointer to variable to store the number of written bytes
 * @buf: the data to write
 */
static int pnand_write(struct mtd_info *mtd, loff_t to, size_t len,
		size_t *retlen, const uint8_t *buf)
{
	struct mtd_oob_ops ops;
	int ret;

	pnand_get_device(mtd, FL_WRITING);

	memset(&ops, 0, sizeof(ops));
	ops.len = len;
	ops.datbuf = (uint8_t *)buf;
	ops.mode = MTD_OPS_PLACE_OOB;
	ret =  pnand_do_write_ops(mtd, to, &ops);

	*retlen = ops.retlen;

	pnand_release_device(mtd);

	return ret;
}


/**
 * pnand_read_oob - [MTD Interface] read data and/or out-of-band
 * @mtd: MTD device structure
 * @from: offset to read from
 * @ops: oob operation description structure
 */
static int
pnand_read_oob(struct mtd_info *mtd, loff_t from,
		struct mtd_oob_ops *ops)
{
	int ret = -ENOTSUPP;

	ops->retlen = 0;

	/* Do not allow reads past end of device */
	if (ops->datbuf && (from + ops->len) > mtd->size) {
		pr_err("%s: attempt to read beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	pnand_get_device(mtd, FL_READING);

	switch (ops->mode) {
		case MTD_OPS_PLACE_OOB:
		case MTD_OPS_AUTO_OOB:
		case MTD_OPS_RAW:
			break;

		default:
			goto out;
	}

	if (!ops->datbuf)
		ret = pnand_do_read_oob(mtd, from, ops);
	else
		ret = pnand_do_read_ops(mtd, from, ops);

out:
	pnand_release_device(mtd);

	return ret;
}

/**
 * pnand_do_write_oob - write out-of-band
 * @mtd: MTD device structure
 * @to: offset to write to
 * @ops: oob operation description structure
 * Description:
 *   Disable internal ECC before writing when MTD_OPS_RAW set.
 */
static int pnand_do_write_oob(struct mtd_info *mtd, loff_t to,
		struct mtd_oob_ops *ops)
{
	int page_addr, len, ret;
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = mtd->priv;
	int writelen = ops->ooblen;
	bool ecc_off = ops->mode == MTD_OPS_RAW;
	int lun_num, lun_mask;
	uint32_t pa_in_out __attribute((unused));


	pr_debug("%s: to = 0x%08x, len = %i\n",
			__func__, (unsigned int)to, (int)writelen);

	len = ops->mode == MTD_OPS_AUTO_OOB ? mtd->oobavail : mtd->oobsize;

	/* Do not allow write past end of page */
	if ((ops->ooboffs + writelen) > len) {
		pr_err("%s: attempt to write past end of page\n",
				__func__);
		return -EINVAL;
	}

	if (unlikely(ops->ooboffs >= len)) {
		pr_err("%s: attempt to start write outside oob\n",
				__func__);
		return -EINVAL;
	}

	/* Do not allow write past end of device */
	if (unlikely(to >= mtd->size ||
				ops->ooboffs + writelen >
				((mtd->size >> chip->page_shift) -
				 (to >> chip->page_shift)) * len)) {
		pr_err("%s: attempt to write beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	/* Shift to get page */
	page_addr = to >> chip->page_shift;
#ifdef CONFIG_NAND_HIDE_BAD
	pa_in_out = page_addr;
	ret = xlat_offset_to_skip_factory_bad(chip, &pa_in_out);
	if(ret == -EINVAL){
		pr_err("%s: offset xlat exceeds chip size\n",__FUNCTION__);
		return ret;
	}
	if(page_addr != pa_in_out){
		pr_debug("Read xlat: 0x%x => 0x%x\n",page_addr,pa_in_out);
	}
	page_addr = pa_in_out;
#endif

	pnand_fill_oob(_chip, ops->oobbuf, writelen, ops);
#if 0
	lun_mask = (1<<chip->lun_shift)-1;

	lun_num = to >> chip->lun_shift;
	if (chip->options & NAND_NEED_DIE_SELECT){
		lun_num = (page_addr<<chip->page_shift) >> chip->lun_shift;
		pnand_lun_select(chip, lun_num);
		if(lun_num){
			page_addr = ((page_addr<<chip->page_shift)&lun_mask)>>chip->page_shift;
		}
	}
#endif

	//TODO: ecc off
#if 0
	if (ecc_off)
		chip->disable_ecc(chip);
#endif

	if (!ecc_off) {
		uint32_t conf;
		if (chip->base.ecc.requirements.strength > 6) {
			conf = ECC_BCH12;
		} else {
			conf = ECC_BCH6;
		}
		_chip->ecc_encode(mtd, _chip->ce, conf, _chip->buf, _chip->oobbuf);
	}
	
	ret = pnand_do_write_page(mtd, page_addr, true);
	if (ret) {
		pr_err("error %d writing page 0x%x\n",
				ret, page_addr);
		goto out;
	}
	ops->oobretlen = writelen;

out:
	//TODO  enable ecc
#if 0
	if (ecc_off)
		chip->enable_ecc(chip);
#endif

	return ret;
}


/**
 * pnand_write_oob - [MTD Interface] write data and/or out-of-band
 * @mtd: MTD device structure
 * @to: offset to write to
 * @ops: oob operation description structure
 */
static int pnand_write_oob(struct mtd_info *mtd, loff_t to,
		struct mtd_oob_ops *ops)
{
	int ret = -ENOTSUPP;

	ops->retlen = 0;

	/* Do not allow writes past end of device */
	if (ops->datbuf && (to + ops->len) > mtd->size) {
		pr_err("%s: attempt to write beyond end of device\n",
				__func__);
		return -EINVAL;
	}

	pnand_get_device(mtd, FL_WRITING);

	switch (ops->mode) {
		case MTD_OPS_PLACE_OOB:
		case MTD_OPS_AUTO_OOB:
		case MTD_OPS_RAW:
			break;

		default:
			goto out;
	}

	if (!ops->datbuf)
		ret = pnand_do_write_oob(mtd, to, ops);
	else
		ret = pnand_do_write_ops(mtd, to, ops);

out:
	pnand_release_device(mtd);

	return ret;
}

/**
 * pnand_sync - [MTD Interface] sync
 * @mtd: MTD device structure
 *
 * Sync is actually a wait for chip ready function
 */
static void pnand_sync(struct mtd_info *mtd)
{
	pr_debug("pnand_sync: called\n");

	/* Grab the lock and see if the device is available */
	pnand_get_device(mtd, FL_SYNCING);

	/* Release it and go back */
	pnand_release_device(mtd);
}

/**
 * pnand_block_isbad - [MTD Interface] Check if block at offset is bad
 * @mtd: MTD device structure
 * @offs: offset relative to mtd start
 */
static int pnand_block_isbad(struct mtd_info *mtd, loff_t offs)
{
	return pnand_block_checkbad(mtd, offs, 1, 0);
}

/**
 * pnand_block_markbad_lowlevel - mark a block bad
 * @mtd: MTD device structure
 * @ofs: offset from device start
 *
 * This function performs the generic bad block marking steps (i.e., bad
 * block table(s) and/or marker(s)). We only allow the hardware driver to
 * specify how to write bad block markers to OOB (chip->block_markbad).
 *
 * We try operations in the following order:
 *  (1) erase the affected block, to allow OOB marker to be written cleanly
 *  (2) write bad block marker to OOB area of affected block (unless flag
 *      NAND_BBT_NO_OOB_BBM is present)
 *  (3) update the BBT
 * Note that we retain the first error encountered in (2) or (3), finish the
 * procedures, and dump the error in the end.
 */
static int pnand_block_markbad_lowlevel(struct mtd_info *mtd, loff_t ofs)
{
	struct nand_chip *chip = mtd->priv;
	// struct nand_bbt *bbt = chip->bbt;
	struct mtd_oob_ops ops = {0};
	struct erase_info einfo = {0};
	uint32_t block_addr;
	uint8_t buf[2] = {0, 0};
	int res, ret = 0;

	if (!chip->bbt || !(chip->bbt_options & NAND_BBT_NO_OOB_BBM)) {
		/*erase bad block before mark bad block*/
		//einfo.mtd = mtd;
		einfo.addr = ofs;
		einfo.len = 1UL << chip->phys_erase_shift;
		pnand_erase(mtd, &einfo);

		block_addr = ofs >> chip->phys_erase_shift;
		ops.mode = MTD_OPS_PLACE_OOB;
		ops.ooblen = 2;
		ops.oobbuf = buf;
		pnand_get_device(mtd, FL_WRITING);
		ret = pnand_do_write_oob(mtd,
				block_addr << chip->phys_erase_shift, &ops);
		pnand_release_device(mtd);
	}

	/* Mark block bad in BBT */
	if (chip->bbt) {
		res = nand_markbad_bbt(chip, ofs);
		if (!ret)
			ret = res;
	}

	if (!ret)
		mtd->ecc_stats.badblocks++;

	return ret;
}


/**
 * pnand_block_markbad - [MTD Interface] Mark block at the given offset
 * as bad
 * @mtd: MTD device structure
 * @ofs: offset relative to mtd start
 */
static int pnand_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	int ret;

	ret = pnand_block_isbad(mtd, ofs);
	if (ret) {
		/* If it was bad already, return success and do nothing */
		if (ret > 0)
			return 0;
		return ret;
	}

	return pnand_block_markbad_lowlevel(mtd, ofs);
}

/**
 * pnand_block_isreserved - [MTD Interface] Check if a block is
 * marked reserved.
 * @mtd: MTD device structure
 * @ofs: offset from device start
 */
static int pnand_block_isreserved(struct mtd_info *mtd, loff_t ofs)
{
	struct nand_chip *chip = mtd->priv;

	if (!chip->bbt)
		return 0;
	/* Return info from the table */
	return nand_isreserved_bbt(chip, ofs);
	//TODO check the same
	//return mtd_block_isreserved(chip->bbt, ofs);
}

/* Extract the bits of per cell from the 3rd byte of the extended ID */
static int nand_get_bits_per_cell(u8 cellinfo)
{
	int bits;

	bits = cellinfo & NAND_CI_CELLTYPE_MSK;
	bits >>= NAND_CI_CELLTYPE_SHIFT;
	return bits + 1;
}


/*
 * Set the bad block marker/indicator (BBM/BBI) patterns according to some
 * heuristic patterns using various detected parameters (e.g., manufacturer,
 * page size, cell-type information).
 */
static void nand_decode_bbm_options(struct mtd_info *mtd,
									struct nand_chip *chip, u8 id_data[8])
{
	int maf_id = id_data[0];

	/* Set the bad block position */
	if (mtd->writesize > 512 || (chip->options & NAND_BUSWIDTH_16))
		chip->badblockpos = NAND_BBM_POS_LARGE;
	else
		chip->badblockpos = NAND_BBM_POS_SMALL;

	/*
	 * Bad block marker is stored in the last page of each block on Samsung
	 * and Hynix MLC devices; stored in first two pages of each block on
	 * Micron devices with 2KiB pages and on SLC Samsung, Hynix, Toshiba,
	 * AMD/Spansion, and Macronix.  All others scan only the first page.
	 */
	/*
	 * We have some legacy Samsung Nand chip. Bad block marker are still
	 * located at first page.
	 */
//	if (!nand_is_slc(chip) &&
//		(maf_id == NAND_MFR_SAMSUNG || maf_id == NAND_MFR_HYNIX))
//		chip->bbt_options |= NAND_BBT_SCANLASTPAGE;
//	else if ((nand_is_slc(chip) &&
	if ((nand_is_slc(chip) &&
			  (maf_id == NAND_MFR_SAMSUNG ||
			   maf_id == NAND_MFR_HYNIX ||
			   maf_id == NAND_MFR_TOSHIBA ||
			   maf_id == NAND_MFR_AMD ||
			   maf_id == NAND_MFR_MACRONIX)) ||
			 (mtd->writesize == 2048 && maf_id == NAND_MFR_MICRON))
		chip->bbt_options |= 0x00008000 ;
}


/*
 * nand_id_has_period - Check if an ID string has a given wraparound period
 * @id_data: the ID string
 * @arrlen: the length of the @id_data array
 * @period: the period of repitition
 *
 * Check if an ID string is repeated within a given sequence of bytes at
 * specific repetition interval period (e.g., {0x20,0x01,0x7F,0x20} has a
 * period of 3). This is a helper function for nand_id_len(). Returns non-zero
 * if the repetition has a period of @period; otherwise, returns zero.
 */
static int nand_id_has_period(u8 * id_data, int arrlen, int period)
{
	int i, j;
	for (i = 0; i < period; i++)
		for (j = i + period; j < arrlen; j += period)
			if (id_data[i] != id_data[j])
				return 0;
	return 1;
}


/*
 * nand_id_len - Get the length of an ID string returned by CMD_READID
 * @id_data: the ID string
 * @arrlen: the length of the @id_data array

 * Returns the length of the ID string, according to known wraparound/trailing
 * zero patterns. If no pattern exists, returns the length of the array.
 */
static int nand_id_len(u8 * id_data, int arrlen)
{
	int last_nonzero, period;

	/* Find last non-zero byte */
	for (last_nonzero = arrlen - 1; last_nonzero >= 0; last_nonzero--)
		if (id_data[last_nonzero])
			break;

	/* All zeros */
	if (last_nonzero < 0)
		return 0;

	/* Calculate wraparound period */
	for (period = 1; period < arrlen; period++)
		if (nand_id_has_period(id_data, arrlen, period))
			break;

	/* There's a repeated pattern */
	if (period < arrlen)
		return period;

	/* There are trailing zeros */
	if (last_nonzero < arrlen - 1)
		return last_nonzero + 1;

	/* No pattern detected */
	return arrlen;
}


static inline bool is_full_id_nand(struct nand_flash_dev *type)
{
	return type->id_len;
}

static bool find_full_id_nand(struct nand_chip *chip, struct mtd_info *mtd,
							  struct nand_flash_dev *type, int *busw)
{
	struct nand_device *base = &chip->base;
	struct nand_ecc_props requirements;
	struct nand_memory_organization *memorg;
	u8 *id_data = chip->id.data;

	memorg = nanddev_get_memorg(&chip->base);
	if (!strncmp(type->id, id_data, type->id_len)) {
		memorg->pagesize = type->pagesize;
		mtd->writesize = memorg->pagesize;
		memorg->pages_per_eraseblock = type->erasesize /
					       type->pagesize;
		mtd->erasesize = type->erasesize;
		memorg->oobsize = type->oobsize;
		mtd->oobsize = memorg->oobsize;

		memorg->bits_per_cell = nand_get_bits_per_cell(id_data[2]);
		memorg->eraseblocks_per_lun =
			DIV_ROUND_DOWN_ULL((u64)type->chipsize << 20,
					   memorg->pagesize *
					   memorg->pages_per_eraseblock);
		chip->options |= type->options;
		requirements.strength = NAND_ECC_STRENGTH(type);
		requirements.step_size = NAND_ECC_STEP(type);
		mtd->ecc_step_size = requirements.step_size;
		nanddev_set_ecc_requirements(base, &requirements);

		chip->parameters.model = kstrdup(type->name, GFP_KERNEL);
		if (!chip->parameters.model)
			return false;
		*busw = type->options & NAND_BUSWIDTH_16;
		return true;
	}
	return false;
}

/*
 * Many new NAND share similar device ID codes, which represent the size of the
 * chip. The rest of the parameters must be decoded according to generic or
 * manufacturer-specific "extended ID" decoding patterns.
 */
static void _nand_decode_ext_id(struct mtd_info *mtd, struct nand_chip *chip,
							   u8 id_data[8], int *busw)
{
	int extid, id_len;
	struct nand_memory_organization *memorg;
	memorg = nanddev_get_memorg(&chip->base);
	/* The 3rd id byte holds MLC / multichip data */
	memorg->bits_per_cell = nand_get_bits_per_cell(id_data[2]);
	/* The 4th id byte is the important one */
	extid = id_data[3];

	id_len = nand_id_len(id_data, 8);

	/*
	 * Field definitions are in the following datasheets:
	 * Old style (4,5 byte ID): Samsung K9GAG08U0M (p.32)
	 * New Samsung (6 byte ID): Samsung K9GAG08U0F (p.44)
	 * Hynix MLC   (6 byte ID): Hynix H27UBG8T2B (p.22)
	 *
	 * Check for ID length, non-zero 6th byte, cell type, and Hynix/Samsung
	 * ID to decide what to do.
	 */
	if (id_len == 6 && id_data[0] == NAND_MFR_SAMSUNG &&
		!nand_is_slc(chip) && id_data[5] != 0x00) {
		/* Calc pagesize */
		mtd->writesize = 2048 << (extid & 0x03);
		extid >>= 2;
		/* Calc oobsize */
		switch (((extid >> 2) & 0x04) | (extid & 0x03)) {
			case 1:
				mtd->oobsize = 128;
				break;
			case 2:
				mtd->oobsize = 218;
				break;
			case 3:
				mtd->oobsize = 400;
				break;
			case 4:
				mtd->oobsize = 436;
				break;
			case 5:
				mtd->oobsize = 512;
				break;
			case 6:
				mtd->oobsize = 640;
				break;
			case 7:
			default:			/* Other cases are "reserved" (unknown) */
				mtd->oobsize = 1024;
				break;
		}
		extid >>= 2;
		/* Calc blocksize */
		mtd->erasesize = (128 * 1024) <<
			(((extid >> 1) & 0x04) | (extid & 0x03));
		*busw = 0;
	} else if (id_len == 6 && id_data[0] == NAND_MFR_HYNIX &&
			   !nand_is_slc(chip)) {
		unsigned int tmp;

		/* Calc pagesize */
		mtd->writesize = 2048 << (extid & 0x03);
		extid >>= 2;
		/* Calc oobsize */
		switch (((extid >> 2) & 0x04) | (extid & 0x03)) {
			case 0:
				mtd->oobsize = 128;
				break;
			case 1:
				mtd->oobsize = 224;
				break;
			case 2:
				mtd->oobsize = 448;
				break;
			case 3:
				mtd->oobsize = 64;
				break;
			case 4:
				mtd->oobsize = 32;
				break;
			case 5:
				mtd->oobsize = 16;
				break;
			default:
				mtd->oobsize = 640;
				break;
		}
		extid >>= 2;
		/* Calc blocksize */
		tmp = ((extid >> 1) & 0x04) | (extid & 0x03);
		if (tmp < 0x03)
			mtd->erasesize = (128 * 1024) << tmp;
		else if (tmp == 0x03)
			mtd->erasesize = 768 * 1024;
		else
			mtd->erasesize = (64 * 1024) << tmp;
		*busw = 0;
	} else {
		/* Calc pagesize */
		mtd->writesize = 1024 << (extid & 0x03);
		extid >>= 2;
		/* Calc oobsize */
		mtd->oobsize = (8 << (extid & 0x01)) * (mtd->writesize >> 9);
		extid >>= 2;
		/* Calc blocksize. Blocksize is multiples of 64KiB */
		mtd->erasesize = (64 * 1024) << (extid & 0x03);
		extid >>= 2;
		/* Get buswidth information */
		*busw = (extid & 0x01) ? NAND_BUSWIDTH_16 : 0;

		/*
		 * Toshiba 24nm raw SLC (i.e., not BENAND) have 32B OOB per
		 * 512B page. For Toshiba SLC, we decode the 5th/6th byte as
		 * follows:
		 * - ID byte 6, bits[2:0]: 100b -> 43nm, 101b -> 32nm,
		 *                         110b -> 24nm
		 * - ID byte 5, bit[7]:    1 -> BENAND, 0 -> raw SLC
		 */
		if (id_len >= 6 && id_data[0] == NAND_MFR_TOSHIBA &&
			nand_is_slc(chip) && (id_data[5] & 0x7) == 0x6 /* 24nm */  &&
			!(id_data[4] & 0x80) /* !BENAND */ ) {
			mtd->oobsize = 32 * mtd->writesize >> 9;
		}

	}
}

/*
 * Old devices have chip data hardcoded in the device ID table. nand_decode_id
 * decodes a matching ID table entry and assigns the MTD size parameters for
 * the chip.
 */
static void nand_decode_id(struct mtd_info *mtd, struct nand_chip *chip,
						   struct nand_flash_dev *type, u8 id_data[8],
						   int *busw)
{
	int maf_id = id_data[0];
	struct nand_memory_organization *memorg;
	memorg = nanddev_get_memorg(&chip->base);

	mtd->erasesize = type->erasesize;
	mtd->writesize = type->pagesize;
	mtd->oobsize = mtd->writesize / 32;
	*busw = type->options & NAND_BUSWIDTH_16;

	/* All legacy ID NAND are small-page, SLC */
	memorg->bits_per_cell = 1;

	/*
	 * Check for Spansion/AMD ID + repeating 5th, 6th byte since
	 * some Spansion chips have erasesize that conflicts with size
	 * listed in nand_ids table.
	 * Data sheet (5 byte ID): Spansion S30ML-P ORNAND (p.39)
	 */
	if (maf_id == NAND_MFR_AMD && id_data[4] != 0x00 && id_data[5] == 0x00
		&& id_data[6] == 0x00 && id_data[7] == 0x00 && mtd->writesize == 512) {
		mtd->erasesize = 128 * 1024;
		mtd->erasesize <<= ((id_data[3] & 0x03) << 1);
	}
}

static int
sw_pnand_read_id(struct mtd_info *mtd, uint8_t *id, int ce);
static struct
nand_flash_dev *nand_get_flash_type(struct mtd_info *mtd,
												  struct nand_chip *chip,
												  int *maf_id, int *dev_id,
												  struct nand_flash_dev *type)
{
	int busw;
	int maf_idx;
	u8 id_data[8];
	u8 id_data2[8];
	int sw_idx_flag;
	int idx;
	int i;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;	
	if (type) {
		sw_idx_flag = 1;
	}

	memset(id_data, 0, sizeof(id_data));
	sw_pnand_read_id(mtd, id_data, 0);

	*maf_id = id_data[0];
	*dev_id = id_data[1];
	for(i = 0; i < NAND_MAX_ID_LEN; i++)
		chip->id.data[i] = id_data[i];
	
	if (!type)
		type = nand_flash_ids;
	for (idx = 0; type->name != NULL; type++, idx++) {
		if (is_full_id_nand(type)) {
			if (find_full_id_nand(chip, mtd, type, &busw)){
				_chip->_numchips = 1;
				if (sw_idx_flag) {
					if (idx_option_flag[idx] & PNAND_OPTION_ONDIE_ECC) {
						_chip->ecc_encode = pnand_ondie_ecc_encode;
						_chip->ecc_decode = pnand_ondie_ecc_decode ;
					} else {
						_chip->ecc_encode = pnand_ecc_encode;
						_chip->ecc_decode = pnand_ecc_decode ;
					}
				}
				sw_pnand_read_id(mtd, id_data2, 1);
				if (!memcmp(id_data, id_data2, type->id_len)) {
					_chip->_numchips = 2;
				}
				if (!_chip->info->page_size_b) {
					_chip->info->page_size_b = type->pagesize;
					_chip->info->row_addr_c = (type->chipsize<<20)/type->erasesize <= 1024 ?  2 : 3;
				}
				nand_chipsize = (uint64_t) type->chipsize << 20;
				goto ident_get_type_done;
			}
		} else if (*dev_id == type->dev_id) {
			break;
		}
	}

	// chip->onfi_version = 0;
	if (!type->name || !type->pagesize) {
		/* Check if the chip is ONFI compliant */
		//if (ca_nand_flash_detect_onfi(mtd, chip, &busw))
		//	goto ident_get_type_done;

		/* Check if the chip is JEDEC compliant */
		//if (ca_nand_flash_detect_jedec(mtd, chip, &busw))
		//	goto ident_get_type_done;
	}

	if (!type->name){
		return ERR_PTR(-ENODEV);
	}



	if (!mtd->name)
		mtd->name = type->name;

	nand_chipsize = (uint64_t) type->chipsize << 20;
	if (!type->pagesize) {
		/* Decode parameters from extended ID */
		_nand_decode_ext_id(mtd, chip, id_data, &busw);
	//		chip->chipsize, chip->chipsize);
	} else {
		nand_decode_id(mtd, chip, type, id_data, &busw);
	}
	/* Get chip options */
	chip->options |= type->options;
	/*
	 * Check if chip is not a Samsung device. Do not clear the
	 * options for chips which do not have an extended id.
	 */
	if (*maf_id != NAND_MFR_SAMSUNG && !type->pagesize)
		chip->options &= ~NAND_SAMSUNG_LP_OPTIONS;
  ident_get_type_done:
	mtd->oobsize &= ~0x7;
	/* Try to identify manufacturer */
	for (maf_idx = 0; nand_manufacturer_descs[maf_idx].id != 0x0; maf_idx++) {
		if (nand_manufacturer_descs[maf_idx].id == *maf_id)
			break;
	}
	if (chip->options & NAND_BUSWIDTH_AUTO) {
		WARN_ON(chip->options & NAND_BUSWIDTH_16);
		chip->options |= busw;
		//nand_set_defaults(chip, busw);
	} else if (busw != (chip->options & NAND_BUSWIDTH_16)) {
		/*
		 * Check, if buswidth is correct. Hardware drivers should set
		 * chip correct!
		 */
		pr_info("device found, Manufacturer ID: 0x%02x, Chip ID: 0x%02x!\n",
				*maf_id, *dev_id);
		pr_info("%s %s\n", nand_manufacturer_descs[maf_idx].name, mtd->name);
		pr_warn("bus width %d instead %d bit\n",
				(chip->options & NAND_BUSWIDTH_16) ? 16 : 8, busw ? 16 : 8);
		return ERR_PTR(-EINVAL);
	}
	nand_decode_bbm_options(mtd, chip, id_data);
	/* Calculate the address shift from the page size */
	chip->page_shift = ffs(mtd->writesize) - 1;
	/* Convert chipsize to number of pages per chip -1 */
	chip->pagemask = (nand_chipsize >> chip->page_shift) - 1;
	chip->bbt_erase_shift = chip->phys_erase_shift = ffs(mtd->erasesize) - 1;
	if (nand_chipsize & 0xffffffff)
		chip->chip_shift = ffs((unsigned)nand_chipsize) - 1;
	else {
		chip->chip_shift = ffs((unsigned)(nand_chipsize >> 32));
		chip->chip_shift += 32 - 1;
	}
	chip->badblockbits = 8;
	//TODO set erase
	//chip->erase = single_erase;

#if 0
	//TODO check cmdfunc
	/* Do not replace user supplied command function! */
	if (mtd->writesize > 512 && chip->cmdfunc == ca_nand_command)
		chip->cmdfunc = nand_command_lp;
#endif
	pr_info("device found, Manufacturer ID: 0x%02x, Chip ID: 0x%02x!!\n",
			*maf_id, *dev_id);

	// if (chip->onfi_version){
	// 	pr_info("%s %s\n", nand_manuf_ids[maf_idx].name,
	// 			chip->onfi_params.model);
	// }else if (chip->jedec_version){
	// 	//pr_info("%s %s\n", nand_manuf_ids[maf_idx].name,
	// 	//		chip->jedec_params.model);
	// }else{
	// 	pr_info("%s %s\n", nand_manuf_ids[maf_idx].name, type->name);
	// }
	pr_info("%s %s\n", nand_manufacturer_descs[maf_idx].name, type->name);
	pr_info("%d MiB, %s, erase size: %d KiB, page size: %d, OOB size: %d!\n",
		(int)(nand_chipsize >> 20), nand_is_slc(chip) ? "SLC" : "MLC",
		mtd->erasesize >> 10, mtd->writesize, mtd->oobsize);
	return type;


	return NULL;

}

uint32_t *biffen_pnand_probe(pnand_info_t *info, bfn_info_t *binfo);

char
read_status_common(struct mtd_info *mtd, pnand_info_t *info)
{
	uint32_t status;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	pnand_write_cmd(_chip->ce, 0x70);
	while(!pnand_ready()){
		;
	}
	pnand_read_buf((uint8_t*)&status, 4);
	return (status>>4);
}



static void sw_pnand_cmdfunc(struct mtd_info *mtd, unsigned command, int column,
		int page_addr);



/**
 * pnand_scan_ident - [NAND Interface] Scan for the NAND device
 * @mtd: MTD device structure
 * Description:
 *   This is the first phase of the initiazation. It reads the flash ID and
 *   sets up nand_chip fields accordingly.
 */
int pnand_scan_ident(struct mtd_info *mtd)
{
	uint8_t id[NAND_MAX_ID_LEN] = {0};
	int nand_maf_id, nand_dev_id;
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	struct nand_flash_dev *type;
	int id_retry = 2;
	pnand_cntlr_init();
	biffen_pnand_probe(&pnand_info, &bfn_info);
	if(is_pnand_info_uninit(&pnand_info)){
		return -ENODEV;
	}
	// diable write protection
	writel(0x00800012, RRVAL(NACFR));

	_chip->info = &pnand_info;

	/* Read the flash type */
	type = nand_get_flash_type(mtd, chip, &nand_maf_id, &nand_dev_id, sw_nand_flash_ids);
ident_done:
	chip->pagemask = (1<<(chip->page_shift))- 1;
	chip->ecc.engine_type = NAND_ECC_ENGINE_TYPE_ON_HOST;
	_chip->info->spare_size_b = mtd->oobsize;
	_chip->block_size = mtd->erasesize;
	/* vmalloc buffer cannot used on DMA.
	 * Define NAND_USE_BOUNCE_BUFFER will use temporary buffer for DMA.
	 * And then use memcpy to move to dest buffer.
	 */
	chip->options |= NAND_USES_DMA | NAND_NO_SUBPAGE_WRITE;
	//_chip->lun = 0;
	//TODO enable/disable ECC
#if 0
	if (!chip->enable_ecc)
		chip->enable_ecc = pnand_enable_ecc;
	if (!chip->disable_ecc)
		chip->disable_ecc = pnand_disable_ecc;
#endif
	_chip->buf = kzalloc((_chip->info->page_size_b + _chip->info->spare_size_b), GFP_ATOMIC);
	if (!_chip->buf)
		return -ENOMEM;

	_chip->oobbuf = _chip->buf + _chip->info->page_size_b;
	chip->oob_poi = _chip->buf + _chip->info->page_size_b;
	return 0;
}
EXPORT_SYMBOL_GPL(pnand_scan_ident);




/**
 * pnand_scan_tail - [NAND Interface] Scan for the NAND device
 * @mtd: MTD device structure
 * Description:
 *   This is the second phase of the initiazation. It fills out all the
 *   uninitialized fields of nand_chip and mtd fields.
 */
int pnand_scan_tail(struct mtd_info *mtd)
{

	struct nand_chip *chip      = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	struct nand_ecc_ctrl *ecc = &chip->ecc;
	uint32_t i __attribute((unused));
	/* Initialize state */
	_chip->_state = FL_READY;
	/* Invalidate the pagebuffer reference */
	_chip->cached_page = -1;
	_chip->_lock = kzalloc((sizeof(spinlock_t)), GFP_ATOMIC);
	_chip->_wq = kzalloc((sizeof(wait_queue_head_t)), GFP_ATOMIC);
	if (!chip->controller) {
		// chip->controller = &chip->hwcontrol;
		spin_lock_init(_chip->_lock);
		init_waitqueue_head(_chip->_wq);
		// chip->controller->active = NULL;
	}
	/*
	 * If no default placement scheme is given, select an appropriate one.
	 */
	_chip->_ecc_layout = kzalloc((sizeof(struct nand_ecclayout_user)), GFP_ATOMIC);
	if (_chip->_ecc_layout && (ecc->algo != NAND_ECC_ALGO_BCH)) {
		switch (mtd->oobsize) {
		case 8:
			_chip->_ecc_layout = &nand_oob_8;
			break;
		case 16:
			_chip->_ecc_layout = &nand_oob_16;
			break;
		case 64:
			_chip->_ecc_layout = &nand_oob_64;
			break;
		case 128:
			_chip->_ecc_layout = &nand_oob_128;
			break;
		case 224:
			_chip->_ecc_layout = &nand_oob_224;
			break;
		case 256:
			_chip->_ecc_layout = &nand_oob_256;
			break;			
		default:
			pr_warn("No oob scheme defined for oobsize %d\n",
				   mtd->oobsize);
			BUG();
		}
	}
	if (_chip->ecc_encode == pnand_ondie_ecc_encode) {
		goto __ecc_layout_end;
	}
	int tag_size_b;
	int ecc_synd_b;
	int step_num;
	if (chip->ecc.strength) {
		if (chip->ecc.strength < 7) {
			_chip->ecc_op.algo = ECC_BCH6;
			ecc_synd_b = 10;
		} else if (chip->ecc.strength < 13) {
			_chip->ecc_op.algo = ECC_BCH12;
			ecc_synd_b = 20;
		} else {
			printk("EE: unsupported ECC strength: %d\n", chip->ecc.strength);
			return -1;
		}
		step_num = mtd->writesize / mtd->ecc_step_size;
		tag_size_b = 6 * step_num;
		_chip->_ecc_layout->eccbytes = ecc_synd_b * step_num;
		_chip->_ecc_layout->oobfree[0].offset = 2;
		_chip->_ecc_layout->oobfree[0].length = tag_size_b - _chip->_ecc_layout->oobfree[0].offset;
		for (i=0; i<_chip->_ecc_layout->eccbytes; i++) {
			_chip->_ecc_layout->eccpos[i] = i + tag_size_b;
		}
		if ((i + tag_size_b) > mtd->oobsize) {
			printk("EE: OOB size is overflow.\n");
			return -1;
		}
	}
__ecc_layout_end:
	_chip->_ecc_layout->oobavail = 0;
	for (i = 0; _chip->_ecc_layout->oobfree[i].length
			&& i < ARRAY_SIZE(_chip->_ecc_layout->oobfree); i++){
		_chip->_ecc_layout->oobavail += _chip->_ecc_layout->oobfree[i].length;
	}
	mtd->oobavail = _chip->_ecc_layout->oobavail;

#if 0
	if (chip->options & NAND_ECC_TYPE_HRADWARE)
		pnand_disable_ecc(chip);
#endif
	mtd->name         = "sw_pnand_flash";
	mtd->size         = nand_chipsize;
	mtd->writebufsize = mtd->writesize;
	mtd->owner        = THIS_MODULE;
	mtd->type         = MTD_NANDFLASH;
	mtd->flags        = MTD_CAP_NANDFLASH;
	if (!mtd->ecc_strength)
		mtd->ecc_strength = chip->ecc.strength ?
			chip->ecc.strength : 1;


	// mtd->ecclayout         = _chip->_ecc_layout;
	mtd->_erase            = pnand_erase;
	mtd->_point            = NULL;
	mtd->_unpoint          = NULL;
	mtd->_read             = pnand_read;
	mtd->_write            = pnand_write;
	mtd->_read_oob         = pnand_read_oob;
	mtd->_write_oob        = pnand_write_oob;
	mtd->_sync             = pnand_sync;
	mtd->_lock             = NULL;
	mtd->_unlock           = NULL;
	mtd->_suspend          = pnand_suspend;
	mtd->_resume           = pnand_resume;
	mtd->_block_isbad      = pnand_block_isbad;
	mtd->_block_markbad    = pnand_block_markbad;
	mtd->_block_isreserved = pnand_block_isreserved;

	if (!mtd->bitflip_threshold)
		mtd->bitflip_threshold = DIV_ROUND_UP(mtd->ecc_strength * 3, 4);
	if (!(chip->options & 0x00020000)) {
		chip->ecc.calc_buf = kzalloc(mtd->oobsize, GFP_ATOMIC);
		chip->ecc.code_buf = kzalloc(mtd->oobsize, GFP_ATOMIC);
		if (!chip->ecc.calc_buf || !chip->ecc.code_buf)
			return -ENOMEM;


	}

#ifdef CONFIG_NAND_HIDE_BAD
	get_pnand_factory_bad_blocks(mtd, 0, mtd->size);
	/* Update the the size information to the availabe capacity */
	mtd->size -= bad_block_cnt*mtd->erasesize;
	printk("Parallel nand total capacity is %dMB(0x%x)\n",(uint32_t)(nand_chipsize/0x100000),(uint32_t)nand_chipsize);
	printk("Parallel nand available size is %dMB(0x%x)\n",(uint32_t)(mtd->size/0x100000),(uint32_t)mtd->size);
	if(bbl_source == FACTORY_1ST_SCAN)
		printk("parallel nand 1st scan factory bad to memory\n");
	else
		printk("parallel nand %s from flash\n",(bbl_source==FACTORY_FROM_BBT0)?"Bbl0":"Bbl1");

	printk("parallel nand hidden: ");
	for(i=0 ; i<bad_block_cnt ; i++){
		printk("0x%x ", bad_block_list[i]);
	}
	printk("\n");

	printk("parallel nand Mode: ");
#endif
	chip->data_buf = kmalloc(mtd->writesize + mtd->oobsize, GFP_ATOMIC);
	if (!chip->data_buf)
		return -ENOMEM;
	/* Check, if we should skip the bad block table scan */
	if (chip->options & NAND_SKIP_BBTSCAN)
		return 0;
	chip->bbt = NULL;
	chip->badblock_pattern = NULL;
	/* Build bad block table */
	return nand_create_bbt(chip);
	return 0;
}
EXPORT_SYMBOL_GPL(pnand_scan_tail);


/**
 * pnand_scan_ident_release - [NAND Interface] Free resources
 * applied by pnand_scan_ident
 * @mtd: MTD device structure
 */
int pnand_scan_ident_release(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;

	kfree(_chip->buf);
	kfree(chip->bbt);

	return 0;
}
EXPORT_SYMBOL_GPL(pnand_scan_ident_release);



/**
 * pnand_scan_tail_release - [NAND Interface] Free resources
 * applied by pnand_scan_tail
 * @mtd: MTD device structure
 */
int pnand_scan_tail_release(struct mtd_info *mtd)
{
	return 0;
}
EXPORT_SYMBOL_GPL(pnand_scan_tail_release);


/**
 * pnand_release - [NAND Interface] Free resources held by the NAND
 * device
 * @mtd: MTD device structure
 */
int pnand_release(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;

	mtd_device_unregister(mtd);
	kfree(_chip->oobbuf);
	kfree(chip->bbt);

	return 0;
}
EXPORT_SYMBOL_GPL(pnand_release);


#if 0
/*
 * sw_pnand_get_feature_reg - send command 0xf with with address <addr> to get register value
 * Description:
 */
	static int
sw_pnand_get_feature_reg(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
	struct sw_pnand_chip *_chip = mtd->priv;
	uint8_t addr = cmd->addr[0];
	uint8_t *value = cmd->rx_buf;
	// TODO: get feature register
	//*value = snand_get_feature(_chip->info, 0, addr);
	return 0;
}
#endif

#if 0
/*
 * sw_pnand_set_feature_reg - send command 0x1f with address <addr> to set register value
 * Description:
 */
	static int
sw_pnand_set_feature_reg(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
	struct sw_pnand_chip *_chip = mtd->priv;
	uint8_t addr = cmd->addr[0];
	const uint8_t *value = cmd->tx_buf;
	int ret = 0;
	// TODO set feature
	//ret = snand_set_feature(_chip->info, 0, addr, *value);

	return ret;
}
#endif

static int
sw_pnand_read_page_to_cache(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = mtd->priv;
	uint8_t addr[6] = {0};

#if 0
  // do all in pnand_read_from_cache
	uint32_t page_idx = cmd->addrs;
	pnand_write_cmd(0, 0);
	pnand_addr_to_array(&addr[2], page_idx);
	pnand_write_addr(addr, _chip->info->row_addr_c + 2);

	pnand_write_cmd(0, 0x30);

	pnand_wait_ready();

	pnand_read_buf(_chip->buf, mtd->writesize+mtd->oobsize);
#endif
	return 0;

}

static int sw_pnand_read_cache(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
	//struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	//uint32_t offset = (uint32_t)(cmd->addr[0] << 8) | (uint32_t)(cmd->addr[1]);
	uint8_t *buf = cmd->rx_buf;
	uint32_t len = cmd->n_rx;

	memcpy(buf, _chip->buf, len);

	return 0;
}

static int sw_pnand_erase_block(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
  struct sw_pnand_chip *_chip = mtd->priv;
  pnand_info_t *info = _chip->info;
  uint8_t status;
  int pageAddr = cmd->addrs;
  uint8_t addr[4] = {0};

  set_ce_of_page_addr(mtd, &pageAddr);
  pnand_write_cmd(_chip->ce, 0x60);
  //printk("DD: row_addr_c: %d, row_addr: %d\n", info->row_addr_c, cur_blk_page);

  pnand_addr_to_array(addr, pageAddr);
  pnand_write_addr(addr, info->row_addr_c);

  pnand_write_cmd(_chip->ce, 0xd0);

  pnand_wait_ready();

  status = read_status_common(mtd, info);
  if (status & 0x01) {
    printk("EE: block_erase_common() blk_page_idx: %d, fail: %02x\n", cmd->addrs, status);
    return status;
  }
  return 0;
}

static int sw_pnand_program_execute(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = mtd->priv;
	int ret =0;

	return ret;
}

static int sw_pnand_write_cache_x1(struct mtd_info *mtd, struct pnand_cmd *cmd)
{
	//struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = mtd->priv;
	//uint32_t offset = (uint32_t)(cmd->addr[0] << 8) | (uint32_t)(cmd->addr[1]);
	uint8_t *buf = (uint8_t *)cmd->tx_buf;
	uint32_t len = cmd->n_tx;

	memcpy(_chip->buf, buf, len);

	return 0;
}

/*
 * sw_pnand_read_id- Read Nand ID
 * Description:
 *    Read ID: read two ID bytes from the Nand device
 */
static int
sw_pnand_read_id(struct mtd_info *mtd, uint8_t *id, int ce)
{
	//struct sw_pnand_chip *_chip = mtd->priv;
	uint8_t addr = 0;

	pnand_write_cmd(ce, 0x90);
	pnand_write_addr(&addr, 1);
	pnand_read_buf((uint8_t *)id, 8);
	return 0;
}

/*
 * sw_pnand_reset - Reset Nand
 * Description:
 *     Reset: reset Nand device
 */
uint32_t
sw_pnand_reset(struct mtd_info *mtd)
{
	// TODO : check reset
	struct sw_pnand_chip *_chip = (struct sw_pnand_chip*)mtd->priv;
	uint8_t cmd = 0xFF;
	pnand_write_cmd(_chip->ce, cmd);
	return 0;
}

/*
 * sw_pnand_cmd_fn - to process a command to send to the -NAND
 * by cortina NAND controller
 * @chip: NAND device structure
 * @cmd: command structure
 * Description:
 *   Set up the command buffer to send to the controller.
 *   The command buffer has to initialized to 0.
 */
static int sw_pnand_cmd_fn(struct sw_pnand_chip *chip,
		struct pnand_cmd *cmd)
{
	struct sw_pnand_chip * _chip = (struct sw_pnand_chip*)chip;
	struct mtd_info *mtd = _chip->mtd;
	int ret = 0;

	switch (cmd->cmd) {
		case NAND_CMD_RESET:
			ret = sw_pnand_reset(mtd);
			break;
#if 0
		case NAND_CMD_GET_FEATURES:
			ret = sw_pnand_get_feature_reg(mtd, cmd);
			break;
#endif
#if 0
		case NAND_CMD_SET_FEATURES:
			ret = sw_pnand_set_feature_reg(mtd, cmd);
			break;
#endif
#if 0
		case NAND_CMD_DIE_SELECT:
			//ret = ca_pnand_die_select(cmd);
			break;
#endif
		case NAND_CMD_READ0:
			ret = sw_pnand_read_page_to_cache(mtd, cmd);
			break;
#if 0
		case NAND_CMD_READ_PAGE_CACHE_RDM:
			/* Todo */
			break;
#endif
#if 0
		case NAND_CMD_READ_PAGE_CACHE_LAST:
			/* Todo */
			break;
#endif
#if 0
		case NAND_CMD_READ_FROM_CACHE:
		case NAND_CMD_READ_FROM_CACHE_FAST:
		case NAND_CMD_READ_FROM_CACHE_X2:
		case NAND_CMD_READ_FROM_CACHE_DUAL_IO:
		case NAND_CMD_READ_FROM_CACHE_X4:
		case NAND_CMD_READ_FROM_CACHE_QUAD_IO:
#endif
			//TODO check read
		case NAND_CMD_READ1:
			ret = sw_pnand_read_cache(mtd, cmd);
			break;
		case NAND_CMD_ERASE1:
			ret = sw_pnand_erase_block(mtd, cmd);
			break;
		case NAND_CMD_PAGEPROG:
			ret = sw_pnand_program_execute(mtd, cmd);
			break;
		case NAND_CMD_SEQIN:
			ret = sw_pnand_write_cache_x1(mtd, cmd);
			break;
#if 0
		case NAND_CMD_PROG_LOAD_RDM_DATA:
			/* Todo */
			break;
#endif
		case NAND_CMD_READID:
			ret = sw_pnand_read_id(mtd, cmd->rx_buf, 0);
			break;
#if 0
		case NAND_CMD_WR_DISABLE:
			/* Todo */
			break;
#endif
#if 0
		case NAND_CMD_WR_ENABLE:
			//ret = ca_pnand_write_enable(cmd);
			break;
#endif
		case NAND_CMD_NONE:
		default:
			break;
	}

	return ret;
}

static void sw_pnand_cmdfunc(struct mtd_info *mtd, unsigned command, int column,
		int page_addr)
{
	struct sw_pnand_chip *chip = mtd->priv;
	//int ret;
	struct pnand_cmd cmd;

	memset(&cmd, 0, sizeof(struct pnand_cmd));
	cmd.cmd = command;
	cmd.addrs = page_addr;
	sw_pnand_cmd_fn(chip, &cmd);
}


#ifdef CONFIG_OF
static struct of_device_id sw_pnand_dt_ids[] = {
	{.compatible = "rtk,sw_pnand",},
	{},
};

MODULE_DEVICE_TABLE(of, sw_pnand_dt_ids);
#endif


#define NANDINFO_PROC
#ifdef NANDINFO_PROC
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
static struct nand_chip *private_chip = NULL;
static int rtk_read_proc_nandinfo(struct seq_file *m, void *v)
{
	if(private_chip){
		// seq_printf(m, "%-15s: %llu\n", "nand_size"   , private_chip->chipsize);
		// seq_printf(m, "%-15s: %llu\n", "chip_size"   , private_chip->chipsize);
		seq_printf(m, "%-15s: %u\n" , "block_size"  , 1<<private_chip->phys_erase_shift);
		seq_printf(m, "%-15s: %u\n" , "chunk_size"  , 1<<private_chip->page_shift);
		seq_printf(m, "%-15s: %u\n" , "oob_size"    , pnand_info.spare_size_b);
		//TODO ecclayout
		//seq_printf(m, "%-15s: %u\n" , "oob_avail"   , private_chip->ecclayout->oobavail);
		seq_printf(m, "%-15s: %u\n" , "PagePerBlock", 1<<(private_chip->phys_erase_shift-private_chip->page_shift));
	} return 0; }


static int rtk_open_proc_nandinfo(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_read_proc_nandinfo, NULL);
}

static const struct proc_ops rtk_nand_proc_ops = {
	.proc_open           = rtk_open_proc_nandinfo,
	.proc_read           = seq_read,
	.proc_lseek         = seq_lseek,
	.proc_release        = single_release,
};
#endif

int
ecc_setting(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct sw_pnand_chip *_chip = mtd->priv;
	int ret;
	int ecc_step_ds;
	ecc_step_ds = chip->base.ecc.requirements.step_size;
	chip->ecc.size =  ecc_step_ds;
	//chip->ecc.layout = &sihnah_nand_oob[sihnah->ce];
	if ((!chip->base.ecc.requirements.strength) && (!ecc_step_ds) || _chip->ecc_encode == pnand_ondie_ecc_encode) {
		chip->base.ecc.requirements.engine_type = NAND_ECC_ENGINE_TYPE_ON_DIE;
	} else {
		chip->base.ecc.requirements.engine_type = NAND_ECC_ENGINE_TYPE_ON_HOST;
		// chip->base.ecc.requirements.algo = NAND_ECC_ALGO_BCH;

		chip->ecc.hwctl = sihnah_ecc_hwctl;
		chip->ecc.calculate = sihnah_ecc_calculate;
		chip->ecc.correct = sihnah_ecc_correct;

		chip->ecc.strength = chip->base.ecc.requirements.strength;

		ret = nand_check_ecc_caps(chip, &sihnah_ecc_caps, mtd->oobsize);
		if (ret) {
			printk("EE: %s:%d: No valid ECC settings set: %d\n", __FUNCTION__, __LINE__, ret);
			return ret;
		}

		ret = nand_match_ecc_req(chip, &sihnah_ecc_caps, mtd->oobsize);
		if (ret) {
			printk("EE: %s:%d: No match ECC settings: %d\n", __FUNCTION__, __LINE__, ret);
			return ret;
		}
	}
	chip->ecc.steps = mtd->writesize / chip->ecc.size;
	if (chip->ecc.steps * chip->ecc.size != mtd->writesize) {
		pr_warn("Invalid ECC parameters\n");
		BUG();
	}

	chip->ecc.total = chip->ecc.steps * chip->ecc.bytes;
    return ret;
}

void init_nand_chip(struct sw_pnand_chip* _chip)
{
	struct nand_chip *chip = (struct nand_chip *)_chip;
	chip->oob_poi = NULL;
	chip->controller = NULL;
	chip->data_buf = NULL;
	chip->bbt = NULL;
	chip->bbt_td = NULL;
	chip->bbt_md = NULL;
	chip->badblock_pattern = NULL;


	_chip->cached_page = 0;
	_chip->buf = NULL;
	_chip->oobbuf = NULL;
	_chip->command_fn = NULL;
	_chip->cached_page_bitflips = 0;
	_chip->cached_page_ecc_off = 0;
	_chip->lun_shift = 0;
	_chip->read_cache_op = 0;
	_chip->block_size = 0;
	_chip->ce = 0;
}

static const char * const part_probes[] = {
  "ofpart", "cmdlinepart", "RedBoot", NULL };
static int
sw_pnand_probe(struct platform_device *pdev)
{
	struct resource mem_resource;
	const struct of_device_id *match;
	struct device_node *np = pdev->dev.of_node;
	struct nand_chip *chip;
	struct sw_pnand_chip *_chip;
	struct mtd_info *mtd;
	struct mtd_part_parser_data ppdata;
	rtk_partition_entry_t partition_table[RTK_MTD_PARTITION_NUM];
	struct mtd_partition rtk_sdk_parts[RTK_MTD_PARTITION_NUM];
	int i ;
	uint8_t ret;

	gpdev = pdev;
	match = of_match_device(sw_pnand_dt_ids, &pdev->dev);
	if (!match)
		return -EINVAL;
	dev_notice(&pdev->dev, "SW PNAND Flash driver\n");
	/* Nand Flash Controller addr base */
	ret = of_address_to_resource(np, 0, &mem_resource);
	if (ret) {
		dev_warn(&pdev->dev, "invalid address %d\n", ret);
		return ret;
	}
	flash_ctl_base = devm_ioremap(&pdev->dev, mem_resource.start,
			resource_size(&mem_resource));
	_chip = kzalloc((sizeof(struct sw_pnand_chip)), GFP_ATOMIC);
	if (!_chip) {
		ret = -ENOMEM;
		goto err1;
	}
	chip = (struct nand_chip*)_chip ;
	mtd = kzalloc((sizeof(struct mtd_info)), GFP_ATOMIC);
	if (!mtd) {
		ret = -ENOMEM;
		goto err2;
	}
	dma_settings(&pdev->dev);
	init_nand_chip(_chip);

	mtd->name = "sw_pnand";
	mtd->priv = _chip;
	_chip->mtd = mtd;
	_chip->command_fn = sw_pnand_cmd_fn;
	// chip->cmdfunc = sw_pnand_cmdfunc;
	chip->priv = pdev;
	mtd_set_of_node(mtd, pdev->dev.of_node);
	rtk_mtd_set_dev_defaults(mtd);
	platform_set_drvdata(pdev, chip);
	ret = pnand_scan_ident(mtd);
	if (ret)
		goto err3;
	ecc_setting(mtd);
#if 0
	ret = pnand_manufacture_init(chip);
	if (ret)
		goto err3;
#endif
	ret = pnand_init_flash_chip(mtd);
	if (ret)
		goto err3;

	ret = pnand_scan_tail(mtd);
	if (ret)
		goto err4;

	// ppdata.of_node = pdev->dev.of_node;
#ifdef NANDINFO_PROC
	private_chip = chip;
	proc_create("nandinfo", 0, NULL, &rtk_nand_proc_ops);
#endif
	// ret = mtd_device_parse_register(mtd, part_probes, &ppdata, NULL, 0);
	// if (!ret)
	// 	return 0;
	rtk_flash_partition_table_get(nand_chipsize, partition_table);

	for (i = 0; i < RTK_MTD_PARTITION_NUM; i++) {
		rtk_sdk_parts[i].name = partition_table[i].name;
		rtk_sdk_parts[i].size = partition_table[i].size;
		rtk_sdk_parts[i].offset = partition_table[i].offset;
		rtk_sdk_parts[i].mask_flags = 0;
		rtk_sdk_parts[i].add_flags = MTD_WRITEABLE;
		rtk_sdk_parts[i].of_node = np;
		rtk_sdk_parts[i].types = (const char *const *)NULL;
	}

	if ( add_mtd_partitions(mtd, rtk_sdk_parts, ARRAY_SIZE(rtk_sdk_parts)) ) {
		printk("%s:%d Add partition Fail\n", __FUNCTION__, __LINE__);
	}
	else {
		printk("%s:%d Add partition Success\n", __FUNCTION__, __LINE__);
		return 0;
	}

	pnand_scan_tail_release(mtd);
err4:
	pnand_scan_ident_release(mtd);
err3:
	platform_set_drvdata(pdev, NULL);
	kfree(mtd);
err2:
	kfree((struct sw_pnand_chip*)chip);
err1:
	return ret;
}

static int
sw_pnand_remove(struct platform_device *pdev)
{
	struct nand_chip *chip = platform_get_drvdata(pdev);
	struct sw_pnand_chip * _chip = (struct sw_pnand_chip*)chip;
	struct mtd_info *mtd = _chip->mtd;

	kfree(mtd);
	kfree((struct sw_pnand_chip*)_chip);

	return 0;
}



static struct platform_driver sw_pnand_driver = {
	.probe	= sw_pnand_probe,
	.remove	= sw_pnand_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = "sw_pnand",
		.of_match_table = of_match_ptr(sw_pnand_dt_ids),
	},
};

static int __init sw_pnand_init(void)
{
	return platform_driver_register(&sw_pnand_driver);
}

static void __exit sw_pnand_exit(void)
{
	platform_driver_unregister(&sw_pnand_driver);
}

module_init(sw_pnand_init);
module_exit(sw_pnand_exit);



MODULE_DESCRIPTION("Parallel NAND framework");
MODULE_LICENSE("GPL v2");
