#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/mtd/flashchip.h>
#include <linux/io.h>
#include <linux/slab.h>


void __iomem *ecc_ctl_base;
static spinlock_t	ecc_chip_lock;
static wait_queue_head_t ecc_wq;
static int ecc_state;

int
ecc_get_device(int new_state)
{
	DECLARE_WAITQUEUE(wait, current);

	/*
	 * Grab the lock and see if the device is available
	 */
	while (1) {
		spin_lock(&(ecc_chip_lock));
		if (ecc_state == FL_READY) {
			ecc_state = new_state;
			spin_unlock(&(ecc_chip_lock));
			break;
		}
		if (new_state == FL_PM_SUSPENDED) {
			spin_unlock(&(ecc_chip_lock));
			return (ecc_state == FL_PM_SUSPENDED) ? 0 : -EAGAIN;
		}
		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&(ecc_wq), &wait);
		spin_unlock(&(ecc_chip_lock));
		schedule();
		remove_wait_queue(&(ecc_wq), &wait);
	}
	return 0;
}

void
ecc_release_device(void)
{

	/* Release the chip */
	spin_lock(&(ecc_chip_lock));
	ecc_state = FL_READY;
	wake_up(&(ecc_wq));
	spin_unlock(&(ecc_chip_lock));
}

static struct of_device_id ecc_ctlr_dt_ids[] = {
	{.compatible = "rtk,ecc_ctlr",},
	{},
};

MODULE_DEVICE_TABLE(of, ecc_ctlr_dt_ids);

static int
ecc_ctlr_probe(struct platform_device *pdev)
{
	struct resource mem_resource;
	const struct of_device_id *match;
	struct device_node *np = pdev->dev.of_node;
	int ret = 0;
	match = of_match_device(ecc_ctlr_dt_ids, &pdev->dev);
	if (!match)
		return -EINVAL;
	dev_notice(&pdev->dev, "ECC ctrl driver\n");
#if 1
	/* Nand Flash Controller addr base */
	ret = of_address_to_resource(np, 0, &mem_resource);
	if (ret) {
		dev_warn(&pdev->dev, "invalid address %d\n", ret);
		return ret;
	}

	ecc_ctl_base = devm_ioremap(&pdev->dev, mem_resource.start,
		resource_size(&mem_resource));
#endif

	init_waitqueue_head(&(ecc_wq));
	spin_lock_init(&(ecc_chip_lock));
	ecc_state = FL_READY;

	return ret;
}


static int
ecc_ctlr_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver ecc_ctlr_driver = {
	.probe	= ecc_ctlr_probe,
	.remove	= ecc_ctlr_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = "ecc_ctlr",
		.of_match_table = of_match_ptr(ecc_ctlr_dt_ids),
	},
};

static int __init ecc_ctrl_init(void)
{
	return platform_driver_register(&ecc_ctlr_driver);
}

static void __exit ecc_ctrl_exit(void)
{
	platform_driver_unregister(&ecc_ctlr_driver);
}

fs_initcall(ecc_ctrl_init);
module_exit(ecc_ctrl_exit);


MODULE_DESCRIPTION("ECC Controller framework");
MODULE_LICENSE("GPL v2");

