/* Virtual terminal interface shell.
 * Copyright (C) 2000 Kunihiro Ishiguro
 *
 * This file is part of GNU Zebra.
 *
 * GNU Zebra is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * GNU Zebra is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Zebra; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <libvty/zebra.h>
#include <sys/un.h>
#include <setjmp.h>
#include <sys/wait.h>
#include <pwd.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>

#include <readline/readline.h>
#include <readline/history.h>
//#include <lib/version.h>
#include <libvty/version.h>
#include "getopt.h"
#include <libvty/command.h>
#include <libvty/memory.h>
#include <libcmd/cmd.h>
#include <libsal/sal_sys.h>
#include <libsal/sal_util.h>
#ifdef CONFIG_SYS_TECH_SUPPORT
#include <libsal/sal_sys_tech.h>
#endif

#include <libfds/fds_export.h>
#include <libfds/fds_default_init.h>
#ifdef CONFIG_SYS_AAA
#include <libsal/sal_aaa.h>
#endif

//#include "vtysh/vtysh.h"
//#include "vtysh/vtysh_user.h"
#include "vtysh.h"
#include "vtysh_user.h"
#include <termios.h>
#include <common/sys_type.h>
#include <common/sys_def.h>
#include <cli/cli_main.h>
// for bfap purpose
#ifdef CONFIG_SYS_SEC
#include <libsal/sal_sec.h>
#endif

#ifdef CONFIG_SYS_UI_CLI_SESSION
#define CLI_MAX_CONNECT CONFIG_SYS_UI_CLI_SESSION
#else
#define CLI_MAX_CONNECT 2
#endif
#ifdef CONFIG_SYS_DEBUG_SHELL
int diagdbg_flag = DIAGDBG_OP_NONE;
int diagdbg_auth = DIAGDBG_AUTH_NONE;
#endif

/* RTK: SSH login CLI directly (Start)*/
char ssh_username[CAPA_SYS_USERNAME_LEN + 1];
/* RTK: SSH login CLI directly (End)*/

/* VTY shell program name. */
char *progname;

/* Configuration file name and directory. */
//char config_default[] = SYSCONFDIR VTYSH_DEFAULT_CONFIG;
char history_file[MAXPATHLEN];

/* Flag for indicate executing child command. */
int execute_flag = 0;

/* For sigsetjmp() & siglongjmp(). */
static sigjmp_buf jmpbuf;

/* Flag for avoid recursive siglongjmp() call. */
static int jmpflag = 0;

/* A static variable for holding the line. */
static char *line_read;

/* Master of threads. */
struct thread_master *master;

/* Record node before execute command for sigint action */
static int current_node = VIEW_NODE;
static int exec_cmd = FALSE;

/* Command logging */
FILE *logfile;

static void
_terminal_initial(void)
{
    struct termios t;

    tcgetattr(STDIN_FILENO, &t);

    /* Mapping backspace key to ascii value '8'
     * (origianl value is '127')
     */
    t.c_cc[VERASE] = 0x8;

    /* Enable echo*/
    t.c_lflag |= (ICANON | ECHO);

    tcsetattr(STDIN_FILENO, TCSANOW, &t);
}

#if 1 /* RTK: Allow timeout for input_str funciton */
int login_idle_timeout = -1;
static struct termios oldt;
static void input_str_timeout_exit(void)
{
    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
    cli_exit();
}

static int
_getch_with_timeout(void)
{
    int ch;
    struct termios newt;

    if (login_idle_timeout < 0)
        login_idle_timeout = (int)DFLT_CLI_LOGIN_IDLE_TIMEOUT;

    tcgetattr(STDIN_FILENO, &oldt);
    newt = oldt;
    newt.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &newt);

    signal(SIGALRM, (void *)input_str_timeout_exit);
    alarm(login_idle_timeout);

    ch = getchar();
    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);

    alarm(0);

    return ch;
}

int _vtysh_main_input_str_with_timeout(char *buf, int size, char echo)
{
    unsigned int pos = 0;
    char c = 0;

    if (buf == NULL)
        return -1;

    while (c != '\n' && c != '\r')
    {
        c = (char)_getch_with_timeout();

        //printf("{%d}", c);

        if ((c == 8) || (c == 127))
        {
            if (pos > 0)
            {
                putc('\b', stdout);
                putc(' ', stdout);
                putc('\b', stdout);
                pos--;
            }
            continue;
        }

        if (c < 32 || c > 126)
            continue;

        if (pos < size)
        {
            if (echo == 0)
                putc(c, stdout);
            else if (echo > 0)
                putc(echo, stdout);

            buf[pos] = c;
            pos++;
        }
    }

    putc('\n', stdout);
    if (pos < size)
    {
        buf[pos] = '\0';
    }

    return pos;
}
#endif /* RTK: Allow timeout for input_str funciton */

#ifdef CONFIG_SYS_UI_CLI_PASSRETRY_SILENT
void _vtysh_main_silent_time(void)
{
    struct termios oldt, newt;
    sys_cliRetrySilent_t retrySilent;
    sys_line_cliType_t type = SYS_LINE_CLI_CONSOLE;

    if (access_flag == SYS_ACCESS_CLI_CONSOLE)
        type = SYS_LINE_CLI_CONSOLE;
    else if (access_flag == SYS_ACCESS_CLI_TELNET)
        type = SYS_LINE_CLI_TELNET;
#ifdef CONFIG_USER_SSH_SSHD
    else if (access_flag == SYS_ACCESS_CLI_SSH)
        type = SYS_LINE_CLI_SSH;
#endif

    SYS_MEM_CLEAR(retrySilent);
    if (SYS_ERR_OK != sal_sys_cliRetrySilent_get(type, &retrySilent))
        retrySilent.silentTime = 0;

    if (retrySilent.silentTime == 0)
    {
        if (SYS_ACCESS_CLI_CONSOLE != access_flag)
            cli_exit();
        else
            return;
    }

    tcgetattr(STDIN_FILENO, &oldt);
    newt = oldt;
    /* Ignore CR */
    newt.c_iflag |= IGNCR;
    tcsetattr(STDIN_FILENO, TCSANOW, &newt);

    sleep(retrySilent.silentTime);

    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
}
#endif /* CONFIG_SYS_UI_CLI_PASSRETRY_SILENT */


void _vtysh_main_get_userinput(char *hint, char *inputUsername, size_t inputMaxSize, char echo)
{
    printf("%s", hint);
    osal_memset(inputUsername, 0, inputMaxSize + 1);
    _vtysh_main_input_str_with_timeout(inputUsername, inputMaxSize, echo);
}

#ifdef CONFIG_SYS_AAA
void _login_fail_init(int32 loginResult, char *inputUsername, uint32 *isInputUser)
{
    if (loginResult != SYS_ERR_OK)
    {
        printf("%% Authentication Failed\n\n");
        *isInputUser = FALSE;

        SYS_LOG(LOG_CAT_AAA, LOG_AAA_AUTH_FAIL,
                "login");

        if (loginResult != SYS_ERR_USERNAME_INVALID)
        {
            sys_trap_trapInfo_t trapInfo;

            SYS_TRAP_INIT(trapInfo, SYS_TRAP_AUTHENTICATE);

            if (SYS_TRAP_SEND(&trapInfo))
            {
                SYS_LOG(LOG_CAT_TRAPMGR, LOG_TRAPMGR_SNMP_TRAP_AUTH_FAIL, DFLT_SYS_ADMIN_USERNAME);
            }

            SYS_LOG(LOG_CAT_AAA, LOG_AAA_USER_REJECT,
                    text_access_type[access_flag].text,
                    inputUsername,
                    (0 == osal_strlen(currHost)) ? "async" : currHost);
        }
        else
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_REJECT,
                    text_access_type[access_flag].text,
                    (0 == osal_strlen(currHost)) ? "async" : currHost);
        }
    }
}



vtysh_login_result_t vtysh_login_authen()
{
    int32 ret = SYS_ERR_FAILED;
    vtysh_login_result_t authenReturn = VTYSH_LOGIN_FAIL;
    char input_username[CAPA_SYS_USERNAME_LEN + 1];
    char input_password[CAPA_SYS_PASSWORD_LEN + 1];
    uint32 userIdx;
    int32 userPriv;
    sys_auth_type_t typePrio[CAPA_AAA_AUTHTYPE_PRI_NUM] = {0};
    uint32 i = 0;
    uint32 isInputUser = FALSE;
    sal_aaa_loginAuthTypePrio_get(access_flag, typePrio);

#ifdef CONFIG_SYS_BFAP
    // get global lockoutPenalty
    sal_sec_bfap_sLockoutPenalty_t *globalLockoutPenalty =
        sal_sec_getBfapLockoutPenalty(SAL_SEC_BFAP_GLOBALTYPE,
                                      SAL_SEC_BFAP_GLOBALID,
                                      SAL_SEC_BFAP_GLOBALNAME);
    if (NULL == globalLockoutPenalty)
    {
        BFAP_DEBUGMESSAGE("%s", "error: get global bfap penalty fail");
        return VTYSH_LOGIN_FAIL;
    }
    if (BFAP_LOCKSTATUS_LOCKED == sal_sec_bfap_checkForPassing(globalLockoutPenalty))
    {
        BFAP_DEBUGMESSAGE("%s", "warning: global locked");
        return VTYSH_LOGIN_LOCK;
    }
#endif /* CONFIG_SYS_BFAP */

    /* It is not allowed to have auth method after 'none'.
        So, if none is in first priority, just return authentication success.
    */
    if (typePrio[0] == SYS_AUTH_TYPE_NONE)
    {
        return VTYSH_LOGIN_SUCCESS;
    }
    /* Behavior:
        1. If 'enable' type is in 1st priority, other types will not be applied.
        2. If 'enable' type is in the middle or last priority, it will be ignored.
        3. 'enable' type allow user to retry 3 times and then re-start session.
        4. 'enable' type will verify the password with privilege level 15
    */
    if (typePrio[0] == SYS_AUTH_TYPE_ENABLE)
    {
        _vtysh_main_get_userinput("Password: ", input_password, CAPA_SYS_PASSWORD_LEN, '*');
        ret = sal_aaa_localEnablePassword_authen(15, input_password);
        if (ret == SYS_ERR_OK)
        {
#ifdef CONFIG_SYS_BFAP
            sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_SUCCESS, globalLockoutPenalty);
#endif /* CONFIG_SYS_BFAP */
            return VTYSH_LOGIN_SUCCESS;
        }
        printf("%% Authentication Failed\n\n");
#ifdef CONFIG_SYS_BFAP
        sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, globalLockoutPenalty);
#endif /* CONFIG_SYS_BFAP */
        return VTYSH_LOGIN_FAIL;
    }

    osal_memset(currUserName, 0, sizeof(currUserName));
    osal_memset(currPassWord, 0, sizeof(currPassWord));

    for (i = 0; i < CAPA_AAA_AUTHTYPE_PRI_NUM; i++)
    {
        if (SYS_AUTH_TYPE_EMPTY == typePrio[i])
            break;

        if ((i > 0) && (typePrio[i] > SYS_AUTH_TYPE_NONE))
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_METHOD_NEW,
                    text_aaa_auth_method[typePrio[i]].text);
        }

        switch (typePrio[i])
        {
        case SYS_AUTH_TYPE_LOCAL:
            userIdx = 0;
            userPriv = -1;
            if (currUserName[0] == '\0')
            {
                _vtysh_main_get_userinput("Username: ", input_username, CAPA_SYS_USERNAME_LEN, 0);
                strlcpy(currUserName, input_username, sizeof(currUserName));
                isInputUser = TRUE;
            }
            ret = sal_aaa_localLoginUsername_authen(input_username, &userIdx);
            if (ret != SYS_ERR_OK)
            {
#ifdef CONFIG_SYS_BFAP
                sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, globalLockoutPenalty);
#endif /* CONFIG_SYS_BFAP */
                authenReturn = VTYSH_LOGIN_FAIL;
                _login_fail_init(ret, input_username, &isInputUser);
                return authenReturn;
            }
#ifdef CONFIG_SYS_BFAP
            BFAP_DEBUGMESSAGE("get user %d", userIdx);
            sal_sec_bfap_sLockoutPenalty_t *userLockoutPenalty =
                sal_sec_getBfapLockoutPenalty(typePrio[i], userIdx, currUserName);
            if (BFAP_LOCKSTATUS_LOCKED == sal_sec_bfap_checkForPassing(userLockoutPenalty))
            {
                BFAP_DEBUGMESSAGE("warning: user %s locked", input_username);
                return VTYSH_LOGIN_LOCK;
            }
#endif /* CONFIG_SYS_BFAP */
            if (currPassWord[0] == '\0')
            {
                _vtysh_main_get_userinput("Password: ", input_password, CAPA_SYS_PASSWORD_LEN, '*');
                strlcpy(currPassWord, input_password, sizeof(currPassWord));
            }
            ret = sal_aaa_localLoginPassword_authen(userIdx, input_password, &userPriv);
            /* If username exist but password auth fail, return auth fail.*/
            if (ret != SYS_ERR_OK)
            {
#ifdef CONFIG_SYS_BFAP
                sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, userLockoutPenalty);
#endif /* CONFIG_SYS_BFAP */
                authenReturn = VTYSH_LOGIN_PASSWORD_FAIL;
                _login_fail_init(ret, input_username, &isInputUser);
                return authenReturn;
            }
            else // login success
            {
#ifdef CONFIG_SYS_BFAP
                sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_SUCCESS, userLockoutPenalty);
                // TO_BE_DISCUSSED
                sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_SUCCESS, globalLockoutPenalty);
#endif /* CONFIG_SYS_BFAP */
                currUserPriv = userPriv;
                currAuthType = SYS_AUTH_TYPE_LOCAL;
                authenReturn = VTYSH_LOGIN_SUCCESS;
                return authenReturn;
            }
            break;
#ifdef CONFIG_SYS_LIB_TACPLUS
        case SYS_AUTH_TYPE_TACPLUS:
        {
            struct session *pTacplusSession = NULL;
            ret = sal_aaa_tacplusSession_init(&pTacplusSession);
            /* If server cannot be connected, just go to next method */
            if (ret != SYS_ERR_OK)
            {
                break;
            }

            if (currUserName[0] == '\0')
            {
                _vtysh_main_get_userinput("Username: ", input_username, CAPA_SYS_USERNAME_LEN, 0);
                strlcpy(currUserName, input_username, sizeof(currUserName));
                isInputUser = TRUE;
            }
            ret = sal_aaa_tacplusLoginUsername_authen(pTacplusSession, input_username);
            if (ret != SYS_ERR_OK)
            {
                if (pTacplusSession != NULL)
                    sal_aaa_tacplusSession_close(pTacplusSession);
                /* If server can be connected but auth fail, do not go to next method. */
                if (ret == SYS_ERR_USERNAME_INVALID)
                {
                    //TODO sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, globalLockoutPenalty);
                    _login_fail_init(ret, input_username, &isInputUser);
                }
                /* If server key is error, just go to next method */
                else
                    break;
            }
            //
            if (currPassWord[0] == '\0')
            {
                _vtysh_main_get_userinput("Password: ", input_password, CAPA_SYS_PASSWORD_LEN, '*');
                strlcpy(currPassWord, input_password, sizeof(currPassWord));
            }
            ret = sal_aaa_tacplusLoginPassword_authen(pTacplusSession, input_password);
            if (pTacplusSession != NULL)
                sal_aaa_tacplusSession_close(pTacplusSession);
            /* If server can be connected but auth fail, do not go to next method. */
            if (ret != SYS_ERR_OK)
            {
                //TODO sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, globalLockoutPenalty);
                _login_fail_init(ret, input_username, &isInputUser);
            }
            currAuthType = SYS_AUTH_TYPE_TACPLUS;
            break;
        }
#endif
#ifdef CONFIG_SYS_LIB_RADIUS
        case SYS_AUTH_TYPE_RADIUS:
            if (currUserName[0] == '\0')
            {
                _vtysh_main_get_userinput("Username: ", input_username, CAPA_SYS_USERNAME_LEN, 0);
                strlcpy(currUserName, input_username, sizeof(currUserName));
                isInputUser = TRUE;
            }
            if (currPassWord[0] == '\0')
            {
                _vtysh_main_get_userinput("Password: ", input_password, CAPA_SYS_PASSWORD_LEN, '*');
                strlcpy(currPassWord, input_password, sizeof(currPassWord));
            }
            ret = sal_aaa_radiusLoginUsername_authen(input_username, input_password);

            if (ret != SYS_ERR_OK)
            {
                /* If server can be connected but auth fail, do not go to next method. */
                if (ret == SYS_ERR_RADIUS_AUTH_REJECT)
                {
                    //TODO sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, globalLockoutPenalty);
                    _login_fail_init(ret, input_username, &isInputUser);
                }
                /* If server key is error or server can't be connected, just go to next method */
                else
                    break;
            }
            currAuthType = SYS_AUTH_TYPE_RADIUS;
            break;
#endif
        case SYS_AUTH_TYPE_NONE:
            /* If user name is empty, return auth fail */
            if ((0 == osal_strlen(currUserName)) && (TRUE == isInputUser))
            {
                //TODO sal_sec_bfap_updateTryStatus(BFAP_TRYSTATUS_FAIL, globalLockoutPenalty);
                _login_fail_init(ret, input_username, &isInputUser);
            }

            ret = SYS_ERR_OK;
            currAuthType = SYS_AUTH_TYPE_NONE;

            currUserPriv = SYS_LOCALUSER_PRIV_ADMIN;

            break;

        default:
            break;
        }

        if (ret == SYS_ERR_OK)
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_AUTH_SUCCESS,
                    text_aaa_auth_method[typePrio[i]].text,
                    "login");

            SYS_LOG(LOG_CAT_AAA, LOG_AAA_CONNECT,
                    text_access_type[access_flag].text,
                    input_username,
                    (0 == osal_strlen(currHost)) ? "async" : currHost);
            return VTYSH_LOGIN_SUCCESS;
        }
        else
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_METHOD_FAIL,
                    text_aaa_auth_method[typePrio[i]].text);
        }
    }
    _login_fail_init(ret, input_username, &isInputUser);
    /* Failed 3 times. Close this CLI session. */
    //kill(0, SIGTERM);
    return 0;
}

int enable_authen(int enblPriv)
{
    int32 ret = SYS_ERR_FAILED;
    char input_password[CAPA_SYS_PASSWORD_LEN + 1];
    sys_auth_type_t typePrio[CAPA_AAA_AUTHTYPE_PRI_NUM] = {0};
    int i = 0;

    sal_aaa_enableAuthTypePrio_get(access_flag, typePrio);

    /* It is not allowed to have auth method after 'none'.
       So, if none is in first priority, just return authentication success. */
    if (typePrio[0] == SYS_AUTH_TYPE_NONE)
    {
        return SYS_ERR_OK;
    }

    printf("Password: ");
    osal_memset(input_password, 0, sizeof(input_password));
    ret = input_str(input_password, CAPA_SYS_PASSWORD_LEN, '*');
    if (ret < 0)
        return SYS_ERR_PASSWORD_INVALID;

    for (i = 0; i < CAPA_AAA_AUTHTYPE_PRI_NUM; i++)
    {
        if (SYS_AUTH_TYPE_EMPTY == typePrio[i])
            break;

        //printf("typePrio[%d]=%d\n", i, typePrio[i]);

        if ((i != 0) && (typePrio[i] > SYS_AUTH_TYPE_NONE))
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_METHOD_NEW,
                    text_aaa_auth_method[typePrio[i]].text);
        }

        switch (typePrio[i])
        {
        case SYS_AUTH_TYPE_ENABLE:
        {
            ret = sal_aaa_localEnablePassword_authen(enblPriv, input_password);
            break;
        }
#ifdef CONFIG_SYS_LIB_TACPLUS
        case SYS_AUTH_TYPE_TACPLUS:
        {
            struct session *pTacplusSession = NULL;
            char enblUsername[10] = {0};

            ret = sal_aaa_tacplusSession_init(&pTacplusSession);
            /* If server cannot be connected, just go to next method */
            if (ret != SYS_ERR_OK)
                break;

            osal_snprintf(enblUsername, sizeof(enblUsername), "$enab%d$", enblPriv);
            ret = sal_aaa_tacplusLoginUsername_authen(pTacplusSession, enblUsername);
            if (ret != SYS_ERR_OK)
            {
                if (pTacplusSession != NULL)
                    sal_aaa_tacplusSession_close(pTacplusSession);
                /* If server can be connected but auth fail, do not go to next method. */
                if (ret == SYS_ERR_USERNAME_INVALID)
                    goto enbl_fail;
                /* If server key is error, just go to next method */
                else
                    break;
            }
            ret = sal_aaa_tacplusLoginPassword_authen(pTacplusSession, input_password);
            if (pTacplusSession != NULL)
                sal_aaa_tacplusSession_close(pTacplusSession);
            /* If server can be connected but auth fail, do not go to next method. */
            if (ret != SYS_ERR_OK)
                goto enbl_fail;
            break;
        }
#endif
#ifdef CONFIG_SYS_LIB_RADIUS
        case SYS_AUTH_TYPE_RADIUS:
        {
            char enblUsername[10] = {0};
            osal_snprintf(enblUsername, sizeof(enblUsername), "$enab%d$", enblPriv);
            ret = sal_aaa_radiusLoginUsername_authen(enblUsername, input_password);
            /* If server can be connected but auth fail, do not go to next method. */
            if (ret != SYS_ERR_OK && ret != SYS_ERR_RADIUS_SRV_ERR)
                goto enbl_fail;
            break;
        }
#endif

        case SYS_AUTH_TYPE_NONE:
            ret = SYS_ERR_OK;
            currAuthType = SYS_AUTH_TYPE_NONE;
            break;

        default:
            break;
        }

        if (ret == SYS_ERR_OK)
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_AUTH_SUCCESS,
                    text_aaa_auth_method[typePrio[i]].text,
                    "enable");

            SYS_LOG(LOG_CAT_AAA, LOG_AAA_ENABLE_SUCCESS,
                    currUserName,
                    text_access_type[access_flag].text,
                    enblPriv);
            return ret;
        }
        else
        {
            SYS_LOG(LOG_CAT_AAA, LOG_AAA_METHOD_FAIL,
                    text_aaa_auth_method[typePrio[i]].text);
        }
    }

#ifdef CONFIG_SYS_LIB_TACPLUS
enbl_fail:
#endif
    SYS_LOG(LOG_CAT_AAA, LOG_AAA_AUTH_FAIL,
            "enable");

    SYS_LOG(LOG_CAT_AAA, LOG_AAA_ENABLE_FAIL,
            currUserName,
            text_access_type[access_flag].text,
            enblPriv);

    return ret;
}

#else /* !CONFIG_SYS_AAA */
int32 vtysh_login_authen(void)
{
    char input_username[CAPA_SYS_USERNAME_LEN + 1] = {0};
    char input_password[CAPA_SYS_PASSWORD_LEN + 1] = {0};
    sys_userInfo_t userInfo;
    uint32 i;
    uint32 isFound = FALSE;
    uint32 retryCnt = 3; /* Retry 3 times */
#ifdef CONFIG_SYS_UI_CLI_PASSRETRY_SILENT
    sys_cliRetrySilent_t retrySilent;
    uint32 cfgRetryCnt = 0;
    sys_line_cliType_t type = 0;

    if (access_flag == SYS_ACCESS_CLI_CONSOLE)
        type = SYS_LINE_CLI_CONSOLE;
    else if (access_flag == SYS_ACCESS_CLI_TELNET)
        type = SYS_LINE_CLI_TELNET;
#ifdef CONFIG_USER_SSH_SSHD
    else if (access_flag == SYS_ACCESS_CLI_SSH)
        type = SYS_LINE_CLI_SSH;
#endif

    sal_sys_cliRetrySilent_get(type, &retrySilent);

    cfgRetryCnt = retrySilent.retryNum;

    if (cfgRetryCnt == 0)
        retryCnt = 0xFFFFFFFF;
    else
        retryCnt = cfgRetryCnt;
#endif /* CONFIG_SYS_UI_CLI_PASSRETRY_SILENT */

    while (retryCnt > 0)
    {
        isFound = FALSE;

#ifdef CONFIG_USER_SSH_SSHD
        if (SYS_ACCESS_CLI_SSH == access_flag)
        {
            /* RTK: SSH login CLI directly*/
            osal_memset(input_username, 0, sizeof(input_username));
            strlcpy(input_username, ssh_username, sizeof(input_username));
        }
        else
#endif
        {
            _vtysh_main_get_userinput("Username: ", input_username, CAPA_SYS_USERNAME_LEN, 0);
        }

        for (i = 0; i < CAPA_SYS_LOCALUSER_NUM; i++)
        {
            osal_memset(&userInfo, 0, sizeof(userInfo));
            SYS_ERR_CONTINUE(sal_sys_localUser_get(i, &userInfo));

            /* priv < 0 means empty account */
            if (userInfo.priv < 0)
                continue;

            if (0 == osal_strcmp(userInfo.username, input_username))
            {
                isFound = TRUE;
                break;
            }
        }

        if (isFound == FALSE)
        {
            printf("Incorrect User Name!!\n\n");
            retryCnt--;

            continue;
        }

        isFound = FALSE;

#ifdef CONFIG_USER_SSH_SSHD
        if (SYS_ACCESS_CLI_SSH == access_flag)
        {
            /* RTK: SSH login CLI directly*/
            isFound = TRUE;
        }
        else
#endif
        {
            _vtysh_main_get_userinput("Password: ", input_password, CAPA_SYS_PASSWORD_LEN, '*');
            if (SYS_PASSWD_TYPE_ENCRYPT == userInfo.passwdType)
            {
                char crypt_password[CAPA_SYS_CRYPT_PASSWORD_LEN] = {0};
                sal_util_str_encrypt(input_password, crypt_password);
                if (0 == osal_strcmp(userInfo.password, crypt_password))
                    isFound = TRUE;
            }
            else if (SYS_PASSWD_TYPE_CLEARTEXT == userInfo.passwdType)
            {
                if (0 == osal_strcmp(userInfo.password, input_password))
                    isFound = TRUE;
            }
            else
            {
                isFound = TRUE;
            }
        }

        if (isFound == TRUE)
        {
            strlcpy(currUserName, input_username, sizeof(currUserName));
            currUserPriv = userInfo.priv;
            currpriv = userInfo.priv;
            if (userInfo.priv == SYS_LOCALUSER_PRIV_ADMIN)
                vty->node = ENABLE_NODE;

            SYS_LOG(LOG_CAT_AAA, LOG_AAA_CONNECT,
                    text_access_type[access_flag].text,
                    input_username,
                    (0 == osal_strlen(currHost)) ? "async" : currHost);
            return 1;
        }

        retryCnt--;
        sys_trap_trapInfo_t trapInfo;

        SYS_TRAP_INIT(trapInfo, SYS_TRAP_AUTHENTICATE);

        if (SYS_TRAP_SEND(&trapInfo))
        {
            SYS_LOG(LOG_CAT_TRAPMGR, LOG_TRAPMGR_SNMP_TRAP_AUTH_FAIL, DFLT_SYS_ADMIN_USERNAME);
        }

        SYS_LOG(LOG_CAT_AAA, LOG_AAA_USER_REJECT,
                text_access_type[access_flag].text,
                input_username,
                (0 == osal_strlen(currHost)) ? "async" : currHost);
    }

    //kill(0, SIGTERM);
    return VTYSH_LOGIN_FAIL;
}

int enable_authen(int enblPriv)
{
    int32 ret = SYS_ERR_PASSWORD_INVALID;
    char input_password[CAPA_SYS_PASSWORD_LEN + 1];
    sys_enblPasswd_t adminEnblPasswd;

    /* Do not need to authentication if the user is already the Admin type */
    if (currpriv == SYS_LOCALUSER_PRIV_ADMIN)
        return SYS_ERR_OK;

    /* Only save admin enable password on index 0 */
    SYS_ERR_CHK(sal_sys_enablePassword_get(0, &adminEnblPasswd));

    if (SYS_PASSWD_TYPE_NOPASSWD == adminEnblPasswd.passwdType)
    {
        currpriv = SYS_LOCALUSER_PRIV_ADMIN;
        return SYS_ERR_OK;
    }

    printf("Password: ");
    osal_memset(input_password, 0, sizeof(input_password));
    ret = input_str(input_password, CAPA_SYS_PASSWORD_LEN, '*');
    if (ret < 0)
        return SYS_ERR_PASSWORD_INVALID;

    ret = SYS_ERR_PASSWORD_INVALID;

    if (SYS_PASSWD_TYPE_ENCRYPT == adminEnblPasswd.passwdType)
    {
        char crypt_password[CAPA_SYS_CRYPT_PASSWORD_LEN] = {0};
        sal_util_str_encrypt(input_password, crypt_password);
        if (0 == osal_strcmp(adminEnblPasswd.password, crypt_password))
            ret = SYS_ERR_OK;
    }
    else if (SYS_PASSWD_TYPE_CLEARTEXT == adminEnblPasswd.passwdType)
    {
        if (0 == osal_strcmp(adminEnblPasswd.password, input_password))
            ret = SYS_ERR_OK;
    }
    if (ret == SYS_ERR_OK)
    {
        currpriv = SYS_LOCALUSER_PRIV_ADMIN;
        SYS_LOG(LOG_CAT_AAA, LOG_AAA_ENABLE_SUCCESS,
                currUserName,
                text_access_type[access_flag].text,
                currpriv);
    }
    else
    {
        SYS_LOG(LOG_CAT_AAA, LOG_AAA_ENABLE_FAIL,
                currUserName,
                text_access_type[access_flag].text,
                currpriv);
    }

    return ret;
}

#endif /* CONFIG_SYS_AAA */

/* After interrupt from signal and not going back, cfg lock may have chance not be released and block. */
static void _cfg_semaphore_unlock(void)
{
    uint32 i;

    for (i = 0; i < CFG_ID_END; i++)
    {
        SYS_ERR_NORET(cfg_close(i));
    }

    return;
}
/* After interrupt from signal and not going back, fd may have chance not be closed. */
static void _process_fd_close(void)
{
#define USER_FD_START 4
#define USER_FD_MAX 1024
    int32 fd;

    for (fd = USER_FD_START; fd < USER_FD_MAX; fd++)
    {
        close(fd);
    }

    return;
}

/* SIGTSTP handler.  This function care user's ^Z input. */
void sigtstp(int sig)
{
    uint32 isCmdIntr = FALSE;
    sys_enable_t enable = DISABLED;

    /* Forbit interrupt */
    if (TRUE == intr_forbid_flag)
        return;

    cmd_textline_enable_get(&enable);
    if (ENABLED == enable)
        return;

    _cfg_semaphore_unlock();
    _process_fd_close();

    if (TRUE == g_isCliFwProcess)
    {
        if (SYS_ERR_OK != sal_sys_fwImageLock_check())
            sal_sys_fwImageLock_give();

        cmd_util_progressPrint_stop();

        g_isCliFwProcess = FALSE;
    }

    /* check if tech-support generation is running or not */
    if (SYS_ERR_OK != sal_tech_fileLock_check())
    {
        unlink(SYS_POLLD_ACT_WEB_TECH_LOCK);
        vty_more_normal();
        cmd_util_progressPrint_stop();
    }

    /* Workaround for Ctrl+z interrupt feature. */
    {
        struct termios t;

        tcgetattr(STDIN_FILENO, &t);

        /* If ICRNL is normal (existing), enable the echo.
         If ICRNL is abnormal (missing), do nothing.
         "show running-config" command is normal and others may abnormal */
        if ((t.c_iflag & ICRNL))
        {
            vty_echo_enable();
            isCmdIntr = TRUE;
        }
    }

    /* Restore node when the command is interruptted */
    if (TRUE == exec_cmd && TRUE == isCmdIntr)
    {
        vty->node = current_node;
        exec_cmd = FALSE;
    }
    /* If Ctrl+z is not acting during command executing, go to privilege mode */
    else
    {
        if (vty->node >= ENABLE_NODE)
        {
            vty->node = ENABLE_NODE;
            current_node = ENABLE_NODE;
        }
    }

    /* Initialize readline. */
    rl_initialize();
    printf("\n");

    /* Check jmpflag for duplicate siglongjmp(). */
    if (!jmpflag)
        return;

    jmpflag = 0;

    /* Back to main command loop. */
    siglongjmp(jmpbuf, 1);
}

/* SIGINT handler.  This function care user's ^C input.  */
void sigint(int sig)
{
    sys_enable_t enable = DISABLED;

    /* Forbit interrupt */
    if (TRUE == intr_forbid_flag)
        return;

    cmd_textline_enable_get(&enable);
    if (ENABLED == enable)
        return;

    /* Check this process is not child process. */
    if (!execute_flag)
    {
        _cfg_semaphore_unlock();
        _process_fd_close();

        if (TRUE == g_isCliFwProcess)
        {
            if (SYS_ERR_OK != sal_sys_fwImageLock_check())
                sal_sys_fwImageLock_give();

            cmd_util_progressPrint_stop();

            g_isCliFwProcess = FALSE;
        }

        if (SYS_ERR_OK != sal_tech_fileLock_check())
        {
            unlink(SYS_POLLD_ACT_WEB_TECH_LOCK);
            vty_more_normal();
            cmd_util_progressPrint_stop();
        }

        /* Only restore node when the command is interruptted */
        if (TRUE == exec_cmd)
        {
            vty->node = current_node;
            exec_cmd = FALSE;
        }
        /* Workaround for Ctrl+c abnormal issue. */
        {
            struct termios t;

            tcgetattr(STDIN_FILENO, &t);

            /* If ICRNL is normal (existing), enable the echo.
               If ICRNL is abnormal (missing), do nothing.
               "show running-config" command is normal and others may abnormal */
            if ((t.c_iflag & ICRNL))
            {
                vty_echo_enable();
            }
        }

        rl_initialize();
        printf("\n");

        /* Check jmpflag for duplicate siglongjmp(). */
        if (!jmpflag)
            return;

        jmpflag = 0;

        /* Back to main command loop. */
        siglongjmp(jmpbuf, 1);
    }
}

#ifdef CONFIG_SYS_BUILD_DEVEL
#ifdef CONFIG_SYS_DEBUG_SHELL
/* SIGQUIT handler.  This function care user's ^\ input.  */
void sigquit(int sig)
{
    /* RTK: password recovery */
    char sel[2] = {0};
    char confirm[2];
    char input_username[CAPA_SYS_USERNAME_LEN + 1] = {0};
    char input_password[CAPA_SYS_PASSWORD_LEN + 1] = {0};

    sys_enable_t enable = DISABLED;

    cmd_textline_enable_get(&enable);
    if (ENABLED == enable)
        return;

    if (access_flag != SYS_ACCESS_CLI_CONSOLE)
    {
        printf("Do not allow remote user to launch password recovery\n");
        return;
    }

    printf("\nPassword Recovery Password: ");
    osal_memset(input_password, 0, sizeof(input_password));
    input_str(input_password, CAPA_SYS_PASSWORD_LEN, '*');

    if (0 == strcmp(input_password, DFLT_CLI_RECOVERY_PASSWORD))
    {
        sys_userInfo_t userInfo;
        uint32 i;

        printf("Password Recovery - Options\n");
        printf("[P] Password recovery for specific user\n");
        printf("[F] Factory default username/password recovery\n");
        printf("[R] Restore all configurations to factory defaults\n");
        printf("Enter Selection: ");
        input_str(sel, 1, 0);

        /* Password Recovery */
        if ('p' == tolower(sel[0]))
        {
            uint32 i;
            uint32 isFound = FALSE;
            printf("User Name: ");
            input_str(input_username, CAPA_SYS_USERNAME_LEN, 0);
            for (i = 0; i < CAPA_SYS_LOCALUSER_NUM; i++)
            {
                osal_memset(&userInfo, 0, sizeof(userInfo));
                SYS_ERR_CONTINUE(sal_sys_localUser_get(i, &userInfo));

                /* priv < 0 means empty account */
                if (userInfo.priv < 0)
                    continue;

                if (0 == osal_strcmp(userInfo.username, input_username))
                {
                    isFound = TRUE;
                    break;
                }
            }

            if (isFound)
            {
                int32 ret = SYS_ERR_FAILED;

                printf("New Password: ");
                osal_memset(input_password, 0, sizeof(input_password));
                input_str(input_password, CAPA_SYS_PASSWORD_LEN, '*');

                osal_memset(userInfo.password, 0, sizeof(userInfo.password));
                strlcpy(userInfo.password, input_password, sizeof(userInfo.password));
                sal_sys_localUser_del(userInfo.username);
                ret = sal_sys_localUser_add(userInfo);

                if (SYS_ERR_OK == ret)
                {
                    printf("\nPassword recovery success.\n");
                }
                else
                {
                    SYS_PRINTF("%s\n", SYS_ERR_STR(ret));
                    return;
                }
            }
            else
            {
                printf("\nUser is not exist\n");
                return;
            }
        }
        else if ('f' == tolower(sel[0]))
        {
            /* Delete all local user */
            for (i = 0; i < CAPA_SYS_LOCALUSER_NUM; i++)
            {
                osal_memset(&userInfo, 0, sizeof(userInfo));
                SYS_ERR_CONTINUE(sal_sys_localUser_get(i, &userInfo));

                /* priv < 0 means empty account */
                if (userInfo.priv < 0)
                    continue;

                sal_sys_localUser_del(userInfo.username);
            }
            /* Initial System user information */
            fds_sys_localuser_dfltEntry_init();
        }
        else if ('r' == tolower(sel[0]))
        {
            printf("\nNeed to reboot to take effect.\nAre you sure? [y/n] ");
            input_str(confirm, 1, 0);
            if (confirm[0] == 'y')
            {
                SYS_ERR_NORET(sal_sys_restore_defaults(SYS_RESTORE_CLI_VTY_RECOVERY));
                printf("Restore Default Success!\n");
                printf("Rebooting now...\n");
                sal_sys_reboot();
                return;
            }

            printf("\nExit password recovery procedure\n\n");
            return;
        }

        printf("Do you want to save recovery password to startup config now? [y/n] ");
        input_str(confirm, 1, 0);
        if (confirm[0] == 'y')
        {
            cmd_conf_startupConfig_save();
        }
    }
    else
    {
        printf("\nAuthentication Fail.\nDo now allow to launch password recovery procedure.\n");
    }
    printf("\nExit password recovery procedure\n\n");
}
#endif
#endif

/* Signale wrapper for vtysh. We don't use sigevent because
 * vtysh doesn't use threads. TODO */
//RETSIGTYPE *
int *vtysh_signal_set(int signo, void (*func)(int))
{
    int ret;
    struct sigaction sig;
    struct sigaction osig;

    sig.sa_handler = func;
    sigemptyset(&sig.sa_mask);
    sig.sa_flags = 0;
#ifdef SA_RESTART
    sig.sa_flags |= SA_RESTART;
#endif /* SA_RESTART */

    ret = sigaction(signo, &sig, &osig);

    if (ret < 0)
        return ((int *)SIG_ERR);
    else
        return ((int *)osig.sa_handler);
}

/* Initialization of signal handles. */
void vtysh_signal_init(void)
{
    vtysh_signal_set(SIGINT, sigint);
    vtysh_signal_set(SIGTSTP, sigtstp);
#ifdef CONFIG_SYS_BUILD_DEVEL
#ifdef CONFIG_SYS_DEBUG_SHELL
    vtysh_signal_set(SIGQUIT, sigquit);
#endif
#endif
    vtysh_signal_set(SIGPIPE, SIG_IGN);
}

/* Help information display. */
static void
usage(int status)
{
    if (status != 0)
        fprintf(stderr, "Try `%s --help' for more information.\n", progname);
    else
        printf("Usage : %s [OPTION...]\n\n"
               "Integrated shell for Quagga routing software suite. \n\n"
               "-b, --boot               Execute boot startup configuration\n"
               "-c, --command            Execute argument as command\n"
               "-d, --daemon             Connect only to the specified daemon\n"
               "-E, --echo               Echo prompt and command in -c mode\n"
               "-C, --dryrun             Check configuration for validity and exit\n"
               "-h, --help               Display this help and exit\n\n"
               "Note that multiple commands may be executed from the command\n"
               "line by passing multiple -c args, or by embedding linefeed\n"
               "characters in one or more of the commands.\n\n"
               "Report bugs to %s\n",
               progname, ZEBRA_BUG_ADDRESS);

    exit(status);
}

/* VTY shell options, we use GNU getopt library. */
struct option longopts[] =
    {
        {"boot", no_argument, NULL, 'b'},
        /* For compatibility with older zebra/quagga versions */
        {"eval", required_argument, NULL, 'e'},
        {"command", required_argument, NULL, 'c'},
        {"daemon", required_argument, NULL, 'd'},
        {"echo", no_argument, NULL, 'E'},
        {"dryrun", no_argument, NULL, 'C'},
        {"help", no_argument, NULL, 'h'},
        {"noerror", no_argument, NULL, 'n'},
        {0}};

/* Read a string, and return a pointer to it.  Returns NULL on EOF. */
char *
vtysh_rl_gets(void)
{
    HIST_ENTRY *last;
    /* If the buffer has already been allocated, return the memory
   * to the free pool. */
    if (line_read)
    {
        free(line_read);
        line_read = NULL;
    }

    /* Get a line from the user.  Change prompt according to node.  XXX. */
    line_read = readline(vtysh_prompt());

    /* If the line has any text in it, save it on the history. But only if
   * last command in history isn't the same one. */
    if (line_read && *line_read)
    {
        using_history();
        last = previous_history();
        if (!last || strcmp(last->line, line_read) != 0)
        {
            add_history(line_read);
            append_history(1, history_file);
        }
    }

    return (line_read);
}

static void log_it(const char *line)
{
    time_t t = time(NULL);
    struct tm *tmp = localtime(&t);
    char *user = getenv("USER") ?: "boot";
    char tod[64];

    strftime(tod, sizeof tod, "%Y%m%d-%H:%M.%S", tmp);

    fprintf(logfile, "%s:%s %s\n", tod, user, line);
}

#ifdef CONFIG_SYS_DEBUG_SHELL
/* RTK: Diagnostic Console operation implementation */
void diagdbgOperation(void)
{
    char sel[2] = {0};
    int oldNode;
    if (DIAGDBG_OP_DEBUG == diagdbg_flag)
    {
        if (access_flag == SYS_ACCESS_CLI_CONSOLE)
        {
            printf("\n");
            printf("[D] Diag\n");
            printf("[E] Engineering\n");
            printf("[S] Shell\n");
            printf("Enter Selection: ");
            input_str(sel, 1, 0);

            /* Enter CLI debug mode */
            if ('e' == tolower(sel[0]))
            {
                char cmd[16] = {0};

                oldNode = vty->node;
                vty->node = DEBUG_NODE;
                osal_snprintf(cmd, sizeof(cmd), "debug %d", oldNode);
                vtysh_execute(cmd);
            }
            /* Enter SDK Diag */
            else if ('d' == tolower(sel[0]))
            {
                oldNode = vty->node;
                vty->node = DEBUG_NODE;
                vtysh_execute("diag");
                vty->node = oldNode;
            }
            /* Enter Linux shell */
            else if ('s' == tolower(sel[0]))
            {
                if (SYS_ACCESS_CLI_CONSOLE == access_flag)
                {
                    unlink(SYS_CONSOLE_CLI_PID_FILE);
                    exit(0);
                }
                else if (SYS_ACCESS_CLI_TELNET == access_flag)
                    SYSTEM_CMD("sh -a telnet");
#ifdef CONFIG_USER_SSH_SSHD
                else if (SYS_ACCESS_CLI_SSH == access_flag)
                    SYSTEM_CMD("sh -a ssh");
#endif
            }
        }
        /* Remote only allow SDK diag */
        else
        {
            oldNode = vty->node;
            vty->node = DEBUG_NODE;
            vtysh_execute("diag");
            vty->node = oldNode;
        }
    }
    diagdbg_flag = DIAGDBG_OP_NONE;
}
#endif

#ifdef CONFIG_SYS_AAA_LOCALCMD_PRIVCHG
void genLocalPrivCmdsFile(void)
{
    vector cmd_vector = NULL;
    vector descvec;
    struct cmd_element *cmd_element;
    uint32 i, j;
    const char *str = NULL;
    struct desc *desc;
    char cmd[MAX_CMD_STRLEN];
    FILE *fp;

    if (NULL == (fp = fopen(SYS_AAA_CMDPRIV_CONF_TMPFILE, "w")))
        return;

    cmd_vector = vector_copy(cmd_node_vector(cmdvec, CONFIG_NODE));
    for (i = 0; i < vector_active(cmd_vector); i++)
    {
        if ((cmd_element = vector_slot(cmd_vector, i)) != NULL)
        {
            if (cmd_element->cmdpriv != cmd_element->cmdcurrpriv)
            {
                memset(cmd, 0, sizeof(cmd));
                osal_snprintf(cmd, sizeof(cmd), "privilege config level %d \"", cmd_element->cmdcurrpriv);
                for (j = 0; j < vector_active(cmd_element->strvec); j++)
                {
                    descvec = vector_slot(cmd_element->strvec, j);
                    if ((desc = vector_slot(descvec, 0)))
                    {
                        str = desc->cmd;
                        strcat_s(cmd, str, sizeof(cmd));
                        strcat_s(cmd, " ", sizeof(cmd));
                    }
                }
                cmd[strlen(cmd) - 1] = '"';
                cmd[strlen(cmd)] = '\n';
                fprintf(fp, "%s", cmd);
            }
        }
    }
    vector_free(cmd_vector);

    cmd_vector = vector_copy(cmd_node_vector(cmdvec, ENABLE_NODE));
    for (i = 0; i < vector_active(cmd_vector); i++)
    {
        if ((cmd_element = vector_slot(cmd_vector, i)) != NULL)
        {
            if (cmd_element->cmdpriv != cmd_element->cmdcurrpriv)
            {
                memset(cmd, 0, sizeof(cmd));
                osal_snprintf(cmd, sizeof(cmd), "privilege exec level %d \"", cmd_element->cmdcurrpriv);
                for (j = 0; j < vector_active(cmd_element->strvec); j++)
                {
                    descvec = vector_slot(cmd_element->strvec, j);
                    if ((desc = vector_slot(descvec, 0)))
                    {
                        str = desc->cmd;
                        strcat_s(cmd, str, sizeof(cmd));
                        strcat_s(cmd, " ", sizeof(cmd));
                    }
                }
                cmd[strlen(cmd) - 1] = '"';
                cmd[strlen(cmd)] = '\n';
                fprintf(fp, "%s", cmd);
            }
        }
    }
    vector_free(cmd_vector);
    fclose(fp);
}

#endif /* CONFIG_SYS_AAA_LOCALCMD_PRIVCHG */

/*
 * vtysh_cmd_loop -- infinite loop to receive commands and execute
 *   them
 */
void vtysh_cmd_loop(void)
{
    /* Reset timeout timer */
    timeout_reset();

    for (;;)
    {
        while (vtysh_rl_gets())
        {
            /* For Ctrl+z feature */
            exec_cmd = FALSE;

            if (access_flag == SYS_ACCESS_CLI_CONSOLE)
                _terminal_initial();

#ifdef CONFIG_SYS_DEBUG_SHELL
            if (diagdbg_flag != DIAGDBG_OP_NONE)
            {
                diagdbgOperation();
            }
#endif

            if (readln_timeout_chk() >= 0)
            {
                SYS_LOG(LOG_CAT_SYSTEM, LOG_SYSTEM_CLI_TIMEOUT,
                        currUserName,
                        (0 == osal_strlen(currHost)) ? "async" : currHost);
#ifdef CONFIG_SYS_BUILD_RELEASE
                SYSTEM_CMD("clear");
#endif
                printf("\n\nTimeout, console is locked.\n\n\n");
                cli_exit();
            }
            /* Clear timeout timer */
            timeout_clear();
            /* Record current node for Ctrl+c/z */
            exec_cmd = TRUE;
            current_node = vty->node;
            /* Execute command */
            CLI_ERR_HDL(vtysh_execute(line_read));
            /* Record current node for Ctrl+c/z */
            current_node = vty->node;
            /* Reset timeout timer */
            timeout_reset();
        }
    }
}

int isOverMaxConn(uint32 *maxSession)
{
    DIR *dir = NULL;
    FILE *fp = NULL;
    int32 sessionNum = 0;
    int32 pid = 0, readSize = 0, i;
    char cmd[SYS_BUF32_LEN + 1];
    char fullname[CAPA_FILE_FULLNAME_LEN];
    struct dirent *entry = NULL;

    SYS_PARAM_CHK(NULL == maxSession, SYS_ERR_NULL_POINTER);

    switch (access_flag)
    {
        case SYS_ACCESS_CLI_CONSOLE:
            *maxSession = 1;
            break;
        case SYS_ACCESS_CLI_TELNET:
            *maxSession = CLI_MAX_CONNECT;
            break;
#ifdef CONFIG_SYS_APP_SSH
        case SYS_ACCESS_CLI_SSH:
            SYS_ERR_CHK(sal_sys_cliMaxSession_get(SYS_ACCESS_CLI_SSH, maxSession));
            break;
#endif
        default:
            *maxSession = 0;
            return 1;
            break;
    }

    /* Don't care console access */
    if (SYS_ACCESS_CLI_CONSOLE == access_flag)
        return 0;

    if ((dir = opendir("/proc")) != NULL)
    {
        while ((entry = readdir(dir)) != NULL)
        {
            if (entry->d_name[0] == '.')
            {
                if ((0 == entry->d_name[1] || (entry->d_name[1] == '.' && 0 == entry->d_name[2])))
                    continue;
            }

            if (!(entry->d_name[0] >= '0' && entry->d_name[0] <= '9'))
                continue;

            pid = SYS_STR2UINT(entry->d_name);

            SYS_MEM_CLEAR(fullname);
            osal_snprintf(fullname, sizeof(fullname), "/proc/%d/cmdline", pid);

            fp = NULL;
            if (NULL == (fp = fopen(fullname, "r")))
                continue;

            SYS_MEM_CLEAR(cmd);
            readSize = fread(cmd, 1, sizeof(cmd) - 1, fp);
            fclose(fp);

            //SYS_PRINTF("cmdline \"%s\"\n", cmd);

            if (NULL != osal_strstr(cmd, "cli"))
            {
                /* replace \0 with space */
                for (i = 0; i < readSize && (i < sizeof(cmd) - 1); i++)
                    if ('\0' == cmd[i])
                        cmd[i] = ' ';

                switch (access_flag)
                {
                    case SYS_ACCESS_CLI_CONSOLE:
                        if (NULL == osal_strstr(cmd, "-a"))
                            sessionNum++;
                        break;
                    case SYS_ACCESS_CLI_TELNET:
                        if (NULL != osal_strstr(cmd, "-a telnet"))
                            sessionNum++;
                        break;
#ifdef CONFIG_SYS_APP_SSH
                    case SYS_ACCESS_CLI_SSH:
                        if (NULL != osal_strstr(cmd, "-a ssh"))
                            sessionNum++;
                        break;
#endif
                    default:
                        return 1;
                        break;
                }
            }
        }
        closedir(dir);
    }

    //SYS_PRINTF("cliNum %d\n", sessionNum);

    if (sessionNum > *maxSession)
        return 1;

    return 0;
}

/* RTK: Get parent process pid */
int processPpidGet(int pid)
{
    FILE *fp = NULL;
    char buf[32] = "";
    char tmp[128] = "";
    char subbuff[122] = ""; // length of tmp[128] - length of "PPid:\t" (6)
    int ppid = -1;
    osal_snprintf(buf, sizeof(buf), "/proc/%d/status", pid);
    if ((fp = fopen(buf, "r")) != NULL)
    {
        while (fgets(tmp, sizeof(tmp), fp))
        {
            if (0 == strncmp(tmp, "PPid:", 5))
            {
                memcpy(subbuff, &tmp[6], (121));
                subbuff[121] = '\0';
                ppid = (int)strtol(subbuff, (char **)NULL, 10);
                break;
            }
            memset(tmp, 0, sizeof(tmp));
        }
        fclose(fp);
    }
    return ppid;
}

#ifdef CONFIG_SYS_UI_MINICLI
int32 changePasswordProcedure(void)
{
    int32 ret = SYS_ERR_FAILED;
    char new_password[CAPA_SYS_PASSWORD_LEN + 1];
    char confirm_password[CAPA_SYS_PASSWORD_LEN + 1];
    sys_userInfo_t userInfo;

    SYS_MEM_CLEAR(new_password);
    SYS_MEM_CLEAR(confirm_password);
    SYS_MEM_CLEAR(userInfo);

    SYS_PRINTF("Enter new password  :");
    _vtysh_main_input_str_with_timeout(new_password, CAPA_SYS_PASSWORD_LEN, '*');
    SYS_PRINTF("Confirm new password:");
    _vtysh_main_input_str_with_timeout(confirm_password, CAPA_SYS_PASSWORD_LEN, '*');

    /* Check new password with confirm password  */
    if (0 != osal_strcmp(new_password, confirm_password))
    {
        SYS_PRINTF("New password confirmation failed..\n");
        return ret;
    }

    osal_strlcpy(userInfo.username, DFLT_SYS_ADMIN_USERNAME, sizeof(userInfo.username));
    osal_strlcpy(userInfo.password, new_password, sizeof(userInfo.password));
    userInfo.priv = SYS_LOCALUSER_PRIV_ADMIN;

    ret = sal_sys_localUser_add(userInfo);

    if (SYS_ERR_OK != ret)
    {
        if (SYS_ERR_STRING == ret)
            SYS_PRINTF("The password length should be %d to %d, and must have uppercase, lowercase and number.\n",
            CAPA_SYS_PASSWORD_MIN, CAPA_SYS_PASSWORD_MAX);
        else
            SYS_PRINTF("%s\n", SYS_ERR_STR(ret));
        return ret;
    }

    return SYS_ERR_OK;

}

#endif
#ifdef CONFIG_SYS_UI_MINICLI
void firstLoginProcedure(void)
{
    while (1)
    {
        SYS_PRINTF("Please change your password from the default settings. Please change the password for better protection of your network.\n");
        if (SYS_ERR_OK == changePasswordProcedure())
            break;
    }

    return;
}
#endif

void cli_exit(void)
{
    if (access_flag == SYS_ACCESS_CLI_CONSOLE)
    {
        int pppid = processPpidGet(getppid());

        unlink(SYS_CONSOLE_CLI_PID_FILE);

        SYS_LOG(LOG_CAT_AAA, LOG_AAA_DISCONNECT,
                text_access_type[access_flag].text,
                currUserName,
                "async");

        /* pid   : cli
           ppid  : /bin/sh
           pppid : init
         */
        if (1 == pppid)
        {
            kill(getppid(), SIGKILL);
        }
        /* pid   : /bin/cli
           ppid  : sh
           pppid : /bin/sh
         */
        else
        {
            kill(pppid, SIGKILL);
        }
    }

    kill(getpid(), SIGTERM);
}

/* VTY shell main routine. */
int main(int argc, char **argv, char **env)
{
    char *p;
    int opt;
    int dryrun = 0;
    //int boot_flag = 0;
    //const char *daemon_name = NULL;
    struct cmd_rec
    {
        const char *line;
        struct cmd_rec *next;
    } *cmd = NULL;
    struct cmd_rec *tail = NULL;
    int echo_command = 0;
    int no_error = 0;
    unsigned int count = 0;
#ifdef CONFIG_SYS_AAA_LOCALCMD_PRIVCHG
    uint32 genLocalPrivCmds = 0;
#endif /* CONFIG_SYS_AAA_LOCALCMD_PRIVCHG */
    char localIp[CAPA_IPV6_STR_LEN];
    uint32 maxSession = 0;

    SYS_MEM_CLEAR(localIp);

  /* Preserve name of myself. */
  progname = ((p = strrchr (argv[0], '/')) ? ++p : argv[0]);

    /* if logging open now */
    if ((p = getenv("VTYSH_LOG")) != NULL)
        logfile = fopen(p, "a");

    /* Option handling. */
    while (1)
    {
#ifdef CONFIG_SYS_AAA_LOCALCMD_PRIVCHG
        opt = getopt_long(argc, argv, "be:c:d:a:u:H:L:nEhCg", longopts, 0);
#else /* CONFIG_SYS_AAA_LOCALCMD_PRIVCHG */
        opt = getopt_long(argc, argv, "be:c:d:a:u:H:L:nEhC", longopts, 0);
#endif

        if (opt == EOF)
            break;

        switch (opt)
        {
        case 0:
            break;
        case 'b':
            //boot_flag = 1;
            break;
        case 'e':
        case 'c':
        {
            struct cmd_rec *cr;
            cr = XMALLOC(0, sizeof(*cr));
            cr->line = optarg;
            cr->next = NULL;
            if (tail)
                tail->next = cr;
            else
                cmd = cr;
            tail = cr;
        }
        break;
        case 'd':
            //daemon_name = optarg;
            break;
        case 'a':
            if (strstr(optarg, "telnet") != 0)
            {
                /* RTK: Record cli pid for telnetd to kill this process before session close */
                char buf[32] = "";
                FILE *fp = NULL;
                int ppid = -1;

                /* Get pid of process "sh -c /bin/cli -a telnet" */
                //ppid = processPpidGet(getpid());
                /* Get pid of process "sh -a telnet" */
                //ppid = processPpidGet(ppid);
                /* Get pid of process "/bin/telnetd" */
                ppid = processPpidGet(getpid());
                /*Write current cli pid to file with telnet pid as file name*/
                osal_snprintf(buf, sizeof(buf), "/tmp/telnet_%d_cli_pid", ppid);
                if ((fp = fopen(buf, "w")) != NULL)
                {
                    fprintf(fp, "%d", getpid());
                    fclose(fp);
                }

                access_flag = SYS_ACCESS_CLI_TELNET;
            }
#ifdef CONFIG_USER_SSH_SSHD
            else if (strstr(optarg, "ssh") != 0)
                access_flag = SYS_ACCESS_CLI_SSH;
#endif
            break;
        /* RTK: SSH login CLI directly (Start)*/
        case 'u':
            memset(ssh_username, 0, sizeof(ssh_username));
            /* Remove [] */
            strncpy(ssh_username, (optarg + 1), strlen(optarg) - 2);
            break;
            /* RTK: SSH login CLI directly (End)*/
        case 'H':
            SYS_MEM_CLEAR(currHost);
            strlcpy(currHost, optarg, sizeof(currHost));
            break;
        case 'L':
            SYS_MEM_CLEAR(localIp);
            strlcpy(localIp, optarg, sizeof(localIp));
            break;
#ifdef CONFIG_SYS_AAA_LOCALCMD_PRIVCHG
        case 'g':
            genLocalPrivCmds = 1;
            break;
#endif /* CONFIG_SYS_AAA_LOCALCMD_PRIVCHG */
        case 'n':
            no_error = 1;
            break;
        case 'E':
            echo_command = 1;
            break;
        case 'C':
            dryrun = 1;
            break;
        case 'h':
            usage(0);
            break;
        default:
            usage(1);
            break;
        }
    }

  if (isOverMaxConn(&maxSession))
  {
      int _i;

      SYS_LOG(LOG_CAT_SYSTEM, LOG_SYSTEM_CLI_SESSION_FULL, maxSession);

      SYSTEM_CMD_EXECUTE("clear");
      printf("\n\nOver maximum cli session !!! Do not allow new session !!!\n\n");

      for(_i = 3; _i > 0; _i--)
      {
          printf("Close after %d seconds\n", _i);
          sleep(1);
          printf("\033[A\033[2K");
      }

      if (SYS_ACCESS_CLI_TELNET == access_flag)
      {
          char filename[SYS_BUF64_LEN + 1];
          SYS_SNPRINTF_SAFE(filename, sizeof(filename), "/tmp/telnet_%d_cli_pid", processPpidGet(getpid()));
          SYS_ERR_CHK(sal_util_file_delete(filename, NULL));
      }

      exit(0);
  }

  /* Initialize user input buffer. */
  line_read = NULL;
  setlinebuf(stdout);

#ifdef CONFIG_SYS_BUILD_RELEASE
#ifdef CONFIG_SYS_AAA_LOCALCMD_PRIVCHG
    if (0 == genLocalPrivCmds)
#endif
        SYSTEM_CMD("clear");
#endif
    /* Signal and others. */
    vtysh_signal_init();

    /* Make vty structure and register commands. */
    vtysh_init_vty();

    vty_init_vtysh();

    /* Install all commands here */
    cli_cmd_init();

    /* Initialize terminal parameters */
    _terminal_initial();

#ifdef CONFIG_SYS_AAA_LOCALCMD_PRIVCHG
    if (genLocalPrivCmds)
    {
        genLocalPrivCmdsFile();
        exit(0);
    }
    {
        FILE *tmpfp;

        if ((tmpfp = fopen(SYS_AAA_CMDPRIV_EXECCONF_TMPFILE, "r")) != NULL)
        {
            fclose(tmpfp);
            if ((tmpfp = fopen(CONF_STARTUP_FILE, "r")) != NULL)
            {
                int oldNode;
                char cmd[MAX_CMD_STRLEN];

                oldNode = vty->node;
                vty->node = CONFIG_NODE;

                while (fgets(cmd, sizeof(cmd), tmpfp))
                {
                    if (0 == strncmp(cmd, "privilege config", strlen("privilege config")) || 0 == strncmp(cmd, "privilege exec", strlen("privilege exec")))
                    {
                        //XMORE("%s", cmd);
                        vtysh_execute(cmd);
                    }
                }
                fclose(tmpfp);

                vty->node = oldNode;
            }
            unlink(SYS_AAA_CMDPRIV_EXECCONF_TMPFILE);
        }
    }
#endif /* CONFIG_SYS_AAA_LOCALCMD_PRIVCHG */

    /* All command installation is done */
    //sort_node ();

    /* Start execution only if not in dry-run mode */
    if (dryrun)
        return (0);

    /* Ignore error messages */
    if (no_error)
        freopen("/dev/null", "w", stdout);

    while (1)
    {
        int32 ret;
        vty_hello(vty);
        if (access_flag == SYS_ACCESS_CLI_CONSOLE)
        {
            /* Press any key to continue */
            vty_out(vty, "%s%s", "Press any key to continue", VTY_NEWLINE);
            vty_getch();
        }
#ifdef CONFIG_USER_SSH_SSHD
        if (access_flag == SYS_ACCESS_CLI_SSH)
        {
            sys_userInfo_t userInfo;

            SYS_MEM_CLEAR(userInfo);
            SYS_ERR_CHK(sal_sys_localUser_getByName(ssh_username, &userInfo));
            currUserPriv = userInfo.priv;
            osal_strlcpy(currUserName, ssh_username, sizeof(currUserName));
#ifdef CONFIG_SYS_UI_MINICLI
            if (IS_PWD_UNCHANGED())
            {
                firstLoginProcedure();
            }
#endif
            break;
        }
        else
#endif
        {
            ret = vtysh_login_authen();
        }
        if (VTYSH_LOGIN_SUCCESS != ret)
        {
#ifdef CONFIG_SYS_UI_CLI_PASSRETRY_SILENT
            _vtysh_main_silent_time();
            vty_getch();
#else /* CONFIG_SYS_UI_CLI_PASSRETRY_SILENT */
            if (SYS_ACCESS_CLI_CONSOLE != access_flag)
            {
                cli_exit();
            }
#endif
        }
        else // authen success
        {
#ifdef CONFIG_SYS_UI_MINICLI
            if (IS_PWD_UNCHANGED())
            {
                firstLoginProcedure();
            }
#endif
            break;
        }
    }
#ifdef CONFIG_SYS_AAA
    currpriv = currUserPriv;
#endif /* CONFIG_SYS_AAA */

    /* If privilege equal to admin, enter privilege mode directly */
    if (currpriv == SYS_LOCALUSER_PRIV_ADMIN)
    {
        vty->node = ENABLE_NODE;
        current_node = ENABLE_NODE;
    }

#if 0
  /* Make sure we pass authentication before proceeding. */
  vtysh_auth ();

  /* Do not connect until we have passed authentication. */
  if (vtysh_connect_all (daemon_name) <= 0)
    {
      fprintf(stderr, "Exiting: failed to connect to any daemons.\n");
      exit(1);
    }
#endif
    /* Create line user file and  */
    {
        sys_line_session_t session;

        SYS_MEM_CLEAR(session);

        session.pid = getpid();
        strlcpy(session.user, currUserName, sizeof(session.user));
        strlcpy(session.host, currHost, sizeof(session.host));
        strlcpy(session.localIp, localIp, sizeof(session.localIp));
        sal_sys_lineSession_set(access_flag, &session);
    }
    /* If eval mode. */
    if (cmd)
    {
        /* Enter into enable node. */
        //vtysh_execute ("enable");

        while (cmd != NULL)
        {
            int ret;
            char *eol;

            while ((eol = strchr(cmd->line, '\n')) != NULL)
            {
                *eol = '\0';

                if (echo_command)
                    printf("%s%s\n", vtysh_prompt(), cmd->line);

                if (logfile)
                    log_it(cmd->line);

                ret = vtysh_execute_no_pager(cmd->line);
                if (!no_error &&
                    !(ret == CMD_SUCCESS ||
                      ret == CMD_SUCCESS_DAEMON ||
                      ret == CMD_WARNING))
                    exit(1);

                cmd->line = eol + 1;
            }

            if (echo_command)
                printf("%s%s\n", vtysh_prompt(), cmd->line);

            if (logfile)
                log_it(cmd->line);

            ret = vtysh_execute_no_pager(cmd->line);
            if (!no_error &&
                !(ret == CMD_SUCCESS ||
                  ret == CMD_SUCCESS_DAEMON ||
                  ret == CMD_WARNING))
                exit(1);

            {
                struct cmd_rec *cr;
                cr = cmd;
                cmd = cmd->next;
                XFREE(0, cr);
            }
        }
        exit(0);
    }

    vtysh_pager_init();
    vtysh_init_timeout(VTY_TIMEOUT_DEFAULT);

    vtysh_readline_init();

    //vty_hello (vty);

    /* Enter into enable node. */
    //vtysh_execute ("enable");

#if 0
  /* Main command loop. */
  while (vtysh_rl_gets ())
    vtysh_execute (line_read);
#endif

    snprintf(history_file, sizeof(history_file), "%s/.history_quagga", getenv("HOME"));
    read_history(history_file);

    {
        sys_line_cliType_t type = SYS_LINE_CLI_CONSOLE;

        if (access_flag == SYS_ACCESS_CLI_CONSOLE)
            type = SYS_LINE_CLI_CONSOLE;
        else if (access_flag == SYS_ACCESS_CLI_TELNET)
            type = SYS_LINE_CLI_TELNET;
#ifdef CONFIG_SYS_APP_SSH
        else if (access_flag == SYS_ACCESS_CLI_SSH)
            type = SYS_LINE_CLI_SSH;
#endif

        if (SYS_ERR_OK != sal_sys_cliHistory_count_get(type, &count))
            count = 0;
        stifle_history(count);
    }

    /* Send log to notify user logn */
    SYS_LOG(LOG_CAT_AAA, LOG_AAA_CONNECT, text_access_type[access_flag].text, currUserName, currHost);

    /* If cli is called by console, create pid file for polld to check */
    if (access_flag == SYS_ACCESS_CLI_CONSOLE)
    {
        FILE *fp = NULL;

        if (NULL != (fp = fopen(SYS_CONSOLE_CLI_PID_FILE, "w")))
        {
            fprintf(fp, "%d", getpid());

            fclose(fp);
        }
    }

    /* Preparation for longjmp() in sigtstp(). */
    sigsetjmp(jmpbuf, 1);
    jmpflag = 1;

    vtysh_cmd_loop();

    history_truncate_file(history_file, 1000);
    printf("\n");

    /* Rest in peace. */
    exit(0);
}
