/* Virtual terminal interface shell.
 * Copyright (C) 2000 Kunihiro Ishiguro
 *
 * This file is part of GNU Zebra.
 *
 * GNU Zebra is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * GNU Zebra is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Zebra; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <libvty/zebra.h>

#include <sys/un.h>
#include <setjmp.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/stat.h>

#include <readline/readline.h>
#include <readline/history.h>

#include <libvty/command.h>
#include <libvty/memory.h>
#include <libfds/fds_export.h>
#include <libsal/sal_util.h>
#include <libcmd/cmd.h>

#include <common/sys_error.h>
#include <common/sys_def.h>
#include <common/sys_type.h>
#include <common/sys_util.h>

#include "vtysh.h"
#include "log.h"


#define CONFIGFILE_MASK 0600

/* Struct VTY. */
struct vty *vty;

/* VTY shell pager name. */
char *vtysh_pager_name = NULL;

/* Defined in lib/vty.c */
extern struct cmd_node vty_node;

/* When '^Z' is received from vty, move down to the enable mode. */
int
vtysh_end (void)
{
  switch (vty->node)
    {
    case VIEW_NODE:
    case ENABLE_NODE:
      /* Nothing to do. */
      break;
    default:
      vty->node = ENABLE_NODE;
      break;
    }
  return CMD_SUCCESS;
}

void
vtysh_pager_init (void)
{
  char *pager_defined;

  pager_defined = getenv ("VTYSH_PAGER");

  if (pager_defined)
    vtysh_pager_name = strdup (pager_defined);
  else
    vtysh_pager_name = strdup ("more");
}

/* Command execution over the vty interface. */
static int
vtysh_execute_func (const char *line, int pager)
{
  int ret, cmd_stat;
#if 0
  u_int i;
#endif
  vector vline;
  struct cmd_element *cmd;
  FILE *fp = NULL;
  int closepager = 0;
  //int tried = 0;
  //int saved_ret;
  //int saved_node;

  /* Split readline string up into the vector. */
  vline = cmd_make_strvec (line, vty->node);

  if (vline == NULL)
    return CMD_SUCCESS;

  ret = cmd_execute_command (vline, vty, &cmd, 1, line);
  //saved_ret = ret;
  //saved_node = vty->node;

#if 0
  /* If command doesn't succeeded in current node, try to walk up in node tree.
   * Changing vty->node is enough to try it just out without actual walkup in
   * the vtysh. */
  while (ret != CMD_SUCCESS && ret != CMD_SUCCESS_DAEMON && ret != CMD_WARNING
     && vty->node > CONFIG_NODE)
    {
      vty->node = node_parent(vty->node);
      ret = cmd_execute_command (vline, vty, &cmd, 1, line);
      tried++;
    }

  vty->node = saved_node;

  /* If command succeeded in any other node than current (tried > 0) we have
   * to move into node in the vtysh where it succeeded. */
  if (ret == CMD_SUCCESS || ret == CMD_SUCCESS_DAEMON || ret == CMD_WARNING)
    {
      if ((saved_node == BGP_VPNV4_NODE || saved_node == BGP_IPV4_NODE
       || saved_node == BGP_IPV6_NODE || saved_node == BGP_IPV4M_NODE
       || saved_node == BGP_IPV6M_NODE)
      && (tried == 1))
    {
      vtysh_execute("exit-address-family");
    }
      else if ((saved_node == KEYCHAIN_KEY_NODE) && (tried == 1))
    {
      vtysh_execute("exit");
    }
      else if (tried)
    {
      vtysh_execute ("end");
      vtysh_execute (VTYSH_CONFIG_MODE_ENTER);
    }
    }
  /* If command didn't succeed in any node, continue with return value from
   * first try. */
  else if (tried)
    {
      ret = saved_ret;
    }
#endif

  cmd_free_strvec (vline);

  cmd_stat = ret;
  switch (ret)
    {
    /* RTK: sync error string handling */
    case CMD_WARNING:
      if (vty->type == VTY_FILE)
          //fprintf (stdout,"Warning...\n");
      break;
    case CMD_ERR_AMBIGUOUS:
      //fprintf (stdout,"%% Ambiguous command.\n");
      break;
    case CMD_ERR_NO_MATCH:
      //fprintf (stdout,"%% Unknown command.\n");
      break;
    case CMD_ERR_INCOMPLETE:
      //fprintf (stdout,"%% Command incomplete.\n");
      break;
    case CMD_ERR_SLAVE_PORT:
      //fprintf (stdout,"%% Trunk slave port can not be configured.\n");
      break;
    case CMD_ERR_TRUNK_MBR:
      //fprintf (stdout,"%% Trunk member port can not be configured.\n");
      break;
    case CMD_ERR_PORTID:
      //fprintf(stdout, "Valid port range should be %d~%d\n", USER_PORT_START, USER_PORT_END(SYS_DEV_ID_DEFAULT));
      break;
    case CMD_SUCCESS_DAEMON:
      {
    /* FIXME: Don't open pager for exit commands. popen() causes problems
     * if exited from vtysh at all. This hack shouldn't cause any problem
     * but is really ugly. */
    if (pager && vtysh_pager_name && (strncmp(line, "exit", 4) != 0))
      {
        fp = popen (vtysh_pager_name, "w");
        if (fp == NULL)
          {
        perror ("popen failed for pager");
        fp = stdout;
          }
        else
          closepager=1;
      }
    else
      fp = stdout;

    if (! strcmp(cmd->string, VTYSH_CONFIG_MODE_ENTER))
    {
#if 0
        for (i = 0; i < VTYSH_INDEX_MAX; i++)
          {
            cmd_stat = vtysh_client_execute(&vtysh_client[i], line, fp);
        if (cmd_stat == CMD_WARNING)
          break;
          }

        if (cmd_stat)
        {
#endif
            line = "end";
            vline = cmd_make_strvec (line, vty->node);

            if (vline == NULL)
            {
                if (pager && vtysh_pager_name && fp && closepager)
                {
                    if (pclose (fp) == -1)
                    {
                        perror ("pclose failed for pager");
                    }
                    fp = NULL;
                }
                return CMD_SUCCESS;
            }

            ret = cmd_execute_command (vline, vty, &cmd, 1, line);
            cmd_free_strvec (vline);
            if (ret != CMD_SUCCESS_DAEMON)
            break;
#if 0
        }
        else
        {
            if (cmd->func)
            {
                (*cmd->func) (cmd, vty, 0, NULL);
                break;
            }
        }
#endif
    }

    cmd_stat = CMD_SUCCESS;
#if 0
    for (i = 0; i < VTYSH_INDEX_MAX; i++)
      {
        if (cmd->daemon & vtysh_client[i].flag)
          {
            cmd_stat = vtysh_client_execute(&vtysh_client[i], line, fp);
        if (cmd_stat != CMD_SUCCESS)
          break;
          }
      }

    if (cmd_stat != CMD_SUCCESS)
      break;
#endif

    if (cmd->func)
      (*cmd->func) (cmd, vty, 0, NULL);
      }
    }
  if (pager && vtysh_pager_name && fp && closepager)
    {
      if (pclose (fp) == -1)
    {
      perror ("pclose failed for pager");
    }
      fp = NULL;
    }
  return cmd_stat;
}

int
vtysh_execute_no_pager (const char *line)
{
  return vtysh_execute_func (line, 0);
}

int
vtysh_execute (const char *line)
{
  char *cmd = NULL;
  int ret = 0;

  cmd = (char*)line;

  /* RTK: Strip ' ' char from head of command.*/
  while (*cmd == ' ')
  {
    cmd++;
  }

  ret = vtysh_execute_func (cmd, 1);

  return ret;
}

/* We don't care about the point of the cursor when '?' is typed. */
#define CLI_CMD_DESCRIPTION_SORT 1
#define CLI_DESC_WIDTH 80
#define CLI_DESC_NEWLINE_KEY '|'
int
vtysh_rl_describe (int count, int ch)
{
  int ret;
  unsigned int i;
  vector vline = NULL;
  vector describe;
  int width;
  struct desc *desc;
  enum node_type old_node;
  sys_enable_t enable = DISABLED;

  cmd_textline_enable_get(&enable);
  if (ENABLED == enable)
  {
      fprintf(stdout, "%c", ch);
      return 0;
  }

  old_node = vty->node;
  if (cmd_try_do_shortcut(vty->node, rl_line_buffer))
  {
      vty->node = ENABLE_NODE;
      vline = cmd_make_strvec (rl_line_buffer + 3, vty->node);
  }
  else
  {
      vline = cmd_make_strvec (rl_line_buffer, vty->node);
  }

  /* In case of '> ?'. */
  if (vline == NULL)
    {
      vline = vector_init (1);
      vector_set (vline, '\0');
    }
  else
    if (rl_end && isspace ((int) rl_line_buffer[rl_end - 1]))
      vector_set (vline, '\0');

  describe = cmd_describe_command (vline, vty, &ret);

  vty->node = old_node;

  fprintf (stdout,"\n");

  /* Ambiguous and no match error. */
#if 0
  switch (ret)
    {
    case CMD_ERR_AMBIGUOUS:
      cmd_free_strvec (vline);
      fprintf (stdout,"Ambiguous command\n");
      rl_on_new_line ();
      return 0;
      break;
    case CMD_ERR_NO_MATCH:
      cmd_free_strvec (vline);
      fprintf (stdout,"There is no matched command\n");
      rl_on_new_line ();
      return 0;
      break;
    case CMD_ERR_SLAVE_PORT:
      cmd_free_strvec (vline);
      fprintf (stdout,"Trunk slave port can not be configured\n");
      rl_on_new_line ();
      return 0;
      break;
    case CMD_ERR_TRUNK_MBR:
      cmd_free_strvec (vline);
      fprintf (stdout,"Trunk member port can not be configured\n");
      rl_on_new_line ();
      return 0;
      break;
    case CMD_ERR_PORTID:
      cmd_free_strvec (vline);
      fprintf(stdout, "Valid port range should be %d~%d\n", USER_PORT_START, USER_PORT_END(SYS_DEV_ID_DEFAULT));
      fprintf (stdout,"Invalid port id\n");
      rl_on_new_line ();
      return 0;
    case CMD_ERR_PORTLIST:
      cmd_free_strvec (vline);
      fprintf (stdout,"Unkonwn port list\n");
      rl_on_new_line ();
      return 0;
      break;
    case CMD_ERR_VALUE:
      cmd_free_strvec (vline);
      fprintf (stdout,"%s\n", SYS_ERR_STR(CMD_ERR_VALUE));
      rl_on_new_line ();
      return 0;
      break;
    }
#else
    if (ret != SYS_ERR_OK)
    {
        cmd_free_strvec (vline);
        fprintf (stdout,"%s\n", SYS_ERR_STR(ret));
        rl_on_new_line ();
        return 0;
    }
#endif

  /* Get width of command string. */
  width = 0;
  for (i = 0; i < vector_active (describe); i++)
    if ((desc = vector_slot (describe, i)) != NULL)
      {
    int len;

    if (desc->cmd[0] == '\0')
      continue;

    len = strlen (desc->cmd);
    if (desc->cmd[0] == '.')
      len--;

    if (width < len)
      width = len;
      }

#if CLI_CMD_DESCRIPTION_SORT
{
    int32 isIntfFaPrt = FALSE;
    int32 isIntfGiPrt = FALSE;
    int32 isIntfTePrt = FALSE;
    int32 isIntfPoPrt = FALSE;
#ifdef CONFIG_SYS_L3_ROUTE
    int32 isIntfVlanPrt = FALSE;
#endif
    typedef struct cli_descp_sort_s
    {
        char *cmd;
        char *str;
        struct cli_descp_sort_s *next;
    } cli_descp_sort_t;

    if (FALSE == vty->descpSortState)
    {
        for (i = 0; i < vector_active (describe); i++)
        {
            if ((desc = vector_slot (describe, i)) != NULL)
            {
                if (desc->cmd[0] == '\0')
                    continue;

                if (! desc->str)
                    fprintf (stdout,"  %-s\n", desc->cmd[0] == '.' ? desc->cmd + 1 : desc->cmd);
                else
                    fprintf (stdout,"  %-*s  %s\n",
                            width,
                            desc->cmd[0] == '.' ? desc->cmd + 1 : desc->cmd,
                            desc->str);
            }
        }
    }
    else
    {
        cli_descp_sort_t *glb_descp = NULL;
        cli_descp_sort_t *sort_descp = NULL;
        cli_descp_sort_t *sort_prev_descp = NULL;
        for (i = 0; i < vector_active(describe); i++)
        {
            /* Create sorting structure for each cmd description object */
            if ((desc = vector_slot(describe, i )) != NULL)
            {
                cli_descp_sort_t *curr_descp = NULL;
                uint32 cmdLen = 0;
                uint32 strLen = 0;

                if (desc->cmd[0] == '\0')
                    continue;
                curr_descp = malloc(sizeof(cli_descp_sort_t));
                memset(curr_descp, 0, sizeof(cli_descp_sort_t));

                if (!desc->str)
                {
                    cmdLen = strlen(desc->cmd);
                    if (desc->cmd[0] == '.')
                    {
                        curr_descp->cmd = malloc(cmdLen +  1);
                        memset(curr_descp->cmd, 0, cmdLen);
                        strncpy(curr_descp->cmd, desc->cmd+1, cmdLen - 1);
                    }
                    else
                    {
                        curr_descp->cmd = malloc(cmdLen + 1);
                        memset(curr_descp->cmd, 0, cmdLen + 1);
                        strncpy(curr_descp->cmd, desc->cmd, cmdLen);
                    }
                }
                else
                {
                    cmdLen = strlen(desc->cmd);
                    strLen = strlen(desc->str);
                    if (desc->cmd[0] == '.')
                    {
                        curr_descp->cmd = malloc(cmdLen + 1);
                        memset(curr_descp->cmd, 0, cmdLen);
                        strncpy(curr_descp->cmd, desc->cmd+1, cmdLen - 1);

                    }
                    else
                    {
                        curr_descp->cmd = malloc(cmdLen + 1);
                        memset(curr_descp->cmd, 0, cmdLen + 1);
                        strncpy(curr_descp->cmd, desc->cmd, cmdLen);
                    }
                    curr_descp->str = malloc(strLen + 1);
                    memset(curr_descp->str, 0, strLen + 1);
                    strncpy(curr_descp->str, desc->str, strLen);
                }

                if (NULL == glb_descp)
                {
                    glb_descp = curr_descp;
                    glb_descp->next = NULL;
                }
                else
                {

                    uint32 i;
                    uint32 isMatchFound = FALSE;
                    cli_descp_sort_t *match_descp = NULL;
                    /* insert current description object to correct position */
                    sort_descp = glb_descp;
                    sort_prev_descp = NULL;
                    while(sort_descp != NULL)
                    {
                        for(i = 0; i < strlen(curr_descp->cmd); i++)
                        {
                            if (curr_descp->cmd[i] == sort_descp->cmd[i])
                                continue;

                            match_descp = sort_descp;
                            /* order is small letter -> large letter */
                            if (curr_descp->cmd[i] < sort_descp->cmd[i])
                            {
                                isMatchFound = TRUE;
                                match_descp = sort_prev_descp;
                            }
                            break;
                        }
                        if (TRUE == isMatchFound)
                            break;
                        sort_prev_descp = sort_descp;
                        sort_descp = sort_descp->next;
                    }

                    if (NULL == match_descp)
                    {
                        curr_descp->next = glb_descp;
                        glb_descp = curr_descp;
                    }
                    else
                    {
                        curr_descp->next = match_descp->next;
                        match_descp->next = curr_descp;
                    }
                }
            }
        }

        /* Print cmd description to terminal and free sorting structures */
        sort_descp = glb_descp;
        while(sort_descp != NULL)
        {
            cli_descp_sort_t *tmp_descp = NULL;

            tmp_descp = sort_descp;

            if (!sort_descp->str)
            {
                fprintf(stdout, "  %-s\n", sort_descp->cmd);
            }
            else
            {
                sys_logic_portmask_t portmask;
                char portmaskStr[CAPA_PORTMASK_STR_LEN];

                osal_memset(portmaskStr, 0, sizeof(portmaskStr));
                osal_memset(&portmask, 0, sizeof(sys_logic_portmask_t));

                /* print port list description with current port list string*/
                if (strstr(sort_descp->cmd, "NML-PORT-LIST"))
                {
                    NORMAL_LOGIC_PORTMASK_SET_ALL(portmask);
                    LPM2STR(&portmask, portmaskStr);

                    fprintf(stdout, "  %-*s  %s\n", width, sort_descp->cmd, portmaskStr);
                }
                else if (strstr(sort_descp->cmd, "NML-WOTRKMBR-PORT-LIST"))
                {
                    sys_logic_port_t port;

                    FOR_EACH_NORMAL_LOGIC_PORT(port)
                    {
                        if (!IS_LP_TRKMBR(port))
                        {
                            LOGIC_PORTMASK_SET_PORT(portmask, port);
                        }
                    }
                    LPM2STR(&portmask, portmaskStr);

                    fprintf(stdout, "  %-*s  %s\n", width, sort_descp->cmd, portmaskStr);
                }
                else if (strstr(sort_descp->cmd, "PORT-LIST"))
                {
                    sys_logic_portmask_t portmask;
                    char portmaskStr[CAPA_PORTMASK_STR_LEN];

                    osal_memset(portmaskStr, 0, sizeof(portmaskStr));

                    LOGIC_PORTMASK_SET_ALL(portmask);

                    sys_port_lPortmask2uStr(&portmask, SYS_PORT_STRFMT_BRIEF, FALSE, portmaskStr);

                    fprintf(stdout, "  %-*s  %s\n", width, sort_descp->cmd, portmaskStr);
                }
                else if (CMD_IF_FA_PORTS(sort_descp->cmd) || CMD_IF_FA_PORT(sort_descp->cmd))
                {
                    if (FALSE == isIntfFaPrt)
                    {
                        isIntfFaPrt = TRUE;
                        fprintf(stdout, "  %-*s  %s\n", width, IF_FA_STR, sort_descp->str);
                    }
                }
                else if (CMD_IF_GI_PORTS(sort_descp->cmd) || CMD_IF_GI_PORT(sort_descp->cmd))
                {
                    if (FALSE == isIntfGiPrt)
                    {
                        isIntfGiPrt = TRUE;
                        fprintf(stdout, "  %-*s  %s\n", width, IF_GI_STR, sort_descp->str);
                    }
                }
                else if (CMD_IF_TE_PORTS(sort_descp->cmd) || CMD_IF_TE_PORT(sort_descp->cmd))
                {
                    if (FALSE == isIntfTePrt)
                    {
                        isIntfTePrt = TRUE;
                        fprintf(stdout, "  %-*s  %s\n", width, IF_TE_STR, sort_descp->str);
                    }
                }
                else if (CMD_IF_PO_PORTS(sort_descp->cmd) || CMD_IF_PO_PORT(sort_descp->cmd))
                {
                    if (FALSE == isIntfPoPrt)
                    {
                        isIntfPoPrt = TRUE;
                        fprintf(stdout, "  %-*s  %s\n", width, IF_PO_STR, sort_descp->str);
                    }
                }
#ifdef CONFIG_SYS_L3_ROUTE
                else if (CMD_IF_VLANS(sort_descp->cmd) || CMD_IF_VLAN(sort_descp->cmd))
                {
                    if (FALSE == isIntfVlanPrt)
                    {
                        isIntfVlanPrt = TRUE;
                        fprintf(stdout, "  %-*s  %s\n", width, IF_VLAN_STR, sort_descp->str);
                    }
                }
                else if (CMD_IF_LOS(sort_descp->cmd) || CMD_IF_LO(sort_descp->cmd))
                {
                    fprintf(stdout, "  %-*s  %s\n", width, IF_LO_STR, sort_descp->str);
                }
#endif
                /* print other descriptions */
                else
                {
                    uint32 isFirstLine = TRUE;
                    uint32 len = 0;
                    char *currStr = NULL;
                    char *newline = NULL;
                    char buf[CLI_DESC_WIDTH + 1];

                    currStr = sort_descp->str;

                    /* if string length is 0, it will print <cr> */
                    if (0 == osal_strlen(currStr))
                        fprintf(stdout, "  %-*s  %s\n", width, sort_descp->cmd, sort_descp->str);

                    while (0 != osal_strlen(currStr))
                    {
                        len = osal_strlen(currStr) + width + 4;
                        if (NULL != (newline = osal_strchr(currStr, CLI_DESC_NEWLINE_KEY)))
                        {
                            osal_memset(buf, 0, sizeof(buf));
                            osal_strncpy(buf, currStr, newline - currStr);
                            fprintf(stdout, "  %-*s  %s\n", width, (isFirstLine) ? sort_descp->cmd : " ", buf);
                            currStr = newline + 1;
                        }
                        else if (len > CLI_DESC_WIDTH)
                        {
                            osal_memset(buf, 0, sizeof(buf));
                            osal_strncpy(buf, currStr, CLI_DESC_WIDTH - width - 4);
                            fprintf(stdout, "  %-*s  %s\n", width, (isFirstLine) ? sort_descp->cmd : " ", buf);
                            currStr += CLI_DESC_WIDTH - width - 4;
                        }
                        else
                        {
                            fprintf(stdout, "  %-*s  %s\n", width, (isFirstLine) ? sort_descp->cmd : " ", currStr);
                            break;
                        }

                        if (isFirstLine)
                            isFirstLine = FALSE;
                    }
                }
            }

            if (sort_descp->cmd != NULL)
                free(sort_descp->cmd);
            if (sort_descp->str != NULL)
                free(sort_descp->str);

            sort_descp = sort_descp->next;
            free(tmp_descp);
        }
    }
}
#else
  for (i = 0; i < vector_active (describe); i++)
    if ((desc = vector_slot (describe, i)) != NULL)
      {
    if (desc->cmd[0] == '\0')
      continue;

    if (! desc->str)
      fprintf (stdout,"  %-s\n",
           desc->cmd[0] == '.' ? desc->cmd + 1 : desc->cmd);
    else
      fprintf (stdout,"  %-*s  %s\n",
           width,
           desc->cmd[0] == '.' ? desc->cmd + 1 : desc->cmd,
           desc->str);
      }
#endif

  cmd_free_strvec (vline);
  vector_free (describe);

  rl_on_new_line();

  return 0;
}

/* Result of cmd_complete_command() call will be stored here
 * and used in new_completion() in order to put the space in
 * correct places only. */
int complete_status;

static char *
command_generator (const char *text, int state)
{
  vector vline;
  static char **matched = NULL;
  static int index = 0;
  enum node_type old_node;

  /* First call. */
  if (! state)
    {
      index = 0;

//      if (vty->node == AUTH_NODE || vty->node == AUTH_ENABLE_NODE)
//    return NULL;

      old_node = vty->node;
      if (cmd_try_do_shortcut(vty->node, rl_line_buffer))
      {
          vty->node = ENABLE_NODE;
          vline = cmd_make_strvec (rl_line_buffer + 3, vty->node);
      }
      else
      {
          vline = cmd_make_strvec (rl_line_buffer, vty->node);
      }

      if (vline == NULL)
    return NULL;

      if (rl_end && isspace ((int) rl_line_buffer[rl_end - 1]))
    vector_set (vline, '\0');

      matched = cmd_complete_command (vline, vty, &complete_status);

      vty->node = old_node;
    }

  if (matched && matched[index])
    return matched[index++];

  return NULL;
}

static char **
new_completion (char *text, int start, int end)
{
  char **matches;
  sys_enable_t enable = DISABLED;

  cmd_textline_enable_get(&enable);
  if (ENABLED == enable)
      return NULL;

  matches = rl_completion_matches (text, command_generator);

  if (matches)
    {
      rl_point = rl_end;
      if (complete_status == CMD_COMPLETE_FULL_MATCH)
    rl_pending_input = ' ';
    }

  return matches;
}

DEFUN(
     vtysh_end_all,
     vtysh_end_all_cmd,
     "end",
     "End current mode and change to enable mode\n")
{
  return vtysh_end ();
}


DEFUN (
     vtysh_line_vty,
     vtysh_line_vty_cmd,
     "line vty",
     "Configure a terminal line\n"
     "Virtual terminal\n")
{
  vty->node = VTY_NODE;
  return CMD_SUCCESS;
}

#ifdef CONFIG_SYS_AAA
DEFUN (
     vtysh_enable,
     vtysh_enable_cmd,
     "enable [<1-15>]",
     "Turn on privileged mode command\n"
     "Privilege level\n")
{
    int32 ret = SYS_ERR_FAILED;

    if (argc > 0)
    {
        int32 priv = SYS_STR2UINT(argv[0]);
        if (priv < currpriv)
        {
            XMORE("Current privilege level %d is higher than %d\n", currpriv, priv);
            return SYS_ERR_OK;
        }
        else if (priv == currpriv)
        {
            XMORE("Equal to current privilege level %d\n", currpriv);
        }
        ret = enable_authen(priv);

        if (ret != SYS_ERR_OK)
        {
            XMORE("%% Authentication Failed\n\n");
            return SYS_ERR_OK;
        }

        currpriv = priv;
    }
    else
    {
        ret = enable_authen(15);

        if (ret != SYS_ERR_OK)
        {
            XMORE("%% Authentication Failed\n\n");
            return SYS_ERR_OK;
        }
        currpriv = 15;
    }

    vty->node = ENABLE_NODE;
    return SYS_ERR_OK;
}

DEFUN (
     vtysh_disable,
     vtysh_disable_cmd,
     "disable [<1-14>]",
     "Turn off privileged mode command\n"
     "Privilege level\n")
{
    if (argc > 0)
    {
        int32 priv = SYS_STR2UINT(argv[0]);
        if (priv > currpriv)
        {
            XMORE("Current privilege level %d is lower than %d\n", currpriv, priv);
            return SYS_ERR_OK;
        }
        currpriv = priv;
    }
    else
    {
        currpriv = 1;
    }

    if (vty->node == ENABLE_NODE)
        vty->node = VIEW_NODE;
  return CMD_SUCCESS;
}
#else
DEFUN (
     vtysh_enable,
     vtysh_enable_cmd,
     "enable",
     "Turn on privileged mode command\n")
{
    SYS_ERR_CHK(enable_authen(0));

  vty->node = ENABLE_NODE;
    return SYS_ERR_OK;
}

DEFUN (
     vtysh_disable,
     vtysh_disable_cmd,
     "disable",
     "Turn off privileged mode command\n")
{
    currpriv = currUserPriv;
  if (vty->node == ENABLE_NODE)
    vty->node = VIEW_NODE;
  return CMD_SUCCESS;
}

#endif /* CONFIG_SYS_AAA */


DEFUN (
     vtysh_config_terminal,
     vtysh_config_terminal_cmd,
     VTYSH_CONFIG_MODE_ENTER,
     "Configuration Mode\n")
{
  vty->node = CONFIG_NODE;
  return CMD_SUCCESS;
}



static int
vtysh_exit (struct vty *vty)
{
    switch (vty->node)
    {
        case VIEW_NODE:
            cli_exit();
            break;

        case ENABLE_NODE:
            currpriv = currUserPriv;
            vty->node = VIEW_NODE;
            break;

        case CONFIG_NODE:
            vty->node = ENABLE_NODE;
            break;

        case IF_NODE:
        case VTY_NODE:
        default:
            vty->node = CONFIG_NODE;
            break;
    }

    return CMD_SUCCESS;
}

DEFUN(
     vtysh_exit_all,
     vtysh_exit_all_cmd,
     "exit",
     "Exit current mode and down to previous mode\n")
{
  return vtysh_exit (vty);
}

#if 0
ALIAS (vtysh_exit_all,
       vtysh_quit_all_cmd,
       "quit",
       "Exit current mode and down to previous mode\n")

DEFUN (vtysh_terminal_length,
       vtysh_terminal_length_cmd,
       "terminal length <0-512>",
       "Set terminal line parameters\n"
       "Set number of lines on a screen\n"
       "Number of lines on screen (0 for no pausing)\n")
{
  int lines;
  char *endptr = NULL;
  char default_pager[10];

  lines = strtol (argv[0], &endptr, 10);
  if (lines < 0 || lines > 512 || *endptr != '\0')
    {
      vty_out (vty, "length is malformed%s", VTY_NEWLINE);
      return CMD_WARNING;
    }

  if (vtysh_pager_name)
    {
      free (vtysh_pager_name);
      vtysh_pager_name = NULL;
    }

  if (lines != 0)
    {
      snprintf(default_pager, 10, "more -%i", lines);
      vtysh_pager_name = strdup (default_pager);
    }

  return CMD_SUCCESS;
}

DEFUN (vtysh_terminal_no_length,
       vtysh_terminal_no_length_cmd,
       "terminal no length",
       "Set terminal line parameters\n"
       NO_STR
       "Set number of lines on a screen\n")
{
  if (vtysh_pager_name)
    {
      free (vtysh_pager_name);
      vtysh_pager_name = NULL;
    }

  vtysh_pager_init();
  return CMD_SUCCESS;
}
#endif

/* Execute command in child process. */
static int
execute_command (const char *command, int argc, const char *arg1,
         const char *arg2)
{
  int ret;
  pid_t pid;
  int status;

  /* Call fork(). */
  pid = fork ();

  if (pid < 0)
    {
      /* Failure of fork(). */
      fprintf (stderr, "Can't fork: %s\n", safe_strerror (errno));
      exit (1);
    }
  else if (pid == 0)
    {
      /* This is child process. */
      switch (argc)
    {
    case 0:
      ret = execlp (command, command, (const char *)NULL);
      break;
    case 1:
      ret = execlp (command, command, arg1, (const char *)NULL);
      break;
    case 2:
      ret = execlp (command, command, arg1, arg2, (const char *)NULL);
      break;
    }

      /* When execlp suceed, this part is not executed. */
      fprintf (stderr, "Can't execute %s: %s\n", command, safe_strerror (errno));
      exit (1);
    }
  else
    {
      /* This is parent. */
      execute_flag = 1;
      ret = wait4 (pid, &status, 0, NULL);
      execute_flag = 0;
    }
    status = ret;
  return 0;
}

DEFUN (vtysh_start_shell,
       vtysh_start_shell_cmd,
       "diag",
       "Start SDK diag\n")

{
    execute_command ("diag", 0, NULL, NULL);

    return CMD_SUCCESS;
}

static void
vtysh_install_default (enum node_type node)
{
  //RTK    install_element (node, &config_list_cmd);
  INSTALL_ELEMENT (node, CMD_PRIV_0, &vtysh_end_all_cmd);
}

/* To disable readline's filename completion. */
static char *
vtysh_completion_entry_function (const char *ignore, int invoking_key)
{
  return NULL;
}


static uint32 readln_last_uptime = 0; /* ms */
static int readln_counter_sec;
static int readln_timeout_sec;

void readln_counter_set(int counter)
{
    readln_counter_sec = counter;
}

void readln_timeout_set(int timeout)
{
    readln_timeout_sec = (timeout > 0)? timeout : 0;
}

int readln_timeout_chk(void)
{
    return (readln_timeout_sec)? (readln_counter_sec - readln_timeout_sec) : -1;
}


/* one tick is 0.1 second */
void readln_event_hook(void)
{
    static uint32 deltaTime = UINT32_MAX;
    uint32 uptime = 0;

    if (SYS_ERR_OK != osal_time_sysUptime_get(&uptime))
        return;

    /* Initialize */
    if (UINT32_MAX == deltaTime)
    {
        readln_last_uptime = uptime;
        deltaTime = 0;
        return;
    }

    /* Accumulate the delta time in milliseconds */
    if (uptime >= readln_last_uptime)
        deltaTime += (uptime - readln_last_uptime);
    else /* Wrap around */
        deltaTime += ((UINT32_MAX - readln_last_uptime) + uptime);

    readln_last_uptime = uptime;

    if (1000 <= deltaTime)
    {
        deltaTime -= 1000;
        readln_counter_sec++;

        /* check timeout */
        if (readln_timeout_sec && (readln_counter_sec >= readln_timeout_sec)) {
            /* readline timeout */
            rl_done = 1;    /* readline done */
        }
    }
}

#ifdef CONFIG_SYS_DEBUG_SHELL
  #ifdef CONFIG_SYS_BUILD_RELEASE
static uint32 _diagDebug_auth(void)
{
    char input_password[CAPA_SYS_PASSWORD_LEN + 1];
    char sha1sum[CAPA_SYS_SHA1STR_LEN + 1];

    printf("\nDiagnostics: ");

    osal_memset(input_password, 0, sizeof(input_password));
    input_str(input_password, CAPA_SYS_PASSWORD_LEN, '*');

    SYS_MEM_CLEAR(sha1sum);
    sal_util_sha1_hash(input_password, osal_strlen(input_password), sha1sum, sizeof(sha1sum));

    if (0 == strcmp(sha1sum, DFLT_CLI_DEBUG_PASSWORD))
        return TRUE;

    return FALSE;
}
  #endif

void vtysh_diagDebug(void)
{
    sys_enable_t enable = DISABLED;

    cmd_textline_enable_get(&enable);
    if (ENABLED == enable)
        return;

    if (DISABLED == DFLT_CLI_REMOTE_DEBUG && access_flag != SYS_ACCESS_CLI_CONSOLE)
    {
        printf("\nDo not allow remote user to enter debug mode\n");
        return;
    }

  #ifdef CONFIG_SYS_BUILD_RELEASE
    if (DIAGDBG_AUTH_DEBUG == diagdbg_auth)
    {
        printf("\n");
        diagdbg_flag = DIAGDBG_OP_DEBUG;
        return;
    }

    if (TRUE == _diagDebug_auth())
    {
        printf("Press ENTER to continue\n");
        diagdbg_flag = DIAGDBG_OP_DEBUG;
        diagdbg_auth = DIAGDBG_AUTH_DEBUG;
    }
    else
    {
        printf("Authentication Fail. Press ENTER to exit\n");
        diagdbg_flag = DIAGDBG_OP_NONE;
    }
  #else
    diagdbg_flag = DIAGDBG_OP_DEBUG;
    diagdbg_auth = DIAGDBG_AUTH_DEBUG;
  #endif

    return;
}
#endif

void
vtysh_readline_init (void)
{
  /* readline related settings. */
  rl_set_keyboard_input_timeout(100000);  /* 0.1 seconds; it's in usec */
  rl_event_hook = (rl_hook_func_t *)readln_event_hook;

//  rl_bind_key ('?', (Function *) vtysh_rl_describe);
  rl_bind_key ('?', (rl_command_func_t *) vtysh_rl_describe);
#ifdef CONFIG_SYS_DEBUG_SHELL
  /* RTK:  Bind keys to enter debug mode*/
  rl_bind_key (CTRL('t'), (rl_command_func_t *) vtysh_diagDebug);
  rl_bind_key (META('t'), (rl_command_func_t *) vtysh_diagDebug);
  rl_bind_keyseq("\\C-\\M-t", (rl_command_func_t *) vtysh_diagDebug);
#endif

  rl_completion_entry_function = vtysh_completion_entry_function;
//  rl_attempted_completion_function = (CPPFunction *)new_completion;
  rl_attempted_completion_function = (rl_completion_func_t *)new_completion;

  /* do not append space after completion. It will be appended
   * in new_completion() function explicitly. */
  rl_completion_append_character = '\0';
}

static int    timeout_duration;

void
timeout_set_duration (
    int    duration
    )
{
    readln_timeout_set(duration);
    timeout_duration = duration;
}

int
timeout_get_duration (void)
{
    return timeout_duration;
}

void
timeout_clear (void)
{
    alarm(0);
}

void
timeout_exit (void)
{
    struct stat st;

    if (timeout_duration <= 0)
        return;

    /* NOTICE:
     * DO NOT REMOVE these functions related of timeout, even readline had implemented timeout feature,
     * but we have to consider the following state:
     * (1) Not enter the readline phase (like: login phase).
     * (2) A command have bugs to make deadlock(while-loop), will not return readline.
     */

    if ((fstat(STDIN_FILENO, &st) == 0) && (S_ISCHR(st.st_mode)) && (major(st.st_rdev) == 4))    /* TTY devices (4 char) */
    {
        /* Local Console */
        //printf("\x1B[H\x1B[2J");    /* Clear Screen */
        return;
    } else {
        /* Remote Login */
        //printf("\nauto-logout\n");
        //user_auth();
        //exit(0);
        return;
    }
}

void
timeout_reset (void)
{
#ifndef CONFIG_SYS_UI_CLI_SESSION_TIMEOUT
    timeout_duration = 0; /* No timeout */
#endif /* CONFIG_SYS_UI_CLI_SESSION_TIMEOUT */

    readln_counter_set(0);
    signal(SIGALRM, (void *)timeout_exit);
    alarm(timeout_duration);
}

void
vtysh_init_timeout(int duration)
{
#ifdef CONFIG_SYS_UI_CLI_SESSION_TIMEOUT
    /* RTK: User defined session timer */
    uint32 timeout = 0;
    sys_line_cliType_t type = SYS_LINE_CLI_CONSOLE;

    if (access_flag == SYS_ACCESS_CLI_CONSOLE)
        type = SYS_LINE_CLI_CONSOLE;
    else if (access_flag == SYS_ACCESS_CLI_TELNET)
        type = SYS_LINE_CLI_TELNET;
  #ifdef CONFIG_SYS_APP_SSH
    else if (access_flag == SYS_ACCESS_CLI_SSH)
        type = SYS_LINE_CLI_SSH;
  #endif

    sal_sys_cliSessionTimeout_get(type, &timeout);

    timeout *= 60;

    timeout_set_duration(timeout);
#else
    timeout_set_duration(duration);
#endif  /* CONFIG_SYS_UI_CLI_SESSION_TIMEOUT */

    /* Reset timeout timer */
    timeout_reset();
}


char *
vtysh_prompt (void)
{
#define MAX_HOST_PROMPT_LEN 20
  static struct utsname names;
  static char buf[100];
  const char*hostname;
  extern struct host host;
  char currSysName[CAPA_SYS_NAME_LEN + 1];
  char showHost[MAX_HOST_PROMPT_LEN + 1];

  osal_memset(currSysName, 0, sizeof(currSysName));
  sal_sys_systemName_get(currSysName);

  if (0 == osal_strlen(currSysName))
  {
      sys_board_conf_t boardConf;
      SYS_MEM_CLEAR(boardConf);
      sal_sys_boardConf_get(&boardConf);

      if (0 != osal_strcmp(boardConf.boardName, host.name))
      {
        XFREE (MTYPE_HOST, host.name);
        host.name = XSTRDUP (MTYPE_HOST, boardConf.boardName);
      }
  }
  else
  {
      if (0 != osal_strcmp(currSysName, host.name))
      {
          XFREE (MTYPE_HOST, host.name);
          host.name = XSTRDUP (MTYPE_HOST, currSysName);
      }
  }

  hostname = host.name;

  if (!hostname)
    {
      if (!names.nodename[0])
    uname (&names);
      hostname = names.nodename;
    }

    SYS_MEM_CLEAR(showHost);
    strlcpy(showHost, hostname, sizeof(showHost));

  SYS_MEM_CLEAR(buf);
  /* Share interface and interface range with the same node */
  if ((IF_NODE == vty->node) || (IF_TRK_NODE == vty->node)
#ifdef CONFIG_SYS_L3_ROUTE
      || (IF_VLAN_NODE == vty->node)
#endif
     )
  {
    if (TRUE == g_isIntfRange)
    {
        snprintf (buf, sizeof(buf), DFLT_CLI_PROMPT_IF_RANGE, showHost);
        return buf;
    }
  }

  snprintf (buf, sizeof(buf), cmd_prompt(vty->node), showHost);

  return buf;
}

void
vtysh_init_vty (void)
{
  /* Make vty structure. */
  vty = vty_new ();
  vty->type = VTY_SHELL;
  vty->node = VIEW_NODE;

  /* Initialize commands. */
  cmd_init (0);

  //vtysh_install_default (VIEW_NODE);
  vtysh_install_default (ENABLE_NODE);
  vtysh_install_default (CONFIG_NODE);
  vtysh_install_default (IF_NODE);
  vtysh_install_default (IF_TRK_NODE);
#ifdef CONFIG_SYS_L3_ROUTE
  vtysh_install_default (IF_VLAN_NODE);
#endif
  vtysh_install_default (VTY_NODE);
  vtysh_install_default (VLAN_DB_NODE);
#if defined(CONFIG_SYS_PROTO_IGMP_FILTERING) || defined(CONFIG_SYS_PROTO_IGMP_THROTTLING)
  vtysh_install_default (IGMP_PROFILE_NODE);
#endif
#ifdef CONFIG_SYS_PROTO_MLD_SNOOPING
  vtysh_install_default (MLD_PROFILE_NODE);
#endif
#ifdef CONFIG_SYS_USER_DEFINED_ACL
  vtysh_install_default (ACL_MAC_NODE);
  vtysh_install_default (ACL_IPV4_NODE);
  vtysh_install_default (ACL_IPV6_NODE);
#endif
#ifdef CONFIG_SYS_MGMT_ACL
  vtysh_install_default (MGMT_ACL_NODE);
#endif
#ifdef CONFIG_SYS_PROTO_STP_MSTP
  vtysh_install_default (MST_CFG_NODE);
#endif

  INSTALL_ELEMENT (VIEW_NODE, CMD_PRIV_1, &vtysh_enable_cmd);
  INSTALL_ELEMENT (ENABLE_NODE, CMD_PRIV_15, &vtysh_config_terminal_cmd);
  INSTALL_ELEMENT (ENABLE_NODE, CMD_PRIV_1, &vtysh_disable_cmd);

  //RTK

  /* "exit" command. */
  INSTALL_ELEMENT (VIEW_NODE, CMD_PRIV_0, &vtysh_exit_all_cmd);
  INSTALL_ELEMENT (CONFIG_NODE, CMD_PRIV_0, &vtysh_exit_all_cmd);
  INSTALL_ELEMENT (ENABLE_NODE, CMD_PRIV_0, &vtysh_exit_all_cmd);
  INSTALL_ELEMENT (VTY_NODE, CMD_PRIV_0, &vtysh_exit_all_cmd);

  INSTALL_ELEMENT (DEBUG_NODE, CMD_PRIV_0, &vtysh_start_shell_cmd);

#if 0
  install_element (VIEW_NODE, CMD_PRIV_1, &vtysh_terminal_length_cmd);
  install_element (ENABLE_NODE, CMD_PRIV_1, &vtysh_terminal_length_cmd);
  install_element (VIEW_NODE, CMD_PRIV_1, &vtysh_terminal_no_length_cmd);
  install_element (ENABLE_NODE, CMD_PRIV_1, &vtysh_terminal_no_length_cmd);
#endif
}

