#include <soc.h>
#include <dram/memcntlr.h>
#include <dram/memcntlr_ag.h>
#include <util.h>

extern mc_info_t meminfo;

MEMCNTLR_SECTION
u32_t r480_calibration(void) {
    puts("\nII: R480 calibration... \n");
    u32_t res = MEM_CNTLR_ZQ_R480_UNINIT;

    u8_t ddr_type = RFLD_DCR(ddr);
    u8_t nocd2, en_dmyi, vref_s;

    switch(ddr_type) {
        case DDR_TYPE_DDR4:
            nocd2=1;
            en_dmyi=1;
            vref_s=0x34;    // 62%
        break;
        case DDR_TYPE_LPDDR3:
            nocd2=0;
            en_dmyi=0;
            vref_s=0x28;    // 50%
        break;
        default:
        case DDR_TYPE_DDR3:
            nocd2=0;
            en_dmyi=0;
            vref_s=0x28;    // 50%
        break;
    }

    RMOD_PAD_CTRL_PROG(dzq_up_sel, 0, zctrl_clk_sel, 0x3, rzq_ext_r240, 0, dzq_auto_up, 1);
    RMOD_ZQ_NOCD2(zq_ena_nocd2_0, nocd2);
    RMOD_PAD_BUS_1(pad_bus_1, 0);
    RMOD_PAD_BUS_0(en_dmyi, en_dmyi, vref_sel, 1, vref_range, 1, vref_s, vref_s); // 62%
    RMOD_ZQ_PAD_CTRL(zq_vref_sel_ini, 1, zq_vref_s_ini, vref_s);
    // for DDR3 board issue
    RMOD_PAD_BUS_2(vref_s_2, vref_s, vref_s_1, vref_s, vref_s_0, vref_s);

    RMOD_DPI_CTRL_0(fw_set_mode, 2);

    //RMOD_DPI_CTRL_1(write_en_0, 1, fw_set_wr_dly, 1);
    DPI_CTRL_1rv = 0x3;
    udelay(1000);

    RMOD_PAD_CTRL_PROG(rzq_cal_en, 0x1);
    u32_t cnt=0;
    while(0==RFLD_PAD_RZCTRL_STATUS(rzq_cal_done)) {
        if(++cnt>0x100000) {udelay(1); res=MEM_CNTLR_R480_TIMEOUT; break;}
    }

    if(0x1F==RFLD_PAD_RZCTRL_STATUS(rzq_480code)) {
        res=MEM_CNTLR_R480_ERROR;
    }

    printf("done, res=%d, status=0x%x\n", res, PAD_RZCTRL_STATUSrv);

    RMOD_PAD_CTRL_PROG(rzq_cal_en, 0x0);

    return res;
}

MEMCNTLR_SECTION
u32_t _X_ZPROG(u32_t ocd_odt) {
    u8_t ocd = ocd_odt>>16;
    u8_t odt = ocd_odt&0xFFFF;
    //FIXME
    u32_t zprog = 0x1234+ocd+odt;
    return zprog;
}

MEMCNTLR_SECTION
u32_t zq_calibration(u32_t ocd_odt[]) {
    u32_t res = MEM_CNTLR_ZQ_R480_UNINIT;
    puts("\nII: ZQ calibration... \n");

    u8_t idx, ddr_type = RFLD_DCR(ddr);
    u8_t nocd2, en_dmyi, vref_s;
    u32_t _zprog[2] = {0x28763B, 0x20763B}; // FIXME
    u32_t zprog;

    //for(idx=zq_dq_odt_sel; idx<zq_end_sel; idx++) {
    for(idx=0; idx<2; idx++) {
        printf("\nDD: ZQ sel %d calibration \n", idx);

        //zprog = _X_ZPROG(ocd_odt[idx]);
        zprog = _zprog[idx];

        if(DDR_TYPE_DDR4 == ddr_type && idx<zq_cke_ocd_sel) {   // DDR4 CMD
            nocd2=1;
            en_dmyi=1;
            vref_s=0x28;    // 62%
        } else {        // DDR3, LPDDR3, DDR4 DQ/DQS
            nocd2=0;
            en_dmyi=0;
            vref_s=0x28;    // 50%
        }

        RMOD_PAD_CTRL_PROG(dzq_up_sel, idx, zctrl_clk_sel, 0x3, rzq_ext_r240, 0, dzq_auto_up, 1);
        RMOD_ZQ_NOCD2(zq_ena_nocd2_0, nocd2);
        RMOD_PAD_BUS_1(pad_bus_1, 0);
        RMOD_PAD_BUS_0(en_dmyi, en_dmyi, vref_sel, 1, vref_range, 1, vref_s, vref_s); // 62%
        RMOD_ZQ_PAD_CTRL(zq_vref_sel_ini, 1, zq_vref_s_ini, vref_s);
        // for DDR3 board issue
        RMOD_PAD_BUS_2(vref_s_2, vref_s, vref_s_1, vref_s, vref_s_0, vref_s);
        RMOD_DPI_CTRL_0(fw_set_mode, 2);
        //RMOD_DPI_CTRL_1(write_en_0, 1, fw_set_wr_dly, 1);
        DPI_CTRL_1rv = 0x3;
        udelay(1000);

        // apply zprog
        RMOD_PAD_CTRL_ZPROG_0(zprog, zprog);

        RMOD_PAD_CTRL_PROG(zctrl_start, 0x1);
        u32_t cnt=0;
        while(0==RFLD_PAD_ZCTRL_STATUS(cali_done)) {
            if(++cnt>0x10000) {udelay(1); res=MEM_CNTLR_ZQ_TIMEOUT; break;}
        }
        if(0!=RFLD_PAD_ZCTRL_STATUS(cali_status)) {
            res=MEM_CNTLR_ZQ_ERROR;
        }

        printf("DD: done, res=%d, status=0x%08x\n", res, PAD_ZCTRL_STATUSrv);
        RMOD_PAD_CTRL_PROG(zctrl_start, 0x0);

    }

    // APPLY ODT/OCD Select
    DQ_ODT_SEL_0rv = 0;
    DQ_ODT_SEL_1rv = 0;
    DQ_OCD_SEL_0rv = 0;
    DQ_OCD_SEL_1rv = 0;
    DQS_P_ODT_SEL_0rv = 0x00110011;
    DQS_P_ODT_SEL_1rv = 0x00110011;
    DQS_N_ODT_SEL_0rv = 0x11001100;
    DQS_N_ODT_SEL_1rv = 0x11001100;
    DQS_OCD_SEL_0rv = 0;
    DQS_OCD_SEL_1rv = 0;
    CKE_OCD_SELrv = 0;
    ADR_OCD_SELrv = 0;
    CK_OCD_SELrv = 0;
    return res;
}

MEMCNTLR_SECTION
void mc_enable_bstc(void)
{
    u32_t cnt;
    // enable BSTC
    RMOD_CSR(bstc_idle, 0, mem_idle, 1);
    // clean error
    RMOD_BCR(crr, 1);
    // flush cmd/wd/rg/rd sram
    RMOD_BCR(fus_rd, 1, fus_rg, 1, fus_wd, 1, fus_cmd, 1);
    //RMOD_BCR(fus_wd, 1, fus_cmd, 1);
    cnt=0x10000;
    while(0!=cnt){cnt--;}
}


MEMCNTLR_SECTION
unsigned int mc_bstc_check_row(BSTC_INFO_T info, u32_t *ary, u32_t len)
{
    BCR_T bcr;
    BSTC_CMD_T cmd = {{0}};     // {{0}}, gcc bug 53119
    const u32_t burst = 1;
    u32_t idx, cnt;
    u8_t dq_factor;

    dq_factor = 1; //RFLD_DCR(dq_32)+1;
    //printf("DD: dq_fac=%d\n", dq_factor);

    mc_enable_bstc();

    // set BCR
    bcr.f.rd_ex_cnt = burst*len*(RXI310_BUS_WIDTH/32)*dq_factor;
    bcr.f.at_err_stop = 0;
    bcr.f.dis_msk = 1;
    bcr.f.loop = 0; //no loop mode
    bcr.f.cmp = 1;
    bcr.f.stop = 0;
    BCRrv = bcr.v;
    RMOD_BCT(loop_cnt, 0);

    u32_t con_addr = BSTC_SRAM_CMD_BASE;
    for (idx=0; idx<len; idx++, con_addr+=0x8) {
        // set CMD_SRAM
        cmd.f.bank = BSTC_PA2BNK(info, ary[idx]);
        cmd.f.col = BSTC_PA2COL(info, ary[idx]);
        cmd.f.row = BSTC_PA2ROW(info, ary[idx]);
        cmd.f.bl = burst;
        cmd.f.cmd = BSTC_WR_CMD;    // write
        BSTC_CMD_WRITE(con_addr, cmd);
        //printf("DD: con_addr=0x%x idx=%d ary[idx]=0x%08x\n", con_addr, idx, ary[idx]);

        // set CMD_SRAM,
        cmd.f.bank = BSTC_PA2BNK(info, ary[len-1-idx]);
        cmd.f.col = BSTC_PA2COL(info, ary[len-1-idx]);
        cmd.f.row = BSTC_PA2ROW(info, ary[len-1-idx]);
        cmd.f.cmd = BSTC_RD_CMD;    // read
        BSTC_CMD_WRITE(con_addr+len*8, cmd);
        //printf("\nDD: con_addr=0x%x idx=%d ary[idx]=0x%08x\n", con_addr+len*8, len-1-idx, ary[len-1-idx]);
    }

    // prepare WD and RG
    con_addr = BSTC_SRAM_WD_BASE;
    for (idx=0; idx<(len*burst); idx++) {
        #define WD_RG_GAP   (BSTC_SRAM_RG_BASE-BSTC_SRAM_WD_BASE)
        for (cnt=0; cnt<BSTC_SRAM_WD_BUSW*dq_factor; cnt+=32, con_addr+=4) { // 32-bit
            REG32(con_addr)=ary[idx];
            REG32(con_addr+WD_RG_GAP)=ary[len-1-idx];
            //printf("DD: WD 0x%x = 0x%08x, RG 0x%x = 0x%08x\n", con_addr, REG32(con_addr), con_addr+WD_RG_GAP, REG32(con_addr+WD_RG_GAP));
        }
    }

    //printf("DD: kick off BSTC WRITE READ... ");
    RMOD_CCR(btt, 1, init, 0, dpit, 0);
    while (0==RFLD_CCR(btt)){puts("*\b");};
    //printf("done, \nDD: BST=0x%x, BER=0x%x\n", BSTrv, BERrv);

    // bstc clear
    mc_disable_bstc();

    u32_t err = BERrv;
    for (idx=0, cnt=0; idx<31; idx++) {
        if (0!= ((1<<idx) & err)) cnt++;
    }

    return (16-cnt);
}

MEMCNTLR_SECTION
unsigned int mc_dram_size_detect(void) {
                    //        8Gb,        4Gb,        2Gb,        1Gb,      512Mb
    u32_t array[] = {  0x3F801200, 0x1F801200, 0x0F801200, 0x07801200, 0x03801200 };
//    u32_t idx, val;

    // bstc init
    BSTC_INFO_T info;

    puts("II: Auto Size Detect... ");
    info.col_size = get_col_size();
    info.bnk_size = get_bnk_size();
    info.wr_ent_cnt = info.rd_ent_cnt = BSTC_SRAM_CMD_ENTY/2;
    u32_t row_size = mc_bstc_check_row(info, array, sizeof(array)/sizeof(u32_t));
    //printf("\nDD: Bank=%d, Row=%d, Col=%d\n", info.bnk_size, row_size, info.col_size);
    printf("x%d %dMB\n", RFLD_DCR(half_dq)?16:32, 1<<(info.col_size+info.bnk_size+row_size)>>20);

    return 0;
}

MEMCNTLR_SECTION
void mc_result_show_dram_config(void) {
    mc_register_set_t *rs = meminfo.register_set;

    printf("II: CCR: 0x%08x, DCR: 0x%08x, IOCR:0x%08x, CSR: 0x%08x\n",
        rs->ccr.v, rs->dcr.v, rs->iocr.v, rs->csr.v);
    printf("II: TPR0:0x%08x, TPR1:0x%08x, TPR2:0x%08x, TPR3:0x%08x\n",
        rs->tpr0.v, rs->tpr1.v, rs->tpr2.v, rs->tpr3.v);
    printf("II: MR0: 0x%08x, MR1: 0x%08x, MR2: 0x%08x, MR3: 0x%08x\n",
        rs->mr0.v, rs->mr1.v, rs->mr2.v, rs->mr3.v);
    printf("II: DRR: 0x%08x, MISC:0x%08x, MRINFO: 0x%08x\n",
        rs->drr.v, rs->misc.v, rs->mrinfo.v);

    return;
}

MEMCNTLR_SECTION
void mc_misc(void) {
	mc_dram_size_detect();
}

//REG_INIT_FUNC(mc_misc, 31);
