#include <soc.h>
#include <dram/memcntlr.h>
#include <dram/memcntlr_ag.h>
#include <util.h>
#include <cg/cg.h>

static mc_dram_param_t   dram_param;
static mc_cntlr_opt_t    cntlr_opt;
static mc_dpi_opt_t      dpi_opt;
static mc_register_set_t register_set;
static mc_register_set_t default_rs;

extern mc_var_t mc_var_begin, mc_var_end;

mc_info_t meminfo = {
    &dram_param, &cntlr_opt, &dpi_opt, &register_set
};

MEMCNTLR_SECTION
__attribute__((weak))
const mc_info_t *
proj_mc_info_select(void) {
    return (mc_info_t *)1;
}

MEMCNTLR_SECTION
void mc_info_probe(void) {
    const mc_info_t *mi;

    mi = proj_mc_info_select();
    ISTAT_SET(probe, MEM_PROBE_UNINIT);

    if (mi == VZERO) {
        ISTAT_SET(probe, MEM_PROBE_FAIL);
    } else if (((u32_t)mi) == 1) {
        ISTAT_SET(probe, MEM_PROBE_FUNC_MISSING);
    } else {
        inline_memcpy(&dram_param,   mi->dram_param,   sizeof(mc_dram_param_t));
        inline_memcpy(&cntlr_opt,    mi->cntlr_opt,    sizeof(mc_cntlr_opt_t));
        inline_memcpy(&dpi_opt,      mi->dpi_opt,      sizeof(mc_dpi_opt_t));
        inline_memcpy(&register_set, mi->register_set, sizeof(mc_register_set_t));
        inline_memcpy(&default_rs,   mi->register_set, sizeof(mc_register_set_t));

        _soc.dram_info = (void *)&meminfo;

        ISTAT_SET(probe, MEM_PROBE_OK);
    }

    return;
}

#define NS2CK(ns, tck_ps)       (((ns)*1000+((tck_ps)-1))/(tck_ps))
#define NS2CK2(ns, freq)        ((ns)*(freq))/1000 //((ns*freq + 999)/1000)
#define ROUNDUP(v, d)           (((v)+((d)-1))/(d))
#define BOUND_CHECK(v, min, max) ({ u32_t __v=v;    \
                                    if (__v > max) __v=max; \
                                    else if (__v < min) __v=min;    \
                                    __v; })

MEMCNTLR_SECTION
void mc_xlat_dram_type(mc_info_t *r) {
    u32_t _v = r->dram_param->dram_type;
    if (DDR_TYPE_SPIN==_v) {    // according to strapped pin
        u8_t dt = RFLD_PSR0(pin_dramtype);
        r->register_set->dcr.f.ddr = (2==dt)?DDR_TYPE_LPDDR3:((1==dt)?DDR_TYPE_DDR4:DDR_TYPE_DDR3);
    } else if ((DDR_TYPE_DDR3==_v) ||
       (DDR_TYPE_DDR4==_v) ||
       (DDR_TYPE_LPDDR3==_v)) {
        r->register_set->dcr.f.ddr = _v;
    } else {
        ISTAT_SET(xlat, MEM_DRAM_TYPE_ERROR);
    }
    return;
}

MEMCNTLR_SECTION
void mc_xlat_bankcnt(mc_info_t *r) {
    u32_t _v = r->dram_param->bankcnt;
    u32_t bc=0;
    for(bc=0; bc<=4; bc++) {
        if (_v==(2<<bc)) break;
    }    
    if (4==bc) {
        ISTAT_SET(xlat, MEM_BANK_NUM_ERROR);
    } else {
        r->register_set->misc.f.bank_size = bc;
    }
    return;
}

MEMCNTLR_SECTION
void mc_xlat_pagesize(mc_info_t *r) {
    u32_t _v = r->dram_param->pagesize;
    u32_t ps=0;
    for(ps=0; ps<=8; ps++) {
        if (_v==(256<<ps)) break;
    }    
    if (8==ps) {
        ISTAT_SET(xlat, MEM_PAGE_SIZE_ERROR);
    } else {
        r->register_set->misc.f.page_size = ps;
    }
    return;
}

MEMCNTLR_SECTION
void mc_xlat_burst(mc_info_t *r) {
    u32_t _v = r->dram_param->burst >> 3;
    if (_v<2) {
        r->register_set->misc.f.bst_size = _v;
        r->register_set->mr0.f.bl = (1==_v)?0:2;    // DDR3, 0b00: 8(fixed), 0b10: 4(fixed)
    } else {
        ISTAT_SET(xlat, MEM_BL_ERROR);
    }
    return;
}

MEMCNTLR_SECTION
void mc_xlat_ref_num(mc_info_t *r) {
    u32_t _v = r->dram_param->ref_num;
    if (_v<16) {
        r->register_set->drr.f.ref_num = r->dram_param->ref_num;
    } else {
        ISTAT_SET(xlat, MEM_REF_NUM_ERROR);
    }
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tref_ns(mc_info_t *r) {
    u32_t _v = (r->dram_param->min_tref_ns)/(r->cntlr_opt->dfi_rate);
    r->register_set->drr.f.tref = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trfc_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trfc_ns, r->cntlr_opt->dfi_rate);
    r->register_set->drr.f.trfc = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tzqcs_tck(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tzqcs_tck, r->cntlr_opt->dfi_rate);
    r->register_set->tpr0.f.tzqcs = _v;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tcke_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tcke_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr0.f.tcke = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trtp_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trtp_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr0.f.trtp = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_twr_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_twr_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr0.f.twr = NS2CK2(_v, GET_MEM_MHZ());
    //_v = r->register_set->tpr0.f.twr;
    r->register_set->mr0.f.wr = (_v>10)?((_v/2)&0x7):(_v-4); 
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tras_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tras_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr0.f.tras = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trp_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trp_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr0.f.trp = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tfaw_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tfaw_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.tfaw = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trtw_tck(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trtw_tck, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.trtw = _v; //NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_twtr_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_twtr_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.twtr = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tccd_tck(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tccd_tck, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.tccd = _v; //NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trcd_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trcd_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.trcd = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trc_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trc_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.trc = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_trrd_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_trrd_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr1.f.trrd = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tmrd_tck(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tmrd_tck, r->cntlr_opt->dfi_rate);
    r->register_set->tpr2.f.tmrd = _v; //NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tccd_r_tck(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tccd_r_tck, r->cntlr_opt->dfi_rate);
    r->register_set->tpr3.f.tccd_r = _v;//NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_twtr_s_ns(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_twtr_s_ns, r->cntlr_opt->dfi_rate);
    r->register_set->tpr3.f.twtr_s = NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_tccd_s_tck(mc_info_t *r) {
    u32_t _v = ROUNDUP(r->dram_param->min_tccd_s_tck, r->cntlr_opt->dfi_rate);
    r->register_set->tpr3.f.tccd_s = _v;//NS2CK2(_v, GET_MEM_MHZ());
    return;
}

MEMCNTLR_SECTION
void mc_xlat_min_dpin_cmd_lat(mc_info_t *r) {
    r->register_set->tpr3.f.dpin_cmd_lat = r->dram_param->min_dpin_cmd_lat;
}

MEMCNTLR_SECTION
void mc_xlat_add_lat(mc_info_t *r) {
    r->register_set->mrinfo.f.add_lat = r->dram_param->add_lat;
    //FIXME
    return;
}

MEMCNTLR_SECTION
void mc_xlat_rd_lat(mc_info_t *r) {
    u32_t _v = r->dram_param->rd_lat;
    r->register_set->mrinfo.f.rd_lat = ROUNDUP(_v, r->cntlr_opt->dfi_rate); // CHECK
    if (_v<12) {
        r->register_set->mr0.f.cl_0 = _v-4;
    } else {
        r->register_set->mr0.f.cl_0 = _v-12;
        r->register_set->mr0.f.cl_1 = 1;
    }
        
    return;
}

MEMCNTLR_SECTION
void mc_xlat_wr_lat(mc_info_t *r) {
    u32_t _v = r->dram_param->wr_lat;
    r->register_set->mrinfo.f.wr_lat = ROUNDUP(_v, r->cntlr_opt->dfi_rate);    // CHECK
    r->register_set->mr2.f.cwl = _v-5;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_flush_fifo(mc_info_t *r) {
    r->register_set->ccr.f.flush_fifo = r->cntlr_opt->flush_fifo;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_dpit_en(mc_info_t *r) {
    r->register_set->ccr.f.dpit = r->cntlr_opt->dpit_en;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_btt_en(mc_info_t *r) {
    r->register_set->ccr.f.btt = r->cntlr_opt->btt_en;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_init(mc_info_t *r) {
    r->register_set->ccr.f.init = r->cntlr_opt->init;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_dq32_en(mc_info_t *r) {
    u8_t dt = r->cntlr_opt->dq32_en;
    if(2==dt) { // from strapped pin
        dt = RFLD_PSR0(ddr_data_buswidth);
    } 
    r->register_set->dcr.f.dq_32 = dt; //r->cntlr_opt->dq32_en;
    r->register_set->dcr.f.half_dq = ~dt; //~(r->cntlr_opt->dq32_en);
    return;
}

MEMCNTLR_SECTION
void mc_xlat_dfi_rate(mc_info_t *r) {
    u32_t _v = r->cntlr_opt->dfi_rate;
    if (2==_v) {    // only support dfi rate = 2:1 now
        r->register_set->dcr.f.dfi_rate = _v;
    } else {
        ISTAT_SET(xlat, MEM_DFI_RATE_ERROR);
    }
}

MEMCNTLR_SECTION
void mc_xlat_zqc_en(mc_info_t *r) {
    r->register_set->dcr.f.zqc = r->cntlr_opt->zqc_en;
}

MEMCNTLR_SECTION
void mc_xlat_rank2_en(mc_info_t *r) {
    r->register_set->dcr.f.rank2 = r->cntlr_opt->rank2_en;
}

MEMCNTLR_SECTION
void mc_xlat_tphy_rdata(mc_info_t *r) {
    r->register_set->iocr.f.thpy_rddate_en = r->cntlr_opt->tphy_rdata;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_tphy_wlat(mc_info_t *r) {
    r->register_set->iocr.f.tphy_wlat = r->cntlr_opt->tphy_wlat;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_tphy_wdata(mc_info_t *r) {
    r->register_set->iocr.f.tphy_wrdata = r->cntlr_opt->tphy_wdata;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_stc_odt_en(mc_info_t *r) {
    r->register_set->iocr.f.stc_odt = r->cntlr_opt->stc_odt_en;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_stc_cke_en(mc_info_t *r) {
    r->register_set->iocr.f.stc_cke = r->cntlr_opt->stc_cke_en;
    r->register_set->tpr0.f.txp = !(r->cntlr_opt->stc_cke_en);
    
    return;
}

MEMCNTLR_SECTION
void mc_xlat_bstc_idle(mc_info_t *r) {
    r->register_set->csr.f.bstc_idle = r->cntlr_opt->bstc_idle;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_mem_idle(mc_info_t *r) {
    r->register_set->csr.f.mem_idle = r->cntlr_opt->mem_idle;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_zqcl_inv(mc_info_t *r) {
    r->register_set->drr.f.zqcl_inv = r->cntlr_opt->zqcl_inv;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_ref_dis(mc_info_t *r) {
    r->register_set->drr.f.ref_dis = r->cntlr_opt->ref_dis;
    return;
}

MEMCNTLR_SECTION
void mc_xlat_rttnom(mc_info_t *r) {
    u32_t rttnom = r->cntlr_opt->rttnom;
    switch (rttnom) {
    case 0:
        r->register_set->mr1.f.rtt_nom_h = 0;
        r->register_set->mr1.f.rtt_nom_m = 0;
        r->register_set->mr1.f.rtt_nom_l = 0;
//        rs->mr[1].f.mr_data = mr1.v;
        break;
    case 20:
    case 30:
    case 40:
    case 60:
    case 120:
        r->register_set->mr1.f.rtt_nom_h = (rttnom <= 30)     ? 1 : 0;
        r->register_set->mr1.f.rtt_nom_m = (rttnom % 40) == 0 ? 1 : 0;
        r->register_set->mr1.f.rtt_nom_l = ((rttnom > 20) && (rttnom < 120)) ? 1 : 0;
//        rs->mr[1].f.mr_data = mr1.v;
        break;
    default:
        ISTAT_SET(xlat, MEM_RTTNOM_ERROR);
        return;
    }
}

MEMCNTLR_SECTION
void mc_xlat_rttwr(mc_info_t *r) {
    u32_t rttwr = r->cntlr_opt->rttwr;
    if (((rttwr % 60) != 0) ||
        (rttwr > 120)) {
        ISTAT_SET(xlat, MEM_RTTWR_ERROR);
    } else {
        r->register_set->mr2.f.rtt_wr = (rttwr / 60);
        //rs->mr[2].f.mr_data = mr2.v;
    }
}     

MEMCNTLR_SECTION
void mc_xlat_dpi_ocd_odt(mc_info_t *r) {
    
}

MEMCNTLR_SECTION
void mc_xlat_rzq_ext(mc_info_t *r) {
    u32_t en = r->dpi_opt->rzq_ext;
    r->register_set->pcpr.f.rzq_ext_r240 = !en; // 0: int R, 1: ext R
}

MEMCNTLR_SECTION
void mc_xlat_dzq_auto_up(mc_info_t *r) {}

DEFINE_RXI310_VAR(dram_type, dram_param, mc_xlat_dram_type);
DEFINE_RXI310_VAR(bankcnt, dram_param, mc_xlat_bankcnt);
DEFINE_RXI310_VAR(pagesize, dram_param, mc_xlat_pagesize);
DEFINE_RXI310_VAR(burst, dram_param, mc_xlat_burst);
DEFINE_RXI310_VAR(ref_num, dram_param, mc_xlat_ref_num);      
DEFINE_RXI310_VAR(min_tref_ns, dram_param, mc_xlat_min_tref_ns);
DEFINE_RXI310_VAR(min_trfc_ns, dram_param, mc_xlat_min_trfc_ns);
DEFINE_RXI310_VAR(min_tzqcs_tck, dram_param, mc_xlat_min_tzqcs_tck);
DEFINE_RXI310_VAR(min_tcke_ns, dram_param, mc_xlat_min_tcke_ns);
DEFINE_RXI310_VAR(min_trtp_ns, dram_param, mc_xlat_min_trtp_ns);
DEFINE_RXI310_VAR(min_twr_ns, dram_param, mc_xlat_min_twr_ns);
DEFINE_RXI310_VAR(min_tras_ns, dram_param, mc_xlat_min_tras_ns);
DEFINE_RXI310_VAR(min_trp_ns, dram_param, mc_xlat_min_trp_ns);
DEFINE_RXI310_VAR(min_tfaw_ns, dram_param, mc_xlat_min_tfaw_ns);
DEFINE_RXI310_VAR(min_trtw_tck, dram_param, mc_xlat_min_trtw_tck);
DEFINE_RXI310_VAR(min_twtr_ns, dram_param, mc_xlat_min_twtr_ns);
DEFINE_RXI310_VAR(min_tccd_tck, dram_param, mc_xlat_min_tccd_tck);
DEFINE_RXI310_VAR(min_trcd_ns, dram_param, mc_xlat_min_trcd_ns);
DEFINE_RXI310_VAR(min_trc_ns, dram_param, mc_xlat_min_trc_ns);
DEFINE_RXI310_VAR(min_trrd_ns, dram_param, mc_xlat_min_trrd_ns);
DEFINE_RXI310_VAR(min_tmrd_tck, dram_param, mc_xlat_min_tmrd_tck); 
DEFINE_RXI310_VAR(min_tccd_r_tck, dram_param, mc_xlat_min_tccd_r_tck);
DEFINE_RXI310_VAR(min_twtr_s_ns, dram_param, mc_xlat_min_twtr_s_ns);
DEFINE_RXI310_VAR(min_tccd_s_tck, dram_param, mc_xlat_min_tccd_s_tck);
DEFINE_RXI310_VAR(min_dpin_cmd_lat, dram_param, mc_xlat_min_dpin_cmd_lat);
DEFINE_RXI310_VAR(add_lat, dram_param, mc_xlat_add_lat);
DEFINE_RXI310_VAR(rd_lat, dram_param, mc_xlat_rd_lat);
DEFINE_RXI310_VAR(wr_lat, dram_param, mc_xlat_wr_lat);

DEFINE_RXI310_VAR(flush_fifo, cntlr_opt, mc_xlat_flush_fifo);
DEFINE_RXI310_VAR(dpit_en, cntlr_opt, mc_xlat_dpit_en);
DEFINE_RXI310_VAR(btt_en, cntlr_opt, mc_xlat_btt_en);
DEFINE_RXI310_VAR(init, cntlr_opt, mc_xlat_init);
DEFINE_RXI310_VAR(dq32_en, cntlr_opt, mc_xlat_dq32_en);
DEFINE_RXI310_VAR(dfi_rate, cntlr_opt, mc_xlat_dfi_rate);
DEFINE_RXI310_VAR(zqc_en, cntlr_opt, mc_xlat_zqc_en);
DEFINE_RXI310_VAR(rank2_en, cntlr_opt, mc_xlat_rank2_en);
DEFINE_RXI310_VAR(tphy_rdata, cntlr_opt, mc_xlat_tphy_rdata);
DEFINE_RXI310_VAR(tphy_wlat, cntlr_opt, mc_xlat_tphy_wlat);
DEFINE_RXI310_VAR(tphy_wdata, cntlr_opt, mc_xlat_tphy_wdata);
DEFINE_RXI310_VAR(stc_odt_en, cntlr_opt, mc_xlat_stc_odt_en);
DEFINE_RXI310_VAR(stc_cke_en, cntlr_opt, mc_xlat_stc_cke_en);
DEFINE_RXI310_VAR(bstc_idle, cntlr_opt, mc_xlat_bstc_idle);
DEFINE_RXI310_VAR(mem_idle, cntlr_opt, mc_xlat_mem_idle);
DEFINE_RXI310_VAR(zqcl_inv, cntlr_opt, mc_xlat_zqcl_inv);     
DEFINE_RXI310_VAR(ref_dis, cntlr_opt, mc_xlat_ref_dis);
DEFINE_RXI310_VAR(rttnom, cntlr_opt, mc_xlat_rttnom);
DEFINE_RXI310_VAR(rttwr, cntlr_opt, mc_xlat_rttwr);
DEFINE_RXI310_VAR(ocd_odt, dpi_opt, mc_xlat_dpi_ocd_odt);
DEFINE_RXI310_VAR(rzq_ext, dpi_opt, mc_xlat_rzq_ext);
DEFINE_RXI310_VAR(dzq_auto_up, dpi_opt, mc_xlat_dzq_auto_up);

MEMCNTLR_SECTION
unsigned int mc_xlat_ocd_odt(unsigned int ocd, unsigned int odt) {
    u32_t zprog = 0;
    return zprog;
}

MEMCNTLR_SECTION
void mc_dram_param_dump(mc_info_t *info,
                    const mc_var_t *begin, 
                    const mc_var_t *end,
                    int order) {
    int step=1;
    if (order<0) {
        const mc_var_t *tmp=begin;
        step=-1;
        begin=end-1;
        end=tmp-1;
    }
    const mc_var_t *idx=begin;
    while (end != idx) {
        idx->dump(info);
        idx+=step;
    }    
}

MEMCNTLR_SECTION
void dram_dump_info(void) {
    return mc_dram_param_dump(&meminfo, &mc_var_begin, &mc_var_end, 1);
}

MEMCNTLR_SECTION
void mc_xlat_n_assign(mc_info_t *info,
                      const mc_var_t *begin, 
                      const mc_var_t *end,
                      int order) {
    int step=1;
    if (order<0) {
        const mc_var_t *tmp=begin;
        step=-1;
        begin=end-1;
        end=tmp-1;
    }
    const mc_var_t *idx=begin;
    while (end != idx) {
        idx->xlat(info);
        idx+=step;
    } 
}

MEMCNTLR_SECTION
void mc_info_translation(void) {
    ISTAT_SET(xlat, MEM_XLAT_UNINIT);

    if (ISTAT_GET(probe) != MEM_PROBE_OK) {
        return;
    }

    mc_xlat_n_assign(&meminfo, &mc_var_begin, &mc_var_end, 1);  

    if (ISTAT_GET(xlat) == MEM_XLAT_UNINIT) {
        ISTAT_SET(xlat, MEM_XLAT_OK);
    }

    return;
}

MEMCNTLR_SECTION
__attribute__((weak))
void mc_cntlr_zq_static_patch(void) {}

/* __attribute__((weak))
u32_t zq_calibration(void) { return 0; } */

MEMCNTLR_SECTION
void mc_cntlr_zq_calibration(void) {
    ISTAT_SET(cntlr_zq, MEM_CNTLR_ZQ_R480_UNINIT);

    if (ISTAT_GET(to_reg) != MEM_TO_REG_OK) {
        return;
    }
#ifndef PROJECT_ON_FPGA
    u32_t res;
    if (0!=meminfo.dpi_opt->rzq_ext) {
        res = r480_calibration();
        if (MEM_CNTLR_ZQ_R480_UNINIT != res) {
            ISTAT_SET(cntlr_zq, res);
            return;
        }
    } else {
        puts("II: using external R480\n");
    }

    res=zq_calibration(meminfo.dpi_opt->ocd_odt);
    if (MEM_CNTLR_ZQ_R480_UNINIT != res) {
        // add to test, 20170830
        u32_t cnt=1;
        WARN_PRINTF("Retry %d times ...\n", cnt);
        while(cnt<5) {
            res=zq_calibration(meminfo.dpi_opt->ocd_odt);
            if (MEM_CNTLR_ZQ_R480_UNINIT == res) {
                goto PASS;
            }
            cnt++;
        }
        // test end
        ISTAT_SET(cntlr_zq, res);
        return;
    } 
PASS:
#endif
    if (ISTAT_GET(cntlr_zq) == MEM_CNTLR_ZQ_R480_UNINIT) {
        ISTAT_SET(cntlr_zq, MEM_CNTLR_ZQ_R480_OK);
    }

    return;
}

MEMCNTLR_SECTION
void mc_info_to_reg(void) {
    ISTAT_SET(to_reg, MEM_TO_REG_UNINIT);

    if (ISTAT_GET(xlat) != MEM_XLAT_OK) {
        return;
    }
    
    mc_register_set_t *rs     = meminfo.register_set;

    // applied parameters
#ifdef DEBUG
    MISCrv = 0x00000063;
    DCRrv = 0x00000223;
    IOCRrv = 0x0010100c;
    MR2rv = 0x00010000;
    MR1rv = 0x00008004;
    MR0rv = 0x00001520;
    MR_INFOrv = 0x00000062;
    DRRrv = 0x0900f30b;
    TPR0rv = 0x0000fef4;
    TPR1rv = 0x10351114;
    TPR2rv = 0x00000042;
#else
    MISCrv = rs->misc.v;
    DCRrv = rs->dcr.v;
    IOCRrv = rs->iocr.v;
    MR3rv = rs->mr3.v;
    MR2rv = rs->mr2.v;
    MR1rv = rs->mr1.v;
    MR0rv = rs->mr0.v;
    MR_INFOrv = rs->mrinfo.v;
    DRRrv = rs->drr.v;
    TPR0rv = rs->tpr0.v;
    TPR1rv = rs->tpr1.v;
    TPR2rv = rs->tpr2.v;
    CCRrv = rs->ccr.v;
#endif

    if (ISTAT_GET(to_reg) == MEM_TO_REG_UNINIT) {
        ISTAT_SET(to_reg, MEM_TO_REG_OK);
    }
    return;
}

extern void mc_dram_size_detect(void);

MEMCNTLR_SECTION
void mc_rxi310_init(void) {
    puts("II: RXI310 init ... ");   // TIMEOUT
    // rxi310 init
    //RMOD_CSR(bstc_idle, 1, mem_idle, 1);    // disable mem access, disable bstc func.
    CSRrv = 0x700;
    
    //RMOD_CCR(init, 1, btt, 0);
    CCRrv = 0x1;
    
    u32_t cntdown = 0x10000000;
    while(0==RFLD_CCR(init)) {
        cntdown--;
        if(0==cntdown) { 
            RMOD_CSR(bstc_idle, 1, mem_idle, 0); 
            puts("timeout\n");   // TIMEOUT
        }
    }
    //RMOD_CSR(bstc_idle, 1, mem_idle, 0);    // enable mem access, disable bstc func.
    CSRrv = 0x600;
    puts("done\n");   // TIMEOUT
    return;
}

MEMCNTLR_SECTION
void mc_calibration(void) {

    ISTAT_SET(cal, MEM_CAL_UNINIT);

    if (ISTAT_GET(cntlr_zq) != MEM_CNTLR_ZQ_R480_OK) {
        return;
    }

    dpi_static_setting();
    //mc_dram_size_detect();
    mc_rxi310_init();

        
    if (ISTAT_GET(cal) == MEM_CAL_UNINIT) {
        ISTAT_SET(cal, MEM_CAL_OK);
    }

    return;
}

MEMCNTLR_SECTION
__attribute__((weak))
void mc_result_show_dram_config(void);

MEMCNTLR_SECTION
void mc_result_decode(void) {
    s8_t *res[5];
    u32_t i;

    res[0] = (s8_t *)ISTAT_STR(probe);
    res[1] = (s8_t *)ISTAT_STR(xlat);
    res[2] = (s8_t *)ISTAT_STR(to_reg);
    res[3] = (s8_t *)ISTAT_STR(cntlr_zq);
    res[4] = (s8_t *)ISTAT_STR(cal);

    for (i=0; i<5; i++) {
        printf("II: %s\n", res[i]);
    }

    if(mc_result_show_dram_config)
        mc_result_show_dram_config();

    return;
}

REG_INIT_FUNC(mc_info_probe,           18);
REG_INIT_FUNC(mc_info_translation,     20);
REG_INIT_FUNC(mc_info_to_reg,          22);
REG_INIT_FUNC(mc_cntlr_zq_calibration, 24);
REG_INIT_FUNC(mc_calibration,          26);
REG_INIT_FUNC(mc_result_decode,        28);
