
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <xtables.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include "../include/linux/netfilter_ipv4/ipt_extlog.h"

static void
help(void)
{
	printf(
"EXTLOG target v%s options:\n"
"  --tcp portrange                   tcp portrange\n"
"  --udp portrange                   udp portrange\n"
"  --tcp_udp  portrange            tcp_udp portrange\n"
"\n",XTABLES_VERSION);
}

static struct option opts[] = {
	{ "TCP", 1, 0, '1' },
	{ "UDP", 1, 0, '2' },
	{ "TCP_UDP", 1, 0, '3' },
	{ 0 }
};

static void
final_check(unsigned int flags)
{
	if (!flags)
		xtables_error(PARAMETER_PROBLEM,
		           "MARK target: Parameter --set/and/or-mark"
			   " is required");
}

/* Function which parses command options; returns true if it
   ate an option */
static int
parse_v1(int c, char **argv, int invert, unsigned int *flags,
     const void *entry, struct xt_entry_target **target)
{
        u_int8_t opt = 0;
	struct extlog_target_info_v1 *markinfo
		= (struct extlog_target_info_v1 *)(*target)->data;

	switch (c) {
	case '1':
	        markinfo->proto = IPT_TCP;
                opt = IPT_TCP;
		break;
	case '2':
	        markinfo->proto = IPT_UDP;
                opt = IPT_UDP;
		break;
	case '3':
	        markinfo->proto = IPT_TCP_UDP;
                opt = IPT_TCP_UDP;
		break;
	default:
		return 0;
	}

      strcpy(&markinfo->portrange, optarg);

	if (*flags & opt)
		xtables_error(PARAMETER_PROBLEM,
			   "EXTLOG target: you use a target twice in one rule");
	*flags |= opt;
	return 1;
}


static void
print_mark(char * mark)
{
	printf("%s ", mark);
}


/* Prints out the targinfo. */
static void
print_v1(const struct ipt_ip *ip,
	 const struct xt_entry_target *target,
	 int numeric)
{
	const struct extlog_target_info_v1 *markinfo =
		(const struct extlog_target_info_v1 *)target->data;

        printf(" EXTLOG");
        
	if (markinfo->proto & IPT_TCP )
        {
            printf(" --tcp ");
        }
        if (markinfo->proto & IPT_UDP )
        {
            printf(" --udp ");
        }
        if (markinfo->proto & IPT_TCP_UDP )
        {
            printf(" --tcp_udp ");
        }
        print_mark(markinfo->portrange);
}

/* Saves the union ipt_targinfo in parsable form to stdout. */
static void
save_v1(const struct ipt_ip *ip, const struct xt_entry_target *target)
{
	const struct extlog_target_info_v1 *markinfo =
		(const struct extlog_target_info_v1 *)target->data;

        printf(" EXTLOG");
        
	if (markinfo->proto & IPT_TCP )
        {
            printf(" --tcp ");
        }
        if (markinfo->proto & IPT_UDP )
        {
            printf(" --udp ");
        }
        if (markinfo->proto & IPT_TCP_UDP )
        {
            printf(" --tcp_udp ");
        }
        print_mark(markinfo->portrange);
}


static
struct xtables_target mark_v1 = {
	.family		= NFPROTO_IPV4,
	.name		= "EXTLOG",
	.version	= XTABLES_VERSION,
	.size		= XT_ALIGN(sizeof(struct extlog_target_info_v1)),
	.userspacesize	= XT_ALIGN(sizeof(struct extlog_target_info_v1)),
	.help		= help,
	.parse		= parse_v1,
	.final_check	= final_check,
	.print		= print_v1,
	.save		= save_v1,
	.extra_opts	= opts
};

void _init(void)
{
	xtables_register_target(&mark_v1);
}
