/* vi: set sw=4 ts=4: */
/*
 * Copyright (c) 1988, 1989, 1991, 1994, 1995, 1996, 1997, 1998, 1999, 2000
 *      The Regents of the University of California.  All rights reserved.
 *
 * Busybox port by Vladimir Oleynik (C) 2005 <dzo@simtreas.ru>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

//#define version "1.4a12"


/*
 * traceroute host  - trace the route ip packets follow going to "host".
 *
 * Attempt to trace the route an ip packet would follow to some
 * internet host.  We find out intermediate hops by launching probe
 * packets with a small ttl (time to live) then listening for an
 * icmp "time exceeded" reply from a gateway.  We start our probes
 * with a ttl of one and increase by one until we get an icmp "port
 * unreachable" (which means we got to "host") or hit a max (which
 * defaults to 30 hops & can be changed with the -m flag).  Three
 * probes (change with -q flag) are sent at each ttl setting and a
 * line is printed showing the ttl, address of the gateway and
 * round trip time of each probe.  If the probe answers come from
 * different gateways, the address of each responding system will
 * be printed.  If there is no response within a 5 sec. timeout
 * interval (changed with the -w flag), a "*" is printed for that
 * probe.
 *
 * Probe packets are UDP format.  We don't want the destination
 * host to process them so the destination port is set to an
 * unlikely value (if some clod on the destination is using that
 * value, it can be changed with the -p flag).
 *
 * A sample use might be:
 *
 *     [yak 71]% traceroute nis.nsf.net.
 *     traceroute to nis.nsf.net (35.1.1.48), 30 hops max, 56 byte packet
 *      1  helios.ee.lbl.gov (128.3.112.1)  19 ms  19 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  39 ms  19 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  39 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  39 ms  40 ms  39 ms
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)  39 ms  39 ms  39 ms
 *      6  128.32.197.4 (128.32.197.4)  40 ms  59 ms  59 ms
 *      7  131.119.2.5 (131.119.2.5)  59 ms  59 ms  59 ms
 *      8  129.140.70.13 (129.140.70.13)  99 ms  99 ms  80 ms
 *      9  129.140.71.6 (129.140.71.6)  139 ms  239 ms  319 ms
 *     10  129.140.81.7 (129.140.81.7)  220 ms  199 ms  199 ms
 *     11  nic.merit.edu (35.1.1.48)  239 ms  239 ms  239 ms
 *
 * Note that lines 2 & 3 are the same.  This is due to a buggy
 * kernel on the 2nd hop system -- lbl-csam.arpa -- that forwards
 * packets with a zero ttl.
 *
 * A more interesting example is:
 *
 *     [yak 72]% traceroute allspice.lcs.mit.edu.
 *     traceroute to allspice.lcs.mit.edu (18.26.0.115), 30 hops max
 *      1  helios.ee.lbl.gov (128.3.112.1)  0 ms  0 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  19 ms  19 ms  19 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  19 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  19 ms  39 ms  39 ms
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)  20 ms  39 ms  39 ms
 *      6  128.32.197.4 (128.32.197.4)  59 ms  119 ms  39 ms
 *      7  131.119.2.5 (131.119.2.5)  59 ms  59 ms  39 ms
 *      8  129.140.70.13 (129.140.70.13)  80 ms  79 ms  99 ms
 *      9  129.140.71.6 (129.140.71.6)  139 ms  139 ms  159 ms
 *     10  129.140.81.7 (129.140.81.7)  199 ms  180 ms  300 ms
 *     11  129.140.72.17 (129.140.72.17)  300 ms  239 ms  239 ms
 *     12  * * *
 *     13  128.121.54.72 (128.121.54.72)  259 ms  499 ms  279 ms
 *     14  * * *
 *     15  * * *
 *     16  * * *
 *     17  * * *
 *     18  ALLSPICE.LCS.MIT.EDU (18.26.0.115)  339 ms  279 ms  279 ms
 *
 * (I start to see why I'm having so much trouble with mail to
 * MIT.)  Note that the gateways 12, 14, 15, 16 & 17 hops away
 * either don't send ICMP "time exceeded" messages or send them
 * with a ttl too small to reach us.  14 - 17 are running the
 * MIT C Gateway code that doesn't send "time exceeded"s.  God
 * only knows what's going on with 12.
 *
 * The silent gateway 12 in the above may be the result of a bug in
 * the 4.[23]BSD network code (and its derivatives):  4.x (x <= 3)
 * sends an unreachable message using whatever ttl remains in the
 * original datagram.  Since, for gateways, the remaining ttl is
 * zero, the icmp "time exceeded" is guaranteed to not make it back
 * to us.  The behavior of this bug is slightly more interesting
 * when it appears on the destination system:
 *
 *      1  helios.ee.lbl.gov (128.3.112.1)  0 ms  0 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  19 ms  39 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  19 ms  39 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  39 ms  40 ms  19 ms
 *      5  ccn-nerif35.Berkeley.EDU (128.32.168.35)  39 ms  39 ms  39 ms
 *      6  csgw.Berkeley.EDU (128.32.133.254)  39 ms  59 ms  39 ms
 *      7  * * *
 *      8  * * *
 *      9  * * *
 *     10  * * *
 *     11  * * *
 *     12  * * *
 *     13  rip.Berkeley.EDU (128.32.131.22)  59 ms !  39 ms !  39 ms !
 *
 * Notice that there are 12 "gateways" (13 is the final
 * destination) and exactly the last half of them are "missing".
 * What's really happening is that rip (a Sun-3 running Sun OS3.5)
 * is using the ttl from our arriving datagram as the ttl in its
 * icmp reply.  So, the reply will time out on the return path
 * (with no notice sent to anyone since icmp's aren't sent for
 * icmp's) until we probe with a ttl that's at least twice the path
 * length.  I.e., rip is really only 7 hops away.  A reply that
 * returns with a ttl of 1 is a clue this problem exists.
 * Traceroute prints a "!" after the time if the ttl is <= 1.
 * Since vendors ship a lot of obsolete (DEC's Ultrix, Sun 3.x) or
 * non-standard (HPUX) software, expect to see this problem
 * frequently and/or take care picking the target host of your
 * probes.
 *
 * Other possible annotations after the time are !H, !N, !P (got a host,
 * network or protocol unreachable, respectively), !S or !F (source
 * route failed or fragmentation needed -- neither of these should
 * ever occur and the associated gateway is busted if you see one).  If
 * almost all the probes result in some kind of unreachable, traceroute
 * will give up and exit.
 *
 * Notes
 * -----
 * This program must be run by root or be setuid.  (I suggest that
 * you *don't* make it setuid -- casual use could result in a lot
 * of unnecessary traffic on our poor, congested nets.)
 *
 * This program requires a kernel mod that does not appear in any
 * system available from Berkeley:  A raw ip socket using proto
 * IPPROTO_RAW must interpret the data sent as an ip datagram (as
 * opposed to data to be wrapped in a ip datagram).  See the README
 * file that came with the source to this program for a description
 * of the mods I made to /sys/netinet/raw_ip.c.  Your mileage may
 * vary.  But, again, ANY 4.x (x < 4) BSD KERNEL WILL HAVE TO BE
 * MODIFIED TO RUN THIS PROGRAM.
 *
 * The udp port usage may appear bizarre (well, ok, it is bizarre).
 * The problem is that an icmp message only contains 8 bytes of
 * data from the original datagram.  8 bytes is the size of a udp
 * header so, if we want to associate replies with the original
 * datagram, the necessary information must be encoded into the
 * udp header (the ip id could be used but there's no way to
 * interlock with the kernel's assignment of ip id's and, anyway,
 * it would have taken a lot more kernel hacking to allow this
 * code to set the ip id).  So, to allow two or more users to
 * use traceroute simultaneously, we use this task's pid as the
 * source port (the high bit is set to move the port number out
 * of the "likely" range).  To keep track of which probe is being
 * replied to (so times and/or hop counts don't get confused by a
 * reply that was delayed in transit), we increment the destination
 * port number before each probe.
 *
 * Don't use this as a coding example.  I was trying to find a
 * routing problem and this code sort-of popped out after 48 hours
 * without sleep.  I was amazed it ever compiled, much less ran.
 *
 * I stole the idea for this program from Steve Deering.  Since
 * the first release, I've learned that had I attended the right
 * IETF working group meetings, I also could have stolen it from Guy
 * Almes or Matt Mathis.  I don't know (or care) who came up with
 * the idea first.  I envy the originators' perspicacity and I'm
 * glad they didn't keep the idea a secret.
 *
 * Tim Seaver, Ken Adelman and C. Philip Wood provided bug fixes and/or
 * enhancements to the original distribution.
 *
 * I've hacked up a round-trip-route version of this that works by
 * sending a loose-source-routed udp datagram through the destination
 * back to yourself.  Unfortunately, SO many gateways botch source
 * routing, the thing is almost worthless.  Maybe one day...
 *
 *  -- Van Jacobson (van@ee.lbl.gov)
 *     Tue Dec 20 03:50:13 PST 1988
 */

#include <net/if.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/udp.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>

#include "libbb.h"
#include "inet_common.h"


/*
 * Definitions for internet protocol version 4.
 * Per RFC 791, September 1981.
 */
#define IPVERSION 4

#ifndef IPPROTO_ICMP
/* Grrrr.... */
#define IPPROTO_ICMP 1
#endif
#ifndef IPPROTO_IP
#define IPPROTO_IP 0
#endif

/*
 * Overlay for ip header used by other protocols (tcp, udp).
 */
struct ipovly {
	unsigned char  ih_x1[9];               /* (unused) */
	unsigned char  ih_pr;                  /* protocol */
	short   ih_len;                 /* protocol length */
	struct  in_addr ih_src;         /* source internet address */
	struct  in_addr ih_dst;         /* destination internet address */
};

/*
 * UDP kernel structures and variables.
 */
struct udpiphdr {
	struct  ipovly ui_i;            /* overlaid ip structure */
	struct  udphdr ui_u;            /* udp header */
};
#define ui_next         ui_i.ih_next
#define ui_prev         ui_i.ih_prev
#define ui_x1           ui_i.ih_x1
#define ui_pr           ui_i.ih_pr
#define ui_len          ui_i.ih_len
#define ui_src          ui_i.ih_src
#define ui_dst          ui_i.ih_dst
#define ui_sport        ui_u.uh_sport
#define ui_dport        ui_u.uh_dport
#define ui_ulen         ui_u.uh_ulen
#define ui_sum          ui_u.uh_sum


/* Host name and address list */
struct hostinfo {
	char *name;
	int n;
	uint32_t *addrs;
};

/* Data section of the probe packet */
struct outdata {
	unsigned char seq;             /* sequence number of this packet */
	unsigned char ttl;             /* ttl packet left with */
	struct timeval tv ATTRIBUTE_PACKED; /* time packet left */
};

struct IFADDRLIST {
	uint32_t addr;
	char device[sizeof(struct ifreq)];
};


static struct ip *outip;               /* last output (udp) packet */
static struct udphdr *outudp;          /* last output (udp) packet */
static struct outdata *outdata;        /* last output (udp) packet */

static struct icmp *outicmp;           /* last output (icmp) packet */

static int s;                          /* receive (icmp) socket file descriptor */
static int sndsock;                    /* send (udp/icmp) socket file descriptor */

static int packlen;                    /* total length of packet */
static int minpacket;                  /* min ip packet size */
static int maxpacket = 32 * 1024;      /* max ip packet size */

static char *hostname;

static uint16_t ident;

static int waittime = 1;               /* time to wait for response (in seconds) */
static int doipcksum = 1;              /* calculate ip checksums by default */

#define optlen 0



/* Keep in sync with getopt32 call! */
#define OPT_DONT_FRAGMNT (1<<0)    /* F */
#define OPT_USE_ICMP     (1<<1)    /* I */
#define OPT_TTL_FLAG     (1<<2)    /* l */
#define OPT_ADDR_NUM     (1<<3)    /* n */
#define OPT_BYPASS_ROUTE (1<<4)    /* r */
#define OPT_DEBUG        (1<<5)    /* d */
#define OPT_VERBOSE      (1<<6)    /* v */
#define OPT_IP_CHKSUM    (1<<7)    /* x */
#define OPT_TOS          (1<<8)    /* t */
#define OPT_DEVICE       (1<<9)    /* i */
#define OPT_MAX_TTL      (1<<10)   /* m */
#define OPT_PORT         (1<<11)   /* p */
#define OPT_NPROBES      (1<<12)   /* q */
#define OPT_SOURCE       (1<<13)   /* s */
#define OPT_WAITTIME     (1<<14)   /* w */
#define OPT_PAUSE_MS     (1<<15)   /* z */
#define OPT_FIRST_TTL    (1<<16)   /* f */

/* use icmp echo instead of udp packets */
#define useicmp 1 //(option_mask32 & OPT_USE_ICMP)

#if ENABLE_FEATURE_TRACEROUTE_VERBOSE
#define verbose (option_mask32 & OPT_VERBOSE)
#endif
#define nflag   (option_mask32 & OPT_ADDR_NUM)


struct globals {
	/* last inbound (icmp) packet */
	unsigned char packet[512];
	struct sockaddr_storage whereto;        /* Who to try to reach */
	struct sockaddr_storage wherefrom;      /* Who we are */
};

#define G (*ptr_to_globals)

#define packet    (G.packet   )
#define whereto   (G.whereto  )
#define wherefrom (G.wherefrom)
#define gwlist    (G.gwlist   )


/*
 * Return the interface list
 */
//static int
//ifaddrlist(struct IFADDRLIST **ipaddrp)
//{
//	int fd, nipaddr;
//	struct ifreq *ifrp, *ifend, *ifnext;
//	struct sockaddr_in *addr_sin;
//	struct IFADDRLIST *al;
//	struct ifconf ifc;
//	struct ifreq ibuf[(32 * 1024) / sizeof(struct ifreq)], ifr;
//	struct IFADDRLIST *st_ifaddrlist;
//
//	fd = xsocket(AF_INET, SOCK_DGRAM, 0);
//
//	ifc.ifc_len = sizeof(ibuf);
//	ifc.ifc_buf = (caddr_t)ibuf;
//
//	if (ioctl(fd, SIOCGIFCONF, (char *)&ifc) < 0 ||
//	    ifc.ifc_len < sizeof(struct ifreq)) {
//		if (errno == EINVAL)
//			bb_error_msg_and_die(
//			    "SIOCGIFCONF: ifreq struct too small (%d bytes)",
//			    (int)sizeof(ibuf));
//		else
//			bb_perror_msg_and_die("SIOCGIFCONF");
//	}
//	ifrp = ibuf;
//	ifend = (struct ifreq *)((char *)ibuf + ifc.ifc_len);
//
//	nipaddr = 1 + (ifc.ifc_len / sizeof(struct ifreq));
//	st_ifaddrlist = xzalloc(nipaddr * sizeof(struct IFADDRLIST));
//	al = st_ifaddrlist;
//	nipaddr = 0;
//
//	for (; ifrp < ifend; ifrp = ifnext) {
//		ifnext = ifrp + 1;
//		/*
//		 * Need a template to preserve address info that is
//		 * used below to locate the next entry.  (Otherwise,
//		 * SIOCGIFFLAGS stomps over it because the requests
//		 * are returned in a union.)
//		 */
//		strncpy(ifr.ifr_name, ifrp->ifr_name, sizeof(ifr.ifr_name));
//		if (ioctl(fd, SIOCGIFFLAGS, (char *)&ifr) < 0) {
//			if (errno == ENXIO)
//				continue;
//			bb_perror_msg_and_die("SIOCGIFFLAGS: %.*s",
//			    (int)sizeof(ifr.ifr_name), ifr.ifr_name);
//		}
//
//		/* Must be up */
//		if ((ifr.ifr_flags & IFF_UP) == 0)
//			continue;
//
//		safe_strncpy(al->device, ifr.ifr_name, sizeof(ifr.ifr_name) + 1);
//
//		if (ioctl(fd, SIOCGIFADDR, (char *)&ifr) < 0)
//			bb_perror_msg_and_die("SIOCGIFADDR: %s", al->device);
//
//		addr_sin = (struct sockaddr_in *)&ifr.ifr_addr;
//		al->addr = addr_sin->sin_addr.s_addr;
//		++al;
//		++nipaddr;
//	}
//	if (nipaddr == 0)
//		bb_error_msg_and_die ("can't find any network interfaces");
//	(void)close(fd);
//
//	*ipaddrp = st_ifaddrlist;
//	return nipaddr;
//}


static void
setsin(struct sockaddr_in *addr_sin, uint32_t addr)
{
	memset(addr_sin, 0, sizeof(*addr_sin));
	addr_sin->sin_family = AF_INET;
	addr_sin->sin_addr.s_addr = addr;
}


/*
 * Return the source address for the given destination address
 */
//static void
//findsaddr(const struct sockaddr_in *to, struct sockaddr_in *from)
//{
//	int i, n;
//	FILE *f;
//	uint32_t mask;
//	uint32_t dest, tmask;
//	struct IFADDRLIST *al;
//	char buf[256], tdevice[256], device[256];
//
//	f = xfopen("/proc/net/route", "r");
//
//	/* Find the appropriate interface */
//	n = 0;
//	mask = 0;
//	device[0] = '\0';
//	while (fgets(buf, sizeof(buf), f) != NULL) {
//		++n;
//		if (n == 1 && strncmp(buf, "Iface", 5) == 0)
//			continue;
//		if ((i = sscanf(buf, "%255s %x %*s %*s %*s %*s %*s %x",
//		    tdevice, &dest, &tmask)) != 3)
//			bb_error_msg_and_die ("junk in buffer");
//		if ((to->sin_addr.s_addr & tmask) == dest &&
//		    (tmask > mask || mask == 0)) {
//			mask = tmask;
//			strcpy(device, tdevice);
//		}
//	}
//	fclose(f);
//
//	if (device[0] == '\0')
//		bb_error_msg_and_die ("can't find interface");
//
//	/* Get the interface address list */
//	n = ifaddrlist(&al);
//
//	/* Find our appropriate source address */
//	for (i = n; i > 0; --i, ++al)
//		if (strcmp(device, al->device) == 0)
//			break;
//	if (i <= 0)
//		bb_error_msg_and_die("can't find interface %s", device);
//
//	setsin(from, al->addr);
//}

/*
"Usage: %s [-dFIlnrvx] [-g gateway] [-i iface] [-f first_ttl]\n"
"\t[-m max_ttl] [ -p port] [-q nqueries] [-s src_addr] [-t tos]\n"
"\t[-w waittime] [-z pausemsecs] host [packetlen]"

*/

/*
 * Subtract 2 timeval structs:  out = out - in.
 * Out is assumed to be >= in.
 */
static inline void
tvsub(struct timeval *out, struct timeval *in)
{

	if ((out->tv_usec -= in->tv_usec) < 0)   {
		--out->tv_sec;
		out->tv_usec += 1000000;
	}
	out->tv_sec -= in->tv_sec;
}

static int
wait_for_reply(int sock, struct sockaddr_in *fromp, const struct timeval *tp)
{
	fd_set fds;
	struct timeval now, tvwait;
	struct timezone tz;
	int cc = 0;
	socklen_t fromlen = sizeof(*fromp);

	FD_ZERO(&fds);
	FD_SET(sock, &fds);

	tvwait.tv_sec = tp->tv_sec + waittime;
	tvwait.tv_usec = tp->tv_usec;
	(void)gettimeofday(&now, &tz);
	tvsub(&tvwait, &now);

	if (select(sock + 1, &fds, NULL, NULL, &tvwait) > 0)
		cc = recvfrom(sock, (char *)packet, sizeof(packet), 0,
			    (struct sockaddr *)fromp, &fromlen);
	return cc;
}

/*
 * Checksum routine for Internet Protocol family headers (C Version)
 */
static uint16_t
in_cksum(uint16_t *addr, int len)
{
	int nleft = len;
	uint16_t *w = addr;
	uint16_t answer;
	int sum = 0;

	/*
	 *  Our algorithm is simple, using a 32 bit accumulator (sum),
	 *  we add sequential 16 bit words to it, and at the end, fold
	 *  back all the carry bits from the top 16 bits into the lower
	 *  16 bits.
	 */
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if (nleft == 1)
		sum += *(unsigned char *)w;

	/*
	 * add back carry outs from top 16 bits to low 16 bits
	 */
	sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
	sum += (sum >> 16);                     /* add carry */
	answer = ~sum;                          /* truncate to 16 bits */
	return answer;
}


static void
send_probe(int seq, int ttl, struct timeval *tp, int flg)
{
	int cc, i;

	outip->ip_ttl = ttl;
	outip->ip_id = htons(ident + seq);

	/*
	 * In most cases, the kernel will recalculate the ip checksum.
	 * But we must do it anyway so that the udp checksum comes out
	 * right.
	 */
	if (doipcksum) {
		outip->ip_sum =
		    in_cksum((uint16_t *)outip, sizeof(*outip) + optlen);
		if (outip->ip_sum == 0)
			outip->ip_sum = 0xffff;
	}

	/* Payload */
//	outdata->seq = seq;
//	outdata->ttl = ttl;
//	memcpy(&outdata->tv, tp, sizeof(outdata->tv));


	outicmp->icmp_seq = htons(seq);



	if (useicmp) {
		/* Always calculate checksum for icmp packets */
		outicmp->icmp_cksum = 0;
		outicmp->icmp_cksum = in_cksum((uint16_t *)outicmp,
		    packlen - (sizeof(*outip) + optlen));
		if (outicmp->icmp_cksum == 0)
			outicmp->icmp_cksum = 0xffff;
	}
	if(flg)
	{
		for(i = 0; i < 100; i++)
		{
			cc = xsendto(sndsock, (char *)outip,
			    packlen, (struct sockaddr *)&whereto, sizeof(whereto));
			if (cc != packlen)  {
				bb_info_msg("wrote %s %d chars, ret=%d", hostname, packlen, cc);
			}
		}
	} else {
		cc = xsendto(sndsock, (char *)outip,
		    packlen, (struct sockaddr *)&whereto, sizeof(whereto));
		if (cc != packlen)  {
			bb_info_msg("wrote %s %d chars, ret=%d", hostname, packlen, cc);
		}
	}
}

static inline double
deltaT(struct timeval *t1p, struct timeval *t2p)
{
	double dt;
	if(t2p->tv_sec < t1p->tv_sec)
		return 0;
	dt = (double)(t2p->tv_sec - t1p->tv_sec) * 1000.0 +
	     (double)(t2p->tv_usec - t1p->tv_usec) / 1000.0;
	return dt;
}

#if ENABLE_FEATURE_TRACEROUTE_VERBOSE
/*
 * Convert an ICMP "type" field to a printable string.
 */
static inline const char *
pr_type(unsigned char t)
{
	static const char * const ttab[] = {
	"Echo Reply",   "ICMP 1",       "ICMP 2",       "Dest Unreachable",
	"Source Quench", "Redirect",    "ICMP 6",       "ICMP 7",
	"Echo",         "Router Advert", "Router Solicit", "Time Exceeded",
	"Param Problem", "Timestamp",   "Timestamp Reply", "Info Request",
	"Info Reply",   "Mask Request", "Mask Reply"
	};

	if (t > 18)
		return "OUT-OF-RANGE";

	return ttab[t];
}
#endif

static int
packet_ok(unsigned char *buf, int cc, struct sockaddr_in *from, int seq)
{
	struct icmp *icp;
	unsigned char type, code;
	int hlen;
	struct ip *ip;

	ip = (struct ip *) buf;
	hlen = ip->ip_hl << 2;
	if (cc < hlen + ICMP_MINLEN) {
		return 0;
	}
	cc -= hlen;
	icp = (struct icmp *)(buf + hlen);
	type = icp->icmp_type;
	code = icp->icmp_code;

	if ((type == ICMP_TIMXCEED && code == ICMP_TIMXCEED_INTRANS)) {
		struct ip *hip;

		hip = &icp->icmp_ip;
		hlen = hip->ip_hl << 2;
		if (useicmp) {
			struct icmp *hicmp;
			hicmp = (struct icmp *)((unsigned char *)hip + hlen);
			/* XXX 8 is a magic number */
//			printf("\nhicmp->icmp_id = %d  htons(ident) = %d\n", hicmp->icmp_id, htons(ident));
//			printf("\nhicmp->icmp_seq = %d  htons(seq)   = %d\n", hicmp->icmp_seq, htons(seq));
			if (hlen + 8 <= cc &&
			    hip->ip_p == IPPROTO_ICMP &&
			    hicmp->icmp_id == htons(ident) &&
			    hicmp->icmp_seq == htons(seq))
				return -1;
		} 
	}
	return 0;
}

/*
 * Construct an Internet address representation.
 * If the nflag has been supplied, give
 * numeric value, otherwise try for symbolic name.
 */
static inline void
inetname(struct sockaddr_in *from)
{
	const char *n = NULL;
	const char *ina;
	char name[257];

	if (!nflag && from->sin_addr.s_addr != INADDR_ANY) {
		if (INET_rresolve(name, sizeof(name), from, 0x4000,
						0xffffffff) >= 0)
			n = name;
	}
	ina = inet_ntoa(from->sin_addr);
	if (nflag)
		printf(" %s\n", ina);
	else
		printf(" %s (%s)\n", (n ? n : ina), ina);
}

static inline void
print(unsigned char *buf, int cc, struct sockaddr_in *from)
{
	struct ip *ip;
	int hlen;

	ip = (struct ip *) buf;
	hlen = ip->ip_hl << 2;
	cc -= hlen;

	inetname(from);
}


static struct hostinfo *
gethostinfo(const char *host)
{
	int n;
	struct hostent *hp;
	struct hostinfo *hi;
	char **p;
	uint32_t addr, *ap;

	hi = xzalloc(sizeof(*hi));
	addr = inet_addr(host);
	if (addr != 0xffffffff) {
		hi->name = xstrdup(host);
		hi->n = 1;
		hi->addrs = xzalloc(sizeof(hi->addrs[0]));
		hi->addrs[0] = addr;
		return hi;
	}

	hp = xgethostbyname(host);
	if (hp->h_addrtype != AF_INET || hp->h_length != 4)
		bb_perror_msg_and_die("bad host %s", host);
	hi->name = xstrdup(hp->h_name);
	for (n = 0, p = hp->h_addr_list; *p != NULL; ++n, ++p)
		continue;
	hi->n = n;
	hi->addrs = xzalloc(n * sizeof(hi->addrs[0]));
	for (ap = hi->addrs, p = hp->h_addr_list; *p != NULL; ++ap, ++p)
		memcpy(ap, *p, sizeof(*ap));
	return hi;
}

static void
freehostinfo(struct hostinfo *hi)
{
	free(hi->name);
	hi->name = NULL;
	free((char *)hi->addrs);
	free((char *)hi);
}

static int find_gateway(int ttl, struct sockaddr_in* from)
{
	int cc,i;
	struct timeval t1, t2;
	struct timezone tz;
	int gw_addr = 0;
	uint32_t lastaddr = 0;
	int gotlastaddr = 0;

	(void)gettimeofday(&t1, &tz);
	send_probe(ttl, ttl, &t1, 0);
	while ((cc = wait_for_reply(s, from, &t1)) != 0) {
		(void)gettimeofday(&t2, &tz);
		i = packet_ok(packet, cc, from, ttl);
		/* Skip short packet */
		
		if (i == 0)
			continue;
		else if ((i == -1) && (!gotlastaddr ||
			from->sin_addr.s_addr != lastaddr)) {
			gw_addr = from->sin_addr.s_addr >> 24;
			print(packet, cc, from);
			
			if((gw_addr != 10) && (gw_addr != 172) && (gw_addr != 192))
			{
				return -1;
			}
			
			lastaddr = from->sin_addr.s_addr;
			++gotlastaddr;
			break;
		}
	}

	return 0;
	
}

static double probe_time(int seq, int ttl, struct sockaddr_in* from, unsigned int addr)
{
	int cc,i;
	struct timeval t1, t2;
	struct timezone tz;
	double gw_time = 0;

	printf(" %d ", ttl);
	(void)gettimeofday(&t1, &tz);
	send_probe(seq, ttl, &t1, 0);
	while ((cc = wait_for_reply(s, from, &t1)) != 0) {
		(void)gettimeofday(&t2, &tz);
		i = packet_ok(packet, cc, from, seq);
		/* Skip short packet */
		if (i == 0)
			continue;
		if (from->sin_addr.s_addr == addr) {
			gw_time = deltaT(&t1, &t2);
			printf("  %.3f ms\n", gw_time);
			return gw_time? gw_time : -1;
		}
		
	}
	if (cc == 0)
		printf(" *\n");
	return -1;
	
}	

int straceroute_main(int argc, char **argv);
int straceroute_main(int argc, char **argv)
{
	unsigned char *outp;
	struct sockaddr_in *from;
	struct sockaddr_in *to;
	struct hostinfo *hi;
	int ttl, i;
	unsigned op;
	uint16_t off  = 0;
	int max_ttl   = 20;
	int first_ttl = 1;
	
	double  bandwidth   = 0;	
	double  total_time1 = 0;
	double  total_time2 = 0;
	double  tmp_time    = 0;
	uint32_t test_num     = 0;
	uint32_t gateway_ttl  = 0;	
	uint32_t gateway_addr = 0;
	
	struct timeval t1, t2;
	struct timezone tz;

	PTR_TO_GLOBALS = xzalloc(sizeof(G));
	from = (struct sockaddr_in *)&wherefrom;
	to = (struct sockaddr_in *)&whereto;

	//opterr = 0;
	opt_complementary = "x-x";


	op = getopt32(argc, argv, "FIlnrdvx");
	
	minpacket = sizeof(*outip) + sizeof(*outdata) + optlen;

	if (useicmp)
		minpacket += 8;                 /* XXX magic number */
	else

		minpacket += sizeof(*outudp);
	packlen = minpacket;                    /* minimum sized packet */

	/* Process destination and optional packet size */
	switch (argc - optind) {

	case 2:
		packlen = xatoul_range(argv[optind + 1], minpacket, maxpacket);
		/* Fall through */

	case 1:
		hostname = argv[optind];
		hi = gethostinfo(hostname);
		setsin(to, hi->addrs[0]);
		if (hi->n > 1)
			bb_error_msg("warning: %s has multiple addresses; using %s",
				hostname, inet_ntoa(to->sin_addr));
		hostname = hi->name;
		hi->name = NULL;
		freehostinfo(hi);
		break;

	default:
		bb_show_usage();
	}
	
	sleep(10);
	/* Ensure the socket fds won't be 0, 1 or 2 */
	bb_sanitize_stdio();

	s = xsocket(AF_INET, SOCK_RAW, IPPROTO_ICMP);

	if (op & OPT_BYPASS_ROUTE)
		setsockopt(s, SOL_SOCKET, SO_DONTROUTE,
				&const_int_1, sizeof(const_int_1));

	sndsock = xsocket(AF_INET, SOCK_RAW, IPPROTO_RAW);

#ifdef SO_SNDBUF
	if (setsockopt(sndsock, SOL_SOCKET, SO_SNDBUF, &packlen, sizeof(packlen)) < 0) {
		bb_perror_msg_and_die("SO_SNDBUF");
	}
#endif

	/* Revert to non-privileged user after opening sockets */
	xsetgid(getgid());
	xsetuid(getuid());

	ident = (getpid() & 0xffff) | 0x8000;
	outip = xzalloc(packlen);

	outip->ip_v = IPVERSION;
	outip->ip_len = htons(packlen);
	outip->ip_off = htons(off);
	outp = (unsigned char *)(outip + 1);
	outip->ip_dst = to->sin_addr;

	outip->ip_hl = (outp - (unsigned char *)outip) >> 2;

	if (useicmp) {
		outip->ip_p = IPPROTO_ICMP;
		outicmp = (struct icmp *)outp;
		outicmp->icmp_type = ICMP_ECHO;
		outicmp->icmp_id = htons(ident);
	}

	outip->ip_src = from->sin_addr;

	printf("Super-route is starting to %s (%s)", hostname, inet_ntoa(to->sin_addr));

	printf(" to find gateway!\n");
	//find gateway
	for (ttl = first_ttl; ttl <= max_ttl; ++ttl) {
		if(find_gateway(ttl, from) == -1)
		{
			printf("\nFind gateway: %s(ttl=%d)\n", inet_ntoa(from->sin_addr), ttl);
			gateway_addr = from->sin_addr.s_addr;
			gateway_ttl = ttl;
			break;
		}
	}
	
	//find gateway failed
	if(ttl > max_ttl)
	{
		printf("\nFind gateway failed!\n\n");
		printf("\ntbs_cannot_estimate: can't estimate");
		return 0;
	}
	
	printf("Super-route is running to %s (TTL=%d)", hostname, gateway_ttl);
	printf(" to calculate bandwidth!\n");
	// probe time 
	for(i = 0; i < 20; i++)
	{	
		tmp_time = probe_time(21 + i, gateway_ttl, from, gateway_addr);
		if(tmp_time > 0)
		{
			if(i == 0) 
			{
				test_num++;
			}
			else
			{
				test_num++;
				total_time2 += tmp_time; 
			}
			
		}
	}
	printf("packlen = %d  test_num = %d  total_time2 = %6.3f\n", packlen, test_num, total_time2);
	
	bandwidth = ((packlen * 8) * test_num)/(1024 * total_time2 / 1000);
	
//	printf("bandwidth = %f\n", bandwidth);
	printf("tbs_rate_detect %6.3fKbps\n",  bandwidth);
	return 0;
}
