/*
 *	Handle incoming frames
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"

#include <linux/udp.h>
#include <linux/tcp.h>
#include <linux/ip.h>
#include <linux/if_ether.h>
#include <linux/netlink.h>
#include <linux/skbuff.h>
#include <generated/autoconf.h>
#include <net/ipv6.h>

/*
 * TBS_TAG: by pengyao 2012-05-23
 * Desc: add IGMP Snooping support
 */
#if defined(CONFIG_IGMP_SNOOPING)
#include <linux/if_vlan.h>
#include <linux/timer.h>
#include <linux/igmp.h>
#include <linux/ip.h>
#endif
#if defined(CONFIG_BR_MLD_SNOOP)
#include "br_mld.h"
#endif
/**/
#if defined(CONFIG_DHCP_FILTER_SUPPORT)
#include <net/udp.h>
extern int br_filter_enter(struct sk_buff *skb);
extern int enable_filter;
#endif
#if defined(CONFIG_DHCP_MON)
int br_dhcp_packet(struct sk_buff *skb, unsigned char *br_dest, unsigned char *dest);
#endif

extern int g_LocalNetwork_access2gg ;
extern int g_LocalNetwork_access5gg ;

/*
 * TBS_TAG: add pengyao 20130720
 * Desc: for pppoe proxy
 */
union ip_array {
    unsigned int ip_addr;
    unsigned char ip_ar[4];
};
/*
 * TBS_END_TAG
 */
/* Bridge group multicast address 802.1d (pg 51). */
const u8 br_group_address[ETH_ALEN] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x00 };

extern int g_Netgear_url_hijack;
/* private netlink operate command list */
enum {
	NL_CMD_WLAN_WSC_STATUS_CHG,	/* wps status change from unconfigured to configured */
	NL_CMD_WLAN_WSC_LOACALPIN_START,
	NL_CMD_WLAN_WSC_LOACALPIN_FAIL,
	NL_CMD_URL_HIJACK_CNT
};

#ifdef CONFIG_DNS_HTTP_HIJACK
#include <net/ip.h>

int sysctl_dns_hijack; //dns hijack switch
__be32 sysctl_hijack_ip = 0xFA01A8C0; //default is 192.168.1.250

struct dns_header {
    unsigned short 	id;
    unsigned short  flags;

    unsigned short	qdcount; //question count
    unsigned short	ancount; //answer counet
    unsigned short	nscount;
    unsigned short	arcount;	/* Till here it would fit perfectly to a real DNS packet if we had big endian. */
};

static int dns_query_name_sizeof(unsigned char *name)
{
	int size = 0;
	if(NULL == name)
		return size;
	
	while(name[size++] != 0);
	return size;	
}

//2017-03-25 linsd add for ipv6 dns hijack
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
unsigned char sysctl_hijack_ipv6[16] = 
{
	0xfe,0x80,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
};

static int br_ipv6_filter_dns_packet(struct sk_buff *skb,u_int32_t  dns_ttl,int aaaa_flag)
{
	struct net_device *dev;
	struct inet6_dev *idev = NULL;
	struct inet6_ifaddr *ifa = NULL;
	
	struct sk_buff *newskb = NULL;
	struct ethhdr *eth = NULL;
	unsigned char tmpmac[ETH_ALEN] = {0};
	struct ipv6hdr *iph = (struct ipv6hdr *)skb_network_header(skb), *newiph = NULL;
	struct udphdr *udph = (void *)iph + sizeof(struct ipv6hdr), *newudph = NULL;
	u_int32_t udplen, newlen, newudplen;
	struct in6_addr  tmpip;
	__be16 tmpport = 0;
	u_int32_t maxQueryLen = 0;
	u_int32_t queryLen = 0;
	// caculate new UDP packet length
	unsigned int size = 0;
	struct dns_header *dns_hdr =(struct dns_header *)((unsigned char *)udph + sizeof(struct udphdr));
	unsigned char *dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
	queryLen = dns_query_name_sizeof((unsigned char *)dns_query);
	if(aaaa_flag == 1)
		size = queryLen + 10 +  16;
	else
		size = queryLen + 14;

	udplen = skb->len - sizeof(struct ipv6hdr);

	maxQueryLen = udplen - sizeof(struct udphdr) - sizeof(struct dns_header) - 4;

	//printk("###############%s %d maxQueryLen:%d queryLen:%d\n",__FUNCTION__,__LINE__,maxQueryLen,queryLen);
	//ѯȵĴСӦСڻmaxQueryLen
	if(queryLen > maxQueryLen)
	{
		return 0;
	}

	newudplen = udplen + size;
	newlen = sizeof(struct ipv6hdr) + newudplen;

	skb_push(skb, ETH_HLEN);
	newskb = skb_copy_expand(skb, skb_headroom(skb), size, GFP_ATOMIC);
	skb_pull(skb, ETH_HLEN);

	if (NULL == newskb)
		return 0;
	
	skb_put(newskb, size);

	// reverse destinate MAC and source MAC
	eth = eth_hdr(newskb);
	memcpy(tmpmac, eth->h_source, ETH_ALEN);
	memcpy(eth->h_source, eth->h_dest, ETH_ALEN);
	memcpy(eth->h_dest, tmpmac, ETH_ALEN);

	// reverse destinate IP and source IP
	newiph = (struct ipv6hdr *)skb_network_header(newskb);
	tmpip = newiph->saddr;
	newiph->saddr = newiph->daddr;
	newiph->daddr = tmpip;

	newiph->payload_len = htons(newudplen);

	// modify UDP packet values
	newudph = (void *)newiph + sizeof(struct ipv6hdr);
	tmpport = newudph->source;
	newudph->source = newudph->dest;
	newudph->dest = tmpport;
	newudph->len = htons(newudplen);

	
	// modify DNS packet values
	dns_hdr = (struct dns_header *)((unsigned char *)newudph + sizeof(struct udphdr));
	dns_hdr->flags = htons(0x8180); // standard query response, no error
	dns_hdr->ancount = htons(1);
	
	{
		unsigned int *tol;
		unsigned short *dl;
		
		unsigned int query_size = 0;
		if(aaaa_flag == 1)
			query_size = size - 22;
		else	
			query_size = size - 10;
		dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
		unsigned char *dns_response = (unsigned char *)dns_query + query_size;	
		if(aaaa_flag == 1)
			memcpy(dns_response, dns_query, size - 12);
		else	
			memcpy(dns_response, dns_query, size);
		tol = (unsigned int *)(dns_response + query_size);
		//DNSЧʱ
		*tol = htonl(dns_ttl);
		dl = (unsigned short *)(dns_response + query_size + 4);

		//ֱӴbr0ӿȡ·ipַ
		if ((dev  = __dev_get_by_name(&init_net, "br0")) != NULL){
			if ((idev = (struct inet6_dev *)(dev->ip6_ptr)) != NULL)
			{
				//for (ifa = idev->addr_list; ifa != NULL;ifa = ifa->if_next) {
				list_for_each_entry(ifa, &idev->addr_list, if_list){
				//	printk("\nsocop=0x%X\n",ifa->scope);
				//	printk("ipv6 addr=0x%X:%X:%X:%X:%X:%X:%X:%X\n",htons(ifa->addr.s6_addr16[0]),htons(ifa->addr.s6_addr16[1]),htons(ifa->addr.s6_addr16[2]),htons(ifa->addr.s6_addr16[3]),
				//									  htons(ifa->addr.s6_addr16[4]),htons(ifa->addr.s6_addr16[5]),htons(ifa->addr.s6_addr16[6]),htons(ifa->addr.s6_addr16[7]));	
					if(htons(ifa->scope)  == htons(IPV6_ADDR_LINKLOCAL))
					{
				//		printk("\nLocal addr=0x%X:%X:%X:%X:%X:%X:%X:%X\n",htons(ifa->addr.s6_addr16[0]),htons(ifa->addr.s6_addr16[1]),htons(ifa->addr.s6_addr16[2]),htons(ifa->addr.s6_addr16[3]),
				//									  htons(ifa->addr.s6_addr16[4]),htons(ifa->addr.s6_addr16[5]),htons(ifa->addr.s6_addr16[6]),htons(ifa->addr.s6_addr16[7]));	
						memcpy(sysctl_hijack_ipv6,ifa->addr.s6_addr,16);
					}
				}
			}	
		}
	
		if(aaaa_flag == 1)
		{	
			unsigned char  *ip;
			*dl = htons(16);
			ip = (unsigned char  *)(dns_response + query_size + 6);
			//*ip = newiph->saddr;
			memcpy(ip ,sysctl_hijack_ipv6,16); //force to the LAN Manage IPv6
	
			//AAAAipv6ѯ
			//return 1;
		}
		else
		{
			__be32 *ip;
			*dl = htons(4);
			ip = (__be32 *)(dns_response + query_size + 6);
			//*ip = newiph->saddr;
			*ip = (__be32)(sysctl_hijack_ip); //force to the LAN Manage IP			
		}
	}


	/* fix udp checksum if udp checksum was previously calculated */
	if (newudph->check != 0) {
		newudph->check = 0;
		newudph->check = csum_ipv6_magic(&newiph->saddr, &newiph->daddr, newudplen,
				     IPPROTO_UDP, csum_partial((char *)newudph,
						             newudplen, 0));		
	}

	if(newskb->dev != NULL && newskb->dev->netdev_ops != NULL &&
	newskb->dev->netdev_ops->ndo_start_xmit != NULL)
	{
	       //printk("\n++++++++++++++++++++++ndo_start_xmit DNS Response+++++++++++++++++++++\n");
		newskb->dev->netdev_ops->ndo_start_xmit(newskb, newskb->dev);
	}

	return 1;
	
}
#endif 

static int br_filter_dns_packet(struct sk_buff *skb,u_int32_t  dns_ttl)
{
	struct sk_buff *newskb = NULL;
	struct ethhdr *eth = NULL;
	unsigned char tmpmac[ETH_ALEN] = {0};
	struct iphdr *iph = (struct iphdr *)skb_network_header(skb), *newiph = NULL;
	struct udphdr *udph = (void *)iph + iph->ihl * 4, *newudph = NULL;
	u_int32_t udplen, newlen, newudplen;
	__be32 tmpip = 0;
	__be16 tmpport = 0;
	u_int32_t maxQueryLen = 0;
	u_int32_t queryLen = 0;

	// caculate new UDP packet length
	unsigned int size = 0;
	struct dns_header *dns_hdr =(struct dns_header *)((unsigned char *)udph + sizeof(struct udphdr));
	unsigned char *dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
	queryLen = dns_query_name_sizeof((unsigned char *)dns_query);
	size = queryLen + 14;

	udplen = skb->len - iph->ihl*4;

	maxQueryLen = udplen - sizeof(struct udphdr) - sizeof(struct dns_header) - 4;

	//printk("###############%s %d maxQueryLen:%d queryLen:%d\n",__FUNCTION__,__LINE__,maxQueryLen,queryLen);
	//ѯȵĴСӦСڻmaxQueryLen
	if(queryLen > maxQueryLen)
	{
		return 0;
	}


	newudplen = udplen + size;
	newlen = iph->ihl*4 + newudplen;

	skb_push(skb, ETH_HLEN);
	newskb = skb_copy_expand(skb, skb_headroom(skb), size, GFP_ATOMIC);
	skb_pull(skb, ETH_HLEN);

	if (NULL == newskb)
		return 0;

	//newiph = (struct iphdr *)skb_network_header(newskb);
	//newudph = (void *)newiph + newiph->ihl*4;
	skb_put(newskb, size);

	// reverse destinate MAC and source MAC
	eth = eth_hdr(newskb);
	memcpy(tmpmac, eth->h_source, ETH_ALEN);
	memcpy(eth->h_source, eth->h_dest, ETH_ALEN);
	memcpy(eth->h_dest, tmpmac, ETH_ALEN);

	// reverse destinate IP and source IP
	newiph = (struct iphdr *)skb_network_header(newskb);
	tmpip = newiph->saddr;
	newiph->saddr = newiph->daddr;
	newiph->daddr = tmpip;

	// modify the rest of IP packet values
	newiph->tot_len = htons(newlen);
	newiph->id = 0;
	newiph->frag_off = htons(0x4000); // do not fragment
	newiph->ttl = 64;
	
	// modify UDP packet values
	newudph = (void *)newiph + newiph->ihl*4;
	tmpport = newudph->source;
	newudph->source = newudph->dest;
	newudph->dest = tmpport;
	newudph->len = htons(newudplen);

	// modify DNS packet values
	dns_hdr = (struct dns_header *)((unsigned char *)newudph + sizeof(struct udphdr));
	dns_hdr->flags = htons(0x8180); // standard query response, no error
	dns_hdr->ancount = htons(1);

	// add DNS answer
	{
		unsigned int *tol;
		unsigned short *dl;
		__be32 *ip;
		unsigned int query_size = size - 10;
		dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
		unsigned char *dns_response = (unsigned char *)dns_query + query_size;	
		memcpy(dns_response, dns_query, size);
		tol = (unsigned int *)(dns_response + query_size);
		//DNSЧʱ
		*tol = htonl(dns_ttl);
		dl = (unsigned short *)(dns_response + query_size + 4);
		*dl = htons(4);
		ip = (__be32 *)(dns_response + query_size + 6);
		//*ip = newiph->saddr;
		*ip = (__be32)(sysctl_hijack_ip); //force to the LAN Manage IP
	}

	/* fix udp checksum if udp checksum was previously calculated */
	if (newudph->check != 0) {
		newudph->check = 0;
		newudph->check = csum_tcpudp_magic(newiph->saddr, newiph->daddr,
						newudplen, IPPROTO_UDP,
						csum_partial((char *)newudph,
						             newudplen, 0));
	}
	ip_send_check(newiph); //ip checksum


	if(newskb->dev != NULL && newskb->dev->netdev_ops != NULL &&
                newskb->dev->netdev_ops->ndo_start_xmit != NULL)
    {
	    //printk("\n++++++++++++++++++++++ndo_start_xmit DNS Response+++++++++++++++++++++\n");
	    newskb->dev->netdev_ops->ndo_start_xmit(newskb, newskb->dev);
    }

	return 1;

}
static int get_real_domain(unsigned char *target, int max_len, unsigned char *pattern, int length)
{
    unsigned char *ch = NULL;
    int len = 0;
    int i = 0;
    unsigned char *a = target;
    ch = pattern;
    while (0 != *ch && length >= 0)
    {
        len = (int)(*ch);
        ch++;
        length--;
        if (length < 0)
            break;
        while (len-- > 0)
        {
            *(a + i) = *(ch);
            ch++;
            i++;
            length--;
            if (length < 0 || i >= max_len)
                break;
        }
        *(a + i) = '.';
        i++;
        if (i >= max_len)
            break;
    }
    *(a + i - 1) = '\0';
    return 0;
}
static int find_white_domain(char *pattern, int length)
{
    static char *white_list = "www.google.com,as.xboxlive.com,tgs.xboxlive.com,macs.xboxlive.com,as.xboxlive.com.local,tgs.xboxlive.com.local,macs.xboxlive.com.local,updates1.netgear.com,captive.apple.com,www.appleiphonecell.com,www.itools.info,www.ibook.info,www.airport.us,www.thinkdifferent.us,www.apple.com,www.thinkdifferent.us,clients1.google.com,clients3.google.com,connectivitycheck.gstatic.com,detectportal.firefox.com,connectivitycheck.android.com,ocapicep-dev.netgear.com,ocapicep-qa.netgear.com,ocapi.netgear.com,itunes.apple.com,play.google.com,appcom-dev.up.netgear.com,devcom-dev.up.netgear.com,appcom-qa.up.netgear.com,devcom-qa.up.netgear.com,appcom-staging.up.netgear.com,devcom-staging.up.netgear.com,appcom.up.netgear.com,devcom.up.netgear.com,http.fw.updates1.netgear.com,connectivity.samsung.com.cn,www.msftconnecttest.com,connectivitycheck.platfrom.hiclould.com,console.netgear.glassboxdigital.io,play.googleapis.com,apple.com,pingid.com,swrve.com,instabug.com,crashlytics.com,firebase.com,firebaseio.com,glassboxdigital.io,glassboxcdn.com,optimizely.com,s3.amazonaws.com,";
    char *start;
    char *pos = NULL;
    char ch = ',';
    int len = 0;
    int found = 0;

    start = white_list;
    while (1)
    {
        if (*start == ch)
            start++;
        if (*start == '\0')
            break;

        pos = strchr(start, ch);
        if (NULL != pos)
        {
            len = pos - start;

            if (0 == memcmp(start, pattern, len))
            {
                found = 1;
                break;
            }
            start = pos + 1;
        }
        else
        {
            break;
        }
    }

    return found;
}

static int findstring(char *source, char * dest, int length, int cmplen)
{
	if(source == NULL || dest == NULL )
		return 0;
		
	char *s1 = source, *d1 = dest;

	/*Ҫҵ̣ͲҪˣֹڴԽ*/
	if( length != cmplen )
		return 0;

	if ( memcmp(s1,d1,cmplen) == 0 ) 
		return 1;
	else
		return 0;
}

//www.mywifiext.com
//mywifiext.com
//www.mywifiext.net
//mywifiext.net
//routerlogin.net
//www.routerlogin.net
//www.routerlogin.com


//DNSЧʱ:0x3 wwwַ3w
#define DNS_TTL 	0			
static unsigned char domain[] = {0x3,'w','w','w',0x09,'m','y','w','i','f','i','e','x','t',0x3,'c','o','m',0};
static unsigned char domain2[] = {0x09,'m','y','w','i','f','i','e','x','t',0x3,'c','o','m',0};
static unsigned char domain3[] = {0x3,'w','w','w',0x09,'m','y','w','i','f','i','e','x','t',0x3,'n','e','t',0};
static unsigned char domain4[] = {0x09,'m','y','w','i','f','i','e','x','t',0x3,'n','e','t',0};
static unsigned char domainNetgearRegSvr[] = {0x3,'w','w','w',0x07,'n','e','t','g','e','a','r',0x3,'c','o','m',0};//www.netgear.com

//static unsigned char domain5[] = {0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'n','e','t',0};
//static unsigned char domain6[] = {0x3,'w','w','w', 0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'n','e','t',0};
//static unsigned char domain7[] = {0x3,'w','w','w', 0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'c','o','m',0};



//2017-03-25 linsd add for ipv6 dns hijack
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)

int br_ipv6_dns_netgear(struct sk_buff * skb)
{
	struct ipv6hdr *hdr;
	struct udphdr                   *udph;
	unsigned char *ptmp = NULL;
	struct dns_header *dns_hdr;
	unsigned short type = 0;
	int querySize = 0;
	hdr = (struct ipv6hdr *) skb->data;
	unsigned char *dns_quest_name;

	if(hdr->nexthdr != IPPROTO_UDP)
		{
		return 1;
		}
	
	udph = (struct udphdr *)(((unsigned char *)hdr) + sizeof(struct ipv6hdr));

	if( udph->dest != htons(53) )
	{
		return 1;
	}	
	ptmp = (unsigned char *)udph + sizeof(struct udphdr);
	dns_hdr = (struct dns_header *)ptmp;

	//flags is query, and the query count is equal 1
    	if (((dns_hdr->flags & htons(0x8000)) != 0)  || (dns_hdr->qdcount != htons(1))) 
    		{
		return 1;
    		}
	

	
	ptmp = (unsigned char *)dns_hdr + sizeof(struct dns_header);
	dns_quest_name = ptmp;
	querySize = dns_query_name_sizeof(ptmp);
	ptmp += querySize;
	type = *((unsigned short *)ptmp);	//get the  query type	

	//DNSʽʾķDNSĺԴ
	if(ptmp + 4 > skb->tail)
	{
		return 1;
	}
	
	/* 
	AAAAͲѯ֧֣ӦĿֹֹ
	AAAAͲѯ
	*/
	if((type != htons(0x0001)) &&  (type != htons(0x001c)))//"Host Address"query
	{
		return 1;
	}
	
	//printk("br_ipv6_dns_netgear------------->dns_quest_name=%s\n",dns_quest_name);
#if 0	
	if(1== g_Netgear_url_hijack)
	{
		if(findstring(dns_quest_name, domainNetgearRegSvr, querySize, sizeof(domainNetgearRegSvr)))
		{
			extern int tbs_netlink_send_msg( int cmd, char *content );
			tbs_netlink_send_msg( NL_CMD_URL_HIJACK_CNT, "br0" );
			printk("br_ipv6_dns_netgear------------->NL_CMD_URL_HIJACK_CNT\n");
		}
	}
#endif	

	if (sysctl_dns_hijack)
	{
		br_ipv6_filter_dns_packet(skb,0,(type == htons(0x001c)) ? 1:0); //construct the dns replay packet
		return 0;
	}
	//ұӦDNSѯЧʱΪһ
	else if(findstring(dns_quest_name, domain, querySize, sizeof(domain))
			|| findstring(dns_quest_name, domain2, querySize, sizeof(domain2)) 
			|| findstring(dns_quest_name, domain3, querySize, sizeof(domain3))
			|| findstring(dns_quest_name, domain4, querySize, sizeof(domain4))
			//|| findstring(dns_quest_name, domain5, querySize, sizeof(domain5))
			//|| findstring(dns_quest_name, domain6, querySize, sizeof(domain6))
			//|| findstring(dns_quest_name, domain7, querySize, sizeof(domain7))
			)
	{
		br_ipv6_filter_dns_packet(skb,DNS_TTL,(type == htons(0x001c)) ? 1:0); //construct the dns replay packet
		return 0;
	}
	
	return 1;
    		
}
#endif
int br_dns_netgear_shouldHijack(const struct sk_buff *skb)
{
	struct ethhdr *eth = eth_hdr(skb);
	struct iphdr *iph;
	struct udphdr *udph;
	struct dns_header *dns_hdr;
	unsigned char *ptmp;
	unsigned char *dns_quest_name;
	unsigned short type = 0;
	int querySize = 0;
	void *ptr = NULL;

	if (eth->h_proto != htons(ETH_P_IP))
		return 0;
		
	/* Check for possible (maliciously) malformed IP frame (thanks Dave) */
	//iph = ip_hdr(skb);
	iph = (struct iphdr *)(skb->data);
	if(iph->protocol != IPPROTO_UDP)
		return 0;

	//udph = (struct udphdr *)(((unsigned char *)iph) + (iph->ihl<<2));
	ptr = (void *)(skb->data+(iph->ihl<<2));
	udph = (struct udphdr*)(ptr);

	if( udph->dest == htons(53) )// the packet is dns packet
	{
		ptmp = (unsigned char *)udph + sizeof(struct udphdr);
    	dns_hdr = (struct dns_header *)ptmp;
    	
    	if ((dns_hdr->flags & htons(0x8000)) == 0 && dns_hdr->qdcount == htons(1)) //flags is query, and the query count is equal 1
    	{
    		ptmp = (unsigned char *)dns_hdr + sizeof(struct dns_header);
    		dns_quest_name = ptmp;
			querySize = dns_query_name_sizeof(ptmp);
    		ptmp += querySize;
    		type = *((unsigned short *)ptmp);	//get the  query type	

		//DNSʽʾķDNSĺԴ
    		if(ptmp + 4 > skb->tail)
    		{
			/*
    			if(printk_ratelimit())
    			{
    				printk("Error DNS packet!\n");
				}
			*/
				return 0;
    		}
			/* 
			AAAAͲѯ֧֣ӦĿֹֹ
			AAAAͲѯ
			*/
    		if(type == htons(0x0001) || type == htons(0x001c))//"Host Address"query
    		{
    		    //printk("br_dns_netgear_shouldHijack------------->dns_quest_name=%s\n",dns_quest_name);
    		   #if 0 
    		    if(1== g_Netgear_url_hijack)
    		    {
					if(findstring(dns_quest_name, domainNetgearRegSvr, querySize, sizeof(domainNetgearRegSvr)))
					{
	                    extern int tbs_netlink_send_msg( int cmd, char *content );
						tbs_netlink_send_msg( NL_CMD_URL_HIJACK_CNT, "br0" );
						printk("br_dns_netgear_shouldHijack------------->NL_CMD_URL_HIJACK_CNT\n");
				    }

				}
    		    //printk("br_dns_netgear_shouldHijack() dns name=%s\n",dns_quest_name);
				#endif
				//ұӦDNSѯЧʱΪһ
				if( sysctl_dns_hijack ) /*dnsطǰжϣЧ*/
				{
				    //printk("%s(%d):src.ip=%x, src.dst=%x dns name=%s \n", __FUNCTION__, __LINE__, iph->saddr, iph->daddr,dns_quest_name);
					return 1;
				}
				/*ƥҪҵdnsֱƥһquest*/
				else if(findstring(dns_quest_name, domain, querySize, sizeof(domain))
						|| findstring(dns_quest_name, domain2, querySize, sizeof(domain2)) 
						|| findstring(dns_quest_name, domain3, querySize, sizeof(domain3))
						|| findstring(dns_quest_name, domain4, querySize, sizeof(domain4))
						//|| findstring(dns_quest_name, domain5, querySize, sizeof(domain5))
						//|| findstring(dns_quest_name, domain6, querySize, sizeof(domain6))
						//|| findstring(dns_quest_name, domain7, querySize, sizeof(domain7))
						)
				{
				    //printk("%s(%d):src.ip=%x, src.dst=%x dns name=%s \n", __FUNCTION__, __LINE__, iph->saddr, iph->daddr,dns_quest_name);
					return 1;
				}
				else
				{
					return 0;
				}
    		}

    	}
	}
	return 0;
}

int br_dns_netgear(const struct sk_buff *skb)
{
	struct ethhdr *eth = eth_hdr(skb);
	struct iphdr *iph;
	struct udphdr *udph;
	struct dns_header *dns_hdr;
	unsigned char *ptmp;
	unsigned char *dns_quest_name;
	unsigned short type = 0;
	int querySize = 0;

#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
		if(eth->h_proto == htons(ETH_P_IPV6))
		{
			return br_ipv6_dns_netgear(skb);
		}
#endif	

	if (eth->h_proto != htons(ETH_P_IP))
		return 1;
		
	/* Check for possible (maliciously) malformed IP frame (thanks Dave) */
	iph = ip_hdr(skb);
	if(iph->protocol != IPPROTO_UDP)
		return 1;

	udph = (struct udphdr *)(((unsigned char *)iph) + (iph->ihl<<2));

	if( udph->dest == htons(53) )// the packet is dns packet
	{
		ptmp = (unsigned char *)udph + sizeof(struct udphdr);
    	dns_hdr = (struct dns_header *)ptmp;
    	
    	if ((dns_hdr->flags & htons(0x8000)) == 0 && dns_hdr->qdcount == htons(1)) //flags is query, and the query count is equal 1
    	{
    		ptmp = (unsigned char *)dns_hdr + sizeof(struct dns_header);
    		dns_quest_name = ptmp;
			querySize = dns_query_name_sizeof(ptmp);
    		ptmp += querySize;
    		type = *((unsigned short *)ptmp);	//get the  query type	

		//DNSʽʾķDNSĺԴ
    		if(ptmp + 4 > skb->tail)
    		{
			/*
    			if(printk_ratelimit())
    			{
    				printk("Error DNS packet!\n");
				}
			*/
				return 1;
    		}
			/* 
			AAAAͲѯ֧֣ӦĿֹֹ
			AAAAͲѯ
			*/
    		if(type == htons(0x0001) || type == htons(0x001c))//"Host Address"query
    		{
    		    //printk("br_dns_netgear------------->dns_quest_name=%s\n",dns_quest_name);
    		    #if 0
    		    if(1== g_Netgear_url_hijack)
    		    {
					if(findstring(dns_quest_name, domainNetgearRegSvr, querySize, sizeof(domainNetgearRegSvr)))
					{
	                    extern int tbs_netlink_send_msg( int cmd, char *content );
						tbs_netlink_send_msg( NL_CMD_URL_HIJACK_CNT, "br0" );
						printk("br_dns_netgear------------->NL_CMD_URL_HIJACK_CNT\n");
				    }

				}
				#endif
                if (sysctl_dns_hijack)
                {
                        static unsigned char tmp_domain[256] = {0};
                        get_real_domain(tmp_domain, sizeof(tmp_domain), (unsigned char *)((unsigned char *)udph + 20), htons(udph->len) - 20);
                        if (1 == find_white_domain(tmp_domain, sizeof(tmp_domain) - 1))
                        {
                                return 1;
                        }
                }
				//ұӦDNSѯЧʱΪһ
				if( sysctl_dns_hijack ) /*dnsطǰжϣЧ*/
				{
					br_filter_dns_packet(skb,0); //construct the dns replay packet
					return 0;
				}
				/*ƥҪҵdnsֱƥһquest*/
				else if(findstring(dns_quest_name, domain, querySize, sizeof(domain))
						|| findstring(dns_quest_name, domain2, querySize, sizeof(domain2)) 
						|| findstring(dns_quest_name, domain3, querySize, sizeof(domain3))
						|| findstring(dns_quest_name, domain4, querySize, sizeof(domain4))
						//|| findstring(dns_quest_name, domain5, querySize, sizeof(domain5))
						//|| findstring(dns_quest_name, domain6, querySize, sizeof(domain6))
						//|| findstring(dns_quest_name, domain7, querySize, sizeof(domain7))
						)
				{
					br_filter_dns_packet(skb,DNS_TTL); //construct the dns replay packet
					return 0;
				}
				else
				{
					return 1;
				}
    		}

    	}
	}
	return 1;
}
#endif

/*************************************************************************
: ڽ뱾vlan ݣƳVLAN ͷЭջ
: skb
: netif_receive_skb(skb);
ע:
*************************************************************************/
static int br_pass_frame_up_finish(struct sk_buff *skb)
{
#ifdef CONFIG_NETFILTER_DEBUG
	skb->nf_debug = 0;
#endif
#if 1
	/* If pass up to IP, remove VLAN header */
	if (skb->protocol == __constant_htons(ETH_P_8021Q)) {
		unsigned short proto;
		struct vlan_hdr *vhdr = (struct vlan_hdr *)(skb->data);

        /* make sure protocol is correct before passing up */
        proto = vhdr->h_vlan_encapsulated_proto;

		skb = skb_share_check(skb, GFP_ATOMIC);
		if (skb) {			
			memmove(skb->data - ETH_HLEN + VLAN_HLEN,
				skb->data - ETH_HLEN, 12);
			skb_pull(skb, VLAN_HLEN);
			skb->mac_header+= VLAN_HLEN;
		}
		/* skb->protocolΪvlanǩӵЭ֮ͣǰprotocolΪ0x8100(ETH_P_8021Q)*/
		skb->protocol = proto;
		/* TODO: do we need to assign skb->priority? */
	}
#endif
	return netif_receive_skb(skb);
}

static int br_pass_frame_up(struct sk_buff *skb)
{
	struct net_device *indev, *brdev = BR_INPUT_SKB_CB(skb)->brdev;
	struct net_bridge *br = netdev_priv(brdev);
	struct br_cpu_netstats *brstats = this_cpu_ptr(br->stats);

	u64_stats_update_begin(&brstats->syncp);
	brstats->rx_packets++;
	brstats->rx_bytes += skb->len;
	u64_stats_update_end(&brstats->syncp);

	indev = skb->dev;
	skb->dev = brdev;

	return NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN, skb, indev, NULL,
		       br_pass_frame_up_finish);
}


/*
 * TBS_TAG: by pengyao 2012-05-23
 * Desc: add IGMP Snooping support
 */
#if 0
void DumpPacket(char *tag,struct sk_buff *skb)
{
    int i=0;
	char *s;
	s=skb->data-14;
    printk("\n###############################################\n");
    printk("\n****** %s ******\n",tag);
	for(i=0;i<14;i++)
		printk("%02x ",*s++);
    for(i=0;i<skb->len;i++)
    {
             if( i%16 == 0)
                     printk("\n0x%08x:",i);
             if( i%4 == 0)
                     printk(" ");

             printk("%02x ",skb->data[i]);
     }

     printk("\n\n###############################################\n\n");
     printk("\n\n");
}
#endif
#if defined(CONFIG_IGMP_SNOOPING)
static void addr_conv(unsigned char *in, unsigned char * out)
{
    sprintf(out, "%02x%02x%02x%02x%02x%02x", in[0], in[1], in[2], in[3], in[4], in[5]);
}

mac_addr upnp_addr = {{0x01, 0x00, 0x5e, 0x7f, 0xff, 0xfa}};
mac_addr sys1_addr = {{0x01, 0x00, 0x5e, 0x00, 0x00, 0x01}};
mac_addr sys2_addr = {{0x01, 0x00, 0x5e, 0x00, 0x00, 0x02}};
mac_addr ospf1_addr = {{0x01, 0x00, 0x5e, 0x00, 0x00, 0x05}};
mac_addr ospf2_addr = {{0x01, 0x00, 0x5e, 0x00, 0x00, 0x06}};
mac_addr ripv2_addr = {{0x01, 0x00, 0x5e, 0x00, 0x00, 0x09}};
mac_addr sys_addr = {{0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

/* Define ipv6 multicast mac address to let them pass through control filtering.
 * All ipv6 multicast mac addresses start with 0x33 0x33. So control_filter
 * only need to compare the first 2 bytes of the address.
 */
mac_addr ipv6_mc_addr = {{0x33, 0x33, 0x00, 0x00, 0x00, 0x00}}; /* only the left two bytes are significant */

static int control_filter(unsigned char *dest, struct iphdr *pip)
{

#if 0
    if (  ( (!memcmp(dest, &upnp_addr, ETH_ALEN))
           && (!memcmp(&pip->daddr, upnp_ip_addr, sizeof(struct in_addr))))
        || ( (!memcmp(dest, &sys1_addr, ETH_ALEN))
            && (!memcmp(&pip->daddr, sys1_ip_addr, sizeof(struct in_addr))))
        || ( (!memcmp(dest, &sys2_addr, ETH_ALEN))
            && (!memcmp(&pip->daddr, sys2_ip_addr, sizeof(struct in_addr))))
        || ( (!memcmp(dest, &ospf1_addr, ETH_ALEN))
            && (!memcmp(&pip->daddr, ospf1_ip_addr, sizeof(struct in_addr))))
        || ( (!memcmp(dest, &ospf2_addr, ETH_ALEN))
            && (!memcmp(&pip->daddr, ospf2_ip_addr, sizeof(struct in_addr))))
        || ( (!memcmp(dest, &ripv2_addr, ETH_ALEN))
            && (!memcmp(&pip->daddr, ripv2_ip_addr, sizeof(struct in_addr))))
        || (!memcmp(dest, &sys_addr, ETH_ALEN))
        || (!memcmp(dest, &ipv6_mc_addr, 2)) )
        return 0;
#else
    if ((!memcmp(dest, &upnp_addr, ETH_ALEN)) ||
            (!memcmp(dest, &sys1_addr, ETH_ALEN)) ||
            (!memcmp(dest, &sys2_addr, ETH_ALEN)) ||
            (!memcmp(dest, &ospf1_addr, ETH_ALEN)) ||
            (!memcmp(dest, &ospf2_addr, ETH_ALEN)) ||
            (!memcmp(dest, &sys_addr, ETH_ALEN)) ||
            (!memcmp(dest, &ripv2_addr, ETH_ALEN)) ||
            (!memcmp(dest, &ipv6_mc_addr, 2)))
        return 0;
#endif
    else
        return 1;
}

static inline void brcm_conv_ip_to_mac(char *ipa, char *maca)
{
    maca[0] = 0x01;
    maca[1] = 0x00;
    maca[2] = 0x5e;
    maca[3] = 0x7F & ipa[1];
    maca[4] = ipa[2];
    maca[5] = ipa[3];

    return;
}

static void br_process_igmpv3(struct net_bridge *br, struct sk_buff *skb, struct igmpv3_report *report)
{
    struct igmpv3_grec *grec;
    int i;
    struct in_addr src;
    union ip_array igmpv3_mcast;
    int num_src;
    unsigned char tmp[6];
    struct net_bridge_mc_fdb_entry *mc_fdb;

    if(report) {
        grec = &report->grec[0];
        for(i = 0; i < ntohs(report->ngrec); i++) {
            igmpv3_mcast.ip_addr = grec->grec_mca;
            brcm_conv_ip_to_mac(igmpv3_mcast.ip_ar, tmp);
            switch(grec->grec_type) {
                case IGMPV3_MODE_IS_INCLUDE:
                case IGMPV3_CHANGE_TO_INCLUDE:
                case IGMPV3_ALLOW_NEW_SOURCES:
                    for(num_src = 0; num_src < ntohs(grec->grec_nsrcs); num_src++) {
                        src.s_addr = grec->grec_src[num_src];
                        mc_fdb = br_mc_fdb_find(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source,
                            &src, &grec->grec_mca);
#if 0
                        if((NULL != mc_fdb) &&
                                (mc_fdb->src_entry.filt_mode == MCAST_EXCLUDE)) {
                            br_mc_fdb_remove(br, br_port_get_rcu(skb->dev), tmp, eth_hdr(skb)->h_source, SNOOP_EX_CLEAR, &src);
                        }
#else
                        if(NULL != mc_fdb){
                            if (mc_fdb->src_entry.filt_mode == MCAST_EXCLUDE)
                                br_mc_fdb_remove(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source,
                                SNOOP_EX_CLEAR, &src, &grec->grec_mca);
                            else if (ip_hdr(skb)->daddr != IGMP_ALL_HOSTS)
                                br_mc_fdb_update(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source,
                                SNOOP_IN_ADD, &src, &grec->grec_mca);
                        }
#endif
                        else {
                            br_mc_fdb_add_by_host(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source,
                                SNOOP_IN_ADD, &src, &grec->grec_mca);
                        }
                    }

                    if(0 == ntohs(grec->grec_nsrcs)) {
                        src.s_addr = 0;
                        br_mc_fdb_remove(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_EX_CLEAR,
                            &src, &grec->grec_mca);
                    }
                    break;

                case IGMPV3_MODE_IS_EXCLUDE:
                case IGMPV3_CHANGE_TO_EXCLUDE:
                case IGMPV3_BLOCK_OLD_SOURCES:
                    for(num_src = 0; num_src < ntohs(grec->grec_nsrcs); num_src++) {
                        src.s_addr = grec->grec_src[num_src];
                        mc_fdb = br_mc_fdb_find(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, &src,
                            &grec->grec_mca);
#if 0
                        if((NULL != mc_fdb) &&
                                (mc_fdb->src_entry.filt_mode == MCAST_INCLUDE)) {
                            br_mc_fdb_remove(br, br_port_get_rcu(skb->dev), tmp, eth_hdr(skb)->h_source, SNOOP_IN_CLEAR, &src);
                        }
#else
                        if(NULL != mc_fdb) {
                            if (mc_fdb->src_entry.filt_mode == MCAST_INCLUDE)
                                br_mc_fdb_remove(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_IN_CLEAR,
                                &src, &grec->grec_mca);
                            else if (ip_hdr(skb)->daddr != IGMP_ALL_HOSTS)
                                br_mc_fdb_update(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source,  SNOOP_EX_ADD,
                                &src, &grec->grec_mca);
                        }
#endif
                        else {
                            br_mc_fdb_add_by_host(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_EX_ADD,
                                &src, &grec->grec_mca);
                        }
                    }

                    if(0 == ntohs(grec->grec_nsrcs)) {
                        src.s_addr = 0;
                        br_mc_fdb_add_by_host(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_EX_ADD,
                            &src, &grec->grec_mca);
                    }
                    break;
            }
            grec = (struct igmpv3_grec *)((char *)grec + IGMPV3_GRP_REC_SIZE(grec));
        }
    }
    return;
}

void multicast2unicast(struct sk_buff *skb, unsigned char *host)
{
    unsigned char *dest = eth_hdr(skb)->h_dest;
    memcpy(dest,host,6);
    return;
}


int mc_forward(struct net_bridge *br, struct sk_buff *skb, unsigned char *dest,int forward)
{
    struct net_bridge_mc_fdb_entry *dst;
    struct net_bridge_mc_fdb_entry fdb;
    struct net_bridge_mc_fdb_entry *dst_tmp = &fdb;
    int status = 0;
    int filter_past = 0;    //info for exclude mode
    struct sk_buff *skb2;
    struct net_bridge_port *p;
    //unsigned char tmp[6];
    const unsigned char bad_mac[ETH_ALEN];
    struct igmpv3_report *report;
    struct iphdr *pip = ip_hdr(skb);
    struct in_addr src;
    unsigned char igmp_type = 0;
    unsigned char mcaddr[6]={0};
    struct igmphdr *igmp_gen = NULL;

    if (!igmpsnooping)
        return 0;

    memcpy(mcaddr,dest,6);
    /*
       if ((igmpsnooping == SNOOPING_BLOCKING_MODE) && control_filter(dest))
       status = 1;
    */

    if(pip->version != 4)
        return 0;

    if (control_filter(mcaddr, pip))
        status = 1;

	//if(printk_ratelimit())
    	//printk("mc forward  proto = %d  proxy = %d\n", skb->data[9], br->proxy);

    if (skb->data[9] == IPPROTO_IGMP) {
        // For proxy; need to add some intelligence here
        if (!br->proxy) {
            if(pip->ihl == 5) {
                igmp_gen = (struct igmphdr *)&skb->data[20];
            } else {
                igmp_gen = (struct igmphdr *)&skb->data[24];
            }

            igmp_type = igmp_gen->type;
            if (((igmp_type == IGMPV2_HOST_MEMBERSHIP_REPORT) ||
                (igmp_type == IGMP_HOST_MEMBERSHIP_REPORT)) &&
                    (skb->protocol == __constant_htons(ETH_P_IP))) {
                src.s_addr = 0;
                br_mc_fdb_add_by_host(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_EX_ADD,
                    &src, &igmp_gen->group);
                if (ip_hdr(skb)->daddr != IGMP_ALL_HOSTS)
                {
                    br_mc_fdb_update(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_EX_ADD,
                        &src, &igmp_gen->group);
                }
            }
            else if((igmp_type == IGMPV3_HOST_MEMBERSHIP_REPORT) &&
                    (skb->protocol == __constant_htons(ETH_P_IP)))
            {
                report = (struct igmpv3_report *)igmp_gen;

                if(report) {
                br_process_igmpv3(br, skb, report);
                }
            }
            else if (igmp_type == IGMP_HOST_LEAVE_MESSAGE)
            {
                src.s_addr = 0;
                br_mc_fdb_remove(br, br_port_get_rcu(skb->dev), eth_hdr(skb)->h_source, SNOOP_EX_CLEAR,
                    &src, &igmp_gen->group);
            }
            else
                ;
        }
        return 0;
    }

    memset(&fdb, 0, sizeof(fdb));
    memset(&bad_mac, 0, sizeof(bad_mac));
    list_for_each_entry_rcu(dst, &br->mc_list, list) {

        if (pip->daddr == dst->group_addr.s_addr) {

            //filter mode as include
            if((dst->src_entry.filt_mode == MCAST_INCLUDE) &&
                    (pip->saddr == dst->src_entry.src.s_addr)) {
                goto deliver_or_forward;
            }
            //filter mode as exclude
            else if(dst->src_entry.filt_mode == MCAST_EXCLUDE) {
                if(0 == dst->src_entry.src.s_addr){
                    goto deliver_or_forward;
                }
                else if(pip->saddr == dst->src_entry.src.s_addr) {
                    status = 1;
                    filter_past = 0;
                }
            }

            /*
             *because mc_list is host-sorted, another host comes
             *with a different host mac.if it comes to another
             *host fdb-entry, it's time to conclude whether a
             *group source is exclude by pre-host
             */
            if (memcmp(&(dst_tmp->host), &(dst->host), ETH_ALEN)){
                if ((memcmp(&(dst_tmp->host), bad_mac, ETH_ALEN))
                        && (dst_tmp->src_entry.filt_mode == MCAST_EXCLUDE)
                        && filter_past == 1){
                    filter_past = 0;
                    dst = dst_tmp;
                    goto deliver_or_forward;
                }

                dst_tmp = dst ;
            }

            continue;
deliver_or_forward:
            if (!dst->dst->dirty) {
                if (forward){
                  	//printk("include mode br_forward to %s\n", dst->dst->dev->name);
                  	
					struct iphdr *iph = NULL;
	                iph = (struct iphdr *)(skb->data);
					//printk("mc_forward(1)  %s--->%s\n",skb->dev->name,dst->dst->dev->name);
					//printk("mc_forward(1)------->iph->saddr=0x%x --> iph->daddr=0x%x \n",iph->saddr,iph->daddr);
                    int shouldFwd = 1;
					if(1)
					{
				        if(0 == g_LocalNetwork_access2gg)
				        {
				            /*Դ˿ ra1,ĿĶ˿ڲra1 ת*/
							if((0 == memcmp(skb->dev->name, "ra1", 3)) && (0 != memcmp(dst->dst->dev->name, "ra1", 3)))
							{
							    //printk("mc_forward(1) %s--->%s  not flood!!\n",skb->dev->name,dst->dst->dev->name);
								shouldFwd = 0;
							}

							/*Դ˿ڲ ra1 ĿĶ˿ڲra1 ת*/
							if((0 != memcmp(skb->dev->name, "ra1", 3)) && (0 == memcmp(dst->dst->dev->name, "ra1", 3)))
							{
							    //printk("mc_forward(2) %s--->%s  not flood!!\n",skb->dev->name,dst->dst->dev->name);
								shouldFwd = 0;
							}

						}
						
						if(0 == g_LocalNetwork_access5gg)
				        {
				             /*Դ˿ rai1,ĿĶ˿ڲrai1 ת*/
							if((0 == memcmp(skb->dev->name, "rai1", 4)) && (0 != memcmp(dst->dst->dev->name, "rai1", 4)))
							{
							    //printk("mc_forward(3) %s--->%s  not flood!!\n",skb->dev->name,dst->dst->dev->name);
								shouldFwd = 0;
							}

							/*Դ˿ڲ rai1ĿĶ˿ڲrai1 ת*/
							if((0 != memcmp(skb->dev->name, "rai1", 4)) && (0 == memcmp(dst->dst->dev->name, "rai1", 4)))
							{
							    //printk("mc_forward(4) %s--->%s  not flood!!\n",skb->dev->name,dst->dst->dev->name);
							    shouldFwd = 0;
							}

						}

					}  
					if(1 == shouldFwd )
                        br_forward(dst->dst, skb, skb);
                }
                else
                {
                    struct iphdr *iph = NULL;
	                iph = (struct iphdr *)(skb->data);
					//printk("mc_forward(2)  %s--->%s\n",skb->dev->name,dst->dst->dev->name);
					//printk("mc_forward(2)------->iph->saddr=0x%x --> iph->daddr=0x%x \n",iph->saddr,iph->daddr);
	                 
                    skb2 = skb_clone(skb, GFP_ATOMIC);

                    br_deliver(dst->dst, skb2);
                }
            }


            dst->dst->dirty = 1;
            status = 1;

            dst_tmp = dst;
            filter_past = 1;

        }
    }

    if (status) {
        list_for_each_entry_rcu(p, &br->port_list, list) {
            p->dirty = 0;
        }
    }

    if ((!forward) && (status))
        kfree_skb(skb);

    return status;
}

#endif
/*
 * TBS_END_TAG
 */
//add by luozf
#ifdef CONFIG_DHCP_MON
#define OPT_CODE 0
#define OPT_LEN 1
#define OPT_DATA 2
#define OPTION_FIELD		0
#define FILE_FIELD		1
#define SNAME_FIELD		2
/* DHCP option codes (partial list) */
#define DHCP_PADDING		0x00
#define DHCP_SUBNET		0x01
#define DHCP_TIME_OFFSET	0x02
#define DHCP_ROUTER		0x03
#define DHCP_TIME_SERVER	0x04
#define DHCP_NAME_SERVER	0x05
#define DHCP_DNS_SERVER		0x06
#define DHCP_LOG_SERVER		0x07
#define DHCP_COOKIE_SERVER	0x08
#define DHCP_LPR_SERVER		0x09
#define DHCP_HOST_NAME		0x0c
#define DHCP_BOOT_SIZE		0x0d
#define DHCP_DOMAIN_NAME	0x0f
#define DHCP_SWAP_SERVER	0x10
#define DHCP_ROOT_PATH		0x11
#define DHCP_IP_TTL		0x17
#define DHCP_MTU		0x1a
#define DHCP_BROADCAST		0x1c
#define DHCP_STATIC_ROUTE	0x21 
#define DHCP_NTP_SERVER		0x2a
#define DHCP_WINS_SERVER	0x2c
#define DHCP_REQUESTED_IP	0x32
#define DHCP_LEASE_TIME		0x33
#define DHCP_OPTION_OVER	0x34
#define DHCP_MESSAGE_TYPE	0x35
#define DHCP_SERVER_ID		0x36
#define DHCP_PARAM_REQ		0x37
#define DHCP_MESSAGE		0x38
#define DHCP_MAX_SIZE		0x39
#define DHCP_T1			0x3a
#define DHCP_T2			0x3b
#define DHCP_VENDOR		0x3c
#define DHCP_CLIENT_ID		0x3d
#define DHCP_NETBIOS_NODETYPE 0x2e
#define DHCP_NETBIOS_SCOPE 0x2F
#define DHCP_END		0xFF
int br_filter_mangle_udp_packet(struct sk_buff **skb, unsigned int match_offset,unsigned int match_len,unsigned char *rep_buffer, unsigned int rep_len);
struct proc_dir_entry *filter_root = NULL;
static struct proc_dir_entry *res1 = NULL;
static struct proc_dir_entry *res2 = NULL;
int enable_filter = 0;

struct dhcpMessage {
	unsigned char op;
	unsigned char htype;
	unsigned char hlen;
	unsigned char hops;
	unsigned int xid;
	unsigned short secs;
	unsigned short flags;
	unsigned int ciaddr;
	unsigned int yiaddr;
	unsigned int siaddr;
	unsigned int giaddr;
	unsigned char chaddr[16];
	unsigned char sname[64];
	unsigned char file[128];
	unsigned int cookie;
	unsigned char options[308]; /* 312 - cookie */ 
};

struct udp_dhcp_packet {
	struct iphdr ip;
	struct udphdr udp;
	struct dhcpMessage data;
};

typedef struct device_list_table
{
	unsigned char ip[4];
	unsigned char mac[6];
	unsigned char hostname[64];
}device_list_table;

extern int g_dhcpmon_pid;
extern struct sock *netlink_dhcpmon_sock;
rwlock_t  br_netlink_lock;

unsigned char *br_get_dhcp_option(struct dhcpMessage *packet, int code, unsigned char *option_len)
{
	int i, length;
	unsigned char *optionptr = NULL;
	int over = 0, done = 0, curr = OPTION_FIELD;
	
	optionptr = packet->options;
	i = 0;
	length = 308;
	while (!done) 
    {
		if (i >= length) 
        {
			return NULL;
		}

		if (optionptr[i + OPT_CODE] == code) 
        {
			if (i + 1 + optionptr[i + OPT_LEN] >= length) 
            {
				return NULL;
			}
			*option_len = optionptr[i + OPT_LEN];
			return optionptr + i + 2;
		}			
		switch (optionptr[i + OPT_CODE]) 
        {
    		case DHCP_PADDING:
    			i++;
    			break;
    		case DHCP_OPTION_OVER:
    			if (i + 1 + optionptr[i + OPT_LEN] >= length) 
                {
    				return NULL;
    			}
    			over = optionptr[i + 3];
    			i += optionptr[OPT_LEN] + 2;
    			break;
    		case DHCP_END:
    			if (curr == OPTION_FIELD && over & FILE_FIELD) 
                {
    				optionptr = packet->file;
    				i = 0;
    				length = 128;
    				curr = FILE_FIELD;
    			} 
                else if (curr == FILE_FIELD && over & SNAME_FIELD) 
    		    {
    				optionptr = packet->sname;
    				i = 0;
    				length = 64;
    				curr = SNAME_FIELD;
    			} 
                else
                {
                    done = 1;
                }
    			break;
    		default:
    			i += optionptr[OPT_LEN + i] + 2;
		}
	}
	return NULL;
}

static void send_msg_to_user(struct device_list_table device_msg)
{
	int size = 0, ret = 0;
	struct device_list_table *s_device = NULL;
	unsigned char *old_tail = NULL;
	struct sk_buff *skb = NULL;
	struct nlmsghdr *nlh = NULL;

	if(g_dhcpmon_pid == 0)
	{
		printk("dhcpmon_pid's pid = %d.Please run dhcpmon_pid.\n", g_dhcpmon_pid);
	}

	size = NLMSG_SPACE(256);

	skb = alloc_skb(size, GFP_ATOMIC);
	old_tail = skb->tail;

	/* init msg header */
	nlh = NLMSG_PUT(skb, 0, 0, NETLINK_DHCP_TBS, size-sizeof(*nlh));

	/* point to msg data area */
	s_device = NLMSG_DATA(nlh);

	/* fill data for sending */
	memset(s_device, 0, size);
	s_device->ip[0] = device_msg.ip[0];
	s_device->ip[1] = device_msg.ip[1];
	s_device->ip[2] = device_msg.ip[2];
	s_device->ip[3] = device_msg.ip[3];

	s_device->mac[0] = device_msg.mac[0];
	s_device->mac[1] = device_msg.mac[1];
	s_device->mac[2] = device_msg.mac[2];
	s_device->mac[3] = device_msg.mac[3];
	s_device->mac[4] = device_msg.mac[4];
	s_device->mac[5] = device_msg.mac[5];

	memcpy(s_device->hostname, device_msg.hostname, 64);
	
	/* get netlink msg length */
	nlh->nlmsg_len = skb->tail - old_tail;

	NETLINK_CB(skb).pid = 0;
	NETLINK_CB(skb).dst_group = 0;
	//NETLINK_CB(skb).dst_pid = blp_cmm_pid;  //by ZhangYu,kernel 2.6.21 has no member dst_pid

	/* send msg */
	read_lock_bh(&br_netlink_lock);
	ret = netlink_unicast(netlink_dhcpmon_sock, skb, g_dhcpmon_pid, MSG_DONTWAIT);
	read_unlock_bh(&br_netlink_lock);

	//printk("message send! ret=%x \n",ret);

	return;
	
nlmsg_failure:
	printk("Fail to send netlink message.\n");

	if(skb)
		kfree_skb(skb);

	return;

}



/*
fuction: br_dhcp_packet

input: udp data, dhcp option code
return:code offset at skb
*/
int br_dhcp_packet(struct sk_buff *skb, unsigned char *br_dest, unsigned char *dest)
{
	struct iphdr *iph = NULL;
	struct udphdr *udph = NULL;
	struct udp_dhcp_packet *dhcp_packet = NULL;
	unsigned char option_len = 0, *temp = NULL;
#ifdef CONFIG_DHCP_MON
	struct device_list_table g_deviceListTable;
#endif
    if(NULL == skb)
		return 0;
	
	if(NULL == br_dest)
		return 0;

	if(NULL == dest)
		return 0;
	
    //get iph
    iph = (struct iphdr *)skb_network_header(skb);
	
	if(NULL == iph)
	    return 0;
	
    //get udph
	udph=(void *) iph + iph->ihl*4;
	if(NULL == udph)
	    return 0;

    //if dhcp server packet send to me
	if(iph->protocol == IPPROTO_UDP)
    {
		if(ntohs(udph->dest) == 67)
		{
			dhcp_packet =(struct udp_dhcp_packet *)skb->data;
			if(NULL == dhcp_packet)
				return 0;
			if(0x01 == dhcp_packet->data.op)
			{
	            if (NULL != (temp = br_get_dhcp_option(&(dhcp_packet->data), DHCP_HOST_NAME, &option_len))) 
	            {
	            	memset(&g_deviceListTable, 0, sizeof(struct device_list_table));
					
	            	memcpy(g_deviceListTable.mac, dhcp_packet->data.chaddr, 6);
					/*
					printk("mac=%02x:%02x:%02x:%02x:%02x:%02x", g_deviceListTable.mac[0], g_deviceListTable.mac[1],
						g_deviceListTable.mac[2], g_deviceListTable.mac[3], g_deviceListTable.mac[04], g_deviceListTable.mac[5]);
					*/
					if(option_len < 64)
					{
						memcpy(g_deviceListTable.hostname, temp, option_len);
						/*
						printk(" hostname=%s", g_deviceListTable.hostname);
						*/
					}

					if (NULL != (temp = br_get_dhcp_option(&(dhcp_packet->data), DHCP_REQUESTED_IP, &option_len))) 
					{
						memcpy(g_deviceListTable.ip, temp, 4);
						/*
						printk(" ip=%d.%d.%d.%d\n", g_deviceListTable.ip[0], g_deviceListTable.ip[1],
							g_deviceListTable.ip[2], g_deviceListTable.ip[3]);
						*/
					}
					send_msg_to_user(g_deviceListTable);
	            }
			}
		}
	}	
					
	return 0;
}


#endif


/* note: already called with rcu_read_lock */
int br_handle_frame_finish(struct sk_buff *skb)
{
	unsigned char *dest = eth_hdr(skb)->h_dest;
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);
	struct net_bridge *br;
	struct net_bridge_fdb_entry *dst = NULL;
	#ifndef CONFIG_IGMP_SNOOPING
	struct net_bridge_mdb_entry *mdst;
	#endif
	struct sk_buff *skb2;

	if (!p || p->state == BR_STATE_DISABLED)
		goto drop;

	/* insert into forwarding database after filtering to avoid spoofing */
	br = p->br;

#ifdef CONFIG_LOOPBACK_CHECK
	if(br_fdb_update(br, p, eth_hdr(skb)->h_source) == -1)
	{
		goto drop;
	}
#else
	br_fdb_update(br, p, eth_hdr(skb)->h_source);
#endif
#if defined(CONFIG_DHCP_FILTER_SUPPORT)	
	if (dest[0] & 1) {
    	if(enable_filter)
        {           
    		if(br_filter_enter(skb))
    			goto drop;
      
    	}	
}	
#elif defined(CONFIG_DHCP_MON)
		br_dhcp_packet(skb, p->br->dev->dev_addr, dest); //add by luozf
#endif

	if (is_multicast_ether_addr(dest) &&
	    br_multicast_rcv(br, p, skb))
		goto drop;

	if (p->state == BR_STATE_LEARNING)
		goto drop;

	BR_INPUT_SKB_CB(skb)->brdev = br->dev;

	/* The packet skb2 goes to the local host (NULL to skip). */
	skb2 = NULL;

	if (br->dev->flags & IFF_PROMISC)
		skb2 = skb;

/*
 * TBS_TAG: by pengyao 2012-05-23
 * Desc:Add IGMP Snooping support
 */
#if defined(CONFIG_BR_MLD_SNOOP)
	if((0x33 == dest[0]) && (0x33 == dest[1])) {
		br->dev->stats.multicast++;
        skb2 = skb;

		br_mld_process_info(br, skb);
		if (br_mld_mc_forward(br, skb, dest, 1))
		{
            if(!skb2)
                kfree_skb(skb );

            skb = NULL;
		}

	} else
#endif
/*
 * TBS_TAG_END
 */

	if (is_multicast_ether_addr(dest)) {
/*
 * TBS_TAG: by pengyao 2012-05-23
 * Desc:Add IGMP Snooping support
 */
#define MULTICAST_MAC(mac) 	   ((mac[0]==0x01)&&(mac[1]==0x00)&&(mac[2]==0x5e))
#if defined(CONFIG_IGMP_SNOOPING)

        skb2 = skb;

        if (igmpsnooping && MULTICAST_MAC(dest) &&
            ntohs(skb->protocol) == ETH_P_IP)
        {
            //printk("----in %s:%d, igmpsnooping enable\n", __FILE__, __LINE__);
            if (mc_forward(br, skb, (unsigned char*)dest, 1))
            {
                //printk("----in %s:%d, mc_forward success\n", __FILE__, __LINE__);
                if(!skb2)
                    kfree_skb(skb);

                skb = NULL;
            }
            else
            {
                //igmpҪת鵽ӿ
                //printk("----in %s:%d, mc_forward fail\n", __FILE__, __LINE__);
            }
        }
#else
		mdst = br_mdb_get(br, skb);
		if (mdst || BR_INPUT_SKB_CB_MROUTERS_ONLY(skb)) {
			if ((mdst && !hlist_unhashed(&mdst->mglist)) ||
			    br_multicast_is_router(br))
				skb2 = skb;
			br_multicast_forward(mdst, skb, skb2);
			skb = NULL;
			if (!skb2)
				goto out;
		} else
			skb2 = skb;
#endif
/*
 * TBS_END_TAG
 */
		
		br->dev->stats.multicast++;
	} 	
	/*modify by wyh at 20140722 to fix the problem that WIFI NMRP function can not work well while in wireless mode, 
	 * because the MAC in the frame is the Virtual MAC, it is not the real MAC, we must get the realMAC here to ensure the 
	 * '__br_fdb_get' function success*/
	else
	{
#ifdef CONFIG_APPS_LOGIC_NMRP
		if (skb->protocol == __constant_htons(0x0912)) //0x0912 means NMRP packet
		{
			//set the br0's MAC as the dest MAC
			dest = br->dev->dev_addr;
			printk("WYH[%s:%d]:MAC:%0x:%0x:%0x:%0x:%0x:%0x\n", __FUNCTION__, __LINE__, 
				dest[0], dest[1], dest[2], dest[3], dest[4], dest[5]);
			
		}
#endif		
		if ((dst = __br_fdb_get(br, dest)) && dst->is_local) {
			skb2 = skb;
			/* Do not forward the packet since it's local. */
			skb = NULL;
		}
	}
	/*modify by wyh end*/
	
	if (skb) {
		if (dst)
			br_forward(dst->dst, skb, skb2);
		else
			br_flood_forward(br, skb, skb2);
	}

	if (skb2)
		return br_pass_frame_up(skb2);

out:
	return 0;
drop:
	kfree_skb(skb);
	goto out;
}

/* note: already called with rcu_read_lock */
static int br_handle_local_finish(struct sk_buff *skb)
{
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);

	br_fdb_update(p->br, p, eth_hdr(skb)->h_source);
	return 0;	 /* process further */
}

/* Does address match the link local multicast address.
 * 01:80:c2:00:00:0X
 */
static inline int is_link_local(const unsigned char *dest)
{
	__be16 *a = (__be16 *)dest;
	static const __be16 *b = (const __be16 *)br_group_address;
	static const __be16 m = cpu_to_be16(0xfff0);

	return ((a[0] ^ b[0]) | (a[1] ^ b[1]) | ((a[2] ^ b[2]) & m)) == 0;
}


/*
 * Return NULL if skb is handled
 * note: already called with rcu_read_lock
 */
struct sk_buff *br_handle_frame(struct sk_buff *skb)
{
	struct net_bridge_port *p;
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	int (*rhook)(struct sk_buff *skb);

	if (skb->pkt_type == PACKET_LOOPBACK)
		return skb;

	if (!is_valid_ether_addr(eth_hdr(skb)->h_source))
		goto drop;

	skb = skb_share_check(skb, GFP_ATOMIC);
	if (!skb)
		return NULL;

	p = br_port_get_rcu(skb->dev);
	if( p == NULL)
		return NULL;

#ifdef CONFIG_SOCKET_BIND_BRPORT
    /* TBS_TAG: by Sudenghai 2014-2-28*/
    struct iphdr *ip;
    ip = ip_hdr(skb);
    /*skb->recvifΪգֱó*/
	if(skb->recvif[0] == '\0' ||
			/*skb->recvifΪգΪigmpģskb->recvif__netif_receive_skbѾó˶˿豸滻Ϊ豸igmpҪõ*/
			((strcmp(skb->recvif, skb->dev->name) == 0) && (ip->protocol == IPPROTO_IGMP)))
    {    	
    	/*滻豸br0*/
    	strncpy(skb->recvif, p->br->dev->name, sizeof(skb->recvif) - 1);
    }
		
#endif

	if (unlikely(is_link_local(dest))) {
		/* Pause frames shouldn't be passed up by driver anyway */
		if (skb->protocol == htons(ETH_P_PAUSE))
			goto drop;

		/* If STP is turned off, then forward */
		if (p->br->stp_enabled == BR_NO_STP && dest[5] == 0)
			goto forward;

		if (NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN, skb, skb->dev,
			    NULL, br_handle_local_finish))
			return NULL;	/* frame consumed by filter */
		else
			return skb;	/* continue processing */
	}

forward:
	switch (p->state) {
	case BR_STATE_FORWARDING:
#ifdef CONFIG_DNS_HTTP_HIJACK
		/*add by wyh at 2014-04-22 hijack routerlogin dns packet*/
		if(br_dns_netgear(skb) == 0)
		{
			goto drop;
		}
#endif
        
#if 0
        extern unsigned int url_direct_func(struct sk_buff *skb);
		extern int g_Netgear_url_hijack;
		if(1== g_Netgear_url_hijack)//hijack all url
		{
			if(0 == url_direct_func(skb))
			{
			    goto drop;
			}
		}
#endif
        
		rhook = rcu_dereference(br_should_route_hook);
		if (rhook != NULL) {
			if (rhook(skb))
				return skb;
			dest = eth_hdr(skb)->h_dest;
		}
		/* fall through */
	case BR_STATE_LEARNING:
		if (!compare_ether_addr(p->br->dev->dev_addr, dest))
			skb->pkt_type = PACKET_HOST;

		NF_HOOK(NFPROTO_BRIDGE, NF_BR_PRE_ROUTING, skb, skb->dev, NULL,
			br_handle_frame_finish);
		break;
	default:
drop:
		kfree_skb(skb);
	}
	return NULL;
}

