/*
 *	Generic parts
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/llc.h>
#include <net/llc.h>
#include <net/stp.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/types.h>
#include <linux/version.h>
#include "br_private.h"
/*

*/
#ifdef CONFIG_DHCP_MON
int g_dhcpmon_pid = 0;
struct sock *netlink_dhcpmon_sock;
#endif
int g_LocalNetwork_access2gg = 0;
int g_LocalNetwork_access5gg = 0;

int g_gw_addr = 0;
int g_gw_mask = 0;
int g_Netgear_url_hijack = 0;
char g_szNetgear_model_name[32] = {"EX6110"};
char g_szNetgear_sn[32] = {"001122334455"};






#if defined(CONFIG_DHCP_FILTER_SUPPORT)
extern int __init br_filter_init(void);
extern void __exit br_filter_exit(void);
#endif

#ifdef PHY_SWITCH_POWER_ON_OFF
static struct proc_dir_entry * proc_port_power;

/*TODO by Sudenghai 2014-03-10
 *ʵֿswitchÿ˿power on/offͻжϿӵЧ
 */
extern void PHY_power_ops(unsigned int port, int optcode);

static void power_up(unsigned int port)
{
	PHY_power_ops(port, 1);
}
static void power_down(unsigned int port)
{
	PHY_power_ops(port, 0);
}

void switch_port_power_mng(int port,int opcode)
{
    if (port < 0 || port > 4)
    {
        printk(KERN_ERR "%s:unkown port(%d)\n", __func__, port);
        return;
    }

    if (opcode)     /* up */
        power_up(port);
    else
        power_down(port);
}
EXPORT_SYMBOL(switch_port_power_mng);

void dev_to_port_power_mng(char *dev_name, int opcode)
{
	int port;
	int dev_id;
	if(dev_name && (strlen(dev_name) > 3) 
				&& (0 == strncmp(dev_name, "eth", 3)))
	{
		dev_id = dev_name[3] - '0';
		if (dev_id < 1 || dev_id > 5)
	    {
	        printk(KERN_ERR "%s:unkown dev id(%d)\n", __func__, port);
	        return;
	    }

		/*eth1 is always WAN dev*/
#ifdef CONFIG_WAN_AT_P4 
		/*ʵ豸eth1~eht5 ˿4~0ת*/
	    port = ((5 - dev_id) > 0) ? (5 - dev_id) : (dev_id - 5);
#else //CONFIG_WAN_AT_P0
		/*ʵ豸eth1~eht5 ˿0~4ת*/
		port = dev_id-1;
#endif

	    switch_port_power_mng(port, opcode);
	}
}
EXPORT_SYMBOL(dev_to_port_power_mng);

void dev_to_all_port_power_mng(char *dev_name, int opcode)
{
	int port;
	int dev_id;
	if(dev_name && (strlen(dev_name) > 3) 
				&& (0 == strncmp(dev_name, "eth", 3)))
	{
		dev_id = dev_name[3] - '0';
		if (dev_id < 1 || dev_id > 5)
	    {
	        printk(KERN_ERR "%s:unkown dev id(%d)\n", __func__, port);
	        return;
	    }

		/*eth1 - WAN down/up*/
		if(1 == dev_id)
		{
			/*eth1 is always WAN dev*/
#ifdef CONFIG_WAN_AT_P4 
			/*ʵ豸eth1~eht5 ˿4~0ת*/
		    port = ((5 - dev_id) > 0) ? (5 - dev_id) : (dev_id - 5);
#else //CONFIG_WAN_AT_P0
			/*ʵ豸eth1~eht5 ˿0~4ת*/
			port = dev_id-1;
#endif
			switch_port_power_mng(port, opcode);
		}
		else
		{
#ifdef CONFIG_WAN_AT_P4 
			for(port=0; port<4; port++)
#else //CONFIG_WAN_AT_P0
			for(port=4; port>0; port--)
#endif
			{
				switch_port_power_mng(port, opcode);
			}
		}

	}
}
EXPORT_SYMBOL(dev_to_all_port_power_mng);

static int port_power_proc_write( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	int ret;
	char str_buf[256];
	int port = -1;
	int on_off = -1;

	if(len > 255)
	{
		printk("Error. Sample: echo 0 1 > /proc/port_power \n");
		return len;
	}

	copy_from_user(str_buf,buf,len);
	str_buf[len] = '\0';

	ret = sscanf(str_buf,"%d %d", &port, &on_off);
	if(ret == -1 || !(port >= 0 && port <= 4) || !(0 == on_off || 1 == on_off))
	{
		printk("Error. Sample: echo 0 1 > /proc/port_power \n");
		return len;
	}

	//printk("Set:port %d, on_off %d\n", port, on_off);
	switch_port_power_mng(port, on_off);

	return len;
}

#endif

#ifdef CONFIG_LOOPBACK_CHECK
static struct proc_dir_entry * loopbackCheckDir;
static struct proc_dir_entry * loopbackport;
static struct proc_dir_entry * loop_port_restore;
static struct proc_dir_entry * loopback_app_ready;


int gLoopbackPort = 0;
int gLoopPortHangup = 0;
int gLoopbackAppReady = 1;

static int loopbackport_read_proc(char *buf,char **start,off_t off,int count,int *eof,void *data)
{
	int len;
	len = sprintf(buf,"0x%08x\n", gLoopbackPort);
	return len;
}

static int loopbackport_write_proc(struct file *file,const char __user *buffer,
		unsigned long count,void *data)
{
	char tempbuf[8] = {0};
	int port_to_clear = 0;
	unsigned long irqflag;

	if(count == 0)
		return -EFAULT;

	if(copy_from_user(&tempbuf,buffer,sizeof(tempbuf))) {
		printk(KERN_ERR "loopbackport: failed to copy from user\n");
		return -EFAULT;
	}

	port_to_clear = tempbuf[0] - '0';
	if(count == 2 && (port_to_clear >= 0 && port_to_clear <= 9)) {
		gLoopbackPort &= ~(1 << port_to_clear);
		gLoopPortHangup |= (1 << port_to_clear);
	}
	else {
		printk(KERN_ERR "loopbackport: invalid port(%d) to clear, count: %d\n", port_to_clear, count);
		return -EFAULT;
	}
	return count;
}

static int loop_port_restore_read_proc(char *buf,char **start,off_t off,int count,int *eof,void *data)
{
	int len;
	len = sprintf(buf,"0x%08x\n", gLoopPortHangup);
	return len;
}

static int loop_port_restore_write_proc(struct file *file,const char __user *buffer,
		unsigned long count,void *data)
{
    struct net_device *dev = NULL;
    struct net *net = NULL;
	char tempbuf[8] = {0};
	int port_to_clear = 0;
	unsigned long irqflag;
	int powerRestoreId;

	if(count == 0)
		return -EFAULT;

	if(copy_from_user(&tempbuf,buffer,sizeof(tempbuf))) {
		printk(KERN_ERR "loop_port_restore: failed to copy from user\n");
		return -EFAULT;
	}

	/*port is 0~4, WAN is always 0 port and is dev eth1*/
	port_to_clear = tempbuf[0] - '0';
	if(count == 2 && (port_to_clear >= 0 && port_to_clear <= 9)) {
	
		powerRestoreId = port_to_clear+1;//Wan dev name number is 1, Lan dev name number is 2~5
		
		if (gLoopPortHangup & (1 << port_to_clear)) {
			gLoopPortHangup &= ~(1 << port_to_clear);
/*
			local_irq_save(irqflag);
			switch_port_power_mng(powerRestoreId, 1);
			local_irq_restore(irqflag);
*/
			/*netdevice*/
			for_each_net(net) {
				for_each_netdev(net, dev) {
					if(powerRestoreId == (dev->name[3] - '0'))
					{
						rtnl_lock();
						dev_open(dev);
						rtnl_unlock();
					}
				}
			}
		}
	}
	else {
		printk(KERN_ERR "loop_port_restore: invalid port(%d) to restore, count: %d\n", port_to_clear, count);
		return -EFAULT;
	}
	return count;
}

static int loopback_app_ready_read_proc(char *buf,char **start,off_t off,int count,int *eof,void *data)
{
	int len;
	len = sprintf(buf,"%d\n", gLoopbackAppReady);
	return len;
}

static int loopback_app_ready_write_proc(struct file *file,const char __user *buffer,
		unsigned long count,void *data)
{
	char tempbuf[8] = {0};

	if(count == 0)
		return -EFAULT;

	if(copy_from_user(&tempbuf,buffer,sizeof(tempbuf))) {
		printk(KERN_ERR "loopbackport: failed to copy from user\n");
		return -EFAULT;
	}

	gLoopbackAppReady = tempbuf[0] - '0';

	return count;
}

#endif
int (*br_should_route_hook)(struct sk_buff *skb);

static const struct stp_proto br_stp_proto = {
	.rcv	= br_stp_rcv,
};

static struct pernet_operations br_net_ops = {
	.exit	= br_net_exit,
};


/* TBS_TAG: by pengyao 2012-05-23 Desc:support MLD SNOOPING */
#if defined(CONFIG_BR_MLD_SNOOP)
#include "br_mld.h"
#endif

/*
 * TBS_TAG:add by pengyao 2012-05-23
 * Desc: support IGMP SNOOPING
 */
#if defined(CONFIG_IGMP_SNOOPING)
int igmpsnooping = 0;
int multicast2unicastEn = 0;
#endif

#if defined(CONFIG_IGMP_SNOOPING) || defined(CONFIG_BR_MLD_SNOOP)
int br_igmp_debug = 0;
EXPORT_SYMBOL(br_igmp_debug);
int br_mcast_to_unicast = 1;
struct proc_dir_entry *proc_snooping;
static int debug_proc_read(char *page, char **start, off_t off,
			  int count, int *eof, void *data)
{
        char *out = page;
        int len = 0;
        struct net_device *dev = NULL;
        struct net_bridge *bridge = NULL;
        struct net_bridge_mc_fdb_entry *mc_entry = NULL;
        struct net *net = NULL;
        //struct net_device *dev = v;
        struct net_br_mld_mc_fdb_entry *dst = NULL;
        struct net_bridge *br = NULL;

        out += sprintf(out, "debug=%d\n", br_igmp_debug);
        out += sprintf(out, "mc_2_unicast=%d\n", br_mcast_to_unicast);
#ifdef CONFIG_IGMP_SNOOPING
        out += sprintf(out, "igmpsnooping=%d\n", igmpsnooping);
        out += sprintf(out, "filt_mode\t sip\t\t mcast port\t ageing timer\t group\n");
#endif

        /*netdevice*/
        for_each_net(net) {
            for_each_netdev(net, dev) {
                if (dev->priv_flags & IFF_EBRIDGE)
                {
                    bridge = netdev_priv(dev);

#ifdef CONFIG_IGMP_SNOOPING
                    list_for_each_entry(mc_entry, &bridge->mc_list, list) {
                        out += sprintf(out, "%s\t "NIPQUAD_FMT"\t %s\t\t %lu\t\t "
                                    //"%02x:%02x:%02x:%02x:%02x:%02x\n",
                                    NIPQUAD_FMT"\n",
                                    (mc_entry->src_entry.filt_mode == MCAST_EXCLUDE)?"MCAST_EXCLUDE":"MCAST_INCLUDE",
                                    NIPQUAD(mc_entry->src_entry.src.s_addr),
                                    mc_entry->dst->dev->name,
                                    (jiffies + QUERY_TIMEOUT*HZ - mc_entry->tstamp)/HZ,
                                    //mc_entry->addr.addr[0], mc_entry->addr.addr[1], mc_entry->addr.addr[2],
                                    //mc_entry->addr.addr[3], mc_entry->addr.addr[4], mc_entry->addr.addr[5]);
                                    NIPQUAD(mc_entry->group_addr.s_addr));
                    }
#endif

                }   /*IFF_EBRIDGE*/
            }   /*for_each_netdev*/
        }   /*for_each_net*/

#ifdef CONFIG_BR_MLD_SNOOP
        out += sprintf(out, "mldsnooping=%d\n", br_mld_snooping);
        out += sprintf(out, "bridge	device	group		   reporter          mode  source timeout\n");
#endif

         /*netdevice*/
        for_each_net(net) {
            for_each_netdev(net, dev) {
                if (dev->priv_flags & IFF_EBRIDGE)
                {
                        br = netdev_priv(dev);
#ifdef CONFIG_BR_MLD_SNOOP
                        list_for_each_entry_rcu(dst, &br->mld_mc_list, list) {
                               out += sprintf(out, "%s %6s    ", br->dev->name, dst->dst->dev->name);
                               out += sprintf(out, "%02x:%02x:%02x:%02x:%02x:%02x   ",
                               dst->addr.addr[0], dst->addr.addr[1],
                               dst->addr.addr[2], dst->addr.addr[3],
                               dst->addr.addr[4], dst->addr.addr[5]);

                               out += sprintf(out, "%02x:%02x:%02x:%02x:%02x:%02x   ",
                               dst->host.addr[0], dst->host.addr[1],
                               dst->host.addr[2], dst->host.addr[3],
                               dst->host.addr[4], dst->host.addr[5]);

                               out += sprintf(out, "%2s  %04x:%04x:%04x:%04x   %d\n",
                               (dst->src_entry.filt_mode == MCAST_EXCLUDE) ?
                               "EX" : "IN",
                               dst->src_entry.src.s6_addr32[0],
                               dst->src_entry.src.s6_addr32[1],
                               dst->src_entry.src.s6_addr32[2],
                               dst->src_entry.src.s6_addr32[3],
                               (int) (dst->tstamp - jiffies)/HZ);
                        }
#endif
                }   /*IFF_EBRIDGE*/
            }   /*for_each_netdev*/
        }   /*for_each_net*/

        len = out - page;
        len -= off;
        if (len < count) {
            *eof = 1;
            if (len <= 0)
                return 0;
        } else
            len = count;

        *start = page + off;
        return len;
}


static int debug_proc_write( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	int ret;
	char str_buf[256];
	char action[20] = {0};
	int val = 0;

	if(len > 255)
	{
		printk("Error. Sample: echo debug 1 > /proc/debug \n");
		return len;
	}

	if (copy_from_user(str_buf,buf,len))
        return -EFAULT;

	str_buf[len] = '\0';

	ret = sscanf(str_buf, "%s %d", action, (int*)&val);
	if(ret == -1 || val < 0 )
	{
		printk("Error.Sample: echo debug 1 > /proc/debug \n");
		return len;
	}
	if (strcmp(action, "debug") == 0)
	{
    	br_igmp_debug = val;
	}
    else if (strcmp(action, "mc") == 0)
    {
        br_mcast_to_unicast = val;
    }
#ifdef CONFIG_IGMP_SNOOPING
    else if (strcmp(action, "igmpsnooping") == 0)
    {
        igmpsnooping = val;
    }
#endif

#ifdef CONFIG_BR_MLD_SNOOP
    else if (strcmp(action, "mldsnooping") == 0)
    {
        br_mld_snooping = val;
    }
#endif

	return len;
}
#endif  /*CONFIG_IGMP_SNOOPING*/
/*
 * TBS_END_TAG
 */

#ifdef CONFIG_DHCP_MON
static int dhcpmon_pid_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%d", &g_dhcpmon_pid);
	return len;
}

static int dhcpmon_pid_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%d", g_dhcpmon_pid);
	return ret;
}
#endif

static int LocalNetWork_access2gg_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%d", &g_LocalNetwork_access2gg);
	return len;
}

static int LocalNetWork_access2gg_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%d", g_LocalNetwork_access2gg);
	return ret;
}

static int LocalNetWork_access5gg_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%d", &g_LocalNetwork_access5gg);
	return len;
}

static int LocalNetWork_access5gg_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%d", g_LocalNetwork_access5gg);
	return ret;
}

static int Netgear_url_hijack_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%d", &g_Netgear_url_hijack);
	return len;
}

static int Netgear_url_hijack_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%d", g_Netgear_url_hijack);
	return ret;
}

#if 0
extern int re_setup_redirect_url();

static int Netgear_model_name_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%s", &g_szNetgear_model_name);
	re_setup_redirect_url();

	return len;
}

static int Netgear_model_name_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%s", g_szNetgear_model_name);
	return ret;
}

static int Netgear_sn_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%s", &g_szNetgear_sn);
	re_setup_redirect_url();
	return len;
}

static int Netgear_sn_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%s", g_szNetgear_sn);
	return ret;
}

#endif

static int gw_addr_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%x", &g_gw_addr);
	return len;
}

static int gw_addr_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%x", g_gw_addr);
	return ret;
}

static int gw_mask_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%x", &g_gw_mask);
	return len;
}

static int gw_mask_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret=0;
	ret += snprintf(buf, len, "%x\n", g_gw_mask);

	ret += snprintf(buf+ret, len, "NetMask:%x\n", g_gw_addr&g_gw_mask);
	return ret;
}





static int __init br_init(void)
{
	int err;
	struct proc_dir_entry *proc_pid_dhcpmon;
	struct proc_dir_entry *proc_LocalNetWork_access2gg;
	struct proc_dir_entry *proc_LocalNetWork_access5gg;
	struct proc_dir_entry *proc_gw_addr;
	struct proc_dir_entry *proc_gw_mask;
#if 0	
	struct proc_dir_entry *proc_netgear_url_hijack;
	struct proc_dir_entry *proc_netgear_model_name;
	struct proc_dir_entry *proc_netgear_sn;
#endif	
	
	err = stp_proto_register(&br_stp_proto);
	if (err < 0) {
		pr_err("bridge: can't register sap for STP\n");
		return err;
	}

	err = br_fdb_init();
	if (err)
		goto err_out;

	err = register_pernet_subsys(&br_net_ops);
	if (err)
		goto err_out1;

	err = br_netfilter_init();
	if (err)
		goto err_out2;

	err = register_netdevice_notifier(&br_device_notifier);
	if (err)
		goto err_out3;

#ifdef CONFIG_LOOPBACK_CHECK
        loopbackCheckDir = proc_mkdir("loopbackcheck", NULL);
        if ( loopbackCheckDir == NULL )
        {
            printk("create loopback proc error\n");
            return 0;
        }

        if((loopbackport = create_proc_entry("loopbackport",S_IRUGO | S_IWUSR,loopbackCheckDir)) != NULL)
        {
            loopbackport->read_proc = loopbackport_read_proc;
            loopbackport->write_proc = loopbackport_write_proc;
        }

        if((loop_port_restore = create_proc_entry("loop_port_restore",S_IRUGO | S_IWUSR,loopbackCheckDir)) != NULL)
        {
            loop_port_restore->read_proc = loop_port_restore_read_proc;
            loop_port_restore->write_proc = loop_port_restore_write_proc;
        }
        if((loopback_app_ready = create_proc_entry("loopback_app_ready",S_IRUGO | S_IWUSR,loopbackCheckDir)) != NULL)
        {
            loopback_app_ready->read_proc = loopback_app_ready_read_proc;
            loopback_app_ready->write_proc = loopback_app_ready_write_proc;
        }
#endif

	err = br_netlink_init();
	if (err)
		goto err_out4;
	
	brioctl_set(br_ioctl_deviceless_stub);

#if defined(CONFIG_ATM_LANE) || defined(CONFIG_ATM_LANE_MODULE)
	br_fdb_test_addr_hook = br_fdb_test_addr;
#endif
/**/
#if defined(CONFIG_DHCP_FILTER_SUPPORT)
	br_filter_init();
#endif

/* TBS_TAG:add by pengyao 2012-05-23 Desc: support IGMP SNOOPING */
#if defined(CONFIG_IGMP_SNOOPING)
    proc_snooping = create_proc_entry( "multicast_snooping", 0644, init_net.proc_net);
    proc_snooping->read_proc  = debug_proc_read;
    proc_snooping->write_proc = debug_proc_write;
#endif

/* TBS_TAG:add by pengyao 2012-05-23 Desc: support MLD SNOOPING */
#if defined(CONFIG_BR_MLD_SNOOP)
	br_mld_snooping_init();
#endif

#ifdef PHY_SWITCH_POWER_ON_OFF
	proc_port_power = create_proc_entry( "port_power", 0644, NULL);                
	proc_port_power->write_proc  = port_power_proc_write;
#endif

#ifdef CONFIG_DHCP_MON
#ifdef CONFIG_PROC_FS
	proc_pid_dhcpmon = create_proc_entry( "dhcpmon_pid", 0644, NULL);
	if(NULL != proc_pid_dhcpmon)
	{
		proc_pid_dhcpmon->read_proc = dhcpmon_pid_read_proc;
		proc_pid_dhcpmon->write_proc = dhcpmon_pid_write_proc;
	}
#endif



#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,22))
	netlink_dhcpmon_sock = netlink_kernel_create(NETLINK_DHCP_TBS, 0, NULL, THIS_MODULE);
#elif(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24))
	netlink_dhcpmon_sock = netlink_kernel_create(NETLINK_DHCP_TBS, 0, NULL, NULL, THIS_MODULE);
#else
	netlink_dhcpmon_sock = netlink_kernel_create(&init_net, NETLINK_DHCP_TBS, 0, NULL, NULL, THIS_MODULE);
#endif
#endif

	proc_LocalNetWork_access2gg = create_proc_entry( "local_access2gg", 0644, NULL);
	if(NULL != proc_LocalNetWork_access2gg)
	{
		proc_LocalNetWork_access2gg->read_proc  = LocalNetWork_access2gg_read_proc;
		proc_LocalNetWork_access2gg->write_proc = LocalNetWork_access2gg_write_proc;
	}

	proc_LocalNetWork_access5gg = create_proc_entry( "local_access5gg", 0644, NULL);
	if(NULL != proc_LocalNetWork_access5gg)
	{
		proc_LocalNetWork_access5gg->read_proc  = LocalNetWork_access5gg_read_proc;
		proc_LocalNetWork_access5gg->write_proc = LocalNetWork_access5gg_write_proc;
	}


	proc_gw_addr = create_proc_entry( "gw_addr", 0644, NULL);
	if(NULL != proc_gw_addr)
	{
		proc_gw_addr->read_proc  = gw_addr_read_proc;
		proc_gw_addr->write_proc = gw_addr_write_proc;
	}
	
	proc_gw_mask = create_proc_entry( "gw_mask", 0644, NULL);
	if(NULL != proc_gw_mask)
	{
		proc_gw_mask->read_proc  = gw_mask_read_proc;
		proc_gw_mask->write_proc = gw_mask_write_proc;
	}
	
#if 0

	proc_netgear_model_name = create_proc_entry( "netgear_model_name", 0644, NULL);
	if(NULL != proc_netgear_model_name)
	{
		proc_netgear_model_name->read_proc  = Netgear_model_name_read_proc;
		proc_netgear_model_name->write_proc = Netgear_model_name_write_proc;
	}

	proc_netgear_sn = create_proc_entry( "netgear_sn", 0644, NULL);
	if(NULL != proc_netgear_sn)
	{
		proc_netgear_sn->read_proc  = Netgear_sn_read_proc;
		proc_netgear_sn->write_proc = Netgear_sn_write_proc;
	}

	

	extern int redirect_url_init(void);
	redirect_url_init();

	
	proc_netgear_url_hijack = create_proc_entry( "netgear_url_hijack", 0644, NULL);
	if(NULL != proc_netgear_url_hijack)
	{
		proc_netgear_url_hijack->read_proc  = Netgear_url_hijack_read_proc;
		proc_netgear_url_hijack->write_proc = Netgear_url_hijack_write_proc;
	}
#endif

	return 0;

err_out4:
	unregister_netdevice_notifier(&br_device_notifier);
err_out3:
	br_netfilter_fini();
err_out2:
	unregister_pernet_subsys(&br_net_ops);
err_out1:
	br_fdb_fini();
err_out:
	stp_proto_unregister(&br_stp_proto);
	return err;
}

static void __exit br_deinit(void)
{
	stp_proto_unregister(&br_stp_proto);

	br_netlink_fini();
	unregister_netdevice_notifier(&br_device_notifier);
	brioctl_set(NULL);

	unregister_pernet_subsys(&br_net_ops);

	rcu_barrier(); /* Wait for completion of call_rcu()'s */

	br_netfilter_fini();
#if defined(CONFIG_ATM_LANE) || defined(CONFIG_ATM_LANE_MODULE)
	br_fdb_test_addr_hook = NULL;
#endif

	br_fdb_fini();
#if defined(CONFIG_DHCP_FILTER_SUPPORT)
	br_filter_exit();
#endif

}

EXPORT_SYMBOL(br_should_route_hook);

module_init(br_init)
module_exit(br_deinit)
MODULE_LICENSE("GPL");
MODULE_VERSION(BR_VERSION);
