/*
 * engine_fastpath.c
 *
 * Copyright:
 *       Copyright (c) 2002-2006 SFNT Finland Oy.
 *       All rights reserved.
 *
 * This file contains the core of the packet processing engine.  The control
 * logic for the "fastpath" (processing using existing flows) is contained
 * in this file.
 */

#include "sshincludes.h"
#include "engine_internal.h"
#ifdef SSH_IPSEC_TCPENCAP
#include "engine_tcp_encaps.h"
#endif /* SSH_IPSEC_TCPENCAP */
#include "fastpath_accel.h"
#include "fastpath_impl.h"
#include "fastpath_swi.h"

#define SSH_DEBUG_MODULE "SshEngineFastpath"


/* A callback function of this type is called once for each fragment,
   generated by ssh_fastpath_fragment.  The argument `pc' belogns to the
   caller and must not be modified in any ways by the callback
   function. */
typedef void (*SshFastpathFragmentCB)(SshFastpath fastpath,
				      SshEnginePacketContext pc,
				      SshInterceptorPacket pp);

void fastpath_execute_send(SshFastpath fastpath,
			   SshEnginePacketContext pc,
			   SshInterceptorPacket pp);

/* Fragments the given packet into the MTU `mtu'.  The function calls
   the callback function once for each fragment in turn. */

void ssh_fastpath_fragment(SshFastpath fastpath,
			   size_t mtu,
			   SshEnginePacketContext pc,
			   SshInterceptorPacket pp,
			   SshFastpathFragmentCB callback)
{
  SshFastpathFragmentContextStruct fragc;
  Boolean df_on_first_fragment;

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  /* One must note that when ssh_fastpath_fragment is called, 'pc->pp != pp'
     in fastpath_execute_send().

     This is a bit unfortunate, as we use 'pc->protocol_offset' for
     book-keeping during this run. This is doable, as the 'pc' will be
     freed after this. This however is quite unfortunate in the longer
     term, even if it works at the moment. */
  if (pc->protocol_offset != 0)
    {
      /* Remove the media header. */
      if (!ssh_interceptor_packet_delete(pp, 0, pc->protocol_offset))
	{
	  SSH_DEBUG(SSH_D_ERROR, ("packet delete failed. packet dropped."));
	  pc->protocol_offset = 0;
	  pc->pp = NULL;
	  return;
	}
      pc->packet_len -= pc->protocol_offset;
      pc->protocol_offset = 0;
    }
#endif /* not SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  /* Enforce minimum fragment size. */
  if (pc->transform_index != SSH_IPSEC_INVALID_INDEX)
    {
      df_on_first_fragment =
	(pc->u.flow.tr.df_bit_processing != SSH_ENGINE_DF_CLEAR);
    }
  else
    /* No IPSec processing.  Do not do PMTU discovery with the first
       fragment. */
    df_on_first_fragment = FALSE;

  if (pp->protocol == SSH_PROTOCOL_IP4)
    { /* Enforce minimum fragment size for IPv4. */
      if (mtu < SSH_ENGINE_MIN_FIRST_FRAGMENT_V4)
        {
          mtu = SSH_ENGINE_MIN_FIRST_FRAGMENT_V4;
          df_on_first_fragment = FALSE;
        }
    }
  else
    { /* Enforce minimum fragment size for IPv6. */
      if (mtu < SSH_ENGINE_MIN_FIRST_FRAGMENT_V6)
        {
          mtu = SSH_ENGINE_MIN_FIRST_FRAGMENT_V6;
          df_on_first_fragment = FALSE;
        }
    }

  /* Initialize the fragmentation context. */
  if (ssh_fastpath_fragc_init(fastpath, &fragc, pc, mtu, df_on_first_fragment))
    { /* The packet has DF set, and cannot be fragmented.  We must send
         back an error ICMP and drop the packet. */
      SSH_DEBUG(SSH_D_FAIL, ("Sending ICMP Frag Needed mtu=%d", (int) mtu));
#if defined (WITH_IPV6)
      if (pp->protocol == SSH_PROTOCOL_IP6)
        ssh_engine_send_icmp_error(fastpath->engine, pc, 
				   SSH_ICMP6_TYPE_TOOBIG, 0, mtu);
      else
#endif /* WITH_IPV6 */
        ssh_engine_send_icmp_error(fastpath->engine, pc,
                                   (SshUInt8) SSH_ICMP_TYPE_UNREACH,
                                   (SshUInt8) SSH_ICMP_CODE_UNREACH_NEEDFRAG,
                                   (SshUInt16) mtu);
      SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_FRAGDROP);
      return;
    }

  /* Get each fragment in turn and call the callback. */
  while ((pp = ssh_fastpath_fragc_next(fastpath, &fragc)) != NULL)
    {
      /* Update `packet_len' and `media_hdr_len' fields in `pc' for 
	 each fragment. */
      pc->packet_len = ssh_interceptor_packet_len(pp);
#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
      pc->media_hdr_len = 0;
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
      (*callback)(fastpath, pc, pp);
    }

  /* Deinitialize the fragmentation context, freeing the
     original `pp'. */
  ssh_fastpath_fragc_uninit(fastpath, &fragc);
}

/* Step function for executing transforms.  This function implements an
   asynchronous state machine for transform execution. */
void fastpath_execute_transform_step(SshEnginePacketContext pc,
				     SshEngineActionRet ret, 
				     void *context)
{
  SshFastpath fastpath = (void *)context;
  SshInterceptorPacket pp;
  Boolean forward, inbound;

  /* Handle errors from the previous step. */
  if (SSH_PREDICT_FALSE(ret != SSH_ENGINE_RET_OK))
    {
      switch (ret)
	{
	case SSH_ENGINE_RET_FAIL:
	case SSH_ENGINE_RET_DROP:
	case SSH_ENGINE_RET_ERROR:
	  /* An error has occurred.  Pass it to fastpath_packet_continue. */
	  SSH_DEBUG(SSH_D_FAIL, ("execute_transform_step fail ret=%d",
				 (int) ret));
	  fastpath_packet_continue(fastpath, pc, ret);
	  return;
	default:
	  ssh_fatal("ssh_engine_execute_transform_step: unexpected ret=%d",
		    ret);
	}
    }

  /* Read some data to local variables. */
  pp = pc->pp;
  forward = (pc->flags & SSH_ENGINE_PC_FORWARD) != 0;
  inbound = (pc->flags & SSH_ENGINE_FLOW_D_IPSECINCOMING) != 0;

  switch (pc->u.flow.tre_state)
    {
    case SSH_ENGINE_TRE_IN:
#ifdef SSHDIST_IPSEC_TRANSFORM
      /* Perform inbound transforms, if any. */
      if (inbound)
        {
          SSH_ASSERT(pc->transform != 0);
          pc->u.flow.tre_state = SSH_ENGINE_TRE_RESTART;

	  /* Increment transform counter. */
	  SSH_ASSERT(pc->transform_counter < SSH_ENGINE_MAX_TUNNEL_NESTING);
	  pc->transform_counter++;

	  /* Assert that inbound transform is applied at correct 
	     nesting level. */ 
	  SSH_ASSERT(pc->transform_counter == pc->u.flow.tr.nesting_level);
	  /* On release builds, drop packet. */
	  if (pc->transform_counter != pc->u.flow.tr.nesting_level)
	    {
	      ssh_warning("Inbound transform applied at incorrect nesting "
			  "level %d (should be %d), dropping packet!",
			  pc->transform_counter, pc->u.flow.tr.nesting_level);
	      fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_FAIL);
	      return;
	    }

          ssh_fastpath_transform_in(fastpath, pc, &pc->u.flow.tr,
				    fastpath_execute_transform_step,
				    context);
          return;
        }
#endif /* SSHDIST_IPSEC_TRANSFORM */

      pc->u.flow.tre_state = SSH_ENGINE_TRE_OUT;

      /* Initialize transform counter to the nesting level of 
	 the innermost transform. */
      if ((pc->flags & SSH_ENGINE_PC_RESTARTED_OUT) == 0)
	{
	  if (pc->transform != 0)
	    pc->transform_counter = pc->u.flow.tr.nesting_level;
	  else
	    pc->transform_counter = 0;
	}
      /* Fall to next case. */
    case SSH_ENGINE_TRE_OUT:

#ifdef SSHDIST_IPSEC_TRANSFORM
      /* Set next state. */
      if (pc->u.flow.tr.restart_after_tre)
	{
	  pc->u.flow.tre_state = SSH_ENGINE_TRE_RESTART;
	  pc->flags |= SSH_ENGINE_PC_RESTARTED_OUT;
	}
      else
	pc->u.flow.tre_state = SSH_ENGINE_TRE_SEND;

      /* Perform outbound transforms, if any. */
      if (pc->transform != 0)
	{

	  /* Assert that outbound transform is applied at correct 
	     nesting level. */ 
	  SSH_ASSERT(pc->transform_counter == pc->u.flow.tr.nesting_level);
	  /* On release builds, drop packet. */
	  if (pc->transform_counter != pc->u.flow.tr.nesting_level)
	    {
	      ssh_warning("Outbound transform applied at incorrect nesting "
			  "level %d (should be %d), dropping packet!",
			  pc->transform_counter, pc->u.flow.tr.nesting_level);
	      fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_FAIL);
	      return;
	    }
      
	  /* Decrement transform counter. */
	  SSH_ASSERT(pc->transform_counter > 0);
	  pc->transform_counter--;

	  ssh_fastpath_transform_out(fastpath, pc, &pc->u.flow.tr,
				     fastpath_execute_transform_step,
				     context);
	  return;
	}
      else
	{
	  if (pc->transform_counter > 0)
	    {
	      ssh_warning("Attempting to forward a packet that should go"
			  " into outer tunnel, nesting level %d"
			  " (should be %d), dropping packet!",
			  pc->transform_counter, pc->u.flow.tr.nesting_level);
	      fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_FAIL);
	      return;
	    }
	}
#endif /* SSHDIST_IPSEC_TRANSFORM */
	  
      /* Perform NAT straight away if no transform is executing,
	 otherwise NAT is the responsibility of the transform. */
#ifdef SSHDIST_IPSEC_NAT
      if (SSH_PREDICT_FALSE(ssh_fastpath_transform_nat(fastpath, pc, forward) 
			    == FALSE))
	{
	  /* ssh_fastpath_transform_nat may free pc->pp */
	  if (pc->pp != NULL)
	    ssh_interceptor_packet_free(pc->pp);
	  pc->pp = NULL;
	  goto error;
	}
#endif /* SSHDIST_IPSEC_NAT */
     
      /* If there was no outbound transform and packet is forwarded, update
         ttl. If transform is defined, ttl is updated there. */
      if (pp->flags & SSH_PACKET_FROMADAPTER && !pc->u.flow.local)
        {
          if (!fastpath_decrement_ttl(pc))
            {
              if (pc->pp)
                ssh_interceptor_packet_free(pc->pp);
              goto error;
            }
        }

      /* Fall to next case. */
    case SSH_ENGINE_TRE_SEND:
      /* Prepare to send the packet out. 
	 First set the outbound interface number
	 and packet direction appropriately. */
      pp->ifnum_out = pc->u.flow.ifnum;
      pp->flags &= ~(SSH_PACKET_FROMPROTOCOL | SSH_PACKET_FROMADAPTER);
      if (SSH_PREDICT_FALSE(pc->u.flow.local))
        pp->flags |= SSH_PACKET_FROMADAPTER;
      else
        pp->flags |= SSH_PACKET_FROMPROTOCOL;
      
      /* Assert that the packet is sent only after all 
	 nested transforms are applied. */
      SSH_ASSERT(pc->transform_counter == 0);
      
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
      /* Copy the route key selector and ifnum used in route lookup,
	 since the interceptor might have to reroute the packet. */
      pp->route_selector = pc->route_selector;

      SSH_DEBUG(SSH_D_MIDSTART,
		("send media: ifnum_out=%d local=%d",
		 pc->u.flow.ifnum, pc->u.flow.local));
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
      SSH_DEBUG(SSH_D_MIDSTART,
		("send media: ifnum_out=%d local=%d mediatype=%d "
		 "media_hdr_len=%d min_packet_len=%d "
		 "media_protocol=%d",
		 (int) pc->u.flow.ifnum,
		 pc->u.flow.local,
		 pc->u.flow.mediatype,
		 pc->u.flow.media_hdr_len,
		 pc->u.flow.min_packet_len,
		 pc->u.flow.media_protocol));

#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

#ifdef SSH_IPSEC_TCPENCAP
      /* Perform TCP encapsulation if needed */
      if (SSH_PREDICT_TRUE(pc->u.flow.local == 0))
	{
	  SshEngineActionRet ret;
	  ret = ssh_engine_tcp_encaps_process_outbound(fastpath->engine, 
						       pc, pp);
	  if (SSH_PREDICT_FALSE(ret != SSH_ENGINE_RET_OK))
	    {
	      switch (ret)
		{
		case SSH_ENGINE_RET_ERROR:
		  goto error;

		case SSH_ENGINE_RET_DROP:
		default:
		  fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_DROP);
		  return;
		}
	    }
	}
#endif /* SSH_IPSEC_TCPENCAP */

      if (SSH_PREDICT_FALSE(pc->packet_len > pc->u.flow.mtu))
        {
          /* Fragment the packet and send each fragment separately. */
	  SSH_DEBUG(SSH_D_HIGHOK,
		    ("Fragmenting packet to %@ length %d into %d byte frags. ",
		     ssh_ipaddr_render, &pc->dst,
		     pc->packet_len,
		     pc->u.flow.mtu));
          ssh_fastpath_fragment(fastpath,
				pc->u.flow.mtu,
				pc, pp, fastpath_execute_send);
        }
      else
        {
          /* Just send the packet out. */
	  SSH_DEBUG(SSH_D_HIGHOK,
		    ("Sending packet to %@ length %d (can fit %d)",
		     ssh_ipaddr_render, &pc->dst,
		     pc->packet_len,
		     pc->u.flow.mtu));
          fastpath_execute_send(fastpath, pc, pp);
        }
      fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_DEINITIALIZE);
      return;

    case SSH_ENGINE_TRE_RESTART:
      if ((pc->flags & SSH_ENGINE_PC_RESTARTED_OUT) == 0)
	{
	  pc->tunnel_id = pc->u.flow.tr.restart_tunnel_id;
	}
      pc->prev_transform_index = pc->transform_index;
      pc->transform_index = SSH_IPSEC_INVALID_INDEX;
      pc->transform = 0;
      SSH_DEBUG(SSH_D_NICETOKNOW,
		("pc %p prev_transform_index 0x%08lx flags 0x%08lx "
		 "transform_counter %d packet length %d",
		 pc, 
		 (unsigned long) pc->prev_transform_index,
		 (unsigned long) pc->flags, pc->transform_counter,
		 (unsigned int) pc->packet_len));

#ifdef SSHDIST_IPSEC_NAT
      if (SSH_IP_DEFINED(&pc->u.flow.internal_nat_ip)
	  && (pc->pp->protocol == SSH_PROTOCOL_IP4
	      || pc->pp->protocol == SSH_PROTOCOL_IP6))
	{
	  SshInterceptorProtocol pid;

	  /* Internal NAT must be performed only after innermost 
	     tunnel decapsulation. I.e. all tranforms are done. */
	  SSH_ASSERT(pc->transform_counter == pc->u.flow.tr.nesting_level);

	  pid = fastpath_packet_context_pullup(fastpath->engine, pc, NULL);

	  /* Pull up the fields required for engine_execute_nat */
	  if (pid == SSH_PROTOCOL_IP6 || pid == SSH_PROTOCOL_IP4)
	    {
	      if (!ssh_fastpath_execute_nat(pc, TRUE,
					    &pc->u.flow.internal_nat_ip,
					    pc->u.flow.internal_nat_port))
		{
		  /* Drop packet */
		  SSH_DEBUG(SSH_D_FAIL, ("source address NAT failed!"));
		  fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_FAIL);
		  return;
		}
	    }
	  else
	    {
	      /* Drop packet */
	      SSH_DEBUG(SSH_D_FAIL, ("pullup resulted in non-ip protocol"));
	      fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_FAIL);
	      return;
	    }
	  SSH_IP_UNDEFINE(&pc->u.flow.internal_nat_ip);
	}
#endif /* SSHDIST_IPSEC_NAT */

      fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_RESTART);
      return;

    default:
      ssh_fatal("ssh_engine_execute_transform_step: bad state %d",
                pc->u.flow.tre_state);
    }
  SSH_NOTREACHED;

 error:
  /* An error occurred that caused pc->pp to be freed. */
  pc->pp = NULL;
  fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_ERROR);
}

/* This function executes the transform indicated by pc->transform,
   and when done, calls fastpath_packet_continue with
   SSH_ENGINE_RET_ERROR if an error caused pc->pp to become invalid,
   SSH_ENGINE_RET_FAIL on other errors, SSH_ENGINE_RET_DEINITIALIZE to
   indicate that the packet has been sent out, ENGINE_RET_DROP to drop
   the packet, and may use other values defined for the
   SshEngineActionRet type. */

void fastpath_execute_transform(SshEnginePacketContext pc)
{
  pc->u.flow.tre_state = SSH_ENGINE_TRE_IN;
  fastpath_execute_transform_step(pc, SSH_ENGINE_RET_OK, 
				  (void *)pc->engine->fastpath);
}

/* This function is called from the ssh_fastpath_fragment when the
   engine is fragmenting packet before applying transform. */

void fastpath_execute_transform_fragmented(SshFastpath fastpath,
					   SshEnginePacketContext pc,
					   SshInterceptorPacket pp)
{
  SshEnginePacketContext fpc;
  SshEngineActionRet ret = SSH_ENGINE_RET_EXECUTE;
  SshEngineFlowData flow = NULL;

  /* Allocate a fresh packet context for the fragment. */
  fpc = ssh_engine_alloc_pc(fastpath->engine);
  if (fpc == NULL)
    {
#ifdef SSH_IPSEC_STATISTICS
      SshFastpathGlobalStats stats;

      ssh_kernel_critical_section_start(fastpath->stats_critical_section);
      stats = &fastpath->stats[ssh_kernel_get_cpu()];
      stats->in_packets++;
      stats->in_octets_uncomp += ssh_interceptor_packet_len(pp);
      stats->in_octets_comp += ssh_interceptor_packet_len(pp);
      ssh_kernel_critical_section_end(fastpath->stats_critical_section);
#endif /* SSH_IPSEC_STATISTICS */
      ssh_interceptor_packet_free(pp);
      return;
    }

  /* Initialize the packet context from the packet context of the
     original packet. */
  *fpc = *pc;

  /* And initialize it for the fragment. */
  fpc->flags &= ~(SSH_ENGINE_PC_RECURSED | SSH_ENGINE_PC_INUSE);
  fpc->recursed_ret = 0xdeadbeee;
  fpc->pp = pp;
  fpc->pending_packets = NULL;
  fpc->next = NULL;

#ifdef SSH_IPSEC_STATISTICS
  memset(fpc->stat_vec, 0, sizeof(fpc->stat_vec));
#endif /* SSH_IPSEC_STATISTICS */
  if (pp->flags & SSH_ENGINE_P_FIRSTFRAG)
    {
      /* Take the header length from the original packet. */
    }
  else
    {
      /* The non-first fragments have only IP header. */
#if defined (WITH_IPV6)
      if (pp->protocol == SSH_PROTOCOL_IP6)
        pc->hdrlen = SSH_IPH6_HDRLEN;
      else
#endif /* WITH_IPV6 */
        pc->hdrlen = SSH_IPH4_HDRLEN;

      FP_LOCK_WRITE(fastpath);

      /* Copy transform data for non-first fragments. If this happens
	 to fail, drop the fragment. */
      if (pc->flow_index != SSH_IPSEC_INVALID_INDEX)
	flow = FP_GET_FLOW(fastpath, pc->flow_index);

      fastpath_copy_flow_data(fastpath, flow, fpc);

      if (pc->flow_index != SSH_IPSEC_INVALID_INDEX)
	FP_RELEASE_FLOW(fastpath, pc->flow_index);

      if (!fastpath_copy_transform_data(fastpath, fpc))
	ret = SSH_ENGINE_RET_DROP;

      FP_UNLOCK_WRITE(fastpath);
    }

  /* Execute transform. */
  fastpath_packet_continue(fastpath, fpc, ret);
}


typedef enum {
  PKT_DISPATCH, /* refetch all (after defrag, rules or async completion) */
  PKT_ERROR,    /* nothing + deinit */
  PKT_CORRUPT,  /* log and drop */
  PKT_DROP,     /* free interceptor packet + deinit */
  PKT_DEINITIALIZE,

  RETURN_UNLESS_RECURSED
} SshEngineFastPathTarget;

#define FASTPATH_DISPATCH_TGT(_target)				\
  do {                                                          \
     switch ((_target))						\
     {								\
     case PKT_DEINITIALIZE: goto deinitialize;			\
     case PKT_DROP: goto drop;					\
     case PKT_CORRUPT: goto corrupt;				\
     case PKT_ERROR: goto error;				\
     case PKT_DISPATCH: goto dispatch;				\
     case RETURN_UNLESS_RECURSED: goto return_unless_recursed;	\
     default:							\
       SSH_NOTREACHED;						\
       goto deinitialize;                                       \
     }                                                          \
    } while(0)

/* Prepare the packet for passing to the engine from the fastpath.
   This function may be called several times for one packet.
   Returns SSH_ENGINE_RET_RESTART_FLOW_LOOKUP if the packet is now ready
   for the engine. Any other return value indicates that the packet
   should goto dispatch: */
static SshEngineActionRet
engine_packet_prepare_pkt(SshEngine engine, SshEnginePacketContext pc)
{
  SshFastpath fastpath = engine->fastpath;

  /* Check the packet direction flags. These are required for rule
     execution. This is a heavy operation, and hence we avoid
     performing it for every packet. */
  if (pc->pp->flags & SSH_PACKET_FROMPROTOCOL)
    {
      /* An outgoing packet coming from our local stack. */
      pc->pp->flags |= SSH_ENGINE_P_FROMLOCAL;

      /* Check if the packet is directed to our local stack.  We
         check only local IP addresses, not broadcast or multicast
         addresses. */
      if (ssh_engine_ip_is_local(engine, &pc->dst))
        pc->pp->flags |= SSH_ENGINE_P_TOLOCAL;
    }
  else
    {
      /* An incoming packet. */

      /* Check if source is a (spoofed) local IP address or 
	 broadcast / multicast. */
      if (ssh_engine_ip_is_local(engine, &pc->src)
	  || ssh_engine_ip_is_broadcast(engine, &pc->src))
        {
          SSH_DEBUG(SSH_D_NETGARB,
                    ("Incoming packet from a local or multicast address"));
          return SSH_ENGINE_RET_DROP;
        }

      /* Check if internal packet forwarding is disabled. */
      /* Check media level broadcast packets. */
      if (engine->flags & SSH_ENGINE_NO_FORWARDING)
        /* Forward packets using the host's IP stack. */
        pc->pp->flags |= SSH_ENGINE_P_TOLOCAL;
      else if (pc->pp->flags & SSH_PACKET_MEDIABCAST)
        pc->pp->flags |= SSH_ENGINE_P_TOLOCAL;
      /* Check local IP addresses and broadcast / multicast addresses. */
      else if (ssh_engine_ip_is_local(engine, &pc->dst) 
	       || ssh_engine_ip_is_broadcast(engine, &pc->dst))
        pc->pp->flags |= SSH_ENGINE_P_TOLOCAL;
      /* Check the null-address. */
      else if (SSH_IP_IS_NULLADDR(&pc->dst))
        pc->pp->flags |= SSH_ENGINE_P_TOLOCAL;
    }

  /* Detach packet from any operating system context data */
  ssh_interceptor_packet_detach(pc->pp);

  /* If it is a first fragment, cause the packet to be reassembled.
     Non-first fragments can arrive here if handled by fragmagic. */
  if (pc->pp->flags & SSH_ENGINE_P_FIRSTFRAG)
    return ssh_fastpath_fragmagic(fastpath, pc, TRUE);

  return SSH_ENGINE_RET_RESTART_FLOW_LOOKUP;
}


static SshEngineFastPathTarget
engine_packet_process_rule(SshEngine engine,
			   SshEnginePacketContext pc,
			   SshInterceptorPacket pp,
			   SshEngineActionRet *ret)
{
  SSH_DEBUG(SSH_D_MIDOK, ("No flow found for this packet"));

  *ret = engine_packet_prepare_pkt(engine, pc);
  if (*ret != SSH_ENGINE_RET_RESTART_FLOW_LOOKUP)
    return PKT_DISPATCH;

  SSH_ASSERT(pc->flow_index == SSH_IPSEC_INVALID_INDEX);
  engine_rule_packet_handler(engine, pc);
  return RETURN_UNLESS_RECURSED;
}

#define FASTPATH_IS_ICMP_ERROR(pc)			 \
  (((pc)->ipproto == SSH_IPPROTO_ICMP			 \
    && ((pc)->icmp_type == SSH_ICMP_TYPE_UNREACH	 \
	|| (pc)->icmp_type == SSH_ICMP_TYPE_SOURCEQUENCH \
	|| (pc)->icmp_type == SSH_ICMP_TYPE_TIMXCEED	 \
	|| (pc)->icmp_type == SSH_ICMP_TYPE_PARAMPROB))	 \
   || ((pc)->ipproto == SSH_IPPROTO_IPV6ICMP		 \
       && ((pc)->icmp_type == SSH_ICMP6_TYPE_UNREACH	 \
	   || (pc)->icmp_type == SSH_ICMP6_TYPE_TOOBIG	 \
	   || (pc)->icmp_type == SSH_ICMP6_TYPE_TIMXCEED \
	   || (pc)->icmp_type == SSH_ICMP6_TYPE_PARAMPROB)))

static SshEngineFastPathTarget
engine_packet_process_special_flow(SshEngine engine,
				   SshEnginePacketContext pc,
				   SshUInt32 flow_data_flags,
				   SshUInt8 flow_type,
				   SshEngineActionRet *ret)
{
  if (flow_data_flags & SSH_ENGINE_FLOW_D_DANGLING)
    {
      /* Prepare packet for engine */
      *ret = engine_packet_prepare_pkt(engine, pc);
      if (*ret != SSH_ENGINE_RET_RESTART_FLOW_LOOKUP)
	return PKT_DISPATCH;

      /* Pass packet to the engine */
      engine_rule_packet_handler(engine, pc);
      return RETURN_UNLESS_RECURSED;
    }
  else if (flow_data_flags & SSH_ENGINE_FLOW_D_REJECT_INBOUND)
    {
      SSH_DEBUG(SSH_D_LOWOK, ("flow mode set to reject"));

      if (flow_type == SSH_ENGINE_FLOW_TYPE_TCP)
	ssh_engine_send_tcp_rst(engine, pc);

      return PKT_CORRUPT;
    }
  else if (flow_data_flags & SSH_ENGINE_FLOW_D_DROP_PKTS)
    {
      SSH_DEBUG(SSH_D_NICETOKNOW, ("Flow mode set to DROP_PKTS!"));
      return PKT_DROP;
    }

  return PKT_DROP;
}

SSH_FASTTEXT
void fastpath_packet_continue(SshFastpath fastpath,
			      SshEnginePacketContext pc,
			      SshEngineActionRet ret)
{
  SshEngine engine;
  SshInterceptorPacket pp;
  SshEnginePacketData pd;
  SshEngineFlowData flow;
  const unsigned char *ucp;
  SshUInt16 discount;
#ifdef SSH_IPSEC_EXECUTE_PROTOCOL_MONITORS
  SshEngineProtocolMonitorRet mret;
#endif /* SSH_IPSEC_EXECUTE_PROTOCOL_MONITORS */
  Boolean forward, reassemble;
  Boolean skip_flow_id_computation, force_frag_reassembly;
  SshInterceptorProtocol icept_proto;
#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  SshUInt32 next_hop_index;
  SshEngineNextHopData nh;
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  SSH_INTERCEPTOR_STACK_MARK();

  /* Load some values from the context into local variables. */
  engine = pc->engine;
  skip_flow_id_computation = FALSE;
  force_frag_reassembly = FALSE;

  /* Eliminate recursion to limit kernel stack usage.  All calls to
     fastpath_packet_continue are tail recursive.  This makes sure
     that we always eliminate the tail recursion to reduce stack
     consumption. */
  if (SSH_PREDICT_FALSE(pc->flags & SSH_ENGINE_PC_INUSE))
    {
      SSH_DEBUG(SSH_D_HIGHOK, ("eliminating tail recursion"));
      pc->recursed_ret = ret;
      pc->flags |= SSH_ENGINE_PC_RECURSED;
      return;
    }
  else
    pc->flags |= SSH_ENGINE_PC_INUSE;

 dispatch:
  /* We jump here after an operation which could have been asynchronous
     (but was not) completes.  After an asynchronous operation completes,
     we get here due to a call to this function.  We must refetch pp and pd
     every time, because they might change e.g. in reassembly. */
  SSH_DEBUG(SSH_D_HIGHOK, ("dispatch: ret=%d", (int)ret));
  pp = pc->pp;
  if (SSH_PREDICT_TRUE(pp != NULL))
    pd = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEnginePacketData);
  else
    pd = NULL;

#ifdef FASTPATH_ACCELERATOR_CONFIGURED
  if (pc->fastpath_accel_ret == ret)
    {
      fastpath_accel_packet_continue(fastpath->accel, pc, ret);
      goto return_unless_recursed;
    }
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */

  /* Switch according to the continuation action (status return). */
  switch (ret)
    {
    case SSH_ENGINE_RET_DEINITIALIZE:
      SSH_DEBUG(SSH_D_MIDOK, ("DEINITIALIZE"));
      goto deinitialize;
    case SSH_ENGINE_RET_RESTART:
      SSH_DEBUG(SSH_D_MIDOK, ("RESTART"));
      SSH_ASSERT(pc->pp != NULL);
      /* Clear state that was possibly set during the first round. */
      pc->pp->flags &= SSH_ENGINE_P_RESET_MASK;
      goto restart;
    case SSH_ENGINE_RET_IS_SANE:
      SSH_DEBUG(SSH_D_MIDOK, ("IS_SANE"));
      goto packet_is_sane;
    case SSH_ENGINE_RET_RESTART_FLOW_LOOKUP:
      SSH_DEBUG(SSH_D_MIDOK, ("RESTART_FLOW_LOOKUP"));
      goto restart_flow_lookup;
    case SSH_ENGINE_RET_EXECUTE:
      SSH_DEBUG(SSH_D_MIDOK, ("EXECUTE"));
      goto execute_transform;
    case SSH_ENGINE_RET_SEND:
      SSH_DEBUG(SSH_D_MIDOK, ("SEND"));
      goto send;
    case SSH_ENGINE_RET_DROP:
      SSH_DEBUG(SSH_D_MIDOK, ("DROP"));
      if (pc->audit.corruption != SSH_PACKET_CORRUPTION_NONE)
        goto corrupt;
      goto drop;
    case SSH_ENGINE_RET_FAIL:
      SSH_DEBUG(SSH_D_MIDOK, ("FAIL"));
      goto drop;
    case SSH_ENGINE_RET_ERROR:
      SSH_DEBUG(SSH_D_MIDOK, ("ERROR"));
      goto error;
    default:
      ssh_fatal("packet_continue: invalid ret: %d", (int)ret);
    }
  SSH_NOTREACHED;

 restart:

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  SSH_ASSERT(pc->protocol_offset == 0);
#endif /* not SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  /* Clear auditing information */
  pc->audit.corruption = SSH_PACKET_CORRUPTION_NONE;
  pc->audit.spi = 0;
  pc->audit.ip_option = 0;

    /* Loop over context_pullup() we have received all we need
     to execute fastpath_packet_context_is_sane() and/or audit
     the packet as corrupt.. */

  for (icept_proto = fastpath_packet_context_pullup(engine, pc, pd);
       SSH_PREDICT_FALSE(icept_proto != SSH_PROTOCOL_IP4
			 && icept_proto != SSH_PROTOCOL_IP6
			 && icept_proto != SSH_PROTOCOL_ARP);
       icept_proto = fastpath_packet_context_pullup(engine, pc, pd))
    {
      /* Here we rely on ssh_engine_context_pullup() terminating
	 on either ARP, IPv4, IPv6 or NUM_PROTOCOLS. If this is
	 not the case, we will busyloop.. */
      if (icept_proto == SSH_PROTOCOL_NUM_PROTOCOLS)
	{
	  /* Refetch `pp' from `pc' since ssh_engine_context_pullup
	     could have freed it.  In that case `pc->pp' has the value
	     NULL. */
	  pp = pc->pp;
	  goto drop;
	}
    }

  /* Perform elementary sanity checks on the final pulled up packet. */
  pc->audit.corruption = fastpath_packet_context_is_sane(engine, icept_proto,
							 pc->pp, pc,
							 &pc->audit.ip_option);

  if (SSH_PREDICT_FALSE(pc->audit.corruption != SSH_PACKET_CORRUPTION_NONE))
    goto corrupt;

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  /* ARP packets are passed through the engine without additional
     processing */
  if (SSH_PREDICT_FALSE(icept_proto == SSH_PROTOCOL_ARP)
#if defined (WITH_IPV6)
      || SSH_PREDICT_FALSE((icept_proto == SSH_PROTOCOL_IP6)
	  && pc->ipproto == SSH_IPPROTO_IPV6ICMP
	  && (pc->icmp_type == SSH_ICMP6_TYPE_NEIGHBOR_ADVERTISEMENT
	      || pc->icmp_type == SSH_ICMP6_TYPE_NEIGHBOR_SOLICITATION))
#endif /* WITH_IPV6 */
      )
    {
      /* Detach packet from any operating system context data */
      ssh_interceptor_packet_detach(pc->pp);

      engine_address_resolution(engine, pc);
      goto return_unless_recursed;
    }
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

 packet_is_sane:
  /* Only IPv4 and IPv6 protocols get here. */
  SSH_ASSERT(pp != NULL);
  SSH_ASSERT(pp->protocol == SSH_PROTOCOL_IP4 ||
             pp->protocol == SSH_PROTOCOL_IP6);

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  SSH_ASSERT(pc->protocol_offset == 0);
#endif /* not SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  ucp = ssh_interceptor_packet_pullup_read(pp, SSH_IPH4_HDRLEN);
  if (SSH_PREDICT_FALSE(!ucp))
    {
      pc->pp = NULL;
      goto error;
    }

  pc->flags |= engine->pc_flags;

#ifdef SSH_IPSEC_TCPENCAP
 perform_fragmagic:
#endif /* SSH_IPSEC_TCPENCAP */

  /* Perform fragment magic and/or reassembly. */
  if (SSH_PREDICT_FALSE((pp->flags & SSH_ENGINE_P_ISFRAG)
			&& (!(pp->flags & SSH_ENGINE_P_FIRSTFRAG)
			    || force_frag_reassembly)))
    {
      /* Perform fragment magic (flow id lookup) or reassembly as
         needed.  Note that first fragments are handled separately
         below after their flow is known. */
      ret = ssh_fastpath_fragmagic(fastpath, pc, force_frag_reassembly);
      if (ret != SSH_ENGINE_RET_RESTART_FLOW_LOOKUP)
        goto dispatch;
    }
  else
    {
      /* Compute the flow id and a 32-bit hash value for it.  However,
         if the packet was pending, then its flow id has already been
         computed, and must not be recomputed (it may have been set by
         fragmagic). */
      if (SSH_PREDICT_TRUE(!skip_flow_id_computation))
        {
          if (SSH_PREDICT_FALSE(!(*engine->flow_id_hash)(fastpath, pc, 
							 pc->pp, 
							 pc->tunnel_id, 
							 pc->flow_id)))
	    {
	      SSH_DEBUG(SSH_D_FAIL, ("computing flow id failed"));
	      goto error;
	    }
	}
    }
  skip_flow_id_computation = FALSE; /* In case we restart repeatedly. */

 restart_flow_lookup:
  FP_LOCK_WRITE(fastpath);

  /* This restarts flow lookup.  We jump here after a packet was passed to
     policy decisions, and the policy processing created a new flow for the
     packet.  Note that an asynchronous call may have been made in between,
     and local variables may not be valid. */
  
  flow = fastpath_lookup_flow(fastpath, pc);

  if (SSH_PREDICT_FALSE(flow == NULL))
    {
      FP_UNLOCK_WRITE(fastpath);

      /* Handle flow lookup error. */
      if (pc->pp == NULL)
	{
	  SSH_DEBUG(SSH_D_FAIL, ("Flow lookup error, dropping packet."));
	  goto drop;
	}

      if ((pp->flags & SSH_ENGINE_P_ISFRAG) &&
	  !(pp->flags & SSH_ENGINE_P_FIRSTFRAG))
        {
          SSH_DEBUG(SSH_D_NETGARB,
                    ("dropping non-first fragment without a matching flow"));
          goto drop;
        }

      pc->flow_index = SSH_IPSEC_INVALID_INDEX;

      /* No flow exists for this packet. */
#ifdef SSH_IPSEC_TCPENCAP






      switch (ssh_engine_tcp_encaps_process_noflow(engine, pc, pp))
        {
          /* Packet is stolen by TCP encaps code */
        case SSH_ENGINE_RET_ASYNC:



	  goto deinitialize;

          /* Packet was decapsulated */
        case SSH_ENGINE_RET_RESTART:
          goto restart;

	  /* Need to reassemble packet before continuing processing */
	case SSH_ENGINE_RET_RESTART_FLOW_LOOKUP:
	  force_frag_reassembly = TRUE;
	  goto perform_fragmagic;

          /* Error */
        case SSH_ENGINE_RET_ERROR:
          SSH_DEBUG(SSH_D_FAIL, ("engine_tcp_encaps_process_inbound() "
                                 "failed, dropping packet"));
	  goto deinitialize;

          /* Drop packet (inbound handshake or non-first triggering packet) */
        case SSH_ENGINE_RET_DROP:
          goto drop;

          /* Let the packet continue */
        case SSH_ENGINE_RET_OK:
          break;

        default:
          SSH_NOTREACHED;
	  goto deinitialize;
        }
#endif /* SSH_IPSEC_TCPENCAP */

      FASTPATH_DISPATCH_TGT(engine_packet_process_rule(engine, pc, pp, &ret));
      SSH_NOTREACHED; /* process rule will always result to jump */
    }

  SSH_ASSERT(flow->data_flags & SSH_ENGINE_FLOW_D_VALID);
  SSH_ASSERT(pc->flow_index != SSH_IPSEC_INVALID_INDEX);

  pc->flow_generation = flow->generation;
  pc->flags |= (flow->data_flags & SSH_ENGINE_FLOW_D_COPY_MASK);
  forward = (pc->flags & SSH_ENGINE_PC_FORWARD) != 0;

  SSH_DEBUG(SSH_D_MIDOK,
            ("Found %s flow %d: %@:%u->%@:%u",
             (pc->flags&SSH_ENGINE_PC_FORWARD?"forward":"backward"),
             (int)pc->flow_index,
             ssh_ipaddr_render,&flow->src_ip,
             flow->src_port,
             ssh_ipaddr_render,&flow->dst_ip,
             flow->dst_port));

#ifdef FASTPATH_PROVIDES_LRU_FLOWS
  /* Mark when the last packet was received on the flow and bump the
     flow up in the flow LRU list. */
  fastpath_bump_lru_flow(fastpath, pc->flow_index);
#endif /* FASTPATH_PROVIDES_LRU_FLOWS */

  flow->last_packet_time = engine->run_time;

  /* Drop unsolicit ICMP errors. */
  if (forward)
    flow->data_flags &= ~SSH_ENGINE_FLOW_D_ERROR_RECEIVED;

  if (SSH_PREDICT_FALSE((flow->data_flags & SSH_ENGINE_FLOW_D_ERROR_RECEIVED)
			&& !(pc->flags & SSH_ENGINE_PC_FORWARD)
			&& FASTPATH_IS_ICMP_ERROR(pc)))
    {
      FP_COMMIT_FLOW_UNLOCK_HASH_READ(fastpath, pc->flow_index, flow);
      FP_UNLOCK_WRITE(fastpath);

      SSH_DEBUG(SSH_D_NETGARB, ("Unsolicit ICMP error message"));
      pc->audit.corruption = SSH_PACKET_CORRUPTION_UNSOLICITED_ICMP_ERROR;
      goto corrupt;
    }

  if (SSH_PREDICT_FALSE(flow->data_flags 
			& (SSH_ENGINE_FLOW_D_DANGLING
			   | SSH_ENGINE_FLOW_D_DROP_PKTS
			   | SSH_ENGINE_FLOW_D_REJECT_INBOUND)))
    {
      SshUInt32 flow_data_flags = flow->data_flags;
      SshUInt8 flow_type = flow->type;
      
      FP_COMMIT_FLOW_UNLOCK_HASH_READ(fastpath, pc->flow_index, flow);
      FP_UNLOCK_WRITE(fastpath);

      FASTPATH_DISPATCH_TGT(engine_packet_process_special_flow(engine,
							       pc,
							       flow_data_flags,
							       flow_type,
							       &ret));
      SSH_NOTREACHED;
    }
  
  /* Verify that the transform index used for decapsulating the packet 
     matches either the flow's transform index in the opposite direction 
     or one of allowed rx transform indexes. Do not consider incoming ipsec 
     flows (note that tunnel_id is non-zero for an incoming ipsec flow only 
     in nested tunnel case). */
  if (SSH_PREDICT_FALSE(pc->tunnel_id != 0
			&& (pc->flags & SSH_ENGINE_FLOW_D_IPSECINCOMING) == 0
			&& (pc->flags & SSH_ENGINE_PC_SKIP_TRD_VERIFY) == 0
			&& pc->prev_transform_index != SSH_IPSEC_INVALID_INDEX)
      )
    {
      int i;
      Boolean found_match = FALSE;
      
      /* Check if packet was decapsulated using transform index in the 
	 opposite direction. */
      if (forward 
	  && (pc->prev_transform_index == flow->reverse_transform_index))
	found_match = TRUE;
      else if (!forward 
	       && (pc->prev_transform_index == flow->forward_transform_index))
	found_match = TRUE;
      
      /* Check if packet was decapsulated using one of allowed rx transform
	 indexes. */
      else
	{
	  for (i = 0; i < SSH_ENGINE_NUM_RX_TRANSFORMS; i++)
	    {
	      if (forward 
		  && (pc->prev_transform_index 
		      == flow->forward_rx_transform_index[i]))
		{
		  found_match = TRUE;
		  break;
		}
	      else if (!forward
		       && (pc->prev_transform_index 
			   == flow->reverse_rx_transform_index[i]))
		{
		  found_match = TRUE;
		  break;
		}
	    }
	}
      
      /* Packets's prev_transform_index did not match any allowed transform 
	 indexes, pass packet to engine. */
      if (found_match == FALSE)
	{
	  SshEngineActionRet retval;
	  
	  /* Transform index mismatch, pass packet to engine. */
	  FP_COMMIT_FLOW_UNLOCK_HASH_READ(fastpath, pc->flow_index, flow);
	  FP_UNLOCK_WRITE(fastpath);
	  
	  /* Prepare packet for engine. This function returns 
	     SSH_ENGINE_RET_RESTART_FLOW_LOOKUP if the packet is ready for
	     to be passed to engine. */
	  retval = engine_packet_prepare_pkt(engine, pc);
	  if (retval != SSH_ENGINE_RET_RESTART_FLOW_LOOKUP)
	    {
	      FASTPATH_DISPATCH_TGT(retval);
	    }
	  else
	    {
	      /* Pass packet to the engine */
	      engine_rule_packet_handler(engine, pc);
	      goto return_unless_recursed;
	    }
	  SSH_NOTREACHED;
	}
    }
  
#ifdef SSH_IPSEC_STATISTICS
  if (forward)
    {
      flow->stats.forward_octets += pc->packet_len;
      flow->stats.forward_packets++;
    }
  else
    {
      flow->stats.reverse_octets += pc->packet_len;
      flow->stats.reverse_packets++;
    }
#endif /* SSH_IPSEC_STATISTICS */

  /* Select the transform that we are going to use.  */
  if ((pc->flags & SSH_ENGINE_FLOW_D_IPSECINCOMING) || forward)
    pc->transform_index = flow->forward_transform_index;
  else
    pc->transform_index = flow->reverse_transform_index;

#ifdef DEBUG_LIGHT
  /* If we hit an IPsec flow, verify there is an associated transform */
  if (pc->flags & SSH_ENGINE_FLOW_D_IPSECINCOMING)
    SSH_ASSERT(pc->transform_index != SSH_IPSEC_INVALID_INDEX);
#endif  /* DEBUG_LIGHT */

  /* Copy other flow data. */
  fastpath_copy_flow_data(fastpath, flow, pc);

  /* Get information related to next hop gateway/routing/media headers. */
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  if (forward)
    {
      pc->u.flow.mtu = flow->forward_mtu;
      pc->u.flow.ifnum = flow->forward_ifnum;
      pc->u.flow.local = flow->forward_local;
      pc->route_selector = flow->forward_route_selector;
    }
  else
    {
      pc->u.flow.mtu = flow->reverse_mtu;
      pc->u.flow.ifnum = flow->reverse_ifnum;
      pc->u.flow.local = flow->reverse_local;
      pc->route_selector = flow->reverse_route_selector;
    }
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  if (forward)
    next_hop_index = flow->forward_nh_index;
  else
    next_hop_index = flow->reverse_nh_index;

  if ((next_hop_index == SSH_IPSEC_INVALID_INDEX) && 
      !(pc->flags & SSH_ENGINE_FLOW_D_IPSECINCOMING))
    {
      SSH_DEBUG(SSH_D_FAIL, ("No next hop index in flow, dropping packet"));
      FP_COMMIT_FLOW_UNLOCK_HASH_READ(fastpath, pc->flow_index, flow);
      FP_UNLOCK_WRITE(fastpath);
      goto drop;
    }
  
  if (next_hop_index != SSH_IPSEC_INVALID_INDEX)
    {
      Boolean dst_is_nulladdr;
      
      SSH_ASSERT(next_hop_index < fastpath->next_hop_hash_size);
      
      nh = FP_GET_NH(fastpath, next_hop_index);
      pc->u.flow.mtu = (SshUInt16) nh->mtu;
      pc->u.flow.ifnum = nh->ifnum;
      pc->u.flow.local = (nh->flags & SSH_ENGINE_NH_LOCAL) != 0;
      pc->u.flow.mediatype = nh->mediatype;
      pc->u.flow.media_protocol = nh->media_protocol;
      pc->u.flow.media_hdr_len = nh->media_hdr_len;
      pc->u.flow.min_packet_len = nh->min_packet_len;
      memcpy(pc->u.flow.mediahdr, nh->mediahdr, nh->media_hdr_len);
      
      /* Update the source/destination media header for special
	 inbound and outbound next-hop nodes. */
      dst_is_nulladdr = SSH_IP_IS_NULLADDR(forward
					   ? &flow->dst_ip
					   : &flow->src_ip);
      
      if (SSH_PREDICT_TRUE(nh->mediatype != SSH_INTERCEPTOR_MEDIA_PLAIN))
	{
	  /* Massage nh->mediahdr in case the nexthop cached one does
	     not suffice (inbound packets to local-stack, null addr packets,
	     etc..) */
	  fastpath_update_media_header(pc, nh, dst_is_nulladdr);
	}
      FP_RELEASE_NH(fastpath, next_hop_index);
    }
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

#ifdef SSHDIST_IPSEC_NAT
  /* Copy NAT related info from flow. */
  if (forward)
    {
      pc->u.flow.nat_src_ip = flow->nat_src_ip;
      pc->u.flow.nat_dst_ip = flow->nat_dst_ip;
      pc->u.flow.nat_src_port = flow->nat_src_port;
      pc->u.flow.nat_dst_port = flow->nat_dst_port;
    }
  else
    {
      pc->u.flow.nat_src_ip = flow->src_ip;
      pc->u.flow.nat_dst_ip = flow->dst_ip;
      pc->u.flow.nat_src_port = flow->src_port;
      pc->u.flow.nat_dst_port = flow->dst_port;
    }
#endif /* SSHDIST_IPSEC_NAT */

  SSH_DEBUG(SSH_D_MY,
            ("pc index 0x%08lx flow forward index 0x%08lx "
	     "reverse index 0x%08lx",
             (unsigned long) pc->transform_index,
             (unsigned long) flow->forward_transform_index,
             (unsigned long) flow->reverse_transform_index));

  /* Require that flow is not dangling before copy_transform_data()
     is called, as we overload the {forward,reverse}_transform_index
     in dangling flows. */
  SSH_ASSERT((flow->data_flags & SSH_ENGINE_FLOW_D_DANGLING) == 0);

#ifdef SSH_IPSEC_EXECUTE_PROTOCOL_MONITORS

  switch (flow->type)
    {
    case SSH_ENGINE_FLOW_TYPE_TCP:
      mret = ssh_engine_tcp_packet(flow, pc);
      break;

    case SSH_ENGINE_FLOW_TYPE_UDP:
      mret = ssh_engine_udp_packet(flow, pc);
      break;

    case SSH_ENGINE_FLOW_TYPE_ICMP:
      mret = ssh_engine_icmp_packet(flow, pc);
      break;

    default:
      mret = SSH_ENGINE_MRET_PASS;
    }

#endif /* SSH_IPSEC_EXECUTE_PROTOCOL_MONITORS */

  /* Commit flow changes back to fastpath and unlock. */
  FP_COMMIT_FLOW_UNLOCK_HASH_READ(fastpath, pc->flow_index, flow);
  FP_UNLOCK_WRITE(fastpath);

#ifdef SSH_IPSEC_EXECUTE_PROTOCOL_MONITORS
 /* Process the protocol monitor result code. */
 switch (mret)
   {
   case SSH_ENGINE_MRET_PASS:
      SSH_DEBUG(SSH_D_LOWOK, ("protocol monitor pass"));
      /* Fall to executing the transform. */
      break;
    case SSH_ENGINE_MRET_DROP:
      /* Fall through "corrupt:" label, incase protocol monitor
         provided any data for logging. */
      SSH_DEBUG(SSH_D_LOWOK, ("protocol monitor drop"));
      SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_MONITORDROP);
      goto corrupt;

    case SSH_ENGINE_MRET_ERROR:
      SSH_DEBUG(SSH_D_LOWOK, ("protocol monitor error"));
      SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_MONITORDROP);
      goto error;

    case SSH_ENGINE_MRET_REJECT:
      /* Fall through "corrupt:" label, incase protocol monitor
         provided any data for logging. */
      SSH_DEBUG(SSH_D_LOWOK, ("protocol monitor reject"));
      SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_MONITORDROP);

      /* If other protocol monitors may return REJECT, then we must
         deduce the protocol type from the packet context. */
      ssh_engine_send_tcp_rst(engine, pc);

      goto corrupt;
    default:
      ssh_fatal("invalid return %d from protocol specific code", (int)mret);
      SSH_NOTREACHED;
    }
#endif /* SSH_IPSEC_EXECUTE_PROTOCOL_MONITORS */

  /* If this is a first fragment, update reasembly / fragment magic
     code.  */ 







  if (SSH_PREDICT_FALSE(pc->pp->flags & SSH_ENGINE_P_FIRSTFRAG))
    {
      if (forward)
        reassemble = (pc->flags & SSH_ENGINE_FLOW_D_FWD_REASSEMBLE) != 0;
      else
        reassemble = (pc->flags & SSH_ENGINE_FLOW_D_REV_REASSEMBLE) != 0;

      SSH_DEBUG(SSH_D_MIDOK, ("first frag, reassemble=%d", (int)reassemble));
      ret = ssh_fastpath_fragmagic(fastpath, pc, reassemble);
      /* If the return value is RESTART_FLOW_LOOKUP, that means that we
         should process it as a fragment.  However, this first fragment
         would get back here.  So we don't actually restart the flow lookup
         for this first fragment. */
      if (ret != SSH_ENGINE_RET_RESTART_FLOW_LOOKUP)
        goto dispatch;
    }

  /* Copy transform data. The outbound sequence number is incremented here,
     and the transform packet timestamp and statistics is updated. */
  if (pc->transform_index != SSH_IPSEC_INVALID_INDEX)
    {
      /* Copy data needed for transform execution. */
      FP_LOCK_WRITE(fastpath);
      if (!fastpath_copy_transform_data(fastpath, pc))
	{
	  FP_UNLOCK_WRITE(fastpath);
	  goto drop;
	}
      FP_UNLOCK_WRITE(fastpath);
    }
  else
    {
      pc->transform = 0;
      
#ifdef SSHDIST_IPSEC_NAT
      /* Allow fragmentation before NAT to ensure that the ICMP frag needed
	 payload is the offending packet before NAT. */
      if (pc->flags & (SSH_ENGINE_FLOW_D_NAT_SRC | SSH_ENGINE_FLOW_D_NAT_DST))
	pc->flags |= SSH_ENGINE_FLOW_D_FRAG_TRANSFORM;
#endif /* SSHDIST_IPSEC_NAT */      
    }
  /* Fall to executing the transform. */

 execute_transform:
  SSH_DEBUG(SSH_D_LOWSTART,
            ("executing transform for packet: transform_idx=%08lx "
             "prev_transform_idx=0x%08lx "
             "pc->flags=0x%08lx",
             (unsigned long) pc->transform_index,
	           (unsigned long) pc->prev_transform_index,
             (unsigned long) pc->flags));
 
  /* Check if the packet must be fragmented before applying the transform. */
  if (pc->transform_index != SSH_IPSEC_INVALID_INDEX &&
      !(pc->flags & SSH_ENGINE_FLOW_D_IPSECINCOMING))
    discount = pc->u.flow.tr.packet_enlargement;
  else
    discount = 0;

  if (SSH_PREDICT_FALSE((pc->packet_len > (pc->u.flow.mtu - discount))
			&& (pc->flags & SSH_ENGINE_FLOW_D_FRAG_TRANSFORM)))
    {
      /* Fragment the packet and execute transform for each
	 fragment separately. */
      SSH_DEBUG(SSH_D_LOWSTART,
		("Fragmenting packet before transform to MTU %u,"
		 "packet len=%d, flow mtu=%d, discount=%u",
		 pc->u.flow.mtu - discount,
		 pc->packet_len,
		 pc->u.flow.mtu,
		 discount));

      ssh_fastpath_fragment(fastpath, pc->u.flow.mtu - discount, pc, pp,
			    fastpath_execute_transform_fragmented);
      /* The pc of the original packet is now ready to be recycled. */
      goto deinitialize;
    }

#ifdef FASTPATH_ACCELERATOR_CONFIGURED
  if (pc->fastpath_accel_ret == SSH_ENGINE_RET_EXECUTE)
    {
      fastpath_accel_packet_continue(fastpath->accel, pc,
				     SSH_ENGINE_RET_EXECUTE);
      goto return_unless_recursed;
    }
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */

  /* Execute the transform on the packet. */
  fastpath_execute_transform(pc);
  goto return_unless_recursed;

 send:
  SSH_DEBUG(SSH_D_LOWSTART, ("sending packet %p", pc->pp));

  /* Set pc flags for statistics accounting. */
  if (SSH_PREDICT_FALSE(pc->pp->flags & SSH_PACKET_FORWARDED))
    pc->flags |= SSH_ENGINE_PC_FORWARDED;
  else if (SSH_PREDICT_TRUE(pc->pp->flags & SSH_PACKET_FROMPROTOCOL))
    pc->flags |= SSH_ENGINE_PC_OUTBOUND;

  /* Check packet length. */
  if (SSH_PREDICT_FALSE(pc->packet_len > pc->u.flow.mtu))
    {
      SSH_DEBUG(SSH_D_FAIL,
		("Packet length %d too large to into MTU %d, "
		 "dropping packet %p",
		 (int) pc->packet_len, (int) pc->u.flow.mtu, pc->pp));
      goto drop;
    }
  
  /* Send the packet. */
  fastpath_execute_send(fastpath, pc, pc->pp);
  goto deinitialize;

 deinitialize:

  /* We jump here to deinitialize after execution.  This code updates
     statistics counters and frees the packet context (if appropriate)
     and does other deinitialization that may be needed after
     processing the packet.  When we get here, the packet itself has
     already been freed (i.e., pp, pd, ucp etc are invalid). */
#ifdef SSH_IPSEC_STATISTICS
  fastpath_update_statistics_counters(fastpath, pc);
#endif /* SSH_IPSEC_STATISTICS */

  /* If we have pending packets, take the first packet from the pending
     list and process it now. */
  if (SSH_PREDICT_FALSE(pc->pending_packets != NULL))
    {
      SshEnginePacketContext next;

      /* NOTE: this code depends on pending packets only being used for
         fragmagic to restart processing of fragments that were received
         out-of-order, when it turns out they should not be reassembled.
         This may need to be changed if pending packets are also used for
         other purposes in the future. */
      pp = pc->pending_packets;
      pd = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEnginePacketData);
      next = pc->next;
      ssh_engine_init_pc(pc, engine, pp, pd->pending_tunnel_id, pp->next);
      pc->next = next;

      memcpy(pc->flow_id, pd->pending_flow_id, sizeof(pc->flow_id));
      ret = pd->pending_ret;

      skip_flow_id_computation = (ret == SSH_ENGINE_RET_RESTART_FLOW_LOOKUP);
      ret = SSH_ENGINE_RET_RESTART;
      goto dispatch;
    }

  /* Mark that the packet context should be put back on the freelist. */
  pc->flags |= SSH_ENGINE_PC_DONE;
  goto return_unless_recursed;

 error:
  /* We jump here after an error that frees pp has occurred.  When we get
     here, pp, pd, ucp etc are invalid.  This should count the error
     and deinitialize.  Note that counter information stored in the packet
     data has been lost when we get here. */



  goto deinitialize;

 corrupt:
 drop:
  pp = pc->pp;

  /* We jump here when sanity checks indicate that the packet is corrupt
     (or possibly an attack packet trying to exploit bugs in TCP/IP
     implementations).  Count the error and drop it. */
  if (pc->audit.corruption != SSH_PACKET_CORRUPTION_NONE)
    {
      /* A packet may enter here without pc->flags having been initialized,
         so make sure we have a fresh copy of the PC_AUDIT_CORRUPT flag. */
      pc->flags &= ~SSH_ENGINE_PC_AUDIT_CORRUPT;
      pc->flags |= engine->pc_flags;

      if (pc->audit.ip_option)
        SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_OPTIONSDROP);
      else
        SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_CORRUPTDROP);
      if (pc->pp != NULL && (pc->flags & SSH_ENGINE_PC_AUDIT_CORRUPT) != 0)
        {
          SSH_DUMP_PACKET(SSH_D_PCKDMP, "corrupt/invalid packet", pc->pp);

          /* Audit this event */
          ssh_engine_audit_packet_context(engine, pc);
          goto return_unless_recursed;
        }
      else
        {
          SSH_DEBUG(SSH_D_NICETOKNOW,
                    ("Omitting packet audit: pp=%p pc->flags=%08lx",
                     pc->pp,
		     (unsigned long) pc->flags));
        }
    }
  /* We jump here when the packet is to be dropped for whatever reason.  This
     frees pp and deinitializes. */
  SSH_DEBUG(SSH_D_MIDOK, ("Dropping packet"));
  SSH_ENGINE_MARK_STAT(pc, SSH_ENGINE_STAT_DROP);
  if (pp != NULL)
    ssh_interceptor_packet_free(pp);
  pp = NULL;
  goto deinitialize;

 return_unless_recursed:
  /* All exists from this function (except for the tail recursion elimination)
     come through here.  This checks if we have recursed, and if so,
     jumps back to beginning as part of the tail recursion elimination
     code. */
  SSH_ASSERT(pc->flags & SSH_ENGINE_PC_INUSE);
  /* Check for recursive calls. */
  if (SSH_PREDICT_FALSE(pc->flags & SSH_ENGINE_PC_RECURSED))
    { /* Recursed - eliminate tail recursion. */
      SSH_DEBUG(SSH_D_HIGHOK, ("faking recursion"));
      pc->flags &= ~SSH_ENGINE_PC_RECURSED;
      SSH_ASSERT(pc->recursed_ret != 0xdeadbeee);
      ret = pc->recursed_ret;
      pc->recursed_ret = 0xdeadbeee;

      goto dispatch;
    }
  /* Mark that we are no longer executing with this pc. */
  pc->flags &= ~SSH_ENGINE_PC_INUSE;
  /* If we are done with the packet, put the packet context back on the
     freelist. */
  if (SSH_PREDICT_TRUE(pc->flags & SSH_ENGINE_PC_DONE))
    {
      SshEnginePacketContext next = pc->next;

      ssh_engine_free_pc(engine, pc);

      if (SSH_PREDICT_FALSE(next != NULL) &&
	  !(next->flags & SSH_ENGINE_PC_DONE))
        {
          pc = next;
          pc->flags |= SSH_ENGINE_PC_INUSE;
	  pc->flags &= ~SSH_ENGINE_PC_RECURSED;
	  SSH_ASSERT(pc->recursed_ret != 0xdeadbeee);
          ret = pc->recursed_ret;
	  pc->recursed_ret = 0xdeadbeee;
          goto dispatch;
        }
    }
}

#ifdef FASTPATH_ACCELERATOR_CONFIGURED
/* Function for sending a packet out via the accelerated fastpath. This
   allocates a SshEnginePacketContext and passes the packet to
   fastpath_accel_packet_continue() with the SSH_ENGINE_RET_SEND action
   code. Otherwise if an error occurs then this must free pp. */
void software_fastpath_execute_send(SshFastpath fastpath,
				    SshInterceptorPacket pp,
				    size_t media_header_len)
{
  SshEnginePacketContext pc;
  
  pc = ssh_engine_alloc_pc(fastpath->engine);
  if (pc == NULL)
    {
      SSH_DEBUG(SSH_D_FAIL, ("Failed to allocate a packet context"));
      ssh_interceptor_packet_free(pp);
      return;
    }
  ssh_engine_init_pc(pc, fastpath->engine, pp, 0, NULL);
  pc->flags = 0;

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  pc->media_hdr_len = media_header_len;
#endif /* !SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  
  SSH_DEBUG(SSH_D_LOWOK, ("Sending packet %p out via accelerated fastpath",
			  pp));
  
  fastpath_accel_packet_continue(fastpath->accel, pc, SSH_ENGINE_RET_SEND);
}
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */

/* Function that is called whenever a packet is received from the
   network or from a protocol.  This is also called when a packet is
   received from the policy manager.  This function must eventually
   free the packet, either by calling ssh_interceptor_packet_free on
   the packet or by passing it to the ssh_interceptor_send function.
   Note that this function may be called asynchronously, concurrently
   with any other functions (including itself).

   When a packet is passed to this callback, the `pp->flags' field may
   contain arbitrary flags in the bits reserved for the interceptor
   (mask 0x000000ff).  This callback is not allowed to modify any of
   those bits; they must be passed intact to ssh_interceptor_send or
   ssh_interceptor_packet_free.  Any other bits (mask 0xffffff00) will
   be zero when the packet is sent to this callback; those bits may be
   used freely by this callback.  They are not used by the
   interceptor.

   See alternative implementations below. */

#ifndef SSH_IPSEC_SEND_IS_SYNC
SSH_FASTTEXT
void fastpath_packet_handler(SshEngine engine, SshInterceptorPacket pp,
			     SshUInt32 tunnel_id,
			     SshUInt32 prev_transform_index,
			     Boolean is_recursive)
{
  SshFastpath fastpath = engine->fastpath;
  SshEnginePacketContext pc;

  /* Some debugging code... */
  SSH_DEBUG(
       SSH_D_MIDSTART,
       ("%s%spacket (pp=%p, flags=0x%08lx, proto=%d, ifnum_in=%d, len=%d)",
	((pp->flags & SSH_PACKET_FROMPROTOCOL) ? "outbound" : "inbound"),
	((pp->flags & SSH_PACKET_FORWARDED)
	 ? " forwarded " : " "),
	pp,
	(unsigned long) pp->flags, pp->protocol,
	(int) pp->ifnum_in,
	ssh_interceptor_packet_len(pp)));
  SSH_DUMP_PACKET(SSH_D_MY + 10, "packet", pp);
  
  /* Note that the SshEnginePacketContext.flags OUTBOUND_CALL denotes
     the context we are processing a packet context in and _NOT_ the
     actual direction of the packet. The reassember should take
     care to require that each packet has the same value of this
     flag. */
  if (pp->flags & SSH_PACKET_FROMADAPTER)
    pp->flags |= SSH_ENGINE_P_FROMADAPTER;

  if ((pc = ssh_engine_alloc_pc(engine)) != NULL)
    {
#ifdef DEBUG_LIGHT
      SshEnginePacketData pd;

      pd = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEnginePacketData);
      memset(pd, 'I', sizeof(*pd));
#endif /* DEBUG_LIGHT */

      ssh_engine_init_pc(pc, engine, pp, tunnel_id, NULL);
      pc->flags = 0;
      pc->prev_transform_index = prev_transform_index;

      /* Start processing the packet using the context. */
      fastpath_packet_continue(fastpath, pc,
			       SSH_ENGINE_RET_RESTART);

    }
  else
    {
#ifdef SSH_IPSEC_STATISTICS
      SshFastpathGlobalStats stats;

      ssh_kernel_critical_section_start(fastpath->stats_critical_section);
      stats = &fastpath->stats[ssh_kernel_get_cpu()];
      stats->in_packets++;
      stats->in_octets_uncomp += ssh_interceptor_packet_len(pp);
      stats->in_octets_comp += ssh_interceptor_packet_len(pp);
      ssh_kernel_critical_section_end(fastpath->stats_critical_section);
#endif /* SSH_IPSEC_STATISTICS */
      ssh_interceptor_packet_free(pp);
    }
}

/* Wraps any required media headers into the packet and sends it out.  This
   ignores `pc->pp', and frees `pp'. */

SSH_FASTTEXT
void fastpath_execute_send(SshFastpath fastpath, 
			   SshEnginePacketContext pc,
			   SshInterceptorPacket pp)
{
  SshEngine engine = fastpath->engine;

  /* Set forwarded flag to TRUE in packets processed across
     the fastpath. */
  if (SSH_PREDICT_TRUE(pp->flags & SSH_PACKET_FROMPROTOCOL))
    {
      if (SSH_PREDICT_TRUE(pp->flags & SSH_ENGINE_P_FROMADAPTER))
	{
	  pp->flags |= SSH_PACKET_FORWARDED;
	  pc->flags |= SSH_ENGINE_PC_FORWARDED;
	}
      else
	pc->flags |= SSH_ENGINE_PC_OUTBOUND;
    }

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  /* Copy the media header into the actual packet. */
  if (SSH_PREDICT_FALSE(!fastpath_insert_media_header(pc, pp)))
    return;
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  /* Send the packet out. */
  SSH_DEBUG(SSH_D_HIGHOK, ("sending packet: ifnum_out=%d, protocol=%d, "
                           "flags=0x%lx, len=%zd",
                           (int) pp->ifnum_out, pp->protocol,
			   (unsigned long) pp->flags,
                           ssh_interceptor_packet_len(pp)));
  SSH_DUMP_PACKET(SSH_D_MY + 10, ("sending packet:"), pp);

#ifdef FASTPATH_ACCELERATOR_CONFIGURED

  /* Pass packet to accelerated fastpath. */
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  software_fastpath_execute_send(fastpath, pp, 0);
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  software_fastpath_execute_send(fastpath, pp, pc->media_hdr_len);
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

#else /* FASTPATH_ACCELERATOR_CONFIGURED */
  
  /* Send packet via interceptor. */
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  ssh_interceptor_send(engine->interceptor, pp, 0);
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  ssh_interceptor_send(engine->interceptor, pp, pc->media_hdr_len);
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

#endif /* FASTPATH_ACCELERATOR_CONFIGURED */
}
#endif /* not SSH_IPSEC_SEND_IS_SYNC */

#ifdef SSH_IPSEC_SEND_IS_SYNC
/* A timeout function that processes queued packets from recursive
   fastpath_packet_handler invocations or packets requiring an
   asynchronous send from ssh_engine_packet_send(). This function is
   called from the timeout and just after we have finished processing
   an inbound packet in fastpath_packet_callback. */
void ssh_engine_process_asynch_packets(void *context)
{
  SshEngine engine = (SshEngine) context;
  SshFastpath fastpath = engine->fastpath;
  void *thread_id;
  SshUInt32 i, free_thread_id_slot;
  SshInterceptorPacket pending_packet;
  SshInterceptorPacket pp;
  SshEngineAsynchPacketData data = NULL;

  SSH_DEBUG(SSH_D_MIDSTART, ("Processing queued packets"));

  free_thread_id_slot = 0xffffffff;
  thread_id = ssh_kernel_thread_id();

  /* Take the queue into a local variable. */
  ssh_kernel_mutex_lock(engine->pp_lock);

  for (i = 0; i < SSH_ENGINE_NUM_CONCURRENT_THREADS; i++)
    {
      if (SSH_PREDICT_FALSE(fastpath->active_thread_ids[i] == thread_id))
        {
          free_thread_id_slot = 0xffffffff;
          break;
        }
      else if (SSH_PREDICT_TRUE(fastpath->active_thread_ids[i] == NULL))
        {
          free_thread_id_slot = i;
        }
    }

  /* Free thread id's not available.. wait awhile till they are.
     Unlike packet_handler it is not possible for us to
     run without a free_thread_id_slot(). */
  if (free_thread_id_slot == 0xffffffff
      || i < SSH_ENGINE_NUM_CONCURRENT_THREADS)
    {
      ssh_kernel_mutex_unlock(engine->pp_lock);

      SSH_DEBUG(SSH_D_NICETOKNOW,
                ("Thread ID not available. Re-scheduling!"));
      ssh_kernel_timeout_register(0, 10,
                                  ssh_engine_process_asynch_packets,
                                  engine);
      return;
    }

  fastpath->active_thread_ids[free_thread_id_slot] = thread_id;
  engine->asynch_timeout_scheduled = FALSE;
  ssh_kernel_mutex_unlock(engine->pp_lock);

  /* Process all queued incoming packets. */
  do
    {
      /* Get one packet from list */
      ssh_kernel_mutex_lock(engine->pp_lock);

      pp = engine->asynch_packets_head;
      if (pp != NULL)
	{
	  engine->asynch_packets_head = pp->next;
	  if (engine->asynch_packets_head == NULL)
	    engine->asynch_packets_tail = NULL;
	  pp->next = NULL;
	}
      ssh_kernel_mutex_unlock(engine->pp_lock);

      /* Break out of loop if there are no more packets to send */
      if (pp == NULL)
	break;

      /* Process this packet. */
      data = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEngineAsynchPacketData);
      if (data->is_icept_send)
        {
#ifdef FASTPATH_ACCELERATOR_CONFIGURED
	  /* Pass packet to accelerated fastpath. */
	  software_fastpath_execute_send(fastpath, pp, data->media_hdr_len);
#else /* FASTPATH_ACCELERATOR_CONFIGURED */
          ssh_interceptor_send(engine->interceptor, pp, data->media_hdr_len);
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */
        }
      else
	{
	  ssh_engine_send_packet(engine, pp,
				 data->hdrlen,
				 data->transform_index,
				 &data->src,
				 &data->dst,
				 data->ipproto,
				 data->ipsec_offset,
				 data->ipsec_offset_prevnh,
				 FALSE);
	  /* The `pp' and `data' are now invalid. */
	}
    } while (pp != NULL);

  /* At this point we might have packets registered for
     "ssh_interceptor_send()". */
  do
    {
      /* Get one packet from list */
      ssh_kernel_mutex_lock(engine->pp_lock);
      pp = fastpath->send_packets_head;
      if (pp != NULL)
	{
	  data = SSH_INTERCEPTOR_PACKET_DATA(pp,
					     SshEngineAsynchPacketData);
	  fastpath->send_packets_head = data->next;
	  if (fastpath->send_packets_head == NULL)
	    fastpath->send_packets_tail = NULL;
	}
      ssh_kernel_mutex_unlock(engine->pp_lock);

      /* Break out of loop if there are no more packets to send */
      if (pp == NULL)
	break;

      /* Process this packet */
      SSH_ASSERT(data != NULL);
#ifdef FASTPATH_ACCELERATOR_CONFIGURED
      /* Pass packet to accelerated fastpath. */
      software_fastpath_execute_send(fastpath, pp, data->media_hdr_len);
#else /* FASTPATH_ACCELERATOR_CONFIGURED */
      ssh_interceptor_send(engine->interceptor, pp, data->media_hdr_len);
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */
    } while (pp != NULL);

  /* At this point we might have packets registered for recursive
     handling. Grab one packet from the list and enter
     regular packet handler, it will take care of the rest. */

  ssh_kernel_mutex_lock(engine->pp_lock);

  pending_packet = fastpath->recursive_packets_head;
  if (pending_packet != NULL)
    {
      fastpath->recursive_packets_head = pending_packet->next;
      if (fastpath->recursive_packets_head == NULL)
        fastpath->recursive_packets_tail = NULL;
    }

  /* Mark that this thread is no longer active before we enter the
     packet handler, otherwise it will put the packets on pending_packets. */
  SSH_ASSERT(fastpath->active_thread_ids[free_thread_id_slot] == thread_id);
  fastpath->active_thread_ids[free_thread_id_slot] = NULL;

  ssh_kernel_mutex_unlock(engine->pp_lock);

  if (pending_packet)
    {
      pending_packet->next = NULL;
      fastpath_packet_handler(engine, pending_packet,
			      0, SSH_IPSEC_INVALID_INDEX,
			      TRUE);
    }
}

/* Queue a packet for later sending */
static void
fastpath_queue_turnaround(SshEngine engine,
			  SshInterceptorPacket pp,
			  SshUInt32 media_hdr_len)
{
  SshEngineAsynchPacketData data;

  ssh_kernel_mutex_lock(engine->pp_lock);

  data = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEngineAsynchPacketData);
  data->is_icept_send = TRUE;
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  data->media_hdr_len = 0;
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  data->media_hdr_len = media_hdr_len;
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  /* Queue packet for the list of pending asynch packets. */
  pp->next = NULL;
  if (engine->asynch_packets_tail)
    engine->asynch_packets_tail->next = pp;
  else
    engine->asynch_packets_head = pp;
  engine->asynch_packets_tail = pp;

  if (!engine->asynch_timeout_scheduled)
    {
      engine->asynch_timeout_scheduled = TRUE;
      ssh_kernel_mutex_unlock(engine->pp_lock);
      ssh_kernel_timeout_register(0, 1,
                                  ssh_engine_process_asynch_packets,
                                  engine);
    }
  else
    {
      ssh_kernel_mutex_unlock(engine->pp_lock);
    }
  return;
}

SSH_FASTTEXT
void fastpath_packet_handler(SshEngine engine,
			     SshInterceptorPacket pp,
			     SshUInt32 tunnel_id,
			     SshUInt32 prev_transform_index,
			     Boolean is_recursive)
{
  SshFastpath fastpath = engine->fastpath;
  SshEnginePacketContext pc;
  SshUInt32 call_outbound;
  SshEngineAsynchPacketData data = NULL;
  SshInterceptorPacket send_packet, pending_packet;
  void *thread_id;
  SshUInt32 i, free_thread_id_slot;
  SshUInt32 flags = pp->flags;

  /* Some debugging code... */
  SSH_DEBUG(
	SSH_D_MIDSTART,
	("%s%spacket (pp=%p, flags=0x%08lx, proto=%u, ifnum_in=%d, len=%zd)",
	 ((pp->flags & SSH_PACKET_FROMPROTOCOL) ? "outbound" : "inbound"),
	 ((pp->flags & SSH_PACKET_FORWARDED)
	  ? " forwarded " : " "),
	 pp, (unsigned long) pp->flags,
	 pp->protocol,
	 (int) pp->ifnum_in,
	 ssh_interceptor_packet_len(pp)));
  SSH_DUMP_PACKET(SSH_D_MY + 10, "packet", pp);

  /* Lock the packet context mutex. It protects the list of recursive
     packets and active thread IDs */
  ssh_kernel_mutex_lock(engine->pp_lock);

  /* Check for recursive call to the packet handler. */
  thread_id = ssh_kernel_thread_id();

  free_thread_id_slot = 0xffffffff;

  for (i = 0; i < SSH_ENGINE_NUM_CONCURRENT_THREADS; i++)
    if (fastpath->active_thread_ids[i] == thread_id)
      break;
    else if (fastpath->active_thread_ids[i] == NULL)
      free_thread_id_slot = i;

  /* Upon a goto restart of 'pending_packet', this value
     is set according to pp->flags of the parameter 'pp'. */
  call_outbound = 0;
  pending_packet = NULL;
  send_packet = NULL;

 restart:
  ssh_kernel_mutex_assert_is_locked(engine->pp_lock);
  SSH_INTERCEPTOR_STACK_MARK();

  if (SSH_PREDICT_FALSE(pp->flags & SSH_PACKET_FROMPROTOCOL))
    call_outbound = SSH_ENGINE_PC_OUTBOUND_CALL;

  /* Note that the SshEnginePacketContext.flags OUTBOUND_CALL denotes
     the context we are processing a packet context in and _NOT_ the
     actual direction of the packet. The reassember should take care
     to require that each packet has the same value of this flag. */
  if (SSH_PREDICT_TRUE(pp->flags & SSH_PACKET_FROMADAPTER))
    pp->flags |= SSH_ENGINE_P_FROMADAPTER;

  /* Terminate all recursions back into quicksec immediately */
  if (SSH_PREDICT_FALSE(i < SSH_ENGINE_NUM_CONCURRENT_THREADS)
      || SSH_PREDICT_FALSE(free_thread_id_slot == 0xffffffff))
    {
      SSH_DEBUG(SSH_D_LOWOK, ("Recursive call for thread ID 0x%p", thread_id));

      /* Store arguments into the packet data. */
      data = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEngineAsynchPacketData);
      data->tunnel_id = tunnel_id;
      data->prev_transform_index = prev_transform_index;

      /* Normal tail-recursion removal case.  Queue packet for the
         list of pending recursive packets. */
      SSH_DEBUG(SSH_D_LOWOK,
                ("Scheduling packet for tail-recursive handling"));
      pp->next = NULL;
      if (fastpath->recursive_packets_tail)
        fastpath->recursive_packets_tail->next = pp;
      else
        fastpath->recursive_packets_head = pp;
      fastpath->recursive_packets_tail = pp;
    }
  else
    {
      /* This is not a recursive call.  Mark this thread running. */
      fastpath->active_thread_ids[free_thread_id_slot] = thread_id;

      ssh_kernel_mutex_unlock(engine->pp_lock);

      if (SSH_PREDICT_TRUE((pc = ssh_engine_alloc_pc(engine)) != NULL))
	{
#ifdef DEBUG_LIGHT
	  SshEnginePacketData pd;

	  pd = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEnginePacketData);
	  memset(pd, 'I', sizeof(*pd));
#endif /* DEBUG_LIGHT */

	  ssh_engine_init_pc(pc, engine, pp, tunnel_id, NULL);
	  pc->flags = 0;
	  pc->prev_transform_index = prev_transform_index;

	  /* Start processing the packet using the context. */
	  pc->flags |= call_outbound;

	  fastpath_packet_continue(fastpath, pc, SSH_ENGINE_RET_RESTART);

	  /* Send queued packets */
	  do
	    {
	      /* Take one packet from list and process it.
		 Do not hog all packets, as other CPUs might
		 be processing packets from the same list. */
	      ssh_kernel_mutex_lock(engine->pp_lock);

	      send_packet = fastpath->send_packets_head;
	      if (send_packet != NULL)
		{
		  data =
		    SSH_INTERCEPTOR_PACKET_DATA(send_packet,
						SshEngineAsynchPacketData);
		  fastpath->send_packets_head = data->next;
		  if (fastpath->send_packets_head == NULL)
		    fastpath->send_packets_tail = NULL;
		  data->next = NULL;
		}
	      ssh_kernel_mutex_unlock(engine->pp_lock);

	      /* Break from loop if there are no more packets to
		 process. */
	      if (send_packet == NULL)
		break;

	      SSH_ASSERT(data != NULL);
	      /* Process packet */
	      if (SSH_PREDICT_FALSE(
		    (call_outbound & SSH_ENGINE_PC_OUTBOUND_CALL))
		  && (send_packet->flags & SSH_PACKET_FROMADAPTER))
		{
		  fastpath_queue_turnaround(engine, send_packet,
					    data->media_hdr_len);
		}
	      else
		{
#ifdef FASTPATH_ACCELERATOR_CONFIGURED
		  /* Pass packet to accelerated fastpath. */
		  software_fastpath_execute_send(fastpath, send_packet,
						 data->media_hdr_len);
#else /* FASTPATH_ACCELERATOR_CONFIGURED */
		  ssh_interceptor_send(engine->interceptor, send_packet,
				       data->media_hdr_len);
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */
		}
	    } while (send_packet != NULL);
	}
      else
	{
#ifdef SSH_IPSEC_STATISTICS
	  SshFastpathGlobalStats stats;

	  ssh_kernel_critical_section_start(fastpath->stats_critical_section);
	  stats = &fastpath->stats[ssh_kernel_get_cpu()];
          stats->in_packets++;
          stats->in_octets_uncomp += ssh_interceptor_packet_len(pp);
          stats->in_octets_comp += ssh_interceptor_packet_len(pp);
	  ssh_kernel_critical_section_end(fastpath->stats_critical_section);
#endif /* SSH_IPSEC_STATISTICS */
          ssh_interceptor_packet_free(pp);
	}

      /* Grab the pp_lock for cleaning up the bookkeeping. */
      ssh_kernel_mutex_lock(engine->pp_lock);

      /* Check if there are any recursive packets to be processed */
      if (SSH_PREDICT_FALSE(fastpath->recursive_packets_head))
	{
	  SSH_DEBUG(SSH_D_MIDSTART,
		    ("Taking a packet from the list of recursive packets"));

	  pending_packet = fastpath->recursive_packets_head;

	  /* Take one packet from list and process it.
	     Do not hog all packets, as other CPUs might be
	     processing packets from the same list. */
	  if (pending_packet != NULL)
	    {
	      fastpath->recursive_packets_head = pending_packet->next;
	      if (fastpath->recursive_packets_head == NULL)
		fastpath->recursive_packets_tail = NULL;
	      pending_packet->next = NULL;
	    }
	}
    }
  /* LOCKED */

  /* One packet processed, dropped because of resource shortages or
     queued as a recursive packet.  Continue processing packets from
     `pending_packet' or we are done. */
  if (SSH_PREDICT_FALSE(pending_packet))
    {
      SSH_DEBUG(SSH_D_MIDSTART,
		("Processing next pending packet"));
      pp = pending_packet;
      pending_packet = NULL;
      /* This is a bit ugly, but unavoidable, without placing
	 the tunnel_id/prev_transform_index always in 'pp' PacketData */
      tunnel_id = 0;
      prev_transform_index = SSH_IPSEC_INVALID_INDEX;
      goto restart;
    }
  else
    {
      if (SSH_PREDICT_TRUE(free_thread_id_slot != 0xffffffff))
	fastpath->active_thread_ids[free_thread_id_slot] = NULL;      
    }
  ssh_kernel_mutex_assert_is_locked(engine->pp_lock);
  ssh_kernel_mutex_unlock(engine->pp_lock);
  
  /* Take a shortcut and process packet queues here, but only if we
     were not called from local stack. */
  if (!is_recursive && (flags & SSH_PACKET_FROMADAPTER))
    ssh_engine_process_asynch_packets(engine);
}


/* Wraps any required media headers into the packet and sends it out.
   This ignores `pc->pp', and frees `pp'. */

void fastpath_execute_send(SshFastpath fastpath, 
			   SshEnginePacketContext pc,
			   SshInterceptorPacket pp)
{
  SshEngine engine = fastpath->engine;
  SshUInt32 i;
  void *thread_id;
  SshEngineAsynchPacketData data, data_prev;

  /* Set forwarded flag to TRUE in packets processed across
     the fastpath. */
  if (pp->flags & SSH_PACKET_FROMPROTOCOL)
    {
      if (pp->flags & SSH_ENGINE_P_FROMADAPTER)
	{
	  pp->flags |= SSH_PACKET_FORWARDED;
	  pc->flags |= SSH_ENGINE_PC_FORWARDED;
	}
      else
	pc->flags |= SSH_ENGINE_PC_OUTBOUND;
    }

#ifndef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  /* Copy the media header into the actual packet. */
  if (!fastpath_insert_media_header(pc, pp))
    return;
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

  /* Send the packet out. */
  SSH_DEBUG(SSH_D_HIGHOK, ("sending packet: ifnum_out=%d, protocol=%d, "
                           "flags=0x%08lx, len=%zd",
                           (int) pp->ifnum_out,
			   pp->protocol,
			   (unsigned long) pp->flags,
                           ssh_interceptor_packet_len(pp)));
  SSH_DUMP_PACKET(SSH_D_MY + 10, ("sending packet:"), pp);

  /* Prevent deadlocks/other misbehaviour. If the quicksec engine was
     entered via a call from the TCP/IP stack (as opposed to a NIC
     driver), then do not send packets back to the TCP/IP stack if we
     have already recursed into quicksec. */

  if (SSH_PREDICT_FALSE((pc->flags & SSH_ENGINE_PC_OUTBOUND_CALL))
      && (pp->flags & SSH_PACKET_FROMADAPTER))
    {
      fastpath_queue_turnaround(fastpath->engine, pp,
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
				0
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
				pc->media_hdr_len
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
				);
      return;
    }

  /* If this is a synchronous packet processing, then
     truncate the callstack by adding the packets to a
     queue for processing before returning. This allows
     us to not overflow the stack on platforms with 8192
     bytes of stack. */



  thread_id = ssh_kernel_thread_id();

  ssh_kernel_mutex_lock(engine->pp_lock);

  for (i = 0; i < SSH_ENGINE_NUM_CONCURRENT_THREADS; i++)
    {
      if (SSH_PREDICT_FALSE(fastpath->active_thread_ids[i] == thread_id))
        {
          data = SSH_INTERCEPTOR_PACKET_DATA(pp, SshEngineAsynchPacketData);
          data->next = NULL;
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
          data->media_hdr_len = 0;
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
          data->media_hdr_len = pc->media_hdr_len;
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

          if (SSH_PREDICT_FALSE(fastpath->send_packets_tail))
            {
              /* pp->next is unavailable to us here, because it used by
                 fastpath_packet_handler() for queueing packets for
                 processing in the engine. */
              data_prev
		= SSH_INTERCEPTOR_PACKET_DATA(fastpath->send_packets_tail,
					      SshEngineAsynchPacketData);
              data_prev->next = pp;
            }
          else
            fastpath->send_packets_head = pp;
          fastpath->send_packets_tail = pp;
	  ssh_kernel_mutex_unlock(engine->pp_lock);
          return;
        }
    }

  ssh_kernel_mutex_unlock(engine->pp_lock);

#ifdef FASTPATH_ACCELERATOR_CONFIGURED

  /* Pass packet to accelerated fastpath. */
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  software_fastpath_execute_send(fastpath, pp, 0);
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  software_fastpath_execute_send(fastpath, pp, pc->media_hdr_len);
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

#else /* FASTPATH_ACCELERATOR_CONFIGURED */

  /* Send packet via interceptor. */
#ifdef SSH_IPSEC_IP_ONLY_INTERCEPTOR
  ssh_interceptor_send(engine->interceptor, pp, 0);
#else /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */
  ssh_interceptor_send(engine->interceptor, pp, pc->media_hdr_len);
#endif /* SSH_IPSEC_IP_ONLY_INTERCEPTOR */

#endif /* FASTPATH_ACCELERATOR_CONFIGURED */
}
#endif /* SSH_IPSEC_SEND_IS_SYNC */


#ifdef FASTPATH_ACCELERATOR_CONFIGURED
/* Function that is called whenever a packet is received from the accelerated
   fastpath. */
void
software_fastpath_packet_handler(SshEngine engine,
				 SshEnginePacketContext pc,
				 SshEngineActionRet input_state,
				 SshEngineActionRet return_state)
{
  pc->fastpath_accel_ret = return_state;
  fastpath_packet_continue(engine->fastpath, pc, input_state);
}
#endif /* FASTPATH_ACCELERATOR_CONFIGURED */

/* Function that is called whenever a packet is received from the
   interceptor.  This function will eventually free `pp', either by
   calling ssh_interceptor_packet_free on the packet or by passing it
   to the ssh_interceptor_send function.  Note that this function may
   be called asynchronously, concurrently with any other functions
   (including itself).

   When a packet is passed to this callback, the `pp->flags' field may
   contain arbitrary flags in the bits reserved for the interceptor
   (mask 0x000000ff).  This callback is not allowed to modify any of
   those bits; they must be passed intact to ssh_interceptor_send or
   ssh_interceptor_packet_free.  Any other bits (mask 0xffffff00) will
   be zero when the packet is sent to this callback; those bits may be
   used freely by this callback.  They are not used by the interceptor. */
void fastpath_packet_callback(SshInterceptorPacket pp, void *context)
{
  SshEngine engine = (SshEngine)context;




  SSH_DEBUG(SSH_D_LOWOK,
            ("packet_callback: pp=0x%p pp->flags=0x%lx engine=0x%p",
             pp, pp->flags, engine));

#ifdef SSH_IPSEC_UNIFIED_ADDRESS_SPACE
#ifdef USERMODE_ENGINE

#ifdef SSH_ENGINE_TOP_OF_STACK
#ifdef HAVE_ALLOCA
  /* Just allocate a meg on the stack, to ensure that we dont have
     any already allocated pages there. */
  alloca(1024*1024);
#endif /* HAVE_ALLOCA */
#endif /* SSH_ENGINE_TOP_OF_STACK */

  /* This really should not be called from the fastpath, but this
     is only used in usermode engine, so it is acceptable, even
     if dirty. This is mostly a debugging aid for stack use. */
  ssh_engine_pme_set_context_kernel(engine);
#endif /* USERMODE_ENGINE */
#endif /* SSH_IPSEC_UNIFIED_ADDRESS_SPACE */

#ifdef SSH_IPSEC_SMALL
  /* Schedule a packet triggered engine age timeout. */
  ssh_engine_age_timeout_schedule(engine, 0L, TRUE, FALSE, FALSE);
#endif /* SSH_IPSEC_SMALL */

  /* Handle incoming packets with tunnel id zero. */
  fastpath_packet_handler(engine, pp, 0, SSH_IPSEC_INVALID_INDEX, FALSE);

#ifdef SSH_IPSEC_UNIFIED_ADDRESS_SPACE
#ifdef USERMODE_ENGINE
  /* This really should not be called from the fastpath, but this
     is only used in usermode engine, so it is acceptable, even
     if dirty. This is mostly a debugging aid for stack use. */
  ssh_engine_pme_set_context_usermode(engine);
#endif /* USERMODE_ENGINE */
#endif /* SSH_IPSEC_UNIFIED_ADDRESS_SPACE */
}
