/*
  File: ndis_render.c

  Description:
        NDIS data type renderer related functions and definitions.

  Copyright:
        Copyright (c) 2008 SFNT Finland Oy.
        All rights reserved
*/

#ifdef DEBUG_LIGHT 

#include "sshincludes.h"
#include "interceptor_i.h"
#ifdef _WIN32_WCE
#include <ndiswwan.h>
#endif /* _WIN32_WCE */

#define SSH_DEBUG_MODULE "SshNdisRender"

typedef struct SshNdisOidRequestTypeStrMapRec
{
  NDIS_REQUEST_TYPE type;
  const unsigned char *str;
} SshNdisOidRequestTypeStrMapStruct, *SshNdisOidRequestTypeStrMap;

#define SSH_OID_REQUEST_TYPE_STR_MAP(request) \
  {(NDIS_REQUEST_TYPE)request, (const unsigned char *)#request}

typedef struct SshNdisOidStrMapRec
{
  NDIS_OID oid;
  const unsigned char *str;
} SshNdisOidStrMapStruct, *SshNdisOidStrMap;

#define SSH_OID_STR_MAP(oid) \
  {(NDIS_OID)oid, (const unsigned char *)#oid}

typedef struct SshNdisStatusStrMapRec
{
  NDIS_STATUS status;
  const unsigned char *str;
} SshNdisStatusStrMapStruct, *SshNdisStatusStrMap;

#define SSH_STATUS_STR_MAP(status) \
  {(NDIS_STATUS)status, (const unsigned char *)#status}

typedef struct SshUInt32StrMapRec
{
  SshUInt32 value;
  const unsigned char *str;
} SshUInt32StrMapStruct, *SshUInt32StrMap;

#define SSH_UINT32_STR_MAP(value) \
  {(SshUInt32)value, (const unsigned char *)#value}

const SshNdisOidRequestTypeStrMapStruct known_oid_req_types[] =
{ 
#ifdef NdisRequestMethod
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestMethod),
#endif /* NdisRequestMethod */
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestQueryInformation),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestSetInformation),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestQueryStatistics),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestOpen),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestClose),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestSend),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestTransferData),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestReset),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestGeneric1),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestGeneric2),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestGeneric3),
  SSH_OID_REQUEST_TYPE_STR_MAP(NdisRequestGeneric4)
};

const SshNdisOidStrMapStruct known_oids[] =
{ SSH_OID_STR_MAP(OID_GEN_SUPPORTED_LIST),
  SSH_OID_STR_MAP(OID_GEN_HARDWARE_STATUS),
  SSH_OID_STR_MAP(OID_GEN_MEDIA_SUPPORTED),
  SSH_OID_STR_MAP(OID_GEN_MEDIA_IN_USE),
  SSH_OID_STR_MAP(OID_GEN_MAXIMUM_LOOKAHEAD),
  SSH_OID_STR_MAP(OID_GEN_MAXIMUM_FRAME_SIZE),
  SSH_OID_STR_MAP(OID_GEN_LINK_SPEED),
  SSH_OID_STR_MAP(OID_GEN_TRANSMIT_BUFFER_SPACE),
  SSH_OID_STR_MAP(OID_GEN_RECEIVE_BUFFER_SPACE),
  SSH_OID_STR_MAP(OID_GEN_TRANSMIT_BLOCK_SIZE),
  SSH_OID_STR_MAP(OID_GEN_RECEIVE_BLOCK_SIZE),
  SSH_OID_STR_MAP(OID_GEN_VENDOR_ID),
  SSH_OID_STR_MAP(OID_GEN_VENDOR_DESCRIPTION),
  SSH_OID_STR_MAP(OID_GEN_CURRENT_PACKET_FILTER),
  SSH_OID_STR_MAP(OID_GEN_CURRENT_LOOKAHEAD),
  SSH_OID_STR_MAP(OID_GEN_DRIVER_VERSION),
  SSH_OID_STR_MAP(OID_GEN_MAXIMUM_TOTAL_SIZE),
  SSH_OID_STR_MAP(OID_GEN_PROTOCOL_OPTIONS),
  SSH_OID_STR_MAP(OID_GEN_MAC_OPTIONS),
  SSH_OID_STR_MAP(OID_GEN_MEDIA_CONNECT_STATUS),
  SSH_OID_STR_MAP(OID_GEN_MAXIMUM_SEND_PACKETS),
  SSH_OID_STR_MAP(OID_GEN_VENDOR_DRIVER_VERSION),
  SSH_OID_STR_MAP(OID_GEN_SUPPORTED_GUIDS),
  SSH_OID_STR_MAP(OID_GEN_NETWORK_LAYER_ADDRESSES),
  SSH_OID_STR_MAP(OID_GEN_TRANSPORT_HEADER_OFFSET),
  SSH_OID_STR_MAP(OID_GEN_MACHINE_NAME),
  SSH_OID_STR_MAP(OID_GEN_RNDIS_CONFIG_PARAMETER),
  SSH_OID_STR_MAP(OID_GEN_VLAN_ID),
  SSH_OID_STR_MAP(OID_GEN_MEDIA_CAPABILITIES),
  SSH_OID_STR_MAP(OID_GEN_PHYSICAL_MEDIUM),
  SSH_OID_STR_MAP(OID_GEN_XMIT_OK),
  SSH_OID_STR_MAP(OID_GEN_RCV_OK),
  SSH_OID_STR_MAP(OID_GEN_XMIT_ERROR),
  SSH_OID_STR_MAP(OID_GEN_RCV_ERROR),
  SSH_OID_STR_MAP(OID_GEN_RCV_NO_BUFFER),
  SSH_OID_STR_MAP(OID_GEN_DIRECTED_BYTES_XMIT),
  SSH_OID_STR_MAP(OID_GEN_DIRECTED_FRAMES_XMIT),
  SSH_OID_STR_MAP(OID_GEN_MULTICAST_BYTES_XMIT),
  SSH_OID_STR_MAP(OID_GEN_MULTICAST_FRAMES_XMIT),
  SSH_OID_STR_MAP(OID_GEN_BROADCAST_BYTES_XMIT),
  SSH_OID_STR_MAP(OID_GEN_BROADCAST_FRAMES_XMIT),
  SSH_OID_STR_MAP(OID_GEN_DIRECTED_BYTES_RCV),
  SSH_OID_STR_MAP(OID_GEN_DIRECTED_FRAMES_RCV),
  SSH_OID_STR_MAP(OID_GEN_MULTICAST_BYTES_RCV),
  SSH_OID_STR_MAP(OID_GEN_MULTICAST_FRAMES_RCV),
  SSH_OID_STR_MAP(OID_GEN_BROADCAST_BYTES_RCV),
  SSH_OID_STR_MAP(OID_GEN_BROADCAST_FRAMES_RCV),
  SSH_OID_STR_MAP(OID_GEN_RCV_CRC_ERROR),
#ifdef OID_GEN_STATISTICS
  SSH_OID_STR_MAP(OID_GEN_STATISTICS),
#endif /* OID_GEN_STATISTICS */
  SSH_OID_STR_MAP(OID_GEN_TRANSMIT_QUEUE_LENGTH),
  SSH_OID_STR_MAP(OID_GEN_GET_TIME_CAPS),
  SSH_OID_STR_MAP(OID_GEN_GET_NETCARD_TIME),
  SSH_OID_STR_MAP(OID_GEN_NETCARD_LOAD),
  SSH_OID_STR_MAP(OID_GEN_DEVICE_PROFILE),
  SSH_OID_STR_MAP(OID_GEN_INIT_TIME_MS),
  SSH_OID_STR_MAP(OID_GEN_RESET_COUNTS),
  SSH_OID_STR_MAP(OID_GEN_MEDIA_SENSE_COUNTS),
  SSH_OID_STR_MAP(OID_GEN_FRIENDLY_NAME),
#ifdef OID_GEN_MINIPORT_INFO
  SSH_OID_STR_MAP(OID_GEN_MINIPORT_INFO),
#endif /* OID_GEN_MINIPORT_INFO */
#ifdef OID_GEN_RESET_VERIFY_PARAMETERS
  SSH_OID_STR_MAP(OID_GEN_RESET_VERIFY_PARAMETERS),
#endif /* OID_GEN_RESET_VERIFY_PARAMETERS */
#ifdef OID_GEN_RECEIVE_SCALE_CAPABILITIES
  SSH_OID_STR_MAP(OID_GEN_RECEIVE_SCALE_CAPABILITIES),
#endif /* OID_GEN_RECEIVE_SCALE_CAPABILITIES */
#ifdef OID_GEN_RECEIVE_SCALE_PARAMETERS
  SSH_OID_STR_MAP(OID_GEN_RECEIVE_SCALE_PARAMETERS),
#endif /* OID_GEN_RECEIVE_SCALE_PARAMETERS */
#ifdef OID_GEN_MAC_ADDRESS
  SSH_OID_STR_MAP(OID_GEN_MAC_ADDRESS),
#endif /* OID_GEN_MAC_ADDRESS */
#ifdef OID_GEN_MAX_LINK_SPEED
  SSH_OID_STR_MAP(OID_GEN_MAX_LINK_SPEED),
#endif /* OID_GEN_MAX_LINK_SPEED */
#ifdef OID_GEN_LINK_STATE
  SSH_OID_STR_MAP(OID_GEN_LINK_STATE),
#endif /* OID_GEN_LINK_STATE */
#ifdef OID_GEN_LINK_PARAMETERS 
  SSH_OID_STR_MAP(OID_GEN_LINK_PARAMETERS),
#endif /* OID_GEN_LINK_PARAMETERS */
#ifdef OID_GEN_INTERRUPT_MODERATION
  SSH_OID_STR_MAP(OID_GEN_INTERRUPT_MODERATION),
#endif /* OID_GEN_INTERRUPT_MODERATION */
#ifdef OID_GEN_NDIS_RESERVED_3
  SSH_OID_STR_MAP(OID_GEN_NDIS_RESERVED_3),
#endif /* OID_GEN_NDIS_RESERVED_3 */
#ifdef OID_GEN_NDIS_RESERVED_4
  SSH_OID_STR_MAP(OID_GEN_NDIS_RESERVED_4),
#endif /* OID_GEN_NDIS_RESERVED_4 */
#ifdef OID_GEN_NDIS_RESERVED_5
  SSH_OID_STR_MAP(OID_GEN_NDIS_RESERVED_5),
#endif /* OID_GEN_NDIS_RESERVED_5 */
#ifdef OID_GEN_ENUMERATE_PORTS
  SSH_OID_STR_MAP(OID_GEN_ENUMERATE_PORTS),
#endif /* OID_GEN_ENUMERATE_PORTS */
#ifdef OID_GEN_PORT_STATE
  SSH_OID_STR_MAP(OID_GEN_PORT_STATE),
#endif /* OID_GEN_PORT_STATE */
#ifdef OID_GEN_PORT_AUTHENTICATION_PARAMETERS
  SSH_OID_STR_MAP(OID_GEN_PORT_AUTHENTICATION_PARAMETERS),
#endif /* OID_GEN_PORT_AUTHENTICATION_PARAMETERS */
#ifdef OID_GEN_TIMEOUT_DPC_REQUEST_CAPABILITIES
  SSH_OID_STR_MAP(OID_GEN_TIMEOUT_DPC_REQUEST_CAPABILITIES),
#endif /* OID_GEN_TIMEOUT_DPC_REQUEST_CAPABILITIES */
#ifdef OID_GEN_PCI_DEVICE_CUSTOM_PROPERTIES
  SSH_OID_STR_MAP(OID_GEN_PCI_DEVICE_CUSTOM_PROPERTIES),
#endif /* OID_GEN_PCI_DEVICE_CUSTOM_PROPERTIES */
#ifdef OID_GEN_NDIS_RESERVED_6
  SSH_OID_STR_MAP(OID_GEN_NDIS_RESERVED_6),
#endif /* OID_GEN_NDIS_RESERVED_6 */
#ifdef OID_GEN_RECEIVE_HASH
  SSH_OID_STR_MAP(OID_GEN_RECEIVE_HASH),
#endif /* OID_GEN_RECEIVE_HASH */
#ifdef OID_GEN_MINIPORT_RESTART_ATTRIBUTES
  SSH_OID_STR_MAP(OID_GEN_MINIPORT_RESTART_ATTRIBUTES),
#endif /* OID_GEN_MINIPORT_RESTART_ATTRIBUTES */
#ifdef OID_GEN_HD_SPLIT_PARAMETERS
  SSH_OID_STR_MAP(OID_GEN_HD_SPLIT_PARAMETERS),
#endif /* OID_GEN_HD_SPLIT_PARAMETERS */
#ifdef OID_GEN_HD_SPLIT_CURRENT_CONFIG
  SSH_OID_STR_MAP(OID_GEN_HD_SPLIT_CURRENT_CONFIG),
#endif /* OID_GEN_HD_SPLIT_CURRENT_CONFIG */
#ifdef OID_GEN_PROMISCUOUS_MODE
  SSH_OID_STR_MAP(OID_GEN_PROMISCUOUS_MODE),
#endif /* OID_GEN_PROMISCUOUS_MODE */
#ifdef OID_GEN_LAST_CHANGE
  SSH_OID_STR_MAP(OID_GEN_LAST_CHANGE),
#endif /* OID_GEN_LAST_CHANGE */
#ifdef OID_GEN_DISCONTINUITY_TIME
  SSH_OID_STR_MAP(OID_GEN_DISCONTINUITY_TIME),
#endif /* OID_GEN_DISCONTINUITY_TIME */
#ifdef OID_GEN_OPERATIONAL_STATUS
  SSH_OID_STR_MAP(OID_GEN_OPERATIONAL_STATUS),
#endif /* OID_GEN_OPERATIONAL_STATUS */
#ifdef OID_GEN_XMIT_LINK_SPEED
  SSH_OID_STR_MAP(OID_GEN_XMIT_LINK_SPEED),
#endif /* OID_GEN_XMIT_LINK_SPEED */
#ifdef OID_GEN_RCV_LINK_SPEED
  SSH_OID_STR_MAP(OID_GEN_RCV_LINK_SPEED),
#endif /* OID_GEN_RCV_LINK_SPEED */
#ifdef OID_GEN_UNKNOWN_PROTOS
  SSH_OID_STR_MAP(OID_GEN_UNKNOWN_PROTOS),
#endif /* OID_GEN_UNKNOWN_PROTOS */
#ifdef OID_GEN_INTERFACE_INFO
  SSH_OID_STR_MAP(OID_GEN_INTERFACE_INFO),
#endif /* OID_GEN_INTERFACE_INFO */
#ifdef OID_GEN_ADMIN_STATUS
  SSH_OID_STR_MAP(OID_GEN_ADMIN_STATUS),
#endif /* OID_GEN_ADMIN_STATUS */
#ifdef OID_GEN_ALIAS
  SSH_OID_STR_MAP(OID_GEN_ALIAS),
#endif /* OID_GEN_ALIAS */
#ifdef OID_GEN_MEDIA_CONNECT_STATUS_EX
  SSH_OID_STR_MAP(OID_GEN_MEDIA_CONNECT_STATUS_EX),
#endif /* OID_GEN_MEDIA_CONNECT_STATUS_EX */
#ifdef OID_GEN_LINK_SPEED_EX
  SSH_OID_STR_MAP(OID_GEN_LINK_SPEED_EX),
#endif /* OID_GEN_LINK_SPEED_EX */
#ifdef OID_GEN_MEDIA_DUPLEX_STATE
  SSH_OID_STR_MAP(OID_GEN_MEDIA_DUPLEX_STATE),
#endif /* OID_GEN_MEDIA_DUPLEX_STATE */

  SSH_OID_STR_MAP(OID_802_3_PERMANENT_ADDRESS),
  SSH_OID_STR_MAP(OID_802_3_CURRENT_ADDRESS),
  SSH_OID_STR_MAP(OID_802_3_MULTICAST_LIST),
  SSH_OID_STR_MAP(OID_802_3_MAXIMUM_LIST_SIZE),
  SSH_OID_STR_MAP(OID_802_3_MAC_OPTIONS),
  SSH_OID_STR_MAP(OID_802_3_RCV_ERROR_ALIGNMENT),
  SSH_OID_STR_MAP(OID_802_3_XMIT_ONE_COLLISION),
  SSH_OID_STR_MAP(OID_802_3_XMIT_MORE_COLLISIONS),
  SSH_OID_STR_MAP(OID_802_3_XMIT_DEFERRED),
  SSH_OID_STR_MAP(OID_802_3_XMIT_MAX_COLLISIONS),
  SSH_OID_STR_MAP(OID_802_3_RCV_OVERRUN),
  SSH_OID_STR_MAP(OID_802_3_XMIT_UNDERRUN),         
  SSH_OID_STR_MAP(OID_802_3_XMIT_HEARTBEAT_FAILURE),
  SSH_OID_STR_MAP(OID_802_3_XMIT_TIMES_CRS_LOST), 
  SSH_OID_STR_MAP(OID_802_3_XMIT_LATE_COLLISIONS),

  SSH_OID_STR_MAP(OID_WAN_PERMANENT_ADDRESS),
  SSH_OID_STR_MAP(OID_WAN_CURRENT_ADDRESS),
  SSH_OID_STR_MAP(OID_WAN_QUALITY_OF_SERVICE),
  SSH_OID_STR_MAP(OID_WAN_PROTOCOL_TYPE),
  SSH_OID_STR_MAP(OID_WAN_MEDIUM_SUBTYPE),
  SSH_OID_STR_MAP(OID_WAN_HEADER_FORMAT),
  SSH_OID_STR_MAP(OID_WAN_GET_INFO),
  SSH_OID_STR_MAP(OID_WAN_SET_LINK_INFO),
  SSH_OID_STR_MAP(OID_WAN_GET_LINK_INFO),
  SSH_OID_STR_MAP(OID_WAN_LINE_COUNT),
  SSH_OID_STR_MAP(OID_WAN_PROTOCOL_CAPS),
  SSH_OID_STR_MAP(OID_WAN_GET_BRIDGE_INFO),
  SSH_OID_STR_MAP(OID_WAN_SET_BRIDGE_INFO),
  SSH_OID_STR_MAP(OID_WAN_GET_COMP_INFO),
  SSH_OID_STR_MAP(OID_WAN_SET_COMP_INFO),
  SSH_OID_STR_MAP(OID_WAN_GET_STATS_INFO),

#ifdef OID_WW_GEN_NETWORK_TYPES_SUPPORTED  
  SSH_OID_STR_MAP(OID_WW_GEN_NETWORK_TYPES_SUPPORTED),
#endif /* OID_WW_GEN_NETWORK_TYPES_SUPPORTED */
#ifdef OID_WW_GEN_NETWORK_TYPE_IN_USE  
  SSH_OID_STR_MAP(OID_WW_GEN_NETWORK_TYPE_IN_USE),
#endif /* OID_WW_GEN_NETWORK_TYPE_IN_USE */
#ifdef OID_WW_GEN_HEADER_FORMATS_SUPPORTED  
  SSH_OID_STR_MAP(OID_WW_GEN_HEADER_FORMATS_SUPPORTED),
#endif /* OID_WW_GEN_HEADER_FORMATS_SUPPORTED */
#ifdef OID_WW_GEN_HEADER_FORMAT_IN_USE  
  SSH_OID_STR_MAP(OID_WW_GEN_HEADER_FORMAT_IN_USE),
#endif /* OID_WW_GEN_HEADER_FORMAT_IN_USE */
#ifdef OID_WW_GEN_INDICATION_REQUEST  
  SSH_OID_STR_MAP(OID_WW_GEN_INDICATION_REQUEST),
#endif /* OID_WW_GEN_INDICATION_REQUEST */
#ifdef OID_WW_GEN_DEVICE_INFO  
  SSH_OID_STR_MAP(OID_WW_GEN_DEVICE_INFO),
#endif /* OID_WW_GEN_DEVICE_INFO */
#ifdef OID_WW_GEN_OPERATION_MODE  
  SSH_OID_STR_MAP(OID_WW_GEN_OPERATION_MODE),
#endif /* OID_WW_GEN_OPERATION_MODE */
#ifdef OID_WW_GEN_LOCK_STATUS  
  SSH_OID_STR_MAP(OID_WW_GEN_LOCK_STATUS),
#endif /* OID_WW_GEN_LOCK_STATUS */
#ifdef OID_WW_GEN_DISABLE_TRANSMITTER  
  SSH_OID_STR_MAP(OID_WW_GEN_DISABLE_TRANSMITTER),
#endif /* OID_WW_GEN_DISABLE_TRANSMITTER */
#ifdef OID_WW_GEN_NETWORK_ID 
  SSH_OID_STR_MAP(OID_WW_GEN_NETWORK_ID),
#endif /* OID_WW_GEN_NETWORK_ID */
#ifdef OID_WW_GEN_PERMANENT_ADDRESS  
  SSH_OID_STR_MAP(OID_WW_GEN_PERMANENT_ADDRESS),
#endif /* OID_WW_GEN_PERMANENT_ADDRESS */
#ifdef OID_WW_GEN_CURRENT_ADDRESS  
  SSH_OID_STR_MAP(OID_WW_GEN_CURRENT_ADDRESS),
#endif /* OID_WW_GEN_CURRENT_ADDRESS */
#ifdef OID_WW_GEN_SUSPEND_DRIVER  
  SSH_OID_STR_MAP(OID_WW_GEN_SUSPEND_DRIVER),
#endif /* OID_WW_GEN_SUSPEND_DRIVER */
#ifdef OID_WW_GEN_BASESTATION_ID  
  SSH_OID_STR_MAP(OID_WW_GEN_BASESTATION_ID),
#endif /* OID_WW_GEN_BASESTATION_ID */
#ifdef OID_WW_GEN_CHANNEL_ID  
  SSH_OID_STR_MAP(OID_WW_GEN_CHANNEL_ID),
#endif /* OID_WW_GEN_CHANNEL_ID */
#ifdef OID_WW_GEN_ENCRYPTION_SUPPORTED  
  SSH_OID_STR_MAP(OID_WW_GEN_ENCRYPTION_SUPPORTED),
#endif /* OID_WW_GEN_ENCRYPTION_SUPPORTED */
#ifdef OID_WW_GEN_ENCRYPTION_IN_USE  
  SSH_OID_STR_MAP(OID_WW_GEN_ENCRYPTION_IN_USE),
#endif /* OID_WW_GEN_ENCRYPTION_IN_USE */
#ifdef OID_WW_GEN_ENCRYPTION_STATE  
  SSH_OID_STR_MAP(OID_WW_GEN_ENCRYPTION_STATE),
#endif /* OID_WW_GEN_ENCRYPTION_STATE */
#ifdef OID_WW_GEN_CHANNEL_QUALITY  
  SSH_OID_STR_MAP(OID_WW_GEN_CHANNEL_QUALITY),
#endif /* OID_WW_GEN_CHANNEL_QUALITY */
#ifdef OID_WW_GEN_REGISTRATION_STATUS  
  SSH_OID_STR_MAP(OID_WW_GEN_REGISTRATION_STATUS),
#endif /* OID_WW_GEN_REGISTRATION_STATUS */
#ifdef OID_WW_GEN_RADIO_LINK_SPEED  
  SSH_OID_STR_MAP(OID_WW_GEN_RADIO_LINK_SPEED),
#endif /* OID_WW_GEN_RADIO_LINK_SPEED */
#ifdef OID_WW_GEN_LATENCY  
  SSH_OID_STR_MAP(OID_WW_GEN_LATENCY),
#endif /* OID_WW_GEN_LATENCY */
#ifdef OID_WW_GEN_BATTERY_LEVEL  
  SSH_OID_STR_MAP(OID_WW_GEN_BATTERY_LEVEL),
#endif /* OID_WW_GEN_BATTERY_LEVEL */
#ifdef OID_WW_GEN_EXTERNAL_POWER  
  SSH_OID_STR_MAP(OID_WW_GEN_EXTERNAL_POWER),
#endif /* OID_WW_GEN_EXTERNAL_POWER */
#ifdef OID_WW_NEGOTIATE_VERSION
  SSH_OID_STR_MAP(OID_WW_NEGOTIATE_VERSION),
#endif /* OID_WW_NEGOTIATE_VERSION */
#ifdef OID_WW_DEVICE_CAPS
  SSH_OID_STR_MAP(OID_WW_DEVICE_CAPS),
#endif /* OID_WW_DEVICE_CAPS */
#ifdef OID_WW_INITIALIZE
  SSH_OID_STR_MAP(OID_WW_INITIALIZE),
#endif /* OID_WW_INITIALIZE */
#ifdef OID_WW_SHUTDOWN
  SSH_OID_STR_MAP(OID_WW_SHUTDOWN),
#endif /* OID_WW_SHUTDOWN */
#ifdef OID_WW_GET_READY_STATE
  SSH_OID_STR_MAP(OID_WW_GET_READY_STATE),
#endif /* OID_WW_GET_READY_STATE */
#ifdef OID_WW_GET_PIN_STATE
  SSH_OID_STR_MAP(OID_WW_GET_PIN_STATE),
#endif /* OID_WW_GET_PIN_STATE */
#ifdef OID_WW_SET_PIN
  SSH_OID_STR_MAP(OID_WW_SET_PIN),
#endif /* OID_WW_SET_PIN */
#ifdef OID_WW_GET_PREFERRED_PROVIDERS
  SSH_OID_STR_MAP(OID_WW_GET_PREFERRED_PROVIDERS),
#endif /* OID_WW_GET_PREFERRED_PROVIDERS */
#ifdef OID_WW_SET_PREFERRED_PROVIDERS
  SSH_OID_STR_MAP(OID_WW_SET_PREFERRED_PROVIDERS),
#endif /* OID_WW_SET_PREFERRED_PROVIDERS */
#ifdef OID_WW_GET_VISIBLE_PROVIDERS
  SSH_OID_STR_MAP(OID_WW_GET_VISIBLE_PROVIDERS),
#endif /* OID_WW_GET_VISIBLE_PROVIDERS */
#ifdef OID_WW_GET_REGISTER_STATE
  SSH_OID_STR_MAP(OID_WW_GET_REGISTER_STATE),
#endif /* OID_WW_GET_REGISTER_STATE */
#ifdef OID_WW_SET_REGISTER_STATE
  SSH_OID_STR_MAP(OID_WW_SET_REGISTER_STATE),
#endif /* OID_WW_SET_REGISTER_STATE */
#ifdef OID_WW_GET_PACKET_SERVICE
  SSH_OID_STR_MAP(OID_WW_GET_PACKET_SERVICE),
#endif /* OID_WW_GET_PACKET_SERVICE */
#ifdef OID_WW_SET_SIGNAL_SUGGESTION
  SSH_OID_STR_MAP(OID_WW_SET_SIGNAL_SUGGESTION),
#endif /* OID_WW_SET_SIGNAL_SUGGESTION */
#ifdef OID_WW_GET_SIGNAL_STATE
  SSH_OID_STR_MAP(OID_WW_GET_SIGNAL_STATE),
#endif /* OID_WW_GET_SIGNAL_STATE */
#ifdef OID_WW_GET_PROVISIONED_CONTEXTS
  SSH_OID_STR_MAP(OID_WW_GET_PROVISIONED_CONTEXTS),
#endif /* OID_WW_GET_PROVISIONED_CONTEXTS */
#ifdef OID_WW_SET_PROVISIONED_CONTEXTS
  SSH_OID_STR_MAP(OID_WW_SET_PROVISIONED_CONTEXTS),
#endif /* OID_WW_SET_PROVISIONED_CONTEXTS */
#ifdef OID_WW_GET_ACTIVATION_STATE
  SSH_OID_STR_MAP(OID_WW_GET_ACTIVATION_STATE),
#endif /* OID_WW_GET_ACTIVATION_STATE */
#ifdef OID_WW_SET_ACTIVATION_STATE
  SSH_OID_STR_MAP(OID_WW_SET_ACTIVATION_STATE),
#endif /* OID_WW_SET_ACTIVATION_STATE */
#ifdef OID_WW_GET_LOCATION_INFO
  SSH_OID_STR_MAP(OID_WW_GET_LOCATION_INFO),
#endif /* OID_WW_GET_LOCATION_INFO */
#ifdef OID_WW_SET_LOCATION_INFO
  SSH_OID_STR_MAP(OID_WW_SET_LOCATION_INFO),
#endif /* OID_WW_SET_LOCATION_INFO */
#ifdef OID_WW_VENDOR_SPECIFIC
  SSH_OID_STR_MAP(OID_WW_VENDOR_SPECIFIC),
#endif /* OID_WW_VENDOR_SPECIFIC */

  SSH_OID_STR_MAP(OID_PNP_CAPABILITIES),
  SSH_OID_STR_MAP(OID_PNP_SET_POWER),
  SSH_OID_STR_MAP(OID_PNP_QUERY_POWER),
  SSH_OID_STR_MAP(OID_PNP_ADD_WAKE_UP_PATTERN),
  SSH_OID_STR_MAP(OID_PNP_REMOVE_WAKE_UP_PATTERN),
  SSH_OID_STR_MAP(OID_PNP_WAKE_UP_PATTERN_LIST),
  SSH_OID_STR_MAP(OID_PNP_ENABLE_WAKE_UP),
  SSH_OID_STR_MAP(OID_PNP_WAKE_UP_OK),
  SSH_OID_STR_MAP(OID_PNP_WAKE_UP_ERROR),

  SSH_OID_STR_MAP(OID_TCP_TASK_OFFLOAD),
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_ADD_SA),
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_DELETE_SA),
  SSH_OID_STR_MAP(OID_TCP_SAN_SUPPORT),
#ifdef OID_TCP_TASK_IPSEC_ADD_UDPESP_SA
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_ADD_UDPESP_SA),
#endif /* OID_TCP_TASK_IPSEC_ADD_UDPESP_SA */
#ifdef OID_TCP_TASK_IPSEC_DELETE_UDPESP_SA
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_DELETE_UDPESP_SA),
#endif /* OID_TCP_TASK_IPSEC_DELETE_UDPESP_SA */
#ifdef OID_TCP4_OFFLOAD_STATS
  SSH_OID_STR_MAP(OID_TCP4_OFFLOAD_STATS),
#endif /* OID_TCP4_OFFLOAD_STATS */
#ifdef OID_TCP6_OFFLOAD_STATS
  SSH_OID_STR_MAP(OID_TCP6_OFFLOAD_STATS),
#endif /* OID_TCP6_OFFLOAD_STATS */
#ifdef OID_IP4_OFFLOAD_STATS
  SSH_OID_STR_MAP(OID_IP4_OFFLOAD_STATS),
#endif /* OID_IP4_OFFLOAD_STATS */
#ifdef OID_IP6_OFFLOAD_STATS
  SSH_OID_STR_MAP(OID_IP6_OFFLOAD_STATS),
#endif /* OID_IP6_OFFLOAD_STATS */
#ifdef OID_TCP_OFFLOAD_CURRENT_CONFIG
  SSH_OID_STR_MAP(OID_TCP_OFFLOAD_CURRENT_CONFIG),
#endif /* OID_TCP_OFFLOAD_CURRENT_CONFIG */
#ifdef OID_TCP_OFFLOAD_PARAMETERS
  SSH_OID_STR_MAP(OID_TCP_OFFLOAD_PARAMETERS),
#endif /* OID_TCP_OFFLOAD_PARAMETERS */
#ifdef OID_TCP_OFFLOAD_HARDWARE_CAPABILITIES
  SSH_OID_STR_MAP(OID_TCP_OFFLOAD_HARDWARE_CAPABILITIES),
#endif /* OID_TCP_OFFLOAD_HARDWARE_CAPABILITIES */
#ifdef OID_TCP_CONNECTION_OFFLOAD_CURRENT_CONFIG
  SSH_OID_STR_MAP(OID_TCP_CONNECTION_OFFLOAD_CURRENT_CONFIG),
#endif /* OID_TCP_CONNECTION_OFFLOAD_CURRENT_CONFIG */
#ifdef OID_TCP_CONNECTION_OFFLOAD_HARDWARE_CAPABILITIES
  SSH_OID_STR_MAP(OID_TCP_CONNECTION_OFFLOAD_HARDWARE_CAPABILITIES),
#endif /* OID_TCP_CONNECTION_OFFLOAD_HARDWARE_CAPABILITIES */
#ifdef OID_OFFLOAD_ENCAPSULATION
  SSH_OID_STR_MAP(OID_OFFLOAD_ENCAPSULATION),
#endif /* OID_OFFLOAD_ENCAPSULATION */
#ifdef OID_TCP_TASK_IPSEC_OFFLOAD_V2_ADD_SA
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_OFFLOAD_V2_ADD_SA),
#endif /* OID_TCP_TASK_IPSEC_OFFLOAD_V2_ADD_SA */
#ifdef OID_TCP_TASK_IPSEC_OFFLOAD_V2_DELETE_SA
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_OFFLOAD_V2_DELETE_SA),
#endif /* OID_TCP_TASK_IPSEC_OFFLOAD_V2_DELETE_SA */
#ifdef OID_TCP_TASK_IPSEC_OFFLOAD_V2_UPDATE_SA
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_OFFLOAD_V2_UPDATE_SA),
#endif /* OID_TCP_TASK_IPSEC_OFFLOAD_V2_UPDATE_SA */

  SSH_OID_STR_MAP(OID_802_11_BSSID),
  SSH_OID_STR_MAP(OID_802_11_SSID),
  SSH_OID_STR_MAP(OID_802_11_NETWORK_TYPES_SUPPORTED),
  SSH_OID_STR_MAP(OID_802_11_NETWORK_TYPE_IN_USE),
  SSH_OID_STR_MAP(OID_802_11_TX_POWER_LEVEL),
  SSH_OID_STR_MAP(OID_802_11_RSSI),
  SSH_OID_STR_MAP(OID_802_11_RSSI_TRIGGER),
  SSH_OID_STR_MAP(OID_802_11_INFRASTRUCTURE_MODE),
  SSH_OID_STR_MAP(OID_802_11_FRAGMENTATION_THRESHOLD),
  SSH_OID_STR_MAP(OID_802_11_RTS_THRESHOLD),
  SSH_OID_STR_MAP(OID_802_11_NUMBER_OF_ANTENNAS),
  SSH_OID_STR_MAP(OID_802_11_RX_ANTENNA_SELECTED),
  SSH_OID_STR_MAP(OID_802_11_TX_ANTENNA_SELECTED),
  SSH_OID_STR_MAP(OID_802_11_SUPPORTED_RATES),
  SSH_OID_STR_MAP(OID_802_11_DESIRED_RATES),
  SSH_OID_STR_MAP(OID_802_11_CONFIGURATION),
  SSH_OID_STR_MAP(OID_802_11_STATISTICS),
  SSH_OID_STR_MAP(OID_802_11_ADD_WEP),
  SSH_OID_STR_MAP(OID_802_11_REMOVE_WEP),
  SSH_OID_STR_MAP(OID_802_11_DISASSOCIATE),
  SSH_OID_STR_MAP(OID_802_11_POWER_MODE),
  SSH_OID_STR_MAP(OID_802_11_BSSID_LIST),
  SSH_OID_STR_MAP(OID_802_11_AUTHENTICATION_MODE),
  SSH_OID_STR_MAP(OID_802_11_PRIVACY_FILTER),
  SSH_OID_STR_MAP(OID_802_11_BSSID_LIST_SCAN),
  SSH_OID_STR_MAP(OID_802_11_ENCRYPTION_STATUS),
  SSH_OID_STR_MAP(OID_802_11_RELOAD_DEFAULTS),
  SSH_OID_STR_MAP(OID_802_11_ADD_KEY),
  SSH_OID_STR_MAP(OID_802_11_REMOVE_KEY),
  SSH_OID_STR_MAP(OID_802_11_ASSOCIATION_INFORMATION),
  SSH_OID_STR_MAP(OID_802_11_TEST),
#ifdef OID_802_11_MEDIA_STREAM_MODE
  SSH_OID_STR_MAP(OID_802_11_MEDIA_STREAM_MODE),
#endif /* OID_802_11_MEDIA_STREAM_MODE */
#ifdef OID_802_11_CAPABILITY
  SSH_OID_STR_MAP(OID_802_11_CAPABILITY),
#endif /* OID_802_11_CAPABILITY */
#ifdef OID_802_11_PMKID
  SSH_OID_STR_MAP(OID_802_11_PMKID),
#endif /* OID_802_11_PMKID */
#ifdef OID_802_11_NON_BCAST_SSID_LIST
  SSH_OID_STR_MAP(OID_802_11_NON_BCAST_SSID_LIST),
#endif /* OID_802_11_NON_BCAST_SSID_LIST */
#ifdef OID_802_11_RADIO_STATUS
  SSH_OID_STR_MAP(OID_802_11_RADIO_STATUS),
#endif /* OID_802_11_RADIO_STATUS */

  SSH_OID_STR_MAP(OID_TAPI_ACCEPT),
  SSH_OID_STR_MAP(OID_TAPI_ANSWER),
  SSH_OID_STR_MAP(OID_TAPI_CLOSE),
  SSH_OID_STR_MAP(OID_TAPI_CLOSE_CALL),
  SSH_OID_STR_MAP(OID_TAPI_CONDITIONAL_MEDIA_DETECTION),
  SSH_OID_STR_MAP(OID_TAPI_CONFIG_DIALOG),
  SSH_OID_STR_MAP(OID_TAPI_DEV_SPECIFIC),
  SSH_OID_STR_MAP(OID_TAPI_DIAL),
  SSH_OID_STR_MAP(OID_TAPI_DROP),
  SSH_OID_STR_MAP(OID_TAPI_GET_ADDRESS_CAPS),
  SSH_OID_STR_MAP(OID_TAPI_GET_ADDRESS_ID),
  SSH_OID_STR_MAP(OID_TAPI_GET_ADDRESS_STATUS),
  SSH_OID_STR_MAP(OID_TAPI_GET_CALL_ADDRESS_ID),
  SSH_OID_STR_MAP(OID_TAPI_GET_CALL_INFO),
  SSH_OID_STR_MAP(OID_TAPI_GET_CALL_STATUS),
  SSH_OID_STR_MAP(OID_TAPI_GET_DEV_CAPS),
  SSH_OID_STR_MAP(OID_TAPI_GET_DEV_CONFIG),
  SSH_OID_STR_MAP(OID_TAPI_GET_EXTENSION_ID),
  SSH_OID_STR_MAP(OID_TAPI_GET_ID),
  SSH_OID_STR_MAP(OID_TAPI_GET_LINE_DEV_STATUS),
  SSH_OID_STR_MAP(OID_TAPI_MAKE_CALL),
  SSH_OID_STR_MAP(OID_TAPI_NEGOTIATE_EXT_VERSION),
  SSH_OID_STR_MAP(OID_TAPI_OPEN),
  SSH_OID_STR_MAP(OID_TAPI_PROVIDER_INITIALIZE),
  SSH_OID_STR_MAP(OID_TAPI_PROVIDER_SHUTDOWN),
  SSH_OID_STR_MAP(OID_TAPI_SECURE_CALL),
  SSH_OID_STR_MAP(OID_TAPI_SELECT_EXT_VERSION),
  SSH_OID_STR_MAP(OID_TAPI_SEND_USER_USER_INFO),
  SSH_OID_STR_MAP(OID_TAPI_SET_APP_SPECIFIC),
  SSH_OID_STR_MAP(OID_TAPI_SET_CALL_PARAMS),
  SSH_OID_STR_MAP(OID_TAPI_SET_DEFAULT_MEDIA_DETECTION),
  SSH_OID_STR_MAP(OID_TAPI_SET_DEV_CONFIG),
  SSH_OID_STR_MAP(OID_TAPI_SET_MEDIA_MODE),
  SSH_OID_STR_MAP(OID_TAPI_SET_STATUS_MESSAGES),
  SSH_OID_STR_MAP(OID_TAPI_GATHER_DIGITS),
  SSH_OID_STR_MAP(OID_TAPI_MONITOR_DIGITS)
#if NDIS_SUPPORT_NDIS61
  , /* NDIS 6.1 */
  SSH_OID_STR_MAP(OID_GEN_HD_SPLIT_PARAMETERS),
  SSH_OID_STR_MAP(OID_GEN_HD_SPLIT_CURRENT_CONFIG),
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_OFFLOAD_V2_ADD_SA),
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_OFFLOAD_V2_DELETE_SA),
  SSH_OID_STR_MAP(OID_TCP_TASK_IPSEC_OFFLOAD_V2_UPDATE_SA)
#endif /* NDIS_SUPPORT_NDIS61 */
#if NDIS_SUPPORT_NDIS620
  , /* NDIS 6.20 */
  SSH_OID_STR_MAP(OID_PM_CURRENT_CAPABILITIES),
  SSH_OID_STR_MAP(OID_PM_HARDWARE_CAPABILITIES),
  SSH_OID_STR_MAP(OID_PM_PARAMETERS),
  SSH_OID_STR_MAP(OID_PM_ADD_WOL_PATTERN),
  SSH_OID_STR_MAP(OID_PM_REMOVE_WOL_PATTERN),
  SSH_OID_STR_MAP(OID_PM_WOL_PATTERN_LIST),
  SSH_OID_STR_MAP(OID_PM_ADD_PROTOCOL_OFFLOAD),
  SSH_OID_STR_MAP(OID_PM_GET_PROTOCOL_OFFLOAD),
  SSH_OID_STR_MAP(OID_PM_REMOVE_PROTOCOL_OFFLOAD),
  SSH_OID_STR_MAP(OID_PM_PROTOCOL_OFFLOAD_LIST),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_HARDWARE_CAPABILITIES),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_GLOBAL_PARAMETERS),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_ALLOCATE_QUEUE),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_FREE_QUEUE),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_ENUM_QUEUES),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_QUEUE_PARAMETERS),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_SET_FILTER),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_CLEAR_FILTER),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_ENUM_FILTERS),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_PARAMETERS),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_QUEUE_ALLOCATION_COMPLETE),
  SSH_OID_STR_MAP(OID_RECEIVE_FILTER_CURRENT_CAPABILITIES),
  SSH_OID_STR_MAP(OID_NIC_SWITCH_HARDWARE_CAPABILITIES),
  SSH_OID_STR_MAP(OID_NIC_SWITCH_CURRENT_CAPABILITIES),
  SSH_OID_STR_MAP(OID_VLAN_RESERVED1),
  SSH_OID_STR_MAP(OID_VLAN_RESERVED2),
  SSH_OID_STR_MAP(OID_VLAN_RESERVED3),
  SSH_OID_STR_MAP(OID_VLAN_RESERVED4),
  SSH_OID_STR_MAP(OID_TUNNEL_INTERFACE_SET_OID),
  SSH_OID_STR_MAP(OID_TUNNEL_INTERFACE_RELEASE_OID)
#endif /* NDIS_SUPPORT_NDIS620 */
};


const SshNdisStatusStrMapStruct known_st_values[] =
{ 
#ifdef NDIS_STATUS_WW_ASYNC_RESPONSE
  SSH_STATUS_STR_MAP(NDIS_STATUS_WW_ASYNC_RESPONSE),
#endif /* NDIS_STATUS_WW_ASYNC_RESPONSE */
#ifdef NDIS_STATUS_WW_INDICATION
  SSH_STATUS_STR_MAP(NDIS_STATUS_WW_INDICATION),
#endif /* NDIS_STATUS_WW_INDICATION */
#ifdef NDIS_STATUS_WW_UNKNOWN_VERSION
  SSH_STATUS_STR_MAP(NDIS_STATUS_WW_UNKNOWN_VERSION),
#endif /* NDIS_STATUS_WW_UNKNOWN_VERSION */
#ifdef NDIS_STATUS_WW_DEVICE_BUSY
  SSH_STATUS_STR_MAP(NDIS_STATUS_WW_DEVICE_BUSY),
#endif /* NDIS_STATUS_WW_DEVICE_BUSY */
#ifdef NDIS_STATUS_WW_INVALID_CAPABILITIES
  SSH_STATUS_STR_MAP(NDIS_STATUS_WW_INVALID_CAPABILITIES),
#endif /* NDIS_STATUS_WW_INVALID_CAPABILITIES */
  SSH_STATUS_STR_MAP(NDIS_STATUS_SUCCESS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_PENDING),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NOT_RECOGNIZED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NOT_COPIED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NOT_ACCEPTED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_CALL_ACTIVE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ONLINE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_RESET_START),
  SSH_STATUS_STR_MAP(NDIS_STATUS_RESET_END),
  SSH_STATUS_STR_MAP(NDIS_STATUS_RING_STATUS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_CLOSED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_WAN_LINE_UP),
  SSH_STATUS_STR_MAP(NDIS_STATUS_WAN_LINE_DOWN),
  SSH_STATUS_STR_MAP(NDIS_STATUS_WAN_FRAGMENT),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MEDIA_CONNECT),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MEDIA_DISCONNECT),
  SSH_STATUS_STR_MAP(NDIS_STATUS_HARDWARE_LINE_UP),
  SSH_STATUS_STR_MAP(NDIS_STATUS_HARDWARE_LINE_DOWN),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INTERFACE_UP),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INTERFACE_DOWN),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MEDIA_BUSY),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MEDIA_SPECIFIC_INDICATION),
  SSH_STATUS_STR_MAP(NDIS_STATUS_LINK_SPEED_CHANGE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_WAN_GET_STATS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_WAN_CO_FRAGMENT),
  SSH_STATUS_STR_MAP(NDIS_STATUS_WAN_CO_LINKPARAMS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NOT_RESETTABLE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_SOFT_ERRORS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_HARD_ERRORS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_BUFFER_OVERFLOW),
  SSH_STATUS_STR_MAP(NDIS_STATUS_FAILURE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_RESOURCES),
  SSH_STATUS_STR_MAP(NDIS_STATUS_CLOSING),
  SSH_STATUS_STR_MAP(NDIS_STATUS_BAD_VERSION),
  SSH_STATUS_STR_MAP(NDIS_STATUS_BAD_CHARACTERISTICS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ADAPTER_NOT_FOUND),
  SSH_STATUS_STR_MAP(NDIS_STATUS_OPEN_FAILED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_DEVICE_FAILED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MULTICAST_FULL),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MULTICAST_EXISTS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_MULTICAST_NOT_FOUND),
  SSH_STATUS_STR_MAP(NDIS_STATUS_REQUEST_ABORTED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_RESET_IN_PROGRESS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_CLOSING_INDICATING),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NOT_SUPPORTED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_PACKET),
  SSH_STATUS_STR_MAP(NDIS_STATUS_OPEN_LIST_FULL),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ADAPTER_NOT_READY),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ADAPTER_NOT_OPEN),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NOT_INDICATING),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_LENGTH),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_DATA),
  SSH_STATUS_STR_MAP(NDIS_STATUS_BUFFER_TOO_SHORT),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_OID),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ADAPTER_REMOVED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_UNSUPPORTED_MEDIA),
  SSH_STATUS_STR_MAP(NDIS_STATUS_GROUP_ADDRESS_IN_USE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_FILE_NOT_FOUND),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ERROR_READING_FILE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_ALREADY_MAPPED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_RESOURCE_CONFLICT),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NO_CABLE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_SAP),
  SSH_STATUS_STR_MAP(NDIS_STATUS_SAP_IN_USE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_ADDRESS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_VC_NOT_ACTIVATED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_DEST_OUT_OF_ORDER),
  SSH_STATUS_STR_MAP(NDIS_STATUS_VC_NOT_AVAILABLE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_CELLRATE_NOT_AVAILABLE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INCOMPATABLE_QOS),
  SSH_STATUS_STR_MAP(NDIS_STATUS_AAL_PARAMS_UNSUPPORTED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NO_ROUTE_TO_DESTINATION),
  SSH_STATUS_STR_MAP(NDIS_STATUS_TOKEN_RING_OPEN_ERROR),
  SSH_STATUS_STR_MAP(NDIS_STATUS_INVALID_DEVICE_REQUEST),
  SSH_STATUS_STR_MAP(NDIS_STATUS_NETWORK_UNREACHABLE)
#if NDIS_SUPPORT_NDIS61
  , /* NDIS 6.1 */
  SSH_STATUS_STR_MAP(NDIS_STATUS_HD_SPLIT_CURRENT_CONFIG)
#endif /* NDIS_SUPPORT_NDIS61 */
#if NDIS_SUPPORT_NDIS620
  , /* NDIS 6.20 */
  SSH_STATUS_STR_MAP(NDIS_STATUS_PM_WOL_PATTERN_REJECTED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_PM_OFFLOAD_REJECTED),
  SSH_STATUS_STR_MAP(NDIS_STATUS_PM_CAPABILITIES_CHANGE),
  SSH_STATUS_STR_MAP(NDIS_STATUS_PM_WOL_PATTERN_LIST_FULL),
  SSH_STATUS_STR_MAP(NDIS_STATUS_PM_PROTOCOL_OFFLOAD_LIST_FULL),
  SSH_STATUS_STR_MAP(NDIS_STATUS_OFFLOAD_CONNECTION_REJECTED)
#endif /* NDIS_SUPPORT_NDIS620 */
};


#ifdef _WIN32_WCE
const SshUInt32StrMapStruct known_ww_notify_types[] =
{ 
  SSH_UINT32_STR_MAP(NDIS_WW_READY_STATE),
  SSH_UINT32_STR_MAP(NDIS_WW_PIN_STATE),
  SSH_UINT32_STR_MAP(NDIS_WW_REGISTER_STATE),
  SSH_UINT32_STR_MAP(NDIS_WW_PACKET_SERVICE),
  SSH_UINT32_STR_MAP(NDIS_WW_SIGNAL_STATE),
  SSH_UINT32_STR_MAP(NDIS_WW_ACTIVATION_STATE),
  SSH_UINT32_STR_MAP(NDIS_WW_LOCATION_INFO),
  SSH_UINT32_STR_MAP(NDIS_WW_VENDOR_SPECIFIC)
};
#endif /* _WIN32_WCE */


/* Render function to render NDIS OID REQUEST types for %@ format string for
   ssh_e*printf */
int ssh_ndis_oid_request_type_render(unsigned char *buf, 
                                     int buf_size, 
                                     int precision,
                                     void *datum)
{
  NDIS_REQUEST_TYPE *request_type = (NDIS_REQUEST_TYPE *)datum;
  int len;
  unsigned int i;
  const unsigned char *request_type_str = NULL;

  for (i = 0; 
       i < sizeof(known_oid_req_types) / sizeof(known_oid_req_types[0]); 
       i++)
    {
      if (known_oid_req_types[i].type == *request_type)
        {
          request_type_str = known_oid_req_types[i].str;
          break;
        }
    }

  if (request_type_str != NULL)
    ssh_snprintf(buf, buf_size + 1, request_type_str);
  else
    ssh_snprintf(buf, buf_size + 1, "%08X", *request_type);

  len = ssh_ustrlen(buf);

  if (precision >= 0)
    if (len > precision)
      len = precision;

  if (len >= buf_size)
    return buf_size + 1;

  return len;
}

/* Render function to render NDIS OIDs for %@ format string for
   ssh_e*printf */
int ssh_ndis_oid_render(unsigned char *buf, 
                        int buf_size, 
                        int precision,
                        void *datum)
{
  NDIS_OID *oid = (NDIS_OID *)datum;
  int len;
  unsigned int i;
  const unsigned char *oid_str = NULL;

  for (i = 0; i < sizeof(known_oids) / sizeof(known_oids[0]); i++)
    {
      if (known_oids[i].oid == *oid)
        {
          oid_str = known_oids[i].str;
          break;
        }
    }

  if (oid_str != NULL)
    ssh_snprintf(buf, buf_size + 1, oid_str);
  else
    ssh_snprintf(buf, buf_size + 1, "%08X", *oid);

  len = ssh_ustrlen(buf);

  if (precision >= 0)
    if (len > precision)
      len = precision;

  if (len >= buf_size)
    return buf_size + 1;

  return len;
}

/* Render function to render NDIS_STATUS values for %@ format string for
   ssh_e*printf */
int ssh_ndis_status_render(unsigned char *buf, 
                           int buf_size, 
                           int precision,
                           void *datum)
{
  NDIS_STATUS *status = (NDIS_STATUS *)datum;
  int len;
  unsigned int i;
  const unsigned char *status_str = NULL;

  for (i = 0; i < sizeof(known_st_values) / sizeof(known_st_values[0]); i++)
    {
      if (known_st_values[i].status == *status)
        {
          status_str = known_st_values[i].str;
          break;
        }
    }

  if (status_str != NULL)
    ssh_snprintf(buf, buf_size + 1, status_str);
  else
    ssh_snprintf(buf, buf_size + 1, "%08X", *status);

  len = ssh_ustrlen(buf);

  if (precision >= 0)
    if (len > precision)
      len = precision;

  if (len >= buf_size)
    return buf_size + 1;

  return len;
}

#ifdef _WIN32_WCE
int ssh_ndis_ww_notification_type_render(unsigned char *buf,
                                         int buf_size,
                                         int precision,
                                         void *datum)
{
  SshUInt32 *value = (SshUInt32 *)datum;
  int len;
  unsigned int i;
  const unsigned char *str = NULL;

  for (i = 0; 
       i < sizeof(known_ww_notify_types) / sizeof(known_ww_notify_types[0]);
       i++)
    {
      if (known_ww_notify_types[i].value == *value)
        {
          str = known_ww_notify_types[i].str;
          break;
        }
    }

  if (str != NULL)
    ssh_snprintf(buf, buf_size + 1, str);
  else
    ssh_snprintf(buf, buf_size + 1, "0x%08X", *value);

  len = ssh_ustrlen(buf);

  if (precision >= 0)
    if (len > precision)
      len = precision;

  if (len >= buf_size)
    return buf_size + 1;

  return len;
} 
#endif /* _WIN32_WCE */

#endif /* DEBUG_LIGHT */
