/*
 * MTD SPI driver for ST M25Pxx flash chips
 *
 * Author: Mike Lavender, mike@steroidmicros.com
 *
 * Copyright (c) 2005, Intec Automation Inc.
 *
 * Some parts are based on lart.c by Abraham Van Der Merwe
 *
 * Cleaned up and generalized based on mtd_dataflash.c
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/mtd/map.h>
#include <linux/mtd/gen_probe.h>
#include <linux/semaphore.h>
#include <linux/slab.h>
#include <linux/spi/spi.h>
#include <linux/spi/flash.h>
#include <asm/mach-ralink/rt_mmap.h>
#include "bbu_spi_flash.h"

//#if defined (CONFIG_RALINK_RT6855A) || defined (CONFIG_RALINK_MT7621) || defined (CONFIG_RALINK_MT7628)
//#include "bbu_spiflash.h"
//#else
//#include "ralink_spi.h"
//#endif

static struct semaphore spi_sem;
struct mtd_info *minfo = NULL;
static struct spi_flash_params *spi_dev;
static const char *part_probes[] = {"cmdlinepart", "RedBoot", NULL};


//#define TEST_CS1_FLASH
//#define RD_MODE_DIOR

#if defined(RD_MODE_DIOR) || defined(RD_MODE_DOR)
#define RD_MODE_DUAL
#elif defined(RD_MODE_QIOR) || defined(RD_MODE_QOR)
#define RD_MODE_QUAD
#endif

/* Choose the SPI flash mode */
#if defined (CONFIG_RALINK_RT6855A) || defined (CONFIG_RALINK_MT7621) || defined (CONFIG_RALINK_MT7628)
#define BBU_MODE		// BBU SPI flash controller
#define MORE_BUF_MODE

// check DUAL/QUAD and MORE_BUF_MODE, they can't be enabled together
#if defined(MORE_BUF_MODE) && (defined(RD_MODE_DIOR) || defined(RD_MODE_DOR) || defined(RD_MODE_QIOR) || defined(RD_MODE_QOR))
#error "DUAL/QUAD mode and MORE_BUF_MODE can't be enabled together\n"
#endif
#else
#define USER_MODE		// SPI flash user mode support by default
//#define COMMAND_MODE		// SPI flash command mode support
#endif

#if !defined USER_MODE && !defined COMMAND_MODE && !defined BBU_MODE
#error "Please choose the correct mode of SPI flash controller"
#endif


/******************************************************************************
 * SPI FLASH elementray definition and function
 ******************************************************************************/

//#define FLASH_PAGESIZE		256

/* Flash opcodes. */
#define OPCODE_WREN		6	/* Write enable */
#define OPCODE_WRDI		4	/* Write disable */
#define OPCODE_RDSR		5	/* Read status register */
#define OPCODE_WRSR		1	/* Write status register */
#define OPCODE_READ		3	/* Read data bytes */
#define OPCODE_PP		2	/* Page program */
#define OPCODE_SE		0xD8	/* Sector erase */
#define OPCODE_RES		0xAB	/* Read Electronic Signature */
#define OPCODE_RDID		0x9F	/* Read JEDEC ID */
#define OPCODE_DOR			0x3B	/* Dual Output Read */
#define OPCODE_QOR			0x6B	/* Quad Output Read */
#define OPCODE_DIOR                     0xBB    /* Dual IO High Performance Read */
#define OPCODE_QIOR                     0xEB    /* Quad IO High Performance Read */

/* Status Register bits. */
#define SR_WIP			1	/* Write in progress */
#define SR_WEL			2	/* Write enable latch */
#define SR_BP0			4	/* Block protect 0 */
#define SR_BP1			8	/* Block protect 1 */
#define SR_BP2			0x10	/* Block protect 2 */
#define SR_EPE			0x20	/* Erase/Program error */
#define SR_SRWD			0x80	/* SR write protect */

#define OPCODE_BRRD		0x16
#define OPCODE_BRWR		0x17
#define OPCODE_RDCR		0x35

//#define SPI_DEBUG
#if !defined (SPI_DEBUG)

#define ra_inl(addr)  (*(volatile unsigned int *)(addr))
#define ra_outl(addr, value)  (*(volatile unsigned int *)(addr) = (value))
#define ra_dbg(args...) do {} while(0)
/*#define ra_dbg(args...) do { printk(args); } while(0)*/

#else

#define ra_dbg(args...) do { printk(args); } while(0)
#define _ra_inl(addr)  (*(volatile unsigned int *)(addr))
#define _ra_outl(addr, value)  (*(volatile unsigned int *)(addr) = (value))

u32 ra_inl(u32 addr)
{	
	u32 retval = _ra_inl(addr);
	printk("%s(%x) => %x \n", __func__, addr, retval);

	return retval;	
}

u32 ra_outl(u32 addr, u32 val)
{
	_ra_outl(addr, val);

	printk("%s(%x, %x) \n", __func__, addr, val);

	return val;	
}

#endif // SPI_DEBUG //

#define ra_aor(addr, a_mask, o_value)  ra_outl(addr, (ra_inl(addr) & (a_mask)) | (o_value))
#define ra_and(addr, a_mask)  ra_aor(addr, a_mask, 0)
#define ra_or(addr, o_value)  ra_aor(addr, -1, o_value)

#define SPIC_READ_BYTES (1<<0)
#define SPIC_WRITE_BYTES (1<<1)
#define SPIC_DEBUG (1 << 7)

#if 1
void usleep(unsigned int usecs)
{
        unsigned long timeout = usecs_to_jiffies(usecs);

        while (timeout)
                timeout = schedule_timeout_interruptible(timeout);
}
#endif

#if defined USER_MODE || defined COMMAND_MODE
static unsigned int spi_wait_nsec = 0;
static int spic_busy_wait(void)
{
	do {
		if ((ra_inl(RT2880_SPISTAT_REG) & 0x01) == 0)
			return 0;
	} while (spi_wait_nsec >> 1);

	printk("%s: fail \n", __func__);
	return -1;
}
#elif defined BBU_MODE
static int bbu_spic_busy_wait(void)
{
	int n = 100000;
	do {
		if ((ra_inl(SPI_REG_CTL) & SPI_CTL_BUSY) == 0)
			return 0;
		udelay(1);
	} while (--n > 0);

	printk("%s: fail \n", __func__);
	return -1;
}
#endif // USER_MODE || COMMAND_MODE //

#ifdef BBU_MODE
#ifdef MORE_BUF_MODE
static int bbu_mb_spic_trans(const u8 code, const u32 addr, u8 *buf, const size_t n_tx, const size_t n_rx, int flag)
{
	u32 reg;
	int i, q, r;
	int rc = -1;

	if (flag != SPIC_READ_BYTES && flag != SPIC_WRITE_BYTES) {
		printk("we currently support more-byte-mode for reading and writing data only\n");
		return -1;
	}

	/* step 0. enable more byte mode */
	ra_or(SPI_REG_MASTER, (1 << 2));

	bbu_spic_busy_wait();

	/* step 1. set opcode & address, and fix cmd bit count to 32 (or 40) */
	if (spi_dev && spi_dev->addr4b) {
		ra_and(SPI_REG_CTL, ~SPI_CTL_ADDREXT_MASK);
		ra_or(SPI_REG_CTL, (code << 24) & SPI_CTL_ADDREXT_MASK);
		ra_outl(SPI_REG_OPCODE, addr);
	}
	else
	{
		ra_outl(SPI_REG_OPCODE, (code << 24) & 0xff000000);
		ra_or(SPI_REG_OPCODE, (addr & 0xffffff));
	}
	ra_and(SPI_REG_MOREBUF, ~SPI_MBCTL_CMD_MASK);
	if (spi_dev && spi_dev->addr4b)
		ra_or(SPI_REG_MOREBUF, (40 << 24));
	else
		ra_or(SPI_REG_MOREBUF, (32 << 24));

	/* step 2. write DI/DO data #0 ~ #7 */
	if (flag & SPIC_WRITE_BYTES) {
		if (buf == NULL) {
			printk("%s: write null buf\n", __func__);
			goto RET_MB_TRANS;
		}
		for (i = 0; i < n_tx; i++) {
			q = i / 4;
			r = i % 4;
			if (r == 0)
				ra_outl(SPI_REG_DATA(q), 0);
			ra_or(SPI_REG_DATA(q), (*(buf + i) << (r * 8)));
		}
	}

	/* step 3. set rx (miso_bit_cnt) and tx (mosi_bit_cnt) bit count */
	ra_and(SPI_REG_MOREBUF, ~SPI_MBCTL_TX_RX_CNT_MASK);
	ra_or(SPI_REG_MOREBUF, (n_rx << 3 << 12));
	ra_or(SPI_REG_MOREBUF, n_tx << 3);

	/* step 4. kick */
	ra_or(SPI_REG_CTL, SPI_CTL_START);

	/* step 5. wait spi_master_busy */
	bbu_spic_busy_wait();
	if (flag & SPIC_WRITE_BYTES) {
		rc = 0;
		goto RET_MB_TRANS;
	}

	/* step 6. read DI/DO data #0 */
	if (flag & SPIC_READ_BYTES) {
		if (buf == NULL) {
			printk("%s: read null buf\n", __func__);
			return -1;
		}
		for (i = 0; i < n_rx; i++) {
			q = i / 4;
			r = i % 4;
			reg = ra_inl(SPI_REG_DATA(q));
			*(buf + i) = (u8)(reg >> (r * 8));
		}
	}

	rc = 0;
RET_MB_TRANS:
	/* step #. disable more byte mode */
	ra_and(SPI_REG_MASTER, ~(1 << 2));
	return rc;
}
#endif // MORE_BUF_MODE //

static int bbu_spic_trans(const u8 code, const u32 addr, u8 *buf, const size_t n_tx, const size_t n_rx, int flag)
{
	u32 reg;

	bbu_spic_busy_wait();

	/* step 1. set opcode & address */
	if (spi_dev && spi_dev->addr4b) {
		ra_and(SPI_REG_CTL, ~SPI_CTL_ADDREXT_MASK);
		ra_or(SPI_REG_CTL, addr & SPI_CTL_ADDREXT_MASK);
	}
	ra_outl(SPI_REG_OPCODE, ((addr & 0xffffff) << 8));
	ra_or(SPI_REG_OPCODE, code);

#if defined(RD_MODE_QUAD) || defined(RD_MODE_DUAL)
	if (flag & SPIC_READ_BYTES)
		ra_outl(SPI_REG_DATA0, 0); // clear data bit for dummy bits in Dual/Quad IO Read
#endif
	/* step 2. write DI/DO data #0 */
	if (flag & SPIC_WRITE_BYTES) {
		if (buf == NULL) {
			printk("%s: write null buf\n", __func__);
			return -1;
		}
		ra_outl(SPI_REG_DATA0, 0);
		switch (n_tx) {
		case 8:
			ra_or(SPI_REG_DATA0, (*(buf+3) << 24));
		case 7:
			ra_or(SPI_REG_DATA0, (*(buf+2) << 16));
		case 6:
			ra_or(SPI_REG_DATA0, (*(buf+1) << 8));
		case 5:
			ra_or(SPI_REG_DATA0, *buf);
			break;
		case 3:
			reg = ra_inl(SPI_REG_CTL);
			if (((reg & (0x3<<19)) == (0x3 << 19)) && (spi_dev && spi_dev->addr4b)) 
			{
				ra_and(SPI_REG_CTL, ~SPI_CTL_ADDREXT_MASK);
				ra_or(SPI_REG_CTL, (*buf << 24) & SPI_CTL_ADDREXT_MASK);
				ra_and(SPI_REG_OPCODE, 0xff);
				ra_or(SPI_REG_OPCODE, (*(buf+1) & 0xff) << 24);
			}
			else
			{
				ra_and(SPI_REG_OPCODE, 0xff);
				ra_or(SPI_REG_OPCODE, (*buf & 0xff) << 24);
				ra_or(SPI_REG_OPCODE, (*(buf+1) & 0xff) << 16);
			}
			break;
		case 2:
			reg = ra_inl(SPI_REG_CTL);
			if (((reg & (0x3<<19)) == (0x3 << 19)) && (spi_dev && spi_dev->addr4b)) 
			{
				ra_and(SPI_REG_CTL, ~SPI_CTL_ADDREXT_MASK);
				ra_or(SPI_REG_CTL, (*buf << 24) & SPI_CTL_ADDREXT_MASK);
			}
			else
			{
				ra_and(SPI_REG_OPCODE, 0xff);
				ra_or(SPI_REG_OPCODE, (*buf & 0xff) << 24);
			}
			break;
		default:
			printk("%s: fixme, write of length %d\n", __func__, n_tx);
			return -1;
		}
	}


	/* step 3. set mosi_byte_cnt */
	ra_and(SPI_REG_CTL, ~SPI_CTL_TX_RX_CNT_MASK);
	ra_or(SPI_REG_CTL, (n_rx << 4));
	if (spi_dev && spi_dev->addr4b && n_tx >= 4)
		ra_or(SPI_REG_CTL, (n_tx + 1));
	else
		ra_or(SPI_REG_CTL, n_tx);


	/* step 4. kick */
	ra_or(SPI_REG_CTL, SPI_CTL_START);

	/* step 5. wait spi_master_busy */
	bbu_spic_busy_wait();
	if (flag & SPIC_WRITE_BYTES)
		return 0;

	/* step 6. read DI/DO data #0 */
	if (flag & SPIC_READ_BYTES) {
		if (buf == NULL) {
			printk("%s: read null buf\n", __func__);
			return -1;
		}
		reg = ra_inl(SPI_REG_DATA0);
		switch (n_rx) {
		case 4:
			*(buf+3) = (u8)(reg >> 24);
		case 3:
			*(buf+2) = (u8)(reg >> 16);
		case 2:
			*(buf+1) = (u8)(reg >> 8);
		case 1:
			*buf = (u8)reg;
			break;
		default:
			printk("%s: fixme, read of length %d\n", __func__, n_rx);
			return -1;
		}
	}


	return 0;
}
#endif // BBU_MODE //

/*
 * Read the status register, returning its value in the location
 */
static int raspi_read_rg(u8 code, u8 *val)
{
	ssize_t retval;

#ifdef USER_MODE
	retval = spic_read(&code, 1, val, 1);
#elif defined BBU_MODE
	retval = bbu_spic_trans(code, 0, val, 1, 1, SPIC_READ_BYTES);
	return retval;
#endif
	if (retval != 1) {
		printk("%s: ret: %x\n", __func__, retval);
		return -EIO;
	}


	return 0;
}

/*
 * write status register
 */
static int raspi_write_rg(u8 code, u8 *val)
{
	ssize_t retval;

#ifdef USER_MODE
	retval = spic_write(&code, 1, val, 1);
#elif defined BBU_MODE
	{
		// put the value to be written in address register, so it will be transfered
		u32 address = (*val) << 24;
		retval = bbu_spic_trans(code, address, val, 2, 0, SPIC_WRITE_BYTES);
	}
	return retval;
#endif
	if (retval != 1) {
		printk("%s: ret: %x\n", __func__, retval);
		return -EIO;
	}

	return 0;
}

#if defined(RD_MODE_DUAL) || defined(RD_MODE_QUAD)

static int raspi_write_rg16(u8 code, u8 *val)
{
	ssize_t retval;

#ifdef USER_MODE
	retval = spic_write(&code, 1, val, 2);
#elif defined BBU_MODE
	{
		// put the value to be written in address register, so it will be transfered
		u32 address = (*val) << 24;
		address |= (*(val+1)) << 16;
		retval = bbu_spic_trans(code, address, val, 3, 0, SPIC_WRITE_BYTES);
	}
	return retval;
#endif
	if (retval != 1) {
		printk("%s: ret: %x\n", __func__, retval);
		return -EIO;
	}

	return 0;
}
#endif

static int raspi_read_sr(u8 *val)
{
	return raspi_read_rg(OPCODE_RDSR, val);
}

static int raspi_write_sr(u8 *val)
{
	return raspi_write_rg(OPCODE_WRSR, val);
}

/*
 * read SPI flash device ID
 */
static int raspi_read_devid(u8 *rxbuf, int n_rx)
{
	u8 code = OPCODE_RDID;
	int retval;

#ifdef USER_MODE
	retval = spic_read(&code, 1, rxbuf, n_rx);
#elif defined BBU_MODE
	retval = bbu_spic_trans(code, 0, rxbuf, 1, 3, SPIC_READ_BYTES);
	if (!retval)
		retval = n_rx;
#endif
	if (retval != n_rx) {
		printk("%s: ret: %x\n", __func__, retval);
		return retval;
	}
	return retval;
}

/*
 * Set write enable latch with Write Enable command.
 * Returns negative if error occurred.
 */
static inline int raspi_write_enable(void)
{
	u8 code = OPCODE_WREN;

#ifdef USER_MODE
	return spic_write(&code, 1, NULL, 0);
#elif defined BBU_MODE
	return bbu_spic_trans(code, 0, NULL, 1, 0, 0);
#endif
}

/*
 * Set all sectors (global) unprotected if they are protected.
 * Returns negative if error occurred.
 */
static inline int raspi_unprotect(void)
{
	u8 sr = 0;

	if (raspi_read_sr(&sr) < 0) {
		printk("%s: read_sr fail: %x\n", __func__, sr);
		return -1;
	}

	if ((sr & (SR_BP0 | SR_BP1 | SR_BP2)) != 0) {
		sr = 0;
		raspi_write_sr(&sr);
	}
	return 0;
}

static int raspi_wait_sleep_ready(int sleep_ms)
{
	int count;
	int sr = 0;

	/*int timeout = sleep_ms * HZ / 1000;
	while (timeout) 
		timeout = schedule_timeout (timeout);*/

	/* one chip guarantees max 5 msec wait here after page writes,
	 * but potentially three seconds (!) after page erase.
	 */
	for (count = 0; count < ((sleep_ms+1) *1000); count++) {
		if ((raspi_read_sr((u8 *)&sr)) < 0)
			break;
		else if (!(sr & SR_WIP)) {
			return 0;
		}

		usleep(1);
		/* REVISIT sometimes sleeping would be best */
	}

	printk("%s: read_sr fail: %x\n", __func__, sr);
	return -EIO;
}


/*
 * Service routine to read status register until ready, or timeout occurs.
 * Returns non-zero if error.
 */
static int raspi_wait_ready(int sleep_ms)
{
	int count;
	int sr = 0;

	/*int timeout = sleep_ms * HZ / 1000;
	while (timeout) 
		timeout = schedule_timeout (timeout);*/

	/* one chip guarantees max 5 msec wait here after page writes,
	 * but potentially three seconds (!) after page erase.
	 */
	for (count = 0; count < ((sleep_ms+1) *1000 * 500); count++) {
		if ((raspi_read_sr((u8 *)&sr)) < 0)
			break;
		else if (!(sr & SR_WIP)) {
			return 0;
		}

		udelay(1);
		/* REVISIT sometimes sleeping would be best */
	}

	printk("%s: read_sr fail: %x\n", __func__, sr);
	return -EIO;
}

static int raspi_4byte_mode(int enable)
{
	ssize_t retval;
	
	raspi_wait_ready(1);
	
	
	if (((spi_dev->id >> 16) & 0xff) == 0x1) // Spansion
	{
		u8 br, br_cfn; // bank register
#ifdef USER_MODE
		if (enable)
		{
			br = 0x81;
			ra_or(RT2880_SPICFG_REG, SPICFG_ADDRMODE);
		}
		else
		{
			br = 0x0;
			ra_and(RT2880_SPICFG_REG, ~(SPICFG_ADDRMODE));
		}
    	
#elif defined BBU_MODE
		if (enable)
		{
			ra_or(SPI_REG_CTL, 0x3 << 19);
			ra_or(SPI_REG_Q_CTL, 0x3 << 8);
			br = 0x81;
		}
		else
		{
			ra_and(SPI_REG_CTL, ~SPI_CTL_SIZE_MASK);
			ra_or(SPI_REG_CTL, 0x2 << 19);
			ra_and(SPI_REG_Q_CTL, ~(0x3 << 8));
			ra_or(SPI_REG_Q_CTL, 0x2 << 8);
			br = 0x0;
		}
#endif
		raspi_write_rg(OPCODE_BRWR, &br);
		raspi_wait_ready(1);
		raspi_read_rg(OPCODE_BRRD, &br_cfn);
		if (br_cfn != br)
		{
			printk("4B mode switch failed %d, %x, %x\n", enable, br_cfn, br);
			return -1;
		}

	}
	else
	{
		u8 code;
	
		code = enable? 0xB7 : 0xE9; /* B7: enter 4B, E9: exit 4B */

#ifdef USER_MODE
		if (enable)
			ra_or(RT2880_SPICFG_REG, SPICFG_ADDRMODE);
		else
			ra_and(RT2880_SPICFG_REG, ~(SPICFG_ADDRMODE));
		retval = spic_read(&code, 1, 0, 0);
#elif defined BBU_MODE
		if (enable) {
			ra_or(SPI_REG_CTL, 0x3 << 19);
			ra_or(SPI_REG_Q_CTL, 0x3 << 8);

		}
		else {
			ra_and(SPI_REG_CTL, ~SPI_CTL_SIZE_MASK);
			ra_or(SPI_REG_CTL, 0x2 << 19);
			ra_and(SPI_REG_Q_CTL, ~(0x3 << 8));
			ra_or(SPI_REG_Q_CTL, 0x2 << 8);
		}
		retval = bbu_spic_trans(code, 0, NULL, 1, 0, 0);
#endif
		if (retval != 0) {
			printk("%s: ret: %x\n", __func__, retval);
			return -1;
		}

	}

	return 0;
}

/*
 * Erase one sector of flash memory at offset ``offset'' which is any
 * address within the sector which should be erased.
 *
 * Returns 0 if successful, non-zero otherwise.
 */
static int raspi_erase_sector(u32 offset)
{
#ifdef USER_MODE
	u8 command[5];
#endif

	/* Wait until finished previous write command. */
	if (raspi_wait_ready(3))
		return -EIO;

	/* Send write enable, then erase commands. */
	raspi_write_enable();
	raspi_unprotect();

#ifdef USER_MODE
	if (spi_dev->addr4b) {
		raspi_4byte_mode(1);
		command[0] = OPCODE_SE;
		command[1] = offset >> 24;
		command[2] = offset >> 16;
		command[3] = offset >> 8;
		command[4] = offset;
		spic_write(command, 5, 0 , 0);
		raspi_wait_sleep_ready(950);
		raspi_4byte_mode(0);
		return 0;
	}

	/* Set up command buffer. */
	command[0] = OPCODE_SE;
	command[1] = offset >> 16;
	command[2] = offset >> 8;
	command[3] = offset;

	spic_write(command, 4, 0 , 0);
	raspi_wait_sleep_ready(950);
#elif defined BBU_MODE
	if (spi_dev->addr4b)
		raspi_4byte_mode(1);
	raspi_write_enable();
	bbu_spic_trans(STM_OP_SECTOR_ERASE, offset, NULL, 4, 0, 0);
	//raspi_wait_ready(950);
	raspi_wait_sleep_ready(950);
	if (spi_dev->addr4b)
		raspi_4byte_mode(0);
#endif
	return 0;
}


/****************************************************************************/
/*
 * Erase an address range on the flash chip.  The address range may extend
 * one or more erase sectors.  Return an error is there is a problem erasing.
 */
static int mtd_spi_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	u32 addr,len;

	//printk("%s: addr:%x len:%x\n", __func__, instr->addr, instr->len);
	/* sanity checks */
	if (instr->addr + instr->len > mtd->size)
		return -EINVAL;

#if 0  //FIXME - undefined reference to `__umoddi3'
	if ((instr->addr % mtd->erasesize) != 0
			|| (instr->len % mtd->erasesize) != 0) {
		return -EINVAL;
	}
#endif

	addr = instr->addr;
	len = instr->len;

  	down(&spi_sem);

	/* now erase those sectors */
	while (len > 0) {
		if (raspi_erase_sector(addr)) {
			instr->state = MTD_ERASE_FAILED;
			up(&spi_sem);
			return -EIO;
		}

		addr += mtd->erasesize;
		len -= mtd->erasesize;
	}

  	up(&spi_sem);

	instr->state = MTD_ERASE_DONE;
	mtd_erase_callback(instr);

	return 0;

}


/*
 * Read an address range from the flash chip.  The address range
 * may be any size provided it is within the physical boundaries.
 */
static int mtd_spi_read(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char *buf)
{
#ifdef USER_MODE
	u8 command[5];
#endif
	size_t rdlen = 0;

	//ra_dbg("%s: from:%x len:%x \n", __func__, from, len);

	/* sanity checks */
	if (len == 0)
		return 0;

	if (from + len > mtd->size)
		return -EINVAL;

	/* Byte count starts at zero. */
	if (retlen)
		*retlen = 0;

	down(&spi_sem);

	/* Wait till previous write/erase is done. */
	if (raspi_wait_ready(1)) {
		/* REVISIT status return?? */
		up(&spi_sem);
		return -EIO;
	}

#ifdef USER_MODE

	/* Set up the write data buffer. */
	command[0] = OPCODE_READ;
	if (spi_dev->addr4b) {
		raspi_4byte_mode(1);
		command[1] = from >> 24;
		command[2] = from >> 16;
		command[3] = from >> 8;
		command[4] = from;
		rdlen = spic_read(command, 5, buf, len);
		raspi_4byte_mode(0);
	}
	else
	{
		command[1] = from >> 16;
		command[2] = from >> 8;
		command[3] = from;
		rdlen = spic_read(command, 4, buf, len);
	}
#elif defined BBU_MODE
	if (spi_dev->addr4b)
		raspi_4byte_mode(1);
#ifndef MORE_BUF_MODE
#if defined(RD_MODE_DUAL)
	// serial mode = dual
	if (((spi_dev->id >> 16) & 0xff) == 0x1) // Spansion
	{
		u8 reg[2], cr;
		raspi_read_rg(OPCODE_RDCR, &reg[1]);
		if (reg[1] & (1 << 1))
		{
			reg[1] &= ~(1 << 1);
			raspi_read_sr(&reg[0]);
                        raspi_write_enable();
                        raspi_write_rg16(OPCODE_WRSR, &reg[0]);
			raspi_wait_ready(1);
			raspi_read_rg(OPCODE_RDCR, &cr);
			if (reg[1] != cr)
				printk("warning: set quad failed %x %x\n", reg[1], cr);
		}
	}
	else // MXIC
	{
                u8 sr;
		raspi_read_sr(&sr);
                if ((sr & (1 << 6)))
                {
			u8 get_sr;
                        sr &= ~(1 << 6);
                        raspi_write_enable();
                        raspi_write_sr(&sr);
			raspi_wait_ready(1);
			raspi_read_sr(&get_sr);
			if (get_sr != sr)
				printk("warning: sr write failed %x %x\n", sr, get_sr);
                }
 	}
	ra_and(SPI_REG_MASTER, ~3);
	ra_or(SPI_REG_MASTER, 1);
#elif defined(RD_MODE_QUAD)
	// serial mode = quad
	if (((spi_dev->id >> 16) & 0xff) == 0x1) // Spansion
	{
		u8 reg[2], cr;
		raspi_read_rg(OPCODE_RDCR, &reg[1]);
		if ((reg[1] & (1 << 1)) == 0)
		{
			reg[1] |= (1 << 1);
			raspi_read_sr(&reg[0]);
                        raspi_write_enable();
                        raspi_write_rg16(OPCODE_WRSR, &reg[0]);
			raspi_wait_ready(1); 
			raspi_read_rg(OPCODE_RDCR, &cr);
			if (reg[1] != cr)
				printk("warning: set quad failed %x %x\n", reg[1], cr);
		}
	}
	else // MXIC
	{
                u8 sr, sr2;
		raspi_read_sr(&sr);
		sr2 = sr;
                if ((sr & (1 << 6)) == 0)
                {
			u8 get_sr;
                        sr |= (1 << 6);
                        raspi_write_enable();
                        raspi_write_sr(&sr);
			raspi_wait_ready(1);
			raspi_read_sr(&get_sr);
			if (get_sr != sr)
				printk("warning: quad sr write failed %x %x %x\n", sr, get_sr, sr2);
			
                }
 	}
	ra_and(SPI_REG_MASTER, ~3);
	ra_or(SPI_REG_MASTER, 2);

#endif
#endif
	do {
		int rc, more;
#ifdef MORE_BUF_MODE
		more = 32;
#else
		more = 4;
#endif
		if (len - rdlen <= more) {
#ifdef MORE_BUF_MODE
			rc = bbu_mb_spic_trans(STM_OP_RD_DATA, from, (buf+rdlen), 0, (len-rdlen), SPIC_READ_BYTES);
#else
#if defined(RD_MODE_DOR)
			rc = bbu_spic_trans(OPCODE_DOR, from, (buf+rdlen), 5, (len-rdlen), SPIC_READ_BYTES);
#elif defined(RD_MODE_DIOR)
			rc = bbu_spic_trans(OPCODE_DIOR, from, (buf+rdlen), 5, (len-rdlen), SPIC_READ_BYTES);
#elif defined(RD_MODE_QOR)
			rc = bbu_spic_trans(OPCODE_QOR, from, (buf+rdlen), 5, (len-rdlen), SPIC_READ_BYTES);
#elif defined(RD_MODE_QIOR)
			rc = bbu_spic_trans(OPCODE_QIOR, from, (buf+rdlen), 7, (len-rdlen), SPIC_READ_BYTES);
#else
			rc = bbu_spic_trans(STM_OP_RD_DATA, from, (buf+rdlen), 4, (len-rdlen), SPIC_READ_BYTES);
#endif
#endif
			if (rc != 0) {
				printk("%s: failed\n", __func__);
				break;
			}
			rdlen = len;
		}
		else {
#ifdef MORE_BUF_MODE
			rc = bbu_mb_spic_trans(STM_OP_RD_DATA, from, (buf+rdlen), 0, more, SPIC_READ_BYTES);
#else
#if defined(RD_MODE_DOR)
			rc = bbu_spic_trans(OPCODE_DOR, from, (buf+rdlen), 5, more, SPIC_READ_BYTES);
#elif defined(RD_MODE_DIOR)
			rc = bbu_spic_trans(OPCODE_DIOR, from, (buf+rdlen), 5, more, SPIC_READ_BYTES);
#elif defined(RD_MODE_QOR)
			rc = bbu_spic_trans(OPCODE_QOR, from, (buf+rdlen), 5, more, SPIC_READ_BYTES);
#elif defined(RD_MODE_QIOR)
			rc = bbu_spic_trans(OPCODE_QIOR, from, (buf+rdlen), 7, more, SPIC_READ_BYTES);
#else
			rc = bbu_spic_trans(STM_OP_RD_DATA, from, (buf+rdlen), 4, more, SPIC_READ_BYTES);
#endif
#endif
			if (rc != 0) {
				printk("%s: failed\n", __func__);
				break;
			}
			rdlen += more;
			from += more;
		}
	} while (rdlen < len);

#ifndef MORE_BUF_MODE
#if defined(RD_MODE_DUAL) || defined(RD_MODE_QUAD)
	// serial mode = normal
	ra_and(SPI_REG_MASTER, ~3);
#if defined(RD_MODE_QUAD)
	// serial mode = quad
	if (((spi_dev->id >> 16) & 0xff) == 0x1) // Spansion
	{
		u8 reg[2], cr;
		raspi_read_rg(OPCODE_RDCR, &reg[1]);
		if (reg[1] & (1 << 1))
		{
			reg[1] &= ~(1 << 1);
			raspi_read_sr(&reg[0]);
                        raspi_write_enable();
                        raspi_write_rg16(OPCODE_WRSR, &reg[0]);
			raspi_wait_ready(1);
			raspi_read_rg(OPCODE_RDCR, &cr);
			if (reg[1] != cr)
				printk("warning: set quad failed %x %x\n", reg[1], cr);
		}
	}
	else // MXIC
	{
                u8 sr;
		raspi_read_sr(&sr);
                if ((sr & (1 << 6)))
                {
                        sr &= ~(1 << 6);
                        raspi_write_enable();
                        raspi_write_sr(&sr);
			raspi_wait_ready(1);
                }
 	}

#endif
#endif
#endif // MORE_BUF_MODE

	if (spi_dev->addr4b)
		raspi_4byte_mode(0);
#endif

  	up(&spi_sem);

	if (retlen) 
		*retlen = rdlen;

	if (rdlen != len) 
		return -EIO;

	return 0;
}

/*
 * Write an address range to the flash chip.  Data must be written in
 * FLASH_PAGESIZE chunks.  The address range may be any size provided
 * it is within the physical boundaries.
 */
static int mtd_spi_write(struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen, const u_char *buf)
{

#ifdef USER_MODE
	u8 command[5];
#endif

	u32 page_offset, page_size;
	int rc = 0;
#if defined BBU_MODE
	int wrto, wrlen, more;
	char *wrbuf;
#endif
	int count = 0;

	//ra_dbg("%s: to:%x len:%x \n", __func__, to, len);
	if (retlen)
		*retlen = 0;

	/* sanity checks */
	if (len == 0)
		return 0;
	if (to + len > mtd->size)
		return -EINVAL;

  	down(&spi_sem);

	/* Wait until finished previous write command. */
	if (raspi_wait_ready(2)) {
		up(&spi_sem);
		return -1;
	}

#ifdef USER_MODE
	/* Set up the opcode in the write buffer. */
	command[0] = OPCODE_PP;
	if (spi_dev->addr4b) {
		command[1] = to >> 24;
		command[2] = to >> 16;
		command[3] = to >> 8;
		command[4] = to;
	}
	else
	{
		command[1] = to >> 16;
		command[2] = to >> 8;
		command[3] = to;
	}
#endif

	/* what page do we start with? */
	//page_offset = to % FLASH_PAGESIZE;
	page_offset = (u32)to % spi_dev->page_size;

	if (spi_dev->addr4b)
		raspi_4byte_mode(1);

	/* write everything in PAGESIZE chunks */
	while (len > 0) {
		//page_size = min_t(size_t, len, FLASH_PAGESIZE-page_offset);
		page_size = min_t(size_t, len, spi_dev->page_size-page_offset);
		page_offset = 0;

		/* write the next page to flash */
#ifdef USER_MODE
		if (spi_dev->addr4b) {
			command[1] = to >> 24;
			command[2] = to >> 16;
			command[3] = to >> 8;
			command[4] = to;
		}
		else
		{
			command[1] = to >> 16;
			command[2] = to >> 8;
			command[3] = to;
		}
#endif

		raspi_wait_ready(3);
		raspi_write_enable();
		raspi_unprotect();

#ifdef USER_MODE

		if (spi_dev->addr4b)
			rc = spic_write(command, 5, buf, page_size);
		else
			rc = spic_write(command, 4, buf, page_size);

#elif defined BBU_MODE
		wrto = to;
		wrlen = page_size;
		wrbuf = (char *)buf;
		rc = wrlen;
		do {
#ifdef MORE_BUF_MODE
			more = 32;
#else
			more = 4;
#endif
			if (wrlen <= more) {
#ifdef MORE_BUF_MODE
				bbu_mb_spic_trans(STM_OP_PAGE_PGRM, wrto, wrbuf, wrlen, 0, SPIC_WRITE_BYTES);
#else
				bbu_spic_trans(STM_OP_PAGE_PGRM, wrto, wrbuf, wrlen+4, 0, SPIC_WRITE_BYTES);
#endif
				wrlen = 0;
			}
			else {
#ifdef MORE_BUF_MODE
				bbu_mb_spic_trans(STM_OP_PAGE_PGRM, wrto, wrbuf, more, 0, SPIC_WRITE_BYTES);
#else
				bbu_spic_trans(STM_OP_PAGE_PGRM, wrto, wrbuf, more+4, 0, SPIC_WRITE_BYTES);
#endif
				wrto += more;
				wrlen -= more;
				wrbuf += more;
			}
			if (wrlen > 0) {
				raspi_wait_ready(100);
				raspi_write_enable();
			}
		} while (wrlen > 0);
#endif
		//printk("%s : to:%llx page_size:%x ret:%x\n", __func__, to, page_size, rc);

		if (rc > 0) {
			if (retlen)
				*retlen += rc;
			if (rc < page_size) {
				up(&spi_sem);
				printk("%s: rc:%x return:%x page_size:%x \n", 
				       __func__, rc, rc, page_size);
				return -EIO;
			}
		}

		len -= page_size;
		to += page_size;
		buf += page_size;
		count++;
		if ((count & 0xf) == 0)
			raspi_wait_sleep_ready(1);
			
	}

	raspi_wait_ready(100);

	if (spi_dev->addr4b)
		raspi_4byte_mode(0);

	up(&spi_sem);

	return 0;
}

extern unsigned long spi_flash_id;
/****************************************************************************/
/*
 * board specific setup should have ensured the SPI clock used here
 * matches what the READ command supports, at least until this driver
 * understands FAST_READ (for clocks over 25 MHz).
 */
static int spi_init(void)
{
	struct mtd_partition *mtd_parts;
	unsigned long flash_id = 0;
	unsigned int temp;	
	int np, i, ret_val = -ENOMEM;
	u8 buf[5];

#if 0
#if defined USER_MODE || defined COMMAND_MODE
	
	// use normal(SPI) mode instead of GPIO mode
	ra_and(RALINK_REG_GPIOMODE, ~(1 << 1));

	// reset spi block
	ra_or(RT2880_RSTCTRL_REG, RSTCTRL_SPI_RESET);
	udelay(1);
	ra_and(RT2880_RSTCTRL_REG, ~RSTCTRL_SPI_RESET);

	// FIXME, clk_div should depend on spi-flash. 
	// mode 0 (SPICLKPOL = 0) & (RXCLKEDGE_FALLING = 0)
	// mode 3 (SPICLKPOL = 1) & (RXCLKEDGE_FALLING = 0)
	ra_outl(RT2880_SPICFG_REG, SPICFG_MSBFIRST | SPICFG_TXCLKEDGE_FALLING | CFG_CLK_DIV | SPICFG_SPICLKPOL );

	// set idle state
	ra_outl(RT2880_SPICTL_REG, SPICTL_HIZSDO | SPICTL_SPIENA_HIGH);
	
	spi_wait_nsec = (8 * 1000 / (128 / CFG_CLK_DIV) ) >> 1 ;
	//printk("spi_wait_nsec: %x \n", spi_wait_nsec);
#elif defined BBU_MODE
#if  defined(CONFIG_RALINK_MT7621) || defined(CONFIG_RALINK_MT7628) 
#else
	// enable SMC bank 0 alias addressing
	ra_or(RALINK_SYSCTL_BASE + 0x38, 0x80000000);
#endif
#endif
#else

#if defined(CONFIG_RALINK_MT7621)
	// set default clock to hclk/5
    ra_and(SPI_REG_MASTER, ~(0xfff << 16));
    ra_or(SPI_REG_MASTER, (0x5 << 16));	//work-around 3-wire SPI issue (3 for RFB, 5 for EVB)
#elif defined(CONFIG_RALINK_MT7628)
	// set default clock to hclk/8
    ra_and(SPI_REG_MASTER, ~(0xfff << 16));
    ra_or(SPI_REG_MASTER, (0x6 << 16));
#endif

#endif

	minfo = kzalloc(sizeof(struct mtd_info), GFP_KERNEL);
	if(NULL == minfo) {
		printk("%s: Failed to allocate memory for MTD device\n", __func__);
		goto out;
	}
	
	sema_init(&spi_sem, 1);
	down(&spi_sem);

	raspi_read_devid(buf, 5);
	
	up(&spi_sem);

	//sdk:
	//jedec = (u32)((u32)(buf[1] << 24) | ((u32)buf[2] << 16) | ((u32)buf[3] <<8) | (u32)buf[4]);
	//info->id == buf[0]
	
	flash_id = (buf[0] << 16) | (buf[1] << 8) | (buf[2] << 0);
		
	for(i = 0; ; i++) {
		spi_dev = (struct spi_flash_params *)&flash_table[i];
		if((flash_id == spi_dev->id) || (0 == spi_dev->id)) {
			break;
		}
	}
	minfo->size      = spi_dev->chip_size;
	minfo->erasesize = spi_dev->erasesize;
	minfo->type      = MTD_NORFLASH;
	minfo->erase     = mtd_spi_erase;
	minfo->write     = mtd_spi_write;
	minfo->read      = mtd_spi_read;
	minfo->flags     = MTD_CAP_NORFLASH;
	minfo->name      = "spi_flash";
	minfo->writesize = 1;
	minfo->owner     = THIS_MODULE;

	spi_flash_id = flash_id;
	printk("FLASH: %2ld MB %s(id:0x%x) at mode %d\r\n", (spi_dev->chip_size >> 20), spi_dev->name, flash_id, spi_dev->mode);
	np = parse_mtd_partitions(minfo, part_probes, &mtd_parts, 0);
	if(np > 0) {		
		add_mtd_partitions(minfo, mtd_parts, np);
		ret_val = 0;
   	} else {
		printk("%s: No partitions found\n", __func__);
	}
out:
	return ret_val;
}

static void __exit spi_exit(void)
{
    /* Do nothing */
}



module_init(spi_init);
module_exit(spi_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jerry Xia");
MODULE_DESCRIPTION("MTD SPI driver for Ralink flash chips");


