/*
 *	Copied from Linux Monitor (LiMon) - Networking.
 *
 *	Copyright 1994 - 2000 Neil Russell.
 *	(See License)
 *	Copyright 2000 Roland Borde
 *	Copyright 2000 Paolo Scaffardi
 *	Copyright 2000-2002 Wolfgang Denk, wd@denx.de
 */

/*
 * General Desription:
 *
 * The user interface supports commands for BOOTP, RARP, and TFTP.
 * Also, we support ARP internally. Depending on available data,
 * these interact as follows:
 *
 * BOOTP:
 *
 *	Prerequisites:	- own ethernet address
 *	We want:	- own IP address
 *			- TFTP server IP address
 *			- name of bootfile
 *	Next step:	ARP
 *
 * RARP:
 *
 *	Prerequisites:	- own ethernet address
 *	We want:	- own IP address
 *			- TFTP server IP address
 *	Next step:	ARP
 *
 * ARP:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *			- TFTP server IP address
 *	We want:	- TFTP server ethernet address
 *	Next step:	TFTP
 *
 * DHCP:
 *
 *     Prerequisites:	- own ethernet address
 *     We want:		- IP, Netmask, ServerIP, Gateway IP
 *			- bootfilename, lease time
 *     Next step:	- TFTP
 *
 * TFTP:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *			- TFTP server IP address
 *			- TFTP server ethernet address
 *			- name of bootfile (if unknown, we use a default name
 *			  derived from our own IP address)
 *	We want:	- load the boot file
 *	Next step:	none
 *
 * NFS:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *			- name of bootfile (if unknown, we use a default name
 *			  derived from our own IP address)
 *	We want:	- load the boot file
 *	Next step:	none
 *
 * SNTP:
 *
 *	Prerequisites:	- own ethernet address
 *			- own IP address
 *	We want:	- network time
 *	Next step:	none
 */


#include <common.h>
#include <command.h>
#include <net.h>
#include <tcp.h>
#include <tbs_common.h>
#include "tftp.h"
#ifdef CFG_NMRP
#include "nmrp.h"
#endif
#include <image.h>

extern ulong get_timer(ulong base);
extern int MoreUpdate ;
static int GetUpdateServerUdp;
static int UpdateServerPort;
static int UpdateOurPort;
static unsigned short UpdateUdpId;
extern int NMRPUpgrade;
#define UPDATE_ACK_TIMEOUT	3		

#ifdef CFG_NMRP
int NmrpIsTimeOut = 0;
#endif

#ifdef CONFIG_STATUS_LED
#include <status_led.h>
#include <miiphy.h>
#endif
#if (CONFIG_COMMANDS & CFG_CMD_SNTP)
#include "sntp.h"
#endif

#if (CONFIG_COMMANDS & CFG_CMD_NET)

DECLARE_GLOBAL_DATA_PTR;

#define ARP_TIMEOUT		5		/* Seconds before trying ARP again */
#ifndef	CONFIG_NET_RETRY_COUNT
# define ARP_TIMEOUT_COUNT	5		/* # of timeouts before giving up  */
#else
# define ARP_TIMEOUT_COUNT  (CONFIG_NET_RETRY_COUNT)
#endif

#if 0
#define ET_DEBUG
#endif

/** BOOTP EXTENTIONS **/

IPaddr_t	NetOurSubnetMask=0;		/* Our subnet mask (0=unknown)	*/
IPaddr_t	NetOurGatewayIP=0;		/* Our gateways IP address	*/
IPaddr_t	NetOurDNSIP=0;			/* Our DNS IP address		*/
#if (CONFIG_BOOTP_MASK & CONFIG_BOOTP_DNS2)
IPaddr_t	NetOurDNS2IP=0;			/* Our 2nd DNS IP address	*/
#endif
char		NetOurNISDomain[32]={0,};	/* Our NIS domain		*/
char		NetOurHostName[32]={0,};	/* Our hostname			*/
char		NetOurRootPath[64]={0,};	/* Our bootpath			*/
ushort		NetBootFileSize=0;		/* Our bootfile size in blocks	*/

/** END OF BOOTP EXTENTIONS **/

ulong		NetBootFileXferSize;	/* The actual transferred size of the bootfile (in bytes) */
uchar		NetOurEther[6];		/* Our ethernet address			*/
uchar		NetServerEther[6] =	/* Boot server enet address		*/
			{ 0, 0, 0, 0, 0, 0 };
IPaddr_t	NetOurIP;		/* Our IP addr (0 = unknown)		*/
IPaddr_t	NetServerIP;		/* Our IP addr (0 = unknown)		*/
volatile uchar *NetRxPkt;		/* Current receive packet		*/
int		NetRxPktLen;		/* Current rx packet length		*/
unsigned	NetIPID;		/* IP packet ID				*/
uchar		NetBcastAddr[6] =	/* Ethernet bcast address		*/
			{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
uchar		NetEtherNullAddr[6] =
			{ 0, 0, 0, 0, 0, 0 };
int		NetState;		/* Network loop state			*/

/* XXX in both little & big endian machines 0xFFFF == ntohs(-1) */
ushort		NetOurVLAN = 0xFFFF;		/* default is without VLAN	*/
ushort		NetOurNativeVLAN = 0xFFFF;	/* ditto			*/

char		BootFile[128];		/* Boot File name			*/
IPaddr_t        NetTftpServerIP = 0;        /* TFTP server IP address                */

#ifdef FIRMWARE_RECOVER_FROM_TFTP_SERVER
uchar NetOurTftpIP[4] = { 192, 168, 1, 1 };
int NetRunTftpServer = 0;
uchar TftpClientEther[6] = { 0, 0, 0, 0, 0, 0};
IPaddr_t TftpClientIP = 0;
#ifdef CFG_NO_FLASH
#include <nand.h>
#else
extern flash_info_t flash_info[];
#endif
#endif

#if (CONFIG_COMMANDS & CFG_CMD_PING)
IPaddr_t	NetPingIP;		/* the ip address to ping 		*/

static void PingStart(void);
#endif

#ifdef CONFIG_NETCONSOLE
void NcStart(void);
int nc_input_packet(uchar *pkt, unsigned dest, unsigned src, unsigned len);
#endif

volatile uchar	PktBuf[(PKTBUFSRX+1) * PKTSIZE_ALIGN + PKTALIGN];

volatile uchar *NetRxPackets[PKTBUFSRX]; /* Receive packets			*/

static rxhand_f *packetHandler;		/* Current RX packet handler		*/
static thand_f *timeHandler;		/* Current timeout handler		*/
static ulong	timeStart;		/* Time base value			*/
static ulong	timeDelta;		/* Current timeout value		*/
volatile uchar *NetTxPacket = 0;	/* THE transmit packet			*/

static int net_check_prereq (proto_t protocol);

/**********************************************************************/

IPaddr_t	NetArpWaitPacketIP;
IPaddr_t	NetArpWaitReplyIP;
uchar	       *NetArpWaitPacketMAC;	/* MAC address of waiting packet's destination	*/
uchar	       *NetArpWaitTxPacket;	/* THE transmit packet			*/
int		NetArpWaitTxPacketSize;
uchar 		NetArpWaitPacketBuf[PKTSIZE_ALIGN + PKTALIGN];
ulong		NetArpWaitTimerStart;
int		NetArpWaitTry;

void ArpRequest (void)
{
	int i;
	volatile uchar *pkt;
	ARP_t *arp;

#ifdef ET_DEBUG
	printf ("ARP broadcast %d\n", NetArpWaitTry);
#endif
	pkt = NetTxPacket;

	pkt += NetSetEther (pkt, NetBcastAddr, PROT_ARP);

	arp = (ARP_t *) pkt;

	arp->ar_hrd = htons (ARP_ETHER);
	arp->ar_pro = htons (PROT_IP);
	arp->ar_hln = 6;
	arp->ar_pln = 4;
	arp->ar_op = htons (ARPOP_REQUEST);

	memcpy (&arp->ar_data[0], NetOurEther, 6);		/* source ET addr	*/
	NetWriteIP ((uchar *) & arp->ar_data[6], NetOurIP);	/* source IP addr	*/
	for (i = 10; i < 16; ++i) {
		arp->ar_data[i] = 0;				/* dest ET addr = 0     */
	}

	if ((NetArpWaitPacketIP & NetOurSubnetMask) !=
	    (NetOurIP & NetOurSubnetMask)) {
		if (NetOurGatewayIP == 0) {
			puts ("## Warning: gatewayip needed but not set\n");
			NetArpWaitReplyIP = NetArpWaitPacketIP;
		} else {
			NetArpWaitReplyIP = NetOurGatewayIP;
		}
	} else {
		NetArpWaitReplyIP = NetArpWaitPacketIP;
	}

	NetWriteIP ((uchar *) & arp->ar_data[16], NetArpWaitReplyIP);

	(void) eth_send (NetTxPacket, (pkt - NetTxPacket) + ARP_HDR_SIZE);

#ifdef ET_DEBUG
	printf ("ARP broadcast sent.\n");
#endif

}

void ArpTimeoutCheck(void)
{
	ulong t;

	if (!NetArpWaitPacketIP)
		return;

	t = get_timer(0);

	/* check for arp timeout */
	if ((t - NetArpWaitTimerStart) > ARP_TIMEOUT * CFG_HZ) {
		NetArpWaitTry++;

		if (NetArpWaitTry >= ARP_TIMEOUT_COUNT) {
			puts ("\nARP Retry count exceeded; starting again\n");
			NetArpWaitTry = 0;
			NetStartAgain();
		} else {
			NetArpWaitTimerStart = t;
			ArpRequest();
		}
	}
}

/**********************************************************************/
/*
 *	Main network processing loop.
 */

int
NetLoop(proto_t protocol)
{
	bd_t *bd = gd->bd;
#ifdef ET_DEBUG
	printf("NetLoop from protocol %x\n", protocol);
#endif
	/* XXX problem with bss workaround */
	NetArpWaitPacketMAC = NULL;
	NetArpWaitTxPacket = NULL;
	NetArpWaitPacketIP = 0;
	NetArpWaitReplyIP = 0;
	NetArpWaitTxPacket = NULL;
	NetTxPacket = NULL;

	if (!NetTxPacket) {
		int	i;
		/*
		 *	Setup packet buffers, aligned correctly.
		 */
		NetTxPacket = &PktBuf[0] + (PKTALIGN - 1);
		NetTxPacket -= (ulong)NetTxPacket % PKTALIGN;
		for (i = 0; i < PKTBUFSRX; i++) {
			NetRxPackets[i] = NetTxPacket + (i+1)*PKTSIZE_ALIGN;
		}
	}

	if (!NetArpWaitTxPacket) {
		NetArpWaitTxPacket = &NetArpWaitPacketBuf[0] + (PKTALIGN - 1);
		NetArpWaitTxPacket -= (ulong)NetArpWaitTxPacket % PKTALIGN;
		NetArpWaitTxPacketSize = 0;
	}
	
restart:
	memcpy (NetOurEther, bd->bi_enetaddr, 6);

	NetState = NETLOOP_CONTINUE;

	/*
	 *	Start the ball rolling with the given start function.  From
	 *	here on, this code is a state machine driven by received
	 *	packets and timer events.
	 */

	switch (protocol) {
#ifdef CFG_NMRP
	case NMRP:
		NetCopyIP(&NetOurIP, NetOurTftpIP);
		//NetOurGatewayIP = getenv_IPaddr ("gatewayip");
		//NetOurSubnetMask= getenv_IPaddr ("netmask");
#ifdef CONFIG_NET_VLAN
		NetOurVLAN = getenv_VLAN("vlan");
		NetOurNativeVLAN = getenv_VLAN("nvlan");
#endif
		break;
#endif

#if (CONFIG_COMMANDS & CFG_CMD_PING)
	case PING:
#endif
	case NETCONS:
	case TFTP:
		NetCopyIP(&NetOurIP, &bd->bi_ip_addr);
		NetOurSubnetMask= getenv_netmask();

                switch (protocol) {

		case NETCONS:
		case TFTP:
			NetServerIP = NetTftpServerIP;
#ifdef FIRMWARE_RECOVER_FROM_TFTP_SERVER
			if(protocol == TFTP && NetRunTftpServer)
			{
				NetServerIP = 1;
				NetCopyIP(&NetOurIP, NetOurTftpIP);
				NetOurGatewayIP = 0;
			}
#endif
			break;

#if (CONFIG_COMMANDS & CFG_CMD_PING)
                case PING:
                        /* nothing */
                        break;
#endif
		default:
			break;
		}

		break;
	default:
		break;
	}

	switch (net_check_prereq (protocol)) {
	case 1:
		/* network not configured */
		eth_halt();
		return (-1);


	case 0:
		switch (protocol) {
		case TFTP:
			/* always use ARP to get server ethernet address */
#ifdef FIRMWARE_RECOVER_FROM_TFTP_SERVER
			if(NetRunTftpServer)
			{
				TftpServerStart();
			}
			else
#endif
			TftpStart();
			break;
#ifdef CFG_NMRP
		case NMRP:
			NmrpStart();
			break;
#endif
#if (CONFIG_COMMANDS & CFG_CMD_PING)
		case PING:
			PingStart();
			break;
#endif
#ifdef CONFIG_NETCONSOLE
		case NETCONS:
			NcStart();
			break;
#endif
		default:
			break;
		}

		NetBootFileXferSize = 0;
		break;
	}

#if defined(CONFIG_MII) || (CONFIG_COMMANDS & CFG_CMD_MII)
#if defined(CFG_FAULT_ECHO_LINK_DOWN) && defined(CONFIG_STATUS_LED) && defined(STATUS_LED_RED)
	/*
	 * Echo the inverted link state to the fault LED.
	 */
	if(miiphy_link(eth_get_dev()->name, CFG_FAULT_MII_ADDR)) {
		status_led_set (STATUS_LED_RED, STATUS_LED_OFF);
	} else {
		status_led_set (STATUS_LED_RED, STATUS_LED_ON);
	}
#endif /* CFG_FAULT_ECHO_LINK_DOWN, ... */
#endif /* CONFIG_MII, ... */

	/*
	 *	Main packet reception loop.  Loop receiving packets until
	 *	someone sets `NetState' to a state that terminates.
	 */
skip_netloop:
	for (;;) {

#ifdef CONFIG_SHOW_ACTIVITY
		{
			extern void show_activity(int arg);
			show_activity(1);
		}
#endif
		/*
		 *	Check the ethernet for a new packet.  The ethernet
		 *	receive routine will process it.
		 */
			eth_rx();

		/*
		 *	Abort if ctrl-c was pressed.
		 */
		if (ctrlc()) {
			eth_halt();
			puts ("\nAbort\n");
			return (-1);
		}

		ArpTimeoutCheck();

		/*
		 *	Check for a timeout, and run the timeout handler
		 *	if we have one.
		 */
		if (timeHandler && ((get_timer(0) - timeStart) > timeDelta)) {
			thand_f *x;

#if defined(CONFIG_MII) || (CONFIG_COMMANDS & CFG_CMD_MII)
#  if defined(CFG_FAULT_ECHO_LINK_DOWN) && \
      defined(CONFIG_STATUS_LED) &&	   \
      defined(STATUS_LED_RED)
			/*
			 * Echo the inverted link state to the fault LED.
			 */
			if(miiphy_link(eth_get_dev()->name, CFG_FAULT_MII_ADDR)) {
				status_led_set (STATUS_LED_RED, STATUS_LED_OFF);
			} else {
				status_led_set (STATUS_LED_RED, STATUS_LED_ON);
			}
#  endif /* CFG_FAULT_ECHO_LINK_DOWN, ... */
#endif /* CONFIG_MII, ... */
			x = timeHandler;
			timeHandler = (thand_f *)0;
			(*x)();
		}


		switch (NetState) {

		case NETLOOP_RESTART:
			goto restart;

		case NETLOOP_SUCCESS:
#ifdef CFG_NMRP
			if (NmrpState == STATE_CLOSING)
				goto skip_netloop;
#endif
			if (NetBootFileXferSize > 0) {
				char buf[10];
				printf("Bytes transferred = %ld (%lx hex)\n",
					NetBootFileXferSize,
					NetBootFileXferSize);
				sprintf(buf, "%lx", NetBootFileXferSize);
				//setenv("filesize", buf);

				sprintf(buf, "%lX", (unsigned long)load_addr);
				//setenv("fileaddr", buf);
			}
			//eth_halt();
			return NetBootFileXferSize;

		case NETLOOP_FAIL:
			return (-1);
		}
	}
}

/**********************************************************************/

static void
startAgainTimeout(void)
{
	NetState = NETLOOP_RESTART;
}

static void
startAgainHandler(uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	/* Totally ignore the packet */
}

void NetStartAgain (void)
{
	//char *nretry;

	NetSetTimeout (10 * CFG_HZ, startAgainTimeout);
	NetSetHandler (startAgainHandler);
}

/**********************************************************************/
/*
 *	Miscelaneous bits.
 */

void
NetSetHandler(rxhand_f * f)
{
	packetHandler = f;
}


void
NetSetTimeout(ulong iv, thand_f * f)
{
	if (iv == 0) {
		timeHandler = (thand_f *)0;
	} else {
		timeHandler = f;
		timeStart = get_timer(0);
		timeDelta = iv;
	}
}

void NetConcelTimeout()
{
	timeHandler = (thand_f *)0;
	timeStart = 0;
	timeDelta = 0;
}

void
NetSendPacket(volatile uchar * pkt, int len)
{
	(void) eth_send(pkt, len);
}

int
NetSendUDPPacket(uchar *ether, IPaddr_t dest, int dport, int sport, int len)
{
	uchar *pkt;

	/* convert to new style broadcast */
	if (dest == 0)
		dest = 0xFFFFFFFF;

	/* if broadcast, make the ether address a broadcast and don't do ARP */
	if (dest == 0xFFFFFFFF)
		ether = NetBcastAddr;

	/* if MAC address was not discovered yet, save the packet and do an ARP request */
	if (memcmp(ether, NetEtherNullAddr, 6) == 0) {

#ifdef ET_DEBUG
		printf("sending ARP for %08lx\n", dest);
#endif
		NetArpWaitPacketIP = dest;
		NetArpWaitPacketMAC = ether;

		pkt = NetArpWaitTxPacket;
		pkt += NetSetEther (pkt, NetArpWaitPacketMAC, PROT_IP);
		NetSetIP (pkt, dest, dport, sport, len);
		memcpy(pkt + IP_HDR_SIZE, (uchar *)NetTxPacket + (pkt - (uchar *)NetArpWaitTxPacket) + IP_HDR_SIZE, len);
		/* size of the waiting packet */
		NetArpWaitTxPacketSize = (pkt - NetArpWaitTxPacket) + IP_HDR_SIZE + len;
		/* and do the ARP request */
		NetArpWaitTry = 1;
		NetArpWaitTimerStart = get_timer(0);
		ArpRequest();
		return 1;	/* waiting */
	}

#ifdef ET_DEBUG
	printf("sending UDP to %08lx/%02x:%02x:%02x:%02x:%02x:%02x\n",
		dest, ether[0], ether[1], ether[2], ether[3], ether[4], ether[5]);
#endif

	pkt = (uchar *)NetTxPacket;
	pkt += NetSetEther (pkt, ether, PROT_IP);
	NetSetIP (pkt, dest, dport, sport, len);
	(void) eth_send(NetTxPacket, (pkt - NetTxPacket) + IP_HDR_SIZE + len);

	return 0;	/* transmitted */
}

#if (CONFIG_COMMANDS & CFG_CMD_PING)
static ushort PingSeqNo;

int PingSend(void)
{
	static uchar mac[6];
	volatile IP_t *ip;
	volatile ushort *s;
	uchar *pkt;

	/* XXX always send arp request */

	memcpy(mac, NetEtherNullAddr, 6);

#ifdef ET_DEBUG
	printf("sending ARP for %08lx\n", NetPingIP);
#endif

	NetArpWaitPacketIP = NetPingIP;
	NetArpWaitPacketMAC = mac;

	pkt = NetArpWaitTxPacket;
	pkt += NetSetEther(pkt, mac, PROT_IP);

	ip = (volatile IP_t *)pkt;

	/*
	 *	Construct an IP and ICMP header.  (need to set no fragment bit - XXX)
	 */
	ip->ip_hl_v  = 0x45;		/* IP_HDR_SIZE / 4 (not including UDP) */
	ip->ip_tos   = 0;
	ip->ip_len   = htons(IP_HDR_SIZE_NO_UDP + 8);
	ip->ip_id    = htons(NetIPID++);
	ip->ip_off   = htons(0x4000);	/* No fragmentation */
	ip->ip_ttl   = 255;
	ip->ip_p     = 0x01;		/* ICMP */
	ip->ip_sum   = 0;
	NetCopyIP((void*)&ip->ip_src, &NetOurIP); /* already in network byte order */
	NetCopyIP((void*)&ip->ip_dst, &NetPingIP);	   /* - "" - */
	ip->ip_sum   = ~NetCksum((uchar *)ip, IP_HDR_SIZE_NO_UDP / 2);

	s = &ip->udp_src;		/* XXX ICMP starts here */
	s[0] = htons(0x0800);		/* echo-request, code */
	s[1] = 0;			/* checksum */
	s[2] = 0; 			/* identifier */
	s[3] = htons(PingSeqNo++);	/* sequence number */
	s[1] = ~NetCksum((uchar *)s, 8/2);

	/* size of the waiting packet */
	NetArpWaitTxPacketSize = (pkt - NetArpWaitTxPacket) + IP_HDR_SIZE_NO_UDP + 8;

	/* and do the ARP request */
	NetArpWaitTry = 1;
	NetArpWaitTimerStart = get_timer(0);
	ArpRequest();
	return 1;	/* waiting */
}

static void
PingTimeout (void)
{
	eth_halt();
	NetState = NETLOOP_FAIL;	/* we did not get the reply */
}

static void
PingHandler (uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	IPaddr_t tmp;
	volatile IP_t *ip = (volatile IP_t *)pkt;

	tmp = NetReadIP((void *)&ip->ip_src);
	if (tmp != NetPingIP)
		return;

	NetState = NETLOOP_SUCCESS;
}

static void PingStart(void)
{
	NetSetTimeout (10 * CFG_HZ, PingTimeout);
	NetSetHandler (PingHandler);

	PingSend();
}
#endif	/* CFG_CMD_PING */


void
NetReceive(volatile uchar * inpkt, int len)
{
	Ethernet_t *et;
	IP_t	*ip;
	ARP_t	*arp;
	IPaddr_t tmp;
	int	x;
	uchar *pkt;
	ushort cti = 0, vlanid = VLAN_NONE, myvlanid, mynvlanid;
	char arp_addr[6] = {0xff,0xff,0xff,0xff,0xff,0xff};

#ifdef ET_DEBUG
	printf("packet received\n");
#endif

	NetRxPkt = inpkt;
	NetRxPktLen = len;
	et = (Ethernet_t *)inpkt;

	/* too small packet? */
	if (len < ETHER_HDR_SIZE)
		return;

	/* Dest Mac isn't ARP ,and not belong to us.Drop it*/	
	if( memcmp( et->et_dest , arp_addr , 6 ) && memcmp( et->et_dest , NetOurEther, 6) )
		return ;

	myvlanid = ntohs(NetOurVLAN);
	if (myvlanid == (ushort)-1)
		myvlanid = VLAN_NONE;
	mynvlanid = ntohs(NetOurNativeVLAN);
	if (mynvlanid == (ushort)-1)
		mynvlanid = VLAN_NONE;

	x = ntohs(et->et_protlen);

#ifdef ET_DEBUG
	printf("packet received\n");
#endif

	if (x < 1514) {
		/*
		 *	Got a 802 packet.  Check the other protocol field.
		 */
		x = ntohs(et->et_prot);

		ip = (IP_t *)(inpkt + E802_HDR_SIZE);
		len -= E802_HDR_SIZE;

	} else if (x != PROT_VLAN) {	/* normal packet */
		ip = (IP_t *)(inpkt + ETHER_HDR_SIZE);
		len -= ETHER_HDR_SIZE;

	} else {			/* VLAN packet */
		VLAN_Ethernet_t *vet = (VLAN_Ethernet_t *)et;

#ifdef ET_DEBUG
		printf("VLAN packet received\n");
#endif
		/* too small packet? */
		if (len < VLAN_ETHER_HDR_SIZE)
			return;

		/* if no VLAN active */
		if ((ntohs(NetOurVLAN) & VLAN_IDMASK) == VLAN_NONE
				)
			return;

		cti = ntohs(vet->vet_tag);
		vlanid = cti & VLAN_IDMASK;
		x = ntohs(vet->vet_type);

		ip = (IP_t *)(inpkt + VLAN_ETHER_HDR_SIZE);
		len -= VLAN_ETHER_HDR_SIZE;
	}

#ifdef ET_DEBUG
	printf("Receive from protocol 0x%x\n", x);
#endif


	if ((myvlanid & VLAN_IDMASK) != VLAN_NONE) {
		if (vlanid == VLAN_NONE)
			vlanid = (mynvlanid & VLAN_IDMASK);
		/* not matched? */
		if (vlanid != (myvlanid & VLAN_IDMASK)){
			printf("NetReceive vlan id error");
			return;
		}
	}

	switch (x) {
#ifdef CFG_NMRP
		case PROT_NMRP:
			if(len <= MIN_ETHER_NMRP_LEN){
				printf("bad packet len@!\n");
				return;
			}
			memcpy(NmrpServerEther,et->et_src,6);
			(*packetHandler)((uchar *)ip,0,0,PROT_NMRP);
			break;
#endif
	case PROT_ARP:
		/*
		 * We have to deal with two types of ARP packets:
		 * - REQUEST packets will be answered by sending  our
		 *   IP address - if we know it.
		 * - REPLY packates are expected only after we asked
		 *   for the TFTP server's or the gateway's ethernet
		 *   address; so if we receive such a packet, we set
		 *   the server ethernet address
		 */
#ifdef ET_DEBUG
		puts ("Got ARP\n");
#endif
		arp = (ARP_t *)ip;
		if (len < ARP_HDR_SIZE) {
			printf("bad length %d < %d\n", len, ARP_HDR_SIZE);
			return;
		}
		if (ntohs(arp->ar_hrd) != ARP_ETHER) {
			return;
		}
		if (ntohs(arp->ar_pro) != PROT_IP) {
			return;
		}
		if (arp->ar_hln != 6) {
			return;
		}
		if (arp->ar_pln != 4) {
			return;
		}

		if (NetOurIP == 0) {
			return;
		}

		if (NetReadIP(&arp->ar_data[16]) != NetOurIP) {
			return;
		}

		switch (ntohs(arp->ar_op)) {
		case ARPOP_REQUEST:		/* reply with our IP address	*/
#ifdef ET_DEBUG
			puts ("Got ARP REQUEST, return our IP\n");
#endif
			pkt = (uchar *)et;
			pkt += NetSetEther(pkt, et->et_src, PROT_ARP);
			arp->ar_op = htons(ARPOP_REPLY);
			memcpy   (&arp->ar_data[10], &arp->ar_data[0], 6);
			NetCopyIP(&arp->ar_data[16], &arp->ar_data[6]);
			memcpy   (&arp->ar_data[ 0], NetOurEther, 6);
			NetCopyIP(&arp->ar_data[ 6], &NetOurIP);
			(void) eth_send((uchar *)et, (pkt - (uchar *)et) + ARP_HDR_SIZE);
			return;

		case ARPOP_REPLY:		/* arp reply */
			/* are we waiting for a reply */
			if (!NetArpWaitPacketIP || !NetArpWaitPacketMAC)
				break;
#ifdef ET_DEBUG
			printf("Got ARP REPLY, set server/gtwy eth addr (%02x:%02x:%02x:%02x:%02x:%02x)\n",
				arp->ar_data[0], arp->ar_data[1],
				arp->ar_data[2], arp->ar_data[3],
				arp->ar_data[4], arp->ar_data[5]);
#endif

			tmp = NetReadIP(&arp->ar_data[6]);

			/* matched waiting packet's address */
			if (tmp == NetArpWaitReplyIP) {
#ifdef ET_DEBUG
				puts ("Got it\n");
#endif
				/* save address for later use */
				memcpy(NetArpWaitPacketMAC, &arp->ar_data[0], 6);

#ifdef CONFIG_NETCONSOLE
				(*packetHandler)(0,0,0,0);
#endif
				/* modify header, and transmit it */
				memcpy(((Ethernet_t *)NetArpWaitTxPacket)->et_dest, NetArpWaitPacketMAC, 6);
				(void) eth_send(NetArpWaitTxPacket, NetArpWaitTxPacketSize);

				/* no arp request pending now */
				NetArpWaitPacketIP = 0;
				NetArpWaitTxPacketSize = 0;
				NetArpWaitPacketMAC = NULL;

			}
			return;
		default:
#ifdef ET_DEBUG
			printf("Unexpected ARP opcode 0x%x\n", ntohs(arp->ar_op));
#endif
			return;
		}
		break;

	case PROT_RARP:
#ifdef ET_DEBUG
		puts ("Got RARP\n");
#endif
		arp = (ARP_t *)ip;
		if (len < ARP_HDR_SIZE) {
			printf("bad length %d < %d\n", len, ARP_HDR_SIZE);
			return;
		}

		if ((ntohs(arp->ar_op) != RARPOP_REPLY) ||
			(ntohs(arp->ar_hrd) != ARP_ETHER)   ||
			(ntohs(arp->ar_pro) != PROT_IP)     ||
			(arp->ar_hln != 6) || (arp->ar_pln != 4)) {

			puts ("invalid RARP header\n");
		} else {
			NetCopyIP(&NetOurIP,    &arp->ar_data[16]);
			if (NetServerIP == 0)
				NetCopyIP(&NetServerIP, &arp->ar_data[ 6]);
			memcpy (NetServerEther, &arp->ar_data[ 0], 6);

			(*packetHandler)(0,0,0,0);
		}
		break;

	case PROT_IP:
#ifdef ET_DEBUG
		puts ("Got IP\n");
#endif
		if (len < IP_HDR_SIZE) {
			debug ("len bad %d < %d\n", len, IP_HDR_SIZE);
			return;
		}
		if (len < ntohs(ip->ip_len)) {
			printf("len bad %d < %d\n", len, ntohs(ip->ip_len));
			return;
		}
		len = ntohs(ip->ip_len);
#ifdef ET_DEBUG
		printf("len=%d, v=%02x\n", len, ip->ip_hl_v & 0xff);
#endif
		if ((ip->ip_hl_v & 0xf0) != 0x40) {
			return;
		}
		if (ip->ip_off & htons(0x1fff)) { /* Can't deal w/ fragments */
			return;
		}
		if (!NetCksumOk((uchar *)ip, IP_HDR_SIZE_NO_UDP / 2)) {
			puts ("checksum bad\n");
			return;
		}
		tmp = NetReadIP(&ip->ip_dst);
		if (NetOurIP && tmp != NetOurIP && tmp != 0xFFFFFFFF) {
			return;
		}
		/*
		 * watch for ICMP host redirects
		 *
		 * There is no real handler code (yet). We just watch
		 * for ICMP host redirect messages. In case anybody
		 * sees these messages: please contact me
		 * (wd@denx.de), or - even better - send me the
		 * necessary fixes :-)
		 *
		 * Note: in all cases where I have seen this so far
		 * it was a problem with the router configuration,
		 * for instance when a router was configured in the
		 * BOOTP reply, but the TFTP server was on the same
		 * subnet. So this is probably a warning that your
		 * configuration might be wrong. But I'm not really
		 * sure if there aren't any other situations.
		 */

		if (ip->ip_p == IPPROTO_ICMP) 
		{
			ICMP_t *icmph = (ICMP_t *)&(ip->udp_src);

			switch (icmph->type) 
			{
			case ICMP_REDIRECT:
				if (icmph->code != ICMP_REDIR_HOST)
					return;
				puts (" ICMP Host Redirect to ");
				print_IPaddr(icmph->un.gateway);
				putc(' ');
				return;
#if (CONFIG_COMMANDS & CFG_CMD_PING)
			case ICMP_ECHO_REPLY:
				/*
				 *	IP header OK.  Pass the packet to the current handler.
				 */
				/* XXX point to ip packet */
				(*packetHandler)((uchar *)ip, 0, 0, 0);
				return;
#endif
			default:
				return;
			}
		}
#ifdef CMD_TCP
		else if (ip->ip_p == IPPROTO_TCP) 
		{
		/* TCP packets */
#ifdef ET_DEBUG
			printf("Got TCP\n");
#endif
			tcp_Handler( ( in_Header * ) ip );
			return;
		}
#endif
		else if (ip->ip_p != IPPROTO_UDP) 
		{
		/* Only UDP packets */
					return;
		}

#ifdef CONFIG_UDP_CHECKSUM
		if (ip->udp_xsum != 0) {
			ulong   xsum;
			ushort *sumptr;
			ushort  sumlen;

			xsum  = ip->ip_p;
			xsum += (ntohs(ip->udp_len));
			xsum += (ntohl(ip->ip_src) >> 16) & 0x0000ffff;
			xsum += (ntohl(ip->ip_src) >>  0) & 0x0000ffff;
			xsum += (ntohl(ip->ip_dst) >> 16) & 0x0000ffff;
			xsum += (ntohl(ip->ip_dst) >>  0) & 0x0000ffff;

			sumlen = ntohs(ip->udp_len);
			sumptr = (ushort *) &(ip->udp_src);

			while (sumlen > 1) {
				ushort sumdata;

				sumdata = *sumptr++;
				xsum += ntohs(sumdata);
				sumlen -= 2;
			}
			if (sumlen > 0) {
				ushort sumdata;

				sumdata = *(unsigned char *) sumptr;
				sumdata = (sumdata << 8) & 0xff00;
				xsum += sumdata;
			}
			while ((xsum >> 16) != 0) {
				xsum = (xsum & 0x0000ffff) + ((xsum >> 16) & 0x0000ffff);
			}
			if ((xsum != 0x00000000) && (xsum != 0x0000ffff)) {
				printf(" UDP wrong checksum %08x %08x\n", xsum, ntohs(ip->udp_xsum));
				return;
			}
		}
#endif

#ifdef FIRMWARE_RECOVER_FROM_TFTP_SERVER
		/* Saved the Client IP address anyway for future use */
		TftpClientIP = NetReadIP(&ip->ip_src);
#endif

#ifdef CONFIG_NETCONSOLE
		nc_input_packet((uchar *)ip +IP_HDR_SIZE,
						ntohs(ip->udp_dst),
						ntohs(ip->udp_src),
						ntohs(ip->udp_len) - 8);
#endif
		/*
		 *	IP header OK.  Pass the packet to the current handler.
		 */
		if( packetHandler == NULL )
		{
			return ;		
		}
		else
		{
			(*packetHandler)((uchar *)ip +IP_HDR_SIZE,
							ntohs(ip->udp_dst),
							ntohs(ip->udp_src),
							ntohs(ip->udp_len) - 8);
		}
		break;
	}
}


/**********************************************************************/

static int net_check_prereq (proto_t protocol)
{
	switch (protocol) {
		/* Fall through */
#if (CONFIG_COMMANDS & CFG_CMD_PING)
	case PING:
		if (NetPingIP == 0) {
			puts ("*** ERROR: ping address not given\n");
			return (1);
		}
		goto common;
#endif
	case NETCONS:
	case TFTP:
		if (NetServerIP == 0) {
			puts ("*** ERROR: `serverip' not set\n");
			return (1);
		}
#if (CONFIG_COMMANDS & (CFG_CMD_PING | CFG_CMD_SNTP))
    common:
#endif

		if (NetOurIP == 0) {
			puts ("*** ERROR: `ipaddr' not set\n");
			return (1);
		}
		/* Fall through */

	case DHCP:
	case RARP:
	case BOOTP:
	case CDP:
		if (memcmp (NetOurEther, "\0\0\0\0\0\0", 6) == 0) {
			puts ("*** ERROR: `ethaddr' not set\n");
			return (1);
		}
		/* Fall through */
	default:
		return (0);
	}
	return (0);		/* OK */
}
/**********************************************************************/

int
NetCksumOk(uchar * ptr, int len)
{
	return !((NetCksum(ptr, len) + 1) & 0xfffe);
}


unsigned
NetCksum(uchar * ptr, int len)
{
	ulong	xsum;
	ushort *p = (ushort *)ptr;

	xsum = 0;
	while (len-- > 0)
		xsum += *p++;
	xsum = (xsum & 0xffff) + (xsum >> 16);
	xsum = (xsum & 0xffff) + (xsum >> 16);
	return (xsum & 0xffff);
}

int
NetEthHdrSize(void)
{
	ushort myvlanid;

	myvlanid = ntohs(NetOurVLAN);
	if (myvlanid == (ushort)-1)
		myvlanid = VLAN_NONE;

	return ((myvlanid & VLAN_IDMASK) == VLAN_NONE) ? ETHER_HDR_SIZE : VLAN_ETHER_HDR_SIZE;
}

int
NetSetEther(volatile uchar * xet, uchar * addr, uint prot)
{
	Ethernet_t *et = (Ethernet_t *)xet;
	ushort myvlanid;

	myvlanid = ntohs(NetOurVLAN);
	if (myvlanid == (ushort)-1)
		myvlanid = VLAN_NONE;

	memcpy (et->et_dest, addr, 6);
	memcpy (et->et_src, NetOurEther, 6);
	if ((myvlanid & VLAN_IDMASK) == VLAN_NONE) {
	et->et_protlen = htons(prot);
		return ETHER_HDR_SIZE;
	} else {
		VLAN_Ethernet_t *vet = (VLAN_Ethernet_t *)xet;

		vet->vet_vlan_type = htons(PROT_VLAN);
		vet->vet_tag = htons((0 << 5) | (myvlanid & VLAN_IDMASK));
		vet->vet_type = htons(prot);
		return VLAN_ETHER_HDR_SIZE;
	}
}

void
NetSetIP(volatile uchar * xip, IPaddr_t dest, int dport, int sport, int len)
{
	volatile IP_t *ip = (IP_t *)xip;

	/*
	 *	If the data is an odd number of bytes, zero the
	 *	byte after the last byte so that the checksum
	 *	will work.
	 */
	if (len & 1)
		xip[IP_HDR_SIZE + len] = 0;

	/*
	 *	Construct an IP and UDP header.
	 *	(need to set no fragment bit - XXX)
	 */
	ip->ip_hl_v  = 0x45;		/* IP_HDR_SIZE / 4 (not including UDP) */
	ip->ip_tos   = 0;
	ip->ip_len   = htons(IP_HDR_SIZE + len);
	ip->ip_id    = htons(NetIPID++);
	ip->ip_off   = htons(0x4000);	/* No fragmentation */
	ip->ip_ttl   = 255;
	ip->ip_p     = 17;		/* UDP */
	ip->ip_sum   = 0;
	NetCopyIP((void*)&ip->ip_src, &NetOurIP); /* already in network byte order */
	NetCopyIP((void*)&ip->ip_dst, &dest);	   /* - "" - */
	ip->udp_src  = htons(sport);
	ip->udp_dst  = htons(dport);
	ip->udp_len  = htons(8 + len);
	ip->udp_xsum = 0;
	ip->ip_sum   = ~NetCksum((uchar *)ip, IP_HDR_SIZE_NO_UDP / 2);
}

void copy_filename (char *dst, char *src, int size)
{
	if (*src && (*src == '"')) {
		++src;
		--size;
	}

	while ((--size > 0) && *src && (*src != '"')) {
		*dst++ = *src++;
	}
	*dst = '\0';
}

#endif /* CFG_CMD_NET */

void ip_to_string (IPaddr_t x, char *s)
{
	x = ntohl (x);
	sprintf (s, "%d.%d.%d.%d",
		 (int) ((x >> 24) & 0xff),
		 (int) ((x >> 16) & 0xff),
		 (int) ((x >> 8) & 0xff), (int) ((x >> 0) & 0xff)
	);
}

IPaddr_t string_to_ip(char *s)
{
	IPaddr_t addr;
	char *e;
	int i;

	if (s == NULL)
		return(0);

	for (addr=0, i=0; i<4; ++i) {
		ulong val = s ? simple_strtoul(s, &e, 10) : 0;
		addr <<= 8;
		addr |= (val & 0xFF);
		if (s) {
			s = (*e) ? e+1 : e;
		}
	}

	return (htonl(addr));
}

void print_IPaddr (IPaddr_t x)
{
	char tmp[16];

	ip_to_string (x, tmp);

	puts (tmp);
}

IPaddr_t getenv_netmask()
{
	char mask[] = MK_STR(CONFIG_NETMASK);
	return (string_to_ip(mask));
}

#ifdef __BIG_ENDIAN
#define UPDATE_SERVER_IP		0xc0a801fd
#define UPDATE_SERVER_PORT	5300
#else
#define UPDATE_SERVER_IP		0xfd01a8c0
#define UPDATE_SERVER_PORT	0x14b4		//5300
#endif

static void
UdpHandler (uchar * pkt, unsigned dest, unsigned src, unsigned len)
{
	bd_t				*bd;
	ushort			id;
	IPaddr_t			ip;

	if (dest != UpdateOurPort) 
	{
		printf("Unknow Packet:dest = %d  TftpOurPort= %d \n", dest,UpdateOurPort);
		return;
	}

#ifdef __BIG_ENDIAN
	id = ( *pkt << 8 ) | ( *( pkt+1 ) );
#else
	id = ( *( pkt+1 ) <<8 ) | *pkt ;
#endif

	if (id != UpdateUdpId)
	{
		printf("Unknow Packet:id = %d  UpdateUdpId= %d \n", id,UpdateUdpId);
		return;
	}

	pkt += 2;

	if( (*pkt !=192) || (*( pkt+1 ) !=168)|| (*( pkt+2 ) !=1) )
	{
		return;
	}
	
#ifdef __BIG_ENDIAN
	ip = ( *pkt << 24 ) | ( *( pkt+1 ) << 16 ) | ( *( pkt+2 ) << 8 ) | ( *( pkt+3 ) ) ;
#else
	ip = ( *( pkt+3 ) << 24 ) | ( *( pkt+2 ) << 16 ) | ( *( pkt+1 ) << 8 ) | *pkt;
#endif

	GetUpdateServerUdp = 0;
	MoreUpdate = 1;

	bd = gd->bd;
	bd->bi_ip_addr = ip;
}

int udp_send()
{
	bd_t				*bd;
	volatile uchar		*pkt;
	volatile uchar		*xp;
	int				len = 0;
	volatile ushort		*s;
	sys_config_t		*syscfg;
	int				i;
	ulong			time;
	unsigned int		crc;

	bd = gd->bd;
	syscfg = ( sys_config_t *  ) SYSCFG_OFFSET;

	if( !is_sysdata( syscfg ) )
	{
		return -1;
	}

	NetSetHandler (UdpHandler);

	bd->bi_ip_addr = syscfg->ip ;
	memcpy (bd->bi_enetaddr, syscfg->mac, 6);

	NetCopyIP(&NetOurIP, &syscfg->ip);
	NetOurSubnetMask= getenv_netmask();
	memcpy (NetOurEther, syscfg->mac, 6);	

	UpdateOurPort = (1024 + (get_timer(0) % 3072)) & 0xFFFF;
	UpdateUdpId= get_timer(0) & 0xFFFF;
		

	NetServerIP = UPDATE_SERVER_IP;
	memset(NetServerEther, 0, 6);
	UpdateServerPort = UPDATE_SERVER_PORT;
	
#ifdef CONFIG_DOUBLE_BACKUP	
	if( syscfg->image_mark == 0 )
	{
		crc = syscfg->first_image_checksum;
	}
	else
	{
		crc = syscfg->second_image_checksum;
	}
#else
	crc = syscfg->first_image_checksum;
#endif


	NetArpWaitPacketMAC = NULL;
	NetArpWaitTxPacket = NULL;
	NetArpWaitPacketIP = 0;
	NetArpWaitReplyIP = 0;
	NetArpWaitTxPacket = NULL;
	NetTxPacket = NULL;

	if (!NetTxPacket) 
	{		
		NetTxPacket = &PktBuf[0] + (PKTALIGN - 1);
		NetTxPacket -= (ulong)NetTxPacket % PKTALIGN;
		for (i = 0; i < PKTBUFSRX; i++) 
		{
			NetRxPackets[i] = NetTxPacket + (i+1)*PKTSIZE_ALIGN;
		}
	}

	if (!NetArpWaitTxPacket) 
	{
		NetArpWaitTxPacket = &NetArpWaitPacketBuf[0] + (PKTALIGN - 1);
		NetArpWaitTxPacket -= (ulong)NetArpWaitTxPacket % PKTALIGN;
		NetArpWaitTxPacketSize = 0;
	}


	pkt = NetTxPacket + NetEthHdrSize() + IP_HDR_SIZE;
	xp = pkt;
	s = (ushort *)pkt;
	*s++ = UpdateUdpId;
	pkt = (uchar *)s;
	strcpy ((char *)pkt, "Reqestchangeip");
	pkt += 14;
	
	memcpy( pkt , &crc, 4);
	pkt += 4;
	pkt += strlen((char *)pkt) ;
	len = pkt - xp;

	NetSendUDPPacket(NetServerEther, NetServerIP, UpdateServerPort, UpdateOurPort, len);
	printf("udp_send finished\n");
	
	//NetSetHandler (UdpHandler);

	GetUpdateServerUdp = 1;

	time = get_timer(0);
	
	while(GetUpdateServerUdp)
	{
		eth_rx();
		
		if ((get_timer(0) - time) > UPDATE_ACK_TIMEOUT * CFG_HZ)
		{
			printf("No answer .auto boot ...\n");
			break;
		}		
	}
		
	return 0;
}

#ifdef FIRMWARE_RECOVER_FROM_TFTP_SERVER
#if defined(CFG_NMRP) && (defined(CFG_SINGLE_FIRMWARE))
void UpgradeFirmwareFromNmrpServer(void)
{
#if 1
	NetRunTftpServer = 1;
	ulong addr;
	image_header_t *hdr;
	int file_size;

	addr = load_addr;
	file_size = NetLoop (TFTP);
	if (file_size < 1)
	{
		printf ("\nFirmware recovering from TFTP server is stopped or failed! :( \n");
		NetRunTftpServer = 0;
		return;
	}
#endif
#if defined(CFG_SINGLE_FIRMWARE)
	NmrpState = STATE_TFTPUPLOADING;
	NetSetHandler(NmrpHandler);
	NmrpSend();
#endif

	printf("Ignore Magic number checking when upgrade via NMRP,Magic number is %x!\n", IH_MAGIC);
	//  copy Image to flash
	if(flash_update_img(( unsigned char * )load_addr,  ( sys_config_t *  ) SYSCFG_OFFSET, file_size, 1) != 0){
		/*طNMRP_CODE_TFTP_UL_REQSTATE_CONFIGING״̬*/
		NmrpState = STATE_CONFIGING; /*ʧ˻tftp waiting״̬*/
	}else{
#if defined(CFG_SINGLE_FIRMWARE)
		/* firmware write to flash done */
		NmrpFwUPOption = 0;
		if (NmrpSTUPOption == 1) {
			NmrpState = STATE_CONFIGING;
		} else {
			NmrpState = STATE_CLOSING;
		}
#else
		NmrpState = STATE_CLOSING;
#endif
	}
	NetSetHandler(NmrpHandler);
	NmrpSend();
	NetRunTftpServer = 0;
}
#endif

#if defined(CFG_NMRP) && defined(CFG_SINGLE_FIRMWARE)
void UpgradeStringTableFromNmrpServer(int table_num)
{
	NetRunTftpServer = 1;
	ulong addr;
	image_header_t *hdr;
	int file_size;

	addr = load_addr;
	memset(addr, 0, CFG_STRING_TABLE_LEN);
	file_size = NetLoop (TFTP);
	if (file_size < 1)
	{
		printf ("\nUpdating string table %d from TFTP server \
			is stopped or failed! :( \n", table_num);
		NetRunTftpServer = 0;
		return;
	}

	/* TFTP Uploading done */
	NmrpState = STATE_TFTPUPLOADING;
	NetSetHandler(NmrpHandler);
	NmrpSend();

	/* Write String Table to flash */
	nmrp_upgrade_string_table((uchar *)addr, table_num, file_size);

	/* upgrade string table done, check if more files */
	NmrpStringTableUpdateIndex++;
	if (NmrpStringTableUpdateIndex == NmrpStringTableUpdateCount)
		NmrpSTUPOption = 0;
	if (NmrpFwUPOption == 0 && NmrpSTUPOption == 0) {
		printf("Upgrading all done\n");
		//workaround_qca955x_gmac_nmrp_hang_action();
		NmrpState = STATE_CLOSING;
		NetSetHandler(NmrpHandler);
		NmrpSend();
	} else {
		printf("More files to be upgrading\n");
		//workaround_qca955x_gmac_nmrp_hang_action();
		//workaround_mt7620a_rt2880_nmrp_print_action();
		NmrpState = STATE_CONFIGING;
		NetSetHandler(NmrpHandler);
		NmrpSend();
	}
	NetRunTftpServer = 0;
}
#endif

void ResetTftpServer(void)
{
	timeHandler = 0;
#ifdef CFG_NMRP
	if(NmrpState != 0)
	{
		NmrpState = STATE_CONFIGING;
		NmrpSend();
	}
	else
#endif
	NetState = NETLOOP_RESTART;
}
#endif


#ifdef CFG_NMRP
void StartNmrpClient(void)
{
	/*NMRPյNMRPٽ*/
	udelay(2000000);
	SetLed(led_lan_1, led_off_trig);  /*led 4*/
	SetLed(led_lan_2, led_off_trig);  /*led 5*/
	SetLed(led_lan_3, led_off_trig);  /*led 6*/
	SetLed(led_lan_4, led_off_trig);  /*led 7*/
	SetLed(led_usb, led_off_trig);     /*led 3*/
	udelay(2000000);/*2NMRP*/
        if( NetLoop(NMRP) < 1)
        {
                printf("\n nmrp server is stopped or failed !\n");
				NmrpIsTimeOut = 1;
                return;
        }
}
void ResetBootup_usual(void)
{
        timeHandler = 0;
        NetState = NETLOOP_SUCCESS;
}

#if defined(CFG_CMD_NMRP)
int do_nmrp (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	StartNmrpClient();
	return 0;
}

U_BOOT_CMD(
	nmrp,	1,	0,	do_nmrp,
	"nmrp - start nmrp mechanism to upgrade firmware-image or string-table.\n",
	"- start nmrp mechanism to upgrade firmware-image or string-table.\n"
);
#endif
#endif



