#include <linux/init.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <asm/uaccess.h>

#include "ralink_gpio.h"
#include "acos_led.h"

/***************************************************************************
*** Common LED control definition and routine
****************************************************************************/
struct gpio_reg {
    int gpio_threshhold;
    int gpio_reg_dir;
    int gpio_reg_data;
};

static struct gpio_reg gpio_reg_list[] = {
#if defined (RALINK_GPIO_HAS_2722)
    {
        .gpio_threshhold = 22,
        .gpio_reg_dir = RALINK_REG_PIO2722DIR,
        .gpio_reg_data = RALINK_REG_PIO2722DATA
    },
#elif defined (RALINK_GPIO_HAS_4524)
    {
        .gpio_threshhold = 40,
        .gpio_reg_dir = RALINK_REG_PIO4540DIR,
        .gpio_reg_data = RALINK_REG_PIO4540DATA
    },
    {
        .gpio_threshhold = 24,
        .gpio_reg_dir = RALINK_REG_PIO3924DIR,
        .gpio_reg_data = RALINK_REG_PIO3924DATA
    },
#elif defined (RALINK_GPIO_HAS_5124)
    {
        .gpio_threshhold = 40,
        .gpio_reg_dir = RALINK_REG_PIO5140DIR,
        .gpio_reg_data = RALINK_REG_PIO5140DATA
    },
    {
        .gpio_threshhold = 24,
        .gpio_reg_dir = RALINK_REG_PIO3924DIR,
        .gpio_reg_data = RALINK_REG_PIO3924DATA
    },
#elif defined (RALINK_GPIO_HAS_9524) || defined (RALINK_GPIO_HAS_7224)
    #if defined (RALINK_GPIO_HAS_7224)
    {
        .gpio_threshhold = 72,
        .gpio_reg_dir = RALINK_REG_PIO72DIR,
        .gpio_reg_data = RALINK_REG_PIO72DATA
    },
    #else
    {
        .gpio_threshhold = 72,
        .gpio_reg_dir = RALINK_REG_PIO9572DIR,
        .gpio_reg_data = RALINK_REG_PIO9572DATA
    },
    #endif
    {
        .gpio_threshhold = 40,
        .gpio_reg_dir = RALINK_REG_PIO7140DIR,
        .gpio_reg_data = RALINK_REG_PIO7140DATA
    },
    {
        .gpio_threshhold = 24,
        .gpio_reg_dir = RALINK_REG_PIO3924DIR,
        .gpio_reg_data = RALINK_REG_PIO3924DATA
    },
#elif defined (RALINK_GPIO_HAS_9532)
    {
        .gpio_threshhold = 64,
        .gpio_reg_dir = RALINK_REG_PIO9564DIR,
        .gpio_reg_data = RALINK_REG_PIO9564DATA
    },
    {
        .gpio_threshhold = 32,
        .gpio_reg_dir = RALINK_REG_PIO6332DIR,
        .gpio_reg_data = RALINK_REG_PIO6332DATA
    },
#endif
    {
        .gpio_threshhold = 0,
        .gpio_reg_dir = RALINK_REG_PIODIR,
        .gpio_reg_data = RALINK_REG_PIODATA
    }
};

static int ralink_gpio_get_reg(int *gpio, int *reg_dir, int *reg_data)
{
    int i, reg_list_size;

    if (!gpio)
        return -1;

    reg_list_size = sizeof(gpio_reg_list)/sizeof(struct gpio_reg);
    for (i = 0; i < reg_list_size; i++) {
        if (*gpio >= gpio_reg_list[i].gpio_threshhold) {
            *gpio -= gpio_reg_list[i].gpio_threshhold;
            if (reg_dir)
                *reg_dir = gpio_reg_list[i].gpio_reg_dir;
            if (reg_data)
                *reg_data = gpio_reg_list[i].gpio_reg_data;
            break;
        }
    }

    return 0;
}

static int adjust_active_mode(int *gpio, unsigned int *active)
{
    if (!gpio)
        return -1;

    if (*gpio < 0 || GPIO_NUM(*gpio) >= RALINK_GPIO_NUMBER) {
        printk("gpio %d out of range (0 ~ %d)\n",
               GPIO_NUM(*gpio), RALINK_GPIO_NUMBER - 1);
        return -1;
    }

    if (active)
        *active = GPIO_ACTIVE_MODE(*gpio);

    *gpio = GPIO_NUM(*gpio);
    return 0;
}

/*
 * Configure the direction of the GPIO pins
 */
int ralink_gpio_set_dir(int gpio_pin, int gpio_dir)
{
    int gpio = gpio_pin;
    int reg_dir;
    unsigned long value;

    if (adjust_active_mode(&gpio, NULL) < 0)
        return -1;

    if (ralink_gpio_get_reg(&gpio, &reg_dir, NULL) < 0)
        return -1;

    value = le32_to_cpu(*(volatile u_long *)reg_dir);
    if (gpio_dir == gpio_out)
        value |= (0x1 << gpio);
    else
        value &= ~(0x1 << gpio);

    *(volatile u_long *)reg_dir = cpu_to_le32(value);
    return 0;
}

/*
 * Read the corresponding active status from the GPIO pins
 */
int ralink_gpio_read(int gpio_pin, unsigned long *value)
{
    int gpio = gpio_pin;
    int reg_data;
    unsigned int active;

    if (!value)
        return -1;

    *value = 0;

    if (adjust_active_mode(&gpio, &active) < 0)
        return -1;

    if (ralink_gpio_get_reg(&gpio, NULL, &reg_data) < 0)
        return -1;

    *value = le32_to_cpu(*(volatile u_long *)reg_data);

    if (active == GPIO_ACTIVE_MODE_HIGH)
        *value &= (0x1 << gpio);
    else
        *value = ~(*value) & (0x1 << gpio);

    return 0;
}

/*
 * Write the corresponding active status to the GPIO pins
 */
int ralink_gpio_write(int gpio_pin, unsigned long value)
{
    int gpio = gpio_pin;
    int reg_data;
    unsigned int active;
    unsigned long data;

    if (adjust_active_mode(&gpio, &active) < 0)
        return -1;

    if (ralink_gpio_get_reg(&gpio, NULL, &reg_data) < 0)
        return -1;

    data = le32_to_cpu(*(volatile u_long *)reg_data);
    if ((active == GPIO_ACTIVE_MODE_HIGH && value)
        || (active == GPIO_ACTIVE_MODE_LOW && value == 0))
        data |= (0x1 << gpio);
    else
        data &= ~(0x1 << gpio);

    *(volatile u_long *)reg_data = cpu_to_le32(data);
    return 0;
}

int gpio_control_dir(int pin, int dir)
{
    return ralink_gpio_set_dir(pin, dir);
}

int gpio_control_normal(int pin, int value)
{
    return ralink_gpio_write(pin, value);
}

/***************************************************************************
*** WPS LED control routine
****************************************************************************/
#define WPS_LED_BLINK_RATE_NORMAL   (25)    /* 500ms on-off */
#define WPS_LED_BLINK_RATE_QUICK    (5)     /* 100ms on-off */
#define WPS_LED_BLINK_LAST_TIME     (250)   /* last 5 sec */

static int wps_led_state = 0;
static int wps_led_state_old = 1;
static int is_wl_secu_mode = 0;
static int wps_led_is_on = 0;

static void wps_led_on_off(int gpio, int value)
{
    wps_led_is_on = value;
    gpio_control_normal(gpio, value);
}

/* 900ms on, 100ms off */
static void wps_ap_lockdown_blink(void)
{
    static int interrupt_count = -1;

    interrupt_count++;
    if (interrupt_count == WPS_LED_BLINK_RATE_QUICK * 10)
        interrupt_count = 0;
    
    if (interrupt_count == 0)
        wps_led_on_off(GPIO_WPS_LED, 1);
    else if (interrupt_count == WPS_LED_BLINK_RATE_QUICK)
        wps_led_on_off(GPIO_WPS_LED, 0);
}

/* 100ms on, 100ms off, last 5s */
static void wps_fail_blink(void)
{
    static int blink_interval = WPS_LED_BLINK_LAST_TIME; /* 5 seconds */
    static int interrupt_count = -1;

    blink_interval--;
    interrupt_count++;
    if (interrupt_count == WPS_LED_BLINK_RATE_QUICK * 2)
        interrupt_count = 0;

    if (interrupt_count == 0)
        wps_led_on_off(GPIO_WPS_LED, 1);
    else if (interrupt_count == WPS_LED_BLINK_RATE_QUICK)
        wps_led_on_off(GPIO_WPS_LED, 0);

    if ( blink_interval <= 0 ) {
        blink_interval = WPS_LED_BLINK_LAST_TIME;
        wps_led_state = 0;
    }
}

/* 500ms on, 500ms off */
static void wps_normal_blink(void)
{
    static int interrupt_count = -1;

    interrupt_count++;
    if (interrupt_count == WPS_LED_BLINK_RATE_NORMAL * 2)
        interrupt_count = 0;
    
    if (interrupt_count == 0)
        wps_led_on_off(GPIO_WPS_LED, 1);
    else if (interrupt_count == WPS_LED_BLINK_RATE_NORMAL)
        wps_led_on_off(GPIO_WPS_LED, 0);
}

/***************************************************************************
*** USB LED control routine
****************************************************************************/
#define USB_LED_BLINK_RATE      (2)    /* 40ms on-off */
int usb_pkt_cnt = 0;
EXPORT_SYMBOL(usb_pkt_cnt);

static int usb_led_state = 0;
static int usb_led_state_old = 1;

static void usb_led_on_off(int gpio, int value)
{
    gpio_control_normal(gpio, value);
}

/*  */
static void usb_normal_blink(void)
{
    static int interrupt_count = -1;
    static int usb_pkt_cnt_old = 0;

    interrupt_count++;
    if (interrupt_count == USB_LED_BLINK_RATE * 2)
        interrupt_count = 0;

    if (interrupt_count == 0)
        usb_led_on_off(GPIO_USB_LED, 1);
    else if (interrupt_count == USB_LED_BLINK_RATE) {
        if (usb_pkt_cnt != usb_pkt_cnt_old) {
            usb_pkt_cnt_old = usb_pkt_cnt;
            usb_led_on_off(GPIO_USB_LED, 0);
        }
    }
}

/***************************************************************************
*** LED timer handler
****************************************************************************/
/* HZ is 250, frequency can't be 10ms,50ms...etc */
#define ACOS_LED_FREQ               (HZ/50) /* 1 cycle = 20ms */
extern unsigned long volatile jiffies;
static struct timer_list acos_led_timer;

static void acos_led_do_timer(unsigned long unused)
{
    if (wps_led_state == 0) {
        if (wps_led_state_old)
            wps_led_on_off(GPIO_WPS_LED, 0);

        if ((!is_wl_secu_mode) && wps_led_is_on)
            wps_led_on_off(GPIO_WPS_LED, 0);

        if (is_wl_secu_mode && (!wps_led_is_on))
            wps_led_on_off(GPIO_WPS_LED, 1);
    } else if (wps_led_state == 1) {
        wps_normal_blink();
    } else if (wps_led_state == 2) {
        wps_fail_blink();
    } else if (wps_led_state == 3) {
        wps_ap_lockdown_blink();
    }

    wps_led_state_old = wps_led_state;

    if (usb_led_state)
        usb_normal_blink();
    else if (usb_led_state_old)
        usb_led_on_off(GPIO_USB_LED, 0);

    usb_led_state_old = usb_led_state;

    init_timer(&acos_led_timer);
    acos_led_timer.expires = jiffies + ACOS_LED_FREQ;
    add_timer(&acos_led_timer);
}

static void acos_led_init_timer(void)
{
    init_timer(&acos_led_timer);
    acos_led_timer.function = acos_led_do_timer;
    acos_led_timer.expires = jiffies + ACOS_LED_FREQ;
    add_timer(&acos_led_timer);
}

/***************************************************************************
*** module routine
****************************************************************************/
static int
acos_led_open(struct inode *inode, struct file * file)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    MOD_INC_USE_COUNT;
#else
    try_module_get(THIS_MODULE);
#endif
    return 0;
}

static int
acos_led_release(struct inode *inode, struct file * file)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    MOD_DEC_USE_COUNT;
#else
    module_put(THIS_MODULE);
#endif
    return 0;
}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
static long
acos_led_ioctl(struct file *file, unsigned int req,
        unsigned long arg)
#else
static int
acos_led_ioctl(struct inode *inode, struct file *file, unsigned int req,
        unsigned long arg)
#endif
{
    switch (req) {
    case USB_LED_STATE_ON:
        usb_led_state = 1;
        return 0;
    case USB_LED_STATE_OFF:
        usb_led_state = 0;
        return 0;
    default:
        break;
    }

    if (arg)
        is_wl_secu_mode = 1;
    else
        is_wl_secu_mode = 0;

    switch (req) {
    case WPS_LED_BLINK_NORMAL:
        wps_led_state = 1;
        break;
    case WPS_LED_BLINK_QUICK:
        wps_led_state = 2;
        break;
    case WPS_LED_BLINK_OFF:
        /* wps_led_state will change to 0 automatically after
         * blinking a few seconds if it's 2 currently
         */
        if (wps_led_state != 2)
            wps_led_state = 0;
        break;
    case WPS_LED_BLINK_AP_LOCKDOWN:
        wps_led_state = 3;
        break;
    default:
        break;
    }

    return 0;
}

static struct file_operations acos_led_fops = {
    owner:      THIS_MODULE,
    open:       acos_led_open,
    release:    acos_led_release,
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
    unlocked_ioctl: acos_led_ioctl,
#else
    ioctl:      acos_led_ioctl,
#endif
};

#ifdef CONFIG_DEVFS_FS
static int acos_led_major;
static devfs_handle_t acos_leddev_handle;
#endif /* CONFIG_DEVFS_FS */

static int __init
acos_led_init(void)
{
#ifdef CONFIG_DEVFS_FS
    if ((acos_led_major = devfs_register_chrdev(ACOS_LED_MAJOR_NUM,
                                ACOS_LED_DEVNAME, &acos_led_fops)) < 0)
        return acos_led_major;

    acos_leddev_handle = devfs_register(NULL, ACOS_LED_DEVNAME,
                                DEVFS_FL_DEFAULT, acos_led_major, 0,
                                S_IFCHR | S_IRUGO | S_IWUGO,
                                &acos_led_fops, NULL);
#else
    int ret_val;
    ret_val = register_chrdev(ACOS_LED_MAJOR_NUM, ACOS_LED_DEVNAME,
                                &acos_led_fops);
    if (ret_val < 0)
    {
        printk(KERN_ERR "acos_led_init: unable to register %s (%d)\n",
                ACOS_LED_DEVNAME, ret_val);
        return ret_val;
    }
#endif

    gpio_control_dir(GPIO_WPS_LED, gpio_out);
    gpio_control_dir(GPIO_USB_LED, gpio_out);
    acos_led_init_timer();
    return 0;
}

static void __exit
acos_led_exit(void)
{
#ifdef CONFIG_DEVFS_FS
    if (acos_leddev_handle != NULL)
        devfs_unregister(acos_leddev_handle);
    acos_leddev_handle = NULL;
    devfs_unregister_chrdev(acos_led_major, ACOS_LED_DEVNAME);
#else
    unregister_chrdev(ACOS_LED_MAJOR_NUM, ACOS_LED_DEVNAME);
#endif

    del_timer(&acos_led_timer);
}

module_init(acos_led_init);
module_exit(acos_led_exit);
