#include <linux/version.h>
#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fs.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
#include <linux/slab.h>
#endif

#ifdef  CONFIG_DEVFS_FS
#include <linux/devfs_fs_kernel.h>
static  devfs_handle_t devfs_handle;
#endif

#include "acos_nvram.h"

char acos_nvram_debug = 0;
#define ACOSNV_PRINT(x, ...) do { \
    if (acos_nvram_debug) \
        printk("acos_nvram: " x, ## __VA_ARGS__); \
} while(0)

#define ACOSNV_ERROR(x, ...) do { \
    printk("acos_nvram: " x, ## __VA_ARGS__); \
} while(0)

extern int ra_mtd_write_nm(char *name, loff_t to, size_t len, const u_char *buf);
extern int ra_mtd_read_nm(char *name, loff_t from, size_t len, u_char *buf);

static int acos_nvram_major = ACOS_NVRAM_MAJOR_NUM;
static spinlock_t acos_nvram_lock = SPIN_LOCK_UNLOCKED;
static struct semaphore acos_nvram_sem;
static int acos_nvram_init = -1;

static char value_buf[NVRAM_SPACE];
static unsigned long value_offset = 0;

static struct nvram_tuple *nvram_hash[257];
static struct nvram_tuple *nvram_dead;

/******************************************************************************
 * Internal Functions
 *****************************************************************************/
static int acos_mtd_write(char *name, loff_t to, size_t len, const u_char *buf)
{
    int ret;

    down(&acos_nvram_sem);
    ret = ra_mtd_write_nm(name, to, len, buf);
    up(&acos_nvram_sem);

    return ret;
}

static int acos_mtd_read(char *name, loff_t from, size_t len, u_char *buf)
{
    int ret;

    down(&acos_nvram_sem);
    ret = ra_mtd_read_nm(name, from, len, buf);
    up(&acos_nvram_sem);

    return ret;
}

/*
 * Checksum routine for Internet Protocol family headers (C Version)
 */
static unsigned short hd_cksum(unsigned short *addr, int len)
{
    unsigned short nleft = len;
    unsigned short *w = addr;
    unsigned short answer;
    unsigned short sum = 0;

    /*
     *  Our algorithm is simple, using a 32 bit accumulator (sum),
     *  we add sequential 16 bit words to it, and at the end, fold
     *  back all the carry bits from the top 16 bits into the lower
     *  16 bits.
     */
    while (nleft > 1) {
        sum += *w++;
        nleft -= 2;
    }

    /* mop up an odd byte, if necessary */
    if (nleft == 1)
        sum += *(unsigned char *)w;

    /*
     * add back carry outs from top 16 bits to low 16 bits
     */
    sum = (sum >> 16) + (sum & 0xffff); /* add hi 16 to low 16 */
    sum += (sum >> 16);                 /* add carry */
    answer = ~sum;                      /* truncate to 16 bits */
    return (answer);
}

static int acos_knvram_read(struct nvram_header *header)
{
    ACOSNV_PRINT("--> acos_knvram_read\n");

    if (!header)
        return -EINVAL;

    if (acos_mtd_read(ACOS_NVRAM_MTDNAME, 0, NVRAM_SPACE, (char *)header) < 0) {
        ACOSNV_ERROR("mtd read error!\n");
        return -EIO;
    }

    if (header->magic != NVRAM_MAGIC) {
        ACOSNV_ERROR("magic error:0x%lx!\n", header->magic);
        return -EIO;
    }

    if (header->len > NVRAM_SPACE) {
        ACOSNV_ERROR("len error:0x%lx!\n", header->len);
        return -EIO;
    }

    if (header->version != NVRAM_VERSION) {
        ACOSNV_ERROR("version error:0x%lx!\n", header->version);
        return -EIO;
    }

    /* verify checksum */
    if (hd_cksum((unsigned short *)header, header->len)) {
        ACOSNV_ERROR("checksum error!\n");
        return -EIO;
    }

    return 0;
}

/* (Re)allocate tuple. Should be locked. */
static struct nvram_tuple *_knvram_realloc(struct nvram_tuple *t,
            const char *name, const char *value)
{
    ACOSNV_PRINT("--> _knvram_realloc\n");

    if ((value_offset + strlen(value) + 1) > NVRAM_SPACE)
        return NULL;

    if (!t) {
        if (!(t = kmalloc(sizeof(struct nvram_tuple) + strlen(name) + 1,
                    GFP_ATOMIC)))
            return NULL;

        /* Copy name */
        t->name = (char *)t + sizeof(struct nvram_tuple);
        strcpy(t->name, name);

        t->value = NULL;
    }

    /* Copy value */
    if (!t->value || strcmp(t->value, value)) {
        t->value = &value_buf[value_offset];
        strcpy(t->value, value);
        value_offset += strlen(value) + 1;
    }

    return t;
}

/* Free tuple. Should be locked. */
static void _knvram_free_tuple(struct nvram_tuple *t)
{
    if (t)
        kfree(t);
    else
        value_offset = 0;
}

/* Free all tuples. Should be locked. */
static void _knvram_free_hash(void)
{
    unsigned int i;
    struct nvram_tuple *t, *next;

    ACOSNV_PRINT("--> _knvram_free_hash\n");

    /* Free hash table */
    for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
        for (t = nvram_hash[i]; t; t = next) {
            next = t->next;
            _knvram_free_tuple(t);
        }
        nvram_hash[i] = NULL;
    }

    /* Free dead table */
    for (t = nvram_dead; t; t = next) {
        next = t->next;
        _knvram_free_tuple(t);
    }
    nvram_dead = NULL;

    /* Indicate to per-port code that all tuples have been freed */
    _knvram_free_tuple(NULL);
}

/* String hash */
static unsigned int hash(const char *s)
{
    unsigned int hash_val = 0;

    while (*s)
        hash_val = 31 * hash_val + *s++;

    return hash_val;
}

/* Get the value of an NVRAM variable. Should be locked. */
static char *_knvram_get(const char *name)
{
    unsigned int i;
    struct nvram_tuple *t;
    char *value;

    if (!name)
        return NULL;

    ACOSNV_PRINT("--> _knvram_get %s\n", name);

    /* Hash the name */
    i = hash(name) % ARRAYSIZE(nvram_hash);

    /* Find the associated tuple in the hash table */
    for (t = nvram_hash[i]; t && strcmp(t->name, name); t = t->next);

    value = t ? t->value : NULL;

    return value;
}

/* Get all NVRAM variables. Should be locked. */
static int _knvram_getall(char *buf, int count)
{
    unsigned int i;
    struct nvram_tuple *t;
    int len = 0;

    if (!buf || count <= 0)
        return -EINVAL;

    ACOSNV_PRINT("--> _knvram_getall\n");

    memset(buf, 0, count);

    /* Write name=value\0 ... \0\0 */
    for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
        for (t = nvram_hash[i]; t; t = t->next) {
            if ((count - len) > (strlen(t->name) + 1 + strlen(t->value) + 1))
                len += sprintf(buf + len, "%s=%s", t->name, t->value) + 1;
            else
                break;
        }
    }

    return 0;
}

/* Set the value of an NVRAM variable. Should be locked. */
static int _knvram_set(const char *name, const char *value)
{
    unsigned int i;
    struct nvram_tuple *t, *u, **prev;

    if (!name || !value)
        return -EINVAL;

    ACOSNV_PRINT("--> _knvram_set %s=%s\n", name, value);

    /* Hash the name */
    i = hash(name) % ARRAYSIZE(nvram_hash);

    /* Find the associated tuple in the hash table */
    for (prev = &nvram_hash[i], t = *prev; t && strcmp(t->name, name);
         prev = &t->next, t = *prev);

    /* (Re)allocate tuple */
    if (!(u = _knvram_realloc(t, name, value)))
        return -ENOMEM;

    /* Value reallocated */
    if (t && t == u)
        return 0;

    /* Move old tuple to the dead table */
    if (t) {
        *prev = t->next;
        t->next = nvram_dead;
        nvram_dead = t;
    }

    /* Add new tuple to the hash table */
    u->next = nvram_hash[i];
    nvram_hash[i] = u;

    return 0;
}

/* Unset the value of an NVRAM variable. Should be locked. */
static int _knvram_unset(const char *name)
{
    unsigned int i;
    struct nvram_tuple *t, **prev;

    if (!name)
        return -EINVAL;

    ACOSNV_PRINT("--> _knvram_unset %s\n", name);

    /* Hash the name */
    i = hash(name) % ARRAYSIZE(nvram_hash);

    /* Find the associated tuple in the hash table */
    for (prev = &nvram_hash[i], t = *prev; t && strcmp(t->name, name);
         prev = &t->next, t = *prev);

    /* Move it to the dead table */
    if (t) {
        *prev = t->next;
        t->next = nvram_dead;
        nvram_dead = t;
    }

    return 0;
}

/* (Re)initialize the hash table. Should be locked. */
static int _knvram_rehash(struct nvram_header *header)
{
    char *name, *value, *end, *eq;

    /* (Re)initialize hash table */
    _knvram_free_hash();

    ACOSNV_PRINT("--> _knvram_rehash\n");

    /* Parse and set "name=value\0 ... \0\0" */
    name = (char *)header + NVRAM_HEADER_SIZE;
    end = (char *)header + NVRAM_SPACE - 2;
    end[0] = end[1] = '\0';
    for (; *name; name = value + strlen(value) + 1) {
        if (!(eq = strchr(name, '=')))
            break;
        *eq = '\0';
        value = eq + 1;
        _knvram_set(name, value);
        *eq = '=';
    }

    return 0;
}

/* Regenerate NVRAM. Should be locked. */
static int _knvram_commit(struct nvram_header *header)
{
    char *ptr, *end;
    unsigned int i;
    struct nvram_tuple *t;

    memset((char *)header, 0xff, NVRAM_SPACE);
    memset((char *)header, 0, NVRAM_HEADER_SIZE);

    /* Regenerate header */
    header->magic = NVRAM_MAGIC;
    header->version = NVRAM_VERSION;

    ptr = (char *)header + NVRAM_HEADER_SIZE;
    /* Leave space for a double NUL at the end */
    end = (char *)header + NVRAM_SPACE - 2;

    /* Write out all tuples */
    for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
        for (t = nvram_hash[i]; t; t = t->next) {
            if ((ptr + strlen(t->name) + 1 + strlen(t->value) + 1) > end)
                break;
            ptr += sprintf(ptr, "%s=%s", t->name, t->value) + 1;
        }
    }

    /* End with a double NUL */
    memset(ptr, 0, 2);

    /* Set new length */
    header->len = ROUNDUP(ptr - (char *)header, NVRAM_HEADER_SIZE);

    /* Set new checksum */
    header->checksum = hd_cksum((unsigned short *)header, header->len);

    /* Reinitialize hash table */
    return _knvram_rehash(header);
}

/******************************************************************************
 * Real functions with spin_lock
 *****************************************************************************/
static char *acos_knvram_get(const char *name)
{
    unsigned long flags;
    char *value;

    spin_lock_irqsave(&acos_nvram_lock, flags);
    value = _knvram_get(name);
    spin_unlock_irqrestore(&acos_nvram_lock, flags);

    return value;
}

static int acos_knvram_getall(char *buf, int count)
{
    unsigned long flags;
    int ret;

    spin_lock_irqsave(&acos_nvram_lock, flags);
    ret = _knvram_getall(buf, count);
    spin_unlock_irqrestore(&acos_nvram_lock, flags);

    return ret;
}

static int acos_knvram_set(const char *name, const char *value)
{
    unsigned long flags;
    int ret;
    struct nvram_header *header;

    spin_lock_irqsave(&acos_nvram_lock, flags);
    if ((ret = _knvram_set(name, value))) {
        /* Consolidate space and try again */
        if ((header = kmalloc(NVRAM_SPACE, GFP_ATOMIC))) {
            if (_knvram_commit(header) == 0)
                ret = _knvram_set(name, value);
            kfree(header);
        }
    }
    spin_unlock_irqrestore(&acos_nvram_lock, flags);

    return ret;
}

static int acos_knvram_unset(const char *name)
{
    unsigned long flags;
    int ret;

    spin_lock_irqsave(&acos_nvram_lock, flags);
    ret = _knvram_unset(name);
    spin_unlock_irqrestore(&acos_nvram_lock, flags);

    return ret;
}

static int acos_knvram_match(const char *name, const char *match)
{
    char *value;

    if (!name || !match)
        return 0;

    value = acos_knvram_get(name);
    return (value && !strcmp(value, match));
}

static int acos_knvram_commit(void)
{
    unsigned long flags;
    struct nvram_header *header;
    int ret;

    ACOSNV_PRINT("--> acos_knvram_commit\n");
    header = (struct nvram_header *)kmalloc(NVRAM_SPACE, GFP_KERNEL);
    if (!header)
        return -ENOMEM;

    /* NOT allow commit after erase nvram partition */
    memset((char *)header, 0, NVRAM_SPACE);
    ret = acos_mtd_read(ACOS_NVRAM_MTDNAME, 0, NVRAM_SPACE, (char *)header);
    if (ret < 0)
        goto done;

    if (header->magic == 0xffffffff) {
        ret = -EPERM;
        goto done;
    }

    /* Regenerate NVRAM */
    spin_lock_irqsave(&acos_nvram_lock, flags);
    ret = _knvram_commit(header);
    spin_unlock_irqrestore(&acos_nvram_lock, flags);

    ret = acos_mtd_write(ACOS_NVRAM_MTDNAME, 0, NVRAM_SPACE, (char *)header);

done:
    kfree(header);
    return ret;
}

/* clear data area by writing all 1's value */
static int acos_knvram_clear(void)
{
    struct nvram_header *header;
    int ret;

    ACOSNV_PRINT("--> acos_knvram_clear\n");
    header = (struct nvram_header *)kmalloc(NVRAM_SPACE, GFP_KERNEL);
    if (!header)
        return -ENOMEM;

    memset((char *)header, 0xff, NVRAM_SPACE);
    memset((char *)header, 0, NVRAM_HEADER_SIZE + 2);

    /* Regenerate header */
    header->magic = NVRAM_MAGIC;
    header->version = NVRAM_VERSION;
    header->len = NVRAM_HEADER_SIZE;
    header->checksum = hd_cksum((unsigned short *)header, header->len);

    ret = acos_mtd_write(ACOS_NVRAM_MTDNAME, 0, NVRAM_SPACE, (char *)header);

    kfree(header);
    return ret;
}

/* clear nvram header */
static int acos_knvram_inval(void)
{
    struct nvram_header *header;
    int ret;

    ACOSNV_PRINT("--> acos_knvram_inval\n");
    header = (struct nvram_header *)kmalloc(NVRAM_HEADER_SIZE, GFP_ATOMIC);
    if (!header)
        return -ENOMEM;

    memset((char *)header, 0xff, NVRAM_HEADER_SIZE);
    ret = acos_mtd_write(ACOS_NVRAM_MTDNAME, 0, NVRAM_HEADER_SIZE,
                (char *)header);

    kfree(header);
    return ret;
}

/* Initialize hash table. */
static int acos_knvram_init(void)
{
    unsigned long flags;
    struct nvram_header *header;
    int ret;

    ACOSNV_PRINT("--> acos_knvram_init\n");
    header = (struct nvram_header *)kmalloc(NVRAM_SPACE, GFP_KERNEL);
    if (!header)
        return -ENOMEM;

    memset((char *)header, 0, NVRAM_SPACE);
    ret = acos_knvram_read(header);
    if (ret == 0) {
        spin_lock_irqsave(&acos_nvram_lock, flags);
        _knvram_rehash(header);
        spin_unlock_irqrestore(&acos_nvram_lock, flags);
    }

    kfree(header);
    return ret;
}

/* Free hash table. */
static int acos_knvram_exit(void)
{
    unsigned long flags;

    ACOSNV_PRINT("--> acos_knvram_exit\n");
    spin_lock_irqsave(&acos_nvram_lock, flags);
    _knvram_free_hash();
    spin_unlock_irqrestore(&acos_nvram_lock, flags);

    return 0;
}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
static long acos_nvram_ioctl(struct file *file, unsigned int req,
        unsigned long arg)
#else
static int acos_nvram_ioctl(struct inode *inode, struct file *file,
        unsigned int req, unsigned long arg)
#endif
{
    acos_nvram_ioctl_t *nvr;
    char *value;
    int ret = -EINVAL;

    switch (req) {
    case ACOS_NVRAM_IOCTL_INIT:
        nvr = (acos_nvram_ioctl_t *)arg;
        if (acos_nvram_init < 0)
            acos_nvram_init = acos_knvram_init();
        ret = acos_nvram_init;
        break;
    case ACOS_NVRAM_IOCTL_GET:
        nvr = (acos_nvram_ioctl_t *)arg;
        value = acos_knvram_get(nvr->name);
        if (!value)
            return 0;

        if (nvr->value_len < strlen(value) + 1)
            return -EFAULT;

        if (copy_to_user(nvr->value, value, strlen(value) + 1))
            return -EFAULT;

        ret = strlen(value) + 1;
        break;
    case ACOS_NVRAM_IOCTL_GETALL:
        nvr = (acos_nvram_ioctl_t *)arg;
        ret = acos_knvram_getall(nvr->value, nvr->value_len);
        break;
    case ACOS_NVRAM_IOCTL_SET:
        nvr = (acos_nvram_ioctl_t *)arg;
        ret = acos_knvram_set(nvr->name, nvr->value);
        break;
    case ACOS_NVRAM_IOCTL_UNSET:
        nvr = (acos_nvram_ioctl_t *)arg;
        ret = acos_knvram_unset(nvr->name);
        break;
    case ACOS_NVRAM_IOCTL_COMMIT:
        nvr = (acos_nvram_ioctl_t *)arg;
        ret = acos_knvram_commit();
        break;
    case ACOS_NVRAM_IOCTL_CLEAR:
        nvr = (acos_nvram_ioctl_t *)arg;
        ret = acos_knvram_clear();
        break;
    case ACOS_NVRAM_IOCTL_INVAL:
        nvr = (acos_nvram_ioctl_t *)arg;
        ret = acos_knvram_inval();
        acos_nvram_init = -1;
        break;
    case ACOS_NVRAM_IOCTL_DEBUG:
        nvr = (acos_nvram_ioctl_t *)arg;
        acos_nvram_debug = *nvr->value - '0';
        ret = 0;
        break;
    default:
        break;
    }

    return ret;
}

static int acos_nvram_open(struct inode *inode, struct file *file)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    MOD_INC_USE_COUNT;
#else
    try_module_get(THIS_MODULE);
#endif
    return 0;
}

static int acos_nvram_release(struct inode *inode, struct file *file)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    MOD_DEC_USE_COUNT;
#else
    module_put(THIS_MODULE);
#endif
    return 0;
}

static struct file_operations acos_nvram_fops =
{
    owner:      THIS_MODULE,
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
    unlocked_ioctl: acos_nvram_ioctl,
#else
    ioctl:      acos_nvram_ioctl,
#endif
    open:       acos_nvram_open,
    release:    acos_nvram_release,
};

int __init dev_nvram_init(void)
{
#ifndef CONFIG_DEVFS_FS
    int r = 0;
#endif

#ifdef CONFIG_DEVFS_FS
    if (devfs_register_chrdev(acos_nvram_major, ACOS_NVRAM_DEVNAME,
                &acos_nvram_fops)) {
        printk(KERN_ERR "acos_nvram: unable to register character device\n");
        return -EIO;
    }
    devfs_handle = devfs_register(NULL, ACOS_NVRAM_DEVNAME,
            DEVFS_FL_DEFAULT, acos_nvram_major, 0,
            S_IFCHR | S_IRUGO | S_IWUGO, &acos_nvram_fops, NULL);
#else
    r = register_chrdev(acos_nvram_major, ACOS_NVRAM_DEVNAME,
            &acos_nvram_fops);
    if (r < 0) {
        printk(KERN_ERR "acos_nvram: unable to register character device\n");
        return r;
    }
    if (acos_nvram_major == 0) {
        acos_nvram_major = r;
        printk(KERN_DEBUG "acos_nvram: got dynamic major %d\n", r);
    }
#endif

    /* Initialize hash table lock */
    spin_lock_init(&acos_nvram_lock);

    /* Initialize commit semaphore */
    init_MUTEX(&acos_nvram_sem);

    acos_nvram_init = acos_knvram_init();

    return 0;
}

static void dev_nvram_exit(void)
{
    acos_knvram_exit();

#ifdef CONFIG_DEVFS_FS
    devfs_unregister_chrdev(acos_nvram_major, ACOS_NVRAM_DEVNAME);
    devfs_unregister(devfs_handle);
#else
    unregister_chrdev(acos_nvram_major, ACOS_NVRAM_DEVNAME);
#endif
}

/******************************************************************************
 * Exported Functions
 *****************************************************************************/
#ifdef ACOS_KERNEL_NVRAM
char *nvram_get(const char *name)
{
    return acos_knvram_get(name);
}
EXPORT_SYMBOL(nvram_get);

int nvram_set(const char *name, const char *value)
{
    return acos_knvram_set(name, value);
}
EXPORT_SYMBOL(nvram_set);

int nvram_unset(const char *name)
{
    return acos_knvram_unset(name);
}
EXPORT_SYMBOL(nvram_unset);

int nvram_match(const char *name, const char *match)
{
    return acos_knvram_match(name, match);
}
EXPORT_SYMBOL(nvram_match);

int nvram_commit(void)
{
    return acos_knvram_commit();
}
EXPORT_SYMBOL(nvram_commit);
#endif

late_initcall(dev_nvram_init);
module_exit(dev_nvram_exit);
