#include <assert.h>
#include <stdarg.h>
#include <stdio.h>
#include <syslog.h>
#include <sys/time.h>
#include <time.h>
#include <string.h>

#include "sys_debug.h"

#define CAM_COLOR_END   "\033[m"
#define CAM_LIGHT_BLUE  "\033[1;34m"
#define CAM_LIGHT_RED   "\033[1;31m"
#define CAM_YELLOW      "\033[1;33m"
#define CAM_LIGHT_GREEN "\033[1;32m"
#define CAM_WHITE       "\033[1;37m"
#define CAM_LIGHT_GRAY  "\033[0;37m"

#define NAME_LEN        10

typedef struct 
{
    unsigned char hour;
    unsigned char min;
    unsigned char sec;
    unsigned short msec;
} _timestamp;

// Default debug level : bit mask
// static unsigned char g_sys_log_lv = (SYS_LOG_LV_VBS|SYS_LOG_LV_INF|SYS_LOG_LV_DBG|SYS_LOG_LV_WRN|SYS_LOG_LV_ERR);
static unsigned char g_sys_log_lv = (SYS_LOG_LV_ERR);

// Default enable log color
static unsigned char g_sys_log_color = 1;

// Default disable time stamp in log
static unsigned char g_sys_log_timestamp = 0;

// Default tag
static unsigned char g_sys_log_enable = 0;

// Default tag
static char g_sys_log_tag[NAME_LEN]="nvram_app";

// Output buffer for bt_trace & bt_log_primitive
#define FMT_BUF_SIZE    384

static void _get_timestamp(_timestamp *ts)
{
    time_t rawtime;
    struct tm* timeinfo;
    struct timeval tv;

    if (ts != NULL)
    {
        time(&rawtime);
        timeinfo = localtime(&rawtime);
        gettimeofday(&tv, NULL);

        ts->hour = (unsigned char) timeinfo->tm_hour;
        ts->min = (unsigned char) timeinfo->tm_min;
        ts->sec = (unsigned char) timeinfo->tm_sec;
        ts->msec = (unsigned short) (tv.tv_usec / 1000);
    }
}

static void SYS_Log_Impl(unsigned char lv, const char *fmt, va_list args, unsigned char color)
{

    char fmt_buf[FMT_BUF_SIZE] = {0};
    char *buf_ptr, *newline, *replace;
    int offset = 0, buf_remain = 0;

    _timestamp ts = {0};
    va_list tmp_args;

    if (g_sys_log_enable)
    {
        openlog(g_sys_log_tag, 0, 0);
        va_copy(tmp_args, args);
        vsyslog(LOG_DEBUG, fmt, tmp_args);
        closelog();
    }

    if ((lv & g_sys_log_lv))
    {        
        buf_ptr = fmt_buf;
        buf_remain = FMT_BUF_SIZE - 1;

        if (g_sys_log_timestamp)
        {
            _get_timestamp(&ts);
            snprintf(buf_ptr, buf_remain, "%02d:%02d:%02d.%03d ", ts.hour, ts.min, ts.sec, ts.msec);

            offset = strlen(fmt_buf);
            buf_ptr += offset;
            buf_remain -= offset;
        }

        if (g_sys_log_color)
        {
            // Apply color prefix.
            switch (lv)
            {
                case SYS_LOG_LV_ERR:
                    snprintf(buf_ptr, buf_remain, "%s", CAM_LIGHT_RED);
                    break;
                case SYS_LOG_LV_WRN:
                    snprintf(buf_ptr, buf_remain, "%s", CAM_YELLOW);
                    break;
                case SYS_LOG_LV_DBG:
                    snprintf(buf_ptr, buf_remain, "%s", CAM_LIGHT_GREEN);
                    break;
                case SYS_LOG_LV_INF:
                    snprintf(buf_ptr, buf_remain, "%s", CAM_WHITE);
                    break;
                default: // LOG_LV_VBS
                    snprintf(buf_ptr, buf_remain, "%s", CAM_LIGHT_GRAY);
            }

            offset = strlen(fmt_buf);
            buf_ptr = fmt_buf + offset;
            buf_remain -= offset;
        }

        // Apply fmt string.
        snprintf(buf_ptr, buf_remain, "%s", fmt);

        // Replacing newline with space character.
        newline = strchr(fmt_buf, '\n');
        replace = NULL;
        while (newline != NULL)
        {
            replace = newline;
            newline = strchr(newline, '\n');
            *replace = ' ';
        }

        offset = strlen(fmt_buf);
        buf_ptr = fmt_buf + offset;
        buf_remain -= offset;

        if (replace == NULL)
        {
            if (g_sys_log_color)
            {
                // Apply color postfix
                snprintf(buf_ptr, buf_remain, "%s", CAM_COLOR_END);
            }
            else
            {
                snprintf(buf_ptr, buf_remain, "%s", "");
            }        
        }
        else
        {
            if (g_sys_log_color)
            {
                // Apply color postfix
                snprintf(buf_ptr, buf_remain, "%s\n", CAM_COLOR_END);
            }
            else
            {
                snprintf(buf_ptr, buf_remain, "\n");
            }
        }

        va_copy(tmp_args, args);
        vprintf(fmt_buf, tmp_args);
        va_end(tmp_args);
    }
}

void SYS_Log(unsigned char lv, const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    SYS_Log_Impl(lv, fmt, args, 1);
    va_end(args);
}

void SYS_Log_SetFlag(const char *str, unsigned char flags)
{
    g_sys_log_lv = flags & SYS_LOG_MASK;

    strcpy(g_sys_log_tag, str);
    
    g_sys_log_color = (flags & SYS_LOG_FLAG_COLOR) ? 1 : 0;
    g_sys_log_timestamp= (flags & SYS_LOG_FLAG_TIMESTAMP) ? 1 : 0;
}

