﻿/****************************************************************************
 * Copyright (c) 2016 Wi-Fi Alliance.  All Rights Reserved

 * Permission to use, copy, modify, and/or distribute this software for any purpose with or
 * without fee is hereby granted, provided that the above copyright notice and this permission
 * notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 *  * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 *
 *
 * Copyright  (C) 2019-2020  MediaTek Inc. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
******************************************************************************/

/*
 * File: mtk_hostapd.c
 *
 *
 * To simplify the current work and avoid any GPL licenses, the functions mostly
 *   1.invoke shell commands by calling linux system call, system("<commands>").
 *   2.write hostapd.conf, then ubus call hostapd config_add <hostapd.con> to apply new config
 *
 *
*/


#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <linux/socket.h>
#include <linux/wireless.h>
#include <poll.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <stdarg.h>
#include <time.h>

#include "mtk_hostapd.h"
#include "wfa_types.h"
#include "wfa_main.h"
#include "wfa_types.h"
#include "mtk_ap.h"
#include "mtk_parse.h"
#include "wfa_sock.h"
#include "wfa_debug.h"
#include "mtk_wmode.h"

#define MACSTR "%02x:%02x:%02x:%02x:%02x:%02x"
#define MAC2STR(a) (a)[0], (a)[1], (a)[2], (a)[3], (a)[4], (a)[5]
#define ETC_CONFIG_WIRELESS	"/etc/config/wireless"

char gCmdStr[WFA_CMD_STR_SZ];
intf_desc_t *cmt_intfs[3] = {NULL, NULL, NULL};


str_to_str_long_tbl_t HOSTAPD_SAE_PK_KeyPair_tbl[] = {
	{"saepk1.pem",
	 "MHcCAQEEIDNNDttjdQmLVyr1DOrWiapbMt15LDn4hnMLIXrBLAN+oAoGCCqGSM49AwEHoUQDQgAENAv4e3IlpYkqQjc/9KM4O4Athh6iY25wlT8Gdg+EhR7yoMR03nHri6QaaLogXxTsa9qGyXj1K9G8DEOyHMQCbg=="},
	{"saepk2.pem",
	 "MHcCAQEEIF4LGkE30VdIeJe1ZVOo3TmkvT9RKRx30+yOx/9nhQY9oAoGCCqGSM49AwEHoUQDQgAE3PNzZH4m41vT5q6W7p5Q6B9owz5MHLwCUnpK84YRTVDLKKZXOPYxSHRh/O5Kz0OnVeOq1QfjEZRhNH79XhHCQQ=="},
	{"saepkP256.pem",
	 "MHcCAQEEIAJIGlfnteonDb7rQyP/SGQjwzrZAnfrXIm4280VWajYoAoGCCqGSM49AwEHoUQDQgAEeRkstKQV+FSAMqBayqFknn2nAQsdsh/MhdX6tiHOTAFin/sUMFRMyspPtIu7YvlKdsexhI0jPVhaYZn1jKWhZg=="},
	{"saepkP384.pem",
	 "MIGkAgEBBDB7iMoR2se0sWriXYCEsiLd8WFEblxWlCqb5kD7JgZfQjjylGwqOgIE7JShOOjE0Z2gBwYFK4EEACKhZANiAATntlmb7rlUopsaA/w5Uhut9jLlcY2sJdT6IzCdQ8uzuxk9Fgh+dwS25pd+lWC91rQ7kyjfZRpoePhwQasnjGRAl6rH2VWI/XtI5Q9iFXbhEaWEdKzWjetd6B5OPWy/BQg="},
	{"saepkP521.pem",
	 "MIHcAgEBBEIBuNKSnOQY5ZVdBgWiXcL1Gr/W+VCw69nOte1gT4sqdVeV3grCl5HJxogVG2LFdtnEDLJrs0AtFoFN9nWnIuMu+ZWgBwYFK4EEACOhgYkDgYYABADuDQkFO2102xXwNnoGpBU+13kNuxZ/gwy8+G0UG75h6iiTqNWRaQIpSWgTmPNER7Ubb7etyXaoOTnsq4v4f9m8wgDt2LMZptHvUkHCq522rRK43ITmCayelbHWY1FhhAE1ETXRItSV8nLymjliEtjdfP45dsr25ySlkSaVCBNUFrAtfw=="}
};

typeDUT_t hostapd_dut_tbl[] = {
	{0, "NO_USED_STRING", NULL, NULL, NULL},
	{WFA_GET_VERSION_TLV, "ca_get_version", parse_ap_ca_version, mtk_ap_ca_version, mtk_ap_ca_version_resp},
	{WFA_DEVICE_GET_INFO_TLV, "device_get_info", parse_device_get_info, mtk_device_get_info, mtk_device_get_info_resp},
	{WFA_AP_CA_VERSION_TLV, "ap_ca_version", parse_ap_ca_version, mtk_ap_ca_version, mtk_ap_ca_version_resp},
	{WFA_AP_CONFIG_COMMIT_TLV, "ap_config_commit", parse_ap_config_commit, hostapd_ap_config_commit,
	 mtk_ap_config_commit_resp},
	{WFA_AP_DEAUTH_STA_TLV, "ap_deauth_sta", parse_ap_deauth_sta, hostapd_ap_deauth_sta, mtk_ap_deauth_sta_resp},
	{WFA_AP_GET_MAC_ADDRESS_TLV, "ap_get_mac_address", parse_ap_get_mac_address, hostapd_ap_get_mac_address,
	 mtk_ap_get_mac_address_resp},
	{WFA_AP_RESET_DEFAULT_TLV, "ap_reset_default", parse_ap_reset_default, hostapd_ap_reset_default,
	 mtk_ap_reset_default_resp},
	{WFA_AP_SET_RFEATURE_TLV, "ap_set_rfeature", parse_ap_set_rfeature, hostapd_ap_set_rfeature,
	 mtk_ap_set_rfeature_resp},
	{WFA_AP_SET_WIRELESS_TLV, "ap_set_wireless", parse_ap_set_wireless, hostapd_ap_set_wireless,
	 mtk_ap_set_wireless_resp},
	{WFA_AP_SET_SECURITY_TLV, "ap_set_security", parse_ap_set_security, hostapd_ap_set_security,
	 mtk_ap_set_security_resp},
	{WFA_AP_SET_RADIUS_TLV, "ap_set_radius", parse_ap_set_radius, hostapd_ap_set_radius, mtk_ap_set_radius_resp},
	{WFA_AP_IGNORE_CAPI_TLV, "AccessPoint", parse_ap_reset_default, hostapd_ap_reset_default,
	 mtk_ap_reset_default_resp},
	{WFA_AP_SET_11N_WIRELESS_TLV, "ap_set_11n_wireless", parse_ap_set_wireless, hostapd_ap_set_wireless,
	 mtk_ap_set_wireless_resp},
	{WFA_AP_SET_PMF_TLV, "ap_set_pmf", parse_ap_set_pmf, hostapd_ap_set_pmf, mtk_ap_set_pmf_resp},
	{WFA_AP_SET_APQOS_TLV, "ap_set_apqos", parse_ap_set_apqos, hostapd_ap_set_apqos, mtk_ap_set_apqos_resp},
	{WFA_AP_SET_STAQOS_TLV, "ap_set_staqos", parse_ap_set_staqos, hostapd_ap_set_staqos, mtk_ap_set_staqos_resp},
	{WFA_DEV_SEND_FRAME_TLV, "dev_send_frame", parse_dev_send_frame, hostapd_dev_send_frame, mtk_dev_send_frame_resp},
	{WFA_TRAFFIC_AGENT_RESET_TLV, "traffic_agent_reset", parse_traffic_agent_reset, mtk_traffic_agent_reset,
	 mtk_traffic_agent_reset_resp},
	{WFA_AP_SET_11H_TLV, "ap_set_11h", parse_ap_set_11h, mtk_ap_set_11h, mtk_ap_set_11h_resp},
	{WFA_AP_SET_11D_TLV, "ap_set_11d", parse_ap_set_11d, hostapd_ap_set_11d, mtk_ap_set_11d_resp},
	{WFA_AP_SET_HS2_TLV, "ap_set_hs2", parse_ap_set_hs2, hostapd_ap_set_hs2, mtk_ap_set_hs2_resp},
	{WFA_AP_SEND_ADDBA_REQ_TLV, "ap_send_addba_req", parse_ap_send_addba_req,
	 mtk_ap_send_addba_req, mtk_ap_send_addba_req_resp},
	{WFA_DEV_EXEC_ACTION_TLV, "dev_exec_action", parse_dev_exec_action, hostapd_dev_exec_action,
	 mtk_dev_exec_action_resp},
/* TODO: other capi cmd */
#if 0
	{WFA_AP_SEND_BCNRPT_REQ_TLV, "ap_send_bcnrpt_req", parse_ap_send_bcnrpt_req, mtk_ap_send_bcnrpt_req,
	 mtk_ap_send_bcnrpt_req_resp},
	{WFA_AP_SEND_BSSTRANS_MGMT_REQ_TLV, "ap_send_bsstrans_mgmt_req", parse_ap_send_bsstrans_mgmt_req,
	 mtk_ap_send_bsstrans_mgmt_req, mtk_ap_send_bsstrans_mgmt_req_resp},
	{WFA_AP_SEND_LINK_MEA_REQ_TLV, "ap_send_link_mea_req", parse_ap_send_link_mea_req, mtk_ap_send_link_mea_req,
	 mtk_ap_send_link_mea_req_resp},
	{WFA_AP_SEND_TSMRPT_REQ_TLV, "ap_send_tsmrpt_req", parse_ap_send_tsmrpt_req, mtk_ap_send_tsmrpt_req,
	 mtk_ap_send_tsmrpt_req_resp},


	{WFA_AP_SET_RRM_TLV, "ap_set_rrm", parse_ap_set_rrm, mtk_ap_set_rrm, mtk_ap_set_rrm_resp},

	{WFA_DEV_CONFIGURE_IE_TLV, "dev_configure_ie", parse_dev_configure_ie, mtk_dev_configure_ie,
	 mtk_dev_configure_ie_resp},
	{WFA_TRAFFIC_SEND_PING_TLV, "traffic_send_ping", parse_traffic_send_ping, mtk_traffic_send_ping,
	 mtk_traffic_send_ping_resp},
	{WFA_TRAFFIC_STOP_PING_TLV, "traffic_stop_ping", parse_traffic_stop_ping, mtk_traffic_stop_ping,
	 mtk_traffic_stop_ping_resp},

#endif
	{WFA_AP_GET_PARAMETER_TLV, "ap_get_parameter", parse_ap_get_parameter, hostapd_ap_get_parameter,
		 mtk_ap_get_parameter_resp},
	{WFA_AP_SET_QOS_TLV, "ap_set_qos", parse_ap_set_qos, mtk_ap_set_qos, mtk_ap_set_qos_resp},
	{-1, "", NULL, NULL, NULL},
};

int hostapd_str_to_hex(char *out_buf, int out_size, char *in_str, int in_len)
{
	int index = 0, offset = 0;
	char *fmt = "%02x";

	DPRINT_INFO(WFA_OUT, "input str %s\n", in_str);
	for (index = 0; index < in_len; index++)
		offset += snprintf(&out_buf[index << 1], out_size - offset - 1, fmt, in_str[index]);

	DPRINT_INFO(WFA_OUT, "out hex str: %s\n", out_buf);
	return offset;
}

int hostapd_parameter_to_tlv(char *out_buf, int out_size, char *in_str, int in_len, enum ENUM_TLV_TYPE type,
			     intf_desc_t *def_intf)
{
	int offset = 0;
	char hex_buf[out_size], chan_cnt = 0;
	char *fmt = "%02x";
	int ret_len = 0;

	memset(hex_buf, 0, sizeof(hex_buf));

	switch (type) {
	case TLV_TYPE_SSID:
		ret_len = hostapd_str_to_hex(hex_buf, sizeof(hex_buf), in_str, strlen(in_str));
		if (ret_len >= out_size)
			DPRINT_INFO(WFA_OUT, "SSID too long\n");

		offset = 0;
		/* t */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, TLV_TYPE_SSID);
		/* l */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, in_len);
		/* v */
		offset += snprintf(out_buf + offset, out_size - offset - 1, "%s", hex_buf);
		DPRINT_INFO(WFA_OUT, "SSID tlv: %s\n", out_buf);
		break;

	case TLV_TYPE_BECON_REPT_INF:
		DPRINT_INFO(WFA_OUT, "RptCond : %s\n", in_str);
		/* t */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, TLV_TYPE_BECON_REPT_INF);
		/* l */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, 2);
		/* v */
		if (!strncmp(in_str, "0", min(in_len, 2)))
			offset += snprintf(out_buf + offset, out_size - offset - 1, "%s", "0000");
		DPRINT_INFO(WFA_OUT, "RptCond tlv: %s\n", out_buf);
		break;

	case TLV_TYPE_BECON_REPT_DET:
		DPRINT_INFO(WFA_OUT, "RptDet : %s\n", in_str);
		/* t */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, TLV_TYPE_BECON_REPT_DET);
		/* l */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, 1);
		/* v */
		if (!strncmp(in_str, "1", min(in_len, 2)))
			offset += snprintf(out_buf + offset, out_size - offset - 1, "%s", "01");
		else if (!strncmp(in_str, "0", min(in_len, 2)))
			offset += snprintf(out_buf + offset, out_size - offset - 1, "%s", "00");
		DPRINT_INFO(WFA_OUT, "RptDet tlv: %s\n", out_buf);
		break;

	case TLV_TYPE_REQ_INFO:
		if (!in_len)
			return 0;

		DPRINT_INFO(WFA_OUT, "ReqInfo : %s\n", in_str);
		if ((strncmp(in_str, "0_54_221", strlen("0_54_221")) == 0) ||
			(strncmp(in_str, "0_221", strlen("0_221"))) == 0) {
			/* t */
			offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, TLV_TYPE_REQ_INFO);
			/* l */
			offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, 13);
			/* v */
			offset += snprintf(out_buf + offset, out_size - offset - 1, "%s", "0001322d3d7fbfc0c3304636dd");
		}
		DPRINT_INFO(WFA_OUT, "ReqInfo tlv: %s\n", out_buf);
		break;

	case TLV_TYPE_AP_CHAN_RPT:
		if (!in_len)
			return 0;
		DPRINT_INFO(WFA_OUT, "APChanRpt : %s\n", in_str);

		if (strstr(in_str, "1")) {
			offset += snprintf(hex_buf + offset, out_size - offset - 1, fmt, 1);
			chan_cnt++;
		}
		if (strstr(in_str, "6")) {
			chan_cnt++;
			offset += snprintf(hex_buf + offset, out_size - offset - 1, fmt, 6);
		}
		if (strstr(in_str, "36")) {
			chan_cnt++;
			offset += snprintf(hex_buf + offset, out_size - offset - 1, fmt, 36);
		}
		if (strstr(in_str, "48")) {
			chan_cnt++;
			offset += snprintf(hex_buf + offset, out_size - offset - 1, fmt, 48);
		}

		offset = 0;
		/* t */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, TLV_TYPE_AP_CHAN_RPT);
		/* l */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, chan_cnt + 1);
		/* v */
		/* opclass */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, def_intf->frame_opclass);
		/* channel list */
		offset += snprintf(out_buf + offset, out_size - offset - 1, "%s", hex_buf);
		DPRINT_INFO(WFA_OUT, "APChanRpt tlv: %s\n", out_buf);
		break;

	case TLV_TYPE_BECON_REPT_IND_REQ:
		DPRINT_INFO(WFA_OUT, "LastBeaconRptIndication : %s\n", in_str);
		/* t */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, TLV_TYPE_BECON_REPT_IND_REQ);
		/* l */
		offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, 1);
		/* v */
		if (strstr(in_str, "1"))
			offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, 1);
		else if (strstr(in_str, "0"))
			offset += snprintf(out_buf + offset, out_size - offset - 1, fmt, 0);
		DPRINT_INFO(WFA_OUT, "LastBeaconRptIndication tlv: %s\n", out_buf);
		break;

	default:
		break;
	}

	return offset;
}

void hostapd_handle_post_cmd(mtk_ap_buf_t *mtk_ap_buf)
{
	handle_post_cmd(mtk_ap_buf);
	sleep(1);
}


char *hostapd_key_mgt_2_string(enum ENUM_KEY_MGNT_TYPE key_mgt)
{
	switch (key_mgt) {
	case KEY_MGNT_TYPE_OPEN:
		return "KEY_MGNT_TYPE_OPEN";
	case KEY_MGNT_TYPE_WPA_PSK:
		return "KEY_MGNT_TYPE_WPA_PSK";
	case KEY_MGNT_TYPE_WPA2_PSK:
		return "KEY_MGNT_TYPE_WPA2_PSK";
	case KEY_MGNT_TYPE_WPA_EAP:
		return "KEY_MGNT_TYPE_WPA_EAP";
	case KEY_MGNT_TYPE_WPA2_EAP:
		return "KEY_MGNT_TYPE_WPA2_EAP";
	case KEY_MGNT_TYPE_WPA2_EAP_MIXED:
		return "KEY_MGNT_TYPE_WPA2_EAP_MIXED";
	case KEY_MGNT_TYPE_WPA2_PSK_MIXED:
		return "KEY_MGNT_TYPE_WPA2_PSK_MIXED";
	case KEY_MGNT_TYPE_SUITEB:
		return "KEY_MGNT_TYPE_SUITEB";
	case KEY_MGNT_TYPE_WPA2_SAE:
		return "KEY_MGNT_TYPE_WPA2_SAE";
	case KEY_MGNT_TYPE_WPA2_PSK_SAE:
		return "KEY_MGNT_TYPE_WPA2_PSK_SAE";
	case KEY_MGNT_TYPE_WPA2_OWE:
		return "KEY_MGNT_TYPE_WPA2_OWE";
	case KEY_MGNT_TYPE_OSEN:
		return "KEY_MGNT_TYPE_OSEN";
	case KEY_MGNT_TYPE_WPA2_ENT_OSEN:
		return "KEY_MGNT_TYPE_WPA2_ENT_OSEN";
	default:
		return "KEY_MGNT_TYPE_UNKNOWN";
	}
}

char *hostapd_enc_type_2_string(enum ENUM_ENCP_TYPE enc_type)
{
	switch (enc_type) {
	case ENCP_TYPE_NONE:
		return "ENCP_TYPE_NONE";
	case ENCP_TYPE_WEP:
		return "ENCP_TYPE_WEP";
	case ENCP_TYPE_TKIP:
		return "ENCP_TYPE_TKIP";
	case ENCP_TYPE_CCMP:
		return "ENCP_TYPE_CCMP";
	case ENCP_TYPE_CCMP_256:
		return "ENCP_TYPE_CCMP_256";
	case ENCP_TYPE_GCMP_128:
		return "ENCP_TYPE_GCMP_128";
	case ENCP_TYPE_GCMP_256:
		return "ENCP_TYPE_GCMP_256";
	case ENCP_TYPE_CCMP_TKIP:
		return "ENCP_TYPE_CCMP_TKIP";
	default:
		return "ENCP_TYPE_UNKNOWN";
	}
}


char *hostapd_apmode_2_string(enum ENUM_AP_MODE ap_mode)
{
	switch (ap_mode) {
	case AP_MODE_11b:
		return "AP_MODE_11b";
	case AP_MODE_11bg:
		return "AP_MODE_11bg";
	case AP_MODE_11bgn:
		return "AP_MODE_11bgn";
	case AP_MODE_11a:
		return "AP_MODE_11a";
	case AP_MODE_11g:
		return "AP_MODE_11g";
	case AP_MODE_11na:
		return "AP_MODE_11na";
	case AP_MODE_11ng:
		return "AP_MODE_11ng";
	case AP_MODE_11ac:
		return "AP_MODE_11ac";
	case AP_MODE_11ad:
		return "AP_MODE_11ad";
	case AP_MODE_11ax:
		return "AP_MODE_11ax";
	default:
		return "AP_MODE_NONE";
	}
}

void hostapd_ch_bw_get_freq(int ch, enum ENUM_CHANNEL_WIDTH bw, int *freq,
	int *cent_freq, int *sec_offset)
{
	if (!freq || !cent_freq)
		return;

	switch (ch) {
	case 36:
		*freq = 5180;
		DPRINT_INFO(WFA_OUT, "freq %d\n", *freq);
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5190;
			DPRINT_INFO(WFA_OUT, "cent_freq %d\n", *cent_freq);
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5210;
			DPRINT_INFO(WFA_OUT, "cent_freq %d\n", *cent_freq);
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			DPRINT_INFO(WFA_OUT, "cent_freq %d\n", *cent_freq);
			break;
		default:
			*cent_freq = 5180;
			DPRINT_INFO(WFA_OUT, "cent_freq %d\n", *cent_freq);
			break;
		}
		break;
	case 40:
		*freq = 5200;
		DPRINT_INFO(WFA_OUT, "freq %d\n", *freq);
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5190;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5210;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5200;
			break;
		}
		break;
	case 44:
		*freq = 5220;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5230;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5210;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5220;
			break;
		}
		break;
	case 48:
		*freq = 5240;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5230;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5210;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5240;
			break;
		}
		break;
	case 52:
		*freq = 5260;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5270;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5290;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5260;
			break;
		}
		break;
	case 56:
		*freq = 5280;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5270;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5290;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5260;
			break;
		}
		break;
	case 60:
		*freq = 5300;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5310;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5290;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5300;
			break;
		}
		break;
	case 64:
		*freq = 5320;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5310;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5290;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5250;
			break;
		default:
			*cent_freq = 5320;
			break;
		}
		break;
	case 100:
		*freq = 5500;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5510;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5530;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5500;
			break;
		}
		break;
	case 104:
		*freq = 5520;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5510;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5530;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5520;
			break;
		}
		break;
	case 108:
		*freq = 5540;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5550;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5530;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5540;
			break;
		}
		break;
	case 112:
		*freq = 5560;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5550;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5530;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5560;
			break;
		}
		break;
	case 116:
		*freq = 5580;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5590;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5610;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5580;
			break;
		}
		break;
	case 120:
		*freq = 5600;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5590;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5610;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5600;
			break;
		}
		break;
	case 124:
		*freq = 5620;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5630;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5610;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5620;
			break;
		}
		break;
	case 128:
		*freq = 5640;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5630;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5610;
			break;
		case CHANNEL_WIDTH_160:
			*cent_freq = 5570;
			break;
		default:
			*cent_freq = 5640;
			break;
		}
		break;
	case 132:
		*freq = 5660;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5670;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5690;
			break;
		default:
			*cent_freq = 5660;
			break;
		}
		break;
	case 136:
		*freq = 5680;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5670;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5690;
			break;
		default:
			*cent_freq = 5680;
			break;
		}
		break;
	case 140:
		*freq = 5700;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5710;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5690;
			break;
		default:
			*cent_freq = 5700;
			break;
		}
		break;
	case 144:
		*freq = 5720;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5710;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5690;
			break;
		default:
			*cent_freq = 5720;
			break;
		}
		break;
	case 149:
		*freq = 5745;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5755;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5775;
			break;
		default:
			*cent_freq = 5745;
			break;
		}
		break;
	case 153:
		*freq = 5765;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5755;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5775;
			break;
		default:
			*cent_freq = 5765;
			break;
		}
		break;
	case 157:
		*freq = 5785;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5795;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5775;
			break;
		default:
			*cent_freq = 5785;
			break;
		}
		break;
	case 161:
		*freq = 5805;
		switch (bw) {
		case CHANNEL_WIDTH_40:
			*cent_freq = 5795;
			break;
		case CHANNEL_WIDTH_80:
			*cent_freq = 5775;
			break;
		default:
			*cent_freq = 5805;
			break;
		}
		break;
	default:
		break;
	}

	if (bw == CHANNEL_WIDTH_80) {
		if (*freq - 10 == *cent_freq ||
			*freq + 30 == *cent_freq)
			*sec_offset = 1;
		else if (*freq + 10 == *cent_freq ||
			*freq - 30 == *cent_freq)
			*sec_offset = -1;
	} else if (bw == CHANNEL_WIDTH_40) {
		if (*freq < *cent_freq)
			*sec_offset = 1;
		else
			*sec_offset = -1;
	}
}


static char hostapd_ap_get_cent_ch(intf_desc_t *def_intf)
{
	char centCh = 0;

	if (def_intf->ch_width == CHANNEL_WIDTH_80) {
		DPRINT_INFO(WFA_OUT, "%s BW 80\n", __func__);
		if (def_intf->mode == WIFI_6G) {
			if (def_intf->channel >= 17 && def_intf->channel <= 29)
				centCh = 23;
			else if (def_intf->channel >= 33 && def_intf->channel <= 45)
				centCh = 39;
			else if (def_intf->channel >= 49 && def_intf->channel <= 61)
				centCh = 55;
			else if (def_intf->channel >= 65 && def_intf->channel <= 77)
				centCh = 71;
			else if (def_intf->channel >= 81 && def_intf->channel <= 93)
				centCh = 87;
			else if (def_intf->channel >= 97 && def_intf->channel <= 109)
				centCh = 103;
			else if (def_intf->channel >= 113 && def_intf->channel <= 125)
				centCh = 119;
			else if (def_intf->channel >= 129 && def_intf->channel <= 141)
				centCh = 135;
			else if (def_intf->channel >= 145 && def_intf->channel <= 157)
				centCh = 151;
			else if (def_intf->channel >= 161 && def_intf->channel <= 173)
				centCh = 167;
			else if (def_intf->channel >= 177 && def_intf->channel <= 189)
				centCh = 183;
			else if (def_intf->channel >= 193 && def_intf->channel <= 205)
				centCh = 199;
			else if (def_intf->channel >= 209 && def_intf->channel <= 221)
				centCh = 215;
			DPRINT_INFO(WFA_OUT, "%s 6g only, cent ch %d\n", __func__, centCh);
		} else if (def_intf->mode == WIFI_5G) {
			if (def_intf->channel >= 36 && def_intf->channel <= 48)
				centCh = 42;
			else if (def_intf->channel >= 52 && def_intf->channel <= 64)
				centCh = 58;
			else if (def_intf->channel >= 100 && def_intf->channel <= 112)
				centCh = 106;
			else if (def_intf->channel >= 116 && def_intf->channel <= 128)
				centCh = 122;
			else if (def_intf->channel >= 132 && def_intf->channel <= 144)
				centCh = 138;
			else if (def_intf->channel >= 149 && def_intf->channel <= 161)
				centCh = 155;
			DPRINT_INFO(WFA_OUT, "%s 5g, cent ch %d\n", __func__, centCh);
		}
	}

	return centCh;
}


char *hostapd_wifi_mode_2_string(wifi_mode mode)
{
	switch (mode) {
	case WIFI_2G:
		return "WIFI_2G";
	case WIFI_5G:
		return "WIFI_5G";
	case WIFI_6G:
		return "WIFI_6G";
	default:
		return "NORMAL";
	}
}

int hostapd_get_cmd_output(char *out_buf, size_t out_len, char *cmd)
{
	FILE *f = NULL;

	DPRINT_INFO(WFA_OUT, "run cmd ==> %s\n", cmd);
	system(cmd);

	f = fopen(HOSTAPD_TEMP_OUTPUT, "r");
	if (f == NULL) {
		DPRINT_ERR(WFA_ERR, "file open fail\n");
		return WFA_ERROR;
	}
	fseek(f, 0, SEEK_END);
	if (!ftell(f))
		return WFA_ERROR;
	fseek(f, 0, SEEK_SET);
	fgets(out_buf, out_len, f);
	fclose(f);
	return WFA_SUCCESS;
}

static void hostapd_cli_cmd(intf_desc_t *def_intf, char *buf)
{
	if (!def_intf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}
	DPRINT_INFO(WFA_OUT, "%s(), buf: %s\n", __func__, buf);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr), "hostapd_cli -i %s -p %s %s", def_intf->name, def_intf->ctrl_inf, buf);
	DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
	system(gCmdStr);
}

int hostapd_get_cmd_tag(char *capi_name)
{
	int cmd_tag = 0;

	if (!capi_name) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return WFA_ERROR;
	}

	while (hostapd_dut_tbl[cmd_tag].type != -1) {
		if (strcmp(hostapd_dut_tbl[cmd_tag].name, capi_name) == 0)
			return cmd_tag;
		cmd_tag++;
	}
	return 0;
}


int hostapd_get_option_from_etc_config_wireless(
	char *option, const char *interface)
{
	FILE *fh;
	char buff[1024];
	char intf_pref[INTF_NAME_LEN+1];
	char intf_idx[2];
	char buf1[IFNAMSIZ+1], buf2[IFNAMSIZ+1], buf3[IFNAMSIZ+1];
	char radio_name[INTF_NAME_LEN+1];
	int channel = -1, wband = W_BAND_INVALID;
	unsigned char found = 0;

	if (sscanf(interface, "%[a-z]%[0-9]", intf_pref, intf_idx) != 2) {
		DPRINT_INFO(WFA_OUT, "%s fail to get intf idx\n", __func__);
		return -1;
	}
	DPRINT_INFO(WFA_OUT, "%s intf_pref %s, intf_idx %s\n", __func__, intf_pref, intf_idx);

	/* Check if /etc/config/wireless is available */
	fh = fopen(ETC_CONFIG_WIRELESS, "r");

	/* Read each device line */
	while (fgets(buff, sizeof(buff), fh)) {
		/* Skip empty or almost empty lines. It seems that in some
		 * cases fgets return a line with only a newline.
		*/
		if ((buff[0] == '\0') || (buff[1] == '\0'))
			continue;

		if (l1_valid) {
			if (strstr(buff, option)) {
				if (sscanf(buff, "%s %s '%[0-9]'", buf1, buf2, buf3) != 3) {
					DPRINT_INFO(WFA_OUT, "%s fail to get %s\n", __func__, option);
					break;
				}
				if (strstr(option, "channel"))
					channel = atoi(buf3);
				if (strstr(option, "band")) {
					if (buf3[0] == '2')
						wband = W_BAND_2G;
					else if (buf3[0] == '5')
						wband = W_BAND_5G;
					else if (buf3[0] == '6')
						wband = W_BAND_6G;
					else
						wband = W_BAND_INVALID;
				}
			} else if (strstr(buff, "wifi-iface")) {
				if (sscanf(buff, "%s %s '%s'", buf1, buf2, buf3) != 3) {
					DPRINT_INFO(WFA_OUT, "%s fail to get wifi-iface\n", __func__);
					break;
				}
				if (strlen(buf3) > 0)
					buf3[strlen(buf3)-1] = '\0';
				if (strncmp(buf3, interface, INTF_NAME_LEN) == 0) {
					found = 1;
					break;
				}
			}
		} else {
			if (strstr(buff, "config wifi-device")) {
				snprintf(radio_name, INTF_NAME_LEN, "radio%s", intf_idx);
				if (!strstr(buff, radio_name))
					memset(radio_name, 0, sizeof(radio_name));
			} else if (strstr(buff, option)) {
				if (!strlen(radio_name))
					continue;
				if (sscanf(buff, "%s %s '%[0-9]'", buf1, buf2, buf3) != 3) {
					DPRINT_INFO(WFA_OUT, "%s fail to get channel/wand\n", __func__);
					break;
				}
				if (strstr(option, "band")) {
					found = 1;
					if (buf3[0] == '2')
						wband = W_BAND_2G;
					else if (buf3[0] == '5')
						wband = W_BAND_5G;
					else if (buf3[0] == '6')
						wband = W_BAND_6G;
					else {
						DPRINT_INFO(WFA_OUT, "%s wband invalid\n", __func__);
						wband = W_BAND_INVALID;
					}
					break;
				}

				if (strstr(option, "channel")) {
					found = 1;
					channel = atoi(buf3);
					break;
				}
			}
		}
	}

	fclose(fh);

	if (found) {
		if (strstr(option, "channel")) {
			DPRINT_INFO(WFA_OUT, "%s channel %d\n", __func__, channel);
			return channel;
		}

		if (strstr(option, "band")) {
			DPRINT_INFO(WFA_OUT, "%s band %d\n", __func__, wband);
			return wband;
		}
	}

	return -1;
}


int hostapd_wifi_interface_chan(int skfd, const char *interface)
{
	DPRINT_INFO(WFA_OUT, "%s get option channel\n", __func__);
	return hostapd_get_option_from_etc_config_wireless("channel", interface);
}


static unsigned int hostapd_wifi_interface_wband(int skfd, const char *interface)
{
	unsigned char wband = W_BAND_INVALID;

	if (!interface)
		return -1;
	DPRINT_INFO(WFA_OUT, "%s get option band\n", __func__);
	wband = hostapd_get_option_from_etc_config_wireless("band", interface);
	return wband;
}


int hostapd_init_profile_name_by_L1profile(int skfd, mtk_ap_buf_t *mtk_ap_buf)
{
	const char *PROFILE_PATH = "profile_path=";
	const char *MAIN_IFNAME = "main_ifname=";
	const int OFFSET = 7;	/* INDEX?_ */
	char card_names[MAX_CARD_NUM][10];
	char prof_names[MAX_CARD_NUM][MAX_INTF_NUM][PROF_NAME_LEN];
	char intf_names[MAX_CARD_NUM][MAX_INTF_NUM][INTF_NAME_LEN];
	int is_intf_up[MAX_CARD_NUM][MAX_INTF_NUM];
	char line[256], *value;
	int i, j, n, i_card = -1, n_prof[MAX_CARD_NUM], n_intf[MAX_CARD_NUM];
	FILE *file = NULL, *file_l2[MAX_CARD_NUM];
	char l3_arch = 0;

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	file = fopen(PROFILE_INF, "r");
	if (file == NULL) {
		printf("Error opening file: %s\n", PROFILE_INF);
		return WFA_ERROR;
	}
	printf("Reading L1 profile: %s\n", PROFILE_INF);

	while (fgets(line, sizeof(line), file) != NULL) {
		if (strlen(line) < OFFSET)
			continue;

		line[strlen(line) - 1] = '\0';		/* '\n' -> '\0' */
		value = line + OFFSET;

		if (line[OFFSET - 1] == '=') {
			if (i_card + 1 == MAX_CARD_NUM) {
				printf("Error: Max cards supported = %d, skipping card: %s\n", MAX_CARD_NUM, value);
				break;
			}

			i_card++;

			strncpy(card_names[i_card], value, 10);
			n_prof[i_card] = 0;
			n_intf[i_card] = 0;
		} else {
			if (strncmp(value, PROFILE_PATH, strlen(PROFILE_PATH)) == 0) {
				value += strlen(PROFILE_PATH);
				for (value = strtok(value, ";"); value != NULL; value = strtok(NULL, ";")) {
					if (n_prof[i_card] < MAX_INTF_NUM)
						strncpy(prof_names[i_card][n_prof[i_card]++], value, PROF_NAME_LEN);
					else
						printf("Error: Max intfs supported per card = %d, skipping: %s\n",
							MAX_INTF_NUM, value);
				}
			} else if (strncmp(value, MAIN_IFNAME, strlen(MAIN_IFNAME)) == 0) {
				value += strlen(MAIN_IFNAME);
				for (value = strtok(value, ";"); value != NULL; value = strtok(NULL, ";")) {
					if (n_intf[i_card] < MAX_INTF_NUM)
						strncpy(intf_names[i_card][n_intf[i_card]++], value, INTF_NAME_LEN);
					else
						printf("Error: Max intfs supported per card = %d, skipping: %s\n",
							MAX_INTF_NUM, value);
				}
			}
		}
	}


	for (i = 0; i <= i_card; i++) {
		if (n_intf[i] != n_prof[i]) {
			l3_arch = 1;
			printf("it's L3 profile arch\n");
			break;
		}
	}

	if (l3_arch) {
		char l2_profile0_path[] = "BN0_profile_path=";
		char l2_profile1_path[] = "BN1_profile_path=";
		char l2_profile2_path[] = "BN2_profile_path=";

		for (i = 0; i <= i_card; i++) {
			/* re-count n_prof and rename profile name */
			n_prof[i_card] = 0;
			n_intf[i_card] = 0;
			file_l2[i_card] = fopen(prof_names[i_card][0], "r");
			if (file_l2[i_card] == NULL) {
				printf("Error opening file: %s\n", prof_names[i_card][0]);
				if ((i_card - 1 >= 0) && file_l2[i_card-1]) {
					fclose(file_l2[i_card-1]);
					printf("close opening file: %s\n", prof_names[i_card-1][0]);
				}
				if ((i_card - 2 >= 0) && file_l2[i_card-2]) {
					fclose(file_l2[i_card-2]);
					printf("close opening file: %s\n", prof_names[i_card-2][0]);
				}
				return WFA_ERROR;
			}
			printf("Reading Card %d L2 profile: %s\n", i_card, prof_names[i_card][0]);

			while (fgets(line, sizeof(line), file_l2[i_card]) != NULL) {
				/* '\n' -> '\0' */
				line[strlen(line) - 1] = '\0';
				value = line;

				if (n_prof[i_card] >= MAX_INTF_NUM)
					break;

				if ((strncmp(line, l2_profile0_path, strlen(l2_profile0_path)) == 0) ||
					(strncmp(line, l2_profile1_path, strlen(l2_profile1_path)) == 0) ||
					(strncmp(line, l2_profile2_path, strlen(l2_profile2_path)) == 0)) {
					value += strlen(l2_profile0_path);
					strncpy(prof_names[i_card][n_prof[i_card]++], value, PROF_NAME_LEN);
					printf("Card %d L3 profile: %s\n", i_card, prof_names[i_card][n_prof[i_card]-1]);
				}
			}
			n_intf[i_card] = n_prof[i_card];
			printf("Card %d interface num %d\n", i_card, n_intf[i_card]);
			fclose(file_l2[i_card]);
		}
	}


	n = 0;
	for (i = 0; i <= i_card; i++) {
		for (j = 0; j < n_intf[i]; j++) {
			if (is_interface_up(skfd, intf_names[i][j])) {
				is_intf_up[i][j] = 1;
				n++;
			} else
				is_intf_up[i][j] = 0;
		}
	}

	mtk_ap_buf->profile_num = n;
	mtk_ap_buf->profile_names_list = calloc(n, sizeof(intf_profile_t));

	if (mtk_ap_buf->profile_num > 3)
		printf("Warning: Max 3 profiles supported, there may be error in sigma profile names\n");

	n = 0;
	for (i = 0; i <= i_card; i++) {
		int n_intf_up = 0;

		printf("card = %s\n", card_names[i]);

		for (j = 0; j < n_intf[i]; j++)
			n_intf_up += is_intf_up[i][j];

		for (j = 0; j < n_intf[i]; j++) {
			intf_profile_t *profile_names;
			char prof_dut[PROF_NAME_LEN], prof_tb[PROF_NAME_LEN];

			if (!is_intf_up[i][j]) {
				printf("  intf = %s, down\n", intf_names[i][j]);
				continue;
			}

			profile_names = &mtk_ap_buf->profile_names_list[n];

			strcpy(profile_names->name, intf_names[i][j]);
			strcpy(profile_names->profile, prof_names[i][j]);
			snprintf(profile_names->profile_bak, PROF_NAME_LEN, "%s.bak", prof_names[i][j]);
			snprintf(profile_names->profile_cmt, PROF_NAME_LEN, "%s.cmt", prof_names[i][j]);

			/* === Sigma DUT Profile ===
			 * single-band card:
			 *	card 1 => /etc/wireless/mediatek/wifi_cert.1.dat
			 *	card 2 => /etc/wireless/mediatek/wifi_cert.2.dat
			 * dual-band card:
			 *	intf 1 => /etc/wireless/sigma_test/wifi_cert_b0.dat
			 *	intf 2 => /etc/wireless/sigma_test/wifi_cert_b1.dat
			 * single-band card, tri-band device:
			 *	card x => /etc/wireless/sigma_test/wifi_cert_b2.dat
			 */

			if (n_intf_up == 1) { /* single-band card */
				if (mtk_ap_buf->profile_num == 3) { /* tri-band device */
					snprintf(prof_dut, PROF_NAME_LEN, "/etc/wireless/sigma_test/wifi_cert_b2.dat");
					strcpy(prof_tb, prof_dut);
				} else {
					snprintf(prof_dut, PROF_NAME_LEN,
						"/etc/wireless/mediatek/wifi_cert.%d.dat", i + 1);
					if (access(prof_dut, F_OK) == -1)
						snprintf(prof_dut, PROF_NAME_LEN,
							"/etc/wireless/sigma_test/wifi_cert_b%d.dat", i);

					mtk_ap_buf->tb_profile_exist = 1;
					snprintf(prof_tb, PROF_NAME_LEN,
						"/etc/wireless/mediatek/wifi_cert_tb.%d.dat", i + 1);
					if (access(prof_tb, F_OK) == -1) {
						mtk_ap_buf->tb_profile_exist = 0;
						strcpy(prof_tb, prof_dut);
					}
				}
			} else { /* dual-band card */
				snprintf(prof_dut, PROF_NAME_LEN, "/etc/wireless/sigma_test/wifi_cert_b%d.dat", j);
				strcpy(prof_tb, prof_dut);
			}

			strcpy(profile_names->sigma_dut_profile, prof_dut);
			strcpy(profile_names->sigma_tb_profile, prof_tb);

			printf("  intf = %s\n", profile_names->name);
			printf("    profile = %s\n", profile_names->profile);
			printf("    profile_bak = %s\n", profile_names->profile_bak);
			printf("    profile_cmt = %s\n", profile_names->profile_cmt);
			printf("    sigma_dut_profile = %s\n", profile_names->sigma_dut_profile);
			printf("    sigma_tb_profile = %s\n", profile_names->sigma_tb_profile);

			if (strcmp(mtk_ap_buf->application, "jedi_hostapd") == 0) {
				char intf0[INTF_NAME_LEN], intf1[INTF_NAME_LEN];

				strcpy(intf0, intf_names[i][j]);
				strcpy(intf1, intf0);
				intf1[strlen(intf1) - 1] = '1';

				snprintf(profile_names->jedi_hostapd_profile, PROF_NAME_LEN,
					"/etc/hostapd_%s.conf", intf0);
				snprintf(profile_names->jedi_hostapd_profile_bak, PROF_NAME_LEN,
					"/etc/hostapd_%s.conf.bak", intf0);
				snprintf(profile_names->jedi_hostapd_sigma_profile, PROF_NAME_LEN,
					"/etc/hostapd_%s_sigma.conf", intf0);
				snprintf(profile_names->jedi_hostapd_mbss_profile, PROF_NAME_LEN,
					"/etc/hostapd_%s.conf", intf1);
				snprintf(profile_names->jedi_hostapd_mbss_sigma_profile, PROF_NAME_LEN,
					"/etc/hostapd_%s_sigma.conf", intf1);

				printf("    jedi_hostapd_profile = %s\n", profile_names->jedi_hostapd_profile);
				printf("    jedi_hostapd_profile_bak = %s\n", profile_names->jedi_hostapd_profile_bak);
				printf("    jedi_hostapd_sigma_profile = %s\n", profile_names->jedi_hostapd_sigma_profile);
				printf("    jedi_hostapd_mbss_profile = %s\n", profile_names->jedi_hostapd_mbss_profile);
				printf("    jedi_hostapd_mbss_sigma_profile = %s\n", profile_names->jedi_hostapd_mbss_sigma_profile);
			}

			n++;
		}
	}

	fclose(file);
	return WFA_SUCCESS;
}


int hostapd_init_profile_name(int skfd, mtk_ap_buf_t *mtk_ap_buf)
{
	intf_profile_t *profile_name = NULL;
	int offset = 0;
	unsigned char i, n = 0;

	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return -1;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr),
		"iwinfo|grep ESSID|awk '{print $1}'|awk '{print length,$0}'|sort -n|awk '{print $2}'>%s"
		, HOSTAPD_WIRELESS_INTF_FILE);
	system(gCmdStr);

	if (l1_valid) {
		hostapd_init_profile_name_by_L1profile(skfd, mtk_ap_buf);

		/* check profile path first */
		/* L1profile valid,
		*  1.restore dat profile name in jedi_hostapd_profile
		*  2.store cert dat profile in sigma_dut_profile
		*  3.store hostapd.conf profile in profile
		*/
		DPRINT_INFO(WFA_OUT, "%s() resave profile for hostapd\n", __func__);
		for (i = 0; i < mtk_ap_buf->profile_num; i++) {
			/* 1. dat profile stored in jedi_hostapd_profile */
			profile_name = &mtk_ap_buf->profile_names_list[i];
			strncpy(profile_name->jedi_hostapd_profile, profile_name->profile, PROF_NAME_LEN-1);
			profile_name->jedi_hostapd_profile[PROF_NAME_LEN-1] = '\0';
			offset = 0;
			offset += snprintf(profile_name->jedi_hostapd_profile_bak+offset, PROF_NAME_LEN-offset-1,
				profile_name->profile);
			offset += snprintf(profile_name->jedi_hostapd_profile_bak+offset, PROF_NAME_LEN-offset-1, ".bak");

			offset = 0;
			offset += snprintf(profile_name->jedi_hostapd_profile_cmt+offset, PROF_NAME_LEN-offset-1,
				profile_name->profile);
			offset += snprintf(profile_name->jedi_hostapd_profile_cmt+offset, PROF_NAME_LEN-offset-1, ".cmt");

			/* 2. reset hostapd.conf profile as /var/run/hostapd-phy%d.conf */
			snprintf(profile_name->profile, PROF_NAME_LEN-1, "/var/run/hostapd-phy%d.conf", i);
			snprintf(profile_name->profile_bak, PROF_NAME_LEN-1, "/var/run/hostapd-phy%d.conf.bak", i);
			snprintf(profile_name->profile_cmt, PROF_NAME_LEN-1, "/var/run/hostapd-phy%d.conf.cmt", i);

			DPRINT_INFO(WFA_OUT, "intf %s, resave dat profile to jedi_hostapd_profile[%s]\n",
				profile_name->name, profile_name->jedi_hostapd_profile);
			DPRINT_INFO(WFA_OUT, "intf %s, reset hostapd profile as[%s]\n",
				profile_name->name, profile_name->profile);
		}
	} else {
		FILE *fh;
		char intf_name[INTF_NUM][IFNAMSIZ + 1];
		char name[IFNAMSIZ + 1];
		unsigned int wband = W_BAND_INVALID;
		unsigned char pf_idx = 0;

		/* Check if /tmp/hostapd_wifi_intf is available */
		fh = fopen(HOSTAPD_WIRELESS_INTF_FILE, "r");
		if (!fh) {
			DPRINT_INFO(WFA_OUT, "fail to open %s\n", HOSTAPD_WIRELESS_INTF_FILE);
			return -1;
		}

		/* Read each device line */
		while (fgets(name, sizeof(name), fh)) {

			/* Skip empty or almost empty lines. It seems that in some
			 * cases fgets return a line with only a newline.
			*/
			if ((name[0] == '\0') || (name[1] == '\0'))
				continue;

			name[strlen(name)-1] = '\0';

			if (strncmp(name, "apcli", 5) != 0) { /* skip apcli interfaces */
				/* Got it, print info about this interface */
				if (n >= 3)
					break;
				strcpy(intf_name[n], name);
				intf_name[n][IFNAMSIZ] = '\0';
				n++;
				DPRINT_INFO(WFA_OUT, "found interface:%s\n", name);
			}
		}

		fclose(fh);

		mtk_ap_buf->profile_num = n;
		mtk_ap_buf->profile_names_list = calloc(n, sizeof(intf_profile_t));
		if (!mtk_ap_buf->profile_names_list) {
			DPRINT_INFO(WFA_OUT, "fail to calloc profile_names_list\n");
			return -1;
		}

		for (i = 0; i < n; i++) {
			/* MT76 no l1 profile, so intf name and profile maybe not sync
			 * need get band before assign name and profile
			*/
			wband = hostapd_wifi_interface_wband(wifi_intf_fd, intf_name[i]);
			if (wband == W_BAND_2G)
				pf_idx = 0;
			else if (wband == W_BAND_5G)
				pf_idx = 1;
			else
				pf_idx = 2;

			profile_name = &mtk_ap_buf->profile_names_list[pf_idx];
			strncpy(profile_name->name, intf_name[i],
				sizeof(profile_name->name) - 1);
			profile_name->name[sizeof(profile_name->name)-1] = '\0';
			DPRINT_INFO(WFA_OUT, "Profile %d: inf_name %s\n", pf_idx+1, profile_name->name);

			snprintf(profile_name->profile, PROF_NAME_LEN-1, "/var/run/hostapd-phy%d.conf", pf_idx);
			DPRINT_INFO(WFA_OUT, "Profile %d: name %s\n", pf_idx+1, profile_name->profile);
			strcpy(profile_name->profile_bak, profile_name->profile);
			strcat(profile_name->profile_bak, ".bak");
			DPRINT_INFO(WFA_OUT, "Profile bak %d: name %s\n", pf_idx+1, profile_name->profile_bak);
			strcpy(profile_name->profile_cmt, profile_name->profile);
			strcat(profile_name->profile_cmt, ".cmt");
			DPRINT_INFO(WFA_OUT, "Profile name %d for last committed : %s\n",
					pf_idx+1, profile_name->profile_cmt);

			strcpy(profile_name->sigma_dut_profile, profile_name->profile);
			strcpy(profile_name->sigma_tb_profile, profile_name->profile);
			DPRINT_INFO(WFA_OUT, "sigma_dut_profile %d: %s\n",
				pf_idx+1, profile_name->sigma_dut_profile);
			DPRINT_INFO(WFA_OUT, "sigma_tb_profile %d: %s\n", pf_idx+1, profile_name->sigma_tb_profile);

		}
	}
	DPRINT_INFO(WFA_OUT, "%s()<====exit===\n", __func__);
	return 0;
}

void hostapd_backup_profile(mtk_ap_buf_t *mtk_ap_buf)
{
	FILE *file;

	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	if (mtk_ap_buf->intf_2G.status && !mtk_ap_buf->intf_5G.status) {
		strcpy(mtk_ap_buf->intf_5G.name, mtk_ap_buf->intf_2G.name);
		mtk_ap_buf->intf_5G.profile_names = mtk_ap_buf->intf_2G.profile_names;
	}
	if (!mtk_ap_buf->intf_2G.status && mtk_ap_buf->intf_5G.status) {
		strcpy(mtk_ap_buf->intf_2G.name, mtk_ap_buf->intf_5G.name);
		mtk_ap_buf->intf_2G.profile_names = mtk_ap_buf->intf_5G.profile_names;
	}

	DPRINT_INFO(WFA_OUT, "%s() intf_2G name %s\n", __func__, mtk_ap_buf->intf_2G.name);
	DPRINT_INFO(WFA_OUT, "%s() intf_5G name %s\n", __func__, mtk_ap_buf->intf_5G.name);
	DPRINT_INFO(WFA_OUT, "%s() intf_6G name %s status %d\n", __func__,
		mtk_ap_buf->intf_6G.name, mtk_ap_buf->intf_6G.status);

	if (mtk_ap_buf->intf_2G.status) {
		file = fopen(mtk_ap_buf->intf_2G.profile_names->profile_bak, "r");
		if (file == NULL) {
			DPRINT_INFO(WFA_OUT, "%s() First time to use daemon, backup 2G profile!\n", __func__);
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->profile,
				mtk_ap_buf->intf_2G.profile_names->profile_bak);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			system(gCmdStr);

			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile,
					mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile_bak);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
				system(gCmdStr);
			}
			system("sync");
		} else {
			DPRINT_INFO(WFA_OUT, "%s() 2G profile has been backup(%s)!\n", __func__,
				    mtk_ap_buf->intf_2G.profile_names->profile_bak);
			fclose(file);
		}
	}
	if (mtk_ap_buf->intf_5G.status) {
		file = fopen(mtk_ap_buf->intf_5G.profile_names->profile_bak, "r");
		if (file == NULL) {
			DPRINT_INFO(WFA_OUT, "%s() First time to use daemon, backup 5G profile!\n", __func__);
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->profile,
				mtk_ap_buf->intf_5G.profile_names->profile_bak);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			system(gCmdStr);

			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile,
					mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile_bak);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
				system(gCmdStr);
			}

			system("sync");
		} else {
			DPRINT_INFO(WFA_OUT, "%s() 5G profile has been backup(%s)!\n", __func__,
				    mtk_ap_buf->intf_5G.profile_names->profile_bak);
			fclose(file);
		}
	}

	if (mtk_ap_buf->intf_6G.status) {
		file = fopen(mtk_ap_buf->intf_6G.profile_names->profile_bak, "r");
		if (file == NULL) {
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->profile,
				mtk_ap_buf->intf_6G.profile_names->profile_bak);
			system(gCmdStr);
			system("sync");
			DPRINT_INFO(WFA_OUT, "%s() First time to use daemon, backup 6G profile from %s to %s!\n", __func__,
				    mtk_ap_buf->intf_6G.profile_names->profile, mtk_ap_buf->intf_6G.profile_names->profile_bak);

			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile,
					mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile_bak);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
				system(gCmdStr);
			}
		} else {
			DPRINT_INFO(WFA_OUT, "%s() 6G profile has been backup(%s)!\n", __func__,
				    mtk_ap_buf->intf_6G.profile_names->profile_bak);
			fclose(file);
		}
	}
	DPRINT_INFO(WFA_OUT, "%s()<====exit===\n", __func__);
}


void hostapd_change_sigma_profile(mtk_ap_buf_t *mtk_ap_buf)
{
	if (!l1_valid)
		return;

	if (mtk_ap_buf->intf_2G.status) {
		if (mtk_ap_buf->program == PROGRAME_TYPE_HE) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set WirelessMode 16\n",
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->program == PROGRAME_TYPE_VHT) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set HT_AMSDU 0\n",
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set MuMimoDlEnable 1;"
				"wificonf -f %s set MuMimoUlEnable 1",
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->program == PROGRAME_TYPE_NONE ||
			mtk_ap_buf->program == PROGRAME_TYPE_11N) {
			/* disable LDPC for 11N 2G */
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set HT_LDPC 0; wificonf -f %s set VHT_LDPC 0\n",
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);

			/* enable TgnWMMCert for 11N 2G */
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set TgnWMMCert 1\n",
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->program == PROGRAME_TYPE_QM) {
			/* enable QoS and DSCP mapping */
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set QoSEnable 1; wificonf -f %s set DscpPriMapEnable 1\n",
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
	}
	if (mtk_ap_buf->intf_5G.status) {
		if (mtk_ap_buf->program == PROGRAME_TYPE_HE) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set WirelessMode 17\n",
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->program == PROGRAME_TYPE_VHT) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set HT_AMSDU 0\n",
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set MuMimoDlEnable 1;"
				"wificonf -f %s set MuMimoUlEnable 1",
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		/* disable LDPC for 11N 5G */
		if (mtk_ap_buf->program == PROGRAME_TYPE_NONE ||
			mtk_ap_buf->program == PROGRAME_TYPE_11N) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set HT_LDPC 0; wificonf -f %s set VHT_LDPC 0\n",
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->program == PROGRAME_TYPE_QM) {
			/* enable QoS and DSCP mapping */
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wificonf -f %s set QoSEnable 1; wificonf -f %s set DscpPriMapEnable 1\n",
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
	}
}

void hostapd_apply_sigma_profile(mtk_ap_buf_t *mtk_ap_buf, device_type dev_type)
{
	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	if (dev_type == TESTBED) {
		if (mtk_ap_buf->intf_2G.status) {
			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->sigma_tb_profile,
				    mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);

			}
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->profile_bak,
				mtk_ap_buf->intf_2G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->intf_5G.status) {
			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->sigma_tb_profile,
				    mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			}
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->profile_bak,
				mtk_ap_buf->intf_5G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->intf_6G.status) {
			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->sigma_tb_profile,
				    mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			}
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->profile_bak,
				mtk_ap_buf->intf_6G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
	} else {
		if (mtk_ap_buf->intf_2G.status) {
			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile,
					mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			}
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->profile_bak,
				mtk_ap_buf->intf_2G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->intf_5G.status) {
			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile,
					mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			}
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->profile_bak,
				mtk_ap_buf->intf_5G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
		if (mtk_ap_buf->intf_6G.status) {
			DPRINT_INFO(WFA_OUT, "%s() intf_6G.profile_names %s\n", __func__, mtk_ap_buf->intf_6G.profile_names->profile);
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->profile_bak,
				mtk_ap_buf->intf_6G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->sigma_dut_profile,
					mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);

				if (mtk_ap_buf->program == PROGRAME_TYPE_HE ||
					mtk_ap_buf->program == PROGRAME_TYPE_MBO) {
					sprintf(gCmdStr,
					    "sed -i '/ctrl_interface=/aoce=4' %s;"
					    "sed -i '/ctrl_interface=/ambo=1' %s",
					    mtk_ap_buf->intf_6G.profile_names->profile,
					    mtk_ap_buf->intf_6G.profile_names->profile);
					system(gCmdStr);
					DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
				}
			}
		}
	}

	if (mtk_ap_buf->intf_2G.status && mtk_ap_buf->intf_5G.status) {
		if (strcasecmp(mtk_ap_buf->intf_2G.profile_names->profile, mtk_ap_buf->intf_5G.profile_names->profile) != 0) {
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->profile_bak,
				mtk_ap_buf->intf_5G.profile_names->profile);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s\n", __func__, gCmdStr);
		}
	}

	hostapd_change_sigma_profile(mtk_ap_buf);

	system("sync");

	mtk_ap_buf->intern_flag.cert_dat_copied = 1;

	DPRINT_INFO(WFA_OUT, "Apply default profile!\n");
}

void hostapd_restore_profile(mtk_ap_buf_t *mtk_ap_buf)
{
	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	if (mtk_ap_buf->intf_2G.status) {
		/*
		 * MT76, phy0 may be 5G, phy1 2G, phy2 6G
		 *
		 * so restore file by phy no
		*/
		DPRINT_INFO(WFA_OUT, "restore 2g profile\n");
		sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE" "HOSTAPD_PROFILE_CMT,
			mtk_ap_buf->intf_2G.phy_no, mtk_ap_buf->intf_2G.phy_no);
		system(gCmdStr);
		DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

		sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE_BAK" "HOSTAPD_PROFILE,
			mtk_ap_buf->intf_2G.phy_no, mtk_ap_buf->intf_2G.phy_no);
		system(gCmdStr);
		DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

		if (l1_valid) {
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile_cmt);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
			system(gCmdStr);

			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile_bak,
				mtk_ap_buf->intf_2G.profile_names->jedi_hostapd_profile);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
			system(gCmdStr);
		}

		system("sync");
	}

	if (mtk_ap_buf->intf_5G.status) {
		DPRINT_INFO(WFA_OUT, "restore 5g profile\n");
		if (mtk_ap_buf->intf_2G.status) {
			if (strcasecmp(mtk_ap_buf->intf_2G.profile_names->profile,
				mtk_ap_buf->intf_5G.profile_names->profile) != 0) {
				sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE" "HOSTAPD_PROFILE_CMT,
					mtk_ap_buf->intf_5G.phy_no, mtk_ap_buf->intf_5G.phy_no);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

				sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE_BAK" "HOSTAPD_PROFILE,
					mtk_ap_buf->intf_5G.phy_no, mtk_ap_buf->intf_5G.phy_no);
				system(gCmdStr);
				DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

				if (l1_valid) {
					sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile,
						mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile_cmt);
					DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
					system(gCmdStr);

					sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile_bak,
						mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
					DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
					system(gCmdStr);
				}
				system("sync");
			}
		} else {
			sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE" "HOSTAPD_PROFILE_CMT,
				mtk_ap_buf->intf_5G.phy_no, mtk_ap_buf->intf_5G.phy_no);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

			sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE_BAK" "HOSTAPD_PROFILE,
				mtk_ap_buf->intf_5G.phy_no, mtk_ap_buf->intf_5G.phy_no);
			system(gCmdStr);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

			if (l1_valid) {
				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile,
					mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile_cmt);
				DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
				system(gCmdStr);

				sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile_bak,
					mtk_ap_buf->intf_5G.profile_names->jedi_hostapd_profile);
				DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
				system(gCmdStr);
			}

			system("sync");
		}
	}

	if (mtk_ap_buf->intf_6G.status) {
		DPRINT_INFO(WFA_OUT, "restore 6g profile\n");

		sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE" "HOSTAPD_PROFILE_CMT,
			mtk_ap_buf->intf_6G.phy_no, mtk_ap_buf->intf_6G.phy_no);
		system(gCmdStr);
		DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

		sprintf(gCmdStr, "cp  "HOSTAPD_PROFILE_BAK" "HOSTAPD_PROFILE,
			mtk_ap_buf->intf_6G.phy_no, mtk_ap_buf->intf_6G.phy_no);
		system(gCmdStr);
		DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);

		if (l1_valid) {
			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile,
				mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile_cmt);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
			system(gCmdStr);

			sprintf(gCmdStr, "cp %s %s", mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile_bak,
				mtk_ap_buf->intf_6G.profile_names->jedi_hostapd_profile);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
			system(gCmdStr);
		}

		system("sync");
	}

	/* golden dat was used after restoring profile */
	mtk_ap_buf->intern_flag.cert_dat_copied = 0;

	DPRINT_INFO(WFA_OUT, "Restore profile!\n");
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

static int hostapd_assign_profile_pointer_to_intf(
	mtk_ap_buf_t *mtk_ap_buf, intf_desc_t *intf_desc)
{
	int i;
	for (i = 0; i < mtk_ap_buf->profile_num; i++) {
		if (strcmp(intf_desc->name, (&mtk_ap_buf->profile_names_list[i])->name) == 0) {
			intf_desc->profile_names = &mtk_ap_buf->profile_names_list[i];
			printf("Assign interface %s with profile: %s\n",
				intf_desc->name, intf_desc->profile_names->profile);
			return 1;
		}
	}
	printf("Profile not found, skipping interface %s\n", intf_desc->name);
	return 0;
}

static void hostapd_ap_set_htcap(FILE *f, intf_desc_t *def_intf)
{
	int ht40plus = 0, ht40minus = 0;
	char ht_cap_str[WFA_BUFF_1K] = {0};
	char ht_cap_offset = 0;

	if (!def_intf || !f) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	memset(ht_cap_str, 0, sizeof(ht_cap_str));

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	if ((def_intf->ap_mode != AP_MODE_11g) && (def_intf->ap_mode != AP_MODE_11na) &&
	    (def_intf->ap_mode != AP_MODE_11ng) && (def_intf->ap_mode != AP_MODE_11ac) &&
	    (def_intf->ap_mode != AP_MODE_11ax)) {
		DPRINT_INFO(WFA_OUT, "mode %s, don't set ht cap\n", hostapd_apmode_2_string(def_intf->ap_mode));
		return;
	}

	if (def_intf->ch_width > CHANNEL_WIDTH_20) {
		if (def_intf->ch_offset == CHANNEL_OFFSET_BELOW) {
			ht40plus = 0;
			ht40minus = 1;
			DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
		} else {
			ht40plus = 1;
			ht40minus = 0;
			DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
		}

		/* TGn: "[HT40+]" or "[HT40-]" should be set if only channel from ucc */
		if (def_intf->ch_width == CHANNEL_WIDTH_40) {
			if ((def_intf->channel == 1) || (def_intf->channel == 2) || (def_intf->channel == 3) ||
			    (def_intf->channel == 4) || (def_intf->channel == 36) ||
			    (def_intf->channel == 44) | (def_intf->channel == 52) || (def_intf->channel == 60) ||
			    (def_intf->channel == 100) || (def_intf->channel == 108) || (def_intf->channel == 116) ||
			    (def_intf->channel == 124) || (def_intf->channel == 132) || (def_intf->channel == 140) ||
			    (def_intf->channel == 149) || (def_intf->channel == 157)) {
				ht40plus = 1;
				ht40minus = 0;
				DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
			} else {
				ht40plus = 0;
				ht40minus = 1;
				DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
			}
		} else if (def_intf->ch_width == CHANNEL_WIDTH_80) {
			if ((def_intf->channel == 36) || (def_intf->channel == 44) ||
				(def_intf->channel == 52) || (def_intf->channel == 60) ||
				(def_intf->channel == 68) || (def_intf->channel == 84) ||
				(def_intf->channel == 100) || (def_intf->channel == 108) ||
				(def_intf->channel == 116) || (def_intf->channel == 124) ||
				(def_intf->channel == 132) || (def_intf->channel == 140) ||
				(def_intf->channel == 149) || (def_intf->channel == 157) ||
				(def_intf->channel == 165)) {
				ht40plus = 1;
				ht40minus = 0;
				DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
			} else {
				ht40plus = 0;
				ht40minus = 1;
				DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
			}
		}
	} else {
		ht40plus = 0;
		ht40minus = 0;
		DPRINT_INFO(WFA_OUT, "%d, ht40plus %d ht40minus %d\n", __LINE__, ht40plus, ht40minus);
	}

	if (!ht40plus && !ht40minus) {
		if (def_intf->sgi20 != WFA_ENABLED) {
			DPRINT_INFO(WFA_OUT, "OH no, def_intf->sgi20 not enabled, no ht40 plus, no ht40 minus\n");
			goto done;
		}
	}
	DPRINT_INFO(WFA_OUT, "ht40plus %d, ht40minus %d, sgi20 %d\n", ht40plus, ht40minus, def_intf->sgi20);

	if (ht40plus)
		ht_cap_offset += snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[HT40+]");
	if (ht40minus)
		ht_cap_offset += snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[HT40-]");
	if (def_intf->mode != WIFI_6G) {
		if (!def_intf->bcc_mode)
			ht_cap_offset += snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[LDPC]");
		if (def_intf->sgi20)
			ht_cap_offset +=
				snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[SHORT-GI-20]");
		if (def_intf->sgi40 && (def_intf->ch_width >= CHANNEL_WIDTH_40))
			ht_cap_offset +=
				snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[SHORT-GI-40]");
		if (def_intf->stbc)
			ht_cap_offset +=
				snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[TX-STBC][RX-STBC1]");
		if (def_intf->max_amsdu == 7935)
			ht_cap_offset +=
				snprintf(ht_cap_str + ht_cap_offset, sizeof(ht_cap_str) - ht_cap_offset, "[MAX-AMSDU-7935]");
	}

done:

	fprintf(f, "ht_capab=%s\n", ht_cap_str);
	DPRINT_INFO(WFA_OUT, "ht_capab=%s\n", ht_cap_str);

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

static void hostapd_ap_set_vhtcap(FILE *f, intf_desc_t *def_intf)
{
	char vht_cap_str[WFA_BUFF_1K] = {0};
	char vht_cap_offset = 0;
	int ucVhtChannelWidth = 1;
	int ucVhtChannelFrequencyS1 = 0;

	if (!def_intf || !f) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	memset(vht_cap_str, 0, sizeof(vht_cap_str));

	if ((def_intf->ap_mode != AP_MODE_11g) && (def_intf->ap_mode != AP_MODE_11na) &&
	    (def_intf->ap_mode != AP_MODE_11ac) && (def_intf->ap_mode != AP_MODE_11ax)) {
		DPRINT_INFO(WFA_OUT, "mode %s, don't set vht cap\n", hostapd_apmode_2_string(def_intf->ap_mode));
		return;
	}

	if (def_intf->channel <= 14) {
		DPRINT_INFO(WFA_OUT, "channel %d, don't set vht cap\n", def_intf->channel);
		return;
	}

	if (def_intf->mode == WIFI_6G) {
		vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
					   "[MAX-A-MPDU-LEN-EXP0]");
		DPRINT_INFO(WFA_OUT, "%s vht_capab=%s\n", __func__, vht_cap_str);
		return;
	}

	if (def_intf->sgi80 && (def_intf->ch_width >= CHANNEL_WIDTH_80))
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[SHORT-GI-80]");

	if (def_intf->sgi160 && (def_intf->ch_width >= CHANNEL_WIDTH_160))
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[SHORT-GI-160]");

	if (def_intf->stbc)
		vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
					   "[TX-STBC-2BY1][RX-STBC-1]");

	if (def_intf->txbf) {
		vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
			"[SU-BEAMFORMER][SU-BEAMFORMEE][MU-BEAMFORMER][MU-BEAMFORMEE]");

		/* need set tx stream to hostapd */
		if (!l1_valid && !def_intf->spatialTxStream) {
			int tx_stream = 0;
			int ret_len = 0;
			char shellResult[WFA_BUFF_64] = {0};

			/* get tx stream by iw phy phy0 info */
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr)-1,
				"iw phy phy%d info | grep \"VHT Capabilities\"|cut -d \"x\" -f2 |"
				"cut -d \")\" -f1 | cut -b 4 > %s", def_intf->phy_no, HOSTAPD_TEMP_OUTPUT);

			if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
				DPRINT_INFO(WFA_OUT, "fail to get phy vht tx stream info\n");
			DPRINT_INFO(WFA_OUT, "%s shellResult=%s\n", __func__, shellResult);

			ret_len = sscanf(shellResult, "%x", &tx_stream);
			if (ret_len != 1)
				DPRINT_INFO(WFA_OUT, "%s warning sscanf len\n", __func__);

			DPRINT_INFO(WFA_OUT, "%s got tx stream=%d\n", __func__, tx_stream);
			tx_stream &= 0x0007;
			DPRINT_INFO(WFA_OUT, "%s got tx stream=%d after &= 0x0007\n", __func__, tx_stream);
			tx_stream += 1;
			DPRINT_INFO(WFA_OUT, "%s got tx stream=%d after ++\n", __func__, tx_stream);
			vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
				"[SOUNDING-DIMENSION-%d][BF-ANTENNA-%d]", tx_stream, tx_stream);
		}
	} else if (def_intf->mimo) {
		if (def_intf->DL)
			vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
						   "[SU-BEAMFORMER][MU-BEAMFORMER]");
		else
			DPRINT_INFO(WFA_OUT, "UL MIMO not support yet\n");
	}

	if (def_intf->max_mpdu == 11454)
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[MAX-MPDU-11454]");

	else if (def_intf->max_mpdu == 7991)
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[MAX-MPDU-7991]");

	if (def_intf->max_ampdu == 7)
		vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
					   "[MAX-A-MPDU-LEN-EXP7]");

	if (!def_intf->bcc_mode)
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[RXLDPC]");

	if (def_intf->ch_width == CHANNEL_WIDTH_80_80)
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[VHT160-80PLUS80]");

	else if (def_intf->ch_width == CHANNEL_WIDTH_160)
		vht_cap_offset +=
		    snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset, "[VHT160]");

	if (def_intf->mimo) {
		if (def_intf->DL)
			vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
						   "[SU-BEAMFORMER][MU-BEAMFORMER]");
		else
			DPRINT_INFO(WFA_OUT, "UL MIMO not support yet\n");
	}

	vht_cap_offset += snprintf(vht_cap_str + vht_cap_offset, sizeof(vht_cap_str) - vht_cap_offset,
				   "[RX-ANTENNA-PATTERN][TX-ANTENNA-PATTERN]");
	fprintf(f, "vht_capab=%s\n", vht_cap_str);
	DPRINT_INFO(WFA_OUT, "vht_capab=%s\n", vht_cap_str);

	DPRINT_INFO(WFA_OUT, "ch_width=%d, channel %d\n", def_intf->ch_width, def_intf->channel);
	if (def_intf->ch_width == CHANNEL_WIDTH_20) {
		ucVhtChannelWidth = 0;
		ucVhtChannelFrequencyS1 = 0;
		DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucVhtChannelWidth);
		DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucVhtChannelFrequencyS1);
	} else if (def_intf->ch_width == CHANNEL_WIDTH_40) {
		ucVhtChannelWidth = 0;
		ucVhtChannelFrequencyS1 = 0;
		DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucVhtChannelWidth);
		DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucVhtChannelFrequencyS1);
	} else if (def_intf->ch_width == CHANNEL_WIDTH_160) {
		ucVhtChannelWidth = 2;
		if (def_intf->channel >= 36 && def_intf->channel <= 64)
			ucVhtChannelFrequencyS1 = 50;
		else if (def_intf->channel >= 100 && def_intf->channel <= 128)
			ucVhtChannelFrequencyS1 = 114;

		DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucVhtChannelWidth);
		DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucVhtChannelFrequencyS1);
	} else {
		ucVhtChannelWidth = 1;
		ucVhtChannelFrequencyS1 = hostapd_ap_get_cent_ch(def_intf);
		DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucVhtChannelWidth);
		DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucVhtChannelFrequencyS1);
	}

	fprintf(f, "vht_oper_centr_freq_seg0_idx=%d\n", ucVhtChannelFrequencyS1);
	fprintf(f, "vht_oper_chwidth=%d\n", ucVhtChannelWidth);

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

static void hostapd_ap_set_he(FILE *f, intf_desc_t *def_intf)
{

	if (!f || !def_intf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	if (def_intf->ap_mode != AP_MODE_11ax && def_intf->program != PROGRAME_TYPE_HE) {

		DPRINT_INFO(WFA_OUT, "mode %s, don't need to set he\n", hostapd_apmode_2_string(def_intf->ap_mode));
		return;
	}

	DPRINT_INFO(WFA_OUT, "Entering %s()\n", __func__);

	fprintf(f, "ieee80211n=1\n");
	fprintf(f, "ieee80211ax=1\n");
	DPRINT_INFO(WFA_OUT, "ieee80211n=1\n");
	DPRINT_INFO(WFA_OUT, "ieee80211ax=1\n");

	DPRINT_INFO(WFA_OUT, "channel=%d\n", def_intf->channel);

	if (def_intf->channel <= 14) {
		/* 2.4G */
		fprintf(f, "hw_mode=g\n");
		DPRINT_INFO(WFA_OUT, "hw_mode=g\n");
	} else {
		int channel = def_intf->channel;
		int ucHeChannelWidth = 1;
		int ucHeChannelFrequencyS1 = 0;

		/* 5G */
		fprintf(f, "hw_mode=a\n");
		DPRINT_INFO(WFA_OUT, "hw_mode=a\n");
		fprintf(f, "ieee80211ac=1\n");
		DPRINT_INFO(WFA_OUT, "ieee80211ac=1\n");

		DPRINT_INFO(WFA_OUT, "ch_width=%d, channel %d\n", def_intf->ch_width, channel);
		if (def_intf->ch_width == CHANNEL_WIDTH_20) {
			ucHeChannelWidth = 0;
			ucHeChannelFrequencyS1 = 0;
			DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucHeChannelWidth);
			DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucHeChannelFrequencyS1);
		} else if (def_intf->ch_width == CHANNEL_WIDTH_40) {
			ucHeChannelWidth = 0;
			ucHeChannelFrequencyS1 = 0;
			DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucHeChannelWidth);
			DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucHeChannelFrequencyS1);
		} else if (def_intf->ch_width == CHANNEL_WIDTH_160) {
			ucHeChannelWidth = 2;
			if (channel >= 36 && channel <= 64)
				ucHeChannelFrequencyS1 = 50;
			else if (channel >= 100 && channel <= 128)
				ucHeChannelFrequencyS1 = 114;

			DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucHeChannelWidth);
			DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucHeChannelFrequencyS1);
		} else {
			ucHeChannelWidth = 1;
			ucHeChannelFrequencyS1 = hostapd_ap_get_cent_ch(def_intf);
			DPRINT_INFO(WFA_OUT, "ucVhtChannelWidth=%d\n", ucHeChannelWidth);
			DPRINT_INFO(WFA_OUT, "ucVhtChannelFrequencyS1=%d\n", ucHeChannelFrequencyS1);
		}

		fprintf(f, "he_oper_centr_freq_seg0_idx=%d\n", ucHeChannelFrequencyS1);
		fprintf(f, "he_oper_chwidth=%d\n", ucHeChannelWidth);
	}

	if (!def_intf->txbf) {
		fprintf(f, "he_su_beamformer=0\n");
		fprintf(f, "he_su_beamformee=0\n");
	} else if (def_intf->mimo) {
		if (def_intf->DL) {
			fprintf(f, "he_su_beamformer=1\n");
			fprintf(f, "he_mu_beamformer=1\n");
		}
		/* MIMO UL not support yet */
	} else if (def_intf->txbf) {
		fprintf(f, "he_su_beamformer=1\n");
		fprintf(f, "he_mu_beamformer=1\n");
	} else {
		fprintf(f, "he_su_beamformer=0\n");
		fprintf(f, "he_mu_beamformer=0\n");
	}

	fprintf(f, "he_su_beamformee=0\n");
	fprintf(f, "he_twt_required=0\n");
	fprintf(f, "he_bss_color=%d\n", (int)(def_intf->bss_color));

	if (def_intf->mode == WIFI_6G) {
		unsigned char opclass = 0;

		if (def_intf->ch_width == CHANNEL_WIDTH_20)
			opclass = 131;
		if (def_intf->ch_width == CHANNEL_WIDTH_40)
			opclass = 132;
		if (def_intf->ch_width == CHANNEL_WIDTH_80)
			opclass = 133;
		if (def_intf->ch_width == CHANNEL_WIDTH_160)
			opclass = 134;

		fprintf(f, "op_class=%d\n", opclass);
		DPRINT_INFO(WFA_OUT, "op_class=%d\n", opclass);

		/* only MT76 support fils_discovery_min_interval */
		if (!l1_valid) {
			fprintf(f, "fils_discovery_min_interval=0\n");
			DPRINT_INFO(WFA_OUT, "fils_discovery_min_interval=0\n");
		}
	}

	if (def_intf->program == PROGRAME_TYPE_HE && def_intf->bcc_mode) {
		if (l1_valid) {
			fprintf(f, "disable_11ax=1\n");
			DPRINT_INFO(WFA_OUT, "disable_11ax=1\n");
		} else {
			fprintf(f, "he_ldpc=0\n");
			DPRINT_INFO(WFA_OUT, "he_ldpc=0\n");
		}
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

static void hostapd_init_intf_default_param(mtk_ap_buf_t *mtk_ap_buf, wifi_mode mode)
{
	intf_desc_t *intf = NULL;
	int nb_idx = 0;

	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);
	DPRINT_INFO(WFA_OUT, "mode %s\n", hostapd_wifi_mode_2_string(mode));
	mtk_ap_buf->WappEnable = 0;

	if (mode == WIFI_2G) {
		intf = &mtk_ap_buf->intf_2G;
		intf->ch_width = CHANNEL_WIDTH_40;
		intf->ch_offset = CHANNEL_OFFSET_ABOVE;
	} else if (mode == WIFI_5G) {
		intf = &mtk_ap_buf->intf_5G;
		intf->ch_width = CHANNEL_WIDTH_80;
		intf->sgi80 = WFA_ENABLED;
	} else if (mode == WIFI_6G) {
		intf = &mtk_ap_buf->intf_6G;
		intf->ch_width = CHANNEL_WIDTH_80;
		intf->sgi80 = WFA_ENABLED;
		intf->unsol_bcast_probe_resp_interval = 20;
		intf->uapsd_advertisement_enabled = 1;
		intf->rnr = 1;
	} else {
		DPRINT_INFO(WFA_OUT, "Error, No default interface is init, check the reason!!!\n");
		return;
	}

	intf->mode = mode;
	intf->sgi20 = WFA_ENABLED;
	intf->sgi40 = WFA_ENABLED;
	intf->mbss_en = 0;
	intf->dot11vmbss_en = 0;
	intf->bss_idx = 0;
	intf->bss_num = 0;
	intf->WLAN_TAG_bss_num = 0;
	memset(intf->passphrase, 0, sizeof(intf->passphrase));
	memset(intf->auth_server_addr, 0, sizeof(intf->auth_server_addr));
	memset(intf->auth_server_port, 0, sizeof(intf->auth_server_port));
	memset(intf->auth_server_shared_secret, 0, sizeof(intf->auth_server_shared_secret));
	intf->bcc_mode = 0;
	intf->stbc = 1;
	intf->txbf = 1;
	intf->max_amsdu = 7935;
	/* vht_max_mpdu_len: [MAX-MPDU-7991] [MAX-MPDU-11454] */
	intf->max_mpdu = 7991;
	/* Maximum A-MPDU Length Exponent:
	 ** [MAX-A-MPDU-LEN-EXP0]..[MAX-A-MPDU-LEN-EXP7]
	 */
	intf->max_ampdu = 7;
	intf->ap_isolate = 0;
	intf->bss_load_update_period = 60;
	intf->chan_util_avg_period = 600;
	intf->disassoc_low_ack = 1;
	intf->skip_inactivity_poll = 0;
	intf->preamble = 1;
	intf->wme = 1;
	intf->ignore_broadcast_ssid = 0;
	intf->utf8_ssid = 1;
	intf->multi_ap = 0;
	intf->auth_algs = 1;
	intf->mimo = 0;
	intf->UL_MUMIMO = 0;
	intf->DL = 0;
	intf->bss_color = 1;
	intf->sha256ad = 0;
	intf->BTMReq_Term_Bit = 0;
	intf->BSS_Term_Duration = 0;
	intf->BTMReq_DisAssoc_Imnt = 0;
	intf->antiCloggingThreshold = -1;
	intf->sae_pwe = 2;
	intf->ap_ecGroupID[0] = 0;
	intf->sae_pk_keypair[0] = 0;
	intf->sae_pk_modifier[0] = 0;
	intf->trans_disable = 0;
	intf->keyMgmtType[0] = KEY_MGNT_TYPE_OPEN;
	intf->keyMgmtType[1] = KEY_MGNT_TYPE_OPEN;
	intf->KeyRotation = 0;
	intf->KeyRotationInterval = 0;
	intf->KeyRotation_BIGTK_STADisassoc = 0;
	intf->ft_oa = 0;
	intf->ft_bss_list[0] = 0;
	intf->nr_count = 0;
	intf->pmf = WFA_DISABLED;

	snprintf(intf->ctrl_inf, sizeof(intf->ctrl_inf) - 1, HOSTAPD_BIN);
	DPRINT_INFO(WFA_OUT, "intf->name:%s, intf->ctrl_inf:%s!\n", intf->name, intf->ctrl_inf);
	memset(intf->ssid[0], 0, sizeof(intf->ssid[0]));
	memset(intf->ssid[1], 0, sizeof(intf->ssid[1]));
	intf->country_code[0] = '\0';

	for (nb_idx = 0; nb_idx < 3; nb_idx++)
		memset(&intf->neighbor[nb_idx], 0, sizeof(struct neighbor_s));
	if (mtk_ap_buf->program == PROGRAME_TYPE_HS2_R2 ||
		mtk_ap_buf->program == PROGRAME_TYPE_HS2_R3) {
		memset(&intf->ppr, 0, sizeof(intf->ppr));
		intf->ppr.hs20[0] = 1;
		intf->ppr.hs20[1] = 1;
		intf->ppr.hs20_release = 3;
		intf->ppr.manage_p2p = 1;
		intf->ppr.vht_bw = 1;
		intf->ppr.interworking[0] = 1;
		intf->ppr.interworking[1] = 1;
		intf->ppr.access_network_type = 2;
		intf->ppr.venue_group = 2;
		intf->ppr.venue_type = 8;

		snprintf(intf->ppr.hessid, sizeof(intf->ppr.hessid), "%s", "50:6f:9a:00:11:22");
		snprintf(intf->ppr.roaming_consortium[0], sizeof(intf->ppr.roaming_consortium[0]), "%s", "506F9A");
		snprintf(intf->ppr.roaming_consortium[1], sizeof(intf->ppr.roaming_consortium[1]), "%s", "001BC504BD");
		snprintf(intf->ppr.hs20_t_c_filename, sizeof(intf->ppr.hs20_t_c_filename), "%s", "tandc-id1-content.txt");
	}

	if (mtk_ap_buf->program == PROGRAME_TYPE_MBO || mode == WIFI_6G) {
		intf->mbo = 1;
		intf->oce = 4;
		DPRINT_INFO(WFA_OUT, "defaul set mbo=1 and oce=4\n");
	}


	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

static void hostapd_set_default_intf(mtk_ap_buf_t *mtk_ap_buf, wifi_mode mode)
{
	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	if (mode == WIFI_2G) {
		mtk_ap_buf->def_mode = WIFI_2G;
		mtk_ap_buf->def_intf = &mtk_ap_buf->intf_2G;
		mtk_ap_buf->def_intf->mode = WIFI_2G;
		mtk_ap_buf->def_intf->status = 1;
		DPRINT_INFO(WFA_OUT, "Set default interface to 2G!\n");
	} else if (mode == WIFI_5G) {
		mtk_ap_buf->def_mode = WIFI_5G;
		mtk_ap_buf->def_intf = &mtk_ap_buf->intf_5G;
		mtk_ap_buf->def_intf->mode = WIFI_5G;
		mtk_ap_buf->def_intf->status = 1;
		DPRINT_INFO(WFA_OUT, "Set default interface to 5G!\n");
	} else if (mode == WIFI_6G) {
		mtk_ap_buf->def_mode = WIFI_6G;
		mtk_ap_buf->def_intf = &mtk_ap_buf->intf_6G;
		mtk_ap_buf->def_intf->mode = WIFI_6G;
		mtk_ap_buf->def_intf->status = 1;
		DPRINT_INFO(WFA_OUT, "Set default interface to 6G!\n");
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}


int hostapd_get_freq_by_phyno(char phy_no, enum wifi_mode mode, int *freq)
{
	char shellResult[WFA_BUFF_64] = {0};
	char freq_str[WFA_BUFF_64] = {0};

	switch (mode) {
	case WIFI_6G:
		snprintf(freq_str, sizeof(freq_str)-1, "%s", "* 6");
		break;

	case WIFI_5G:
		snprintf(freq_str, sizeof(freq_str)-1, "%s", "* 5");
		break;

	default:
		snprintf(freq_str, sizeof(freq_str)-1, "%s", "* 2");
		break;
	}

	snprintf(gCmdStr, sizeof(gCmdStr)-1,
		"iw phy phy%d channels | grep '\\[' | grep \"%s\" | awk '!(/disabled/)'|awk '{print $2}'|awk 'NR==1' > %s",
		phy_no, freq_str, HOSTAPD_TEMP_OUTPUT);
	if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
		DPRINT_INFO(WFA_OUT, "fail to get phy channels info, check another phy\n");

	if (!strlen(shellResult))
		return WFA_ERROR;

	*freq = atoi(shellResult);
	if ((mode == WIFI_6G && *freq > 6000) ||
		(mode == WIFI_5G && *freq > 5000) ||
		(mode == WIFI_2G && *freq < 5000 && *freq > 2000)) {
		DPRINT_INFO(WFA_OUT, "got %s freq %d\n", hostapd_wifi_mode_2_string(mode), *freq);
		return WFA_SUCCESS;
	} else
		return WFA_ERROR;
}

void hostapd_set_phy_no_by_mode(mtk_ap_buf_t *mtk_ap_buf, enum wifi_mode mode)
{
	char phy_no = 0;
	int ret = 0;
	int freq = 0;

	for (phy_no = 0; phy_no < 3; phy_no++) {
		ret = hostapd_get_freq_by_phyno(phy_no, mode, &freq);
		if (ret != WFA_SUCCESS)
			continue;

		if (freq > 6000 && mtk_ap_buf->intf_6G.status) {
			mtk_ap_buf->intf_6G.phy_no = phy_no;
			DPRINT_INFO(WFA_OUT, "[%s] set 6G phy no %d\n", mtk_ap_buf->intf_6G.name, phy_no);
			break;
		} else if (freq > 5000 && mtk_ap_buf->intf_5G.status) {
			mtk_ap_buf->intf_5G.phy_no = phy_no;
			DPRINT_INFO(WFA_OUT, "[%s] set 5G phy no %d\n", mtk_ap_buf->intf_5G.name, phy_no);
			break;
		} else if (freq > 2000 && freq < 5000 && mtk_ap_buf->intf_2G.status) {
			ret = hostapd_get_freq_by_phyno(phy_no, WIFI_5G, &freq);
			if (ret != WFA_SUCCESS) {
				mtk_ap_buf->intf_2G.phy_no = phy_no;
				DPRINT_INFO(WFA_OUT, "[%s] set 2G phy no %d\n", mtk_ap_buf->intf_5G.name, phy_no);
				break;
			}

			/* set 5g phy no if support 5g freq, continue to check 2g phy no */
			if (freq > 5000 && mtk_ap_buf->intf_5G.status) {
				mtk_ap_buf->intf_5G.phy_no = phy_no;
				DPRINT_INFO(WFA_OUT, "[%s] set 5G phy no %d\n", mtk_ap_buf->intf_5G.name, phy_no);
				continue;
			}
			break;
		}
	}
}


void hostapd_set_phy_no(mtk_ap_buf_t *mtk_ap_buf)
{
	hostapd_set_phy_no_by_mode(mtk_ap_buf, WIFI_6G);
	hostapd_set_phy_no_by_mode(mtk_ap_buf, WIFI_5G);
	hostapd_set_phy_no_by_mode(mtk_ap_buf, WIFI_2G);
	return;
}

static int hostapd_ap_init(mtk_ap_buf_t *mtk_ap_buf)
{
	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return WFA_FAILURE;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	if (mtk_ap_buf->intf_2G.status) {
		hostapd_set_default_intf(mtk_ap_buf, WIFI_2G);
		hostapd_init_intf_default_param(mtk_ap_buf, WIFI_2G);
		mtk_ap_buf->intf_2G.bss_num = 1;
		if (mtk_ap_buf->intf_2G.bss_num > 1)
			mtk_ap_buf->intf_2G.mbss_en = 1;
	}

	if (mtk_ap_buf->intf_5G.status) {
		hostapd_set_default_intf(mtk_ap_buf, WIFI_5G);
		hostapd_init_intf_default_param(mtk_ap_buf, WIFI_5G);
		mtk_ap_buf->intf_5G.bss_num = 1;
		if (mtk_ap_buf->intf_5G.bss_num > 1)
			mtk_ap_buf->intf_5G.mbss_en = 1;
	}

	if (mtk_ap_buf->intf_6G.status) {
		hostapd_init_intf_default_param(mtk_ap_buf, WIFI_6G);
		mtk_ap_buf->intf_6G.bss_num = 1;
		if (mtk_ap_buf->intf_6G.bss_num > 1)
			mtk_ap_buf->intf_6G.mbss_en = 1;
	}

	if (!(mtk_ap_buf->intf_2G.status || mtk_ap_buf->intf_5G.status)) {
		DPRINT_INFO(WFA_OUT, "%s No valid interface!!!\n", __func__);
		DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
		return WFA_FAILURE;
	}

	hostapd_set_phy_no(mtk_ap_buf);

	/* check if need set OWN_IP_ADDR and BssidNum */

	mtk_ap_buf->WLAN_TAG = 0;
	mtk_ap_buf->post_cmd_idx = 0;
	strncpy(mtk_ap_buf->Reg_Domain, "", sizeof(mtk_ap_buf->Reg_Domain));
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

static void hostapd_write_conf(intf_desc_t *def_intf)
{
	FILE *f = NULL;
	int bss_idx = 0, bss_end_idx = 0;
	int ret_len = 0;
	char shellResult[WFA_BUFF_64] = {0};
	char shellResult_interface[WFA_BUFF_64] = {0};
	char shellResult_bss[WFA_BUFF_64] = {0};
	/* hostapd_qos_t *ap_qos = &(def_intf->ap_qos); */
	hostapd_qos_t *sta_qos = &(def_intf->sta_qos);
	char *keymgnt = NULL;
	char bss_intf_name[INTF_NAME_LEN];
	unsigned char is_owe = 0;
	char hapd_profile[PROF_NAME_LEN] = {0};
	char mbss_intf_name[WFA_NAME_LEN];
	int offset = 0;

	DPRINT_INFO(WFA_OUT, "*********************************\n");
	DPRINT_INFO(WFA_OUT, "%s() enter\n", __func__);

	/*
	 * MT76, phy0 may be 5G, phy1 2G, phy2 6G
	 *
	 * so write file by phy no
	*/
	snprintf(hapd_profile, sizeof(hapd_profile), "/var/run/hostapd-phy%d.conf", def_intf->phy_no);


	f = fopen(hapd_profile, "w");
	if (f == NULL) {
		DPRINT_ERR(WFA_ERR, "file open fail\n");
		return;
	}
	DPRINT_INFO(WFA_OUT, "open %s()\n", hapd_profile);

	fprintf(f, "driver=nl80211\n");
	fprintf(f, "logger_syslog=127\n");
	fprintf(f, "logger_syslog_level=2\n");
	fprintf(f, "logger_stdout=127\n");
	fprintf(f, "logger_stdout_level=2\n");

	if (def_intf->ap_mode == AP_MODE_11ng) {
		fprintf(f, "noscan=0\n");
		DPRINT_INFO(WFA_OUT, "noscan=0\n");
	} else {
		fprintf(f, "noscan=1\n");
		DPRINT_INFO(WFA_OUT, "noscan=1\n");
	}

	switch (def_intf->ap_mode) {
	case AP_MODE_11b:
		fprintf(f, "hw_mode=b\n");
		DPRINT_INFO(WFA_OUT, "hw_mode=b\n");
		break;
	case AP_MODE_11g:
	case AP_MODE_11ng:
		fprintf(f, "hw_mode=g\n");
		DPRINT_INFO(WFA_OUT, "hw_mode=g\n");
		if (def_intf->ap_mode == AP_MODE_11ng) {
			fprintf(f, "ieee80211n=1\n");
			DPRINT_INFO(WFA_OUT, "ieee80211n=1\n");
		}
		break;
	case AP_MODE_11a:
	case AP_MODE_11na:
	case AP_MODE_11ac:
		fprintf(f, "hw_mode=a\n");
		DPRINT_INFO(WFA_OUT, "hw_mode=a\n");
		if ((!l1_valid) && (def_intf->country_code[0] == '\0'))
			snprintf(def_intf->country_code, sizeof(def_intf->country_code), "US");

		if (def_intf->ap_mode == AP_MODE_11na) {
			fprintf(f, "ieee80211n=1\n");
			DPRINT_INFO(WFA_OUT, "ieee80211n=1\n");
		} else if (def_intf->ap_mode == AP_MODE_11ac) {
			fprintf(f, "ieee80211n=1\n");
			DPRINT_INFO(WFA_OUT, "ieee80211n=1\n");
			fprintf(f, "ieee80211ac=1\n");
			DPRINT_INFO(WFA_OUT, "ieee80211ac=1\n");
		}
		break;
	case AP_MODE_11ax:
		hostapd_ap_set_he(f, def_intf);
		break;
	default:
		DPRINT_WARNING(WFA_WNG, "unsupport mode: %d\n", def_intf->ap_mode);
		break;
	}

	if (def_intf->channel) {
		fprintf(f, "channel=%d\n", def_intf->channel);
		DPRINT_INFO(WFA_OUT, "channel=%d\n", def_intf->channel);

		if ((def_intf->channel <= 14) && (def_intf->ch_width == CHANNEL_WIDTH_40)) {
			fprintf(f, "ht_coex=1\n");
			fprintf(f, "obss_interval=300\n");
			DPRINT_INFO(WFA_OUT, "ht_coex=1\n");
			DPRINT_INFO(WFA_OUT, "obss_interval=300\n");
		}
	}
	if (def_intf->bcnint)
		fprintf(f, "beacon_int=%d\n", def_intf->bcnint);
	DPRINT_INFO(WFA_OUT, "beacon_int=%d\n", def_intf->bcnint);

	if (def_intf->dtim_period)
		fprintf(f, "dtim_period=%d\n", def_intf->dtim_period);
	DPRINT_INFO(WFA_OUT, "dtim_period=%d\n", def_intf->dtim_period);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr), "md5sum %s | cut -d\" \" -f1 > %s",
		hapd_profile, HOSTAPD_TEMP_OUTPUT);
	if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
		DPRINT_INFO(WFA_OUT, "no md5 found\n");

	fprintf(f, "radio_config_id=%s\n", shellResult);
	DPRINT_INFO(WFA_OUT, "radio_config_id %s\n", shellResult);

	hostapd_ap_set_htcap(f, def_intf);

	hostapd_ap_set_vhtcap(f, def_intf);

	if (def_intf->bss_idx)
		bss_end_idx = 2;
	else
		bss_end_idx = 1;

	for (bss_idx = 0; bss_idx < bss_end_idx; bss_idx++) {
		if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_OWE)
			is_owe = 1;
	}
	if (is_owe) {
		for (bss_idx = 0; bss_idx < bss_end_idx; bss_idx++) {
			if (!l1_valid) {
				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
					 def_intf->name, def_intf->name, HOSTAPD_TEMP_OUTPUT);

				memset(shellResult, 0, sizeof(shellResult));
				if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
					DPRINT_INFO(WFA_OUT, "no bssid found\n");
				if (bss_idx) {
					unsigned int mac_hex[6] = {0};

					/* last bytes plus one for multi bss */
					ret_len =
					    sscanf(shellResult, "%02x:%02x:%02x:%02x:%02x:%02x", &mac_hex[0],
						   &mac_hex[1], &mac_hex[2], &mac_hex[3], &mac_hex[4], &mac_hex[5]);
					if (ret_len != 6)
						DPRINT_INFO(WFA_OUT, "sscanf of BSSID not match\n");
					mac_hex[5] += 1;
					snprintf(shellResult, sizeof(shellResult), "%02x:%02x:%02x:%02x:%02x:%02x",
						 mac_hex[0], mac_hex[1], mac_hex[2], mac_hex[3], mac_hex[4],
						 mac_hex[5]);

					strncpy(shellResult_bss, shellResult, sizeof(shellResult_bss)-1);
					shellResult_bss[sizeof(shellResult_bss)-1] = '\0';
					DPRINT_INFO(WFA_OUT, "bss_bssid=%s\n", shellResult);
				} else {
					strncpy(shellResult_interface, shellResult, sizeof(shellResult_interface)-1);
					shellResult_interface[sizeof(shellResult_interface)-1] = '\0';
					DPRINT_INFO(WFA_OUT, "interface_bssid=%s\n", shellResult);
				}
			} else {

				memset(bss_intf_name, 0, sizeof(bss_intf_name));
				memcpy(bss_intf_name, def_intf->name, sizeof(bss_intf_name));

				if (bss_idx) {
					memset(gCmdStr, 0, sizeof(gCmdStr));

					snprintf(bss_intf_name+strlen(def_intf->name)-1, sizeof(def_intf->name), "%d", bss_idx);
					snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
						 bss_intf_name, bss_intf_name, HOSTAPD_TEMP_OUTPUT);

					if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
						printf("no bssid found\n");

					strncpy(shellResult_bss, shellResult, sizeof(shellResult_bss)-1);
					shellResult_bss[sizeof(shellResult_bss)-1] = '\0';
					DPRINT_INFO(WFA_OUT, "bss_bssid=%s\n", shellResult);
				} else {
					memset(gCmdStr, 0, sizeof(gCmdStr));
					snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
						 def_intf->name, def_intf->name, HOSTAPD_TEMP_OUTPUT);

					if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
						printf("no bssid found\n");

					strncpy(shellResult_interface, shellResult, sizeof(shellResult_interface)-1);
					shellResult_interface[sizeof(shellResult_interface)-1] = '\0';
					DPRINT_INFO(WFA_OUT, "interface_bssid=%s\n", shellResult);
				}
			}
		}
	}

	if (l1_valid)
		fprintf(f, "use_driver_iface_addr=1\n");

	for (bss_idx = 0; bss_idx < bss_end_idx; bss_idx++) {
		if (bss_idx) {
			memset(mbss_intf_name, 0, sizeof(mbss_intf_name));
			offset += snprintf(mbss_intf_name + offset, sizeof(mbss_intf_name) - offset - 1,
				def_intf->name);
			if (strncmp(def_intf->name, "wlan", 4) == 0) {
				/* MT76: main intf, wlan0; mbss: wlan0-1 */
				offset += snprintf(mbss_intf_name + offset, sizeof(mbss_intf_name) - offset - 1,
				    "-%d", bss_idx);
			} else {
				/* MT76 main intf: phy0-ap0; mbss: phy0-ap1 */
				/* none-MT76 main intf: ra0; mbss: ra1 */
				offset += snprintf(mbss_intf_name + offset - 1, sizeof(mbss_intf_name) - offset - 1,
				    "%d", bss_idx);
			}
			DPRINT_INFO(WFA_OUT, "mbss_intf_name=%s\n", mbss_intf_name);
			/* interface name for mbss: ra1/rai1??? */
			fprintf(f, "bss=%s\n", mbss_intf_name);
			DPRINT_INFO(WFA_OUT, "bss=%s\n", mbss_intf_name);
		} else {
			fprintf(f, "interface=%s\n", def_intf->name);
			DPRINT_INFO(WFA_OUT, "interface=%s\n", def_intf->name);
		}

		if (l1_valid) {
			if (def_intf->dot11vmbss_en) {
				fprintf(f, "dot11vmbssid=1\n");
				DPRINT_INFO(WFA_OUT, "dot11vmbssid=1\n");
			}
		}

		fprintf(f, "ctrl_interface=%s\n", def_intf->ctrl_inf);
		DPRINT_INFO(WFA_OUT, "ctrl_interface=%s\n", def_intf->ctrl_inf);

		if (bss_end_idx == 1) {
			fprintf(f, "ssid=%s\n", def_intf->ssid[0]);
			DPRINT_INFO(WFA_OUT, "ssid=%s\n", def_intf->ssid[0]);
		} else {
			if (bss_idx) {
				if (strlen(def_intf->ssid[1])) {
					fprintf(f, "ssid=%s\n", def_intf->ssid[1]);
					DPRINT_INFO(WFA_OUT, "multi ssid=%s\n", def_intf->ssid[1]);
				} else {
					if (is_owe) {
						fprintf(f, "ssid=%s-owe\n", def_intf->ssid[0]);
						DPRINT_INFO(WFA_OUT, "OWE multi ssid=%s-owe\n", def_intf->ssid[0]);
					} else {
						fprintf(f, "ssid=sigma-unconfig\n");
						DPRINT_INFO(WFA_OUT, "multi ssid=sigma-unconfig\n");
					}
				}
			} else {
				if (strlen(def_intf->ssid[0])) {
					fprintf(f, "ssid=%s\n", def_intf->ssid[0]);
					DPRINT_INFO(WFA_OUT, "main ssid=%s\n", def_intf->ssid[0]);
				} else {
					fprintf(f, "ssid=sigma-unconfig\n");
					DPRINT_INFO(WFA_OUT, "main ssid=sigma-unconfig\n");
				}
			}
		}

		/* set the same channel for main bss and bss[bss_idx] */

		if (def_intf->rts)
			fprintf(f, "rts_threshold=%d\n", def_intf->rts);

		if (def_intf->frgmnt)
			fprintf(f, "fragm_threshold=%d\n", def_intf->frgmnt);

		if (def_intf->wme)
			fprintf(f, "wmm_enabled=1\n");

		if (def_intf->wmmps)
			fprintf(f, "uapsd_advertisement_enabled=1\n");

		if (def_intf->p2p_mgmt)
			fprintf(f, "manage_p2p=1\n");

		switch (def_intf->keyMgmtType[bss_idx]) {
		case KEY_MGNT_TYPE_OPEN:
			fprintf(f, "wpa=0\n");
			if (def_intf->wepkey[0]) {
				fprintf(f, "wep_key0=%s\n", def_intf->wepkey);
				fprintf(f, "wep_default_key=0\n");
			}
			if (is_owe) {
				DPRINT_INFO(WFA_OUT, "set owe bssid %s\n", shellResult_bss);
				if (!l1_valid)
					fprintf(f, "bssid=%s\n", shellResult_interface);
				else
					fprintf(f, "owe_transition_ifname=%s\n", bss_intf_name);
				fprintf(f, "owe_transition_bssid=%s\n", shellResult_bss);
				fprintf(f, "owe_transition_ssid=\"%s-owe\"\n", def_intf->ssid[0]);
				fprintf(f, "ignore_broadcast_ssid=0\n");
				fprintf(f, "wpa=0\n");
				fprintf(f, "ieee8021x=0\n");
				fprintf(f, "ieee80211w=0\n");
			}
			break;
		case KEY_MGNT_TYPE_WPA_PSK:
		case KEY_MGNT_TYPE_WPA2_PSK:
		case KEY_MGNT_TYPE_WPA2_PSK_MIXED:
			/* config wpa */
			if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_PSK) {
				fprintf(f, "rsn_pairwise=CCMP\n");
				fprintf(f, "wpa=2\n");
				DPRINT_INFO(WFA_OUT, "rsn_pairwise=CCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa=2\n");
			} else if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_PSK_MIXED) {
				fprintf(f, "wpa=3\n");
				DPRINT_INFO(WFA_OUT, "wpa=3\n");
			} else {
				fprintf(f, "wpa=1\n");
				DPRINT_INFO(WFA_OUT, "wpa=1\n");
			}

			switch (def_intf->encpType[bss_idx]) {
			case ENCP_TYPE_NONE:
				/* do nothing */
				DPRINT_INFO(WFA_OUT, "do nothing, ENCP_TYPE_NONE\n");
				break;
			case ENCP_TYPE_TKIP:
				fprintf(f, "wpa_pairwise=TKIP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=TKIP\n");
				break;
			case ENCP_TYPE_CCMP:
				fprintf(f, "wpa_pairwise=CCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=CCMP\n");
				break;
			case ENCP_TYPE_GCMP_128:
				fprintf(f, "wpa_pairwise=GCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=GCMP\n");
				break;
			case ENCP_TYPE_CCMP_TKIP:
				fprintf(f, "wpa_pairwise=TKIP CCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=TKIP CCMP\n");
				break;
			default:
				DPRINT_INFO(WFA_OUT, "unknown encpType: %d", def_intf->encpType[bss_idx]);
				break;
			}

			if (def_intf->passphrase[0] && def_intf->encpType[bss_idx] > ENCP_TYPE_WEP) {
				fprintf(f, "wpa_passphrase=%s\n", def_intf->passphrase);
				DPRINT_INFO(WFA_OUT, "wpa_passphrase=%s\n", def_intf->passphrase);
			}

			switch (def_intf->pmf) {
			case WFA_DISABLED:
			case WFA_OPTIONAL:
				if (def_intf->sha256ad) {
					fprintf(f, "wpa_key_mgmt=WPA-PSK WPA-PSK-SHA256\n");
					DPRINT_INFO(WFA_OUT, "wpa_key_mgmt=WPA-PSK WPA-PSK-SHA256\n");
				} else {
					if (def_intf->ft_oa) {
						fprintf(f, "wpa_key_mgmt=WPA-PSK FT-PSK\n");
						DPRINT_INFO(WFA_OUT, "wpa_key_mgmt=WPA-PSK FT-PSK\n");
					} else {
						fprintf(f, "wpa_key_mgmt=WPA-PSK\n");
						DPRINT_INFO(WFA_OUT, "wpa_key_mgmt=WPA-PSK\n");
					}
				}
				break;
			case WFA_REQUIRED:
				fprintf(f, "wpa_key_mgmt=WPA-PSK-SHA256\n");
				DPRINT_INFO(WFA_OUT, "wpa_key_mgmt=WPA-PSK-SHA256\n");
				break;
			default:
				break;
			}
			break;

		case KEY_MGNT_TYPE_WPA_EAP:
		case KEY_MGNT_TYPE_WPA2_EAP:
		case KEY_MGNT_TYPE_WPA2_EAP_MIXED:
		case KEY_MGNT_TYPE_SUITEB:
			/* config wpa */
			if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_EAP) {
				fprintf(f, "rsn_pairwise=CCMP\n");
				fprintf(f, "wpa=2\n");
				DPRINT_INFO(WFA_OUT, "rsn_pairwise=CCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa=2\n");
			} else if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_SUITEB) {
				fprintf(f, "wpa=2\n");
				DPRINT_INFO(WFA_OUT, "wpa=2\n");
			} else if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_EAP_MIXED) {
				fprintf(f, "wpa=3\n");
				DPRINT_INFO(WFA_OUT, "wpa=3\n");
				if (def_intf->program == PROGRAME_TYPE_MBO) {
					/* PMF needs to be enabled whenever using WPA2 with MBO */
					fprintf(f, "ieee80211w=2\n");
				}
			} else {
				fprintf(f, "wpa=1\n");
				DPRINT_INFO(WFA_OUT, "wpa=1\n");
			}

			fprintf(f, "ieee8021x=1\n");
			if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_SUITEB) {
				fprintf(f, "wpa_key_mgmt=WPA-EAP-SUITE-B-192\n");
				fprintf(f, "group_mgmt_cipher=BIP-GMAC-256\n");
			} else {
				if (def_intf->program == PROGRAME_TYPE_MBO)
					fprintf(f, "wpa_key_mgmt=WPA-EAP FT-EAP\n");
				else
					fprintf(f, "wpa_key_mgmt=WPA-EAP\n");
			}

			switch (def_intf->encpType[bss_idx]) {
			case ENCP_TYPE_NONE:
				/* do nothing */
				DPRINT_INFO(WFA_OUT, "do nothing, ENCP_TYPE_NONE\n");
				break;
			case ENCP_TYPE_TKIP:
				fprintf(f, "wpa_pairwise=TKIP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=TKIP\n");
				break;
			case ENCP_TYPE_CCMP:
				fprintf(f, "wpa_pairwise=CCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=CCMP\n");
				break;
			case ENCP_TYPE_CCMP_256:
				fprintf(f, "wpa_pairwise=CCMP-256\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=CCMP-256\n");
				break;
			case ENCP_TYPE_GCMP_128:
				fprintf(f, "wpa_pairwise=GCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=GCMP\n");
				break;
			case ENCP_TYPE_GCMP_256:
				fprintf(f, "wpa_pairwise=GCMP-256\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=GCMP-256\n");
				break;
			case ENCP_TYPE_CCMP_TKIP:
				fprintf(f, "wpa_pairwise=TKIP CCMP\n");
				DPRINT_INFO(WFA_OUT, "wpa_pairwise=TKIP CCMP\n");
				break;
			default:
				DPRINT_INFO(WFA_OUT, "unknown encpType: %d", def_intf->encpType[bss_idx]);
				break;
			}
			break;

		case KEY_MGNT_TYPE_WPA2_PSK_SAE:
		case KEY_MGNT_TYPE_WPA2_SAE: {
			const char *key_mgmt;

			fprintf(f, "wpa=2\n");

			switch (def_intf->pmf) {
			case WFA_DISABLED:
			case WFA_OPTIONAL:
				if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_SAE)
					key_mgmt = "SAE";
				else
					key_mgmt = "WPA-PSK SAE";

				fprintf(f, "wpa_key_mgmt=%s%s\n", key_mgmt,
					def_intf->sha256ad ? " WPA-PSK-SHA256" : "");
				break;
			case WFA_REQUIRED:
				if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_SAE)
					key_mgmt = "SAE";
				else
					key_mgmt = "WPA-PSK-SHA256 SAE";

				fprintf(f, "wpa_key_mgmt=%s\n", key_mgmt);
				break;
			default:
				break;
			}

			switch (def_intf->encpType[bss_idx]) {
			case ENCP_TYPE_NONE:
				/* do nothing */
				break;
			case ENCP_TYPE_TKIP:
				fprintf(f, "wpa_pairwise=TKIP\n");
				break;
			case ENCP_TYPE_CCMP:
				fprintf(f, "wpa_pairwise=CCMP\n");
				break;
			case ENCP_TYPE_GCMP_128:
				fprintf(f, "wpa_pairwise=GCMP\n");
				break;
			case ENCP_TYPE_CCMP_TKIP:
				fprintf(f, "wpa_pairwise=CCMP TKIP\n");
				break;
			default:
				DPRINT_WARNING(WFA_WNG, "unknown encpType: %d", def_intf->encpType[bss_idx]);
				break;
			}

			if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_SAE) {
				if (def_intf->sae_pk_keypair[0] && def_intf->sae_pk_modifier[0]) {
					fprintf(f, "sae_password=%s|pk=%s:%s\n",
						def_intf->passphrase, def_intf->sae_pk_modifier, def_intf->sae_pk_keypair);
					DPRINT_INFO(WFA_OUT, "sae_password=%s|pk=%s:%s\n",
						def_intf->passphrase, def_intf->sae_pk_modifier, def_intf->sae_pk_keypair);
				} else if (def_intf->passphrase[0]) {
					fprintf(f, "sae_password=%s\n", def_intf->passphrase);
					DPRINT_INFO(WFA_OUT, "sae_password=%s\n", def_intf->passphrase);
				} else {
					DPRINT_INFO(WFA_OUT, "error: no sae password set\n");
					break;
				}
				/* if (def_intf->sae_pk)
				*	fprintf(f, "sae_pk=%d\n", def_intf->sae_pk);
				*/
			} else {
				if (def_intf->passphrase[0]) {
					fprintf(f, "wpa_passphrase=%s\n", def_intf->passphrase);
					fprintf(f, "sae_password=%s\n", def_intf->passphrase);
				}
			}
			fprintf(f, "sae_pwe=%d\n", def_intf->sae_pwe);
			fprintf(f, "group_mgmt_cipher=AES-128-CMAC\n");
			if (def_intf->ap_ecGroupID[0]) {
				fprintf(f, "sae_groups=%s\n", def_intf->ap_ecGroupID);
				DPRINT_INFO(WFA_OUT, "SAE groups=%s\n", def_intf->ap_ecGroupID);
			}
			break;
		}

		case KEY_MGNT_TYPE_WPA2_OWE:
			keymgnt = def_intf->keymgnt;
			if (strcasecmp(keymgnt, "OWE") != 0) {
				DPRINT_INFO(WFA_OUT, "auth is not OWE, break\n");
				break;
			}
			DPRINT_INFO(WFA_OUT, "set owe bssid %s\n", shellResult_interface);
			if (bss_end_idx > 1) {
				if (!l1_valid)
					fprintf(f, "bssid=%s\n", shellResult_bss);
				else
					fprintf(f, "owe_transition_ifname=%s\n", def_intf->name);
				fprintf(f, "owe_transition_bssid=%s\n", shellResult_interface);
				fprintf(f, "owe_transition_ssid=\"%s\"\n", def_intf->ssid[0]);
				fprintf(f, "ignore_broadcast_ssid=1\n");
				fprintf(f, "wpa=2\n");
				fprintf(f, "wpa_key_mgmt=OWE\n");
				fprintf(f, "ieee8021x=0\n");
				fprintf(f, "ieee80211w=2\n");
				fprintf(f, "rsn_pairwise=CCMP\n");
				if (def_intf->ap_ecGroupID[0]) {
					fprintf(f, "owe_groups=%s\n", def_intf->ap_ecGroupID);
					fprintf(f, "owe_ptk_workaround=1\n");
					DPRINT_INFO(WFA_OUT, "OWE groups=%s\n", def_intf->ap_ecGroupID);
				}
			} else if (bss_end_idx == 1) {
				if (!l1_valid)
					fprintf(f, "bssid=%s\n", shellResult_interface);
				fprintf(f, "ignore_broadcast_ssid=0\n");
				fprintf(f, "wpa=2\n");
				fprintf(f, "wpa_key_mgmt=OWE\n");
				fprintf(f, "ieee8021x=0\n");
				fprintf(f, "ieee80211w=2\n");
				fprintf(f, "rsn_pairwise=CCMP\n");
				if (def_intf->ap_ecGroupID[0]) {
					fprintf(f, "owe_groups=%s\n", def_intf->ap_ecGroupID);
					fprintf(f, "owe_ptk_workaround=1\n");
					DPRINT_INFO(WFA_OUT, "OWE groups=%s\n", def_intf->ap_ecGroupID);
				}
			}
			break;

			case KEY_MGNT_TYPE_OSEN:
			case KEY_MGNT_TYPE_WPA2_ENT_OSEN:
				if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_OSEN) {
					fprintf(f, "wpa=0\n");
					fprintf(f, "wpa_key_mgmt=OSEN\n");
				} else {
					fprintf(f, "wpa=2\n");
					fprintf(f, "wpa_key_mgmt=WPA-EAP OSEN\n");
					fprintf(f, "rsn_pairwise=CCMP\n");
				}

				fprintf(f, "ieee8021x=1\n");
				fprintf(f, "osen=1\n");
				fprintf(f, "group_mgmt_cipher=AES-128-CMAC\n");
				break;

		case KEY_MGNT_TYPE_UNKNOWN:
			keymgnt = def_intf->keymgnt;
			if (strcasecmp(keymgnt, "SAE FT-SAE") == 0) {
				fprintf(f, "wpa=2\n");
				fprintf(f, "wpa_key_mgmt=%s\n", keymgnt);
				fprintf(f, "ieee8021x=0\n");
				fprintf(f, "ieee80211w=2\n");
				fprintf(f, "rsn_pairwise=CCMP\n");
				fprintf(f, "disable_pmksa_caching=0\n");
				fprintf(f, "sae_pwe=%d\n", def_intf->sae_pwe);
				fprintf(f, "mobility_domain=0101\n");
				fprintf(f, "ft_r0_key_lifetime=10000\n");
				fprintf(f, "bss_transition=1\n");
				fprintf(f, "sae_password=%s\n", def_intf->passphrase);
				fprintf(f, "r1_key_holder=%s\n", def_intf->mac_addr);
				fprintf(f, "pmk_r1_push=1\n");
				fprintf(f, "nas_identifier=ap%c.mtk.com\n", def_intf->cmd_name[2]);
				fprintf(f, "ft_over_ds=%d\n", def_intf->ft_oa ? 0 : 1);
			} else if (strcasecmp(keymgnt, "WPA-EAP-SHA256 FT-EAP") == 0) {
				fprintf(f, "wpa=2\n");
				fprintf(f, "wpa_key_mgmt=%s\n", keymgnt);
				fprintf(f, "ieee8021x=1\n");
				fprintf(f, "ieee80211w=2\n");
				fprintf(f, "rsn_pairwise=CCMP\n");
				fprintf(f, "disable_pmksa_caching=0\n");
				fprintf(f, "mobility_domain=0101\n");
				fprintf(f, "ft_r0_key_lifetime=10000\n");
				fprintf(f, "own_ip_addr=%s\n", gIPaddr);
				fprintf(f, "bss_transition=1\n");
				fprintf(f, "r1_key_holder=%s\n", def_intf->mac_addr);
				fprintf(f, "pmk_r1_push=1\n");
				fprintf(f, "nas_identifier=ap%c.mtk.com\n", def_intf->cmd_name[2]);
				fprintf(f, "ft_over_ds=%d\n", def_intf->ft_oa ? 0 : 1);
			} else if (strcasecmp(keymgnt, "WPA-EAP FT-EAP WPA-EAP-SHA256") == 0) {
				fprintf(f, "wpa=2\n");
				fprintf(f, "wpa_key_mgmt=%s\n", keymgnt);
				fprintf(f, "ieee8021x=1\n");
				fprintf(f, "ieee80211w=1\n");
				fprintf(f, "rsn_pairwise=CCMP\n");
				fprintf(f, "disable_pmksa_caching=0\n");
				fprintf(f, "mobility_domain=0101\n");
				fprintf(f, "ft_r0_key_lifetime=10000\n");
				fprintf(f, "own_ip_addr=%s\n", gIPaddr);
				fprintf(f, "bss_transition=1\n");
				fprintf(f, "r1_key_holder=%s\n", def_intf->mac_addr);
				fprintf(f, "pmk_r1_push=1\n");
				fprintf(f, "nas_identifier=ap%c.mtk.com\n", def_intf->cmd_name[2]);
				fprintf(f, "ft_over_ds=%d\n", def_intf->ft_oa ? 0 : 1);
			} else if (strcasecmp(keymgnt, "WPA-PSK-SHA256 FT-PSK WPA-PSK SAE FT-SAE") == 0) {
				fprintf(f, "wpa=2\n");
				fprintf(f, "wpa_key_mgmt=%s\n", keymgnt);
				fprintf(f, "ieee8021x=0\n");
				fprintf(f, "ieee80211w=1\n");
				fprintf(f, "rsn_pairwise=CCMP\n");
				fprintf(f, "disable_pmksa_caching=0\n");
				fprintf(f, "sae_pwe=%d\n", def_intf->sae_pwe);
				fprintf(f, "mobility_domain=0101\n");
				fprintf(f, "ft_r0_key_lifetime=10000\n");
				fprintf(f, "wpa_passphrase=%s\n", def_intf->passphrase);
				fprintf(f, "bss_transition=1\n");
				fprintf(f, "r1_key_holder=%s\n", def_intf->mac_addr);
				fprintf(f, "pmk_r1_push=1\n");
				fprintf(f, "nas_identifier=ap%c.mtk.com\n", def_intf->cmd_name[2]);
				fprintf(f, "ft_over_ds=%d\n", def_intf->ft_oa ? 0 : 1);
			} else {
				DPRINT_WARNING(WFA_WNG, "unknown keymgnt: %s\n", def_intf->keymgnt);
			}
			break;
		default:
			DPRINT_WARNING(WFA_WNG, "unknown keyMgmtType: %s\n",
				hostapd_key_mgt_2_string(def_intf->keyMgmtType[bss_idx]));
			break;
		}

		if (def_intf->ft_bss_list[0]) {
			char *list = def_intf->ft_bss_list;
			char kh[150];

			strcpy(kh, list);
			if (strcasecmp(def_intf->cmd_name, "ap1mbo") == 0) {
				strcat(kh, " ");
				strcat(kh, list);
				strcat(kh, " 000102030405060708090a0b0c0d0e0f");
				fprintf(f, "r1kh=%s\n", kh);
				strcpy(kh, list);
				strcat(kh, " ap2.mtk.com 000102030405060708090a0b0c0d0e0f");
				fprintf(f, "r0kh=%s\n", kh);
			} else if (strcasecmp(def_intf->cmd_name, "ap2mbo") == 0) {
				strcat(kh, " ");
				strcat(kh, list);
				strcat(kh, " 000102030405060708090a0b0c0d0e0f");
				fprintf(f, "r1kh=%s\n", kh);
				strcpy(kh, list);
				strcat(kh, " ap1.mtk.com 000102030405060708090a0b0c0d0e0f");
				fprintf(f, "r0kh=%s\n", kh);
			}
		}

		switch (def_intf->pmf) {
		case WFA_DISABLED:
			/* do nothing */
			break;
		case WFA_OPTIONAL:
			fprintf(f, "ieee80211w=1\n");
			if (def_intf->keyMgmtType[bss_idx] == KEY_MGNT_TYPE_WPA2_PSK_SAE)
				fprintf(f, "sae_require_mfp=1\n");
			break;
		case WFA_REQUIRED:
			fprintf(f, "ieee80211w=2\n");
			break;
		default:
			break;
		}

		if (def_intf->trans_disable)
			fprintf(f, "transition_disable=0x%x\n", def_intf->trans_disable);

		if (def_intf->preauthentication)
			fprintf(f, "rsn_preauth=1\n");

		if (def_intf->ap_sae_commit_override[0])
			fprintf(f, "sae_commit_override=%s\n", def_intf->ap_sae_commit_override);

		if (def_intf->antiCloggingThreshold >= 0)
			fprintf(f, "anti_clogging_threshold=%d\n", def_intf->antiCloggingThreshold);

		if (def_intf->ap_reflection)
			fprintf(f, "sae_reflection_attack=1\n");

		if (strlen(def_intf->auth_server_addr)) {
			fprintf(f, "auth_server_addr=%s\n", def_intf->auth_server_addr);
			DPRINT_INFO(WFA_OUT, "auth_server_addr=%s\n", def_intf->auth_server_addr);
		}

		if (strlen(def_intf->auth_server_port)) {
			fprintf(f, "auth_server_port=%s\n", def_intf->auth_server_port);
			DPRINT_INFO(WFA_OUT, "auth_server_port=%s\n", def_intf->auth_server_port);
		}

		if (strlen(def_intf->auth_server_shared_secret)) {
			fprintf(f, "auth_server_shared_secret=%s\n", def_intf->auth_server_shared_secret);
			DPRINT_INFO(WFA_OUT, "auth_server_shared_secret=%s\n", def_intf->auth_server_shared_secret);
		}

		if (def_intf->KeyRotation) {
			unsigned char rekey_intv = 30;

			if (def_intf->KeyRotationInterval)
				rekey_intv = def_intf->KeyRotationInterval;

			fprintf(f, "wpa_group_rekey=%d\n", rekey_intv);
			DPRINT_INFO(WFA_OUT, "wpa_group_rekey=%d\n", rekey_intv);
		}

		if (def_intf->KeyRotation_BIGTK_STADisassoc) {
			fprintf(f, "wpa_strict_rekey=%d\n", 1);
			DPRINT_INFO(WFA_OUT, "wpa_strict_rekey=%d\n", 1);
		}

		if ((def_intf->program == PROGRAME_TYPE_HS2_R2 ||
			def_intf->program == PROGRAME_TYPE_HS2_R3) && def_intf->ppr.proxy_arp) {
			/* ap_isolate must be enabled if proxy arp enabled */
			fprintf(f, "ap_isolate=1\n");
		} else
			fprintf(f, "ap_isolate=%d\n", (int)(def_intf->ap_isolate));
		fprintf(f, "bss_load_update_period=%d\n", (int)(def_intf->bss_load_update_period));
		fprintf(f, "chan_util_avg_period=%d\n", (int)(def_intf->chan_util_avg_period));
		fprintf(f, "disassoc_low_ack=%d\n", (int)(def_intf->disassoc_low_ack));
		fprintf(f, "skip_inactivity_poll=%d\n", (int)(def_intf->skip_inactivity_poll));
		fprintf(f, "preamble=%d\n", (int)(def_intf->preamble));
		if (strcasecmp(def_intf->keymgnt, "OWE") != 0)
			fprintf(f, "ignore_broadcast_ssid=%d\n", (int)(def_intf->ignore_broadcast_ssid));
		fprintf(f, "uapsd_advertisement_enabled=%d\n", (int)(def_intf->uapsd_advertisement_enabled));
		fprintf(f, "utf8_ssid=%d\n", (int)(def_intf->utf8_ssid));
		fprintf(f, "multi_ap=%d\n", (int)(def_intf->multi_ap));
		fprintf(f, "auth_algs=%d\n", (int)(def_intf->auth_algs));
		/* only MT76 support unsol_bcast_probe_resp and enable unsol_bcast_probe_resp when 6g band only*/
		if (def_intf->unsol_bcast_probe_resp_interval && !l1_valid && def_intf->intf_6G_only)
			fprintf(f, "unsol_bcast_probe_resp_interval=%d\n", (int)(def_intf->unsol_bcast_probe_resp_interval));
		if (def_intf->rnr)
			fprintf(f, "rnr=%d\n", (int)(def_intf->rnr));

		fprintf(f, "bridge=br-lan\n");

		if (sta_qos->ACM_BE || sta_qos->ACM_BK || sta_qos->ACM_VI || sta_qos->ACM_VO || sta_qos->AIFS_BE ||
		    sta_qos->AIFS_BK || sta_qos->AIFS_VI || sta_qos->AIFS_VO || sta_qos->cwmax_BE ||
		    sta_qos->cwmax_BK || sta_qos->cwmax_VI || sta_qos->cwmax_VO || sta_qos->cwmin_BE ||
		    sta_qos->cwmin_BK || sta_qos->cwmin_VI || sta_qos->cwmin_VO) {
			fprintf(f, "wmm_ac_bk_aifs=%d\n", sta_qos->AIFS_BK);
			fprintf(f, "wmm_ac_bk_cwmin=%d\n", sta_qos->cwmin_BK);
			fprintf(f, "wmm_ac_bk_cwmax=%d\n", sta_qos->cwmax_BK);
			fprintf(f, "wmm_ac_bk_txop_limit=%d\n", 0);
			fprintf(f, "wmm_ac_bk_acm=%d\n", sta_qos->ACM_BK);

			fprintf(f, "wmm_ac_be_aifs=%d\n", sta_qos->AIFS_BE);
			fprintf(f, "wmm_ac_be_cwmin=%d\n", sta_qos->cwmin_BE);
			fprintf(f, "wmm_ac_be_cwmax=%d\n", sta_qos->cwmax_BE);
			fprintf(f, "wmm_ac_be_txop_limit=%d\n", 0);
			fprintf(f, "wmm_ac_be_acm=%d\n", sta_qos->ACM_BE);

			fprintf(f, "wmm_ac_vi_aifs=%d\n", sta_qos->AIFS_VI);
			fprintf(f, "wmm_ac_vi_cwmin=%d\n", sta_qos->cwmin_VI);
			fprintf(f, "wmm_ac_vi_cwmax=%d\n", sta_qos->cwmax_VI);
			fprintf(f, "wmm_ac_vi_txop_limit=%d\n", 94);
			fprintf(f, "wmm_ac_vi_acm=%d\n", sta_qos->ACM_VI);

			fprintf(f, "wmm_ac_vo_aifs=%d\n", sta_qos->AIFS_VO);
			fprintf(f, "wmm_ac_vo_cwmin=%d\n", sta_qos->cwmin_VO);
			fprintf(f, "wmm_ac_vo_cwmax=%d\n", sta_qos->cwmax_VO);
			fprintf(f, "wmm_ac_vo_txop_limit=%d\n", 47);
			fprintf(f, "wmm_ac_vo_acm=%d\n", sta_qos->ACM_VO);
		}


		if ((def_intf->program == PROGRAME_TYPE_MBO) || (def_intf->mode == WIFI_6G)) {
			/* MBO-OCE IE */
			fprintf(f, "oce=%d\n", def_intf->oce);
			fprintf(f, "mbo=%d\n", def_intf->mbo);
		}
		if ((def_intf->program == PROGRAME_TYPE_MBO) || (def_intf->mode == WIFI_6G)) {
			/* 11k */
			fprintf(f, "rrm_neighbor_report=%d\n", 1);
			fprintf(f, "rrm_beacon_report=%d\n", 1);
			/* 11v N/A */
			/* 11r */
			if (def_intf->ft_oa && l1_valid) {
				fprintf(f, "mobility_domain=0101\n");
				fprintf(f, "pmk_r1_push=1\n");
				fprintf(f, "ft_r0_key_lifetime=10000\n");
				fprintf(f, "r1_key_holder=%s\n", def_intf->mac_addr);
				if (strcasecmp(def_intf->cmd_name, "ap1mbo") == 0)
					fprintf(f, "nas_identifier=ap2.mtk.com\n");
				else
					fprintf(f, "nas_identifier=ap1.mtk.com\n");
				fprintf(f, "rkh_pull_timeout=100\n");
				fprintf(f, "rkh_pull_retries=4\n");

			} else {
				fprintf(f, "mobility_domain=ea91\n");
				fprintf(f, "nas_identifier=000c432a57ab\n");
				fprintf(f, "ft_psk_generate_local=%d\n", 1);
			}
			fprintf(f, "ft_over_ds=%d\n", def_intf->ft_oa ? 0 : 1);
			fprintf(f, "reassociation_deadline=%d\n", 1000);
			/* Interworking IE & Extented Capabilities bit31 */
			fprintf(f, "interworking=%d\n", 1);
			/* BSS Transition Management IE & Extended Capabilities bit19 */
			fprintf(f, "bss_transition=%d\n", 1);
			/* Country Information IE */
			if (def_intf->country_code[0] == '\0')
				snprintf(def_intf->country_code, sizeof(def_intf->country_code), "US");
			fprintf(f, "country3=0x04\n");

			/* MBO required configuration for each bss once no ft_bss_list available */
			if (def_intf->program == PROGRAME_TYPE_MBO) {
				if ((def_intf->ft_bss_list[0] == 0) && def_intf->ft_oa && l1_valid) {
					fprintf(f, "r1kh=00:00:00:00:00:00 00:00:00:00:00:00 000102030405060708090a0b0c0d0e0f\n");
					fprintf(f, "r0kh=ff:ff:ff:ff:ff:ff * 000102030405060708090a0b0c0d0e0f\n");
				}
			}
		}

		if (def_intf->program == PROGRAME_TYPE_HE) {
			if ((!l1_valid) && (def_intf->country_code[0] == '\0')) {
				if (def_intf->mode == WIFI_6G)
					snprintf(def_intf->country_code, sizeof(def_intf->country_code), "DE");
				else {
					/* HE-4.60.1, ch100, no DFS purpose */
					snprintf(def_intf->country_code, sizeof(def_intf->country_code), "ZA");
				}
			}

			if (!l1_valid)
				fprintf(f, "mu_onoff=13\n"); /* Default enable mu dl/ul and ru dl */

			/* For OFDMA, pe duration: 16us */
			fprintf(f, "he_default_pe_duration=4\n");

			/* For UL OFDMA, MU EDCA */
			fprintf(f, "he_mu_edca_ac_be_aifsn=0\n");
			fprintf(f, "he_mu_edca_ac_be_ecwmin=15\n");
			fprintf(f, "he_mu_edca_ac_be_ecwmax=15\n");
			fprintf(f, "he_mu_edca_ac_be_timer=255\n");
			fprintf(f, "he_mu_edca_ac_bk_aifsn=0\n");
			fprintf(f, "he_mu_edca_ac_bk_aci=1\n");
			fprintf(f, "he_mu_edca_ac_bk_ecwmin=15\n");
			fprintf(f, "he_mu_edca_ac_bk_ecwmax=15\n");
			fprintf(f, "he_mu_edca_ac_bk_timer=255\n");
			fprintf(f, "he_mu_edca_ac_vi_aifsn=0\n");
			fprintf(f, "he_mu_edca_ac_vi_aci=2\n");
			fprintf(f, "he_mu_edca_ac_vi_ecwmin=15\n");
			fprintf(f, "he_mu_edca_ac_vi_ecwmax=15\n");
			fprintf(f, "he_mu_edca_ac_vi_timer=255\n");
			fprintf(f, "he_mu_edca_ac_vo_aifsn=0\n");
			fprintf(f, "he_mu_edca_ac_vo_aci=3\n");
			fprintf(f, "he_mu_edca_ac_vo_ecwmin=15\n");
			fprintf(f, "he_mu_edca_ac_vo_ecwmax=15\n");
			fprintf(f, "he_mu_edca_ac_vo_timer=255\n");
		}

#if 1
		if (def_intf->program == PROGRAME_TYPE_WPA3 && !def_intf->ft_ds) {
			/* Beacon protection Enable */
			fprintf(f, "beacon_prot=1\n");

			/* Operating channel validation */
			fprintf(f, "ocv=2\n");
		}
#endif


		if (def_intf->program == PROGRAME_TYPE_HS2 ||
			def_intf->program == PROGRAME_TYPE_HS2_R2 ||
			def_intf->program == PROGRAME_TYPE_HS2_R3) {
			int param_idx = 0;


			fprintf(f, "hs20=%d\n", def_intf->ppr.hs20[bss_idx]);
			fprintf(WFA_OUT, "hs20=%d\n", def_intf->ppr.hs20[bss_idx]);

			fprintf(f, "interworking=%d\n", def_intf->ppr.interworking[bss_idx]);
			fprintf(WFA_OUT, "interworking=%d\n", def_intf->ppr.interworking[bss_idx]);

			if (def_intf->ppr.hs20[bss_idx]) {

				fprintf(f, "hs20_release=%d\n", def_intf->ppr.hs20_release);
				fprintf(WFA_OUT, "hs20_release=%d\n", def_intf->ppr.hs20_release);

				fprintf(f, "manage_p2p=%d\n", def_intf->ppr.manage_p2p);
				fprintf(WFA_OUT, "manage_p2p=%d\n", def_intf->ppr.manage_p2p);

				/* fprintf(f, "ht_bw=%d\n", def_intf->ppr.ht_bw); */
				/* fprintf(f, "vht_bw=%d\n", def_intf->ppr.vht_bw); */

				fprintf(f, "access_network_type=%d\n", def_intf->ppr.access_network_type);
				fprintf(WFA_OUT, "access_network_type=%d\n", def_intf->ppr.access_network_type);

				fprintf(f, "internet=%d\n", def_intf->ppr.internet);
				fprintf(WFA_OUT, "internet=%d\n", def_intf->ppr.internet);

				fprintf(f, "venue_group=%d\n", def_intf->ppr.venue_group);
				fprintf(WFA_OUT, "venue_group=%d\n", def_intf->ppr.venue_group);

				fprintf(f, "venue_type=%d\n", def_intf->ppr.venue_type);
				fprintf(WFA_OUT, "venue_type=%d\n", def_intf->ppr.venue_type);

				if (strlen(def_intf->ppr.venue_name)) {
					fprintf(f, "venue_name=%s\n", def_intf->ppr.venue_name);
					fprintf(WFA_OUT, "venue_name=%s\n", def_intf->ppr.venue_name);
				}
				if (strlen(def_intf->ppr.hessid)) {
					fprintf(f, "hessid=%s\n", def_intf->ppr.hessid);
					fprintf(WFA_OUT, "hessid=%s\n", def_intf->ppr.hessid);
				}

				/*
				* fprintf(f, "advertisement_proto_id=%d\n", def_intf->ppr.advertisement_proto_id);
				* fprintf(WFA_OUT, "advertisement_proto_id=%d\n", def_intf->ppr.advertisement_proto_id);
				*/

				if (strlen(def_intf->ppr.domain_name)) {
					fprintf(f, "domain_name=%s\n", def_intf->ppr.domain_name);
					fprintf(WFA_OUT, "domain_name=%s\n", def_intf->ppr.domain_name);
				}
				if (strlen(def_intf->ppr.network_auth_type)) {
					fprintf(f, "network_auth_type=%s\n", def_intf->ppr.network_auth_type);
					fprintf(WFA_OUT, "network_auth_type=%s\n", def_intf->ppr.network_auth_type);
				}
				if (strlen(def_intf->ppr.operator_icon)) {
					fprintf(f, "operator_icon=%s\n", def_intf->ppr.operator_icon);
					fprintf(WFA_OUT, "operator_icon=%s\n", def_intf->ppr.operator_icon);
				}
				if (strlen(def_intf->ppr.osu_ssid)) {
					fprintf(f, "osu_ssid=%s\n", def_intf->ppr.osu_ssid);
					fprintf(WFA_OUT, "osu_ssid=%s\n", def_intf->ppr.osu_ssid);
				}
				if (strlen(def_intf->ppr.hs20_oper_friendly_name)) {
					fprintf(f, "hs20_oper_friendly_name=%s\n", def_intf->ppr.hs20_oper_friendly_name);
					fprintf(WFA_OUT, "hs20_oper_friendly_name=%s\n", def_intf->ppr.hs20_oper_friendly_name);
				}
				if (strlen(def_intf->ppr.hs20_wan_metrics)) {
					fprintf(f, "hs20_wan_metrics=%s\n", def_intf->ppr.hs20_wan_metrics);
					fprintf(WFA_OUT, "hs20_wan_metrics=%s\n", def_intf->ppr.hs20_wan_metrics);
				}
				/*
				* fprintf(f, "hs20_operating_class=%d\n", def_intf->ppr.hs20_operating_class);
				* fprintf(WFA_OUT, "hs20_operating_class=%d\n", def_intf->ppr.hs20_operating_class);
				*/
				if (strlen(def_intf->ppr.hs20_t_c_filename)) {
					fprintf(f, "hs20_t_c_filename=%s\n", def_intf->ppr.hs20_t_c_filename);
					fprintf(WFA_OUT, "hs20_t_c_filename=%s\n", def_intf->ppr.hs20_t_c_filename);
				}
				if (strlen(def_intf->ppr.hs20_t_c_server_url)) {
					fprintf(f, "hs20_t_c_server_url=%s\n", def_intf->ppr.hs20_t_c_server_url);
					fprintf(WFA_OUT, "hs20_t_c_server_url=%s\n", def_intf->ppr.hs20_t_c_server_url);
				}
				if (strlen(def_intf->ppr.hs20_t_c_timestamp)) {
					fprintf(f, "hs20_t_c_timestamp=%s\n", def_intf->ppr.hs20_t_c_timestamp);
					fprintf(WFA_OUT, "hs20_t_c_timestamp=%s\n", def_intf->ppr.hs20_t_c_timestamp);
				}
				if (strlen(def_intf->ppr.ipaddr_type_availability)) {
					fprintf(f, "ipaddr_type_availability=%s\n", def_intf->ppr.ipaddr_type_availability);
					fprintf(WFA_OUT, "ipaddr_type_availability=%s\n", def_intf->ppr.ipaddr_type_availability);
				}
				fprintf(f, "disable_dgaf=%d\n", def_intf->ppr.disable_dgaf);
				fprintf(WFA_OUT, "disable_dgaf=%d\n", def_intf->ppr.disable_dgaf);

				fprintf(f, "proxy_arp=%d\n", def_intf->ppr.proxy_arp);
				fprintf(WFA_OUT, "proxy_arp=%d\n", def_intf->ppr.proxy_arp);

				/* fprintf(f, "l2_filter=%d\n", def_intf->ppr.l2_filter); */
				/* fprintf(f, "icmpv4_deny=%d\n", def_intf->ppr.icmpv4_deny); */
				if (def_intf->ppr.mmpdu_size) {
					fprintf(f, "mmpdu_size=%d\n", def_intf->ppr.mmpdu_size);
					fprintf(WFA_OUT, "mmpdu_size=%d\n", def_intf->ppr.mmpdu_size);
				}

				fprintf(f, "external_anqp_server_test=%d\n", def_intf->ppr.external_anqp_server_test);
				fprintf(WFA_OUT, "external_anqp_server_test=%d\n", def_intf->ppr.external_anqp_server_test);

				if (def_intf->ppr.gas_comeback_delay) {
					fprintf(f, "gas_comeback_delay=%d\n", def_intf->ppr.gas_comeback_delay);
					fprintf(WFA_OUT, "gas_comeback_delay=%d\n", def_intf->ppr.gas_comeback_delay);
				}
				if (def_intf->ppr.gas_frag_limit) {
					fprintf(f, "gas_frag_limit=%d\n", def_intf->ppr.gas_frag_limit);
					fprintf(WFA_OUT, "gas_frag_limit=%d\n", def_intf->ppr.gas_frag_limit);
				}
				if (def_intf->ppr.anqp_3gpp_cell_net) {
					fprintf(f, "anqp_3gpp_cell_net=%s\n", def_intf->ppr.anqp_3gpp_cell_net);
					fprintf(WFA_OUT, "anqp_3gpp_cell_net=%s\n", def_intf->ppr.anqp_3gpp_cell_net);
				}
				if (def_intf->ppr.anqp_domain_id) {
					fprintf(f, "anqp_domain_id=%d\n", def_intf->ppr.anqp_domain_id);
					fprintf(WFA_OUT, "anqp_domain_id=%d\n", def_intf->ppr.anqp_domain_id);
				}
				if (def_intf->ppr.hs2_openmode_test) {
					fprintf(f, "hs2_openmode_test=%d\n", def_intf->ppr.hs2_openmode_test);
					fprintf(WFA_OUT, "hs2_openmode_test=%d\n", def_intf->ppr.hs2_openmode_test);
				}
				if (strlen(def_intf->ppr.osu_server_uri)) {
					fprintf(f, "osu_server_uri=%s\n", def_intf->ppr.osu_server_uri);
					fprintf(WFA_OUT, "osu_server_uri=%s\n", def_intf->ppr.osu_server_uri);
				}
				if (def_intf->ppr.legacy_osu) {
					fprintf(f, "legacy_osu=%d\n", def_intf->ppr.legacy_osu);
					fprintf(WFA_OUT, "legacy_osu=%d\n", def_intf->ppr.legacy_osu);
				}
				if (def_intf->ppr.qosmap) {
					fprintf(f, "qosmap=%d\n", def_intf->ppr.qosmap);
					fprintf(WFA_OUT, "qosmap=%d\n", def_intf->ppr.qosmap);
				}
				if (strlen(def_intf->ppr.qos_map_set)) {
					fprintf(f, "qos_map_set=%s\n", def_intf->ppr.qos_map_set);
					fprintf(WFA_OUT, "qos_map_set=%s\n", def_intf->ppr.qos_map_set);
				}
				if (def_intf->ppr.dscp_range) {
					fprintf(f, "dscp_range=%d\n", def_intf->ppr.dscp_range);
					fprintf(WFA_OUT, "dscp_range=%d\n", def_intf->ppr.dscp_range);
				}
				if (def_intf->ppr.dscp_exception) {
					fprintf(f, "dscp_exception=%d\n", def_intf->ppr.dscp_exception);
					fprintf(WFA_OUT, "dscp_exception=%d\n", def_intf->ppr.dscp_exception);
				}
				if (def_intf->ppr.qload_test) {
					fprintf(f, "qload_test=%d\n", def_intf->ppr.qload_test);
					fprintf(WFA_OUT, "qload_test=%d\n", def_intf->ppr.qload_test);
				}
				if (def_intf->ppr.qload_cu) {
					fprintf(f, "qload_cu=%d\n", def_intf->ppr.qload_cu);
					fprintf(WFA_OUT, "qload_cu=%d\n", def_intf->ppr.qload_cu);
				}
				if (def_intf->ppr.qload_sta_cnt) {
					fprintf(f, "qload_sta_cnt=%d\n", def_intf->ppr.qload_sta_cnt);
					fprintf(WFA_OUT, "qload_sta_cnt=%d\n", def_intf->ppr.qload_sta_cnt);
				}
				if (def_intf->ppr.icon_tag) {
					fprintf(f, "icon_tag=%d\n", def_intf->ppr.icon_tag);
					fprintf(WFA_OUT, "icon_tag=%d\n", def_intf->ppr.icon_tag);
				}

				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (strlen(def_intf->ppr.hs20_icon[param_idx])) {
						fprintf(f, "hs20_icon=%s\n", def_intf->ppr.hs20_icon[param_idx]);
						fprintf(WFA_OUT, "hs20_icon=%s\n", def_intf->ppr.hs20_icon[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (strlen(def_intf->ppr.roaming_consortium[param_idx])) {
						fprintf(f, "roaming_consortium=%s\n", def_intf->ppr.roaming_consortium[param_idx]);
						fprintf(WFA_OUT, "roaming_consortium=%s\n", def_intf->ppr.roaming_consortium[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (def_intf->ppr.osu_method_list[param_idx]) {
						fprintf(f, "osu_method_list=%d\n", def_intf->ppr.osu_method_list[param_idx]);
						fprintf(WFA_OUT, "osu_method_list=%d\n", def_intf->ppr.osu_method_list[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (strlen(def_intf->ppr.osu_icon[param_idx])) {
						fprintf(f, "osu_icon=%s\n", def_intf->ppr.osu_icon[param_idx]);
						fprintf(WFA_OUT, "osu_icon=%s\n", def_intf->ppr.osu_icon[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (strlen(def_intf->ppr.osu_nai[param_idx])) {
						fprintf(f, "osu_nai=%s\n", def_intf->ppr.osu_nai[param_idx]);
						fprintf(WFA_OUT, "osu_nai=%s\n", def_intf->ppr.osu_nai[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (strlen(def_intf->ppr.osu_nai2[param_idx])) {
						fprintf(f, "osu_nai2=%s\n", def_intf->ppr.osu_nai2[param_idx]);
						fprintf(WFA_OUT, "osu_nai2=%s\n", def_intf->ppr.osu_nai2[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 2; param_idx++) {
					if (strlen(def_intf->ppr.osu_service_desc[param_idx])) {
						fprintf(f, "osu_service_desc=%s\n", def_intf->ppr.osu_service_desc[param_idx]);
						fprintf(WFA_OUT, "osu_service_desc=%s\n", def_intf->ppr.osu_service_desc[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 4; param_idx++) {
					if (strlen(def_intf->ppr.nai_realm[param_idx])) {
						fprintf(f, "nai_realm=%s\n", def_intf->ppr.nai_realm[param_idx]);
						fprintf(WFA_OUT, "nai_realm=%s\n", def_intf->ppr.nai_realm[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 4; param_idx++) {
					if (strlen(def_intf->ppr.osu_friendly_name[param_idx])) {
						fprintf(f, "osu_friendly_name=%s\n", def_intf->ppr.osu_friendly_name[param_idx]);
						fprintf(WFA_OUT, "osu_friendly_name=%s\n", def_intf->ppr.osu_friendly_name[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 11; param_idx++) {
					if (strlen(def_intf->ppr.hs20_conn_capab[param_idx])) {
						fprintf(f, "hs20_conn_capab=%s\n", def_intf->ppr.hs20_conn_capab[param_idx]);
						fprintf(WFA_OUT, "hs20_conn_capab=%s\n", def_intf->ppr.hs20_conn_capab[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 10; param_idx++) {
					if (strlen(def_intf->ppr.venue_url[param_idx])) {
						fprintf(f, "venue_url=%s\n", def_intf->ppr.venue_url[param_idx]);
						fprintf(WFA_OUT, "venue_url=%s\n", def_intf->ppr.venue_url[param_idx]);
					}
				}
				for (param_idx = 0; param_idx < 3; param_idx++) {
					if (strlen(def_intf->ppr.advice_of_charge[param_idx])) {
						fprintf(f, "advice_of_charge=%s\n", def_intf->ppr.advice_of_charge[param_idx]);
						fprintf(WFA_OUT, "advice_of_charge=%s\n", def_intf->ppr.advice_of_charge[param_idx]);
					}
				}
			}
		}

		if (def_intf->country_code[0] != '\0') {
			fprintf(f, "country_code=%s\n", def_intf->country_code);
			DPRINT_INFO(WFA_OUT, "country_code=%s\n", def_intf->country_code);
			fprintf(f, "ieee80211h=%d\n", 1);
			fprintf(f, "ieee80211d=%d\n", 1);
			fprintf(f, "local_pwr_constraint=%d\n", 3);
			fprintf(f, "spectrum_mgmt_required=%d\n", 1);
		}

		if (!l1_valid) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
				 def_intf->name, def_intf->name, HOSTAPD_TEMP_OUTPUT);

			if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
				DPRINT_INFO(WFA_OUT, "no bssid found\n");
			if (bss_idx) {
				unsigned int mac_hex[6] = {0};

				/* last bytes plus one for multi bss */
				ret_len = sscanf(shellResult, "%02x:%02x:%02x:%02x:%02x:%02x", &mac_hex[0], &mac_hex[1],
						 &mac_hex[2], &mac_hex[3], &mac_hex[4], &mac_hex[5]);
				if (ret_len != 6)
					DPRINT_INFO(WFA_OUT, "sscanf of BSSID not match\n");
				mac_hex[5] += 1;
				snprintf(shellResult, sizeof(shellResult), "%02x:%02x:%02x:%02x:%02x:%02x", mac_hex[0],
					 mac_hex[1], mac_hex[2], mac_hex[3], mac_hex[4], mac_hex[5]);
			}

			if (strcasecmp(def_intf->keymgnt, "OWE") != 0) {
				fprintf(f, "bssid=%s\n", shellResult);
				DPRINT_INFO(WFA_OUT, "bssid=%s\n", shellResult);
			}
		}
	}

	if (f != NULL)
		fclose(f);

	snprintf(gCmdStr, sizeof(gCmdStr), "chmod 0777 %s", hapd_profile);
	if (system(gCmdStr) != 0)
		DPRINT_INFO(WFA_OUT, "chmod 0777 %s fail\n", hapd_profile);

	DPRINT_INFO(WFA_OUT, "chmod 0777 %s success\n", hapd_profile);
	DPRINT_INFO(WFA_OUT, "*********************************\n");

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

void hostapd_handle_special_cmd(mtk_ap_buf_t *mtk_ap_buf)
{
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	int offset = 0;
	unsigned char phy_id = 0;

	if (def_intf->program == PROGRAME_TYPE_HS2_R2 ||
			def_intf->program == PROGRAME_TYPE_HS2_R3) {
		if (def_intf->mode == WIFI_6G)
			phy_id = 2;
		else if (def_intf->mode == WIFI_5G)
			phy_id = 1;
		else
			phy_id = 0;

		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "mwctl phy phy%d set channel bw=20", phy_id);

		add_post_cmd(mtk_ap_buf);
	}

	if ((def_intf->spatialRxStream < 1) || (def_intf->spatialRxStream > 4)) {
		DPRINT_INFO(WFA_OUT, "%s(), skip to set tx rx stream\n", __func__);
		return;
	}

	memset(gCmdStr, 0, sizeof(gCmdStr));

	if (mtk_ap_buf->intern_flag.capi_dual_pf) {
		if (def_intf->spatialRxStream == 1) {
			DPRINT_INFO(WFA_OUT, "%s(), spacial tx rx stream 1\n", __func__);
			offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset, "iw phy phy0 set antenna 1 1;");

			offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset, "iw phy phy0 set antenna 2 2;");

			offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset, "iw phy phy1 set antenna 1 1;");

			offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset, "iw phy phy1 set antenna 2 2");
		} else if (def_intf->spatialRxStream == 2) {
			DPRINT_INFO(WFA_OUT, "%s(), spacial tx rx stream 2\n", __func__);

			offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset, "iw phy phy0 set antenna 3 3;");

			offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset, "iw phy phy1 set antenna 3 3");
		}
	} else {
		DPRINT_INFO(WFA_OUT, "%s(), spacial rx stream %d tx stream %d\n", __func__, def_intf->spatialRxStream,
			    def_intf->spatialTxStream);
		if (def_intf->mode == WIFI_2G) {
			if (def_intf->spatialRxStream == 1) {
				DPRINT_INFO(WFA_OUT, "%s(), spacial tx rx stream 1\n", __func__);
				offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset,
						   "iw phy phy%d set antenna 1 1;", def_intf->phy_no);

				offset +=
					snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset,
					    "iw phy phy%d set antenna 2 2", def_intf->phy_no);
			} else if (def_intf->spatialRxStream == 2) {
				DPRINT_INFO(WFA_OUT, "%s(), spacial tx rx stream 2\n", __func__);

				offset +=
					snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset,
					    "iw phy phy%d set antenna 3 3", def_intf->phy_no);
			}
		} else {
			if (def_intf->spatialRxStream == 1) {
				DPRINT_INFO(WFA_OUT, "%s(), spacial tx rx stream 1\n", __func__);

				offset += snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset,
						   "iw phy phy%d set antenna 1 1;", def_intf->phy_no);

				offset +=
					snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset,
					    "iw phy phy%d set antenna 2 2", def_intf->phy_no);
			} else if (def_intf->spatialRxStream == 2) {
				DPRINT_INFO(WFA_OUT, "%s(), spacial tx rx stream 2\n", __func__);
				snprintf(gCmdStr, sizeof(gCmdStr), "iw phy phy%d set antenna 3 3;", def_intf->phy_no);
				offset +=
					snprintf(gCmdStr + offset, sizeof(gCmdStr) - offset,
					    "iw phy phy%d set antenna 3 3", def_intf->phy_no);
			}
		}
	}

	system(gCmdStr);
	DPRINT_INFO(WFA_OUT, "run cmd===>%s\n", gCmdStr);
	sleep(2);
}

int hostapd_turn_interface_down(int skfd, char *ifname, void *args, int count)
{
	struct ifreq intf_req;
	int sock_fd = 0;

	sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
	bzero(&intf_req, sizeof(intf_req));
	strncpy(intf_req.ifr_name, ifname, IFNAMSIZ-1);
	ioctl(sock_fd, SIOCGIFFLAGS, &intf_req);
	intf_req.ifr_flags &= ~IFF_UP;
	DPRINT_INFO(WFA_OUT, "turn interface:%s down!\n", ifname);
	ioctl(sock_fd, SIOCSIFFLAGS, &intf_req);
	close(sock_fd);

	return 0;
}


int hostapd_turn_interface_up(int skfd, char *ifname, void *args, int count)
{
	struct ifreq intf_req;
	int sock_fd = 0;

	DPRINT_INFO(WFA_OUT, "turn interface:%s up(may take a few secs)!\n", ifname);
	sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
	bzero(&intf_req, sizeof(intf_req));
	strncpy(intf_req.ifr_name, ifname, IFNAMSIZ-1);
	ioctl(sock_fd, SIOCGIFFLAGS, &intf_req);
	intf_req.ifr_flags |= IFF_UP;
	ioctl(sock_fd, SIOCSIFFLAGS, &intf_req);
	DPRINT_INFO(WFA_OUT, "turn interface:%s up !complete!\n", ifname);
	close(sock_fd);

	return 0;
}

int hostapd_wifi_enum_devices(int skfd, iw_enum_handler fn, void *args, int reset_stat)
{
	char buff[1024];
	FILE *file = NULL;
	int intf_found = 0;
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)args;
	int ret = WFA_SUCCESS;

	if (reset_stat) {
		mtk_ap_buf->intf_2G.status = 0;
		mtk_ap_buf->intf_5G.status = 0;
	}

	file = fopen(HOSTAPD_WIRELESS_INTF_FILE, "r");

	if (file) {
		while (fgets(buff, sizeof(buff), file)) {
			if ((buff[0] == '\0') || (buff[1] == '\0'))
				continue;

			buff[strlen(buff)-1] = '\0';
			DPRINT_INFO(WFA_OUT, "**************************************\n");
			DPRINT_INFO(WFA_OUT, "intf name: %s\n", buff);
			intf_found = 1;
			(*fn)(skfd, buff, args, 0);
			DPRINT_INFO(WFA_OUT, "**************************************\n");
		}

		fclose(file);

		if (intf_found)
			ret = WFA_SUCCESS;
		else {
			DPRINT_INFO(WFA_OUT, "!!!Can't find interface entry in %s!\n", HOSTAPD_WIRELESS_INTF_FILE);
			ret = WFA_ERROR;
		}
	} else {
		DPRINT_INFO(WFA_OUT, "!!!Can't open %s!\n", HOSTAPD_WIRELESS_INTF_FILE);
		ret = WFA_ERROR;
	}

	return ret;
}


void hostapd_remove_conf(mtk_ap_buf_t *mtk_ap_buf)
{
	if (mtk_ap_buf->intf_2G.status) {
		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
			mtk_ap_buf->intf_2G.name);
		DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
		system(gCmdStr);
		sleep(2);
	}
	if (mtk_ap_buf->intf_5G.status) {
		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
			mtk_ap_buf->intf_5G.name);
		DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
		system(gCmdStr);
		sleep(2);
	}
	if (mtk_ap_buf->intf_6G.status) {
		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
			mtk_ap_buf->intf_6G.name);
		DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
		system(gCmdStr);
		sleep(2);
	}
}


void hostapd_add_conf(mtk_ap_buf_t *mtk_ap_buf, intf_desc_t *intfs[3])
{
	int i = 0;

	for (i = 0; i < 3; i++) {
		if (!intfs[i])
			continue;

		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr),
			"ubus call hostapd config_add '{\"iface\":\"%s\",\"config\":\"/var/run/hostapd-phy%d.conf\"}'",
			intfs[i]->name, intfs[i]->phy_no);
		system(gCmdStr);
		DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
		sleep(3);
	}

	DPRINT_INFO(WFA_OUT,
		"\033[1;31m!!!!maybe take 10 secs to apply hostapd conf!!!!\033[0m\n");
	sleep(10);
}


void hostapd_app_bringup(mtk_ap_buf_t *mtk_ap_buf, intf_desc_t *intfs[3])
{
	int i = 0, j = 0, offset = 0;
	intf_desc_t *intf_2g = NULL, *intf_5g = NULL, *intf_6g = NULL;
	intf_desc_t *cmt_2g = NULL, *cmt_5g = NULL, *cmt_6g = NULL;
	unsigned char conf_num = 0;
	unsigned char same_band = 0;

	memset(gCmdStr, 0, sizeof(gCmdStr));
	offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
		"/usr/sbin/hostapd -B -t -dddd -f /root/hostapd.log");

	for (i = 0; i < 3; i++) {
		same_band = 0;
		for (j = 0; j < 3; j++) {
			/* same band, not apply conf.cmt again */
			if (cmt_intfs[j] && intfs[i] &&
				cmt_intfs[j]->mode == intfs[i]->mode) {
				same_band = 1;
			}
		}
		if (intfs[i]) {
			if (intfs[i]->mode == WIFI_2G)
				intf_2g = intfs[i];
			else if (intfs[i]->mode == WIFI_5G)
				intf_5g = intfs[i];
			else
				intf_6g = intfs[i];
		}
		/* append committed config */
		if (cmt_intfs[i] && !same_band) {
			if (cmt_intfs[i]->mode == WIFI_2G)
				cmt_2g = cmt_intfs[i];
			else if (cmt_intfs[i]->mode == WIFI_5G)
				cmt_5g = cmt_intfs[i];
			else
				cmt_6g = cmt_intfs[i];
		}
	}

	/* bring up hostapd with hostapd cert config
	 * once no any valid config
	*/
	if ((!intf_2g) && (!intf_5g) && (!intf_6g)) {
		if (mtk_ap_buf->intf_2G.status) {
			conf_num++;
			offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
				" /var/run/hostapd-phy%d.conf", mtk_ap_buf->intf_2G.phy_no);
			DPRINT_INFO(WFA_OUT, "%s insert 2g default config\n", __func__);
		}
		if (mtk_ap_buf->intf_5G.status) {
			conf_num++;
			offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
				" /var/run/hostapd-phy%d.conf", mtk_ap_buf->intf_5G.phy_no);
			DPRINT_INFO(WFA_OUT, "%s insert 5g default config\n", __func__);
		}
		if (mtk_ap_buf->intf_6G.status) {
			conf_num++;
			offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
				" /var/run/hostapd-phy%d.conf", mtk_ap_buf->intf_6G.phy_no);
			DPRINT_INFO(WFA_OUT, "%s insert 6g default config\n", __func__);
		}
	} else {
		if (intf_2g || cmt_2g) {
			conf_num++;
			if (intf_2g) {
				offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
					" /var/run/hostapd-phy%d.conf", intf_2g->phy_no);
				DPRINT_INFO(WFA_OUT, "%s insert 2g config\n", __func__);
			} else if (cmt_2g) {
				offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
					" /var/run/hostapd-phy%d.conf.cmt", cmt_2g->phy_no);
				DPRINT_INFO(WFA_OUT, "%s insert 2g commtted config\n", __func__);
			}
		}
		if (intf_5g || cmt_5g) {
			conf_num++;
			if (intf_5g) {
				offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
					" /var/run/hostapd-phy%d.conf", intf_5g->phy_no);
				DPRINT_INFO(WFA_OUT, "%s insert 5g config\n", __func__);
			} else if (cmt_5g) {
				offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
					" /var/run/hostapd-phy%d.conf.cmt", cmt_5g->phy_no);
				DPRINT_INFO(WFA_OUT, "%s insert 5g commtted config\n", __func__);
			}
		}
		if (intf_6g || cmt_6g) {
			conf_num++;
			if (intf_6g) {
				offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
					" /var/run/hostapd-phy%d.conf", intf_6g->phy_no);
				DPRINT_INFO(WFA_OUT, "%s insert 6g config\n", __func__);
			} else if (cmt_6g) {
				offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset,
					" /var/run/hostapd-phy%d.conf.cmt", cmt_6g->phy_no);
				DPRINT_INFO(WFA_OUT, "%s insert 6g commtted config\n", __func__);
			}
		}

		/* store intf as TGn will config 2G and 5G separately */
		for (i = 0; i < 3; i++) {
			if (!intfs[i] || (intfs[i]->ap_mode >= AP_MODE_11ac))
				continue;

			for (j = 0; j < 3; j++) {
				if (cmt_intfs[j]) {
					/* do not save again for same band */
					if (cmt_intfs[j]->mode == intfs[i]->mode)
						break;
				} else {
					cmt_intfs[j] = intfs[i];
					break;
				}
			}
		}
	}
	offset += snprintf(gCmdStr+offset, sizeof(gCmdStr)-offset, " &");
	DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
	system(gCmdStr);

	DPRINT_INFO(WFA_OUT,
		"\033[1;31m!!!!maybe take %d secs to bring up hostapd for sigma!!!!\033[0m\n",
		30*conf_num);
	sleep(30*conf_num);
}

void hostapd_apply_config(mtk_ap_buf_t *mtk_ap_buf, intf_desc_t *intfs[3])
{
	if (l1_valid) {
		DPRINT_INFO(WFA_OUT, "non-MT76 kill and bringup hostapd\n");
		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "/etc/init.d/wpad stop;killall hostapd;rm /root/hostapd.log\n");
		DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
		system(gCmdStr);


		/* turn down all interface including both apcli and ap */
		hostapd_wifi_enum_devices(wifi_intf_fd, &hostapd_turn_interface_down, mtk_ap_buf, 0);
	} else {
		DPRINT_INFO(WFA_OUT, "MT76 use ubus to apply config\n");
		hostapd_remove_conf(mtk_ap_buf);
	}

	sleep(3);
	if (l1_valid)
		hostapd_app_bringup(mtk_ap_buf, intfs);
	else
		hostapd_add_conf(mtk_ap_buf, intfs);
}



void hostapd_owe_special_step(uint8_t *ap_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	FILE *f = NULL, *tmp_f = NULL;
	char line[256];
	char match = 0;
	char tmp_profile[PROF_NAME_LEN+4];
	char shellResult[WFA_BUFF_64] = {0};
	char shellResult_interface[WFA_BUFF_64] = {0};
	char shellResult_bss[WFA_BUFF_64] = {0};
	char mbss_intf_name[INTF_NAME_LEN];

	/* OWE MBSS case
	** get Main intf and mbss mac again, and update owe_transition_bssid
	*/

	printf("OWE mbss case, update owe_transition_bssid\n");

	memset(mbss_intf_name, 0, sizeof(mbss_intf_name));
	memcpy(mbss_intf_name, def_intf->name, sizeof(mbss_intf_name));

	snprintf(mbss_intf_name+strlen(def_intf->name)-1, sizeof(def_intf->name), "%d", 1);
	DPRINT_INFO(WFA_OUT, "prepare mbss itf name %s\n", mbss_intf_name);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	memset(shellResult, 0, sizeof(shellResult));

	snprintf(tmp_profile, sizeof(tmp_profile), "%s.tmp", def_intf->profile_names->profile);

	snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
		 def_intf->name, def_intf->name, HOSTAPD_TEMP_OUTPUT);
	if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
		printf("no bssid found\n");

	strncpy(shellResult_interface, shellResult, sizeof(shellResult_interface)-1);
	shellResult_interface[sizeof(shellResult_interface)-1] = '\0';
	shellResult_interface[strlen(shellResult_interface)-1] = '\0';

	DPRINT_INFO(WFA_OUT, "main_bssid=%s\n", shellResult_interface);

	snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
		 mbss_intf_name, mbss_intf_name, HOSTAPD_TEMP_OUTPUT);

	if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
		printf("no bssid found\n");

	strncpy(shellResult_bss, shellResult, sizeof(shellResult_bss)-1);
	shellResult_bss[sizeof(shellResult_bss)-1] = '\0';
	shellResult_bss[strlen(shellResult_bss)-1] = '\0';
	DPRINT_INFO(WFA_OUT, "bss_bssid=%s\n", shellResult_bss);

	f = fopen(def_intf->profile_names->profile, "r");
	tmp_f = fopen(tmp_profile, "w");

	if (!f || !tmp_f) {
		if (f)
			fclose(f);
		if (tmp_f)
			fclose(tmp_f);
		return;
	}
	while (fgets(line, sizeof(line), f) != NULL) {
		if (strstr(line, "owe_transition_bssid=")) {
			match++;
			if (match == 1) {
				snprintf(line, sizeof(line), "owe_transition_bssid=%s", shellResult_bss);
				DPRINT_INFO(WFA_OUT, "update %s owe_transition_bssid=%s\n", def_intf->name, shellResult_bss);
			} else {
				snprintf(line, sizeof(line), "owe_transition_bssid=%s", shellResult_interface);
				DPRINT_INFO(WFA_OUT, "update %s owe_transition_bssid=%s\n", mbss_intf_name, shellResult_interface);
			}
		}
		if ((strstr(line, "ht_capab=")) || (strstr(line, "ht_capab=")))
			continue;
		fputs(line, tmp_f);
	}
	fclose(f);
	fclose(tmp_f);

	sleep(1);
	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr), "cp %s %s; rm %s",
		tmp_profile, def_intf->profile_names->profile, tmp_profile);
	system(gCmdStr);
	DPRINT_INFO(WFA_OUT, "cmd: %s\n", gCmdStr);

	system("sync");
	sleep(1);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr), "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
		 def_intf->name);
	DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
	system(gCmdStr);
	sleep(2);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s down", def_intf->name);
	DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
	system(gCmdStr);
	sleep(2);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr),
		"ubus call hostapd config_add '{\"iface\":\"%s\",\"config\":\"%s\"}'", def_intf->name,
		def_intf->profile_names->profile);
	DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);

	system(gCmdStr);
	sleep(5);
}


int hostapd_ap_config_commit(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	int radio_index = 0, wlan_tag_idx = 0;
	int radio_end_idx = 0;
	intf_desc_t *intfs[3] = {NULL, NULL, NULL};

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);
	DPRINT_INFO(WFA_OUT, "%s def_intf.name=%s\n", __func__, def_intf->name);

	if (mtk_ap_buf->program)
		def_intf->program = mtk_ap_buf->program;

	/* apply sigma profile as no cert dat was copied for PMF TC */
	if (!mtk_ap_buf->intern_flag.cert_dat_copied)
		hostapd_apply_sigma_profile(mtk_ap_buf, mtk_ap_buf->dev_type);

	if (mtk_ap_buf->intern_flag.capi_dual_pf)
		radio_end_idx = 2;
	else if (mtk_ap_buf->WLAN_TAG > 0) {
		for (wlan_tag_idx = 0; wlan_tag_idx < 3; wlan_tag_idx++) {
			if (mtk_ap_buf->WLAN_TAG_inf[wlan_tag_idx])
				radio_end_idx++;
		}
	} else
		radio_end_idx = 1;
	DPRINT_INFO(WFA_OUT, "radio_end_idx [%d]\n", radio_end_idx);

	for (radio_index = 0; radio_index < radio_end_idx; radio_index++) {
		if (mtk_ap_buf->WLAN_TAG_inf[radio_index]) {
			def_intf = mtk_ap_buf->WLAN_TAG_inf[radio_index];
			if (radio_index) {
				/* single band, mbss case */
				if (mtk_ap_buf->WLAN_TAG_inf[radio_index-1]->channel ==
					mtk_ap_buf->WLAN_TAG_inf[radio_index]->channel) {
					/* mbss handled in main bss */
					DPRINT_INFO(WFA_OUT, "single band 2nd bss, skip write and apply config\n");
					break;
				}
			}
		}
		/* dual band case */
		else if (mtk_ap_buf->intern_flag.capi_dual_pf) {
			if (radio_index)
				def_intf = &mtk_ap_buf->intf_2G;
			else
				def_intf = &mtk_ap_buf->intf_5G;
		}
		/* single band sigle bss case */
		else
			def_intf = mtk_ap_buf->def_intf;

		/* reset mtk_ap_buf->def_intf for dual band case */
		mtk_ap_buf->def_intf = def_intf;
		intfs[radio_index] = def_intf;

		switch (def_intf->mode) {
		case WIFI_6G:
			DPRINT_INFO(WFA_OUT, "commit 6G band\n");
			break;
		case WIFI_5G:
			DPRINT_INFO(WFA_OUT, "commit 5G band\n");
			break;
		default:
			DPRINT_INFO(WFA_OUT, "commit 2G band\n");
			break;
		}

		hostapd_handle_special_cmd(mtk_ap_buf);
		sleep(2);

		if (mtk_ap_buf->program)
			def_intf->program = mtk_ap_buf->program;
		def_intf->intf_6G_only = mtk_ap_buf->Band6Gonly.intf_6G_only;
		hostapd_write_conf(def_intf);
		sleep(2);
	}

	hostapd_apply_config(mtk_ap_buf, intfs);

	if (def_intf->program == PROGRAME_TYPE_HS2_R2 ||
			def_intf->program == PROGRAME_TYPE_HS2_R3)
		mtk_ap_buf->cmd_cfg.post_intf_rst_delay = 10;

	if (mtk_ap_buf->cmd_cfg.post_intf_rst_delay) {
		DPRINT_INFO(WFA_OUT, "Delay %d seconds after interfaces turn on!\n",
			    mtk_ap_buf->cmd_cfg.post_intf_rst_delay);
		sleep(mtk_ap_buf->cmd_cfg.post_intf_rst_delay);
	}

	hostapd_handle_post_cmd(mtk_ap_buf);

	if (strcasecmp(mtk_ap_buf->cmd_cfg.program, "QM") == 0) {
		mtk_send_qos_cmd_to_wapp(def_intf->name, "qos_ap_commit_config", NULL, NULL);
		return WFA_SUCCESS;
	}

	/* hairpin_mode=1 make sure success to ping
	 ** between two stas in the same bss
	 */
	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr), "echo 1 > /sys/devices/virtual/net/br-lan/lower_%s/brport/hairpin_mode",
		 def_intf->name);
	DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
	if (system(gCmdStr) != 0)
		DPRINT_INFO(WFA_OUT, "%s fail\n", gCmdStr);

	hostapd_restore_profile(mtk_ap_buf);

	mtk_ap_buf->intern_flag.capi_dual_pf = 0;

	def_intf->sha256ad = 0;
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

static int hostapd_capi_set_intf(mtk_ap_buf_t *mtk_ap_buf)
{
	capi_data_t *data = mtk_ap_buf->capi_data;

	data = mtk_ap_buf->capi_data;

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	DPRINT_INFO(WFA_OUT, "%s() interface %s\n", __func__, data->interface);

	DPRINT_INFO(WFA_OUT, "mtk_ap_buf->intf_2G.status=%d\n", mtk_ap_buf->intf_2G.status);
	DPRINT_INFO(WFA_OUT, "mtk_ap_buf->intf_5G.status=%d\n", mtk_ap_buf->intf_5G.status);
	DPRINT_INFO(WFA_OUT, "mtk_ap_buf->intf_6G.status=%d\n", mtk_ap_buf->intf_6G.status);
	/* set 5G default interface */
	if ((strcasecmp(data->interface, "5G") == 0) || (strcasecmp(data->interface, "5.0") == 0)) {
		if (mtk_ap_buf->intf_5G.status)
			hostapd_set_default_intf(mtk_ap_buf, WIFI_5G);
		else if (mtk_ap_buf->intf_2G.status) {
			/* Single interface, swtich to another band */
			hostapd_set_default_intf(mtk_ap_buf, WIFI_5G);
			strcpy(mtk_ap_buf->intf_5G.name, mtk_ap_buf->intf_2G.name);
			mtk_ap_buf->intf_2G.status = 0;
			DPRINT_INFO(WFA_OUT, "Single interface, Switch 2G to 5G, interface=%s\n",
				    mtk_ap_buf->intf_5G.name);
		} else {
			DPRINT_INFO(WFA_OUT, "No valid interface exist, skip!\n");
		}
	} else if ((strcasecmp(data->interface, "2G") == 0) || (strcasecmp(data->interface, "24G") == 0) ||
		   (strcasecmp(data->interface, "2.4") == 0)) {
		/* set 2G default interface */
		if (mtk_ap_buf->intf_2G.status)
			hostapd_set_default_intf(mtk_ap_buf, WIFI_2G);
		else if (mtk_ap_buf->intf_5G.status) {
			/* Single interface, swtich to another band */
			hostapd_set_default_intf(mtk_ap_buf, WIFI_2G);
			strcpy(mtk_ap_buf->intf_2G.name, mtk_ap_buf->intf_5G.name);
			mtk_ap_buf->intf_5G.status = 0;
			DPRINT_INFO(WFA_OUT, "Single interface, Switch 5G to 2G, interface=%s\n",
				    mtk_ap_buf->intf_2G.name);
		} else
			DPRINT_INFO(WFA_OUT, "No valid interface exist, skip!\n");
	} else if (strcasecmp(data->interface, "6G") == 0) {
		/* set 6G default interface */
		DPRINT_INFO(WFA_OUT, "mtk_ap_buf->intf_6G.name=%s\n", mtk_ap_buf->intf_6G.name);
		if (mtk_ap_buf->profile_num > 2) {
			if (mtk_ap_buf->intf_6G.mode == WIFI_6G) {
				if (!hostapd_assign_profile_pointer_to_intf(
					mtk_ap_buf, &mtk_ap_buf->intf_6G)) {
					DPRINT_INFO(WFA_OUT, "Profile name can't be found, use 6G default!\n");
					return WFA_ERROR;
				}
			}
		} else {
			if (!mtk_ap_buf->intf_6G.status && mtk_ap_buf->intf_5G.status) {
				strcpy(mtk_ap_buf->intf_6G.name, mtk_ap_buf->intf_5G.name);
				mtk_ap_buf->intf_5G.status = 0;
				mtk_ap_buf->intf_2G.status = 1;
				if (!hostapd_assign_profile_pointer_to_intf(
					mtk_ap_buf,	&mtk_ap_buf->intf_6G)) {
					DPRINT_INFO(WFA_OUT, "Profile name can't be found, use 5G default!\n");
					return WFA_ERROR;
				}
			}
		}

		hostapd_set_default_intf(mtk_ap_buf, WIFI_6G);

		DPRINT_INFO(WFA_OUT, "%s mtk_ap_buf->def_intf.name=%s\n", __func__, mtk_ap_buf->def_intf->name);
	} else
		return WFA_ERROR;

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

int hostapd_ap_exec(uint8_t *ap_buf, uint8_t *capi_data, uint8_t *resp_buf, int cmd_len, int *resp_len_ptr, int cmd_tag)
{
	mtk_ap_buf_t *mtk_ap_buf = NULL;
	retType_t status;

	mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	mtk_ap_buf->capi_data = (capi_data_t *)capi_data;

	if (!ap_buf || !capi_data || !resp_buf || !resp_len_ptr) {
		DPRINT_INFO(WFA_OUT, "%s() NULL pointer\n", __func__);
		return WFA_ERROR;
	}

	if (strlen(mtk_ap_buf->capi_data->program) &&
	    (strlen(mtk_ap_buf->capi_data->program) != strlen(mtk_ap_buf->cmd_cfg.program)) &&
	    (strlen(mtk_ap_buf->capi_data->program) < sizeof(mtk_ap_buf->cmd_cfg.program))) {
		strcpy(mtk_ap_buf->cmd_cfg.program, mtk_ap_buf->capi_data->program);
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);
	status = hostapd_dut_tbl[cmd_tag].cmd_mtk(cmd_len, (uint8_t *)mtk_ap_buf, resp_len_ptr, (uint8_t *)resp_buf);
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return status;
}


void hostapd_read_mac_address_file(char *mac_address_buf, char *interface, mtk_ap_buf_t *mtk_ap_buf)
{

	char file_name[100];
	char line[256];
	FILE *file;

	snprintf(file_name, sizeof(file_name), "/sys/class/net/%s/address", interface);
	file = fopen(file_name, "r");
	if (file == NULL) {
		snprintf(gCmdStr, sizeof(gCmdStr), "interface %s not exist:\n", interface);
		snprintf(mac_address_buf, WFA_MAC_ADDR_STR_LEN, gCmdStr);
		return;
	}

	/*write the first 2 lines */
	if (fgets(line, sizeof(line), file))
		snprintf(mac_address_buf, WFA_MAC_ADDR_STR_LEN, line);

	fclose(file);
}

int hostapd_ap_get_mac_address(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr;
	intf_desc_t *def_intf = NULL;
	char mac_address_buf[WFA_MAC_ADDR_STR_LEN];
	char intf[INTF_NAME_LEN];
	char tmp_str[10] = {0};
	int i = 0, j = 0;
	int non_tx_bss_idx = 0;

	DPRINT_INFO(WFA_OUT, "===== running %s function =====\n", __func__);

	value_ptr = data->values;

	if (strcasecmp(data->interface, "2G") == 0 || strcasecmp(data->interface, "24G") == 0 ||
	    strcasecmp(data->interface, "2.4") == 0)
		mtk_ap_buf->def_intf = &(mtk_ap_buf->intf_2G);
	else if (strcasecmp(data->interface, "5G") == 0 || strcasecmp(data->interface, "5.0") == 0)
		mtk_ap_buf->def_intf = &(mtk_ap_buf->intf_5G);
	else if (strcasecmp(data->interface, "6G") == 0 || strcasecmp(data->interface, "6.0") == 0)
		mtk_ap_buf->def_intf = &(mtk_ap_buf->intf_6G);

	def_intf = mtk_ap_buf->def_intf;
	snprintf(intf, sizeof(intf), mtk_ap_buf->def_intf->name);

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "WLAN_TAG") == 0) {
			if (mtk_ap_buf->def_intf->mbss_en) {
				for (j = 0; j < mtk_ap_buf->def_intf->WLAN_TAG_bss_num; j++) {
					DPRINT_INFO(WFA_OUT, "WLAN_TAG[%d]=%d\n", j, mtk_ap_buf->def_intf->WLAN_TAG[j]);
					if (mtk_ap_buf->def_intf->WLAN_TAG[j] == atoi((value_ptr)[i])) {
						non_tx_bss_idx = j;
						break;
					}
				}
			}
		} else if (strcasecmp((data->params)[i], "NonTxBSSIndex") == 0)
			non_tx_bss_idx = atoi((value_ptr)[i]);
	}

	if ((non_tx_bss_idx > 0) && (non_tx_bss_idx < 10)) {
		if (!l1_valid) {
			snprintf(tmp_str, sizeof(tmp_str), "-%d ", non_tx_bss_idx);
			strcat(intf, tmp_str);
			DPRINT_INFO(WFA_OUT, "bss=%s\n", intf);
		} else
			intf[strlen(intf) - 1] = '0' + non_tx_bss_idx;
	}
	DPRINT_INFO(WFA_OUT, "non_tx_bss_idx is %d, inf is %s\n", non_tx_bss_idx, intf);

	if (!l1_valid) {
		char shellResult[WFA_BUFF_64] = {0};
		int ret_len = 0;

		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s", def_intf->name,
			 def_intf->name, HOSTAPD_TEMP_OUTPUT);

		if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr))
			printf("no bssid found\n");
		if (non_tx_bss_idx) {
			unsigned int mac_hex[6] = {0};

			/* last bytes plus one for multi bss */
			ret_len = sscanf(shellResult, "%02x:%02x:%02x:%02x:%02x:%02x", &mac_hex[0], &mac_hex[1],
					 &mac_hex[2], &mac_hex[3], &mac_hex[4], &mac_hex[5]);
			if (ret_len != 6)
				DPRINT_INFO(WFA_OUT, "sscanf of BSSID not match\n");
			mac_hex[5] += 1;
			snprintf(shellResult, sizeof(shellResult), "%02x:%02x:%02x:%02x:%02x:%02x", mac_hex[0],
				 mac_hex[1], mac_hex[2], mac_hex[3], mac_hex[4], mac_hex[5]);
		}
		sprintf((char *)resp_buf, "%s", shellResult);
	} else {
		hostapd_read_mac_address_file(mac_address_buf, intf, mtk_ap_buf);
		sprintf((char *)resp_buf, "%s", mac_address_buf);
	}
	DPRINT_INFO(WFA_OUT, "%s\n", resp_buf);

	return WFA_SUCCESS;
}


int hostapd_ap_deauth_sta(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr = data->values;
	int i = 0;
	char deauth_str[WFA_BUFF_128] = {0};

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "STA_MAC_Address") == 0) {
			memset(deauth_str, 0, sizeof(deauth_str));
			snprintf(deauth_str, sizeof(deauth_str), "deauthenticate %s", (value_ptr)[i]);
			hostapd_cli_cmd(def_intf, deauth_str);
		}
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}


static void hostapd_get_channel_from_cert_file(char *file, int *ch)
{
	char line[256];
	char *token;
	FILE *f = NULL;

	if (strlen(file) == 0)
		goto fail;

	DPRINT_INFO(WFA_OUT, "%s: file %s\n", __func__, file);

	f = fopen(file, "r");
	if (!f)
		goto fail;

	while (fgets(line, sizeof(line), f)) {
		if (strncmp(line, "Channel", 7) == 0) {
			token = strtok(line, "=");
			token = strtok(NULL, "=");
			if (token) {
				DPRINT_INFO(WFA_OUT, "%s: channel %s\n", __func__, token);
				*ch = atoi(token);
				break;
			}
		}
	}
fail:
	if (f)
		fclose(f);
}


static void hostapd_get_wmode_from_cert_file(char *file, int *wm)
{
	char line[256];
	char *token;
	FILE *f = NULL;

	if (strlen(file) == 0)
		goto fail;

	DPRINT_INFO(WFA_OUT, "%s: file %s\n", __func__, file);

	f = fopen(file, "r");
	if (!f)
		goto fail;

	while (fgets(line, sizeof(line), f)) {
		if (strncmp(line, "WirelessMode", 12) == 0) {
			token = strtok(line, "=");
			token = strtok(NULL, "=");
			if (token) {
				DPRINT_INFO(WFA_OUT, "%s: WirelessMode %s\n", __func__, token);
				*wm = atoi(token);
				break;
			}
		}
	}
fail:
	if (f)
		fclose(f);
}


int hostapd_is_wifi_interface_exist(int skfd, const char *interface)
{
	struct iwreq wrq;

	memset((char *)&wrq, 0, sizeof(struct iwreq));
	strncpy(wrq.ifr_name, interface, IFNAMSIZ-1);
	if (ioctl(skfd, SIOCGIWNAME, &wrq) < 0)
		return 0;

	return 1;
}


int hostapd_fillup_intf(int skfd, char *ifname, void *args, int count)
{
	mtk_ap_buf_t *mtk_ap_buf;
	int chan;
	static unsigned int wireless_band = W_BAND_INVALID;

	/* Avoid "Unused parameter" warning */
	mtk_ap_buf = (mtk_ap_buf_t *)args;
	count = count;

	/* only check ra intf */
	if (strstr(ifname, "apcli"))
		return -1;

	chan = hostapd_wifi_interface_chan(skfd, ifname);
	wireless_band = hostapd_wifi_interface_wband(skfd, ifname);
	DPRINT_INFO(WFA_OUT, "%s intf:%s, wireless_band=%d\n\n", __func__, ifname, wireless_band);

	if (wireless_band != W_BAND_INVALID) {
		if (wireless_band == W_BAND_6G && !mtk_ap_buf->intf_6G.status) {
			strcpy(mtk_ap_buf->intf_6G.name, ifname);
			DPRINT_INFO(WFA_OUT, "%s 6G intf:%s, channel=%d\n\n", __func__, mtk_ap_buf->intf_6G.name, chan);
			if (hostapd_assign_profile_pointer_to_intf(mtk_ap_buf, &mtk_ap_buf->intf_6G)) {
				mtk_ap_buf->intf_6G.status = 1;
				mtk_ap_buf->intf_6G.mode = WIFI_6G;
				mtk_ap_buf->intf_6G.mbss_en = 0;
				return 1;
			}
		}

		if ((wireless_band == W_BAND_2G) && !mtk_ap_buf->intf_2G.status) {
			strcpy(mtk_ap_buf->intf_2G.name, ifname);
			DPRINT_INFO(WFA_OUT, "%s 2G intf:%s, channel=%d\n", __func__, mtk_ap_buf->intf_2G.name, chan);
			if (hostapd_assign_profile_pointer_to_intf(mtk_ap_buf, &mtk_ap_buf->intf_2G)) {
				mtk_ap_buf->intf_2G.status = 1;
				mtk_ap_buf->intf_2G.mode = WIFI_2G;
				mtk_ap_buf->intf_2G.mbss_en = 0;
				return 1;
			}
		} else if ((wireless_band == W_BAND_5G) && !mtk_ap_buf->intf_5G.status) {
			strcpy(mtk_ap_buf->intf_5G.name, ifname);
			DPRINT_INFO(WFA_OUT, "%s 5G intf:%s, channel=%d\n", __func__, mtk_ap_buf->intf_5G.name, chan);
			if (hostapd_assign_profile_pointer_to_intf(mtk_ap_buf, &mtk_ap_buf->intf_5G)) {
				mtk_ap_buf->intf_5G.status = 1;
				mtk_ap_buf->intf_5G.mode = WIFI_5G;
				mtk_ap_buf->intf_5G.mbss_en = 0;
				return 1;
			}
		}
	} else {
		if ((chan >= 1 && chan <= 14) && !mtk_ap_buf->intf_2G.status) {
			strcpy(mtk_ap_buf->intf_2G.name, ifname);
			DPRINT_INFO(WFA_OUT, "%s 2G intf:%s, channel=%d\n", __func__, mtk_ap_buf->intf_2G.name, chan);
			if (hostapd_assign_profile_pointer_to_intf(mtk_ap_buf, &mtk_ap_buf->intf_2G)) {
				mtk_ap_buf->intf_2G.status = 1;
				mtk_ap_buf->intf_2G.mode = WIFI_2G;
				mtk_ap_buf->intf_2G.mbss_en = 0;
				return 1;
			}
		} else if ((chan >= 36) && !mtk_ap_buf->intf_5G.status) {
			strcpy(mtk_ap_buf->intf_5G.name, ifname);
			DPRINT_INFO(WFA_OUT, "%s 5G intf:%s, channel=%d\n", __func__, mtk_ap_buf->intf_5G.name, chan);
			if (hostapd_assign_profile_pointer_to_intf(mtk_ap_buf, &mtk_ap_buf->intf_5G)) {
				mtk_ap_buf->intf_5G.status = 1;
				mtk_ap_buf->intf_5G.mode = WIFI_5G;
				mtk_ap_buf->intf_5G.mbss_en = 0;
				return 1;
			}
		}
	}
	return 0;
}


void hostapd_reassign_sigma_dut_profile(mtk_ap_buf_t *mtk_ap_buf)
{
	int i = 0;
	int chan[INTF_NUM] = {0, 0, 0};
	int wmode[INTF_NUM] = {0, 0, 0};
	char sigma_dut_file[INTF_NUM][WFA_NAME_LEN];

	memset(sigma_dut_file, 0, sizeof(sigma_dut_file));
	memset(chan, 0, sizeof(chan));
	memset(wmode, 0, sizeof(wmode));

	for (i = 0; i < INTF_NUM; i++) {
		snprintf(sigma_dut_file[i],
			sizeof(mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile),
			mtk_ap_buf->profile_names_list[i].sigma_dut_profile);
		hostapd_get_channel_from_cert_file(sigma_dut_file[i], &chan[i]);
		hostapd_get_wmode_from_cert_file(sigma_dut_file[i], &wmode[i]);
	}

	for (i = 0; i < INTF_NUM; i++) {
		if (wmode[i] < WIFI_PHY_11AX_24G) {
			/* 11n or 11ac, reassign cert dat by channel */
			if (chan[i] > 0 && chan[i] < 36 && mtk_ap_buf->intf_2G.status) {
				snprintf(mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile,
					sizeof(mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile), sigma_dut_file[i]);
				DPRINT_INFO(WFA_OUT, "%s: reassign %s sigma dut profile %s\n", __func__,
					mtk_ap_buf->intf_2G.profile_names->name,
					mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile);
			}
			if (chan[i] >= 36 && mtk_ap_buf->intf_5G.status) {
				snprintf(mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile,
					sizeof(mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile), sigma_dut_file[i]);
				DPRINT_INFO(WFA_OUT, "%s: reassign %s sigma dut profile %s\n", __func__,
					mtk_ap_buf->intf_5G.profile_names->name,
					mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile);
			}
		} else {
			/* 11ax, 11he, reassign cert dat by wmode */
			switch (wmode[i]) {
			case WIFI_PHY_11AX_24G:
			case WIFI_PHY_11BE_24G:
				if (mtk_ap_buf->intf_2G.status) {
					snprintf(mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile,
						sizeof(mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile), sigma_dut_file[i]);
					DPRINT_INFO(WFA_OUT, "%s: reassign %s sigma dut profile %s\n", __func__,
						mtk_ap_buf->intf_2G.profile_names->name,
						mtk_ap_buf->intf_2G.profile_names->sigma_dut_profile);
				}
				break;
			case WIFI_PHY_11AX_5G:
			case WIFI_PHY_11BE_5G:
				if (mtk_ap_buf->intf_5G.status) {
					snprintf(mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile,
						sizeof(mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile), sigma_dut_file[i]);
					DPRINT_INFO(WFA_OUT, "%s: reassign %s sigma dut profile %s\n", __func__,
						mtk_ap_buf->intf_5G.profile_names->name,
						mtk_ap_buf->intf_5G.profile_names->sigma_dut_profile);
				}
				break;
			case WIFI_PHY_11AX_6G:
			case WIFI_PHY_11BE_6G:
				if (mtk_ap_buf->intf_6G.status) {
					snprintf(mtk_ap_buf->intf_6G.profile_names->sigma_dut_profile,
						sizeof(mtk_ap_buf->intf_6G.profile_names->sigma_dut_profile), sigma_dut_file[i]);
					DPRINT_INFO(WFA_OUT, "%s: reassign %s sigma dut profile %s\n", __func__,
						mtk_ap_buf->intf_6G.profile_names->name,
						mtk_ap_buf->intf_6G.profile_names->sigma_dut_profile);
				}
				break;
			default:
				break;
			}
		}
	}
}



int hostapd_init(void *ap_buf)
{
	int ret = WFA_ERROR;
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;

	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		goto fail;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	memset(gCmdStr, 0, sizeof(gCmdStr));
	snprintf(gCmdStr, sizeof(gCmdStr),
		"cp %s %s;\n cp %s %s;\n cp %s %s",
		ETC_HOSTAPD_RADIO1_PROFILE, HOSTAPD_RADIO1_PROFILE,
		ETC_HOSTAPD_RADIO2_PROFILE, HOSTAPD_RADIO2_PROFILE,
		ETC_HOSTAPD_RADIO3_PROFILE, HOSTAPD_RADIO3_PROFILE);
	DPRINT_INFO(WFA_OUT, "%s\n", gCmdStr);
	system(gCmdStr);

	/* init profile name */
	if (hostapd_init_profile_name(wifi_intf_fd, mtk_ap_buf)) {
		DPRINT_INFO(WFA_OUT, "%s(), init profile fail\n", __func__);
		goto fail;
	}

	hostapd_wifi_enum_devices(wifi_intf_fd, &hostapd_fillup_intf, mtk_ap_buf, 0);
	if (l1_valid)
		hostapd_reassign_sigma_dut_profile(mtk_ap_buf);

	/* backup profile */
	hostapd_backup_profile(mtk_ap_buf);

	/* ap init interface and parameters */
	if (hostapd_ap_init(mtk_ap_buf) != WFA_SUCCESS) {
		DPRINT_INFO(WFA_OUT, "fail to hostapd_ap_init.\n");
		goto fail;
	}

	ret = WFA_SUCCESS;
fail:

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return ret;
}

int hostapd_ap_reset_default(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	retType_t status;
	char **value_ptr;
	int i = 0;

	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return WFA_ERROR;
	}

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	value_ptr = data->values;
	mtk_ap_buf->dev_type = UNKNOWN;
	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "type") == 0) {
			if (strcasecmp((value_ptr)[i], "DUT") == 0)
				mtk_ap_buf->dev_type = DUT;
			else if (strcasecmp((value_ptr)[i], "TestBed") == 0)
				mtk_ap_buf->dev_type = TESTBED;
		}
	}

	if (strcasecmp(data->program, "wpa2") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_WPA2;
	else if (strcasecmp(data->program, "wmm") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_WMM;
	else if (strcasecmp(data->program, "hs2") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_HS2;
	else if (strcasecmp(data->program, "hs2-r2") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_HS2_R2;
	else if (strcasecmp(data->program, "hs2-r3") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_HS2_R3;
	else if (strcasecmp(data->program, "vht") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_VHT;
	else if (strcasecmp(data->program, "11n") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_11N;
	else if (strcasecmp(data->program, "60ghz") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_60GHZ;
	else if (strcasecmp(data->program, "loc") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_LOC;
	else if (strcasecmp(data->program, "wpa3") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_WPA3;
	else if (strcasecmp(data->program, "he") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_HE;
	else if (strcasecmp(data->program, "MBO") == 0)
		mtk_ap_buf->program = PROGRAME_TYPE_MBO;
	else if (strcasecmp(data->program, "QM") == 0) {
		mtk_ap_buf->program = PROGRAME_TYPE_QM;
		mtk_send_qos_cmd_to_wapp(mtk_ap_buf->def_intf->name, "reset_default", NULL, NULL);
	} else
		mtk_ap_buf->program = PROGRAME_TYPE_NONE;
	DPRINT_INFO(WFA_OUT, "program: %d\n", mtk_ap_buf->program);

	if (mtk_ap_buf->Band6Gonly.intf_6G_only) {
		mtk_ap_buf->Band6Gonly.intf_6G_only = 0;
		if (mtk_ap_buf->Band6Gonly.intf_2G_orig_stat) {
			snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s up", mtk_ap_buf->intf_2G.name);
			DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
			system(gCmdStr);

			sleep(3);
			mtk_ap_buf->Band6Gonly.intf_2G_orig_stat = 0;
			mtk_ap_buf->intf_2G.status = 1;
		}
		if (mtk_ap_buf->Band6Gonly.intf_5G_orig_stat) {
			if (strcasecmp(mtk_ap_buf->intf_2G.name, mtk_ap_buf->intf_5G.name) != 0) {
				snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s up", mtk_ap_buf->intf_5G.name);
				DPRINT_INFO(WFA_OUT, "%s() %s!\n", __func__, gCmdStr);
				system(gCmdStr);

				sleep(3);
			}
			mtk_ap_buf->Band6Gonly.intf_5G_orig_stat = 0;
			mtk_ap_buf->intf_5G.status = 1;
		}
	}

	hostapd_apply_sigma_profile(mtk_ap_buf, mtk_ap_buf->dev_type);

	if (l1_valid) {
		intf_desc_t *intfs[3] = {NULL, NULL, NULL};

		/* bringup hostapd with hostapd_cert_bx.dat */
		DPRINT_INFO(WFA_OUT, "\033[1;31m!!!!bring-up hostapd for sigma!!!!\033[0m\n");
		sleep(1);
		hostapd_apply_config(mtk_ap_buf, intfs);
		mtk_ap_buf->intern_flag.wifi_restarted = 1;
	}

	if (mtk_ap_buf->intf_6G.status) {
		if (mtk_ap_buf->profile_num <= 2) {
			strncpy(mtk_ap_buf->intf_5G.name, mtk_ap_buf->intf_6G.name, sizeof(mtk_ap_buf->intf_5G.name)-1);
			mtk_ap_buf->intf_5G.name[INTF_NAME_LEN-1] = '\0';
			if (!hostapd_assign_profile_pointer_to_intf(mtk_ap_buf, &mtk_ap_buf->intf_5G)) {
				DPRINT_INFO(WFA_OUT, "Profile name can't be found, use default!\n");
				status = WFA_SUCCESS;
				goto ret;
			}
			mtk_ap_buf->intf_6G.status = 0;
		}
		mtk_ap_buf->intf_5G.status = 1;
		mtk_ap_buf->intf_2G.status = 1;

		DPRINT_INFO(WFA_OUT, "%s intf_6G.status = 0!\n", __func__);
		DPRINT_INFO(WFA_OUT, "%s intf_6G.profile %s!\n", __func__, mtk_ap_buf->intf_6G.profile_names->profile);
	}

	status = hostapd_ap_init(mtk_ap_buf);


	mtk_ap_buf->intern_flag.committed = 0;
	mtk_ap_buf->intern_flag.BW_5G_set = 0;
	mtk_ap_buf->WLAN_TAG = -1;
	for (i = 0; i < 3; i++) {
		mtk_ap_buf->WLAN_TAG_inf[i] = NULL;
		cmt_intfs[i] = NULL;
	}

	if (mtk_ap_buf->program == PROGRAME_TYPE_HE) {
		char shellResult[WFA_BUFF_64] = {0};

		memset(shellResult, 0, sizeof(shellResult));
		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr) - 1, "lsmod | grep mt76 > %s", HOSTAPD_TEMP_OUTPUT);

		if (!hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr)) {
			DPRINT_INFO(WFA_OUT, "it's MT76 SKU\n");
			/* it's workaround for MT76, Add for fw reload */
			DPRINT_INFO(WFA_OUT, "system: rmmod mt7615e\n");
			system("rmmod mt7615e");
			DPRINT_INFO(WFA_OUT, "system: rmmod mt7615_common\n");
			system("rmmod mt7615_common");
			DPRINT_INFO(WFA_OUT, "system: rmmod mt7915e\n");
			system("rmmod mt7915e");
			DPRINT_INFO(WFA_OUT, "system: rmmod mt76-connac-lib\n");
			system("rmmod mt76-connac-lib");
			DPRINT_INFO(WFA_OUT, "system: rmmod mt76\n");
			system("rmmod mt76");
			DPRINT_INFO(WFA_OUT, "system: rmmod mac80211\n");
			system("rmmod mac80211");
			DPRINT_INFO(WFA_OUT, "system: rmmod cfg80211\n");
			system("rmmod cfg80211");
			DPRINT_INFO(WFA_OUT, "system: modprobe cfg80211\n");
			system("modprobe cfg80211");
			DPRINT_INFO(WFA_OUT, "system: modprobe mac80211\n");
			system("modprobe mac80211");
			DPRINT_INFO(WFA_OUT, "system: modprobe mt76\n");
			system("modprobe mt76");
			DPRINT_INFO(WFA_OUT, "system: modprobe mt76-connac-lib\n");
			system("modprobe mt76-connac-lib");
			DPRINT_INFO(WFA_OUT, "system: modprobe mt7915e\n");
			system("modprobe mt7915e");

			DPRINT_INFO(WFA_OUT, "system: Sleep 2...\n");
			sleep(2);
			DPRINT_INFO(WFA_OUT, "system: killall hostapd\n");
			system("killall hostapd");
			DPRINT_INFO(WFA_OUT, "system: Sleep 1...\n");
			sleep(1);
		}
	}

	/* Flush arp */
	system("ip -s -s neigh flush all");
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);

ret:
	return status;
}


void hostapd_update_key_mgt(
	mtk_ap_buf_t *mtk_ap_buf, enum ENUM_KEY_MGNT_TYPE key_mgt)
{
	int bss_idx = 0;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;

	fprintf(WFA_OUT, "def_intf->bss_idx: %d, mtk_ap_buf->WLAN_TAG %d\n", def_intf->bss_idx, mtk_ap_buf->WLAN_TAG);

	if ((def_intf->bss_idx > 1) ||
		(mtk_ap_buf->WLAN_TAG > 0 &&
		(mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]->mode ==
		mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG-1]->mode))) {
		bss_idx = 1;
		fprintf(WFA_OUT, "2nd keyMgmtType: %s\n", hostapd_key_mgt_2_string(key_mgt));
	} else
		fprintf(WFA_OUT, "1st keyMgmtType: %s\n", hostapd_key_mgt_2_string(key_mgt));

	def_intf->keyMgmtType[bss_idx] = key_mgt;
}

void hostapd_update_enc_type(
	mtk_ap_buf_t *mtk_ap_buf, enum ENUM_ENCP_TYPE enc_type)
{
	int bss_idx = 0;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;

	if ((def_intf->bss_idx > 1) ||
		(mtk_ap_buf->WLAN_TAG > 0 &&
		(mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]->mode ==
		mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG-1]->mode))) {
		bss_idx = 1;
		fprintf(WFA_OUT, "2nd encpType: %s\n", hostapd_enc_type_2_string(enc_type));
	} else
		fprintf(WFA_OUT, "1st encpType: %s\n", hostapd_enc_type_2_string(enc_type));

	def_intf->encpType[bss_idx] = enc_type;
}


void hostapd_update_capi_dual_table(mtk_ap_buf_t *mtk_ap_buf, char *key_ptr, char *value_ptr)
{
	intf_desc_t *def_intf = NULL;
	intf_desc_t *def_intf2 = NULL;
	char value[WFA_SSID_NAME_LEN];
	char param1[WFA_SSID_NAME_LEN];
	char param2[WFA_SSID_NAME_LEN];
	char *token = NULL;

	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return;
	}
	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	strncpy(value, value_ptr, sizeof(value) - 1);
	if (NULL != (strstr(value, ";"))) {
		token = strtok(value, ";");
		strncpy(param1, token, sizeof(param1) - 1);
		param1[sizeof(param1) - 1] = '\0';
		DPRINT_INFO(WFA_OUT, "param1: %s\n", param1);
		token = strtok(NULL, ";");
		strncpy(param2, token, sizeof(param2) - 1);
		param2[sizeof(param2) - 1] = '\0';
		DPRINT_INFO(WFA_OUT, "param2: %s\n", param2);
	} else {
		strncpy(param1, value, sizeof(param1) - 1);
		param1[sizeof(param1) - 1] = '\0';
		strncpy(param2, value, sizeof(param2) - 1);
		param2[sizeof(param2) - 1] = '\0';
	}

	def_intf = mtk_ap_buf->def_intf;
	if (mtk_ap_buf->def_mode == WIFI_5G)
		def_intf2 = &mtk_ap_buf->intf_2G;
	else if (mtk_ap_buf->def_mode == WIFI_2G)
		def_intf2 = &mtk_ap_buf->intf_5G;

	if (strcasecmp(key_ptr, "SSID") == 0) {
		strncpy(def_intf->ssid[0], param1, sizeof(def_intf->ssid[0]) - 1);
		def_intf->ssid[0][sizeof(def_intf->ssid[0]) - 1] = '\0';
		strncpy(def_intf2->ssid[0], param2, sizeof(def_intf2->ssid[0]) - 1);
		def_intf2->ssid[0][sizeof(def_intf2->ssid[0]) - 1] = '\0';
		DPRINT_INFO(WFA_OUT, "ssid1: %s\n", def_intf->ssid[0]);
		DPRINT_INFO(WFA_OUT, "ssid2: %s\n", def_intf2->ssid[0]);
	} else if (strcasecmp(key_ptr, "CHANNEL") == 0) {
		def_intf->channel = (char)atoi(param1);
		def_intf2->channel = (char)atoi(param2);
		DPRINT_INFO(WFA_OUT, "channel 1: %d\n", def_intf->channel);
		DPRINT_INFO(WFA_OUT, "channel 2: %d\n", def_intf2->channel);
	} else if (strcasecmp(key_ptr, "Mode") == 0) {
		if (strcasecmp(param1, "11b") == 0)
			def_intf->ap_mode = AP_MODE_11b;
		else if (strcasecmp(param1, "11bg") == 0)
			def_intf->ap_mode = AP_MODE_11bg;
		else if (strcasecmp(param1, "11bgn") == 0)
			def_intf->ap_mode = AP_MODE_11bgn;
		else if (strcasecmp(param1, "11a") == 0)
			def_intf->ap_mode = AP_MODE_11a;
		else if (strcasecmp(param1, "11g") == 0)
			def_intf->ap_mode = AP_MODE_11g;
		else if (strcasecmp(param1, "11na") == 0)
			def_intf->ap_mode = AP_MODE_11na;
		else if (strcasecmp(param1, "11ng") == 0)
			def_intf->ap_mode = AP_MODE_11ng;
		else if (strcasecmp(param1, "11ac") == 0)
			def_intf->ap_mode = AP_MODE_11ac;
		else if (strcasecmp(param1, "11ad") == 0)
			def_intf->ap_mode = AP_MODE_11ad;
		else if (strcasecmp(param1, "11ax") == 0)
			def_intf->ap_mode = AP_MODE_11ax;
		else
			def_intf->ap_mode = AP_MODE_NONE;
		DPRINT_INFO(WFA_OUT, "mode1 %s\n", hostapd_apmode_2_string(def_intf->ap_mode));

		if (strcasecmp(param2, "11b") == 0)
			def_intf2->ap_mode = AP_MODE_11b;
		else if (strcasecmp(param2, "11bg") == 0)
			def_intf2->ap_mode = AP_MODE_11bg;
		else if (strcasecmp(param2, "11bgn") == 0)
			def_intf2->ap_mode = AP_MODE_11bgn;
		else if (strcasecmp(param2, "11a") == 0)
			def_intf2->ap_mode = AP_MODE_11a;
		else if (strcasecmp(param2, "11g") == 0)
			def_intf2->ap_mode = AP_MODE_11g;
		else if (strcasecmp(param2, "11na") == 0)
			def_intf2->ap_mode = AP_MODE_11na;
		else if (strcasecmp(param2, "11ng") == 0)
			def_intf2->ap_mode = AP_MODE_11ng;
		else if (strcasecmp(param2, "11ac") == 0)
			def_intf2->ap_mode = AP_MODE_11ac;
		else if (strcasecmp(param2, "11ad") == 0)
			def_intf2->ap_mode = AP_MODE_11ad;
		else if (strcasecmp(param2, "11ax") == 0)
			def_intf2->ap_mode = AP_MODE_11ax;
		else
			def_intf2->ap_mode = AP_MODE_NONE;
		DPRINT_INFO(WFA_OUT, "mode2 %s\n", hostapd_apmode_2_string(def_intf2->ap_mode));
	} else if (strcasecmp(key_ptr, "BCNINT") == 0) {
		def_intf->bcnint = atoi(param1);
		def_intf2->bcnint = atoi(param2);
		DPRINT_INFO(WFA_OUT, "bcnint %d\n", def_intf->bcnint);
		DPRINT_INFO(WFA_OUT, "bcnint2 %d\n", def_intf2->bcnint);
	} else if (strcasecmp(key_ptr, "PSK") == 0) {
		strncpy(def_intf->passphrase, param1, sizeof(def_intf->passphrase) - 1);
		strncpy(def_intf2->passphrase, param2, sizeof(def_intf2->passphrase) - 1);
		def_intf->passphrase[sizeof(def_intf->passphrase) - 1] = '\0';
		def_intf2->passphrase[sizeof(def_intf2->passphrase) - 1] = '\0';
		DPRINT_INFO(WFA_OUT, "PSK %s\n", def_intf->passphrase);
		DPRINT_INFO(WFA_OUT, "PSK2 %s\n", def_intf2->passphrase);
	} else if (strcasecmp(key_ptr, "None") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_OPEN;
		def_intf->encpType[0] = ENCP_TYPE_NONE;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_OPEN;
		def_intf2->encpType[0] = ENCP_TYPE_NONE;
		DPRINT_INFO(WFA_OUT, "keyMgmtType KEY_MGNT_TYPE_OPEN, encpType ENCP_TYPE_NONE\n");
		DPRINT_INFO(WFA_OUT, "keyMgmtType2 KEY_MGNT_TYPE_OPEN, encpType2 ENCP_TYPE_NONE\n");
	} else if (strcasecmp(key_ptr, "WPA-PSK") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_WPA_PSK;
		def_intf->encpType[0] = ENCP_TYPE_TKIP;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_WPA_PSK;
		def_intf2->encpType[0] = ENCP_TYPE_TKIP;
		DPRINT_INFO(WFA_OUT, "keyMgmtType KEY_MGNT_TYPE_WPA_PSK, encpType ENCP_TYPE_TKIP\n");
		DPRINT_INFO(WFA_OUT, "keyMgmtType2 KEY_MGNT_TYPE_WPA_PSK, encpType2 ENCP_TYPE_TKIP\n");
	} else if (strcasecmp(key_ptr, "WPA2-PSK") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_PSK;
		def_intf->encpType[0] = ENCP_TYPE_CCMP;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_PSK;
		def_intf2->encpType[0] = ENCP_TYPE_CCMP;
		DPRINT_INFO(WFA_OUT, "keyMgmtType KEY_MGNT_TYPE_WPA2_PSK, encpType ENCP_TYPE_CCMP\n");
		DPRINT_INFO(WFA_OUT, "keyMgmtType2 KEY_MGNT_TYPE_WPA2_PSK, encpType2 ENCP_TYPE_CCMP\n");
	} else if (strcasecmp(key_ptr, "WPA2-PSK-Mixed") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_PSK_MIXED;
		def_intf->encpType[0] = ENCP_TYPE_CCMP_TKIP;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_PSK_MIXED;
		def_intf2->encpType[0] = ENCP_TYPE_CCMP_TKIP;
		DPRINT_INFO(WFA_OUT, "keyMgmtType KEY_MGNT_TYPE_WPA2_PSK_MIXED, encpType ENCP_TYPE_CCMP_TKIP\n");
		DPRINT_INFO(WFA_OUT, "keyMgmtType2 KEY_MGNT_TYPE_WPA2_PSK_MIXED, encpType2 ENCP_TYPE_CCMP_TKIP\n");
	} else if (strcasecmp(key_ptr, "WPA2-ENT") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_EAP;
		def_intf->encpType[0] = ENCP_TYPE_CCMP;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_EAP;
		def_intf2->encpType[0] = ENCP_TYPE_CCMP;
		DPRINT_INFO(WFA_OUT, "keyMgmtType KEY_MGNT_TYPE_WPA2_EAP, encpType ENCP_TYPE_CCMP\n");
		DPRINT_INFO(WFA_OUT, "keyMgmtType2 KEY_MGNT_TYPE_WPA2_EAP, encpType2 ENCP_TYPE_CCMP\n");
	} else if (strcasecmp(key_ptr, "wpa2-psk-sae") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_PSK_SAE;
		def_intf->encpType[0] = ENCP_TYPE_CCMP;
		def_intf->pmf = WFA_OPTIONAL;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_PSK_SAE;
		def_intf2->encpType[0] = ENCP_TYPE_CCMP;
		def_intf2->pmf = WFA_OPTIONAL;
		DPRINT_INFO(WFA_OUT,
			    "keyMgmtType KEY_MGNT_TYPE_WPA2_PSK_SAE, encpType ENCP_TYPE_CCMP, pmf WFA_OPTIONAL\n");
		DPRINT_INFO(WFA_OUT,
			    "keyMgmtType2 KEY_MGNT_TYPE_WPA2_PSK_SAE, encpType2 ENCP_TYPE_CCMP pmf WFA_OPTIONAL\n");
	} else if (strcasecmp(key_ptr, "SAE") == 0) {
		def_intf->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_SAE;
		def_intf->encpType[0] = ENCP_TYPE_CCMP;
		def_intf->pmf = WFA_REQUIRED;
		def_intf2->keyMgmtType[0] = KEY_MGNT_TYPE_WPA2_SAE;
		def_intf2->encpType[0] = ENCP_TYPE_CCMP;
		def_intf2->pmf = WFA_REQUIRED;
		DPRINT_INFO(WFA_OUT, "keyMgmtType KEY_MGNT_TYPE_WPA2_SAE, encpType ENCP_TYPE_CCMP, pmf WFA_REQUIRED\n");
		DPRINT_INFO(WFA_OUT,
			    "keyMgmtType2 KEY_MGNT_TYPE_WPA2_SAE, encpType2 ENCP_TYPE_CCMP pmf WFA_REQUIRED\n");
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
}

int hostapd_ap_set_security(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = NULL;
	char **value_ptr;
	char pairwise_cipher[42], group_cipher[42];
	int i = 0, j = 0;
	int intf_set = 1;

	DPRINT_INFO(WFA_OUT, "===== running %s function =====\n", __func__);
	if (!mtk_ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return WFA_ERROR;
	}

	strcpy(pairwise_cipher, "");
	strcpy(group_cipher, "");
	value_ptr = data->values;

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	intf_set = hostapd_capi_set_intf(mtk_ap_buf);
	def_intf = mtk_ap_buf->def_intf;

	DPRINT_INFO(WFA_OUT, "%s def_intf.name=%s\n", __func__, def_intf->name);

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "WLAN_TAG") == 0) {
			DPRINT_INFO(WFA_OUT, "intf_set %d\n", intf_set);
			if (intf_set == WFA_ERROR) {
				mtk_ap_buf->WLAN_TAG = atoi((value_ptr)[i]) - 1;
				if (mtk_ap_buf->WLAN_TAG < 0)
					mtk_ap_buf->WLAN_TAG = 0;
				if (mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]) {
					hostapd_set_default_intf(mtk_ap_buf,
								 mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]->mode);
					def_intf = mtk_ap_buf->def_intf;
					DPRINT_INFO(WFA_OUT, "reset default intf %s\n", def_intf->name);
				}
			}
			DPRINT_INFO(WFA_OUT, "Default interface=%s\n", mtk_ap_buf->def_intf->name);

			if (def_intf->mbss_en) {
				for (j = 0; j < def_intf->WLAN_TAG_bss_num; j++) {
					if (def_intf->WLAN_TAG[j] == atoi((value_ptr)[i])) {
						DPRINT_INFO(WFA_OUT, "found WLAN_TAG [%d]\n", def_intf->WLAN_TAG[j]);
						break;
					}
				}
			}
			DPRINT_INFO(WFA_OUT, "WLAN_TAG: %d\n", mtk_ap_buf->WLAN_TAG);
			DPRINT_INFO(WFA_OUT, "bss_idx %d, WLAN_TAG_bss_num %d\n", def_intf->bss_idx,
				def_intf->WLAN_TAG_bss_num);
			continue;
		} else if ((strcasecmp((data->params)[i], "KEYMGNT") == 0) ||
			   (strcasecmp((data->params)[i], "AKMSuiteType") == 0)) {
			char *keymgnt = def_intf->keymgnt;
			if (strcasecmp((data->params)[i], "AKMSuiteType") == 0) {
				DPRINT_INFO(WFA_OUT, "AKMSuiteType: %s\n", (value_ptr)[i]);
				strcpy(keymgnt, table_search_lower(hostapd_AKM_keymgnt_tbl, (value_ptr)[i]) ?: "");
			} else {
				strcpy(keymgnt, (value_ptr)[i]);
			}
			DPRINT_INFO(WFA_OUT, "KEYMGNT: %s\n", keymgnt);

			if (strcasecmp(keymgnt, "None") == 0) {
				DPRINT_INFO(WFA_OUT, "set none attr!!\n");
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "None", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_OPEN);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_NONE);
					def_intf->ppr.hs20[mtk_ap_buf->WLAN_TAG] = 0;
				}
			} else if (strcasecmp(keymgnt, "WPA-PSK") == 0) {
				hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA_PSK);
				DPRINT_INFO(WFA_OUT, "keyMgmtType: KEY_MGNT_TYPE_WPA_PSK, encpType: ENCP_TYPE_TKIP\n");
			} else if (strcasecmp(keymgnt, "WPA2-PSK") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "WPA2-PSK", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_PSK);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
					def_intf->pmf = WFA_OPTIONAL;
					DPRINT_INFO(WFA_OUT, "pmf: WFA_OPTIONAL\n");
				}
			} else if (strcasecmp(keymgnt, "WPA2-PSK-Mixed") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "WPA2-PSK-Mixed", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_PSK_MIXED);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP_TKIP);
				}
			} else if (strcasecmp(keymgnt, "SAE") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "SAE", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_SAE);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
					def_intf->pmf = WFA_REQUIRED;
					DPRINT_INFO(WFA_OUT, "pmf: WFA_REQUIRED\n");
				}
			} else if (strcasecmp(keymgnt, "wpa2-psk-sae") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "wpa2-psk-sae", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_PSK_SAE);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
					def_intf->pmf = WFA_OPTIONAL;
					DPRINT_INFO(WFA_OUT, "pmf: WFA_OPTIONAL\n");
				}
			} else if (strcasecmp(keymgnt, "WPA2-Ent") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "WPA2-Ent", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_EAP);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
					def_intf->pmf = WFA_OPTIONAL;
					DPRINT_INFO(WFA_OUT, "pmf: WFA_OPTIONAL\n");
				}
			} else if (strcasecmp(keymgnt, "WPA-Ent") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "WPA-Ent", "0");
				else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA_EAP);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_TKIP);
				}
			} else if (strcasecmp(keymgnt, "SuiteB") == 0) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf) {
					hostapd_update_capi_dual_table(mtk_ap_buf, "SuiteB", "0");
				} else {
					hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_SUITEB);
					hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_GCMP_256);
					def_intf->pmf = WFA_REQUIRED;
					DPRINT_INFO(WFA_OUT, "pmf: WFA_REQUIRED\n");
				}
			} else if (strcasecmp(keymgnt, "OWE") == 0) {
				hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_OWE);
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
				def_intf->auth_algs = 1;
				DPRINT_INFO(WFA_OUT, "pmf: DISABLE or OPTIONAL or REQUIRED\n");
			} else if (strcasecmp(keymgnt, "WPA2-Mixed") == 0) {
				hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_EAP_MIXED);
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP_TKIP);
			} else if (strcasecmp(keymgnt, "OSEN") == 0) {
				hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_OSEN);
				def_intf->auth_algs = 1;
			} else if (strcasecmp(keymgnt, "WPA2-ENT-OSEN") == 0) {
				hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_WPA2_ENT_OSEN);
				def_intf->auth_algs = 1;
			} else {
				hostapd_update_key_mgt(mtk_ap_buf, KEY_MGNT_TYPE_UNKNOWN);
				if ((strcasecmp(keymgnt, "SAE FT-SAE") == 0) ||
					(strcasecmp(keymgnt, "WPA-PSK-SHA256 FT-PSK WPA-PSK SAE FT-SAE") == 0))
					def_intf->auth_algs = 1;
				else if ((strcasecmp(keymgnt, "WPA-EAP-SHA256 FT-EAP") == 0) ||
					(strcasecmp(keymgnt, "WPA-EAP FT-EAP WPA-EAP-SHA256") == 0))
					def_intf->auth_algs = 3;
			}
		} else if ((strcasecmp((data->params)[i], "PairwiseCipher") == 0)) {
			if (strcasecmp((value_ptr)[i], "AES-CCMP-128") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
			else if (strcasecmp((value_ptr)[i], "AES-CCMP-256") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP_256);
			else if (strcasecmp((value_ptr)[i], "AES-GCMP-128") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_GCMP_128);
			else if (strcasecmp((value_ptr)[i], "AES-GCMP-256") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_GCMP_256);
			else
				DPRINT_INFO(WFA_OUT, "invalid PairwCipher: %s, use defalut\n", (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "Transition_Disable") == 0) {
			if ((strcasecmp((value_ptr)[i], "1") == 0) && (i + 1 < data->count) &&
			    (strcasecmp((data->params)[i + 1], "Transition_Disable_Index") == 0)) {
				char *index = (value_ptr)[i + 1];
				char idx_num[2];
				int j, bitmask = 0;

				memset(idx_num, 0, sizeof(idx_num));
				for (j = 0; j < strlen(index); j++) {
					idx_num[0] = index[j];
					bitmask |= 1 << atoi(idx_num);
				}
				def_intf->trans_disable = bitmask;
				i++;
			}
			DPRINT_INFO(WFA_OUT, "trans_disable: 0x%x\n", def_intf->trans_disable);
		} else if ((strcasecmp((data->params)[i], "PSK") == 0) ||
			   (strcasecmp((data->params)[i], "PSKPassPhrase") == 0) ||
			   (strcasecmp((data->params)[i], "SAEPasswords") == 0)) {
			if ((def_intf->bss_idx > 1) && (def_intf->bss_idx < 6))
				sprintf(gCmdStr, "PSK%d", def_intf->bss_idx);
			else if (def_intf->bss_idx <= 1) {
				if (mtk_ap_buf->intern_flag.capi_dual_pf)
					hostapd_update_capi_dual_table(mtk_ap_buf, "PSK", (value_ptr)[i]);
				else {
					strncpy(def_intf->passphrase, (value_ptr)[i], sizeof(def_intf->passphrase) - 1);
					def_intf->passphrase[sizeof(def_intf->passphrase) - 1] = '\0';
					DPRINT_INFO(WFA_OUT, "wpa_passphrase: %s\n", (value_ptr)[i]);
				}
				strcpy(def_intf->PWDIDR, "0");
			}
		} else if (strcasecmp((data->params)[i], "WEPKey") == 0) {
			strncpy(def_intf->wepkey, (value_ptr)[i], sizeof(def_intf->wepkey) - 1);
			def_intf->wepkey[sizeof(def_intf->wepkey) - 1] = '\0';
			DPRINT_INFO(WFA_OUT, "wep_key0:%s\n", def_intf->wepkey);
		} else if (strcasecmp((data->params)[i], "ENCRYPT") == 0) {
			if (strcasecmp((value_ptr)[i], "wep") == 0) {
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_WEP);
				/*
				** auth_algs=1: open,wep open
				** auth_algs=2: wep share key(ucc not test it)
				*/
				def_intf->auth_algs = 1;
			} else if (strcasecmp((value_ptr)[i], "tkip") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_TKIP);
			if (strcasecmp((value_ptr)[i], "aes") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_CCMP);
			else if (strcasecmp((value_ptr)[i], "aes-gcmp") == 0)
				hostapd_update_enc_type(mtk_ap_buf, ENCP_TYPE_GCMP_128);
			DPRINT_INFO(WFA_OUT, "ENCRYPT:%s\n", (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "NonTxBSSIndex") == 0) {
			def_intf->bss_idx = atoi((value_ptr)[i]) + 1;
			DPRINT_INFO(WFA_OUT, "NonTxBSSIndex:%d\n", def_intf->bss_idx);
		} else if (strcasecmp((data->params)[i], "PMF") == 0) {
			if (strcasecmp((value_ptr)[i], "Required") == 0) {
				if (def_intf->program == PROGRAME_TYPE_HS2_R2)
					def_intf->pmf = WFA_OPTIONAL;
				else
					def_intf->pmf = WFA_REQUIRED;
			}
			else if (strcasecmp((value_ptr)[i], "Optional") == 0)
				def_intf->pmf = WFA_OPTIONAL;
			else if (strcasecmp((value_ptr)[i], "Disabled") == 0)
				def_intf->pmf = WFA_DISABLED;
			DPRINT_INFO(WFA_OUT, "pmf:%s\n", (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "SHA256AD") == 0) {
			if (strcasecmp((value_ptr)[i], "Enabled") == 0)
				def_intf->sha256ad = WFA_ENABLED;
			DPRINT_INFO(WFA_OUT, "SHA256AD:%s\n", (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "ECGroupID") == 0) {
			snprintf(def_intf->ap_ecGroupID, sizeof(def_intf->ap_ecGroupID) - 1, (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "ECGroupID:%s\n", def_intf->ap_ecGroupID);
		} else if (strcasecmp((data->params)[i], "sae_pwe") == 0) {
			const char *str = table_search_lower(PweMethod_tbl, (value_ptr)[i]);

			def_intf->sae_pwe = str ? atoi(str) - 1 : 2;
			DPRINT_INFO(WFA_OUT, "sae_pwe: %d\n", def_intf->sae_pwe);
		} else if (strcasecmp((data->params)[i], "SAE_PK") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0)
				def_intf->sae_pk = 1;
			else
				def_intf->sae_pk = 0;
		} else if ((strcasecmp((data->params)[i], "SAE_PK_KeyPair") == 0) ||
			   (strcasecmp((data->params)[i], "SAE_PK_KeyPairMism") == 0)) {
			for (j = 0; j < 5; j++) {
				if (strcasecmp((value_ptr)[i], HOSTAPD_SAE_PK_KeyPair_tbl[j].str_key) == 0) {
					strncpy(def_intf->sae_pk_keypair, HOSTAPD_SAE_PK_KeyPair_tbl[j].str_val,
						sizeof(def_intf->sae_pk_keypair)-1);
					DPRINT_INFO(WFA_OUT, "sae_pk_val: %s\n", def_intf->sae_pk_keypair);
				}
				if (strcasecmp((value_ptr)[i], SAE_PK_Group_tbl[j].str_key) == 0) {
					strncpy(def_intf->ap_ecGroupID, SAE_PK_Group_tbl[j].str_val, sizeof(def_intf->ap_ecGroupID)-1);
					DPRINT_INFO(WFA_OUT, "ap_ecGroupID: %s\n", def_intf->ap_ecGroupID);
				}
			}
		} else if (strcasecmp((data->params)[i], "SAE_PK_Modifier") == 0) {
			strncpy(def_intf->sae_pk_modifier, (value_ptr)[i], sizeof(def_intf->sae_pk_modifier)-1);
			DPRINT_INFO(WFA_OUT, "sae_pk_modifier: %s\n", def_intf->sae_pk_modifier);
		} else if (strcasecmp((data->params)[i], "KeyRotation") == 0) {
			def_intf->KeyRotation = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "KeyRotation:%d\n", def_intf->KeyRotation);
		} else if (strcasecmp((data->params)[i], "KeyRotationInterval") == 0) {
			def_intf->KeyRotationInterval = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "KeyRotationInterval:%d\n", def_intf->KeyRotationInterval);
		} else if (strcasecmp((data->params)[i], "KeyRotation_BIGTK_STADisassoc") == 0) {
			def_intf->KeyRotation_BIGTK_STADisassoc = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "KeyRotation_BIGTK_STADisassoc:%d\n", def_intf->KeyRotation_BIGTK_STADisassoc);
		}
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

int hostapd_ap_set_rfeature(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = NULL;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr;
	int i = 0, j = 0;
	char ltf = 0, ltf_set = 0;
	char gi[4] = {0}, gi_set = 0;
	int Nebor_Pref = 0, Nebor_Op_Class = 0, Nebor_Op_Ch = 0;
	unsigned char BSSID[6];
	unsigned int mac_int[6] = {0};
	char cmd_content[WFA_CMD_STR_SZ];
	char Nebor_BSSID[WFA_BUFF_64];
	char SSID[WFA_BUFF_64], hex_ssid[WFA_BUFF_64];
	int offset = 0;
	int ret_len = 0;
	unsigned char Own_BSSID[6];
	unsigned int mac_hex[6] = {0};
	int idx1 = -1, idx2 = -1;

	memset(BSSID, 0, sizeof(BSSID));
	memset(mac_int, 0, sizeof(mac_int));
	memset(cmd_content, 0, sizeof(cmd_content));
	memset(Nebor_BSSID, 0, sizeof(Nebor_BSSID));
	memset(SSID, 0, sizeof(SSID));
	memset(hex_ssid, 0, sizeof(hex_ssid));

	if (!ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return WFA_ERROR;
	}

	def_intf = mtk_ap_buf->def_intf;
	value_ptr = data->values;

	if (strcasecmp(mtk_ap_buf->cmd_cfg.program, "QM") == 0) {
		for (i = 0; i < data->count; i++) {
			if (strcasecmp((data->params)[i], "RequestControl_Reset") == 0)
				mtk_send_qos_cmd_to_wapp(def_intf->name, "qos_ap_set_config", "ReqCtrlReset",
							 (value_ptr)[i]);
			else if (strcasecmp((data->params)[i], "DSCPPolicies") == 0)
				mtk_send_qos_cmd_to_wapp(def_intf->name, "qos_ap_set_config", "DSCPPolicies",
							 (value_ptr)[i]);
			else if (strcasecmp((data->params)[i], "MSCS") == 0 &&
				 strcasecmp((value_ptr)[i], "teardownclient") == 0)
				idx1 = i;
			else if (strcasecmp((data->params)[i], "MSCSClientMac") == 0)
				idx2 = i;
		}

		if ((idx1 >= 0) && (idx2 >= 0))
			mtk_send_qos_cmd_to_wapp(def_intf->name, "qos_ap_set_config", "teardownclient",
						 (value_ptr)[idx2]);

		return WFA_SUCCESS;
	}

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "LTF") == 0) {
			if (strncmp((value_ptr)[i], "3.2", 4) == 0)
				ltf = 1;
			else if (strncmp((value_ptr)[i], "6.4", 4) == 0)
				ltf = 2;
			else if (strncmp((value_ptr)[i], "12.8", 4) == 0)
				ltf = 4;
			ltf_set = 1;
			DPRINT_INFO(WFA_OUT, "ltf %d\n", ltf);
		} else if (strcasecmp((data->params)[i], "GI") == 0) {
			snprintf(gi, sizeof(gi), (value_ptr)[i]);
			gi[3] = '\0';
			gi_set = 1;
			DPRINT_INFO(WFA_OUT, "gi %s\n", gi);
		} else if (strcasecmp((data->params)[i], "txBandwidth") == 0) {
			if (strcasecmp((value_ptr)[i], "20") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_20;
			else if (strcasecmp((value_ptr)[i], "40") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_40;
			else if (strcasecmp((value_ptr)[i], "80") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_80;
			else if (strcasecmp((value_ptr)[i], "160") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_160;
			else
				def_intf->ch_width = CHANNEL_WIDTH_AUTO;
			DPRINT_INFO(WFA_OUT, "txBandwidth %s, channel %d\n", (value_ptr)[i], def_intf->channel);

			/* Re-run hostapd */
			hostapd_ap_config_commit(len, ap_buf, resp_len_ptr, resp_buf);
		} else if (strcasecmp((data->params)[i], "chnum_band") == 0) {
			char buf[WFA_NAME_LEN];
			char channel[4], bw[4], *token;
			int intf_idx = 0;
			int ch, bandwidth, freq, cent_freq, sec_offset = 0;

			memset(buf, 0, sizeof(buf));
			memset(channel, 0, sizeof(channel));
			memset(bw, 0, sizeof(bw));
			strncpy(buf, (value_ptr)[i], sizeof(buf)-1);
			token = strtok(buf, ";");
			if (token) {
				strncpy(channel, token, sizeof(channel)-1);
				channel[sizeof(channel)-1] = '\0';
				DPRINT_INFO(WFA_OUT, "channel: %s\n", channel);
			}
			token = strtok(NULL, ";");
			if (token) {
				strncpy(bw, token, sizeof(bw)-1);
				bw[sizeof(bw)-1] = '\0';
				DPRINT_INFO(WFA_OUT, "bw: %s\n", bw);
			}
			if (def_intf->mode == WIFI_6G)
				intf_idx = 2;
			else if (def_intf->mode == WIFI_5G)
				intf_idx = 1;
			else
				intf_idx = 0;

			ch = atoi(channel);
			bandwidth = atoi(bw);

			memset(gCmdStr, 0, sizeof(gCmdStr));
			if (l1_valid)
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl phy phy%d set channel num=%s bw=%s\n",
					intf_idx, channel, bw);
			else {
				switch (bandwidth) {
				case 20:
					hostapd_ch_bw_get_freq(ch, CHANNEL_WIDTH_20, &freq, &cent_freq, &sec_offset);
					if (def_intf->program == PROGRAME_TYPE_VHT)
						snprintf(gCmdStr, sizeof(gCmdStr),
						"hostapd_cli -i %s chan_switch 10 %d skip_cac\n", def_intf->name, freq);
					else
						snprintf(gCmdStr, sizeof(gCmdStr),
						"hostapd_cli -i %s chan_switch 10 %d\n", def_intf->name, freq);
					break;
				case 40:
					hostapd_ch_bw_get_freq(ch, CHANNEL_WIDTH_40, &freq, &cent_freq, &sec_offset);
					if (def_intf->program == PROGRAME_TYPE_VHT)
						snprintf(gCmdStr, sizeof(gCmdStr),
						"hostapd_cli -i %s chan_switch 10 %d sec_channel_offset=%d center_freq1=%d bandwidth=%s ht skip_cac\n",
							def_intf->name, freq, sec_offset, cent_freq, bw);
					else
						snprintf(gCmdStr, sizeof(gCmdStr),
						"hostapd_cli -i %s chan_switch 10 %d sec_channel_offset=%d center_freq1=%d bandwidth=%s ht\n",
							def_intf->name, freq, sec_offset, cent_freq, bw);
					break;
				case 80:
					hostapd_ch_bw_get_freq(ch, CHANNEL_WIDTH_80, &freq, &cent_freq, &sec_offset);
					if (def_intf->program == PROGRAME_TYPE_VHT)
						snprintf(gCmdStr, sizeof(gCmdStr),
						"hostapd_cli -i %s chan_switch 10 %d sec_channel_offset=%d center_freq1=%d bandwidth=%s vht skip_cac\n",
							def_intf->name, freq, sec_offset, cent_freq, bw);
					else
						snprintf(gCmdStr, sizeof(gCmdStr),
						"hostapd_cli -i %s chan_switch 10 %d sec_channel_offset=%d center_freq1=%d bandwidth=%s vht\n",
							def_intf->name, freq, sec_offset, cent_freq, bw);

					break;
				}
			}
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);
		} else if (strcasecmp((data->params)[i], "PPDUTxType") == 0) {
			char ppdu_type = 0;

			if (strcasecmp((value_ptr)[i], "SU") == 0)
				ppdu_type = 0;
			else if (strcasecmp((value_ptr)[i], "MU") == 0)
				ppdu_type = 1;
			else if (strcasecmp((value_ptr)[i], "ER") == 0)
				ppdu_type = 2;
			else if (strcasecmp((value_ptr)[i], "TB") == 0)
				ppdu_type = 3;
			else
				ppdu_type = 4;

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_rfeatures ppdu_type=%d\n",
				 def_intf->name, (int)ppdu_type);
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);
		} else if (strcasecmp((data->params)[i], "TriggerType") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_rfeatures trig_type=1,%s\n",
				 def_intf->name, (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);
		} else if (strcasecmp((data->params)[i], "AckPolicy") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_rfeatures ack_policy=%s\n",
				 def_intf->name, (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);
		} else if (strcasecmp((data->params)[i], "DisableTriggerType") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_rfeatures trig_type=0,%s\n",
				 def_intf->name, (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);
		} else if (strcasecmp((data->params)[i], "BTMReq_DisAssoc_Imnt") == 0)
			def_intf->BTMReq_DisAssoc_Imnt = 1;
		else if (strcasecmp((data->params)[i], "BTMReq_Term_Bit") == 0)
			def_intf->BTMReq_Term_Bit = 1;
		else if (strcasecmp((data->params)[i], "BSS_Term_Duration") == 0)
			def_intf->BSS_Term_Duration = atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "BSS_Term_TSF") == 0) {
			/* This is special for MBO TC 4.2.5.4 */
			char shellResult[WFA_CMD_STR_SZ];

			def_intf->BTMReq_BSS_Term_TSF = atoi((value_ptr)[i]);
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "hostapd_cli set bss_termination_tsf %s\n", (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "ifconfig %s | grep %s | awk '{print $5}' > %s",
				 def_intf->name, def_intf->name, HOSTAPD_TEMP_OUTPUT);

			if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr)) {
				DPRINT_INFO(WFA_OUT, "no bssid found\n");
			}

			ret_len = sscanf(shellResult, "%02x:%02x:%02x:%02x:%02x:%02x", &mac_hex[0], &mac_hex[1],
					 &mac_hex[2], &mac_hex[3], &mac_hex[4], &mac_hex[5]);
			if (ret_len != 6) {
				DPRINT_INFO(WFA_OUT, "sscanf of BSSID of BSS_Term_TSF not match\n");
			}

			for (j = 0; j < 6; j++)
				Own_BSSID[j] = (unsigned char)mac_hex[j];

			DPRINT_INFO(WFA_OUT, "Delete Neighbor Report for own BSS %s", shellResult);
			offset = 0;
			memset(cmd_content, 0, sizeof(cmd_content));
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "remove_neighbor ");

			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1,
					   " %02x:%02x:%02x:%02x:%02x:%02x ", MAC2STR(Own_BSSID));
			hostapd_cli_cmd(def_intf, cmd_content);

			DPRINT_INFO(WFA_OUT, "Add Neighbor Report for own BSS %s", shellResult);
			offset = 0;
			memset(cmd_content, 0, sizeof(cmd_content));
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "set_neighbor ");

			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1,
					   " %02x:%02x:%02x:%02x:%02x:%02x ", MAC2STR(Own_BSSID));

			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1,
					   " ssid=\"6f6f6f6f6f\" nr=");
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1,
					   "%02x%02x%02x%02x%02x%02x", MAC2STR(Own_BSSID));
			offset +=
			    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "931c00007324090301ff");
			hostapd_cli_cmd(def_intf, cmd_content);

			if (l1_valid) {
				def_intf->nr_count++;
				memcpy(def_intf->neighbor[0].bssid, Own_BSSID, 6);
				def_intf->neighbor[0].op_class = 115;
				def_intf->neighbor[0].op_ch = 36;
				def_intf->neighbor[0].pref = 0;
			}
		} else if (strcasecmp((data->params)[i], "Assoc_Disallow") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "hostapd_cli set mbo_assoc_disallow 1\n");
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			system(gCmdStr);
		} else if (strcasecmp((data->params)[i], "Nebor_BSSID") == 0) {
			snprintf(Nebor_BSSID, sizeof(Nebor_BSSID) - 1, (value_ptr)[i]);
			ret_len = sscanf((value_ptr)[i], "%02x:%02x:%02x:%02x:%02x:%02x", mac_int, (mac_int + 1),
					 (mac_int + 2), (mac_int + 3), (mac_int + 4), (mac_int + 5));
			if (ret_len != 6) {
				DPRINT_INFO(WFA_OUT, "sscanf of Nebor_BSSID not match\n");
			}
			for (j = 0; j < 6; j++)
				BSSID[j] = (unsigned char)mac_int[j];
			DPRINT_INFO(WFA_OUT, "Nebor_BSSID " MACSTR "\n", MAC2STR(BSSID));

			memset(cmd_content, 0, sizeof(cmd_content));
			snprintf(cmd_content, sizeof(cmd_content), "remove_neighbor %s\n", (value_ptr)[i]);
			hostapd_cli_cmd(def_intf, cmd_content);
		} else if (strcasecmp((data->params)[i], "Nebor_Pref") == 0) {
			ret_len = sscanf((value_ptr)[i], "%d", &Nebor_Pref);
			if (ret_len != 1) {
				DPRINT_INFO(WFA_OUT, "sscanf of Nebor_Pref not match\n");
			}
		} else if (strcasecmp((data->params)[i], "Nebor_Op_Class") == 0) {
			ret_len = sscanf((value_ptr)[i], "%d", &Nebor_Op_Class);
			if (ret_len != 1) {
				DPRINT_INFO(WFA_OUT, "sscanf of Nebor_Op_Class not match\n");
			}
		} else if (strcasecmp((data->params)[i], "Nebor_Op_Ch") == 0) {
			ret_len = sscanf((value_ptr)[i], "%d", &Nebor_Op_Ch);
			if (ret_len != 1) {
				DPRINT_INFO(WFA_OUT, "sscanf of Nebor_Op_Ch not match\n");
			}
		} else if (strcasecmp((data->params)[i], "UnsolicitedProbeResp") == 0) {
			char enable = 0, interval = 20;

			DPRINT_INFO(WFA_OUT, "[%s] [%s]\n", data->params[i], (value_ptr)[i]);
			memset(cmd_content, 0, sizeof(cmd_content));
			if (strcasecmp((value_ptr)[i], "disable") == 0)
				enable = 0;
			else
				enable = 1;

			snprintf(cmd_content, sizeof(cmd_content),
				"inband_discovery %d %d\n", enable, interval);
			hostapd_cli_cmd(def_intf, cmd_content);
		} else if (strcasecmp((data->params)[i], "RTS_FORCE") == 0) {
			DPRINT_INFO(WFA_OUT, "[%s] [%s]\n", data->params[i], (value_ptr)[i]);
			if (strcasecmp((value_ptr)[i], "enable") == 0) {
				char shellResult[5];

				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "iwinfo %s info | grep PHY | awk '{print $6}' > %s",
					def_intf->name, HOSTAPD_TEMP_OUTPUT);
				system(gCmdStr);
				if (hostapd_get_cmd_output(shellResult, sizeof(shellResult), gCmdStr)) {
					DPRINT_ERR(WFA_ERR, "no phy result\n");
					break;
				}

				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "iw %s set rts 500", shellResult);
				DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
				system(gCmdStr);
			}
		} else {
			DPRINT_INFO(WFA_OUT, "[%s] [%s]\n", data->params[i], (value_ptr)[i]);
		}
	}

	if (ltf_set || gi_set) {
		char band[4] = {0};

		if (def_intf->mode == WIFI_6G)
			snprintf(band, sizeof(band), "6");
		else if (def_intf->mode == WIFI_5G)
			snprintf(band, sizeof(band), "5");
		else if (def_intf->mode == WIFI_2G)
			snprintf(band, sizeof(band), "2.4");
		band[3] = '\0';

		memset(gCmdStr, 0, sizeof(gCmdStr));
		snprintf(gCmdStr, sizeof(gCmdStr), "iw dev %s set bitrates he-gi-%s %s he-ltf-%s %d\n", def_intf->name,
			 band, gi, band, ltf);
		DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
		system(gCmdStr);
	}

	if (strlen(Nebor_BSSID) || Nebor_Pref) {
		memset(cmd_content, 0, sizeof(cmd_content));
		/* set_neighbor BSSID */
		offset +=
		    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "set_neighbor %s", Nebor_BSSID);
		/* ssid="MBOxx" with random num */
		srand((unsigned int)time(NULL));
		snprintf(SSID, sizeof(SSID) - 1, "MBO%d", rand());
		ret_len = hostapd_str_to_hex(hex_ssid, sizeof(hex_ssid), SSID, strlen(SSID));
		if (ret_len >= sizeof(hex_ssid))
			DPRINT_INFO(WFA_OUT, "SSID too long\n");
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, " ssid=\"%s\"", hex_ssid);
		/* BSSID */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1,
				   " nr=%02x%02x%02x%02x%02x%02x", MAC2STR(BSSID));

		if (Nebor_Op_Class && Nebor_Op_Ch) {
			/* BSSID info: 00000000 */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "00000000");
			/* Nebor_Op_Class */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x",
					   (unsigned char)Nebor_Op_Class);
			/* Nebor_Op_Ch */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x",
					   (unsigned char)Nebor_Op_Ch);
			/* Phy type: 0 */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", 0);
		} else {
			/* BSSID info: 931c0000 */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "931c0000");
			/* Nebor_Op_Class: 115 */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", 115);
			/* Nebor_Op_Ch: 36 */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", 36);
			/* Phy type: 09 */
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", 9);
		}
		/* BSS Transition Preference tlv */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "0301%02x\n",
				   (unsigned char)Nebor_Pref);
		def_intf->nr_count++;
		hostapd_cli_cmd(def_intf, cmd_content);

		if (l1_valid) {
			int nb_idx = 0;

			for (nb_idx = 0; nb_idx < def_intf->nr_count; nb_idx++) {
				if (nb_idx == def_intf->nr_count)
					break;

				if (def_intf->neighbor[nb_idx].op_ch)
					continue;

				memcpy(def_intf->neighbor[nb_idx].bssid, BSSID, 6);
				def_intf->neighbor[nb_idx].op_class = (unsigned char)Nebor_Op_Class;
				def_intf->neighbor[nb_idx].op_ch = (unsigned char)Nebor_Op_Ch;
				def_intf->neighbor[nb_idx].pref = (unsigned char)Nebor_Pref;
				break;
			}
		}
	}

	return 0;
}

int hostapd_ap_set_wireless(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = NULL;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr = NULL;
	int i = 0, j = 0;
	int intf_set = 1;

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);
	intf_set = hostapd_capi_set_intf(mtk_ap_buf);

	def_intf = mtk_ap_buf->def_intf;

	DPRINT_INFO(WFA_OUT, "%s def_intf.name=%s\n", __func__, def_intf->name);

	if (((mtk_ap_buf->def_mode == WIFI_5G) || (mtk_ap_buf->def_mode == WIFI_6G)) &&
	    (!mtk_ap_buf->intern_flag.BW_5G_set)) {
		DPRINT_INFO(WFA_OUT, "Set 5G/6G interface default VTH BW to 80MHz!\n");
		def_intf->ch_width = CHANNEL_WIDTH_80;
	}

	value_ptr = data->values;
	DPRINT_INFO(WFA_OUT, "-----start looping: %d\n", data->count);

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "WLAN_TAG") == 0) {
			mtk_ap_buf->WLAN_TAG = atoi((value_ptr)[i]) - 1;
			if (mtk_ap_buf->WLAN_TAG < 0)
				mtk_ap_buf->WLAN_TAG = 0;
			if (mtk_ap_buf->WLAN_TAG > 2)
				mtk_ap_buf->WLAN_TAG = 2;
			DPRINT_INFO(WFA_OUT, "WLAN_TAG: %d\n", mtk_ap_buf->WLAN_TAG);
			DPRINT_INFO(WFA_OUT, "intf_set: %d\n", intf_set);

			if (intf_set == WFA_SUCCESS) {
				mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG] = def_intf;

				for (j = 0; j < def_intf->WLAN_TAG_bss_num; j++) {
					if (def_intf->WLAN_TAG[j] == atoi((value_ptr)[i]))
						break;
				}

				if (j == def_intf->WLAN_TAG_bss_num) {
					def_intf->WLAN_TAG[def_intf->WLAN_TAG_bss_num++] = atoi((value_ptr)[i]);
					if (def_intf->WLAN_TAG_bss_num > 1) {
						def_intf->bss_idx = def_intf->WLAN_TAG_bss_num - 1;
						if (def_intf->mbss_en == 0)
							def_intf->mbss_en = 1;
					}
				}
				DPRINT_INFO(WFA_OUT, "WLAN_TAG_bss_num [%d]\n", def_intf->WLAN_TAG_bss_num);
				DPRINT_INFO(WFA_OUT, "mbss_en: %d\n", def_intf->mbss_en);
			} else {
				if (mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]) {
					hostapd_set_default_intf(mtk_ap_buf,
								 mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]->mode);
					def_intf = mtk_ap_buf->def_intf;
				}
			}
			DPRINT_INFO(WFA_OUT, "Default interface=%s\n", mtk_ap_buf->def_intf->name);
			continue;
		} else if (strcasecmp((data->params)[i], "MBSSID") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->mbss_en = 1;
			else if (strcasecmp((value_ptr)[i], "disable") == 0)
				def_intf->mbss_en = 0;
			continue;
		} else if (strcasecmp((data->params)[i], "ssid") == 0) {
			if (def_intf->bss_idx > 1) {
				strncpy(def_intf->ssid[1], (value_ptr)[i], sizeof(def_intf->ssid[1]) - 1);
				def_intf->ssid[1][WFA_IF_NAME_LEN - 1] = '\0';
				DPRINT_INFO(WFA_OUT, "[%s] multi ssid: [%s]\n", def_intf->name, def_intf->ssid[1]);
				continue;
			}

			if (mtk_ap_buf->intern_flag.capi_dual_pf)
				hostapd_update_capi_dual_table(mtk_ap_buf, "SSID", (value_ptr)[i]);
			else {
				if (def_intf->mbss_en && mtk_ap_buf->WLAN_TAG > 0) {
					strncpy(def_intf->ssid[1], (value_ptr)[i], sizeof(def_intf->ssid[1]) - 1);
					def_intf->ssid[1][WFA_IF_NAME_LEN - 1] = '\0';
					DPRINT_INFO(WFA_OUT, "[%s] 2nd ssid: [%s]\n", def_intf->name, def_intf->ssid[1]);
				} else {
					strncpy(def_intf->ssid[0], (value_ptr)[i], sizeof(def_intf->ssid[0]) - 1);
					def_intf->ssid[0][WFA_IF_NAME_LEN - 1] = '\0';
					DPRINT_INFO(WFA_OUT, "[%s] main ssid: [%s]\n", def_intf->name, def_intf->ssid[0]);
				}
			}
			continue;
		} else if (strcasecmp((data->params)[i], "channel") == 0) {
			if (strcasecmp(mtk_ap_buf->cmd_cfg.program, "QM") == 0) {
				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr),
					"wappctrl %s qm qos_ap_set_config channel=%s", def_intf->name, (value_ptr)[i]);
				DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
				system(gCmdStr);
			}
			if (NULL != (strstr((value_ptr)[i], ";"))) {
				mtk_ap_buf->intern_flag.capi_dual_pf = 1;
				DPRINT_INFO(WFA_OUT, "set capi_dual_pf %d\n", mtk_ap_buf->intern_flag.capi_dual_pf);
			}

			if (mtk_ap_buf->intern_flag.capi_dual_pf)
				hostapd_update_capi_dual_table(mtk_ap_buf, "CHANNEL", (value_ptr)[i]);
			else {
				def_intf->channel = atoi((value_ptr)[i]);
				DPRINT_INFO(WFA_OUT, "channel %d\n", def_intf->channel);
			}
			continue;
		} else if (strcasecmp((data->params)[i], "width") == 0) {
			if (strcasecmp((value_ptr)[i], "20") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_20;
			else if (strcasecmp((value_ptr)[i], "40") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_40;
			else if (strcasecmp((value_ptr)[i], "80") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_80;
			else if (strcasecmp((value_ptr)[i], "160") == 0)
				def_intf->ch_width = CHANNEL_WIDTH_160;
			else
				def_intf->ch_width = CHANNEL_WIDTH_AUTO;
			DPRINT_INFO(WFA_OUT, "width %s\n", (value_ptr)[i]);
			mtk_ap_buf->intern_flag.BW_5G_set = 1;
		} else if (strcasecmp((data->params)[i], "mode") == 0) {
			if (mtk_ap_buf->intern_flag.capi_dual_pf)
				hostapd_update_capi_dual_table(mtk_ap_buf, "MODE", (value_ptr)[i]);
			else {
				char drv_cert_en = 1, fw_cert_en = 1;

				if (strcasecmp((value_ptr)[i], "11b") == 0) {
					def_intf->ap_mode = AP_MODE_11b;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11bg") == 0) {
					def_intf->ap_mode = AP_MODE_11bg;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11bgn") == 0) {
					def_intf->ap_mode = AP_MODE_11bgn;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11a") == 0) {
					def_intf->ap_mode = AP_MODE_11a;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11g") == 0) {
					def_intf->ap_mode = AP_MODE_11g;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11na") == 0) {
					def_intf->ap_mode = AP_MODE_11na;
					fw_cert_en = 0;
					if (!l1_valid) {
						/* Disable bridge netfilter */
						snprintf(gCmdStr, sizeof(gCmdStr),
							 "echo 0 > /proc/sys/net/bridge/bridge-nf-call-iptables\n");
						add_post_cmd(mtk_ap_buf);
					}
				} else if (strcasecmp((value_ptr)[i], "11ng") == 0) {
					def_intf->ap_mode = AP_MODE_11ng;
					fw_cert_en = 0;
					if (!l1_valid) {
						/* Disable bridge netfilter */
						snprintf(gCmdStr, sizeof(gCmdStr),
							 "echo 0 > /proc/sys/net/bridge/bridge-nf-call-iptables\n");
						add_post_cmd(mtk_ap_buf);
					}
				} else if (strcasecmp((value_ptr)[i], "11ac") == 0) {
					def_intf->ap_mode = AP_MODE_11ac;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11ad") == 0) {
					def_intf->ap_mode = AP_MODE_11ad;
					fw_cert_en = 0;
				} else if (strcasecmp((value_ptr)[i], "11ax") == 0) {
					def_intf->ap_mode = AP_MODE_11ax;
					/* Only support BW20 in 2.4 GHz*/
					if (def_intf->channel <= 14)
						def_intf->ch_width = CHANNEL_WIDTH_20;
				} else
					def_intf->ap_mode = AP_MODE_NONE;

				DPRINT_INFO(WFA_OUT, "mode %s\n", hostapd_apmode_2_string(def_intf->ap_mode));
				/* To enable OMI and bypass bf smooth*/
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless cert=%d-%d\n",
						def_intf->name, drv_cert_en, fw_cert_en);
				add_post_cmd(mtk_ap_buf);
			}
		} else if (strcasecmp((data->params)[i], "rts") == 0) {
			def_intf->rts = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "rts %d\n", def_intf->rts);
		} else if (strcasecmp((data->params)[i], "frgmnt") == 0) {
			def_intf->frgmnt = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "fragment: %d\n", def_intf->frgmnt);
		} else if (strcasecmp((data->params)[i], "bcnint") == 0) {
			if (mtk_ap_buf->intern_flag.capi_dual_pf)
				hostapd_update_capi_dual_table(mtk_ap_buf, "BCNINT", (value_ptr)[i]);
			else {
				def_intf->bcnint = atoi((value_ptr)[i]);
				DPRINT_INFO(WFA_OUT, "bcnint %d\n", def_intf->bcnint);
			}
		} else if (strcasecmp((data->params)[i], "offset") == 0) {
			if (strcasecmp((value_ptr)[i], "above") == 0)
				def_intf->ch_offset = CHANNEL_OFFSET_ABOVE;
			else if (strcasecmp((value_ptr)[i], "below") == 0)
				def_intf->ch_offset = CHANNEL_OFFSET_BELOW;
			else {
				DPRINT_INFO(WFA_OUT, "invalid offset value: %s\n", (value_ptr)[i]);
				def_intf->ch_offset = CHANNEL_OFFSET_NONE;
			}
			DPRINT_INFO(WFA_OUT, "offset %d\n", def_intf->ch_offset);
		} else if (strcasecmp((data->params)[i], "dtim") == 0) {
			def_intf->dtim_period = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "dtim: %d\n", def_intf->dtim_period);
		} else if (strcasecmp((data->params)[i], "sgi20") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->sgi20 = WFA_ENABLED;
			else
				def_intf->sgi20 = WFA_DISABLED;
			DPRINT_INFO(WFA_OUT, "sgi20: %d\n", def_intf->sgi20);
		} else if (strcasecmp((data->params)[i], "prog") == 0 ||
			   strcasecmp((data->params)[i], "program") == 0) {
			if (strcasecmp((value_ptr)[i], "wpa2") == 0)
				def_intf->program = PROGRAME_TYPE_WPA2;
			else if (strcasecmp((value_ptr)[i], "wmm") == 0)
				def_intf->program = PROGRAME_TYPE_WMM;
			else if (strcasecmp((value_ptr)[i], "hs2") == 0)
				def_intf->program = PROGRAME_TYPE_HS2;
			else if (strcasecmp((value_ptr)[i], "hs2-r2") == 0)
				def_intf->program = PROGRAME_TYPE_HS2_R2;
			else if (strcasecmp((value_ptr)[i], "hs2-r3") == 0)
				def_intf->program = PROGRAME_TYPE_HS2_R3;
			else if (strcasecmp((value_ptr)[i], "vht") == 0)
				def_intf->program = PROGRAME_TYPE_VHT;
			else if (strcasecmp((value_ptr)[i], "11n") == 0)
				def_intf->program = PROGRAME_TYPE_11N;
			else if (strcasecmp((value_ptr)[i], "60ghz") == 0)
				def_intf->program = PROGRAME_TYPE_60GHZ;
			else if (strcasecmp((value_ptr)[i], "loc") == 0)
				def_intf->program = PROGRAME_TYPE_LOC;
			else if (strcasecmp((value_ptr)[i], "wpa3") == 0)
				def_intf->program = PROGRAME_TYPE_WPA3;
			else if (strcasecmp((value_ptr)[i], "he") == 0)
				def_intf->program = PROGRAME_TYPE_HE;
			else if (strcasecmp((value_ptr)[i], "MBO") == 0)
				def_intf->program = PROGRAME_TYPE_MBO;
			else
				def_intf->program = PROGRAME_TYPE_NONE;
			DPRINT_INFO(WFA_OUT, "program: %d\n", def_intf->program);
		} else if (strcasecmp((data->params)[i], "DYN_BW_SGNL") == 0) {
			int bw_sgnl = 0;

			if (strcasecmp((value_ptr)[i], "enable") == 0)
				bw_sgnl = 2;
			else
				bw_sgnl = 1;

			if (l1_valid) {
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set bw_signaling %d", def_intf->name, bw_sgnl);
				add_post_cmd(mtk_ap_buf);
			} else {
				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless rts_sigta=%d",
					def_intf->name, bw_sgnl);
				add_post_cmd(mtk_ap_buf);
			}
		} else if (strcasecmp((data->params)[i], "BW_SGNL") == 0) {
			int bw_sgnl = 0;

			if (strcasecmp((value_ptr)[i], "enable") == 0)
				bw_sgnl = 1;
			else
				bw_sgnl = 0;

			if (l1_valid) {
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set bw_signaling %d", def_intf->name, bw_sgnl);
				add_post_cmd(mtk_ap_buf);
			} else {
				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless rts_sigta=%d",
					def_intf->name, bw_sgnl);
				add_post_cmd(mtk_ap_buf);
			}
		} else if (strcasecmp((data->params)[i], "TxBF") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->txbf = WFA_ENABLED;
			else
				def_intf->txbf = WFA_DISABLED;
			DPRINT_INFO(WFA_OUT, "txbf: %d\n", def_intf->txbf);
		} else if (strcasecmp((data->params)[i], "mcs_fixedrate") == 0) {
			char band[4] = {0};

			DPRINT_INFO(WFA_OUT, "mcs_fixed_rate: %s\n", (value_ptr)[i]);

			if (def_intf->mode == WIFI_6G)
				snprintf(band, sizeof(band), "6");
			else if (def_intf->mode == WIFI_5G)
				snprintf(band, sizeof(band), "5");
			else if (def_intf->mode == WIFI_2G)
				snprintf(band, sizeof(band), "2.4");

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "iw dev %s set bitrates he-mcs-%s %d:%s", def_intf->name,
				 band, 1, (value_ptr)[i]);

			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "NonTxBSSIndex") == 0) {
			def_intf->dot11vmbss_en = 1;
			def_intf->bss_idx = atoi((value_ptr)[i]) + 1;
			DPRINT_INFO(WFA_OUT, "NonTxBSSIndex:%d\n", def_intf->bss_idx);
			continue;
		} else if (strcasecmp((data->params)[i], "NumNonTxBSS") == 0) {
			def_intf->bss_num = atoi((value_ptr)[i]) + 1;
			DPRINT_INFO(WFA_OUT, "NumNonTxBSS: in str %s, in integer %d\n", (value_ptr)[i],
				    def_intf->bss_num - 1);
		} else if (strcasecmp((data->params)[i], "LDPC") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->bcc_mode = 0;
			DPRINT_INFO(WFA_OUT, "LDPC: %d\n", !def_intf->bcc_mode);
		} else if (strcasecmp((data->params)[i], "BCC") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->bcc_mode = 1;
			else if (strcasecmp((value_ptr)[i], "disable") == 0)
				def_intf->bcc_mode = 0;
			DPRINT_INFO(WFA_OUT, "bcc_mode: %d\n", def_intf->bcc_mode);
		} else if (strcasecmp((data->params)[i], "OFDMA") == 0) {
			int ofdma = 0;

			if (strcasecmp((value_ptr)[i], "DL") == 0) {
				def_intf->DL = 1;
				ofdma = 1;
			} else if (strcasecmp((value_ptr)[i], "UL") == 0)
				ofdma = 2;
			else if (strcasecmp((value_ptr)[i], "DL-20and80") == 0) {
				def_intf->DL = 1;
				ofdma = 3;
			} else
				ofdma = 0;

			memset(gCmdStr, 0, sizeof(gCmdStr));
			sprintf(gCmdStr, "mwctl %s set ap_wireless ofdma=%d\n", def_intf->name, ofdma);
			add_post_cmd(mtk_ap_buf);

			memset(gCmdStr, 0, sizeof(gCmdStr));
			sprintf(gCmdStr, "mwctl %s set ap_wireless add_ba_req_bufsize=64\n", def_intf->name);
			add_post_cmd(mtk_ap_buf);
		} else if ((strcasecmp((data->params)[i], "ADDBAReq_BufSize") == 0) ||
					(strcasecmp((data->params)[i], "ADDBAResp_BufSize") == 0)) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			if (strcasecmp((value_ptr)[i], "gt64") == 0)
				sprintf(gCmdStr, "mwctl %s set ap_wireless add_ba_req_bufsize=256\n", def_intf->name);
			else
				sprintf(gCmdStr, "mwctl %s set ap_wireless add_ba_req_bufsize=64\n", def_intf->name);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "PPDUTxType") == 0) {
			char ppdu_tx_type = 0;

			if (strcasecmp((value_ptr)[i], "MU") == 0)
				ppdu_tx_type = 1;
			else if (strcasecmp((value_ptr)[i], "SU") == 0)
				ppdu_tx_type = 0;
			else
				ppdu_tx_type = 4;

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless ppdu_type=%d\n",
				 def_intf->name, (int)ppdu_tx_type);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "MIMO") == 0) {
			if (strcasecmp((value_ptr)[i], "DL") == 0) {
				def_intf->DL = 1;
				def_intf->mimo = 1;
				memset(gCmdStr, 0, sizeof(gCmdStr));
				sprintf(gCmdStr, "mwctl %s set ap_wireless mimo=0\n", def_intf->name); /* MIMO DL Direction */
				add_post_cmd(mtk_ap_buf);
			} else if (strcasecmp((value_ptr)[i], "UL") == 0)
				def_intf->UL_MUMIMO = 1;
			DPRINT_INFO(WFA_OUT, "MIMO UL %d, MIMO DL %d\n", def_intf->UL_MUMIMO, def_intf->DL);
		} else if (strcasecmp((data->params)[i], "NumUsersOFDMA") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			sprintf(gCmdStr, "mwctl %s set ap_wireless nusers_ofdma=%s\n", def_intf->name,
				(value_ptr)[i]);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "MU_TxBF") == 0) {
			int enable = 0;

			if (strcasecmp((value_ptr)[i], "enable") == 0)
				enable = 1;
			def_intf->DL = enable;
			def_intf->mimo = enable;
			memset(gCmdStr, 0, sizeof(gCmdStr));
			if (enable)
				snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless mimo=%d\n",
						def_intf->name, 0); /* MIMO DL Direction */
			add_post_cmd(mtk_ap_buf);

			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless ppdu_type=%d\n",
					def_intf->name, enable);
			add_post_cmd(mtk_ap_buf);
		} else if ((strcasecmp((data->params)[i], "SPATIAL_RX_STREAM") == 0) ||
					(strcasecmp((data->params)[i], "SPATIAL_TX_STREAM") == 0)) {
			if (!l1_valid) {
				if ((strcasecmp((data->params)[i], "SPATIAL_RX_STREAM") == 0)) {
					def_intf->spatialRxStream = (char)atoi((value_ptr)[i]);
					DPRINT_INFO(WFA_OUT, "spatial_rx_stream %d\n", def_intf->spatialRxStream);
				} else {
					def_intf->spatialTxStream = (char)atoi((value_ptr)[i]);
					DPRINT_INFO(WFA_OUT, "spatial_tx_stream %d\n", def_intf->spatialTxStream);
				}
			} else {
				int stream_num = 0;

				if ((strcasecmp((value_ptr)[i], "1SS") == 0) || (strcasecmp((value_ptr)[i], "1") == 0)) {
					stream_num = 0x1;
				} else if ((strcasecmp((value_ptr)[i], "2SS") == 0) || (strcasecmp((value_ptr)[i], "2") == 0)) {
					stream_num = 0x3;
				} else if ((strcasecmp((value_ptr)[i], "3SS") == 0) || (strcasecmp((value_ptr)[i], "3") == 0)) {
					stream_num = 0x7;
				} else if ((strcasecmp((value_ptr)[i], "4SS") == 0) || (strcasecmp((value_ptr)[i], "4") == 0)) {
					stream_num = 0xf;
				} else {
					DPRINT_INFO(WFA_OUT, "%s value is wrong %s, must be 1SS,2SS,3SS,4SS,1,2,3,4\n",
						    (data->params)[i], (value_ptr)[i]);
					continue;
				}
				if ((strcasecmp((data->params)[i], "SPATIAL_TX_STREAM") == 0))
					def_intf->spatialTxStream = stream_num;
				else
					def_intf->spatialRxStream = stream_num;

				if ((def_intf->spatialTxStream > 0) && (def_intf->spatialRxStream > 0)) {
					memset(gCmdStr, 0, sizeof(gCmdStr));
					snprintf(gCmdStr, sizeof(gCmdStr), "iw phy phy%d set antenna %d %d\n",
							def_intf->phy_no, def_intf->spatialTxStream, def_intf->spatialRxStream);
					add_post_cmd(mtk_ap_buf);
					/*
					 *	Reset spatialTxStream and spatialRxStream to 0 due to prevent
					 *	hostapd_handle_special_cmd executed unexpectly.
					*/
					def_intf->spatialTxStream = 0;
					def_intf->spatialRxStream = 0;
				}
			}
		} else if (strcasecmp((data->params)[i], "HE_TXOPDurRTSThr") == 0) {
			int txop_rts_thld = 0;

			memset(gCmdStr, 0, sizeof(gCmdStr));
			if (strcasecmp((value_ptr)[i], "Enable") == 0)
				txop_rts_thld = 100;
			else
				txop_rts_thld = 1023;
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless he_txop_rts_thld=%d\n",
					def_intf->name, txop_rts_thld);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "radio") == 0) {
			if (strcasecmp(mtk_ap_buf->cmd_cfg.program, "QM") == 0) {
				mtk_send_qos_cmd_to_wapp(def_intf->name, "qos_ap_set_config",
							 (data->params)[i], (value_ptr)[i]);
				continue;
			}

			if (strcasecmp((value_ptr)[i], "off") == 0) {
				if (!def_intf) {
					/* no interface ready, off all the radios */
					if (l1_valid) {
						if (mtk_ap_buf->intf_2G.status) {
							memset(gCmdStr, 0, sizeof(gCmdStr));
							snprintf(gCmdStr, sizeof(gCmdStr),
								 "mwctl %s set RadioOn=0", mtk_ap_buf->intf_2G.name);
							DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
							system(gCmdStr);
						}

						if (mtk_ap_buf->intf_5G.status) {
							memset(gCmdStr, 0, sizeof(gCmdStr));
							snprintf(gCmdStr, sizeof(gCmdStr),
								 "mwctl %s set RadioOn=0", mtk_ap_buf->intf_5G.name);
							DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
							system(gCmdStr);
						}

						if (mtk_ap_buf->intf_6G.status) {
							memset(gCmdStr, 0, sizeof(gCmdStr));
							snprintf(gCmdStr, sizeof(gCmdStr),
								 "mwctl %s set RadioOn=0", mtk_ap_buf->intf_6G.name);
							DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
							system(gCmdStr);
						}
					} else {
						if (mtk_ap_buf->intf_2G.status) {
							memset(gCmdStr, 0, sizeof(gCmdStr));
							snprintf(gCmdStr, sizeof(gCmdStr),
								 "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
								 mtk_ap_buf->intf_2G.name);
							DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
							system(gCmdStr);
						}
						if (mtk_ap_buf->intf_5G.status) {
							memset(gCmdStr, 0, sizeof(gCmdStr));
							snprintf(gCmdStr, sizeof(gCmdStr),
								 "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
								 mtk_ap_buf->intf_5G.name);
							DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
							system(gCmdStr);
						}
						if (mtk_ap_buf->intf_6G.status) {
							memset(gCmdStr, 0, sizeof(gCmdStr));
							snprintf(gCmdStr, sizeof(gCmdStr),
								 "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
								 mtk_ap_buf->intf_6G.name);
							DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
							system(gCmdStr);
						}
					}
				} else {
					/* radio off the specified radio */
					if (l1_valid) {
						memset(gCmdStr, 0, sizeof(gCmdStr));
						snprintf(gCmdStr, sizeof(gCmdStr),
							 "mwctl %s set RadioOn=0", def_intf->name);
						DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
						system(gCmdStr);
					} else {
						memset(gCmdStr, 0, sizeof(gCmdStr));
						snprintf(gCmdStr, sizeof(gCmdStr),
							 "ubus call hostapd config_remove '{\"iface\":\"%s\"}'",
							 def_intf->name);
						DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
						system(gCmdStr);
					}
				}
			} else if (strcasecmp((value_ptr)[i], "on") == 0) {
				if (l1_valid) {
					memset(gCmdStr, 0, sizeof(gCmdStr));
					snprintf(gCmdStr, sizeof(gCmdStr),
						 "mwctl %s set RadioOn=1", def_intf->name);
					DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
					system(gCmdStr);
				} else {
					memset(gCmdStr, 0, sizeof(gCmdStr));
					snprintf(gCmdStr, sizeof(gCmdStr),
						 "ubus call hostapd config_add '{\"iface\":\"%s\",\"config\":\"%s\"}'",
						 def_intf->name, def_intf->profile_names->profile_cmt);
					DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
					system(gCmdStr);
				}
			}
		} else if (strcasecmp((data->params)[i], "Reg_Domain") == 0) {
			char Reg_Domain[WFA_NAME_LEN] = "00";

			if (def_intf->mode != WIFI_6G) {
				DPRINT_INFO(WFA_OUT, "param[Reg_Domain] %s\n", (value_ptr)[i]);
				if (strcasecmp((value_ptr)[i], "Global") == 0)
					strncpy(Reg_Domain, "00", sizeof(Reg_Domain) - 1);
				Reg_Domain[sizeof(Reg_Domain) - 1] = '\0';
				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "iw reg set %s", Reg_Domain);
				add_post_cmd(mtk_ap_buf);
			}
		} else if (strcasecmp((data->params)[i], "ADDBAReq_BufSize") == 0) {
			DPRINT_INFO(WFA_OUT, "ADDBAReq_BufSize %s\n", (value_ptr)[i]);
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set ap_wireless add_ba_req_bufsize=%s",
				 def_intf->name, (value_ptr)[i]);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "GAS_CB_Delay") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			/* set the neighbour report for GASInit response packet
			 * Any dummy value will work for this case
			*/
			snprintf(gCmdStr, sizeof(gCmdStr),
				"hostapd_cli -i %s set anqp_elem 272:341200104cb1cd36beac040000007324000301ff;"
				"hostapd_cli -i %s set gas_comeback_delay %s",
				def_intf->name, def_intf->name, (value_ptr)[i]);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "CountryCode") == 0) {
			strncpy(def_intf->country_code, (value_ptr)[i], 2);
			def_intf->country_code[2] = '\0';
			DPRINT_INFO(WFA_OUT, "CountryCode %s\n", def_intf->country_code);
		} else if (strcasecmp((data->params)[i], "FT_OA") == 0 || strcasecmp((data->params)[i], "FT_DS") == 0) {
			strcpy(def_intf->cmd_name, data->name);
			read_mac_address_file(def_intf->mac_addr, def_intf->name, mtk_ap_buf);
			strip_char(def_intf->mac_addr, ':');
			if (strcasecmp((data->params)[i], "FT_OA") == 0)
				def_intf->ft_oa = 1;
			else
				def_intf->ft_ds = 1;
			DPRINT_INFO(WFA_OUT, "NAME: %s, ft_oa: %s, mac_addr: %s\n", data->name, (value_ptr)[i],
				    def_intf->mac_addr);
			/* Ft is time critical feature, extra logs may cause roaming fail */
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr), "mwctl %s set Debug=0",
					def_intf->name);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "FT_BSS_LIST") == 0) {
			strcpy(def_intf->cmd_name, data->name);
			strcpy(def_intf->ft_bss_list, (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "NAME: %s, FT_BSS_LIST: %s\n", data->name, (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "UnsolicitedProbeResp") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->unsol_bcast_probe_resp_interval = 20;
			DPRINT_INFO(WFA_OUT, "unsol_bcast_probe_resp_interval %d\n",
				def_intf->unsol_bcast_probe_resp_interval);
		} else if (strcasecmp((data->params)[i], "ActiveInd_UnsolicitedProbeResp") == 0) {
			if (strcasecmp((value_ptr)[i], "enable") == 0)
				def_intf->rnr = 1;
			DPRINT_INFO(WFA_OUT, "rnr %d\n", def_intf->rnr);
		}  else if (strcasecmp((data->params)[i], "Band6Gonly") == 0) {
			DPRINT_INFO(WFA_OUT, "%s, mtk_ap_buf->intf_2G.name=%s\n", __func__, mtk_ap_buf->intf_2G.name);
			DPRINT_INFO(WFA_OUT, "%s, mtk_ap_buf->intf_5G.name=%s\n", __func__,  mtk_ap_buf->intf_5G.name);
			DPRINT_INFO(WFA_OUT, "%s, mtk_ap_buf->intf_6G.name=%s\n", __func__,  mtk_ap_buf->intf_6G.name);
			if (strcasecmp((value_ptr)[i], "enable") == 0) {
				mtk_ap_buf->Band6Gonly.intf_6G_only = 1;
				def_intf->sae_pwe = 1;
				if (mtk_ap_buf->intf_2G.status) {
					ifconfig_interface_down(mtk_ap_buf->intf_2G.name);
					mtk_ap_buf->Band6Gonly.intf_2G_orig_stat = mtk_ap_buf->intf_2G.status;
					mtk_ap_buf->intf_2G.status = 0;
					DPRINT_INFO(WFA_OUT, "turn off 2G\n");
				}
				if (mtk_ap_buf->intf_5G.status) {
					if (strcasecmp(mtk_ap_buf->intf_2G.name, mtk_ap_buf->intf_5G.name) != 0)
						ifconfig_interface_down(mtk_ap_buf->intf_5G.name);
					mtk_ap_buf->Band6Gonly.intf_5G_orig_stat = mtk_ap_buf->intf_5G.status;
					mtk_ap_buf->intf_5G.status = 0;
					DPRINT_INFO(WFA_OUT, "turn off 5G\n");
				}
				strcpy(def_intf->country_code, "DE");
				def_intf->country_code[2] = '\0';

				DPRINT_INFO(WFA_OUT, "CountryCode %s\n", def_intf->country_code);
			}
			continue;
		} else if (strcasecmp((data->params)[i], "ChnlFreq") == 0) {
			int freq = 0;

			freq = atoi((value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, "ChnlFreq %d\n", freq);
			if (freq > 6000) {
				def_intf->sae_pwe = 1;
				snprintf(def_intf->country_code, sizeof(def_intf->country_code), "DE");
				def_intf->country_code[2] = '\0';
				DPRINT_INFO(WFA_OUT, "6G, set sae_pwe %d country code = %s\n",
					def_intf->sae_pwe, def_intf->country_code);
			}
		} else if (strcasecmp((data->params)[i], "DSCPPolicyCapability") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wappctrl %s qm qos_ap_set_config dscppolicycapa=%s",
				def_intf->name, (strcasecmp((value_ptr)[i], "Enable") == 0) ? "1" : "0");
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "QoSMapCapability") == 0) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
				"wappctrl %s qm qos_ap_set_config qosmapcapa=%s",
				def_intf->name, (strcasecmp((value_ptr)[i], "Enable") == 0) ? "1" : "0");
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "HS2") == 0) {
			if (mtk_ap_buf->def_intf->WLAN_TAG_bss_num > 1) {
				if ((strcasecmp((value_ptr)[i], "0") == 0) ||
				    (strcasecmp((value_ptr)[i], "Disabled") == 0)) {
					def_intf->ppr.hs20[1] = 0;
					def_intf->ppr.interworking[1] = 0;
				} else if ((strcasecmp((value_ptr)[i], "1") == 0) ||
					   (strcasecmp((value_ptr)[i], "Enabled") == 0)) {
					def_intf->ppr.hs20[1] = 1;
					def_intf->ppr.interworking[1] = 1;
				}

				fprintf(WFA_OUT, "bss1 hs20 %d interworking %d\n", def_intf->ppr.hs20[1], def_intf->ppr.interworking[1]);
			} else {
				if ((strcasecmp((value_ptr)[i], "0") == 0) ||
				    (strcasecmp((value_ptr)[i], "Disabled") == 0)) {
					def_intf->ppr.hs20[0] = 0;
					def_intf->ppr.interworking[0] = 0;
				} else if ((strcasecmp((value_ptr)[i], "1") == 0) ||
				    (strcasecmp((value_ptr)[i], "Enabled") == 0)) {
					def_intf->ppr.hs20[0] = 1;
					def_intf->ppr.interworking[0] = 1;
				}

				fprintf(WFA_OUT, "bss0 hs20 %d interworking %d\n", def_intf->ppr.hs20[0], def_intf->ppr.interworking[0]);
			}
			continue;
		} else
			DPRINT_INFO(WFA_OUT, "param[%s] to be done\n", data->params[i]);
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

int hostapd_ap_set_radius(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = NULL;
	intf_desc_t *def_intf2 = NULL;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr;
	int i;
	int wlan_tag_idx = 0;

	DPRINT_INFO(WFA_OUT, "===== running %s function =====\n", __func__);
	if (!ap_buf) {
		DPRINT_INFO(WFA_OUT, "%s(), NULL pointer\n", __func__);
		return WFA_ERROR;
	}

	value_ptr = data->values;
	def_intf = mtk_ap_buf->def_intf;
	if (mtk_ap_buf->intern_flag.capi_dual_pf) {
		if (def_intf->mode == WIFI_2G)
			def_intf2 = &mtk_ap_buf->intf_5G;

		if (def_intf->mode == WIFI_5G)
			def_intf2 = &mtk_ap_buf->intf_2G;

	} else if (mtk_ap_buf->WLAN_TAG > 0) {
		for (wlan_tag_idx = 0; wlan_tag_idx < 3; wlan_tag_idx++) {
			if (wlan_tag_idx == 0)
				def_intf = mtk_ap_buf->WLAN_TAG_inf[wlan_tag_idx];
			else if (wlan_tag_idx == 1)
				def_intf2 = mtk_ap_buf->WLAN_TAG_inf[wlan_tag_idx];
		}
	}

	DPRINT_INFO(WFA_OUT, "-----start looping: %d\n", data->count);
	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "IPADDR") == 0) {
			DPRINT_INFO(WFA_OUT, "found! radius server IPADDR %s\n", value_ptr[i]);
			strncpy(def_intf->auth_server_addr, value_ptr[i], sizeof(def_intf->auth_server_addr) - 1);
			if (def_intf2)
				strncpy(def_intf2->auth_server_addr, value_ptr[i],
					sizeof(def_intf2->auth_server_addr) - 1);
			if (mtk_ap_buf->intf_6G.status)
				strncpy(mtk_ap_buf->intf_6G.auth_server_addr, value_ptr[i],
					sizeof(mtk_ap_buf->intf_6G.auth_server_addr) - 1);
		} else if (strcasecmp((data->params)[i], "PASSWORD") == 0) {
			DPRINT_INFO(WFA_OUT, "found! radius server PASSWORD %s\n", value_ptr[i]);
			strncpy(def_intf->auth_server_shared_secret, value_ptr[i],
				sizeof(def_intf->auth_server_shared_secret) - 1);
			if (def_intf2)
				strncpy(def_intf2->auth_server_shared_secret, value_ptr[i],
					sizeof(def_intf2->auth_server_shared_secret) - 1);
			if (mtk_ap_buf->intf_6G.status) {
				strncpy(mtk_ap_buf->intf_6G.auth_server_shared_secret, value_ptr[i],
					sizeof(mtk_ap_buf->intf_6G.auth_server_shared_secret) - 1);
			}
		} else if (strcasecmp((data->params)[i], "PORT") == 0) {
			DPRINT_INFO(WFA_OUT, "found! radius server PORT %s\n", value_ptr[i]);
			strncpy(def_intf->auth_server_port, value_ptr[i], sizeof(def_intf->auth_server_port) - 1);
			if (def_intf2)
				strncpy(def_intf2->auth_server_port, value_ptr[i],
					sizeof(def_intf2->auth_server_port) - 1);
			if (mtk_ap_buf->intf_6G.status) {
				strncpy(mtk_ap_buf->intf_6G.auth_server_port, value_ptr[i],
					sizeof(mtk_ap_buf->intf_6G.auth_server_port) - 1);
			}
		}
	}

	return WFA_SUCCESS;
}

int hostapd_ap_set_pmf(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);

	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	char **value_ptr = NULL;
	int i = 0;

	value_ptr = data->values;

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "PMF") == 0) {
			if (strcasecmp((value_ptr)[i], "Required") == 0)
				def_intf->pmf = WFA_REQUIRED;
			else if (strcasecmp((value_ptr)[i], "Optional") == 0)
				def_intf->pmf = WFA_OPTIONAL;
			else if (strcasecmp((value_ptr)[i], "Disabled") == 0)
				def_intf->pmf = WFA_DISABLED;

			DPRINT_INFO(WFA_OUT, "pmf:%s\n", (value_ptr)[i]);
		}
	}

	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

int hostapd_ap_set_apqos(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	char **value_ptr = NULL;
	hostapd_qos_t *ap_qos = &(def_intf->ap_qos);
	int i = 0;

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);
	value_ptr = data->values;

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "cwmin_VO") == 0)
			ap_qos->cwmin_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmin_VI") == 0)
			ap_qos->cwmin_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmin_BE") == 0)
			ap_qos->cwmin_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmin_BK") == 0)
			ap_qos->cwmin_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_VO") == 0)
			ap_qos->cwmax_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_VI") == 0)
			ap_qos->cwmax_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_BE") == 0)
			ap_qos->cwmax_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_BK") == 0)
			ap_qos->cwmax_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_VO") == 0)
			ap_qos->AIFS_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_VI") == 0)
			ap_qos->AIFS_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_BE") == 0)
			ap_qos->AIFS_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_BK") == 0)
			ap_qos->AIFS_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_VO") == 0)
			ap_qos->TXOP_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_VI") == 0)
			ap_qos->TXOP_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_BE") == 0)
			ap_qos->TXOP_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_BK") == 0)
			ap_qos->TXOP_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "ACM_VO") == 0) {
			if (strcasecmp((value_ptr)[i], "off") == 0)
				ap_qos->ACM_VO = 0;
			else
				ap_qos->ACM_VO = (char)atoi((value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "ACM_VI") == 0) {
			if (strcasecmp((value_ptr)[i], "off") == 0)
				ap_qos->ACM_VI = 0;
			else
				ap_qos->ACM_VI = (char)atoi((value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "ACM_BE") == 0) {
			if (strcasecmp((value_ptr)[i], "off") == 0)
				ap_qos->ACM_BE = 0;
			else
				ap_qos->ACM_BE = (char)atoi((value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "ACM_BK") == 0) {
			if (strcasecmp((value_ptr)[i], "off") == 0)
				ap_qos->ACM_BK = 0;
			else
				ap_qos->ACM_BK = (char)atoi((value_ptr)[i]);
		}
	}
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

int hostapd_ap_set_staqos(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	char **value_ptr = NULL;
	hostapd_qos_t *sta_qos = &(def_intf->sta_qos);
	int i = 0;

	DPRINT_INFO(WFA_OUT, "%s()===enter====>\n", __func__);
	value_ptr = data->values;

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "cwmin_VO") == 0)
			sta_qos->cwmin_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmin_VI") == 0)
			sta_qos->cwmin_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmin_BE") == 0)
			sta_qos->cwmin_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmin_BK") == 0)
			sta_qos->cwmin_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_VO") == 0)
			sta_qos->cwmax_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_VI") == 0)
			sta_qos->cwmax_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_BE") == 0)
			sta_qos->cwmax_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "cwmax_BK") == 0)
			sta_qos->cwmax_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_VO") == 0)
			sta_qos->AIFS_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_VI") == 0)
			sta_qos->AIFS_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_BE") == 0)
			sta_qos->AIFS_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "AIFS_BK") == 0)
			sta_qos->AIFS_BK = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_VO") == 0)
			sta_qos->TXOP_VO = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_VI") == 0)
			sta_qos->TXOP_VI = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_BE") == 0)
			sta_qos->TXOP_BE = (char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "TXOP_BK") == 0)
			sta_qos->TXOP_BK = (char)atoi((value_ptr)[i]);
	}
	DPRINT_INFO(WFA_OUT, "%s()<===exit====\n", __func__);
	return WFA_SUCCESS;
}

int hostapd_ap_set_hs2(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	char **value_ptr = NULL;
	char intf[16];
	int i = 0, j = 0;
	int intf_set = 1;
	int WLAN_TAG_bss_idx = 0;
	char prefix[10];
	int param_idx = 0;
	int offset = 0;

	fprintf(WFA_OUT, "===== running %s function =====\n", __func__);

	value_ptr = data->values;

	fprintf(WFA_OUT, "-----start looping: %d\n", data->count);
	strcpy(intf, mtk_ap_buf->def_intf->name);
	intf_set = hostapd_capi_set_intf(mtk_ap_buf);

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "WLAN_TAG") == 0) {
			if (intf_set == 0) {
				mtk_ap_buf->WLAN_TAG = atoi((value_ptr)[i]) - 1;
				if (mtk_ap_buf->WLAN_TAG < 0)
					mtk_ap_buf->WLAN_TAG = 0;
				if (mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]) {
					hostapd_set_default_intf(mtk_ap_buf,
							 mtk_ap_buf->WLAN_TAG_inf[mtk_ap_buf->WLAN_TAG]->mode);
				}
			}
			fprintf(WFA_OUT, "Default interface=%s\n", mtk_ap_buf->def_intf->name);

			WLAN_TAG_bss_idx = 0;
			if (mtk_ap_buf->def_intf->mbss_en) {

				for (j = 0; j < mtk_ap_buf->def_intf->WLAN_TAG_bss_num; j++) {
					fprintf(WFA_OUT, "WLAN_TAG[%d]=%d\n", j, mtk_ap_buf->def_intf->WLAN_TAG[j]);
					if (mtk_ap_buf->def_intf->WLAN_TAG[j] == atoi((value_ptr)[i])) {
						WLAN_TAG_bss_idx = j + 1;
						break;
					}
				}
			}
			fprintf(WFA_OUT, "WLAN_TAG_bss_idx=%d\n", WLAN_TAG_bss_idx);
			if (WLAN_TAG_bss_idx > 1) {
				strcpy(prefix, intf);
				prefix[strlen(intf) - 1] = '\0';
				sprintf(intf, "%s%d", prefix, WLAN_TAG_bss_idx - 1);
				intf[strlen(intf)] = '\0';
			}
		} else if (strcasecmp((data->params)[i], "Accs_Net_Type") == 0) {
			def_intf->ppr.access_network_type = atoi((value_ptr)[i]);
			fprintf(WFA_OUT, "access_network_type=%d\n", def_intf->ppr.access_network_type);
		} else if (strcasecmp((data->params)[i], "Advice_of_Charge") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.advice_of_charge[0],
				    sizeof(def_intf->ppr.advice_of_charge[0]), "%s", "0:0::ENG:USD:0;FRA:CAD:1");
				snprintf(def_intf->ppr.advice_of_charge[1],
				    sizeof(def_intf->ppr.advice_of_charge[1]), "%s", "1:0::ENG:USD:2;FRA:CAD:3");
				snprintf(def_intf->ppr.advice_of_charge[2],
				    sizeof(def_intf->ppr.advice_of_charge[2]), "%s",
				    "3:0:service-provider.com;federation.example.com:ENG:USD:4");

				for (param_idx = 0; param_idx < 3; param_idx++)
					fprintf(WFA_OUT, "advice_of_charge[%d] %s\n",
					    param_idx, def_intf->ppr.advice_of_charge[param_idx]);
			} else
				fprintf(WFA_OUT, "Wrong value of advice of charge id\n");
		} else if (strcasecmp((data->params)[i], "ANQP") == 0) {
			if (strcasecmp((value_ptr)[i], "0") == 0) {
				def_intf->ppr.external_anqp_server_test = 1;
				fprintf(WFA_OUT, "external_anqp_server_test %d\n", 1);
			}
		} else if (strcasecmp((data->params)[i], "BSS_LOAD") == 0) {
			/* ToDo: add for hostapd */
			;
		} else if (strcasecmp((data->params)[i], "Conn_Cap") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.hs20_conn_capab[0], sizeof(def_intf->ppr.hs20_conn_capab[0]), "%s", "1:0:0");
				snprintf(def_intf->ppr.hs20_conn_capab[1], sizeof(def_intf->ppr.hs20_conn_capab[1]), "%s", "6:20:1");
				snprintf(def_intf->ppr.hs20_conn_capab[2], sizeof(def_intf->ppr.hs20_conn_capab[2]), "%s", "6:22:0");
				snprintf(def_intf->ppr.hs20_conn_capab[3], sizeof(def_intf->ppr.hs20_conn_capab[3]), "%s", "6:80:1");
				snprintf(def_intf->ppr.hs20_conn_capab[4], sizeof(def_intf->ppr.hs20_conn_capab[4]), "%s", "6:443:1");
				snprintf(def_intf->ppr.hs20_conn_capab[5], sizeof(def_intf->ppr.hs20_conn_capab[5]), "%s", "6:1723:0");
				snprintf(def_intf->ppr.hs20_conn_capab[6], sizeof(def_intf->ppr.hs20_conn_capab[6]), "%s", "6:5060:0");
				snprintf(def_intf->ppr.hs20_conn_capab[7], sizeof(def_intf->ppr.hs20_conn_capab[7]), "%s", "17:500:1");
				snprintf(def_intf->ppr.hs20_conn_capab[8], sizeof(def_intf->ppr.hs20_conn_capab[8]), "%s", "17:5060:0");
				snprintf(def_intf->ppr.hs20_conn_capab[9], sizeof(def_intf->ppr.hs20_conn_capab[9]), "%s", "17:4500:1");
				snprintf(def_intf->ppr.hs20_conn_capab[10], sizeof(def_intf->ppr.hs20_conn_capab[10]), "%s", "50:0:1");

				for (param_idx = 0; param_idx < 11; param_idx++)
					fprintf(WFA_OUT, "hs20_conn_capab[%d] %s\n",
					    param_idx, def_intf->ppr.hs20_conn_capab[param_idx]);
			} else if (strcasecmp((value_ptr)[i], "3") == 0) {
				snprintf(def_intf->ppr.hs20_conn_capab[0], sizeof(def_intf->ppr.hs20_conn_capab[0]), "%s", "6:80:1");
				snprintf(def_intf->ppr.hs20_conn_capab[1], sizeof(def_intf->ppr.hs20_conn_capab[1]), "%s", "6:443:1");

				for (param_idx = 0; param_idx < 2; param_idx++)
					fprintf(WFA_OUT, "hs20_conn_capab[%d] %s\n",
					    param_idx, def_intf->ppr.hs20_conn_capab[param_idx]);
			} else if (strcasecmp((value_ptr)[i], "4") == 0) {
				snprintf(def_intf->ppr.hs20_conn_capab[0], sizeof(def_intf->ppr.hs20_conn_capab[0]), "%s", "6:80:1");
				snprintf(def_intf->ppr.hs20_conn_capab[1], sizeof(def_intf->ppr.hs20_conn_capab[1]), "%s", "6:443:1");
				snprintf(def_intf->ppr.hs20_conn_capab[2], sizeof(def_intf->ppr.hs20_conn_capab[2]), "%s", "6:5060:1");
				snprintf(def_intf->ppr.hs20_conn_capab[3], sizeof(def_intf->ppr.hs20_conn_capab[3]), "%s", "17:443:1");

				for (param_idx = 0; param_idx < 4; param_idx++)
					fprintf(WFA_OUT, "hs20_conn_capab[%d] %s\n",
					    param_idx, def_intf->ppr.hs20_conn_capab[param_idx]);
			} else
				fprintf(WFA_OUT, "Unknown Conn Cap!!!\n");
		} else if (strcasecmp((data->params)[i], "DGAF_Disable") == 0) {
			def_intf->ppr.disable_dgaf = atoi((value_ptr)[i]);
			fprintf(WFA_OUT, "disable_dgaf %d\n", def_intf->ppr.disable_dgaf);

			if (strcasecmp((value_ptr)[i], "1") == 0) {
				def_intf->ppr.proxy_arp = atoi((value_ptr)[i]);
				fprintf(WFA_OUT, "proxy_arp %d\n", def_intf->ppr.proxy_arp);
			}
		} else if (strcasecmp((data->params)[i], "Domain_List") == 0) {
			/* ToDo: add for hostapd */
			;
		} else if (strcasecmp((data->params)[i], "GAS_CB_Delay") == 0) {
			def_intf->ppr.gas_comeback_delay = atoi((value_ptr)[i]);
			fprintf(WFA_OUT, "gas_comeback_delay %d\n", def_intf->ppr.gas_comeback_delay);
		} else if (strcasecmp((data->params)[i], "HESSID") == 0) {
			memcpy(def_intf->ppr.hessid, (value_ptr)[i], sizeof(def_intf->ppr.hessid));
			fprintf(WFA_OUT, "hessid %s\n", def_intf->ppr.hessid);
		} else if (strcasecmp((data->params)[i], "ICMPv4_Echo") == 0) {
			def_intf->ppr.disable_dgaf = 0;
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				def_intf->ppr.icmpv4_deny = 0;
				fprintf(WFA_OUT, "icmpv4_deny %d\n", def_intf->ppr.icmpv4_deny);
				fprintf(WFA_OUT, "disable_dgaf %d\n", def_intf->ppr.disable_dgaf);

			} else if (strcasecmp((value_ptr)[i], "0") == 0) {
				def_intf->ppr.icmpv4_deny = 1;
				fprintf(WFA_OUT, "icmpv4_deny %d\n", def_intf->ppr.icmpv4_deny);
				fprintf(WFA_OUT, "disable_dgaf %d\n", def_intf->ppr.disable_dgaf);
			}

			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set icmpv4_deny=%d\n",
				def_intf->name, def_intf->ppr.icmpv4_deny);
			system(gCmdStr);
			add_post_cmd(mtk_ap_buf);
			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set dgaf_disable=%d\n",
				def_intf->name, def_intf->ppr.disable_dgaf);
			system(gCmdStr);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "Interworking") == 0) {
			if ((strcasecmp((value_ptr)[i], "enabled") == 0) || (strcasecmp((value_ptr)[i], "1") == 0)) {
				if (WLAN_TAG_bss_idx <= 1)
					def_intf->ppr.interworking[0] = 1;
				else
					def_intf->ppr.interworking[1] = 1;
			} else if ((strcasecmp((value_ptr)[i], "disabled") == 0) ||
				   (strcasecmp((value_ptr)[i], "0") == 0)) {
				if (WLAN_TAG_bss_idx <= 1) {
					def_intf->ppr.interworking[0] = 0;
					def_intf->ppr.hs20[0] = 0;
				} else {
					def_intf->ppr.interworking[1] = 0;
					def_intf->ppr.hs20[1] = 0;
				}
			}

			for (param_idx = 0; param_idx < 2; param_idx++)
				fprintf(WFA_OUT, "bss[%d] hs20 %d, interworking %d\n",
				    param_idx, def_intf->ppr.hs20[param_idx], def_intf->ppr.interworking[param_idx]);
		} else if (strcasecmp((data->params)[i], "IP_Add_Type_Avail") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.ipaddr_type_availability,
				    sizeof(def_intf->ppr.ipaddr_type_availability), "%s", "0C");
				fprintf(WFA_OUT, "ipaddr_type_availability %s\n", def_intf->ppr.ipaddr_type_availability);
			} else
				fprintf(WFA_OUT, "Unknown IP Add Type\n");
		} else if (strcasecmp((data->params)[i], "L2_Traffic_Inspect") == 0) {
			def_intf->ppr.l2_filter = (char)atoi((value_ptr)[i]);
			def_intf->ppr.icmpv4_deny = 1;
			def_intf->ppr.disable_dgaf = 1;

			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set l2_filter=%s",
			    def_intf->name, (value_ptr)[i]);
			system(gCmdStr);
			add_post_cmd(mtk_ap_buf);
			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set icmpv4_deny=%s",
			    def_intf->name, (value_ptr)[i]);
			system(gCmdStr);
			add_post_cmd(mtk_ap_buf);
			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set dgaf_disable=%s",
				 def_intf->name, (value_ptr)[i]);
			system(gCmdStr);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "NAI_Realm_List") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,mail.example.com,21[2:4][5:7]");
				snprintf(def_intf->ppr.nai_realm[1], sizeof(def_intf->ppr.nai_realm[1]),
				    "%s", "0,cisco.com,21[2:4][5:7]");
				snprintf(def_intf->ppr.nai_realm[2], sizeof(def_intf->ppr.nai_realm[2]),
				    "%s", "0,wi-fi.org,21[2:4][5:7],13[5:6]");
				snprintf(def_intf->ppr.nai_realm[3], sizeof(def_intf->ppr.nai_realm[3]),
				    "%s", "0,example.com,13[5:6]");

				for (param_idx = 0; param_idx < 4; param_idx++)
					fprintf(WFA_OUT, "nai_realm[%d] %s\n",
					    param_idx, def_intf->ppr.nai_realm[param_idx]);
			} else if (strcasecmp((value_ptr)[i], "2") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,mail.example.com,21[2:4][5:7]");
				fprintf(WFA_OUT, "nai_realm[0] %s\n", def_intf->ppr.nai_realm[0]);
			} else if (strcasecmp((value_ptr)[i], "3") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,cisco.com,21[2:4][5:7]");
				snprintf(def_intf->ppr.nai_realm[1], sizeof(def_intf->ppr.nai_realm[1]),
				    "%s", "0,wi-fi.org,21[2:4][5:7],13[5:6]");
				snprintf(def_intf->ppr.nai_realm[2], sizeof(def_intf->ppr.nai_realm[2]),
				    "%s", "0,example.com,13[5:6]");

				for (param_idx = 0; param_idx < 3; param_idx++)
					fprintf(WFA_OUT, "nai_realm[%d] %s\n",
					    param_idx, def_intf->ppr.nai_realm[param_idx]);
			} else if (strcasecmp((value_ptr)[i], "4") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,mail.example.com,21[2:4][5:7],13[5:6]");
				fprintf(WFA_OUT, "nai_realm[0] %s\n", def_intf->ppr.nai_realm[0]);
			} else if (strcasecmp((value_ptr)[i], "5") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,wi-fi.org,21[2:4][5:7]");
				snprintf(def_intf->ppr.nai_realm[1], sizeof(def_intf->ppr.nai_realm[1]),
				    "%s", "0,ruckuswireless.com,21[2:4][5:7]");
				fprintf(WFA_OUT, "nai_realm[0] %s\n", def_intf->ppr.nai_realm[0]);
				fprintf(WFA_OUT, "nai_realm[1] %s\n", def_intf->ppr.nai_realm[1]);
			} else if (strcasecmp((value_ptr)[i], "6") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,wi-fi.org,21[2:4][5:7]");
				snprintf(def_intf->ppr.nai_realm[1], sizeof(def_intf->ppr.nai_realm[1]),
				    "%s", "0,mail.example.com,21[2:4][5:7]");
				fprintf(WFA_OUT, "nai_realm[0] %s\n", def_intf->ppr.nai_realm[0]);
				fprintf(WFA_OUT, "nai_realm[1] %s\n", def_intf->ppr.nai_realm[1]);
			} else if (strcasecmp((value_ptr)[i], "7") == 0) {
				snprintf(def_intf->ppr.nai_realm[0], sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "0,wi-fi.org,21[2:4][5:7],13[5:6]");
				fprintf(WFA_OUT, "nai_realm[0] %s\n", def_intf->ppr.nai_realm[0]);
			} else
				fprintf(WFA_OUT, "Unknown NAI Realm List ID\n");
			mtk_ap_buf->IsMediatekHS2NAISet = 1;
		} else if (strcasecmp((data->params)[i], "Net_Auth_Type") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.network_auth_type, sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "00https://tandc-server.wi-fi.org");
				fprintf(WFA_OUT, "network_auth_type %s\n", def_intf->ppr.network_auth_type);
			} else if (strcasecmp((value_ptr)[i], "2") == 0) {
				snprintf(def_intf->ppr.network_auth_type, sizeof(def_intf->ppr.nai_realm[0]),
				    "%s", "01");
				fprintf(WFA_OUT, "network_auth_type %s\n", def_intf->ppr.network_auth_type);
			} else
				fprintf(WFA_OUT, "Unknown Network Auth Type\n");
		} else if (strcasecmp((data->params)[i], "Oper_Class") == 0) {
			/* ToDo: add for hostapd */
		} else if (strcasecmp((data->params)[i], "Oper_Name") == 0) {
			/* ToDo: add for hostapd */
		} else if (strcasecmp((data->params)[i], "Operator_Icon_Metadata") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.operator_icon, sizeof(def_intf->ppr.operator_icon),
				    "%s", "icon_red_eng.png");
				fprintf(WFA_OUT, "operator_icon %s\n", def_intf->ppr.operator_icon);
			} else
				fprintf(WFA_OUT, "Unknown Operator Icon Metadata\n");
		} else if (strcasecmp((data->params)[i], "OSU") == 0) {
			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set osu=%d",
			    def_intf->name, 1);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "OSU_SSID") == 0) {
			char osu_ssid[20];

			sprintf(osu_ssid, "\"%s\"", (value_ptr)[i]);

			snprintf(def_intf->ppr.osu_ssid, sizeof(def_intf->ppr.osu_ssid), "%s", osu_ssid);
			fprintf(WFA_OUT, "osu_ssid %s\n", def_intf->ppr.osu_ssid);
		} else if (strcasecmp((data->params)[i], "OSU_METHOD") == 0) {
			if (strcasecmp((value_ptr)[i], "SOAP") == 0)
				def_intf->ppr.osu_method_list[0] = 1;
			else
				def_intf->ppr.osu_method_list[0] = 0;

			fprintf(WFA_OUT, "osu_method_list[0] %d\n", def_intf->ppr.osu_method_list[0]);
		} else if (strcasecmp((data->params)[i], "OSU_ICON_TAG") == 0) {
			if (strcasecmp((value_ptr)[i], "2") == 0) {
				sprintf(gCmdStr, "cp /etc/wifi-abgn-logo_270x73.png /etc/icon_red_zxx.png");
				fprintf(WFA_OUT, "run command ==> %s\n", gCmdStr);
				system(gCmdStr);
				sprintf(gCmdStr, "cp /etc/wifi-abgn-logo_270x73.png /etc/icon_red_eng.png");
				fprintf(WFA_OUT, "run command ==> %s\n", gCmdStr);
				system(gCmdStr);
			} else {
				sprintf(gCmdStr, "cp /etc/icon_red_zxx_default.png /etc/icon_red_zxx.png");
				DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
				system(gCmdStr);
				sprintf(gCmdStr, "cp /etc/icon_red_eng_default.png /etc/icon_red_eng.png");
				DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
				system(gCmdStr);
			}
		} else if (strcasecmp((data->params)[i], "OSU_PROVIDER_LIST") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.hs20_icon[0], sizeof(def_intf->ppr.hs20_icon[0]),
				    "%s", "128:61:zxx:image/png:icon_red_zxx.png:/etc/icon_red_zxx.png");
				snprintf(def_intf->ppr.hs20_icon[1], sizeof(def_intf->ppr.hs20_icon[1]),
				    "%s", "160:76:eng:image/png:icon_red_eng.png:/etc/icon_red_eng.png");
				for (param_idx = 0; param_idx < 2; param_idx++)
					fprintf(WFA_OUT, "hs20_icon[%d] %s\n",
					    param_idx, def_intf->ppr.hs20_icon[param_idx]);

				snprintf(def_intf->ppr.osu_icon[0], sizeof(def_intf->ppr.osu_icon[0]),
				    "%s", "icon_red_zxx.png");
				snprintf(def_intf->ppr.osu_icon[1], sizeof(def_intf->ppr.osu_icon[1]),
				    "%s", "icon_red_eng.png");
				for (param_idx = 0; param_idx < 2; param_idx++)
					fprintf(WFA_OUT, "osu_icon[%d] %s\n",
					    param_idx, def_intf->ppr.osu_icon[param_idx]);

				snprintf(def_intf->ppr.osu_friendly_name[0], sizeof(def_intf->ppr.osu_friendly_name[0]),
				    "%s", "eng:SP Red Test Only");
				snprintf(def_intf->ppr.osu_friendly_name[1], sizeof(def_intf->ppr.osu_friendly_name[1]),
				    "%s", "kor:SP 빨강 테스트 전용");
				for (param_idx = 0; param_idx < 2; param_idx++)
					fprintf(WFA_OUT, "osu_friendly_name[%d] %s\n",
					    param_idx, def_intf->ppr.osu_friendly_name[param_idx]);

				snprintf(def_intf->ppr.osu_service_desc[0], sizeof(def_intf->ppr.osu_service_desc[0]),
				    "%s", "eng:Free service for test purpose");
				snprintf(def_intf->ppr.osu_service_desc[1], sizeof(def_intf->ppr.osu_service_desc[1]),
				    "%s", "kor:테스트 목적으로 무료 서비스");
				for (param_idx = 0; param_idx < 2; param_idx++)
					fprintf(WFA_OUT, "osu_service_desc[%d] %s\n",
					    param_idx, def_intf->ppr.osu_service_desc[param_idx]);
			} else if (strcasecmp((value_ptr)[i], "2") == 0) {
				snprintf(def_intf->ppr.hs20_icon[0], sizeof(def_intf->ppr.hs20_icon[0]),
				    "%s", "128:61:zxx:image/png:icon_orange_zxx:/etc/icon_orange_zxx.png");
				fprintf(WFA_OUT, "hs20_icon[0] %s\n", def_intf->ppr.hs20_icon[0]);

				snprintf(def_intf->ppr.osu_icon[0], sizeof(def_intf->ppr.osu_icon[0]),
				    "%s", "icon_orange_zxx.png");
				fprintf(WFA_OUT, "osu_icon[0] %s\n", def_intf->ppr.osu_icon[0]);

				snprintf(def_intf->ppr.osu_friendly_name[0], sizeof(def_intf->ppr.osu_friendly_name[0]),
				    "%s", "eng:Wireless Broadband Alliance");
				snprintf(def_intf->ppr.osu_friendly_name[1], sizeof(def_intf->ppr.osu_friendly_name[1]),
				    "%s", "kor:와이어리스 브로드밴드 얼라이언스");
				fprintf(WFA_OUT, "osu_friendly_name[0] %s\n", def_intf->ppr.osu_friendly_name[0]);
				fprintf(WFA_OUT, "osu_friendly_name[1] %s\n", def_intf->ppr.osu_friendly_name[1]);

				snprintf(def_intf->ppr.osu_service_desc[0], sizeof(def_intf->ppr.osu_service_desc[0]),
				    "%s", "eng:Free service for test purpose");
				snprintf(def_intf->ppr.osu_service_desc[1], sizeof(def_intf->ppr.osu_service_desc[1]),
				    "%s", "kor:테스트 목적으로 무료 서비스");
				fprintf(WFA_OUT, "osu_service_desc[0] %s\n", def_intf->ppr.osu_service_desc[0]);
				fprintf(WFA_OUT, "osu_service_desc[1] %s\n", def_intf->ppr.osu_service_desc[1]);
			} else if (strcasecmp((value_ptr)[i], "8") == 0) {
				snprintf(def_intf->ppr.hs20_icon[0], sizeof(def_intf->ppr.hs20_icon[0]),
				    "%s", "128:61:zxx:image/png:icon_red_zxx:/etc/icon_red_zxx.png");
				fprintf(WFA_OUT, "hs20_icon[0] %s\n", def_intf->ppr.hs20_icon[0]);

				snprintf(def_intf->ppr.osu_nai[0], sizeof(def_intf->ppr.osu_nai[0]),
				    "%s", "anonymous@hotspot.net");
				fprintf(WFA_OUT, "osu_nai[0] %s\n", def_intf->ppr.osu_nai[0]);

				snprintf(def_intf->ppr.osu_icon[0], sizeof(def_intf->ppr.osu_icon[0]),
				    "%s", "icon_red_zxx.png");
				fprintf(WFA_OUT, "osu_icon[0] %s\n", def_intf->ppr.osu_icon[0]);

				snprintf(def_intf->ppr.osu_friendly_name[0], sizeof(def_intf->ppr.osu_friendly_name[0]),
				   "%s", "eng:SP Red Test Only");
				snprintf(def_intf->ppr.osu_friendly_name[1], sizeof(def_intf->ppr.osu_friendly_name[1]),
				    "%s", "kor:SP 빨강 테스트 전용");
				fprintf(WFA_OUT, "osu_friendly_name[0] %s\n", def_intf->ppr.osu_friendly_name[0]);
				fprintf(WFA_OUT, "osu_friendly_name[1] %s\n", def_intf->ppr.osu_friendly_name[1]);

				snprintf(def_intf->ppr.osu_service_desc[0], sizeof(def_intf->ppr.osu_service_desc[0]),
				    "%s", "eng:Free service for test purpose");
				snprintf(def_intf->ppr.osu_service_desc[1], sizeof(def_intf->ppr.osu_service_desc[1]),
				    "%s", "kor:테스트 목적으로 무료 서비스");
				fprintf(WFA_OUT, "osu_service_desc[0] %s\n", def_intf->ppr.osu_service_desc[0]);
				fprintf(WFA_OUT, "osu_service_desc[1] %s\n", def_intf->ppr.osu_service_desc[1]);
			} else if (strcasecmp((value_ptr)[i], "9") == 0) {
				snprintf(def_intf->ppr.hs20_icon[0], sizeof(def_intf->ppr.hs20_icon[0]),
				    "%s", "128:61:zxx:image/png:icon_orange_zxx.png:/etc/icon_orange_zxx.png");
				fprintf(WFA_OUT, "hs20_icon[0] %s\n", def_intf->ppr.hs20_icon[0]);

				snprintf(def_intf->ppr.osu_nai[0], sizeof(def_intf->ppr.osu_nai[0]),
				    "%s", "test-anonymous@wi-fi.org");
				fprintf(WFA_OUT, "osu_nai[0] %s\n", def_intf->ppr.osu_nai[0]);

				snprintf(def_intf->ppr.osu_icon[0], sizeof(def_intf->ppr.osu_icon[0]),
				    "%s", "icon_orange_zxx.png");
				fprintf(WFA_OUT, "osu_icon[0] %s\n", def_intf->ppr.osu_icon[0]);

				snprintf(def_intf->ppr.osu_friendly_name[0], sizeof(def_intf->ppr.osu_friendly_name[0]),
				    "%s", "eng:SP Orange Test Only");
				snprintf(def_intf->ppr.osu_friendly_name[1], sizeof(def_intf->ppr.osu_friendly_name[1]),
				    "%s", "kor:SP 빨강 테스트 전용");
				fprintf(WFA_OUT, "osu_friendly_name[0] %s\n", def_intf->ppr.osu_friendly_name[0]);
				fprintf(WFA_OUT, "osu_friendly_name[1] %s\n", def_intf->ppr.osu_friendly_name[1]);

				snprintf(def_intf->ppr.osu_service_desc[0], sizeof(def_intf->ppr.osu_service_desc[0]),
				    "%s", "eng:Free service for test purpose");
				snprintf(def_intf->ppr.osu_service_desc[1], sizeof(def_intf->ppr.osu_service_desc[1]),
				    "%s", "kor:테스트 목적으로 무료 서비스");
				fprintf(WFA_OUT, "osu_service_desc[0] %s\n", def_intf->ppr.osu_service_desc[0]);
				fprintf(WFA_OUT, "osu_service_desc[1] %s\n", def_intf->ppr.osu_service_desc[1]);
			} else if (strcasecmp((value_ptr)[i], "10") == 0) {
				snprintf(def_intf->ppr.osu_nai2[0], sizeof(def_intf->ppr.osu_nai2[0]),
				    "%s", "anonymous@hotspot.net");
				snprintf(def_intf->ppr.osu_nai2[1], sizeof(def_intf->ppr.osu_nai2[1]),
				    "%s", "test-anonymous@wi-fi.org");

				snprintf(def_intf->ppr.osu_icon[0], sizeof(def_intf->ppr.osu_icon[0]),
				    "%s", "icon_red_zxx.png");
				snprintf(def_intf->ppr.osu_icon[1], sizeof(def_intf->ppr.osu_icon[1]),
				    "%s", "icon_orange_zxx.png");

				def_intf->ppr.osu_method_list[0] = 1;
				def_intf->ppr.osu_method_list[1] = 1;

				snprintf(def_intf->ppr.osu_nai[0], sizeof(def_intf->ppr.osu_nai),
				    "%s", "anonymous@hotspot.net");
				snprintf(def_intf->ppr.osu_nai[1], sizeof(def_intf->ppr.osu_nai),
				    "%s", "test-anonymous@wi-fi.org");
				snprintf(def_intf->ppr.osu_friendly_name[0],
				    sizeof(def_intf->ppr.osu_friendly_name[0]),
				    "%s", "kor:SP 빨강 테스트 전용");
				snprintf(def_intf->ppr.osu_friendly_name[1],
				    sizeof(def_intf->ppr.osu_friendly_name[1]),
				    "%s", "eng:SP Red Test Only");
				snprintf(def_intf->ppr.osu_friendly_name[2],
				    sizeof(def_intf->ppr.osu_friendly_name[2]),
				    "%s", "kor:SP 오렌지 테스트 전용");
				snprintf(def_intf->ppr.osu_friendly_name[3],
				    sizeof(def_intf->ppr.osu_friendly_name[3]),
				    "%s", "eng:SP Orange Test Only");

				snprintf(def_intf->ppr.osu_service_desc[0],
				    sizeof(def_intf->ppr.osu_service_desc[0]),
				    "%s", "eng:Free service for test purpose");
				snprintf(def_intf->ppr.osu_service_desc[1],
				    sizeof(def_intf->ppr.osu_service_desc[1]),
				    "%s", "kor:테스트 목적으로 무료 서비스");
				snprintf(def_intf->ppr.osu_server_uri,
				    sizeof(def_intf->ppr.osu_server_uri), "%s",
				    "https://osu-server.r2-testbed-rks.wi-fi.org:9446/OnlineSignup/services/newUser/digest");


				def_intf->ppr.osu_nai2_configured = 1;

				snprintf(def_intf->ppr.hs20_icon[0], sizeof(def_intf->ppr.hs20_icon[0]),
				    "%s", "128:61:zxx:image/png:icon_red_zxx.png:/etc/icon_red_zxx.png");
				snprintf(def_intf->ppr.hs20_icon[1], sizeof(def_intf->ppr.hs20_icon[1]),
				    "%s", "128:61:zxx:image/png:icon_orange_zxx.png:/etc/icon_orange_zxx.png");

				for (param_idx = 0; param_idx < 2; param_idx++) {
					fprintf(WFA_OUT, "osu_nai2[%d] %s\n",
					    param_idx, def_intf->ppr.osu_nai2[param_idx]);
					fprintf(WFA_OUT, "osu_icon[%d] %s\n",
					    param_idx, def_intf->ppr.osu_icon[param_idx]);
					fprintf(WFA_OUT, "osu_method_list[%d] %d\n",
					    param_idx, def_intf->ppr.osu_method_list[param_idx]);
					fprintf(WFA_OUT, "osu_nai[%d] %s\n",
					    param_idx, def_intf->ppr.osu_nai[param_idx]);
					fprintf(WFA_OUT, "osu_service_desc[%d] %s\n",
					    param_idx, def_intf->ppr.osu_service_desc[param_idx]);
					fprintf(WFA_OUT, "osu_friendly_name[%d] %s\n",
					    param_idx, def_intf->ppr.osu_friendly_name[param_idx]);
				}
				fprintf(WFA_OUT, "osu_friendly_name[2] %s\n",
					def_intf->ppr.osu_friendly_name[2]);
				fprintf(WFA_OUT, "osu_friendly_name[3] %s\n",
					def_intf->ppr.osu_friendly_name[3]);
			} else
				fprintf(WFA_OUT, "HOSTAPD: OSU_PROVIDER_LIST not configured for ID#%s\n",
					    (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "OSU_PROVIDER_NAI_LIST") == 0) {
			if (def_intf->ppr.osu_nai2_configured == 1)
				continue;

			memset(def_intf->ppr.osu_nai2[0], 0, sizeof(def_intf->ppr.osu_nai2[0]));
			memset(def_intf->ppr.osu_nai2[1], 0, sizeof(def_intf->ppr.osu_nai2[1]));
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.osu_nai2[0], sizeof(def_intf->ppr.osu_nai2[0]),
				    "%s", "anonymous@hotspot.net");
				fprintf(WFA_OUT, "osu_nai2[0] %s\n", def_intf->ppr.osu_nai2[0]);
			} else if (strcasecmp((value_ptr)[i], "2") == 0) {
				snprintf(def_intf->ppr.osu_nai2[0], sizeof(def_intf->ppr.osu_nai2[0]),
				    "%s", "test-anonymous@wi-fi.org");
				fprintf(WFA_OUT, "osu_nai2[0] %s\n", def_intf->ppr.osu_nai2[0]);
			} else if (strcasecmp((value_ptr)[i], "3") == 0) {
				snprintf(def_intf->ppr.osu_nai2[0], sizeof(def_intf->ppr.osu_nai2[0]),
				    "%s", "test-anonymous@wi-fi.org");
				snprintf(def_intf->ppr.osu_nai2[1], sizeof(def_intf->ppr.osu_nai2[1]),
				    "%s", "anonymous@hotspot.net");
				fprintf(WFA_OUT, "osu_nai2[0] %s\n", def_intf->ppr.osu_nai2[0]);
				fprintf(WFA_OUT, "osu_nai2[1] %s\n", def_intf->ppr.osu_nai2[1]);
			} else if (strcasecmp((value_ptr)[i], "4") == 0) {
				snprintf(def_intf->ppr.osu_nai2[0], sizeof(def_intf->ppr.osu_nai2[0]),
				    "%s", "random@hotspot.net");
				snprintf(def_intf->ppr.osu_nai2[1], sizeof(def_intf->ppr.osu_nai2[1]),
				    "%s", "anonymous@hotspot.net");
				fprintf(WFA_OUT, "osu_nai2[0] %s\n", def_intf->ppr.osu_nai2[0]);
				fprintf(WFA_OUT, "osu_nai2[1] %s\n", def_intf->ppr.osu_nai2[1]);
			} else
				fprintf(WFA_OUT,
					    "HOSTAPD: OSU_PROVIDER_NAI_LIST not configured for ID#%s\n",
					    (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "OSU_SERVER_URI") == 0) {
			strcpy(mtk_ap_buf->osu_server_uri, (value_ptr)[i]);
			snprintf(def_intf->ppr.osu_server_uri, sizeof(def_intf->ppr.osu_server_uri),
			    "%s", (value_ptr)[i]);
			fprintf(WFA_OUT, "osu_server_uri %s\n", def_intf->ppr.osu_server_uri);
		} else if (strcasecmp((data->params)[i], "PLMN_MCC") == 0) {
			strcpy(mtk_ap_buf->plm_mcc, (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "PLMN_MNC") == 0) {
			char *token = NULL, *token1 = NULL;
			int mcc[10] = {0};
			char *p = def_intf->ppr.anqp_3gpp_cell_net;

			offset = 0;
			j = 0;
			token = strtok(mtk_ap_buf->plm_mcc, ";");
			while (token != NULL) {
				mcc[j] = atoi(token);
				token = strtok(NULL, ";");
				j++;
			}
			token1 = strtok((value_ptr)[i], ";");
			j = 0;
			while (token1 != NULL) {
				offset += snprintf(p+offset,
					sizeof(def_intf->ppr.anqp_3gpp_cell_net)-offset,
					"%d,%s;", mcc[j], token1);
				token1 = strtok(NULL, ";");
				j++;
			}

			if (*(p+offset-1) == ';')
				*(p+offset-1) = '\0';
		} else if (strcasecmp((data->params)[i], "Proxy_ARP") == 0) {
			if ((strcasecmp((value_ptr)[i], "enabled") == 0) || (strcasecmp((value_ptr)[i], "1") == 0)) {
				def_intf->ppr.proxy_arp = 1;
				fprintf(WFA_OUT, "proxy_arp %d\n", def_intf->ppr.proxy_arp);
			} else if ((strcasecmp((value_ptr)[i], "disabled") == 0) ||
				   (strcasecmp((value_ptr)[i], "0") == 0)) {
				def_intf->ppr.proxy_arp = 0;
				fprintf(WFA_OUT, "proxy_arp %d\n", def_intf->ppr.proxy_arp);
			}

			snprintf(gCmdStr, sizeof(gCmdStr), "iwpriv %s set proxyarp_enable=%d",
			    def_intf->name, def_intf->ppr.proxy_arp);
			add_post_cmd(mtk_ap_buf);
		} else if (strcasecmp((data->params)[i], "QoS_MAP_SET") == 0) {
			if (mtk_ap_buf->client_mac != NULL) {
				if (strcasecmp((value_ptr)[i], "1") == 0) {
					snprintf(def_intf->ppr.qos_map_set, sizeof(def_intf->ppr.qos_map_set),
					    "%s", "53,2,22,6,8,15,0,7,255,255,16,31,32,39,255,255,40,47,255,255");
					snprintf(gCmdStr, sizeof(gCmdStr),
					    "iwpriv %s set configure_qosmap=\"%s 53:2:22:6 "
					    "8:15:0:7:255:255:16:31:32:39:255:255:40:47:48:63\"\n",
					    intf, mtk_ap_buf->client_mac);
					fprintf(WFA_OUT, "run command ==> %s", gCmdStr);
					system(gCmdStr);
				} else if (strcasecmp((value_ptr)[i], "2") == 0) {
					snprintf(def_intf->ppr.qos_map_set, sizeof(def_intf->ppr.qos_map_set),
					    "%s", "8,15,0,7,255,255,16,31,32,39,255,255,40,47,48,63");
					snprintf(gCmdStr, sizeof(gCmdStr),
						"iwpriv %s set configure_qosmap=\"%s "
						"8:15:0:7:255:255:16:31:32:39:255:255:40:47:48:63\"\n",
						intf, mtk_ap_buf->client_mac);
					fprintf(WFA_OUT, "run command ==> %s", gCmdStr);
					system(gCmdStr);
				}
				DPRINT_INFO(WFA_OUT, "qos_map_set %s\n", def_intf->ppr.qos_map_set);
			} else {
				if (strcasecmp((value_ptr)[i], "1") == 0)
					snprintf(def_intf->ppr.qos_map_set, sizeof(def_intf->ppr.qos_map_set),
					    "%s", "53,2,22,6,8,15,0,7,255,255,16,31,32,39,255,255,40,47,255,255");
				else if (strcasecmp((value_ptr)[i], "2") == 0)
					snprintf(def_intf->ppr.qos_map_set, sizeof(def_intf->ppr.qos_map_set),
					    "%s", "8,15,0,7,255,255,16,31,32,39,255,255,40,47,48,63");

				fprintf(WFA_OUT, "qos_map_set %s\n", def_intf->ppr.qos_map_set);
			}
		} else if (strcasecmp((data->params)[i], "Roaming_Cons") == 0) {
			memset(def_intf->ppr.roaming_consortium, 0, sizeof(def_intf->ppr.roaming_consortium));

			if (strcasecmp((value_ptr)[i], "Disabled") != 0) {
				int offset = 0, Len = 0, k = 0;
				char cmd_rc[60], *roaming_consortium = NULL;

				memset(cmd_rc, 0, sizeof(cmd_rc));
				Len = strlen((value_ptr)[i]);
				roaming_consortium = (value_ptr)[i];
				for (k = 0; k < Len; k += 2) {
					offset += snprintf(cmd_rc+offset, sizeof(cmd_rc)-offset-1, &roaming_consortium[k], 2);
					if (k < Len - 2)
						offset += snprintf(cmd_rc+offset, sizeof(cmd_rc)-offset-1, "-");
				}

				snprintf(def_intf->ppr.roaming_consortium[0],
				    sizeof(def_intf->ppr.roaming_consortium),
				    "%s", roaming_consortium);
				fprintf(WFA_OUT, "roaming_consortium[0] %s\n", def_intf->ppr.roaming_consortium[0]);
			}
		} else if (strcasecmp((data->params)[i], "STA_MAC") == 0) {
			mtk_ap_buf->client_mac = (value_ptr)[i];
			fprintf(WFA_OUT, "STA_MAC %s\n", mtk_ap_buf->client_mac);
		} else if (strcasecmp((data->params)[i], "TnC_File_Name") == 0) {
			snprintf(def_intf->ppr.hs20_t_c_filename, sizeof(def_intf->ppr.hs20_t_c_filename),
			    "%s", "tandc-id1-content.txt");
			fprintf(WFA_OUT, "hs20_t_c_filename %s\n", def_intf->ppr.hs20_t_c_filename);
		} else if (strcasecmp((data->params)[i], "TnC_File_Time_Stamp") == 0) {
			snprintf(def_intf->ppr.hs20_t_c_timestamp, sizeof(def_intf->ppr.hs20_t_c_timestamp),
			    "%s", (value_ptr)[i]);
			fprintf(WFA_OUT, "hs20_t_c_timestamp %s\n", def_intf->ppr.hs20_t_c_timestamp);
		} else if (strcasecmp((data->params)[i], "Venue_Name") == 0) {
			/* ToDo: add for hostapd */
		} else if (strcasecmp((data->params)[i], "Venue_Type") == 0) {
			/* ToDo: add for hostapd */
		} else if (strcasecmp((data->params)[i], "Venue_URL") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0) {
				snprintf(def_intf->ppr.venue_url[0], sizeof(def_intf->ppr.venue_url[0]),
				    "%s", "1:https://venue-server.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[1], sizeof(def_intf->ppr.venue_url[1]),
				    "%s", "1:https://venue-server.r2m-testbed.wi-fi.org/directory/index.html");
				for (param_idx = 0; param_idx < 2; param_idx++)
					fprintf(WFA_OUT, "venue_url[%d] %s\n",
					    param_idx, def_intf->ppr.venue_url[param_idx]);
			} else if (strcasecmp((value_ptr)[i], "2") == 0) {
				snprintf(def_intf->ppr.venue_url[0], sizeof(def_intf->ppr.venue_url[0]),
				    "%s", "1:https://the-great-mall.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[1], sizeof(def_intf->ppr.venue_url[1]),
				    "%s", "2:https://abercrombie.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[2], sizeof(def_intf->ppr.venue_url[2]),
				    "%s", "3:https://adidas.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[3], sizeof(def_intf->ppr.venue_url[3]),
				    "%s", "4:https://aeropostale.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[4], sizeof(def_intf->ppr.venue_url[4]),
				    "%s", "5:https://agaci.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[5], sizeof(def_intf->ppr.venue_url[5]), "%s",
				    "6:https://aldo-shoes.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[6], sizeof(def_intf->ppr.venue_url[6]),
				    "%s", "7:https://american-eagle-outfitters.r2m-testbed.wi-fi.org/floorplans/");
				snprintf(def_intf->ppr.venue_url[7], sizeof(def_intf->ppr.venue_url[7]),
				    "%s", "8:https://anderson-bakery.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[8], sizeof(def_intf->ppr.venue_url[8]), "%s",
				    "9:https://banana-republic-factory-store.r2m-testbed.wi-fi.org/floorplans/index.html");
				snprintf(def_intf->ppr.venue_url[9], sizeof(def_intf->ppr.venue_url[9]), "%s",
				    "10:https://bed-bath-and-beyond.r2m-testbed.wi-fi.org/floorplans/index.html");

				for (param_idx = 0; param_idx < 10; param_idx++)
					fprintf(WFA_OUT, "venue_url[%d] %s\n",
						param_idx, def_intf->ppr.venue_url[param_idx]);
			} else
				fprintf(WFA_OUT, "HOSTAPD: Venue_URL not configured for ID#%s\n",
					    (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "WAN_Metrics") == 0) {
			if (strcasecmp((value_ptr)[i], "1") == 0)
				snprintf(def_intf->ppr.hs20_wan_metrics,
				    sizeof(def_intf->ppr.hs20_wan_metrics), "%s", "01:2500:384:0:0:10");
			else if (strcasecmp((value_ptr)[i], "2") == 0)
				snprintf(def_intf->ppr.hs20_wan_metrics,
				    sizeof(def_intf->ppr.hs20_wan_metrics), "%s", "01:1500:384:20:20:10");
			else if (strcasecmp((value_ptr)[i], "3") == 0)
				snprintf(def_intf->ppr.hs20_wan_metrics,
				    sizeof(def_intf->ppr.hs20_wan_metrics), "%s", "01:2000:1000:20:20:10");
			else if (strcasecmp((value_ptr)[i], "4") == 0)
				snprintf(def_intf->ppr.hs20_wan_metrics,
				    sizeof(def_intf->ppr.hs20_wan_metrics), "%s", "01:8000:1000:20:20:10");
			else if (strcasecmp((value_ptr)[i], "5") == 0)
				snprintf(def_intf->ppr.hs20_wan_metrics,
				    sizeof(def_intf->ppr.hs20_wan_metrics), "%s", "01:9000:5000:20:20:10");
			else
				fprintf(WFA_OUT, "Unknown WAN Type\n");
			fprintf(WFA_OUT, "hs20_wan_metrics %s\n", def_intf->ppr.hs20_wan_metrics);
		}
	}

	return WFA_SUCCESS;
}


int hostapd_ap_set_11d(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = NULL;
	char **value_ptr = NULL;
	int i = 0;

	DPRINT_INFO(WFA_OUT, "===== running %s function =====\n", __func__);

	value_ptr = data->values;
	def_intf = mtk_ap_buf->def_intf;

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "CountryCode") == 0) {
			strncpy(def_intf->country_code, (value_ptr)[i], sizeof(def_intf->country_code)-1);
			def_intf->country_code[sizeof(def_intf->country_code)-1] = '\0';
			DPRINT_INFO(WFA_OUT, "country code %s\n", def_intf->country_code);
		} else if (strcasecmp((data->params)[i], "Regulatory_Mode") == 0) {
			DPRINT_INFO(WFA_OUT, "Set %s to %s", (data->params)[i], (value_ptr)[i]);
			DPRINT_INFO(WFA_OUT, ",do nothing!\n");
		} else {
			DPRINT_INFO(WFA_OUT, "param %s", (data->params)[i]);
			DPRINT_INFO(WFA_OUT, " not support!\n");
		}
	}
	return WFA_SUCCESS;
}

int hostapd_dev_send_frame(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr;
	char intf[10], FrameName[20], Dest_MAC[20];
	unsigned char RegClass = 0, Channel = 0, BSSID[6];
	unsigned short RandInt = 0, MeaDur = 0;
	char MeaMode = 0, ssid_hex_str[WFA_BUFF_64];
	char RptCond_hex_str[WFA_BUFF_64], RptDet_hex_str[WFA_BUFF_64];
	char APChanRpt_hex_str[WFA_BUFF_64], ReqInfo_hdex_str[WFA_BUFF_64];
	char LastBeaconRptInd_hex_str[WFA_BUFF_64];
	int CandList = 0;
	char cmd_content[WFA_CMD_STR_SZ];
	unsigned int mac_int[6] = {0};
	int i = 0, j = 0;
	int offset = 0, ret_len = 0;
	char keyword[20] = {0}, framebody[256] = {0};

	DPRINT_INFO(WFA_OUT, "===== running %s function =====\n", __func__);
	value_ptr = data->values;

	if (strcasecmp(mtk_ap_buf->cmd_cfg.program, "QM") == 0) {
		for (i = 0; i < data->count; i++) {
			if ((strcasecmp((data->params)[i], "FrameName") == 0) &&
			    ((strcasecmp((value_ptr)[i], "DSCPPolicyReq") == 0) ||
			     (strcasecmp((value_ptr)[i], "SCSResp") == 0))) {
				memset(keyword, 0, sizeof(keyword));
				memset(framebody, 0, sizeof(framebody));

				memcpy(keyword, (value_ptr)[i], min(sizeof(keyword), strlen((value_ptr)[i])));
				while (i++ < data->count && (data->params)[i] && (value_ptr)[i]) {
					snprintf(framebody + strlen(framebody), sizeof(framebody) - strlen(framebody),
						 "%s,%s,", (data->params)[i], (value_ptr)[i]);
				}

				mtk_send_qos_cmd_to_wapp(def_intf->name, "qos_ap_set_config", keyword,
							 framebody);
			}
		}

		return WFA_SUCCESS;
	}

	memset(RptCond_hex_str, 0, sizeof(RptCond_hex_str));
	memset(RptDet_hex_str, 0, sizeof(RptDet_hex_str));
	memset(APChanRpt_hex_str, 0, sizeof(APChanRpt_hex_str));
	memset(ReqInfo_hdex_str, 0, sizeof(ReqInfo_hdex_str));
	memset(LastBeaconRptInd_hex_str, 0, sizeof(LastBeaconRptInd_hex_str));
	memset(ssid_hex_str, 0, sizeof(ssid_hex_str));
	memset(cmd_content, 0, sizeof(cmd_content));
	memset(BSSID, 0, sizeof(BSSID));
	memset(mac_int, 0, sizeof(mac_int));
	snprintf(intf, sizeof(intf) - 1, mtk_ap_buf->def_intf->name);

	if ((strcasecmp(data->interface, "5G") == 0) || (strcasecmp(data->interface, "5.0") == 0)) {
		if (mtk_ap_buf->intf_5G.status)
			snprintf(intf, sizeof(intf) - 1, mtk_ap_buf->intf_5G.name);
		else
			DPRINT_INFO(WFA_OUT, "5G interface is not supported, skip!\n");
	} else if ((strcasecmp(data->interface, "2G") == 0) || (strcasecmp(data->interface, "24G") == 0) ||
		   (strcasecmp(data->interface, "2.4") == 0)) {
		if (mtk_ap_buf->intf_2G.status)
			snprintf(intf, sizeof(intf) - 1, mtk_ap_buf->intf_2G.name);
		else
			DPRINT_INFO(WFA_OUT, "2G interface is not supported, skip!\n");
	} else if (strcasecmp(data->interface, "6G") == 0) {
		if (mtk_ap_buf->intf_6G.status)
			snprintf(intf, sizeof(intf) - 1, mtk_ap_buf->intf_6G.name);
		else
			DPRINT_INFO(WFA_OUT, "6G interface is not supported, skip!\n");
	}

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "FrameName") == 0)
			snprintf(FrameName, sizeof(FrameName) - 1, (value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "Dest_MAC") == 0)
			snprintf(Dest_MAC, sizeof(Dest_MAC) - 1, (value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "DestMAC") == 0)
			snprintf(Dest_MAC, sizeof(Dest_MAC) - 1, (value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "Cand_List") == 0)
			CandList = 1;
		else if (strcasecmp((data->params)[i], "RegClass") == 0) {
			RegClass = (unsigned char)atoi((value_ptr)[i]);
			def_intf->frame_opclass = RegClass;
		} else if (strcasecmp((data->params)[i], "Channel") == 0)
			Channel = (unsigned char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "OCIChannel") == 0)
			Channel = (unsigned char)atoi((value_ptr)[i]);
		else if (strcasecmp((data->params)[i], "RandInt") == 0) {
			RandInt = (unsigned short)atoi((value_ptr)[i]);
			RandInt = ((RandInt & 0xff) << 8) | (RandInt >> 8);
		} else if (strcasecmp((data->params)[i], "MeaDur") == 0) {
			MeaDur = (unsigned short)atoi((value_ptr)[i]);
			MeaDur = ((MeaDur & 0xff) << 8) | (MeaDur >> 8);
		} else if (strcasecmp((data->params)[i], "MeaMode") == 0) {
			if (strcasecmp((value_ptr)[i], "PASSIVE") == 0)
				MeaMode = 0;
			else if (strcasecmp((value_ptr)[i], "ACTIVE") == 0)
				MeaMode = 1;
			else if (strcasecmp((value_ptr)[i], "TABLE") == 0)
				MeaMode = 2;
		} else if (strcasecmp((data->params)[i], "BSSID") == 0) {
			ret_len = sscanf((value_ptr)[i], "%02x:%02x:%02x:%02x:%02x:%02x", mac_int, (mac_int + 1),
					 (mac_int + 2), (mac_int + 3), (mac_int + 4), (mac_int + 5));
			if (ret_len != 6)
				DPRINT_INFO(WFA_OUT, "sscanf of BSSID not match\n");

			for (j = 0; j < 6; j++)
				BSSID[j] = (unsigned char)mac_int[j];
			DPRINT_INFO(WFA_OUT, "BSSID " MACSTR "\n", MAC2STR(BSSID));
		} else if (strcasecmp((data->params)[i], "SSID") == 0) {
			ret_len = hostapd_parameter_to_tlv(ssid_hex_str, sizeof(ssid_hex_str), (value_ptr)[i],
							   strlen((value_ptr)[i]), TLV_TYPE_SSID, def_intf);
			if (ret_len >= sizeof(ssid_hex_str)) {
				DPRINT_INFO(WFA_OUT, "SSID is too long!\n");
			}
		} else if (strcasecmp((data->params)[i], "RptCond") == 0) {
			ret_len = hostapd_parameter_to_tlv(RptCond_hex_str, sizeof(RptCond_hex_str), (value_ptr)[i],
							   strlen((value_ptr)[i]), TLV_TYPE_BECON_REPT_INF, def_intf);
			if (ret_len >= sizeof(RptCond_hex_str)) {
				DPRINT_INFO(WFA_OUT, "RptCond is too long!\n");
			}
		} else if (strcasecmp((data->params)[i], "RptDet") == 0) {
			ret_len = hostapd_parameter_to_tlv(RptDet_hex_str, sizeof(RptDet_hex_str), (value_ptr)[i],
							   strlen((value_ptr)[i]), TLV_TYPE_BECON_REPT_DET, def_intf);
			if (ret_len >= sizeof(RptDet_hex_str)) {
				DPRINT_INFO(WFA_OUT, "RptDet is too long!\n");
			}
		} else if (strcasecmp((data->params)[i], "MeaDurMand") == 0) {
			DPRINT_INFO(WFA_OUT, "MeaDurMand %s\n", (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "APChanRpt") == 0) {
			ret_len = hostapd_parameter_to_tlv(APChanRpt_hex_str, sizeof(APChanRpt_hex_str), (value_ptr)[i],
							   strlen((value_ptr)[i]), TLV_TYPE_AP_CHAN_RPT, def_intf);
			if (ret_len >= sizeof(APChanRpt_hex_str))
				DPRINT_INFO(WFA_OUT, "APChanRpt is too long!\n");
		} else if (strcasecmp((data->params)[i], "LastBeaconRptIndication") == 0) {
			ret_len = hostapd_parameter_to_tlv(LastBeaconRptInd_hex_str, sizeof(LastBeaconRptInd_hex_str),
							   (value_ptr)[i], strlen((value_ptr)[i]),
							   TLV_TYPE_BECON_REPT_IND_REQ, def_intf);
			if (ret_len >= sizeof(LastBeaconRptInd_hex_str)) {
				DPRINT_INFO(WFA_OUT, "LastBeaconRptIndication is too long!\n");
			}
		} else if (strcasecmp((data->params)[i], "ReqInfo") == 0) {
			ret_len = hostapd_parameter_to_tlv(ReqInfo_hdex_str, sizeof(ReqInfo_hdex_str), (value_ptr)[i],
							   strlen((value_ptr)[i]), TLV_TYPE_REQ_INFO, def_intf);
			if (ret_len >= sizeof(ReqInfo_hdex_str)) {
				DPRINT_INFO(WFA_OUT, "ReqInfo is too long!\n");
			}
		} else if (strcasecmp((data->params)[i], "type") == 0)
			continue;
		else {
			DPRINT_INFO(WFA_OUT, "dev_send_frame %s  Command is ignored or invalid!\n", data->params[i]);
		}
	}

	if (strcasecmp(FrameName, "BTMReq") == 0) {
		offset = 0;
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "bss_tm_req");
		/* dst mac */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, " %s ", Dest_MAC);
		/* disassoc_timer */
		if (def_intf->BTMReq_Term_Bit)
			offset +=
			    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "disassoc_timer=5 ");
		else if (def_intf->BTMReq_DisAssoc_Imnt)
			offset +=
			    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "disassoc_timer=10 ");
		else
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "disassoc_timer=0 ");

		/* valid_int */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "valid_int=200 ");
		/* bss_term */
		if (def_intf->BTMReq_Term_Bit)
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "bss_term=%d, %x ",
					   def_intf->BTMReq_BSS_Term_TSF, def_intf->BSS_Term_Duration);

		if (l1_valid) {
			int nb_idx = 0;
			/*
			* BSS Transition Candidate List Entries - Neighbor Report elements
			* neighbor=<BSSID eg: 00:20:30:40:50:60>,<BSSID Information - e.g 000>,<Operating Class - e.g 73>,
			* <Channel Number - e.g 24>,<PHY Type - 09>[,<Optional Subelements as pref - e.g - 0301ff>]
			*/
			for (nb_idx = 0; nb_idx < def_intf->nr_count; nb_idx++) {
				if (nb_idx == def_intf->nr_count)
					break;

				if (!def_intf->neighbor[nb_idx].op_ch)
					break;

				/* BSSID */
				offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "neighbor="MACSTR",",
					MAC2STR(def_intf->neighbor[nb_idx].bssid));
				/* BSSID Information - e.g 000 */
				offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "000,");
				/* <Operating Class> */
				offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%d,",
					def_intf->neighbor[nb_idx].op_class);
				/* <Channel Number> */
				offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%d,",
					def_intf->neighbor[nb_idx].op_ch);
				/* <PHY Type> */
				offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "09,");
				/* <PHY Type> */
				offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "0301%02x ",
					def_intf->neighbor[nb_idx].pref);
			}
		}

		/* pref */
		if (CandList)
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "pref=1 ");

		/* disassoc_imminent */
		if (def_intf->BTMReq_DisAssoc_Imnt)
			offset +=
			    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "disassoc_imminent=1 ");

		/* mbo */
		if (def_intf->BTMReq_DisAssoc_Imnt)
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "mbo=0:10:1 ");
		else
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "mbo=0:0:1 ");

		hostapd_cli_cmd(def_intf, cmd_content);

		if (def_intf->BTMReq_BSS_Term_TSF) {
			memset(gCmdStr, 0, sizeof(gCmdStr));
			snprintf(gCmdStr, sizeof(gCmdStr),
					"mwctl %s set mbo_beacon_off=1", def_intf->name);
			DPRINT_INFO(WFA_OUT, "run command ==> %s\n", gCmdStr);
			system(gCmdStr);
		}
		def_intf->BTMReq_BSS_Term_TSF = 0;
		def_intf->BTMReq_Term_Bit = 0;
		def_intf->BSS_Term_Duration = 0;
		def_intf->BTMReq_DisAssoc_Imnt = 0;
	} else if (strcasecmp(FrameName, "BcnRptReq") == 0) {
		offset = 0;
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "req_beacon");
		/* dst mac */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, " %s ", Dest_MAC);
		/* req mode */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "req_mode=00 ");
		/* RegClass */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", RegClass);
		/* Channel */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", Channel);
		/* RandInt */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%04x", RandInt);
		/* MeaDur */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%04x", MeaDur);
		/* MeaMode */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x", MeaMode);
		/* BSSID */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%02x%02x%02x%02x%02x%02x",
				   BSSID[0], BSSID[1], BSSID[2], BSSID[3], BSSID[4], BSSID[5]);
		/* hex ssid */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%s", ssid_hex_str);
		/* RptCond */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%s", RptCond_hex_str);
		/* RptDet */
		offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%s", RptDet_hex_str);
		/* APChanRpt */
		if (strlen(APChanRpt_hex_str)) {
			offset +=
			    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%s", APChanRpt_hex_str);
		}
		/* ReqInfo */
		if (strlen(ReqInfo_hdex_str)) {
			offset +=
			    snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%s", ReqInfo_hdex_str);
		}
		/* LastBeaconRptIndication */
		if (strlen(LastBeaconRptInd_hex_str))
			offset += snprintf(cmd_content + offset, sizeof(cmd_content) - offset - 1, "%s",
					   LastBeaconRptInd_hex_str);
		hostapd_cli_cmd(def_intf, cmd_content);
	}
	return WFA_SUCCESS;
}

int hostapd_dev_exec_action(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	char **value_ptr;
	char result[48];
	char intf[10];
	int i, j;
	unsigned int rejected_group = 0;
	char STA_MAC[20];
	FILE *fp;
	char *start;

	strcpy(result, "");

	memset(STA_MAC, 0, sizeof(STA_MAC));
	strcpy(intf, mtk_ap_buf->def_intf->name);
	value_ptr = data->values;
	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "Rejected_DH_Groups") == 0) {
			if (STA_MAC == NULL) {
				printf("STA_MAC_Address can't be NULL, return!!!\n");
				return WFA_ERROR;
			}
			sprintf(gCmdStr, "iwpriv %s show sae_rejected_group=%s\n", intf, STA_MAC);
			DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
			fp = popen(gCmdStr, "r");
			if (fp == NULL) {
				printf("Failed to run command!!!\n");
				return WFA_ERROR;
			}

			while (fgets(gCmdStr, sizeof(gCmdStr), fp) != NULL) {
				start = strstr(gCmdStr, "show:");
				if (start == NULL)
					continue;
				/* Replace the last character from '\n' to '\0' */
				start = start + 5;
				start[strlen(start) - 1] = '\0';
				rejected_group = (unsigned int)strtol(start, NULL, 16);
				strcpy(result, ",Rejected_DH_Groups,");
				for (j = 0; j < 32; j++) {
					if (rejected_group & (1 << j))
						sprintf(result + strlen(result), "%d ", j);
				}
				if (strlen(result) > 1)
					result[strlen(result) - 1] = '\0';

				break;
			}
			pclose(fp);
		} else if (strcasecmp((data->params)[i], "Dest_MAC") == 0) {
			strcpy(STA_MAC, (value_ptr)[i]);
		} else if (strcasecmp((data->params)[i], "KeyRotation") == 0) {
			char rekey_str[WFA_BUFF_128] = {0};

			if ((strcasecmp((value_ptr)[i], "1") == 0)) {
				memset(rekey_str, 0, sizeof(rekey_str));
				strcpy(rekey_str, "force_rekey_gtk");
				sprintf(gCmdStr, "hostapd_cli -i %s -p %s %s\n",
					mtk_ap_buf->def_intf->name, mtk_ap_buf->def_intf->ctrl_inf, rekey_str);
				DPRINT_INFO(WFA_OUT, "run command ==> %s", gCmdStr);
				system(gCmdStr);
			}
		}
	}
	sprintf((char *)resp_buf, "%s", result);
	printf("%s\n", resp_buf);
	return WFA_SUCCESS;
}

int hostapd_ap_get_parameter(int len, uint8_t *ap_buf, int *resp_len_ptr, uint8_t *resp_buf)
{
	char **value_ptr = NULL;
	char result[200] = {0};
	int i = 0;
	char STA_MAC[20] = {0};
	char PMK[100] = {0};
	FILE *fp = NULL;
	char *start = NULL;
	mtk_ap_buf_t *mtk_ap_buf = (mtk_ap_buf_t *)ap_buf;
	capi_data_t *data = mtk_ap_buf->capi_data;
	intf_desc_t *def_intf = mtk_ap_buf->def_intf;

	DPRINT_INFO(WFA_OUT, "===== running hostapd_ap_get_parameter function =====\n");

	memset(result, 0, sizeof(result));
	memset(STA_MAC, 0, sizeof(STA_MAC));
	value_ptr = data->values;

	for (i = 0; i < data->count; i++) {
		if (strcasecmp((data->params)[i], "Parameter") == 0) {
			if (strcasecmp((value_ptr)[i], "SSID") == 0) {
				fp = popen("iwconfig ra0", "r");
				if (fp == NULL) {
					printf("Failed to run command!!!\n");
					return WFA_ERROR;
				}

				while (fgets(gCmdStr, sizeof(gCmdStr), fp) != NULL) {
					start = strstr(gCmdStr, "ESSID:");
					if (start == NULL)
						continue;
					/* Replace the last character from '\n' to '\0' */
					start = start + 7;
					start[strlen(start) - 4] = '\0';
					sprintf(result, "SSID,%s", start);
					break;
				}

				pclose(fp);
			} else if (strcasecmp((value_ptr)[i], "PSK") == 0) {
			} else if (strcasecmp((value_ptr)[i], "PMK") == 0) {
				if (!strlen(STA_MAC)) {
					printf("STA_MAC_Address can't be NULL, return!!!\n");
					return WFA_ERROR;
				}
				/* GET PMK with hostapd commands
				* ex, "hostapd_cli -i rai0 -p /var/run/hostapd raw GET_PMK 32:d8:57:5d:36:c2"
				*/

				memset(gCmdStr, 0, sizeof(gCmdStr));
				snprintf(gCmdStr, sizeof(gCmdStr), "hostapd_cli -i %s -p %s raw GET_PMK %s",
					def_intf->name, def_intf->ctrl_inf, STA_MAC);
				fp = popen(gCmdStr, "r");
				fgets(PMK, sizeof(PMK), fp);
				pclose(fp);

				snprintf(result, sizeof(result), "PMK,%s", PMK);
				break;
			}
		} else if (strcasecmp((data->params)[i], "STA_MAC_Address") == 0) {
			strcpy(STA_MAC, (value_ptr)[i]);
		}
	}
	sprintf((char *)resp_buf, "%s", result);
	printf("%s\n", resp_buf);
	return WFA_SUCCESS;
}
