// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (c) 2015 Google, Inc
 * Written by Simon Glass <sjg@chromium.org>
 */

#include <common.h>
#include <dm.h>
#include <errno.h>
#include <led.h>
#include <log.h>
#include <malloc.h>
#include <asm/gpio.h>
#include <dm/lists.h>
#include <time.h>

struct led_gpio_priv {
	struct gpio_desc gpio;
#ifdef CONFIG_LED_BLINK
	int delay_on;
	int delay_off;
	int passed_ms;
#endif

};

static int gpio_led_set_state(struct udevice *dev, enum led_state_t state)
{
	struct led_gpio_priv *priv = dev_get_priv(dev);
	int ret;

	if (!dm_gpio_is_valid(&priv->gpio))
		return -EREMOTEIO;
	switch (state) {
	case LEDST_OFF:
	case LEDST_ON:
		break;
	case LEDST_TOGGLE:
		ret = dm_gpio_get_value(&priv->gpio);
		if (ret < 0)
			return ret;
		state = !ret;
		break;
#ifdef CONFIG_LED_BLINK
	case LEDST_BLINK:
		return 0;
#endif
	default:
		return -ENOSYS;
	}

	return dm_gpio_set_value(&priv->gpio, state);
}

static enum led_state_t gpio_led_get_state(struct udevice *dev)
{
	struct led_gpio_priv *priv = dev_get_priv(dev);
	int ret;

	if (!dm_gpio_is_valid(&priv->gpio))
		return -EREMOTEIO;
	ret = dm_gpio_get_value(&priv->gpio);
	if (ret < 0)
		return ret;

	return ret ? LEDST_ON : LEDST_OFF;
}

#ifdef CONFIG_LED_BLINK
void gpio_led_tick(ulong tick_ms)
{
	//int ret;
	struct udevice *dev;
	struct led_gpio_priv *priv;
	static ulong s_latest_tick_ms;

	uclass_find_first_device(UCLASS_LED, &dev);
	if (!dev)
	{
		printk(KERN_DEBUG "gpio_led_tick first dev do not found\n");
		return;
	}

	for (uclass_first_device(UCLASS_LED, &dev);
			dev;
			uclass_next_device(&dev)) {
		priv = dev_get_priv(dev);
		if (priv->delay_on > 0 && priv->delay_off > 0) {
			if (time_after(tick_ms, s_latest_tick_ms))
				priv->passed_ms += (tick_ms - s_latest_tick_ms);

			if (priv->passed_ms < priv->delay_on) {
				gpio_led_set_state(dev, LEDST_ON);
			} else if (priv->passed_ms < (priv->delay_on + priv->delay_off)) {
				gpio_led_set_state(dev, LEDST_OFF);
			} else {
				gpio_led_set_state(dev, LEDST_ON);
				priv->passed_ms -= (priv->delay_on + priv->delay_off);
			}
		}
		/*printk(KERN_DEBUG "Led: %s, State: %d, delay_on: %d delay_off: %d passed:%d\n",
				dev->name,
				dm_gpio_get_value(&priv->gpio),
				priv->delay_on, priv->delay_off, priv->passed_ms);*/
	}

	s_latest_tick_ms = tick_ms;
}

static int gpio_led_set_period(struct udevice *dev, int delayon, int delayoff)
{
	struct led_gpio_priv *priv = dev_get_priv(dev);

	if (delayon < 0 || delayoff < 0)
		return -EREMOTEIO;

	priv->delay_on = delayon;
	priv->delay_off = delayoff;
	priv->passed_ms = 0;

	return 0;
}
#endif

static int led_gpio_probe(struct udevice *dev)
{
	struct led_gpio_priv *priv = dev_get_priv(dev);

	return gpio_request_by_name(dev, "gpios", 0, &priv->gpio, GPIOD_IS_OUT);
}

static int led_gpio_remove(struct udevice *dev)
{
	/*
	 * The GPIO driver may have already been removed. We will need to
	 * address this more generally.
	 */
#ifndef CONFIG_SANDBOX
	struct led_gpio_priv *priv = dev_get_priv(dev);

	if (dm_gpio_is_valid(&priv->gpio))
		dm_gpio_free(dev, &priv->gpio);
#endif

	return 0;
}

static int led_gpio_bind(struct udevice *parent)
{
	struct udevice *dev;
	ofnode node;
	int ret;

	dev_for_each_subnode(node, parent) {
		ret = device_bind_driver_to_node(parent, "gpio_led",
						 ofnode_get_name(node),
						 node, &dev);
		if (ret)
			return ret;
	}

	return 0;
}

static const struct led_ops gpio_led_ops = {
	.set_state	= gpio_led_set_state,
	.get_state	= gpio_led_get_state,
#ifdef CONFIG_LED_BLINK
	.set_period = gpio_led_set_period,
#endif
};

U_BOOT_DRIVER(led_gpio) = {
	.name	= "gpio_led",
	.id	= UCLASS_LED,
	.ops	= &gpio_led_ops,
	.priv_auto	= sizeof(struct led_gpio_priv),
	.probe	= led_gpio_probe,
	.remove	= led_gpio_remove,
};

static const struct udevice_id led_gpio_ids[] = {
	{ .compatible = "gpio-leds" },
	{ }
};

U_BOOT_DRIVER(led_gpio_wrap) = {
	.name	= "gpio_led_wrap",
	.id	= UCLASS_NOP,
	.of_match = led_gpio_ids,
	.bind	= led_gpio_bind,
};
