#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <stdint.h>
#include <sys/un.h>
#include <sys/socket.h>
#include <ctype.h>
#include <sys/ioctl.h>

#include <sys/select.h>
#include <sys/time.h>
#include <sys/wait.h>
//#include <linux/autoconf.h>
#include "nvram.h"
#include "nvram_env.h"
#include "nvram_sock.h"
#include "sys_debug.h"
#include "linux_socket.h"


#define NVRAM_CLIENT_TEST_PROGRAM 0
#define TEST_LIB		0
#define DEV_NUM 3

#define RALINK_NVRAM_IOCTL_GET_CMD	"nvram_get"
#define RALINK_NVRAM_IOCTL_GETALL_CMD	"ralink_init show"
#define RALINK_NVRAM_IOCTL_SET_CMD	"nvram_set"
#define RALINK_NVRAM_IOCTL_COMMIT_CMD	"nvram_commit"
#define RALINK_NVRAM_IOCTL_CLEAR_CMD	"ralink_init clear"
#define RALINK_NVRAM_IOCTL_READ_MAC_CMD	"flash_read_mac"
#define RALINK_NVRAM_CMD_NULL		"null_cmd"

#define LIBNV_PRINT(x, ...) do { if (libnvram_debug) printf("%s %d: " x, __FILE__, __LINE__, ## __VA_ARGS__); } while(0)
#define LIBNV_ERROR(x, ...) do { fprintf(stderr, "%s %d: ERROR! " x, __FILE__, __LINE__, ## __VA_ARGS__); } while(0)

#define FREE(x) do { if (x != NULL) {free(x); x=NULL;} } while(0)

//x is the value returned if the check failed
#define LIBNV_CHECK_INDEX(x) do { \
	if (index < 0 || index >= FLASH_BLOCK_NUM+EXTEND_BLOCK_NUM) { \
		LIBNV_PRINT("index(%d) is out of range\n", index); \
		return x; \
	} \
} while (0)

#define LIBNV_CHECK_VALID() do { \
	if (!fb[index].valid) { \
		LIBNV_PRINT("fb[%d] invalid, init again\n", index); \
		nvram_init(index); \
	} \
} while (0)
/*****************************************************************/
/*                      FUNCTION PROTOTYPE                       */
/*****************************************************************/
static int nvram_cli_init(void);
static int nvram_cli_main(int index, char* cmd, char* name, char* value, size_t value_len);

/*****************************************************************/
/*              VARIABLE DECLARATION (GLOBAL & LOCAL)            */
/*****************************************************************/
char libnvram_debug = 0;
static char get_value[MAX_VALUE_LEN];

struct nvram_cli_struct {
    int nvram_cli_sock;

	char path[128];

    /* max select socket */
    int max_sock_no;

    /* agent select set */
    fd_set readfds;
};

struct nvram_cli_struct nvram_client = {
	.nvram_cli_sock = -1
};

static struct legacy_nvram_mapping_struct {
        char legacy_name[8];
        char name[8];
} legacy_nvram_mapping[DEV_NUM] = {
        {
                .legacy_name = "2860",
                .name = "dev1"
        },
        {
                .legacy_name = "rtdev",
                .name = "dev2"
        },
        {
                .legacy_name = "wifi3",
                .name = "dev3"
        }
};
/*****************************************************************/
/*                 LOCAL FUNTION IMPLEMENTATION                  */
/*****************************************************************/

static int cmd_to_msg(char *cmd)
{
	if(!strcmp(cmd, "get"))
		return RALINK_NVRAM_IOCTL_GET;
	else if(!strcmp(cmd, "getall"))
		return RALINK_NVRAM_IOCTL_GETALL;
	else if(!strcmp(cmd, "set"))
		return RALINK_NVRAM_IOCTL_SET;
	else if(!strcmp(cmd, "commit"))
		return RALINK_NVRAM_IOCTL_COMMIT;
	else if(!strcmp(cmd, "clear"))
		return RALINK_NVRAM_IOCTL_CLEAR;
	else if(!strcmp(cmd, "mac"))
		return RALINK_NVRAM_IOCTL_READ_MAC;
	else
		return RALINK_NVRAM_IOCTL_GET;
}

static int nvram_cli_init(void)
{
	struct sockaddr_un cli_addr;

	LIBNV_PRINT("nvram_client.nvram_cli_sock=%d\n", nvram_client.nvram_cli_sock);

	if(nvram_client.nvram_cli_sock >= 0)
		return 0;

	nvram_client.nvram_cli_sock = socket_gram_clnt_create();

	memset(&cli_addr, 0, sizeof(struct sockaddr_un));
	cli_addr.sun_family = AF_UNIX;
	snprintf(cli_addr.sun_path, sizeof(cli_addr.sun_path),
			"/tmp/nvram_cli_%d.sock", (int) getpid());
	snprintf(nvram_client.path, sizeof(nvram_client.path), "%s", cli_addr.sun_path);
	if (access(nvram_client.path, F_OK) == 0){
		LIBNV_ERROR("socket file %s is exists. It might be caused by unexpected exit of previous process with the same pid. Unlink it anyway.\n", nvram_client.path);
		unlink(nvram_client.path);
	}

	if (bind(nvram_client.nvram_cli_sock, (struct sockaddr *) &cli_addr, sizeof(struct sockaddr_un)) == -1){
		LIBNV_ERROR("bind socket[%d] NG\n", nvram_client.nvram_cli_sock);
		return -1;
	}
		
	nvram_client.max_sock_no = nvram_client.nvram_cli_sock;
	//SYS_Logd("nvram_client.max_sock_no=%d ---------->\n", nvram_client.nvram_cli_sock);

	return 0;
}

static void nvram_cli_deinit(void)
{
	struct stat st;

	if(!stat(nvram_client.path, &st))
		unlink(nvram_client.path);

	nvram_client.nvram_cli_sock = -1;
}

static int nvram_cli_main(int index, char* cmd, char* name, char* value, size_t value_len)
{
	struct nvram_cli_struct *nvram_cli = &nvram_client;
	static struct nvram_srv_msg nvram_msg;
	struct nvram_srv_msg *nmsg;
	size_t send_len = 12; //sizeof(nmsg->msg_id + index + data_len)

	//SYS_Logd("<--------\n");
	nmsg = &nvram_msg;
	memset(nmsg, 0, sizeof(struct nvram_srv_msg));

	nmsg->msg_id = cmd_to_msg(cmd);
	nmsg->index = index;

	/* sanity check */
	switch (nmsg->msg_id){
	case RALINK_NVRAM_IOCTL_GET:
		if(!name)
			return -1;
		snprintf(nmsg->name, sizeof(nmsg->name), "%s", name);
		nmsg->data_len = strlen(name) + 1;
		send_len += nmsg->data_len;
		break;
	case RALINK_NVRAM_IOCTL_SET:
		if(!name || !value)
			return -1;
		snprintf(nmsg->name, sizeof(nmsg->name), "%s", name);
		snprintf(nmsg->value, sizeof(nmsg->value), "%s", value);
		send_len += sizeof(nmsg->name) + sizeof(nmsg->value);
		break;
	case RALINK_NVRAM_IOCTL_GETALL:
	case RALINK_NVRAM_IOCTL_COMMIT:
	case RALINK_NVRAM_IOCTL_CLEAR:
	default:
		break;
	}

	//SYS_Logd("msg_id=%d, index=%d, name=%s value=%s,\n", nmsg->msg_id, nmsg->index, nmsg->name, nmsg->value);

	/* Send command to nvram_server */
	//socket_gram_clnt_sendto(nvram_cli->nvram_cli_sock, NVRAM_SERVER_NAME, (char *)nmsg, sizeof(struct nvram_srv_msg));
	socket_gram_clnt_sendto(nvram_cli->nvram_cli_sock, NVRAM_SERVER_NAME, (char *)nmsg, send_len);

	/* Handle the return msg from nvram_server */
	ssize_t recv_len;
	uint32_t buflen;
	char *buf, *p, *q;
	int i;
	switch (nmsg->msg_id){
	case RALINK_NVRAM_IOCTL_GET:
	case RALINK_NVRAM_IOCTL_READ_MAC:
		//socket_gram_clnt_recv(nvram_cli->nvram_cli_sock, nmsg->value, sizeof(nmsg->value));
		recv_len = socket_gram_clnt_recv(nvram_cli->nvram_cli_sock, (char *)nmsg, sizeof(struct nvram_srv_msg));

		//SYS_Logd("nvram_get recv_len=%zd, msg_id=%d, index=%d, name=%s value=%s,\n",
		//		recv_len, nmsg->msg_id, nmsg->index, nmsg->name, nmsg->value);
#if 0 /* TODO: fine tune the field storing the value. */
		if(value != NULL)
			snprintf(value, value_len, "%s", nmsg->value);
#else
		if(value != NULL)
			snprintf(value, value_len, "%s", nmsg->name);
		
#endif
		break;
	case RALINK_NVRAM_IOCTL_GETALL:

		recv_len = socket_gram_clnt_recv(nvram_cli->nvram_cli_sock, (char *)nmsg, sizeof(struct nvram_srv_msg));
		buflen = nmsg->data_len;

		//SYS_Logd("nvram_getall recv_len=%zd, msg_id=%d, index=%d, next_data_len=%u\n", recv_len, nmsg->msg_id, nmsg->index, buflen);

		buf = malloc(sizeof(char) * buflen);
		if(!buf){
			SYS_Loge("malloc error\n");
		}

		recv_len = socket_gram_clnt_recv(nvram_cli->nvram_cli_sock, buf, buflen);
		//SYS_Logd("nvram_getall recv_data_len=%zd buflen=%u\n", recv_len, buflen);

		p = buf;
		for (i = 0; i < MAX_CACHE_ENTRY; i++) {
			if (NULL == (q = strchr(p, '='))) {
				SYS_Loge("parsed failed - cannot find '='\n");
				break;
			}
			*q = '\0'; //strip '='
			fprintf(stdout, "%s=", p);

			p = q + 1; //value
			if (NULL == (q = strchr(p, '\0'))) {
				SYS_Loge("parsed failed - cannot find '\\0'\n");
				break;
			}
			fprintf(stdout, "%s\n", p);

			p = q + 1; //next entry
			if (p - buf + 1 >= buflen) //end of block
				break;
			if (*p == '\0') //end of env
				break;
		}
		free(buf);
		break;
	}

	return 0;	
}

/*
 * return idx (0 ~ iMAX_CACHE_ENTRY)
 * return -1 if no such value or empty cache
 */
static int cache_idx(int index, char *name)
{
	int i;

	for (i = 0; i < MAX_CACHE_ENTRY; i++) {
		if (!fb[index].cache[i].name)
			return -1;
		if (!strcmp(name, fb[index].cache[i].name))
			return i;
	}
	return -1;
}

void nvram_init(int index)
{
	LIBNV_PRINT("--> nvram_init %d\n", index);
	/* init socket client */
	if(nvram_cli_init() < 0)
		return ;
}

void nvram_close(int index)
{
	size_t i;

	LIBNV_PRINT("--> nvram_close %d\n", index);
	nvram_commit(index);

	//free cache
	for (i = 0; i < MAX_CACHE_ENTRY; i++) {
		FREE(fb[index].cache[i].name);
		FREE(fb[index].cache[i].value);
	}
	nvram_cli_deinit();
}

const char *nvram_get(int index, char *name)
{
	LIBNV_PRINT("--> nvram_get %d\n", index);
	/* Get the fresh value from nvram_server 
	 * so there is no need to do nvram_close() and nvram_init() again
	 */
	//nvram_close(index);
	//nvram_init(index);
	return nvram_bufget(index, name);
}

int nvram_set(int index, char *name, char *value)
{
	LIBNV_PRINT("--> nvram_set\n");

	if (-1 == nvram_bufset(index, name, value))
		return -1;

	return nvram_commit(index);
}

const char *nvram_bufget(int index, char *name)
{
	int idx;
	static char const *ret;
	size_t max_cache_idx = fb[index].max_cache_idx;

	LIBNV_PRINT("--> nvram_bufget %d\n", index);
	LIBNV_CHECK_INDEX("");

	if(!name)
		return "";

	if(-1 == nvram_cli_init()){
		LIBNV_PRINT("");
	}
	nvram_cli_main(index, "get", name, get_value, MAX_VALUE_LEN);

	idx = cache_idx(index, name);
	if (idx < 0) {
		if(max_cache_idx == MAX_CACHE_ENTRY) {
			LIBNV_ERROR("run out of env cache, please increase MAX_CACHE_ENTRY:%d\n", MAX_CACHE_ENTRY);
			return "";
		}
		fb[index].cache[max_cache_idx].name= strdup(name);
		fb[index].cache[max_cache_idx].value = strdup(get_value);
		ret = fb[index].cache[max_cache_idx].value;
		LIBNV_PRINT("bufget %d '%s'->'%s'\n", index, name, ret);
		max_cache_idx++;
		fb[index].max_cache_idx = max_cache_idx;
		return ret;
	} else {
		if (fb[index].cache[idx].value)
			FREE(fb[index].cache[idx].value);

		fb[index].cache[idx].value = strdup(get_value);
		ret = fb[index].cache[idx].value;
		LIBNV_PRINT("bufget %d '%s'->'%s'\n", index, name, ret);
		return ret;
	}
	
	return "";
//	return &get_value[0];
}

int nvram_bufset(int index, char *name, const char *value)
{
	LIBNV_PRINT("--> nvram_bufset\n");
	LIBNV_CHECK_INDEX(-1);

	nvram_cli_init();
	nvram_cli_main(index, "set", name, value, strlen(value));

	return 0;
}

void nvram_buflist(int index)
{
	LIBNV_PRINT("--> nvram_buflist\n");
	LIBNV_CHECK_INDEX();

	nvram_cli_init();
	nvram_cli_main(index, "getall", NULL, NULL, 0);
}

/*
 * write flash from cache
 */
int nvram_commit(int index)
{
	LIBNV_PRINT("--> nvram_commit\n");
	LIBNV_CHECK_INDEX(-1);

	nvram_cli_init();
	nvram_cli_main(index, "commit", NULL, NULL, 0);

	return 0;
}

/*
 * clear flash by writing all 1's value
 */
int nvram_clear(int index)
{
	int i;

	LIBNV_PRINT("--> nvram_clear %d\n", index);
	LIBNV_CHECK_INDEX(-1);

	nvram_cli_init();
	nvram_cli_main(index, "clear", NULL, NULL, 0);
	nvram_commit(index);

	//free cache
	for (i = 0; i < MAX_CACHE_ENTRY; i++) {
		FREE(fb[index].cache[i].name);
		FREE(fb[index].cache[i].value);
	}

	return 0;
}

int flash_read_mac(char *buf)
{
	int index = getNvramIndex("dev1");

	nvram_cli_init();
	nvram_cli_main(index, "mac", "", buf, 6);

	/* Return expected length */
	return 6;
}

int getNvramNum(void)
{
	return FLASH_BLOCK_NUM+EXTEND_BLOCK_NUM;
}

size_t getNvramOffset(int index)
{
	LIBNV_CHECK_INDEX(0);
	return fb[index].flash_offset;
}

char *getNvramName(int index)
{
	LIBNV_CHECK_INDEX(NULL);
	return fb[index].name;
}

size_t getNvramBlockSize(int index)
{
	LIBNV_CHECK_INDEX(0);
	return fb[index].flash_max_len;
}

static char *get_nvram_name(char *name)
{
	int i;

	for(i=0; i < DEV_NUM; i++){
		if(!strncmp(name, legacy_nvram_mapping[i].legacy_name,
					strlen(legacy_nvram_mapping[i].legacy_name)))
			return legacy_nvram_mapping[i].name;
	}

	return name;
}

unsigned int getNvramIndex(char *name)
{
	int i;
	int *real_name = get_nvram_name(name);

	for (i = 0; i < FLASH_BLOCK_NUM+EXTEND_BLOCK_NUM; i++) {
		if (!strcmp(fb[i].name, real_name)) {
			return i;
		}
	}
	return -1;
}

void toggleNvramDebug()
{
	if (libnvram_debug) {
		libnvram_debug = 0;
		printf("%s: turn off debugging\n", __FILE__);
	}
	else {
		libnvram_debug = 1;
		printf("%s: turn ON debugging\n", __FILE__);
	}
}

#if defined(NVRAM_CLIENT_TEST_PROGRAM)
static void usage(void)
{
	printf("Usage: nvram_client 2860 get Channel\n");
//	printf("       nvram_client 2860 getall\n");
	printf("       nvram_client 2860 set Channel 6\n");
	printf("       nvram_client 2860 commit\n");
	printf("       nvram_client 2860 clear\n");
}

int main(int argc, char *argv[])
{
	int index;

	if(nvram_cli_init() < 0)
		return -1;

	//SYS_Logd("argc=%d, \n", argc);

	if (argc < 2 ){
		usage();
		return 1;
	}

	index = getNvramIndex(argv[1]);
#if TEST_LIB
	const char *str;
	if (index < 0){
		index = getNvramIndex("dev1");
		switch (argc){
			case 2: /* default nvram zone is 2860 */
				nvram_cli_main(index, argv[1], NULL, NULL, 0);
				break;
			case 3: /* ommit, clear  */
				nvram_cli_main(index, argv[1], NULL, NULL, 0);
				break;
			case 4: /* get, getall */
				nvram_cli_main(index, argv[1], argv[2], NULL, 0);
				break;
			case 5: /* set */
				nvram_cli_main(index, argv[1], argv[2], 
						argv[3], strlen(argv[3]) + 1);
				break;
		}
	} else {
		switch (argc){
			case 3: /* commit, clear, getall */
				if(!strncmp(argv[2], "commit", strlen("commit")))
					nvram_commit(index);
				else if(!strncmp(argv[2], "clear", strlen("clear")))
					nvram_clear(index);
				else if(!strncmp(argv[2], "getall", strlen("getall")))
					nvram_clear(index);
				else
					SYS_Loge("unknown command:[%s]", argv[0]);
				break;
			case 4: /* get */
				if(!strncmp(argv[2], "get", strlen("get"))){
					str = nvram_get(index, argv[3]);
					fprintf(stdout, "%s", str);
				} else {
					SYS_Loge("unknown command:[%s]", argv[0]);
				}
				break;
			case 5: /* set */
				if(!strncmp(argv[2], "set", strlen("set")))
					nvram_set(index, argv[3], argv[4]);
				else
					SYS_Loge("unknown command:[%s]", argv[0]);
				break;
		}
	}
#else
	if (index < 0){
		index = getNvramIndex("dev1");
		switch (argc){
			case 2: /* default nvram zone is dev1 */
				nvram_cli_main(index, argv[1], NULL, NULL, 0);
				break;
			case 3: /* commit, clear  */
				nvram_cli_main(index, argv[1], NULL, NULL, 0);
				break;
			case 4: /* get, getall */
				nvram_cli_main(index, argv[1], argv[2], NULL, 0);
				break;
			case 5: /* set */
				nvram_cli_main(index, argv[1], argv[2], 
						argv[3], strlen(argv[3]) + 1);
				break;
		}
	} else {
		switch (argc){
			case 3: /* commit, clear */
				nvram_cli_main(index, argv[2], NULL, NULL, 0);
				break;
			case 4: /* get, getall */
				nvram_cli_main(index, argv[2], argv[3], NULL, 0);
				break;
			case 5: /* set */
				nvram_cli_main(index, argv[2], argv[3],
						argv[4], strlen(argv[4]) + 1);
				break;
		}
	}
#endif
	return 0;
}
#endif
