#!/usr/bin/env lua
--[[
 * A lua library to translate from /etc/config/wireless into
 * MediaTek's WiFi profile (mt76xx.dat) and vice-versa.
 *
 * Copyright (C) 2018 Hua Shao <nossiac@163.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
]]

package.path = '../?.lua;'..package.path

local uci = require("shuci")
local l1profile = "/etc/wireless/l1profile.dat"
local ucifile = "/etc/config/wireless"

-- this flag is used in uci2dat and dat2uci
-- if set to true, the tool will NOT actually modify the target file,
-- but dump the result to stdout
local testmode = false

-- this flag is used in datdiff
-- if false, will dump diff result in plain text
-- if true, will generate "iwpriv" commands according to diff result
local iwpriv_mode = false

function __trim(s)
  if s then return (s:gsub("^%s*(.-)%s*$", "%1")) end
end


-- lua sorted pairs
function __spairs(t, order)
    local keys = {}
    for k in pairs(t) do keys[#keys+1] = k end
    table.sort(keys, order)

    -- return a closure as an iterator
    local i = 0
    return function()
        i = i + 1
        if keys[i] then
            return keys[i], t[keys[i]]
        end
    end
end

function string:split(sep)
    local sep, fields = sep or ":", {}
    local pattern = string.format("([^%s]+)", sep)
    self:gsub(pattern, function(c) fields[#fields+1] = c end)
    return fields
end

-- load MediaTek WiFi profile as a lua table
function load_profile(path)
    local cfgs = {}
    local fd = io.open(path, "r")
    if not fd then return cfgs end

    for line in fd:lines() do
        line = __trim(line)
        if string.byte(line) ~= string.byte("#") then
            local i = string.find(line, "=")
            if i then
                local k,v
                k = string.sub(line, 1, i-1)
                v = string.sub(line, i+1)
                -- if cfgs[__trim(k)] then
                --     print("warning", "skip repeated key"..line)
                -- end
                cfgs[__trim(k)] = __trim(v) or ""
            -- else
            --     print("warning", "skip line without '=' "..line)
            end
        -- else
        --     print("warning", "skip comment line "..line)
        end
    end
    fd:close()
    return cfgs
end


-- save configuration table as MediaTek wifi profile
function save_profile(cfgs, path)
    local fd = io.stdout
    if path then fd = io.open(path, "w") end

    fd:write("# Generated by uci2dat\n")
    fd:write("Default\n")
    for k,v in __spairs(cfgs) do
        fd:write(k.."="..v.."\n")
    end
    fd:close()
end


function merge_profile(t1, t2)
    if not t1 or not t2 or not next(t2) then return end
    for k,v in pairs(t2) do
        t1[k] = v
    end
end

function vifs_cfg_param(param)
    local vifs_cfg_params = {"Auth", "EncrypType", "Key", "WPAPSK", ";"}
    for _, pat in ipairs(vifs_cfg_params) do
        if string.find(param, pat) then
            return true
        end
    end
    return false
end

-- cfgs,   config data parsed from wifi old profile
-- diff,   difference between old profile and new profile
-- device, device data parsed from l1profile
function __set_wifi_misc(cfgs, diff, device)
    local vifname = device.main_ifname
    local vifext = device.ext_ifname
    local last_command = string.format([[
                iwpriv %s set SSID="%s";]], vifname, cfgs.SSID1)

    local vifidx = cfgs.vifidx:split(";")
    local commands_vifs_ssid = false
    local commands_ssid = false --per ssid

    local token = function(str, n, default)
        local i = 1
        local list = {}
        for k in string.gmatch(str, "([^;]+)") do
            list[i] = k
            i = i + 1
        end
        return list[tonumber(n)] or default
    end

    for k,v in pairs(diff) do
        local commands
        local commands_1
        local commands_vifs

        if k:find("^SSID") then
            local _,_,i = string.find(k, "^SSID([%d]+)")
            if i == "1" then
                commands = string.format([[
                iwpriv %s set SSID="%s";]], vifname, tostring(v[2]))
            else
                commands = string.format([[
                iwpriv %s set SSID="%s";]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
            end
            last_command = commands
        elseif k == "WirelessMode" or k == "wirelessmode" then
            commands = string.format([[
                iwpriv %s set WirelessMode=%s;]], vifname, tostring(v[2]))
        elseif k == "Channel" or k == "channel" then
            commands = string.format([[
                iwpriv %s set Channel=%s;]], vifname, tostring(v[2]))
        elseif k == "CountryCode" or k == "countrycode" then
            commands = string.format([[
                iwpriv %s set CountryCode=%s;]], vifname, tostring(v[2]))
        elseif k == "ShortSlot" or  k == "shortslot" then
            commands = string.format([[
                iwpriv %s set ShortSlot=%s;]], vifname, tostring(v[2]))
        elseif k == "PktAggregate" or  k == "pktaggregate" then
            commands = string.format([[
                iwpriv %s set PktAggregate=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_BADecline" or k == "ht_badecline" then
            commands = string.format([[
                iwpriv %s set BADecline=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_DisallowTKIP" or k == "ht_disallowtkip" then
            commands = string.format([[
                iwpriv %s set HtDisallowTKIP=%s;]], vifname, tostring(v[2]))
        elseif k == "BeaconPeriod" or  k == "beaconperiod" then
            commands = string.format([[
                iwpriv %s set BeaconPeriod=%s;]], vifname, tostring(v[2]))
        elseif k == "DtimPeriod" or  k == "dtimperiod" then
            commands = string.format([[
                iwpriv %s set DtimPeriod=%s;]], vifname, tostring(v[2]))
        elseif k == "FragThreshold" or  k == "fragthreshold" then
            commands = string.format([[
                iwpriv %s set FragThreshold=%s;]], vifname, tostring(v[2]))
        elseif k == "RTSThreshold" or  k == "rtsthreshold" then
            commands = string.format([[
                iwpriv %s set RTSThreshold=%s;]], vifname, tostring(v[2]))
        elseif k == "TxPreamble" or  k == "txpreamble" then
            commands = string.format([[
                iwpriv %s set TxPreamble=%s;]], vifname, tostring(v[2]))
        elseif k == "TxBurst" or  k == "txburst" then
            commands = string.format([[
                iwpriv %s set TxBurst=%s;]], vifname, tostring(v[2]))
        elseif k == "NoForwarding" or  k == "noforwarding" then
            commands = string.format([[
                iwpriv %s set NoForwarding=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_MCS" or k == "ht_mcs" then
            commands = string.format([[
                iwpriv %s set HtMcs=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_EXTCHA" or k == "ht_extcha" then
            commands = string.format([[
                iwpriv %s set HtExtCha=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_AutoBA" or k == "ht_autoba" then
            commands = string.format([[
                iwpriv %s set HtAutoBa=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_MpduDensity" or k == "ht_mpdudensity" then
            commands = string.format([[
                iwpriv %s set HtMpduDensity=%s;]], vifname, tostring(v[2]))
        elseif k == "HT_RDG" or k == "ht_rdg" then
            commands = string.format([[
                iwpriv %s set HtRdg=%s;]], vifname, tostring(v[2]))
        --Need parse vifs
        elseif k == "CountryRegion" or k == "countryregion" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set CountryRegion=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "CountryRegionABand" or k == "countryregionaband" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set CountryRegionABand=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
        elseif k == "BGProtection" or k == "bgprotection" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set BGProtection=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "TxPower" or  k == "txpower" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set TxPower=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "HT_OpMode" or k == "ht_opmode" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set HtOpMode=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "HT_AMSDU" or k == "ht_amsdu" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set HtAmsdu=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "VHT_STBC" or k == "vht_stbc" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set VhtStbc=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "HT_STBC" or k == "ht_stbc" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set HtStbc=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "HT_GI" or  k == "ht_gi" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set HtGi=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "HT_BW" or k == "ht_bw" then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set HtBw=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                commands_1 = string.format([[
                iwpriv %s set VhtBw=%s;]], vifext..tostring(tonumber(i)-1), diff["VHT_BW"] and tostring(diff["VHT_BW"][2])
                    or tostring(cfgs["VHT_BW"]))
                if commands_vifs then print(commands_vifs) end
                if commands_1 then print(commands_1) end
            end
            commands_vifs_ssid = true
        elseif k == "VHT_BW" or k == "vht_bw" then
           for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set VhtBw=%s;]], vifext..tostring(tonumber(i)-1), tostring(v[2]))
                commands_1 = string.format([[
                iwpriv %s set HtBw=%s;]], vifext..tostring(tonumber(i)-1), diff["HT_BW"] and tostring(diff["HT_BW"][2])
                    or tostring(cfgs["HT_BW"]))
                if commands_1 then print(commands_1) end
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        elseif k == "HideSSID" or k == "hidessid" then
            for i=1, #vifidx  do
              if token(cfgs.HideSSID, i) ~= token(diff.HideSSID[2], i) then
                commands_vifs = string.format([[
                iwpriv %s set HideSSID=%s;]], vifext..tostring(tonumber(i)-1), token(diff.HideSSID[2], i))
                commands_ssid = string.format([[
                iwpriv %s set SSID=%s;]], vifext..tostring(tonumber(i)-1), cfgs["SSID"..tostring(i)])
                print(commands_vifs)
                print(commands_ssid)
             end
            end
        elseif not vifs_cfg_param(k) and not vifs_cfg_param(v[2]) then
            for i=1, #vifidx  do
                commands_vifs = string.format([[
                iwpriv %s set %s=%s;]], vifext..tostring(tonumber(i)-1), k, tostring(v[2]))
                if commands_vifs then print(commands_vifs) end
            end
            commands_vifs_ssid = true
        end
        if commands then print(commands) end
    end

    if commands_vifs_ssid then
        for i=1, #vifidx  do
            commands_vifs_ssid = string.format([[
                iwpriv %s set SSID="%s";]], vifext..tostring(tonumber(i)-1), cfgs["SSID"..tostring(i)])
            print(commands_vifs_ssid)
        end
    elseif not commands_ssid then -- HideSSID will set ssid
       if commands and last_command then print(last_command) end
    end
end

function __set_wifi_security(cfgs, diff, device)
    -- to keep it simple, we always reconf the security if anything related is changed.
    -- do optimization only if there's significant performance defect.

    local vifs = {} -- changed vifs

    -- figure out which vif is changed
    -- since multi-bssid is possible, both AuthMode and EncrypType can be a group
    local auth_old = cfgs.AuthMode:split()
    local encr_old = cfgs.EncrypType:split()
    local auth_new = {}
    local auth_new1 = {}
    local encr_new = {}
    local encr_new1 = {}
    if diff.EncrypType then
        encr_new = diff.EncrypType[2]:split()
        encr_new1 = encr_new[1]:split(";")
    end
    if diff.AuthMode then
        auth_new = diff.AuthMode[2]:split()
        auth_new1 = auth_new[1]:split(";")
    end

    -- For WPA/WPA2
    local RadiusS_old = cfgs.RADIUS_Server:split() or {}
    local RadiusP_old = cfgs.RADIUS_Port:split() or {}
    local RadiusS_old_i = (RadiusS_old[1] or ''):split(";")
    local RadiusP_old_i = (RadiusP_old[1] or ''):split(";")
    local RadiusS_new = diff.RADIUS_Server and diff.RADIUS_Server[2]:split() or RadiusS_old
    local RadiusP_new = diff.RADIUS_Port and diff.RADIUS_Port[2]:split() or RadiusP_old
    local RadiusS_new_i = (RadiusS_new[1] or ''):split(";") --split by ";"
    local RadiusP_new_i = (RadiusP_new[1] or ''):split(";")

    local auth_old1 = auth_old[1]:split(";") --auth_old1[1]=OPEN,auth_old1[2]=WPA2PSK
    local encr_old1 = encr_old[1]:split(";")
    for i = 1, #encr_old1 do
        local changed = false
        if next(auth_new) and auth_old1[i] ~= auth_new1[i] then
            changed = true
        elseif next(encr_new) and encr_old1[i] ~= encr_new1[i] then
            changed = true
        elseif diff["WPAPSK"..tostring(i)] then
            changed = true
        elseif next(RadiusS_new) and RadiusS_old_i[i] ~= RadiusS_new_i[i] then
            changed = true
        elseif next(RadiusP_new) and RadiusP_old_i[i] ~= RadiusP_new_i[i] then
            changed = true
        elseif diff["RADIUS_Key"..tostring(i)] then
            changed = true
        else
            for j = 1, 4 do
                if diff["Key"..tostring(j).."Str"..tostring(i)] then
                    changed = true
                    break
                end
            end
        end
        if changed then
            local vif = {}
            vif.idx = i
            vif.vifname = device.ext_ifname..tostring(i-1)
            vif.AuthMode = auth_new1 and auth_new1[i] or auth_old1[i]
            vif.EncrypType = encr_new1 and encr_new1[i] or encr_old1[i]
            vif.Key1 = diff["Key1Str"..tostring(i)] and diff["Key1Str"..tostring(i)][2] or cfgs["Key1Str"..tostring(i)]
            vif.WPAPSK = diff["WPAPSK"..tostring(i)] and diff["WPAPSK"..tostring(i)][2] or cfgs["WPAPSK"..tostring(i)]
            vif.SSID = diff["SSID"..tostring(i)] and diff["SSID"..tostring(i)][2] or cfgs["SSID"..tostring(i)]
            vif.RADIUS_Server = RadiusS_new_i and RadiusS_new_i[i] or RadiusS_old_i[i]
            vif.RADIUS_Port = RadiusP_new_i and RadiusP_new_i[i] or RadiusP_old_i[i]
            vif.RADIUS_Key = diff["RADIUS_Key"..tostring(i)] and diff["RADIUS_Key"..tostring(i)][2] or cfgs["RADIUS_Key"..tostring(i)]
            table.insert(vifs, vif)
        end
    end

    -- iwpriv here
    for i, vif in ipairs(vifs) do
        if vif.AuthMode == "OPEN" then
            if vif.EncrypType == "WEP" then
                commands = string.format([[
                iwpriv %s set AuthMode=OPEN;
                iwpriv %s set EncrypType=WEP;
                iwpriv %s set Key1="%s";
                iwpriv %s set DefaultKeyID=1;
                iwpriv %s set IEEE8021X=0;]],
            vif.vifname, vif.vifname, vif.vifname, vif.Key1,
            vif.vifname, vif.vifname)
            else
                commands = string.format([[
                iwpriv %s set AuthMode=OPEN;
                iwpriv %s set EncrypType=NONE;
                iwpriv %s set IEEE8021X=0;]],
            vif.vifname, vif.vifname, vif.vifname)
            end
        elseif vif.AuthMode == "SHARED" then
            commands = string.format([[
                iwpriv %s set AuthMode=SHARED;
                iwpriv %s set EncrypType=WEP;
                iwpriv %s set Key1="%s";
                iwpriv %s set DefaultKeyID=1;
                iwpriv %s set IEEE8021X=0;]],
            vif.vifname, vif.vifname, vif.vifname, vif.Key1,
            vif.vifname, vif.vifname)
        elseif vif.AuthMode == "WPA2PSK" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPA2PSK;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set SSID="%s";
                iwpriv %s set WPAPSK="%s";]],
            vif.vifname, vif.vifname, vif.EncrypType, vif.vifname, vif.SSID,
            vif.vifname, vif.WPAPSK)
        elseif vif.AuthMode == "WPAPSKWPA2PSK" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPAPSKWPA2PSK;
                iwpriv %s set EncrypType=TKIPAES;
                iwpriv %s set SSID="%s";
                iwpriv %s set WpaMixPairCipher=WPA_TKIP_WPA2_AES;
                iwpriv %s set WPAPSK="%s";]],
            vif.vifname, vif.vifname, vif.vifname, vif.SSID,
            vif.vifname, vif.vifname, vif.WPAPSK)
        elseif vif.AuthMode == "WPAPSK" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPAPSK;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set WPAPSK="%s";]],
            vif.vifname, vif.vifname, vif.EncrypType,
            vif.vifname, vif.WPAPSK)
        elseif vif.AuthMode == "WPA" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPA;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set RADIUS_Server=%s;
                iwpriv %s set RADIUS_Port=%s;
                iwpriv %s set RADIUS_Key=%s;
                iwpriv %s set IEEE8021X=0;]],
            vif.vifname, vif.vifname, vif.EncrypType,vif.vifname, vif.RADIUS_Server,
            vif.vifname, vif.RADIUS_Port, vif.vifname, vif.RADIUS_Key, vif.vifname)
        elseif vif.AuthMode == "WPA2" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPA2;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set RADIUS_Server=%s;
                iwpriv %s set RADIUS_Port=%s;
                iwpriv %s set RADIUS_Key=%s;
                iwpriv %s set IEEE8021X=0;]],
            vif.vifname, vif.vifname, vif.EncrypType,vif.vifname, vif.RADIUS_Server,
            vif.vifname,vif.RADIUS_Port, vif.vifname, vif.RADIUS_Key, vif.vifname)
        elseif vif.AuthMode == "WPA1WPA2" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPA1WPA2;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set RADIUS_Server=%s;
                iwpriv %s set RADIUS_Port=%s;
                iwpriv %s set RADIUS_Key=%s;
                iwpriv %s set IEEE8021X=0;]],
            vif.vifname, vif.vifname, vif.EncrypType,vif.vifname, vif.RADIUS_Server,
            vif.vifname,vif.RADIUS_Port, vif.vifname, vif.RADIUS_Key, vif.vifname)
        elseif vif.AuthMode == "WPA3PSK" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPA3PSK;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set WPAPSK="%s";]],
            vif.vifname, vif.vifname, vif.EncrypType, vif.vifname, vif.WPAPSK)
        elseif vif.AuthMode == "WPA2PSKWPA3PSK" then
            commands = string.format([[
                iwpriv %s set AuthMode=WPA2PSKWPA3PSK;
                iwpriv %s set EncrypType=%s;
                iwpriv %s set WPAPSK="%s";]],
            vif.vifname, vif.vifname, vif.EncrypType, vif.vifname, vif.WPAPSK)
        else
            error(string.format("invalid AuthMode \"%s\"", vif.AuthMode))
        end

        -- must append extra SSID command to make changes take effect
        commands = commands .."\n".. string.format([[
                iwpriv %s set SSID="%s";]], vif.vifname, vif.SSID)
        print(commands)
    end
end


function diff_profile(datold, datnew)
    if not datold or not datnew then
        error("Usage: datdiff <dat-old> <dat-new>")
        return
    end

    local diff = {}
    local cfg1 = load_profile(datold) or {}
    local cfg2 = load_profile(datnew) or {}

    for k,v in pairs(cfg1) do
        if cfg2[k] ~= cfg1[k] then
            diff[k] = {cfg1[k] or "", cfg2[k] or ""}
        end
    end

    for k,v in pairs(cfg2) do
        if cfg2[k] ~= cfg1[k] then
            diff[k] = {cfg1[k] or "", cfg2[k] or ""}
        end
    end

    if not iwpriv_mode then
        for k,v in pairs(diff) do
            print(datold..":"..k.."="..v[1])
            print(datnew..":"..k.."="..v[2])
        end
    end

    return diff
end


-- mapping dat-diff into iwpriv
function diff_profile_iwpriv(datold, datnew)
    if not datold or not datnew then
        error("Usage: datdiff -x <dat-old> <dat-new>")
        return
    end
    local l1dat = load_l1profile(l1profile)
    local cfgs = load_profile(datold)
    local diff = diff_profile(datold, datnew)
    local device = nil

    for i, dev in ipairs(l1dat) do
        if dev.profile_path == datold then
            device = dev
            break
        elseif dev.profile_path == datnew then
            device = dev
            break
        end
    end

    if not device then
        error("cannot find device of "..datnew)
    end
    __set_wifi_misc(cfgs, diff, device)

    -- security is complicated enough to get a special API
    __set_wifi_security(cfgs, diff, device)
end


function get_vif_by_ifname(ucicfg, ifname)
    for vifname, vif in pairs(ucicfg["wifi-iface"]) do
        if vifname == ifname then
            return vif
        end
    end
end

function get_vifs_by_dev(ucicfg, devname)
    local vifs = {}
    for vifname, vif in pairs(ucicfg["wifi-iface"]) do
        if vif.device == devname then
            if tonumber(vif.vifidx) then
                vifs[tonumber(vif.vifidx)] = vif
            end
        end
    end

    for vifname, vif in pairs(ucicfg["wifi-iface"]) do
        if vif.device == devname then
            if tonumber(vif.vifidx) == nil  then
                vifs[#vifs+1] = vif
            end
        end
    end

    return vifs
end


function vif_cfg_ssid(vifs)
    assert(vifs)
    assert(type(vifs) == "table")

    local t = {}
    t["BssidNum"] = #vifs
    for i,vif in ipairs(vifs) do
        if vif.radio == "1" then
            t["SSID"..tostring(i)] = vif.ssid
            vif.ssid = nil
        end
    end

    return t
end

function vif_cfg_guestNetWork(vifs)
    assert(vifs)
    assert(type(vifs) == "table")

    local t = {}
    for i,vif in ipairs(vifs) do
        if vif.allowLocalNetwork == "0" then
            t["NoForwardingBTNBSSID"] = 1
        else
            t["NoForwardingBTNBSSID"] = 0
        end
    end

    return t
end

function dat2uci_vif_cfg_encryption(auth,encr)
    if auth == "OPEN" and encr == "NONE" then
        encr = "none"
    elseif auth == "OPEN" and encr == "WEP" then
        encr = "wep-open"
    elseif auth == "SHARED" and encr == "WEP" then
        encr = "wep-shared"
    elseif auth == "WPAPSK" and encr == "AES" then
        encr = "psk"
    elseif auth == "WPAPSK" and encr == "TKIP" then
        encr = "psk+tkip"
    elseif auth == "WPAPSK" and encr == "TKIPAES" then
        encr = "psk+tkip+ccmp"
    elseif auth == "WPA2PSK" and encr == "AES" then
        encr = "psk2"
    elseif auth == "WPA2PSK" and encr == "TKIP" then
        encr = "psk2+tkip"
    elseif auth == "WPA2PSK" and encr == "TKIPAES" then
        encr = "psk2+tkip+ccmp"
    elseif auth == "WPAPSKWPA2PSK" and encr == "AES" then
        encr = "psk+psk2"
    elseif auth == "WPA" and encr == "AES" then
        encr = "wpa"
    elseif auth == "WPA2" and encr == "AES" then
        encr = "wpa2"
    elseif auth == "WPA1WPA2" and encr == "AES" then
        encr = "wpa+wpa2"
    elseif auth == "WPA3PSK" and encr == "AES" then
        encr = "psk3"
    elseif auth == "WPA2PSKWPA3PSK" and encr == "AES" then
        encr = "psk2+psk3"
    else -- default, open
        encr = "none"
    end
    return encr
end

function dev_cfg_encryption(v)
    local t = {}

    local auth = ""
    local encr = ""
    local mfpc = ""
    local mfpr = ""
    local sha = ""

    if v == "None" then
        auth = auth.."OPEN"
        encr = encr.."NONE"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    elseif v == "WEP" then
        auth = auth.."SHARED"
        encr = encr.."WEP"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    elseif (v == "WPA2-PSK-AES" or v == "WPA2-Personal") then
        auth = auth.."WPA2PSK"
        encr = encr.."AES"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    elseif (v == "WPA-PSK-TKIP|WPA2-PSK-AES" or v == "WPA/WPA2-Personal") then
        auth = auth.."WPAPSKWPA2PSK"
        encr = encr.."TKIPAES"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    elseif v == "WPA3-Personal" then
        auth = auth.."WPA3PSK"
        encr = encr.."AES"
        mfpc = mfpc.."1"
        mfpr = mfpr.."1"
        sha = sha.."1"
    elseif v == "WPA2/WPA3-Personal" then
        auth = auth.."WPA2PSKWPA3PSK"
        encr = encr.."AES"
        mfpc = mfpc.."1"
        mfpr = mfpr.."1"
        sha = sha.."1"
    elseif v == "WPA-PSK-TKIP" then
        auth = auth.."WPAPSK"
        encr = encr.."TKIP"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    elseif v == "WPA-PSK-AES" then
        auth = auth.."WPAPSK"
        encr = encr.."AES"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    elseif v == "WPA/WPA2-Enterprise" then
        auth = auth.."WPA1WPA2"
        encr = encr.."TKIPAES"
        mfpc = mfpc.."1"
        mfpr = mfpr.."0"
        sha = sha.."0"
    else
        auth = auth.."OPEN"
        encr = encr.."NONE"
        mfpc = mfpc.."0"
        mfpr = mfpr.."0"
        sha = sha.."0"
    end
    
    t["ApCliAuthMode"] = auth
    t["ApCliEncrypType"] = encr
    t["ApCliPMFMFPC"] = mfpc
    t["ApCliPMFMFPR"] = mfpr
    t["ApCliPMFSHA256"] = sha

    return t
end

function vif_cfg_encryption(vifs)
    assert(vifs)
    assert(type(vifs) == "table")

    local t = {}
    local auth = ""
    local encr = ""
    local auth_server = ""
    local auth_port = ""
    local auth_secret = ""
    local mfpc = ""
    local mfpr = ""
    local sha = ""

    function __aestkip(m)
        if m == "ccmp" then
            return "AES"
        elseif m == "tkip" then
            return "TKIP"
        elseif m == "ccmp+tkip" or m == "tkip+ccmp" then
            return "TKIPAES"
        else
            return "AES"
        end
    end

    for i, vif in ipairs(vifs) do
        if vif.security == "None" then
            auth = auth.."OPEN"
            encr = encr.."NONE"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif vif.security == "WEP" then
            if vif.wepAuthType == "1" then
                auth = auth.."SHARED"
                encr = encr.."WEP"
            else 
                auth = auth.."WEPAUTO"
                encr = encr.."WEP"
            end
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif (vif.security == "WPA2-PSK-AES" or vif.security == "WPA2-Personal") then
            auth = auth.."WPA2PSK"
            encr = encr.."AES"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif vif.security == "WPA2-PSK-TKIP" then
            auth = auth.."WPA2PSK"
            encr = encr.."TKIP"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif (vif.security == "WPA-PSK-TKIP|WPA2-PSK-AES" or vif.security == "WPA/WPA2-Personal") then
            auth = auth.."WPAPSKWPA2PSK"
            encr = encr.."TKIPAES"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif vif.security == "WPA3-Personal" then
            auth = auth.."WPA3PSK"
            encr = encr.."AES"
            mfpc = mfpc.."1"
            mfpr = mfpr.."1"
            sha = sha.."1"
        elseif vif.security == "WPA2/WPA3-Personal" then
            auth = auth.."WPA2PSKWPA3PSK"
            encr = encr.."AES"
            mfpc = mfpc.."1"
            mfpr = mfpr.."1"
            sha = sha.."1"
        elseif vif.security == "WPA-PSK-TKIP" then
            auth = auth.."WPAPSK"
            encr = encr.."TKIP"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif vif.security == "WPA-PSK-AES" then
            auth = auth.."WPAPSK"
            encr = encr.."AES"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        elseif vif.security == "WPA/WPA2-Enterprise" then
            auth = auth.."WPA1WPA2"
            encr = encr.."TKIPAES"
            mfpc = mfpc.."1"
            mfpr = mfpr.."0"
            sha = sha.."0"
        else
            auth = auth.."OPEN"
            encr = encr.."NONE"
            mfpc = mfpc.."0"
            mfpr = mfpr.."0"
            sha = sha.."0"
        end

        if vif.auth_server == nil then
            vif.auth_server='0'
        end
        if vif.auth_port == nil then
            vif.auth_port='0'
        end
        auth_server = auth_server..vif.auth_server
        auth_port = auth_port..vif.auth_port

        -- don't forget the delimiter ";"
        if i<#vifs then
            auth = auth..";"
            encr = encr..";"
            mfpc = mfpc..";"
            mfpr = mfpr..";"
            sha = sha..";"
            -- for wpa*
            auth_server = auth_server..";"
            auth_port = auth_port..";"
        end

        t["PMFMFPC"] = mfpc
        t["PMFMFPR"] = mfpr
        t["PMFSHA256"] = sha
        t["AuthMode"] = auth
        t["EncrypType"] = encr

        t["RADIUS_Server"] = auth_server
        t["RADIUS_Port"] = auth_port
        t["RADIUS_Key"..tostring(i)] = vif.auth_secret or auth_secret

        -- WEP keys
        t["Key"..tostring(i).."Type"] = "1" -- 0=hex, 1=string
        if vif.key1 ~= nil and (#vif.key1 == 7 or #vif.key1 == 15) then
            vif.key1 = string.sub(vif.key1,3)
        end

        t["Key1Str"..tostring(i)] = vif.key1
        t["Key2Str"..tostring(i)] = vif.key2
        t["Key3Str"..tostring(i)] = vif.key3
        t["Key4Str"..tostring(i)] = vif.key4
        t["DefaultKeyID"] = vif.wepKeyID

        -- PSK keys
        t["WPAPSK"..tostring(i)] = vif.key

        -- unset keys already done
        vif.key = nil
        vif.key1 = nil
        vif.key2 = nil
        vif.key3 = nil
        vif.key4 = nil
        vif.auth_secret = nil
        vif.security = nil
    end

    return t
end


--[[
    Traslate ucikey into datkey by vifs (per SSID)

    vifs:      wifi interfaces(per SSID)
    ucikey:    the keyword defined in uci config
    datkey:    coresponding keyword in dat, optional.
    default:   default value if key is not specified
]]

function vif_cfg_by_key(vifs, ucikey, datkey, default)
    local t = {}

    local key = datkey
    local strbuf = ""
    if not key then key = ucikey end
    if not default then default = 0 end

    for i,vif in ipairs(vifs) do
        if type(vif[ucikey]) == "string" or type(vif[ucikey]) == "number" then
            strbuf = strbuf..tostring(vif[ucikey])
        else
            strbuf = strbuf..tostring(default)
        end
        -- don't forget the delimiter ";"
        if i<#vifs then strbuf = strbuf..";" end

        -- unset keys already done
        vif[ucikey] = nil
    end

    t[key] = strbuf

    return t
end


function vif_cfg_ignore(vifs, ucikey)
    if type(ucikey) == "string" then
        for i,vif in ipairs(vifs) do
            vif[ucikey] = nil
        end
    elseif type(ucikey) == "table" then
        for _,key in ipairs(ucikey) do
            for i,vif in ipairs(vifs) do
                vif[key] = nil
            end
        end
    end
end


function vif_cfg_all(vifs)
    if #vifs < 1 then return end
    local t = {}

    local keys = {}
    for _,vif in pairs(vifs) do
        for k,_ in pairs(vif) do
            keys[k] = true
        end
    end

    for k,_ in pairs(keys) do
        local strbuf = ""
        for i=1, #vifs do
            v = vifs[i][k]
            if type(v) == "string" or type(v) == "number" then
                strbuf = strbuf..tostring(v)
            else
                strbuf = strbuf..tostring(0)
            end
            -- don't forget the delimiter ";"
            if i<#vifs then strbuf = strbuf..";" end

            -- unset keys already done
            vifs[i][k] = nil
        end
        t[k] = strbuf
    end

    return t
end

function load_l1profile(path, dump)
    local l1dat = {}
    local fd = io.open(path, "rb")
    if not fd then error("failed to open "..path) return end
    for line in fd:lines() do
        local m,n,i,devname = string.find(line, "INDEX(%d)=([^%s\'\"]+)")
        if i and devname then
            local devidx = 1
            for i,dev in ipairs(l1dat) do
                if dev.chipname == devname then
                    devidx = devidx + 1
                end
            end
            l1dat[tonumber(i)+1] = {}
            l1dat[tonumber(i)+1]["chipname"] = devname
            l1dat[tonumber(i)+1]["devname"] = devname.."."..tostring(devidx)
        end
        local m,n,i,k,v = string.find(line, "INDEX(%d)_([%w-_]+)=([^%s\'\"]*)")
        if i and k then
            -- print(i,k,v)
            l1dat[tonumber(i)+1][k] = v
        end
    end

    -- split dbdc dev into 2 devs
    local idx = 1
    local dbdc = {}
    for i,dev in ipairs(l1dat) do
            if dev.profile_path:find(";") then
                    -- make a dup first
                    dbdc[idx] = {}
                    for k,v in pairs(dev) do
                            if v:find(";") then
                                    local v1v2 = v:split(";")
                                    assert(2 == #v1v2)
                                    l1dat[i][k] = v1v2[1]
                                    dbdc[idx][k] = v1v2[2]
                            else
                                    dbdc[idx][k] = v
                            end
                    end
                    dbdc[idx]["devname"] = l1dat[i].devname..".2"
                    l1dat[i]["devname"] = l1dat[i].devname..".1"
                    idx = idx + 1
            end
    end

    local idx = #l1dat + 1
    for _,dev in ipairs(dbdc) do
            l1dat[idx] = {}
            for k,v in pairs(dev) do
                    l1dat[idx][k] = v
            end
            idx = idx + 1
    end


    if dump then
        for i,dev in ipairs(l1dat) do
            for k,v in pairs(dev) do
                print(string.format("l1dat[%d].%s=%s", i, k, v))
            end
        end
    end

    return l1dat
end

function getCountryRegionABandWithoutDFS(CountryCode)
    local CountryRegionABand = ""

    CountryRegionABand = getCountryRegionABand(CountryCode)

    if CountryCode == "US" then
        CountryRegionABand = 29
    elseif CountryCode == "NA" then
        CountryRegionABand = 29
    elseif CountryCode == "CA" then
        CountryRegionABand = 14
    end

    return CountryRegionABand
end

function getCountryRegionABand(CountryCode)
    local CountryRegionABand = ""

    if CountryCode == "ZA" then
        CountryRegionABand = 6
    elseif CountryCode == "AS" then
        CountryRegionABand = 10
    elseif CountryCode == "AU" then
        CountryRegionABand = 9
    elseif CountryCode == "CA" then
        CountryRegionABand = 25
    elseif CountryCode == "CN" then
        CountryRegionABand = 0
    elseif CountryCode == "WW" then
        CountryRegionABand = 1
    elseif CountryCode == "EU" then
        CountryRegionABand = 1
    elseif CountryCode == "IN" then
        CountryRegionABand = 10
    elseif CountryCode == "JP" then
        CountryRegionABand = 1
    elseif CountryCode == "KR" then
        CountryRegionABand = 17
    elseif CountryCode == "MY" then
        CountryRegionABand = 10
    elseif CountryCode == "MX" then
        CountryRegionABand = 10
    elseif CountryCode == "TR" then
        CountryRegionABand = 6
    elseif CountryCode == "SA" then
        CountryRegionABand = 10
    elseif CountryCode == "AE" then
        CountryRegionABand = 6
    elseif CountryCode == "RU" then
        CountryRegionABand = 28
    elseif CountryCode == "SG" then
        CountryRegionABand = 10
    elseif CountryCode == "BR" then
        CountryRegionABand = 10
    elseif CountryCode == "TW" then
        CountryRegionABand = 27
    elseif CountryCode == "US" then
        CountryRegionABand = 26
    elseif CountryCode == "NA" then
        CountryRegionABand = 26
    elseif CountryCode == "HK" then
        CountryRegionABand = 7
    end

    return CountryRegionABand
end

function getCountryRegion(CountryCode)
    local CountryRegion = ""

    if CountryCode == "US" or CountryCode == "MX" or CountryCode == "CA" or CountryCode == "NA" then
        CountryRegion = 0
    else
        CountryRegion = 1
    end

    return CountryRegion
end

function getCountryRDRegion(CountryCode)
    local CountryRDRegion = ""

    if CountryCode == "CA" then
        CountryRDRegion = ""
    elseif CountryCode == "US" or CountryCode == "NA" then
        CountryRDRegion = "FCC"
    elseif CountryCode == "EU" then
        CountryRDRegion = "CE"
    end

    return CountryRDRegion
end

function getDfsEnable(CountryCode)
    local dfsEnable = "1"

    if CountryCode == "CA" then
        dfsEnable = "1"
    end

    return dfsEnable
end

function getIEEE80211H(CountryCode)
    local ieee80211h = "1"

    if CountryCode == "CA" then
        ieee80211h = "1"
    end

    return ieee80211h
end

function convertCountryCode(CountryCode)
    local ccValue = CountryCode
    if CountryCode == "WW" then
        ccValue = "EU"
    end
    return ccValue;
end

function getSkuTableIdx(CountryCode)
    local SkuTableIdx = "0"
    if CountryCode == "US" or CountryCode == "NA" then
        SkuTableIdx = "0"
    elseif CountryCode == "CA" then
        SkuTableIdx = "1"
    elseif CountryCode == "EU" or CountryCode == "WW" then
        SkuTableIdx = "2"
    elseif CountryCode == "AU" or CountryCode == "HK" then
        SkuTableIdx = "3"
    end
    return SkuTableIdx 
end

function uci2dat(ucifile, datfile, devname)
    if not ucifile then error("ucifile not specified!") return end
    if not datfile then error("datfile not specified!") return end
    if not devname then error("devname not specified!") return end
    if devname then devname = devname:gsub("%.", "_") end
    local ucicfg = uci.decode(ucifile)
    if not ucicfg then error("unable to decode "..ucifile) return end

    --uci.dump(ucicfg)

    local datcfg = load_profile(datfile)
	
    for _,dev in pairs(ucicfg["wifi-device"]) do while true do
        if devname ~= dev.device and dev.device ~= "MT7915D_1_1;MT7915D_1_2" and dev.device ~= "MT7986_1_1;MT7986_1_2" and dev.device ~= "MT7981_1_1;MT7981_1_2" then break end

        local devapclicfg = {}
        local devscfg = {}
        local t2 = {}
        local dfsEnable = ""

        for k,v in pairs(dev) do
            --------------- device apcli level ---------------
            if k == "rootap_ssid" then
                devapclicfg.ApCliSsid1 = v
            elseif k == "apcli_security" then
                t2 = dev_cfg_encryption(v)
                merge_profile(devapclicfg, t2)
            elseif k == "apcli_key" then
                devapclicfg.ApCliWPAPSK = v
            elseif k == "apcli_existOption" then
                devapclicfg.ApCliEnable = v
            elseif k == "apcli_bssid" then
                devapclicfg.ApCliBssid = v
            elseif k == "apcli_defaultKeyID" then
                devapclicfg.ApCliDefaultKeyID = v
            end
            
            --------------- device level ---------------
            -- The following keywords are used by openwrt luci ui:
            -- ref: https://wiki.openwrt.org/doc/uci/wireless
            -- We follow its convention.
            if k == "channel" then
                devscfg.Channel = v
                if v == "0" then
                    devscfg.AutoChannelSelect = 3
                else
                    devscfg.AutoChannelSelect = 0
                end
            elseif k == "wifimode" then
                devscfg.WirelessMode = v
            elseif k == "country" then
                devscfg.CountryCode = convertCountryCode(v)
                devscfg.CountryRegionABand = getCountryRegionABand(v)
                devscfg.CountryRegion = getCountryRegion(v)
                devscfg.RDRegion = getCountryRDRegion(v)
                devscfg.SkuTableIdx = getSkuTableIdx(v)
                devscfg.IEEE80211H = getIEEE80211H(v)
                devscfg.DfsEnable = getDfsEnable(v)
            elseif k == "dfsEnable" then
                dfsEnable = v
            elseif k == "region" then
                devscfg.CountryRegion = v
            elseif k == "aregion" then
                devscfg.CountryRegionABand = v
            elseif k == "txpower" then
                devscfg.TxPower = v
            elseif k == "bgprotect" then
                devscfg.BGProtection = v
            elseif k == "mapMode" then
                devscfg.MapMode = v
            elseif k == "macAddr" then
                devscfg.MacAddress = v
            elseif k == "pktaggre" then
                devscfg.PktAggregate = v
            elseif k == "wirelessMode" then
                devscfg.Vht1024QamSupport = 0

                if devname == "MT7915D_1_1" then
                    --devscfg.G_BAND_256QAM = 0
                    devscfg.WirelessMode  = 16
                elseif devname == "MT7986_1_1" then
                    devscfg.WirelessMode  = 16
                elseif devname == "MT7981_1_1" then
                    devscfg.WirelessMode  = 16
                    devscfg.G_BAND_256QAM = 1
                else
                    --devscfg.G_BAND_256QAM = 1
                    devscfg.WirelessMode  = 17
                end
                if v == "54Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.WirelessMode = 0
                    elseif devname == "MT7986_1_1" then
                        devscfg.WirelessMode = 0
                    elseif devname == "MT7981_1_1" then
                        devscfg.WirelessMode = 0
                    else
                        devscfg.WirelessMode = 2
                    end
                elseif v == "150Mbps" then
                    devscfg.HT_BW = 0
                    devscfg.VHT_BW = 0
                elseif v == "173Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7986_1_1" then
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7981_1_1" then
                        devscfg.WirelessMode  = 9
                    else
                        devscfg.WirelessMode  = 14
                    end
                    devscfg.HT_BW = 0
                    devscfg.VHT_BW = 0
                elseif v == "216Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.HT_BW = 1
                    elseif devname == "MT7986_1_1" then
                        devscfg.HT_BW = 1
                    elseif devname == "MT7981_1_1" then
                        devscfg.HT_BW = 1
                    else
                        devscfg.HT_BW = 0
                    end
                    devscfg.VHT_BW = 0
                elseif v == "286Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.HT_BW = 0
                    elseif devname == "MT7986_1_1" then
                        devscfg.HT_BW = 0
                    elseif devname == "MT7981_1_1" then
                        devscfg.HT_BW = 0
                    elseif devname == "MT7986_1_2" then
                        devscfg.HT_BW = 0
                        devscfg.WirelessMode = 17
                        devscfg.Vht1024QamSupport = 1
                    elseif devname == "MT7981_1_2" then
                        devscfg.HT_BW = 0
                        devscfg.WirelessMode = 17
                        devscfg.Vht1024QamSupport = 1
                    else
                        devscfg.HT_BW = 0
                    end
                    devscfg.VHT_BW = 0
                elseif v == "300Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.HT_BW = 1
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7986_1_1" then
                        devscfg.HT_BW = 1
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7981_1_1" then
                        devscfg.HT_BW = 1
                        devscfg.WirelessMode  = 9
                        devscfg.G_BAND_256QAM = 0
                    else
                        devscfg.HT_BW = 0
                    end
                    devscfg.VHT_BW = 0
                elseif v == "400Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7986_1_1" then
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7981_1_1" then
                        devscfg.WirelessMode  = 9
                    else
                        devscfg.WirelessMode  = 14
                    end
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 0
                elseif v == "450Mbps" then
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 0
                elseif v == "600Mbps" then
                    --devscfg.G_BAND_256QAM = 1
                    if devname == "MT7986_1_2" then
                        devscfg.Vht1024QamSupport = 1
                    elseif devname == "MT7981_1_2" then
                        devscfg.Vht1024QamSupport = 1
                    end
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 0
                elseif v == "867Mbps" then
                    if devname == "MT7915D_1_1" then
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7986_1_1" then
                        devscfg.WirelessMode  = 9
                    elseif devname == "MT7981_1_1" then
                        devscfg.WirelessMode  = 9
                    else
                        devscfg.WirelessMode  = 14
                    end
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 1
                elseif v == "1200Mbps" then
                    devscfg.Vht1024QamSupport = 1
                    devscfg.WirelessMode = 17
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 1
                elseif v == "1300Mbps" then
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 1
                elseif v == "1733Mbps" then
                    devscfg.WirelessMode  = 14
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 2
                elseif v == "1800Mbps" then
                    devscfg.WirelessMode  = 17
                    devscfg.Vht1024QamSupport = 1
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 2
                elseif v == "2400Mbps" then
                    devscfg.Vht1024QamSupport = 1
                    devscfg.WirelessMode  = 17
                    devscfg.HT_BW = 1
                    devscfg.VHT_BW = 2
                end
            elseif k == "igmpsnenable" then
                devscfg.IgmpSnEnable = v
            elseif k == "deviceName" then
                devscfg.WscDeviceName = v
            elseif k == "wifiCoext" then
                devscfg.HT_BSSCoexistence = v
            --elseif k == "wpsPinCode" then
                --devscfg.WscVendorPinCode = v
            elseif k == "skuEnable" then
                devscfg.SKUenable = v
            elseif k == "mumimo" then
                devscfg.MUTxRxEnable = v
                devscfg.ETxBfEnCond = v
                devscfg.MuMimoDlEnable = v
                devscfg.MuMimoUlEnable = v
            elseif k == "wifi6" then
                devscfg.MuOfdmaDlEnable = v
                devscfg.MuOfdmaUlEnable = v
            elseif k == "maxStaNum" then
                devscfg.MbssMaxStaNum = v
            end
        end
        if dfsEnable == "0" then
            devscfg.DfsEnable = dfsEnable
            devscfg.IEEE80211H = dfsEnable
            devscfg.CountryRegionABand = getCountryRegionABandWithoutDFS(devscfg.CountryCode)
        end

        -- for k,v in __spairs(devscfg) do print(k.."="..v) end
        merge_profile(datcfg, devapclicfg)
        merge_profile(datcfg, devscfg)

        local vifs = get_vifs_by_dev(ucicfg, devname)
        if not vifs or #vifs < 1 then break end
        local vifscfg = {}
        local t = {}

        -- special apis to deal with complicated keys
        t = vif_cfg_encryption(vifs)
        merge_profile(vifscfg, t)

        t = vif_cfg_ssid(vifs)
        merge_profile(vifscfg, t)

        -- t = vif_cfg_guestNetWork(vifs)
        -- merge_profile(vifscfg, t)

        -- common api to fetch a key
        t = vif_cfg_by_key(vifs, "hidden", "HideSSID")
        merge_profile(vifscfg, t)

        -- ignore a single key
        t = vif_cfg_ignore(vifs, "device")
        merge_profile(vifscfg, t)

        -- ignore a group of keys
        t = vif_cfg_ignore(vifs, {"mode", "network", "conf_mode", 
                                  "same_sec", "wpsPinAttackCheck", "wpsPinAttackNum",
                                  "wepAuthType," , "wepKeyLength", "wpsLockdown"})
        merge_profile(vifscfg, t)

        t = vif_cfg_ignore(vifs, {"auth_port", "auth_secret", "auth_server"})
        merge_profile(vifscfg, t)

        -- fetch all rest keys
        t = vif_cfg_all(vifs)
        merge_profile(vifscfg, t)

        -- dump all we've got
        --for k,v in __spairs(vifscfg) do print(k.."="..v) end

        merge_profile(datcfg, vifscfg)
    break end end

    os.execute("mkdir -p /tmp/mtk/wifi/")
    local _,_,datbak = datfile:find("([%w%.]+%.dat)")
    if datbak then
        datbak = datbak..".bak"
        os.execute("cp "..datfile.." /tmp/mtk/wifi/"..datbak)
    end

    save_profile(datcfg, (not testmode) and datfile)
end


function dat2uci(datfile, ucifile, devname, ifpre)
    local datcfg = load_profile(datfile)
    if not datcfg then error("unable to open "..datfile) return end
    if devname then devname = devname:gsub("%.", "_") end

    ucicfg = {}
    ucicfg["wifi-device"] = {}
    ucicfg["wifi-iface"] = {}


    dev = {}
    -- luci keys
    dev.vendor = "mediatek"
    dev.type = "mediatek"
    dev.device = devname
    dev.mode = "ap"
    dev.network = "lan"
    dev.txpower = datcfg.TxPower
    dev.wifimode = datcfg.WirelessMode
    dev.channel = datcfg.Channel

    if datcfg.HT_BW == "1" then
        if datcfg.VHT_BW == "0" then
            dev.band = "2G"
            dev.bw = "40"
        elseif datcfg.VHT_BW == "1" then
            dev.band = "5G"
            dev.bw = "80"
        elseif datcfg.VHT_BW == "2" then
            dev.band = "5G"
            dev.bw = "160"
        elseif datcfg.VHT_BW == "3" then
            dev.band = "5G"
            dev.bw = "80+80"
        end
    else
        dev.band = "2G"
        dev.bw = "20"
    end


    -- mediatek keys
    dev.country = datcfg.CountryCode
    dev.WmmCapable = datcfg.WmmCapable
    dev.bgprotect = datcfg.BGProtection
    dev.shortslot = datcfg.ShortSlot
    dev.pktaggre = datcfg.BGProtection
    dev.E2pAccessMode = datcfg.E2pAccessMode
    dev.HT_OpMode = datcfg.HT_OpMode
    dev.HT_GI = datcfg.HT_GI
    dev.HT_STBC = datcfg.HT_STBC
    dev.HT_AMSDU = datcfg.HT_AMSDU
    dev.HT_AutoBA = datcfg.HT_AutoBA
    dev.HT_BADecline = datcfg.HT_BADecline
    dev.HT_DisallowTKIP = datcfg.HT_DisallowTKIP
    dev.HT_LDPC = datcfg.HT_LDPC
    ucicfg["wifi-device"][devname] = dev
    if dev.band == "5G" then
        dev.aregion = datcfg.CountryRegionABand
        dev.VHT_STBC = datcfg.VHT_STBC
        dev.VHT_SGI = datcfg.VHT_SGI
        dev.VHT_BW_SIGNAL = datcfg.VHT_BW_SIGNAL
        dev.VHT_LDPC = datcfg.VHT_LDPC
    else
        dev.region = datcfg.CountryRegion
        dev.HT_BSSCoexistence = datcfg.HT_BSSCoexistence
    end


    local token = function(str, n, default)
        local i = 1
        local list = {}
        for k in string.gmatch(str, "([^;]+)") do
            list[i] = k
            i = i + 1
        end
        return list[tonumber(n)] or default
    end

    local encr1 = datcfg.EncrypType:split()
    local auth1 = datcfg.AuthMode:split()
    local encr2 = encr1[1]:split(";")
    local auth2 = auth1[1]:split(";")

    for i=1, tonumber(datcfg.BssidNum) do
        local encr = dat2uci_vif_cfg_encryption(auth2[i],encr2[i])
        if encr == "wep-shared" or encr == "wep-open" then
            ucicfg["wifi-iface"][ifpre..tostring(i-1)] = {
            ssid = datcfg["SSID"..tostring(i)],
            encryption = encr,
            device = devname,
            vifidx = i,
            ifname = ifpre..tostring(i-1),
            key1 = datcfg["Key1Str"..tostring(i)],
            hidden = token(datcfg.HideSSID, i),
            mode = "ap",
            wmm = WmmCapable,
            }
        elseif encr == "wpa" or encr == "wpa2" or encr == "wpa+wpa2" then
            ucicfg["wifi-iface"][ifpre..tostring(i-1)] = {
            ssid = datcfg["SSID"..tostring(i)],
            encryption = encr,
            device = devname,
            vifidx = i,
            ifname = ifpre..tostring(i-1),
            auth_server = token(datcfg.RADIUS_Server, i),
            auth_port = token(datcfg.RADIUS_Port, i),
            auth_secret = datcfg["RADIUS_Key"..tostring(i)],
            hidden = token(datcfg.HideSSID, i),
            mode = "ap",
            wmm = WmmCapable,
            }
       else
            ucicfg["wifi-iface"][ifpre..tostring(i-1)] = {
            ssid = datcfg["SSID"..tostring(i)],
            encryption = encr,
            device = devname,
            vifidx = i,
            ifname = ifpre..tostring(i-1),
            key = datcfg["WPAPSK"..tostring(i)],
            hidden = token(datcfg.HideSSID, i),
            mode = "ap",
            wmm = WmmCapable,
            }
        end
    end

    uci.encode(ucicfg, (not testmode) and ucifile)
end


--[[
    A basic imitation of posix "getopt".
]]
function getopt(argv, fmt)
    local tab = {}
    local i = 1

    while i <= #argv do
        local arg = argv[i]
        local m = arg:match("-(%w+)")
        if m then
            if fmt:find(m..":") then
                tab[m] = argv[i+1]
                i = i + 1
            elseif fmt:find(m) then
                tab[m] = true
            else
                error("invalid option "..m)
            end
        else
            table.insert(tab, arg)
        end
        i = i + 1
    end

    return tab
end



----------------------------  TIME TO GO ----------------------------
----------------------------  TIME TO GO ----------------------------
----------------------------  TIME TO GO ----------------------------
----------------------------  TIME TO GO ----------------------------
----------------------------  TIME TO GO ----------------------------

local opts = getopt(arg, "l:u:d:hf:tx")
local datfile = opts["f"]
local devname = opts["d"]
if opts["l"] then l1profile = opts["l"] end
if opts["u"] then ucifile = opts["u"] end
if opts["t"] then testmode = true end
if opts["x"] then iwpriv_mode = true end

-- help message
if opts["h"] then
    print([[

This tool translates between openwrt uci configuration and MediaTek WiFi profile.
It can be used in 4 ways as below:

    uci2dat [options] [datfile]
    dat2uci [options] [datfile]
    datdiff <old-dat> <new-dat>
    l1prof  [-l l1profile]

Options:
    -h   display help message.
    -t   test mode, will not modify target files but dump result to stdout.
    -x   iwpriv mode, "datdiff" will generate "iwpriv" commands according to diff result.
    -l   specify l1profile path if available. ("/etc/wireless/l1profile")
    -u   specify uci configuration path. ("/etc/config/wireless")
    -f   specify wifi profile path. ("/etc/wireless/mt76*/mt76*.dat")
    -d   specify device name. It should match the "device" value in both uci configuration
         and l1profile.

Examples: UCI ==> DAT

    1. if l1profile is present.
    uci2dat -u /etc/config/wireless -l /etc/wireless/l1profile.dat

    2. without l1prfile, you need to specify the device name and its profile path.
    uci2dat -u /etc/config/wireless -d mt7628 -f /etc/wireless/mt7628/mt7628.dat

Examples: DAT ==> UCI

    1. if l1profile is present.
    dat2uci -l /etc/wireless/l1profile.dat -u /etc/config/wireless

    2. without l1prfile, you need to specify device and its profile path.
    dat2uci -d mt7628 -u /etc/config/wireless -f /etc/wireless/mt7628/mt7628.dat

    3. without "-u", the result will be print to stdout.
    dat2uci -l /etc/wireless/l1profile.dat
    dat2uci -d mt7613 -f /etc/wireless/mt7613/mt7613.dat

Examples: Dump DAT difference
    1. for human readable result
    datdiff /etc/wireless/mt7615.1.2G.dat.bak /etc/wireless/mt7615.1.2G.dat

    2. translate into executable "iwpriv" commands
    datdiff -x /etc/wireless/mt7615.1.2G.dat.bak /etc/wireless/mt7615.1.2G.dat

Examples: Dump l1profile content
    1. dump default l1profile "/etc/wireless/l1profile.dat"
    l1prof

    2. specify l1profile path
    l1prof -l /tmp/l1profile.dat

]])
    return
end



if arg[0]:match("uci2dat") then
    if datfile then
        uci2dat(ucifile, datfile, devname)
    else
        -- try l1profile
        local l1dat = load_l1profile(l1profile)
        for i, dev in ipairs(l1dat) do
            uci2dat(ucifile, dev.profile_path, dev.devname)
        end
    end
elseif arg[0]:match("dat2uci") then
    if datfile then
        dat2uci(datfile, ucifile, devname, "ra")
    else
        -- try l1profile
        local l1dat = load_l1profile(l1profile)
        os.execute("mkdir -p /tmp/mtk/wifi")
        os.execute("cp "..ucifile.." /tmp/mtk/wifi/wireless.uci.bak 2>/dev/null")
        os.execute("echo > "..ucifile)
        for i, dev in ipairs(l1dat) do
            dat2uci(dev.profile_path, ucifile, dev.devname, dev.ext_ifname)
        end
    end
elseif arg[0]:match("datdiff") then
    if iwpriv_mode then
        diff_profile_iwpriv(opts[1], opts[2])
    else
        diff_profile(opts[1], opts[2])
    end
elseif arg[0]:match("l1prof") then
    load_l1profile(l1profile, true)
else
    print("Illegal alias!", arg[0])
end
