/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * Copyright  (C) 2020-2021  MediaTek Inc. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <linux/if_ether.h>
#include <linux/etherdevice.h>
#include <net/netlink.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/smp.h>
#include <linux/spinlock.h>
#include <linux/kobject.h>
#include <linux/version.h>
#include <linux/jhash.h>
#include <linux/ip.h>
#include <linux/ipv6.h>
#include <linux/tcp.h>
#include <linux/udp.h>

#include "mtqos.h"

MODULE_LICENSE("Dual BSD/GPL");

#define MTQOS_VERSION "1.0"

#define QOS_NETLINK_EXT 27
#define MAX_MSGSIZE 1024
#define TIMER_INTERVAL 5

/*CB offset should be sync with wifi driver*/
#define CB_OFF  10
#define UP_OFFSET_1  CB_OFF
#define UP_OFFSET_2  (CB_OFF+33)

#define DSCP_TBL_SIZE 64

/*Macro definition*/
#define HASH_TABLE_SIZE 256
#define MAC_ADDR_HASH(addr) (addr[0]^addr[1]^addr[2]^addr[3]^addr[4]^addr[5])
#define MAC_ADDR_HASH_INDEX(addr) (MAC_ADDR_HASH(addr) & (HASH_TABLE_SIZE - 1))
#define MAX(a, b) ((a > b) ? (a) : (b))
#define MIN(a, b) ((a < b) ? (a) : (b))

UCHAR ZERO_MAC_ADDR[MAC_ADDR_LEN]  = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

#define MAC_ADDR_EQUAL(Addr1, Addr2)	(!memcmp((void*)(Addr1), (void*)(Addr2), MAC_ADDR_LEN))
#define COPY_MAC_ADDR(Addr1, Addr2)		memcpy((void*)(Addr1), (void*)(Addr2), MAC_ADDR_LEN)
#define IPV6_ADDR_EQUAL(Addr1, Addr2)	(!memcmp((void*)(Addr1), (void*)(Addr2), IPV6_ADDR_LEN))
#define COPY_IPV6_ADDR(Addr1, Addr2)	memcpy((void*)(Addr1), (void*)(Addr2), IPV6_ADDR_LEN)

#define is_valid_dscp(dscp) (dscp < DSCP_TBL_SIZE)
#define is_valid_up(up) (up < UP_MAX)

#define os_snprintf_error(size, res) ((res < 0) || (res >= (size)))

#define NIP4(addr) \
    ((unsigned char *)&addr)[0], \
    ((unsigned char *)&addr)[1], \
    ((unsigned char *)&addr)[2], \
    ((unsigned char *)&addr)[3]

#define NIP6(addr) \
    ntohs((addr)->s6_addr16[0]), \
    ntohs((addr)->s6_addr16[1]), \
    ntohs((addr)->s6_addr16[2]), \
    ntohs((addr)->s6_addr16[3]), \
    ntohs((addr)->s6_addr16[4]), \
    ntohs((addr)->s6_addr16[5]), \
    ntohs((addr)->s6_addr16[6]), \
    ntohs((addr)->s6_addr16[7])

#define IPV4STR "%d.%d.%d.%d"
#define IPV6STR "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x"

/*Global variable*/
int mscs_config_tbl_cnt = 0;
spinlock_t mscs_config_tbl_lock;
struct list_head mscs_config_tbl[HASH_TABLE_SIZE];

#ifdef QOS_R2
int scs_up_tuple_tbl_cnt = 0;
spinlock_t scs_up_tuple_tbl_lock;
struct list_head scs_up_tuple_tbl[HASH_TABLE_SIZE];
#endif

int mscs_up_tuple_tbl_cnt = 0;
spinlock_t mscs_up_tuple_tbl_lock;
struct list_head mscs_up_tuple_tbl[HASH_TABLE_SIZE];

int vendor_specific_tbl_cnt = 0;
spinlock_t vendor_specific_tbl_lock;
struct list_head vendor_specific_tbl[HASH_TABLE_SIZE];

UCHAR expired_idx[HASH_TABLE_SIZE];

UCHAR QoSMapEnable = 0;
struct qos_map QoS_Map;
struct timer_list QoS_timer;

UCHAR g_QoSMgmtEnabled = 1;
UCHAR g_hk_registered = 0;
int g_pid = 0;
UCHAR send_buf[MAX_MSGSIZE];

UCHAR g_force_up = 0xFF;

UCHAR dft_dscp2up_tbl[DSCP_TBL_SIZE] = {
	0,	0,	0,	0,	0,	0,	0,	0, /* 0 ~ 7  */
	1,	1,	0,	1,	0,	1,	0,	1, /* 8 ~ 15 */
	2,	0,	3,	0,	3,	0,	3,	0, /*16 ~ 23 */
	4,	4,	4,	4,	4,	4,	4,	4, /*24 ~ 31 */
	4,	4,	4,	4,	4,	4,	4,	4, /*32 ~ 39 */
	5,	5,	5,	5,	6,	5,	6,	5, /*40 ~ 47 */
	7,	0,	0,	0,	0,	0,	0,	0, /*48 ~ 55 */
	7,	0,	0,	0,	0,	0,	0,	0  /*56 ~ 63 */
};

#ifdef DSCP_TO_UP_OFFLOAD
UCHAR dft_up2dscp[UP_MAX] = {0, 8, 16, 22, 30, 40, 46, 48};
#endif

/*function declaration*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,14,0)
static void periodic_timer_function(unsigned long data);
#else
static void periodic_timer_function(struct timer_list *data);
#endif

static ssize_t qos_show_info(struct kobject *kobj, struct kobj_attribute *attr, char *buf);
static ssize_t qos_set_dbg_level(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count);
static ssize_t qos_set_enable(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count);
static ssize_t qos_force_up(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count);


void recv_nlmsg(struct sk_buff *skb);
void send_nlmsg(char *buf, int buflen);
int update_mscs_configuration(UCHAR *buf);
int update_mscs_up_tuple(struct mscs_up_tuple *up_tuple);
int update_vendor_specific_up_tuple(UCHAR *buf);
int remove_mscs_up_tuple(UCHAR *dmac);
void free_mscs_config_entry(struct rcu_head *head);
void free_mscs_up_tuple_entry(struct rcu_head *head);
void free_vendor_specific_entry(struct rcu_head *head);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 1, 0))
static unsigned int mtk_qos_pre_routing(void *priv,
			       struct sk_buff *skb, const struct nf_hook_state *state);
static unsigned int mtk_qos_post_routing(void *priv,
			       struct sk_buff *skb, const struct nf_hook_state *state);
#else
static unsigned int mtk_qos_pre_routing(unsigned int hooknum, struct sk_buff *skb,
	const struct net_device *in, const struct net_device *out, int (*okfn)(struct sk_buff *));
static unsigned int mtk_qos_post_routing(unsigned int hooknum, struct sk_buff *skb,
	const struct net_device *in,const struct net_device *out, int (*okfn)(struct sk_buff *));
#endif
static inline INT32 match_mscs_up_tuple_to_stream(struct mscs_up_tuple *up_tuple,
	struct classifier_type4 *pkt_tuple);
static inline INT32 match_vend_specific_up_tuple_to_stream(
	struct wapp_vend_spec_classifier_para_report *vs_up_tuple, struct classifier_type4 *pkt_tuple);

#ifdef QOS_R2
static inline INT32 match_scs_up_tuple_to_stream(struct scs_classifier_parameter *scs_tuple,
	struct classifier_type4 *pkt_type4, struct classifier_type10 *pkt_type10);
#endif

static struct kobj_attribute qos_sysfs_show_setting =
		__ATTR(show_setting, S_IRUGO, qos_show_info, NULL);

static struct kobj_attribute qos_sysfs_set_dbg_level =
		__ATTR(debug_level, S_IWUSR, NULL, qos_set_dbg_level);

static struct kobj_attribute qos_sysfs_enable_mtqos =
		__ATTR(enable_mtqos, S_IWUSR, NULL, qos_set_enable);

static struct kobj_attribute qos_sysfs_force_up =
		__ATTR(force_up, S_IWUSR, NULL, qos_force_up);

static struct attribute *qos_sysfs[] = {
	&qos_sysfs_show_setting.attr,
	&qos_sysfs_set_dbg_level.attr,
	&qos_sysfs_enable_mtqos.attr,
	&qos_sysfs_force_up.attr,
	NULL,
};
static struct attribute_group qos_attr_group = {
	.attrs = qos_sysfs,
};
struct kobject *qos_kobj;
struct sock *nl_sk;

struct netlink_kernel_cfg nl_kernel_cfg = {
	.groups = 0,
	.flags = 0,
	.input = recv_nlmsg,
	.cb_mutex = NULL,
	.bind = NULL,
};

static struct nf_hook_ops mtk_qos_nf_ops[] = {
	{
		.hook		= mtk_qos_pre_routing,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_PRE_ROUTING,
		.priority	= NF_BR_PRI_FIRST,
	},
	{
		.hook		= mtk_qos_post_routing,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_POST_ROUTING,
		.priority	= NF_BR_PRI_LAST - 2,
	}
};

#if (LINUX_VERSION_CODE > KERNEL_VERSION(4,12,0))
int mtqos_nf_reg_hook(struct nf_hook_ops *reg)
{
	struct net *map_net;
	struct net *last;
	int ret;

	rtnl_lock();
	for_each_net(map_net) {
		ret = nf_register_net_hook(map_net, reg);
		if (ret && ret != -ENOENT) {
			pr_info("map_nf_reg_hook failed\n");
			goto rollback;
		}
	}
	rtnl_unlock();
	return 0;
rollback:
	last = map_net;
	for_each_net(map_net) {
		if (map_net == last)
			break;
		nf_unregister_net_hook(map_net, reg);
	}
	rtnl_unlock();
	return ret;
}

void mtqos_nf_unreg_hook(struct nf_hook_ops *reg)
{
	struct net *map_net;

	rtnl_lock();
	for_each_net(map_net)
		nf_unregister_net_hook(map_net, reg);
	rtnl_unlock();
}

void nf_unregister_hooks(struct nf_hook_ops *reg, unsigned int count)
{
	while (count-- > 0)
		mtqos_nf_unreg_hook(&reg[count]);
}

int nf_register_hooks(struct nf_hook_ops *reg, unsigned int count)
{
	int error = 0;
	unsigned int i;

	for (i = 0; i < count; i++) {
		error = mtqos_nf_reg_hook(&reg[i]);
		if (error)
			goto error;
	}
	return error;

error:
	if (i > 0)
		nf_unregister_hooks(reg, i);
	return error;
}
#endif

static int __init mtqos_init(void)
{
	int ret = -1, i = 0;

	DBGPRINT(DBG_LVL_OFF, "-->%s(), mtqos ver:%s", __func__, MTQOS_VERSION);

	mscs_config_tbl_cnt = 0;
	mscs_up_tuple_tbl_cnt = 0;
	vendor_specific_tbl_cnt = 0;
	QoSMapEnable = 0;

	memset(&QoS_Map, 0, sizeof(QoS_Map));

#ifdef QOS_R2
	scs_up_tuple_tbl_cnt = 0;
	spin_lock_init(&scs_up_tuple_tbl_lock);
	for (i = 0; i < HASH_TABLE_SIZE; i++)
		INIT_LIST_HEAD(&scs_up_tuple_tbl[i]);
#endif

	/*init hash table.*/
	spin_lock_init(&mscs_config_tbl_lock);
	for (i = 0; i < HASH_TABLE_SIZE; i++)
		INIT_LIST_HEAD(&mscs_config_tbl[i]);

	spin_lock_init(&mscs_up_tuple_tbl_lock);
	for (i = 0; i < HASH_TABLE_SIZE; i++)
		INIT_LIST_HEAD(&mscs_up_tuple_tbl[i]);

	spin_lock_init(&vendor_specific_tbl_lock);
	for (i = 0; i < HASH_TABLE_SIZE; i++)
		INIT_LIST_HEAD(&vendor_specific_tbl[i]);

	/*register hook function to bridge.*/
	ret = nf_register_hooks(&mtk_qos_nf_ops[0], ARRAY_SIZE(mtk_qos_nf_ops));
	if (ret < 0) {
		DBGPRINT(DBG_LVL_ERROR, "register nf hook fail, ret = %d\n", ret);
		goto error1;
	}
	g_hk_registered = 1;

	/*register netlink interface.*/
	nl_sk = netlink_kernel_create(&init_net, QOS_NETLINK_EXT, &nl_kernel_cfg);
	if (!nl_sk) {
		DBGPRINT(DBG_LVL_ERROR, "create netlink socket error.\n");
		ret = -EFAULT;
		goto error2;
	}

	qos_kobj = kobject_create_and_add("mtqos", NULL);
	if (!qos_kobj) {
		ret = -EFAULT;
		goto error3;
	}

	ret = sysfs_create_group(qos_kobj, &qos_attr_group);
	if (ret)
		goto error4;

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,14,0)
	init_timer(&QoS_timer);
	QoS_timer.function = periodic_timer_function;
	QoS_timer.expires = jiffies + TIMER_INTERVAL*HZ;
	add_timer(&QoS_timer);
#else
	QoS_timer.expires = jiffies + TIMER_INTERVAL*HZ;
	timer_setup(&QoS_timer, periodic_timer_function, 0);
	add_timer(&QoS_timer);
#endif

	DBGPRINT(DBG_LVL_OFF, "<--\n");

	return ret;
error4:
	kobject_put(qos_kobj);
error3:
	sock_release(nl_sk->sk_socket);
error2:
	nf_unregister_hooks(&mtk_qos_nf_ops[0], ARRAY_SIZE(mtk_qos_nf_ops));
	g_hk_registered = 0;
error1:
	return ret;
}

static void __exit mtqos_exit(void)
{
	int i;
	struct mscs_config_entry *pos1 = NULL, *tmp1 = NULL;
	struct mscs_up_tuple_entry *pos2 = NULL, *tmp2 = NULL;
	struct vendor_specific_up_entry *pos3 = NULL, *tmp3 = NULL;
#ifdef QOS_R2
	struct scs_up_tuple_entry *pos4 = NULL, *tmp4 = NULL;
#endif
	DBGPRINT(DBG_LVL_OFF, "-->%s()", __func__);

	sysfs_remove_group(qos_kobj, &qos_attr_group);
	kobject_put(qos_kobj);

	if (nl_sk != NULL)
		sock_release(nl_sk->sk_socket);

	nf_unregister_hooks(&mtk_qos_nf_ops[0], ARRAY_SIZE(mtk_qos_nf_ops));
	g_hk_registered = 0;

	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		list_for_each_entry_safe(pos1, tmp1, &mscs_config_tbl[i], list) {
			list_del_rcu(&pos1->list);
			mscs_config_tbl_cnt--;
			kfree(pos1);
		}
	}

	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		list_for_each_entry_safe(pos2, tmp2, &mscs_up_tuple_tbl[i], list) {
			list_del_rcu(&pos2->list);
			mscs_up_tuple_tbl_cnt--;
			kfree(pos2);
		}
	}

	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		list_for_each_entry_safe(pos3, tmp3, &vendor_specific_tbl[i], list) {
			list_del_rcu(&pos3->list);
			vendor_specific_tbl_cnt--;
			kfree(pos3);
		}
	}

#ifdef QOS_R2
	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		list_for_each_entry_safe(pos4, tmp4, &scs_up_tuple_tbl[i], list) {
			list_del_rcu(&pos4->list);
			scs_up_tuple_tbl_cnt--;
			kfree(pos4);
		}
	}
#endif

	del_timer(&QoS_timer);
	DBGPRINT(DBG_LVL_OFF, "<--%s()\n", __func__);

	return;
}

void hex_dump(char *str, unsigned char *buf, unsigned int len)
{
	unsigned char *pt;
	int x, tmplen = 512, ret = 0, offset = 0;
	unsigned char tmp[512] = {0};

	if(len > 200)
		return ;

	pt = buf;
	memset(tmp, 0, tmplen);
	ret = snprintf(tmp, tmplen, "%s: %p, len = %d\n", str,  buf, len);
	if (os_snprintf_error(tmplen, ret))
		goto error;
	offset += ret;

	for (x = 0; x < len; x++) {
		if (x % 16 == 0) {
			ret = snprintf(tmp+offset, tmplen-offset, "0x%04x : ", x);
			if (os_snprintf_error(tmplen-offset, ret))
				goto error;
			offset += ret;
		}
		ret = snprintf(tmp+offset, tmplen-offset, "%02x ", ((unsigned char)pt[x]));
		if (os_snprintf_error(tmplen-offset, ret))
			goto error;
		offset += ret;

		if (x%16 == 15) {
			ret = snprintf(tmp+offset, tmplen-offset, "\n");
			if (os_snprintf_error(tmplen-offset, ret))
				goto error;
			offset += ret;
		}
	}
	DBGPRINT(DBG_LVL_ERROR, "%s \n", tmp);
error:
	DBGPRINT(DBG_LVL_ERROR, "%s(), error happen, ret:%d\n", __func__, ret);
}

static inline void get_system_uptime(ULONG *time)
{
	*time = jiffies;
}

static inline UINT8 dscp_to_up(UINT8 dscp)
{
	UINT8 tmp = 0, up = 0;
	switch (dscp) {
		case DSCP_CS7:
		case DSCP_CS6:
			up = 7;
			break;

		case DSCP_EF:
		case DSCP_VA:
			up = 6;
			break;
		default:{
			tmp = dscp & 0x38;
			if (tmp == DSCP_CS5)
				up = 5;
			else if (tmp == DSCP_CS4 || tmp == DSCP_CS3)
				up = 4;
			else
				up = 0;
			break;
		}
	}
	return up;
}

static inline UCHAR* protocol2str(UINT8 protocol)
{
	int ret = 0;
	static unsigned char tmp[8] = {0};

	memset(tmp, 0, sizeof(tmp));
	switch (protocol) {
		case IP_TYPE_ICMP:
			ret = snprintf(tmp, sizeof(tmp), "%s", "ICMP");
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
		case IP_TYPE_IGMP:
			ret = snprintf(tmp, sizeof(tmp), "%s", "IGMP");
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
		case IP_TYPE_TCP:
			ret = snprintf(tmp, sizeof(tmp), "%s", "TCP");
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
		case IP_TYPE_UDP:
			ret = snprintf(tmp, sizeof(tmp), "%s", "UDP");
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
		case IP_TYPE_IGRP:
			ret = snprintf(tmp, sizeof(tmp), "%s", "IGRP");
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
		case IP_TYPE_OSPF:
			ret = snprintf(tmp, sizeof(tmp), "%s", "OSPF");
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
		default:
			ret = snprintf(tmp, sizeof(tmp), "%d", protocol);
			if (os_snprintf_error(sizeof(tmp), ret))
				goto error;
			break;
	}
	return tmp;

error:
	memset(tmp, 0, sizeof(tmp));
	DBGPRINT(DBG_LVL_ERROR, "%s(), error happen, ret:%d\n", __func__, ret);
	return tmp;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 1, 0))
static unsigned int mtk_qos_pre_routing(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
static unsigned int mtk_qos_pre_routing(unsigned int hooknum,
	struct sk_buff *skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
#endif
{
	struct mscs_up_tuple up_tuple;
	static struct mirror_classifier_type4 *pmrtype4 = NULL;
	int hash_idx = 0;
	const struct net_device *indev = NULL;
	struct ethhdr *ehdr = eth_hdr(skb);
	struct iphdr *ihdr = ip_hdr(skb);
	struct ipv6hdr *iv6hdr = ipv6_hdr(skb);
	struct tcphdr *thdr = NULL;

	struct mscs_config_entry *pos = NULL;
	char up = skb->cb[UP_OFFSET_1];
	UCHAR is_mscs_active = 0;

	if (!g_QoSMgmtEnabled)
		return NF_ACCEPT;

	if (mscs_config_tbl_cnt < 1)
		return NF_ACCEPT;

	if (up == UP_0)
		up = skb->cb[UP_OFFSET_2];

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 1, 0))
	indev = state->in;
#else
	indev = in;
#endif

	if (ihdr->version == VER_IPV4)
		thdr = (struct tcphdr*)(((void*)ihdr) + ihdr->ihl * 4);
	else if (ihdr->version == VER_IPV6)
		thdr = (struct tcphdr*)(((void*)ihdr) + sizeof(struct ipv6hdr));
	else
		return NF_ACCEPT;

	DBGPRINT(DBG_LVL_TRACE, "[mtqos] >>>pre, recv on %s, DA:%pM, SA:%pM; proto:0x%x, ipv%d, up:%d\n",
		indev->name, ehdr->h_dest, ehdr->h_source, ntohs(ehdr->h_proto), ihdr->version, up);

	if (up < UP_1 || up > UP_MAX || (ehdr->h_dest[0] & 1))
		return NF_ACCEPT;

	hash_idx = MAC_ADDR_HASH_INDEX(ehdr->h_source);
	rcu_read_lock();
	list_for_each_entry(pos, &mscs_config_tbl[hash_idx], list)
		if (MAC_ADDR_EQUAL(ehdr->h_source, pos->cs_param.sta_mac)) {
			is_mscs_active = 1;
			break;
		}
	rcu_read_unlock();

	if (!is_mscs_active) {
		DBGPRINT(DBG_LVL_WARN, "[mtqos] >>>pre, mscs is not active on this STA.\n");
		return NF_ACCEPT;
	}
	/*Only support Classifier Type 4 currently*/
	if (pos->cs_param.cs.header.cs_type != CLASSIFIER_TYPE4) {
		DBGPRINT(DBG_LVL_ERROR, "[mtqos] >>pre, not support classifier type.\n");
		return NF_ACCEPT;
	}

	if (!(pos->cs_param.up_bitmap & bit(up))) {
		DBGPRINT(DBG_LVL_WARN,
			"[mtqos] >>>pre (pos->cs_param.up_bitmap & bit(up))=%x, bit(up)=%d\n",
			(pos->cs_param.up_bitmap & bit(up)),bit(up));
		return NF_ACCEPT;
	}

	memset(&up_tuple, 0, sizeof(up_tuple));
	up_tuple.up = up;
	up_tuple.up_limit = pos->cs_param.up_limit;
	up_tuple.timeout = pos->cs_param.timeout;
	up_tuple.cs.header.cs_type = pos->cs_param.cs.header.cs_type;
	up_tuple.cs.header.cs_mask = pos->cs_param.cs.header.cs_mask;
	COPY_MAC_ADDR(up_tuple.sta_mac, pos->cs_param.sta_mac);

	/*the version field will always be added to tuple*/
	pmrtype4 = &up_tuple.cs.mr_type4;
	pmrtype4->version = ihdr->version;
	if (pmrtype4->version == VER_IPV4) {
		pmrtype4->srcIp.ipv4 = ihdr->saddr;
		pmrtype4->destIp.ipv4 = ihdr->daddr;
		pmrtype4->DSCP = ((ihdr->tos & 0xFC) >> 2);
		pmrtype4->u.protocol = ihdr->protocol;

		DBGPRINT(DBG_LVL_WARN,
			"[mtqos] >>>pre, %s, src ip:"IPV4STR", dst ip:"IPV4STR", srcP:%d, dstP:%d, dscp:%d\n",
			protocol2str(pmrtype4->u.protocol), NIP4(ihdr->saddr), NIP4(ihdr->daddr), ntohs(thdr->source),
			ntohs(thdr->dest), pmrtype4->DSCP);
	} else {
		COPY_IPV6_ADDR(pmrtype4->srcIp.ipv6, &iv6hdr->saddr);
		COPY_IPV6_ADDR(pmrtype4->destIp.ipv6, &iv6hdr->daddr);
		pmrtype4->DSCP = ((((iv6hdr->priority) << 4) | ((iv6hdr->flow_lbl[0] & 0xF0) >> 4)) >> 2);
		pmrtype4->u.nextheader = iv6hdr->nexthdr;
		memcpy(pmrtype4->flowLabel, iv6hdr->flow_lbl, 3);
		pmrtype4->flowLabel[0] &= 0x0F;

		DBGPRINT(DBG_LVL_WARN,
			"[mtqos] >>>pre, src ip:"IPV6STR", dst ip:"IPV6STR", srcP:%d, dstP:%d, dscp:%d\n",
			NIP6(&iv6hdr->saddr), NIP6(&iv6hdr->daddr), ntohs(thdr->source), ntohs(thdr->dest),
			pmrtype4->DSCP);
	}
	pmrtype4->srcPort = thdr->source;
	pmrtype4->destPort = thdr->dest;

	update_mscs_up_tuple(&up_tuple);

	return NF_ACCEPT;
}

static inline void calculate_ipcksum(struct iphdr *ihdr)
{
	unsigned int sum=0;
	unsigned short *addr = (unsigned short *)ihdr;
	int len = ihdr->ihl*4;

	ihdr->check = 0;
	while (len > 1) {
		sum += *addr++;
		len -= 2;
	}
	if(len==1)
		sum += *(unsigned char*)addr;

	sum = (sum >> 16) + (sum & 0xffff);
	sum += (sum>>16);
	ihdr->check = ~sum;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 1, 0))
static unsigned int mtk_qos_post_routing(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
static unsigned int mtk_qos_post_routing(unsigned int hooknum,
		struct sk_buff *skb,
		const struct net_device *in,
		const struct net_device *out,
		int (*okfn)(struct sk_buff *))
#endif
{
	int hash_idx = 0;
	const struct net_device *outdev = NULL;
	struct ethhdr *ehdr = eth_hdr(skb);
	struct iphdr *ihdr = ip_hdr(skb);
	struct ipv6hdr *iv6hdr = ipv6_hdr(skb);
	struct tcphdr *thdr = NULL;
	struct udphdr *uhdr = NULL;
	static struct classifier_type4 pkt_type4;
	struct vendor_specific_up_entry *pos1 = NULL, *tmp1 = NULL;
	struct mscs_up_tuple_entry *pos2 = NULL, *tmp2 = NULL;

#ifdef QOS_R2
	UCHAR *spihdr = NULL;
	struct scs_up_tuple_entry *pos3 = NULL, *tmp3 = NULL;
	static struct classifier_type10 pkt_type10;
#endif
	UCHAR dmac[MAC_ADDR_LEN] = {0};
	UCHAR up = 0;
	ULONG now = 0;
	UCHAR matched = 0, i = 0, dscp = 0, dscp_l = 0, dscp_h = 0;

	if (!g_QoSMgmtEnabled)
		return NF_ACCEPT;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 1, 0))
	outdev = state->out;
#else
	outdev = out;
#endif

	if (ehdr->h_dest[0] & 1)
		return NF_ACCEPT;

	DBGPRINT(DBG_LVL_TRACE, "[mtqos] >>>post, send on %s, DA:%pM, SA:%pM; proto:0x%x, ipv%d\n",
		outdev->name, ehdr->h_dest, ehdr->h_source, ntohs(ehdr->h_proto), ihdr->version);

	COPY_MAC_ADDR(dmac, ehdr->h_dest);

	memset(&pkt_type4, 0, sizeof(pkt_type4));
	pkt_type4.version = ihdr->version;
	if (ihdr->version == VER_IPV4) {
		if (ihdr->protocol == IP_TYPE_TCP) {
			thdr = (struct tcphdr*)(((void*)ihdr) + ihdr->ihl * 4);
			pkt_type4.srcPort = thdr->source;
			pkt_type4.destPort = thdr->dest;
		} else if (ihdr->protocol == IP_TYPE_UDP) {
			uhdr = (struct udphdr*)(((void*)ihdr) + ihdr->ihl * 4);
			pkt_type4.srcPort = uhdr->source;
			pkt_type4.destPort = uhdr->dest;
#ifdef QOS_R2
			spihdr = (UCHAR *)(uhdr) + sizeof(*uhdr);
#endif
		}
#ifdef QOS_R2
		else if (ihdr->protocol == IP_TYPE_ESP)
			spihdr = (UCHAR *)(((void*)ihdr) + ihdr->ihl * 4);
#endif
		pkt_type4.srcIp.ipv4 = ihdr->saddr;
		pkt_type4.destIp.ipv4 = ihdr->daddr;
		pkt_type4.DSCP = ((ihdr->tos & 0xFC) >> 2);
		pkt_type4.u.protocol = ihdr->protocol;

		DBGPRINT(DBG_LVL_TRACE,
			"[mtqos] >>>post, %s, src ip:"IPV4STR", dst ip:"IPV4STR", srcP:%d, dstP:%d, dscp:%d\n",
			protocol2str(pkt_type4.u.protocol), NIP4(pkt_type4.srcIp.ipv4),
			NIP4(pkt_type4.destIp.ipv4), ntohs(pkt_type4.srcPort), ntohs(pkt_type4.destPort),
			pkt_type4.DSCP);
	} else if (ihdr->version == VER_IPV6) {
		if (iv6hdr->nexthdr == IP_TYPE_TCP) {
			thdr = (struct tcphdr*)(((void*)ihdr) + sizeof(struct ipv6hdr));
			pkt_type4.srcPort = thdr->source;
			pkt_type4.destPort = thdr->dest;
		}
		else if (iv6hdr->nexthdr == IP_TYPE_UDP) {
			uhdr = (struct udphdr*)(((void*)ihdr) + sizeof(struct ipv6hdr));
			pkt_type4.srcPort = uhdr->source;
			pkt_type4.destPort = uhdr->dest;
#ifdef QOS_R2
			spihdr = (UCHAR *)(uhdr) + sizeof(*uhdr);
#endif
		}
#ifdef QOS_R2
		else if (iv6hdr->nexthdr == IP_TYPE_ESP)
			spihdr = (UCHAR *)(((void*)ihdr) + sizeof(struct ipv6hdr));
#endif
		COPY_IPV6_ADDR(pkt_type4.srcIp.ipv6, &iv6hdr->saddr);
		COPY_IPV6_ADDR(pkt_type4.destIp.ipv6, &iv6hdr->daddr);
		pkt_type4.DSCP = ((((iv6hdr->priority) << 4) | ((iv6hdr->flow_lbl[0] & 0xF0) >> 4)) >> 2);
		pkt_type4.u.nextheader = iv6hdr->nexthdr;
		memcpy(pkt_type4.flowLabel, iv6hdr->flow_lbl, 3);
		pkt_type4.flowLabel[0] &= 0x0F;

		DBGPRINT(DBG_LVL_TRACE,
			"[mtqos] >>>post, src ip:"IPV6STR", dst ip:"IPV6STR", srcP:%d, dstP:%d, dscp:%d, nxthdr:%d\n",
			NIP6((struct in6_addr*)pkt_type4.srcIp.ipv6), NIP6((struct in6_addr*)pkt_type4.destIp.ipv6),
			ntohs(pkt_type4.srcPort), ntohs(pkt_type4.destPort), pkt_type4.DSCP, pkt_type4.u.nextheader);
	} else {
		return NF_ACCEPT;
	}

#ifdef QOS_R2
	memset(&pkt_type10, 0, sizeof(pkt_type10));
	if (spihdr) {
		pkt_type10.u.protocol = pkt_type4.u.protocol;
		pkt_type10.filterlen = 4;
		memcpy(pkt_type10.filterValue, spihdr, 4);
		memset(pkt_type10.filterMask, 0xFF, 4);
		DBGPRINT(DBG_LVL_TRACE,
			"ESP pkt, protocol:%d filtervalue:0x%02x%02x%02x%02x, filterMask:0x%02x%02x%02x%02x\n",
			pkt_type4.u.protocol, pkt_type10.filterValue[0], pkt_type10.filterValue[1],
			pkt_type10.filterValue[2], pkt_type10.filterValue[3], pkt_type10.filterMask[0],
			pkt_type10.filterMask[1], pkt_type10.filterMask[2], pkt_type10.filterMask[3]);
	}
#endif

	if ((!matched) && (vendor_specific_tbl_cnt > 0)) {
		hash_idx = MAC_ADDR_HASH_INDEX(dmac);
		rcu_read_lock();
		list_for_each_entry_safe(pos1, tmp1, &vendor_specific_tbl[hash_idx], list) {
			if (MAC_ADDR_EQUAL(pos1->vs_cs_param.sta_mac, dmac) &&
				(match_vend_specific_up_tuple_to_stream(&pos1->vs_cs_param, &pkt_type4) == 1)) {
				get_system_uptime(&now);
				pos1->vs_cs_param.expired = now + pos1->vs_cs_param.timeout*HZ;
				up = pos1->vs_cs_param.up;
				matched = 1;
				break;
			}
		}
		rcu_read_unlock();
	}

#ifdef QOS_R2
	if ((!matched) && (scs_up_tuple_tbl_cnt > 0)) {
		hash_idx = MAC_ADDR_HASH_INDEX(dmac);
		rcu_read_lock();
		list_for_each_entry_safe(pos3 , tmp3, &scs_up_tuple_tbl[hash_idx], list) {
			if (MAC_ADDR_EQUAL(pos3->scs_param.sta_mac, dmac) &&
				(match_scs_up_tuple_to_stream(&pos3->scs_param, &pkt_type4, &pkt_type10) == 1)) {
				up = pos3->scs_param.up;
				matched = 1;
				DBGPRINT(DBG_LVL_ERROR, "scs up tuple is matched, up:%d\n", up);
				break;
			}
		}
		rcu_read_unlock();
	}
#endif

	if ((!matched) && (mscs_up_tuple_tbl_cnt > 0)) {
		hash_idx = MAC_ADDR_HASH_INDEX(dmac);
		rcu_read_lock();
		list_for_each_entry_safe(pos2 , tmp2, &mscs_up_tuple_tbl[hash_idx], list) {
			if (MAC_ADDR_EQUAL(pos2->mscstuple.sta_mac, dmac) &&
				(match_mscs_up_tuple_to_stream(&pos2->mscstuple, &pkt_type4) == 1)) {
				get_system_uptime(&now);
				pos2->mscstuple.expired = now + pos2->mscstuple.timeout*HZ;
				up = MIN(pos2->mscstuple.up, pos2->mscstuple.up_limit);
				matched = 1;
				DBGPRINT(DBG_LVL_ERROR, "mscs up tuple is matched, up:%d\n", up);
				break;
			}
		}
		rcu_read_unlock();
	}

	dscp = pkt_type4.DSCP;
	if ((!matched) && QoSMapEnable) {
		for (i = 0; i < QoS_Map.num; i++) {
			if (dscp == QoS_Map.dscp_ex[i].dscp) {
				up = QoS_Map.dscp_ex[i].up;
				matched = 1;
				break;
			}
		}
		for (i = 0; i < UP_MAX && !matched; i++) {
			dscp_l = QoS_Map.dscp_rg[i].low;
			dscp_h = QoS_Map.dscp_rg[i].high;
			if (is_valid_dscp(dscp_l) && is_valid_dscp(dscp_h) &&
				dscp_l <= dscp && dscp <= dscp_h) {
				up = i;
				matched = 1;
				break;
			}
		}
	}
#ifndef DSCP_TO_UP_OFFLOAD
	if ((!matched) && (dscp && (dscp < DSCP_TBL_SIZE))) {
		matched = 1;
		up = dft_dscp2up_tbl[dscp];
	}
#endif

	/*for debug use*/
	if (is_valid_up(g_force_up)) {
		up = g_force_up;
		matched = 1;
	}

	/*set up to DSCP field */
	if ((pkt_type4.version == VER_IPV4) && matched) {
		DBGPRINT(DBG_LVL_WARN, "[mtqos] >>>post, output up is:%d\n", up);
		ihdr->tos &= 0x03;
#ifndef DSCP_TO_UP_OFFLOAD
		ihdr->tos |= (up << 5);
#else
		if (is_valid_dscp(dft_up2dscp[up]))
			ihdr->tos |= (dft_up2dscp[up] << 2);
#endif
		calculate_ipcksum(ihdr);
	} else if ((pkt_type4.version == VER_IPV6) && matched) {
		DBGPRINT(DBG_LVL_WARN, "[mtqos] >>>post, output up is:%d\n", up);
		iv6hdr->priority = 0;
#ifndef DSCP_TO_UP_OFFLOAD
		iv6hdr->priority |= ((up << 1) & 0x0E);
#else
		if (is_valid_dscp(dft_up2dscp[up])) {
			iv6hdr->priority = ((dft_up2dscp[up] & 0x3C) >> 2);
			iv6hdr->flow_lbl[0] &= 0x3F;
			iv6hdr->flow_lbl[0] |= ((dft_up2dscp[up] & 0x03) << 6);
		}
#endif
	}

	return NF_ACCEPT;
}

void free_mscs_config_entry(struct rcu_head *head)
{
	struct mscs_config_entry *entry = container_of(head, struct mscs_config_entry, rcu);
	kfree(entry);
}

int update_mscs_configuration(UCHAR *buf)
{
	UCHAR i, num = 0;
	INT32 hash_idx = 0;
	struct mscs_config_entry *pos = NULL, *tmp = NULL;
	struct classifier_parameter *pcs_param = NULL;

	if (!buf || buf[0] < 1) {
		DBGPRINT(DBG_LVL_ERROR, "mscs configuration data is invalid.\n");
		return -1;
	}

	num = buf[0];
	for (i = 0; i < num; i++) {
		pcs_param = (struct classifier_parameter *)(&buf[1] + sizeof(*pcs_param) * i);
		if (!pcs_param) {
			DBGPRINT(DBG_LVL_ERROR, "pcs_param is NULL.\n");
			return -1;
		}

		hash_idx = MAC_ADDR_HASH_INDEX(pcs_param->sta_mac);
		if (pcs_param->requet_type == REQUEST_TYPE_ADD) {
			rcu_read_lock();
			list_for_each_entry(pos, &mscs_config_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pcs_param->sta_mac, pos->cs_param.sta_mac)) {
					DBGPRINT(DBG_LVL_ERROR,	"mscs config is already active for the STA (%pM)\n",
						pcs_param->sta_mac);
					break;
				}
			}
			rcu_read_unlock();

			if (&pos->list == &mscs_config_tbl[hash_idx]) {
				/*add a new entry to table.*/
				pos = kmalloc(sizeof(struct mscs_config_entry), GFP_ATOMIC);
				if (pos == NULL)
					return -1;

				memset(pos, 0, sizeof(struct mscs_config_entry));
				memcpy(&pos->cs_param, pcs_param, sizeof(struct classifier_parameter));
				spin_lock(&mscs_config_tbl_lock);
				list_add_tail_rcu(&pos->list, &mscs_config_tbl[hash_idx]);
				mscs_config_tbl_cnt++;
				spin_unlock(&mscs_config_tbl_lock);

				DBGPRINT(DBG_LVL_ERROR,	"mscs config is added for the STA (%pM)\n", pcs_param->sta_mac);
			}

		} else if (pcs_param->requet_type == REQUEST_TYPE_REMOVE) {
			if (mscs_config_tbl_cnt < 1)
				return 0;

			spin_lock(&mscs_config_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &mscs_config_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pcs_param->sta_mac, pos->cs_param.sta_mac)) {
					list_del_rcu(&pos->list);
					mscs_config_tbl_cnt--;
					call_rcu(&pos->rcu, free_mscs_config_entry);
					DBGPRINT(DBG_LVL_ERROR,
						"mscs config is removed for the STA (%pM)\n", pcs_param->sta_mac);

					remove_mscs_up_tuple(pcs_param->sta_mac);
					break;
				}
			}
			spin_unlock(&mscs_config_tbl_lock);
		} else if (pcs_param->requet_type == REQUEST_TYPE_CHANGE) {
			spin_lock(&mscs_config_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &mscs_config_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pcs_param->sta_mac, pos->cs_param.sta_mac)) {
					memcpy(&pos->cs_param, pcs_param, sizeof(*pcs_param));
					DBGPRINT(DBG_LVL_ERROR,
						"mscs config is updated for the STA (%pM)\n", pcs_param->sta_mac);
					break;
				}
			}
			spin_unlock(&mscs_config_tbl_lock);
		} else {
			DBGPRINT(DBG_LVL_ERROR, "%s, unknown requet_type:%d\n", __func__, pcs_param->requet_type);
		}
	}

	return 0;
}

#ifdef QOS_R2
void free_scs_tuple_entry(struct rcu_head *head)
{
	struct scs_up_tuple_entry *entry = container_of(head, struct scs_up_tuple_entry, rcu);
	kfree(entry);
}

int update_scs_up_tuple(UCHAR *buf)
{
	UCHAR i, num = 0;
	INT32 hash_idx = 0;
	struct scs_up_tuple_entry *pos = NULL, *tmp = NULL;
	struct scs_classifier_parameter *pscs_param = NULL;

	if (!buf || buf[0] < 1) {
		DBGPRINT(DBG_LVL_ERROR, "scs configuration data is invalid.\n");
		return -1;
	}

	num = buf[0];
	for (i = 0; i < num; i++) {
		pscs_param = (struct scs_classifier_parameter *)(&buf[1] + sizeof(*pscs_param) * i);

		if (!pscs_param) {
			DBGPRINT(DBG_LVL_ERROR, "pscs_param is NULL.\n");
			return -1;
		}

		hash_idx = MAC_ADDR_HASH_INDEX(pscs_param->sta_mac);
		if (pscs_param->requet_type == REQUEST_TYPE_ADD) {
			rcu_read_lock();
			list_for_each_entry(pos, &scs_up_tuple_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pscs_param->sta_mac, pos->scs_param.sta_mac) &&
					pscs_param->scsid == pos->scs_param.scsid) {
					DBGPRINT(DBG_LVL_ERROR,	"scs tuple is already active, scsid:%d, mac:(%pM)\n",
						pscs_param->scsid, pscs_param->sta_mac);
					break;
				}
			}
			rcu_read_unlock();

			if (&pos->list == &scs_up_tuple_tbl[hash_idx]) {
				/*add a new entry to table.*/
				pos = kmalloc(sizeof(struct scs_up_tuple_entry), GFP_ATOMIC);
				if (pos == NULL)
					return -1;

				memset(pos, 0, sizeof(struct scs_up_tuple_entry));
				memcpy(&pos->scs_param, pscs_param, sizeof(*pscs_param));
				spin_lock(&scs_up_tuple_tbl_lock);
				list_add_tail_rcu(&pos->list, &scs_up_tuple_tbl[hash_idx]);
				scs_up_tuple_tbl_cnt++;
				spin_unlock(&scs_up_tuple_tbl_lock);

				DBGPRINT(DBG_LVL_ERROR, "scs tuple is added, scsid:%d, mac:(%pM)\n",
					pscs_param->scsid, pscs_param->sta_mac);
			}
		} else if (pscs_param->requet_type == REQUEST_TYPE_REMOVE) {
			if (scs_up_tuple_tbl_cnt < 1)
				return 0;

			spin_lock(&scs_up_tuple_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &scs_up_tuple_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pscs_param->sta_mac, pos->scs_param.sta_mac) &&
					pscs_param->scsid == pos->scs_param.scsid) {
					list_del_rcu(&pos->list);
					scs_up_tuple_tbl_cnt--;
					call_rcu(&pos->rcu, free_scs_tuple_entry);
					DBGPRINT(DBG_LVL_ERROR, "scs tuple is removed, scsid:%d, mac:(%pM)\n",
							pscs_param->scsid, pscs_param->sta_mac);
					break;
				}
			}
			spin_unlock(&scs_up_tuple_tbl_lock);
		} else if (pscs_param->requet_type == REQUEST_TYPE_CHANGE) {
			spin_lock(&scs_up_tuple_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &scs_up_tuple_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pscs_param->sta_mac, pos->scs_param.sta_mac) &&
					pscs_param->scsid == pos->scs_param.scsid) {
					memcpy(&pos->scs_param, pscs_param, sizeof(*pscs_param));
					DBGPRINT(DBG_LVL_ERROR, "scs config is updated, scsid:%d, mac:(%pM)\n",
							pscs_param->scsid, pscs_param->sta_mac);
					break;
				}
			}
			spin_unlock(&scs_up_tuple_tbl_lock);
		} else {
			DBGPRINT(DBG_LVL_ERROR, "%s, unknown requet_type:%d\n", __func__, pscs_param->requet_type);
		}
	}

	return 0;
}
#endif

void reset_qos_configuration(void)
{
	INT32 i = 0;
	struct mscs_config_entry *pos1 = NULL, *tmp1 = NULL;
	struct mscs_up_tuple_entry *pos2 = NULL, *tmp2 = NULL;
#ifdef QOS_R2
	struct scs_up_tuple_entry *pos3 = NULL, *tmp3 = NULL;
#endif
	for (i = 0; (i < HASH_TABLE_SIZE) && (mscs_config_tbl_cnt > 0); i++) {
		list_for_each_entry_safe(pos1, tmp1, &mscs_config_tbl[i], list) {
			list_del_rcu(&pos1->list);
			call_rcu(&pos1->rcu, free_mscs_config_entry);
			mscs_config_tbl_cnt --;
		}
	}

	for (i = 0; (i < HASH_TABLE_SIZE) && (mscs_up_tuple_tbl_cnt > 0); i++) {
		list_for_each_entry_safe(pos2, tmp2, &mscs_up_tuple_tbl[i], list) {
			list_del_rcu(&pos2->list);
			call_rcu(&pos2->rcu, free_mscs_up_tuple_entry);
			mscs_up_tuple_tbl_cnt--;
		}
	}

#ifdef QOS_R2
	for (i = 0; (i < HASH_TABLE_SIZE) && (scs_up_tuple_tbl_cnt > 0); i++) {
		list_for_each_entry_safe(pos3, tmp3, &scs_up_tuple_tbl[i], list) {
			list_del_rcu(&pos3->list);
			call_rcu(&pos3->rcu, free_scs_tuple_entry);
			scs_up_tuple_tbl_cnt --;
		}
	}
#endif
}

int update_qos_map(UCHAR *buf, unsigned short buflen)
{
	UCHAR i;
	struct qos_map *pQos_Map = NULL;

	DBGPRINT(DBG_LVL_ERROR, "-->%s()\n", __func__);

	if (!buf) {
		DBGPRINT(DBG_LVL_ERROR, "qos map data is invalid.\n");
		return -1;
	}

	/*reset setting*/
	if (buflen == 0) {
		QoSMapEnable = 0;
		memset(&QoS_Map, 0, sizeof(QoS_Map));
		return 0;
	}

	pQos_Map = (struct qos_map *)buf;
	if (!pQos_Map) {
		DBGPRINT(DBG_LVL_ERROR, "pQos_Map is NULL.\n");
		return -1;
	}

	if (pQos_Map->num > MAX_EXCEPT_CNT) {
		DBGPRINT(DBG_LVL_ERROR, "invalid pQos_Map->num:%d.\n", pQos_Map->num);
		return -1;
	}

	memset(&QoS_Map, 0, sizeof(QoS_Map));

	QoS_Map.num = pQos_Map->num;

	for (i = 0; i < QoS_Map.num; i++) {
		if (is_valid_dscp(pQos_Map->dscp_ex[i].dscp) && is_valid_up(pQos_Map->dscp_ex[i].up))
			memcpy(&QoS_Map.dscp_ex[i], &pQos_Map->dscp_ex[i], sizeof(struct dscp_exception));
	}

	for (i = 0; i < UP_MAX; i++) {
		if (is_valid_dscp(pQos_Map->dscp_rg[i].low) && is_valid_dscp(pQos_Map->dscp_rg[i].high) &&
			pQos_Map->dscp_rg[i].low < pQos_Map->dscp_rg[i].high)
			memcpy(&QoS_Map.dscp_rg[i], &pQos_Map->dscp_rg[i], sizeof(struct dscp_range));
		else
			memset(&QoS_Map.dscp_rg[i], 0xFF, sizeof(struct dscp_range));
	}

	QoSMapEnable = 1;

	return 0;
}


void free_vendor_specific_entry(struct rcu_head *head)
{
	struct vendor_specific_up_entry *entry = container_of(head, struct vendor_specific_up_entry, rcu);
	kfree(entry);
}

int update_vendor_specific_up_tuple(UCHAR *buf)
{
	ULONG now = 0;
	UCHAR i, num = 0;
	INT32 hash_idx = 0;
	struct vendor_specific_up_entry *pos = NULL, *tmp = NULL;
	struct wapp_vend_spec_classifier_para_report *pvs_cs_param = NULL;

	if (!buf || buf[0] < 1) {
		DBGPRINT(DBG_LVL_ERROR, "vendor specific data is invalid.\n");
		return -1;
	}

	num = buf[0];

	for (i = 0; i < num; i++) {
		pvs_cs_param = (struct wapp_vend_spec_classifier_para_report *)(&buf[1] + sizeof(*pvs_cs_param) * i);
		if (!pvs_cs_param) {
			DBGPRINT(DBG_LVL_ERROR, "pvs_cs_param is NULL.\n");
			return -1;
		}

		get_system_uptime(&now);
		hash_idx = MAC_ADDR_HASH_INDEX(pvs_cs_param->sta_mac);
		if (pvs_cs_param->requet_type == REQUEST_TYPE_ADD) {
			rcu_read_lock();
			list_for_each_entry(pos, &vendor_specific_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pvs_cs_param->sta_mac, pos->vs_cs_param.sta_mac) &&
					pvs_cs_param->id == pos->vs_cs_param.id) {
					DBGPRINT(DBG_LVL_ERROR,
						"vendor specific up tuple is already existed, id:%d, mac:%pM\n",
						pvs_cs_param->id, pvs_cs_param->sta_mac);
					break;
				}
			}
			rcu_read_unlock();

			if (&pos->list == &vendor_specific_tbl[hash_idx]) {
				/*add a new entry to table.*/
				pos = kmalloc(sizeof(struct vendor_specific_up_entry), GFP_ATOMIC);
				if (pos == NULL)
					return -1;

				memset(pos, 0, sizeof(struct vendor_specific_up_entry));
				memcpy(&pos->vs_cs_param, pvs_cs_param, sizeof(*pvs_cs_param));

				pos->vs_cs_param.expired = now + pos->vs_cs_param.timeout*HZ;

				spin_lock_bh(&vendor_specific_tbl_lock);
				list_add_tail_rcu(&pos->list, &vendor_specific_tbl[hash_idx]);
				vendor_specific_tbl_cnt++;
				spin_unlock_bh(&vendor_specific_tbl_lock);

				DBGPRINT(DBG_LVL_ERROR,
					"vendor specific up tuple is added, id:%d, mac:%pM, cnt:%d\n",
					pvs_cs_param->id, pvs_cs_param->sta_mac, vendor_specific_tbl_cnt);
			}
		} else if (pvs_cs_param->requet_type == REQUEST_TYPE_REMOVE) {
			if (vendor_specific_tbl_cnt < 1)
				return 0;

			spin_lock_bh(&vendor_specific_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &vendor_specific_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pvs_cs_param->sta_mac, pos->vs_cs_param.sta_mac) &&
					pvs_cs_param->id == pos->vs_cs_param.id) {
					list_del_rcu(&pos->list);
					vendor_specific_tbl_cnt--;
					call_rcu(&pos->rcu, free_vendor_specific_entry);

					DBGPRINT(DBG_LVL_ERROR,	"vendor specific is removed, id:%d, mac:%pM\n",
						pvs_cs_param->id, pvs_cs_param->sta_mac);
					break;
				}
			}
			spin_unlock_bh(&vendor_specific_tbl_lock);
		} else if (pvs_cs_param->requet_type == REQUEST_TYPE_CHANGE) {
			spin_lock_bh(&vendor_specific_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &vendor_specific_tbl[hash_idx], list) {
				if (MAC_ADDR_EQUAL(pvs_cs_param->sta_mac, pos->vs_cs_param.sta_mac) &&
					pvs_cs_param->id == pos->vs_cs_param.id) {
					memcpy(&pos->vs_cs_param, pvs_cs_param, sizeof(*pvs_cs_param));
					pos->vs_cs_param.expired = now + pos->vs_cs_param.timeout*HZ;

					DBGPRINT(DBG_LVL_ERROR,	"vendor specific is updated, id:%d, mac:%pM\n",
						pvs_cs_param->id, pvs_cs_param->sta_mac);
					break;
				}
			}
			spin_unlock_bh(&vendor_specific_tbl_lock);
		} else {
			DBGPRINT(DBG_LVL_ERROR, "%s, unknown requet_type:%d\n", __func__, pvs_cs_param->requet_type);
		}
	}

	return 0;
}
/*  compare two classifier of type4,
 *  mask: bitmap to indicate parameter to be compared.
 *  ptype4_1/ptype4_2 to be compared.
 *  return: 1: ptype4_1/ptype4_2 was matched, others: not matched.
 *
 */
static inline INT32 compare_classifier_type4(struct classifier_type4 *ptype4_1, struct classifier_type4 *ptype4_2)
{
	UCHAR mask;

	if (!ptype4_1 || !ptype4_2)
		return -1;

	mask = ptype4_1->cs_mask;
	if ((mask & TYPE4_CS_MASK_VERSION) && (ptype4_1->version != ptype4_2->version))
		return 0;

	if (ptype4_1->version == VER_IPV4) {
		if (mask & TYPE4_CS_MASK_SRC_IP) {
			if (ptype4_1->srcIp.ipv4 != ptype4_2->srcIp.ipv4)
				return 0;
		}
		if (mask & TYPE4_CS_MASK_DST_IP) {
			if (ptype4_1->destIp.ipv4 != ptype4_2->destIp.ipv4)
				return 0;
		}
		if (mask & TYPE4_CS_MASK_PROTOCOL) {
			if (ptype4_1->u.protocol != ptype4_2->u.protocol)
				return 0;
		}
	}

	if (ptype4_1->version == VER_IPV6) {
		if (mask & TYPE4_CS_MASK_SRC_IP) {
			if (!IPV6_ADDR_EQUAL(ptype4_1->srcIp.ipv6, ptype4_2->srcIp.ipv6))
				return 0;
		}
		if (mask & TYPE4_CS_MASK_DST_IP) {
			if (!IPV6_ADDR_EQUAL(ptype4_1->destIp.ipv6, ptype4_2->destIp.ipv6))
				return 0;
		}
		if (mask & TYPE4_CS_MASK_NXT_HEAD) {
			if (ptype4_1->u.nextheader != ptype4_2->u.nextheader)
				return 0;
		}
		if (mask & TYPE4_CS_MASK_FLOW_LABEL) {
			if (memcmp(ptype4_1->flowLabel, ptype4_2->flowLabel, 3))
				return 0;
		}
	}
	if (mask & TYPE4_CS_MASK_SRC_PORT) {
		if (ptype4_1->srcPort != ptype4_2->srcPort)
			return 0;
	}
	if (mask & TYPE4_CS_MASK_DST_PORT) {
		if (ptype4_1->destPort != ptype4_2->destPort)
			return 0;
	}
	if (mask & TYPE4_CS_MASK_DSCP) {
		if (ptype4_1->DSCP != ptype4_2->DSCP)
			return 0;
	}

	/*all parameters are matched*/
	return 1;
}

static inline INT32 compare_classifier_type10(
	struct classifier_type10 *ptype10_1, struct classifier_type10 *ptype10_2)
{
	UCHAR mask = 0, i = 0, fltlen = 0;

	if (!ptype10_1 || !ptype10_2)
		return -1;

	if (ptype10_1->protocolInstance != ptype10_2->protocolInstance ||
		ptype10_1->u.protocol != ptype10_2->u.protocol ||
		ptype10_1->filterlen != ptype10_2->filterlen)
		return 0;

	fltlen = MIN(ptype10_1->filterlen, MAX_FILTER_LEN);

	for (i = 0; i < fltlen; i++) {
		mask = (ptype10_1->filterMask[i] & ptype10_2->filterMask[i]);
		if ((ptype10_1->filterValue[i] & mask) != (ptype10_2->filterValue[i] & mask))
			return 0;
	}
	/*all parameters are matched*/
	return 1;
}

static inline INT32 is_mscs_up_tuple_equal(struct mscs_up_tuple *tuple1, struct mscs_up_tuple *tuple2)
{
	if (!tuple1 || !tuple2)
		return -1;

	if (!MAC_ADDR_EQUAL(tuple1->sta_mac, tuple2->sta_mac) ||
		/*tuple1->up != tuple2->up ||*/
		tuple1->up_limit != tuple2->up_limit ||
		tuple1->timeout != tuple2->timeout ||
		tuple1->cs.header.cs_type != tuple2->cs.header.cs_type ||
		tuple1->cs.header.cs_mask != tuple2->cs.header.cs_mask)
		return 0;

	if (tuple1->cs.header.cs_type == CLASSIFIER_TYPE4) {
		return compare_classifier_type4(&tuple1->cs.type4, &tuple2->cs.type4);
	} else
		return -1; /*not support request type other than type4*/
}

static inline INT32 match_mscs_up_tuple_to_stream(struct mscs_up_tuple *up_tuple,
	struct classifier_type4 *pkt_tuple)
{
	if (!up_tuple || !pkt_tuple)
		return -1;

	if (up_tuple->cs.header.cs_type == CLASSIFIER_TYPE4) {
		return compare_classifier_type4(&up_tuple->cs.type4, pkt_tuple);
	} else
		return -1; /*not support cs type other than type4*/
}

#ifdef QOS_R2
static inline INT32 match_scs_up_tuple_to_stream(struct scs_classifier_parameter *scs_tuple,
	struct classifier_type4 *pkt_type4, struct classifier_type10 *pkt_type10)
{
	UCHAR i = 0;
	INT32 matched = 0;
	ptclas_element ptclas = NULL;

	if (!scs_tuple || !pkt_type4 || !pkt_type10)
		return -1;

	if (scs_tuple->tclas_num == 0) {
		DBGPRINT(DBG_LVL_ERROR, "no scs classfier parameter.\n");
		return -1;
	}

	for (i = 0; (i < scs_tuple->tclas_num) && (i < MAX_TCLAS_NUM); i++) {
		ptclas = &scs_tuple->tclas_elem[i];
		if (scs_tuple->processing == 0) { /*match all tclas elements*/
			if (ptclas->header.cs_type == CLASSIFIER_TYPE4)
				matched = compare_classifier_type4(&ptclas->type4, pkt_type4);
			else if (ptclas->header.cs_type == CLASSIFIER_TYPE10)
				matched = compare_classifier_type10(&ptclas->type10, pkt_type10);
			else
				matched = 0; /*not support request type other than type4*/

			if (matched == 0)
				break; /*not matched*/
		} else { /*match at least one tclas element*/
			if (ptclas->header.cs_type == CLASSIFIER_TYPE4) {
				matched = compare_classifier_type4(&ptclas->type4, pkt_type4);
			} else if (ptclas->header.cs_type == CLASSIFIER_TYPE10) {
				matched = compare_classifier_type10(&ptclas->type10, pkt_type10);
			} else
				matched = 0; /*not support request type other than type4*/

			if (matched == 1)
				break; /*matched*/
		}
	}

	return matched;
}
#endif

static inline INT32 match_vend_specific_up_tuple_to_stream(
	struct wapp_vend_spec_classifier_para_report *vs_up_tuple, struct classifier_type4 *pkt_tuple)
{
	if (!vs_up_tuple || !pkt_tuple)
		return -1;

	if (pkt_tuple->version == VER_IPV4) {
		if (vs_up_tuple->protocol != pkt_tuple->u.protocol ||
			vs_up_tuple->srcIp.ipv4 != pkt_tuple->srcIp.ipv4 ||
			vs_up_tuple->destIp.ipv4 != pkt_tuple->destIp.ipv4 ||
			vs_up_tuple->srcPort != pkt_tuple->srcPort ||
			vs_up_tuple->destPort != pkt_tuple->destPort)
			return 0;
	} else if (pkt_tuple->version == VER_IPV6) {
		if (!IPV6_ADDR_EQUAL(vs_up_tuple->srcIp.ipv6, pkt_tuple->srcIp.ipv6) ||
			!IPV6_ADDR_EQUAL(vs_up_tuple->destIp.ipv6, pkt_tuple->destIp.ipv6) ||
			vs_up_tuple->srcPort != pkt_tuple->srcPort ||
			vs_up_tuple->destPort != pkt_tuple->destPort)
			return 0;
	} else
		return -1;

	return 1;
}


int update_mscs_up_tuple(struct mscs_up_tuple *up_tuple)
{
	INT32 hash_idx = 0;
	struct mscs_up_tuple_entry *pos = NULL, *tmp = NULL;
	ULONG now = 0;

	if (!up_tuple ||
		MAC_ADDR_EQUAL(ZERO_MAC_ADDR, up_tuple->sta_mac)) {
		DBGPRINT(DBG_LVL_ERROR, "mir_cs_param is invalid.\n");
		return -1;
	}

	if (up_tuple->cs.header.cs_type != CLASSIFIER_TYPE4) {
		DBGPRINT(DBG_LVL_ERROR, "add mscs up tuple fail, only type4 was supportted currently.\n");
		return -1;
	}

	get_system_uptime(&now);

	hash_idx = MAC_ADDR_HASH_INDEX(up_tuple->sta_mac);

	spin_lock_bh(&mscs_up_tuple_tbl_lock);
	list_for_each_entry_safe(pos, tmp, &mscs_up_tuple_tbl[hash_idx], list) {
		if (is_mscs_up_tuple_equal(&pos->mscstuple, up_tuple) == 1) {
			pos->mscstuple.expired = now + pos->mscstuple.timeout*HZ;
			pos->mscstuple.up = up_tuple->up;
			DBGPRINT(DBG_LVL_WARN, "mscs up tuple was existed\n");
			break;
		}
	}
	spin_unlock_bh(&mscs_up_tuple_tbl_lock);

	if (&pos->list == &mscs_up_tuple_tbl[hash_idx]) {
		/*add a new entry to table.*/
		pos = kmalloc(sizeof(struct mscs_up_tuple_entry), GFP_ATOMIC);
		if (pos == NULL)
			return -1;

		memcpy(&pos->mscstuple, up_tuple, sizeof(*up_tuple));
		pos->mscstuple.expired = now + pos->mscstuple.timeout*HZ;

		spin_lock_bh(&mscs_up_tuple_tbl_lock);
		list_add_tail_rcu(&pos->list, &mscs_up_tuple_tbl[hash_idx]);
		mscs_up_tuple_tbl_cnt++;
		spin_unlock_bh(&mscs_up_tuple_tbl_lock);

		DBGPRINT(DBG_LVL_ERROR, "mscs up tuple is added for STA (%pM), cnt:%d\n",
			up_tuple->sta_mac, mscs_up_tuple_tbl_cnt);
	}

	return 0;
}

int remove_mscs_up_tuple(UCHAR *dmac)
{
	INT32 hash_idx = 0;
	struct mscs_up_tuple_entry *pos = NULL, *tmp = NULL;

	if (!dmac || MAC_ADDR_EQUAL(ZERO_MAC_ADDR, dmac)) {
		DBGPRINT(DBG_LVL_ERROR, "dmac is invalid.\n");
		return -1;
	}

	if (mscs_up_tuple_tbl_cnt > 0) {
		hash_idx = MAC_ADDR_HASH_INDEX(dmac);
		spin_lock_bh(&mscs_up_tuple_tbl_lock);
		list_for_each_entry_safe(pos, tmp, &mscs_up_tuple_tbl[hash_idx], list) {
			if (MAC_ADDR_EQUAL(&pos->mscstuple.sta_mac, dmac)) {
				list_del_rcu(&pos->list);
				mscs_up_tuple_tbl_cnt--;
				call_rcu(&pos->rcu, free_mscs_up_tuple_entry);
				DBGPRINT(DBG_LVL_ERROR, "%s(), remove:%pM, cnt:%d\n",
					__func__, dmac, mscs_up_tuple_tbl_cnt);
			}
		}
		spin_unlock_bh(&mscs_up_tuple_tbl_lock);
	}
	return 0;
}

void free_mscs_up_tuple_entry(struct rcu_head *head)
{
	struct mscs_up_tuple_entry *entry = container_of(head, struct mscs_up_tuple_entry, rcu);
	kfree(entry);
}

void mscs_up_tuple_timeout(void)
{
	INT32 i = 0, j = 0;
	struct mscs_up_tuple_entry *pos = NULL, *tmp = NULL;
	ULONG now = 0;

	if (mscs_up_tuple_tbl_cnt > 0) {
		j = 0;
		memset(expired_idx, 0, sizeof(expired_idx));
		get_system_uptime(&now);

		/*1. Find exipred hash index.*/
		rcu_read_lock();
		for (i = 0; i < HASH_TABLE_SIZE; i++) {
			list_for_each_entry(pos, &mscs_up_tuple_tbl[i], list) {
				if (time_after(now, pos->mscstuple.expired))
					expired_idx[j++] = i;
			}
		}
		rcu_read_unlock();

		/*2. delete entry according to exipred hash index.*/
		for (i = 0; i < j; i++) {
			spin_lock_bh(&mscs_up_tuple_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &mscs_up_tuple_tbl[expired_idx[i]], list) {
				if (time_after(now, pos->mscstuple.expired)) {
					list_del_rcu(&pos->list);
					mscs_up_tuple_tbl_cnt--;
					call_rcu(&pos->rcu, free_mscs_up_tuple_entry);
					DBGPRINT(DBG_LVL_WARN, "-->%s(), cnt:%d\n", __func__, mscs_up_tuple_tbl_cnt);
				}
			}
			spin_unlock_bh(&mscs_up_tuple_tbl_lock);
		}
	}
}

void vendor_specific_up_tuple_timeout(void)
{
	INT32 i = 0, j = 0;
	ULONG now = 0;
	struct vendor_specific_up_entry *pos = NULL, *tmp = NULL;
	struct qos_netlink_message *msg = NULL;

	if (vendor_specific_tbl_cnt > 0) {
		j = 0;
		memset(expired_idx, 0, sizeof(expired_idx));

		msg = (struct qos_netlink_message *)send_buf;
		msg->type = NL_VEND_SPECIFIC_CONFIG_EXPIRED;

		get_system_uptime(&now);

		/*1. Find exipred hash index.*/
		rcu_read_lock();
		for (i = 0; i < HASH_TABLE_SIZE; i++) {
			list_for_each_entry(pos, &vendor_specific_tbl[i], list) {
				if (time_after(now, pos->vs_cs_param.expired))
					expired_idx[j++] = i;
			}
		}
		rcu_read_unlock();

		for (i = 0; i < j; i++) {
			spin_lock_bh(&vendor_specific_tbl_lock);
			list_for_each_entry_safe(pos, tmp, &vendor_specific_tbl[expired_idx[i]], list) {
				if (time_after(now, pos->vs_cs_param.expired)) {
					msg->len = sizeof(pos->vs_cs_param);
					memcpy(&msg->data[0], &pos->vs_cs_param, msg->len);
					send_nlmsg((char *)msg, sizeof(*msg) + msg->len);

					list_del_rcu(&pos->list);
					vendor_specific_tbl_cnt--;
					call_rcu(&pos->rcu, free_vendor_specific_entry);

					DBGPRINT(DBG_LVL_WARN, "-->%s(), expired id:%d, left cnt:%d\n",
						__func__, pos->vs_cs_param.id, vendor_specific_tbl_cnt);
				}
			}
			spin_unlock_bh(&vendor_specific_tbl_lock);
		}
	}
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,14,0)
static void periodic_timer_function(unsigned long data)
#else
static void periodic_timer_function(struct timer_list *data)
#endif
{
	/*mscs_up_tuple_timeout();*/
	vendor_specific_up_tuple_timeout();

	mod_timer(&QoS_timer, (jiffies + TIMER_INTERVAL * HZ));
/*
	del_timer(&QoS_timer);
	QoS_timer.function = periodic_timer_function;
	QoS_timer.expires = jiffies + TIMER_INTERVAL * HZ;
	add_timer(&QoS_timer);
*/
}

int mtqos_event_handler(struct qos_netlink_message *msg)
{
	int ret = 0;

	DBGPRINT(DBG_LVL_WARN, "[mtqos] -->%s(), msg->type:%d, msg->len:%d\n", __func__, msg->type, msg->len);
	switch(msg->type) {
		case NL_SET_MSCS_CONFIG:
			update_mscs_configuration(&msg->data[0]);
			break;
		case NL_SET_QOS_MAP:
			update_qos_map(&msg->data[0], msg->len);
			break;
		case NL_SET_VEND_SPECIFIC_CONFIG:
			update_vendor_specific_up_tuple(&msg->data[0]);
			break;
		case NL_RESET_QOS_CONFIG:
			reset_qos_configuration();
			break;
#ifdef QOS_R2
		case NL_SET_SCS_CONFIG:
			update_scs_up_tuple(&msg->data[0]);
			break;
#endif
		default:
			DBGPRINT(DBG_LVL_ERROR, "[mtqos] unknown msg type(%02x)\n", msg->type);
			break;
	}

	return ret;
}

void recv_nlmsg(struct sk_buff *skb)
{
	struct nlmsghdr *nlh = nlmsg_hdr(skb);
	struct qos_netlink_message *msg = NULL;

	if (nlh->nlmsg_len < NLMSG_HDRLEN || skb->len < nlh->nlmsg_len)
		return;

	g_pid = nlh->nlmsg_pid;

	msg = (struct qos_netlink_message *)NLMSG_DATA(nlh);
	mtqos_event_handler(msg);
}

void send_nlmsg(char *buf, int buflen)
{
	struct nlmsghdr *nlh = NULL;
	struct sk_buff *skb = NULL;

	if (g_pid <= 0)
		return;

	skb = alloc_skb(NLMSG_SPACE(MAX_MSGSIZE), GFP_KERNEL);
	if(!skb){
		DBGPRINT(DBG_LVL_ERROR, "alloc skb error\n");
		return;
	}

	nlh = nlmsg_put(skb, 0, 0, 0, MAX_MSGSIZE, 0);
	NETLINK_CB(skb).portid = 0;
	NETLINK_CB(skb).dst_group = 0;

	memcpy(NLMSG_DATA(nlh), buf, buflen);
	netlink_unicast(nl_sk, skb, g_pid, MSG_DONTWAIT);
}

static inline UINT8 expired_check(ULONG time)
{
	ULONG now = 0;
	get_system_uptime(&now);
	if (time_after(now, time))
		return 0;
	else
		return ((time-now)/HZ);
}

static ssize_t qos_show_info(struct kobject *kobj,
		struct kobj_attribute *attr, char *buffer)
{
	UINT16 i = 0, No = 0, buflen = 2048;
	struct mscs_config_entry *pos1 = NULL;
	struct classifier_parameter *pConf = NULL;
	struct mscs_up_tuple_entry *pos2 = NULL;
	struct mscs_up_tuple *pTuple = NULL;
	struct vendor_specific_up_entry *pos3 = NULL;
	struct wapp_vend_spec_classifier_para_report *pvend_Tuple = NULL;
#ifdef QOS_R2
	UINT8 j = 0, k = 0;
	struct scs_up_tuple_entry *pos4 = NULL;
	struct scs_classifier_parameter *pScsTuple = NULL;
	ptclas_element ptclas = NULL;
#endif
	unsigned char *buf = NULL, *vs_buf = NULL;
	int ret = 0, offset = 0, vs_offset = 0;

	if (!buffer)
		return 0;

	buf = kmalloc(buflen, GFP_ATOMIC);
	if (!buf)
		return 0;

	memset(buf, 0, buflen);

	if (mscs_config_tbl_cnt) {
		ret = snprintf(buf + offset, buflen - offset, "[mtqos]---------MSCS Configuration---------\n");
		if (os_snprintf_error(buflen - offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf + offset, buflen - offset, " NO.      sta_mac      requet_type up_bitmap up_limit");
		if (os_snprintf_error(buflen - offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf + offset, buflen - offset, " timeout cs_type cs_mask\n");
		if (os_snprintf_error(buflen - offset, ret))
			goto error;
		offset += ret;
		rcu_read_lock();
		for (i = 0, No = 0; i < HASH_TABLE_SIZE; i++) {
			list_for_each_entry(pos1, &mscs_config_tbl[i], list) {
				pConf = &pos1->cs_param;
				ret = snprintf(buf + offset, buflen - offset,
					" %2d  %pM      %d        0x%x %8d %8d  %5d      0x%x\n",
					No++, pConf->sta_mac, pConf->requet_type, pConf->up_bitmap, pConf->up_limit,
					pConf->timeout, pConf->cs.header.cs_type, pConf->cs.header.cs_mask);
				if (os_snprintf_error(buflen - offset, ret))
					goto error1;
				offset += ret;
			}
		}
		rcu_read_unlock();
	}

	if (mscs_up_tuple_tbl_cnt) {
		ret = snprintf(buf + offset, buflen - offset, "\n[mtqos]---------MSCS UP{tuple}---------\n");
		if (os_snprintf_error(buflen - offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf + offset, buflen - offset,
			" NO.      sta_mac       up up_limit timeout cs_type cs_mask");
		if (os_snprintf_error(buflen - offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf + offset, buflen - offset,
			" expired version srcPort destPort protocol   srcIp           destIp\n");
		if (os_snprintf_error(buflen - offset, ret))
			goto error;
		offset += ret;
		rcu_read_lock();
		for (i = 0, No = 0; i < HASH_TABLE_SIZE; i++) {
			list_for_each_entry(pos2, &mscs_up_tuple_tbl[i], list) {
				pTuple = &pos2->mscstuple;
				ret = snprintf(buf+offset, buflen-offset,
					" %2d  %pM %3d %4d %8d %7d     0x%x %7d %6d %9d %7d %8s",
					No++, pTuple->sta_mac, pTuple->up, pTuple->up_limit, pTuple->timeout,
					pTuple->cs.type4.cs_type, pTuple->cs.type4.cs_mask,
					expired_check(pTuple->expired),
					pTuple->cs.type4.version, ntohs(pTuple->cs.type4.srcPort),
					ntohs(pTuple->cs.type4.destPort), protocol2str(pTuple->cs.type4.u.protocol));
				if (os_snprintf_error(buflen - offset, ret))
					goto error1;
				offset += ret;

				if (pTuple->cs.type4.version == VER_IPV4) {
					ret = snprintf(buf+offset, buflen-offset,
						"   "IPV4STR, NIP4(pTuple->cs.type4.srcIp.ipv4));
					if (os_snprintf_error(buflen - offset, ret))
						goto error1;
					offset += ret;

					ret = snprintf(buf+offset, buflen-offset,
						"   "IPV4STR, NIP4(pTuple->cs.type4.destIp.ipv4));
					if (os_snprintf_error(buflen - offset, ret))
						goto error1;
					offset += ret;

				} else {
					ret = snprintf(buf+offset, buflen-offset, IPV6STR,
						NIP6((struct in6_addr*)(pTuple->cs.type4.srcIp.ipv6)));
					if (os_snprintf_error(buflen - offset, ret))
						goto error1;
					offset += ret;

					ret = snprintf(buf+offset, buflen-offset, IPV6STR,
						NIP6((struct in6_addr*)(pTuple->cs.type4.destIp.ipv6)));
					if (os_snprintf_error(buflen - offset, ret))
						goto error1;
					offset += ret;
				}
				ret = snprintf(buf+offset, buflen-offset, "\n");
				if (os_snprintf_error(buflen - offset, ret))
					goto error1;
				offset += ret;
			}
		}
		rcu_read_unlock();
	}

	if (vendor_specific_tbl_cnt) {
		vs_buf = kmalloc(buflen, GFP_ATOMIC);
		if (vs_buf) {
			memset(vs_buf, 0, buflen);
			ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
				"\n[mtqos]---------Vendor Specific  UP{tuple}---------\n");
			if (os_snprintf_error(buflen-vs_offset, ret)) {
				DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n", __func__, __LINE__, ret);
				goto exit;
			}
			vs_offset += ret;
			ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
				"vendor_specific_tbl_cnt:%d\n", vendor_specific_tbl_cnt);
			if (os_snprintf_error(buflen-vs_offset, ret)) {
				DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n", __func__, __LINE__, ret);
				goto exit;
			}
			vs_offset += ret;
			ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
				" NO.  id        sta_mac        up delay_bound protocol timeout");
			if (os_snprintf_error(buflen-vs_offset, ret)) {
				DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n", __func__, __LINE__, ret);
				goto exit;
			}
			vs_offset += ret;
			ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
				" expired version srcPort destPort    srcIp           destIp         hashidx   ifname\n");
			if (os_snprintf_error(buflen-vs_offset, ret)) {
				DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n", __func__, __LINE__, ret);
				goto exit;
			}
			vs_offset += ret;
			rcu_read_lock();
			for (i = 0, No = 0; i < HASH_TABLE_SIZE; i++) {
				list_for_each_entry(pos3, &vendor_specific_tbl[i], list) {
					pvend_Tuple = &pos3->vs_cs_param;
					ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
						" %2d  %04d  %pM %4d %7d %9d %7d %7d %7d %9d %7d",
						No++, pvend_Tuple->id, pvend_Tuple->sta_mac, pvend_Tuple->up,
						pvend_Tuple->delay_bound, pvend_Tuple->protocol, pvend_Tuple->timeout,
						expired_check(pvend_Tuple->expired), pvend_Tuple->version,
						ntohs(pvend_Tuple->srcPort), ntohs(pvend_Tuple->destPort));
					if (os_snprintf_error(buflen-vs_offset, ret)) {
						DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
							__func__, __LINE__, ret);
						goto exit1;
					}
					vs_offset += ret;
					if (pvend_Tuple->version == VER_IPV4) {
						ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
							"   "IPV4STR, NIP4(pvend_Tuple->srcIp.ipv4));
						if (os_snprintf_error(buflen-vs_offset, ret)) {
							DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
								__func__, __LINE__, ret);
							goto exit1;
						}
						vs_offset += ret;
						ret = snprintf(vs_buf+vs_offset, buflen-vs_offset,
							"   "IPV4STR, NIP4(pvend_Tuple->destIp.ipv4));
						if (os_snprintf_error(buflen-vs_offset, ret)) {
							DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
								__func__, __LINE__, ret);
							goto exit1;
						}
						vs_offset += ret;
					} else {
						ret = snprintf(vs_buf+vs_offset, buflen-vs_offset, IPV6STR,
							NIP6((struct in6_addr*)(pvend_Tuple->srcIp.ipv6)));
						if (os_snprintf_error(buflen-vs_offset, ret)) {
							DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
								__func__, __LINE__, ret);
							goto exit1;
						}
						vs_offset += ret;
						ret = snprintf(vs_buf+vs_offset, buflen-vs_offset, IPV6STR,
							NIP6((struct in6_addr*)(pvend_Tuple->destIp.ipv6)));
						if (os_snprintf_error(buflen-vs_offset, ret)) {
							DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
								__func__, __LINE__, ret);
							goto exit1;
						}
						vs_offset += ret;
					}
					ret = snprintf(vs_buf+vs_offset, buflen-vs_offset, "%7d   %s", i, pvend_Tuple->ifname);
					if (os_snprintf_error(buflen-vs_offset, ret)) {
						DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
							__func__, __LINE__, ret);
						goto exit1;
					}
					vs_offset += ret;
					ret = snprintf(vs_buf+vs_offset, buflen-vs_offset, "\n");
					if (os_snprintf_error(buflen-vs_offset, ret)) {
						DBGPRINT(DBG_LVL_ERROR, "%s(), line:%d error happen, ret:%d\n",
							__func__, __LINE__, ret);
						goto exit1;
					}
					vs_offset += ret;
				}
			}
exit1:
			rcu_read_unlock();
exit:
			DBGPRINT(DBG_LVL_ERROR, "%s\n", vs_buf);
			kfree(vs_buf);
		}
	}

#ifdef QOS_R2
	if (scs_up_tuple_tbl_cnt) {
		ret = snprintf(buf+offset, buflen-offset, "\n[mtqos]---------SCS UP{tuple}---------\n");
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf+offset, buflen-offset,
			" scsid	    sta_mac	   up alt_queue drop_elig processing\n");
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;

		rcu_read_lock();
		for (i = 0; i < HASH_TABLE_SIZE; i++) {
			list_for_each_entry(pos4, &scs_up_tuple_tbl[i], list) {
				pScsTuple = &pos4->scs_param;
				ret = snprintf(buf+offset, buflen-offset, " %3d   %pM %3d %5d %8d %10d",
					pScsTuple->scsid, pScsTuple->sta_mac, pScsTuple->up,
					pScsTuple->alt_queue, pScsTuple->drop_elig, pScsTuple->processing);
				if (os_snprintf_error(buflen-offset, ret))
					goto error1;
				offset += ret;

				for (j = 0; (j < pScsTuple->tclas_num) && (j < MAX_TCLAS_NUM); j++) {
					ptclas = &pScsTuple->tclas_elem[j];
					if (ptclas->header.cs_type == CLASSIFIER_TYPE4) {
						ret = snprintf(buf+offset, buflen-offset,
							" {type4, mask:0x%x, version:%d, srcP:%d, dstP:%d, dscp:%d,",
							ptclas->type4.cs_mask, ptclas->type4.version,
							ntohs(ptclas->type4.srcPort), ntohs(ptclas->type4.destPort),
							ptclas->type4.DSCP);
						if (os_snprintf_error(buflen-offset, ret))
							goto error1;
						offset += ret;

						if (ptclas->type4.version == VER_IPV4) {
							ret = snprintf(buf+offset, buflen-offset,
								"srcIP:"IPV4STR", dstIP:"IPV4STR" protocol:%d},",
								NIP4(ptclas->type4.srcIp.ipv4),
								NIP4(ptclas->type4.destIp.ipv4),
								ptclas->type4.u.protocol);
							if (os_snprintf_error(buflen-offset, ret))
								goto error1;
							offset += ret;
						} else {
							ret = snprintf(buf+offset, buflen-offset,
								"srcIP:"IPV6STR", dstIP:"IPV6STR", proto/nxthd:%d,",
								NIP6((struct in6_addr*)ptclas->type4.srcIp.ipv6),
								NIP6((struct in6_addr*)ptclas->type4.destIp.ipv6),
								ptclas->type4.u.nextheader);
							if (os_snprintf_error(buflen-offset, ret))
								goto error1;
							offset += ret;

							ret = snprintf(buf+offset, buflen-offset,
								" flabel:0x%02x%02x%02x},",
								ptclas->type4.flowLabel[0], ptclas->type4.flowLabel[1],
								ptclas->type4.flowLabel[2]);
							if (os_snprintf_error(buflen-offset, ret))
								goto error1;
							offset += ret;
						}
					} else if (ptclas->header.cs_type == CLASSIFIER_TYPE10) {
						ret = snprintf(buf+offset, buflen-offset,
							" {type10, protoInst:%d, proto/nxthd:%d,",
							ptclas->type10.protocolInstance, ptclas->type10.u.protocol);
						if (os_snprintf_error(buflen-offset, ret))
							goto error1;
						offset += ret;
						ret = snprintf(buf+offset, buflen-offset,
							" filterlen:%d, filterValue:0x",
							ptclas->type10.filterlen);
						if (os_snprintf_error(buflen-offset, ret))
							goto error1;
						offset += ret;
						for (k = 0; (k < ptclas->type10.filterlen) && (k < MAX_FILTER_LEN); k++) {
							ret = snprintf(buf+offset, buflen-offset, "%02x",
								ptclas->type10.filterValue[k]);
							if (os_snprintf_error(buflen-offset, ret))
								goto error1;
							offset += ret;
						}
						ret = snprintf(buf+offset, buflen-offset, ", filterMask:0x");
						if (os_snprintf_error(buflen-offset, ret))
							goto error1;
						offset += ret;
						for (k = 0; (k < ptclas->type10.filterlen) && (k < MAX_FILTER_LEN); k++) {
							ret = snprintf(buf+offset, buflen-offset, "%02x",
								ptclas->type10.filterMask[k]);
							if (os_snprintf_error(buflen-offset, ret))
								goto error1;
							offset += ret;
						}
						ret = snprintf(buf+offset, buflen-offset, "},");
						if (os_snprintf_error(buflen-offset, ret))
							goto error1;
						offset += ret;
					}
				}
				ret = snprintf(buf+offset, buflen-offset, "\n");
				if (os_snprintf_error(buflen-offset, ret))
					goto error1;
				offset += ret;
			}
		}
		rcu_read_unlock();
	}
#endif

	if (QoSMapEnable) {
		ret = snprintf(buf+offset, buflen-offset, "\n[QoSR1]: DSCP Exception, Count:%d.\n", QoS_Map.num);
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf+offset, buflen-offset, "\t DSCP \t UP\n");
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;
		for (i = 0; i < QoS_Map.num; i++) {
			ret = snprintf(buf+offset, buflen-offset, "\t %3d \t %2d \n",
				QoS_Map.dscp_ex[i].dscp, QoS_Map.dscp_ex[i].up);
			if (os_snprintf_error(buflen-offset, ret))
				goto error;
			offset += ret;
		}

		ret = snprintf(buf+offset, buflen-offset, "\n[QoSR1]: DSCP Range.\n");
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;
		ret = snprintf(buf+offset, buflen-offset, "\t UP \t DSCP_L \t DSCP_H\n");
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;
		for (i = 0; i < UP_MAX; i++) {
			ret = snprintf(buf+offset, buflen-offset, "\t %2d \t %4d \t\t %4d \n",
				i, QoS_Map.dscp_rg[i].low, QoS_Map.dscp_rg[i].high);
			if (os_snprintf_error(buflen-offset, ret))
				goto error;
			offset += ret;
		}
	}

#ifdef DSCP_TO_UP_OFFLOAD
	ret = snprintf(buf+offset, buflen-offset, "\n[QoSR1]: up to dscp mapping table.\n");
	if (os_snprintf_error(buflen-offset, ret))
		goto error;
	offset += ret;
	ret = snprintf(buf+offset, buflen-offset, "\t UP \t DSCP\n");
	if (os_snprintf_error(buflen-offset, ret))
		goto error;
	offset += ret;
	for (i = 0; i < UP_MAX; i++) {
		ret = snprintf(buf+offset, buflen-offset, "\t %2d \t %4d\n", i, dft_up2dscp[i]);
		if (os_snprintf_error(buflen-offset, ret))
			goto error;
		offset += ret;
	}
#endif

	if (strlen(buf))
		memcpy(buffer, buf, strlen(buf));
	kfree(buf);

	return strlen(buffer);
error1:
	rcu_read_unlock();

error:
	kfree(buf);
	DBGPRINT(DBG_LVL_ERROR, "%s(), error happen, ret:%d\n", __func__, ret);
	return 0;
}

static ssize_t qos_set_dbg_level(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count)
{
	int dbglvl = simple_strtol(buf, 0, 10);
	if (DBG_LVL_OFF <= dbglvl && dbglvl <= DBG_LVL_LOUD) {
		dbg_level = dbglvl;
		DBGPRINT(DBG_LVL_OFF, "qos dbg level set to %d\n", dbglvl);
	} else
		DBGPRINT(DBG_LVL_OFF, "value is invalid, it should be 0~5\n");

	return count;
}

static ssize_t qos_set_enable(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count)
{
	int enable = simple_strtol(buf, 0, 10);

	if (enable == 1) {
		g_QoSMgmtEnabled = enable;
		DBGPRINT(DBG_LVL_OFF, "mtqos was enabled.\n");
	} else if (enable == 0) {
		g_QoSMgmtEnabled = enable;
		DBGPRINT(DBG_LVL_OFF, "mtqos was disabled.\n");
	} else
		DBGPRINT(DBG_LVL_OFF, "valid value is 0/1, 0:disable mtqos, 1:enable mtqos.\n");

	return count;
}

static ssize_t qos_force_up(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count)
{
	int up = simple_strtol(buf, 0, 10);
	if (UP_0 < up && up < UP_MAX) {
		g_force_up = up;
		DBGPRINT(DBG_LVL_OFF, "force up to %d\n", up);
	} else {
		g_force_up = 0xFF;
		DBGPRINT(DBG_LVL_OFF, "up value is invalid, it should be 1~7\n");
	}

	return count;
}

module_init(mtqos_init);
module_exit(mtqos_exit);
