//==========================================================================
//
//      sync/mutex.cxx
//
//      Mutex and condition variable implementation
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
// ----------------------------------------------------------------------
// Portions of this software Copyright (c) 2003-2011 Broadcom Corporation
// ----------------------------------------------------------------------
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    nickg
// Contributors: nickg, jlarmour
// Date:         1999-02-17
// Purpose:      Mutex implementation
// Description:  This file contains the implementations of the mutex
//               and condition variable classes.
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/kernel.h>

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros
#include <cyg/kernel/instrmnt.h>       // instrumentation
#include <cyg/error/codes.h>           // error codes

#include <cyg/kernel/mutex.hxx>        // our header

#include <cyg/kernel/thread.inl>       // thread inlines
#include <cyg/kernel/sched.inl>        // scheduler inlines
#include <cyg/kernel/clock.inl>        // clock inlines


#ifdef CYGPKG_HAL_MIPS
extern "C" void MipsStackTrace( cyg_uint32 pc, cyg_uint32 sp, cyg_uint32 ra );
#endif

// -------------------------------------------------------------------------
// Mutex protocol test macros.
// If the dynamic protocol option is enabled, then these generate appropriate
// tests on the protocol field. If there is no dynamic choice then they simply
// result in empty statements.

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DYNAMIC

#define IF_PROTOCOL_INHERIT if( protocol == INHERIT )
#define IF_PROTOCOL_CEILING if( protocol == CEILING )
#define IF_PROTOCOL_ACTIVE  if( protocol != NONE )

#else

#define IF_PROTOCOL_INHERIT
#define IF_PROTOCOL_CEILING
#define IF_PROTOCOL_ACTIVE

#endif

// -------------------------------------------------------------------------
// Constructor

Cyg_Mutex::Cyg_Mutex()
{
    CYG_REPORT_FUNCTION();

    locked      = false;
    owner       = NULL;

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT
    next        = NULL;
#endif

#if defined(CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT) && \
    defined(CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DYNAMIC)

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_INHERIT
    protocol    = INHERIT;
#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_CEILING
    protocol    = CEILING;
    ceiling     = CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_PRIORITY;
#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_NONE
    protocol    = NONE;
#endif

#else // not (DYNAMIC and DEFAULT defined)

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_PRIORITY

    // if there is a default priority ceiling defined, use that to initialize
    // the ceiling.
    ceiling = CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_PRIORITY;

#else

    // Otherwise set it to zero.
    ceiling = 0;

#endif
#endif

#endif // DYNAMIC and DEFAULT defined

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------
// Construct with defined protocol

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DYNAMIC

Cyg_Mutex::Cyg_Mutex( cyg_protcol protocol_arg )
{
    CYG_REPORT_FUNCTION();

    locked      = false;
    owner       = NULL;

    protocol    = protocol_arg;

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT
    next        = NULL;
#endif

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_PRIORITY

    // if there is a default priority ceiling defined, use that to initialize
    // the ceiling.
    ceiling = CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DEFAULT_PRIORITY;

#else

    // Otherwise set it to zero.
    ceiling = 0;

#endif
#endif

    CYG_REPORT_RETURN();
}

#endif

// -------------------------------------------------------------------------
// Destructor

Cyg_Mutex::~Cyg_Mutex()
{
    CYG_REPORT_FUNCTION();

    CYG_ASSERT( owner == NULL, "Deleting mutex with owner");
    CYG_ASSERT( queue.empty(), "Deleting mutex with waiting threads");
    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL
/// cyg_bool detect_mutex_loop()
///
/// Detects a mutex deadlock or loop.  Recursive.
/// NOTE: Must be called with kernel locked.
///
/// Parameters: thread - Start point for loop detection.
///
/// Returns:  true if loop detected
///
cyg_bool
Cyg_Mutex::detect_mutex_loop( Cyg_Thread *thread )
{
    cyg_bool loop_detected = false;
    Cyg_Thread *owner=NULL;

    if ( thread->marked() )
    {
        diag_printf( "ERROR!  Mutex deadlock detected!\n");
        return true;
    }

    // Mark the thread so we can tell we've been here before.
    thread->mark();

    // Get the mutex this thread is waiting on, or NULL if it isn't.
    Cyg_Mutex *mutex_waiting = thread->get_mutex_waiting();
    if ( ( mutex_waiting != NULL ) &&
         ( ( owner = mutex_waiting->get_owner() ) != NULL ) )
    {
        // Check the owner of the mutex we're waiting on.
        loop_detected = detect_mutex_loop( owner );
        if ( loop_detected )
        {
            diag_printf( "ERROR!!! Mutex Deadlock detected, thread=%08p waiting for mutex=%08p\n", thread, mutex_waiting);
            // Print a list of mutexes held by this thread.
            Cyg_Mutex *mutex_list = thread->get_mutex_list();
            while ( mutex_list != NULL )
            {
                diag_printf( "    Holding %08p\n", mutex_list );
                mutex_list = mutex_list->get_next();
            }
#ifdef CYGPKG_HAL_MIPS
            {
                // dump callstack of the deadlock threads.
                cyg_uint32 ra, pc, sp;
                if ( thread == Cyg_Thread::self())
                {
                    asm( "move %0, $31;"
                         "la   %1, 1f;"  // local label with "f - searches forward only".
                         "move %2, $29;"
                         "1:" :  // this is the local label for the "la" instruction for PC
                         "=r" (ra),
                         "=r" (pc),
                         "=r" (sp) );
                }
                else
                {
                    HAL_SavedRegisters *exc_regs = (HAL_SavedRegisters*) thread->get_stack_ptr();
                    ra = exc_regs->d[31];
                    pc = exc_regs->pc;
                    sp = exc_regs->d[29];
                }
                diag_printf( "thread %08p callstack:\n", thread );
                MipsStackTrace( pc, sp, ra );
            }
#endif
        }
    }

    thread->unmark();

    return loop_detected;
}

#endif

#ifdef CYGDBG_USE_ASSERTS

cyg_bool
Cyg_Mutex::check_this( cyg_assert_class_zeal zeal) const
{
//    CYG_REPORT_FUNCTION();

    // check that we have a non-NULL pointer first
    if( this == NULL ) return false;

    switch( zeal )
    {
    case cyg_system_test:
    case cyg_extreme:
    case cyg_thorough:
    case cyg_quick:
    case cyg_trivial:
        if(  locked && owner == NULL ) return false;
        if( !locked && owner != NULL ) return false;
    case cyg_none:
    default:
        break;
    };

    return true;
}

#endif

// -------------------------------------------------------------------------
// Lock and/or wait

cyg_bool
Cyg_Mutex::lock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_bool result = true;
    Cyg_Thread *self = Cyg_Thread::self();

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_ASSERTCLASS( this, "Bad this pointer");

    CYG_INSTRUMENT_MUTEX(LOCK, this, 0);

    // Loop while the mutex is locked, sleeping each time around
    // the loop. This copes with the possibility of a higher priority
    // thread grabbing the mutex between the wakeup in unlock() and
    // this thread actually starting.

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL

    self->wait_mutex( this );
    IF_PROTOCOL_ACTIVE
        self->count_mutex();

#endif

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

    IF_PROTOCOL_CEILING
        self->set_priority_ceiling(ceiling);

#endif

    while( locked && result )
    {
        CYG_ASSERT( self != owner, "Locking mutex I already own");

        self->set_sleep_reason( Cyg_Thread::WAIT );

        self->sleep();

        queue.enqueue( self );

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT

        IF_PROTOCOL_INHERIT
            owner->inherit_priority(self);

#endif

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL
        // Deadlock loop detection
        detect_mutex_loop(self);
#endif
        CYG_INSTRUMENT_MUTEX(WAIT, this, 0);

        // Allow other threads to run
        Cyg_Scheduler::reschedule();

        CYG_ASSERTCLASS( this, "Bad this pointer");

        switch( self->get_wake_reason() )
        {
        case Cyg_Thread::DESTRUCT:
        case Cyg_Thread::BREAK:
            result = false;
            break;

        case Cyg_Thread::EXIT:
            self->exit();
            break;

        default:
            break;
        }

    }

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL
    self->unwait_mutex();
#endif

    if( result )
    {
        locked      = true;
        owner       = self;

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT
        IF_PROTOCOL_INHERIT
            owner->own_mutex( this );
#endif
        CYG_INSTRUMENT_MUTEX(LOCKED, this, 0);
    }
    else
    {
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL

       IF_PROTOCOL_ACTIVE
           self->uncount_mutex();

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT

        IF_PROTOCOL_INHERIT
            self->disinherit_priority();

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

        IF_PROTOCOL_CEILING
            self->clear_priority_ceiling();

#endif
    }

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");

    CYG_REPORT_RETVAL(result);

    return result;
}

// -------------------------------------------------------------------------
// Lock and/or wait until timeout expiry

cyg_bool
Cyg_Mutex::lock( cyg_tick_count timeout )
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_bool result = true;
    Cyg_Thread *self = Cyg_Thread::self();

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_ASSERTCLASS( this, "Bad this pointer");

    CYG_INSTRUMENT_MUTEX(LOCK, this, 0);

    // Loop while the mutex is locked, sleeping each time around
    // the loop. This copes with the possibility of a higher priority
    // thread grabbing the mutex between the wakeup in unlock() and
    // this thread actually starting.

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL

    self->wait_mutex( this );

    IF_PROTOCOL_ACTIVE
        self->count_mutex();

#endif

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

    IF_PROTOCOL_CEILING
        self->set_priority_ceiling(ceiling);

#endif

    // Set the timer and sleep reason
    self->set_timer( timeout, Cyg_Thread::TIMEOUT );

    if( self->get_wake_reason() != Cyg_Thread::NONE )
        result = false;

    while( locked && result )
    {
        CYG_ASSERT( self != owner, "Locking mutex I already own");

        // must reset the sleep reason every time
        self->set_sleep_reason( Cyg_Thread::TIMEOUT );

        self->sleep();

        queue.enqueue( self );

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT

        IF_PROTOCOL_INHERIT
            owner->inherit_priority(self);

#endif

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL
        // Deadlock loop detection
        detect_mutex_loop(self);
#endif
        CYG_INSTRUMENT_MUTEX(WAIT, this, 0);

        // Allow other threads to run
        Cyg_Scheduler::reschedule();

        CYG_ASSERTCLASS( this, "Bad this pointer");

        switch( self->get_wake_reason() )
        {
        case Cyg_Thread::TIMEOUT:
        case Cyg_Thread::DESTRUCT:
        case Cyg_Thread::BREAK:
            result = false;
            break;

        case Cyg_Thread::EXIT:

            self->exit();
            break;

        default:
            break;
        }

    }

    // clear the timer; if it actually fired, no worries.
    self->clear_timer();

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL
    self->unwait_mutex();
#endif

    if( result )
    {
        locked      = true;
        owner       = self;

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT
        IF_PROTOCOL_INHERIT
            owner->own_mutex( this );
#endif
        CYG_INSTRUMENT_MUTEX(LOCKED, this, 0);
    }
    else
    {
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL

       IF_PROTOCOL_ACTIVE
           self->uncount_mutex();

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT

        IF_PROTOCOL_INHERIT
            self->disinherit_priority();

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

        IF_PROTOCOL_CEILING
            self->clear_priority_ceiling();

#endif
    }

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");

    CYG_REPORT_RETVAL(result);

    return result;
}

// -------------------------------------------------------------------------
// Try to lock and return success

cyg_bool
Cyg_Mutex::trylock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    CYG_ASSERTCLASS( this, "Bad this pointer");

    cyg_bool result = true;

    // Prevent preemption
    Cyg_Scheduler::lock();

    // If the mutex is not locked, grab it
    // for ourself. Otherwise return failure.
    if( !locked )
    {
        Cyg_Thread *self = Cyg_Thread::self();

        locked  = true;
        owner   = self;

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL

       IF_PROTOCOL_ACTIVE
            self->count_mutex();

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT
        IF_PROTOCOL_INHERIT
            owner->own_mutex( this );
#endif

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

        IF_PROTOCOL_CEILING
            self->set_priority_ceiling(ceiling);

#endif

    }
    else result = false;

    CYG_INSTRUMENT_MUTEX(TRY, this, result);

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETVAL(result);
    return result;
}

// -------------------------------------------------------------------------
// unlock

void
Cyg_Mutex::unlock(void)
{
    CYG_REPORT_FUNCTION();

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_INSTRUMENT_MUTEX(UNLOCK, this, 0);

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERT( locked, "Unlock mutex that is not locked");
    CYG_ASSERT( owner == Cyg_Thread::self(), "Unlock mutex I do not own");

    if( !queue.empty() ) {

        // The queue is non-empty, so grab the next
        // thread from it and wake it up.

        Cyg_Thread *thread = queue.dequeue();

        CYG_ASSERTCLASS( thread, "Bad thread pointer");

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT

        // Give the owner-to-be a chance to inherit from the remaining
        // queue or the relinquishing thread:

        IF_PROTOCOL_INHERIT
            thread->relay_priority(owner, &queue);

#endif

        thread->set_wake_reason( Cyg_Thread::DONE );

        thread->wake();

        CYG_INSTRUMENT_MUTEX(WAKE, this, thread);

    }

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL

    IF_PROTOCOL_ACTIVE
	owner->uncount_mutex();

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_INHERIT

    IF_PROTOCOL_INHERIT
    {
        // We have to disown this mutex before disinheriting the priority,
        // or else we'll re-inherit the priority from this mutex.
        owner->disown_mutex( this );
        owner->disinherit_priority();
    }

#endif
#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

    IF_PROTOCOL_CEILING
        owner->clear_priority_ceiling();

#endif

    locked      = false;
    owner       = NULL;

    CYG_ASSERTCLASS( this, "Bad this pointer");

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------
// Release all waiting threads.

void Cyg_Mutex::release()
{
    CYG_REPORT_FUNCTION();

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_INSTRUMENT_MUTEX(RELEASE, this, 0);

    CYG_ASSERTCLASS( this, "Bad this pointer");

    while( !queue.empty() )
    {
        // The queue is non-empty, so grab each
        // thread from it and release it.

        Cyg_Thread *thread = queue.dequeue();

        CYG_ASSERTCLASS( thread, "Bad thread pointer");

        thread->release();

        CYG_INSTRUMENT_MUTEX(RELEASED, this, thread);

    }

    CYG_ASSERTCLASS( this, "Bad this pointer");

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------
// Set ceiling priority for priority ceiling protocol

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_CEILING

void Cyg_Mutex::set_ceiling( cyg_priority priority )
{
    CYG_REPORT_FUNCTION();

//    CYG_ASSERT( priority >=  CYG_THREAD_MAX_PRIORITY, "Priority out of range");
//    CYG_ASSERT( priority <=  CYG_THREAD_MIN_PRIORITY, "Priority out of range");

    // Prevent preemption
    Cyg_Scheduler::lock();

    ceiling = priority;

    // Unlock the scheduler
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETURN();
}

#endif

// -------------------------------------------------------------------------
// Set priority inversion protocol

#ifdef CYGSEM_KERNEL_SYNCH_MUTEX_PRIORITY_INVERSION_PROTOCOL_DYNAMIC
void Cyg_Mutex::set_protocol( cyg_protcol new_protocol )
{
    CYG_REPORT_FUNCTION();

    // Prevent preemption
    Cyg_Scheduler::lock();

    protocol = new_protocol;

    // Unlock the scheduler
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETURN();
}

#endif


//==========================================================================
// Condition variables

Cyg_Condition_Variable::Cyg_Condition_Variable(
    Cyg_Mutex   &mx                // linked mutex
    )
{
    CYG_REPORT_FUNCTION();

    mutex       = &mx;

    CYG_ASSERTCLASS( mutex, "Invalid mutex argument");

    CYG_REPORT_RETURN();
}

Cyg_Condition_Variable::Cyg_Condition_Variable()
{
    CYG_REPORT_FUNCTION();

    mutex       = NULL;

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------
// Destructor

Cyg_Condition_Variable::~Cyg_Condition_Variable()
{
    CYG_REPORT_FUNCTION();

    CYG_ASSERT( queue.empty(), "Deleting condvar with waiting threads");

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------

#ifdef CYGDBG_USE_ASSERTS

cyg_bool
Cyg_Condition_Variable::check_this( cyg_assert_class_zeal zeal) const
{
    bool result = true;

    CYG_REPORT_FUNCTYPE("returning %d");
    CYG_REPORT_FUNCARG1("zeal = %d", zeal);

    // check that we have a non-NULL pointer first
    if( this == NULL )
        result = false;
    else {

        switch( zeal )
        {
        case cyg_system_test:
        case cyg_extreme:
        case cyg_thorough:
            if( mutex != NULL && !mutex->check_this(zeal) )
                result = false;
        case cyg_quick:
        case cyg_trivial:
        case cyg_none:
        default:
            break;
        }
    }

    CYG_REPORT_RETVAL(result);
    return result;
}

#endif

// -------------------------------------------------------------------------
// Wait for condition to be true
// Note: if this function is entered with the scheduler locked (e.g. to
// suspend DSR processing) then there is no need to take the lock.  Also
// in this case, exit with the scheduler locked, which allows this function
// to be used in a totally thread-safe manner.

cyg_bool
Cyg_Condition_Variable::wait_inner( Cyg_Mutex *mx )
{
    CYG_REPORT_FUNCTION();

    cyg_bool result = true;
    Cyg_Thread *self = Cyg_Thread::self();

    Cyg_Scheduler::lock();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERTCLASS( mx, "Corrupt mutex");
    CYG_ASSERTCLASS( self, "Bad self thread");

    CYG_INSTRUMENT_CONDVAR(WAIT, this, 0);

    mx->unlock();

    self->set_sleep_reason( Cyg_Thread::WAIT );

    self->sleep();

    queue.enqueue( self );

    // Avoid calling ASRs during the following unlock.
    self->set_asr_inhibit();

    // Unlock the scheduler and switch threads
    Cyg_Scheduler::unlock_reschedule();

    // Allow ASRs again
    self->clear_asr_inhibit();

    CYG_INSTRUMENT_CONDVAR(WOKE, this, self->get_wake_reason());

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERTCLASS( mx, "Corrupt mutex");

    switch( self->get_wake_reason() )
    {
    case Cyg_Thread::DESTRUCT:          // which, the cv or the mutex?
    case Cyg_Thread::BREAK:
        result = false;
        break;

    case Cyg_Thread::EXIT:
        self->exit();
        break;

    default:
        break;
    }

    // When we awake, we must re-acquire the mutex.  Note that while
    // it is essential to release the mutex and queue on the CV
    // atomically relative to other threads, to avoid races, it is not
    // necessary for us to re-acquire the mutex in the same atomic
    // action. Hence we can do it after unlocking the scheduler.
    // We need to loop here in case the thread is released while waiting
    // for the mutex. It is essential that we exit this function with the
    // mutex claimed.

    while ( !mx->lock() )
        continue;

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERTCLASS( mx, "Corrupt mutex");
    CYG_ASSERT( mx->owner == self, "Not mutex owner");

    CYG_REPORT_RETURN();

    return result;
}

// -------------------------------------------------------------------------
// Wake one thread

void
Cyg_Condition_Variable::signal(void)
{
    CYG_REPORT_FUNCTION();

    CYG_ASSERTCLASS( this, "Bad this pointer");

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_INSTRUMENT_CONDVAR(SIGNAL, this, 0);

    if( !queue.empty() )
    {
        // The queue is non-empty, so grab the next
        // thread from it and wake it up.

        Cyg_Thread *thread = queue.dequeue();

        CYG_ASSERTCLASS( thread, "Bad thread pointer");

        thread->set_wake_reason( Cyg_Thread::DONE );

        thread->wake();

        CYG_INSTRUMENT_CONDVAR(WAKE, this, thread);

    }

    CYG_ASSERTCLASS( this, "Bad this pointer");

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------
// Set cond true, wake all threads

void
Cyg_Condition_Variable::broadcast(void)
{
    CYG_REPORT_FUNCTION();

    CYG_ASSERTCLASS( this, "Bad this pointer");

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_INSTRUMENT_CONDVAR(BROADCAST, this, 0);

    // Grab all the threads from the queue and let them
    // go.

    while( !queue.empty() )
    {
        Cyg_Thread *thread = queue.dequeue();

        CYG_ASSERTCLASS( thread, "Bad thread pointer");

        thread->set_wake_reason( Cyg_Thread::DONE );

        thread->wake();

        CYG_INSTRUMENT_CONDVAR(WAKE, this, thread);
    }

    CYG_ASSERTCLASS( this, "Bad this pointer");

    // Unlock the scheduler and maybe switch threads
    Cyg_Scheduler::unlock();

    CYG_REPORT_RETURN();
}

// -------------------------------------------------------------------------
// Optional timed wait on a CV

#if defined(CYGMFN_KERNEL_SYNCH_CONDVAR_TIMED_WAIT)

cyg_bool
Cyg_Condition_Variable::wait_inner( Cyg_Mutex *mx, cyg_tick_count timeout )
{
    CYG_REPORT_FUNCTYPE("returning %d");
    CYG_REPORT_FUNCARG1("timeout = %d", timeout);

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERTCLASS( mx, "Corrupt mutex");

    cyg_bool result = true;

    Cyg_Thread *self = Cyg_Thread::self();

    CYG_ASSERTCLASS( self, "Bad self thread");

    // Prevent preemption
    Cyg_Scheduler::lock();

    CYG_INSTRUMENT_CONDVAR(TIMED_WAIT, this, 0 );

    mx->unlock();

    // The ordering of sleep() and set_timer() here are
    // important. If the timeout is in the past, the thread
    // will be woken up immediately and will not sleep.

    self->sleep();

    // Set the timer and sleep reason
    self->set_timer( timeout, Cyg_Thread::TIMEOUT );

    // Only enqueue if the timeout has not already fired.
    if( self->get_wake_reason() == Cyg_Thread::NONE )
        queue.enqueue( self );

    // Avoid calling ASRs during the following unlock.
    self->set_asr_inhibit();

    // Unlock the scheduler and switch threads
    Cyg_Scheduler::unlock_reschedule();

    // Allow ASRs again
    self->clear_asr_inhibit();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERTCLASS( mx, "Corrupt mutex");

    self->clear_timer();

    CYG_INSTRUMENT_CONDVAR(WOKE, this, self->get_wake_reason());

    switch( self->get_wake_reason() )
    {
    case Cyg_Thread::TIMEOUT:
    case Cyg_Thread::DESTRUCT:          // which, the cv or the mutex?
    case Cyg_Thread::BREAK:
        result = false;
        break;

    case Cyg_Thread::EXIT:
        self->exit();
        break;

    default:
        break;
    }


    // When we awake, we must re-acquire the mutex.  Note that while
    // it is essential to release the mutex and queue on the CV
    // atomically relative to other threads, to avoid races, it is not
    // necessary for us to re-acquire the mutex in the same atomic
    // action. Hence we can do it after unlocking the scheduler.

    while ( !mx->lock() )
        continue;

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_ASSERTCLASS( mx, "Corrupt mutex");

    CYG_REPORT_RETVAL(result);

    return result;
}

#endif

//==========================================================================
// Read-write lock
// Write-preferred read-write lock based on a mutex and condition variable
Cyg_RW_Lock::Cyg_RW_Lock()
{
    CYG_REPORT_FUNCTION();
    int i;

    mutex       = new Cyg_Mutex();
    CYG_ASSERTCLASS( mutex, "Failed to create mutex");
    rcond        = new Cyg_Condition_Variable(*mutex);
    CYG_ASSERTCLASS( rcond, "Failed to reader cond var");
    wcond        = new Cyg_Condition_Variable(*mutex);
    CYG_ASSERTCLASS( wcond, "Failed to writer cond var");

    num_pend_wrs = 0;
    num_pend_rds = 0;
    num_active_rds = 0;
    write_owner = NULL;
    for (i = 0; i < MAX_READ_OWNERS_LIST_SIZE; i++)
        read_owners[i] = NULL;
    kind = eKIND_DEAULT;

    CYG_REPORT_RETURN();
}
// -------------------------------------------------------------------------
// Destructor

Cyg_RW_Lock::~Cyg_RW_Lock()
{
    CYG_REPORT_FUNCTION();
    CYG_ASSERT( num_pend_wrs == 0, "Deleting rw_lock with waiting write threads");
    CYG_ASSERT( num_pend_rds == 0, "Deleting rw_lock with waiting write threads");
    CYG_ASSERT( num_active_rds == 0, "Deleting rw_lock with active read threads");
    CYG_ASSERT( write_owner == NULL, "Deleting rw_lock with active write thread");

    delete mutex;
    delete rcond;
    delete wcond;

    CYG_REPORT_RETURN();
}


#ifdef CYGDBG_USE_ASSERTS

cyg_bool
Cyg_RW_Lock::check_this( cyg_assert_class_zeal zeal) const
{
    bool result = true;

    CYG_REPORT_FUNCTYPE("returning %d");
    CYG_REPORT_FUNCARG1("zeal = %d", zeal);

    // check that we have a non-NULL pointer first
    if( this == NULL )
        result = false;
    else {

        switch( zeal )
        {
        case cyg_system_test:
        case cyg_extreme:
        case cyg_thorough:
            if( mutex != NULL && !mutex->check_this(zeal) )
                result = false;
        case cyg_quick:
        case cyg_trivial:
            if( num_active_rds && write_owner != NULL ) return false;
            if( num_active_rds > MAX_READ_OWNERS_LIST_SIZE ) return false;
        case cyg_none:
        default:
            break;
        }
    }

    CYG_REPORT_RETVAL(result);
    return result;
}

#endif


cyg_int32
Cyg_RW_Lock::rdlock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_int32 result = ENOERR;
    Cyg_Thread *self = Cyg_Thread::self();
    mutex->lock();
    // The current thread already owns the read-write lock for writing.
    if ( self == write_owner )
        result = EDEADLK;
    // The read lock could not be acquired because the maximum number of read locks for rwlock has been exceeded.
    else if ( (num_active_rds + num_pend_rds) >= MAX_READ_OWNERS_LIST_SIZE )
        result = EAGAIN;
    else
    {
        num_pend_rds++;
        // keep waiting when there is a writer or (pending writer in write-preferred, not read-preferred)
        while ( write_owner ||
                (num_pend_wrs && (kind != eKIND_READ_PREFERRED)) )
        {
            rcond->wait();
        }
        num_pend_rds--;
        read_owners[num_active_rds++] = self;
    }
    mutex->unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_REPORT_RETVAL(result);
    return result;
}

cyg_int32
Cyg_RW_Lock::wrlock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_int32 result = ENOERR;
    Cyg_Thread *self = Cyg_Thread::self();
    int read_owner_idx = -1;
    int i;

    mutex->lock();
    for (i = (num_active_rds-1); i >= 0 ; i--)
    {
        if (read_owners[i] == self)
            read_owner_idx = i;
    }
    // The current thread already owns the read-write lock for writing or reading.
    if ( (self == write_owner) || (read_owner_idx != -1) )
        result = EDEADLK;
    else
    {
        num_pend_wrs++;
        // keep waiting when there is a writer or active reader
        while ( write_owner || num_active_rds )
        {
            wcond->wait();
        }
        num_pend_wrs--;
        write_owner = Cyg_Thread::self();
    }
    mutex->unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_REPORT_RETVAL(result);
    return result;
}

cyg_int32
Cyg_RW_Lock::tryrdlock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_int32 result = ENOERR;
    Cyg_Thread *self = Cyg_Thread::self();

    mutex->lock();
    // The read-write lock could not be acquired for reading because a writer holds the lock or was blocked on it.
    if ( write_owner ||num_pend_wrs )
        result = EBUSY;
    // The read lock could not be acquired because the maximum number of read locks for rwlock has been exceeded.
    else if ( (num_active_rds + num_pend_rds) >= MAX_READ_OWNERS_LIST_SIZE )
        result = EAGAIN;
    else
        read_owners[num_active_rds++] = self;
    mutex->unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_REPORT_RETVAL(result);
    return result;
}


cyg_int32
Cyg_RW_Lock::trywrlock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_int32 result = ENOERR;
    Cyg_Thread *self = Cyg_Thread::self();
    int read_owner_idx = -1;
    int i;

    mutex->lock();
    for (i = (num_active_rds-1); i >= 0 ; i--)
    {
        if (read_owners[i] == self)
            read_owner_idx = i;
    }
    // The read-write lock could not be acquired for writing because it was already locked for reading or writing.
    if ( write_owner || num_active_rds )
        result = EBUSY;
    else
        write_owner = self;
    mutex->unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_REPORT_RETVAL(result);
    return result;
}

cyg_int32
Cyg_RW_Lock::unlock(void)
{
    CYG_REPORT_FUNCTYPE("returning %d");

    cyg_int32 result = ENOERR;
    Cyg_Thread *self = Cyg_Thread::self();
    int read_owner_idx = -1;
    int i;
    cyg_bool do_unlock = false;

    mutex->lock();
    for (i = (num_active_rds-1); i >= 0 ; i--)
    {
        if (read_owners[i] == self)
            read_owner_idx = i;
    }
    // The current thread does not own the read-write lock.
    if (write_owner != self && (read_owner_idx == -1))
        result = EPERM;
    else
    {
        // write unlock
        if (write_owner)
        {
           do_unlock = true;
           write_owner = NULL;
        }
        else
        // read unlock
        {
            for (i = read_owner_idx; i < (num_active_rds-1); i++)
                read_owners[i] = read_owners[i+1];
            num_active_rds--;
            if (num_active_rds == 0)
                do_unlock = true;
        }

        if (do_unlock)
        {
            // write-preferred
            if (kind != eKIND_READ_PREFERRED)
            {
                if (num_pend_wrs)
                    wcond->signal();      // wake highest priority pending writer or first pending writer
                else if (num_pend_rds)
                    rcond->broadcast();   // wake all pending reader
            }
            // read-preferred
            else
            {
                if (num_pend_rds)
                    rcond->broadcast();   // wake all pending reader
                else if (num_pend_wrs)
                {
                    wcond->signal();      // wake highest priority pending writer or first pending writer
                }
            }
        }
    }
    mutex->unlock();

    CYG_ASSERTCLASS( this, "Bad this pointer");
    CYG_REPORT_RETVAL(result);
    return result;
};

// -------------------------------------------------------------------------
// EOF sync/mutex.cxx
