//****************************************************************************
//
// Copyright (c) 2021 Broadcom Corporation
//
// This program is the proprietary software of Broadcom Corporation and/or
// its licensors, and may only be used, duplicated, modified or distributed
// pursuant to the terms and conditions of a separate, written license
// agreement executed between you and Broadcom (an "Authorized License").
// Except as set forth in an Authorized License, Broadcom grants no license
// (express or implied), right to use, or waiver of any kind with respect to
// the Software, and Broadcom expressly reserves all rights in and to the
// Software and all intellectual property rights therein.  IF YOU HAVE NO
// AUTHORIZED LICENSE, THEN YOU HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY,
// AND SHOULD IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE
// SOFTWARE.
//
// Except as expressly set forth in the Authorized License,
//
// 1.     This program, including its structure, sequence and organization,
// constitutes the valuable trade secrets of Broadcom, and you shall use all
// reasonable efforts to protect the confidentiality thereof, and to use this
// information only in connection with your use of Broadcom integrated circuit
// products.
//
// 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED
// "AS IS" AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS
// OR WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH
// RESPECT TO THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL
// IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR
// A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET
// ENJOYMENT, QUIET POSSESSION OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME
// THE ENTIRE RISK ARISING OUT OF USE OR PERFORMANCE OF THE SOFTWARE.
//
// 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM
// OR ITS LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL,
// INDIRECT, OR EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY
// RELATING TO YOUR USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM
// HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN
// EXCESS OF THE AMOUNT ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1,
// WHICHEVER IS GREATER. THESE LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY
// FAILURE OF ESSENTIAL PURPOSE OF ANY LIMITED REMEDY.
//
//*****************************************************************************
//
//  Filename:       poll.cxx
//  Author:         Ignatius Cheng
//  Creation Date:  July 12, 2021
//
//****************************************************************************
//  Description:
//      This is BRCM implmenetation of poll() for eCos
//
//****************************************************************************
//==========================================================================
//
//      poll.cxx
//
//      Fileio poll() support
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros

#include <stdarg.h>                    // for fcntl()

#include "fio.h"                       // Private header

#include <sys/poll.h>                // poll header

#include <cyg/kernel/sched.hxx>        // scheduler definitions
#include <cyg/kernel/thread.hxx>       // thread definitions
#include <cyg/kernel/mutex.hxx>        // mutex definitions
#include <cyg/kernel/clock.hxx>        // clock definitions

#include <cyg/kernel/sched.inl>
#include <cyg/kernel/thread.inl>
#include <cyg/kernel/clock.inl>

//==========================================================================
// millisecond to ticks conversion support

// Converters from millisecond to ticks
static struct Cyg_Clock::converter ms_converter;

static cyg_bool converters_initialized = false;

externC cyg_tick_count cyg_ms_to_ticks( const int ms )
{
    if( !converters_initialized )
    {
        // Create the converters we need.
        Cyg_Clock::real_time_clock->get_other_to_clock_converter( 1000000, &ms_converter );
        converters_initialized = true;
    }

    // Short circuit zero ms
    if( ms == 0 )
    {
        return 0;
    }

    // Convert the millisecond field to ticks.
    cyg_tick_count ticks = Cyg_Clock::convert( ms, &ms_converter );

    return ticks;
}

//==========================================================================
// Poll API function

static int
cyg_ppoll (struct pollfd *fds, nfds_t nfds,
           int ms, const sigset_t *mask)
{
    FILEIO_ENTRY();

    int error = ENOERR;
    int fd, num;
    unsigned long int i;
    cyg_file *fp;
    cyg_tick_count ticks;
    struct pollfd *cur_fd;
    cyg_uint32 wake_count;
    sigset_t oldmask;

    // Compute end time
    if (ms >= 0)
        ticks = cyg_ms_to_ticks( ms );
    else ticks = ((cyg_tick_count)-1);

    // Lock the mutex
    cyg_selmutexlock();

    // Scan sets for possible I/O until something found, timeout or error.
    while (!error)
    {
        wake_count = cyg_selwakecount();
        num = 0;  // Total file descriptors "ready"
        for (i = 0, cur_fd = fds; i < nfds;  i++, cur_fd++)
        {
            fd = cur_fd->fd;
            cur_fd->revents = 0;  // clear the result
            if ( fd < 0)  // ignore fd
                continue;

            fp = cyg_fp_get( fd );
            if( fp == NULL )
            {
                cur_fd->revents |= POLLNVAL;
                continue;
            }
            if (cur_fd->events & POLLIN)
            {
                if ((*fp->f_ops->fo_select)(fp, CYG_FREAD, 0))
                    cur_fd->revents |= POLLIN;
            }
            if (cur_fd->events & POLLOUT)
            {
                if ((*fp->f_ops->fo_select)(fp, CYG_FWRITE, 0))
                    cur_fd->revents |= POLLOUT;
            }
            if (cur_fd->events & POLLPRI)
            {
                if ((*fp->f_ops->fo_select)(fp, 0, 0))
                    cur_fd->revents |= POLLPRI;
            }
            cyg_fp_free( fp );

            if (cur_fd->revents)
                num++;
        }

        if (num)
        {
            // Found something, return
            cyg_selmutexunlock();
            CYG_FILEIO_DELIVER_SIGNALS( mask );
            FILEIO_RETURN_VALUE(num);
        }

        Cyg_Scheduler::lock();

        // Switch to the supplied signal mask. This will permit delivery
        // of any signals that might terminate this poll operation.
        CYG_FILEIO_SIGMASK_SET( mask, &oldmask );

        do
        {

            // We need to see if any signals have been posted while we
            // were testing all those files. The handlers will not
            // have run because we have ASRs inhibited but the signal
            // will have been set pending.

            if( CYG_FILEIO_SIGPENDING() )
            {
                // There are pending signals so we need to terminate
                // the poll operation and return EINTR. Handlers for
                // the pending signals will be called just before we
                // return.
                error = EINTR;
                break;
            }

            if( wake_count == cyg_selwakecount() )
            {
                // Special case of "poll"
                if (ticks == 0) {
                    error = EAGAIN;  // timeout
                    break;
                }

                // Nothing found, see if we want to wait
                if (ms > 0)
                {
                    ticks += Cyg_Clock::real_time_clock->current_value();
                    if( !cyg_seltimedwait( ticks ) )
                    {
                        // A non-standard wakeup, if the current time is equal to
                        // or past the timeout, return zero. Otherwise return
                        // EINTR, since we have been released.

                        if( Cyg_Clock::real_time_clock->current_value() >= ticks )
                        {
                            error = EAGAIN;  // timeout
                            break;
                        }
                        else error = EINTR;
                    }

                    ticks -= Cyg_Clock::real_time_clock->current_value();
                }
                else
                {
                    // Wait forever (until something happens)
                    if( !cyg_selwait() )
                        error = EINTR;
                }
            }
        } while(0);

        CYG_FILEIO_SIGMASK_SET( &oldmask, NULL );

        Cyg_Scheduler::unlock();

    } // while(!error)

    cyg_selmutexunlock();

    // If the error code is EAGAIN, this means that a timeout has
    // happened. We return zero in that case, rather than a proper
    // error code.
    // If the error code is EINTR, then a signal may be pending
    // delivery. Call back into the POSIX package to handle it.

    if( error == EAGAIN )
        FILEIO_RETURN_VALUE(0);
    else if( error == EINTR )
        CYG_FILEIO_DELIVER_SIGNALS( mask );

    FILEIO_RETURN(error);
}

// -------------------------------------------------------------------------
// Select API function

__externC int
poll (struct pollfd *fds, nfds_t nfds, int timeout)
{
    return cyg_ppoll(fds, nfds, timeout, NULL);
}

// -------------------------------------------------------------------------
// Pselect API function
//
// This is derived from the POSIX-200X specification.

#ifdef CYGPKG_POSIX

__externC int
ppoll (struct pollfd *fds, nfds_t nfds,
       const struct timespec *ts, const sigset_t *sigmask)
{
    int ms = -1;
    if (ts != NULL) {
        ms = ts->tv_sec * 1000;
        ms += ts->tv_nsec / 1000000;
    }

    return cyg_ppoll(fds, nfds, ms, sigmask);
}

#endif

// -------------------------------------------------------------------------
// EOF poll.cxx
